from autocoder.common.auto_coder_lang import get_message_with_format
from typing import List, Dict, Union

def read_file(file_path):
    """Read a file with automatic encoding detection.
    
    Tries common encodings in sequence (UTF-8 > GBK > UTF-16 > Latin-1) to handle
    cross-platform encoding issues between Windows and Linux systems.
    
    Args:
        file_path (str): Path to the file to read
        
    Returns:
        str: The file contents as a string
        
    Raises:
        ValueError: If the file cannot be decoded with any of the tried encodings
    """
    encodings = ['utf-8', 'gbk', 'utf-16', 'latin-1']
    
    for encoding in encodings:
        try:
            with open(file_path, 'r', encoding=encoding) as f:
                content = f.read()
                return content
        except UnicodeDecodeError:
            continue
            
    raise ValueError(get_message_with_format("file_decode_error", 
        file_path=file_path, 
        encodings=", ".join(encodings)))



def save_file(file_path: str, content: Union[str, List[str]]) -> None:
    """Save content to a file using UTF-8 encoding.
    
    Args:
        file_path (str): Path to the file to write
        content (Union[str, List[str]]): Content to write to the file. 
            Can be a string or list of strings (will be joined with newlines)
        
    Raises:
        IOError: If the file cannot be written
        TypeError: If content is neither str nor List[str]
    """
    try:
        with open(file_path, 'w', encoding='utf-8') as f:
            if isinstance(content, str):
                f.write(content)
            elif isinstance(content, list):
                f.write('\n'.join(content))
            else:
                raise TypeError("Content must be either str or List[str]")
    except IOError as e:
        raise IOError(get_message_with_format("file_write_error",
            file_path=file_path,
            error=str(e)))