import geoip2.database
import geoip2.webservice

import os
from pydantic import BaseModel
from tracardi.domain.entity import Entity

local = os.path.dirname(__file__)


class PluginConfiguration(BaseModel):
    source: Entity


class GeoLiteCredentials(BaseModel):
    accountId: int
    license: str
    host: str = 'geolite.info'


class GeoIpConfiguration(BaseModel):
    webservice: GeoLiteCredentials = None
    database: str = None

    def get_local_database(self):
        return os.path.join(local, f'../database/{self.database}')

    def is_local(self):
        return self.database is not None

    def is_remote(self):
        return self.webservice is not None


# class MaxMindGeoLite2Client(metaclass=Singleton):
class MaxMindGeoLite2Client:

    def __init__(self, credentials: GeoLiteCredentials):
        self.client = geoip2.webservice.AsyncClient(credentials.accountId, credentials.license, host=credentials.host)

    async def read(self, ip: str):
        return await self.client.city(ip)

    async def close(self):
        await self.client.close()


class MaxMindGeoLite2Reader:

    def __init__(self, database):
        self.reader = geoip2.database.Reader(database)

    def read(self, ip: str):
        return self.reader.city(ip)

    def __del__(self):
        self.reader.close()


class MaxMindGeoLite2:

    def __init__(self, config: GeoIpConfiguration):
        self.config = config
        if self.config.is_local():
            self.client = MaxMindGeoLite2Reader(database=self.config.get_local_database())
        elif self.config.is_remote():
            self.client = MaxMindGeoLite2Client(credentials=self.config.webservice)
        else:
            raise ValueError("Misconfiguration of MaxMindGeoLite2.")

    async def fetch(self, ip):
        result = self.client.read(ip)
        if isinstance(self.client, MaxMindGeoLite2Client):
            result = await result
        return result
