# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/core.ipynb.

# %% auto 0
__all__ = ['session', 'set_headers', 'get_session', 'prc_request_cf_data', 'prc_clean_cf_data', 'prc_get_cf_fields',
           'get_cf_info', 'get_cf_options', 'cf_option_name']

# %% ../nbs/core.ipynb 5
import requests, json
from . import config
import pandas as pd
from tqdm import tqdm

# %% ../nbs/core.ipynb 6
def set_headers(AccessToken:str, #Access Token (API Key) provided by Copper
                 UserEmail:str): #Email associated with your API key
    """
    Sets global variable to hold header information for Copper API requests.
    This must be set before any requests are made. See documentation for more info.

    Parameters:
    AccessToken: type=str
        This is your Access Token provided by Copper. These can be found and created 
        under Settings > Intergrations > API Keys. 

    UserEmail: type=str
        This is your the email associated with your Copper account.
        
    """

    assert isinstance(AccessToken, str), 'AccessToken must be a string. \nSee help(set_headers) for more info.'
    assert isinstance(UserEmail, str), 'UserEmail must be a string. \nSee help(set_headers) for more info.'

    config.COPPER_HEADERS = {
        'X-PW-AccessToken':AccessToken,
        'X-PW-Application':'developer_api',
        'X-PW-UserEmail': UserEmail,
        'Content-Type':'application/json'
        }

# %% ../nbs/core.ipynb 8
session = None

def get_session(**kwargs):
    """
    Function to get current session or create one if one doesn't exist. If both keyword 
    arguments are provided, then the set_headers() function is called.
    
    Optional Keyword Arguments:
    
    AccessToken: type=str
        This is your Access Token provided by Copper. These can be found and created 
        under Settings > Integrations > API Keys. 

    UserEmail: type=str
        This is your the email associated with your Copper account.
    """
    
    if 'AccessToken' in kwargs and 'UserEmail' in kwargs:
        set_headers(kwargs.get('AccessToken'), kwargs.get('UserEmail'))
    
    if not hasattr(config, 'COPPER_HEADERS'):
        raise NameError("Header information is not set. Please use set_header() function to do so. \n\nSee ?set_headers for more information.")
    
    if not hasattr(config, 'SESSION') or not isinstance(config.SESSION, requests.Session):
        # No session exists or the global variable is not of type requests.Session, create a new session
        config.SESSION = requests.Session()
        config.SESSION.headers.update(config.COPPER_HEADERS)  # Assuming config.COPPER_HEADERS is defined elsewhere
    
    return config.SESSION

# %% ../nbs/core.ipynb 11
def prc_request_cf_data():
    """
    Helpter function to request the custom field data.
    """
    Sess = get_session()

    try:
        cf_request = Sess.get('https://api.copper.com/developer_api/v1/custom_field_definitions')
        cf_request.raise_for_status()
    except requests.exceptions.HTTPError as err:
        raise Exception(f"Failed to fetch custom field data: {err}") from err
    else:
        custom_fields_data = cf_request.json()
        return custom_fields_data

def prc_clean_cf_data(data:json):
    """
    Function to process the custom field data.
    """

    output_dict = {}

    for item in data:
        item_dict = {}

        item_dict['name'] = item.get('name')
        item_dict['data_type'] = item.get('data_type')
        item_dict['available_on'] = item.get('available_on')
        item_dict['is_filterable'] = item.get('is_filterable')

        if 'options' in item:
            item_options = {}
            for sub_item in item['options']:
                sub_item_name = sub_item['name']
                sub_item_id = sub_item['id']

                item_options[sub_item_id] = sub_item_name
        
            item_dict['options'] = item_options
        
        item_id = item['id']
        output_dict[item_id] = item_dict

    return output_dict

# %% ../nbs/core.ipynb 13
def prc_get_cf_fields():
    """
    Checks if custom_fields has been set and creates it if it hasn't been set. 
    Called by all Custom Field Functions.
    """

    if  not hasattr(config,"CUSTOM_FIELDS") or not hasattr(config,"CUSTOM_FIELDS_DICT") or not hasattr(config,"CF_ID_LOOKUP"):
        
        raw_cf_data = prc_request_cf_data()
        custom_fields = prc_clean_cf_data(raw_cf_data)
    
        # Setup Items:

        custom_fields_dict = {}
        reverse_id_lookup = {}

        for key in custom_fields.keys():
            item_name = custom_fields[key].get('name')
            custom_fields_dict[key] = item_name
            reverse_id_lookup[item_name] = key

        custom_fields_list = list(custom_fields_dict.values())

        config.CUSTOM_FIELDS = custom_fields              # Full data in JSON form
        config.CUSTOM_FIELDS_DICT = custom_fields_dict    # ID -> Name
        config.LIST_CF_NAMES = custom_fields_list         # List of Names
        config.CF_ID_LOOKUP = reverse_id_lookup           # Name -> ID

# %% ../nbs/core.ipynb 15
def get_cf_info(cf_id:str,     # ID of custom field
                cf_info:list = None,  # Designed information about field, list if multiple items
               )->list: #Returns list if cf_info is list. Otherwise, returns value
    """
    Function to get the custom field information based on the field id.
    Can be supplied with a sigular value for 'cf_info' or multiple as a list.

    Possible itmes are ['name','data_type','available_on','is_filterable','options']
    """

    prc_get_cf_fields()

    if isinstance(cf_id,str):   cf_id = int(cf_id)    

    # Check if cf_info item(s) are valid
    if not cf_info:
        cf_info = ['name','data_type','available_on','is_filterable','options']  
    elif isinstance(cf_info, list):
        for item_name in cf_info:
            # Remove faulty items:
            if item_name not in ['name','data_type','available_on','is_filterable','options']: cf_info.pop(item_name)
            
            # Cancel function if no valid items
            if len(cf_info) == 0: return None
    elif cf_info not in ['name','data_type','available_on','is_filterable','options']:
        print(f'Invalid cf_info: {cf_info}'); return None

    if isinstance(cf_info,list):
        return_list = []
        for item_name in cf_info:
            value = config.CUSTOM_FIELDS[cf_id].get(item_name)
            return_list.append(value)
        return return_list
    else:
        value = config.CUSTOM_FIELDS[cf_id].get(cf_info)
        return value

def get_cf_options(cf_id:int, #Coppper ID for custom field
    )->dict: # Returns dictionary of options for field
    return config.CUSTOM_FIELDS.get(cf_id).get('options')

def cf_option_name(cf_id:int, #Coppper ID for custom field
                   option_id:int, #Coppper ID for option
                  )->str: #Returns name/value of option
    return get_cf_options(cf_id).get(option_id)

