"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const log_retention_1 = require("./log-retention");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _a;
                    (_a = this.role) === null || _a === void 0 ? void 0 : _a.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfilingEnvironmentVariables(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfilingEnvironmentVariables(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup');
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        this.environment = { ...profilingGroupEnvironmentVariables, ...(props.environment || {}) };
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logretention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            resource.addPropertyOverride('FileSystemConfigs', [
                {
                    LocalMountPath: config.localMountPath,
                    Arn: config.arn,
                },
            ]);
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        return this._currentVersion;
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     */
    addEnvironment(key, value) {
        this.environment[key] = value;
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logretention = new log_retention_1.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logretention.logGroupArn);
        }
        return this._logGroup;
    }
    prepare() {
        super.prepare();
        // if we have a current version resource, override it's logical id
        // so that it includes the hash of the function code and it's configuration.
        if (this._currentVersion) {
            const stack = core_1.Stack.of(this);
            const cfn = this._currentVersion.node.defaultChild;
            const originalLogicalId = stack.resolve(cfn.logicalId);
            const hash = function_hash_1.calculateFunctionHash(this);
            const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
            cfn.overrideLogicalId(`${logicalId}${hash}`);
        }
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        // for backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        if (!this._currentVersion) {
            return {
                variables: this.environment,
            };
        }
        // sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it).
        const variables = {};
        for (const key of Object.keys(this.environment).sort()) {
            variables[key] = this.environment[key];
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        // Pick subnets, make sure they're not Public. Routing through an IGW
        // won't work because the ENIs don't get a Public IP.
        // Why are we not simply forcing vpcSubnets? Because you might still be choosing
        // Isolated networks or selecting among 2 sets of Private subnets by name.
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId)) {
                throw new Error('Not possible to place Lambda Functions in a Public subnet');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfilingEnvironmentVariables(props) {
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error('lambda.Code must specify one of "inlineCode" or "s3Location" but not both');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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