"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.fromAsset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
            role: props.role,
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        provider.addToRolePolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: (_a = retryOptions.base) === null || _a === void 0 ? void 0 : _a.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === logs.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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