"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
const roleAccount = '123456789012';
const notRoleAccount = '012345678901';
describe('IAM Role.fromRoleArn', () => {
    let app;
    beforeEach(() => {
        app = new core_1.App();
    });
    let roleStack;
    let importedRole;
    describe('imported with a static ARN', () => {
        const roleName = 'MyRole';
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(`arn:aws:iam::${roleAccount}:role/${roleName}`);
            });
            test('correctly parses the imported role name', () => {
                expect(importedRole.roleName).toBe(roleName);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack, with account set to', () => {
                    describe('the same account as in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("correctly attaches the Policy to the imported role", () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than in the ARN of the imported role', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("does NOT attach the Policy to the imported role", () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                });
            });
        });
        describe('into a targeted stack with account set to', () => {
            describe('the same account as in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('returns true', () => {
                        expect(addToPolicyResult).toBe(true);
                    });
                    test("generates a default Policy resource pointing at the imported role's physical name", () => {
                        assertRoleHasDefaultPolicy(roleStack, roleName);
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test("correctly attaches the Policy to the imported role", () => {
                            assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("correctly attaches the Policy to the imported role", () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test("correctly attaches the Policy to the imported role", () => {
                                assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                            });
                        });
                        describe('a different account than in the imported role ARN and in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test("does NOT attach the Policy to the imported role", () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
            describe('a different account than in the ARN the role was imported with', () => {
                beforeEach(() => {
                    roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: notRoleAccount } });
                    importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`);
                });
                describe('then adding a PolicyStatement to it', () => {
                    let addToPolicyResult;
                    beforeEach(() => {
                        addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                    });
                    test('returns false', () => {
                        expect(addToPolicyResult).toBe(false);
                    });
                    test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                        expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                    });
                });
                describe('then attaching a Policy to it', () => {
                    describe('that belongs to the same stack as the imported role', () => {
                        beforeEach(() => {
                            importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                        });
                        test("does NOT attach the Policy to the imported role", () => {
                            assertPolicyDidNotAttachToRole(roleStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to an env-agnostic stack', () => {
                        let policyStack;
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack');
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("does NOT attach the Policy to the imported role", () => {
                            assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                        });
                    });
                    describe('that belongs to a different targeted stack, with account set to', () => {
                        let policyStack;
                        describe('the same account as in the ARN of the imported role', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test("does NOT attach the Policy to the imported role", () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('the same account as in the stack the imported role belongs to', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test("does NOT attach the Policy to the imported role", () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                        describe('a third account, different from both the role and scope stack accounts', () => {
                            beforeEach(() => {
                                policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: 'some-random-account' } });
                                importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                            });
                            test("does NOT attach the Policy to the imported role", () => {
                                assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                            });
                        });
                    });
                });
            });
        });
        describe('and with mutable=false', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', `arn:aws:iam::${roleAccount}:role/${roleName}`, { mutable: false });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns false', () => {
                    expect(addToPolicyResult).toBe(false);
                });
                test("does NOT generate a default Policy resource pointing at the imported role's physical name", () => {
                    expect(roleStack).not.toHaveResourceLike('AWS::IAM::Policy');
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to a stack with account equal to the account in the imported role ARN', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test("does NOT attach the Policy to the imported role", () => {
                        assertPolicyDidNotAttachToRole(policyStack, 'MyPolicy');
                    });
                });
            });
        });
    });
    describe('imported with a dynamic ARN', () => {
        const dynamicValue = core_1.Lazy.stringValue({ produce: () => 'role-arn' });
        const roleName = {
            "Fn::Select": [1,
                {
                    "Fn::Split": ["/",
                        {
                            "Fn::Select": [5,
                                { "Fn::Split": [":", "role-arn"] },
                            ]
                        },
                    ],
                },
            ],
        };
        describe('into an env-agnostic stack', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack');
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            test('correctly parses the imported role ARN', () => {
                expect(importedRole.roleArn).toBe(dynamicValue);
            });
            test('correctly parses the imported role name', () => {
                new lib_1.Role(roleStack, 'AnyRole', {
                    roleName: 'AnyRole',
                    assumedBy: new lib_1.ArnPrincipal(importedRole.roleName),
                });
                expect(roleStack).toHaveResourceLike('AWS::IAM::Role', {
                    "AssumeRolePolicyDocument": {
                        "Statement": [
                            {
                                "Action": "sts:AssumeRole",
                                "Effect": "Allow",
                                "Principal": {
                                    "AWS": roleName,
                                },
                            },
                        ],
                    },
                });
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a targeted stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
            });
        });
        describe('into a targeted stack with account set', () => {
            beforeEach(() => {
                roleStack = new core_1.Stack(app, 'RoleStack', { env: { account: roleAccount } });
                importedRole = lib_1.Role.fromRoleArn(roleStack, 'ImportedRole', dynamicValue);
            });
            describe('then adding a PolicyStatement to it', () => {
                let addToPolicyResult;
                beforeEach(() => {
                    addToPolicyResult = importedRole.addToPolicy(somePolicyStatement());
                });
                test('returns true', () => {
                    expect(addToPolicyResult).toBe(true);
                });
                test("generates a default Policy resource pointing at the imported role's physical name", () => {
                    assertRoleHasDefaultPolicy(roleStack, roleName);
                });
            });
            describe('then attaching a Policy to it', () => {
                let policyStack;
                describe('that belongs to the same stack as the imported role', () => {
                    beforeEach(() => {
                        importedRole.attachInlinePolicy(somePolicy(roleStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(roleStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to an env-agnostic stack', () => {
                    beforeEach(() => {
                        policyStack = new core_1.Stack(app, 'PolicyStack');
                        importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                    });
                    test("correctly attaches the Policy to the imported role", () => {
                        assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                    });
                });
                describe('that belongs to a different targeted stack, with account set to', () => {
                    describe('the same account as the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: roleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("correctly attaches the Policy to the imported role", () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                    describe('a different account than the stack the role was imported into', () => {
                        beforeEach(() => {
                            policyStack = new core_1.Stack(app, 'PolicyStack', { env: { account: notRoleAccount } });
                            importedRole.attachInlinePolicy(somePolicy(policyStack, 'MyPolicy'));
                        });
                        test("correctly attaches the Policy to the imported role", () => {
                            assertRoleHasAttachedPolicy(policyStack, roleName, 'MyPolicy');
                        });
                    });
                });
            });
        });
    });
});
function somePolicyStatement() {
    return new lib_1.PolicyStatement({
        actions: ['s3:*'],
        resources: ['xyz'],
    });
}
function somePolicy(policyStack, policyName) {
    const someRole = new lib_1.Role(policyStack, 'SomeExampleRole', {
        assumedBy: new lib_1.AnyPrincipal(),
    });
    const roleResource = someRole.node.defaultChild;
    roleResource.overrideLogicalId('SomeRole'); // force a particular logical ID in the Ref expression
    return new lib_1.Policy(policyStack, 'MyPolicy', {
        policyName,
        statements: [somePolicyStatement()],
        // need at least one of user/group/role, otherwise validation fails
        roles: [someRole],
    });
}
function assertRoleHasDefaultPolicy(stack, roleName) {
    _assertStackContainsPolicyResource(stack, [roleName], undefined);
}
function assertRoleHasAttachedPolicy(stack, roleName, attachedPolicyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }, roleName], attachedPolicyName);
}
function assertPolicyDidNotAttachToRole(stack, policyName) {
    _assertStackContainsPolicyResource(stack, [{ Ref: 'SomeRole' }], policyName);
}
function _assertStackContainsPolicyResource(stack, roleNames, nameOfPolicy) {
    const expected = {
        PolicyDocument: {
            Statement: [
                {
                    Action: "s3:*",
                    Effect: "Allow",
                    Resource: "xyz",
                },
            ],
        },
        Roles: roleNames,
    };
    if (nameOfPolicy) {
        expected.PolicyName = nameOfPolicy;
    }
    expect(stack).toHaveResourceLike('AWS::IAM::Policy', expected);
}
//# sourceMappingURL=data:application/json;base64,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