"""GangaCron submit / finish / extract / report implementations.

The classes GangaCronSubmitter, GangaCronExtractor and GangaCronReporter
together replicate the functionality of GangaCron, which was the predecessor to
GangaRobot. As such they may be interesting as an example of how to specialise
the generic submit / finish / extract / report implementations provided in
GangaRobot.Lib.Core.

See GANGACRON.INI for suitable configuration options.

"""

from GangaRobot.Lib.Base.BaseSubmitter import BaseSubmitter
from GangaRobot.Lib.Base.BaseExtractor import BaseExtractor
from GangaRobot.Lib.Base.BaseReporter import BaseReporter
from GangaRobot.Lib.Base.Extract import Node
from GangaRobot.Lib.Base.Report import Heading, Pre, Table
from GangaRobot.Lib.Core.CoreExtractor import CoreExtractor
from GangaRobot.Lib.Core.CoreReporter import CoreReporter
from GangaRobot.Lib.Ext.DiracExtractor import DiracExtractor
from GangaRobot.Framework import Utility
from Ganga.Utility import files
from Ganga.GPI import *
import glob
import os



class GangaCronSubmitter(BaseSubmitter):

    def handlesubmit(self, jobids, runid):
        """Submit 10 DaVinci/Dirac jobs with 4 LFN files, for each CPUTime 3599 and 86399 seconds."""
        JOB_COUNT = 1 #default 10
        CHUNK_SIZE = 1 # default 4
        CPU_TIMES = [3599, 86399]
        for cputime in CPU_TIMES:
            for index in range(JOB_COUNT):
                j = self._createjob(cputime, index, CHUNK_SIZE)
                j.submit()
                jobids.append(j.id)

    def _createjob(self, cputime, chunkindex, chunksize):
        import time
        j = Job()
        j.name = '%d-%d' % (cputime, time.time())
        cmt_user_path = files.expandfilename('~/cmtuserRobot')
        optsfile = os.path.join(cmt_user_path, 'DaVinci_v23r3p1/Analysis/solutions/DaVinci2/DVTutorial_2.py')
        j.application = DaVinci(
            extraopts = 'DaVinci().EvtMax = 2500;\nDaVinci().DataType = "DC06"',
            platform = 'slc4_ia32_gcc34',
            package = 'Phys',
            masterpackage = None,
            version = 'v23r3p1',
            optsfile = optsfile)
        j.backend = Dirac(
            CPUTime = cputime)
        j.inputdata = _getinputdata(chunkindex, chunksize)
        return j


class GangaCronExtractor(BaseExtractor):

    def __init__(self):
        self.chain = [CoreExtractor(), DiracExtractor(), self]

    def handlerunnode(self, runnode, runid):
        """Add gangaCron node with mask node of OK / BANNED LCG sites."""
        gcnode = runnode.addnode('gangaCron')
        try:
            import urllib
            mask = urllib.urlopen('http://lhcb-project-dirac.web.cern.ch/lhcb-project-dirac/t1AnalysisMask.txt').read()
        except IOError:
            mask = 'failed to load url'
        gcnode.addnode('mask', mask)

    def handlejobnode(self, jobnode, job):
        """Add gangaCron node with events node of number of events processed (from stdout log)."""
        gcnode = jobnode.addnode('gangaCron')
        gcnode.addnode('events', self._extractevents(job))

    def _extractevents(self, job):
        events = None #default to None
        import re
        eventsre = re.compile('^ *(?P<events>\d+) events processed$')
        try:
            # search for most recent events match in stdout file
            filename = os.path.join(job.outputdir, 'stdout')
            file = open(filename, 'r')
            try:
                for line in file:
                    match = eventsre.search(line)
                    if match:
                        events = int(match.group('events'))
            finally:
                file.close()
        except IOError:
            pass #file not found or unreadable
        return events


class GangaCronReporter(BaseReporter):

    def __init__(self):
        self.chain = [self, CoreReporter()]

    def handlereport(self, report, runnode):
        """Add mask of OK / BANNED LCG sites and Successes / Successes History tables based on Ganga status, Dirac status and events processed."""
        HISTORY_COUNT = 10
        
        runid = runnode.getvalue('core.id')
        
        #GangaCron Report id.
        report.title = 'GangaCron Report ' + runid
        
        #GangaCron Analysis
        report.addline(Heading('GangaCron Analysis', 2))
        report.addline()
        
        #t1AnalysisMask
        report.addline(Heading('t1AnalysisMask'))
        report.addline(Pre(runnode.getvalue('gangaCron.mask')))
        report.addline()
        
        #Successes
        report.addline(Heading('Successes'))
        #Criterion | Total
        #...
        report.addline(self._getsuccessestable(runnode))
        report.addline()
        
        #Successes History (upto 10 previous runs)
        report.addline(Heading('Successes History (upto %d previous runs)' % HISTORY_COUNT))
        #Start time | Ganga=completed | Dirac=outputready | Events=2500 | #jobs
        #...
        report.addline(self._getsuccesseshistorytable(runid, HISTORY_COUNT))
        report.addline()

    def _getsuccessestable(self, runnode):
        t = Table()
        t.hformat = '%-17s | %5s'
        t.header = ('Criterion', 'Total')
        t.rformat = '%-17s | %5d'
        successestuple = self.__getsuccessestuple(runnode)
        t.addrow('Ganga=completed', successestuple[0])
        t.addrow('Dirac=outputready', successestuple[1])
        t.addrow('Events=2500', successestuple[2])
        t.addrow('Job count', successestuple[3])
        return t

    def _getsuccesseshistorytable(self, runid, historycount):
        t = Table()
        t.hformat = '%-19s | %15s | %17s | %11s | %9s'
        t.header = ('Start time', 'Ganga=completed', 'Dirac=outputready', 'Events=2500', 'Job count')
        t.rformat = '%-19s | %15d | %17d | %11d | %9d'
        historyrunnodes = self.__loadhistoryrunnodes(runid, historycount)
        for runnode in historyrunnodes:
            successestuple = self.__getsuccessestuple(runnode)
            t.addrow(runnode.getvalue('core.start-time'), successestuple[0],
                     successestuple[1], successestuple[2], successestuple[3])
        return t

    def __loadhistoryrunnodes(self, runid, historycount):
        """Return the previous runnodes up to a maximum of historycount."""
        #find current xml filename
        xmltokenpattern = self.getoption('BaseExtractor_XmlFile')
        currentxmlfilename = files.expandfilename(Utility.expand(xmltokenpattern, runid = runid))
        #find all xml filenames
        xmlglobpattern = files.expandfilename(Utility.expand(xmltokenpattern, runid = '*'))
        allxmlfilenames = glob.glob(xmlglobpattern)
        #sort into latest first
        allxmlfilenames.sort()
        allxmlfilenames.reverse()
        #find index of current xml filename
        i = allxmlfilenames.index(currentxmlfilename)
        #select previous xmlfilenames upto historycount
        historyxmlfilenames = allxmlfilenames[i+1:i+1+historycount]
        #load xml as nodes
        historyxmls = [Utility.readfile(filename) for filename in historyxmlfilenames]
        return [Node.fromxml(xml) for xml in historyxmls]

    def __getsuccessestuple(self, runnode):
        """Return tuple of totals (Ganga=completed, Dirac=outputready, Events=2500, Jobs) for given runnode."""
        jobnodes = runnode.getnodes('job')
        gangasuccesses = len([jobnode for jobnode in jobnodes if jobnode.getvalue('core.status') == 'completed'])
        diracsuccesses = len([jobnode for jobnode in jobnodes if jobnode.getvalue('dirac.status') == 'outputready'])
        eventsuccesses = len([jobnode for jobnode in jobnodes if jobnode.getvalue('gangaCron.events') == '2500'])
        jobs = len(jobnodes)
        return (gangasuccesses, diracsuccesses, eventsuccesses, jobs)


def _getinputdata(chunkindex, chunksize):
    """Return an LHCbDataset containing a chunk of LHCbDataFile with LFN references.
    
    Keyword Arguments:
    chunkindex -- The index of the chunk.
    chunksize -- The number of LFN references to include in each chunk.
    
    """
    lfns = [
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000001_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000002_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000003_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000004_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000005_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000007_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000008_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000009_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000010_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000011_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000012_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000013_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000014_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000015_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000016_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000017_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000018_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000019_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000020_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000021_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000022_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000024_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000025_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000026_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000027_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000028_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000029_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000030_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000031_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000032_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000033_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000034_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000035_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000036_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000037_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000038_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000039_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000040_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000041_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000042_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000043_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000044_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000046_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000047_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000048_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000049_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000050_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000051_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000052_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000053_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000054_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000056_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000057_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000058_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000059_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000060_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000061_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000062_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000063_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000064_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000065_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000066_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000068_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000070_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000072_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000074_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000075_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000076_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000077_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000078_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000079_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000080_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000081_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000083_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000085_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000086_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000087_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000089_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000090_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000091_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000092_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000093_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000094_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000095_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000096_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000097_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000100_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000101_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000102_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000103_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000104_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000105_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000106_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000108_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000109_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000112_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000114_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000115_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000116_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000117_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000118_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000119_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000120_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000121_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000122_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000124_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000125_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000127_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000128_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000129_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000130_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000131_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000132_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000134_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000135_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000136_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000137_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000138_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000139_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000141_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000142_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000143_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000144_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000145_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000147_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000149_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000150_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000151_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000152_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000153_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000154_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000156_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000157_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000159_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000160_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000162_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000163_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000165_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000166_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000167_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000170_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000171_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000172_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000174_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000175_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000176_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000177_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000178_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000179_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000180_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000181_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000182_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000183_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000184_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000185_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000186_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000187_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000188_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000189_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000190_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000191_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000192_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000193_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000196_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000197_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000198_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000199_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000200_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000202_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000203_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000205_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000206_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000207_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000208_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000209_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000210_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000211_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000212_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000214_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000215_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000216_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000218_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000219_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000220_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000221_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000222_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000223_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000225_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000226_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000227_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000228_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000229_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000230_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000231_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000232_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000233_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000234_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000235_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000236_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000237_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000240_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000241_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000242_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000243_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000244_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000245_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000246_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000247_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000249_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000250_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000252_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000253_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000254_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000255_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000256_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000258_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000259_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000260_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000261_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000262_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000263_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000264_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000265_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000266_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000267_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000268_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000269_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000270_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000273_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000274_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000275_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000276_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000277_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000278_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000279_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000280_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000282_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000283_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000284_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000285_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000286_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000287_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000288_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000290_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000291_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000292_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000293_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000294_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000295_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000296_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000298_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000299_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000301_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000303_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000304_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000305_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000306_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000307_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000308_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000309_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000310_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000311_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000312_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000313_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000314_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000315_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000316_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000317_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000320_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000321_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000322_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000323_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000325_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000326_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000327_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000328_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000329_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000330_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000331_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000332_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000333_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000334_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000335_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000336_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000337_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000338_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000340_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000341_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000342_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000343_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000344_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000345_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000346_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000347_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000348_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000350_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000351_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000352_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000353_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000354_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000355_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000356_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000358_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000359_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000360_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000363_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000364_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000365_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000366_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000367_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000368_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000369_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000370_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000371_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000372_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000373_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000374_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000375_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000377_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000378_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000379_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000380_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000382_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000383_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000384_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000385_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000386_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000387_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000388_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000389_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000390_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000391_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000392_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000393_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000394_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000395_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000396_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000397_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000398_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000399_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000401_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000403_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000404_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000405_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000406_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000408_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000409_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000410_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000411_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000413_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000414_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000415_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000416_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000417_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000418_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000419_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000420_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000421_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000422_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000423_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000424_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000425_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000426_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000427_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000428_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000429_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000430_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000431_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000432_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000433_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000435_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000436_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000437_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000438_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000439_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000440_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000441_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000442_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000443_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000444_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000445_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000447_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000448_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000449_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000450_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000451_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000452_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000453_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000454_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000455_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000456_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000457_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000458_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000459_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000460_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000461_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000462_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000463_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000464_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000465_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000466_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000467_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000469_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000470_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000471_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000472_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000473_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000474_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000476_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000477_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000478_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000479_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000481_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000482_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000483_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000484_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000485_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000486_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000487_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000488_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000489_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000490_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000491_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000493_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000494_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000495_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000496_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000497_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000498_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000499_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000501_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000503_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000504_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000505_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000506_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000507_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000508_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000509_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000510_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000511_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000512_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000513_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000514_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000516_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000517_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000518_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000519_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000520_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000521_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000522_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000524_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000525_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000526_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000528_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000529_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000530_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000531_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000533_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000534_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000535_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000536_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000537_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000538_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000539_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000540_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000541_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000542_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000544_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000545_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000546_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000547_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000548_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000549_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000550_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000551_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000552_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000553_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000554_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000555_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000556_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000557_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000559_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000560_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000561_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000563_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000564_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000565_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000568_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000569_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000570_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000571_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000572_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000574_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000575_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000576_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000577_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000578_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000581_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000583_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000584_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000585_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000586_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000589_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000591_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000592_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000593_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000594_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000595_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000596_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000599_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000600_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000601_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000602_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000603_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000604_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000605_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000606_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000607_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000608_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000609_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000610_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000611_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000612_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000613_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000614_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000615_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000616_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000621_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000623_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000625_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000626_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000628_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000629_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000630_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000631_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000632_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000633_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000634_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000635_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000636_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000637_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000638_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000639_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000641_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000643_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000644_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000645_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000646_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000647_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000649_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000650_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000651_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000652_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000653_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000654_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000655_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000656_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000658_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000659_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000662_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000663_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000664_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000665_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000666_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000667_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000668_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000670_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000671_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000672_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000674_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000675_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000676_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000677_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000678_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000680_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000681_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000682_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000683_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000684_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000685_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000686_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000687_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000688_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000689_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000691_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000692_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000693_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000694_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000695_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000696_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000697_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000699_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000700_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000702_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000704_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000705_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000706_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000707_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000708_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000709_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000710_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000711_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000713_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000714_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000715_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000716_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000717_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000718_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000719_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000720_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000721_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000722_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000723_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000724_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000725_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000726_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000727_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000728_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000729_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000730_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000731_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000732_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000734_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000735_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000736_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000737_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000738_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000740_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000741_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000742_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000743_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000744_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000745_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000747_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000748_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000749_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000750_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000751_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000752_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000754_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000755_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000756_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000758_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000759_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000760_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000761_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000762_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000763_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000765_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000766_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000767_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000768_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000769_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000771_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000772_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000773_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000774_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000775_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000777_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000778_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000780_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000783_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000784_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000785_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000786_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000789_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000791_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000794_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000795_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000796_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000797_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000798_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000800_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000801_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000802_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000803_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000804_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000805_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000806_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000808_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000809_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000810_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000811_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000812_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000813_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000815_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000816_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000817_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000818_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000819_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000822_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000823_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000824_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000825_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000826_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000827_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000830_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000832_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000833_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000834_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000835_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000836_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000837_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000839_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000840_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000842_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000843_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000845_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000847_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000849_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000850_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000851_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000852_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000853_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000854_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000855_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000857_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000858_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000859_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000860_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000861_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000862_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000863_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000864_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000866_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000867_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000868_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000869_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000871_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000872_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000873_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000874_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000875_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000876_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000877_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000879_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000880_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000881_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000882_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000883_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000884_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000885_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000886_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000887_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000888_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000889_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000891_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000893_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000894_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000895_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000898_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000899_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000900_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000901_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000904_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000905_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000906_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000907_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000908_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000909_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000910_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000911_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000913_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000914_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000915_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000916_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000917_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000918_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000919_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000920_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000921_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000922_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000923_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000924_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000925_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000926_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000927_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000928_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000929_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000930_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000931_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000933_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000934_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000935_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000937_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000938_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000939_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000940_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000941_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000942_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000943_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000944_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000945_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000946_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000948_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000950_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000951_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000952_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000953_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000955_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000956_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000958_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000959_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000960_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000961_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000962_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000963_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000964_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000965_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000966_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000967_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000968_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000970_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000971_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000973_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000975_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000976_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000977_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000978_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000979_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000981_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000982_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000983_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000984_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000985_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000987_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000988_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000991_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000992_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000993_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000994_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000996_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000997_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000998_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00000999_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001001_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001002_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001004_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001005_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001006_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001007_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001008_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001009_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001010_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001011_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001012_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001013_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001014_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001015_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001016_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001017_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001018_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001019_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001020_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001021_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001023_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001024_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001025_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001026_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001027_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001029_5.dst',
        'LFN:/lhcb/production/DC06/phys-lumi2/00001586/DST/0000/00001586_00001030_5.dst'    ]
    start = (chunkindex * chunksize) % len(lfns)
    chunk = lfns[start:start + chunksize]
    return LHCbDataset(files = [LHCbDataFile(name = lfn) for lfn in chunk])
