"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("./token");
/**
 * Represents the amount of digital storage.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 */
class Size {
    constructor(amount, unit) {
        if (!token_1.Token.isUnresolved(amount) && amount < 0) {
            throw new Error(`Storage amounts cannot be negative. Received: ${amount}`);
        }
        this.amount = amount;
        this.unit = unit;
    }
    /**
     * Create a Storage representing an amount kibibytes.
     * 1 KiB = 1024 bytes
     */
    static kibibytes(amount) {
        return new Size(amount, StorageUnit.Kibibytes);
    }
    /**
     * Create a Storage representing an amount mebibytes.
     * 1 MiB = 1024 KiB
     */
    static mebibytes(amount) {
        return new Size(amount, StorageUnit.Mebibytes);
    }
    /**
     * Create a Storage representing an amount gibibytes.
     * 1 GiB = 1024 MiB
     */
    static gibibytes(amount) {
        return new Size(amount, StorageUnit.Gibibytes);
    }
    /**
     * Create a Storage representing an amount tebibytes.
     * 1 TiB = 1024 GiB
     */
    static tebibytes(amount) {
        return new Size(amount, StorageUnit.Tebibytes);
    }
    /**
     * Create a Storage representing an amount pebibytes.
     * 1 PiB = 1024 TiB
     */
    static pebibyte(amount) {
        return new Size(amount, StorageUnit.Pebibytes);
    }
    /**
     * Return this storage as a total number of kibibytes.
     */
    toKibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Kibibytes, opts);
    }
    /**
     * Return this storage as a total number of mebibytes.
     */
    toMebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Mebibytes, opts);
    }
    /**
     * Return this storage as a total number of gibibytes.
     */
    toGibibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Gibibytes, opts);
    }
    /**
     * Return this storage as a total number of tebibytes.
     */
    toTebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Tebibytes, opts);
    }
    /**
     * Return this storage as a total number of pebibytes.
     */
    toPebibytes(opts = {}) {
        return convert(this.amount, this.unit, StorageUnit.Pebibytes, opts);
    }
}
exports.Size = Size;
/**
 * Rounding behaviour when converting between units of `Size`.
 */
var SizeRoundingBehavior;
(function (SizeRoundingBehavior) {
    /** Fail the conversion if the result is not an integer. */
    SizeRoundingBehavior[SizeRoundingBehavior["FAIL"] = 0] = "FAIL";
    /** If the result is not an integer, round it to the closest integer less than the result */
    SizeRoundingBehavior[SizeRoundingBehavior["FLOOR"] = 1] = "FLOOR";
    /** Don't round. Return even if the result is a fraction. */
    SizeRoundingBehavior[SizeRoundingBehavior["NONE"] = 2] = "NONE";
})(SizeRoundingBehavior = exports.SizeRoundingBehavior || (exports.SizeRoundingBehavior = {}));
class StorageUnit {
    constructor(label, inKibiBytes) {
        this.label = label;
        this.inKibiBytes = inKibiBytes;
        // MAX_SAFE_INTEGER is 2^53, so by representing storage in kibibytes,
        // the highest storage we can represent is 8 exbibytes.
    }
    toString() {
        return this.label;
    }
}
StorageUnit.Kibibytes = new StorageUnit('kibibytes', 1);
StorageUnit.Mebibytes = new StorageUnit('mebibytes', 1024);
StorageUnit.Gibibytes = new StorageUnit('gibibytes', 1024 * 1024);
StorageUnit.Tebibytes = new StorageUnit('tebibytes', 1024 * 1024 * 1024);
StorageUnit.Pebibytes = new StorageUnit('pebibytes', 1024 * 1024 * 1024 * 1024);
function convert(amount, fromUnit, toUnit, options = {}) {
    var _a;
    const rounding = (_a = options.rounding) !== null && _a !== void 0 ? _a : SizeRoundingBehavior.FAIL;
    if (fromUnit.inKibiBytes === toUnit.inKibiBytes) {
        return amount;
    }
    if (token_1.Token.isUnresolved(amount)) {
        throw new Error(`Unable to perform time unit conversion on un-resolved token ${amount}.`);
    }
    const multiplier = fromUnit.inKibiBytes / toUnit.inKibiBytes;
    const value = amount * multiplier;
    switch (rounding) {
        case SizeRoundingBehavior.NONE:
            return value;
        case SizeRoundingBehavior.FLOOR:
            return Math.floor(value);
        default:
        case SizeRoundingBehavior.FAIL:
            if (!Number.isInteger(value)) {
                throw new Error(`'${amount} ${fromUnit}' cannot be converted into a whole number of ${toUnit}.`);
            }
            return value;
    }
}
//# sourceMappingURL=data:application/json;base64,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