"use strict";
const cdk = require("@aws-cdk/cdk");
const util_1 = require("../lib/util");
module.exports = {
    parseBucketArn: {
        'explicit arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'my:bucket:arn';
            test.deepEqual(util_1.parseBucketArn(stack, { bucketArn }), bucketArn);
            test.done();
        },
        'produce arn from bucket name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'hello';
            test.deepEqual(stack.node.resolve(util_1.parseBucketArn(stack, { bucketName })), { 'Fn::Join': ['',
                    ['arn:',
                        { Ref: 'AWS::Partition' },
                        ':s3:::hello']] });
            test.done();
        },
        'fails if neither arn nor name are provided'(test) {
            const stack = new cdk.Stack();
            test.throws(() => util_1.parseBucketArn(stack, {}), /Cannot determine bucket ARN. At least `bucketArn` or `bucketName` is needed/);
            test.done();
        }
    },
    parseBucketName: {
        'explicit name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'foo';
            test.deepEqual(stack.node.resolve(util_1.parseBucketName(stack, { bucketName })), 'foo');
            test.done();
        },
        'extract bucket name from string arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            test.deepEqual(stack.node.resolve(util_1.parseBucketName(stack, { bucketArn })), 'my-bucket');
            test.done();
        },
        'undefined if cannot extract name from a non-string arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = `arn:aws:s3:::${new cdk.Token({ Ref: 'my-bucket' })}`;
            test.deepEqual(stack.node.resolve(util_1.parseBucketName(stack, { bucketArn })), undefined);
            test.done();
        },
        'fails if arn uses a non "s3" service'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:xx:::my-bucket';
            test.throws(() => util_1.parseBucketName(stack, { bucketArn }), /Invalid ARN/);
            test.done();
        },
        'fails if ARN has invalid format'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'invalid-arn';
            test.throws(() => util_1.parseBucketName(stack, { bucketArn }), /ARNs must have at least 6 components/);
            test.done();
        },
        'fails if ARN has path'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket/path';
            test.throws(() => util_1.parseBucketName(stack, { bucketArn }), /Bucket ARN must not contain a path/);
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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