"""Service loader for registering community-devops-agent with boto3."""

import json
from pathlib import Path
from typing import Dict, Any

from botocore.loaders import Loader


class ServiceLoader:
    """Loads service model files for the community-devops-agent service."""

    SERVICE_NAME = "community-devops-agent"
    API_VERSION = "2025-12-09"

    def __init__(self) -> None:
        """Initialize the service loader."""
        self.data_path = self._get_data_path()
        self._service_data = None

    def _get_data_path(self) -> Path:
        """Get the path to the service model data directory."""
        # Get the directory containing this file
        current_dir = Path(__file__).parent.parent
        data_dir = current_dir / "data" / self.SERVICE_NAME / self.API_VERSION

        if not data_dir.exists():
            raise RuntimeError(
                f"Service model directory not found: {data_dir}. "
                "Ensure the package is properly installed."
            )

        return data_dir

    def load_service_model(
        self, type_name: str = "service-2"
    ) -> Dict[str, Any]:
        """
        Load a service model file.

        Args:
            type_name: The type of model to load (e.g., 'service-2', 'paginators-1')

        Returns:
            The loaded service model as a dictionary
        """
        file_path = self.data_path / f"{type_name}.json"

        if not file_path.exists():
            raise FileNotFoundError(
                f"Service model file not found: {file_path}"
            )

        with open(file_path, "r") as f:
            return json.load(f)

    def get_service_data(self) -> Dict[str, Any]:
        """
        Get the complete service data including all model files.

        Returns:
            Dictionary containing service model, paginators, waiters, etc.
        """
        if self._service_data is None:
            self._service_data = {
                "service": self.load_service_model("service-2"),
                "paginators": self.load_service_model("paginators-1"),
                "waiters": self.load_service_model("waiters-2"),
                "endpoint-rule-set": self.load_service_model(
                    "endpoint-rule-set-1"
                ),
            }

        return self._service_data

    @classmethod
    def register_with_botocore(cls) -> None:
        """
        Register the service with botocore's loader.

        This modifies botocore's loader to include our custom service models.
        """
        loader = ServiceLoader()

        # Get the original methods
        original_load_data = Loader.load_data
        original_list_available_services = Loader.list_available_services
        original_list_api_versions = Loader.list_api_versions

        def custom_load_data(self, data_path):
            """Custom load_data that includes our service."""
            # Check if this is a request for our service
            parts = data_path.split("/")

            if len(parts) >= 2 and parts[0] == cls.SERVICE_NAME:
                # This is a request for our service
                if len(parts) == 2:
                    # Request for API versions list
                    if parts[1] == "api-versions":
                        return [cls.API_VERSION]
                elif len(parts) == 3:
                    # Request for a specific model file
                    api_version = parts[1]
                    model_type = parts[2]

                    if api_version == cls.API_VERSION:
                        try:
                            return loader.load_service_model(model_type)
                        except FileNotFoundError:
                            pass

            # Fall back to original loader
            return original_load_data(self, data_path)

        def custom_list_available_services(self, type_name):
            """Custom list_available_services that includes our service."""
            # Get the original list
            services = original_list_available_services(self, type_name)

            # Add our service if not already present
            if cls.SERVICE_NAME not in services:
                services.append(cls.SERVICE_NAME)

            return services

        def custom_list_api_versions(self, service_name, type_name):
            """Custom list_api_versions that includes our service."""
            if service_name == cls.SERVICE_NAME:
                # Return our API version
                return [cls.API_VERSION]

            # Fall back to original loader
            return original_list_api_versions(self, service_name, type_name)

        # Monkey-patch the Loader class
        Loader.load_data = custom_load_data
        Loader.list_available_services = custom_list_available_services
        Loader.list_api_versions = custom_list_api_versions
