"""
Displays a dependency graph for a given project, using pygraphviz.

Copyright (c) 2007 by Enthought, Inc.
License: BSD Style.

"""

# Standard library imports
import os
import sys
import types

# Project library imports
from enthought.ets.base_subcommand import BaseSubcommand
from pkg_resources import Requirement


class Graph(BaseSubcommand):
    """
    The ets graph command.

    """

    def __init__(self, subparsers):
        """
        Constructor.

        Overloaded to customize our parser configuration.

        """

        # Create our parser.
        desc = ('Uses graphiz to save a dependency graph of the specified'
            'projects. At least one project must be specified.  '
            'The project dependencies are found by consulting a project map.  '
            'That map may be explicitly provided or generated by'
            'inspecting a set of repositories.')
        parser = subparsers.add_parser('graph',
            description = desc,
            help = '. . . %s' % desc,
            )

        # Add arguments
        parser.add_argument('project',
            nargs = "+",
            help = 'Specifications of project(s) to retrieve.  These are of '
                'the same form as setuptools\' version specifications.  For '
                'example "ets==2.7.0" or "ets >=2.7, <3.0a"',
            )

        # Add the options
        parser.add_argument('-e', '--extra',
            dest = 'extra',
            default = False,
            action = 'store_true',
            help = '''Display the project[extra] requirements as separate
            nodes.'''
            )
        self.use_project_map(parser)
        parser.add_argument('-o', '--output',
            dest = 'path',
            help = '''Specify the file in which the graph will be saved.
            Filetype is guessed from extension (defaults to png).
            Possibilities are png, ps, svg, jpg, svgz, tga, plain, or jpeg'''
            )
        self.proxy(parser)

        # Save the entry point for running this command.
        parser.set_defaults(func = self.main)

        return


    def main(self, args, cfg):
        """
        Execute the ets graph command.

        """

        try:
            from pygraphviz import AGraph
        except ImportError:
            print >>sys.stderr, ('Cannot import pygraphviz. '
                'You need pygraphviz for this operation.')
            return

        # Build up the set of projects we're interested in.
        project_set = self.build_project_set(args, cfg)
        project_set.add_dependencies()

        # Only continue if the user specified at least one project.
        if len(args.project) == 0:
            return

        # Build the graph, storing the nodes with extras, to be able to clean
        # up the graph later on.
        graph = AGraph(strict=True, directed=True)
        extra_nodes = set()
        # FIXME: Should not be accessing a protected member of the ProjectSet
        # class!
        for pinfo in project_set._projects.itervalues():
            project_id = "%s %s" % (pinfo['name'], pinfo['version'])

            # Add all the extra nodes and connect them to the core package
            for extra in pinfo['extras_require']:
                graph.add_edge( ('%s[%s]' % (project_id, extra), project_id) )
                extra_nodes.add('%s[%s]' % (project_id, extra))

            # Satisfy the requirements for each extra needed
            for extra in pinfo['extras']:
                extra_requires = pinfo['extras_require'][extra]
                for req in extra_requires:
                    for dependency_id in self.requirement_labels(req,
                        project_set):
                        graph.add_edge(('%s[%s]' % (project_id, extra),
                            dependency_id)
                                                    )
            # Satisfy the install requirements
            for req in pinfo['install_requires']:
                for dependency_id in self.requirement_labels(req, project_set):
                    graph.add_edge((project_id, dependency_id))

        if not args.extra:
            self.remove_nodes(graph, extra_nodes)

        graph.layout(prog='dot')

        # Write to the output file.  If no filename was specified, use the name
        # and version of the root project.
        if args.path is None or args.path == '':
            name, version = project_set.get_root_project_info()
            args.path = '%s_%s' % (name, version)
            name, extension = os.path.splitext(args.path)
            if not extension in ('png', 'ps', 'svg', 'jpg', 'svgz', 'tga',
                'plain', 'jpeg'):
                args.path += '.png'
        graph.draw(args.path)
        print "Dependency graph saved to %s" % args.path

        return


    def remove_nodes(self, graph, nodes):
        """
        Remove nodes from a pygraphviz graph, reconnecting the edges
        without the nodes.

        """

        for node in nodes:
            in_neighbors = graph.in_neighbors(node)
            out_neighbors = graph.out_neighbors(node)
            graph.delete_node(node)
            for in_neighbor in in_neighbors:
                for out_neighbor in out_neighbors:
                    graph.add_edge(in_neighbor, out_neighbor)

        return


    def requirement_labels(self, requirement, project_set):
        """
        Return the list of labels of the bubles for a given requirement.

        """

        requirement_info = Requirement.parse(requirement)
        requirement_name = requirement_info.project_name
        if requirement_name in project_set._projects:
            dependency_info  = project_set._projects[requirement_name]
            dependency_id    =  "%s %s" % (
                                        dependency_info['name'],
                                        dependency_info['version'],
                                            )
            if len(requirement_info.extras)>0:
                dependency_id = ["%s[%s]" % (dependency_id, extra)
                            for extra in requirement_info.extras]
                return dependency_id
        else:
            # FIXME: Here we should mark this node as being
            # different: the project is not hosted by us.
            # but we need a recent version of pygraphviz.
            #dependency_id = requirement_name
            #graph.add_node(dependency_id, color='blue')
            dependency_id = '%s -- External' % requirement_name

        return (dependency_id, )

