import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as lambda from 'aws-cdk-lib/aws-lambda';
/**
 * Represents an AppSpec to be used for ECS services.
 *
 * see: https://docs.aws.amazon.com/codedeploy/latest/userguide/reference-appspec-file-structure-resources.html#reference-appspec-file-structure-resources-ecs
 */
export declare class EcsAppSpec {
    /**
     * Service to target for the deployment
     */
    private readonly targetService;
    /**
     * Optional lifecycle hooks
     */
    private readonly hooks?;
    constructor(targetService: TargetService, hooks?: AppSpecHooks);
    /**
     * Render JSON string for this AppSpec to be used
     *
     * @returns string representation of this AppSpec
     */
    toString(): string;
    private hooksSection;
    private configureAwsVpcNetworkingWithSecurityGroups;
}
/**
 * Describe the target for CodeDeploy to use when creating a deployment for an ecs.EcsDeploymentGroup.
 */
export interface TargetService {
    /**
     * The TaskDefintion to deploy to the target services.
     */
    readonly taskDefinition: ecs.ITaskDefinition;
    /**
     * The name of the Amazon ECS container that contains your Amazon ECS application. It must be a container specified in your Amazon ECS task definition.
     */
    readonly containerName: string;
    /**
     * The port on the container where traffic will be routed to.
     */
    readonly containerPort: number;
    /**
     * The platform version of the Fargate tasks in the deployed Amazon ECS service.
     * see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
     *
     * @default LATEST
     */
    readonly platformVersion?: ecs.FargatePlatformVersion;
    /**
     * Network configuration for ECS services that have a network type of `awsvpc`.
     *
     * @default reuse current network settings for ECS service.
     */
    readonly awsvpcConfiguration?: AwsvpcConfiguration;
    /**
     * A list of Amazon ECS capacity providers to use for the deployment.
     *
     * @default reuse current capcity provider strategy for ECS service.
     */
    readonly capacityProviderStrategy?: ecs.CapacityProviderStrategy[];
}
/**
 * Network configuration for ECS services that have a network type of `awsvpc`.
 */
export interface AwsvpcConfiguration {
    /**
     * The VPC to use for the task.
     */
    readonly vpc: ec2.IVpc;
    /**
     * The Subnets to use for the task.
     */
    readonly vpcSubnets: ec2.SubnetSelection;
    /**
     * The Security Groups to use for the task.
     */
    readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * Assign a public IP address to the task.
     */
    readonly assignPublicIp: boolean;
}
/**
 * Lifecycle hooks configuration
 */
export interface AppSpecHooks {
    /**
     * Lambda or ARN of a lambda to run tasks before the replacement task set is created.
     */
    readonly beforeInstall?: string | lambda.IFunction;
    /**
     * Lambda or ARN of a lambda to run tasks after the replacement task set is created and one of the target groups is associated with it.
     */
    readonly afterInstall?: string | lambda.IFunction;
    /**
     * Lambda or ARN of a lambda to run tasks after the test listener serves traffic to the replacement task set.
     */
    readonly afterAllowTestTraffic?: string | lambda.IFunction;
    /**
     * Lambda or ARN of a lambda to run tasks after the second target group is associated with the replacement task set, but before traffic is shifted to the replacement task set.
     */
    readonly beforeAllowTraffic?: string | lambda.IFunction;
    /**
     * Lambda or ARN of a lambda to run tasks after the second target group serves traffic to the replacement task set.
     */
    readonly afterAllowTraffic?: string | lambda.IFunction;
}
