"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const resources = require("@aws-solutions-constructs/resources");
class OpenApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.apiDefinitionAsset && (props.apiDefinitionBucket || props.apiDefinitionKey)) {
            throw new Error('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both');
        }
        const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
        const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
        if (apiDefinitionBucket === undefined || apiDefinitionKey === undefined) {
            throw new Error('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
        }
        if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
            throw new Error('At least one ApiIntegration must be specified in the apiIntegrations property');
        }
        // store a counter to be able to uniquely name lambda functions to avoid naming collisions
        let lambdaCounter = 0;
        this.apiLambdaFunctions = props.apiIntegrations.map(apiIntegration => {
            if (apiIntegration.existingLambdaObj || apiIntegration.lambdaFunctionProps) {
                return {
                    id: apiIntegration.id,
                    lambdaFunction: defaults.buildLambdaFunction(this, {
                        existingLambdaObj: apiIntegration.existingLambdaObj,
                        lambdaFunctionProps: apiIntegration.lambdaFunctionProps
                    }, `ApiFunction${lambdaCounter++}`)
                };
            }
            else {
                throw new Error(`One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: ${apiIntegration.id}`);
            }
        });
        // Map each id and lambda function pair to the required format for the template writer custom resource
        const apiIntegrationUris = this.apiLambdaFunctions.map(apiLambdaFunction => {
            // the placeholder string that will be replaced in the OpenAPI Definition
            const uriPlaceholderString = apiLambdaFunction.id;
            // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
            const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
            return {
                id: uriPlaceholderString,
                value: uriResolvedValue
            };
        });
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        const apiDefinitionWriter = resources.createTemplateWriterCustomResource(this, 'Api', {
            templateBucket: apiDefinitionBucket,
            templateKey: apiDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? cdk.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        const specRestApiResponse = defaults.CreateSpecRestApi(this, {
            ...props.apiGatewayProps,
            apiDefinition: apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key)
        }, props.logGroupProps);
        this.apiGateway = specRestApiResponse.api;
        this.apiGatewayCloudWatchRole = specRestApiResponse.role;
        this.apiGatewayLogGroup = specRestApiResponse.logGroup;
        // Redeploy the API any time the incoming API definition changes (from asset or s3 object)
        this.apiGateway.latestDeployment?.addToLogicalId(apiDefinitionKey);
        this.apiLambdaFunctions.forEach(apiLambdaFunction => {
            // Redeploy the API any time one of the lambda functions changes
            this.apiGateway.latestDeployment?.addToLogicalId(apiLambdaFunction.lambdaFunction.functionArn);
            // Grant APIGW invocation rights for each lambda function
            apiLambdaFunction.lambdaFunction.addPermission('PermitAPIGInvocation', {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.apiGateway.arnForExecuteApi('*')
            });
        });
    }
}
exports.OpenApiGatewayToLambda = OpenApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-openapigateway-lambda.OpenApiGatewayToLambda", version: "2.42.0" };
//# sourceMappingURL=data:application/json;base64,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