"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const assertions_1 = require("aws-cdk-lib/assertions");
const core_1 = require("@aws-solutions-constructs/core");
test('Simple deployment works', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiGateway).toBeDefined();
    expect(construct.apiGatewayCloudWatchRole).toBeDefined();
    expect(construct.apiGatewayLogGroup).toBeDefined();
    expect(construct.apiLambdaFunctions.length).toEqual(1);
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from Asset', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('API Definition can be specified from S3 Bucket and Key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const apiDefinitionKey = 'api.yaml';
    const construct = new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionBucket,
        apiDefinitionKey,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }
        ]
    });
    expect(construct.apiLambdaFunctions[0].id).toEqual('MessagesHandler');
    expect(construct.apiLambdaFunctions[0].lambdaFunction).toBeDefined();
});
test('Throws error when both api definition asset and s3 object are specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket: new s3.Bucket(stack, 'ApiDefinitionBucket'),
            apiDefinitionKey: 'key',
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: lambda.Runtime.NODEJS_18_X,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both');
});
test('Multiple Lambda Functions can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                }
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
});
test('Existing lambda function can be specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const existingLambdaObj = new lambda.Function(stack, 'ExistingLambda', {
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        functionName: 'ExistingLambdaFunction',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    });
    new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
        apiDefinitionAsset,
        apiIntegrations: [
            {
                id: 'MessagesHandler',
                existingLambdaObj
            }, {
                id: 'PhotosHandler',
                lambdaFunctionProps: {
                    runtime: lambda.Runtime.NODEJS_18_X,
                    handler: 'index.handler',
                    functionName: 'NewLambdaFunction',
                    code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Lambda::Function', 4);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'ExistingLambdaFunction'
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: 'NewLambdaFunction'
    });
});
test('Throws error when neither existingLambdaObj or lambdaFunctionProps is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: [
                {
                    id: 'MessagesHandler'
                }
            ]
        });
    };
    expect(app).toThrowError('One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: MessagesHandler');
});
test('Throws error when no api definition is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiIntegrations: [
                {
                    id: 'MessagesHandler',
                    lambdaFunctionProps: {
                        runtime: lambda.Runtime.NODEJS_18_X,
                        handler: 'index.handler',
                        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
                    }
                }
            ]
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when no api integration is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('At least one ApiIntegration must be specified in the apiIntegrations property');
});
test('Throws error when api definition s3 bucket is specified but s3 object key is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api definition s3 object key is specified but s3 bucket is missing', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 bucket is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const apiDefinitionBucket = core_1.CreateScrapBucket(stack, {});
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionBucket,
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
test('Throws error when api is defined as asset and s3 key is specified', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'OpenApiAsset', {
        path: path.join(__dirname, 'openapi/apiDefinition.yaml')
    });
    const app = () => {
        new lib_1.OpenApiGatewayToLambda(stack, 'test-apigateway-lambda', {
            apiDefinitionAsset,
            apiDefinitionKey: 'prefix/api-definition.yml',
            apiIntegrations: []
        });
    };
    expect(app).toThrowError('Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified');
});
//# sourceMappingURL=data:application/json;base64,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