"use strict";
// Copyright (c) TileDB
// Distributed under the terms of the MIT License.
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DagVisualizeView = exports.DagVisualizeModel = void 0;
const base_1 = require("@jupyter-widgets/base");
const d3 = __importStar(require("d3"));
const debounce_1 = __importDefault(require("./debounce"));
const version_1 = require("./version");
require("../css/widget.css");
const worker_js_1 = __importDefault(require("file-loader!../lib/worker.js"));
const poll_1 = require("./poll");
// const BOX_ASPECT_RATIO = 0.36;
class DagVisualizeModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: DagVisualizeModel.model_name, _model_module: DagVisualizeModel.model_module, _model_module_version: DagVisualizeModel.model_module_version, _view_name: DagVisualizeModel.view_name, _view_module: DagVisualizeModel.view_module, _view_module_version: DagVisualizeModel.view_module_version, value: '' });
    }
}
exports.DagVisualizeModel = DagVisualizeModel;
DagVisualizeModel.serializers = Object.assign({}, base_1.DOMWidgetModel.serializers);
DagVisualizeModel.model_name = 'DagVisualizeModel';
DagVisualizeModel.model_module = version_1.MODULE_NAME;
DagVisualizeModel.model_module_version = version_1.MODULE_VERSION;
DagVisualizeModel.view_name = 'DagVisualizeView'; // Set to null if no view
DagVisualizeModel.view_module = version_1.MODULE_NAME; // Set to null if no view
DagVisualizeModel.view_module_version = version_1.MODULE_VERSION;
const PADDING = 40;
class DagVisualizeView extends base_1.DOMWidgetView {
    render() {
        this.el.classList.add('tiledb-widget');
        this.createSVG();
        this.value_changed();
        /**
         * Debounce rendering function so it won't rerender too fast
         */
        const debouncedOnChange = debounce_1.default(this.value_changed.bind(this), 1500);
        this.model.on('change:value', debouncedOnChange, this);
    }
    value_changed() {
        this.data = JSON.parse(this.model.get('value'));
        /**
         * Reset html and build new graph
         */
        this.createDag();
    }
    calculateBounds(positions) {
        if (typeof this.bounds === 'undefined') {
            const xNums = Object.keys(positions).map((pos) => positions[pos][0]);
            const yNums = Object.keys(positions).map((pos) => positions[pos][1]);
            const padding = 30;
            const verticalPadding = 60;
            const maxHorizontalCoordinate = Math.max(...xNums);
            let maxVerticalCoordinate = Math.max(...yNums);
            this.bounds = [maxHorizontalCoordinate + padding, maxVerticalCoordinate + verticalPadding];
        }
        return this.bounds;
    }
    createSVG() {
        this.wrapper = d3.select(this.el).append('svg').append('g');
        this.svg = d3.select(this.el).select('svg');
        this.createControls();
        this.createTooltip();
    }
    getScale() {
        const [maxWidth, height] = this.bounds;
        const scaleX = this.el.offsetWidth / (maxWidth + PADDING);
        const scaleY = 400 / (height + PADDING);
        return [scaleX, scaleY];
    }
    zoom() {
        const [width, height] = this.bounds;
        const [scaleX, scaleY] = this.getScale();
        const svg = this.svg;
        const zoom = d3.zoom().translateExtent([[0, 0], [width * scaleX + PADDING, height * scaleY + PADDING]]).on('zoom', () => {
            this.wrapper.attr('transform', d3.event.transform);
        });
        svg.call(zoom).on('wheel.zoom', null);
        function zoomHandler() {
            d3.event.preventDefault();
            const direction = (this.id === 'zoom_in') ? .2 : -.2;
            /**
             * In SVG 1.1 <svg> elements did not support transform attributes. In SVG 2 it is proposed that they should.
             * Chrome and Firefox implement this part of the SVG 2 specification, Safari does not yet do so and IE11 never will.
             * That's why we apply transform to the "g" element instead of the "svg"
             */
            svg.transition().duration(300).call(zoom.scaleBy, 1 + direction);
        }
        function resetHandler() {
            svg.transition().duration(300).call(zoom.transform, d3.zoomIdentity);
        }
        setTimeout(() => {
            d3.select(this.el).selectAll('.zoomControl').on('click', zoomHandler);
            d3.select(this.el).selectAll('.resetControl').on('click', resetHandler);
        }, 0);
        this.initialized = true;
    }
    createTooltip() {
        this.tooltip = d3.select(this.el).append('div')
            .attr('class', 'tooltip')
            .style("opacity", 0);
    }
    /**
     * Method to calculate vertical offset from the hidden "fake" root node
     * @param descendants The hierarchy points
     * @param circleSize The size of the circle
     * @param fauxRootNode The name of the fake root node
     */
    calculateYOffset(descendants, circleSize, fauxRootNode) {
        /**
         * If we have already calculated the offset just return it.
         * If not calculate the offset
         */
        if (typeof this.verticalOffset === 'undefined') {
            const originalRoot = descendants.find((node) => Boolean(node.parent && node.parent.id === fauxRootNode));
            /**
             * Calculate the offset of the original root node,
             * since the first faux root node that we added,
             * we hide it and create an empty space to the top.
             */
            const yOffset = originalRoot ? originalRoot.y : 0;
            this.verticalOffset = circleSize - yOffset / 2;
        }
        return this.verticalOffset;
    }
    getRootNodes(nodes, edges) {
        const hasNoParent = (node) => edges.every(([, parent]) => node !== parent);
        return nodes.filter(hasNoParent);
    }
    /**
     * Calculate the size of the nodes in the graph depending on the number of nodes
     * As the number of nodes in the tree gets bigger and bigger, the tree is becoming
     * squeezed and the size of the nodes have to get smaller in order not to overlap
     * witch each other.
     * @param numberOfNodes Number of the nodes in the tree
     */
    getNodeSize(numberOfNodes) {
        const howManyTens = Math.floor(numberOfNodes / 10);
        /**
         * Don't let node size go bellow 5
         */
        return Math.max(320 / (20 + howManyTens), 5);
    }
    createDag() {
        return __awaiter(this, void 0, void 0, function* () {
            const { nodes, edges, node_details, positions } = this.data;
            const bounds = this.calculateBounds(positions);
            const maxHeight = bounds[1];
            /**
             * During initialization the wrapper elemete (this.el) has no width,
             * we wait for that before we do any DOM calculations.
             */
            if (!this.initialized) {
                yield poll_1.poll(() => this.el.offsetWidth > 0, 300);
            }
            const numberOfNodes = nodes.length;
            const lessThanThirtyNodes = numberOfNodes < 30;
            const [scaleX, scaleY] = this.getScale();
            /**
             * Sometimes during updates we are getting different/weird positions object
             * So we save and re-use the first positions object we are getting
             */
            this.positions = this.positions || positions;
            if (!this.initialized) {
                this.zoom();
            }
            const circleSize = this.getNodeSize(numberOfNodes);
            const links = edges.map(([parent, child]) => ({
                source: parent,
                target: child,
            }));
            const nodeDetails = Object.keys(node_details).map((node, i) => ({
                index: i,
                status: node_details[node].status,
                id: node,
                fx: this.positions[node][0] * scaleX,
                /** For Y position we flip tree upside down (that's why: maxHeight - node's Y position) */
                fy: (maxHeight - this.positions[node][1]) * scaleY,
            }));
            const worker = new Worker(worker_js_1.default);
            worker.postMessage({
                nodes: nodeDetails,
                links
            });
            worker.onmessage = (event) => {
                if (event.data.type !== 'end') {
                    return;
                }
                /**
                 * Remove previous contents
                 */
                this.wrapper.selectAll("*").remove();
                const { nodes, links } = event.data;
                this.wrapper.append("g")
                    .selectAll("path")
                    .data(links)
                    .enter().append("path")
                    .attr('d', (d) => {
                    return `M${d.source.x},${d.source.y} C ${d.source.x},${(d.source.y + d.target.y) / 2} ${d.target.x},${(d.source.y + d.target.y) / 2} ${d.target.x},${d.target.y}`;
                })
                    .attr('class', (d) => `path-${d.target.status}`);
                this.wrapper.append("g")
                    .selectAll("circle")
                    .data(nodes)
                    .enter().append("circle")
                    .attr("cx", (d) => d.x)
                    .attr("cy", (d) => d.y)
                    .attr("r", circleSize)
                    .attr('class', (d) => `${d.status} ${lessThanThirtyNodes ? 'node--small' : ''}`)
                    .on('mouseover', (d) => {
                    this.tooltip.transition()
                        .duration(200)
                        .style('opacity', .9);
                    this.tooltip.html(`<p>${d.id}: ${d.status}</p>`)
                        .style('left', `${d3.event.clientX + 10}px`)
                        .style('top', `${d3.event.clientY + 10}px`);
                }).on('mouseout', () => {
                    this.tooltip.transition()
                        .duration(500)
                        .style('opacity', 0);
                });
            };
        });
    }
    createControls() {
        const zoomInButton = document.createElement('button');
        const zoomOutButton = document.createElement('button');
        const resetButton = document.createElement('button');
        const className = 'zoomControl';
        zoomInButton.id = 'zoom_in';
        zoomOutButton.id = 'zoom_out';
        resetButton.className = 'resetControl';
        zoomInButton.className = className;
        zoomOutButton.className = className;
        this.el.append(zoomInButton);
        this.el.append(zoomOutButton);
        this.el.append(resetButton);
    }
}
exports.DagVisualizeView = DagVisualizeView;
//# sourceMappingURL=widget.js.map