# import package/modules
from typing import Dict, List, Union, Literal, Optional, Tuple, Any
import yaml
import pycuc
# internals
from .fugacitycore import FugacityCore
from .thermolinkdb import ThermoLinkDB
from ..plugin import ReferenceManager, EQUATION_OF_STATE_MODELS
from .eosutils import EOSUtils
from ..utils import eos_model_name, add_attributes


class eosCore(ThermoLinkDB, ReferenceManager):
    '''
    eosCore class for calculating properties of fluids using different equations of state (EOS).
    '''

    # NOTE: class variables
    _references = {}
    # results
    _results = {}

    def __init__(self, **kwargs):
        # init
        ThermoLinkDB.__init__(self)
        ReferenceManager.__init__(self)

        # NOTE: load reference
        # reference plugin
        self._references = self.load_reference()

    def __str__(self):
        model_ = "eosCore class for calculating properties of fluids using different equations of state (EOS)."
        return model_

    def parse_model_inputs(self, model_inputs: str) -> Dict[str, Any]:
        '''
        Convert model inputs from string to dictionary format.

        Parameters
        ----------
        model_inputs: str
            Model inputs in string format, such as:
            - `{"component": "CO2", "phase": "VAPOR", "pressure": [1.2*1e5, 'Pa'], "temperature": [300, 'K']}`
            - `{"feed-specification": {"CO2": 0.5, "N2": 0.5}, "pressure": [1.2*1e5, 'Pa'], "temperature": [300, 'K']}`

        Returns
        -------
        model_input_parsed: dict
            Parsed model inputs in dictionary format.
        '''
        try:
            # check if model_inputs is None or model_inputs == 'None'
            if model_inputs is None or model_inputs == 'None':
                raise Exception('Model inputs are not provided!')

            # strip
            model_inputs = model_inputs.strip()
            model_input_parsed = yaml.safe_load(model_inputs)

            return model_input_parsed
        except Exception as e:
            raise Exception("Parsing model inputs failed!, ", e)

    @add_attributes(metadata=EQUATION_OF_STATE_MODELS)
    def cal_fugacity(
        self,
        model_name: Literal[
            'SRK', 'PR', 'RK', 'vdW'
        ],
        model_input: Dict,
        model_source: Dict,
        solver_method: Literal[
            'ls', 'newton', 'fsolve', 'root'
        ] = 'ls',
        liquid_fugacity_mode: Literal[
            'EOS', 'Poynting'
        ] = 'EOS',
        **kwargs
    ) -> Dict[str, Any]:
        '''
        Starts calculating fugacity for the single component

        Parameters
        ----------
        model_name: str
            eos model name,
                - `SRK`: Soave-Redlich-Kwong
                - `PR`: Peng-Robinson
                - `vdW`: Van der Waals
                - `RK`: Redlich-Kwong
        model_input: dict
            model input
                - `phase` (Optional): str, `VAPOR`: vapor phase, `LIQUID`: liquid phase, `VAPOR-LIQUID`: vapor-liquid phase, `SUPERCRITICAL`: supercritical phase
                - `pressure`: list, pressure and a unit, such as `[1.2*1e5, 'Pa']`
                - `temperature`: list, temperature and a unit, such as `[300, 'K']`
        solver_method: str
            solver method as:
                - `ls`: least square method (default)
                - `root`: root method
                - `newton`: newton method
                - `fsolve`: fsolve method
        model_source: dict
            datasource and equationsource needed for fugacity calculation
                - datasource: dict, datasource for the component (`generated by PyThermoDB`)
                - equationsource: dict, equationsource for the component (`generated by PyThermoDB`)
        liquid_fugacity_mode: str
            liquid fugacity method,
                - `Poynting`: Poynting method (soon)
                - `EOS`: Equation of state (lowest Z)
        **kwargs: Optional[Dict]
            additional arguments
                - tolerance: float, tolerance for the calculation (default: 1e-1)

        Returns
        -------
        res: Dict
            results of fugacity calculation

        Notes
        -----
        ### solver_method:
            - `ls`: least square method
            - `newton`: newton method
            - `fsolve`: fsolve method
            - `root`: root method (default)

        ### root_analysis_set:

        root analysis set, `None`: default (calculation performed according to phase provided), `1`: 3 roots (VAPOR-LIQUID), `2`: 1 root (LIQUID), `3`: 1 root (VAPOR), `4`: 1 root (SUPERCRITICAL). Otherwise, eos root analysis is performed to determine the phase of the component at the provided operating conditions including temperature and pressure.

            - set [1]: 3 roots (VAPOR-LIQUID)
                - At `T < Tc` and `P = Psat`, 3 real roots → smallest is liquid, largest is vapor.
            - set [2]: 1 root (LIQUID)
                - At `T < Tc` and `P > Psat`, EOS may give 1 or 3 roots → use smallest (liquid).
            - set [3]: 1 root (VAPOR)
                - At `T < Tc` and `P < Psat`, EOS may give 1 or 3 roots → use largest (vapor).
            - set [4]: 1 root (SUPERCRITICAL)
                - At `T > Tc`, only 1 real root → fluid is supercritical (vapor-like or liquid-like).

        Examples
        --------
        `model_input` defines as:

        ```python
        # single component
        component = 'CO2

        # model input
        # eos model
        eos_model = 'SRK'

        # component phase (Optional)
        phase = "VAPOR"

        # temperature [K] (other units are automatically converted to K)
        T = 300

        # pressure [Pa] (other units are automatically converted to Pa)
        P = 1.2*1e5

        # model input
        model_input = {
            "component": component,
            "phase": phase, # optional
            "pressure": [P, 'Pa'],
            "temperature": [T, 'K'],
        }

        # model source
        model_source = {
            "datasource": datasource,
            "equationsource": equationsource
        }

        # fugacity calculation
        res = tm.cal_fugacity(
            model_name=eos_model,
            model_input=model_input,
            model_source=model_source)
        ```
        '''
        try:
            # SECTION: keywords
            # tolerance
            tolerance = kwargs.get('tolerance', 1e-1)

            # SECTION: set input parameters
            # eos
            eos_model = model_name.upper()
            eos_model = eos_model_name(eos_model)

            # SECTION: phase
            phase = model_input.get('phase', None)

            # check
            if phase is not None:
                phase = phase.upper()
                # check phase
                if phase not in [
                    'VAPOR', 'LIQUID', 'VAPOR-LIQUID', 'SUPERCRITICAL'
                ]:
                    raise Exception('Invalid phase provided!')

            # SECTION: calculation mode
            calculation_mode = 'single'

            # SECTION: component list
            component = model_input.get('component', None)
            # check
            if component is None or component == 'None':
                raise Exception('Component name is not provided!')

            # set
            components = [component.strip()]

            # SECTION: mole fraction
            mole_fraction = [1]

            # SECTION: operating conditions
            # NOTE: check temperature and pressure
            if 'pressure' not in model_input.keys():
                raise Exception('No pressure in operating conditions!')

            if 'temperature' not in model_input.keys():
                raise Exception('No temperature in operating conditions!')

            # operating conditions
            operating_conditions = {
                "pressure": model_input["pressure"],
                "temperature": model_input["temperature"]
            }

            # SECTION: check eos roots
            if phase is None:
                # check eos roots for single component
                eos_roots_analysis = self.check_eos_roots_single_component(
                    model_name=eos_model,
                    model_input=model_input,
                    model_source=model_source,
                    tolerance=tolerance
                )

                # NOTE: eos parms
                # set phase
                phase = str(eos_roots_analysis['phase'])

            # check phase
            if phase is None:
                raise Exception('Phase is not provided!')

            # user set phase for the calculation
            eos_parms = {
                'phase': phase,
                'eos-model': eos_model,
                'mode': calculation_mode,
                'liquid-fugacity-mode': liquid_fugacity_mode
            }

            # SECTION: set datasource and equationsource
            # NOTE: check if datasource and equationsource are provided in model_input
            # datasource
            datasource = model_source.get('datasource', {})
            # equationsource
            equationsource = model_source.get('equationsource', {})
            # set thermodb link
            link_status = self.set_thermodb_link(datasource, equationsource)
            # check
            if not link_status:
                raise Exception('Thermodb link failed!')

            # SECTION: reference for eos
            reference = self._references.get(eos_model, None)

            # build datasource
            component_datasource = self.set_datasource(components, reference)
            # build equation source
            equation_equationsource = self.set_equationsource(
                components, reference)

            # SECTION: init fugacity core
            FugacityCoreC = FugacityCore(
                component_datasource,
                equation_equationsource,
                components,
                operating_conditions,
                eos_parms
            )

            # SECTION: calculation mode
            res = FugacityCoreC.fugacity_cal(
                mole_fraction,
                solver_method=solver_method
            )

            return res
        except Exception as e:
            raise Exception("Fugacity calculation failed!, ", e)

    def check_eos_roots_single_component(
        self,
        model_name: Literal[
            'SRK', 'PR', 'vdW', 'RK'
        ],
        model_input: Dict,
        model_source: Dict,
        **kwargs
    ) -> Dict[str, Any]:
        '''
        Check eos roots for the single component at different temperature and pressure.

        Parameters
        ----------
        model_name: str
            eos model name,
                - `SRK`: Soave-Redlich-Kwong
                - `PR`: Peng-Robinson
                - `RK`: Redlich-Kwong
                - `vdW`: Van der Waals
        model_input: Dict
            model input as:
                - component: str, component name
                - phase: Optional[str] as:
                    1. `VAPOR`: vapor phase,
                    2. `LIQUID`: liquid phase,
                    3. `VAPOR-LIQUID`: vapor-liquid phase,
                    4. `SUPERCRITICAL`: supercritical phase
                - pressure: list, pressure and unit, such as `[1.2, 'Pa']`
                - temperature: list, temperature and unit, such as `[300, 'K']`
        model_source: Dict
            datasource and equationsource needed for fugacity calculation as:
                - datasource: dict, datasource for the component (`generated by PyThermoDB`)
                - equationsource: dict, equationsource for the component (`generated by PyThermoDB`)
        **kwargs: Optional[Dict]
            additional arguments
                - tolerance: float, tolerance for the calculation (default: 1e-1)

        Returns
        -------
        res: dict
            eos root analysis

        Notes
        -----
        The phase of the component is determined based on the temperature and pressure provided in the model_input.

        1. At T < Tc and P = Psat, 3 real roots → smallest is liquid, largest is vapor (vapor-liquid).
        2. At T < Tc and P > Psat, EOS may give 1 or 3 roots → use smallest (liquid).
        3. At T < Tc and P < Psat, EOS may give 1 or 3 roots → use largest (vapor).
        4. At T = Tc, one real root (critical point) → fluid is at critical state.
        5. At T > Tc, only 1 real root → fluid is supercritical (vapor-like or liquid-like).

        Pressure and temperature are in SI units (Pa, K), if not provided in the model_input, they are automatically converted.

        References
        ----------
        1. Introductory Chemical Engineering Thermodynamics
        '''
        try:
            # SECTION: keywords
            # tolerance
            tolerance = kwargs.get('tolerance', 1e-1)

            # SECTION: set input parameters
            # NOTE: eos model
            eos_model = model_name.upper()
            eos_model = eos_model_name(eos_model)

            # NOTE: component list
            component = model_input.get('component', None)
            if component is None or component == 'None':
                raise Exception('Component name is not provided!')

            # set
            components = [component]

            # NOTE: check temperature and pressure
            if 'pressure' not in model_input.keys():
                raise Exception('No pressure in operating conditions!')

            if 'temperature' not in model_input.keys():
                raise Exception('No temperature in operating conditions!')

            # NOTE: operating conditions
            operating_conditions = {
                "pressure": model_input["pressure"],
                "temperature": model_input["temperature"]
            }

            # SECTION: set datasource and equationsource
            # NOTE: check if datasource and equationsource are provided in model_input
            # datasource
            datasource = model_source.get('datasource', {})
            # equationsource
            equationsource = model_source.get('equationsource', {})
            # set thermodb link
            link_status = self.set_thermodb_link(
                datasource,
                equationsource
            )
            # check
            if not link_status:
                raise Exception('Thermodb link failed!')

            # SECTION: reference for eos
            reference = self._references.get(eos_model, None)

            # build datasource
            component_datasource = self.set_datasource(
                components,
                reference
            )
            # build equation source
            equation_equationsource = self.set_equationsource(
                components,
                reference
            )

            # SECTION: operating conditions
            # pressure [Pa]
            P = pycuc.to(
                operating_conditions["pressure"][0],
                f"{operating_conditions['pressure'][1]} => Pa")
            # temperature [K]
            T = pycuc.to(
                operating_conditions["temperature"][0],
                f"{operating_conditions['temperature'][1]} => K")

            # SECTION: init
            EOSUtilsC = EOSUtils(
                component_datasource,
                equation_equationsource
            )

            # SECTION: eos root analysis
            res = EOSUtilsC.eos_root_analysis(
                P,
                T,
                components,
                tolerance=tolerance
            )

            return res[component]
        except Exception as e:
            raise Exception("Fugacity calculation failed!, ", e)

    def cal_fugacity_mixture(
        self,
        model_name: Literal[
            'SRK', 'PR', 'RK', 'vdW'
        ],
        model_input: Dict,
        model_source: Dict,
        solver_method: Literal[
            'ls', 'newton', 'fsolve', 'root'
        ] = 'ls',
        liquid_fugacity_mode: Literal[
            'EOS', 'Poynting'
        ] = 'EOS',
        **kwargs
    ) -> Dict[str, Any]:
        '''
        Starts calculating fugacity for the single and multi-component systems

        Parameters
        ----------
        model_name: str
            eos model name,
                - `SRK`: Soave-Redlich-Kwong
                - `PR`: Peng-Robinson
                - `RK`: Redlich-Kwong
                - `vdW`: Van der Waals
        model_input: dict
            model input
                - `phase`: str | None, `VAPOR`: vapor phase, `LIQUID`: liquid phase, `VAPOR-LIQUID`: vapor-liquid phase, `SUPERCRITICAL`: supercritical phase
                - `feed-specification`: dict, such as `{'CO2': 1.0}` or `{'CO2': 0.5, 'N2': 0.5}`
                - `pressure`: list, pressure in SI unit, such as `[1.2*1e5, 'Pa']`
                - `temperature`: list, temperature in SI unit, such as `[300, 'K']`
        solver_method: str
            solver method as:
                - `ls`: least square method (default)
                - `root`: root method
                - `newton`: newton method
                - `fsolve`: fsolve method
        model_source: dict
            datasource and equationsource needed for fugacity calculation
                - datasource: dict, datasource for the component (`generated by PyThermoDB`)
                - equationsource: dict, equationsource for the component (`generated by PyThermoDB`)
        liquid_fugacity_mode: str
            liquid fugacity method as:
                - `Poynting`: Poynting method (soon),
                - `EOS`: Equation of state (lowest Z)
        **kwargs: Optional[Dict]
            additional arguments
                - tolerance: float, tolerance for the calculation (default: 1e-1)

        Returns
        -------
        res: Dict
            results of fugacity calculation

        Notes
        -----
        ### solver_method:
            - `ls`: least square method
            - `newton`: newton method
            - `fsolve`: fsolve method

        ### root_analysis_set:
            - set [1]: 3 roots (VAPOR-LIQUID)
                - At `T < Tc` and `P = Psat`, 3 real roots → smallest is liquid, largest is vapor (vapor-liquid).
            - set [2]: 1 root (LIQUID)
                - At `T < Tc` and `P > Psat`, EOS may give 1 or 3 roots → use smallest (liquid).
            - set [3]: 1 root (VAPOR)
                - At `T < Tc` and `P < Psat`, EOS may give 1 or 3 roots → use largest (vapor).
            - set [4]: 1 root (SUPERCRITICAL)
                - At `T > Tc`, only 1 real root → fluid is supercritical (vapor-like or liquid-like).

        Examples
        --------
        `model_input` defines as:

        ```python
        # single component
        N0s = {
            "CO2": 1.0
        }

        # multi-component
        N0s = {
            "CO2": 0.5,
            "N2": 0.5
        }

        # Others
        # model input
        # eos model
        eos_model = 'SRK'
        # component phase
        phase = "VAPOR"
        # temperature [K]
        T = 300
        # pressure [Pa]
        P = 1.2*1e5

        # model input
        model_input = {
            "phase": phase,
            "feed-specification": N0s,
            "pressure": [P, 'Pa'],
            "temperature": [T, 'K'],
        }

        # model source
        model_source = {
            "datasource": datasource,
            "equationsource": equationsource
        }
        ```
        '''
        try:
            # SECTION: keywords
            # tolerance
            tolerance = kwargs.get('tolerance', 1e-1)

            # SECTION: set input parameters
            # eos
            eos_model = model_name.upper()
            eos_model = eos_model_name(eos_model)

            # SECTION: phase
            phase = model_input.get('phase', None)

            # check
            if phase is not None:
                phase = phase.upper()
                # check phase
                if phase not in [
                    'VAPOR', 'LIQUID', 'VAPOR-LIQUID', 'SUPERCRITICAL'
                ]:
                    raise Exception('Invalid phase provided!')

            # NOTE: calculation mode
            calculation_mode = 'mixture'
            # NOTE: component number
            component_num = 0

            # SECTION: input
            feed_spec = model_input.get('feed-specification', None)
            # check
            if feed_spec is None or feed_spec == 'None':
                raise Exception('Feed specification is not provided!')

            # component list
            components = []
            # mole fraction
            mole_fraction = []
            # looping through feed-specification
            for key, value in feed_spec.items():
                components.append(key)
                mole_fraction.append(value)

            # SECTION: check component
            if isinstance(components, list):
                # set
                component_num = len(components)
                # single
                if len(components) == 1:
                    raise Exception(
                        'Single component calculation is not allowed!')
            else:
                raise Exception('Components list not provided!')

            if len(mole_fraction) != component_num:
                raise Exception('Mole fraction list not provided!')

            # NOTE: check temperature and pressure
            if 'pressure' not in model_input.keys():
                raise Exception('No pressure in operating conditions!')

            if 'temperature' not in model_input.keys():
                raise Exception('No temperature in operating conditions!')

            # operating conditions
            operating_conditions = {
                "pressure": model_input["pressure"],
                "temperature": model_input["temperature"]
            }

            # SECTION: check eos roots
            if phase is None:
                # check eos roots for single component
                eos_roots_analysis = self.check_eos_roots_multi_component(
                    model_name=eos_model,
                    model_input=model_input,
                    model_source=model_source
                )

                # NOTE: eos parms
                # set phase
                phase = str(eos_roots_analysis['mixture']['phase'])

            # check phase
            if phase is None:
                raise Exception('Phase is not provided!')

            # NOTE: set eos parameters
            eos_parms = {
                'phase': phase,
                'eos-model': eos_model,
                'mode': calculation_mode,
                'liquid-fugacity-mode': liquid_fugacity_mode
            }

            # SECTION: set datasource and equationsource
            # NOTE: check if datasource and equationsource are provided in model_input
            # datasource
            datasource = model_source.get('datasource', {})
            # equationsource
            equationsource = model_source.get('equationsource', {})
            # set thermodb link
            link_status = self.set_thermodb_link(datasource, equationsource)
            # check
            if not link_status:
                raise Exception('Thermodb link failed!')

            # SECTION: reference for eos
            reference = self._references.get(eos_model, None)

            # build datasource
            component_datasource = self.set_datasource(components, reference)
            # build equation source
            equation_equationsource = self.set_equationsource(
                components, reference)

            # SECTION: init fugacity core
            FugacityCoreC = FugacityCore(
                component_datasource,
                equation_equationsource,
                components,
                operating_conditions,
                eos_parms,
                **kwargs
            )

            # SECTION: calculation mode
            res = FugacityCoreC.fugacity_cal(
                mole_fraction,
                solver_method=solver_method
            )

            return res
        except Exception as e:
            raise Exception("Fugacity calculation failed!, ", e)

    def check_eos_roots_multi_component(
        self,
        model_name: Literal[
            'SRK', 'PR', 'RK', 'vdW'
        ],
        model_input: Dict,
        model_source: Dict,
        bubble_point_pressure_mode: Literal[
            "Raoult"
        ] = "Raoult",
        dew_point_pressure_mode: Literal[
            "Raoult"
        ] = "Raoult",
        **kwargs
    ) -> Dict[str, Any]:
        '''
        Check eos roots for the multi-components at different temperature and pressure. To do so, the bubble point and dew point pressure are calculated using Raoult's law. Assuming that the mixture is ideal, the bubble point pressure is equal to the vapor pressure of the component at the bubble point temperature. The dew point pressure is equal to the vapor pressure of the component at the dew point temperature.

        Parameters
        ----------
        model_name: str
            eos model name,
                - `SRK`: Soave-Redlich-Kwong
                - `PR`: Peng-Robinson,
                - `RK`: Redlich-Kwong
                - `vdW`: Van der Waals
        model_input: Dict
            model input
                - `feed-specification`: dict, feed specification such as `{'CO2': 1.0}` or `{'CO2': 0.5, 'N2': 0.5}`
                - `pressure`: list, pressure and unit, such as `[1.2*1e5, 'Pa']`
                - `temperature`: list, temperature and unit, such as `[300, 'K']`
        model_source: Dict
            datasource and equationsource needed for fugacity calculation
                `datasource`: dict, datasource for the component (`generated by PyThermoDB`)
                `equationsource`: dict, equationsource for the component (`generated by PyThermoDB`)
        bubble_point_pressure_mode: str
            bubble point pressure calculation mode, `Raoult`: Raoult's law
        dew_point_pressure_mode: str
            dew point pressure calculation mode, `Raoult`: Raoult's law

        Returns
        -------
        res: Dict
            eos root analysis

        Notes
        -----
        * The phase of the component is determined based on the temperature and pressure provided in the model_input.

        1. At T < Tc and P = Psat, 3 real roots → smallest is liquid, largest is vapor (vapor-liquid).
        2. At T < Tc and P > Psat, EOS may give 1 or 3 roots → use smallest (liquid).
        3. At T < Tc and P < Psat, EOS may give 1 or 3 roots → use largest (vapor).
        4. At T = Tc, one real root (critical point) → fluid is at critical state.
        5. At T > Tc, only 1 real root → fluid is supercritical (vapor-like or liquid-like).

        * Pressure and temperature are in SI units (Pa, K), if not provided in the model_input, they are automatically converted.

        * Raoult's law analysis does not inherently account for the critical point of the mixture (or even of individual components). It's based on assumptions that:

        - The liquid behaves ideally (activity coefficients = 1)
        - The vapor behaves ideally (fugacity coefficients = 1)
        - It is far from the critical point, where such assumptions break down

        '''
        try:
            # SECTION: keywords
            # tolerance
            tolerance = kwargs.get('tolerance', 1e-1)

            # SECTION: set input parameters
            # NOTE: eos model
            eos_model = model_name.upper()
            eos_model = eos_model_name(eos_model)

            # NOTE: component number
            component_num = 0

            # SECTION: feed specification
            feed_spec = model_input.get('feed-specification')
            # check
            if feed_spec is None or feed_spec == 'None':
                raise Exception('Feed specification is not provided!')

            # NOTE: component list
            components = []
            # mole fraction
            mole_fraction = []
            # looping through feed-specification
            for key, value in feed_spec.items():
                components.append(key)
                mole_fraction.append(value)

            # NOTE: check
            if isinstance(components, list):
                # set
                component_num = len(components)
                # single
                if len(components) == 1:
                    raise Exception(
                        'Single component calculation is not allowed!')
            else:
                raise Exception('Components list not provided!')

            # NOTE: check if multi
            if len(mole_fraction) != component_num:
                raise Exception('Mole fraction list not provided!')

            # SECTION: check temperature and pressure
            if 'pressure' not in model_input.keys():
                raise Exception('No pressure in operating conditions!')

            if 'temperature' not in model_input.keys():
                raise Exception('No temperature in operating conditions!')

            # NOTE: operating conditions
            operating_conditions = {
                "pressure": model_input["pressure"],
                "temperature": model_input["temperature"]
            }

            # SECTION: set datasource and equationsource
            # NOTE: check if datasource and equationsource are provided in model_input
            # datasource
            datasource = model_source.get('datasource', {})
            # equationsource
            equationsource = model_source.get('equationsource', {})
            # set thermodb link
            link_status = self.set_thermodb_link(datasource, equationsource)
            # check
            if not link_status:
                raise Exception('Thermodb link failed!')

            # SECTION: reference for eos
            reference = self._references.get(eos_model, None)

            # build datasource
            component_datasource = self.set_datasource(components, reference)
            # build equation source
            equation_equationsource = self.set_equationsource(
                components, reference)

            # SECTION: operating conditions
            # pressure [Pa]
            P = pycuc.to(
                operating_conditions["pressure"][0],
                f"{operating_conditions['pressure'][1]} => Pa")
            # temperature [K]
            T = pycuc.to(
                operating_conditions["temperature"][0],
                f"{operating_conditions['temperature'][1]} => K")

            # SECTION: init
            EOSUtilsC = EOSUtils(component_datasource, equation_equationsource)

            # SECTION: eos root analysis
            res = EOSUtilsC.eos_root_analysis(
                P,
                T,
                components,
                tolerance=tolerance,
                bubble_point_pressure_mode=bubble_point_pressure_mode,
                dew_point_pressure_mode=dew_point_pressure_mode,
                mole_fraction=mole_fraction
            )
            # res
            return res
        except Exception as e:
            raise Exception("Fugacity calculation failed!, ", e)
