"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const minimatch = require("minimatch");
const assets_1 = require("./assets");
const bundling_1 = require("./bundling");
const fs_1 = require("./fs");
const names_1 = require("./names");
const cache_1 = require("./private/cache");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * Stages a file or directory from a location on the file system into a staging directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 *
 * @stability stable
 */
class AssetStaging extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.sourcePath = path.resolve(props.sourcePath);
        this.fingerprintOptions = props;
        if (!fs.existsSync(this.sourcePath)) {
            throw new Error(`Cannot find asset at ${this.sourcePath}`);
        }
        this.sourceStats = fs.statSync(this.sourcePath);
        const outdir = (_b = stage_1.Stage.of(this)) === null || _b === void 0 ? void 0 : _b.assetOutdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly asset output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        this.assetOutdir = outdir;
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        this.customSourceFingerprint = props.assetHash;
        this.hashType = determineHashType(props.assetHashType, this.customSourceFingerprint);
        // Decide what we're going to do, without actually doing it yet
        let stageThisAsset;
        let skip = false;
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_c = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _c !== void 0 ? _c : ['*'];
            skip = !bundlingStacks.find(pattern => minimatch(stack_1.Stack.of(this).stackName, pattern));
            const bundling = props.bundling;
            stageThisAsset = () => this.stageByBundling(bundling, skip);
        }
        else {
            stageThisAsset = () => this.stageByCopying();
        }
        // Calculate a cache key from the props. This way we can check if we already
        // staged this asset and reuse the result (e.g. the same asset with the same
        // configuration is used in multiple stacks). In this case we can completely
        // skip file system and bundling operations.
        //
        // The output directory and whether this asset is skipped or not should also be
        // part of the cache key to make sure we don't accidentally return the wrong
        // staged asset from the cache.
        this.cacheKey = calculateCacheKey({
            outdir: this.assetOutdir,
            sourcePath: path.resolve(props.sourcePath),
            bundling: props.bundling,
            assetHashType: this.hashType,
            customFingerprint: this.customSourceFingerprint,
            extraHash: props.extraHash,
            exclude: props.exclude,
            ignoreMode: props.ignoreMode,
            skip,
        });
        const staged = AssetStaging.assetCache.obtain(this.cacheKey, stageThisAsset);
        this.stagedPath = staged.stagedPath;
        this.absoluteStagedPath = staged.stagedPath;
        this.assetHash = staged.assetHash;
        this.packaging = staged.packaging;
        this.isArchive = staged.isArchive;
    }
    /**
     * Clears the asset hash cache.
     *
     * @stability stable
     */
    static clearAssetHashCache() {
        this.assetCache.clear();
    }
    /**
     * (deprecated) A cryptographic hash of the asset.
     *
     * @deprecated see `assetHash`.
     */
    get sourceHash() {
        return this.assetHash;
    }
    /**
     * Return the path to the staged asset, relative to the Cloud Assembly (manifest) directory of the given stack.
     *
     * Only returns a relative path if the asset was staged, returns an absolute path if
     * it was not staged.
     *
     * A bundled asset might end up in the outDir and still not count as
     * "staged"; if asset staging is disabled we're technically expected to
     * reference source directories, but we don't have a source directory for the
     * bundled outputs (as the bundle output is written to a temporary
     * directory). Nevertheless, we will still return an absolute path.
     *
     * A non-obvious directory layout may look like this:
     *
     * ```
     *    CLOUD ASSEMBLY ROOT
     *      +-- asset.12345abcdef/
     *      +-- assembly-Stage
     *            +-- MyStack.template.json
     *            +-- MyStack.assets.json <- will contain { "path": "../asset.12345abcdef" }
     * ```
     *
     * @stability stable
     */
    relativeStagedPath(stack) {
        var _b;
        const asmManifestDir = (_b = stage_1.Stage.of(stack)) === null || _b === void 0 ? void 0 : _b.outdir;
        if (!asmManifestDir) {
            return this.stagedPath;
        }
        const isOutsideAssetDir = path.relative(this.assetOutdir, this.stagedPath).startsWith('..');
        if (isOutsideAssetDir || this.stagingDisabled) {
            return this.stagedPath;
        }
        return path.relative(asmManifestDir, this.stagedPath);
    }
    /**
     * Stage the source to the target by copying
     *
     * Optionally skip if staging is disabled, in which case we pretend we did something but we don't really.
     */
    stageByCopying() {
        const assetHash = this.calculateHash(this.hashType);
        const stagedPath = this.stagingDisabled
            ? this.sourcePath
            : path.resolve(this.assetOutdir, renderAssetFilename(assetHash, path.extname(this.sourcePath)));
        if (!this.sourceStats.isDirectory() && !this.sourceStats.isFile()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be either a directory or a regular file`);
        }
        this.stageAsset(this.sourcePath, stagedPath, 'copy');
        return {
            assetHash,
            stagedPath,
            packaging: this.sourceStats.isDirectory() ? assets_1.FileAssetPackaging.ZIP_DIRECTORY : assets_1.FileAssetPackaging.FILE,
            isArchive: this.sourceStats.isDirectory() || ARCHIVE_EXTENSIONS.includes(path.extname(this.sourcePath).toLowerCase()),
        };
    }
    /**
     * Stage the source to the target by bundling
     *
     * Optionally skip, in which case we pretend we did something but we don't really.
     */
    stageByBundling(bundling, skip) {
        var _b;
        if (!this.sourceStats.isDirectory()) {
            throw new Error(`Asset ${this.sourcePath} is expected to be a directory when bundling`);
        }
        if (skip) {
            // We should have bundled, but didn't to save time. Still pretend to have a hash.
            // If the asset uses OUTPUT or BUNDLE, we use a CUSTOM hash to avoid fingerprinting
            // a potentially very large source directory. Other hash types are kept the same.
            let hashType = this.hashType;
            if (hashType === assets_1.AssetHashType.OUTPUT || hashType === assets_1.AssetHashType.BUNDLE) {
                this.customSourceFingerprint = names_1.Names.uniqueId(this);
                hashType = assets_1.AssetHashType.CUSTOM;
            }
            return {
                assetHash: this.calculateHash(hashType, bundling),
                stagedPath: this.sourcePath,
                packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
                isArchive: true,
            };
        }
        // Try to calculate assetHash beforehand (if we can)
        let assetHash = this.hashType === assets_1.AssetHashType.SOURCE || this.hashType === assets_1.AssetHashType.CUSTOM
            ? this.calculateHash(this.hashType, bundling)
            : undefined;
        const bundleDir = this.determineBundleDir(this.assetOutdir, assetHash);
        this.bundle(bundling, bundleDir);
        // Check bundling output content and determine if we will need to archive
        const bundlingOutputType = (_b = bundling.outputType) !== null && _b !== void 0 ? _b : bundling_1.BundlingOutput.AUTO_DISCOVER;
        const bundledAsset = determineBundledAsset(bundleDir, bundlingOutputType);
        // Calculate assetHash afterwards if we still must
        assetHash = assetHash !== null && assetHash !== void 0 ? assetHash : this.calculateHash(this.hashType, bundling, bundledAsset.path);
        const stagedPath = path.resolve(this.assetOutdir, renderAssetFilename(assetHash, bundledAsset.extension));
        this.stageAsset(bundledAsset.path, stagedPath, 'move');
        // If bundling produced a single archive file we "touch" this file in the bundling
        // directory after it has been moved to the staging directory. This way if bundling
        // is skipped because the bundling directory already exists we can still determine
        // the correct packaging type.
        if (bundledAsset.packaging === assets_1.FileAssetPackaging.FILE) {
            fs.closeSync(fs.openSync(bundledAsset.path, 'w'));
        }
        return {
            assetHash,
            stagedPath,
            packaging: bundledAsset.packaging,
            isArchive: true,
        };
    }
    /**
     * Whether staging has been disabled
     */
    get stagingDisabled() {
        return !!this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
    }
    /**
     * Copies or moves the files from sourcePath to targetPath.
     *
     * Moving implies the source directory is temporary and can be trashed.
     *
     * Will not do anything if source and target are the same.
     */
    stageAsset(sourcePath, targetPath, style) {
        // Is the work already done?
        const isAlreadyStaged = fs.existsSync(targetPath);
        if (isAlreadyStaged) {
            if (style === 'move' && sourcePath !== targetPath) {
                fs.removeSync(sourcePath);
            }
            return;
        }
        // Moving can be done quickly
        if (style == 'move') {
            fs.renameSync(sourcePath, targetPath);
            return;
        }
        // Copy file/directory to staging directory
        if (this.sourceStats.isFile()) {
            fs.copyFileSync(sourcePath, targetPath);
        }
        else if (this.sourceStats.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${sourcePath}`);
        }
    }
    /**
     * Determine the directory where we're going to write the bundling output
     *
     * This is the target directory where we're going to write the staged output
     * files if we can (if the hash is fully known), or a temporary directory
     * otherwise.
     */
    determineBundleDir(outdir, sourceHash) {
        if (sourceHash) {
            return path.resolve(outdir, renderAssetFilename(sourceHash));
        }
        // When the asset hash isn't known in advance, bundler outputs to an
        // intermediate directory named after the asset's cache key
        return path.resolve(outdir, `bundling-temp-${this.cacheKey}`);
    }
    /**
     * Bundles an asset to the given directory
     *
     * If the given directory already exists, assume that everything's already
     * in order and don't do anything.
     *
     * @param options Bundling options
     * @param bundleDir Where to create the bundle directory
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, bundleDir) {
        var _b, _c, _d, _e;
        if (fs.existsSync(bundleDir)) {
            return;
        }
        fs.ensureDirSync(bundleDir);
        // Chmod the bundleDir to full access.
        fs.chmodSync(bundleDir, 0o777);
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_b = options.volumes) !== null && _b !== void 0 ? _b : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_c = options.local) === null || _c === void 0 ? void 0 : _c.tryBundle(bundleDir, options);
            if (!localBundling) {
                let user;
                if (options.user) {
                    user = options.user;
                }
                else { // Default to current user
                    const userInfo = os.userInfo();
                    user = userInfo.uid !== -1 // uid is -1 on Windows
                        ? `${userInfo.uid}:${userInfo.gid}`
                        : '1000:1000';
                }
                options.image.run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_d = options.workingDirectory) !== null && _d !== void 0 ? _d : AssetStaging.BUNDLING_INPUT_DIR,
                    securityOpt: (_e = options.securityOpt) !== null && _e !== void 0 ? _e : '',
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
    }
    calculateHash(hashType, bundling, outputDir) {
        var _b;
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update((_b = this.customSourceFingerprint) !== null && _b !== void 0 ? _b : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!outputDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(outputDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
_a = JSII_RTTI_SYMBOL_1;
AssetStaging[_a] = { fqn: "@aws-cdk/core.AssetStaging", version: "1.129.0" };
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 *
 * @stability stable
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 *
 * @stability stable
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
/**
 * Cache of asset hashes based on asset configuration to avoid repeated file
 * system and bundling operations.
 */
AssetStaging.assetCache = new cache_1.Cache();
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param customSourceFingerprint Asset hash seed given in the construct props
 */
function determineHashType(assetHashType, customSourceFingerprint) {
    const hashType = customSourceFingerprint
        ? (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.CUSTOM)
        : (assetHashType !== null && assetHashType !== void 0 ? assetHashType : assets_1.AssetHashType.SOURCE);
    if (customSourceFingerprint && hashType !== assets_1.AssetHashType.CUSTOM) {
        throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
    }
    if (hashType === assets_1.AssetHashType.CUSTOM && !customSourceFingerprint) {
        throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
    }
    return hashType;
}
/**
 * Calculates a cache key from the props. Normalize by sorting keys.
 */
function calculateCacheKey(props) {
    return crypto.createHash('sha256')
        .update(JSON.stringify(sortObject(props)))
        .digest('hex');
}
/**
 * Recursively sort object keys
 */
function sortObject(object) {
    if (typeof object !== 'object' || object instanceof Array) {
        return object;
    }
    const ret = {};
    for (const key of Object.keys(object).sort()) {
        ret[key] = sortObject(object[key]);
    }
    return ret;
}
/**
 * Returns the single archive file of a directory or undefined
 */
function singleArchiveFile(directory) {
    if (!fs.existsSync(directory)) {
        throw new Error(`Directory ${directory} does not exist.`);
    }
    if (!fs.statSync(directory).isDirectory()) {
        throw new Error(`${directory} is not a directory.`);
    }
    const content = fs.readdirSync(directory);
    if (content.length === 1) {
        const file = path.join(directory, content[0]);
        const extension = path.extname(content[0]).toLowerCase();
        if (fs.statSync(file).isFile() && ARCHIVE_EXTENSIONS.includes(extension)) {
            return file;
        }
    }
    return undefined;
}
/**
 * Returns the bundled asset to use based on the content of the bundle directory
 * and the type of output.
 */
function determineBundledAsset(bundleDir, outputType) {
    const archiveFile = singleArchiveFile(bundleDir);
    // auto-discover means that if there is an archive file, we take it as the
    // bundle, otherwise, we will archive here.
    if (outputType === bundling_1.BundlingOutput.AUTO_DISCOVER) {
        outputType = archiveFile ? bundling_1.BundlingOutput.ARCHIVED : bundling_1.BundlingOutput.NOT_ARCHIVED;
    }
    switch (outputType) {
        case bundling_1.BundlingOutput.NOT_ARCHIVED:
            return { path: bundleDir, packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY };
        case bundling_1.BundlingOutput.ARCHIVED:
            if (!archiveFile) {
                throw new Error('Bundling output directory is expected to include only a single .zip or .jar file when `output` is set to `ARCHIVED`');
            }
            return { path: archiveFile, packaging: assets_1.FileAssetPackaging.FILE, extension: path.extname(archiveFile) };
    }
}
//# sourceMappingURL=data:application/json;base64,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