/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
import * as sqs from 'aws-cdk-lib/aws-sqs';
import * as sns from 'aws-cdk-lib/aws-sns';
import * as subscriptions from 'aws-cdk-lib/aws-sns-subscriptions';
import * as kms from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
/**
 * @summary The properties for the SnsToSqs class.
 */
export interface SnsToSqsProps {
    /**
     * Existing instance of SNS topic object, providing both this and topicProps will cause an error..
     *
     * @default - Default props are used
     */
    readonly existingTopicObj?: sns.Topic;
    /**
     * Optional user provided properties to override the default properties for the SNS topic.
     *
     * @default - Default properties are used.
     */
    readonly topicProps?: sns.TopicProps;
    /**
     * Existing instance of SQS queue object, Providing both this and queueProps will cause an error.
     *
     * @default - Default props are used
     */
    readonly existingQueueObj?: sqs.Queue;
    /**
     * Optional user provided properties
     *
     * @default - Default props are used
     */
    readonly queueProps?: sqs.QueueProps;
    /**
     * Optional user provided properties for the dead letter queue
     *
     * @default - Default props are used
     */
    readonly deadLetterQueueProps?: sqs.QueueProps;
    /**
     * Whether to deploy a secondary queue to be used as a dead letter queue.
     *
     * @default - true.
     */
    readonly deployDeadLetterQueue?: boolean;
    /**
     * The number of times a message can be unsuccessfully dequeued before being moved to the dead-letter queue.
     *
     * @default - required field if deployDeadLetterQueue=true.
     */
    readonly maxReceiveCount?: number;
    /**
     * If no keys are provided, this flag determines whether both the topic and queue are encrypted with a new CMK or an AWS managed key.
     * This flag is ignored if any of the following are defined:
     * topicProps.masterKey, queueProps.encryptionMasterKey, encryptionKey or encryptionKeyProps.
     *
     * @default - True if topicProps.masterKey, queueProps.encryptionMasterKey, encryptionKey, and encryptionKeyProps are all undefined.
     *
     * @deprecated Use the separate attributes for Queue and Topic encryption.
     */
    readonly enableEncryptionWithCustomerManagedKey?: boolean;
    /**
     * An optional, imported encryption key to encrypt the SQS Queue and SNS Topic with. We recommend
     * you migrate your code to use  queueEncryptionKey and topicEncryptionKey in place of this prop value.
     *
     * @default - None
     *
     * @deprecated Use the separate attributes for Queue and Topic encryption.
     */
    readonly encryptionKey?: kms.Key;
    /**
     * Optional user provided properties to override the default properties for the KMS encryption key used to
     * encrypt the SQS topic and queue with. We recommend you migrate your code to use queueEncryptionKeyProps
     * and topicEncryptionKeyProps in place of this prop value.
     *
     * @default - None
     *
     * @deprecated Use the separate attributes for Queue and Topic encryption.
     */
    readonly encryptionKeyProps?: kms.KeyProps;
    /**
     * Optional user-provided props to override the default props for sqsSubscriptionProps.
     *
     * @default - Default props are used.
     */
    readonly sqsSubscriptionProps?: subscriptions.SqsSubscriptionProps;
    /**
     * Whether to encrypt the Queue with a customer managed KMS key (CMK). This is the default
     * behavior, and this property defaults to true - if it is explicitly set to false then the
     * Queue is encrypted with an Amazon managed KMS key. For a completely unencrypted Queue (not recommended),
     * create the Queue separately from the construct and pass it in using the existingQueueObject.
     * Since SNS subscriptions do not currently support SQS queues with AWS managed encryption keys,
     * setting this to false will always result in an error from the underlying CDK - we have still
     * included this property for consistency with topics and to be ready if the services one day support
     * this functionality.
     *
     * @default - false
     */
    readonly encryptQueueWithCmk?: boolean;
    /**
     * An optional CMK that will be used by the construct to encrypt the new SQS queue.
     *
     * @default - None
     */
    readonly existingQueueEncryptionKey?: kms.Key;
    /**
     * An optional subset of key properties to override the default properties used by constructs (`enableKeyRotation: true`).
     * These properties will be used in constructing the CMK used to encrypt the SQS queue.
     *
     * @default - None
     */
    readonly queueEncryptionKeyProps?: kms.KeyProps;
    /**
     * Whether to encrypt the Topic with a customer managed KMS key (CMK). This is the
     * default behavior, and this property defaults to true - if it is explicitly set
     * to false then the Topic is encrypted with an Amazon managed KMS key. For a completely unencrypted
     * Topic (not recommended), create the Topic separately from the construct and pass it in using the existingTopicObject.
     *
     * @default - false
     */
    readonly encryptTopicWithCmk?: boolean;
    /**
     * An optional CMK that will be used by the construct to encrypt the new SNS Topic.
     *
     * @default - None
     */
    readonly existingTopicEncryptionKey?: kms.Key;
    /**
     * An optional subset of key properties to override the default properties used by constructs
     * (`enableKeyRotation: true`). These properties will be used in constructing the CMK used to
     * encrypt the SNS topic.
     *
     * @default - None
     */
    readonly topicEncryptionKeyProps?: kms.KeyProps;
}
export interface ConstructKeys {
    readonly useDeprecatedInterface: boolean;
    readonly singleKey?: kms.Key;
    readonly topicKey?: kms.Key;
    readonly queueKey?: kms.Key;
    readonly encryptTopicWithCmk: boolean;
    readonly encryptQueueWithCmk: boolean;
}
/**
 * @summary The SnsToSqs class.
 */
export declare class SnsToSqs extends Construct {
    readonly snsTopic: sns.Topic;
    readonly encryptionKey?: kms.Key;
    readonly queueEncryptionKey?: kms.Key;
    readonly topicEncryptionKey?: kms.Key;
    readonly sqsQueue: sqs.Queue;
    readonly deadLetterQueue?: sqs.DeadLetterQueue;
    /**
     * @summary Constructs a new instance of the SnsToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {SnsToSqsProps} props - user provided props for the construct.
     * @since 1.62.0
     * @access public
     */
    constructor(scope: Construct, id: string, props: SnsToSqsProps);
    static createRequiredKeys(scope: Construct, id: string, props: SnsToSqsProps): ConstructKeys;
    private uniquePropChecks;
}
