"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs FIFO Queue';
stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
// Definitions
const snsManagedKey = kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns');
const topic = new sns.Topic(stack, 'TestTopic', {
    contentBasedDeduplication: true,
    displayName: 'Customer subscription topic',
    fifo: true,
    masterKey: snsManagedKey
});
const queueEncryptionKeyProps = {
    enableKeyRotation: true,
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
};
const existingQueueKey = new kms.Key(stack, 'ImportedEncryptionKey', queueEncryptionKeyProps);
const deadLetterQueueKey = new kms.Key(stack, 'ImportedDLQEncryptionKey', queueEncryptionKeyProps);
const props = {
    encryptQueueWithCmk: true,
    existingQueueEncryptionKey: existingQueueKey,
    existingTopicObj: topic,
    queueProps: {
        fifo: true,
    },
    deadLetterQueueProps: {
        encryption: sqs.QueueEncryption.KMS,
        fifo: true,
        encryptionMasterKey: deadLetterQueueKey
    },
};
new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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