"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../../lib");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs';
stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", false);
const sqsEncryptionKey = new kms.Key(stack, 'ImportedSQSEncryptionKey', {
    enableKeyRotation: true,
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
});
const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {
    encryptionMasterKey: sqsEncryptionKey
});
// Definitions
const props = {
    sqsSubscriptionProps: {
        rawMessageDelivery: true,
        // To test this stack, when publishing the SNS message, add a message attribute:
        // Type: string, Name: store, Value: (any value)
        filterPolicy: {
            store: sns.SubscriptionFilter.existsFilter(),
        },
        deadLetterQueue: dlq
    }
};
const snsToSqsStack = new lib_1.SnsToSqs(stack, 'test-sns-sqs-stack', props);
// Grant yourself permissions to use the Customer Managed KMS Key
const policyStatement = new iam.PolicyStatement({
    actions: ["kms:Encrypt", "kms:Decrypt"],
    effect: iam.Effect.ALLOW,
    principals: [new iam.AccountRootPrincipal()],
    resources: ["*"]
});
snsToSqsStack.encryptionKey?.addToResourcePolicy(policyStatement);
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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