"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const sns = require("aws-cdk-lib/aws-sns");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const utils_1 = require("./utils");
test('Pattern deployment w/ new Topic, new Queue and default props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {};
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    // Confirm subscription has proper target
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
test('Pattern deployment w/ new topic, new queue, and overridden props', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testTopicName = "topicehdjs";
    const testQueueName = "queueehdjs.fifo";
    const props = {
        topicProps: {
            topicName: testTopicName,
        },
        queueProps: {
            queueName: testQueueName,
            fifo: true
        },
        queueEncryptionKeyProps: {
            enableKeyRotation: false
        },
        encryptQueueWithCmk: true,
        encryptTopicWithCmk: true,
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).not.toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        TopicName: testTopicName,
    });
    template.hasResourceProperties("AWS::SQS::Queue", {
        QueueName: testQueueName,
        FifoQueue: true,
    });
    // One key has rotation (default), one does not (override)
    // NOTE - THIS IS FOR TESTING, Key Rotation is a best practice
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false
    });
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
test('Test getter methods', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Test deployment w/ existing queue, and topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testTopicName = 'existing-topic-adfa';
    const testQueueName = 'existing-queue-zfx';
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: testTopicName
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: testQueueName
    });
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingTopicObj: topic,
        existingQueueObj: queue
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    // Since we used simple CDK calls to make topic/queue, there are no CMKs nor DLQ
    template.resourceCountIs("AWS::KMS::Key", 0);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
    // Confirm subscription has proper target
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
test('Test deployment with imported encryption key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testKeyAlias = 'key-asdg';
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false,
        alias: testKeyAlias
    });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        existingQueueEncryptionKey: kmsKey
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
    });
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: true,
    });
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Topic", {
        KmsMasterKeyId: assertions_1.Match.anyValue()
    });
});
test('Test deployment with SNS managed KMS key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: kms.Alias.fromAliasName(stack, 'sns-managed-key', 'alias/aws/sns')
        },
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'test-key', {}),
        },
        enableEncryptionWithCustomerManagedKey: false
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.sse);
});
test('Test deployment with CMK encrypted SNS Topic (avoids interface)', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const testDescription = 'someString-llasdj';
    const testTopicName = 'topic-hyuyj';
    const cmk = new kms.Key(stack, 'cmk', { description: testDescription });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        topicProps: {
            masterKey: cmk,
            topicName: testTopicName
        }
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::KMS::Key", {
        Description: testDescription
    });
});
test('Pattern deployment w/ existing topic and FIFO queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const topic = new sns.Topic(stack, 'TestTopic', {
        contentBasedDeduplication: true,
        displayName: 'Customer subscription topic',
        fifo: true,
        topicName: 'customerTopic',
    });
    const props = {
        encryptQueueWithCmk: false,
        existingTopicObj: topic,
        queueProps: {
            fifo: true,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: true,
        }
    };
    const app = () => {
        new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    };
    // Assertion
    expect(app).toThrowError("SQS queue encrypted by AWS managed KMS key cannot be used as SNS subscription");
});
test('Pattern deployment w/ existing topic and Standard queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const topic = new sns.Topic(stack, 'TestTopic', {
        displayName: 'Customer subscription topic',
        fifo: false,
        topicName: 'customerTopic',
    });
    const props = {
        encryptQueueWithCmk: false,
        existingTopicObj: topic,
        queueProps: {
            fifo: false,
        },
        deadLetterQueueProps: {
            encryption: sqs.QueueEncryption.UNENCRYPTED,
            fifo: false,
        }
    };
    const app = () => {
        new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    };
    // Assertion
    expect(app).toThrowError("SQS queue encrypted by AWS managed KMS key cannot be used as SNS subscription");
});
test('Check sqsSubscriptionProps are used', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        sqsSubscriptionProps: {
            rawMessageDelivery: true,
            filterPolicy: {
                color: sns.SubscriptionFilter.stringFilter({
                    allowlist: ['red', 'orange'],
                    matchPrefixes: ['bl'],
                })
            }
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
        RawMessageDelivery: true,
        FilterPolicy: {
            color: [
                "red",
                "orange",
                {
                    prefix: "bl"
                }
            ]
        }
    });
});
test('Check SNS dead letter queue', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {});
    const props = {
        sqsSubscriptionProps: {
            deadLetterQueue: dlq
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    const template = assertions_1.Template.fromStack(stack);
    // The SNS DLQ is a third queue and should be attached to the Subscription
    template.resourceCountIs("AWS::SQS::Queue", 3);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        RedrivePolicy: assertions_1.Match.anyValue()
    });
});
test('Construct uses topicProps.masterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const cmk = new kms.Key(stack, 'cmkfortopic');
    const noKeyQueue = new sqs.Queue(stack, 'placeHolderQueue', {});
    const props = {
        topicProps: {
            masterKey: cmk
        },
        existingQueueObj: noKeyQueue,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.deadLetterQueue).not.toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 1);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.none);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Construct uses queueProps.encryptionMasterKey when specified (avoids interface)', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const cmk = new kms.Key(stack, 'cmkforqueue', {});
    const props = {
        queueProps: {
            encryptionMasterKey: cmk
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
    expect(testConstruct.queueEncryptionKey).not.toBeDefined();
    expect(testConstruct.topicEncryptionKey).toBeDefined();
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 1);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.none);
});
test('Test deployment with existing encryption key for Topic', () => {
    const testDescription = "someValue";
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Setup
    const kmsKey = new kms.Key(stack, 'imported-key', {
        enableKeyRotation: false,
        description: testDescription
    });
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true,
        existingTopicEncryptionKey: kmsKey
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
        Description: testDescription
    });
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Test deployment with key props for Topic', () => {
    const testDescription = "test-description-lkjh";
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true,
        topicEncryptionKeyProps: {
            enableKeyRotation: false,
            description: testDescription
        }
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    template.hasResourceProperties("AWS::KMS::Key", {
        EnableKeyRotation: false,
        Description: testDescription
    });
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Test deployment with no key props or key for Topic', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const testConstruct = new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
        encryptTopicWithCmk: true
    });
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.cmk);
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.none);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.sqsQueue).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
    expect(testConstruct.queueEncryptionKey).toBeDefined();
    expect(testConstruct.topicEncryptionKey).toBeDefined();
    expect(testConstruct.queueEncryptionKey?.keyId).not.toEqual(testConstruct.topicEncryptionKey?.keyId);
    expect(testConstruct.deadLetterQueue).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.resourceCountIs("AWS::SNS::Subscription", 1);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
});
test('Old interface use with feature flag enabled', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const props = {
        // Force a second key to avoid circular reference problem
        enableEncryptionWithCustomerManagedKey: true,
        queueProps: {
            encryptionMasterKey: new kms.Key(stack, 'queueKey', {})
        }
    };
    const testConstruct = new lib_1.SnsToSqs(stack, 'test-sns-sqs', props);
    (0, utils_1.CheckKeyProperty)(testConstruct.queueEncryptionKey, utils_1.keyType.none);
    (0, utils_1.CheckKeyProperty)(testConstruct.topicEncryptionKey, utils_1.keyType.none);
    // this is the key created for the topic
    (0, utils_1.CheckKeyProperty)(testConstruct.encryptionKey, utils_1.keyType.cmk);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    (0, utils_1.CheckQueueKeyType)(template, utils_1.keyType.cmk);
    (0, utils_1.CheckTopicKeyType)(template, utils_1.keyType.cmk);
    template.resourceCountIs("AWS::KMS::Key", 2);
    template.resourceCountIs("AWS::SQS::Queue", 2);
    template.resourceCountIs("AWS::SNS::Topic", 1);
    template.hasResourceProperties("AWS::SNS::Subscription", {
        Protocol: "sqs",
    });
});
/*******************************
 *
 * Input tests - these tests check that invalid inputs are caught
 * before any processing is attempted. Having both SNS and SQS in the same
 * Construct means the associated properties have unique names in the construct to
 * avoid collisions, so the standard checks will not find these.
 *
 *******************************/
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    // Assertion
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Confirm that Construct checks for mixed deprecated and active props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            enableEncryptionWithCustomerManagedKey: true,
            encryptQueueWithCmk: true
        });
    };
    // Assertion
    expect(app).toThrowError(/Cannot specify both deprecated key props and new key props/);
});
test('Confirm that queueProps and existingQueue is caught', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Confirm that existingTopic and topicProps is caught', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    // Helper declaration
    const topic = new sns.Topic(stack, "existing-topic-obj", {
        topicName: 'existing-topic-obj'
    });
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicObj: topic,
            topicProps: {
                topicName: 'topic-name'
            },
            existingQueueObj: queue
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps or existingTopicObj, but not both.\n/);
});
test('Confirm that CheckSqsProps is called', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            },
            existingQueueObj: new sqs.Queue(stack, 'test', {})
        });
    };
    // Assertion
    expect(app).toThrowError("Error - Either provide queueProps or existingQueueObj, but not both.\n");
});
test('Catch queue key and queue key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingQueueEncryptionKey: new kms.Key(stack, 'test', {}),
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingQueueEncryptionKey or queueEncryptionKeyProps, but not both.\n/);
});
test('Catch queueProps key and construct props key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingQueueEncryptionKey: new kms.Key(stack, 'firstKey', {}),
            queueProps: {
                encryptionMasterKey: new kms.Key(stack, 'secondKey', {})
            }
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide queueProps.encryptionMasterKey or existingQueueEncryptionKey, but not both.\n/);
});
test('Catch queueProps key and construct props queue Key Props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                encryptionMasterKey: new kms.Key(stack, 'secondKey', {})
            },
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide queueProps.encryptionMasterKey or queueEncryptionKeyProps, but not both./);
});
test('Catch topic key and topic key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            existingTopicEncryptionKey: {},
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide existingTopicEncryptionKey or topicEncryptionKeyProps, but not both.\n/);
});
test('Catch topicProps key and construct props key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            topicProps: {
                masterKey: new kms.Key(stack, 'firstKey', {})
            },
            existingTopicEncryptionKey: new kms.Key(stack, 'secondKey', {})
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps.masterKey or existingTopicEncryptionKey, but not both.\n/);
});
test('Catch topicProps key and construct props Key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            topicProps: {
                masterKey: new kms.Key(stack, 'firstKey', {})
            },
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide topicProps.masterKey or topicEncryptionKeyProps, but not both.\n/);
});
test('Catch encryptTopicWithCmk false with topic key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptTopicWithCmk: false,
            topicEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptTopicWithCmk is false, submitting topicEncryptionKeyProps is invalid\n/);
});
test('Catch encryptTopicWithCmk false with topic key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptTopicWithCmk: false,
            existingTopicEncryptionKey: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptTopicWithCmk is false, submitting existingTopicEncryptionKey is invalid\n/);
});
test('Catch encryptQueueWithCmk false with queue key props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptQueueWithCmk: false,
            queueEncryptionKeyProps: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptQueueWithCmk is false, submitting queueEncryptionKeyProps is invalid\n/);
});
test('Catch encryptQueueWithCmk false with queue key', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            encryptQueueWithCmk: false,
            existingQueueEncryptionKey: {}
        });
    };
    // Assertion
    expect(app).toThrowError(/Error - if encryptQueueWithCmk is false, submitting existingQueueEncryptionKey is invalid\n/);
});
test('Catch queueProps.encryption on new interface', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const app = () => {
        new lib_1.SnsToSqs(stack, 'sns-to-sqs-stack', {
            queueProps: {
                encryption: sqs.QueueEncryption.UNENCRYPTED
            }
        });
    };
    // Assertion
    expect(app).toThrowError();
});
/*******************************
 * Tests for CreateRequiredKeys()
 *******************************/
test('test CreateRequiredKeys for no arguments', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {});
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingTopicObj: {}
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue is provided', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        existingQueueObj: {}
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue encryption is explicitly disabled', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        encryptQueueWithCmk: false
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic encryption is explicitly disabled', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        encryptTopicWithCmk: false
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Topic props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        topicProps: {
            masterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeTruthy();
    expect(result.encryptTopicWithCmk).toBeFalsy();
    expect(result.queueKey).toBeDefined();
    expect(result.topicKey).not.toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
test('test CreateRequiredKeys when Queue props have a key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    stack.node.setContext("@aws-cdk/aws-sns-subscriptions:restrictSqsDescryption", true);
    const result = lib_1.SnsToSqs.createRequiredKeys(stack, 'test', {
        queueProps: {
            encryptionMasterKey: {}
        }
    });
    expect(result.useDeprecatedInterface).toBeFalsy();
    expect(result.encryptQueueWithCmk).toBeFalsy();
    expect(result.encryptTopicWithCmk).toBeTruthy();
    expect(result.queueKey).not.toBeDefined();
    expect(result.topicKey).toBeDefined();
    expect(result.singleKey).not.toBeDefined();
});
//# sourceMappingURL=data:application/json;base64,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