"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lib_2 = require("../lib");
describe('IAM polocy document', () => {
    test('the Permission class is a programming model for iam', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addActions('sqs:SendMessage');
        p.addActions('dynamodb:CreateTable', 'dynamodb:DeleteTable');
        p.addResources('myQueue');
        p.addResources('yourQueue');
        p.addAllResources();
        p.addAwsAccountPrincipal(`my${core_1.Token.asString({ account: 'account' })}name`);
        p.addAccountCondition('12221121221');
        expect(stack.resolve(p.toStatementJson())).toEqual({ Action: ['sqs:SendMessage',
                'dynamodb:CreateTable',
                'dynamodb:DeleteTable'],
            Resource: ['myQueue', 'yourQueue', '*'],
            Effect: 'Allow',
            Principal: { AWS: { 'Fn::Join': ['',
                        ['arn:',
                            { Ref: 'AWS::Partition' },
                            ':iam::my',
                            { account: 'account' },
                            'name:root']] } },
            Condition: { StringEquals: { 'sts:ExternalId': '12221121221' } } });
    });
    test('the PolicyDocument class is a dom for iam policy documents', () => {
        const stack = new core_1.Stack();
        const doc = new lib_1.PolicyDocument();
        const p1 = new lib_1.PolicyStatement();
        p1.addActions('sqs:SendMessage');
        p1.addNotResources('arn:aws:sqs:us-east-1:123456789012:forbidden_queue');
        const p2 = new lib_1.PolicyStatement();
        p2.effect = lib_1.Effect.DENY;
        p2.addActions('cloudformation:CreateStack');
        const p3 = new lib_1.PolicyStatement();
        p3.effect = lib_1.Effect.ALLOW;
        p3.addNotActions('cloudformation:UpdateTerminationProtection');
        const p4 = new lib_1.PolicyStatement();
        p4.effect = lib_1.Effect.DENY;
        p4.addNotPrincipals(new lib_1.CanonicalUserPrincipal('OnlyAuthorizedUser'));
        doc.addStatements(p1);
        doc.addStatements(p2);
        doc.addStatements(p3);
        doc.addStatements(p4);
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [{ Effect: 'Allow', Action: 'sqs:SendMessage', NotResource: 'arn:aws:sqs:us-east-1:123456789012:forbidden_queue' },
                { Effect: 'Deny', Action: 'cloudformation:CreateStack' },
                { Effect: 'Allow', NotAction: 'cloudformation:UpdateTerminationProtection' },
                { Effect: 'Deny', NotPrincipal: { CanonicalUser: 'OnlyAuthorizedUser' } }]
        });
    });
    test('Cannot combine Actions and NotActions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['abc:def'],
                notActions: ['abc:def'],
            });
        }).toThrow(/Cannot add 'NotActions' to policy statement if 'Actions' have been added/);
    });
    test('Throws with invalid actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                actions: ['service:action', '*', 'service:acti*', 'in:val:id']
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Throws with invalid not actions', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                notActions: ['service:action', '*', 'service:acti*', 'in:val:id']
            });
        }).toThrow(/Action 'in:val:id' is invalid/);
    });
    test('Cannot combine Resources and NotResources', () => {
        expect(() => {
            new lib_1.PolicyStatement({
                resources: ['abc'],
                notResources: ['def'],
            });
        }).toThrow(/Cannot add 'NotResources' to policy statement if 'Resources' have been added/);
    });
    test('Cannot add NotPrincipals when Principals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            principals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addNotPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added/);
    });
    test('Cannot add Principals when NotPrincipals exist', () => {
        const stmt = new lib_1.PolicyStatement({
            notPrincipals: [new lib_1.CanonicalUserPrincipal('abc')],
        });
        expect(() => {
            stmt.addPrincipals(new lib_1.CanonicalUserPrincipal('def'));
        }).toThrow(/Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added/);
    });
    test('Permission allows specifying multiple actions upon construction', () => {
        const stack = new core_1.Stack();
        const perm = new lib_1.PolicyStatement();
        perm.addResources('MyResource');
        perm.addActions('Action1', 'Action2', 'Action3');
        expect(stack.resolve(perm.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['Action1', 'Action2', 'Action3'],
            Resource: 'MyResource'
        });
    });
    test('PolicyDoc resolves to undefined if there are no permissions', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        expect(stack.resolve(p)).toBeUndefined();
    });
    test('canonicalUserPrincipal adds a principal to a policy with the passed canonical user id', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        const canoncialUser = "averysuperduperlongstringfor";
        p.addPrincipals(new lib_1.CanonicalUserPrincipal(canoncialUser));
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: "Allow",
            Principal: {
                CanonicalUser: canoncialUser
            }
        });
    });
    test('addAccountRootPrincipal adds a principal with the current account root', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAccountRootPrincipal();
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: "Allow",
            Principal: {
                AWS: {
                    "Fn::Join": [
                        "",
                        [
                            "arn:",
                            { Ref: "AWS::Partition" },
                            ":iam::",
                            { Ref: "AWS::AccountId" },
                            ":root"
                        ]
                    ]
                }
            }
        });
    });
    test('addFederatedPrincipal adds a Federated principal with the passed value', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addFederatedPrincipal("com.amazon.cognito", { StringEquals: { key: 'value' } });
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: "Allow",
            Principal: {
                Federated: "com.amazon.cognito"
            },
            Condition: {
                StringEquals: { key: 'value' }
            }
        });
    });
    test('addAwsAccountPrincipal can be used multiple times', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyStatement();
        p.addAwsAccountPrincipal('1234');
        p.addAwsAccountPrincipal('5678');
        expect(stack.resolve(p.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::1234:root']] },
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::5678:root']] }
                ]
            }
        });
    });
    describe('hasResource', () => {
        test('false if there are no resources', () => {
            expect(new lib_1.PolicyStatement().hasResource).toEqual(false);
        });
        test('true if there is one resource', () => {
            expect(new lib_1.PolicyStatement({ resources: ['one-resource'] }).hasResource).toEqual(true);
        });
        test('true for multiple resources', () => {
            const p = new lib_1.PolicyStatement();
            p.addResources('r1');
            p.addResources('r2');
            expect(p.hasResource).toEqual(true);
        });
    });
    describe('hasPrincipal', () => {
        test('false if there is no principal', () => {
            expect(new lib_1.PolicyStatement().hasPrincipal).toEqual(false);
        });
        test('true if there is a principal', () => {
            const p = new lib_1.PolicyStatement();
            p.addArnPrincipal('bla');
            expect(p.hasPrincipal).toEqual(true);
        });
        test('true if there is a notPrincipal', () => {
            const p = new lib_1.PolicyStatement();
            p.addNotPrincipals(new lib_1.CanonicalUserPrincipal('test'));
            expect(p.hasPrincipal).toEqual(true);
        });
    });
    test('statementCount returns the number of statement in the policy document', () => {
        const p = new lib_1.PolicyDocument();
        expect(p.statementCount).toEqual(0);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'] }));
        expect(p.statementCount).toEqual(1);
        p.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'] }));
        expect(p.statementCount).toEqual(2);
    });
    describe('{ AWS: "*" } principal', () => {
        test('is represented as `Anyone`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.Anyone()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
        });
        test('is represented as `AnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            p.addStatements(new lib_1.PolicyStatement({ principals: [new lib_1.AnyPrincipal()] }));
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
        });
        test('is represented as `addAnyPrincipal`', () => {
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const s = new lib_1.PolicyStatement();
            s.addAnyPrincipal();
            p.addStatements(s);
            expect(stack.resolve(p)).toEqual({
                Statement: [
                    { Effect: 'Allow', Principal: '*' }
                ],
                Version: '2012-10-17'
            });
        });
    });
    test('addResources() will not break a list-encoded Token', () => {
        const stack = new core_1.Stack();
        const statement = new lib_1.PolicyStatement();
        statement.addActions(...core_1.Lazy.listValue({ produce: () => ['a', 'b', 'c'] }));
        statement.addResources(...core_1.Lazy.listValue({ produce: () => ['x', 'y', 'z'] }));
        expect(stack.resolve(statement.toStatementJson())).toEqual({
            Effect: 'Allow',
            Action: ['a', 'b', 'c'],
            Resource: ['x', 'y', 'z'],
        });
    });
    test('addCanonicalUserPrincipal can be used to add cannonical user principals', () => {
        const stack = new core_1.Stack();
        const p = new lib_1.PolicyDocument();
        const s1 = new lib_1.PolicyStatement();
        s1.addCanonicalUserPrincipal('cannonical-user-1');
        const s2 = new lib_1.PolicyStatement();
        s2.addPrincipals(new lib_1.CanonicalUserPrincipal('cannonical-user-2'));
        p.addStatements(s1);
        p.addStatements(s2);
        expect(stack.resolve(p)).toEqual({
            Statement: [
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-1' } },
                { Effect: 'Allow', Principal: { CanonicalUser: 'cannonical-user-2' } }
            ],
            Version: '2012-10-17'
        });
    });
    test('addPrincipal correctly merges array in', () => {
        const stack = new core_1.Stack();
        const arrayPrincipal = {
            get grantPrincipal() { return this; },
            assumeRoleAction: 'sts:AssumeRole',
            policyFragment: new lib_2.PrincipalPolicyFragment({ AWS: ['foo', 'bar'] }),
            addToPolicy() { return false; }
        };
        const s = new lib_1.PolicyStatement();
        s.addAccountRootPrincipal();
        s.addPrincipals(arrayPrincipal);
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Effect: 'Allow',
            Principal: {
                AWS: [
                    { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::', { Ref: 'AWS::AccountId' }, ':root']] },
                    'foo', 'bar'
                ]
            }
        });
    });
    // https://github.com/aws/aws-cdk/issues/1201
    test('policy statements with multiple principal types can be created using multiple addPrincipal calls', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addArnPrincipal('349494949494');
        s.addServicePrincipal('test.service');
        s.addResources('resource');
        s.addActions('action');
        expect(stack.resolve(s.toStatementJson())).toEqual({
            Action: 'action',
            Effect: 'Allow',
            Principal: { AWS: '349494949494', Service: 'test.service' },
            Resource: 'resource'
        });
    });
    describe('Service principals', () => {
        test('regional service principals resolve appropriately', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
        });
        test('regional service principals resolve appropriately (with user-set region)', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-northeast-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('codedeploy.amazonaws.com', { region: 'cn-north-1' });
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'codedeploy.cn-north-1.amazonaws.com.cn' }
            });
        });
        test('obscure service principals resolve to the user-provided value', () => {
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'cn-north-1' } });
            const s = new lib_1.PolicyStatement();
            s.addActions('test:Action');
            s.addServicePrincipal('test.service-principal.dev');
            expect(stack.resolve(s.toStatementJson())).toEqual({
                Effect: 'Allow',
                Action: 'test:Action',
                Principal: { Service: 'test.service-principal.dev' }
            });
        });
    });
    describe('CompositePrincipal can be used to represent a principal that has multiple types', () => {
        test('with a single principal', () => {
            const stack = new core_1.Stack();
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            expect(stack.resolve(statement.toStatementJson())).toEqual({ Effect: 'Allow', Principal: { AWS: 'i:am:an:arn' } });
        });
        test('conditions are not allowed on individual principals of a composite', () => {
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am'));
            expect(() => p.addPrincipals(new lib_2.FederatedPrincipal('federated', { condition: 1 })))
                .toThrow(/Components of a CompositePrincipal must not have conditions/);
        });
        test('principals and conditions are a big nice merge', () => {
            const stack = new core_1.Stack();
            // add via ctor
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('i:am:an:arn'), new lib_2.ServicePrincipal('amazon.com'));
            // add via `addPrincipals` (with condition)
            p.addPrincipals(new lib_1.Anyone(), new lib_2.ServicePrincipal('another.service'));
            const statement = new lib_1.PolicyStatement();
            statement.addPrincipals(p);
            // add via policy statement
            statement.addArnPrincipal('aws-principal-3');
            statement.addCondition('cond2', { boom: 123 });
            expect(stack.resolve(statement.toStatementJson())).toEqual({
                Condition: {
                    cond2: { boom: 123 }
                },
                Effect: 'Allow',
                Principal: {
                    AWS: ['i:am:an:arn', '*', 'aws-principal-3'],
                    Service: ['amazon.com', 'another.service'],
                }
            });
        });
        test('cannot mix types of assumeRoleAction in a single composite', () => {
            // GIVEN
            const p = new lib_2.CompositePrincipal(new lib_2.ArnPrincipal('arn')); // assumeRoleAction is "sts:AssumeRule"
            // THEN
            expect(() => p.addPrincipals(new lib_2.FederatedPrincipal('fed', {}, 'sts:Boom')))
                .toThrow(/Cannot add multiple principals with different "assumeRoleAction". Expecting "sts:AssumeRole", got "sts:Boom"/);
        });
    });
    describe('duplicate statements', () => {
        test('without tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement = new lib_1.PolicyStatement();
            statement.addResources('resource1', 'resource2');
            statement.addActions('action1', 'action2');
            statement.addServicePrincipal('service');
            statement.addConditions({
                a: {
                    b: 'c'
                },
                d: {
                    e: 'f'
                }
            });
            // WHEN
            p.addStatements(statement);
            p.addStatements(statement);
            p.addStatements(statement);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
        test('with tokens', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const p = new lib_1.PolicyDocument();
            const statement1 = new lib_1.PolicyStatement();
            statement1.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement1.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            const statement2 = new lib_1.PolicyStatement();
            statement2.addResources(core_1.Lazy.stringValue({ produce: () => 'resource' }));
            statement2.addActions(core_1.Lazy.stringValue({ produce: () => 'action' }));
            // WHEN
            p.addStatements(statement1);
            p.addStatements(statement2);
            // THEN
            expect(stack.resolve(p).Statement).toHaveLength(1);
        });
    });
    test('autoAssignSids enables auto-assignment of a unique SID for each statement', () => {
        // GIVEN
        const doc = new lib_1.PolicyDocument({
            assignSids: true
        });
        // WHEN
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action1'], resources: ['resource1'] }));
        doc.addStatements(new lib_1.PolicyStatement({ actions: ['service:action2'], resources: ['resource2'] }));
        // THEN
        const stack = new core_1.Stack();
        expect(stack.resolve(doc)).toEqual({
            Version: '2012-10-17',
            Statement: [
                { Action: 'service:action1', Effect: 'Allow', Resource: 'resource1', Sid: '0' },
                { Action: 'service:action2', Effect: 'Allow', Resource: 'resource2', Sid: '1' }
            ],
        });
    });
    test('constructor args are equivalent to mutating in-place', () => {
        const stack = new core_1.Stack();
        const s = new lib_1.PolicyStatement();
        s.addActions('service:action1', 'service:action2');
        s.addAllResources();
        s.addArnPrincipal('arn');
        s.addCondition('key', { equals: 'value' });
        const doc1 = new lib_1.PolicyDocument();
        doc1.addStatements(s);
        const doc2 = new lib_1.PolicyDocument();
        doc2.addStatements(new lib_1.PolicyStatement({
            actions: ['service:action1', 'service:action2'],
            resources: ['*'],
            principals: [new lib_2.ArnPrincipal('arn')],
            conditions: {
                key: { equals: 'value' }
            }
        }));
        expect(stack.resolve(doc1)).toEqual(stack.resolve(doc2));
    });
});
//# sourceMappingURL=data:application/json;base64,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