"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM role', () => {
    test('default role', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com')
        });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } } } });
    });
    test('a role can grant PassRole permissions', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.ServicePrincipal('henk.amazonaws.com') });
        const user = new lib_1.User(stack, 'User');
        // WHEN
        role.grantPassRole(user);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        });
    });
    test('can supply externalId', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalId: 'SomeSecret',
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply single externalIds', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalIds: ['SomeSecret'],
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply multiple externalIds', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalIds: ['SomeSecret', 'AnotherSecret'],
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": ["SomeSecret", "AnotherSecret"] }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('policy is created automatically when permissions are added', () => {
        // by default we don't expect a role policy
        const before = new core_1.Stack();
        new lib_1.Role(before, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        expect(before).not.toHaveResource('AWS::IAM::Policy');
        // add a policy to the role
        const after = new core_1.Stack();
        const afterRole = new lib_1.Role(after, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        afterRole.addToPolicy(new lib_1.PolicyStatement({ resources: ['myresource'], actions: ['service:myaction'] }));
        expect(after).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "service:myaction",
                        Effect: "Allow",
                        Resource: "myresource"
                    }
                ],
                Version: "2012-10-17"
            },
            PolicyName: "MyRoleDefaultPolicyA36BE1DD",
            Roles: [
                {
                    Ref: "MyRoleF48FFE04"
                }
            ]
        });
    });
    test('managed policy arns can be supplied upon initialization and also added later', () => {
        const stack = new core_1.Stack();
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
            managedPolicies: [{ managedPolicyArn: 'managed1' }, { managedPolicyArn: 'managed2' }]
        });
        role.addManagedPolicy({ managedPolicyArn: 'managed3' });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: ['managed1', 'managed2', 'managed3'] } } } });
    });
    test('federated principal can change AssumeRoleAction', () => {
        const stack = new core_1.Stack();
        const cognitoPrincipal = new lib_1.FederatedPrincipal('foo', { StringEquals: { key: 'value' } }, 'sts:AssumeSomething');
        new lib_1.Role(stack, 'MyRole', { assumedBy: cognitoPrincipal });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Version: "2012-10-17",
                Statement: [
                    {
                        Principal: { Federated: "foo" },
                        Condition: {
                            StringEquals: { key: "value" }
                        },
                        Action: "sts:AssumeSomething",
                        Effect: "Allow",
                    }
                ],
            }
        });
    });
    describe('maxSessionDuration', () => {
        test('is not specified by default', () => {
            const stack = new core_1.Stack();
            new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            expect(stack).toMatchTemplate({
                Resources: {
                    MyRoleF48FFE04: {
                        Type: "AWS::IAM::Role",
                        Properties: {
                            AssumeRolePolicyDocument: {
                                Statement: [
                                    {
                                        Action: "sts:AssumeRole",
                                        Effect: "Allow",
                                        Principal: {
                                            Service: "sns.amazonaws.com"
                                        }
                                    }
                                ],
                                Version: "2012-10-17"
                            }
                        }
                    }
                }
            });
        });
        test('can be used to specify the maximum session duration for assuming the role', () => {
            const stack = new core_1.Stack();
            new lib_1.Role(stack, 'MyRole', { maxSessionDuration: core_1.Duration.seconds(3700), assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            expect(stack).toHaveResource('AWS::IAM::Role', {
                MaxSessionDuration: 3700
            });
        });
        test('must be between 3600 and 43200', () => {
            const stack = new core_1.Stack();
            const assumedBy = new lib_1.ServicePrincipal('bla');
            new lib_1.Role(stack, 'MyRole1', { assumedBy, maxSessionDuration: core_1.Duration.hours(1) });
            new lib_1.Role(stack, 'MyRole2', { assumedBy, maxSessionDuration: core_1.Duration.hours(12) });
            const expected = (val) => `maxSessionDuration is set to ${val}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`;
            expect(() => new lib_1.Role(stack, 'MyRole3', { assumedBy, maxSessionDuration: core_1.Duration.minutes(1) }))
                .toThrow(expected(60));
            expect(() => new lib_1.Role(stack, 'MyRole4', { assumedBy, maxSessionDuration: core_1.Duration.seconds(3599) }))
                .toThrow(expected(3599));
            expect(() => new lib_1.Role(stack, 'MyRole5', { assumedBy, maxSessionDuration: core_1.Duration.seconds(43201) }))
                .toThrow(expected(43201));
        });
    });
    test('allow role with multiple principals', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.CompositePrincipal(new lib_1.ServicePrincipal('boom.amazonaws.test'), new lib_1.ArnPrincipal('1111111'))
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "boom.amazonaws.test",
                            AWS: "1111111"
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply permissions boundary managed policy', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const permissionsBoundary = lib_1.ManagedPolicy.fromAwsManagedPolicyName('managed-policy');
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            permissionsBoundary,
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            PermissionsBoundary: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/managed-policy"
                    ]
                ]
            }
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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