"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')
            ]
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*']
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs10.x',
                Timeout: 300,
            }
        });
        resource.node.addDependency(role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = core_1.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    const s3 = new (require('aws-sdk').S3)();
    const https = require("https");
    const url = require("url");
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse("FAILED", err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse("SUCCESS");
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || "See the details in CloudWatch Log Stream: " + context.logStreamName,
            PhysicalResourceId: event.PhysicalResourceId || event.LogicalResourceId,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "content-type": "",
                "content-length": responseBody.length
            }
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on("error", (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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