"""

--------------------

```../examples/bpy.types.RenderEngine.py```


--------------------

PropertyGroups are the base class for dynamically defined sets of properties.

They can be used to extend existing blender data with your own types which can
be animated, accessed from the user interface and from python.

[NOTE]
The values assigned to blender data are saved to disk but the class
definitions are not, this means whenever you load blender the class needs
to be registered too.
This is best done by creating an addon which loads on startup and registers
your properties.

[NOTE]
PropertyGroups must be registered before assigning them to blender data.

Property types used in class declarations are all in bpy.props



```../examples/bpy.types.PropertyGroup.py```


--------------------

This script demonstrates basic operations on object like creating new
object, placing it into scene, selecting it and making it active.

```../examples/bpy.types.Object.py```


--------------------

Here is an example of a simple menu. Menus differ from panels in that they must
reference from a header, panel or another menu.

Notice the 'CATEGORY_MT_name' in  Menu.bl_idname, this is a naming
convention for menus.

[NOTE]
Menu subclasses must be registered before referencing them from blender.

[NOTE]
Menus have their Layout.operator_context initialized as
'EXEC_REGION_WIN' rather than 'INVOKE_DEFAULT' (see Execution Context <operator-execution_context>).
If the operator context needs to initialize inputs from the
Operator.invoke function, then this needs to be explicitly set.

```../examples/bpy.types.Menu.py```


--------------------

This menu demonstrates some different functions.

```../examples/bpy.types.Menu.1.py```


--------------------

When creating menus for addons you can't reference menus in Blender's default
scripts.
Instead, the addon can add menu items to existing menus.

The function menu_draw acts like Menu.draw.

```../examples/bpy.types.Menu.2.py```


--------------------

Preset menus are simply a convention that uses a menu sub-class
to perform the common task of managing presets.

This example shows how you can add a preset menu.

This example uses the object draw options,
however you can use properties defined by your own scripts too.

```../examples/bpy.types.Menu.3.py```

```../examples/bpy.types.AddonPreferences.1.py```


--------------------

This script is a simple panel which will draw into the object properties
section.

Notice the 'CATEGORY_PT_name' Panel.bl_idname, this is a naming
convention for panels.

[NOTE]
Panel subclasses must be registered for blender to use them.

```../examples/bpy.types.Panel.py```


--------------------

This panel has a Panel.poll and Panel.draw_header function,
even though the contents is basic this closely resembles blenders panels.

```../examples/bpy.types.Panel.1.py```


--------------------

A mix-in parent class can be used to share common properties and
Menu.poll function.

```../examples/bpy.types.Panel.2.py```


--------------------

The NodeTree.poll function determines if a node tree is visible
in the given context (similar to how Panel.poll
and Menu.poll define visibility). If it returns False,
the node tree type will not be selectable in the node editor.

A typical condition for shader nodes would be to check the active render engine
of the scene and only show nodes of the renderer they are designed for.

```../examples/bpy.types.NodeTree.py```


--------------------

This script shows simple operator which prints a message.

Since the operator only has an Operator.execute function it takes no
user input.

[NOTE]
Operator subclasses must be registered before accessing them from blender.

```../examples/bpy.types.Operator.py```


--------------------

Operator.invoke is used to initialize the operator from the context
at the moment the operator is called.
invoke() is typically used to assign properties which are then used by
execute().
Some operators don't have an execute() function, removing the ability to be
repeated from a script or macro.

This example shows how to define an operator which gets mouse input to
execute a function and that this operator can be invoked or executed from
the python api.

Also notice this operator defines its own properties, these are different
to typical class properties because blender registers them with the
operator, to use as arguments when called, saved for operator undo/redo and
automatically added into the user interface.

```../examples/bpy.types.Operator.1.py```


--------------------

This example shows how an operator can use the file selector.

Notice the invoke function calls a window manager method and returns
{'RUNNING_MODAL'}

, this means the file selector stays open and the operator does not
exit immediately after invoke finishes.

The file selector runs the operator, calling Operator.execute when the
user confirms.

The Operator.poll function is optional, used to check if the operator
can run.

```../examples/bpy.types.Operator.2.py```


--------------------

This operator uses its Operator.invoke function to call a popup.

```../examples/bpy.types.Operator.3.py```


--------------------

By default operator properties use an automatic user interface layout.
If you need more control you can create your own layout with a
Operator.draw function.

This works like the Panel and Menu draw functions, its used
for dialogs and file selectors.

```../examples/bpy.types.Operator.4.py```


--------------------

This operator defines a Operator.modal function which running,
handling events until it returns {'FINISHED'}

 or {'CANCELLED'}

.

Grab, Rotate, Scale and Fly-Mode are examples of modal operators.
They are especially useful for interactive tools,
your operator can have its own state where keys toggle options as the operator
runs.

Operator.invoke is used to initialize the operator as being by
returning {'RUNNING_MODAL'}

, initializing the modal loop.

Notice __init__()

 and __del__()

 are declared.
For other operator types they are not useful but for modal operators they will
be called before the Operator.invoke and after the operator finishes.

```../examples/bpy.types.Operator.5.py```


--------------------

This script is the UIList subclass used to show material slots, with a bunch of additional commentaries.

Notice the name of the class, this naming convention is similar as the one for panels or menus.

[NOTE]
UIList subclasses must be registered for blender to use them.

```../examples/bpy.types.UIList.1.py```


--------------------

This script is an extended version of the UIList

 subclass used to show vertex groups. It is not used 'as is',
because iterating over all vertices in a 'draw' function is a very bad idea for UI performances! However, it's a good
example of how to create/use filtering/reordering callbacks.

```../examples/bpy.types.UIList.2.py```


--------------------

The mesh data is accessed in object mode and intended for compact storage,
for more flexible mesh editing from python see bmesh.

Blender stores 4 main arrays to define mesh geometry.

* Mesh.vertices (3 points in space)
* Mesh.edges (reference 2 vertices)
* Mesh.loops (reference a single vertex and edge)
* Mesh.polygons: (reference a range of loops)

Each polygon reference a slice in the loop array, this way, polygons do not store vertices or corner data such as UV's directly,
only a reference to loops that the polygon uses.

Mesh.loops, Mesh.uv_layers Mesh.vertex_colors are all aligned so the same polygon loop
indices can be used to find the UV's and vertex colors as with as the vertices.

To compare mesh API options see: NGons and Tessellation Faces <info_gotcha_mesh_faces>

This example script prints the vertices and UV's for each polygon, assumes the active object is a mesh with UVs.

```../examples/bpy.types.Mesh.py```

"""

import typing
import collections.abc
import typing_extensions
import bl_operators.anim
import bl_operators.clip
import bl_operators.file
import bl_operators.freestyle
import bl_operators.mask
import bl_operators.node
import bl_operators.presets
import bl_operators.view3d
import bl_operators.wm
import bl_ui
import bl_ui.properties_constraint
import bl_ui.properties_data_armature
import bl_ui.properties_data_bone
import bl_ui.properties_data_camera
import bl_ui.properties_data_curve
import bl_ui.properties_data_empty
import bl_ui.properties_data_lamp
import bl_ui.properties_data_lattice
import bl_ui.properties_data_mesh
import bl_ui.properties_data_metaball
import bl_ui.properties_data_modifier
import bl_ui.properties_data_speaker
import bl_ui.properties_freestyle
import bl_ui.properties_game
import bl_ui.properties_grease_pencil_common
import bl_ui.properties_mask_common
import bl_ui.properties_material
import bl_ui.properties_object
import bl_ui.properties_paint_common
import bl_ui.properties_particle
import bl_ui.properties_physics_cloth
import bl_ui.properties_physics_common
import bl_ui.properties_physics_dynamicpaint
import bl_ui.properties_physics_field
import bl_ui.properties_physics_fluid
import bl_ui.properties_physics_rigidbody
import bl_ui.properties_physics_rigidbody_constraint
import bl_ui.properties_physics_smoke
import bl_ui.properties_physics_softbody
import bl_ui.properties_render
import bl_ui.properties_render_layer
import bl_ui.properties_scene
import bl_ui.properties_texture
import bl_ui.properties_world
import bl_ui.space_clip
import bl_ui.space_console
import bl_ui.space_dopesheet
import bl_ui.space_filebrowser
import bl_ui.space_graph
import bl_ui.space_image
import bl_ui.space_info
import bl_ui.space_logic
import bl_ui.space_nla
import bl_ui.space_node
import bl_ui.space_outliner
import bl_ui.space_properties
import bl_ui.space_sequencer
import bl_ui.space_text
import bl_ui.space_time
import bl_ui.space_userpref
import bl_ui.space_view3d
import bl_ui.space_view3d_toolbar
import keyingsets_builtins
import mathutils

class bpy_prop_collection[_GenericType1]:
    """built-in class used for all collections."""

    def find(self, key: str) -> int:
        """Returns the index of a key in a collection or -1 when not found
        (matches pythons string find function of the same name).

                :param key: The identifier for the collection member.
                :type key: str
                :return: index of the key.
                :rtype: int
        """

    def foreach_get(
        self,
        attr: str,
        seq: collections.abc.MutableSequence[bool]
        | collections.abc.MutableSequence[int]
        | collections.abc.MutableSequence[float]
        | typing_extensions.Buffer,
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        Multi-dimensional arrays (like array of vectors) will be flattened into seq.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.MutableSequence[bool] | collections.abc.MutableSequence[int] | collections.abc.MutableSequence[float] | typing_extensions.Buffer
        """

    def foreach_set(
        self,
        attr: str,
        seq: collections.abc.Sequence[bool]
        | collections.abc.Sequence[int]
        | collections.abc.Sequence[float]
        | typing_extensions.Buffer,
    ):
        """This is a function to give fast access to attributes within a collection.Only works for 'basic type' properties (bool, int and float)!
        seq must be uni-dimensional, multi-dimensional arrays (like array of vectors) will be re-created from it.

                :param attr:
                :type attr: str
                :param seq:
                :type seq: collections.abc.Sequence[bool] | collections.abc.Sequence[int] | collections.abc.Sequence[float] | typing_extensions.Buffer
        """

    def get[_GenericType2](
        self, key: str, default: _GenericType2 = None
    ) -> _GenericType1 | _GenericType2:
        """Returns the value of the item assigned to key or default when not found
        (matches pythons dictionary function of the same name).

                :param key: The identifier for the collection member.
                :type key: str
                :param default: Optional argument for the value to return if
        key is not found.
                :type default: _GenericType2
                :return:
                :rtype: _GenericType1 | _GenericType2
        """

    def items(self) -> list[tuple[str, _GenericType1]]:
        """Return the identifiers of collection members
        (matching pythons dict.items() functionality).

                :return: (key, value) pairs for each member of this collection.
                :rtype: list[tuple[str, _GenericType1]]
        """

    def keys(self) -> list[str]:
        """Return the identifiers of collection members
        (matching pythons dict.keys() functionality).

                :return: the identifiers for each member of this collection.
                :rtype: list[str]
        """

    def values(self) -> list[_GenericType1 | None]:
        """Return the values of collection
        (matching pythons dict.values() functionality).

                :return: the members of this collection.
                :rtype: list[_GenericType1 | None]
        """

    @typing.overload
    def __getitem__(self, key: int | str) -> _GenericType1:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: _GenericType1
        """

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[_GenericType1, ...]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[_GenericType1, ...]
        """

    def __setitem__(self, key: int | str, value: GenericType1 | None):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: GenericType1 | None
        """

    @typing.overload
    def __setitem__(self, key: int, value: _GenericType1 | None):
        """

        :param key:
        :type key: int
        :param value:
        :type value: _GenericType1 | None
        """

    @typing.overload
    def __setitem__(self, key: str, value: _GenericType1):
        """

        :param key:
        :type key: str
        :param value:
        :type value: _GenericType1
        """

    def __iter__(self) -> collections.abc.Iterator[_GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[_GenericType1]
        """

    def __next__(self) -> _GenericType1:
        """

        :return:
        :rtype: _GenericType1
        """

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """

    def __contains__(self, key: str | tuple[str, ...] | _GenericType1) -> bool:
        """

        :param key:
        :type key: str | tuple[str, ...] | _GenericType1
        :return:
        :rtype: bool
        """

class bpy_struct[_GenericType1]:
    """built-in base class for all classes in bpy.types."""

    id_data: typing.Any
    """ The `bpy.types.ID` object this datablock is from or None, (not available for all data types)"""

    bl_rna: BlenderRNA
    """ 

    :type: BlenderRNA
    """

    def as_pointer(self) -> int:
        """Returns the memory address which holds a pointer to blenders internal data

        :return: int (memory address).
        :rtype: int
        """

    def driver_add(self, path: str | None, index: int | None = -1) -> FCurve:
        """Adds driver(s) to the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: The driver(s) added.
        :rtype: FCurve
        """

    def driver_remove(self, path: str | None, index: int | None = -1) -> bool:
        """Remove driver(s) from the given property

        :param path: path to the property to drive, analogous to the fcurve's data path.
        :type path: str | None
        :param index: array index of the property drive. Defaults to -1 for all indices or a single channel if the property is not an array.
        :type index: int | None
        :return: Success of driver removal.
        :rtype: bool
        """

    def get(self, key: str | None, default=None):
        """Returns the value of the custom property assigned to key or default
        when not found (matches pythons dictionary function of the same name).

                :param key: The key associated with the custom property.
                :type key: str | None
                :param default: Optional argument for the value to return if
        key is not found.
        """

    def is_property_hidden(self, property) -> bool:
        """Check if a property is hidden.

        :param property:
        :return: True when the property is hidden.
        :rtype: bool
        """

    def is_property_readonly(self, property) -> bool:
        """Check if a property is readonly.

        :param property:
        :return: True when the property is readonly (not writable).
        :rtype: bool
        """

    def is_property_set(self, property) -> bool:
        """Check if a property is set, use for testing operator properties.

        :param property:
        :return: True when the property has been set.
        :rtype: bool
        """

    def items(self):
        """Returns the items of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property key, value pairs.
        """

    def keyframe_delete(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = bpy.context.scene.frame_current,
        group: str | None = "",
    ) -> bool:
        """Remove a keyframe from this properties fcurve.

        :param data_path: path to the property to remove a key, analogous to the fcurve's data path.
        :type data_path: str | None
        :param index: array index of the property to remove a key. Defaults to -1 removing all indices or a single channel if the property is not an array.
        :type index: int | None
        :param frame: The frame on which the keyframe is deleted, defaulting to the current frame.
        :type frame: float | None
        :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
        :type group: str | None
        :return: Success of keyframe deleation.
        :rtype: bool
        """

    def keyframe_insert(
        self,
        data_path: str | None,
        index: int | None = -1,
        frame: float | None = bpy.context.scene.frame_current,
        group: str | None = "",
    ) -> bool:
        """Insert a keyframe on the property given, adding fcurves and animation data when necessary.This is the most simple example of inserting a keyframe from python.Note that when keying data paths which contain nested properties this must be
        done from the `ID` subclass, in this case the `Armature` rather
        than the bone.

                :param data_path: path to the property to key, analogous to the fcurve's data path.
                :type data_path: str | None
                :param index: array index of the property to key.
        Defaults to -1 which will key all indices or a single channel if the property is not an array.
                :type index: int | None
                :param frame: The frame on which the keyframe is inserted, defaulting to the current frame.
                :type frame: float | None
                :param group: The name of the group the F-Curve should be added to if it doesn't exist yet.
                :type group: str | None
                :return: Success of keyframe insertion.
                :rtype: bool
        """

    def keys(self) -> list[str]:
        """Returns the keys of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property keys.
                :rtype: list[str]
        """

    def path_from_id(self, property: str | None = "") -> str:
        """Returns the data path from the ID to this object (string).

                :param property: Optional property name which can be used if the path is
        to a property of this object.
                :type property: str | None
                :return: The path from `bpy.types.bpy_struct.id_data`
        to this struct and property (when given).
                :rtype: str
        """

    def path_resolve(self, path: str | None, coerce: bool | None = True):
        """Returns the property from the path, raise an exception when not found.

                :param path: path which this property resolves.
                :type path: str | None
                :param coerce: optional argument, when True, the property will be converted
        into its python representation.
                :type coerce: bool | None
        """

    def property_unset(self, property):
        """Unset a property, will use default value afterward.

        :param property:
        """

    def type_recast(self):
        """Return a new instance, this is needed because types
        such as textures can be changed at runtime.

                :return: a new instance of this object with the type initialized again.
        """

    def values(self) -> list:
        """Returns the values of this objects custom properties (matches pythons
        dictionary function of the same name).

                :return: custom property values.
                :rtype: list
        """

    def __getitem__(self, key: int | str) -> typing.Any:
        """

        :param key:
        :type key: int | str
        :return:
        :rtype: typing.Any
        """

    def __setitem__(self, key: int | str, value: typing.Any):
        """

        :param key:
        :type key: int | str
        :param value:
        :type value: typing.Any
        """

    def __delitem__(self, key: int | str):
        """

        :param key:
        :type key: int | str
        """

class bpy_prop_array[_GenericType1]:
    def __get__(self, instance, owner) -> bpy_prop_array[_GenericType1]:
        """

        :param instance:
        :param owner:
        :return:
        :rtype: bpy_prop_array[_GenericType1]
        """

    def __set__(self, instance, value: collections.abc.Iterable[_GenericType1]):
        """

        :param instance:
        :param value:
        :type value: collections.abc.Iterable[_GenericType1]
        """

    def foreach_get(
        self,
        seq: collections.abc.MutableSequence[_GenericType1] | typing_extensions.Buffer,
    ):
        """

        :param seq:
        :type seq: collections.abc.MutableSequence[_GenericType1] | typing_extensions.Buffer
        """

    def foreach_set(
        self, seq: collections.abc.Sequence[_GenericType1] | typing_extensions.Buffer
    ):
        """

        :param seq:
        :type seq: collections.abc.Sequence[_GenericType1] | typing_extensions.Buffer
        """

    @typing.overload
    def __getitem__(self, key: int) -> _GenericType1:
        """

        :param key:
        :type key: int
        :return:
        :rtype: _GenericType1
        """

    @typing.overload
    def __getitem__(self, key: slice) -> tuple[_GenericType1, ...]:
        """

        :param key:
        :type key: slice
        :return:
        :rtype: tuple[_GenericType1, ...]
        """

    @typing.overload
    def __setitem__(self, key: int, value: _GenericType1):
        """

        :param key:
        :type key: int
        :param value:
        :type value: _GenericType1
        """

    @typing.overload
    def __setitem__(self, key: slice, value: collections.abc.Iterable[_GenericType1]):
        """

        :param key:
        :type key: slice
        :param value:
        :type value: collections.abc.Iterable[_GenericType1]
        """

    def __delitem__(self, key: int):
        """

        :param key:
        :type key: int
        """

    def __iter__(self) -> collections.abc.Iterator[_GenericType1]:
        """

        :return:
        :rtype: collections.abc.Iterator[_GenericType1]
        """

    def __next__(self) -> _GenericType1:
        """

        :return:
        :rtype: _GenericType1
        """

    def __len__(self) -> int:
        """

        :return:
        :rtype: int
        """

class ActionFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of action F-Curves"""

    def new(
        self, data_path: str, index: int | None = 0, action_group: str = ""
    ) -> FCurve:
        """Add an F-Curve to the action

        :param data_path: Data Path, F-Curve data path to use
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :param action_group: Action Group, Acton group to add this F-Curve into
        :type action_group: str
        :return: Newly created F-Curve
        :rtype: FCurve
        """

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the action.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

    def remove(self, fcurve: FCurve):
        """Remove action group

        :param fcurve: F-Curve to remove
        :type fcurve: FCurve
        """

class ActionGroups(bpy_prop_collection[ActionGroup], bpy_struct):
    """Collection of action groups"""

    def new(self, name: str) -> ActionGroup:
        """Create a new action group and add it to the action

        :param name: New name for the action group
        :type name: str
        :return: Newly created action group
        :rtype: ActionGroup
        """

    def remove(self, action_group: ActionGroup):
        """Remove action group

        :param action_group: Action group to remove
        :type action_group: ActionGroup
        """

class ActionPoseMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    active: TimelineMarker | None
    """ Active pose marker for this action

    :type: TimelineMarker | None
    """

    active_index: int | None
    """ Index of active pose marker

    :type: int | None
    """

    def new(self, name: str) -> TimelineMarker:
        """Add a pose marker to the action

        :param name: New name for the marker (not unique)
        :type name: str
        :return: Newly created marker
        :rtype: TimelineMarker
        """

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """

class Addons(bpy_prop_collection[Addon], bpy_struct):
    """Collection of add-ons"""

    @classmethod
    def new(cls) -> Addon:
        """Add a new add-on

        :return: Add-on data
        :rtype: Addon
        """

    @classmethod
    def remove(cls, addon: Addon):
        """Remove add-on

        :param addon: Add-on to remove
        :type addon: Addon
        """

class AlembicObjectPaths(bpy_prop_collection[AlembicObjectPath], bpy_struct):
    """Collection of object paths"""

class AnimDataDrivers(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of Driver F-Curves"""

    def from_existing(self, src_driver: FCurve | None = None) -> FCurve:
        """Add a new driver given an existing one

        :param src_driver: Existing Driver F-Curve to use as template for a new one
        :type src_driver: FCurve | None
        :return: New Driver F-Curve
        :rtype: FCurve
        """

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find a driver F-Curve. Note that this function performs a linear scan of all driver F-Curves.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

class AreaSpaces(bpy_prop_collection[Space], bpy_struct):
    """Collection of spaces"""

    active: Space | None
    """ Space currently being displayed in this area

    :type: Space | None
    """

class ArmatureBones(bpy_prop_collection[Bone], bpy_struct):
    """Collection of armature bones"""

    active: Bone | None
    """ Armature's active bone

    :type: Bone | None
    """

class ArmatureEditBones(bpy_prop_collection[EditBone], bpy_struct):
    """Collection of armature edit bones"""

    active: EditBone | None
    """ Armatures active edit bone

    :type: EditBone | None
    """

    def new(self, name: str) -> EditBone:
        """Add a new bone

        :param name: New name for the bone
        :type name: str
        :return: Newly created edit bone
        :rtype: EditBone
        """

    def remove(self, bone: EditBone):
        """Remove an existing bone from the armature

        :param bone: EditBone to remove
        :type bone: EditBone
        """

class BackgroundImages(bpy_prop_collection[BackgroundImage], bpy_struct):
    """Collection of background images"""

    def new(self) -> BackgroundImage:
        """Add new background image

        :return: Image displayed as viewport background
        :rtype: BackgroundImage
        """

    def remove(self, image: BackgroundImage):
        """Remove background image

        :param image: Image displayed as viewport background
        :type image: BackgroundImage
        """

    def clear(self):
        """Remove all background images"""

class BlendDataActions(bpy_prop_collection[Action], bpy_struct):
    """Collection of actions"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Action:
        """Add a new action to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New action data-block
        :rtype: Action
        """

    def remove(self, action: Action, do_unlink: bool | None = False):
        """Remove a action from the current blendfile

        :param action: Action to remove
        :type action: Action
        :param do_unlink: Unlink all usages of this action before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataArmatures(bpy_prop_collection[Armature], bpy_struct):
    """Collection of armatures"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Armature:
        """Add a new armature to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New armature data-block
        :rtype: Armature
        """

    def remove(self, armature: Armature, do_unlink: bool | None = False):
        """Remove a armature from the current blendfile

        :param armature: Armature to remove
        :type armature: Armature
        :param do_unlink: Unlink all usages of this armature before deleting it (WARNING: will also delete objects instancing that armature data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataBrushes(bpy_prop_collection[Brush], bpy_struct):
    """Collection of brushes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self,
        name: str,
        mode: typing.Literal[
            "OBJECT",
            "EDIT",
            "POSE",
            "SCULPT",
            "VERTEX_PAINT",
            "WEIGHT_PAINT",
            "TEXTURE_PAINT",
            "PARTICLE_EDIT",
            "GPENCIL_EDIT",
        ]
        | None = "TEXTURE_PAINT",
    ) -> Brush:
        """Add a new brush to the main database

                :param name: New name for the data-block
                :type name: str
                :param mode: Paint Mode for the new brush

        OBJECT Object Mode.

        EDIT Edit Mode.

        POSE Pose Mode.

        SCULPT Sculpt Mode.

        VERTEX_PAINT Vertex Paint.

        WEIGHT_PAINT Weight Paint.

        TEXTURE_PAINT Texture Paint.

        PARTICLE_EDIT Particle Edit.

        GPENCIL_EDIT Edit Strokes, Edit Grease Pencil Strokes.
                :type mode: typing.Literal['OBJECT','EDIT','POSE','SCULPT','VERTEX_PAINT','WEIGHT_PAINT','TEXTURE_PAINT','PARTICLE_EDIT','GPENCIL_EDIT'] | None
                :return: New brush data-block
                :rtype: Brush
        """

    def remove(self, brush: Brush, do_unlink: bool | None = False):
        """Remove a brush from the current blendfile

        :param brush: Brush to remove
        :type brush: Brush
        :param do_unlink: Unlink all usages of this brush before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataCacheFiles(bpy_prop_collection[CacheFile], bpy_struct):
    """Collection of cache files"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataCameras(bpy_prop_collection[Camera], bpy_struct):
    """Collection of cameras"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Camera:
        """Add a new camera to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New camera data-block
        :rtype: Camera
        """

    def remove(self, camera: Camera, do_unlink: bool | None = False):
        """Remove a camera from the current blendfile

        :param camera: Camera to remove
        :type camera: Camera
        :param do_unlink: Unlink all usages of this camera before deleting it (WARNING: will also delete objects instancing that camera data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataCurves(bpy_prop_collection[Curve], bpy_struct):
    """Collection of curves"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self, name: str, type: typing.Literal["CURVE", "SURFACE", "FONT"] | None
    ) -> Curve:
        """Add a new curve to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of curve to add
        :type type: typing.Literal['CURVE','SURFACE','FONT'] | None
        :return: New curve data-block
        :rtype: Curve
        """

    def remove(self, curve: Curve, do_unlink: bool | None = False):
        """Remove a curve from the current blendfile

        :param curve: Curve to remove
        :type curve: Curve
        :param do_unlink: Unlink all usages of this curve before deleting it (WARNING: will also delete objects instancing that curve data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataFonts(bpy_prop_collection[VectorFont], bpy_struct):
    """Collection of fonts"""

    is_updated: bool
    """ 

    :type: bool
    """

    def load(self, filepath: str, check_existing: bool | None = False) -> VectorFont:
        """Load a new font into the main database

        :param filepath: path of the font to load
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New font data-block
        :rtype: VectorFont
        """

    def remove(self, vfont: VectorFont, do_unlink: bool | None = False):
        """Remove a font from the current blendfile

        :param vfont: Font to remove
        :type vfont: VectorFont
        :param do_unlink: Unlink all usages of this font before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataGreasePencils(bpy_prop_collection[GreasePencil], bpy_struct):
    """Collection of grease pencils"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    @classmethod
    def new(cls, name: str) -> GreasePencil:
        """new

        :param name: New name for the data-block
        :type name: str
        :return: New grease pencil data-block
        :rtype: GreasePencil
        """

    def remove(self, grease_pencil: GreasePencil, do_unlink: bool | None = False):
        """Remove a grease pencil instance from the current blendfile

        :param grease_pencil: Grease Pencil to remove
        :type grease_pencil: GreasePencil
        :param do_unlink: Unlink all usages of this grease pencil before deleting it
        :type do_unlink: bool | None
        """

class BlendDataGroups(bpy_prop_collection[Group], bpy_struct):
    """Collection of groups"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Group:
        """Add a new group to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New group data-block
        :rtype: Group
        """

    def remove(self, group: Group, do_unlink: bool | None = False):
        """Remove a group from the current blendfile

        :param group: Group to remove
        :type group: Group
        :param do_unlink: Unlink all usages of this group before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataImages(bpy_prop_collection[Image], bpy_struct):
    """Collection of images"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self,
        name: str,
        width: int | None,
        height: int | None,
        alpha: bool | None = False,
        float_buffer: bool | None = False,
        stereo3d: bool | None = False,
    ) -> Image:
        """Add a new image to the main database

        :param name: New name for the data-block
        :type name: str
        :param width: Width of the image
        :type width: int | None
        :param height: Height of the image
        :type height: int | None
        :param alpha: Alpha, Use alpha channel
        :type alpha: bool | None
        :param float_buffer: Float Buffer, Create an image with floating point color
        :type float_buffer: bool | None
        :param stereo3d: Stereo 3D, Create left and right views
        :type stereo3d: bool | None
        :return: New image data-block
        :rtype: Image
        """

    def load(self, filepath: str, check_existing: bool | None = False) -> Image:
        """Load a new image into the main database

        :param filepath: path of the file to load
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New image data-block
        :rtype: Image
        """

    def remove(self, image: Image, do_unlink: bool | None = False):
        """Remove an image from the current blendfile

        :param image: Image to remove
        :type image: Image
        :param do_unlink: Unlink all usages of this image before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataLamps(bpy_prop_collection[Lamp], bpy_struct):
    """Collection of lamps"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self,
        name: str,
        type: typing.Literal["POINT", "SUN", "SPOT", "HEMI", "AREA"] | None,
    ) -> Lamp:
        """Add a new lamp to the main database

                :param name: New name for the data-block
                :type name: str
                :param type: Type, The type of texture to add

        POINT Point, Omnidirectional point light source.

        SUN Sun, Constant direction parallel ray light source.

        SPOT Spot, Directional cone light source.

        HEMI Hemi, 180 degree constant light source.

        AREA Area, Directional area light source.
                :type type: typing.Literal['POINT','SUN','SPOT','HEMI','AREA'] | None
                :return: New lamp data-block
                :rtype: Lamp
        """

    def remove(self, lamp: Lamp, do_unlink: bool | None = False):
        """Remove a lamp from the current blendfile

        :param lamp: Lamp to remove
        :type lamp: Lamp
        :param do_unlink: Unlink all usages of this lamp before deleting it (WARNING: will also delete objects instancing that lamp data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataLattices(bpy_prop_collection[Lattice], bpy_struct):
    """Collection of lattices"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Lattice:
        """Add a new lattice to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New lattices data-block
        :rtype: Lattice
        """

    def remove(self, lattice: Lattice, do_unlink: bool | None = False):
        """Remove a lattice from the current blendfile

        :param lattice: Lattice to remove
        :type lattice: Lattice
        :param do_unlink: Unlink all usages of this lattice before deleting it (WARNING: will also delete objects instancing that lattice data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataLibraries(bpy_prop_collection[Library], bpy_struct):
    """Collection of libraries"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def load(
        self,
        filepath: str | None,
        link: bool | None = False,
        relative: bool | None = False,
    ):
        """Returns a context manager which exposes 2 library objects on entering.
        Each object has attributes matching bpy.data which are lists of strings to be linked.

                :param filepath: The path to a blend file.
                :type filepath: str | None
                :param link: When False reference to the original file is lost.
                :type link: bool | None
                :param relative: When True the path is stored relative to the open blend file.
                :type relative: bool | None
        """

    def write(
        self,
        filepath: str | None,
        datablocks: set | None,
        relative_remap: bool | None = False,
        fake_user: bool | None = False,
        compress: bool | None = False,
    ):
        """Write data-blocks into a blend file.

        :param filepath: The path to write the blend-file.
        :type filepath: str | None
        :param datablocks: set of data-blocks (`bpy.types.ID` instances).
        :type datablocks: set | None
        :param relative_remap: When True, remap the paths relative to the current blend-file.
        :type relative_remap: bool | None
        :param fake_user: When True, data-blocks will be written with fake-user flag enabled.
        :type fake_user: bool | None
        :param compress: When True, write a compressed blend file.
        :type compress: bool | None
        """

class BlendDataLineStyles(bpy_prop_collection[FreestyleLineStyle], bpy_struct):
    """Collection of line styles"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def new(self, name: str) -> FreestyleLineStyle:
        """Add a new line style instance to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New line style data-block
        :rtype: FreestyleLineStyle
        """

    def remove(self, linestyle: FreestyleLineStyle, do_unlink: bool | None = False):
        """Remove a line style instance from the current blendfile

        :param linestyle: Line style to remove
        :type linestyle: FreestyleLineStyle
        :param do_unlink: Unlink all usages of this line style before deleting it
        :type do_unlink: bool | None
        """

class BlendDataMasks(bpy_prop_collection[Mask], bpy_struct):
    """Collection of masks"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def new(self, name: str = "") -> Mask:
        """Add a new mask with a given name to the main database

        :param name: Mask, Name of new mask data-block
        :type name: str
        :return: New mask data-block
        :rtype: Mask
        """

    def remove(self, mask: Mask, do_unlink: bool | None = False):
        """Remove a masks from the current blendfile.

        :param mask: Mask to remove
        :type mask: Mask
        :param do_unlink: Unlink all usages of this mask before deleting it
        :type do_unlink: bool | None
        """

class BlendDataMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Material:
        """Add a new material to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New material data-block
        :rtype: Material
        """

    def remove(self, material: Material, do_unlink: bool | None = False):
        """Remove a material from the current blendfile

        :param material: Material to remove
        :type material: Material
        :param do_unlink: Unlink all usages of this material before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataMeshes(bpy_prop_collection[Mesh], bpy_struct):
    """Collection of meshes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Mesh:
        """Add a new mesh to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New mesh data-block
        :rtype: Mesh
        """

    def new_from_object(
        self,
        scene: Scene,
        object: Object,
        apply_modifiers: bool | None,
        settings: typing.Literal["PREVIEW", "RENDER"] | None,
        calc_tessface: bool | None = True,
        calc_undeformed: bool | None = False,
    ) -> Mesh:
        """Add a new mesh created from object with modifiers applied

                :param scene: Scene within which to evaluate modifiers
                :type scene: Scene
                :param object: Object to create mesh from
                :type object: Object
                :param apply_modifiers: Apply modifiers
                :type apply_modifiers: bool | None
                :param settings: Modifier settings to apply

        PREVIEW Preview, Apply modifier preview settings.

        RENDER Render, Apply modifier render settings.
                :type settings: typing.Literal['PREVIEW','RENDER'] | None
                :param calc_tessface: Calculate Tessellation, Calculate tessellation faces
                :type calc_tessface: bool | None
                :param calc_undeformed: Calculate Undeformed, Calculate undeformed vertex coordinates
                :type calc_undeformed: bool | None
                :return: Mesh created from object, remove it if it is only used for export
                :rtype: Mesh
        """

    def remove(self, mesh: Mesh, do_unlink: bool | None = False):
        """Remove a mesh from the current blendfile

        :param mesh: Mesh to remove
        :type mesh: Mesh
        :param do_unlink: Unlink all usages of this mesh before deleting it (WARNING: will also delete objects instancing that mesh data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataMetaBalls(bpy_prop_collection[MetaBall], bpy_struct):
    """Collection of metaballs"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> MetaBall:
        """Add a new metaball to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New metaball data-block
        :rtype: MetaBall
        """

    def remove(self, metaball: MetaBall, do_unlink: bool | None = False):
        """Remove a metaball from the current blendfile

        :param metaball: Metaball to remove
        :type metaball: MetaBall
        :param do_unlink: Unlink all usages of this metaball before deleting it (WARNING: will also delete objects instancing that metaball data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataMovieClips(bpy_prop_collection[MovieClip], bpy_struct):
    """Collection of movie clips"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

    def remove(self, clip: MovieClip, do_unlink: bool | None = False):
        """Remove a movie clip from the current blendfile.

        :param clip: Movie clip to remove
        :type clip: MovieClip
        :param do_unlink: Unlink all usages of this movie clip before deleting it
        :type do_unlink: bool | None
        """

    def load(self, filepath: str, check_existing: bool | None = False) -> MovieClip:
        """Add a new movie clip to the main database from a file (while check_existing is disabled for consistency with other load functions, behavior with multiple movie-clips using the same file may incorrectly generate proxies)

        :param filepath: path for the data-block
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New movie clip data-block
        :rtype: MovieClip
        """

class BlendDataNodeTrees(bpy_prop_collection[NodeTree], bpy_struct):
    """Collection of node trees"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str, type: typing.Literal["DUMMY"] | None) -> NodeTree:
        """Add a new node tree to the main database

        :param name: New name for the data-block
        :type name: str
        :param type: Type, The type of node_group to add
        :type type: typing.Literal['DUMMY'] | None
        :return: New node tree data-block
        :rtype: NodeTree
        """

    def remove(self, tree: NodeTree, do_unlink: bool | None = False):
        """Remove a node tree from the current blendfile

        :param tree: Node tree to remove
        :type tree: NodeTree
        :param do_unlink: Unlink all usages of this node tree before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of objects"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str, object_data: ID | None) -> Object:
        """Add a new object to the main database

        :param name: New name for the data-block
        :type name: str
        :param object_data: Object data or None for an empty object
        :type object_data: ID | None
        :return: New object data-block
        :rtype: Object
        """

    def remove(self, object: Object, do_unlink: bool | None = False):
        """Remove a object from the current blendfile

        :param object: Object to remove
        :type object: Object
        :param do_unlink: Unlink all usages of this object before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataPaintCurves(bpy_prop_collection[PaintCurve], bpy_struct):
    """Collection of paint curves"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataPalettes(bpy_prop_collection[Palette], bpy_struct):
    """Collection of palettes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Palette:
        """Add a new palette to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New palette data-block
        :rtype: Palette
        """

    def remove(self, palette: Palette, do_unlink: bool | None = False):
        """Remove a palette from the current blendfile

        :param palette: Palette to remove
        :type palette: Palette
        :param do_unlink: Unlink all usages of this palette before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataParticles(bpy_prop_collection[ParticleSettings], bpy_struct):
    """Collection of particle settings"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> ParticleSettings:
        """Add a new particle settings instance to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New particle settings data-block
        :rtype: ParticleSettings
        """

    def remove(self, particle: ParticleSettings, do_unlink: bool | None = False):
        """Remove a particle settings instance from the current blendfile

        :param particle: Particle Settings to remove
        :type particle: ParticleSettings
        :param do_unlink: Unlink all usages of those particle settings before deleting them
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataScenes(bpy_prop_collection[Scene], bpy_struct):
    """Collection of scenes"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Scene:
        """Add a new scene to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New scene data-block
        :rtype: Scene
        """

    def remove(self, scene: Scene, do_unlink: bool | None = False):
        """Remove a scene from the current blendfile

        :param scene: Scene to remove
        :type scene: Scene
        :param do_unlink: Unlink all usages of this scene before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataScreens(bpy_prop_collection[Screen], bpy_struct):
    """Collection of screens"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataSounds(bpy_prop_collection[Sound], bpy_struct):
    """Collection of sounds"""

    is_updated: bool
    """ 

    :type: bool
    """

    def load(self, filepath: str, check_existing: bool | None = False) -> Sound:
        """Add a new sound to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str
        :param check_existing: Using existing data-block if this file is already loaded
        :type check_existing: bool | None
        :return: New text data-block
        :rtype: Sound
        """

    def remove(self, sound: Sound, do_unlink: bool | None = False):
        """Remove a sound from the current blendfile

        :param sound: Sound to remove
        :type sound: Sound
        :param do_unlink: Unlink all usages of this sound before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataSpeakers(bpy_prop_collection[Speaker], bpy_struct):
    """Collection of speakers"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Speaker:
        """Add a new speaker to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New speaker data-block
        :rtype: Speaker
        """

    def remove(self, speaker: Speaker, do_unlink: bool | None = False):
        """Remove a speaker from the current blendfile

        :param speaker: Speaker to remove
        :type speaker: Speaker
        :param do_unlink: Unlink all usages of this speaker before deleting it (WARNING: will also delete objects instancing that speaker data)
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataTexts(bpy_prop_collection[Text], bpy_struct):
    """Collection of texts"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> Text:
        """Add a new text to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New text data-block
        :rtype: Text
        """

    def remove(self, text: Text, do_unlink: bool | None = False):
        """Remove a text from the current blendfile

        :param text: Text to remove
        :type text: Text
        :param do_unlink: Unlink all usages of this text before deleting it
        :type do_unlink: bool | None
        """

    def load(self, filepath: str, internal: bool | None = False) -> Text:
        """Add a new text to the main database from a file

        :param filepath: path for the data-block
        :type filepath: str
        :param internal: Make internal, Make text file internal after loading
        :type internal: bool | None
        :return: New text data-block
        :rtype: Text
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataTextures(bpy_prop_collection[Texture], bpy_struct):
    """Collection of groups"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(
        self,
        name: str,
        type: typing.Literal[
            "NONE",
            "BLEND",
            "CLOUDS",
            "DISTORTED_NOISE",
            "ENVIRONMENT_MAP",
            "IMAGE",
            "MAGIC",
            "MARBLE",
            "MUSGRAVE",
            "NOISE",
            "OCEAN",
            "POINT_DENSITY",
            "STUCCI",
            "VORONOI",
            "VOXEL_DATA",
            "WOOD",
        ]
        | None,
    ) -> Texture:
        """Add a new texture to the main database

                :param name: New name for the data-block
                :type name: str
                :param type: Type, The type of texture to add

        NONE None.

        BLEND Blend, Procedural - create a ramp texture.

        CLOUDS Clouds, Procedural - create a cloud-like fractal noise texture.

        DISTORTED_NOISE Distorted Noise, Procedural - noise texture distorted by two noise algorithms.

        ENVIRONMENT_MAP Environment Map, Create a render of the environment mapped to a texture.

        IMAGE Image or Movie, Allow for images or movies to be used as textures.

        MAGIC Magic, Procedural - color texture based on trigonometric functions.

        MARBLE Marble, Procedural - marble-like noise texture with wave generated bands.

        MUSGRAVE Musgrave, Procedural - highly flexible fractal noise texture.

        NOISE Noise, Procedural - random noise, gives a different result every time, for every frame, for every pixel.

        OCEAN Ocean, Use a texture generated by an Ocean modifier.

        POINT_DENSITY Point Density.

        STUCCI Stucci, Procedural - create a fractal noise texture.

        VORONOI Voronoi, Procedural - create cell-like patterns based on Worley noise.

        VOXEL_DATA Voxel Data, Create a 3D texture based on volumetric data.

        WOOD Wood, Procedural - wave generated bands or rings, with optional noise.
                :type type: typing.Literal['NONE','BLEND','CLOUDS','DISTORTED_NOISE','ENVIRONMENT_MAP','IMAGE','MAGIC','MARBLE','MUSGRAVE','NOISE','OCEAN','POINT_DENSITY','STUCCI','VORONOI','VOXEL_DATA','WOOD'] | None
                :return: New texture data-block
                :rtype: Texture
        """

    def remove(self, texture: Texture, do_unlink: bool | None = False):
        """Remove a texture from the current blendfile

        :param texture: Texture to remove
        :type texture: Texture
        :param do_unlink: Unlink all usages of this texture before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataWindowManagers(bpy_prop_collection[WindowManager], bpy_struct):
    """Collection of window managers"""

    is_updated: bool
    """ 

    :type: bool
    """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BlendDataWorlds(bpy_prop_collection[World], bpy_struct):
    """Collection of worlds"""

    is_updated: bool
    """ 

    :type: bool
    """

    def new(self, name: str) -> World:
        """Add a new world to the main database

        :param name: New name for the data-block
        :type name: str
        :return: New world data-block
        :rtype: World
        """

    def remove(self, world: World, do_unlink: bool | None = False):
        """Remove a world from the current blendfile

        :param world: World to remove
        :type world: World
        :param do_unlink: Unlink all usages of this world before deleting it
        :type do_unlink: bool | None
        """

    def tag(self, value: bool | None):
        """tag

        :param value: Value
        :type value: bool | None
        """

class BoneGroups(bpy_prop_collection[BoneGroup], bpy_struct):
    """Collection of bone groups"""

    active: BoneGroup | None
    """ Active bone group for this pose

    :type: BoneGroup | None
    """

    active_index: int | None
    """ Active index in bone groups array

    :type: int | None
    """

    def new(self, name: str = "Group") -> BoneGroup:
        """Add a new bone group to the object

        :param name: Name of the new group
        :type name: str
        :return: New bone group
        :rtype: BoneGroup
        """

    def remove(self, group: BoneGroup):
        """Remove a bone group from this object

        :param group: Removed bone group
        :type group: BoneGroup
        """

class ChannelDriverVariables(bpy_prop_collection[DriverVariable], bpy_struct):
    """Collection of channel driver Variables"""

    def new(self) -> DriverVariable:
        """Add a new variable for the driver

        :return: Newly created Driver Variable
        :rtype: DriverVariable
        """

    def remove(self, variable: DriverVariable):
        """Remove an existing variable from the driver

        :param variable: Variable to remove from the driver
        :type variable: DriverVariable
        """

class ColorRampElements(bpy_prop_collection[ColorRampElement], bpy_struct):
    """Collection of Color Ramp Elements"""

    def new(self, position: float | None) -> ColorRampElement:
        """Add element to ColorRamp

        :param position: Position, Position to add element
        :type position: float | None
        :return: New element
        :rtype: ColorRampElement
        """

    def remove(self, element: ColorRampElement):
        """Delete element from ColorRamp

        :param element: Element to remove
        :type element: ColorRampElement
        """

class CompositorNodeOutputFileFileSlots(
    bpy_prop_collection[NodeOutputFileSlotFile], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all file slots from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class CompositorNodeOutputFileLayerSlots(
    bpy_prop_collection[NodeOutputFileSlotLayer], bpy_struct
):
    """Collection of File Output node slots"""

    def new(self, name: str) -> NodeSocket:
        """Add a file slot to this node

        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a file slot from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all file slots from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a file slot to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class CurveMapPoints(bpy_prop_collection[CurveMapPoint], bpy_struct):
    """Collection of Curve Map Points"""

    def new(self, position: float | None, value: float | None) -> CurveMapPoint:
        """Add point to CurveMap

        :param position: Position, Position to add point
        :type position: float | None
        :param value: Value, Value of point
        :type value: float | None
        :return: New point
        :rtype: CurveMapPoint
        """

    def remove(self, point: CurveMapPoint):
        """Delete point from CurveMap

        :param point: PointElement to remove
        :type point: CurveMapPoint
        """

class CurveSplines(bpy_prop_collection[Spline], bpy_struct):
    """Collection of curve splines"""

    active: Spline | None
    """ Active curve spline

    :type: Spline | None
    """

    def new(
        self,
        type: typing.Literal["POLY", "BEZIER", "BSPLINE", "CARDINAL", "NURBS"] | None,
    ) -> Spline:
        """Add a new spline to the curve

        :param type: type for the new spline
        :type type: typing.Literal['POLY','BEZIER','BSPLINE','CARDINAL','NURBS'] | None
        :return: The newly created spline
        :rtype: Spline
        """

    def remove(self, spline: Spline):
        """Remove a spline from a curve

        :param spline: The spline to remove
        :type spline: Spline
        """

    def clear(self):
        """Remove all splines from a curve"""

class DynamicPaintSurfaces(bpy_prop_collection[DynamicPaintSurface], bpy_struct):
    """Collection of Dynamic Paint Canvas surfaces"""

    active: DynamicPaintSurface | None
    """ Active Dynamic Paint surface being displayed

    :type: DynamicPaintSurface | None
    """

    active_index: int | None
    """ 

    :type: int | None
    """

class FCurveKeyframePoints(bpy_prop_collection[Keyframe], bpy_struct):
    """Collection of keyframe points"""

    def insert(
        self,
        frame: float | None,
        value: float | None,
        options: set[typing.Literal["REPLACE", "NEEDED", "FAST"]] | None = {},
        keyframe_type: typing.Literal[
            "KEYFRAME", "BREAKDOWN", "MOVING_HOLD", "EXTREME", "JITTER"
        ]
        | None = "KEYFRAME",
    ) -> Keyframe:
        """Add a keyframe point to a F-Curve

                :param frame: X Value of this keyframe point
                :type frame: float | None
                :param value: Y Value of this keyframe point
                :type value: float | None
                :param options: Keyframe options

        REPLACE Replace, Don't add any new keyframes, but just replace existing ones.

        NEEDED Needed, Only adds keyframes that are needed.

        FAST Fast, Fast keyframe insertion to avoid recalculating the curve each time.
                :type options: set[typing.Literal['REPLACE','NEEDED','FAST']] | None
                :param keyframe_type: Type of keyframe to insert

        KEYFRAME Keyframe, Normal keyframe - e.g. for key poses.

        BREAKDOWN Breakdown, A breakdown pose - e.g. for transitions between key poses.

        MOVING_HOLD Moving Hold, A keyframe that is part of a moving hold.

        EXTREME Extreme, An 'extreme' pose, or some other purpose as needed.

        JITTER Jitter, A filler or baked keyframe for keying on ones, or some other purpose as needed.
                :type keyframe_type: typing.Literal['KEYFRAME','BREAKDOWN','MOVING_HOLD','EXTREME','JITTER'] | None
                :return: Newly created keyframe
                :rtype: Keyframe
        """

    def add(self, count: int | None = 1):
        """Add a keyframe point to a F-Curve

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    def remove(self, keyframe: Keyframe, fast: bool | None = False):
        """Remove keyframe from an F-Curve

        :param keyframe: Keyframe to remove
        :type keyframe: Keyframe
        :param fast: Fast, Fast keyframe removal to avoid recalculating the curve each time
        :type fast: bool | None
        """

class FCurveModifiers(bpy_prop_collection[FModifier], bpy_struct):
    """Collection of F-Curve Modifiers"""

    active: FModifier | None
    """ Active F-Curve Modifier

    :type: FModifier | None
    """

    def new(
        self,
        type: typing.Literal[
            "NULL",
            "GENERATOR",
            "FNGENERATOR",
            "ENVELOPE",
            "CYCLES",
            "NOISE",
            "LIMITS",
            "STEPPED",
        ]
        | None,
    ) -> FModifier:
        """Add a constraint to this object

                :param type: Constraint type to add

        NULL Invalid.

        GENERATOR Generator, Generate a curve using a factorized or expanded polynomial.

        FNGENERATOR Built-In Function, Generate a curve using standard math functions such as sin and cos.

        ENVELOPE Envelope, Reshape F-Curve values - e.g. change amplitude of movements.

        CYCLES Cycles, Cyclic extend/repeat keyframe sequence.

        NOISE Noise, Add pseudo-random noise on top of F-Curves.

        LIMITS Limits, Restrict maximum and minimum values of F-Curve.

        STEPPED Stepped Interpolation, Snap values to nearest grid-step - e.g. for a stop-motion look.
                :type type: typing.Literal['NULL','GENERATOR','FNGENERATOR','ENVELOPE','CYCLES','NOISE','LIMITS','STEPPED'] | None
                :return: New fmodifier
                :rtype: FModifier
        """

    def remove(self, modifier: FModifier):
        """Remove a modifier from this F-Curve

        :param modifier: Removed modifier
        :type modifier: FModifier
        """

class FModifierEnvelopeControlPoints(
    bpy_prop_collection[FModifierEnvelopeControlPoint], bpy_struct
):
    """Control points defining the shape of the envelope"""

    def add(self, frame: float | None) -> FModifierEnvelopeControlPoint:
        """Add a control point to a FModifierEnvelope

        :param frame: Frame to add this control-point
        :type frame: float | None
        :return: Newly created control-point
        :rtype: FModifierEnvelopeControlPoint
        """

    def remove(self, point: FModifierEnvelopeControlPoint):
        """Remove a control-point from an FModifierEnvelope

        :param point: Control-point to remove
        :type point: FModifierEnvelopeControlPoint
        """

class FreestyleModules(bpy_prop_collection[FreestyleModuleSettings], bpy_struct):
    """A list of style modules (to be applied from top to bottom)"""

    def new(self) -> FreestyleModuleSettings:
        """Add a style module to scene render layer Freestyle settings

        :return: Newly created style module
        :rtype: FreestyleModuleSettings
        """

    def remove(self, module: FreestyleModuleSettings):
        """Remove a style module from scene render layer Freestyle settings

        :param module: Style module to remove
        :type module: FreestyleModuleSettings
        """

class GPencilFrames(bpy_prop_collection[GPencilFrame], bpy_struct):
    """Collection of grease pencil frames"""

    def new(self, frame_number: int | None) -> GPencilFrame:
        """Add a new grease pencil frame

        :param frame_number: Frame Number, The frame on which this sketch appears
        :type frame_number: int | None
        :return: The newly created frame
        :rtype: GPencilFrame
        """

    def remove(self, frame: GPencilFrame):
        """Remove a grease pencil frame

        :param frame: Frame, The frame to remove
        :type frame: GPencilFrame
        """

    def copy(self, source: GPencilFrame) -> GPencilFrame:
        """Copy a grease pencil frame

        :param source: Source, The source frame
        :type source: GPencilFrame
        :return: The newly copied frame
        :rtype: GPencilFrame
        """

class GPencilPaletteColors(bpy_prop_collection[GPencilPaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: GPencilPaletteColor | None
    """ Current active color

    :type: GPencilPaletteColor | None
    """

    active_index: int | None
    """ Index of active palette color

    :type: int | None
    """

    def new(self) -> GPencilPaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: GPencilPaletteColor
        """

    def remove(self, color: GPencilPaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: GPencilPaletteColor
        """

class GPencilStrokePoints(bpy_prop_collection[GPencilStrokePoint], bpy_struct):
    """Collection of grease pencil stroke points"""

    def add(
        self,
        count: int | None = 1,
        pressure: float | None = 1.0,
        strength: float | None = 1.0,
    ):
        """Add a new grease pencil stroke point

        :param count: Number, Number of points to add to the stroke
        :type count: int | None
        :param pressure: Pressure, Pressure for newly created points
        :type pressure: float | None
        :param strength: Strength, Color intensity (alpha factor) for newly created points
        :type strength: float | None
        """

    def pop(self, index: int | None = -1):
        """Remove a grease pencil stroke point

        :param index: Index, point index
        :type index: int | None
        """

class GPencilStrokes(bpy_prop_collection[GPencilStroke], bpy_struct):
    """Collection of grease pencil stroke"""

    def new(self, colorname: str = "") -> GPencilStroke:
        """Add a new grease pencil stroke

        :param colorname: Color, Name of the color
        :type colorname: str
        :return: The newly created stroke
        :rtype: GPencilStroke
        """

    def remove(self, stroke: GPencilStroke):
        """Remove a grease pencil stroke

        :param stroke: Stroke, The stroke to remove
        :type stroke: GPencilStroke
        """

class GreasePencilBrushes(bpy_prop_collection[GPencilBrush], bpy_struct):
    """Collection of grease pencil brushes"""

    active: GPencilBrush | None
    """ Current active brush

    :type: GPencilBrush | None
    """

    active_index: int | None
    """ Index of active brush

    :type: int | None
    """

    def new(self, name: str, set_active: bool | None = False) -> GPencilBrush:
        """Add a new grease pencil brush

        :param name: Name, Name of the brush
        :type name: str
        :param set_active: Set Active, Set the newly created brush to the active brush
        :type set_active: bool | None
        :return: The newly created brush
        :rtype: GPencilBrush
        """

    def remove(self, brush: GPencilBrush):
        """Remove a grease pencil brush

        :param brush: The brush to remove
        :type brush: GPencilBrush
        """

class GreasePencilLayers(bpy_prop_collection[GPencilLayer], bpy_struct):
    """Collection of grease pencil layers"""

    active: GPencilLayer | None
    """ Active grease pencil layer

    :type: GPencilLayer | None
    """

    active_index: int | None
    """ Index of active grease pencil layer

    :type: int | None
    """

    def new(self, name: str, set_active: bool | None = False) -> GPencilLayer:
        """Add a new grease pencil layer

        :param name: Name, Name of the layer
        :type name: str
        :param set_active: Set Active, Set the newly created layer to the active layer
        :type set_active: bool | None
        :return: The newly created layer
        :rtype: GPencilLayer
        """

    def remove(self, layer: GPencilLayer):
        """Remove a grease pencil layer

        :param layer: The layer to remove
        :type layer: GPencilLayer
        """

class GreasePencilPalettes(bpy_prop_collection[GPencilPalette], bpy_struct):
    """Collection of grease pencil palettes"""

    active: GPencilPalette | None
    """ Current active palette

    :type: GPencilPalette | None
    """

    active_index: int | None
    """ Index of active palette

    :type: int | None
    """

    def new(self, name: str, set_active: bool | None = False) -> GPencilPalette:
        """Add a new grease pencil palette

        :param name: Name, Name of the palette
        :type name: str
        :param set_active: Set Active, Activate the newly created palette
        :type set_active: bool | None
        :return: The newly created palette
        :rtype: GPencilPalette
        """

    def remove(self, palette: GPencilPalette):
        """Remove a grease pencil palette

        :param palette: The palette to remove
        :type palette: GPencilPalette
        """

class GroupObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of group objects"""

    def link(self, object: Object):
        """Add this object to a group

        :param object: Object to add
        :type object: Object
        """

    def unlink(self, object: Object | None):
        """Remove this object to a group

        :param object: Object to remove
        :type object: Object | None
        """

class IDMaterials(bpy_prop_collection[Material], bpy_struct):
    """Collection of materials"""

    def append(self, material: Material | None):
        """Add a new material to the data block

        :param material: Material to add
        :type material: Material | None
        """

    def pop(self, *, index: int = -1, update_data: bool | None = False) -> Material:
        """Remove a material from the data block

        :param index: Index of material to remove
        :type index: int
        :param update_data: Update data by re-adjusting the material slots assigned
        :type update_data: bool | None
        :return: Material to remove
        :rtype: Material
        """

    def clear(self, update_data: bool | None = False):
        """Remove all materials from the data block

        :param update_data: Update data by re-adjusting the material slots assigned
        :type update_data: bool | None
        """

class KeyConfigurations(bpy_prop_collection[KeyConfig], bpy_struct):
    """Collection of KeyConfigs"""

    active: KeyConfig | None
    """ Active key configuration (preset)

    :type: KeyConfig | None
    """

    addon: KeyConfig
    """ Key configuration that can be extended by add-ons, and is added to the active configuration when handling events

    :type: KeyConfig
    """

    default: KeyConfig
    """ Default builtin key configuration

    :type: KeyConfig
    """

    user: KeyConfig
    """ Final key configuration that combines keymaps from the active and add-on configurations, and can be edited by the user

    :type: KeyConfig
    """

    def new(self, name: str) -> KeyConfig:
        """new

        :param name: Name
        :type name: str
        :return: Key Configuration, Added key configuration
        :rtype: KeyConfig
        """

    def remove(self, keyconfig: KeyConfig):
        """remove

        :param keyconfig: Key Configuration, Removed key configuration
        :type keyconfig: KeyConfig
        """

class KeyMapItems(bpy_prop_collection[KeyMapItem], bpy_struct):
    """Collection of keymap items"""

    def new(
        self,
        idname: str,
        type: typing.Literal[
            "NONE",
            "LEFTMOUSE",
            "MIDDLEMOUSE",
            "RIGHTMOUSE",
            "BUTTON4MOUSE",
            "BUTTON5MOUSE",
            "BUTTON6MOUSE",
            "BUTTON7MOUSE",
            "ACTIONMOUSE",
            "SELECTMOUSE",
            "PEN",
            "ERASER",
            "MOUSEMOVE",
            "INBETWEEN_MOUSEMOVE",
            "TRACKPADPAN",
            "TRACKPADZOOM",
            "MOUSEROTATE",
            "WHEELUPMOUSE",
            "WHEELDOWNMOUSE",
            "WHEELINMOUSE",
            "WHEELOUTMOUSE",
            "EVT_TWEAK_L",
            "EVT_TWEAK_M",
            "EVT_TWEAK_R",
            "EVT_TWEAK_A",
            "EVT_TWEAK_S",
            "A",
            "B",
            "C",
            "D",
            "E",
            "F",
            "G",
            "H",
            "I",
            "J",
            "K",
            "L",
            "M",
            "N",
            "O",
            "P",
            "Q",
            "R",
            "S",
            "T",
            "U",
            "V",
            "W",
            "X",
            "Y",
            "Z",
            "ZERO",
            "ONE",
            "TWO",
            "THREE",
            "FOUR",
            "FIVE",
            "SIX",
            "SEVEN",
            "EIGHT",
            "NINE",
            "LEFT_CTRL",
            "LEFT_ALT",
            "LEFT_SHIFT",
            "RIGHT_ALT",
            "RIGHT_CTRL",
            "RIGHT_SHIFT",
            "OSKEY",
            "GRLESS",
            "ESC",
            "TAB",
            "RET",
            "SPACE",
            "LINE_FEED",
            "BACK_SPACE",
            "DEL",
            "SEMI_COLON",
            "PERIOD",
            "COMMA",
            "QUOTE",
            "ACCENT_GRAVE",
            "MINUS",
            "PLUS",
            "SLASH",
            "BACK_SLASH",
            "EQUAL",
            "LEFT_BRACKET",
            "RIGHT_BRACKET",
            "LEFT_ARROW",
            "DOWN_ARROW",
            "RIGHT_ARROW",
            "UP_ARROW",
            "NUMPAD_2",
            "NUMPAD_4",
            "NUMPAD_6",
            "NUMPAD_8",
            "NUMPAD_1",
            "NUMPAD_3",
            "NUMPAD_5",
            "NUMPAD_7",
            "NUMPAD_9",
            "NUMPAD_PERIOD",
            "NUMPAD_SLASH",
            "NUMPAD_ASTERIX",
            "NUMPAD_0",
            "NUMPAD_MINUS",
            "NUMPAD_ENTER",
            "NUMPAD_PLUS",
            "F1",
            "F2",
            "F3",
            "F4",
            "F5",
            "F6",
            "F7",
            "F8",
            "F9",
            "F10",
            "F11",
            "F12",
            "F13",
            "F14",
            "F15",
            "F16",
            "F17",
            "F18",
            "F19",
            "PAUSE",
            "INSERT",
            "HOME",
            "PAGE_UP",
            "PAGE_DOWN",
            "END",
            "MEDIA_PLAY",
            "MEDIA_STOP",
            "MEDIA_FIRST",
            "MEDIA_LAST",
            "TEXTINPUT",
            "WINDOW_DEACTIVATE",
            "TIMER",
            "TIMER0",
            "TIMER1",
            "TIMER2",
            "TIMER_JOBS",
            "TIMER_AUTOSAVE",
            "TIMER_REPORT",
            "TIMERREGION",
            "NDOF_MOTION",
            "NDOF_BUTTON_MENU",
            "NDOF_BUTTON_FIT",
            "NDOF_BUTTON_TOP",
            "NDOF_BUTTON_BOTTOM",
            "NDOF_BUTTON_LEFT",
            "NDOF_BUTTON_RIGHT",
            "NDOF_BUTTON_FRONT",
            "NDOF_BUTTON_BACK",
            "NDOF_BUTTON_ISO1",
            "NDOF_BUTTON_ISO2",
            "NDOF_BUTTON_ROLL_CW",
            "NDOF_BUTTON_ROLL_CCW",
            "NDOF_BUTTON_SPIN_CW",
            "NDOF_BUTTON_SPIN_CCW",
            "NDOF_BUTTON_TILT_CW",
            "NDOF_BUTTON_TILT_CCW",
            "NDOF_BUTTON_ROTATE",
            "NDOF_BUTTON_PANZOOM",
            "NDOF_BUTTON_DOMINANT",
            "NDOF_BUTTON_PLUS",
            "NDOF_BUTTON_MINUS",
            "NDOF_BUTTON_ESC",
            "NDOF_BUTTON_ALT",
            "NDOF_BUTTON_SHIFT",
            "NDOF_BUTTON_CTRL",
            "NDOF_BUTTON_1",
            "NDOF_BUTTON_2",
            "NDOF_BUTTON_3",
            "NDOF_BUTTON_4",
            "NDOF_BUTTON_5",
            "NDOF_BUTTON_6",
            "NDOF_BUTTON_7",
            "NDOF_BUTTON_8",
            "NDOF_BUTTON_9",
            "NDOF_BUTTON_10",
            "NDOF_BUTTON_A",
            "NDOF_BUTTON_B",
            "NDOF_BUTTON_C",
        ]
        | None,
        value: typing.Literal[
            "ANY",
            "NOTHING",
            "PRESS",
            "RELEASE",
            "CLICK",
            "DOUBLE_CLICK",
            "NORTH",
            "NORTH_EAST",
            "EAST",
            "SOUTH_EAST",
            "SOUTH",
            "SOUTH_WEST",
            "WEST",
            "NORTH_WEST",
        ]
        | None,
        any: bool | None = False,
        shift: bool | None = False,
        ctrl: bool | None = False,
        alt: bool | None = False,
        oskey: bool | None = False,
        key_modifier: typing.Literal[
            "NONE",
            "LEFTMOUSE",
            "MIDDLEMOUSE",
            "RIGHTMOUSE",
            "BUTTON4MOUSE",
            "BUTTON5MOUSE",
            "BUTTON6MOUSE",
            "BUTTON7MOUSE",
            "ACTIONMOUSE",
            "SELECTMOUSE",
            "PEN",
            "ERASER",
            "MOUSEMOVE",
            "INBETWEEN_MOUSEMOVE",
            "TRACKPADPAN",
            "TRACKPADZOOM",
            "MOUSEROTATE",
            "WHEELUPMOUSE",
            "WHEELDOWNMOUSE",
            "WHEELINMOUSE",
            "WHEELOUTMOUSE",
            "EVT_TWEAK_L",
            "EVT_TWEAK_M",
            "EVT_TWEAK_R",
            "EVT_TWEAK_A",
            "EVT_TWEAK_S",
            "A",
            "B",
            "C",
            "D",
            "E",
            "F",
            "G",
            "H",
            "I",
            "J",
            "K",
            "L",
            "M",
            "N",
            "O",
            "P",
            "Q",
            "R",
            "S",
            "T",
            "U",
            "V",
            "W",
            "X",
            "Y",
            "Z",
            "ZERO",
            "ONE",
            "TWO",
            "THREE",
            "FOUR",
            "FIVE",
            "SIX",
            "SEVEN",
            "EIGHT",
            "NINE",
            "LEFT_CTRL",
            "LEFT_ALT",
            "LEFT_SHIFT",
            "RIGHT_ALT",
            "RIGHT_CTRL",
            "RIGHT_SHIFT",
            "OSKEY",
            "GRLESS",
            "ESC",
            "TAB",
            "RET",
            "SPACE",
            "LINE_FEED",
            "BACK_SPACE",
            "DEL",
            "SEMI_COLON",
            "PERIOD",
            "COMMA",
            "QUOTE",
            "ACCENT_GRAVE",
            "MINUS",
            "PLUS",
            "SLASH",
            "BACK_SLASH",
            "EQUAL",
            "LEFT_BRACKET",
            "RIGHT_BRACKET",
            "LEFT_ARROW",
            "DOWN_ARROW",
            "RIGHT_ARROW",
            "UP_ARROW",
            "NUMPAD_2",
            "NUMPAD_4",
            "NUMPAD_6",
            "NUMPAD_8",
            "NUMPAD_1",
            "NUMPAD_3",
            "NUMPAD_5",
            "NUMPAD_7",
            "NUMPAD_9",
            "NUMPAD_PERIOD",
            "NUMPAD_SLASH",
            "NUMPAD_ASTERIX",
            "NUMPAD_0",
            "NUMPAD_MINUS",
            "NUMPAD_ENTER",
            "NUMPAD_PLUS",
            "F1",
            "F2",
            "F3",
            "F4",
            "F5",
            "F6",
            "F7",
            "F8",
            "F9",
            "F10",
            "F11",
            "F12",
            "F13",
            "F14",
            "F15",
            "F16",
            "F17",
            "F18",
            "F19",
            "PAUSE",
            "INSERT",
            "HOME",
            "PAGE_UP",
            "PAGE_DOWN",
            "END",
            "MEDIA_PLAY",
            "MEDIA_STOP",
            "MEDIA_FIRST",
            "MEDIA_LAST",
            "TEXTINPUT",
            "WINDOW_DEACTIVATE",
            "TIMER",
            "TIMER0",
            "TIMER1",
            "TIMER2",
            "TIMER_JOBS",
            "TIMER_AUTOSAVE",
            "TIMER_REPORT",
            "TIMERREGION",
            "NDOF_MOTION",
            "NDOF_BUTTON_MENU",
            "NDOF_BUTTON_FIT",
            "NDOF_BUTTON_TOP",
            "NDOF_BUTTON_BOTTOM",
            "NDOF_BUTTON_LEFT",
            "NDOF_BUTTON_RIGHT",
            "NDOF_BUTTON_FRONT",
            "NDOF_BUTTON_BACK",
            "NDOF_BUTTON_ISO1",
            "NDOF_BUTTON_ISO2",
            "NDOF_BUTTON_ROLL_CW",
            "NDOF_BUTTON_ROLL_CCW",
            "NDOF_BUTTON_SPIN_CW",
            "NDOF_BUTTON_SPIN_CCW",
            "NDOF_BUTTON_TILT_CW",
            "NDOF_BUTTON_TILT_CCW",
            "NDOF_BUTTON_ROTATE",
            "NDOF_BUTTON_PANZOOM",
            "NDOF_BUTTON_DOMINANT",
            "NDOF_BUTTON_PLUS",
            "NDOF_BUTTON_MINUS",
            "NDOF_BUTTON_ESC",
            "NDOF_BUTTON_ALT",
            "NDOF_BUTTON_SHIFT",
            "NDOF_BUTTON_CTRL",
            "NDOF_BUTTON_1",
            "NDOF_BUTTON_2",
            "NDOF_BUTTON_3",
            "NDOF_BUTTON_4",
            "NDOF_BUTTON_5",
            "NDOF_BUTTON_6",
            "NDOF_BUTTON_7",
            "NDOF_BUTTON_8",
            "NDOF_BUTTON_9",
            "NDOF_BUTTON_10",
            "NDOF_BUTTON_A",
            "NDOF_BUTTON_B",
            "NDOF_BUTTON_C",
        ]
        | None = "NONE",
        head: bool | None = False,
    ) -> KeyMapItem:
        """new

                :param idname: Operator Identifier
                :type idname: str
                :param type: Type

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C'] | None
                :param value: Value
                :type value: typing.Literal['ANY','NOTHING','PRESS','RELEASE','CLICK','DOUBLE_CLICK','NORTH','NORTH_EAST','EAST','SOUTH_EAST','SOUTH','SOUTH_WEST','WEST','NORTH_WEST'] | None
                :param any: Any
                :type any: bool | None
                :param shift: Shift
                :type shift: bool | None
                :param ctrl: Ctrl
                :type ctrl: bool | None
                :param alt: Alt
                :type alt: bool | None
                :param oskey: OS Key
                :type oskey: bool | None
                :param key_modifier: Key Modifier

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type key_modifier: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C'] | None
                :param head: At Head, Force item to be added at start (not end) of key map so that it doesn't get blocked by an existing key map item
                :type head: bool | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """

    def new_modal(
        self,
        propvalue: str,
        type: typing.Literal[
            "NONE",
            "LEFTMOUSE",
            "MIDDLEMOUSE",
            "RIGHTMOUSE",
            "BUTTON4MOUSE",
            "BUTTON5MOUSE",
            "BUTTON6MOUSE",
            "BUTTON7MOUSE",
            "ACTIONMOUSE",
            "SELECTMOUSE",
            "PEN",
            "ERASER",
            "MOUSEMOVE",
            "INBETWEEN_MOUSEMOVE",
            "TRACKPADPAN",
            "TRACKPADZOOM",
            "MOUSEROTATE",
            "WHEELUPMOUSE",
            "WHEELDOWNMOUSE",
            "WHEELINMOUSE",
            "WHEELOUTMOUSE",
            "EVT_TWEAK_L",
            "EVT_TWEAK_M",
            "EVT_TWEAK_R",
            "EVT_TWEAK_A",
            "EVT_TWEAK_S",
            "A",
            "B",
            "C",
            "D",
            "E",
            "F",
            "G",
            "H",
            "I",
            "J",
            "K",
            "L",
            "M",
            "N",
            "O",
            "P",
            "Q",
            "R",
            "S",
            "T",
            "U",
            "V",
            "W",
            "X",
            "Y",
            "Z",
            "ZERO",
            "ONE",
            "TWO",
            "THREE",
            "FOUR",
            "FIVE",
            "SIX",
            "SEVEN",
            "EIGHT",
            "NINE",
            "LEFT_CTRL",
            "LEFT_ALT",
            "LEFT_SHIFT",
            "RIGHT_ALT",
            "RIGHT_CTRL",
            "RIGHT_SHIFT",
            "OSKEY",
            "GRLESS",
            "ESC",
            "TAB",
            "RET",
            "SPACE",
            "LINE_FEED",
            "BACK_SPACE",
            "DEL",
            "SEMI_COLON",
            "PERIOD",
            "COMMA",
            "QUOTE",
            "ACCENT_GRAVE",
            "MINUS",
            "PLUS",
            "SLASH",
            "BACK_SLASH",
            "EQUAL",
            "LEFT_BRACKET",
            "RIGHT_BRACKET",
            "LEFT_ARROW",
            "DOWN_ARROW",
            "RIGHT_ARROW",
            "UP_ARROW",
            "NUMPAD_2",
            "NUMPAD_4",
            "NUMPAD_6",
            "NUMPAD_8",
            "NUMPAD_1",
            "NUMPAD_3",
            "NUMPAD_5",
            "NUMPAD_7",
            "NUMPAD_9",
            "NUMPAD_PERIOD",
            "NUMPAD_SLASH",
            "NUMPAD_ASTERIX",
            "NUMPAD_0",
            "NUMPAD_MINUS",
            "NUMPAD_ENTER",
            "NUMPAD_PLUS",
            "F1",
            "F2",
            "F3",
            "F4",
            "F5",
            "F6",
            "F7",
            "F8",
            "F9",
            "F10",
            "F11",
            "F12",
            "F13",
            "F14",
            "F15",
            "F16",
            "F17",
            "F18",
            "F19",
            "PAUSE",
            "INSERT",
            "HOME",
            "PAGE_UP",
            "PAGE_DOWN",
            "END",
            "MEDIA_PLAY",
            "MEDIA_STOP",
            "MEDIA_FIRST",
            "MEDIA_LAST",
            "TEXTINPUT",
            "WINDOW_DEACTIVATE",
            "TIMER",
            "TIMER0",
            "TIMER1",
            "TIMER2",
            "TIMER_JOBS",
            "TIMER_AUTOSAVE",
            "TIMER_REPORT",
            "TIMERREGION",
            "NDOF_MOTION",
            "NDOF_BUTTON_MENU",
            "NDOF_BUTTON_FIT",
            "NDOF_BUTTON_TOP",
            "NDOF_BUTTON_BOTTOM",
            "NDOF_BUTTON_LEFT",
            "NDOF_BUTTON_RIGHT",
            "NDOF_BUTTON_FRONT",
            "NDOF_BUTTON_BACK",
            "NDOF_BUTTON_ISO1",
            "NDOF_BUTTON_ISO2",
            "NDOF_BUTTON_ROLL_CW",
            "NDOF_BUTTON_ROLL_CCW",
            "NDOF_BUTTON_SPIN_CW",
            "NDOF_BUTTON_SPIN_CCW",
            "NDOF_BUTTON_TILT_CW",
            "NDOF_BUTTON_TILT_CCW",
            "NDOF_BUTTON_ROTATE",
            "NDOF_BUTTON_PANZOOM",
            "NDOF_BUTTON_DOMINANT",
            "NDOF_BUTTON_PLUS",
            "NDOF_BUTTON_MINUS",
            "NDOF_BUTTON_ESC",
            "NDOF_BUTTON_ALT",
            "NDOF_BUTTON_SHIFT",
            "NDOF_BUTTON_CTRL",
            "NDOF_BUTTON_1",
            "NDOF_BUTTON_2",
            "NDOF_BUTTON_3",
            "NDOF_BUTTON_4",
            "NDOF_BUTTON_5",
            "NDOF_BUTTON_6",
            "NDOF_BUTTON_7",
            "NDOF_BUTTON_8",
            "NDOF_BUTTON_9",
            "NDOF_BUTTON_10",
            "NDOF_BUTTON_A",
            "NDOF_BUTTON_B",
            "NDOF_BUTTON_C",
        ]
        | None,
        value: typing.Literal[
            "ANY",
            "NOTHING",
            "PRESS",
            "RELEASE",
            "CLICK",
            "DOUBLE_CLICK",
            "NORTH",
            "NORTH_EAST",
            "EAST",
            "SOUTH_EAST",
            "SOUTH",
            "SOUTH_WEST",
            "WEST",
            "NORTH_WEST",
        ]
        | None,
        any: bool | None = False,
        shift: bool | None = False,
        ctrl: bool | None = False,
        alt: bool | None = False,
        oskey: bool | None = False,
        key_modifier: typing.Literal[
            "NONE",
            "LEFTMOUSE",
            "MIDDLEMOUSE",
            "RIGHTMOUSE",
            "BUTTON4MOUSE",
            "BUTTON5MOUSE",
            "BUTTON6MOUSE",
            "BUTTON7MOUSE",
            "ACTIONMOUSE",
            "SELECTMOUSE",
            "PEN",
            "ERASER",
            "MOUSEMOVE",
            "INBETWEEN_MOUSEMOVE",
            "TRACKPADPAN",
            "TRACKPADZOOM",
            "MOUSEROTATE",
            "WHEELUPMOUSE",
            "WHEELDOWNMOUSE",
            "WHEELINMOUSE",
            "WHEELOUTMOUSE",
            "EVT_TWEAK_L",
            "EVT_TWEAK_M",
            "EVT_TWEAK_R",
            "EVT_TWEAK_A",
            "EVT_TWEAK_S",
            "A",
            "B",
            "C",
            "D",
            "E",
            "F",
            "G",
            "H",
            "I",
            "J",
            "K",
            "L",
            "M",
            "N",
            "O",
            "P",
            "Q",
            "R",
            "S",
            "T",
            "U",
            "V",
            "W",
            "X",
            "Y",
            "Z",
            "ZERO",
            "ONE",
            "TWO",
            "THREE",
            "FOUR",
            "FIVE",
            "SIX",
            "SEVEN",
            "EIGHT",
            "NINE",
            "LEFT_CTRL",
            "LEFT_ALT",
            "LEFT_SHIFT",
            "RIGHT_ALT",
            "RIGHT_CTRL",
            "RIGHT_SHIFT",
            "OSKEY",
            "GRLESS",
            "ESC",
            "TAB",
            "RET",
            "SPACE",
            "LINE_FEED",
            "BACK_SPACE",
            "DEL",
            "SEMI_COLON",
            "PERIOD",
            "COMMA",
            "QUOTE",
            "ACCENT_GRAVE",
            "MINUS",
            "PLUS",
            "SLASH",
            "BACK_SLASH",
            "EQUAL",
            "LEFT_BRACKET",
            "RIGHT_BRACKET",
            "LEFT_ARROW",
            "DOWN_ARROW",
            "RIGHT_ARROW",
            "UP_ARROW",
            "NUMPAD_2",
            "NUMPAD_4",
            "NUMPAD_6",
            "NUMPAD_8",
            "NUMPAD_1",
            "NUMPAD_3",
            "NUMPAD_5",
            "NUMPAD_7",
            "NUMPAD_9",
            "NUMPAD_PERIOD",
            "NUMPAD_SLASH",
            "NUMPAD_ASTERIX",
            "NUMPAD_0",
            "NUMPAD_MINUS",
            "NUMPAD_ENTER",
            "NUMPAD_PLUS",
            "F1",
            "F2",
            "F3",
            "F4",
            "F5",
            "F6",
            "F7",
            "F8",
            "F9",
            "F10",
            "F11",
            "F12",
            "F13",
            "F14",
            "F15",
            "F16",
            "F17",
            "F18",
            "F19",
            "PAUSE",
            "INSERT",
            "HOME",
            "PAGE_UP",
            "PAGE_DOWN",
            "END",
            "MEDIA_PLAY",
            "MEDIA_STOP",
            "MEDIA_FIRST",
            "MEDIA_LAST",
            "TEXTINPUT",
            "WINDOW_DEACTIVATE",
            "TIMER",
            "TIMER0",
            "TIMER1",
            "TIMER2",
            "TIMER_JOBS",
            "TIMER_AUTOSAVE",
            "TIMER_REPORT",
            "TIMERREGION",
            "NDOF_MOTION",
            "NDOF_BUTTON_MENU",
            "NDOF_BUTTON_FIT",
            "NDOF_BUTTON_TOP",
            "NDOF_BUTTON_BOTTOM",
            "NDOF_BUTTON_LEFT",
            "NDOF_BUTTON_RIGHT",
            "NDOF_BUTTON_FRONT",
            "NDOF_BUTTON_BACK",
            "NDOF_BUTTON_ISO1",
            "NDOF_BUTTON_ISO2",
            "NDOF_BUTTON_ROLL_CW",
            "NDOF_BUTTON_ROLL_CCW",
            "NDOF_BUTTON_SPIN_CW",
            "NDOF_BUTTON_SPIN_CCW",
            "NDOF_BUTTON_TILT_CW",
            "NDOF_BUTTON_TILT_CCW",
            "NDOF_BUTTON_ROTATE",
            "NDOF_BUTTON_PANZOOM",
            "NDOF_BUTTON_DOMINANT",
            "NDOF_BUTTON_PLUS",
            "NDOF_BUTTON_MINUS",
            "NDOF_BUTTON_ESC",
            "NDOF_BUTTON_ALT",
            "NDOF_BUTTON_SHIFT",
            "NDOF_BUTTON_CTRL",
            "NDOF_BUTTON_1",
            "NDOF_BUTTON_2",
            "NDOF_BUTTON_3",
            "NDOF_BUTTON_4",
            "NDOF_BUTTON_5",
            "NDOF_BUTTON_6",
            "NDOF_BUTTON_7",
            "NDOF_BUTTON_8",
            "NDOF_BUTTON_9",
            "NDOF_BUTTON_10",
            "NDOF_BUTTON_A",
            "NDOF_BUTTON_B",
            "NDOF_BUTTON_C",
        ]
        | None = "NONE",
    ) -> KeyMapItem:
        """new_modal

                :param propvalue: Property Value
                :type propvalue: str
                :param type: Type

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C'] | None
                :param value: Value
                :type value: typing.Literal['ANY','NOTHING','PRESS','RELEASE','CLICK','DOUBLE_CLICK','NORTH','NORTH_EAST','EAST','SOUTH_EAST','SOUTH','SOUTH_WEST','WEST','NORTH_WEST'] | None
                :param any: Any
                :type any: bool | None
                :param shift: Shift
                :type shift: bool | None
                :param ctrl: Ctrl
                :type ctrl: bool | None
                :param alt: Alt
                :type alt: bool | None
                :param oskey: OS Key
                :type oskey: bool | None
                :param key_modifier: Key Modifier

        NONE .

        LEFTMOUSE Left Mouse, LMB.

        MIDDLEMOUSE Middle Mouse, MMB.

        RIGHTMOUSE Right Mouse, RMB.

        BUTTON4MOUSE Button4 Mouse, MB4.

        BUTTON5MOUSE Button5 Mouse, MB5.

        BUTTON6MOUSE Button6 Mouse, MB6.

        BUTTON7MOUSE Button7 Mouse, MB7.

        ACTIONMOUSE Action Mouse, MBA.

        SELECTMOUSE Select Mouse, MBS.

        PEN Pen.

        ERASER Eraser.

        MOUSEMOVE Mouse Move, MsMov.

        INBETWEEN_MOUSEMOVE In-between Move, MsSubMov.

        TRACKPADPAN Mouse/Trackpad Pan, MsPan.

        TRACKPADZOOM Mouse/Trackpad Zoom, MsZoom.

        MOUSEROTATE Mouse/Trackpad Rotate, MsRot.

        WHEELUPMOUSE Wheel Up, WhUp.

        WHEELDOWNMOUSE Wheel Down, WhDown.

        WHEELINMOUSE Wheel In, WhIn.

        WHEELOUTMOUSE Wheel Out, WhOut.

        EVT_TWEAK_L Tweak Left, TwkL.

        EVT_TWEAK_M Tweak Middle, TwkM.

        EVT_TWEAK_R Tweak Right, TwkR.

        EVT_TWEAK_A Tweak Action, TwkA.

        EVT_TWEAK_S Tweak Select, TwkS.

        A A.

        B B.

        C C.

        D D.

        E E.

        F F.

        G G.

        H H.

        I I.

        J J.

        K K.

        L L.

        M M.

        N N.

        O O.

        P P.

        Q Q.

        R R.

        S S.

        T T.

        U U.

        V V.

        W W.

        X X.

        Y Y.

        Z Z.

        ZERO 0.

        ONE 1.

        TWO 2.

        THREE 3.

        FOUR 4.

        FIVE 5.

        SIX 6.

        SEVEN 7.

        EIGHT 8.

        NINE 9.

        LEFT_CTRL Left Ctrl, CtrlL.

        LEFT_ALT Left Alt, AltL.

        LEFT_SHIFT Left Shift, ShiftL.

        RIGHT_ALT Right Alt, AltR.

        RIGHT_CTRL Right Ctrl, CtrlR.

        RIGHT_SHIFT Right Shift, ShiftR.

        OSKEY OS Key, Cmd.

        GRLESS Grless.

        ESC Esc.

        TAB Tab.

        RET Return, Enter.

        SPACE Spacebar, Space.

        LINE_FEED Line Feed.

        BACK_SPACE Back Space, BkSpace.

        DEL Delete, Del.

        SEMI_COLON ;.

        PERIOD ..

        COMMA ,.

        QUOTE ".

        ACCENT_GRAVE `.

        MINUS -.

        PLUS +.

        SLASH /.

        BACK_SLASH \.

        EQUAL =.

        LEFT_BRACKET [.

        RIGHT_BRACKET ].

        LEFT_ARROW Left Arrow, ←.

        DOWN_ARROW Down Arrow, ↓.

        RIGHT_ARROW Right Arrow, →.

        UP_ARROW Up Arrow, ↑.

        NUMPAD_2 Numpad 2, Pad2.

        NUMPAD_4 Numpad 4, Pad4.

        NUMPAD_6 Numpad 6, Pad6.

        NUMPAD_8 Numpad 8, Pad8.

        NUMPAD_1 Numpad 1, Pad1.

        NUMPAD_3 Numpad 3, Pad3.

        NUMPAD_5 Numpad 5, Pad5.

        NUMPAD_7 Numpad 7, Pad7.

        NUMPAD_9 Numpad 9, Pad9.

        NUMPAD_PERIOD Numpad ., Pad..

        NUMPAD_SLASH Numpad /, Pad/.

        NUMPAD_ASTERIX Numpad *, Pad*.

        NUMPAD_0 Numpad 0, Pad0.

        NUMPAD_MINUS Numpad -, Pad-.

        NUMPAD_ENTER Numpad Enter, PadEnter.

        NUMPAD_PLUS Numpad +, Pad+.

        F1 F1.

        F2 F2.

        F3 F3.

        F4 F4.

        F5 F5.

        F6 F6.

        F7 F7.

        F8 F8.

        F9 F9.

        F10 F10.

        F11 F11.

        F12 F12.

        F13 F13.

        F14 F14.

        F15 F15.

        F16 F16.

        F17 F17.

        F18 F18.

        F19 F19.

        PAUSE Pause.

        INSERT Insert, Ins.

        HOME Home.

        PAGE_UP Page Up, PgUp.

        PAGE_DOWN Page Down, PgDown.

        END End.

        MEDIA_PLAY Media Play/Pause, >/||.

        MEDIA_STOP Media Stop, Stop.

        MEDIA_FIRST Media First, |<<.

        MEDIA_LAST Media Last, >>|.

        TEXTINPUT Text Input, TxtIn.

        WINDOW_DEACTIVATE Window Deactivate.

        TIMER Timer, Tmr.

        TIMER0 Timer 0, Tmr0.

        TIMER1 Timer 1, Tmr1.

        TIMER2 Timer 2, Tmr2.

        TIMER_JOBS Timer Jobs, TmrJob.

        TIMER_AUTOSAVE Timer Autosave, TmrSave.

        TIMER_REPORT Timer Report, TmrReport.

        TIMERREGION Timer Region, TmrReg.

        NDOF_MOTION NDOF Motion, NdofMov.

        NDOF_BUTTON_MENU NDOF Menu, NdofMenu.

        NDOF_BUTTON_FIT NDOF Fit, NdofFit.

        NDOF_BUTTON_TOP NDOF Top, Ndof↑.

        NDOF_BUTTON_BOTTOM NDOF Bottom, Ndof↓.

        NDOF_BUTTON_LEFT NDOF Left, Ndof←.

        NDOF_BUTTON_RIGHT NDOF Right, Ndof→.

        NDOF_BUTTON_FRONT NDOF Front, NdofFront.

        NDOF_BUTTON_BACK NDOF Back, NdofBack.

        NDOF_BUTTON_ISO1 NDOF Isometric 1, NdofIso1.

        NDOF_BUTTON_ISO2 NDOF Isometric 2, NdofIso2.

        NDOF_BUTTON_ROLL_CW NDOF Roll CW, NdofRCW.

        NDOF_BUTTON_ROLL_CCW NDOF Roll CCW, NdofRCCW.

        NDOF_BUTTON_SPIN_CW NDOF Spin CW, NdofSCW.

        NDOF_BUTTON_SPIN_CCW NDOF Spin CCW, NdofSCCW.

        NDOF_BUTTON_TILT_CW NDOF Tilt CW, NdofTCW.

        NDOF_BUTTON_TILT_CCW NDOF Tilt CCW, NdofTCCW.

        NDOF_BUTTON_ROTATE NDOF Rotate, NdofRot.

        NDOF_BUTTON_PANZOOM NDOF Pan/Zoom, NdofPanZoom.

        NDOF_BUTTON_DOMINANT NDOF Dominant, NdofDom.

        NDOF_BUTTON_PLUS NDOF Plus, Ndof+.

        NDOF_BUTTON_MINUS NDOF Minus, Ndof-.

        NDOF_BUTTON_ESC NDOF Esc, NdofEsc.

        NDOF_BUTTON_ALT NDOF Alt, NdofAlt.

        NDOF_BUTTON_SHIFT NDOF Shift, NdofShift.

        NDOF_BUTTON_CTRL NDOF Ctrl, NdofCtrl.

        NDOF_BUTTON_1 NDOF Button 1, NdofB1.

        NDOF_BUTTON_2 NDOF Button 2, NdofB2.

        NDOF_BUTTON_3 NDOF Button 3, NdofB3.

        NDOF_BUTTON_4 NDOF Button 4, NdofB4.

        NDOF_BUTTON_5 NDOF Button 5, NdofB5.

        NDOF_BUTTON_6 NDOF Button 6, NdofB6.

        NDOF_BUTTON_7 NDOF Button 7, NdofB7.

        NDOF_BUTTON_8 NDOF Button 8, NdofB8.

        NDOF_BUTTON_9 NDOF Button 9, NdofB9.

        NDOF_BUTTON_10 NDOF Button 10, NdofB10.

        NDOF_BUTTON_A NDOF Button A, NdofBA.

        NDOF_BUTTON_B NDOF Button B, NdofBB.

        NDOF_BUTTON_C NDOF Button C, NdofBC.
                :type key_modifier: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C'] | None
                :return: Item, Added key map item
                :rtype: KeyMapItem
        """

    def remove(self, item: KeyMapItem):
        """remove

        :param item: Item
        :type item: KeyMapItem
        """

    def from_id(self, id: int | None) -> KeyMapItem:
        """from_id

        :param id: id, ID of the item
        :type id: int | None
        :return: Item
        :rtype: KeyMapItem
        """

class KeyMaps(bpy_prop_collection[KeyMap], bpy_struct):
    """Collection of keymaps"""

    def new(
        self,
        name: str,
        space_type: typing.Literal[
            "EMPTY",
            "VIEW_3D",
            "TIMELINE",
            "GRAPH_EDITOR",
            "DOPESHEET_EDITOR",
            "NLA_EDITOR",
            "IMAGE_EDITOR",
            "SEQUENCE_EDITOR",
            "CLIP_EDITOR",
            "TEXT_EDITOR",
            "NODE_EDITOR",
            "LOGIC_EDITOR",
            "PROPERTIES",
            "OUTLINER",
            "USER_PREFERENCES",
            "INFO",
            "FILE_BROWSER",
            "CONSOLE",
        ]
        | None = "EMPTY",
        region_type: typing.Literal[
            "WINDOW",
            "HEADER",
            "CHANNELS",
            "TEMPORARY",
            "UI",
            "TOOLS",
            "TOOL_PROPS",
            "PREVIEW",
        ]
        | None = "WINDOW",
        modal: bool | None = False,
    ) -> KeyMap:
        """new

                :param name: Name
                :type name: str
                :param space_type: Space Type

        EMPTY Empty.

        VIEW_3D 3D View, 3D viewport.

        TIMELINE Timeline, Timeline and playback controls.

        GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.

        DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.

        NLA_EDITOR NLA Editor, Combine and layer Actions.

        IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.

        SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.

        CLIP_EDITOR Movie Clip Editor, Motion tracking tools.

        TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.

        NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.

        LOGIC_EDITOR Logic Editor, Game logic editing.

        PROPERTIES Properties, Edit properties of active object and related data-blocks.

        OUTLINER Outliner, Overview of scene graph and all available data-blocks.

        USER_PREFERENCES User Preferences, Edit persistent configuration settings.

        INFO Info, Main menu bar and list of error messages (drag down to expand and display).

        FILE_BROWSER File Browser, Browse for files and assets.

        CONSOLE Python Console, Interactive programmatic console for advanced editing and script development.
                :type space_type: typing.Literal['EMPTY','VIEW_3D','TIMELINE','GRAPH_EDITOR','DOPESHEET_EDITOR','NLA_EDITOR','IMAGE_EDITOR','SEQUENCE_EDITOR','CLIP_EDITOR','TEXT_EDITOR','NODE_EDITOR','LOGIC_EDITOR','PROPERTIES','OUTLINER','USER_PREFERENCES','INFO','FILE_BROWSER','CONSOLE'] | None
                :param region_type: Region Type
                :type region_type: typing.Literal['WINDOW','HEADER','CHANNELS','TEMPORARY','UI','TOOLS','TOOL_PROPS','PREVIEW'] | None
                :param modal: Modal
                :type modal: bool | None
                :return: Key Map, Added key map
                :rtype: KeyMap
        """

    def remove(self, keymap: KeyMap):
        """remove

        :param keymap: Key Map, Removed key map
        :type keymap: KeyMap
        """

    def find(
        self,
        name: str,
        space_type: typing.Literal[
            "EMPTY",
            "VIEW_3D",
            "TIMELINE",
            "GRAPH_EDITOR",
            "DOPESHEET_EDITOR",
            "NLA_EDITOR",
            "IMAGE_EDITOR",
            "SEQUENCE_EDITOR",
            "CLIP_EDITOR",
            "TEXT_EDITOR",
            "NODE_EDITOR",
            "LOGIC_EDITOR",
            "PROPERTIES",
            "OUTLINER",
            "USER_PREFERENCES",
            "INFO",
            "FILE_BROWSER",
            "CONSOLE",
        ]
        | None = "EMPTY",
        region_type: typing.Literal[
            "WINDOW",
            "HEADER",
            "CHANNELS",
            "TEMPORARY",
            "UI",
            "TOOLS",
            "TOOL_PROPS",
            "PREVIEW",
        ]
        | None = "WINDOW",
    ) -> KeyMap:
        """find

                :param name: Name
                :type name: str
                :param space_type: Space Type

        EMPTY Empty.

        VIEW_3D 3D View, 3D viewport.

        TIMELINE Timeline, Timeline and playback controls.

        GRAPH_EDITOR Graph Editor, Edit drivers and keyframe interpolation.

        DOPESHEET_EDITOR Dope Sheet, Adjust timing of keyframes.

        NLA_EDITOR NLA Editor, Combine and layer Actions.

        IMAGE_EDITOR UV/Image Editor, View and edit images and UV Maps.

        SEQUENCE_EDITOR Video Sequence Editor, Video editing tools.

        CLIP_EDITOR Movie Clip Editor, Motion tracking tools.

        TEXT_EDITOR Text Editor, Edit scripts and in-file documentation.

        NODE_EDITOR Node Editor, Editor for node-based shading and compositing tools.

        LOGIC_EDITOR Logic Editor, Game logic editing.

        PROPERTIES Properties, Edit properties of active object and related data-blocks.

        OUTLINER Outliner, Overview of scene graph and all available data-blocks.

        USER_PREFERENCES User Preferences, Edit persistent configuration settings.

        INFO Info, Main menu bar and list of error messages (drag down to expand and display).

        FILE_BROWSER File Browser, Browse for files and assets.

        CONSOLE Python Console, Interactive programmatic console for advanced editing and script development.
                :type space_type: typing.Literal['EMPTY','VIEW_3D','TIMELINE','GRAPH_EDITOR','DOPESHEET_EDITOR','NLA_EDITOR','IMAGE_EDITOR','SEQUENCE_EDITOR','CLIP_EDITOR','TEXT_EDITOR','NODE_EDITOR','LOGIC_EDITOR','PROPERTIES','OUTLINER','USER_PREFERENCES','INFO','FILE_BROWSER','CONSOLE'] | None
                :param region_type: Region Type
                :type region_type: typing.Literal['WINDOW','HEADER','CHANNELS','TEMPORARY','UI','TOOLS','TOOL_PROPS','PREVIEW'] | None
                :return: Key Map, Corresponding key map
                :rtype: KeyMap
        """

    def find_modal(self, name: str) -> KeyMap:
        """find_modal

        :param name: Operator Name
        :type name: str
        :return: Key Map, Corresponding key map
        :rtype: KeyMap
        """

class KeyingSetPaths(bpy_prop_collection[KeyingSetPath], bpy_struct):
    """Collection of keying set paths"""

    active: KeyingSetPath | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSetPath | None
    """

    active_index: int | None
    """ Current Keying Set index

    :type: int | None
    """

    def add(
        self,
        target_id: ID | None,
        data_path: str,
        index: int | None = -1,
        group_method: typing.Literal["NAMED", "NONE", "KEYINGSET"] | None = "KEYINGSET",
        group_name: str = "",
    ) -> KeyingSetPath:
        """Add a new path for the Keying Set

        :param target_id: Target ID, ID-Datablock for the destination
        :type target_id: ID | None
        :param data_path: Data-Path, RNA-Path to destination property
        :type data_path: str
        :param index: Index, The index of the destination property (i.e. axis of Location/Rotation/etc.), or -1 for the entire array
        :type index: int | None
        :param group_method: Grouping Method, Method used to define which Group-name to use
        :type group_method: typing.Literal['NAMED','NONE','KEYINGSET'] | None
        :param group_name: Group Name, Name of Action Group to assign destination to (only if grouping mode is to use this name)
        :type group_name: str
        :return: New Path, Path created and added to the Keying Set
        :rtype: KeyingSetPath
        """

    def remove(self, path: KeyingSetPath):
        """Remove the given path from the Keying Set

        :param path: Path
        :type path: KeyingSetPath
        """

    def clear(self):
        """Remove all the paths from the Keying Set"""

class KeyingSets(bpy_prop_collection[KeyingSet], bpy_struct):
    """Scene keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

    def new(self, idname: str = "KeyingSet", name: str = "KeyingSet") -> KeyingSet:
        """Add a new Keying Set to Scene

        :param idname: IDName, Internal identifier of Keying Set
        :type idname: str
        :param name: Name, User visible name of Keying Set
        :type name: str
        :return: Newly created Keying Set
        :rtype: KeyingSet
        """

class KeyingSetsAll(bpy_prop_collection[KeyingSet], bpy_struct):
    """All available keying sets"""

    active: KeyingSet | None
    """ Active Keying Set used to insert/delete keyframes

    :type: KeyingSet | None
    """

    active_index: int | None
    """ Current Keying Set index (negative for 'builtin' and positive for 'absolute')

    :type: int | None
    """

class LampTextureSlots(bpy_prop_collection[LampTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LampTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LampTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> LampTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LampTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

class LineStyleAlphaModifiers(bpy_prop_collection[LineStyleAlphaModifier], bpy_struct):
    """Alpha modifiers for changing line alphas"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "ALONG_STROKE",
            "CREASE_ANGLE",
            "CURVATURE_3D",
            "DISTANCE_FROM_CAMERA",
            "DISTANCE_FROM_OBJECT",
            "MATERIAL",
            "NOISE",
            "TANGENT",
        ]
        | None,
    ) -> LineStyleAlphaModifier:
        """Add a alpha modifier to line style

        :param name: New name for the alpha modifier (not unique)
        :type name: str
        :param type: Alpha modifier type to add
        :type type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT'] | None
        :return: Newly added alpha modifier
        :rtype: LineStyleAlphaModifier
        """

    def remove(self, modifier: LineStyleAlphaModifier):
        """Remove a alpha modifier from line style

        :param modifier: Alpha modifier to remove
        :type modifier: LineStyleAlphaModifier
        """

class LineStyleColorModifiers(bpy_prop_collection[LineStyleColorModifier], bpy_struct):
    """Color modifiers for changing line colors"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "ALONG_STROKE",
            "CREASE_ANGLE",
            "CURVATURE_3D",
            "DISTANCE_FROM_CAMERA",
            "DISTANCE_FROM_OBJECT",
            "MATERIAL",
            "NOISE",
            "TANGENT",
        ]
        | None,
    ) -> LineStyleColorModifier:
        """Add a color modifier to line style

        :param name: New name for the color modifier (not unique)
        :type name: str
        :param type: Color modifier type to add
        :type type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT'] | None
        :return: Newly added color modifier
        :rtype: LineStyleColorModifier
        """

    def remove(self, modifier: LineStyleColorModifier):
        """Remove a color modifier from line style

        :param modifier: Color modifier to remove
        :type modifier: LineStyleColorModifier
        """

class LineStyleGeometryModifiers(
    bpy_prop_collection[LineStyleGeometryModifier], bpy_struct
):
    """Geometry modifiers for changing line geometries"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "2D_OFFSET",
            "2D_TRANSFORM",
            "BACKBONE_STRETCHER",
            "BEZIER_CURVE",
            "BLUEPRINT",
            "GUIDING_LINES",
            "PERLIN_NOISE_1D",
            "PERLIN_NOISE_2D",
            "POLYGONIZATION",
            "SAMPLING",
            "SIMPLIFICATION",
            "SINUS_DISPLACEMENT",
            "SPATIAL_NOISE",
            "TIP_REMOVER",
        ]
        | None,
    ) -> LineStyleGeometryModifier:
        """Add a geometry modifier to line style

        :param name: New name for the geometry modifier (not unique)
        :type name: str
        :param type: Geometry modifier type to add
        :type type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER'] | None
        :return: Newly added geometry modifier
        :rtype: LineStyleGeometryModifier
        """

    def remove(self, modifier: LineStyleGeometryModifier):
        """Remove a geometry modifier from line style

        :param modifier: Geometry modifier to remove
        :type modifier: LineStyleGeometryModifier
        """

class LineStyleTextureSlots(bpy_prop_collection[LineStyleTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> LineStyleTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> LineStyleTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: LineStyleTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

class LineStyleThicknessModifiers(
    bpy_prop_collection[LineStyleThicknessModifier], bpy_struct
):
    """Thickness modifiers for changing line thickness"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "ALONG_STROKE",
            "CALLIGRAPHY",
            "CREASE_ANGLE",
            "CURVATURE_3D",
            "DISTANCE_FROM_CAMERA",
            "DISTANCE_FROM_OBJECT",
            "MATERIAL",
            "NOISE",
            "TANGENT",
        ]
        | None,
    ) -> LineStyleThicknessModifier:
        """Add a thickness modifier to line style

        :param name: New name for the thickness modifier (not unique)
        :type name: str
        :param type: Thickness modifier type to add
        :type type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT'] | None
        :return: Newly added thickness modifier
        :rtype: LineStyleThicknessModifier
        """

    def remove(self, modifier: LineStyleThicknessModifier):
        """Remove a thickness modifier from line style

        :param modifier: Thickness modifier to remove
        :type modifier: LineStyleThicknessModifier
        """

class Linesets(bpy_prop_collection[FreestyleLineSet], bpy_struct):
    """Line sets for associating lines and style parameters"""

    active: FreestyleLineSet | None
    """ Active line set being displayed

    :type: FreestyleLineSet | None
    """

    active_index: int | None
    """ Index of active line set slot

    :type: int | None
    """

    def new(self, name: str) -> FreestyleLineSet:
        """Add a line set to scene render layer Freestyle settings

        :param name: New name for the line set (not unique)
        :type name: str
        :return: Newly created line set
        :rtype: FreestyleLineSet
        """

    def remove(self, lineset: FreestyleLineSet):
        """Remove a line set from scene render layer Freestyle settings

        :param lineset: Line set to remove
        :type lineset: FreestyleLineSet
        """

class LoopColors(bpy_prop_collection[MeshLoopColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshLoopColorLayer | None
    """ Active vertex color layer

    :type: MeshLoopColorLayer | None
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(self, name: str = "Col") -> MeshLoopColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str
        :return: The newly created layer
        :rtype: MeshLoopColorLayer
        """

    def remove(self, layer: MeshLoopColorLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshLoopColorLayer
        """

class MaskLayers(bpy_prop_collection[MaskLayer], bpy_struct):
    """Collection of layers used by mask"""

    active: MaskLayer | None
    """ Active layer in this mask

    :type: MaskLayer | None
    """

    def new(self, name: str = "") -> MaskLayer:
        """Add layer to this mask

        :param name: Name, Name of new layer
        :type name: str
        :return: New mask layer
        :rtype: MaskLayer
        """

    def remove(self, layer: MaskLayer):
        """Remove layer from this mask

        :param layer: Shape to be removed
        :type layer: MaskLayer
        """

    def clear(self):
        """Remove all mask layers"""

class MaskSplinePoints(bpy_prop_collection[MaskSplinePoint], bpy_struct):
    """Collection of masking spline points"""

    def add(self, count: int | None = 1):
        """Add a number of point to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

    def remove(self, point: MaskSplinePoint):
        """Remove a point from a spline

        :param point: The point to remove
        :type point: MaskSplinePoint
        """

class MaskSplines(bpy_prop_collection[MaskSpline], bpy_struct):
    """Collection of masking splines"""

    active: MaskSpline | None
    """ Active spline of masking layer

    :type: MaskSpline | None
    """

    active_point: MaskSplinePoint | None
    """ Active spline of masking layer

    :type: MaskSplinePoint | None
    """

    def new(self) -> MaskSpline:
        """Add a new spline to the layer

        :return: The newly created spline
        :rtype: MaskSpline
        """

    def remove(self, spline: MaskSpline):
        """Remove a spline from a layer

        :param spline: The spline to remove
        :type spline: MaskSpline
        """

class MaterialTextureSlots(bpy_prop_collection[MaterialTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> MaterialTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: MaterialTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> MaterialTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: MaterialTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

class MeshEdges(bpy_prop_collection[MeshEdge], bpy_struct):
    """Collection of mesh edges"""

    def add(self, count: int | None = 0):
        """add

        :param count: Count, Number of edges to add
        :type count: int | None
        """

class MeshLoops(bpy_prop_collection[MeshLoop], bpy_struct):
    """Collection of mesh loops"""

    def add(self, count: int | None = 0):
        """add

        :param count: Count, Number of loops to add
        :type count: int | None
        """

class MeshPolygons(bpy_prop_collection[MeshPolygon], bpy_struct):
    """Collection of mesh polygons"""

    active: int | None
    """ The active polygon for this mesh

    :type: int | None
    """

    def add(self, count: int | None = 0):
        """add

        :param count: Count, Number of polygons to add
        :type count: int | None
        """

class MeshTessFaces(bpy_prop_collection[MeshTessFace], bpy_struct):
    """Collection of mesh faces"""

    active: int | None
    """ The active face for this mesh

    :type: int | None
    """

    def add(self, count: int | None = 0):
        """add

        :param count: Count, Number of faces to add
        :type count: int | None
        """

class MeshVertices(bpy_prop_collection[MeshVertex], bpy_struct):
    """Collection of mesh vertices"""

    def add(self, count: int | None = 0):
        """add

        :param count: Count, Number of vertices to add
        :type count: int | None
        """

class MetaBallElements(bpy_prop_collection[MetaElement], bpy_struct):
    """Collection of metaball elements"""

    active: MetaElement | None
    """ Last selected element

    :type: MetaElement | None
    """

    def new(
        self,
        type: typing.Literal["BALL", "CAPSULE", "PLANE", "ELLIPSOID", "CUBE"]
        | None = "BALL",
    ) -> MetaElement:
        """Add a new element to the metaball

        :param type: type for the new meta-element
        :type type: typing.Literal['BALL','CAPSULE','PLANE','ELLIPSOID','CUBE'] | None
        :return: The newly created meta-element
        :rtype: MetaElement
        """

    def remove(self, element: MetaElement):
        """Remove an element from the metaball

        :param element: The element to remove
        :type element: MetaElement
        """

    def clear(self):
        """Remove all elements from the metaball"""

class MovieTrackingMarkers(bpy_prop_collection[MovieTrackingMarker], bpy_struct):
    """Collection of markers for movie tracking track"""

    def find_frame(
        self, frame: int | None, exact: bool | None = True
    ) -> MovieTrackingMarker:
        """Get marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get marker at exact frame number rather than get estimated marker
        :type exact: bool | None
        :return: Marker for specified frame
        :rtype: MovieTrackingMarker
        """

    def insert_frame(
        self, frame: int | None, co: collections.abc.Iterable[float] | None = (0.0, 0.0)
    ) -> MovieTrackingMarker:
        """Insert a new marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :param co: Coordinate, Place new marker at the given frame using specified in normalized space coordinates
        :type co: collections.abc.Iterable[float] | None
        :return: Newly created marker
        :rtype: MovieTrackingMarker
        """

    def delete_frame(self, frame: int | None):
        """Delete marker at specified frame

        :param frame: Frame, Frame number to delete marker from
        :type frame: int | None
        """

class MovieTrackingObjectPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of tracking plane tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

class MovieTrackingObjectTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(self, name: str = "", frame: int | None = 1) -> MovieTrackingTrack:
        """create new motion track in this movie clip

        :param name: Name of new track
        :type name: str
        :param frame: Frame, Frame number to add tracks on
        :type frame: int | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """

class MovieTrackingObjects(bpy_prop_collection[MovieTrackingObject], bpy_struct):
    """Collection of movie tracking objects"""

    active: MovieTrackingObject | None
    """ Active object in this tracking data object

    :type: MovieTrackingObject | None
    """

    def new(self, name: str) -> MovieTrackingObject:
        """Add tracking object to this movie clip

        :param name: Name of new object
        :type name: str
        :return: New motion tracking object
        :rtype: MovieTrackingObject
        """

    def remove(self, object: MovieTrackingObject):
        """Remove tracking object from this movie clip

        :param object: Motion tracking object to be removed
        :type object: MovieTrackingObject
        """

class MovieTrackingPlaneMarkers(
    bpy_prop_collection[MovieTrackingPlaneMarker], bpy_struct
):
    """Collection of markers for movie tracking plane track"""

    def find_frame(
        self, frame: int | None, exact: bool | None = True
    ) -> MovieTrackingPlaneMarker:
        """Get plane marker for specified frame

        :param frame: Frame, Frame number to find marker for
        :type frame: int | None
        :param exact: Exact, Get plane marker at exact frame number rather than get estimated marker
        :type exact: bool | None
        :return: Plane marker for specified frame
        :rtype: MovieTrackingPlaneMarker
        """

    def insert_frame(self, frame: int | None) -> MovieTrackingPlaneMarker:
        """Insert a new plane marker at the specified frame

        :param frame: Frame, Frame number to insert marker to
        :type frame: int | None
        :return: Newly created plane marker
        :rtype: MovieTrackingPlaneMarker
        """

    def delete_frame(self, frame: int | None):
        """Delete plane marker at specified frame

        :param frame: Frame, Frame number to delete plane marker from
        :type frame: int | None
        """

class MovieTrackingPlaneTracks(
    bpy_prop_collection[MovieTrackingPlaneTrack], bpy_struct
):
    """Collection of movie tracking plane tracks"""

    active: MovieTrackingPlaneTrack | None
    """ Active plane track in this tracking data object

    :type: MovieTrackingPlaneTrack | None
    """

class MovieTrackingReconstructedCameras(
    bpy_prop_collection[MovieReconstructedCamera], bpy_struct
):
    """Collection of solved cameras"""

    def find_frame(self, frame: int | None = 1) -> MovieReconstructedCamera:
        """Find a reconstructed camera for a give frame number

        :param frame: Frame, Frame number to find camera for
        :type frame: int | None
        :return: Camera for a given frame
        :rtype: MovieReconstructedCamera
        """

    def matrix_from_frame(self, frame: int | None = 1) -> bpy_prop_array[float]:
        """Return interpolated camera matrix for a given frame

        :param frame: Frame, Frame number to find camera for
        :type frame: int | None
        :return: Matrix, Interpolated camera matrix for a given frame
        :rtype: bpy_prop_array[float]
        """

class MovieTrackingTracks(bpy_prop_collection[MovieTrackingTrack], bpy_struct):
    """Collection of movie tracking tracks"""

    active: MovieTrackingTrack | None
    """ Active track in this tracking data object

    :type: MovieTrackingTrack | None
    """

    def new(self, name: str = "", frame: int | None = 1) -> MovieTrackingTrack:
        """Create new motion track in this movie clip

        :param name: Name of new track
        :type name: str
        :param frame: Frame, Frame number to add track on
        :type frame: int | None
        :return: Newly created track
        :rtype: MovieTrackingTrack
        """

class NlaStripFCurves(bpy_prop_collection[FCurve], bpy_struct):
    """Collection of NLA strip F-Curves"""

    def find(self, data_path: str, index: int | None = 0) -> FCurve:
        """Find an F-Curve. Note that this function performs a linear scan of all F-Curves in the NLA strip.

        :param data_path: Data Path, F-Curve data path
        :type data_path: str
        :param index: Index, Array index
        :type index: int | None
        :return: The found F-Curve, or None if it doesn't exist
        :rtype: FCurve
        """

class NlaStrips(bpy_prop_collection[NlaStrip], bpy_struct):
    """Collection of Nla Strips"""

    def new(self, name: str, start: int | None, action: Action) -> NlaStrip:
        """Add a new Action-Clip strip to the track

        :param name: Name for the NLA Strips
        :type name: str
        :param start: Start Frame, Start frame for this strip
        :type start: int | None
        :param action: Action to assign to this strip
        :type action: Action
        :return: New NLA Strip
        :rtype: NlaStrip
        """

    def remove(self, strip: NlaStrip):
        """Remove a NLA Strip

        :param strip: NLA Strip to remove
        :type strip: NlaStrip
        """

class NlaTracks(bpy_prop_collection[NlaTrack], bpy_struct):
    """Collection of NLA Tracks"""

    active: NlaTrack | None
    """ Active Object constraint

    :type: NlaTrack | None
    """

    def new(self, prev: NlaTrack | None = None) -> NlaTrack:
        """Add a new NLA Track

        :param prev: NLA Track to add the new one after
        :type prev: NlaTrack | None
        :return: New NLA Track
        :rtype: NlaTrack
        """

    def remove(self, track: NlaTrack):
        """Remove a NLA Track

        :param track: NLA Track to remove
        :type track: NlaTrack
        """

class NodeInputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(self, type: str, name: str, identifier: str = "") -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all sockets from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class NodeLinks(bpy_prop_collection[NodeLink], bpy_struct):
    """Collection of Node Links"""

    def new(
        self, input: NodeSocket, output: NodeSocket, verify_limits: bool | None = True
    ) -> NodeLink:
        """Add a node link to this node tree

        :param input: The input socket
        :type input: NodeSocket
        :param output: The output socket
        :type output: NodeSocket
        :param verify_limits: Verify Limits, Remove existing links if connection limit is exceeded
        :type verify_limits: bool | None
        :return: New node link
        :rtype: NodeLink
        """

    def remove(self, link: NodeLink):
        """remove a node link from the node tree

        :param link: The node link to remove
        :type link: NodeLink
        """

    def clear(self):
        """remove all node links from the node tree"""

class NodeOutputs(bpy_prop_collection[NodeSocket], bpy_struct):
    """Collection of Node Sockets"""

    def new(self, type: str, name: str, identifier: str = "") -> NodeSocket:
        """Add a socket to this node

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :param identifier: Identifier, Unique socket identifier
        :type identifier: str
        :return: New socket
        :rtype: NodeSocket
        """

    def remove(self, socket: NodeSocket | None):
        """Remove a socket from this node

        :param socket: The socket to remove
        :type socket: NodeSocket | None
        """

    def clear(self):
        """Remove all sockets from this node"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class NodeTreeInputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(self, type: str, name: str) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocketInterface
        """

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """

    def clear(self):
        """Remove all sockets from this node tree"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class NodeTreeOutputs(bpy_prop_collection[NodeSocketInterface], bpy_struct):
    """Collection of Node Tree Sockets"""

    def new(self, type: str, name: str) -> NodeSocketInterface:
        """Add a socket to this node tree

        :param type: Type, Data type
        :type type: str
        :param name: Name
        :type name: str
        :return: New socket
        :rtype: NodeSocketInterface
        """

    def remove(self, socket: NodeSocketInterface | None):
        """Remove a socket from this node tree

        :param socket: The socket to remove
        :type socket: NodeSocketInterface | None
        """

    def clear(self):
        """Remove all sockets from this node tree"""

    def move(self, from_index: int | None, to_index: int | None):
        """Move a socket to another position

        :param from_index: From Index, Index of the socket to move
        :type from_index: int | None
        :param to_index: To Index, Target index for the socket
        :type to_index: int | None
        """

class Nodes(bpy_prop_collection[Node], bpy_struct):
    """Collection of Nodes"""

    active: Node | None
    """ Active node in this tree

    :type: Node | None
    """

    def new(self, type: str) -> Node:
        """Add a node to this node tree

        :param type: Type, Type of node to add (Warning: should be same as node.bl_idname, not node.type!)
        :type type: str
        :return: New node
        :rtype: Node
        """

    def remove(self, node: Node):
        """Remove a node from this node tree

        :param node: The node to remove
        :type node: Node
        """

    def clear(self):
        """Remove all nodes from this node tree"""

class ObjectConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of object constraints"""

    active: Constraint | None
    """ Active Object constraint

    :type: Constraint | None
    """

    def new(
        self,
        type: typing.Literal[
            "CAMERA_SOLVER",
            "FOLLOW_TRACK",
            "OBJECT_SOLVER",
            "COPY_LOCATION",
            "COPY_ROTATION",
            "COPY_SCALE",
            "COPY_TRANSFORMS",
            "LIMIT_DISTANCE",
            "LIMIT_LOCATION",
            "LIMIT_ROTATION",
            "LIMIT_SCALE",
            "MAINTAIN_VOLUME",
            "TRANSFORM",
            "TRANSFORM_CACHE",
            "CLAMP_TO",
            "DAMPED_TRACK",
            "IK",
            "LOCKED_TRACK",
            "SPLINE_IK",
            "STRETCH_TO",
            "TRACK_TO",
            "ACTION",
            "CHILD_OF",
            "FLOOR",
            "FOLLOW_PATH",
            "PIVOT",
            "RIGID_BODY_JOINT",
            "SHRINKWRAP",
        ]
        | None,
    ) -> Constraint:
        """Add a new constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER Camera Solver.

        FOLLOW_TRACK Follow Track.

        OBJECT_SOLVER Object Solver.

        COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.

        LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.

        LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.

        CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.

        IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK Locked Track, Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).

        STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.

        TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.

        ACTION Action, Use transform property of target to look up pose for owner from an Action.

        CHILD_OF Child Of, Make target the 'detachable' parent of owner.

        FLOOR Floor, Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.

        PIVOT Pivot, Change pivot point for transforms (buggy).

        RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).

        SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh.
                :type type: typing.Literal['CAMERA_SOLVER','FOLLOW_TRACK','OBJECT_SOLVER','COPY_LOCATION','COPY_ROTATION','COPY_SCALE','COPY_TRANSFORMS','LIMIT_DISTANCE','LIMIT_LOCATION','LIMIT_ROTATION','LIMIT_SCALE','MAINTAIN_VOLUME','TRANSFORM','TRANSFORM_CACHE','CLAMP_TO','DAMPED_TRACK','IK','LOCKED_TRACK','SPLINE_IK','STRETCH_TO','TRACK_TO','ACTION','CHILD_OF','FLOOR','FOLLOW_PATH','PIVOT','RIGID_BODY_JOINT','SHRINKWRAP'] | None
                :return: New constraint
                :rtype: Constraint
        """

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """

    def clear(self):
        """Remove all constraint from this object"""

class ObjectModifiers(bpy_prop_collection[Modifier], bpy_struct):
    """Collection of object modifiers"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "DATA_TRANSFER",
            "MESH_CACHE",
            "MESH_SEQUENCE_CACHE",
            "NORMAL_EDIT",
            "UV_PROJECT",
            "UV_WARP",
            "VERTEX_WEIGHT_EDIT",
            "VERTEX_WEIGHT_MIX",
            "VERTEX_WEIGHT_PROXIMITY",
            "ARRAY",
            "BEVEL",
            "BOOLEAN",
            "BUILD",
            "DECIMATE",
            "EDGE_SPLIT",
            "MASK",
            "MIRROR",
            "MULTIRES",
            "REMESH",
            "SCREW",
            "SKIN",
            "SOLIDIFY",
            "SUBSURF",
            "TRIANGULATE",
            "WIREFRAME",
            "ARMATURE",
            "CAST",
            "CORRECTIVE_SMOOTH",
            "CURVE",
            "DISPLACE",
            "HOOK",
            "LAPLACIANSMOOTH",
            "LAPLACIANDEFORM",
            "LATTICE",
            "MESH_DEFORM",
            "SHRINKWRAP",
            "SIMPLE_DEFORM",
            "SMOOTH",
            "WARP",
            "WAVE",
            "CLOTH",
            "COLLISION",
            "DYNAMIC_PAINT",
            "EXPLODE",
            "FLUID_SIMULATION",
            "OCEAN",
            "PARTICLE_INSTANCE",
            "PARTICLE_SYSTEM",
            "SMOKE",
            "SOFT_BODY",
            "SURFACE",
        ]
        | None,
    ) -> Modifier:
        """Add a new modifier

                :param name: New name for the modifier
                :type name: str
                :param type: Modifier type to add

        DATA_TRANSFER Data Transfer.

        MESH_CACHE Mesh Cache.

        MESH_SEQUENCE_CACHE Mesh Sequence Cache.

        NORMAL_EDIT Normal Edit.

        UV_PROJECT UV Project.

        UV_WARP UV Warp.

        VERTEX_WEIGHT_EDIT Vertex Weight Edit.

        VERTEX_WEIGHT_MIX Vertex Weight Mix.

        VERTEX_WEIGHT_PROXIMITY Vertex Weight Proximity.

        ARRAY Array.

        BEVEL Bevel.

        BOOLEAN Boolean.

        BUILD Build.

        DECIMATE Decimate.

        EDGE_SPLIT Edge Split.

        MASK Mask.

        MIRROR Mirror.

        MULTIRES Multiresolution.

        REMESH Remesh.

        SCREW Screw.

        SKIN Skin.

        SOLIDIFY Solidify.

        SUBSURF Subdivision Surface.

        TRIANGULATE Triangulate.

        WIREFRAME Wireframe, Generate a wireframe on the edges of a mesh.

        ARMATURE Armature.

        CAST Cast.

        CORRECTIVE_SMOOTH Corrective Smooth.

        CURVE Curve.

        DISPLACE Displace.

        HOOK Hook.

        LAPLACIANSMOOTH Laplacian Smooth.

        LAPLACIANDEFORM Laplacian Deform.

        LATTICE Lattice.

        MESH_DEFORM Mesh Deform.

        SHRINKWRAP Shrinkwrap.

        SIMPLE_DEFORM Simple Deform.

        SMOOTH Smooth.

        WARP Warp.

        WAVE Wave.

        CLOTH Cloth.

        COLLISION Collision.

        DYNAMIC_PAINT Dynamic Paint.

        EXPLODE Explode.

        FLUID_SIMULATION Fluid Simulation.

        OCEAN Ocean.

        PARTICLE_INSTANCE Particle Instance.

        PARTICLE_SYSTEM Particle System.

        SMOKE Smoke.

        SOFT_BODY Soft Body.

        SURFACE Surface.
                :type type: typing.Literal['DATA_TRANSFER','MESH_CACHE','MESH_SEQUENCE_CACHE','NORMAL_EDIT','UV_PROJECT','UV_WARP','VERTEX_WEIGHT_EDIT','VERTEX_WEIGHT_MIX','VERTEX_WEIGHT_PROXIMITY','ARRAY','BEVEL','BOOLEAN','BUILD','DECIMATE','EDGE_SPLIT','MASK','MIRROR','MULTIRES','REMESH','SCREW','SKIN','SOLIDIFY','SUBSURF','TRIANGULATE','WIREFRAME','ARMATURE','CAST','CORRECTIVE_SMOOTH','CURVE','DISPLACE','HOOK','LAPLACIANSMOOTH','LAPLACIANDEFORM','LATTICE','MESH_DEFORM','SHRINKWRAP','SIMPLE_DEFORM','SMOOTH','WARP','WAVE','CLOTH','COLLISION','DYNAMIC_PAINT','EXPLODE','FLUID_SIMULATION','OCEAN','PARTICLE_INSTANCE','PARTICLE_SYSTEM','SMOKE','SOFT_BODY','SURFACE'] | None
                :return: Newly created modifier
                :rtype: Modifier
        """

    def remove(self, modifier: Modifier):
        """Remove an existing modifier from the object

        :param modifier: Modifier to remove
        :type modifier: Modifier
        """

    def clear(self):
        """Remove all modifiers from the object"""

class PaletteColors(bpy_prop_collection[PaletteColor], bpy_struct):
    """Collection of palette colors"""

    active: PaletteColor | None
    """ 

    :type: PaletteColor | None
    """

    def new(self) -> PaletteColor:
        """Add a new color to the palette

        :return: The newly created color
        :rtype: PaletteColor
        """

    def remove(self, color: PaletteColor):
        """Remove a color from the palette

        :param color: The color to remove
        :type color: PaletteColor
        """

    def clear(self):
        """Remove all colors from the palette"""

class ParticleSettingsTextureSlots(
    bpy_prop_collection[ParticleSettingsTextureSlot], bpy_struct
):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> ParticleSettingsTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> ParticleSettingsTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: ParticleSettingsTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

class ParticleSystems(bpy_prop_collection[ParticleSystem], bpy_struct):
    """Collection of particle systems"""

    active: ParticleSystem | None
    """ Active particle system being displayed

    :type: ParticleSystem | None
    """

    active_index: int | None
    """ Index of active particle system slot

    :type: int | None
    """

class PathCompareCollection(bpy_prop_collection[PathCompare], bpy_struct):
    """Collection of paths"""

    @classmethod
    def new(cls) -> PathCompare:
        """Add a new path

        :return:
        :rtype: PathCompare
        """

    @classmethod
    def remove(cls, pathcmp: PathCompare):
        """Remove path

        :param pathcmp:
        :type pathcmp: PathCompare
        """

class PointCaches(bpy_prop_collection[PointCache], bpy_struct):
    """Collection of point caches"""

    active_index: int | None
    """ 

    :type: int | None
    """

class PolygonFloatProperties(
    bpy_prop_collection[MeshPolygonFloatPropertyLayer], bpy_struct
):
    """Collection of float properties"""

    def new(self, name: str = "Float Prop") -> MeshPolygonFloatPropertyLayer:
        """Add a float property layer to Mesh

        :param name: Float property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshPolygonFloatPropertyLayer
        """

class PolygonIntProperties(
    bpy_prop_collection[MeshPolygonIntPropertyLayer], bpy_struct
):
    """Collection of int properties"""

    def new(self, name: str = "Int Prop") -> MeshPolygonIntPropertyLayer:
        """Add a integer property layer to Mesh

        :param name: Int property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshPolygonIntPropertyLayer
        """

class PolygonStringProperties(
    bpy_prop_collection[MeshPolygonStringPropertyLayer], bpy_struct
):
    """Collection of string properties"""

    def new(self, name: str = "String Prop") -> MeshPolygonStringPropertyLayer:
        """Add a string property layer to Mesh

        :param name: String property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshPolygonStringPropertyLayer
        """

class PoseBoneConstraints(bpy_prop_collection[Constraint], bpy_struct):
    """Collection of pose bone constraints"""

    active: Constraint | None
    """ Active PoseChannel constraint

    :type: Constraint | None
    """

    def new(
        self,
        type: typing.Literal[
            "CAMERA_SOLVER",
            "FOLLOW_TRACK",
            "OBJECT_SOLVER",
            "COPY_LOCATION",
            "COPY_ROTATION",
            "COPY_SCALE",
            "COPY_TRANSFORMS",
            "LIMIT_DISTANCE",
            "LIMIT_LOCATION",
            "LIMIT_ROTATION",
            "LIMIT_SCALE",
            "MAINTAIN_VOLUME",
            "TRANSFORM",
            "TRANSFORM_CACHE",
            "CLAMP_TO",
            "DAMPED_TRACK",
            "IK",
            "LOCKED_TRACK",
            "SPLINE_IK",
            "STRETCH_TO",
            "TRACK_TO",
            "ACTION",
            "CHILD_OF",
            "FLOOR",
            "FOLLOW_PATH",
            "PIVOT",
            "RIGID_BODY_JOINT",
            "SHRINKWRAP",
        ]
        | None,
    ) -> Constraint:
        """Add a constraint to this object

                :param type: Constraint type to add

        CAMERA_SOLVER Camera Solver.

        FOLLOW_TRACK Follow Track.

        OBJECT_SOLVER Object Solver.

        COPY_LOCATION Copy Location, Copy the location of a target (with an optional offset), so that they move together.

        COPY_ROTATION Copy Rotation, Copy the rotation of a target (with an optional offset), so that they rotate together.

        COPY_SCALE Copy Scale, Copy the scale factors of a target (with an optional offset), so that they are scaled by the same amount.

        COPY_TRANSFORMS Copy Transforms, Copy all the transformations of a target, so that they move together.

        LIMIT_DISTANCE Limit Distance, Restrict movements to within a certain distance of a target (at the time of constraint evaluation only).

        LIMIT_LOCATION Limit Location, Restrict movement along each axis within given ranges.

        LIMIT_ROTATION Limit Rotation, Restrict rotation along each axis within given ranges.

        LIMIT_SCALE Limit Scale, Restrict scaling along each axis with given ranges.

        MAINTAIN_VOLUME Maintain Volume, Compensate for scaling one axis by applying suitable scaling to the other two axes.

        TRANSFORM Transformation, Use one transform property from target to control another (or same) property on owner.

        TRANSFORM_CACHE Transform Cache, Look up the transformation matrix from an external file.

        CLAMP_TO Clamp To, Restrict movements to lie along a curve by remapping location along curve's longest axis.

        DAMPED_TRACK Damped Track, Point towards a target by performing the smallest rotation necessary.

        IK Inverse Kinematics, Control a chain of bones by specifying the endpoint target (Bones only).

        LOCKED_TRACK Locked Track, Rotate around the specified ('locked') axis to point towards a target.

        SPLINE_IK Spline IK, Align chain of bones along a curve (Bones only).

        STRETCH_TO Stretch To, Stretch along Y-Axis to point towards a target.

        TRACK_TO Track To, Legacy tracking constraint prone to twisting artifacts.

        ACTION Action, Use transform property of target to look up pose for owner from an Action.

        CHILD_OF Child Of, Make target the 'detachable' parent of owner.

        FLOOR Floor, Use position (and optionally rotation) of target to define a 'wall' or 'floor' that the owner can not cross.

        FOLLOW_PATH Follow Path, Use to animate an object/bone following a path.

        PIVOT Pivot, Change pivot point for transforms (buggy).

        RIGID_BODY_JOINT Rigid Body Joint, Use to define a Rigid Body Constraint (for Game Engine use only).

        SHRINKWRAP Shrinkwrap, Restrict movements to surface of target mesh.
                :type type: typing.Literal['CAMERA_SOLVER','FOLLOW_TRACK','OBJECT_SOLVER','COPY_LOCATION','COPY_ROTATION','COPY_SCALE','COPY_TRANSFORMS','LIMIT_DISTANCE','LIMIT_LOCATION','LIMIT_ROTATION','LIMIT_SCALE','MAINTAIN_VOLUME','TRANSFORM','TRANSFORM_CACHE','CLAMP_TO','DAMPED_TRACK','IK','LOCKED_TRACK','SPLINE_IK','STRETCH_TO','TRACK_TO','ACTION','CHILD_OF','FLOOR','FOLLOW_PATH','PIVOT','RIGID_BODY_JOINT','SHRINKWRAP'] | None
                :return: New constraint
                :rtype: Constraint
        """

    def remove(self, constraint: Constraint):
        """Remove a constraint from this object

        :param constraint: Removed constraint
        :type constraint: Constraint
        """

class RenderLayers(bpy_prop_collection[SceneRenderLayer], bpy_struct):
    """Collection of render layers"""

    active: SceneRenderLayer | None
    """ Active Render Layer

    :type: SceneRenderLayer | None
    """

    active_index: int | None
    """ Active index in render layer array

    :type: int | None
    """

    def new(self, name: str) -> SceneRenderLayer:
        """Add a render layer to scene

        :param name: New name for the render layer (not unique)
        :type name: str
        :return: Newly created render layer
        :rtype: SceneRenderLayer
        """

    def remove(self, layer: SceneRenderLayer):
        """Remove a render layer

        :param layer: Render layer to remove
        :type layer: SceneRenderLayer
        """

class RenderPasses(bpy_prop_collection[RenderPass], bpy_struct):
    """Collection of render passes"""

    def find_by_type(
        self,
        pass_type: typing.Literal[
            "COMBINED",
            "Z",
            "COLOR",
            "DIFFUSE",
            "SPECULAR",
            "SHADOW",
            "AO",
            "REFLECTION",
            "NORMAL",
            "VECTOR",
            "REFRACTION",
            "OBJECT_INDEX",
            "UV",
            "MIST",
            "EMIT",
            "ENVIRONMENT",
            "MATERIAL_INDEX",
            "DIFFUSE_DIRECT",
            "DIFFUSE_INDIRECT",
            "DIFFUSE_COLOR",
            "GLOSSY_DIRECT",
            "GLOSSY_INDIRECT",
            "GLOSSY_COLOR",
            "TRANSMISSION_DIRECT",
            "TRANSMISSION_INDIRECT",
            "TRANSMISSION_COLOR",
            "SUBSURFACE_DIRECT",
            "SUBSURFACE_INDIRECT",
            "SUBSURFACE_COLOR",
        ]
        | None,
        view: str,
    ) -> RenderPass:
        """Get the render pass for a given type and view

        :param pass_type: Pass
        :type pass_type: typing.Literal['COMBINED','Z','COLOR','DIFFUSE','SPECULAR','SHADOW','AO','REFLECTION','NORMAL','VECTOR','REFRACTION','OBJECT_INDEX','UV','MIST','EMIT','ENVIRONMENT','MATERIAL_INDEX','DIFFUSE_DIRECT','DIFFUSE_INDIRECT','DIFFUSE_COLOR','GLOSSY_DIRECT','GLOSSY_INDIRECT','GLOSSY_COLOR','TRANSMISSION_DIRECT','TRANSMISSION_INDIRECT','TRANSMISSION_COLOR','SUBSURFACE_DIRECT','SUBSURFACE_INDIRECT','SUBSURFACE_COLOR'] | None
        :param view: View, Render view to get pass from
        :type view: str
        :return: The matching render pass
        :rtype: RenderPass
        """

class RenderSlots(bpy_prop_collection[RenderSlot], bpy_struct):
    """Collection of the render slots"""

    active: RenderSlot | None
    """ Active render slot of the image

    :type: RenderSlot | None
    """

    active_index: int | None
    """ Index of an active render slot of the image

    :type: int | None
    """

class RenderViews(bpy_prop_collection[SceneRenderView], bpy_struct):
    """Collection of render views"""

    active: SceneRenderView | None
    """ Active Render View

    :type: SceneRenderView | None
    """

    active_index: int | None
    """ Active index in render view array

    :type: int | None
    """

    def new(self, name: str) -> SceneRenderView:
        """Add a render view to scene

        :param name: New name for the marker (not unique)
        :type name: str
        :return: Newly created render view
        :rtype: SceneRenderView
        """

    def remove(self, view: SceneRenderView):
        """Remove a render view

        :param view: Render view to remove
        :type view: SceneRenderView
        """

class SceneBases(bpy_prop_collection[ObjectBase], bpy_struct):
    """Collection of scene bases"""

    active: ObjectBase | None
    """ Active object base in the scene

    :type: ObjectBase | None
    """

class SceneObjects(bpy_prop_collection[Object], bpy_struct):
    """Collection of scene objects"""

    active: Object | None
    """ Active object for this scene

    :type: Object | None
    """

    def link(self, object: Object) -> ObjectBase:
        """Link object to scene, run scene.update() after

        :param object: Object to add to scene
        :type object: Object
        :return: The newly created base
        :rtype: ObjectBase
        """

    def unlink(self, object: Object):
        """Unlink object from scene

        :param object: Object to remove from scene
        :type object: Object
        """

class SequenceElements(bpy_prop_collection[SequenceElement], bpy_struct):
    """Collection of SequenceElement"""

    def append(self, filename: str) -> SequenceElement:
        """Push an image from ImageSequence.directory

        :param filename: Filepath to image
        :type filename: str
        :return: New SequenceElement
        :rtype: SequenceElement
        """

    def pop(self, index: int | None):
        """Pop an image off the collection

        :param index: Index of image to remove
        :type index: int | None
        """

class SequenceModifiers(bpy_prop_collection[SequenceModifier], bpy_struct):
    """Collection of strip modifiers"""

    def new(
        self,
        name: str,
        type: typing.Literal[
            "COLOR_BALANCE",
            "CURVES",
            "HUE_CORRECT",
            "BRIGHT_CONTRAST",
            "MASK",
            "WHITE_BALANCE",
            "TONEMAP",
        ]
        | None,
    ) -> SequenceModifier:
        """Add a new modifier

        :param name: New name for the modifier
        :type name: str
        :param type: Modifier type to add
        :type type: typing.Literal['COLOR_BALANCE','CURVES','HUE_CORRECT','BRIGHT_CONTRAST','MASK','WHITE_BALANCE','TONEMAP'] | None
        :return: Newly created modifier
        :rtype: SequenceModifier
        """

    def remove(self, modifier: SequenceModifier):
        """Remove an existing modifier from the sequence

        :param modifier: Modifier to remove
        :type modifier: SequenceModifier
        """

    def clear(self):
        """Remove all modifiers from the sequence"""

class Sequences(bpy_prop_collection[Sequence], bpy_struct):
    """Collection of Sequences"""

    def new_clip(
        self, name: str, clip: MovieClip, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new movie clip sequence

        :param name: Name for the new sequence
        :type name: str
        :param clip: Movie clip to add
        :type clip: MovieClip
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_mask(
        self, name: str, mask: Mask, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new mask sequence

        :param name: Name for the new sequence
        :type name: str
        :param mask: Mask to add
        :type mask: Mask
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_scene(
        self, name: str, scene: Scene, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new scene sequence

        :param name: Name for the new sequence
        :type name: str
        :param scene: Scene to add
        :type scene: Scene
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_image(
        self, name: str, filepath: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new image sequence

        :param name: Name for the new sequence
        :type name: str
        :param filepath: Filepath to image
        :type filepath: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_movie(
        self, name: str, filepath: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new movie sequence

        :param name: Name for the new sequence
        :type name: str
        :param filepath: Filepath to movie
        :type filepath: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_sound(
        self, name: str, filepath: str, channel: int | None, frame_start: int | None
    ) -> Sequence:
        """Add a new sound sequence

        :param name: Name for the new sequence
        :type name: str
        :param filepath: Filepath to movie
        :type filepath: str
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :return: New Sequence
        :rtype: Sequence
        """

    def new_effect(
        self,
        name: str,
        type: typing.Literal[
            "CROSS",
            "ADD",
            "SUBTRACT",
            "ALPHA_OVER",
            "ALPHA_UNDER",
            "GAMMA_CROSS",
            "MULTIPLY",
            "OVER_DROP",
            "WIPE",
            "GLOW",
            "TRANSFORM",
            "COLOR",
            "SPEED",
            "MULTICAM",
            "ADJUSTMENT",
            "GAUSSIAN_BLUR",
            "TEXT",
        ]
        | None,
        channel: int | None,
        frame_start: int | None,
        frame_end: int | None = 0,
        seq1: Sequence | None = None,
        seq2: Sequence | None = None,
        seq3: Sequence | None = None,
    ) -> Sequence:
        """Add a new effect sequence

        :param name: Name for the new sequence
        :type name: str
        :param type: Type, type for the new sequence
        :type type: typing.Literal['CROSS','ADD','SUBTRACT','ALPHA_OVER','ALPHA_UNDER','GAMMA_CROSS','MULTIPLY','OVER_DROP','WIPE','GLOW','TRANSFORM','COLOR','SPEED','MULTICAM','ADJUSTMENT','GAUSSIAN_BLUR','TEXT'] | None
        :param channel: Channel, The channel for the new sequence
        :type channel: int | None
        :param frame_start: The start frame for the new sequence
        :type frame_start: int | None
        :param frame_end: The end frame for the new sequence
        :type frame_end: int | None
        :param seq1: Sequence 1 for effect
        :type seq1: Sequence | None
        :param seq2: Sequence 2 for effect
        :type seq2: Sequence | None
        :param seq3: Sequence 3 for effect
        :type seq3: Sequence | None
        :return: New Sequence
        :rtype: Sequence
        """

    def remove(self, sequence: Sequence):
        """Remove a Sequence

        :param sequence: Sequence to remove
        :type sequence: Sequence
        """

class SpaceNodeEditorPath(bpy_prop_collection[NodeTreePath], bpy_struct):
    """Get the node tree path as a string"""

    to_string: str
    """ 

    :type: str
    """

    def clear(self):
        """Reset the node tree path"""

    def start(self, node_tree: NodeTree | None):
        """Set the root node tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        """

    def append(self, node_tree: NodeTree | None, node: Node | None = None):
        """Append a node group tree to the path

        :param node_tree: Node Tree, Node tree to append to the node editor path
        :type node_tree: NodeTree | None
        :param node: Node, Group node linking to this node tree
        :type node: Node | None
        """

    def pop(self):
        """Remove the last node tree from the path"""

class SplineBezierPoints(bpy_prop_collection[BezierSplinePoint], bpy_struct):
    """Collection of spline Bezier points"""

    def add(self, count: int | None = 1):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

class SplinePoints(bpy_prop_collection[SplinePoint], bpy_struct):
    """Collection of spline points"""

    def add(self, count: int | None = 1):
        """Add a number of points to this spline

        :param count: Number, Number of points to add to the spline
        :type count: int | None
        """

class TessfaceUVTextures(bpy_prop_collection[MeshTextureFaceLayer], bpy_struct):
    """Collection of UV maps for tessellated faces"""

    active: MeshTextureFaceLayer | None
    """ Active UV Map

    :type: MeshTextureFaceLayer | None
    """

    active_index: int | None
    """ Active UV Map index

    :type: int | None
    """

    def new(self, name: str = "UVMap") -> MeshTextureFaceLayer:
        """Add a UV tessface-texture layer to Mesh (only for meshes with no polygons)

        :param name: UV map name
        :type name: str
        :return: The newly created layer
        :rtype: MeshTextureFaceLayer
        """

class TimelineMarkers(bpy_prop_collection[TimelineMarker], bpy_struct):
    """Collection of timeline markers"""

    def new(self, name: str, frame: int | None = 1) -> TimelineMarker:
        """Add a keyframe to the curve

        :param name: New name for the marker (not unique)
        :type name: str
        :param frame: The frame for the new marker
        :type frame: int | None
        :return: Newly created timeline marker
        :rtype: TimelineMarker
        """

    def remove(self, marker: TimelineMarker):
        """Remove a timeline marker

        :param marker: Timeline marker to remove
        :type marker: TimelineMarker
        """

    def clear(self):
        """Remove all timeline markers"""

class UVLoopLayers(bpy_prop_collection[MeshUVLoopLayer], bpy_struct):
    """Collection of uv loop layers"""

    active: MeshUVLoopLayer | None
    """ Active UV loop layer

    :type: MeshUVLoopLayer | None
    """

    active_index: int | None
    """ Active UV loop layer index

    :type: int | None
    """

class UVTextures(bpy_prop_collection[MeshTexturePolyLayer], bpy_struct):
    """Collection of UV maps"""

    active: MeshTexturePolyLayer | None
    """ Active UV Map

    :type: MeshTexturePolyLayer | None
    """

    active_index: int | None
    """ Active UV Map index

    :type: int | None
    """

    def new(self, name: str = "UVMap") -> MeshTexturePolyLayer:
        """Add a UV map layer to Mesh

        :param name: UV map name
        :type name: str
        :return: The newly created layer
        :rtype: MeshTexturePolyLayer
        """

    def remove(self, layer: MeshTexturePolyLayer):
        """Remove a vertex color layer

        :param layer: The layer to remove
        :type layer: MeshTexturePolyLayer
        """

class VertexColors(bpy_prop_collection[MeshColorLayer], bpy_struct):
    """Collection of vertex colors"""

    active: MeshColorLayer | None
    """ Active vertex color layer

    :type: MeshColorLayer | None
    """

    active_index: int | None
    """ Active vertex color index

    :type: int | None
    """

    def new(self, name: str = "Col") -> MeshColorLayer:
        """Add a vertex color layer to Mesh

        :param name: Vertex color name
        :type name: str
        :return: The newly created layer
        :rtype: MeshColorLayer
        """

class VertexFloatProperties(
    bpy_prop_collection[MeshVertexFloatPropertyLayer], bpy_struct
):
    """Collection of float properties"""

    def new(self, name: str = "Float Prop") -> MeshVertexFloatPropertyLayer:
        """Add a float property layer to Mesh

        :param name: Float property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshVertexFloatPropertyLayer
        """

class VertexGroups(bpy_prop_collection[VertexGroup], bpy_struct):
    """Collection of vertex groups"""

    active: VertexGroup | None
    """ Vertex groups of the object

    :type: VertexGroup | None
    """

    active_index: int | None
    """ Active index in vertex group array

    :type: int | None
    """

    def new(self, name: str = "Group") -> VertexGroup:
        """Add vertex group to object

        :param name: Vertex group name
        :type name: str
        :return: New vertex group
        :rtype: VertexGroup
        """

    def remove(self, group: VertexGroup):
        """Delete vertex group from object

        :param group: Vertex group to remove
        :type group: VertexGroup
        """

    def clear(self):
        """Delete all vertex groups from object"""

class VertexIntProperties(bpy_prop_collection[MeshVertexIntPropertyLayer], bpy_struct):
    """Collection of int properties"""

    def new(self, name: str = "Int Prop") -> MeshVertexIntPropertyLayer:
        """Add a integer property layer to Mesh

        :param name: Int property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshVertexIntPropertyLayer
        """

class VertexStringProperties(
    bpy_prop_collection[MeshVertexStringPropertyLayer], bpy_struct
):
    """Collection of string properties"""

    def new(self, name: str = "String Prop") -> MeshVertexStringPropertyLayer:
        """Add a string property layer to Mesh

        :param name: String property name
        :type name: str
        :return: The newly created layer
        :rtype: MeshVertexStringPropertyLayer
        """

class WorldTextureSlots(bpy_prop_collection[WorldTextureSlot], bpy_struct):
    """Collection of texture slots"""

    @classmethod
    def add(cls) -> WorldTextureSlot:
        """add

        :return: The newly initialized mtex
        :rtype: WorldTextureSlot
        """

    @classmethod
    def create(cls, index: int | None) -> WorldTextureSlot:
        """create

        :param index: Index, Slot index to initialize
        :type index: int | None
        :return: The newly initialized mtex
        :rtype: WorldTextureSlot
        """

    @classmethod
    def clear(cls, index: int | None):
        """clear

        :param index: Index, Slot index to clear
        :type index: int | None
        """

class Action(ID, bpy_struct):
    """A collection of F-Curves for animation"""

    fcurves: ActionFCurves
    """ The individual F-Curves that make up the action

    :type: ActionFCurves
    """

    frame_range: bpy_prop_array[float]
    """ The final frame range of all F-Curves within this action

    :type: bpy_prop_array[float]
    """

    groups: ActionGroups
    """ Convenient groupings of F-Curves

    :type: ActionGroups
    """

    id_root: typing.Literal[
        "ACTION",
        "ARMATURE",
        "BRUSH",
        "CAMERA",
        "CACHEFILE",
        "CURVE",
        "FONT",
        "GREASEPENCIL",
        "GROUP",
        "IMAGE",
        "KEY",
        "LAMP",
        "LIBRARY",
        "LINESTYLE",
        "LATTICE",
        "MASK",
        "MATERIAL",
        "META",
        "MESH",
        "MOVIECLIP",
        "NODETREE",
        "OBJECT",
        "PAINTCURVE",
        "PALETTE",
        "PARTICLE",
        "SCENE",
        "SCREEN",
        "SOUND",
        "SPEAKER",
        "TEXT",
        "TEXTURE",
        "WINDOWMANAGER",
        "WORLD",
    ]
    """ Type of ID block that action can be used on - DO NOT CHANGE UNLESS YOU KNOW WHAT YOU ARE DOING

    :type: typing.Literal['ACTION','ARMATURE','BRUSH','CAMERA','CACHEFILE','CURVE','FONT','GREASEPENCIL','GROUP','IMAGE','KEY','LAMP','LIBRARY','LINESTYLE','LATTICE','MASK','MATERIAL','META','MESH','MOVIECLIP','NODETREE','OBJECT','PAINTCURVE','PALETTE','PARTICLE','SCENE','SCREEN','SOUND','SPEAKER','TEXT','TEXTURE','WINDOWMANAGER','WORLD']
    """

    pose_markers: ActionPoseMarkers
    """ Markers specific to this action, for labeling poses

    :type: ActionPoseMarkers
    """

class ActionActuator(Actuator, bpy_struct):
    """Actuator to control the object movement"""

    action: Action | None
    """ 

    :type: Action | None
    """

    apply_to_children: bool
    """ Update Action on all children Objects as well

    :type: bool
    """

    blend_mode: typing.Literal["BLEND", "ADD"]
    """ How this layer is blended with previous layers

    :type: typing.Literal['BLEND','ADD']
    """

    frame_blend_in: int
    """ Number of frames of motion blending

    :type: int
    """

    frame_end: float
    """ 

    :type: float
    """

    frame_property: str
    """ Assign the action's current frame number to this property

    :type: str
    """

    frame_start: float
    """ 

    :type: float
    """

    layer: int
    """ The animation layer to play the action on

    :type: int
    """

    layer_weight: float
    """ How much of the previous layer to blend into this one

    :type: float
    """

    play_mode: typing.Literal[
        "PLAY", "PINGPONG", "FLIPPER", "LOOPSTOP", "LOOPEND", "PROPERTY"
    ]
    """ Action playback type

    :type: typing.Literal['PLAY','PINGPONG','FLIPPER','LOOPSTOP','LOOPEND','PROPERTY']
    """

    priority: int
    """ Execution priority - lower numbers will override actions with higher numbers (with 2 or more actions at once, the overriding channels must be lower in the stack)

    :type: int
    """

    property: str
    """ Use this property to define the Action position

    :type: str
    """

    use_additive: bool
    """ Action is added to the current loc/rot/scale in global or local coordinate according to Local flag

    :type: bool
    """

    use_continue_last_frame: bool
    """ Restore last frame when switching on/off, otherwise play from the start each time

    :type: bool
    """

    use_force: bool
    """ Apply Action as a global or local force depending on the local option (dynamic objects only)

    :type: bool
    """

    use_local: bool
    """ Let the Action act in local coordinates, used in Force and Add mode

    :type: bool
    """

class ActionConstraint(Constraint, bpy_struct):
    """Map an action to the transform axes of a bone"""

    action: Action | None
    """ The constraining action

    :type: Action | None
    """

    frame_end: int
    """ Last frame of the Action to use

    :type: int
    """

    frame_start: int
    """ First frame of the Action to use

    :type: int
    """

    max: float
    """ Maximum value for target channel range

    :type: float
    """

    min: float
    """ Minimum value for target channel range

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    transform_channel: typing.Literal[
        "LOCATION_X",
        "LOCATION_Y",
        "LOCATION_Z",
        "ROTATION_X",
        "ROTATION_Y",
        "ROTATION_Z",
        "SCALE_X",
        "SCALE_Y",
        "SCALE_Z",
    ]
    """ Transformation channel from the target that is used to key the Action

    :type: typing.Literal['LOCATION_X','LOCATION_Y','LOCATION_Z','ROTATION_X','ROTATION_Y','ROTATION_Z','SCALE_X','SCALE_Y','SCALE_Z']
    """

    use_bone_object_action: bool
    """ Bones only: apply the object's transformation channels of the action to the constrained bone, instead of bone's channels

    :type: bool
    """

class ActionGroup(bpy_struct):
    """Groups of F-Curves"""

    channels: bpy_prop_collection[FCurve]
    """ F-Curves in this group

    :type: bpy_prop_collection[FCurve]
    """

    color_set: typing.Literal[
        "DEFAULT",
        "THEME01",
        "THEME02",
        "THEME03",
        "THEME04",
        "THEME05",
        "THEME06",
        "THEME07",
        "THEME08",
        "THEME09",
        "THEME10",
        "THEME11",
        "THEME12",
        "THEME13",
        "THEME14",
        "THEME15",
        "THEME16",
        "THEME17",
        "THEME18",
        "THEME19",
        "THEME20",
        "CUSTOM",
    ]
    """ Custom color set to use

    :type: typing.Literal['DEFAULT','THEME01','THEME02','THEME03','THEME04','THEME05','THEME06','THEME07','THEME08','THEME09','THEME10','THEME11','THEME12','THEME13','THEME14','THEME15','THEME16','THEME17','THEME18','THEME19','THEME20','CUSTOM']
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    lock: bool
    """ Action group is locked

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ Action group is selected

    :type: bool
    """

    show_expanded: bool
    """ Action group is expanded

    :type: bool
    """

class Actuator(bpy_struct):
    """Actuator to apply actions in the game engine"""

    active: bool | None
    """ Set the active state of the actuator

    :type: bool | None
    """

    name: str
    """ 

    :type: str
    """

    pin: bool
    """ Display when not linked to a visible states controller

    :type: bool
    """

    show_expanded: bool
    """ Set actuator expanded in the user interface

    :type: bool
    """

    type: typing.Literal[
        "ACTION",
        "ARMATURE",
        "CAMERA",
        "CONSTRAINT",
        "EDIT_OBJECT",
        "FILTER_2D",
        "GAME",
        "MESSAGE",
        "MOTION",
        "MOUSE",
        "PARENT",
        "PROPERTY",
        "RANDOM",
        "SCENE",
        "SOUND",
        "STATE",
        "STEERING",
        "VISIBILITY",
    ]
    """ 

    :type: typing.Literal['ACTION','ARMATURE','CAMERA','CONSTRAINT','EDIT_OBJECT','FILTER_2D','GAME','MESSAGE','MOTION','MOUSE','PARENT','PROPERTY','RANDOM','SCENE','SOUND','STATE','STEERING','VISIBILITY']
    """

    def link(self, controller: Controller | None):
        """Link the actuator to a controller

        :param controller: Controller to link to
        :type controller: Controller | None
        """

    def unlink(self, controller: Controller | None):
        """Unlink the actuator from a controller

        :param controller: Controller to unlink from
        :type controller: Controller | None
        """

class ActuatorSensor(Sensor, bpy_struct):
    """Sensor to detect state modifications of actuators"""

    actuator: str
    """ Actuator name, actuator active state modifications will be detected

    :type: str
    """

class AddSequence(EffectSequence, Sequence, bpy_struct):
    """Add Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class Addon(bpy_struct):
    """Python add-ons to be loaded automatically"""

    module: str
    """ Module name

    :type: str
    """

    preferences: AddonPreferences
    """ 

    :type: AddonPreferences
    """

class AddonPreferences(bpy_struct):
    bl_idname: str
    """ 

    :type: str
    """

    layout: UILayout
    """ 

    :type: UILayout
    """

class AdjustmentSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip to perform filter adjustments to layers below"""

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    input_count: int
    """ 

    :type: int
    """

class AlembicObjectPath(bpy_struct):
    """Path of an object inside of an Alembic archive"""

    path: str
    """ Object path

    :type: str
    """

class AlphaOverSequence(EffectSequence, Sequence, bpy_struct):
    """Alpha Over Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class AlphaUnderSequence(EffectSequence, Sequence, bpy_struct):
    """Alpha Under Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class AlwaysSensor(Sensor, bpy_struct):
    """Sensor to generate continuous pulses"""

class AndController(Controller, bpy_struct):
    """Controller passing on events based on a logical AND operation"""

class AnimData(bpy_struct):
    """Animation data for data-block"""

    action: Action | None
    """ Active Action for this data-block

    :type: Action | None
    """

    action_blend_type: typing.Literal["REPLACE", "ADD", "SUBTRACT", "MULTIPLY"]
    """ Method used for combining Active Action's result with result of NLA stack

    :type: typing.Literal['REPLACE','ADD','SUBTRACT','MULTIPLY']
    """

    action_extrapolation: typing.Literal["NOTHING", "HOLD", "HOLD_FORWARD"]
    """ Action to take for gaps past the Active Action's range (when evaluating with NLA)

    :type: typing.Literal['NOTHING','HOLD','HOLD_FORWARD']
    """

    action_influence: float
    """ Amount the Active Action contributes to the result of the NLA stack

    :type: float
    """

    drivers: AnimDataDrivers
    """ The Drivers/Expressions for this data-block

    :type: AnimDataDrivers
    """

    nla_tracks: NlaTracks
    """ NLA Tracks (i.e. Animation Layers)

    :type: NlaTracks
    """

    use_nla: bool
    """ NLA stack is evaluated when evaluating this block

    :type: bool
    """

    use_tweak_mode: bool
    """ Whether to enable or disable tweak mode in NLA

    :type: bool
    """

class AnimViz(bpy_struct):
    """Settings for the visualization of motion"""

    motion_path: AnimVizMotionPaths
    """ Motion Path settings for visualization

    :type: AnimVizMotionPaths
    """

    onion_skin_frames: AnimVizOnionSkinning
    """ Onion Skinning (ghosting) settings for visualization

    :type: AnimVizOnionSkinning
    """

class AnimVizMotionPaths(bpy_struct):
    """Motion Path settings for animation visualization"""

    bake_location: typing.Literal["HEADS", "TAILS"]
    """ When calculating Bone Paths, use Head or Tips

    :type: typing.Literal['HEADS','TAILS']
    """

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of paths to display/calculate (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between paths shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    show_frame_numbers: bool
    """ Show frame numbers on Motion Paths

    :type: bool
    """

    show_keyframe_action_all: bool
    """ For bone motion paths, search whole Action for keyframes instead of in group with matching name only (is slower)

    :type: bool
    """

    show_keyframe_highlight: bool
    """ Emphasize position of keyframes on Motion Paths

    :type: bool
    """

    show_keyframe_numbers: bool
    """ Show frame numbers of Keyframes on Motion Paths

    :type: bool
    """

    type: typing.Literal["CURRENT_FRAME", "RANGE"]
    """ Type of range to show for Motion Paths

    :type: typing.Literal['CURRENT_FRAME','RANGE']
    """

class AnimVizOnionSkinning(bpy_struct):
    """Onion Skinning settings for animation visualization"""

    frame_after: int
    """ Number of frames to show after the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_before: int
    """ Number of frames to show before the current frame (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_end: int
    """ End frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_start: int
    """ Starting frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    frame_step: int
    """ Number of frames between ghosts shown (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    show_only_selected: bool
    """ For Pose-Mode drawing, only draw ghosts for selected bones

    :type: bool
    """

    type: typing.Literal["NONE", "CURRENT_FRAME", "RANGE", "KEYS"]
    """ Method used for determining what ghosts get drawn

    :type: typing.Literal['NONE','CURRENT_FRAME','RANGE','KEYS']
    """

class AnyType(bpy_struct):
    """RNA type used for pointers to any possible data"""

class Area(bpy_struct):
    """Area in a subdivided screen, containing an editor"""

    height: int
    """ Area height

    :type: int
    """

    regions: bpy_prop_collection[Region]
    """ Regions this area is subdivided in

    :type: bpy_prop_collection[Region]
    """

    show_menus: bool
    """ Show menus in the header

    :type: bool
    """

    spaces: AreaSpaces
    """ Spaces contained in this area, the first being the active space (NOTE: Useful for example to restore a previously used 3D view space in a certain area to get the old view orientation)

    :type: AreaSpaces
    """

    type: typing.Literal[
        "EMPTY",
        "VIEW_3D",
        "TIMELINE",
        "GRAPH_EDITOR",
        "DOPESHEET_EDITOR",
        "NLA_EDITOR",
        "IMAGE_EDITOR",
        "SEQUENCE_EDITOR",
        "CLIP_EDITOR",
        "TEXT_EDITOR",
        "NODE_EDITOR",
        "LOGIC_EDITOR",
        "PROPERTIES",
        "OUTLINER",
        "USER_PREFERENCES",
        "INFO",
        "FILE_BROWSER",
        "CONSOLE",
    ]
    """ Current editor type for this area

    :type: typing.Literal['EMPTY','VIEW_3D','TIMELINE','GRAPH_EDITOR','DOPESHEET_EDITOR','NLA_EDITOR','IMAGE_EDITOR','SEQUENCE_EDITOR','CLIP_EDITOR','TEXT_EDITOR','NODE_EDITOR','LOGIC_EDITOR','PROPERTIES','OUTLINER','USER_PREFERENCES','INFO','FILE_BROWSER','CONSOLE']
    """

    width: int
    """ Area width

    :type: int
    """

    x: int
    """ The window relative vertical location of the area

    :type: int
    """

    y: int
    """ The window relative horizontal location of the area

    :type: int
    """

    def tag_redraw(self):
        """tag_redraw"""

    def header_text_set(self, text: str = ""):
        """Set the header text

        :param text: Text, New string for the header, no argument clears the text
        :type text: str
        """

class AreaLamp(Lamp, ID, bpy_struct):
    """Directional area lamp"""

    compression_threshold: float
    """ Deep shadow map compression threshold

    :type: float
    """

    gamma: float
    """ Light gamma correction value

    :type: float
    """

    ge_shadow_buffer_type: typing.Literal["SIMPLE", "VARIANCE"]
    """ The shadow mapping algorithm used

    :type: typing.Literal['SIMPLE','VARIANCE']
    """

    shadow_adaptive_threshold: float
    """ Threshold for Adaptive Sampling (Raytraced shadows)

    :type: float
    """

    shadow_buffer_bias: float
    """ Shadow buffer sampling bias

    :type: float
    """

    shadow_buffer_bleed_bias: float
    """ Bias for reducing light-bleed on variance shadow maps

    :type: float
    """

    shadow_buffer_clip_end: float
    """ Shadow map clip end, beyond which objects will not generate shadows

    :type: float
    """

    shadow_buffer_clip_start: float
    """ Shadow map clip start, below which objects will not generate shadows

    :type: float
    """

    shadow_buffer_samples: int
    """ Number of shadow buffer samples

    :type: int
    """

    shadow_buffer_size: int
    """ Resolution of the shadow buffer, higher values give crisper shadows but use more memory

    :type: int
    """

    shadow_buffer_soft: float
    """ Size of shadow buffer sampling area

    :type: float
    """

    shadow_buffer_type: typing.Literal["REGULAR", "HALFWAY", "IRREGULAR", "DEEP"]
    """ Type of shadow buffer

    :type: typing.Literal['REGULAR','HALFWAY','IRREGULAR','DEEP']
    """

    shadow_color: bpy_prop_array[float]
    """ Color of shadows cast by the lamp

    :type: bpy_prop_array[float]
    """

    shadow_filter_type: typing.Literal["BOX", "TENT", "GAUSS"]
    """ Type of shadow filter (Buffer Shadows)

    :type: typing.Literal['BOX','TENT','GAUSS']
    """

    shadow_method: typing.Literal["NOSHADOW", "RAY_SHADOW"]
    """ 

    :type: typing.Literal['NOSHADOW','RAY_SHADOW']
    """

    shadow_ray_sample_method: typing.Literal[
        "ADAPTIVE_QMC", "CONSTANT_QMC", "CONSTANT_JITTERED"
    ]
    """ Method for generating shadow samples: Adaptive QMC is fastest, Constant QMC is less noisy but slower

    :type: typing.Literal['ADAPTIVE_QMC','CONSTANT_QMC','CONSTANT_JITTERED']
    """

    shadow_ray_samples_x: int
    """ Number of samples taken extra (samples x samples)

    :type: int
    """

    shadow_ray_samples_y: int
    """ Number of samples taken extra (samples x samples)

    :type: int
    """

    shadow_sample_buffers: typing.Literal["BUFFERS_1", "BUFFERS_4", "BUFFERS_9"]
    """ Number of shadow buffers to render for better AA, this increases memory usage

    :type: typing.Literal['BUFFERS_1','BUFFERS_4','BUFFERS_9']
    """

    shadow_soft_size: float
    """ Light size for ray shadow sampling (Raytraced shadows)

    :type: float
    """

    shape: typing.Literal["SQUARE", "RECTANGLE"]
    """ Shape of the area lamp

    :type: typing.Literal['SQUARE','RECTANGLE']
    """

    size: float
    """ Size of the area of the area Lamp, X direction size for Rectangle shapes

    :type: float
    """

    size_y: float
    """ Size of the area of the area Lamp in the Y direction for Rectangle shapes

    :type: float
    """

    use_auto_clip_end: bool
    """ Automatic calculation of clipping-end, based on visible vertices

    :type: bool
    """

    use_auto_clip_start: bool
    """ Automatic calculation of clipping-start, based on visible vertices

    :type: bool
    """

    use_dither: bool
    """ Use 2x2 dithering for sampling  (Constant Jittered sampling)

    :type: bool
    """

    use_jitter: bool
    """ Use noise for sampling  (Constant Jittered sampling)

    :type: bool
    """

    use_only_shadow: bool
    """ Cast shadows only, without illuminating objects

    :type: bool
    """

    use_shadow: bool
    """ 

    :type: bool
    """

    use_shadow_layer: bool
    """ Objects on the same layers only cast shadows

    :type: bool
    """

    use_umbra: bool
    """ Emphasize parts that are fully shadowed (Constant Jittered sampling)

    :type: bool
    """

class Armature(ID, bpy_struct):
    """Armature data-block containing a hierarchy of bones, usually used for rigging characters"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bones: ArmatureBones
    """ 

    :type: ArmatureBones
    """

    deform_method: typing.Literal["BLENDER", "BGE_CPU"]
    """ Vertex Deformer Method (Game Engine only)

    :type: typing.Literal['BLENDER','BGE_CPU']
    """

    draw_type: typing.Literal["OCTAHEDRAL", "STICK", "BBONE", "ENVELOPE", "WIRE"]
    """ 

    :type: typing.Literal['OCTAHEDRAL','STICK','BBONE','ENVELOPE','WIRE']
    """

    edit_bones: ArmatureEditBones
    """ 

    :type: ArmatureEditBones
    """

    ghost_frame_end: int
    """ End frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_frame_start: int
    """ Starting frame of range of Ghosts to display (not for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_size: int
    """ Frame step for Ghosts (not for 'On Keyframes' Onion-skinning method)

    :type: int
    """

    ghost_step: int
    """ Number of frame steps on either side of current frame to show as ghosts (only for 'Around Current Frame' Onion-skinning method)

    :type: int
    """

    ghost_type: typing.Literal["CURRENT_FRAME", "RANGE", "KEYS"]
    """ Method of Onion-skinning for active Action

    :type: typing.Literal['CURRENT_FRAME','RANGE','KEYS']
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    layers: bpy_prop_array[bool]
    """ Armature layer visibility

    :type: bpy_prop_array[bool]
    """

    layers_protected: bpy_prop_array[bool]
    """ Protected layers in Proxy Instances are restored to Proxy settings on file reload and undo

    :type: bpy_prop_array[bool]
    """

    pose_position: typing.Literal["POSE", "REST"]
    """ Show armature in binding pose or final posed state

    :type: typing.Literal['POSE','REST']
    """

    show_axes: bool
    """ Draw bone axes

    :type: bool
    """

    show_bone_custom_shapes: bool
    """ Draw bones with their custom shapes

    :type: bool
    """

    show_group_colors: bool
    """ Draw bone group colors

    :type: bool
    """

    show_names: bool
    """ Draw bone names

    :type: bool
    """

    show_only_ghost_selected: bool
    """ 

    :type: bool
    """

    use_auto_ik: bool
    """ Add temporary IK constraints while grabbing bones in Pose Mode

    :type: bool
    """

    use_deform_delay: bool
    """ Don't deform children when manipulating bones in Pose Mode

    :type: bool
    """

    use_mirror_x: bool
    """ Apply changes to matching bone on opposite side of X-Axis

    :type: bool
    """

    def transform(self, matrix: collections.abc.Iterable[float] | None):
        """Transform armature bones by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        """

class ArmatureActuator(Actuator, bpy_struct):
    bone: str
    """ Bone on which the constraint is defined

    :type: str
    """

    constraint: str
    """ Name of the constraint to control

    :type: str
    """

    influence: float
    """ Influence of this constraint

    :type: float
    """

    mode: typing.Literal[
        "RUN", "ENABLE", "DISABLE", "SETTARGET", "SETWEIGHT", "SETINFLUENCE"
    ]
    """ 

    :type: typing.Literal['RUN','ENABLE','DISABLE','SETTARGET','SETWEIGHT','SETINFLUENCE']
    """

    secondary_target: Object | None
    """ Set this object as the secondary target of the constraint (only IK polar target at the moment)

    :type: Object | None
    """

    target: Object | None
    """ Set this object as the target of the constraint

    :type: Object | None
    """

    weight: float
    """ Weight of this constraint

    :type: float
    """

class ArmatureModifier(Modifier, bpy_struct):
    """Armature deformation modifier"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    object: Object | None
    """ Armature object to deform with

    :type: Object | None
    """

    use_bone_envelopes: bool
    """ Bind Bone envelopes to armature modifier

    :type: bool
    """

    use_deform_preserve_volume: bool
    """ Deform rotation interpolation with quaternions

    :type: bool
    """

    use_multi_modifier: bool
    """ Use same input as previous modifier, and mix results using overall vgroup

    :type: bool
    """

    use_vertex_groups: bool
    """ Bind vertex groups to armature modifier

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class ArmatureSensor(Sensor, bpy_struct):
    """Sensor to detect values and changes in values of IK solver"""

    bone: str
    """ Identify the bone to check value from

    :type: str
    """

    constraint: str
    """ Identify the bone constraint to check value from

    :type: str
    """

    test_type: typing.Literal[
        "STATECHG", "LINERRORBELOW", "LINERRORABOVE", "ROTERRORBELOW", "ROTERRORABOVE"
    ]
    """ Type of value and test

    :type: typing.Literal['STATECHG','LINERRORBELOW','LINERRORABOVE','ROTERRORBELOW','ROTERRORABOVE']
    """

    value: float
    """ Value to be used in comparison

    :type: float
    """

class ArrayModifier(Modifier, bpy_struct):
    """Array duplication modifier"""

    constant_offset_displace: bpy_prop_array[float]
    """ Value for the distance between arrayed items

    :type: bpy_prop_array[float]
    """

    count: int
    """ Number of duplicates to make

    :type: int
    """

    curve: Object | None
    """ Curve object to fit array length to

    :type: Object | None
    """

    end_cap: Object | None
    """ Mesh object to use as an end cap

    :type: Object | None
    """

    fit_length: float
    """ Length to fit array within

    :type: float
    """

    fit_type: typing.Literal["FIXED_COUNT", "FIT_LENGTH", "FIT_CURVE"]
    """ Array length calculation method

    :type: typing.Literal['FIXED_COUNT','FIT_LENGTH','FIT_CURVE']
    """

    merge_threshold: float
    """ Limit below which to merge vertices

    :type: float
    """

    offset_object: Object | None
    """ Use the location and rotation of another object to determine the distance and rotational change between arrayed items

    :type: Object | None
    """

    relative_offset_displace: bpy_prop_array[float]
    """ The size of the geometry will determine the distance between arrayed items

    :type: bpy_prop_array[float]
    """

    start_cap: Object | None
    """ Mesh object to use as a start cap

    :type: Object | None
    """

    use_constant_offset: bool
    """ Add a constant offset

    :type: bool
    """

    use_merge_vertices: bool
    """ Merge vertices in adjacent duplicates

    :type: bool
    """

    use_merge_vertices_cap: bool
    """ Merge vertices in first and last duplicates

    :type: bool
    """

    use_object_offset: bool
    """ Add another object's transformation to the total offset

    :type: bool
    """

    use_relative_offset: bool
    """ Add an offset relative to the object's bounding box

    :type: bool
    """

class Available(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """

class BackgroundImage(bpy_struct):
    """Image and settings for display in the 3D View background"""

    clip: MovieClip | None
    """ Movie clip displayed and edited in this space

    :type: MovieClip | None
    """

    clip_user: MovieClipUser
    """ Parameters defining which frame of the movie clip is displayed

    :type: MovieClipUser
    """

    draw_depth: typing.Literal["BACK", "FRONT"]
    """ Draw under or over everything

    :type: typing.Literal['BACK','FRONT']
    """

    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the image fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    image: Image | None
    """ Image displayed and edited in this space

    :type: Image | None
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    offset_x: float
    """ Offset image horizontally from the world origin

    :type: float
    """

    offset_y: float
    """ Offset image vertically from the world origin

    :type: float
    """

    opacity: float
    """ Image opacity to blend the image against the background color

    :type: float
    """

    rotation: float
    """ Rotation for the background image (ortho view only)

    :type: float
    """

    show_background_image: bool
    """ Show this image as background

    :type: bool
    """

    show_expanded: bool
    """ Show the expanded in the user interface

    :type: bool
    """

    show_on_foreground: bool
    """ Show this image in front of objects in viewport

    :type: bool
    """

    size: float
    """ Size of the background image (ortho view only)

    :type: float
    """

    source: typing.Literal["IMAGE", "MOVIE_CLIP"]
    """ Data source used for background

    :type: typing.Literal['IMAGE','MOVIE_CLIP']
    """

    use_camera_clip: bool
    """ Use movie clip from active scene camera

    :type: bool
    """

    use_flip_x: bool
    """ Flip the background image horizontally

    :type: bool
    """

    use_flip_y: bool
    """ Flip the background image vertically

    :type: bool
    """

    view_axis: typing.Literal[
        "LEFT", "RIGHT", "BACK", "FRONT", "BOTTOM", "TOP", "ALL", "CAMERA"
    ]
    """ The axis to display the image on

    :type: typing.Literal['LEFT','RIGHT','BACK','FRONT','BOTTOM','TOP','ALL','CAMERA']
    """

class BakePixel(bpy_struct):
    du_dx: float
    """ 

    :type: float
    """

    du_dy: float
    """ 

    :type: float
    """

    dv_dx: float
    """ 

    :type: float
    """

    dv_dy: float
    """ 

    :type: float
    """

    next: typing_extensions.Self
    """ 

    :type: typing_extensions.Self
    """

    object_id: int
    """ 

    :type: int
    """

    primitive_id: int
    """ 

    :type: int
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class BakeSettings(bpy_struct):
    """Bake data for a Scene data-block"""

    cage_extrusion: float
    """ Distance to use for the inward ray cast when using selected to active

    :type: float
    """

    cage_object: str
    """ Object to use as cage instead of calculating the cage from the active object with cage extrusion

    :type: str
    """

    filepath: str
    """ Image filepath to use when saving externally

    :type: str
    """

    height: int
    """ Vertical dimension of the baking map

    :type: int
    """

    image_settings: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    margin: int
    """ Extends the baked result as a post process filter

    :type: int
    """

    normal_b: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ Axis to bake in blue channel

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    normal_g: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ Axis to bake in green channel

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    normal_r: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ Axis to bake in red channel

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    normal_space: typing.Literal["OBJECT", "TANGENT"]
    """ Choose normal space for baking

    :type: typing.Literal['OBJECT','TANGENT']
    """

    pass_filter: set[str]
    """ Passes to include in the active baking pass

    :type: set[str]
    """

    save_mode: typing.Literal["INTERNAL", "EXTERNAL"]
    """ Choose how to save the baking map

    :type: typing.Literal['INTERNAL','EXTERNAL']
    """

    use_automatic_name: bool
    """ Automatically name the output file with the pass type (external only)

    :type: bool
    """

    use_cage: bool
    """ Cast rays to active object from a cage

    :type: bool
    """

    use_clear: bool
    """ Clear Images before baking (internal only)

    :type: bool
    """

    use_pass_ambient_occlusion: bool
    """ Add ambient occlusion contribution

    :type: bool
    """

    use_pass_color: bool
    """ Color the pass

    :type: bool
    """

    use_pass_diffuse: bool
    """ Add diffuse contribution

    :type: bool
    """

    use_pass_direct: bool
    """ Add direct lighting contribution

    :type: bool
    """

    use_pass_emit: bool
    """ Add emission contribution

    :type: bool
    """

    use_pass_glossy: bool
    """ Add glossy contribution

    :type: bool
    """

    use_pass_indirect: bool
    """ Add indirect lighting contribution

    :type: bool
    """

    use_pass_subsurface: bool
    """ Add subsurface contribution

    :type: bool
    """

    use_pass_transmission: bool
    """ Add transmission contribution

    :type: bool
    """

    use_selected_to_active: bool
    """ Bake shading on the surface of selected objects to the active object

    :type: bool
    """

    use_split_materials: bool
    """ Split external images per material (external only)

    :type: bool
    """

    width: int
    """ Horizontal dimension of the baking map

    :type: int
    """

class BevelModifier(Modifier, bpy_struct):
    """Bevel modifier to make edges and vertices more rounded"""

    angle_limit: float
    """ Angle above which to bevel edges

    :type: float
    """

    edge_weight_method: typing.Literal["AVERAGE", "SHARPEST", "LARGEST"]
    """ What edge weight to use for weighting a vertex

    :type: typing.Literal['AVERAGE','SHARPEST','LARGEST']
    """

    limit_method: typing.Literal["NONE", "ANGLE", "WEIGHT", "VGROUP"]
    """ 

    :type: typing.Literal['NONE','ANGLE','WEIGHT','VGROUP']
    """

    loop_slide: bool
    """ Prefer sliding along edges to having even widths

    :type: bool
    """

    material: int
    """ Material index of generated faces, -1 for automatic

    :type: int
    """

    offset_type: typing.Literal["OFFSET", "WIDTH", "DEPTH", "PERCENT"]
    """ What distance Width measures

    :type: typing.Literal['OFFSET','WIDTH','DEPTH','PERCENT']
    """

    profile: float
    """ The profile shape (0.5 = round)

    :type: float
    """

    segments: int
    """ Number of segments for round edges/verts

    :type: int
    """

    use_clamp_overlap: bool
    """ Clamp the width to avoid overlap

    :type: bool
    """

    use_only_vertices: bool
    """ Bevel verts/corners, not edges

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

    width: float
    """ Bevel value/amount

    :type: float
    """

class BezierSplinePoint(bpy_struct):
    """Bezier curve point with two handles"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: typing.Literal["FREE", "VECTOR", "ALIGNED", "AUTO"]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO']
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: typing.Literal["FREE", "VECTOR", "ALIGNED", "AUTO"]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO']
    """

    hide: bool
    """ Visibility status

    :type: bool
    """

    radius: float
    """ Radius for beveling

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Handle 1 selection status

    :type: bool
    """

    select_right_handle: bool
    """ Handle 2 selection status

    :type: bool
    """

    tilt: float
    """ Tilt in 3D View

    :type: float
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

class BlendData(bpy_struct):
    """Main data structure representing a .blend file and all its datablocks"""

    actions: BlendDataActions
    """ Action datablocks

    :type: BlendDataActions
    """

    armatures: BlendDataArmatures
    """ Armature datablocks

    :type: BlendDataArmatures
    """

    brushes: BlendDataBrushes
    """ Brush datablocks

    :type: BlendDataBrushes
    """

    cache_files: BlendDataCacheFiles
    """ Cache Files datablocks

    :type: BlendDataCacheFiles
    """

    cameras: BlendDataCameras
    """ Camera datablocks

    :type: BlendDataCameras
    """

    curves: BlendDataCurves
    """ Curve datablocks

    :type: BlendDataCurves
    """

    filepath: str
    """ Path to the .blend file

    :type: str
    """

    fonts: BlendDataFonts
    """ Vector font datablocks

    :type: BlendDataFonts
    """

    grease_pencil: BlendDataGreasePencils
    """ Grease Pencil datablocks

    :type: BlendDataGreasePencils
    """

    groups: BlendDataGroups
    """ Group datablocks

    :type: BlendDataGroups
    """

    images: BlendDataImages
    """ Image datablocks

    :type: BlendDataImages
    """

    is_dirty: bool
    """ Have recent edits been saved to disk

    :type: bool
    """

    is_saved: bool
    """ Has the current session been saved to disk as a .blend file

    :type: bool
    """

    lamps: BlendDataLamps
    """ Lamp datablocks

    :type: BlendDataLamps
    """

    lattices: BlendDataLattices
    """ Lattice datablocks

    :type: BlendDataLattices
    """

    libraries: BlendDataLibraries
    """ Library datablocks

    :type: BlendDataLibraries
    """

    linestyles: BlendDataLineStyles
    """ Line Style datablocks

    :type: BlendDataLineStyles
    """

    masks: BlendDataMasks
    """ Masks datablocks

    :type: BlendDataMasks
    """

    materials: BlendDataMaterials
    """ Material datablocks

    :type: BlendDataMaterials
    """

    meshes: BlendDataMeshes
    """ Mesh datablocks

    :type: BlendDataMeshes
    """

    metaballs: BlendDataMetaBalls
    """ Metaball datablocks

    :type: BlendDataMetaBalls
    """

    movieclips: BlendDataMovieClips
    """ Movie Clip datablocks

    :type: BlendDataMovieClips
    """

    node_groups: BlendDataNodeTrees
    """ Node group datablocks

    :type: BlendDataNodeTrees
    """

    objects: BlendDataObjects
    """ Object datablocks

    :type: BlendDataObjects
    """

    paint_curves: BlendDataPaintCurves
    """ Paint Curves datablocks

    :type: BlendDataPaintCurves
    """

    palettes: BlendDataPalettes
    """ Palette datablocks

    :type: BlendDataPalettes
    """

    particles: BlendDataParticles
    """ Particle datablocks

    :type: BlendDataParticles
    """

    scenes: BlendDataScenes
    """ Scene datablocks

    :type: BlendDataScenes
    """

    screens: BlendDataScreens
    """ Screen datablocks

    :type: BlendDataScreens
    """

    shape_keys: bpy_prop_collection[Key]
    """ Shape Key datablocks

    :type: bpy_prop_collection[Key]
    """

    sounds: BlendDataSounds
    """ Sound datablocks

    :type: BlendDataSounds
    """

    speakers: BlendDataSpeakers
    """ Speaker datablocks

    :type: BlendDataSpeakers
    """

    texts: BlendDataTexts
    """ Text datablocks

    :type: BlendDataTexts
    """

    textures: BlendDataTextures
    """ Texture datablocks

    :type: BlendDataTextures
    """

    use_autopack: bool
    """ Automatically pack all external data into .blend file

    :type: bool
    """

    version: bpy_prop_array[int]
    """ Version of Blender the .blend was saved with

    :type: bpy_prop_array[int]
    """

    window_managers: BlendDataWindowManagers
    """ Window manager datablocks

    :type: BlendDataWindowManagers
    """

    worlds: BlendDataWorlds
    """ World datablocks

    :type: BlendDataWorlds
    """

    def user_map(
        self,
        subset: collections.abc.Sequence | None,
        key_types: set[str] | None,
        value_types: set[str] | None,
    ) -> dict:
        """Returns a mapping of all ID datablocks in current bpy.data to a set of all datablocks using them.For list of valid set members for key_types & value_types, see: `bpy.types.KeyingSetPath.id_type`.

        :param subset: When passed, only these data-blocks and their users will be included as keys/values in the map.
        :type subset: collections.abc.Sequence | None
        :param key_types: Filter the keys mapped by ID types.
        :type key_types: set[str] | None
        :param value_types: Filter the values in the set by ID types.
        :type value_types: set[str] | None
        :return: dictionary of `bpy.types.ID` instances, with sets of ID's as their values.
        :rtype: dict
        """

class BlendTexture(Texture, ID, bpy_struct):
    """Procedural color blending texture"""

    progression: typing.Literal[
        "LINEAR",
        "QUADRATIC",
        "EASING",
        "DIAGONAL",
        "SPHERICAL",
        "QUADRATIC_SPHERE",
        "RADIAL",
    ]
    """ Style of the color blending

    :type: typing.Literal['LINEAR','QUADRATIC','EASING','DIAGONAL','SPHERICAL','QUADRATIC_SPHERE','RADIAL']
    """

    use_flip_axis: typing.Literal["HORIZONTAL", "VERTICAL"]
    """ Flip the texture's X and Y axis

    :type: typing.Literal['HORIZONTAL','VERTICAL']
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class BlenderRNA(bpy_struct):
    """Blender RNA structure definitions"""

    structs: bpy_prop_collection[Struct]
    """ 

    :type: bpy_prop_collection[Struct]
    """

    properties: bpy_prop_collection[Property]
    """ 

    :type: bpy_prop_collection[Property]
    """

class BoidRule(bpy_struct):
    name: str
    """ Boid rule name

    :type: str
    """

    type: typing.Literal[
        "GOAL",
        "AVOID",
        "AVOID_COLLISION",
        "SEPARATE",
        "FLOCK",
        "FOLLOW_LEADER",
        "AVERAGE_SPEED",
        "FIGHT",
    ]
    """ 

    :type: typing.Literal['GOAL','AVOID','AVOID_COLLISION','SEPARATE','FLOCK','FOLLOW_LEADER','AVERAGE_SPEED','FIGHT']
    """

    use_in_air: bool
    """ Use rule when boid is flying

    :type: bool
    """

    use_on_land: bool
    """ Use rule when boid is on land

    :type: bool
    """

class BoidRuleAverageSpeed(BoidRule, bpy_struct):
    level: float
    """ How much velocity's z-component is kept constant

    :type: float
    """

    speed: float
    """ Percentage of maximum speed

    :type: float
    """

    wander: float
    """ How fast velocity's direction is randomized

    :type: float
    """

class BoidRuleAvoid(BoidRule, bpy_struct):
    fear_factor: float
    """ Avoid object if danger from it is above this threshold

    :type: float
    """

    object: Object | None
    """ Object to avoid

    :type: Object | None
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

class BoidRuleAvoidCollision(BoidRule, bpy_struct):
    look_ahead: float
    """ Time to look ahead in seconds

    :type: float
    """

    use_avoid: bool
    """ Avoid collision with other boids

    :type: bool
    """

    use_avoid_collision: bool
    """ Avoid collision with deflector objects

    :type: bool
    """

class BoidRuleFight(BoidRule, bpy_struct):
    distance: float
    """ Attack boids at max this distance

    :type: float
    """

    flee_distance: float
    """ Flee to this distance

    :type: float
    """

class BoidRuleFollowLeader(BoidRule, bpy_struct):
    distance: float
    """ Distance behind leader to follow

    :type: float
    """

    object: Object | None
    """ Follow this object instead of a boid

    :type: Object | None
    """

    queue_count: int
    """ How many boids in a line

    :type: int
    """

    use_line: bool
    """ Follow leader in a line

    :type: bool
    """

class BoidRuleGoal(BoidRule, bpy_struct):
    object: Object | None
    """ Goal object

    :type: Object | None
    """

    use_predict: bool
    """ Predict target movement

    :type: bool
    """

class BoidSettings(bpy_struct):
    """Settings for boid physics"""

    accuracy: float
    """ Accuracy of attack

    :type: float
    """

    active_boid_state: BoidRule | None
    """ 

    :type: BoidRule | None
    """

    active_boid_state_index: int | None
    """ 

    :type: int | None
    """

    aggression: float
    """ Boid will fight this times stronger enemy

    :type: float
    """

    air_acc_max: float
    """ Maximum acceleration in air (relative to maximum speed)

    :type: float
    """

    air_ave_max: float
    """ Maximum angular velocity in air (relative to 180 degrees)

    :type: float
    """

    air_personal_space: float
    """ Radius of boids personal space in air (% of particle size)

    :type: float
    """

    air_speed_max: float
    """ Maximum speed in air

    :type: float
    """

    air_speed_min: float
    """ Minimum speed in air (relative to maximum speed)

    :type: float
    """

    bank: float
    """ Amount of rotation around velocity vector on turns

    :type: float
    """

    health: float
    """ Initial boid health when born

    :type: float
    """

    height: float
    """ Boid height relative to particle size

    :type: float
    """

    land_acc_max: float
    """ Maximum acceleration on land (relative to maximum speed)

    :type: float
    """

    land_ave_max: float
    """ Maximum angular velocity on land (relative to 180 degrees)

    :type: float
    """

    land_jump_speed: float
    """ Maximum speed for jumping

    :type: float
    """

    land_personal_space: float
    """ Radius of boids personal space on land (% of particle size)

    :type: float
    """

    land_smooth: float
    """ How smoothly the boids land

    :type: float
    """

    land_speed_max: float
    """ Maximum speed on land

    :type: float
    """

    land_stick_force: float
    """ How strong a force must be to start effecting a boid on land

    :type: float
    """

    pitch: float
    """ Amount of rotation around side vector

    :type: float
    """

    range: float
    """ Maximum distance from which a boid can attack

    :type: float
    """

    states: bpy_prop_collection[BoidState]
    """ 

    :type: bpy_prop_collection[BoidState]
    """

    strength: float
    """ Maximum caused damage on attack per second

    :type: float
    """

    use_climb: bool
    """ Allow boids to climb goal objects

    :type: bool
    """

    use_flight: bool
    """ Allow boids to move in air

    :type: bool
    """

    use_land: bool
    """ Allow boids to move on land

    :type: bool
    """

class BoidState(bpy_struct):
    """Boid state for boid physics"""

    active_boid_rule: BoidRule | None
    """ 

    :type: BoidRule | None
    """

    active_boid_rule_index: int | None
    """ 

    :type: int | None
    """

    falloff: float
    """ 

    :type: float
    """

    name: str
    """ Boid state name

    :type: str
    """

    rule_fuzzy: float
    """ 

    :type: float
    """

    rules: bpy_prop_collection[BoidRule]
    """ 

    :type: bpy_prop_collection[BoidRule]
    """

    ruleset_type: typing.Literal["FUZZY", "RANDOM", "AVERAGE"]
    """ How the rules in the list are evaluated

    :type: typing.Literal['FUZZY','RANDOM','AVERAGE']
    """

    volume: float
    """ 

    :type: float
    """

class Bone(bpy_struct):
    """Bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveiny: float
    """ Y-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveouty: float
    """ Y-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_in: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_out: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: float
    """ Scale factor for start of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_scaleout: float
    """ Scale factor for end of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    children: bpy_prop_collection[Bone]
    """ Bones which are children of this bone

    :type: bpy_prop_collection[Bone]
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone relative to its parent

    :type: bpy_prop_array[float]
    """

    head_local: bpy_prop_array[float]
    """ Location of head end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when it is not in Edit Mode (i.e. in Object or Pose Modes)

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    layers: bpy_prop_array[bool]
    """ Layers bone exists in

    :type: bpy_prop_array[bool]
    """

    matrix: bpy_prop_array[float]
    """ 3x3 bone matrix

    :type: bpy_prop_array[float]
    """

    matrix_local: bpy_prop_array[float]
    """ 4x4 bone matrix relative to armature

    :type: bpy_prop_array[float]
    """

    name: str
    """ 

    :type: str
    """

    parent: typing_extensions.Self
    """ Parent bone (in same Armature)

    :type: typing_extensions.Self
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone

    :type: bpy_prop_array[float]
    """

    tail_local: bpy_prop_array[float]
    """ Location of tail end of the bone relative to armature

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Use Roll Out of parent bone as Roll In of its children

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character
(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.
(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.
(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.
(readonly)"""

    length: typing.Any
    """ The distance from head to tail,
when set the head is moved to fit the length."""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent
(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)
(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.
(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.
(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.
(readonly)"""

    def evaluate_envelope(self, point: collections.abc.Iterable[float] | None) -> float:
        """Calculate bone envelope at given point

        :param point: Point, Position in 3d space to evaluate
        :type point: collections.abc.Iterable[float] | None
        :return: Factor, Envelope factor
        :rtype: float
        """

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """

class BoneGroup(bpy_struct):
    """Groups of Pose Channels (Bones)"""

    color_set: typing.Literal[
        "DEFAULT",
        "THEME01",
        "THEME02",
        "THEME03",
        "THEME04",
        "THEME05",
        "THEME06",
        "THEME07",
        "THEME08",
        "THEME09",
        "THEME10",
        "THEME11",
        "THEME12",
        "THEME13",
        "THEME14",
        "THEME15",
        "THEME16",
        "THEME17",
        "THEME18",
        "THEME19",
        "THEME20",
        "CUSTOM",
    ]
    """ Custom color set to use

    :type: typing.Literal['DEFAULT','THEME01','THEME02','THEME03','THEME04','THEME05','THEME06','THEME07','THEME08','THEME09','THEME10','THEME11','THEME12','THEME13','THEME14','THEME15','THEME16','THEME17','THEME18','THEME19','THEME20','CUSTOM']
    """

    colors: ThemeBoneColorSet
    """ Copy of the colors associated with the group's color set

    :type: ThemeBoneColorSet
    """

    is_custom_color_set: bool
    """ Color set is user-defined instead of a fixed theme color set

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

class BoolProperty(Property, bpy_struct):
    """RNA boolean property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: bool
    """ Default value for this number

    :type: bool
    """

    default_array: bpy_prop_array[bool]
    """ Default value for this array

    :type: bpy_prop_array[bool]
    """

    is_array: bool
    """ 

    :type: bool
    """

class BooleanModifier(Modifier, bpy_struct):
    """Boolean operations modifier"""

    double_threshold: float
    """ Threshold for checking overlapping geometry

    :type: float
    """

    object: Object | None
    """ Mesh object to use for Boolean operation

    :type: Object | None
    """

    operation: typing.Literal["INTERSECT", "UNION", "DIFFERENCE"]
    """ 

    :type: typing.Literal['INTERSECT','UNION','DIFFERENCE']
    """

    solver: typing.Literal["BMESH", "CARVE"]
    """ 

    :type: typing.Literal['BMESH','CARVE']
    """

class BrightContrastModifier(SequenceModifier, bpy_struct):
    """Bright/contrast modifier data for sequence strip"""

    bright: float
    """ 

    :type: float
    """

    contrast: float
    """ 

    :type: float
    """

class Brush(ID, bpy_struct):
    """Brush data-block for storing brush settings for painting and sculpting"""

    auto_smooth_factor: float
    """ Amount of smoothing to automatically apply to each stroke

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUB",
        "MUL",
        "LIGHTEN",
        "DARKEN",
        "ERASE_ALPHA",
        "ADD_ALPHA",
        "OVERLAY",
        "HARDLIGHT",
        "COLORBURN",
        "LINEARBURN",
        "COLORDODGE",
        "SCREEN",
        "SOFTLIGHT",
        "PINLIGHT",
        "VIVIDLIGHT",
        "LINEARLIGHT",
        "DIFFERENCE",
        "EXCLUSION",
        "HUE",
        "SATURATION",
        "LUMINOSITY",
        "COLOR",
    ]
    """ Brush blending mode

    :type: typing.Literal['MIX','ADD','SUB','MUL','LIGHTEN','DARKEN','ERASE_ALPHA','ADD_ALPHA','OVERLAY','HARDLIGHT','COLORBURN','LINEARBURN','COLORDODGE','SCREEN','SOFTLIGHT','PINLIGHT','VIVIDLIGHT','LINEARLIGHT','DIFFERENCE','EXCLUSION','HUE','SATURATION','LUMINOSITY','COLOR']
    """

    blur_kernel_radius: int
    """ Radius of kernel used for soften and sharpen in pixels

    :type: int
    """

    blur_mode: typing.Literal["BOX", "GAUSSIAN"]
    """ 

    :type: typing.Literal['BOX','GAUSSIAN']
    """

    brush_capabilities: BrushCapabilities
    """ Brush's capabilities

    :type: BrushCapabilities
    """

    clone_alpha: float
    """ Opacity of clone image display

    :type: float
    """

    clone_image: Image | None
    """ Image for clone tool

    :type: Image | None
    """

    clone_offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    crease_pinch_factor: float
    """ How much the crease brush pinches

    :type: float
    """

    cursor_color_add: bpy_prop_array[float]
    """ Color of cursor when adding

    :type: bpy_prop_array[float]
    """

    cursor_color_subtract: bpy_prop_array[float]
    """ Color of cursor when subtracting

    :type: bpy_prop_array[float]
    """

    cursor_overlay_alpha: int
    """ 

    :type: int
    """

    curve: CurveMapping
    """ Editable falloff curve

    :type: CurveMapping
    """

    direction: typing.Literal["ADD", "SUBTRACT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT']
    """

    fill_threshold: float
    """ Threshold above which filling is not propagated

    :type: float
    """

    grad_spacing: int
    """ Spacing before brush gradient goes full circle

    :type: int
    """

    gradient: ColorRamp
    """ 

    :type: ColorRamp
    """

    gradient_fill_mode: typing.Literal["LINEAR", "RADIAL"]
    """ 

    :type: typing.Literal['LINEAR','RADIAL']
    """

    gradient_stroke_mode: typing.Literal["PRESSURE", "SPACING_REPEAT", "SPACING_CLAMP"]
    """ 

    :type: typing.Literal['PRESSURE','SPACING_REPEAT','SPACING_CLAMP']
    """

    height: float
    """ Affectable height of brush (layer height for layer tool, i.e.)

    :type: float
    """

    icon_filepath: str
    """ File path to brush icon

    :type: str
    """

    image_paint_capabilities: ImapaintToolCapabilities
    """ Brush's capabilities in image paint mode

    :type: ImapaintToolCapabilities
    """

    image_tool: typing.Literal["DRAW", "SOFTEN", "SMEAR", "CLONE", "FILL", "MASK"]
    """ 

    :type: typing.Literal['DRAW','SOFTEN','SMEAR','CLONE','FILL','MASK']
    """

    jitter: float
    """ Jitter the position of the brush while painting

    :type: float
    """

    jitter_absolute: int
    """ Jitter the position of the brush in pixels while painting

    :type: int
    """

    mask_overlay_alpha: int
    """ 

    :type: int
    """

    mask_stencil_dimension: bpy_prop_array[float]
    """ Dimensions of mask stencil in viewport

    :type: bpy_prop_array[float]
    """

    mask_stencil_pos: bpy_prop_array[float]
    """ Position of mask stencil in viewport

    :type: bpy_prop_array[float]
    """

    mask_texture: Texture | None
    """ 

    :type: Texture | None
    """

    mask_texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    mask_tool: typing.Literal["DRAW", "SMOOTH"]
    """ 

    :type: typing.Literal['DRAW','SMOOTH']
    """

    normal_weight: float
    """ How much grab will pull vertexes out of surface during a grab

    :type: float
    """

    paint_curve: PaintCurve | None
    """ Active Paint Curve

    :type: PaintCurve | None
    """

    plane_offset: float
    """ Adjust plane on which the brush acts towards or away from the object surface

    :type: float
    """

    plane_trim: float
    """ If a vertex is further away from offset plane than this, then it is not affected

    :type: float
    """

    rake_factor: float
    """ How much grab will follow cursor rotation

    :type: float
    """

    rate: float
    """ Interval between paints for Airbrush

    :type: float
    """

    sculpt_capabilities: SculptToolCapabilities
    """ Brush's capabilities in sculpt mode

    :type: SculptToolCapabilities
    """

    sculpt_plane: typing.Literal["AREA", "VIEW", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['AREA','VIEW','X','Y','Z']
    """

    sculpt_tool: typing.Literal[
        "BLOB",
        "CLAY",
        "CLAY_STRIPS",
        "CREASE",
        "DRAW",
        "FILL",
        "FLATTEN",
        "GRAB",
        "INFLATE",
        "LAYER",
        "MASK",
        "NUDGE",
        "PINCH",
        "ROTATE",
        "SCRAPE",
        "SIMPLIFY",
        "SMOOTH",
        "SNAKE_HOOK",
        "THUMB",
    ]
    """ 

    :type: typing.Literal['BLOB','CLAY','CLAY_STRIPS','CREASE','DRAW','FILL','FLATTEN','GRAB','INFLATE','LAYER','MASK','NUDGE','PINCH','ROTATE','SCRAPE','SIMPLIFY','SMOOTH','SNAKE_HOOK','THUMB']
    """

    secondary_color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    sharp_threshold: float
    """ Threshold below which, no sharpening is done

    :type: float
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    smooth_stroke_factor: float
    """ Higher values give a smoother stroke

    :type: float
    """

    smooth_stroke_radius: int
    """ Minimum distance from last point before stroke continues

    :type: int
    """

    spacing: int
    """ Spacing between brush daubs as a percentage of brush diameter

    :type: int
    """

    stencil_dimension: bpy_prop_array[float]
    """ Dimensions of stencil in viewport

    :type: bpy_prop_array[float]
    """

    stencil_pos: bpy_prop_array[float]
    """ Position of stencil in viewport

    :type: bpy_prop_array[float]
    """

    strength: float
    """ How powerful the effect of the brush is when applied

    :type: float
    """

    stroke_method: typing.Literal[
        "DOTS", "DRAG_DOT", "SPACE", "AIRBRUSH", "ANCHORED", "LINE", "CURVE"
    ]
    """ 

    :type: typing.Literal['DOTS','DRAG_DOT','SPACE','AIRBRUSH','ANCHORED','LINE','CURVE']
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_overlay_alpha: int
    """ 

    :type: int
    """

    texture_sample_bias: float
    """ Value added to texture samples

    :type: float
    """

    texture_slot: BrushTextureSlot
    """ 

    :type: BrushTextureSlot
    """

    unprojected_radius: float
    """ Radius of brush in Blender units

    :type: float
    """

    use_accumulate: bool
    """ Accumulate stroke daubs on top of each other

    :type: bool
    """

    use_adaptive_space: bool
    """ Space daubs according to surface orientation instead of screen space

    :type: bool
    """

    use_airbrush: bool
    """ Keep applying paint effect while holding mouse (spray)

    :type: bool
    """

    use_alpha: bool
    """ When this is disabled, lock alpha while painting

    :type: bool
    """

    use_anchor: bool
    """ Keep the brush anchored to the initial location

    :type: bool
    """

    use_cursor_overlay: bool
    """ Show cursor in viewport

    :type: bool
    """

    use_cursor_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_curve: bool
    """ Define the stroke curve with a bezier curve. Dabs are separated according to spacing

    :type: bool
    """

    use_custom_icon: bool
    """ Set the brush icon from an image file

    :type: bool
    """

    use_edge_to_edge: bool
    """ Drag anchor brush from edge-to-edge

    :type: bool
    """

    use_frontface: bool
    """ Brush only affects vertexes that face the viewer

    :type: bool
    """

    use_gradient: bool
    """ Use Gradient by utilizing a sampling method

    :type: bool
    """

    use_inverse_smooth_pressure: bool
    """ Lighter pressure causes more smoothing to be applied

    :type: bool
    """

    use_line: bool
    """ Draw a line with dabs separated according to spacing

    :type: bool
    """

    use_locked_size: bool
    """ When locked brush stays same size relative to object; when unlocked brush size is given in pixels

    :type: bool
    """

    use_offset_pressure: bool
    """ Enable tablet pressure sensitivity for offset

    :type: bool
    """

    use_original_normal: bool
    """ When locked keep using normal of surface where stroke was initiated

    :type: bool
    """

    use_paint_image: bool
    """ Use this brush in texture paint mode

    :type: bool
    """

    use_paint_sculpt: bool
    """ Use this brush in sculpt mode

    :type: bool
    """

    use_paint_vertex: bool
    """ Use this brush in vertex paint mode

    :type: bool
    """

    use_paint_weight: bool
    """ Use this brush in weight paint mode

    :type: bool
    """

    use_persistent: bool
    """ Sculpt on a persistent layer of the mesh

    :type: bool
    """

    use_plane_trim: bool
    """ Enable Plane Trim

    :type: bool
    """

    use_pressure_jitter: bool
    """ Enable tablet pressure sensitivity for jitter

    :type: bool
    """

    use_pressure_masking: typing.Literal["NONE", "RAMP", "CUTOFF"]
    """ Pen pressure makes texture influence smaller

    :type: typing.Literal['NONE','RAMP','CUTOFF']
    """

    use_pressure_size: bool
    """ Enable tablet pressure sensitivity for size

    :type: bool
    """

    use_pressure_spacing: bool
    """ Enable tablet pressure sensitivity for spacing

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

    use_primary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_primary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_relative_jitter: bool
    """ Jittering happens in screen space, not relative to brush size

    :type: bool
    """

    use_restore_mesh: bool
    """ Allow a single dot to be carefully positioned

    :type: bool
    """

    use_secondary_overlay: bool
    """ Show texture in viewport

    :type: bool
    """

    use_secondary_overlay_override: bool
    """ Don't show overlay during a stroke

    :type: bool
    """

    use_smooth_stroke: bool
    """ Brush lags behind mouse and follows a smoother path

    :type: bool
    """

    use_space: bool
    """ Limit brush application to the distance specified by spacing

    :type: bool
    """

    use_space_attenuation: bool
    """ Automatically adjust strength to give consistent results for different spacings

    :type: bool
    """

    vertex_tool: typing.Literal["MIX", "ADD", "SUB", "MUL", "BLUR", "LIGHTEN", "DARKEN"]
    """ Brush blending mode

    :type: typing.Literal['MIX','ADD','SUB','MUL','BLUR','LIGHTEN','DARKEN']
    """

    weight: float
    """ Vertex weight when brush is applied

    :type: float
    """

class BrushCapabilities(bpy_struct):
    """Read-only indications of which brush operations are supported by the current brush"""

    has_overlay: bool
    """ 

    :type: bool
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_smooth_stroke: bool
    """ 

    :type: bool
    """

    has_spacing: bool
    """ 

    :type: bool
    """

class BrushTextureSlot(TextureSlot, bpy_struct):
    """Texture slot for textures in a Brush data-block"""

    angle: float
    """ Brush texture rotation

    :type: float
    """

    has_random_texture_angle: bool
    """ 

    :type: bool
    """

    has_texture_angle: bool
    """ 

    :type: bool
    """

    has_texture_angle_source: bool
    """ 

    :type: bool
    """

    map_mode: typing.Literal[
        "VIEW_PLANE", "AREA_PLANE", "TILED", "3D", "RANDOM", "STENCIL"
    ]
    """ 

    :type: typing.Literal['VIEW_PLANE','AREA_PLANE','TILED','3D','RANDOM','STENCIL']
    """

    mask_map_mode: typing.Literal["VIEW_PLANE", "TILED", "RANDOM", "STENCIL"]
    """ 

    :type: typing.Literal['VIEW_PLANE','TILED','RANDOM','STENCIL']
    """

    random_angle: float
    """ Brush texture random angle

    :type: float
    """

    tex_paint_map_mode: typing.Literal["VIEW_PLANE", "TILED", "3D", "RANDOM", "STENCIL"]
    """ 

    :type: typing.Literal['VIEW_PLANE','TILED','3D','RANDOM','STENCIL']
    """

    use_rake: bool
    """ 

    :type: bool
    """

    use_random: bool
    """ 

    :type: bool
    """

class BuildModifier(Modifier, bpy_struct):
    """Build effect modifier"""

    frame_duration: float
    """ Total time the build effect requires

    :type: float
    """

    frame_start: float
    """ Start frame of the effect

    :type: float
    """

    seed: int
    """ Seed for random if used

    :type: int
    """

    use_random_order: bool
    """ Randomize the faces or edges during build

    :type: bool
    """

    use_reverse: bool
    """ Deconstruct the mesh instead of building it

    :type: bool
    """

class CLIP_UL_tracking_objects(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class CYCLES(RenderEngine, bpy_struct):
    def bake(
        self,
        scene,
        obj,
        pass_type,
        pass_filter,
        object_id,
        pixel_array,
        num_pixels,
        depth,
        result,
    ):
        """

        :param scene:
        :param obj:
        :param pass_type:
        :param pass_filter:
        :param object_id:
        :param pixel_array:
        :param num_pixels:
        :param depth:
        :param result:
        """

    def render(self, scene):
        """

        :param scene:
        """

    def update(self, data, scene):
        """

        :param data:
        :param scene:
        """

    def update_script_node(self, node):
        """

        :param node:
        """

    def view_draw(self, context):
        """

        :param context:
        """

    def view_update(self, context):
        """

        :param context:
        """

class CacheFile(ID, bpy_struct):
    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    forward_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ 

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    frame: float
    """ The time to use for looking up the data in the cache file, or to determine which file to use in a file sequence

    :type: float
    """

    is_sequence: bool
    """ Whether the cache is separated in a series of files

    :type: bool
    """

    object_paths: AlembicObjectPaths
    """ Paths of the objects inside the Alembic archive

    :type: AlembicObjectPaths
    """

    override_frame: bool
    """ Whether to use a custom frame for looking up data in the cache file, instead of using the current scene frame

    :type: bool
    """

    scale: float
    """ Value by which to enlarge or shrink the object with respect to the world's origin (only applicable through a Transform Cache constraint)

    :type: float
    """

    up_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ 

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

class Camera(ID, bpy_struct):
    """Camera data-block for storing camera settings"""

    angle: float
    """ Camera lens field of view

    :type: float
    """

    angle_x: float
    """ Camera lens horizontal field of view

    :type: float
    """

    angle_y: float
    """ Camera lens vertical field of view

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    clip_end: float
    """ Camera far clipping distance

    :type: float
    """

    clip_start: float
    """ Camera near clipping distance

    :type: float
    """

    cycles: CyclesCameraSettings
    """ Cycles camera settings

    :type: CyclesCameraSettings
    """

    dof_distance: float
    """ Distance to the focus point for depth of field

    :type: float
    """

    dof_object: Object | None
    """ Use this object to define the depth of field focal point

    :type: Object | None
    """

    draw_size: float
    """ Apparent size of the Camera object in the 3D View

    :type: float
    """

    gpu_dof: GPUDOFSettings
    """ 

    :type: GPUDOFSettings
    """

    lens: float
    """ Perspective Camera lens value in millimeters

    :type: float
    """

    lens_unit: typing.Literal["MILLIMETERS", "FOV"]
    """ Unit to edit lens in for the user interface

    :type: typing.Literal['MILLIMETERS','FOV']
    """

    ortho_scale: float
    """ Orthographic Camera scale (similar to zoom)

    :type: float
    """

    passepartout_alpha: float
    """ Opacity (alpha) of the darkened overlay in Camera view

    :type: float
    """

    sensor_fit: typing.Literal["AUTO", "HORIZONTAL", "VERTICAL"]
    """ Method to fit image and field of view angle inside the sensor

    :type: typing.Literal['AUTO','HORIZONTAL','VERTICAL']
    """

    sensor_height: float
    """ Vertical size of the image sensor area in millimeters

    :type: float
    """

    sensor_width: float
    """ Horizontal size of the image sensor area in millimeters

    :type: float
    """

    shift_x: float
    """ Camera horizontal shift

    :type: float
    """

    shift_y: float
    """ Camera vertical shift

    :type: float
    """

    show_guide: set[
        typing.Literal[
            "CENTER",
            "CENTER_DIAGONAL",
            "THIRDS",
            "GOLDEN",
            "GOLDEN_TRIANGLE_A",
            "GOLDEN_TRIANGLE_B",
            "HARMONY_TRIANGLE_A",
            "HARMONY_TRIANGLE_B",
        ]
    ]
    """ Draw overlay

    :type: set[typing.Literal['CENTER','CENTER_DIAGONAL','THIRDS','GOLDEN','GOLDEN_TRIANGLE_A','GOLDEN_TRIANGLE_B','HARMONY_TRIANGLE_A','HARMONY_TRIANGLE_B']]
    """

    show_limits: bool
    """ Draw the clipping range and focus point on the camera

    :type: bool
    """

    show_mist: bool
    """ Draw a line from the Camera to indicate the mist area

    :type: bool
    """

    show_name: bool
    """ Show the active Camera's name in Camera view

    :type: bool
    """

    show_passepartout: bool
    """ Show a darkened overlay outside the image area in Camera view

    :type: bool
    """

    show_safe_areas: bool
    """ Show TV title safe and action safe areas in Camera view

    :type: bool
    """

    show_safe_center: bool
    """ Show safe areas to fit content in a different aspect ratio

    :type: bool
    """

    show_sensor: bool
    """ Show sensor size (film gate) in Camera view

    :type: bool
    """

    stereo: CameraStereoData
    """ 

    :type: CameraStereoData
    """

    type: typing.Literal["PERSP", "ORTHO", "PANO"]
    """ Camera types

    :type: typing.Literal['PERSP','ORTHO','PANO']
    """

    def view_frame(self, scene: Scene | None = None):
        """Return 4 points for the cameras frame (before object transformation)

                :param scene: Scene to use for aspect calculation, when omitted 1:1 aspect is used
                :type scene: Scene | None
                :return: result_1, Result, float array of 3 items in [-inf, inf]

        result_2, Result, float array of 3 items in [-inf, inf]

        result_3, Result, float array of 3 items in [-inf, inf]

        result_4, Result, float array of 3 items in [-inf, inf]
        """

class CameraActuator(Actuator, bpy_struct):
    axis: typing.Literal["POS_X", "POS_Y", "NEG_X", "NEG_Y"]
    """ Axis the Camera will try to get behind

    :type: typing.Literal['POS_X','POS_Y','NEG_X','NEG_Y']
    """

    damping: float
    """ Strength of the constraint that drives the camera behind the target

    :type: float
    """

    height: float
    """ 

    :type: float
    """

    max: float
    """ 

    :type: float
    """

    min: float
    """ 

    :type: float
    """

    object: Object | None
    """ Look at this Object

    :type: Object | None
    """

class CameraSolverConstraint(Constraint, bpy_struct):
    """Lock motion to the reconstructed camera movement"""

    clip: MovieClip | None
    """ Movie Clip to get tracking data from

    :type: MovieClip | None
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

class CameraStereoData(bpy_struct):
    """Stereoscopy settings for a Camera data-block"""

    convergence_distance: float
    """ The converge point for the stereo cameras (often the distance between a projector and the projection screen)

    :type: float
    """

    convergence_mode: typing.Literal["OFFAXIS", "PARALLEL", "TOE"]
    """ 

    :type: typing.Literal['OFFAXIS','PARALLEL','TOE']
    """

    interocular_distance: float
    """ Set the distance between the eyes - the stereo plane distance / 30 should be fine

    :type: float
    """

    pivot: typing.Literal["LEFT", "RIGHT", "CENTER"]
    """ 

    :type: typing.Literal['LEFT','RIGHT','CENTER']
    """

    pole_merge_angle_from: float
    """ Angle at which interocular distance starts to fade to 0

    :type: float
    """

    pole_merge_angle_to: float
    """ Angle at which interocular distance is 0

    :type: float
    """

    use_pole_merge: bool
    """ Fade interocular distance to 0 after the given cutoff angle

    :type: bool
    """

    use_spherical_stereo: bool
    """ Render every pixel rotating the camera around the middle of the interocular distance

    :type: bool
    """

class CastModifier(Modifier, bpy_struct):
    """Modifier to cast to other shapes"""

    cast_type: typing.Literal["SPHERE", "CYLINDER", "CUBOID"]
    """ Target object shape

    :type: typing.Literal['SPHERE','CYLINDER','CUBOID']
    """

    factor: float
    """ 

    :type: float
    """

    object: Object | None
    """ Control object: if available, its location determines the center of the effect

    :type: Object | None
    """

    radius: float
    """ Only deform vertices within this distance from the center of the effect (leave as 0 for infinite.)

    :type: float
    """

    size: float
    """ Size of projection shape (leave as 0 for auto)

    :type: float
    """

    use_radius_as_size: bool
    """ Use radius as size of projection shape (0 = auto)

    :type: bool
    """

    use_transform: bool
    """ Use object transform to control projection shape

    :type: bool
    """

    use_x: bool
    """ 

    :type: bool
    """

    use_y: bool
    """ 

    :type: bool
    """

    use_z: bool
    """ 

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class ChildOfConstraint(Constraint, bpy_struct):
    """Create constraint-based parent-child relationship"""

    inverse_matrix: bpy_prop_array[float]
    """ Transformation matrix to apply before

    :type: bpy_prop_array[float]
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_location_x: bool
    """ Use X Location of Parent

    :type: bool
    """

    use_location_y: bool
    """ Use Y Location of Parent

    :type: bool
    """

    use_location_z: bool
    """ Use Z Location of Parent

    :type: bool
    """

    use_rotation_x: bool
    """ Use X Rotation of Parent

    :type: bool
    """

    use_rotation_y: bool
    """ Use Y Rotation of Parent

    :type: bool
    """

    use_rotation_z: bool
    """ Use Z Rotation of Parent

    :type: bool
    """

    use_scale_x: bool
    """ Use X Scale of Parent

    :type: bool
    """

    use_scale_y: bool
    """ Use Y Scale of Parent

    :type: bool
    """

    use_scale_z: bool
    """ Use Z Scale of Parent

    :type: bool
    """

class ChildParticle(bpy_struct):
    """Child particle interpolated from simulated or edited particles"""

class ClampToConstraint(Constraint, bpy_struct):
    """Constrain an object's location to the nearest point along the target path"""

    main_axis: typing.Literal["CLAMPTO_AUTO", "CLAMPTO_X", "CLAMPTO_Y", "CLAMPTO_Z"]
    """ Main axis of movement

    :type: typing.Literal['CLAMPTO_AUTO','CLAMPTO_X','CLAMPTO_Y','CLAMPTO_Z']
    """

    target: Object | None
    """ Target Object (Curves only)

    :type: Object | None
    """

    use_cyclic: bool
    """ Treat curve as cyclic curve (no clamping to curve bounding box)

    :type: bool
    """

class ClothCollisionSettings(bpy_struct):
    """Cloth simulation settings for self collision and collision with other objects"""

    collision_quality: int
    """ How many collision iterations should be done. (higher is better quality but slower)

    :type: int
    """

    damping: float
    """ Amount of velocity lost on collision

    :type: float
    """

    distance_min: float
    """ Minimum distance between collision objects before collision response takes in

    :type: float
    """

    distance_repel: float
    """ Maximum distance to apply repulsion force, must be greater than minimum distance

    :type: float
    """

    friction: float
    """ Friction force if a collision happened (higher = less movement)

    :type: float
    """

    group: Group | None
    """ Limit colliders to this Group

    :type: Group | None
    """

    repel_force: float
    """ Repulsion force to apply on cloth when close to colliding

    :type: float
    """

    self_collision_quality: int
    """ How many self collision iterations should be done (higher is better quality but slower)

    :type: int
    """

    self_distance_min: float
    """ 0.5 means no distance at all, 1.0 is maximum distance

    :type: float
    """

    self_friction: float
    """ Friction/damping with self contact

    :type: float
    """

    use_collision: bool
    """ Enable collisions with other objects

    :type: bool
    """

    use_self_collision: bool
    """ Enable self collisions

    :type: bool
    """

    vertex_group_self_collisions: str
    """ Vertex group to define vertices which are not used during self collisions

    :type: str
    """

class ClothModifier(Modifier, bpy_struct):
    """Cloth simulation modifier"""

    collision_settings: ClothCollisionSettings
    """ 

    :type: ClothCollisionSettings
    """

    hair_grid_max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hair_grid_resolution: bpy_prop_array[int]
    """ 

    :type: bpy_prop_array[int]
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    settings: ClothSettings
    """ 

    :type: ClothSettings
    """

    solver_result: ClothSolverResult
    """ 

    :type: ClothSolverResult
    """

class ClothSettings(bpy_struct):
    """Cloth simulation settings for an object"""

    air_damping: float
    """ Air has normally some thickness which slows falling things down

    :type: float
    """

    bending_damping: float
    """ Damping of bending motion

    :type: float
    """

    bending_stiffness: float
    """ Wrinkle coefficient (higher = less smaller but more big wrinkles)

    :type: float
    """

    bending_stiffness_max: float
    """ Maximum bending stiffness value

    :type: float
    """

    collider_friction: float
    """ 

    :type: float
    """

    density_strength: float
    """ Influence of target density on the simulation

    :type: float
    """

    density_target: float
    """ Maximum density of hair

    :type: float
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    goal_default: float
    """ Default Goal (vertex target position) value, when no Vertex Group used

    :type: float
    """

    goal_friction: float
    """ Goal (vertex target position) friction

    :type: float
    """

    goal_max: float
    """ Goal maximum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_min: float
    """ Goal minimum, vertex group weights are scaled to match this range

    :type: float
    """

    goal_spring: float
    """ Goal (vertex target position) spring stiffness

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity or external force vector

    :type: bpy_prop_array[float]
    """

    internal_friction: float
    """ 

    :type: float
    """

    mass: float
    """ Mass of cloth material

    :type: float
    """

    pin_stiffness: float
    """ Pin (vertex target position) spring stiffness

    :type: float
    """

    quality: int
    """ Quality of the simulation in steps per frame (higher is better quality but slower)

    :type: int
    """

    rest_shape_key: ShapeKey | None
    """ Shape key to use the rest spring lengths from

    :type: ShapeKey | None
    """

    sewing_force_max: float
    """ Maximum sewing force

    :type: float
    """

    shrink_max: float
    """ Max amount to shrink cloth by

    :type: float
    """

    shrink_min: float
    """ Min amount to shrink cloth by

    :type: float
    """

    spring_damping: float
    """ Damping of cloth velocity (higher = more smooth, less jiggling)

    :type: float
    """

    structural_stiffness: float
    """ Overall stiffness of structure

    :type: float
    """

    structural_stiffness_max: float
    """ Maximum structural stiffness value

    :type: float
    """

    time_scale: float
    """ Cloth speed is multiplied by this value

    :type: float
    """

    use_dynamic_mesh: bool
    """ Make simulation respect deformations in the base mesh

    :type: bool
    """

    use_pin_cloth: bool
    """ Enable pinning of cloth vertices to other objects/positions

    :type: bool
    """

    use_sewing_springs: bool
    """ Pulls loose edges together

    :type: bool
    """

    use_stiffness_scale: bool
    """ If enabled, stiffness can be scaled along a weight painted vertex group

    :type: bool
    """

    vel_damping: float
    """ Damp velocity to help cloth reach the resting position faster (1.0 = no damping, 0.0 = fully dampened)

    :type: float
    """

    vertex_group_bending: str
    """ Vertex group for fine control over bending stiffness

    :type: str
    """

    vertex_group_mass: str
    """ Vertex Group for pinning of vertices

    :type: str
    """

    vertex_group_shrink: str
    """ Vertex Group for shrinking cloth

    :type: str
    """

    vertex_group_structural_stiffness: str
    """ Vertex group for fine control over structural stiffness

    :type: str
    """

    voxel_cell_size: float
    """ Size of the voxel grid cells for interaction effects

    :type: float
    """

class ClothSolverResult(bpy_struct):
    """Result of cloth solver iteration"""

    avg_error: float
    """ Average error during substeps

    :type: float
    """

    avg_iterations: float
    """ Average iterations during substeps

    :type: float
    """

    max_error: float
    """ Maximum error during substeps

    :type: float
    """

    max_iterations: int
    """ Maximum iterations during substeps

    :type: int
    """

    min_error: float
    """ Minimum error during substeps

    :type: float
    """

    min_iterations: int
    """ Minimum iterations during substeps

    :type: int
    """

    status: set[
        typing.Literal["SUCCESS", "NUMERICAL_ISSUE", "NO_CONVERGENCE", "INVALID_INPUT"]
    ]
    """ Status of the solver iteration

    :type: set[typing.Literal['SUCCESS','NUMERICAL_ISSUE','NO_CONVERGENCE','INVALID_INPUT']]
    """

class CloudsTexture(Texture, ID, bpy_struct):
    """Procedural noise texture"""

    cloud_type: typing.Literal["GRAYSCALE", "COLOR"]
    """ Determine whether Noise returns grayscale or RGB values

    :type: typing.Literal['GRAYSCALE','COLOR']
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_depth: int
    """ Depth of the cloud calculation

    :type: int
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    noise_type: typing.Literal["SOFT_NOISE", "HARD_NOISE"]
    """ 

    :type: typing.Literal['SOFT_NOISE','HARD_NOISE']
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class CollectionProperty(Property, bpy_struct):
    """RNA collection property to define lists, arrays and mappings"""

    fixed_type: Struct
    """ Fixed pointer type, empty if variable type

    :type: Struct
    """

class CollisionModifier(Modifier, bpy_struct):
    """Collision modifier defining modifier stack position used for collision"""

    settings: CollisionSettings
    """ 

    :type: CollisionSettings
    """

class CollisionSensor(Sensor, bpy_struct):
    """Sensor to detect objects colliding with the current object, with more settings than the Touch sensor"""

    material: str
    """ Only look for objects with this material (blank = all objects)

    :type: str
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    use_material: bool
    """ Toggle collision on material or property

    :type: bool
    """

    use_pulse: bool
    """ Change to the set of colliding objects generates pulse

    :type: bool
    """

class CollisionSettings(bpy_struct):
    """Collision settings for object in physics simulation"""

    absorption: float
    """ How much of effector force gets lost during collision with this object (in percent)

    :type: float
    """

    damping: float
    """ Amount of damping during collision

    :type: float
    """

    damping_factor: float
    """ Amount of damping during particle collision

    :type: float
    """

    damping_random: float
    """ Random variation of damping

    :type: float
    """

    friction_factor: float
    """ Amount of friction during particle collision

    :type: float
    """

    friction_random: float
    """ Random variation of friction

    :type: float
    """

    permeability: float
    """ Chance that the particle will pass through the mesh

    :type: float
    """

    stickiness: float
    """ Amount of stickiness to surface collision

    :type: float
    """

    thickness_inner: float
    """ Inner face thickness (only used by softbodies)

    :type: float
    """

    thickness_outer: float
    """ Outer face thickness

    :type: float
    """

    use: bool
    """ Enable this objects as a collider for physics systems

    :type: bool
    """

    use_particle_kill: bool
    """ Kill collided particles

    :type: bool
    """

class ColorBalanceModifier(SequenceModifier, bpy_struct):
    """Color balance modifier for sequence strip"""

    color_balance: SequenceColorBalanceData
    """ 

    :type: SequenceColorBalanceData
    """

    color_multiply: float
    """ 

    :type: float
    """

class ColorManagedDisplaySettings(bpy_struct):
    """Color management specific to display device"""

    display_device: str
    """ Display device name

    :type: str
    """

class ColorManagedInputColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: typing.Literal[
        "Linear", "Linear ACES", "Non-Color", "Raw", "sRGB", "VD16", "XYZ"
    ]
    """ Color space of the image or movie on disk

    :type: typing.Literal['Linear','Linear ACES','Non-Color','Raw','sRGB','VD16','XYZ']
    """

class ColorManagedSequencerColorspaceSettings(bpy_struct):
    """Input color space settings"""

    name: typing.Literal[
        "Linear", "Linear ACES", "Non-Color", "Raw", "sRGB", "VD16", "XYZ"
    ]
    """ Color space that the sequencer operates in

    :type: typing.Literal['Linear','Linear ACES','Non-Color','Raw','sRGB','VD16','XYZ']
    """

class ColorManagedViewSettings(bpy_struct):
    """Color management settings used for displaying images on the display"""

    curve_mapping: CurveMapping
    """ Color curve mapping applied before display transform

    :type: CurveMapping
    """

    exposure: float
    """ Exposure (stops) applied before display transform

    :type: float
    """

    gamma: float
    """ Amount of gamma modification applied after display transform

    :type: float
    """

    look: typing.Literal["NONE"]
    """ Additional transform applied before view transform for an artistic needs

    :type: typing.Literal['NONE']
    """

    use_curve_mapping: bool
    """ Use RGB curved for pre-display transformation

    :type: bool
    """

    view_transform: typing.Literal["NONE"]
    """ View used when converting image to a display space

    :type: typing.Literal['NONE']
    """

class ColorMapping(bpy_struct):
    """Color mapping settings"""

    blend_color: bpy_prop_array[float]
    """ Blend color to mix with texture output color

    :type: bpy_prop_array[float]
    """

    blend_factor: float
    """ 

    :type: float
    """

    blend_type: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "SCREEN",
        "OVERLAY",
        "DIFFERENCE",
        "DIVIDE",
        "DARKEN",
        "LIGHTEN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Mode used to mix with texture output color

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','SCREEN','OVERLAY','DIFFERENCE','DIVIDE','DARKEN','LIGHTEN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    brightness: float
    """ Adjust the brightness of the texture

    :type: float
    """

    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    contrast: float
    """ Adjust the contrast of the texture

    :type: float
    """

    saturation: float
    """ Adjust the saturation of colors in the texture

    :type: float
    """

    use_color_ramp: bool
    """ Toggle color ramp operations

    :type: bool
    """

class ColorRamp(bpy_struct):
    """Color ramp mapping a scalar value to a color"""

    color_mode: typing.Literal["RGB", "HSV", "HSL"]
    """ Set color mode to use for interpolation

    :type: typing.Literal['RGB','HSV','HSL']
    """

    elements: ColorRampElements
    """ 

    :type: ColorRampElements
    """

    hue_interpolation: typing.Literal["NEAR", "FAR", "CW", "CCW"]
    """ Set color interpolation

    :type: typing.Literal['NEAR','FAR','CW','CCW']
    """

    interpolation: typing.Literal["EASE", "CARDINAL", "LINEAR", "B_SPLINE", "CONSTANT"]
    """ Set interpolation between color stops

    :type: typing.Literal['EASE','CARDINAL','LINEAR','B_SPLINE','CONSTANT']
    """

    def evaluate(self, position: float | None) -> bpy_prop_array[float]:
        """Evaluate ColorRamp

        :param position: Position, Evaluate ColorRamp at position
        :type position: float | None
        :return: Color, Color at given position
        :rtype: bpy_prop_array[float]
        """

class ColorRampElement(bpy_struct):
    """Element defining a color at a position in the color ramp"""

    alpha: float
    """ Set alpha of selected color stop

    :type: float
    """

    color: bpy_prop_array[float]
    """ Set color of selected color stop

    :type: bpy_prop_array[float]
    """

    position: float
    """ Set position of selected color stop

    :type: float
    """

class ColorSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip creating an image filled with a single color"""

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    input_count: int
    """ 

    :type: int
    """

class CompositorNode(NodeInternal, Node, bpy_struct):
    def tag_need_exec(self):
        """Tag the node for compositor update"""

    def update(self): ...

class CompositorNodeAlphaOver(CompositorNode, NodeInternal, Node, bpy_struct):
    premul: float
    """ Mix Factor

    :type: float
    """

    use_premultiply: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBilateralblur(CompositorNode, NodeInternal, Node, bpy_struct):
    iterations: int
    """ 

    :type: int
    """

    sigma_color: float
    """ 

    :type: float
    """

    sigma_space: float
    """ 

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    aspect_correction: typing.Literal["NONE", "Y", "X"]
    """ Type of aspect correction to use

    :type: typing.Literal['NONE','Y','X']
    """

    factor: float
    """ 

    :type: float
    """

    factor_x: float
    """ 

    :type: float
    """

    factor_y: float
    """ 

    :type: float
    """

    filter_type: typing.Literal[
        "FLAT", "TENT", "QUAD", "CUBIC", "GAUSS", "FAST_GAUSS", "CATROM", "MITCH"
    ]
    """ 

    :type: typing.Literal['FLAT','TENT','QUAD','CUBIC','GAUSS','FAST_GAUSS','CATROM','MITCH']
    """

    size_x: int
    """ 

    :type: int
    """

    size_y: int
    """ 

    :type: int
    """

    use_bokeh: bool
    """ Use circular filter (slower)

    :type: bool
    """

    use_extended_bounds: bool
    """ Extend bounds of the input image to fully fit blurred image

    :type: bool
    """

    use_gamma_correction: bool
    """ Apply filter on gamma corrected values

    :type: bool
    """

    use_relative: bool
    """ Use relative (percent) values to define blur radius

    :type: bool
    """

    use_variable_size: bool
    """ Support variable blur per-pixel when using an image for size input

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBokehBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    blur_max: float
    """ Blur limit, maximum CoC radius

    :type: float
    """

    use_extended_bounds: bool
    """ Extend bounds of the input image to fully fit blurred image

    :type: bool
    """

    use_variable_size: bool
    """ Support variable blur per-pixel when using an image for size input

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBokehImage(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ Angle of the bokeh

    :type: float
    """

    catadioptric: float
    """ Level of catadioptric of the bokeh

    :type: float
    """

    flaps: int
    """ Number of flaps

    :type: int
    """

    rounding: float
    """ Level of rounding of the bokeh

    :type: float
    """

    shift: float
    """ Shift of the lens components

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBoxMask(CompositorNode, NodeInternal, Node, bpy_struct):
    height: float
    """ Height of the box

    :type: float
    """

    mask_type: typing.Literal["ADD", "SUBTRACT", "MULTIPLY", "NOT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT','MULTIPLY','NOT']
    """

    rotation: float
    """ Rotation angle of the box

    :type: float
    """

    width: float
    """ Width of the box

    :type: float
    """

    x: float
    """ X position of the middle of the box

    :type: float
    """

    y: float
    """ Y position of the middle of the box

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeBrightContrast(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeChannelMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    color_space: typing.Literal["RGB", "HSV", "YUV", "YCC"]
    """ 

    :type: typing.Literal['RGB','HSV','YUV','YCC']
    """

    limit_channel: typing.Literal["R", "G", "B"]
    """ Limit by this channel's value

    :type: typing.Literal['R','G','B']
    """

    limit_max: float
    """ Values higher than this setting are 100% opaque

    :type: float
    """

    limit_method: typing.Literal["SINGLE", "MAX"]
    """ Algorithm to use to limit channel

    :type: typing.Literal['SINGLE','MAX']
    """

    limit_min: float
    """ Values lower than this setting are 100% keyed

    :type: float
    """

    matte_channel: typing.Literal["R", "G", "B"]
    """ Channel used to determine matte

    :type: typing.Literal['R','G','B']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeChromaMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    gain: float
    """ Alpha falloff

    :type: float
    """

    lift: float
    """ Alpha lift

    :type: float
    """

    shadow_adjust: float
    """ Adjusts the brightness of any shadows captured

    :type: float
    """

    threshold: float
    """ Tolerance below which colors will be considered as exact matches

    :type: float
    """

    tolerance: float
    """ Tolerance for a color to be considered a keying color

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeColorBalance(CompositorNode, NodeInternal, Node, bpy_struct):
    correction_method: typing.Literal["LIFT_GAMMA_GAIN", "OFFSET_POWER_SLOPE"]
    """ 

    :type: typing.Literal['LIFT_GAMMA_GAIN','OFFSET_POWER_SLOPE']
    """

    gain: bpy_prop_array[float]
    """ Correction for Highlights

    :type: bpy_prop_array[float]
    """

    gamma: bpy_prop_array[float]
    """ Correction for Midtones

    :type: bpy_prop_array[float]
    """

    lift: bpy_prop_array[float]
    """ Correction for Shadows

    :type: bpy_prop_array[float]
    """

    offset: bpy_prop_array[float]
    """ Correction for Shadows

    :type: bpy_prop_array[float]
    """

    offset_basis: float
    """ Support negative color by using this as the RGB basis

    :type: float
    """

    power: bpy_prop_array[float]
    """ Correction for Midtones

    :type: bpy_prop_array[float]
    """

    slope: bpy_prop_array[float]
    """ Correction for Highlights

    :type: bpy_prop_array[float]
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeColorCorrection(CompositorNode, NodeInternal, Node, bpy_struct):
    blue: bool
    """ Blue channel active

    :type: bool
    """

    green: bool
    """ Green channel active

    :type: bool
    """

    highlights_contrast: float
    """ Highlights contrast

    :type: float
    """

    highlights_gain: float
    """ Highlights gain

    :type: float
    """

    highlights_gamma: float
    """ Highlights gamma

    :type: float
    """

    highlights_lift: float
    """ Highlights lift

    :type: float
    """

    highlights_saturation: float
    """ Highlights saturation

    :type: float
    """

    master_contrast: float
    """ Master contrast

    :type: float
    """

    master_gain: float
    """ Master gain

    :type: float
    """

    master_gamma: float
    """ Master gamma

    :type: float
    """

    master_lift: float
    """ Master lift

    :type: float
    """

    master_saturation: float
    """ Master saturation

    :type: float
    """

    midtones_contrast: float
    """ Midtones contrast

    :type: float
    """

    midtones_end: float
    """ End of midtones

    :type: float
    """

    midtones_gain: float
    """ Midtones gain

    :type: float
    """

    midtones_gamma: float
    """ Midtones gamma

    :type: float
    """

    midtones_lift: float
    """ Midtones lift

    :type: float
    """

    midtones_saturation: float
    """ Midtones saturation

    :type: float
    """

    midtones_start: float
    """ Start of midtones

    :type: float
    """

    red: bool
    """ Red channel active

    :type: bool
    """

    shadows_contrast: float
    """ Shadows contrast

    :type: float
    """

    shadows_gain: float
    """ Shadows gain

    :type: float
    """

    shadows_gamma: float
    """ Shadows gamma

    :type: float
    """

    shadows_lift: float
    """ Shadows lift

    :type: float
    """

    shadows_saturation: float
    """ Shadows saturation

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeColorMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    color_hue: float
    """ Hue tolerance for colors to be considered a keying color

    :type: float
    """

    color_saturation: float
    """ Saturation Tolerance for the color

    :type: float
    """

    color_value: float
    """ Value Tolerance for the color

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeColorSpill(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["R", "G", "B"]
    """ 

    :type: typing.Literal['R','G','B']
    """

    limit_channel: typing.Literal["R", "G", "B"]
    """ 

    :type: typing.Literal['R','G','B']
    """

    limit_method: typing.Literal["SIMPLE", "AVERAGE"]
    """ 

    :type: typing.Literal['SIMPLE','AVERAGE']
    """

    ratio: float
    """ Scale limit by value

    :type: float
    """

    unspill_blue: float
    """ Blue spillmap scale

    :type: float
    """

    unspill_green: float
    """ Green spillmap scale

    :type: float
    """

    unspill_red: float
    """ Red spillmap scale

    :type: float
    """

    use_unspill: bool
    """ Compensate all channels (differently) by hand

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCombHSVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCombRGBA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCombYCCA(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ 

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCombYUVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeComposite(CompositorNode, NodeInternal, Node, bpy_struct):
    use_alpha: bool
    """ Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCornerPin(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCrop(CompositorNode, NodeInternal, Node, bpy_struct):
    max_x: int
    """ 

    :type: int
    """

    max_y: int
    """ 

    :type: int
    """

    min_x: int
    """ 

    :type: int
    """

    min_y: int
    """ 

    :type: int
    """

    rel_max_x: float
    """ 

    :type: float
    """

    rel_max_y: float
    """ 

    :type: float
    """

    rel_min_x: float
    """ 

    :type: float
    """

    rel_min_y: float
    """ 

    :type: float
    """

    relative: bool
    """ Use relative values to crop image

    :type: bool
    """

    use_crop_size: bool
    """ Whether to crop the size of the input image

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCurveRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeCurveVec(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ 

    :type: float
    """

    center_x: float
    """ 

    :type: float
    """

    center_y: float
    """ 

    :type: float
    """

    distance: float
    """ 

    :type: float
    """

    iterations: int
    """ 

    :type: int
    """

    spin: float
    """ 

    :type: float
    """

    use_wrap: bool
    """ 

    :type: bool
    """

    zoom: float
    """ 

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDefocus(CompositorNode, NodeInternal, Node, bpy_struct):
    angle: float
    """ Bokeh shape rotation offset

    :type: float
    """

    blur_max: float
    """ Blur limit, maximum CoC radius

    :type: float
    """

    bokeh: typing.Literal[
        "OCTAGON", "HEPTAGON", "HEXAGON", "PENTAGON", "SQUARE", "TRIANGLE", "CIRCLE"
    ]
    """ 

    :type: typing.Literal['OCTAGON','HEPTAGON','HEXAGON','PENTAGON','SQUARE','TRIANGLE','CIRCLE']
    """

    f_stop: float
    """ Amount of focal blur, 128=infinity=perfect focus, half the value doubles the blur radius

    :type: float
    """

    scene: Scene | None
    """ Scene from which to select the active camera (render scene if undefined)

    :type: Scene | None
    """

    threshold: float
    """ CoC radius threshold, prevents background bleed on in-focus midground, 0=off

    :type: float
    """

    use_gamma_correction: bool
    """ Enable gamma correction before and after main process

    :type: bool
    """

    use_preview: bool
    """ Enable low quality mode, useful for preview

    :type: bool
    """

    use_zbuffer: bool
    """ Disable when using an image as input instead of actual z-buffer (auto enabled if node not image based, eg. time node)

    :type: bool
    """

    z_scale: float
    """ Scale the Z input when not using a z-buffer, controls maximum blur designated by the color white or input value 1

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDespeckle(CompositorNode, NodeInternal, Node, bpy_struct):
    threshold: float
    """ Threshold for detecting pixels to despeckle

    :type: float
    """

    threshold_neighbor: float
    """ Threshold for the number of neighbor pixels that must match

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDiffMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    falloff: float
    """ Color distances below this additional threshold are partially keyed

    :type: float
    """

    tolerance: float
    """ Color distances below this threshold are keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDilateErode(CompositorNode, NodeInternal, Node, bpy_struct):
    distance: int
    """ Distance to grow/shrink (number of iterations)

    :type: int
    """

    edge: float
    """ Edge to inset

    :type: float
    """

    falloff: typing.Literal[
        "SMOOTH", "SPHERE", "ROOT", "INVERSE_SQUARE", "SHARP", "LINEAR"
    ]
    """ Falloff type the feather

    :type: typing.Literal['SMOOTH','SPHERE','ROOT','INVERSE_SQUARE','SHARP','LINEAR']
    """

    mode: typing.Literal["STEP", "THRESHOLD", "DISTANCE", "FEATHER"]
    """ Growing/shrinking mode

    :type: typing.Literal['STEP','THRESHOLD','DISTANCE','FEATHER']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDisplace(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDistanceMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["RGB", "YCC"]
    """ 

    :type: typing.Literal['RGB','YCC']
    """

    falloff: float
    """ Color distances below this additional threshold are partially keyed

    :type: float
    """

    tolerance: float
    """ Color distances below this threshold are keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeDoubleEdgeMask(CompositorNode, NodeInternal, Node, bpy_struct):
    edge_mode: typing.Literal["BLEED_OUT", "KEEP_IN"]
    """ 

    :type: typing.Literal['BLEED_OUT','KEEP_IN']
    """

    inner_mode: typing.Literal["ALL", "ADJACENT_ONLY"]
    """ 

    :type: typing.Literal['ALL','ADJACENT_ONLY']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeEllipseMask(CompositorNode, NodeInternal, Node, bpy_struct):
    height: float
    """ Height of the ellipse

    :type: float
    """

    mask_type: typing.Literal["ADD", "SUBTRACT", "MULTIPLY", "NOT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT','MULTIPLY','NOT']
    """

    rotation: float
    """ Rotation angle of the ellipse

    :type: float
    """

    width: float
    """ Width of the ellipse

    :type: float
    """

    x: float
    """ X position of the middle of the ellipse

    :type: float
    """

    y: float
    """ Y position of the middle of the ellipse

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeFilter(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: typing.Literal[
        "SOFTEN", "SHARPEN", "LAPLACE", "SOBEL", "PREWITT", "KIRSCH", "SHADOW"
    ]
    """ 

    :type: typing.Literal['SOFTEN','SHARPEN','LAPLACE','SOBEL','PREWITT','KIRSCH','SHADOW']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeFlip(CompositorNode, NodeInternal, Node, bpy_struct):
    axis: typing.Literal["X", "Y", "XY"]
    """ 

    :type: typing.Literal['X','Y','XY']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeGamma(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeGlare(CompositorNode, NodeInternal, Node, bpy_struct):
    angle_offset: float
    """ Streak angle offset

    :type: float
    """

    color_modulation: float
    """ Amount of Color Modulation, modulates colors of streaks and ghosts for a spectral dispersion effect

    :type: float
    """

    fade: float
    """ Streak fade-out factor

    :type: float
    """

    glare_type: typing.Literal["GHOSTS", "STREAKS", "FOG_GLOW", "SIMPLE_STAR"]
    """ 

    :type: typing.Literal['GHOSTS','STREAKS','FOG_GLOW','SIMPLE_STAR']
    """

    iterations: int
    """ 

    :type: int
    """

    mix: float
    """ -1 is original image only, 0 is exact 50/50 mix, 1 is processed image only

    :type: float
    """

    quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ If not set to high quality, the effect will be applied to a low-res copy of the source image

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    size: int
    """ Glow/glare size (not actual size; relative to initial size of bright area of pixels)

    :type: int
    """

    streaks: int
    """ Total number of streaks

    :type: int
    """

    threshold: float
    """ The glare filter will only be applied to pixels brighter than this value

    :type: float
    """

    use_rotate_45: bool
    """ Simple star filter: add 45 degree rotation offset

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeGroup(CompositorNode, NodeInternal, Node, bpy_struct):
    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    node_tree: CompositorNodeTree | None
    """ 

    :type: CompositorNodeTree | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeHueCorrect(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeHueSat(CompositorNode, NodeInternal, Node, bpy_struct):
    color_hue: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    color_value: float
    """ 

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeIDMask(CompositorNode, NodeInternal, Node, bpy_struct):
    index: int
    """ Pass index number to convert to alpha

    :type: int
    """

    use_antialiasing: bool
    """ Apply an anti-aliasing filter to the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeImage(CompositorNode, NodeInternal, Node, bpy_struct):
    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    has_layers: bool
    """ True if this image has any named layer

    :type: bool
    """

    has_views: bool
    """ True if this image has multiple views

    :type: bool
    """

    image: Image | None
    """ 

    :type: Image | None
    """

    layer: typing.Literal["PLACEHOLDER"]
    """ 

    :type: typing.Literal['PLACEHOLDER']
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

    use_straight_alpha_output: bool
    """ Put Node output buffer to straight alpha instead of premultiplied

    :type: bool
    """

    view: typing.Literal["ALL"]
    """ 

    :type: typing.Literal['ALL']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeInpaint(CompositorNode, NodeInternal, Node, bpy_struct):
    distance: int
    """ Distance to inpaint (number of iterations)

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeInvert(CompositorNode, NodeInternal, Node, bpy_struct):
    invert_alpha: bool
    """ 

    :type: bool
    """

    invert_rgb: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeKeying(CompositorNode, NodeInternal, Node, bpy_struct):
    blur_post: int
    """ Matte blur size which applies after clipping and dilate/eroding

    :type: int
    """

    blur_pre: int
    """ Chroma pre-blur size which applies before running keyer

    :type: int
    """

    clip_black: float
    """ Value of non-scaled matte pixel which considers as fully background pixel

    :type: float
    """

    clip_white: float
    """ Value of non-scaled matte pixel which considers as fully foreground pixel

    :type: float
    """

    despill_balance: float
    """ Balance between non-key colors used to detect amount of key color to be removed

    :type: float
    """

    despill_factor: float
    """ Factor of despilling screen color from image

    :type: float
    """

    dilate_distance: int
    """ Matte dilate/erode side

    :type: int
    """

    edge_kernel_radius: int
    """ Radius of kernel used to detect whether pixel belongs to edge

    :type: int
    """

    edge_kernel_tolerance: float
    """ Tolerance to pixels inside kernel which are treating as belonging to the same plane

    :type: float
    """

    feather_distance: int
    """ Distance to grow/shrink the feather

    :type: int
    """

    feather_falloff: typing.Literal[
        "SMOOTH", "SPHERE", "ROOT", "INVERSE_SQUARE", "SHARP", "LINEAR"
    ]
    """ Falloff type the feather

    :type: typing.Literal['SMOOTH','SPHERE','ROOT','INVERSE_SQUARE','SHARP','LINEAR']
    """

    screen_balance: float
    """ Balance between two non-primary channels primary channel is comparing against

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeKeyingScreen(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    tracking_object: str
    """ 

    :type: str
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeLensdist(CompositorNode, NodeInternal, Node, bpy_struct):
    use_fit: bool
    """ For positive distortion factor only: scale image such that black areas are not visible

    :type: bool
    """

    use_jitter: bool
    """ Enable/disable jittering (faster, but also noisier)

    :type: bool
    """

    use_projector: bool
    """ Enable/disable projector mode (the effect is applied in horizontal direction only)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeLevels(CompositorNode, NodeInternal, Node, bpy_struct):
    channel: typing.Literal["COMBINED_RGB", "RED", "GREEN", "BLUE", "LUMINANCE"]
    """ 

    :type: typing.Literal['COMBINED_RGB','RED','GREEN','BLUE','LUMINANCE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeLumaMatte(CompositorNode, NodeInternal, Node, bpy_struct):
    limit_max: float
    """ Values higher than this setting are 100% opaque

    :type: float
    """

    limit_min: float
    """ Values lower than this setting are 100% keyed

    :type: float
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMapRange(CompositorNode, NodeInternal, Node, bpy_struct):
    use_clamp: bool
    """ Clamp result of the node to 0..1 range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMapUV(CompositorNode, NodeInternal, Node, bpy_struct):
    alpha: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMapValue(CompositorNode, NodeInternal, Node, bpy_struct):
    max: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    min: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    offset: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    size: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    use_max: bool
    """ 

    :type: bool
    """

    use_min: bool
    """ 

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMask(CompositorNode, NodeInternal, Node, bpy_struct):
    mask: Mask | None
    """ 

    :type: Mask | None
    """

    motion_blur_samples: int
    """ Number of motion blur samples

    :type: int
    """

    motion_blur_shutter: float
    """ Exposure for motion blur as a factor of FPS

    :type: float
    """

    size_source: typing.Literal["SCENE", "FIXED", "FIXED_SCENE"]
    """ Where to get the mask size from for aspect/size information

    :type: typing.Literal['SCENE','FIXED','FIXED_SCENE']
    """

    size_x: int
    """ 

    :type: int
    """

    size_y: int
    """ 

    :type: int
    """

    use_antialiasing: bool
    """ Apply an anti-aliasing filter to the mask

    :type: bool
    """

    use_feather: bool
    """ Use feather information from the mask

    :type: bool
    """

    use_motion_blur: bool
    """ Use multi-sampled motion blur of the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMath(CompositorNode, NodeInternal, Node, bpy_struct):
    operation: typing.Literal[
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "SINE",
        "COSINE",
        "TANGENT",
        "ARCSINE",
        "ARCCOSINE",
        "ARCTANGENT",
        "POWER",
        "LOGARITHM",
        "MINIMUM",
        "MAXIMUM",
        "ROUND",
        "LESS_THAN",
        "GREATER_THAN",
        "MODULO",
        "ABSOLUTE",
    ]
    """ 

    :type: typing.Literal['ADD','SUBTRACT','MULTIPLY','DIVIDE','SINE','COSINE','TANGENT','ARCSINE','ARCCOSINE','ARCTANGENT','POWER','LOGARITHM','MINIMUM','MAXIMUM','ROUND','LESS_THAN','GREATER_THAN','MODULO','ABSOLUTE']
    """

    use_clamp: bool
    """ Clamp result of the node to 0..1 range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMixRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    blend_type: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ 

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    use_alpha: bool
    """ Include alpha of second input in this operation

    :type: bool
    """

    use_clamp: bool
    """ Clamp result of the node to 0..1 range

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMovieClip(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeMovieDistortion(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    distortion_type: typing.Literal["UNDISTORT", "DISTORT"]
    """ Distortion to use to filter image

    :type: typing.Literal['UNDISTORT','DISTORT']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeNormal(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeNormalize(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeOutputFile(CompositorNode, NodeInternal, Node, bpy_struct):
    active_input_index: int | None
    """ Active input index in details view list

    :type: int | None
    """

    base_path: str
    """ Base output path for the image

    :type: str
    """

    file_slots: CompositorNodeOutputFileFileSlots
    """ 

    :type: CompositorNodeOutputFileFileSlots
    """

    format: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    layer_slots: CompositorNodeOutputFileLayerSlots
    """ 

    :type: CompositorNodeOutputFileLayerSlots
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodePixelate(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodePlaneTrackDeform(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    motion_blur_samples: int
    """ Number of motion blur samples

    :type: int
    """

    motion_blur_shutter: float
    """ Exposure for motion blur as a factor of FPS

    :type: float
    """

    plane_track_name: str
    """ 

    :type: str
    """

    tracking_object: str
    """ 

    :type: str
    """

    use_motion_blur: bool
    """ Use multi-sampled motion blur of the mask

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodePremulKey(CompositorNode, NodeInternal, Node, bpy_struct):
    mapping: typing.Literal["STRAIGHT_TO_PREMUL", "PREMUL_TO_STRAIGHT"]
    """ Conversion between premultiplied alpha and key alpha

    :type: typing.Literal['STRAIGHT_TO_PREMUL','PREMUL_TO_STRAIGHT']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeRGBToBW(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeRLayers(CompositorNode, NodeInternal, Node, bpy_struct):
    layer: typing.Literal["PLACEHOLDER"]
    """ 

    :type: typing.Literal['PLACEHOLDER']
    """

    scene: Scene | None
    """ 

    :type: Scene | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeRotate(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter rotation

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeScale(CompositorNode, NodeInternal, Node, bpy_struct):
    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the image fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    offset_x: float
    """ Offset image horizontally (factor of image size)

    :type: float
    """

    offset_y: float
    """ Offset image vertically (factor of image size)

    :type: float
    """

    space: typing.Literal["RELATIVE", "ABSOLUTE", "SCENE_SIZE", "RENDER_SIZE"]
    """ Coordinate space to scale relative to

    :type: typing.Literal['RELATIVE','ABSOLUTE','SCENE_SIZE','RENDER_SIZE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSepHSVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSepRGBA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSepYCCA(CompositorNode, NodeInternal, Node, bpy_struct):
    mode: typing.Literal["ITUBT601", "ITUBT709", "JFIF"]
    """ 

    :type: typing.Literal['ITUBT601','ITUBT709','JFIF']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSepYUVA(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSetAlpha(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSplitViewer(CompositorNode, NodeInternal, Node, bpy_struct):
    axis: typing.Literal["X", "Y"]
    """ 

    :type: typing.Literal['X','Y']
    """

    factor: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeStabilize(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter stabilization

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    invert: bool
    """ Invert stabilization to re-introduce motion to the frame

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSunBeams(CompositorNode, NodeInternal, Node, bpy_struct):
    ray_length: float
    """ Length of rays as a factor of the image size

    :type: float
    """

    source: bpy_prop_array[float]
    """ Source point of rays as a factor of the image width & height

    :type: bpy_prop_array[float]
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSwitch(CompositorNode, NodeInternal, Node, bpy_struct):
    check: bool
    """ Off: first socket, On: second socket

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeSwitchView(CompositorNode, NodeInternal, Node, bpy_struct):
    check: bool
    """ Off: first socket, On: second socket

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTexture(CompositorNode, NodeInternal, Node, bpy_struct):
    node_output: int
    """ For node-based textures, which output node to use

    :type: int
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTime(CompositorNode, NodeInternal, Node, bpy_struct):
    curve: CurveMapping
    """ 

    :type: CurveMapping
    """

    frame_end: int
    """ 

    :type: int
    """

    frame_start: int
    """ 

    :type: int
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTonemap(CompositorNode, NodeInternal, Node, bpy_struct):
    adaptation: float
    """ If 0, global; if 1, based on pixel intensity

    :type: float
    """

    contrast: float
    """ Set to 0 to use estimate from input image

    :type: float
    """

    correction: float
    """ If 0, same for all channels; if 1, each independent

    :type: float
    """

    gamma: float
    """ If not used, set to 1

    :type: float
    """

    intensity: float
    """ If less than zero, darkens image; otherwise, makes it brighter

    :type: float
    """

    key: float
    """ The value the average luminance is mapped to

    :type: float
    """

    offset: float
    """ Normally always 1, but can be used as an extra control to alter the brightness curve

    :type: float
    """

    tonemap_type: typing.Literal["RD_PHOTORECEPTOR", "RH_SIMPLE"]
    """ 

    :type: typing.Literal['RD_PHOTORECEPTOR','RH_SIMPLE']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTrackPos(CompositorNode, NodeInternal, Node, bpy_struct):
    clip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    frame_relative: int
    """ Frame to be used for relative position

    :type: int
    """

    position: typing.Literal[
        "ABSOLUTE", "RELATIVE_START", "RELATIVE_FRAME", "ABSOLUTE_FRAME"
    ]
    """ Which marker position to use for output

    :type: typing.Literal['ABSOLUTE','RELATIVE_START','RELATIVE_FRAME','ABSOLUTE_FRAME']
    """

    track_name: str
    """ 

    :type: str
    """

    tracking_object: str
    """ 

    :type: str
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTransform(CompositorNode, NodeInternal, Node, bpy_struct):
    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Method to use to filter transform

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTranslate(CompositorNode, NodeInternal, Node, bpy_struct):
    use_relative: bool
    """ Use relative (fraction of input image size) values to define translation

    :type: bool
    """

    wrap_axis: typing.Literal["NONE", "XAXIS", "YAXIS", "BOTH"]
    """ Wrap image on a specific axis

    :type: typing.Literal['NONE','XAXIS','YAXIS','BOTH']
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeTree(NodeTree, ID, bpy_struct):
    """Node tree consisting of linked nodes used for compositing"""

    chunk_size: typing.Literal["32", "64", "128", "256", "512", "1024"]
    """ Max size of a tile (smaller values gives better distribution of multiple threads, but more overhead)

    :type: typing.Literal['32','64','128','256','512','1024']
    """

    edit_quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ Quality when editing

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    render_quality: typing.Literal["HIGH", "MEDIUM", "LOW"]
    """ Quality when rendering

    :type: typing.Literal['HIGH','MEDIUM','LOW']
    """

    use_groupnode_buffer: bool
    """ Enable buffering of group nodes

    :type: bool
    """

    use_opencl: bool
    """ Enable GPU calculations

    :type: bool
    """

    use_two_pass: bool
    """ Use two pass execution during editing: first calculate fast nodes, second pass calculate all nodes

    :type: bool
    """

    use_viewer_border: bool
    """ Use boundaries for viewer nodes and composite backdrop

    :type: bool
    """

class CompositorNodeValToRGB(CompositorNode, NodeInternal, Node, bpy_struct):
    color_ramp: ColorRamp
    """ 

    :type: ColorRamp
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeValue(CompositorNode, NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeVecBlur(CompositorNode, NodeInternal, Node, bpy_struct):
    factor: float
    """ Scaling factor for motion vectors (actually, 'shutter speed', in frames)

    :type: float
    """

    samples: int
    """ 

    :type: int
    """

    speed_max: int
    """ Maximum speed, or zero for none

    :type: int
    """

    speed_min: int
    """ Minimum speed for a pixel to be blurred (used to separate background from foreground)

    :type: int
    """

    use_curved: bool
    """ Interpolate between frames in a Bezier curve, rather than linearly

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeViewer(CompositorNode, NodeInternal, Node, bpy_struct):
    center_x: float
    """ 

    :type: float
    """

    center_y: float
    """ 

    :type: float
    """

    tile_order: typing.Literal["CENTEROUT", "RANDOM", "BOTTOMUP", "RULE_OF_THIRDS"]
    """ Tile order

    :type: typing.Literal['CENTEROUT','RANDOM','BOTTOMUP','RULE_OF_THIRDS']
    """

    use_alpha: bool
    """ Colors are treated alpha premultiplied, or colors output straight (alpha gets set to 1)

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class CompositorNodeZcombine(CompositorNode, NodeInternal, Node, bpy_struct):
    use_alpha: bool
    """ Take Alpha channel into account when doing the Z operation

    :type: bool
    """

    use_antialias_z: bool
    """ Anti-alias the z-buffer to try to avoid artifacts, mostly useful for Blender renders

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    def update(self): ...

class ConsoleLine(bpy_struct):
    """Input line for the interactive console"""

    body: str
    """ Text in the line

    :type: str
    """

    current_character: int
    """ 

    :type: int
    """

    type: typing.Literal["OUTPUT", "INPUT", "INFO", "ERROR"]
    """ Console line type when used in scrollback

    :type: typing.Literal['OUTPUT','INPUT','INFO','ERROR']
    """

class Constraint(bpy_struct):
    """Constraint modifying the transformation of objects and bones"""

    active: bool | None
    """ Constraint is the one being edited

    :type: bool | None
    """

    error_location: float
    """ Amount of residual error in Blender space unit for constraints that work on position

    :type: float
    """

    error_rotation: float
    """ Amount of residual error in radians for constraints that work on orientation

    :type: float
    """

    influence: float
    """ Amount of influence constraint will have on the final solution

    :type: float
    """

    is_proxy_local: bool
    """ Constraint was added in this proxy instance (i.e. did not belong to source Armature)

    :type: bool
    """

    is_valid: bool
    """ Constraint has valid settings and can be evaluated

    :type: bool
    """

    mute: bool
    """ Enable/Disable Constraint

    :type: bool
    """

    name: str
    """ Constraint name

    :type: str
    """

    owner_space: typing.Literal["WORLD", "POSE", "LOCAL_WITH_PARENT", "LOCAL"]
    """ Space that owner is evaluated in

    :type: typing.Literal['WORLD','POSE','LOCAL_WITH_PARENT','LOCAL']
    """

    show_expanded: bool
    """ Constraint's panel is expanded in UI

    :type: bool
    """

    target_space: typing.Literal["WORLD", "POSE", "LOCAL_WITH_PARENT", "LOCAL"]
    """ Space that target is evaluated in

    :type: typing.Literal['WORLD','POSE','LOCAL_WITH_PARENT','LOCAL']
    """

    type: typing.Literal[
        "CAMERA_SOLVER",
        "FOLLOW_TRACK",
        "OBJECT_SOLVER",
        "COPY_LOCATION",
        "COPY_ROTATION",
        "COPY_SCALE",
        "COPY_TRANSFORMS",
        "LIMIT_DISTANCE",
        "LIMIT_LOCATION",
        "LIMIT_ROTATION",
        "LIMIT_SCALE",
        "MAINTAIN_VOLUME",
        "TRANSFORM",
        "TRANSFORM_CACHE",
        "CLAMP_TO",
        "DAMPED_TRACK",
        "IK",
        "LOCKED_TRACK",
        "SPLINE_IK",
        "STRETCH_TO",
        "TRACK_TO",
        "ACTION",
        "CHILD_OF",
        "FLOOR",
        "FOLLOW_PATH",
        "PIVOT",
        "RIGID_BODY_JOINT",
        "SHRINKWRAP",
    ]
    """ 

    :type: typing.Literal['CAMERA_SOLVER','FOLLOW_TRACK','OBJECT_SOLVER','COPY_LOCATION','COPY_ROTATION','COPY_SCALE','COPY_TRANSFORMS','LIMIT_DISTANCE','LIMIT_LOCATION','LIMIT_ROTATION','LIMIT_SCALE','MAINTAIN_VOLUME','TRANSFORM','TRANSFORM_CACHE','CLAMP_TO','DAMPED_TRACK','IK','LOCKED_TRACK','SPLINE_IK','STRETCH_TO','TRACK_TO','ACTION','CHILD_OF','FLOOR','FOLLOW_PATH','PIVOT','RIGID_BODY_JOINT','SHRINKWRAP']
    """

class ConstraintActuator(Actuator, bpy_struct):
    """Actuator to handle Constraints"""

    angle_max: float
    """ Maximum angle allowed with target direction (no correction is done if angle with target direction is between min and max)

    :type: float
    """

    angle_min: float
    """ Minimum angle to maintain with target direction (no correction is done if angle with target direction is between min and max)

    :type: float
    """

    damping: int
    """ Damping factor: time constant (in frame) of low pass filter

    :type: int
    """

    damping_rotation: int
    """ Use a different damping for orientation

    :type: int
    """

    direction: typing.Literal[
        "NONE", "DIRPX", "DIRPY", "DIRPZ", "DIRNX", "DIRNY", "DIRNZ"
    ]
    """ Direction of the ray

    :type: typing.Literal['NONE','DIRPX','DIRPY','DIRPZ','DIRNX','DIRNY','DIRNZ']
    """

    direction_axis: typing.Literal[
        "NONE", "DIRPX", "DIRPY", "DIRPZ", "DIRNX", "DIRNY", "DIRNZ"
    ]
    """ Select the axis to be aligned along the reference direction

    :type: typing.Literal['NONE','DIRPX','DIRPY','DIRPZ','DIRNX','DIRNY','DIRNZ']
    """

    direction_axis_pos: typing.Literal["NONE", "DIRPX", "DIRPY", "DIRPZ"]
    """ Select the axis to be aligned along the reference direction

    :type: typing.Literal['NONE','DIRPX','DIRPY','DIRPZ']
    """

    distance: float
    """ Keep this distance to target

    :type: float
    """

    fh_damping: float
    """ Damping factor of the force field spring

    :type: float
    """

    fh_force: float
    """ Spring force within the force field area

    :type: float
    """

    fh_height: float
    """ Height of the force field area

    :type: float
    """

    limit: typing.Literal["NONE", "LOCX", "LOCY", "LOCZ"]
    """ 

    :type: typing.Literal['NONE','LOCX','LOCY','LOCZ']
    """

    limit_max: float
    """ 

    :type: float
    """

    limit_min: float
    """ 

    :type: float
    """

    material: str
    """ Ray detects only Objects with this material

    :type: str
    """

    mode: typing.Literal["LOC", "DIST", "ORI", "FH"]
    """ The type of the constraint

    :type: typing.Literal['LOC','DIST','ORI','FH']
    """

    property: str
    """ Ray detects only Objects with this property

    :type: str
    """

    range: float
    """ Maximum length of ray

    :type: float
    """

    rotation_max: bpy_prop_array[float]
    """ Reference Direction

    :type: bpy_prop_array[float]
    """

    time: int
    """ Maximum activation time in frame, 0 for unlimited

    :type: int
    """

    use_fh_normal: bool
    """ Add a horizontal spring force on slopes

    :type: bool
    """

    use_fh_paralel_axis: bool
    """ Keep object axis parallel to normal

    :type: bool
    """

    use_force_distance: bool
    """ Force distance of object to point of impact of ray

    :type: bool
    """

    use_local: bool
    """ Set ray along object's axis or global axis

    :type: bool
    """

    use_material_detect: bool
    """ Detect material instead of property

    :type: bool
    """

    use_normal: bool
    """ Set object axis along (local axis) or parallel (global axis) to the normal at hit position

    :type: bool
    """

    use_persistent: bool
    """ Persistent actuator: stays active even if ray does not reach target

    :type: bool
    """

class ConstraintTarget(bpy_struct):
    """Target object for multi-target constraints"""

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

class Context(bpy_struct):
    """Current windowmanager and data context"""

    area: Area
    """ 

    :type: Area
    """

    blend_data: BlendData
    """ 

    :type: BlendData
    """

    mode: typing.Literal[
        "EDIT_MESH",
        "EDIT_CURVE",
        "EDIT_SURFACE",
        "EDIT_TEXT",
        "EDIT_ARMATURE",
        "EDIT_METABALL",
        "EDIT_LATTICE",
        "POSE",
        "SCULPT",
        "PAINT_WEIGHT",
        "PAINT_VERTEX",
        "PAINT_TEXTURE",
        "PARTICLE",
        "OBJECT",
    ]
    """ 

    :type: typing.Literal['EDIT_MESH','EDIT_CURVE','EDIT_SURFACE','EDIT_TEXT','EDIT_ARMATURE','EDIT_METABALL','EDIT_LATTICE','POSE','SCULPT','PAINT_WEIGHT','PAINT_VERTEX','PAINT_TEXTURE','PARTICLE','OBJECT']
    """

    region: Region
    """ 

    :type: Region
    """

    region_data: RegionView3D
    """ 

    :type: RegionView3D
    """

    scene: Scene
    """ 

    :type: Scene
    """

    screen: Screen
    """ 

    :type: Screen
    """

    space_data: Space
    """ 

    :type: Space
    """

    tool_settings: ToolSettings
    """ 

    :type: ToolSettings
    """

    user_preferences: UserPreferences
    """ 

    :type: UserPreferences
    """

    window: Window
    """ 

    :type: Window
    """

    window_manager: WindowManager
    """ 

    :type: WindowManager
    """

    visible_objects: list[Object]
    """ 

    :type: list[Object]
    """

    visible_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selectable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selectable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selected_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    editable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    selected_editable_objects: list[Object]
    """ 

    :type: list[Object]
    """

    selected_editable_bases: list[ObjectBase]
    """ 

    :type: list[ObjectBase]
    """

    visible_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    selected_editable_bones: list[EditBone]
    """ 

    :type: list[EditBone]
    """

    visible_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    selected_pose_bones: list[PoseBone]
    """ 

    :type: list[PoseBone]
    """

    active_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    active_pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    active_base: ObjectBase | None
    """ 

    :type: ObjectBase | None
    """

    active_object: Object | None
    """ 

    :type: Object | None
    """

    object: Object | None
    """ 

    :type: Object | None
    """

    edit_object: Object | None
    """ 

    :type: Object | None
    """

    sculpt_object: Object | None
    """ 

    :type: Object | None
    """

    vertex_paint_object: Object | None
    """ 

    :type: Object | None
    """

    weight_paint_object: Object | None
    """ 

    :type: Object | None
    """

    image_paint_object: Object | None
    """ 

    :type: Object | None
    """

    particle_edit_object: Object | None
    """ 

    :type: Object | None
    """

    sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    selected_editable_sequences: list[Sequence]
    """ 

    :type: list[Sequence]
    """

    gpencil_data: typing.Any
    gpencil_data_owner: ID | None
    """ 

    :type: ID | None
    """

    visible_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_layers: list[GPencilLayer]
    """ 

    :type: list[GPencilLayer]
    """

    editable_gpencil_strokes: list[GPencilStroke]
    """ 

    :type: list[GPencilStroke]
    """

    active_gpencil_layer: list[GPencilLayer] | None
    """ 

    :type: list[GPencilLayer] | None
    """

    active_gpencil_frame: typing.Any
    active_gpencil_palette: list[GPencilPalette] | None
    """ 

    :type: list[GPencilPalette] | None
    """

    active_gpencil_palettecolor: list[GPencilPaletteColor] | None
    """ 

    :type: list[GPencilPaletteColor] | None
    """

    active_gpencil_brush: GPencilSculptBrush | None
    """ 

    :type: GPencilSculptBrush | None
    """

    active_operator: Operator | None
    """ 

    :type: Operator | None
    """

    texture_slot: MaterialTextureSlot | None
    """ 

    :type: MaterialTextureSlot | None
    """

    world: World | None
    """ 

    :type: World | None
    """

    mesh: Mesh | None
    """ 

    :type: Mesh | None
    """

    armature: Armature | None
    """ 

    :type: Armature | None
    """

    lattice: Lattice | None
    """ 

    :type: Lattice | None
    """

    curve: Curve | None
    """ 

    :type: Curve | None
    """

    meta_ball: MetaBall | None
    """ 

    :type: MetaBall | None
    """

    lamp: Lamp | None
    """ 

    :type: Lamp | None
    """

    speaker: Speaker | None
    """ 

    :type: Speaker | None
    """

    camera: Camera | None
    """ 

    :type: Camera | None
    """

    material: Material | None
    """ 

    :type: Material | None
    """

    material_slot: MaterialSlot | None
    """ 

    :type: MaterialSlot | None
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_user: ID | None
    """ 

    :type: ID | None
    """

    texture_user_property: Property | None
    """ 

    :type: Property | None
    """

    bone: Bone | None
    """ 

    :type: Bone | None
    """

    edit_bone: EditBone | None
    """ 

    :type: EditBone | None
    """

    pose_bone: PoseBone | None
    """ 

    :type: PoseBone | None
    """

    particle_system: ParticleSystem | None
    """ 

    :type: ParticleSystem | None
    """

    particle_system_editable: ParticleSystem | None
    """ 

    :type: ParticleSystem | None
    """

    particle_settings: ParticleSettings | None
    """ 

    :type: ParticleSettings | None
    """

    cloth: ClothModifier | None
    """ 

    :type: ClothModifier | None
    """

    soft_body: SoftBodyModifier | None
    """ 

    :type: SoftBodyModifier | None
    """

    fluid: FluidSimulationModifier | None
    """ 

    :type: FluidSimulationModifier | None
    """

    smoke: SmokeModifier | None
    """ 

    :type: SmokeModifier | None
    """

    collision: CollisionModifier | None
    """ 

    :type: CollisionModifier | None
    """

    brush: Brush | None
    """ 

    :type: Brush | None
    """

    dynamic_paint: DynamicPaintModifier | None
    """ 

    :type: DynamicPaintModifier | None
    """

    line_style: FreestyleLineStyle | None
    """ 

    :type: FreestyleLineStyle | None
    """

    edit_image: Image | None
    """ 

    :type: Image | None
    """

    edit_mask: Mask | None
    """ 

    :type: Mask | None
    """

    selected_nodes: list[Node]
    """ 

    :type: list[Node]
    """

    active_node: Node | None
    """ 

    :type: Node | None
    """

    edit_text: Text | None
    """ 

    :type: Text | None
    """

    edit_movieclip: MovieClip | None
    """ 

    :type: MovieClip | None
    """

    def copy(self): ...

class ControlFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects controlling the motion of fluid in the simulation"""

    attraction_radius: float
    """ Force field radius around the control object

    :type: float
    """

    attraction_strength: float
    """ Force strength for directional attraction towards the control object

    :type: float
    """

    end_time: float
    """ Time when the control particles are deactivated

    :type: float
    """

    quality: float
    """ Quality which is used for object sampling (higher = better but slower)

    :type: float
    """

    start_time: float
    """ Time when the control particles are activated

    :type: float
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse control object movement

    :type: bool
    """

    velocity_radius: float
    """ Force field radius around the control object

    :type: float
    """

    velocity_strength: float
    """ Force strength of how much of the control object's velocity is influencing the fluid velocity

    :type: float
    """

class Controller(bpy_struct):
    """Game engine logic brick to process events, connecting sensors to actuators"""

    active: bool | None
    """ Set the active state of the controller

    :type: bool | None
    """

    actuators: bpy_prop_collection[Actuator]
    """ The list containing the actuators connected to the controller

    :type: bpy_prop_collection[Actuator]
    """

    name: str
    """ 

    :type: str
    """

    show_expanded: bool
    """ Set controller expanded in the user interface

    :type: bool
    """

    states: int
    """ Set Controller state index (1 to 30)

    :type: int
    """

    type: typing.Literal[
        "LOGIC_AND",
        "LOGIC_OR",
        "LOGIC_NAND",
        "LOGIC_NOR",
        "LOGIC_XOR",
        "LOGIC_XNOR",
        "EXPRESSION",
        "PYTHON",
    ]
    """ 

    :type: typing.Literal['LOGIC_AND','LOGIC_OR','LOGIC_NAND','LOGIC_NOR','LOGIC_XOR','LOGIC_XNOR','EXPRESSION','PYTHON']
    """

    use_priority: bool
    """ Mark controller for execution before all non-marked controllers (good for startup scripts)

    :type: bool
    """

    def link(self, sensor: Sensor | None = None, actuator: Actuator | None = None):
        """Link the controller with a sensor/actuator

        :param sensor: Sensor to link the controller to
        :type sensor: Sensor | None
        :param actuator: Actuator to link the controller to
        :type actuator: Actuator | None
        """

    def unlink(self, sensor: Sensor | None = None, actuator: Actuator | None = None):
        """Unlink the controller from a sensor/actuator

        :param sensor: Sensor to unlink the controller from
        :type sensor: Sensor | None
        :param actuator: Actuator to unlink the controller from
        :type actuator: Actuator | None
        """

class CopyLocationConstraint(Constraint, bpy_struct):
    """Copy the location of the target"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    invert_x: bool
    """ Invert the X location

    :type: bool
    """

    invert_y: bool
    """ Invert the Y location

    :type: bool
    """

    invert_z: bool
    """ Invert the Z location

    :type: bool
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_offset: bool
    """ Add original location into copied location

    :type: bool
    """

    use_x: bool
    """ Copy the target's X location

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y location

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z location

    :type: bool
    """

class CopyRotationConstraint(Constraint, bpy_struct):
    """Copy the rotation of the target"""

    invert_x: bool
    """ Invert the X rotation

    :type: bool
    """

    invert_y: bool
    """ Invert the Y rotation

    :type: bool
    """

    invert_z: bool
    """ Invert the Z rotation

    :type: bool
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_offset: bool
    """ Add original rotation into copied rotation

    :type: bool
    """

    use_x: bool
    """ Copy the target's X rotation

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y rotation

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z rotation

    :type: bool
    """

class CopyScaleConstraint(Constraint, bpy_struct):
    """Copy the scale of the target"""

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_offset: bool
    """ Add original scale into copied scale

    :type: bool
    """

    use_x: bool
    """ Copy the target's X scale

    :type: bool
    """

    use_y: bool
    """ Copy the target's Y scale

    :type: bool
    """

    use_z: bool
    """ Copy the target's Z scale

    :type: bool
    """

class CopyTransformsConstraint(Constraint, bpy_struct):
    """Copy all the transforms of the target"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class CorrectiveSmoothModifier(Modifier, bpy_struct):
    """Correct distortion caused by deformation"""

    factor: float
    """ Smooth factor effect

    :type: float
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    is_bind: bool
    """ 

    :type: bool
    """

    iterations: int
    """ 

    :type: int
    """

    rest_source: typing.Literal["ORCO", "BIND"]
    """ Select the source of rest positions

    :type: typing.Literal['ORCO','BIND']
    """

    smooth_type: typing.Literal["SIMPLE", "LENGTH_WEIGHTED"]
    """ Method used for smoothing

    :type: typing.Literal['SIMPLE','LENGTH_WEIGHTED']
    """

    use_only_smooth: bool
    """ Apply smoothing without reconstructing the surface

    :type: bool
    """

    use_pin_boundary: bool
    """ Excludes boundary vertices from being smoothed

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class CrossSequence(EffectSequence, Sequence, bpy_struct):
    """Cross Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class Curve(ID, bpy_struct):
    """Curve data-block storing curves, splines and NURBS"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bevel_depth: float
    """ Bevel depth when not using a bevel object

    :type: float
    """

    bevel_factor_end: float
    """ Factor that defines to where beveling of spline happens (0=to the very beginning, 1=to the very end)

    :type: float
    """

    bevel_factor_mapping_end: typing.Literal["RESOLUTION", "SEGMENTS", "SPLINE"]
    """ Determines how the end bevel factor is mapped to a spline

    :type: typing.Literal['RESOLUTION','SEGMENTS','SPLINE']
    """

    bevel_factor_mapping_start: typing.Literal["RESOLUTION", "SEGMENTS", "SPLINE"]
    """ Determines how the start bevel factor is mapped to a spline

    :type: typing.Literal['RESOLUTION','SEGMENTS','SPLINE']
    """

    bevel_factor_start: float
    """ Factor that defines from where beveling of spline happens (0=from the very beginning, 1=from the very end)

    :type: float
    """

    bevel_object: Object | None
    """ Curve object name that defines the bevel shape

    :type: Object | None
    """

    bevel_resolution: int
    """ Bevel resolution when depth is non-zero and no specific bevel object has been defined

    :type: int
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    dimensions: typing.Literal["2D", "3D"]
    """ Select 2D or 3D curve type

    :type: typing.Literal['2D','3D']
    """

    eval_time: float
    """ Parametric position along the length of the curve that Objects 'following' it should be at (position is evaluated by dividing by the 'Path Length' value)

    :type: float
    """

    extrude: float
    """ Amount of curve extrusion when not using a bevel object

    :type: float
    """

    fill_mode: typing.Literal["FULL", "BACK", "FRONT", "HALF"]
    """ Mode of filling curve

    :type: typing.Literal['FULL','BACK','FRONT','HALF']
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    offset: float
    """ Offset the curve to adjust the width of a text

    :type: float
    """

    path_duration: int
    """ The number of frames that are needed to traverse the path, defining the maximum value for the 'Evaluation Time' setting

    :type: int
    """

    render_resolution_u: int
    """ Surface resolution in U direction used while rendering (zero uses preview resolution)

    :type: int
    """

    render_resolution_v: int
    """ Surface resolution in V direction used while rendering (zero uses preview resolution)

    :type: int
    """

    resolution_u: int
    """ Surface resolution in U direction

    :type: int
    """

    resolution_v: int
    """ Surface resolution in V direction

    :type: int
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    show_handles: bool
    """ Display Bezier handles in editmode

    :type: bool
    """

    show_normal_face: bool
    """ Display 3D curve normals in editmode

    :type: bool
    """

    splines: CurveSplines
    """ Collection of splines in this curve data object

    :type: CurveSplines
    """

    taper_object: Object | None
    """ Curve object name that defines the taper (width)

    :type: Object | None
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    twist_mode: typing.Literal["Z_UP", "MINIMUM", "TANGENT"]
    """ The type of tilt calculation for 3D Curves

    :type: typing.Literal['Z_UP','MINIMUM','TANGENT']
    """

    twist_smooth: float
    """ Smoothing iteration for tangents

    :type: float
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    use_deform_bounds: bool
    """ Option for curve-deform: Use the mesh bounds to clamp the deformation

    :type: bool
    """

    use_fill_caps: bool
    """ Fill caps for beveled curves

    :type: bool
    """

    use_fill_deform: bool
    """ Fill curve after applying shape keys and all modifiers

    :type: bool
    """

    use_map_taper: bool
    """ Map effect of taper object on actually beveled curve

    :type: bool
    """

    use_path: bool
    """ Enable the curve to become a translation path

    :type: bool
    """

    use_path_follow: bool
    """ Make curve path children to rotate along the path

    :type: bool
    """

    use_radius: bool
    """ Option for paths and curve-deform: apply the curve radius with path following it and deforming

    :type: bool
    """

    use_stretch: bool
    """ Option for curve-deform: make deformed child to stretch along entire path

    :type: bool
    """

    use_uv_as_generated: bool
    """ Uses the UV values as Generated textured coordinates

    :type: bool
    """

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | None = False,
    ):
        """Transform curve by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | None
        """

    def validate_material_indices(self) -> bool:
        """Validate material indices of splines or letters, return True when the curve has had invalid indices corrected (to default 0)

        :return: Result
        :rtype: bool
        """

class CurveMap(bpy_struct):
    """Curve in a curve mapping"""

    extend: typing.Literal["HORIZONTAL", "EXTRAPOLATED"]
    """ Extrapolate the curve or extend it horizontally

    :type: typing.Literal['HORIZONTAL','EXTRAPOLATED']
    """

    points: CurveMapPoints
    """ 

    :type: CurveMapPoints
    """

    def evaluate(self, position: float | None) -> float:
        """Evaluate curve at given location

        :param position: Position, Position to evaluate curve at
        :type position: float | None
        :return: Value, Value of curve at given location
        :rtype: float
        """

class CurveMapPoint(bpy_struct):
    """Point of a curve used for a curve mapping"""

    handle_type: typing.Literal["AUTO", "AUTO_CLAMPED", "VECTOR"]
    """ Curve interpolation at this point: Bezier or vector

    :type: typing.Literal['AUTO','AUTO_CLAMPED','VECTOR']
    """

    location: bpy_prop_array[float]
    """ X/Y coordinates of the curve point

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection state of the curve point

    :type: bool
    """

class CurveMapping(bpy_struct):
    """Curve mapping to map color, vector and scalar values to other values using a user defined curve"""

    black_level: bpy_prop_array[float]
    """ For RGB curves, the color that black is mapped to

    :type: bpy_prop_array[float]
    """

    clip_max_x: float
    """ 

    :type: float
    """

    clip_max_y: float
    """ 

    :type: float
    """

    clip_min_x: float
    """ 

    :type: float
    """

    clip_min_y: float
    """ 

    :type: float
    """

    curves: bpy_prop_collection[CurveMap]
    """ 

    :type: bpy_prop_collection[CurveMap]
    """

    use_clip: bool
    """ Force the curve view to fit a defined boundary

    :type: bool
    """

    white_level: bpy_prop_array[float]
    """ For RGB curves, the color that white is mapped to

    :type: bpy_prop_array[float]
    """

    def update(self):
        """Update curve mapping after making changes"""

    def initialize(self):
        """Initialize curve"""

class CurveModifier(Modifier, bpy_struct):
    """Curve deformation modifier"""

    deform_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ The axis that the curve deforms along

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    object: Object | None
    """ Curve object to deform with

    :type: Object | None
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class CurvePaintSettings(bpy_struct):
    corner_angle: float
    """ Angles above this are considered corners

    :type: float
    """

    curve_type: typing.Literal["POLY", "BEZIER"]
    """ Type of curve to use for new strokes

    :type: typing.Literal['POLY','BEZIER']
    """

    depth_mode: typing.Literal["CURSOR", "SURFACE"]
    """ Method of projecting depth

    :type: typing.Literal['CURSOR','SURFACE']
    """

    error_threshold: int
    """ Allow deviation for a smoother, less precise line

    :type: int
    """

    fit_method: typing.Literal["REFIT", "SPLIT"]
    """ Curve fitting method

    :type: typing.Literal['REFIT','SPLIT']
    """

    radius_max: float
    """ Radius to use when the maximum pressure is applied (or when a tablet isn't used)

    :type: float
    """

    radius_min: float
    """ Minimum radius when the minimum pressure is applied (also the minimum when tapering)

    :type: float
    """

    radius_taper_end: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    radius_taper_start: float
    """ Taper factor for the radius of each point along the curve

    :type: float
    """

    surface_offset: float
    """ Offset the stroke from the surface

    :type: float
    """

    surface_plane: typing.Literal["NORMAL_VIEW", "NORMAL_SURFACE", "VIEW"]
    """ Plane for projected stroke

    :type: typing.Literal['NORMAL_VIEW','NORMAL_SURFACE','VIEW']
    """

    use_corners_detect: bool
    """ Detect corners and use non-aligned handles

    :type: bool
    """

    use_offset_absolute: bool
    """ Apply a fixed offset (don't scale by the radius)

    :type: bool
    """

    use_pressure_radius: bool
    """ Map tablet pressure to curve radius

    :type: bool
    """

    use_stroke_endpoints: bool
    """ Use the start of the stroke for the depth

    :type: bool
    """

class CurvesModifier(SequenceModifier, bpy_struct):
    """RGB curves modifier for sequence strip"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

class CyclesCameraSettings(PropertyGroup, bpy_struct):
    aperture_blades: int
    """ Number of blades in aperture for polygonal bokeh (at least 3)

    :type: int
    """

    aperture_fstop: float
    """ F-stop ratio (lower numbers give more defocus, higher numbers give a sharper image)

    :type: float
    """

    aperture_ratio: float
    """ Distortion to simulate anamorphic lens bokeh

    :type: float
    """

    aperture_rotation: float
    """ Rotation of blades in aperture

    :type: float
    """

    aperture_size: float
    """ Radius of the aperture for depth of field (higher values give more defocus)

    :type: float
    """

    aperture_type: typing.Literal["RADIUS", "FSTOP"]
    """ Use f-stop number or aperture radius

    :type: typing.Literal['RADIUS','FSTOP']
    """

    fisheye_fov: float
    """ Field of view for the fisheye lens

    :type: float
    """

    fisheye_lens: float
    """ Lens focal length (mm)

    :type: float
    """

    latitude_max: float
    """ Maximum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    latitude_min: float
    """ Minimum latitude (vertical angle) for the equirectangular lens

    :type: float
    """

    longitude_max: float
    """ Maximum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    longitude_min: float
    """ Minimum longitude (horizontal angle) for the equirectangular lens

    :type: float
    """

    panorama_type: typing.Literal[
        "EQUIRECTANGULAR", "FISHEYE_EQUIDISTANT", "FISHEYE_EQUISOLID", "MIRRORBALL"
    ]
    """ Distortion to use for the calculation

    :type: typing.Literal['EQUIRECTANGULAR','FISHEYE_EQUIDISTANT','FISHEYE_EQUISOLID','MIRRORBALL']
    """

class CyclesCurveRenderSettings(PropertyGroup, bpy_struct):
    cull_backfacing: bool
    """ Do not test the back-face of each strand

    :type: bool
    """

    maximum_width: float
    """ Maximum extension that strand radius can be increased by

    :type: float
    """

    minimum_width: float
    """ Minimal pixel width for strands (0 - deactivated)

    :type: float
    """

    primitive: typing.Literal["TRIANGLES", "LINE_SEGMENTS", "CURVE_SEGMENTS"]
    """ Type of primitive used for hair rendering

    :type: typing.Literal['TRIANGLES','LINE_SEGMENTS','CURVE_SEGMENTS']
    """

    resolution: int
    """ Resolution of generated mesh

    :type: int
    """

    shape: typing.Literal["RIBBONS", "THICK"]
    """ Form of hair

    :type: typing.Literal['RIBBONS','THICK']
    """

    subdivisions: int
    """ Number of subdivisions used in Cardinal curve intersection (power of 2)

    :type: int
    """

    use_curves: bool
    """ Activate Cycles hair rendering for particle system

    :type: bool
    """

class CyclesCurveSettings(PropertyGroup, bpy_struct):
    radius_scale: float
    """ Multiplier of width properties

    :type: float
    """

    root_width: float
    """ Strand's width at root

    :type: float
    """

    shape: float
    """ Strand shape parameter

    :type: float
    """

    tip_width: float
    """ Strand's width at tip

    :type: float
    """

    use_closetip: bool
    """ Set tip radius to zero

    :type: bool
    """

class CyclesDeviceSettings(PropertyGroup, bpy_struct):
    id: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    type: typing.Literal["CPU", "CUDA", "OPENCL"]
    """ 

    :type: typing.Literal['CPU','CUDA','OPENCL']
    """

    use: bool
    """ 

    :type: bool
    """

class CyclesLampSettings(PropertyGroup, bpy_struct):
    cast_shadow: bool
    """ Lamp casts shadows

    :type: bool
    """

    is_portal: bool
    """ Use this area lamp to guide sampling of the background, note that this will make the lamp invisible

    :type: bool
    """

    max_bounces: int
    """ Maximum number of bounces the light will contribute to the render

    :type: int
    """

    samples: int
    """ Number of light samples to render for each AA sample

    :type: int
    """

    use_multiple_importance_sampling: bool
    """ Use multiple importance sampling for the lamp, reduces noise for area lamps and sharp glossy materials

    :type: bool
    """

class CyclesMaterialSettings(PropertyGroup, bpy_struct):
    displacement_method: typing.Literal["BUMP", "TRUE", "BOTH"]
    """ Method to use for the displacement

    :type: typing.Literal['BUMP','TRUE','BOTH']
    """

    homogeneous_volume: bool
    """ When using volume rendering, assume volume has the same density everywhere (not using any textures), for faster rendering

    :type: bool
    """

    sample_as_light: bool
    """ Use multiple importance sampling for this material, disabling may reduce overall noise for large objects that emit little light compared to other light sources

    :type: bool
    """

    use_transparent_shadow: bool
    """ Use transparent shadows for this material if it contains a Transparent BSDF, disabling will render faster but not give accurate shadows

    :type: bool
    """

    volume_interpolation: typing.Literal["LINEAR", "CUBIC"]
    """ Interpolation method to use for smoke/fire volumes

    :type: typing.Literal['LINEAR','CUBIC']
    """

    volume_sampling: typing.Literal["DISTANCE", "EQUIANGULAR", "MULTIPLE_IMPORTANCE"]
    """ Sampling method to use for volumes

    :type: typing.Literal['DISTANCE','EQUIANGULAR','MULTIPLE_IMPORTANCE']
    """

class CyclesMeshSettings(PropertyGroup, bpy_struct): ...

class CyclesObjectSettings(PropertyGroup, bpy_struct):
    dicing_rate: float
    """ Multiplier for scene dicing rate (located in the Geometry Panel)

    :type: float
    """

    motion_steps: int
    """ Control accuracy of deformation motion blur, more steps gives more memory usage (actual number of steps is 2^(steps - 1))

    :type: int
    """

    use_adaptive_subdivision: bool
    """ Use adaptive render time subdivision

    :type: bool
    """

    use_camera_cull: bool
    """ Allow this object and its duplicators to be culled by camera space culling

    :type: bool
    """

    use_deform_motion: bool
    """ Use deformation motion blur for this object

    :type: bool
    """

    use_distance_cull: bool
    """ Allow this object and its duplicators to be culled by distance from camera

    :type: bool
    """

    use_motion_blur: bool
    """ Use motion blur for this object

    :type: bool
    """

class CyclesPreferences(AddonPreferences, bpy_struct):
    compute_device_type: str
    """ Device to use for computation (rendering with Cycles)

    :type: str
    """

    devices: bpy_prop_collection[CyclesDeviceSettings]
    """ 

    :type: bpy_prop_collection[CyclesDeviceSettings]
    """

    def draw(self, context):
        """

        :param context:
        """

    def draw_impl(self, layout, context):
        """

        :param layout:
        :param context:
        """

    def get_device_types(self, context):
        """

        :param context:
        """

    def get_devices(self): ...
    def get_num_gpu_devices(self): ...
    def has_active_device(self): ...

class CyclesRenderSettings(PropertyGroup, bpy_struct):
    aa_samples: int
    """ Number of antialiasing samples to render for each pixel

    :type: int
    """

    ao_samples: int
    """ Number of ambient occlusion samples to render for each AA sample

    :type: int
    """

    bake_type: typing.Literal[
        "COMBINED",
        "AO",
        "SHADOW",
        "NORMAL",
        "UV",
        "EMIT",
        "ENVIRONMENT",
        "DIFFUSE",
        "GLOSSY",
        "TRANSMISSION",
        "SUBSURFACE",
    ]
    """ Type of pass to bake

    :type: typing.Literal['COMBINED','AO','SHADOW','NORMAL','UV','EMIT','ENVIRONMENT','DIFFUSE','GLOSSY','TRANSMISSION','SUBSURFACE']
    """

    blur_glossy: float
    """ Adaptively blur glossy shaders after blurry bounces, to reduce noise at the cost of accuracy

    :type: float
    """

    camera_cull_margin: float
    """ Margin for the camera space culling

    :type: float
    """

    caustics_reflective: bool
    """ Use reflective caustics, resulting in a brighter image (more noise but added realism)

    :type: bool
    """

    caustics_refractive: bool
    """ Use refractive caustics, resulting in a brighter image (more noise but added realism)

    :type: bool
    """

    debug_bvh_time_steps: int
    """ Split BVH primitives by this number of time steps to speed up render time in cost of memory

    :type: int
    """

    debug_bvh_type: typing.Literal["DYNAMIC_BVH", "STATIC_BVH"]
    """ Choose between faster updates, or faster render

    :type: typing.Literal['DYNAMIC_BVH','STATIC_BVH']
    """

    debug_cancel_timeout: float
    """ 

    :type: float
    """

    debug_opencl_device_type: typing.Literal[
        "NONE", "ALL", "DEFAULT", "CPU", "GPU", "ACCELERATOR"
    ]
    """ 

    :type: typing.Literal['NONE','ALL','DEFAULT','CPU','GPU','ACCELERATOR']
    """

    debug_opencl_kernel_type: typing.Literal["DEFAULT", "MEGA", "SPLIT"]
    """ 

    :type: typing.Literal['DEFAULT','MEGA','SPLIT']
    """

    debug_reset_timeout: float
    """ 

    :type: float
    """

    debug_text_timeout: float
    """ 

    :type: float
    """

    debug_tile_size: int
    """ 

    :type: int
    """

    debug_use_cpu_avx: bool
    """ 

    :type: bool
    """

    debug_use_cpu_avx2: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse2: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse3: bool
    """ 

    :type: bool
    """

    debug_use_cpu_sse41: bool
    """ 

    :type: bool
    """

    debug_use_cuda_adaptive_compile: bool
    """ 

    :type: bool
    """

    debug_use_hair_bvh: bool
    """ Use special type BVH optimized for hair (uses more ram but renders faster)

    :type: bool
    """

    debug_use_opencl_debug: bool
    """ 

    :type: bool
    """

    debug_use_qbvh: bool
    """ 

    :type: bool
    """

    debug_use_spatial_splits: bool
    """ Use BVH spatial splits: longer builder time, faster render

    :type: bool
    """

    device: typing.Literal["CPU", "GPU"]
    """ Device to use for rendering

    :type: typing.Literal['CPU','GPU']
    """

    dicing_rate: float
    """ Size of a micropolygon in pixels

    :type: float
    """

    diffuse_bounces: int
    """ Maximum number of diffuse reflection bounces, bounded by total maximum

    :type: int
    """

    diffuse_samples: int
    """ Number of diffuse bounce samples to render for each AA sample

    :type: int
    """

    distance_cull_margin: float
    """ Cull objects which are further away from camera than this distance

    :type: float
    """

    feature_set: typing.Literal["SUPPORTED", "EXPERIMENTAL"]
    """ Feature set to use for rendering

    :type: typing.Literal['SUPPORTED','EXPERIMENTAL']
    """

    film_exposure: float
    """ Image brightness scale

    :type: float
    """

    film_transparent: bool
    """ World background is transparent with premultiplied alpha

    :type: bool
    """

    filter_type: typing.Literal["BOX", "GAUSSIAN", "BLACKMAN_HARRIS"]
    """ Pixel filter type

    :type: typing.Literal['BOX','GAUSSIAN','BLACKMAN_HARRIS']
    """

    filter_width: float
    """ Pixel filter width

    :type: float
    """

    glossy_bounces: int
    """ Maximum number of glossy reflection bounces, bounded by total maximum

    :type: int
    """

    glossy_samples: int
    """ Number of glossy bounce samples to render for each AA sample

    :type: int
    """

    light_sampling_threshold: float
    """ Probabilistically terminate light samples when the light contribution is below this threshold (more noise but faster rendering). Zero disables the test and never ignores lights

    :type: float
    """

    max_bounces: int
    """ Total maximum number of bounces

    :type: int
    """

    max_subdivisions: int
    """ Stop subdividing when this level is reached even if the dice rate would produce finer tessellation

    :type: int
    """

    mesh_light_samples: int
    """ Number of mesh emission light samples to render for each AA sample

    :type: int
    """

    min_bounces: int
    """ Minimum number of bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier)

    :type: int
    """

    motion_blur_position: typing.Literal["START", "CENTER", "END"]
    """ Offset for the shutter's time interval, allows to change the motion blur trails

    :type: typing.Literal['START','CENTER','END']
    """

    pixel_filter_type: typing.Literal["BOX", "GAUSSIAN", "BLACKMAN_HARRIS"]
    """ Pixel filter type

    :type: typing.Literal['BOX','GAUSSIAN','BLACKMAN_HARRIS']
    """

    preview_aa_samples: int
    """ Number of antialiasing samples to render in the viewport, unlimited if 0

    :type: int
    """

    preview_active_layer: bool
    """ Preview active render layer in viewport

    :type: bool
    """

    preview_dicing_rate: float
    """ Size of a micropolygon in pixels during preview render

    :type: float
    """

    preview_pause: bool
    """ Pause all viewport preview renders

    :type: bool
    """

    preview_samples: int
    """ Number of samples to render in the viewport, unlimited if 0

    :type: int
    """

    preview_start_resolution: int
    """ Resolution to start rendering preview at, progressively increasing it to the full viewport size

    :type: int
    """

    progressive: typing.Literal["BRANCHED_PATH", "PATH"]
    """ Method to sample lights and materials

    :type: typing.Literal['BRANCHED_PATH','PATH']
    """

    rolling_shutter_duration: float
    """ Scanline "exposure" time for the rolling shutter effect

    :type: float
    """

    rolling_shutter_type: typing.Literal["NONE", "TOP"]
    """ Type of rolling shutter effect matching CMOS-based cameras

    :type: typing.Literal['NONE','TOP']
    """

    sample_all_lights_direct: bool
    """ Sample all lights (for direct samples), rather than randomly picking one

    :type: bool
    """

    sample_all_lights_indirect: bool
    """ Sample all lights (for indirect samples), rather than randomly picking one

    :type: bool
    """

    sample_clamp_direct: float
    """ If non-zero, the maximum value for a direct sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy

    :type: float
    """

    sample_clamp_indirect: float
    """ If non-zero, the maximum value for an indirect sample, higher values will be scaled down to avoid too much noise and slow convergence at the cost of accuracy

    :type: float
    """

    samples: int
    """ Number of samples to render for each pixel

    :type: int
    """

    sampling_pattern: typing.Literal["SOBOL", "CORRELATED_MUTI_JITTER"]
    """ Random sampling pattern used by the integrator

    :type: typing.Literal['SOBOL','CORRELATED_MUTI_JITTER']
    """

    seed: int
    """ Seed value for integrator to get different noise patterns

    :type: int
    """

    shading_system: bool
    """ Use Open Shading Language (CPU rendering only)

    :type: bool
    """

    subsurface_samples: int
    """ Number of subsurface scattering samples to render for each AA sample

    :type: int
    """

    texture_limit: typing.Literal[
        "OFF", "128", "256", "512", "1024", "2048", "4096", "8192"
    ]
    """ Limit texture size used by viewport rendering

    :type: typing.Literal['OFF','128','256','512','1024','2048','4096','8192']
    """

    texture_limit_render: typing.Literal[
        "OFF", "128", "256", "512", "1024", "2048", "4096", "8192"
    ]
    """ Limit texture size used by final rendering

    :type: typing.Literal['OFF','128','256','512','1024','2048','4096','8192']
    """

    tile_order: typing.Literal[
        "CENTER",
        "RIGHT_TO_LEFT",
        "LEFT_TO_RIGHT",
        "TOP_TO_BOTTOM",
        "BOTTOM_TO_TOP",
        "HILBERT_SPIRAL",
    ]
    """ Tile order for rendering

    :type: typing.Literal['CENTER','RIGHT_TO_LEFT','LEFT_TO_RIGHT','TOP_TO_BOTTOM','BOTTOM_TO_TOP','HILBERT_SPIRAL']
    """

    transmission_bounces: int
    """ Maximum number of transmission bounces, bounded by total maximum

    :type: int
    """

    transmission_samples: int
    """ Number of transmission bounce samples to render for each AA sample

    :type: int
    """

    transparent_max_bounces: int
    """ Maximum number of transparent bounces

    :type: int
    """

    transparent_min_bounces: int
    """ Minimum number of transparent bounces, setting this lower than the maximum enables probabilistic path termination (faster but noisier)

    :type: int
    """

    use_animated_seed: bool
    """ Use different seed values (and hence noise patterns) at different frames

    :type: bool
    """

    use_camera_cull: bool
    """ Allow objects to be culled based on the camera frustum

    :type: bool
    """

    use_distance_cull: bool
    """ Allow objects to be culled based on the distance from camera

    :type: bool
    """

    use_layer_samples: typing.Literal["USE", "BOUNDED", "IGNORE"]
    """ How to use per render layer sample settings

    :type: typing.Literal['USE','BOUNDED','IGNORE']
    """

    use_progressive_refine: bool
    """ Instead of rendering each tile until it is finished, refine the whole image progressively (this renders somewhat slower, but time can be saved by manually stopping the render when the noise is low enough)

    :type: bool
    """

    use_square_samples: bool
    """ Square sampling values for easier artist control

    :type: bool
    """

    use_transparent_shadows: bool
    """ Use transparency of surfaces for rendering shadows

    :type: bool
    """

    volume_bounces: int
    """ Maximum number of volumetric scattering events

    :type: int
    """

    volume_max_steps: int
    """ Maximum number of steps through the volume before giving up, to avoid extremely long render times with big objects or small step sizes

    :type: int
    """

    volume_samples: int
    """ Number of volume scattering samples to render for each AA sample

    :type: int
    """

    volume_step_size: float
    """ Distance between volume shader samples when rendering the volume (lower values give more accurate and detailed results, but also increased render time)

    :type: float
    """

class CyclesVisibilitySettings(PropertyGroup, bpy_struct):
    camera: bool
    """ Object visibility for camera rays

    :type: bool
    """

    diffuse: bool
    """ Object visibility for diffuse reflection rays

    :type: bool
    """

    glossy: bool
    """ Object visibility for glossy reflection rays

    :type: bool
    """

    scatter: bool
    """ Object visibility for volume scatter rays

    :type: bool
    """

    shadow: bool
    """ Object visibility for shadow rays

    :type: bool
    """

    transmission: bool
    """ Object visibility for transmission rays

    :type: bool
    """

class CyclesWorldSettings(PropertyGroup, bpy_struct):
    homogeneous_volume: bool
    """ When using volume rendering, assume volume has the same density everywhere(not using any textures), for faster rendering

    :type: bool
    """

    max_bounces: int
    """ Maximum number of bounces the background light will contribute to the render

    :type: int
    """

    sample_as_light: bool
    """ Use multiple importance sampling for the environment, enabling for non-solid colors is recommended

    :type: bool
    """

    sample_map_resolution: int
    """ Importance map size is resolution x resolution; higher values potentially produce less noise, at the cost of memory and speed

    :type: int
    """

    samples: int
    """ Number of light samples to render for each AA sample

    :type: int
    """

    volume_interpolation: typing.Literal["LINEAR", "CUBIC"]
    """ Interpolation method to use for volumes

    :type: typing.Literal['LINEAR','CUBIC']
    """

    volume_sampling: typing.Literal["DISTANCE", "EQUIANGULAR", "MULTIPLE_IMPORTANCE"]
    """ Sampling method to use for volumes

    :type: typing.Literal['DISTANCE','EQUIANGULAR','MULTIPLE_IMPORTANCE']
    """

class DampedTrackConstraint(Constraint, bpy_struct):
    """Point toward target by taking the shortest rotation path"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    track_axis: typing.Literal[
        "TRACK_X",
        "TRACK_Y",
        "TRACK_Z",
        "TRACK_NEGATIVE_X",
        "TRACK_NEGATIVE_Y",
        "TRACK_NEGATIVE_Z",
    ]
    """ Axis that points to the target object

    :type: typing.Literal['TRACK_X','TRACK_Y','TRACK_Z','TRACK_NEGATIVE_X','TRACK_NEGATIVE_Y','TRACK_NEGATIVE_Z']
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class DataTransferModifier(Modifier, bpy_struct):
    """Modifier transferring some data from a source mesh"""

    data_types_edges: set[
        typing.Literal[
            "SHARP_EDGE", "SEAM", "CREASE", "BEVEL_WEIGHT_EDGE", "FREESTYLE_EDGE"
        ]
    ]
    """ Which edge data layers to transfer

    :type: set[typing.Literal['SHARP_EDGE','SEAM','CREASE','BEVEL_WEIGHT_EDGE','FREESTYLE_EDGE']]
    """

    data_types_loops: set[typing.Literal["CUSTOM_NORMAL", "VCOL", "UV"]]
    """ Which face corner data layers to transfer

    :type: set[typing.Literal['CUSTOM_NORMAL','VCOL','UV']]
    """

    data_types_polys: set[typing.Literal["SMOOTH", "FREESTYLE_FACE"]]
    """ Which poly data layers to transfer

    :type: set[typing.Literal['SMOOTH','FREESTYLE_FACE']]
    """

    data_types_verts: set[typing.Literal["VGROUP_WEIGHTS", "BEVEL_WEIGHT_VERT"]]
    """ Which vertex data layers to transfer

    :type: set[typing.Literal['VGROUP_WEIGHTS','BEVEL_WEIGHT_VERT']]
    """

    edge_mapping: typing.Literal[
        "TOPOLOGY", "VERT_NEAREST", "NEAREST", "POLY_NEAREST", "EDGEINTERP_VNORPROJ"
    ]
    """ Method used to map source edges to destination ones

    :type: typing.Literal['TOPOLOGY','VERT_NEAREST','NEAREST','POLY_NEAREST','EDGEINTERP_VNORPROJ']
    """

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    islands_precision: float
    """ Factor controlling precision of islands handling (typically, 0.1 should be enough, higher values can make things really slow)

    :type: float
    """

    layers_uv_select_dst: typing.Literal["ACTIVE", "NAME", "INDEX"]
    """ How to match source and destination layers

    :type: typing.Literal['ACTIVE','NAME','INDEX']
    """

    layers_uv_select_src: typing.Literal["ACTIVE", "ALL", "BONE_SELECT", "BONE_DEFORM"]
    """ Which layers to transfer, in case of multi-layers types

    :type: typing.Literal['ACTIVE','ALL','BONE_SELECT','BONE_DEFORM']
    """

    layers_vcol_select_dst: typing.Literal["ACTIVE", "NAME", "INDEX"]
    """ How to match source and destination layers

    :type: typing.Literal['ACTIVE','NAME','INDEX']
    """

    layers_vcol_select_src: typing.Literal[
        "ACTIVE", "ALL", "BONE_SELECT", "BONE_DEFORM"
    ]
    """ Which layers to transfer, in case of multi-layers types

    :type: typing.Literal['ACTIVE','ALL','BONE_SELECT','BONE_DEFORM']
    """

    layers_vgroup_select_dst: typing.Literal["ACTIVE", "NAME", "INDEX"]
    """ How to match source and destination layers

    :type: typing.Literal['ACTIVE','NAME','INDEX']
    """

    layers_vgroup_select_src: typing.Literal[
        "ACTIVE", "ALL", "BONE_SELECT", "BONE_DEFORM"
    ]
    """ Which layers to transfer, in case of multi-layers types

    :type: typing.Literal['ACTIVE','ALL','BONE_SELECT','BONE_DEFORM']
    """

    loop_mapping: typing.Literal[
        "TOPOLOGY",
        "NEAREST_NORMAL",
        "NEAREST_POLYNOR",
        "NEAREST_POLY",
        "POLYINTERP_NEAREST",
        "POLYINTERP_LNORPROJ",
    ]
    """ Method used to map source faces' corners to destination ones

    :type: typing.Literal['TOPOLOGY','NEAREST_NORMAL','NEAREST_POLYNOR','NEAREST_POLY','POLYINTERP_NEAREST','POLYINTERP_LNORPROJ']
    """

    max_distance: float
    """ Maximum allowed distance between source and destination element, for non-topology mappings

    :type: float
    """

    mix_factor: float
    """ Factor to use when applying data to destination (exact behavior depends on mix mode)

    :type: float
    """

    mix_mode: typing.Literal[
        "REPLACE", "ABOVE_THRESHOLD", "BELOW_THRESHOLD", "MIX", "ADD", "SUB", "MUL"
    ]
    """ How to affect destination elements with source values

    :type: typing.Literal['REPLACE','ABOVE_THRESHOLD','BELOW_THRESHOLD','MIX','ADD','SUB','MUL']
    """

    object: Object | None
    """ Object to transfer data from

    :type: Object | None
    """

    poly_mapping: typing.Literal["TOPOLOGY", "NEAREST", "NORMAL", "POLYINTERP_PNORPROJ"]
    """ Method used to map source faces to destination ones

    :type: typing.Literal['TOPOLOGY','NEAREST','NORMAL','POLYINTERP_PNORPROJ']
    """

    ray_radius: float
    """ 'Width' of rays (especially useful when raycasting against vertices or edges)

    :type: float
    """

    use_edge_data: bool
    """ Enable edge data transfer

    :type: bool
    """

    use_loop_data: bool
    """ Enable face corner data transfer

    :type: bool
    """

    use_max_distance: bool
    """ Source elements must be closer than given distance from destination one

    :type: bool
    """

    use_object_transform: bool
    """ Evaluate source and destination meshes in global space

    :type: bool
    """

    use_poly_data: bool
    """ Enable face data transfer

    :type: bool
    """

    use_vert_data: bool
    """ Enable vertex data transfer

    :type: bool
    """

    vert_mapping: typing.Literal[
        "TOPOLOGY",
        "NEAREST",
        "EDGE_NEAREST",
        "EDGEINTERP_NEAREST",
        "POLY_NEAREST",
        "POLYINTERP_NEAREST",
        "POLYINTERP_VNORPROJ",
    ]
    """ Method used to map source vertices to destination ones

    :type: typing.Literal['TOPOLOGY','NEAREST','EDGE_NEAREST','EDGEINTERP_NEAREST','POLY_NEAREST','POLYINTERP_NEAREST','POLYINTERP_VNORPROJ']
    """

    vertex_group: str
    """ Vertex group name for selecting the affected areas

    :type: str
    """

class DecimateModifier(Modifier, bpy_struct):
    """Decimation modifier"""

    angle_limit: float
    """ Only dissolve angles below this (planar only)

    :type: float
    """

    decimate_type: typing.Literal["COLLAPSE", "UNSUBDIV", "DISSOLVE"]
    """ 

    :type: typing.Literal['COLLAPSE','UNSUBDIV','DISSOLVE']
    """

    delimit: set[typing.Literal["NORMAL", "MATERIAL", "SEAM", "SHARP", "UV"]]
    """ Limit merging geometry

    :type: set[typing.Literal['NORMAL','MATERIAL','SEAM','SHARP','UV']]
    """

    face_count: int
    """ The current number of faces in the decimated mesh

    :type: int
    """

    invert_vertex_group: bool
    """ Invert vertex group influence (collapse only)

    :type: bool
    """

    iterations: int
    """ Number of times reduce the geometry (unsubdivide only)

    :type: int
    """

    ratio: float
    """ Ratio of triangles to reduce to (collapse only)

    :type: float
    """

    symmetry_axis: typing.Literal["X", "Y", "Z"]
    """ Axis of symmetry

    :type: typing.Literal['X','Y','Z']
    """

    use_collapse_triangulate: bool
    """ Keep triangulated faces resulting from decimation (collapse only)

    :type: bool
    """

    use_dissolve_boundaries: bool
    """ Dissolve all vertices inbetween face boundaries (planar only)

    :type: bool
    """

    use_symmetry: bool
    """ Maintain symmetry on an axis

    :type: bool
    """

    vertex_group: str
    """ Vertex group name (collapse only)

    :type: str
    """

    vertex_group_factor: float
    """ Vertex group strength

    :type: float
    """

class DelaySensor(Sensor, bpy_struct):
    """Sensor to send delayed events"""

    delay: int
    """ Delay in number of logic tics before the positive trigger (default 60 per second)

    :type: int
    """

    duration: int
    """ If >0, delay in number of logic tics before the negative trigger following the positive trigger

    :type: int
    """

    use_repeat: bool
    """ Toggle repeat option (if selected, the sensor restarts after Delay+Duration logic tics)

    :type: bool
    """

class Depsgraph(bpy_struct):
    def debug_graphviz(self, filename: str):
        """debug_graphviz

        :param filename: File Name, File in which to store graphviz debug output
        :type filename: str
        """

    def debug_rebuild(self):
        """debug_rebuild"""

    def debug_stats(self):
        """Report the number of elements in the Dependency Graph"""

class DisplaceModifier(Modifier, bpy_struct):
    """Displacement modifier"""

    direction: typing.Literal["X", "Y", "Z", "NORMAL", "CUSTOM_NORMAL", "RGB_TO_XYZ"]
    """ 

    :type: typing.Literal['X','Y','Z','NORMAL','CUSTOM_NORMAL','RGB_TO_XYZ']
    """

    mid_level: float
    """ Material value that gives no displacement

    :type: float
    """

    strength: float
    """ Amount to displace geometry

    :type: float
    """

    texture: Texture | None
    """ 

    :type: Texture | None
    """

    texture_coords: typing.Literal["LOCAL", "GLOBAL", "OBJECT", "UV"]
    """ 

    :type: typing.Literal['LOCAL','GLOBAL','OBJECT','UV']
    """

    texture_coords_object: Object | None
    """ Object to set the texture coordinates

    :type: Object | None
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class DisplaySafeAreas(bpy_struct):
    """Safe Areas used in 3D view and the VSE"""

    action: bpy_prop_array[float]
    """ Safe area for general elements

    :type: bpy_prop_array[float]
    """

    action_center: bpy_prop_array[float]
    """ Safe area for general elements in a different aspect ratio

    :type: bpy_prop_array[float]
    """

    title: bpy_prop_array[float]
    """ Safe area for text and graphics

    :type: bpy_prop_array[float]
    """

    title_center: bpy_prop_array[float]
    """ Safe area for text and graphics in a different aspect ratio

    :type: bpy_prop_array[float]
    """

class DistortedNoiseTexture(Texture, ID, bpy_struct):
    """Procedural distorted noise texture"""

    distortion: float
    """ Amount of distortion

    :type: float
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_distortion: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis for the distortion

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class DomainFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for the domain of a fluid simulation"""

    compressibility: float
    """ Allowed compressibility due to gravitational force for standing fluid (directly affects simulation step size)

    :type: float
    """

    end_time: float
    """ Simulation time of the last blender frame (in seconds)

    :type: float
    """

    filepath: str
    """ Directory (and/or filename prefix) to store baked fluid simulation files in

    :type: str
    """

    fluid_mesh_vertices: bpy_prop_collection[FluidVertexVelocity]
    """ Vertices of the fluid mesh generated by simulation

    :type: bpy_prop_collection[FluidVertexVelocity]
    """

    frame_offset: int
    """ Offset when reading baked cache

    :type: int
    """

    generate_particles: float
    """ Amount of particles to generate (0=off, 1=normal, >1=more)

    :type: float
    """

    gravity: bpy_prop_array[float]
    """ Gravity in X, Y and Z direction

    :type: bpy_prop_array[float]
    """

    grid_levels: int
    """ Number of coarsened grids to use (-1 for automatic)

    :type: int
    """

    memory_estimate: str
    """ Estimated amount of memory needed for baking the domain

    :type: str
    """

    partial_slip_factor: float
    """ Amount of mixing between no- and free-slip, 0 is no slip and 1 is free slip

    :type: float
    """

    preview_resolution: int
    """ Preview resolution in X,Y and Z direction

    :type: int
    """

    render_display_mode: typing.Literal["GEOMETRY", "PREVIEW", "FINAL"]
    """ How to display the mesh for rendering

    :type: typing.Literal['GEOMETRY','PREVIEW','FINAL']
    """

    resolution: int
    """ Domain resolution in X,Y and Z direction

    :type: int
    """

    simulation_rate: float
    """ Fluid motion rate (0 = stationary, 1 = normal speed)

    :type: float
    """

    simulation_scale: float
    """ Size of the simulation domain in meters

    :type: float
    """

    slip_type: typing.Literal["NOSLIP", "PARTIALSLIP", "FREESLIP"]
    """ 

    :type: typing.Literal['NOSLIP','PARTIALSLIP','FREESLIP']
    """

    start_time: float
    """ Simulation time of the first blender frame (in seconds)

    :type: float
    """

    surface_smooth: float
    """ Amount of surface smoothing (a value of 0 is off, 1 is normal smoothing and more than 1 is extra smoothing)

    :type: float
    """

    surface_subdivisions: int
    """ Number of isosurface subdivisions (this is necessary for the inclusion of particles into the surface generation - WARNING: can lead to longer computation times !)

    :type: int
    """

    threads: int
    """ Override number of threads for the simulation, 0 is automatic

    :type: int
    """

    tracer_particles: int
    """ Number of tracer particles to generate

    :type: int
    """

    use_reverse_frames: bool
    """ Reverse fluid frames

    :type: bool
    """

    use_speed_vectors: bool
    """ Generate speed vectors for vector blur

    :type: bool
    """

    use_surface_noobs: bool
    """ Removes the air gap between fluid surface and obstacles - WARNING: Can result in a dissolving surface in other areas

    :type: bool
    """

    use_time_override: bool
    """ Use a custom start and end time (in seconds) instead of the scene's timeline

    :type: bool
    """

    viewport_display_mode: typing.Literal["GEOMETRY", "PREVIEW", "FINAL"]
    """ How to display the mesh in the viewport

    :type: typing.Literal['GEOMETRY','PREVIEW','FINAL']
    """

    viscosity_base: float
    """ Viscosity setting: value that is multiplied by 10 to the power of (exponent*-1)

    :type: float
    """

    viscosity_exponent: int
    """ Negative exponent for the viscosity value (to simplify entering small values e.g. 5*10^-6)

    :type: int
    """

class DopeSheet(bpy_struct):
    """Settings for filtering the channels shown in animation editors"""

    filter_fcurve_name: str
    """ F-Curve live filtering string

    :type: str
    """

    filter_group: Group | None
    """ Group that included object should be a member of

    :type: Group | None
    """

    filter_text: str
    """ Live filtering string

    :type: str
    """

    show_armatures: bool
    """ Include visualization of armature related animation data

    :type: bool
    """

    show_cameras: bool
    """ Include visualization of camera related animation data

    :type: bool
    """

    show_curves: bool
    """ Include visualization of curve related animation data

    :type: bool
    """

    show_datablock_filters: bool
    """ Show options for whether channels related to certain types of data are included

    :type: bool
    """

    show_expanded_summary: bool
    """ Collapse summary when shown, so all other channels get hidden (Dope Sheet editors only)

    :type: bool
    """

    show_gpencil: bool
    """ Include visualization of Grease Pencil related animation data and frames

    :type: bool
    """

    show_gpencil_3d_only: bool
    """ Only show Grease Pencil datablocks used as part of the active scene

    :type: bool
    """

    show_hidden: bool
    """ Include channels from objects/bone that are not visible

    :type: bool
    """

    show_lamps: bool
    """ Include visualization of lamp related animation data

    :type: bool
    """

    show_lattices: bool
    """ Include visualization of lattice related animation data

    :type: bool
    """

    show_linestyles: bool
    """ Include visualization of Line Style related Animation data

    :type: bool
    """

    show_materials: bool
    """ Include visualization of material related animation data

    :type: bool
    """

    show_meshes: bool
    """ Include visualization of mesh related animation data

    :type: bool
    """

    show_metaballs: bool
    """ Include visualization of metaball related animation data

    :type: bool
    """

    show_missing_nla: bool
    """ Include animation data blocks with no NLA data (NLA editor only)

    :type: bool
    """

    show_modifiers: bool
    """ Include visualization of animation data related to data-blocks linked to modifiers

    :type: bool
    """

    show_nodes: bool
    """ Include visualization of node related animation data

    :type: bool
    """

    show_only_errors: bool
    """ Only include F-Curves and drivers that are disabled or have errors

    :type: bool
    """

    show_only_group_objects: bool
    """ Only include channels from objects in the specified group

    :type: bool
    """

    show_only_matching_fcurves: bool
    """ Only include F-Curves with names containing search text

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    show_particles: bool
    """ Include visualization of particle related animation data

    :type: bool
    """

    show_scenes: bool
    """ Include visualization of scene related animation data

    :type: bool
    """

    show_shapekeys: bool
    """ Include visualization of shape key related animation data

    :type: bool
    """

    show_speakers: bool
    """ Include visualization of speaker related animation data

    :type: bool
    """

    show_summary: bool
    """ Display an additional 'summary' line (Dope Sheet editors only)

    :type: bool
    """

    show_textures: bool
    """ Include visualization of texture related animation data

    :type: bool
    """

    show_transforms: bool
    """ Include visualization of object-level animation data (mostly transforms)

    :type: bool
    """

    show_worlds: bool
    """ Include visualization of world related animation data

    :type: bool
    """

    source: ID
    """ ID-Block representing source data, usually ID_SCE (i.e. Scene)

    :type: ID
    """

    use_datablock_sort: bool
    """ Alphabetically sorts datablocks - mainly objects in the scene (disable to increase viewport speed)

    :type: bool
    """

    use_filter_text: bool
    """ Only include channels with names containing search text

    :type: bool
    """

    use_multi_word_filter: bool
    """ Perform fuzzy/multi-word matching (WARNING: May be slow)

    :type: bool
    """

class Driver(bpy_struct):
    """Driver for the value of a setting based on an external value"""

    expression: str
    """ Expression to use for Scripted Expression

    :type: str
    """

    is_valid: bool
    """ Driver could not be evaluated in past, so should be skipped

    :type: bool
    """

    show_debug_info: bool
    """ Show intermediate values for the driver calculations to allow debugging of drivers

    :type: bool
    """

    type: typing.Literal["AVERAGE", "SUM", "SCRIPTED", "MIN", "MAX"]
    """ Driver type

    :type: typing.Literal['AVERAGE','SUM','SCRIPTED','MIN','MAX']
    """

    use_self: bool
    """ Include a 'self' variable in the name-space, so drivers can easily reference the data being modified (object, bone, etc...)

    :type: bool
    """

    variables: ChannelDriverVariables
    """ Properties acting as inputs for this driver

    :type: ChannelDriverVariables
    """

class DriverTarget(bpy_struct):
    """Source of input values for driver variables"""

    bone_target: str
    """ Name of PoseBone to use as target

    :type: str
    """

    data_path: str
    """ RNA Path (from ID-block) to property used

    :type: str
    """

    id: ID | None
    """ ID-block that the specific property used can be found from (id_type property must be set first)

    :type: ID | None
    """

    id_type: typing.Literal[
        "ACTION",
        "ARMATURE",
        "BRUSH",
        "CAMERA",
        "CACHEFILE",
        "CURVE",
        "FONT",
        "GREASEPENCIL",
        "GROUP",
        "IMAGE",
        "KEY",
        "LAMP",
        "LIBRARY",
        "LINESTYLE",
        "LATTICE",
        "MASK",
        "MATERIAL",
        "META",
        "MESH",
        "MOVIECLIP",
        "NODETREE",
        "OBJECT",
        "PAINTCURVE",
        "PALETTE",
        "PARTICLE",
        "SCENE",
        "SCREEN",
        "SOUND",
        "SPEAKER",
        "TEXT",
        "TEXTURE",
        "WINDOWMANAGER",
        "WORLD",
    ]
    """ Type of ID-block that can be used

    :type: typing.Literal['ACTION','ARMATURE','BRUSH','CAMERA','CACHEFILE','CURVE','FONT','GREASEPENCIL','GROUP','IMAGE','KEY','LAMP','LIBRARY','LINESTYLE','LATTICE','MASK','MATERIAL','META','MESH','MOVIECLIP','NODETREE','OBJECT','PAINTCURVE','PALETTE','PARTICLE','SCENE','SCREEN','SOUND','SPEAKER','TEXT','TEXTURE','WINDOWMANAGER','WORLD']
    """

    transform_space: typing.Literal["WORLD_SPACE", "TRANSFORM_SPACE", "LOCAL_SPACE"]
    """ Space in which transforms are used

    :type: typing.Literal['WORLD_SPACE','TRANSFORM_SPACE','LOCAL_SPACE']
    """

    transform_type: typing.Literal[
        "LOC_X",
        "LOC_Y",
        "LOC_Z",
        "ROT_X",
        "ROT_Y",
        "ROT_Z",
        "SCALE_X",
        "SCALE_Y",
        "SCALE_Z",
    ]
    """ Driver variable type

    :type: typing.Literal['LOC_X','LOC_Y','LOC_Z','ROT_X','ROT_Y','ROT_Z','SCALE_X','SCALE_Y','SCALE_Z']
    """

class DriverVariable(bpy_struct):
    """Variable from some source/target for driver relationship"""

    is_name_valid: bool
    """ Is this a valid name for a driver variable

    :type: bool
    """

    name: str
    """ Name to use in scripted expressions/functions (no spaces or dots are allowed, and must start with a letter)

    :type: str
    """

    targets: bpy_prop_collection[DriverTarget]
    """ Sources of input data for evaluating this variable

    :type: bpy_prop_collection[DriverTarget]
    """

    type: typing.Literal["SINGLE_PROP", "TRANSFORMS", "ROTATION_DIFF", "LOC_DIFF"]
    """ Driver variable type

    :type: typing.Literal['SINGLE_PROP','TRANSFORMS','ROTATION_DIFF','LOC_DIFF']
    """

class DupliObject(bpy_struct):
    """An object duplicate"""

    hide: bool
    """ Don't show dupli object in viewport or render

    :type: bool
    """

    index: int
    """ Index in the lowest-level dupli list

    :type: int
    """

    matrix: bpy_prop_array[float]
    """ Object duplicate transformation matrix

    :type: bpy_prop_array[float]
    """

    object: Object
    """ Object being duplicated

    :type: Object
    """

    orco: bpy_prop_array[float]
    """ Generated coordinates in parent object space

    :type: bpy_prop_array[float]
    """

    particle_system: ParticleSystem
    """ Particle system that this dupli object was instanced from

    :type: ParticleSystem
    """

    persistent_id: bpy_prop_array[int]
    """ Persistent identifier for inter-frame matching of objects with motion blur

    :type: bpy_prop_array[int]
    """

    type: typing.Literal["NONE", "FRAMES", "VERTS", "FACES", "GROUP"]
    """ Duplicator type that generated this dupli object

    :type: typing.Literal['NONE','FRAMES','VERTS','FACES','GROUP']
    """

    uv: bpy_prop_array[float]
    """ UV coordinates in parent object space

    :type: bpy_prop_array[float]
    """

class DynamicPaintBrushSettings(bpy_struct):
    """Brush settings"""

    invert_proximity: bool
    """ Proximity falloff is applied inside the volume

    :type: bool
    """

    material: Material | None
    """ Material to use (if not defined, material linked to the mesh is used)

    :type: Material | None
    """

    paint_alpha: float
    """ Paint alpha

    :type: float
    """

    paint_color: bpy_prop_array[float]
    """ Color of the paint

    :type: bpy_prop_array[float]
    """

    paint_distance: float
    """ Maximum distance from brush to mesh surface to affect paint

    :type: float
    """

    paint_ramp: ColorRamp
    """ Color ramp used to define proximity falloff

    :type: ColorRamp
    """

    paint_source: typing.Literal[
        "PARTICLE_SYSTEM", "POINT", "DISTANCE", "VOLUME_DISTANCE", "VOLUME"
    ]
    """ 

    :type: typing.Literal['PARTICLE_SYSTEM','POINT','DISTANCE','VOLUME_DISTANCE','VOLUME']
    """

    paint_wetness: float
    """ Paint wetness, visible in wetmap (some effects only affect wet paint)

    :type: float
    """

    particle_system: ParticleSystem | None
    """ The particle system to paint with

    :type: ParticleSystem | None
    """

    proximity_falloff: typing.Literal["SMOOTH", "CONSTANT", "RAMP"]
    """ Proximity falloff type

    :type: typing.Literal['SMOOTH','CONSTANT','RAMP']
    """

    ray_direction: typing.Literal["CANVAS", "BRUSH", "Z_AXIS"]
    """ Ray direction to use for projection (if brush object is located in that direction it's painted)

    :type: typing.Literal['CANVAS','BRUSH','Z_AXIS']
    """

    smooth_radius: float
    """ Smooth falloff added after solid radius

    :type: float
    """

    smudge_strength: float
    """ Smudge effect strength

    :type: float
    """

    solid_radius: float
    """ Radius that will be painted solid

    :type: float
    """

    use_absolute_alpha: bool
    """ Only increase alpha value if paint alpha is higher than existing

    :type: bool
    """

    use_material: bool
    """ Use object material to define color and influence

    :type: bool
    """

    use_negative_volume: bool
    """ Negate influence inside the volume

    :type: bool
    """

    use_paint_erase: bool
    """ Erase / remove paint instead of adding it

    :type: bool
    """

    use_particle_radius: bool
    """ Use radius from particle settings

    :type: bool
    """

    use_proximity_project: bool
    """ Brush is projected to canvas from defined direction within brush proximity

    :type: bool
    """

    use_proximity_ramp_alpha: bool
    """ Only read color ramp alpha

    :type: bool
    """

    use_smudge: bool
    """ Make this brush to smudge existing paint as it moves

    :type: bool
    """

    use_velocity_alpha: bool
    """ Multiply brush influence by velocity color ramp alpha

    :type: bool
    """

    use_velocity_color: bool
    """ Replace brush color by velocity color ramp

    :type: bool
    """

    use_velocity_depth: bool
    """ Multiply brush intersection depth (displace, waves) by velocity ramp alpha

    :type: bool
    """

    velocity_max: float
    """ Velocity considered as maximum influence (Blender units per frame)

    :type: float
    """

    velocity_ramp: ColorRamp
    """ Color ramp used to define brush velocity effect

    :type: ColorRamp
    """

    wave_clamp: float
    """ Maximum level of surface intersection used to influence waves (use 0.0 to disable)

    :type: float
    """

    wave_factor: float
    """ Multiplier for wave influence of this brush

    :type: float
    """

    wave_type: typing.Literal["CHANGE", "DEPTH", "FORCE", "REFLECT"]
    """ 

    :type: typing.Literal['CHANGE','DEPTH','FORCE','REFLECT']
    """

class DynamicPaintCanvasSettings(bpy_struct):
    """Dynamic Paint canvas settings"""

    canvas_surfaces: DynamicPaintSurfaces
    """ Paint surface list

    :type: DynamicPaintSurfaces
    """

class DynamicPaintModifier(Modifier, bpy_struct):
    """Dynamic Paint modifier"""

    brush_settings: DynamicPaintBrushSettings
    """ 

    :type: DynamicPaintBrushSettings
    """

    canvas_settings: DynamicPaintCanvasSettings
    """ 

    :type: DynamicPaintCanvasSettings
    """

    ui_type: typing.Literal["CANVAS", "BRUSH"]
    """ 

    :type: typing.Literal['CANVAS','BRUSH']
    """

class DynamicPaintSurface(bpy_struct):
    """A canvas surface layer"""

    brush_group: Group | None
    """ Only use brush objects from this group

    :type: Group | None
    """

    brush_influence_scale: float
    """ Adjust influence brush objects have on this surface

    :type: float
    """

    brush_radius_scale: float
    """ Adjust radius of proximity brushes or particles for this surface

    :type: float
    """

    color_dry_threshold: float
    """ The wetness level when colors start to shift to the background

    :type: float
    """

    color_spread_speed: float
    """ How fast colors get mixed within wet paint

    :type: float
    """

    depth_clamp: float
    """ Maximum level of depth intersection in object space (use 0.0 to disable)

    :type: float
    """

    displace_factor: float
    """ Strength of displace when applied to the mesh

    :type: float
    """

    displace_type: typing.Literal["DISPLACE", "DEPTH"]
    """ 

    :type: typing.Literal['DISPLACE','DEPTH']
    """

    dissolve_speed: int
    """ Approximately in how many frames should dissolve happen

    :type: int
    """

    drip_acceleration: float
    """ How much surface acceleration affects dripping

    :type: float
    """

    drip_velocity: float
    """ How much surface velocity affects dripping

    :type: float
    """

    dry_speed: int
    """ Approximately in how many frames should drying happen

    :type: int
    """

    effect_ui: typing.Literal["SPREAD", "DRIP", "SHRINK"]
    """ 

    :type: typing.Literal['SPREAD','DRIP','SHRINK']
    """

    effector_weights: EffectorWeights
    """ 

    :type: EffectorWeights
    """

    frame_end: int
    """ Simulation end frame

    :type: int
    """

    frame_start: int
    """ Simulation start frame

    :type: int
    """

    frame_substeps: int
    """ Do extra frames between scene frames to ensure smooth motion

    :type: int
    """

    image_fileformat: typing.Literal["PNG", "OPENEXR"]
    """ 

    :type: typing.Literal['PNG','OPENEXR']
    """

    image_output_path: str
    """ Directory to save the textures

    :type: str
    """

    image_resolution: int
    """ Output image resolution

    :type: int
    """

    init_color: bpy_prop_array[float]
    """ Initial color of the surface

    :type: bpy_prop_array[float]
    """

    init_color_type: typing.Literal["NONE", "COLOR", "TEXTURE", "VERTEX_COLOR"]
    """ 

    :type: typing.Literal['NONE','COLOR','TEXTURE','VERTEX_COLOR']
    """

    init_layername: str
    """ 

    :type: str
    """

    init_texture: Texture | None
    """ 

    :type: Texture | None
    """

    is_active: bool
    """ Toggle whether surface is processed or ignored

    :type: bool
    """

    is_cache_user: bool
    """ 

    :type: bool
    """

    name: str
    """ Surface name

    :type: str
    """

    output_name_a: str
    """ Name used to save output from this surface

    :type: str
    """

    output_name_b: str
    """ Name used to save output from this surface

    :type: str
    """

    point_cache: PointCache
    """ 

    :type: PointCache
    """

    preview_id: typing.Literal["PAINT", "WETMAP"]
    """ 

    :type: typing.Literal['PAINT','WETMAP']
    """

    show_preview: bool
    """ Display surface preview in 3D-views

    :type: bool
    """

    shrink_speed: float
    """ How fast shrink effect moves on the canvas surface

    :type: float
    """

    spread_speed: float
    """ How fast spread effect moves on the canvas surface

    :type: float
    """

    surface_format: typing.Literal["VERTEX", "IMAGE"]
    """ Surface Format

    :type: typing.Literal['VERTEX','IMAGE']
    """

    surface_type: typing.Literal["PAINT"]
    """ Surface Type

    :type: typing.Literal['PAINT']
    """

    use_antialiasing: bool
    """ Use 5x multisampling to smooth paint edges

    :type: bool
    """

    use_color_preview: bool
    """ Whether this surface has some color preview for 3D view

    :type: bool
    """

    use_dissolve: bool
    """ Enable to make surface changes disappear over time

    :type: bool
    """

    use_dissolve_log: bool
    """ Use logarithmic dissolve (makes high values to fade faster than low values)

    :type: bool
    """

    use_drip: bool
    """ Process drip effect (drip wet paint to gravity direction)

    :type: bool
    """

    use_dry_log: bool
    """ Use logarithmic drying (makes high values to dry faster than low values)

    :type: bool
    """

    use_drying: bool
    """ Enable to make surface wetness dry over time

    :type: bool
    """

    use_incremental_displace: bool
    """ New displace is added cumulatively on top of existing

    :type: bool
    """

    use_output_a: bool
    """ Save this output layer

    :type: bool
    """

    use_output_b: bool
    """ Save this output layer

    :type: bool
    """

    use_premultiply: bool
    """ Multiply color by alpha (recommended for Blender input)

    :type: bool
    """

    use_shrink: bool
    """ Process shrink effect (shrink paint areas)

    :type: bool
    """

    use_spread: bool
    """ Process spread effect (spread wet paint around surface)

    :type: bool
    """

    use_wave_open_border: bool
    """ Pass waves through mesh edges

    :type: bool
    """

    uv_layer: str
    """ UV map name

    :type: str
    """

    wave_damping: float
    """ Wave damping factor

    :type: float
    """

    wave_smoothness: float
    """ Limit maximum steepness of wave slope between simulation points (use higher values for smoother waves at expense of reduced detail)

    :type: float
    """

    wave_speed: float
    """ Wave propagation speed

    :type: float
    """

    wave_spring: float
    """ Spring force that pulls water level back to zero

    :type: float
    """

    wave_timescale: float
    """ Wave time scaling factor

    :type: float
    """

    def output_exists(self, object: Object, index: int | None) -> bool:
        """Checks if surface output layer of given name exists

        :param object:
        :type object: Object
        :param index: Index
        :type index: int | None
        :return:
        :rtype: bool
        """

class EdgeSplitModifier(Modifier, bpy_struct):
    """Edge splitting modifier to create sharp edges"""

    split_angle: float
    """ Angle above which to split edges

    :type: float
    """

    use_edge_angle: bool
    """ Split edges with high angle between faces

    :type: bool
    """

    use_edge_sharp: bool
    """ Split edges that are marked as sharp

    :type: bool
    """

class EditBone(bpy_struct):
    """Editmode bone in an Armature data-block"""

    bbone_curveinx: float
    """ X-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveiny: float
    """ Y-axis handle offset for start of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveoutx: float
    """ X-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_curveouty: float
    """ Y-axis handle offset for end of the B-Bone's curve, adjusts curvature

    :type: float
    """

    bbone_in: float
    """ Length of first Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_out: float
    """ Length of second Bezier Handle (for B-Bones only)

    :type: float
    """

    bbone_rollin: float
    """ Roll offset for the start of the B-Bone, adjusts twist

    :type: float
    """

    bbone_rollout: float
    """ Roll offset for the end of the B-Bone, adjusts twist

    :type: float
    """

    bbone_scalein: float
    """ Scale factor for start of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_scaleout: float
    """ Scale factor for end of the B-Bone, adjusts thickness (for tapering effects)

    :type: float
    """

    bbone_segments: int
    """ Number of subdivisions of bone (for B-Bones only)

    :type: int
    """

    bbone_x: float
    """ B-Bone X size

    :type: float
    """

    bbone_z: float
    """ B-Bone Z size

    :type: float
    """

    envelope_distance: float
    """ Bone deformation distance (for Envelope deform only)

    :type: float
    """

    envelope_weight: float
    """ Bone deformation weight (for Envelope deform only)

    :type: float
    """

    head: bpy_prop_array[float]
    """ Location of head end of the bone

    :type: bpy_prop_array[float]
    """

    head_radius: float
    """ Radius of head of bone (for Envelope deform only)

    :type: float
    """

    hide: bool
    """ Bone is not visible when in Edit Mode

    :type: bool
    """

    hide_select: bool
    """ Bone is able to be selected

    :type: bool
    """

    layers: bpy_prop_array[bool]
    """ Layers bone exists in

    :type: bpy_prop_array[bool]
    """

    lock: bool
    """ Bone is not able to be transformed when in Edit Mode

    :type: bool
    """

    matrix: bpy_prop_array[float]
    """ Matrix combining loc/rot of the bone (head position, direction and roll), in armature space (WARNING: does not include/support bone's length/size)

    :type: bpy_prop_array[float]
    """

    name: str
    """ 

    :type: str
    """

    parent: typing_extensions.Self | None
    """ Parent edit bone (in same Armature)

    :type: typing_extensions.Self | None
    """

    roll: float
    """ Bone rotation around head-tail axis

    :type: float
    """

    select: bool
    """ 

    :type: bool
    """

    select_head: bool
    """ 

    :type: bool
    """

    select_tail: bool
    """ 

    :type: bool
    """

    show_wire: bool
    """ Bone is always drawn as Wireframe regardless of viewport draw mode (useful for non-obstructive custom bone shapes)

    :type: bool
    """

    tail: bpy_prop_array[float]
    """ Location of tail end of the bone

    :type: bpy_prop_array[float]
    """

    tail_radius: float
    """ Radius of tail of bone (for Envelope deform only)

    :type: float
    """

    use_connect: bool
    """ When bone has a parent, bone's head is stuck to the parent's tail

    :type: bool
    """

    use_cyclic_offset: bool
    """ When bone doesn't have a parent, it receives cyclic offset effects (Deprecated)

    :type: bool
    """

    use_deform: bool
    """ Enable Bone to deform geometry

    :type: bool
    """

    use_endroll_as_inroll: bool
    """ Use Roll Out of parent bone as Roll In of its children

    :type: bool
    """

    use_envelope_multiply: bool
    """ When deforming bone, multiply effects of Vertex Group weights with Envelope influence

    :type: bool
    """

    use_inherit_rotation: bool
    """ Bone inherits rotation or scale from parent bone

    :type: bool
    """

    use_inherit_scale: bool
    """ Bone inherits scaling from parent bone

    :type: bool
    """

    use_local_location: bool
    """ Bone location is set in local space

    :type: bool
    """

    use_relative_parent: bool
    """ Object children will use relative transform, like deform

    :type: bool
    """

    basename: typing.Any
    """ The name of this bone before any '.' character
(readonly)"""

    center: typing.Any
    """ The midpoint between the head and the tail.
(readonly)"""

    children: typing.Any
    """ A list of all the bones children.
(readonly)"""

    children_recursive: typing.Any
    """ A list of all children from this bone.
(readonly)"""

    children_recursive_basename: typing.Any
    """ Returns a chain of children with the same base name as this bone.
Only direct chains are supported, forks caused by multiple children
with matching base names will terminate the function
and not be returned.
(readonly)"""

    length: typing.Any
    """ The distance from head to tail,
when set the head is moved to fit the length."""

    parent_recursive: typing.Any
    """ A list of parents, starting with the immediate parent
(readonly)"""

    vector: typing.Any
    """ The direction this bone is pointing.
Utility function for (tail - head)
(readonly)"""

    x_axis: typing.Any
    """ Vector pointing down the x-axis of the bone.
(readonly)"""

    y_axis: typing.Any
    """ Vector pointing down the y-axis of the bone.
(readonly)"""

    z_axis: typing.Any
    """ Vector pointing down the z-axis of the bone.
(readonly)"""

    def align_roll(self, vector: collections.abc.Iterable[float] | None):
        """Align the bone to a localspace roll so the Z axis points in the direction of the vector given

        :param vector: Vector
        :type vector: collections.abc.Iterable[float] | None
        """

    def align_orientation(self, other):
        """Align this bone to another by moving its tail and settings its roll
        the length of the other bone is not used.

                :param other:
        """

    def parent_index(self, parent_test):
        """The same as 'bone in other_bone.parent_recursive'
        but saved generating a list.

                :param parent_test:
        """

    def transform(
        self,
        matrix: collections.abc.Sequence[collections.abc.Sequence[float]]
        | mathutils.Matrix
        | None,
        scale: bool | None = True,
        roll: bool | None = True,
    ):
        """Transform the the bones head, tail, roll and envelope
        (when the matrix has a scale component).

                :param matrix: 3x3 or 4x4 transformation matrix.
                :type matrix: collections.abc.Sequence[collections.abc.Sequence[float]] | mathutils.Matrix | None
                :param scale: Scale the bone envelope by the matrix.
                :type scale: bool | None
                :param roll: Correct the roll to point in the same relative
        direction to the head and tail.
                :type roll: bool | None
        """

    def translate(self, vec):
        """Utility function to add vec to the head and tail of this bone

        :param vec:
        """

class EditObjectActuator(Actuator, bpy_struct):
    """Actuator used to edit objects"""

    angular_velocity: bpy_prop_array[float]
    """ Angular velocity upon creation

    :type: bpy_prop_array[float]
    """

    dynamic_operation: typing.Literal[
        "RESTOREDYN", "SUSPENDDYN", "ENABLERIGIDBODY", "DISABLERIGIDBODY", "SETMASS"
    ]
    """ 

    :type: typing.Literal['RESTOREDYN','SUSPENDDYN','ENABLERIGIDBODY','DISABLERIGIDBODY','SETMASS']
    """

    linear_velocity: bpy_prop_array[float]
    """ Velocity upon creation

    :type: bpy_prop_array[float]
    """

    mass: float
    """ The mass of the object

    :type: float
    """

    mesh: Mesh | None
    """ Replace the existing, when left blank 'Phys' will remake the existing physics mesh

    :type: Mesh | None
    """

    mode: typing.Literal["ADDOBJECT", "ENDOBJECT", "REPLACEMESH", "TRACKTO", "DYNAMICS"]
    """ The mode of the actuator

    :type: typing.Literal['ADDOBJECT','ENDOBJECT','REPLACEMESH','TRACKTO','DYNAMICS']
    """

    object: Object | None
    """ Add this Object and all its children (can't be on a visible layer)

    :type: Object | None
    """

    time: int
    """ Duration the new Object lives or the track takes

    :type: int
    """

    track_axis: typing.Literal[
        "TRACKAXISX",
        "TRACKAXISY",
        "TRACKAXISZ",
        "TRACKAXISNEGX",
        "TRACKAXISNEGY",
        "TRACKAXISNEGZ",
    ]
    """ The axis that points to the target object

    :type: typing.Literal['TRACKAXISX','TRACKAXISY','TRACKAXISZ','TRACKAXISNEGX','TRACKAXISNEGY','TRACKAXISNEGZ']
    """

    track_object: Object | None
    """ Track to this Object

    :type: Object | None
    """

    up_axis: typing.Literal["UPAXISX", "UPAXISY", "UPAXISZ"]
    """ The axis that points upward

    :type: typing.Literal['UPAXISX','UPAXISY','UPAXISZ']
    """

    use_3d_tracking: bool
    """ Enable 3D tracking

    :type: bool
    """

    use_local_angular_velocity: bool
    """ Apply the rotation locally

    :type: bool
    """

    use_local_linear_velocity: bool
    """ Apply the transformation locally

    :type: bool
    """

    use_replace_display_mesh: bool
    """ Replace the display mesh

    :type: bool
    """

    use_replace_physics_mesh: bool
    """ Replace the physics mesh (triangle bounds only - compound shapes not supported)

    :type: bool
    """

class EffectSequence(Sequence, bpy_struct):
    """Sequence strip applying an effect on the images created by other strips"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class EffectorWeights(bpy_struct):
    """Effector weights for physics simulation"""

    all: float
    """ All effector's weight

    :type: float
    """

    apply_to_hair_growing: bool
    """ Use force fields when growing hair

    :type: bool
    """

    boid: float
    """ Boid effector weight

    :type: float
    """

    charge: float
    """ Charge effector weight

    :type: float
    """

    curve_guide: float
    """ Curve guide effector weight

    :type: float
    """

    drag: float
    """ Drag effector weight

    :type: float
    """

    force: float
    """ Force effector weight

    :type: float
    """

    gravity: float
    """ Global gravity weight

    :type: float
    """

    group: Group | None
    """ Limit effectors to this Group

    :type: Group | None
    """

    harmonic: float
    """ Harmonic effector weight

    :type: float
    """

    lennardjones: float
    """ Lennard-Jones effector weight

    :type: float
    """

    magnetic: float
    """ Magnetic effector weight

    :type: float
    """

    smokeflow: float
    """ Smoke Flow effector weight

    :type: float
    """

    texture: float
    """ Texture effector weight

    :type: float
    """

    turbulence: float
    """ Turbulence effector weight

    :type: float
    """

    vortex: float
    """ Vortex effector weight

    :type: float
    """

    wind: float
    """ Wind effector weight

    :type: float
    """

class EnumProperty(Property, bpy_struct):
    """RNA enumeration property definition, to choose from a number of predefined options"""

    default: typing.Literal["DUMMY"]
    """ Default value for this enum

    :type: typing.Literal['DUMMY']
    """

    default_flag: set[str]
    """ Default value for this enum

    :type: set[str]
    """

    enum_items: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property

    :type: bpy_prop_collection[EnumPropertyItem]
    """

    enum_items_static: bpy_prop_collection[EnumPropertyItem]
    """ Possible values for the property (never calls optional dynamic generation of those)

    :type: bpy_prop_collection[EnumPropertyItem]
    """

class EnumPropertyItem(bpy_struct):
    """Definition of a choice in an RNA enum property"""

    description: str
    """ Description of the item's purpose

    :type: str
    """

    icon: typing.Literal[
        "NONE",
        "QUESTION",
        "ERROR",
        "CANCEL",
        "TRIA_RIGHT",
        "TRIA_DOWN",
        "TRIA_LEFT",
        "TRIA_UP",
        "ARROW_LEFTRIGHT",
        "PLUS",
        "DISCLOSURE_TRI_DOWN",
        "DISCLOSURE_TRI_RIGHT",
        "RADIOBUT_OFF",
        "RADIOBUT_ON",
        "MENU_PANEL",
        "BLENDER",
        "GRIP",
        "DOT",
        "COLLAPSEMENU",
        "X",
        "GO_LEFT",
        "PLUG",
        "UI",
        "NODE",
        "NODE_SEL",
        "FULLSCREEN",
        "SPLITSCREEN",
        "RIGHTARROW_THIN",
        "BORDERMOVE",
        "VIEWZOOM",
        "ZOOMIN",
        "ZOOMOUT",
        "PANEL_CLOSE",
        "COPY_ID",
        "EYEDROPPER",
        "LINK_AREA",
        "AUTO",
        "CHECKBOX_DEHLT",
        "CHECKBOX_HLT",
        "UNLOCKED",
        "LOCKED",
        "UNPINNED",
        "PINNED",
        "SCREEN_BACK",
        "RIGHTARROW",
        "DOWNARROW_HLT",
        "DOTSUP",
        "DOTSDOWN",
        "LINK",
        "INLINK",
        "PLUGIN",
        "HELP",
        "GHOST_ENABLED",
        "COLOR",
        "LINKED",
        "UNLINKED",
        "HAND",
        "ZOOM_ALL",
        "ZOOM_SELECTED",
        "ZOOM_PREVIOUS",
        "ZOOM_IN",
        "ZOOM_OUT",
        "RENDER_REGION",
        "BORDER_RECT",
        "BORDER_LASSO",
        "FREEZE",
        "STYLUS_PRESSURE",
        "GHOST_DISABLED",
        "NEW",
        "FILE_TICK",
        "QUIT",
        "URL",
        "RECOVER_LAST",
        "FULLSCREEN_ENTER",
        "FULLSCREEN_EXIT",
        "BLANK1",
        "LAMP",
        "MATERIAL",
        "TEXTURE",
        "ANIM",
        "WORLD",
        "SCENE",
        "EDIT",
        "GAME",
        "RADIO",
        "SCRIPT",
        "PARTICLES",
        "PHYSICS",
        "SPEAKER",
        "TEXTURE_SHADED",
        "VIEW3D",
        "IPO",
        "OOPS",
        "BUTS",
        "FILESEL",
        "IMAGE_COL",
        "INFO",
        "SEQUENCE",
        "TEXT",
        "IMASEL",
        "SOUND",
        "ACTION",
        "NLA",
        "SCRIPTWIN",
        "TIME",
        "NODETREE",
        "LOGIC",
        "CONSOLE",
        "PREFERENCES",
        "CLIP",
        "ASSET_MANAGER",
        "OBJECT_DATAMODE",
        "EDITMODE_HLT",
        "FACESEL_HLT",
        "VPAINT_HLT",
        "TPAINT_HLT",
        "WPAINT_HLT",
        "SCULPTMODE_HLT",
        "POSE_HLT",
        "PARTICLEMODE",
        "LIGHTPAINT",
        "SCENE_DATA",
        "RENDERLAYERS",
        "WORLD_DATA",
        "OBJECT_DATA",
        "MESH_DATA",
        "CURVE_DATA",
        "META_DATA",
        "LATTICE_DATA",
        "LAMP_DATA",
        "MATERIAL_DATA",
        "TEXTURE_DATA",
        "ANIM_DATA",
        "CAMERA_DATA",
        "PARTICLE_DATA",
        "LIBRARY_DATA_DIRECT",
        "GROUP",
        "ARMATURE_DATA",
        "POSE_DATA",
        "BONE_DATA",
        "CONSTRAINT",
        "SHAPEKEY_DATA",
        "CONSTRAINT_BONE",
        "CAMERA_STEREO",
        "PACKAGE",
        "UGLYPACKAGE",
        "BRUSH_DATA",
        "IMAGE_DATA",
        "FILE",
        "FCURVE",
        "FONT_DATA",
        "RENDER_RESULT",
        "SURFACE_DATA",
        "EMPTY_DATA",
        "SETTINGS",
        "RENDER_ANIMATION",
        "RENDER_STILL",
        "LIBRARY_DATA_BROKEN",
        "BOIDS",
        "STRANDS",
        "LIBRARY_DATA_INDIRECT",
        "GREASEPENCIL",
        "LINE_DATA",
        "GROUP_BONE",
        "GROUP_VERTEX",
        "GROUP_VCOL",
        "GROUP_UVS",
        "RNA",
        "RNA_ADD",
        "OUTLINER_OB_EMPTY",
        "OUTLINER_OB_MESH",
        "OUTLINER_OB_CURVE",
        "OUTLINER_OB_LATTICE",
        "OUTLINER_OB_META",
        "OUTLINER_OB_LAMP",
        "OUTLINER_OB_CAMERA",
        "OUTLINER_OB_ARMATURE",
        "OUTLINER_OB_FONT",
        "OUTLINER_OB_SURFACE",
        "OUTLINER_OB_SPEAKER",
        "RESTRICT_COLOR_OFF",
        "RESTRICT_COLOR_ON",
        "RESTRICT_VIEW_OFF",
        "RESTRICT_VIEW_ON",
        "RESTRICT_SELECT_OFF",
        "RESTRICT_SELECT_ON",
        "RESTRICT_RENDER_OFF",
        "RESTRICT_RENDER_ON",
        "OUTLINER_DATA_EMPTY",
        "OUTLINER_DATA_MESH",
        "OUTLINER_DATA_CURVE",
        "OUTLINER_DATA_LATTICE",
        "OUTLINER_DATA_META",
        "OUTLINER_DATA_LAMP",
        "OUTLINER_DATA_CAMERA",
        "OUTLINER_DATA_ARMATURE",
        "OUTLINER_DATA_FONT",
        "OUTLINER_DATA_SURFACE",
        "OUTLINER_DATA_SPEAKER",
        "OUTLINER_DATA_POSE",
        "MESH_PLANE",
        "MESH_CUBE",
        "MESH_CIRCLE",
        "MESH_UVSPHERE",
        "MESH_ICOSPHERE",
        "MESH_GRID",
        "MESH_MONKEY",
        "MESH_CYLINDER",
        "MESH_TORUS",
        "MESH_CONE",
        "MESH_CAPSULE",
        "LAMP_POINT",
        "LAMP_SUN",
        "LAMP_SPOT",
        "LAMP_HEMI",
        "LAMP_AREA",
        "META_EMPTY",
        "META_PLANE",
        "META_CUBE",
        "META_BALL",
        "META_ELLIPSOID",
        "META_CAPSULE",
        "SURFACE_NCURVE",
        "SURFACE_NCIRCLE",
        "SURFACE_NSURFACE",
        "SURFACE_NCYLINDER",
        "SURFACE_NSPHERE",
        "SURFACE_NTORUS",
        "CURVE_BEZCURVE",
        "CURVE_BEZCIRCLE",
        "CURVE_NCURVE",
        "CURVE_NCIRCLE",
        "CURVE_PATH",
        "COLOR_RED",
        "COLOR_GREEN",
        "COLOR_BLUE",
        "TRIA_RIGHT_BAR",
        "TRIA_DOWN_BAR",
        "TRIA_LEFT_BAR",
        "TRIA_UP_BAR",
        "FORCE_FORCE",
        "FORCE_WIND",
        "FORCE_VORTEX",
        "FORCE_MAGNETIC",
        "FORCE_HARMONIC",
        "FORCE_CHARGE",
        "FORCE_LENNARDJONES",
        "FORCE_TEXTURE",
        "FORCE_CURVE",
        "FORCE_BOID",
        "FORCE_TURBULENCE",
        "FORCE_DRAG",
        "FORCE_SMOKEFLOW",
        "NODE_INSERT_ON",
        "NODE_INSERT_OFF",
        "MODIFIER",
        "MOD_WAVE",
        "MOD_BUILD",
        "MOD_DECIM",
        "MOD_MIRROR",
        "MOD_SOFT",
        "MOD_SUBSURF",
        "HOOK",
        "MOD_PHYSICS",
        "MOD_PARTICLES",
        "MOD_BOOLEAN",
        "MOD_EDGESPLIT",
        "MOD_ARRAY",
        "MOD_UVPROJECT",
        "MOD_DISPLACE",
        "MOD_CURVE",
        "MOD_LATTICE",
        "CONSTRAINT_DATA",
        "MOD_ARMATURE",
        "MOD_SHRINKWRAP",
        "MOD_CAST",
        "MOD_MESHDEFORM",
        "MOD_BEVEL",
        "MOD_SMOOTH",
        "MOD_SIMPLEDEFORM",
        "MOD_MASK",
        "MOD_CLOTH",
        "MOD_EXPLODE",
        "MOD_FLUIDSIM",
        "MOD_MULTIRES",
        "MOD_SMOKE",
        "MOD_SOLIDIFY",
        "MOD_SCREW",
        "MOD_VERTEX_WEIGHT",
        "MOD_DYNAMICPAINT",
        "MOD_REMESH",
        "MOD_OCEAN",
        "MOD_WARP",
        "MOD_SKIN",
        "MOD_TRIANGULATE",
        "MOD_WIREFRAME",
        "MOD_DATA_TRANSFER",
        "MOD_NORMALEDIT",
        "REC",
        "PLAY",
        "FF",
        "REW",
        "PAUSE",
        "PREV_KEYFRAME",
        "NEXT_KEYFRAME",
        "PLAY_AUDIO",
        "PLAY_REVERSE",
        "PREVIEW_RANGE",
        "ACTION_TWEAK",
        "PMARKER_ACT",
        "PMARKER_SEL",
        "PMARKER",
        "MARKER_HLT",
        "MARKER",
        "SPACE2",
        "SPACE3",
        "KEYINGSET",
        "KEY_DEHLT",
        "KEY_HLT",
        "MUTE_IPO_OFF",
        "MUTE_IPO_ON",
        "VISIBLE_IPO_OFF",
        "VISIBLE_IPO_ON",
        "DRIVER",
        "SOLO_OFF",
        "SOLO_ON",
        "FRAME_PREV",
        "FRAME_NEXT",
        "NLA_PUSHDOWN",
        "IPO_CONSTANT",
        "IPO_LINEAR",
        "IPO_BEZIER",
        "IPO_SINE",
        "IPO_QUAD",
        "IPO_CUBIC",
        "IPO_QUART",
        "IPO_QUINT",
        "IPO_EXPO",
        "IPO_CIRC",
        "IPO_BOUNCE",
        "IPO_ELASTIC",
        "IPO_BACK",
        "IPO_EASE_IN",
        "IPO_EASE_OUT",
        "IPO_EASE_IN_OUT",
        "VERTEXSEL",
        "EDGESEL",
        "FACESEL",
        "LOOPSEL",
        "ROTATE",
        "CURSOR",
        "ROTATECOLLECTION",
        "ROTATECENTER",
        "ROTACTIVE",
        "ALIGN",
        "SMOOTHCURVE",
        "SPHERECURVE",
        "ROOTCURVE",
        "SHARPCURVE",
        "LINCURVE",
        "NOCURVE",
        "RNDCURVE",
        "PROP_OFF",
        "PROP_ON",
        "PROP_CON",
        "SCULPT_DYNTOPO",
        "PARTICLE_POINT",
        "PARTICLE_TIP",
        "PARTICLE_PATH",
        "MAN_TRANS",
        "MAN_ROT",
        "MAN_SCALE",
        "MANIPUL",
        "SNAP_OFF",
        "SNAP_ON",
        "SNAP_NORMAL",
        "SNAP_GRID",
        "SNAP_VERTEX",
        "SNAP_EDGE",
        "SNAP_FACE",
        "SNAP_VOLUME",
        "SNAP_INCREMENT",
        "STICKY_UVS_LOC",
        "STICKY_UVS_DISABLE",
        "STICKY_UVS_VERT",
        "CLIPUV_DEHLT",
        "CLIPUV_HLT",
        "SNAP_PEEL_OBJECT",
        "GRID",
        "PASTEDOWN",
        "COPYDOWN",
        "PASTEFLIPUP",
        "PASTEFLIPDOWN",
        "SNAP_SURFACE",
        "AUTOMERGE_ON",
        "AUTOMERGE_OFF",
        "RETOPO",
        "UV_VERTEXSEL",
        "UV_EDGESEL",
        "UV_FACESEL",
        "UV_ISLANDSEL",
        "UV_SYNC_SELECT",
        "BBOX",
        "WIRE",
        "SOLID",
        "SMOOTH",
        "POTATO",
        "ORTHO",
        "LOCKVIEW_OFF",
        "LOCKVIEW_ON",
        "AXIS_SIDE",
        "AXIS_FRONT",
        "AXIS_TOP",
        "NDOF_DOM",
        "NDOF_TURN",
        "NDOF_FLY",
        "NDOF_TRANS",
        "LAYER_USED",
        "LAYER_ACTIVE",
        "SORTALPHA",
        "SORTBYEXT",
        "SORTTIME",
        "SORTSIZE",
        "LONGDISPLAY",
        "SHORTDISPLAY",
        "GHOST",
        "IMGDISPLAY",
        "SAVE_AS",
        "SAVE_COPY",
        "BOOKMARKS",
        "FONTPREVIEW",
        "FILTER",
        "NEWFOLDER",
        "OPEN_RECENT",
        "FILE_PARENT",
        "FILE_REFRESH",
        "FILE_FOLDER",
        "FILE_BLANK",
        "FILE_BLEND",
        "FILE_IMAGE",
        "FILE_MOVIE",
        "FILE_SCRIPT",
        "FILE_SOUND",
        "FILE_FONT",
        "FILE_TEXT",
        "RECOVER_AUTO",
        "SAVE_PREFS",
        "LINK_BLEND",
        "APPEND_BLEND",
        "IMPORT",
        "EXPORT",
        "EXTERNAL_DATA",
        "LOAD_FACTORY",
        "LOOP_BACK",
        "LOOP_FORWARDS",
        "BACK",
        "FORWARD",
        "FILE_HIDDEN",
        "FILE_BACKUP",
        "DISK_DRIVE",
        "MATPLANE",
        "MATSPHERE",
        "MATCUBE",
        "MONKEY",
        "HAIR",
        "ALIASED",
        "ANTIALIASED",
        "MAT_SPHERE_SKY",
        "WORDWRAP_OFF",
        "WORDWRAP_ON",
        "SYNTAX_OFF",
        "SYNTAX_ON",
        "LINENUMBERS_OFF",
        "LINENUMBERS_ON",
        "SCRIPTPLUGINS",
        "SEQ_SEQUENCER",
        "SEQ_PREVIEW",
        "SEQ_LUMA_WAVEFORM",
        "SEQ_CHROMA_SCOPE",
        "SEQ_HISTOGRAM",
        "SEQ_SPLITVIEW",
        "IMAGE_RGB",
        "IMAGE_RGB_ALPHA",
        "IMAGE_ALPHA",
        "IMAGE_ZDEPTH",
        "IMAGEFILE",
        "BRUSH_ADD",
        "BRUSH_BLOB",
        "BRUSH_BLUR",
        "BRUSH_CLAY",
        "BRUSH_CLAY_STRIPS",
        "BRUSH_CLONE",
        "BRUSH_CREASE",
        "BRUSH_DARKEN",
        "BRUSH_FILL",
        "BRUSH_FLATTEN",
        "BRUSH_GRAB",
        "BRUSH_INFLATE",
        "BRUSH_LAYER",
        "BRUSH_LIGHTEN",
        "BRUSH_MASK",
        "BRUSH_MIX",
        "BRUSH_MULTIPLY",
        "BRUSH_NUDGE",
        "BRUSH_PINCH",
        "BRUSH_SCRAPE",
        "BRUSH_SCULPT_DRAW",
        "BRUSH_SMEAR",
        "BRUSH_SMOOTH",
        "BRUSH_SNAKE_HOOK",
        "BRUSH_SOFTEN",
        "BRUSH_SUBTRACT",
        "BRUSH_TEXDRAW",
        "BRUSH_TEXFILL",
        "BRUSH_TEXMASK",
        "BRUSH_THUMB",
        "BRUSH_ROTATE",
        "BRUSH_VERTEXDRAW",
        "MATCAP_01",
        "MATCAP_02",
        "MATCAP_03",
        "MATCAP_04",
        "MATCAP_05",
        "MATCAP_06",
        "MATCAP_07",
        "MATCAP_08",
        "MATCAP_09",
        "MATCAP_10",
        "MATCAP_11",
        "MATCAP_12",
        "MATCAP_13",
        "MATCAP_14",
        "MATCAP_15",
        "MATCAP_16",
        "MATCAP_17",
        "MATCAP_18",
        "MATCAP_19",
        "MATCAP_20",
        "MATCAP_21",
        "MATCAP_22",
        "MATCAP_23",
        "MATCAP_24",
        "VIEW3D_VEC",
        "EDIT_VEC",
        "EDITMODE_VEC_DEHLT",
        "EDITMODE_VEC_HLT",
        "DISCLOSURE_TRI_RIGHT_VEC",
        "DISCLOSURE_TRI_DOWN_VEC",
        "MOVE_UP_VEC",
        "MOVE_DOWN_VEC",
        "X_VEC",
        "SMALL_TRI_RIGHT_VEC",
        "KEYTYPE_KEYFRAME_VEC",
        "KEYTYPE_BREAKDOWN_VEC",
        "KEYTYPE_EXTREME_VEC",
        "KEYTYPE_JITTER_VEC",
        "KEYTYPE_MOVING_HOLD_VEC",
        "COLORSET_01_VEC",
        "COLORSET_02_VEC",
        "COLORSET_03_VEC",
        "COLORSET_04_VEC",
        "COLORSET_05_VEC",
        "COLORSET_06_VEC",
        "COLORSET_07_VEC",
        "COLORSET_08_VEC",
        "COLORSET_09_VEC",
        "COLORSET_10_VEC",
        "COLORSET_11_VEC",
        "COLORSET_12_VEC",
        "COLORSET_13_VEC",
        "COLORSET_14_VEC",
        "COLORSET_15_VEC",
        "COLORSET_16_VEC",
        "COLORSET_17_VEC",
        "COLORSET_18_VEC",
        "COLORSET_19_VEC",
        "COLORSET_20_VEC",
    ]
    """ Icon of the item

    :type: typing.Literal['NONE','QUESTION','ERROR','CANCEL','TRIA_RIGHT','TRIA_DOWN','TRIA_LEFT','TRIA_UP','ARROW_LEFTRIGHT','PLUS','DISCLOSURE_TRI_DOWN','DISCLOSURE_TRI_RIGHT','RADIOBUT_OFF','RADIOBUT_ON','MENU_PANEL','BLENDER','GRIP','DOT','COLLAPSEMENU','X','GO_LEFT','PLUG','UI','NODE','NODE_SEL','FULLSCREEN','SPLITSCREEN','RIGHTARROW_THIN','BORDERMOVE','VIEWZOOM','ZOOMIN','ZOOMOUT','PANEL_CLOSE','COPY_ID','EYEDROPPER','LINK_AREA','AUTO','CHECKBOX_DEHLT','CHECKBOX_HLT','UNLOCKED','LOCKED','UNPINNED','PINNED','SCREEN_BACK','RIGHTARROW','DOWNARROW_HLT','DOTSUP','DOTSDOWN','LINK','INLINK','PLUGIN','HELP','GHOST_ENABLED','COLOR','LINKED','UNLINKED','HAND','ZOOM_ALL','ZOOM_SELECTED','ZOOM_PREVIOUS','ZOOM_IN','ZOOM_OUT','RENDER_REGION','BORDER_RECT','BORDER_LASSO','FREEZE','STYLUS_PRESSURE','GHOST_DISABLED','NEW','FILE_TICK','QUIT','URL','RECOVER_LAST','FULLSCREEN_ENTER','FULLSCREEN_EXIT','BLANK1','LAMP','MATERIAL','TEXTURE','ANIM','WORLD','SCENE','EDIT','GAME','RADIO','SCRIPT','PARTICLES','PHYSICS','SPEAKER','TEXTURE_SHADED','VIEW3D','IPO','OOPS','BUTS','FILESEL','IMAGE_COL','INFO','SEQUENCE','TEXT','IMASEL','SOUND','ACTION','NLA','SCRIPTWIN','TIME','NODETREE','LOGIC','CONSOLE','PREFERENCES','CLIP','ASSET_MANAGER','OBJECT_DATAMODE','EDITMODE_HLT','FACESEL_HLT','VPAINT_HLT','TPAINT_HLT','WPAINT_HLT','SCULPTMODE_HLT','POSE_HLT','PARTICLEMODE','LIGHTPAINT','SCENE_DATA','RENDERLAYERS','WORLD_DATA','OBJECT_DATA','MESH_DATA','CURVE_DATA','META_DATA','LATTICE_DATA','LAMP_DATA','MATERIAL_DATA','TEXTURE_DATA','ANIM_DATA','CAMERA_DATA','PARTICLE_DATA','LIBRARY_DATA_DIRECT','GROUP','ARMATURE_DATA','POSE_DATA','BONE_DATA','CONSTRAINT','SHAPEKEY_DATA','CONSTRAINT_BONE','CAMERA_STEREO','PACKAGE','UGLYPACKAGE','BRUSH_DATA','IMAGE_DATA','FILE','FCURVE','FONT_DATA','RENDER_RESULT','SURFACE_DATA','EMPTY_DATA','SETTINGS','RENDER_ANIMATION','RENDER_STILL','LIBRARY_DATA_BROKEN','BOIDS','STRANDS','LIBRARY_DATA_INDIRECT','GREASEPENCIL','LINE_DATA','GROUP_BONE','GROUP_VERTEX','GROUP_VCOL','GROUP_UVS','RNA','RNA_ADD','OUTLINER_OB_EMPTY','OUTLINER_OB_MESH','OUTLINER_OB_CURVE','OUTLINER_OB_LATTICE','OUTLINER_OB_META','OUTLINER_OB_LAMP','OUTLINER_OB_CAMERA','OUTLINER_OB_ARMATURE','OUTLINER_OB_FONT','OUTLINER_OB_SURFACE','OUTLINER_OB_SPEAKER','RESTRICT_COLOR_OFF','RESTRICT_COLOR_ON','RESTRICT_VIEW_OFF','RESTRICT_VIEW_ON','RESTRICT_SELECT_OFF','RESTRICT_SELECT_ON','RESTRICT_RENDER_OFF','RESTRICT_RENDER_ON','OUTLINER_DATA_EMPTY','OUTLINER_DATA_MESH','OUTLINER_DATA_CURVE','OUTLINER_DATA_LATTICE','OUTLINER_DATA_META','OUTLINER_DATA_LAMP','OUTLINER_DATA_CAMERA','OUTLINER_DATA_ARMATURE','OUTLINER_DATA_FONT','OUTLINER_DATA_SURFACE','OUTLINER_DATA_SPEAKER','OUTLINER_DATA_POSE','MESH_PLANE','MESH_CUBE','MESH_CIRCLE','MESH_UVSPHERE','MESH_ICOSPHERE','MESH_GRID','MESH_MONKEY','MESH_CYLINDER','MESH_TORUS','MESH_CONE','MESH_CAPSULE','LAMP_POINT','LAMP_SUN','LAMP_SPOT','LAMP_HEMI','LAMP_AREA','META_EMPTY','META_PLANE','META_CUBE','META_BALL','META_ELLIPSOID','META_CAPSULE','SURFACE_NCURVE','SURFACE_NCIRCLE','SURFACE_NSURFACE','SURFACE_NCYLINDER','SURFACE_NSPHERE','SURFACE_NTORUS','CURVE_BEZCURVE','CURVE_BEZCIRCLE','CURVE_NCURVE','CURVE_NCIRCLE','CURVE_PATH','COLOR_RED','COLOR_GREEN','COLOR_BLUE','TRIA_RIGHT_BAR','TRIA_DOWN_BAR','TRIA_LEFT_BAR','TRIA_UP_BAR','FORCE_FORCE','FORCE_WIND','FORCE_VORTEX','FORCE_MAGNETIC','FORCE_HARMONIC','FORCE_CHARGE','FORCE_LENNARDJONES','FORCE_TEXTURE','FORCE_CURVE','FORCE_BOID','FORCE_TURBULENCE','FORCE_DRAG','FORCE_SMOKEFLOW','NODE_INSERT_ON','NODE_INSERT_OFF','MODIFIER','MOD_WAVE','MOD_BUILD','MOD_DECIM','MOD_MIRROR','MOD_SOFT','MOD_SUBSURF','HOOK','MOD_PHYSICS','MOD_PARTICLES','MOD_BOOLEAN','MOD_EDGESPLIT','MOD_ARRAY','MOD_UVPROJECT','MOD_DISPLACE','MOD_CURVE','MOD_LATTICE','CONSTRAINT_DATA','MOD_ARMATURE','MOD_SHRINKWRAP','MOD_CAST','MOD_MESHDEFORM','MOD_BEVEL','MOD_SMOOTH','MOD_SIMPLEDEFORM','MOD_MASK','MOD_CLOTH','MOD_EXPLODE','MOD_FLUIDSIM','MOD_MULTIRES','MOD_SMOKE','MOD_SOLIDIFY','MOD_SCREW','MOD_VERTEX_WEIGHT','MOD_DYNAMICPAINT','MOD_REMESH','MOD_OCEAN','MOD_WARP','MOD_SKIN','MOD_TRIANGULATE','MOD_WIREFRAME','MOD_DATA_TRANSFER','MOD_NORMALEDIT','REC','PLAY','FF','REW','PAUSE','PREV_KEYFRAME','NEXT_KEYFRAME','PLAY_AUDIO','PLAY_REVERSE','PREVIEW_RANGE','ACTION_TWEAK','PMARKER_ACT','PMARKER_SEL','PMARKER','MARKER_HLT','MARKER','SPACE2','SPACE3','KEYINGSET','KEY_DEHLT','KEY_HLT','MUTE_IPO_OFF','MUTE_IPO_ON','VISIBLE_IPO_OFF','VISIBLE_IPO_ON','DRIVER','SOLO_OFF','SOLO_ON','FRAME_PREV','FRAME_NEXT','NLA_PUSHDOWN','IPO_CONSTANT','IPO_LINEAR','IPO_BEZIER','IPO_SINE','IPO_QUAD','IPO_CUBIC','IPO_QUART','IPO_QUINT','IPO_EXPO','IPO_CIRC','IPO_BOUNCE','IPO_ELASTIC','IPO_BACK','IPO_EASE_IN','IPO_EASE_OUT','IPO_EASE_IN_OUT','VERTEXSEL','EDGESEL','FACESEL','LOOPSEL','ROTATE','CURSOR','ROTATECOLLECTION','ROTATECENTER','ROTACTIVE','ALIGN','SMOOTHCURVE','SPHERECURVE','ROOTCURVE','SHARPCURVE','LINCURVE','NOCURVE','RNDCURVE','PROP_OFF','PROP_ON','PROP_CON','SCULPT_DYNTOPO','PARTICLE_POINT','PARTICLE_TIP','PARTICLE_PATH','MAN_TRANS','MAN_ROT','MAN_SCALE','MANIPUL','SNAP_OFF','SNAP_ON','SNAP_NORMAL','SNAP_GRID','SNAP_VERTEX','SNAP_EDGE','SNAP_FACE','SNAP_VOLUME','SNAP_INCREMENT','STICKY_UVS_LOC','STICKY_UVS_DISABLE','STICKY_UVS_VERT','CLIPUV_DEHLT','CLIPUV_HLT','SNAP_PEEL_OBJECT','GRID','PASTEDOWN','COPYDOWN','PASTEFLIPUP','PASTEFLIPDOWN','SNAP_SURFACE','AUTOMERGE_ON','AUTOMERGE_OFF','RETOPO','UV_VERTEXSEL','UV_EDGESEL','UV_FACESEL','UV_ISLANDSEL','UV_SYNC_SELECT','BBOX','WIRE','SOLID','SMOOTH','POTATO','ORTHO','LOCKVIEW_OFF','LOCKVIEW_ON','AXIS_SIDE','AXIS_FRONT','AXIS_TOP','NDOF_DOM','NDOF_TURN','NDOF_FLY','NDOF_TRANS','LAYER_USED','LAYER_ACTIVE','SORTALPHA','SORTBYEXT','SORTTIME','SORTSIZE','LONGDISPLAY','SHORTDISPLAY','GHOST','IMGDISPLAY','SAVE_AS','SAVE_COPY','BOOKMARKS','FONTPREVIEW','FILTER','NEWFOLDER','OPEN_RECENT','FILE_PARENT','FILE_REFRESH','FILE_FOLDER','FILE_BLANK','FILE_BLEND','FILE_IMAGE','FILE_MOVIE','FILE_SCRIPT','FILE_SOUND','FILE_FONT','FILE_TEXT','RECOVER_AUTO','SAVE_PREFS','LINK_BLEND','APPEND_BLEND','IMPORT','EXPORT','EXTERNAL_DATA','LOAD_FACTORY','LOOP_BACK','LOOP_FORWARDS','BACK','FORWARD','FILE_HIDDEN','FILE_BACKUP','DISK_DRIVE','MATPLANE','MATSPHERE','MATCUBE','MONKEY','HAIR','ALIASED','ANTIALIASED','MAT_SPHERE_SKY','WORDWRAP_OFF','WORDWRAP_ON','SYNTAX_OFF','SYNTAX_ON','LINENUMBERS_OFF','LINENUMBERS_ON','SCRIPTPLUGINS','SEQ_SEQUENCER','SEQ_PREVIEW','SEQ_LUMA_WAVEFORM','SEQ_CHROMA_SCOPE','SEQ_HISTOGRAM','SEQ_SPLITVIEW','IMAGE_RGB','IMAGE_RGB_ALPHA','IMAGE_ALPHA','IMAGE_ZDEPTH','IMAGEFILE','BRUSH_ADD','BRUSH_BLOB','BRUSH_BLUR','BRUSH_CLAY','BRUSH_CLAY_STRIPS','BRUSH_CLONE','BRUSH_CREASE','BRUSH_DARKEN','BRUSH_FILL','BRUSH_FLATTEN','BRUSH_GRAB','BRUSH_INFLATE','BRUSH_LAYER','BRUSH_LIGHTEN','BRUSH_MASK','BRUSH_MIX','BRUSH_MULTIPLY','BRUSH_NUDGE','BRUSH_PINCH','BRUSH_SCRAPE','BRUSH_SCULPT_DRAW','BRUSH_SMEAR','BRUSH_SMOOTH','BRUSH_SNAKE_HOOK','BRUSH_SOFTEN','BRUSH_SUBTRACT','BRUSH_TEXDRAW','BRUSH_TEXFILL','BRUSH_TEXMASK','BRUSH_THUMB','BRUSH_ROTATE','BRUSH_VERTEXDRAW','MATCAP_01','MATCAP_02','MATCAP_03','MATCAP_04','MATCAP_05','MATCAP_06','MATCAP_07','MATCAP_08','MATCAP_09','MATCAP_10','MATCAP_11','MATCAP_12','MATCAP_13','MATCAP_14','MATCAP_15','MATCAP_16','MATCAP_17','MATCAP_18','MATCAP_19','MATCAP_20','MATCAP_21','MATCAP_22','MATCAP_23','MATCAP_24','VIEW3D_VEC','EDIT_VEC','EDITMODE_VEC_DEHLT','EDITMODE_VEC_HLT','DISCLOSURE_TRI_RIGHT_VEC','DISCLOSURE_TRI_DOWN_VEC','MOVE_UP_VEC','MOVE_DOWN_VEC','X_VEC','SMALL_TRI_RIGHT_VEC','KEYTYPE_KEYFRAME_VEC','KEYTYPE_BREAKDOWN_VEC','KEYTYPE_EXTREME_VEC','KEYTYPE_JITTER_VEC','KEYTYPE_MOVING_HOLD_VEC','COLORSET_01_VEC','COLORSET_02_VEC','COLORSET_03_VEC','COLORSET_04_VEC','COLORSET_05_VEC','COLORSET_06_VEC','COLORSET_07_VEC','COLORSET_08_VEC','COLORSET_09_VEC','COLORSET_10_VEC','COLORSET_11_VEC','COLORSET_12_VEC','COLORSET_13_VEC','COLORSET_14_VEC','COLORSET_15_VEC','COLORSET_16_VEC','COLORSET_17_VEC','COLORSET_18_VEC','COLORSET_19_VEC','COLORSET_20_VEC']
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    name: str
    """ Human readable name

    :type: str
    """

    value: int
    """ Value of the item

    :type: int
    """

class EnvironmentMap(bpy_struct):
    """Environment map created by the renderer and cached for subsequent renders"""

    clip_end: float
    """ Objects further than this are not visible to map

    :type: float
    """

    clip_start: float
    """ Objects nearer than this are not visible to map

    :type: float
    """

    depth: int
    """ Number of times a map will be rendered recursively (mirror effects)

    :type: int
    """

    is_valid: bool
    """ True if this map is ready for use, False if it needs rendering

    :type: bool
    """

    layers_ignore: bpy_prop_array[bool]
    """ Hide objects on these layers when generating the Environment Map

    :type: bpy_prop_array[bool]
    """

    mapping: typing.Literal["CUBE", "PLANE"]
    """ 

    :type: typing.Literal['CUBE','PLANE']
    """

    resolution: int
    """ Pixel resolution of the rendered environment map

    :type: int
    """

    source: typing.Literal["STATIC", "ANIMATED", "IMAGE_FILE"]
    """ 

    :type: typing.Literal['STATIC','ANIMATED','IMAGE_FILE']
    """

    viewpoint_object: Object | None
    """ Object to use as the environment map's viewpoint location

    :type: Object | None
    """

    zoom: float
    """ 

    :type: float
    """

    def clear(self):
        """Discard the environment map and free it from memory"""

    def save(
        self,
        filepath: str,
        scene: Scene | None = None,
        layout: collections.abc.Iterable[float] | None = (
            0.0,
            0.0,
            1.0,
            0.0,
            2.0,
            0.0,
            0.0,
            1.0,
            1.0,
            1.0,
            2.0,
            1.0,
        ),
    ):
        """Save the environment map to disc using the scene render settings

        :param filepath: File path, Location of the output file
        :type filepath: str
        :param scene: Overrides the scene from which image parameters are taken
        :type scene: Scene | None
        :param layout: File layout, Flat array describing the X,Y position of each cube face in the output image, where 1 is the size of a face - order is [+Z -Z +Y -X -Y +X] (use -1 to skip a face)
        :type layout: collections.abc.Iterable[float] | None
        """

class EnvironmentMapTexture(Texture, ID, bpy_struct):
    """Environment map texture"""

    environment_map: EnvironmentMap
    """ Get the environment map associated with this texture

    :type: EnvironmentMap
    """

    filter_eccentricity: int
    """ Maximum eccentricity (higher gives less blur at distant/oblique angles, but is also slower)

    :type: int
    """

    filter_probes: int
    """ Maximum number of samples (higher gives less blur at distant/oblique angles, but is also slower)

    :type: int
    """

    filter_size: float
    """ Multiply the filter size used by MIP Map and Interpolation

    :type: float
    """

    filter_type: typing.Literal["BOX", "EWA", "FELINE", "AREA"]
    """ Texture filter to use for sampling image

    :type: typing.Literal['BOX','EWA','FELINE','AREA']
    """

    image: Image | None
    """ Source image file to read the environment map from

    :type: Image | None
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    use_filter_size_min: bool
    """ Use Filter Size as a minimal filter value in pixels

    :type: bool
    """

    use_mipmap: bool
    """ Use auto-generated MIP maps for the image

    :type: bool
    """

    use_mipmap_gauss: bool
    """ Use Gauss filter to sample down MIP maps

    :type: bool
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class Event(bpy_struct):
    """Window Manager Event"""

    alt: bool
    """ True when the Alt/Option key is held

    :type: bool
    """

    ascii: str
    """ Single ASCII character for this event

    :type: str
    """

    ctrl: bool
    """ True when the Ctrl key is held

    :type: bool
    """

    is_tablet: bool
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: bool
    """

    mouse_prev_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_prev_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    mouse_region_x: int
    """ The region relative horizontal location of the mouse

    :type: int
    """

    mouse_region_y: int
    """ The region relative vertical location of the mouse

    :type: int
    """

    mouse_x: int
    """ The window relative horizontal location of the mouse

    :type: int
    """

    mouse_y: int
    """ The window relative vertical location of the mouse

    :type: int
    """

    oskey: bool
    """ True when the Cmd key is held

    :type: bool
    """

    pressure: float
    """ The pressure of the tablet or 1.0 if no tablet present

    :type: float
    """

    shift: bool
    """ True when the Shift key is held

    :type: bool
    """

    tilt: bpy_prop_array[float]
    """ The pressure of the tablet or zeroes if no tablet present

    :type: bpy_prop_array[float]
    """

    type: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ 

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    unicode: str
    """ Single unicode character for this event

    :type: str
    """

    value: typing.Literal[
        "ANY",
        "NOTHING",
        "PRESS",
        "RELEASE",
        "CLICK",
        "DOUBLE_CLICK",
        "NORTH",
        "NORTH_EAST",
        "EAST",
        "SOUTH_EAST",
        "SOUTH",
        "SOUTH_WEST",
        "WEST",
        "NORTH_WEST",
    ]
    """ The type of event, only applies to some

    :type: typing.Literal['ANY','NOTHING','PRESS','RELEASE','CLICK','DOUBLE_CLICK','NORTH','NORTH_EAST','EAST','SOUTH_EAST','SOUTH','SOUTH_WEST','WEST','NORTH_WEST']
    """

class ExplodeModifier(Modifier, bpy_struct):
    """Explosion effect modifier based on a particle system"""

    particle_uv: str
    """ UV map to change with particle age

    :type: str
    """

    protect: float
    """ Clean vertex group edges

    :type: float
    """

    show_alive: bool
    """ Show mesh when particles are alive

    :type: bool
    """

    show_dead: bool
    """ Show mesh when particles are dead

    :type: bool
    """

    show_unborn: bool
    """ Show mesh when particles are unborn

    :type: bool
    """

    use_edge_cut: bool
    """ Cut face edges for nicer shrapnel

    :type: bool
    """

    use_size: bool
    """ Use particle size for the shrapnel

    :type: bool
    """

    vertex_group: str
    """ 

    :type: str
    """

class ExpressionController(Controller, bpy_struct):
    """Controller passing on events based on the evaluation of an expression"""

    expression: str
    """ 

    :type: str
    """

class FCurve(bpy_struct):
    """F-Curve defining values of a period of time"""

    array_index: int
    """ Index to the specific property affected by F-Curve if applicable

    :type: int
    """

    color: bpy_prop_array[float]
    """ Color of the F-Curve in the Graph Editor

    :type: bpy_prop_array[float]
    """

    color_mode: typing.Literal["AUTO_RAINBOW", "AUTO_RGB", "AUTO_YRGB", "CUSTOM"]
    """ Method used to determine color of F-Curve in Graph Editor

    :type: typing.Literal['AUTO_RAINBOW','AUTO_RGB','AUTO_YRGB','CUSTOM']
    """

    data_path: str
    """ RNA Path to property affected by F-Curve

    :type: str
    """

    driver: Driver
    """ Channel Driver (only set for Driver F-Curves)

    :type: Driver
    """

    extrapolation: typing.Literal["CONSTANT", "LINEAR"]
    """ Method used for evaluating value of F-Curve outside first and last keyframes

    :type: typing.Literal['CONSTANT','LINEAR']
    """

    group: ActionGroup | None
    """ Action Group that this F-Curve belongs to

    :type: ActionGroup | None
    """

    hide: bool
    """ F-Curve and its keyframes are hidden in the Graph Editor graphs

    :type: bool
    """

    is_valid: bool
    """ False when F-Curve could not be evaluated in past, so should be skipped when evaluating

    :type: bool
    """

    keyframe_points: FCurveKeyframePoints
    """ User-editable keyframes

    :type: FCurveKeyframePoints
    """

    lock: bool
    """ F-Curve's settings cannot be edited

    :type: bool
    """

    modifiers: FCurveModifiers
    """ Modifiers affecting the shape of the F-Curve

    :type: FCurveModifiers
    """

    mute: bool
    """ F-Curve is not evaluated

    :type: bool
    """

    sampled_points: bpy_prop_collection[FCurveSample]
    """ Sampled animation data

    :type: bpy_prop_collection[FCurveSample]
    """

    select: bool
    """ F-Curve is selected for editing

    :type: bool
    """

    def evaluate(self, frame: float | None) -> float:
        """Evaluate F-Curve

        :param frame: Frame, Evaluate F-Curve at given frame
        :type frame: float | None
        :return: Value, Value of F-Curve specific frame
        :rtype: float
        """

    def update(self):
        """Ensure keyframes are sorted in chronological order and handles are set correctly"""

    def range(self) -> bpy_prop_array[float]:
        """Get the time extents for F-Curve

        :return: Range, Min/Max values
        :rtype: bpy_prop_array[float]
        """

    def update_autoflags(self, data: typing.Any):
        """Update FCurve flags set automatically from affected property (currently, integer/discrete flags set when the property is not a float)

        :param data: Data, Data containing the property controlled by given FCurve
        :type data: typing.Any
        """

    def convert_to_samples(self, start: int | None, end: int | None):
        """Convert current FCurve from keyframes to sample points, if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """

    def convert_to_keyframes(self, start: int | None, end: int | None):
        """Convert current FCurve from sample points to keyframes (linear interpolation), if necessary

        :param start: Start Frame
        :type start: int | None
        :param end: End Frame
        :type end: int | None
        """

class FCurveSample(bpy_struct):
    """Sample point for F-Curve"""

    co: bpy_prop_array[float]
    """ Point coordinates

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Selection status

    :type: bool
    """

class FFmpegSettings(bpy_struct):
    """FFmpeg related settings for the scene"""

    audio_bitrate: int
    """ Audio bitrate (kb/s)

    :type: int
    """

    audio_channels: typing.Literal[
        "MONO", "STEREO", "SURROUND4", "SURROUND51", "SURROUND71"
    ]
    """ Audio channel count

    :type: typing.Literal['MONO','STEREO','SURROUND4','SURROUND51','SURROUND71']
    """

    audio_codec: typing.Literal[
        "NONE", "MP2", "MP3", "AC3", "AAC", "VORBIS", "FLAC", "PCM"
    ]
    """ FFmpeg audio codec to use

    :type: typing.Literal['NONE','MP2','MP3','AC3','AAC','VORBIS','FLAC','PCM']
    """

    audio_mixrate: int
    """ Audio samplerate(samples/s)

    :type: int
    """

    audio_volume: float
    """ Audio volume

    :type: float
    """

    buffersize: int
    """ Rate control: buffer size (kb)

    :type: int
    """

    codec: typing.Literal[
        "NONE",
        "MPEG1",
        "MPEG2",
        "MPEG4",
        "HUFFYUV",
        "DV",
        "H264",
        "THEORA",
        "FLASH",
        "FFV1",
        "QTRLE",
        "DNXHD",
        "PNG",
    ]
    """ FFmpeg codec to use

    :type: typing.Literal['NONE','MPEG1','MPEG2','MPEG4','HUFFYUV','DV','H264','THEORA','FLASH','FFV1','QTRLE','DNXHD','PNG']
    """

    format: typing.Literal[
        "MPEG1",
        "MPEG2",
        "MPEG4",
        "AVI",
        "QUICKTIME",
        "DV",
        "H264",
        "XVID",
        "OGG",
        "MKV",
        "FLASH",
    ]
    """ Output file format

    :type: typing.Literal['MPEG1','MPEG2','MPEG4','AVI','QUICKTIME','DV','H264','XVID','OGG','MKV','FLASH']
    """

    gopsize: int
    """ Distance between key frames

    :type: int
    """

    maxrate: int
    """ Rate control: max rate (kb/s)

    :type: int
    """

    minrate: int
    """ Rate control: min rate (kb/s)

    :type: int
    """

    muxrate: int
    """ Mux rate (bits/s(!))

    :type: int
    """

    packetsize: int
    """ Mux packet size (byte)

    :type: int
    """

    use_autosplit: bool
    """ Autosplit output at 2GB boundary

    :type: bool
    """

    use_lossless_output: bool
    """ Use lossless output for video streams

    :type: bool
    """

    video_bitrate: int
    """ Video bitrate (kb/s)

    :type: int
    """

class FILEBROWSER_UL_dir(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class FModifier(bpy_struct):
    """Modifier for values of F-Curve"""

    active: bool | None
    """ F-Curve Modifier is the one being edited

    :type: bool | None
    """

    blend_in: float
    """ Number of frames from start frame for influence to take effect

    :type: float
    """

    blend_out: float
    """ Number of frames from end frame for influence to fade out

    :type: float
    """

    frame_end: float
    """ Frame that modifier's influence ends (if Restrict Frame Range is in use)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if Restrict Frame Range is in use)

    :type: float
    """

    influence: float
    """ Amount of influence F-Curve Modifier will have when not fading in/out

    :type: float
    """

    is_valid: bool
    """ F-Curve Modifier has invalid settings and will not be evaluated

    :type: bool
    """

    mute: bool
    """ F-Curve Modifier will not be evaluated

    :type: bool
    """

    show_expanded: bool
    """ F-Curve Modifier's panel is expanded in UI

    :type: bool
    """

    type: typing.Literal[
        "NULL",
        "GENERATOR",
        "FNGENERATOR",
        "ENVELOPE",
        "CYCLES",
        "NOISE",
        "LIMITS",
        "STEPPED",
    ]
    """ F-Curve Modifier Type

    :type: typing.Literal['NULL','GENERATOR','FNGENERATOR','ENVELOPE','CYCLES','NOISE','LIMITS','STEPPED']
    """

    use_influence: bool
    """ F-Curve Modifier's effects will be tempered by a default factor

    :type: bool
    """

    use_restricted_range: bool
    """ F-Curve Modifier is only applied for the specified frame range to help mask off effects in order to chain them

    :type: bool
    """

class FModifierCycles(FModifier, bpy_struct):
    """Repeat the values of the modified F-Curve"""

    cycles_after: int
    """ Maximum number of cycles to allow after last keyframe (0 = infinite)

    :type: int
    """

    cycles_before: int
    """ Maximum number of cycles to allow before first keyframe (0 = infinite)

    :type: int
    """

    mode_after: typing.Literal["NONE", "REPEAT", "REPEAT_OFFSET", "MIRROR"]
    """ Cycling mode to use after last keyframe

    :type: typing.Literal['NONE','REPEAT','REPEAT_OFFSET','MIRROR']
    """

    mode_before: typing.Literal["NONE", "REPEAT", "REPEAT_OFFSET", "MIRROR"]
    """ Cycling mode to use before first keyframe

    :type: typing.Literal['NONE','REPEAT','REPEAT_OFFSET','MIRROR']
    """

class FModifierEnvelope(FModifier, bpy_struct):
    """Scale the values of the modified F-Curve"""

    control_points: FModifierEnvelopeControlPoints
    """ Control points defining the shape of the envelope

    :type: FModifierEnvelopeControlPoints
    """

    default_max: float
    """ Upper distance from Reference Value for 1:1 default influence

    :type: float
    """

    default_min: float
    """ Lower distance from Reference Value for 1:1 default influence

    :type: float
    """

    reference_value: float
    """ Value that envelope's influence is centered around / based on

    :type: float
    """

class FModifierEnvelopeControlPoint(bpy_struct):
    """Control point for envelope F-Modifier"""

    frame: float
    """ Frame this control-point occurs on

    :type: float
    """

    max: float
    """ Upper bound of envelope at this control-point

    :type: float
    """

    min: float
    """ Lower bound of envelope at this control-point

    :type: float
    """

class FModifierFunctionGenerator(FModifier, bpy_struct):
    """Generate values using a Built-In Function"""

    amplitude: float
    """ Scale factor determining the maximum/minimum values

    :type: float
    """

    function_type: typing.Literal["SIN", "COS", "TAN", "SQRT", "LN", "SINC"]
    """ Type of built-in function to use

    :type: typing.Literal['SIN','COS','TAN','SQRT','LN','SINC']
    """

    phase_multiplier: float
    """ Scale factor determining the 'speed' of the function

    :type: float
    """

    phase_offset: float
    """ Constant factor to offset time by for function

    :type: float
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

    value_offset: float
    """ Constant factor to offset values by

    :type: float
    """

class FModifierGenerator(FModifier, bpy_struct):
    """Deterministically generate values for the modified F-Curve"""

    coefficients: bpy_prop_array[float]
    """ Coefficients for 'x' (starting from lowest power of x^0)

    :type: bpy_prop_array[float]
    """

    mode: typing.Literal["POLYNOMIAL", "POLYNOMIAL_FACTORISED"]
    """ Type of generator to use

    :type: typing.Literal['POLYNOMIAL','POLYNOMIAL_FACTORISED']
    """

    poly_order: int
    """ The highest power of 'x' for this polynomial (number of coefficients - 1)

    :type: int
    """

    use_additive: bool
    """ Values generated by this modifier are applied on top of the existing values instead of overwriting them

    :type: bool
    """

class FModifierLimits(FModifier, bpy_struct):
    """Limit the time/value ranges of the modified F-Curve"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

class FModifierNoise(FModifier, bpy_struct):
    """Give randomness to the modified F-Curve"""

    blend_type: typing.Literal["REPLACE", "ADD", "SUBTRACT", "MULTIPLY"]
    """ Method of modifying the existing F-Curve

    :type: typing.Literal['REPLACE','ADD','SUBTRACT','MULTIPLY']
    """

    depth: int
    """ Amount of fine level detail present in the noise

    :type: int
    """

    offset: float
    """ Time offset for the noise effect

    :type: float
    """

    phase: float
    """ A random seed for the noise effect

    :type: float
    """

    scale: float
    """ Scaling (in time) of the noise

    :type: float
    """

    strength: float
    """ Amplitude of the noise - the amount that it modifies the underlying curve

    :type: float
    """

class FModifierPython(FModifier, bpy_struct):
    """Perform user-defined operation on the modified F-Curve"""

class FModifierStepped(FModifier, bpy_struct):
    """Hold each interpolated value from the F-Curve for several frames without changing the timing"""

    frame_end: float
    """ Frame that modifier's influence ends (if applicable)

    :type: float
    """

    frame_offset: float
    """ Reference number of frames before frames get held (use to get hold for '1-3' vs '5-7' holding patterns)

    :type: float
    """

    frame_start: float
    """ Frame that modifier's influence starts (if applicable)

    :type: float
    """

    frame_step: float
    """ Number of frames to hold each value

    :type: float
    """

    use_frame_end: bool
    """ Restrict modifier to only act before its 'end' frame

    :type: bool
    """

    use_frame_start: bool
    """ Restrict modifier to only act after its 'start' frame

    :type: bool
    """

class FieldSettings(bpy_struct):
    """Field settings for an object in physics simulation"""

    apply_to_location: bool
    """ Effect particles' location

    :type: bool
    """

    apply_to_rotation: bool
    """ Effect particles' dynamic rotation

    :type: bool
    """

    distance_max: float
    """ Maximum distance for the field to work

    :type: float
    """

    distance_min: float
    """ Minimum distance for the field's fall-off

    :type: float
    """

    falloff_power: float
    """ Falloff power (real gravitational falloff = 2)

    :type: float
    """

    falloff_type: typing.Literal["SPHERE", "TUBE", "CONE"]
    """ 

    :type: typing.Literal['SPHERE','TUBE','CONE']
    """

    flow: float
    """ Convert effector force into air flow velocity

    :type: float
    """

    guide_clump_amount: float
    """ Amount of clumping

    :type: float
    """

    guide_clump_shape: float
    """ Shape of clumping

    :type: float
    """

    guide_free: float
    """ Guide-free time from particle life's end

    :type: float
    """

    guide_kink_amplitude: float
    """ The amplitude of the offset

    :type: float
    """

    guide_kink_axis: typing.Literal["X", "Y", "Z"]
    """ Which axis to use for offset

    :type: typing.Literal['X','Y','Z']
    """

    guide_kink_frequency: float
    """ The frequency of the offset (1/total length)

    :type: float
    """

    guide_kink_shape: float
    """ Adjust the offset to the beginning/end

    :type: float
    """

    guide_kink_type: typing.Literal[
        "NONE", "CURL", "RADIAL", "WAVE", "BRAID", "ROTATION", "ROLL"
    ]
    """ Type of periodic offset on the curve

    :type: typing.Literal['NONE','CURL','RADIAL','WAVE','BRAID','ROTATION','ROLL']
    """

    guide_minimum: float
    """ The distance from which particles are affected fully

    :type: float
    """

    harmonic_damping: float
    """ Damping of the harmonic force

    :type: float
    """

    inflow: float
    """ Inwards component of the vortex force

    :type: float
    """

    linear_drag: float
    """ Drag component proportional to velocity

    :type: float
    """

    noise: float
    """ Amount of noise for the force strength

    :type: float
    """

    quadratic_drag: float
    """ Drag component proportional to the square of velocity

    :type: float
    """

    radial_falloff: float
    """ Radial falloff power (real gravitational falloff = 2)

    :type: float
    """

    radial_max: float
    """ Maximum radial distance for the field to work

    :type: float
    """

    radial_min: float
    """ Minimum radial distance for the field's fall-off

    :type: float
    """

    rest_length: float
    """ Rest length of the harmonic force

    :type: float
    """

    seed: int
    """ Seed of the noise

    :type: int
    """

    shape: typing.Literal["POINT", "PLANE", "SURFACE", "POINTS"]
    """ Which direction is used to calculate the effector force

    :type: typing.Literal['POINT','PLANE','SURFACE','POINTS']
    """

    size: float
    """ Size of the turbulence

    :type: float
    """

    source_object: Object | None
    """ Select domain object of the smoke simulation

    :type: Object | None
    """

    strength: float
    """ Strength of force field

    :type: float
    """

    texture: Texture | None
    """ Texture to use as force

    :type: Texture | None
    """

    texture_mode: typing.Literal["RGB", "GRADIENT", "CURL"]
    """ How the texture effect is calculated (RGB & Curl need a RGB texture, else Gradient will be used instead)

    :type: typing.Literal['RGB','GRADIENT','CURL']
    """

    texture_nabla: float
    """ Defines size of derivative offset used for calculating gradient and curl

    :type: float
    """

    type: typing.Literal[
        "NONE",
        "FORCE",
        "WIND",
        "VORTEX",
        "MAGNET",
        "HARMONIC",
        "CHARGE",
        "LENNARDJ",
        "TEXTURE",
        "GUIDE",
        "BOID",
        "TURBULENCE",
        "DRAG",
        "SMOKE_FLOW",
    ]
    """ Type of field

    :type: typing.Literal['NONE','FORCE','WIND','VORTEX','MAGNET','HARMONIC','CHARGE','LENNARDJ','TEXTURE','GUIDE','BOID','TURBULENCE','DRAG','SMOKE_FLOW']
    """

    use_2d_force: bool
    """ Apply force only in 2D

    :type: bool
    """

    use_absorption: bool
    """ Force gets absorbed by collision objects

    :type: bool
    """

    use_global_coords: bool
    """ Use effector/global coordinates for turbulence

    :type: bool
    """

    use_guide_path_add: bool
    """ Based on distance/falloff it adds a portion of the entire path

    :type: bool
    """

    use_guide_path_weight: bool
    """ Use curve weights to influence the particle influence along the curve

    :type: bool
    """

    use_max_distance: bool
    """ Use a maximum distance for the field to work

    :type: bool
    """

    use_min_distance: bool
    """ Use a minimum distance for the field's fall-off

    :type: bool
    """

    use_multiple_springs: bool
    """ Every point is effected by multiple springs

    :type: bool
    """

    use_object_coords: bool
    """ Use object/global coordinates for texture

    :type: bool
    """

    use_radial_max: bool
    """ Use a maximum radial distance for the field to work

    :type: bool
    """

    use_radial_min: bool
    """ Use a minimum radial distance for the field's fall-off

    :type: bool
    """

    use_root_coords: bool
    """ Texture coordinates from root particle locations

    :type: bool
    """

    use_smoke_density: bool
    """ Adjust force strength based on smoke density

    :type: bool
    """

    z_direction: typing.Literal["BOTH", "POSITIVE", "NEGATIVE"]
    """ Effect in full or only positive/negative Z direction

    :type: typing.Literal['BOTH','POSITIVE','NEGATIVE']
    """

class FileBrowserFSMenuEntry(bpy_struct):
    """File Select Parameters"""

    is_valid: bool
    """ Whether this path is currently reachable

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    path: str
    """ 

    :type: str
    """

    use_save: bool
    """ Whether this path is saved in bookmarks, or generated from OS

    :type: bool
    """

class FileSelectParams(bpy_struct):
    """File Select Parameters"""

    directory: str
    """ Directory displayed in the file browser

    :type: str
    """

    display_size: typing.Literal["TINY", "SMALL", "NORMAL", "LARGE"]
    """ Change the size of the display (width of columns or thumbnails size)

    :type: typing.Literal['TINY','SMALL','NORMAL','LARGE']
    """

    display_type: typing.Literal["LIST_SHORT", "LIST_LONG", "THUMBNAIL"]
    """ Display mode for the file list

    :type: typing.Literal['LIST_SHORT','LIST_LONG','THUMBNAIL']
    """

    filename: str
    """ Active file in the file browser

    :type: str
    """

    filter_glob: str
    """ 

    :type: str
    """

    filter_id: set[
        typing.Literal[
            "ACTION",
            "ARMATURE",
            "BRUSH",
            "CAMERA",
            "CACHEFILE",
            "CURVE",
            "GREASE_PENCIL",
            "GROUP",
            "IMAGE",
            "LAMP",
            "LINESTYLE",
            "LATTICE",
            "MATERIAL",
            "METABALL",
            "MOVIE_CLIP",
            "MESH",
            "MASK",
            "NODE_TREE",
            "OBJECT",
            "PARTICLE_SETTINGS",
            "PALETTE",
            "PAINT_CURVE",
            "SCENE",
            "SPEAKER",
            "SOUND",
            "TEXTURE",
            "TEXT",
            "FONT",
            "WORLD",
        ]
    ]
    """ Which ID types to show/hide, when browsing a library

    :type: set[typing.Literal['ACTION','ARMATURE','BRUSH','CAMERA','CACHEFILE','CURVE','GREASE_PENCIL','GROUP','IMAGE','LAMP','LINESTYLE','LATTICE','MATERIAL','METABALL','MOVIE_CLIP','MESH','MASK','NODE_TREE','OBJECT','PARTICLE_SETTINGS','PALETTE','PAINT_CURVE','SCENE','SPEAKER','SOUND','TEXTURE','TEXT','FONT','WORLD']]
    """

    filter_id_category: set[
        typing.Literal[
            "SCENE",
            "ANIMATION",
            "OBJECT",
            "GEOMETRY",
            "SHADING",
            "IMAGE",
            "ENVIRONMENT",
            "MISC",
        ]
    ]
    """ Which ID categories to show/hide, when browsing a library

    :type: set[typing.Literal['SCENE','ANIMATION','OBJECT','GEOMETRY','SHADING','IMAGE','ENVIRONMENT','MISC']]
    """

    filter_search: str
    """ Filter by name, supports '*' wildcard

    :type: str
    """

    recursion_level: typing.Literal["NONE", "BLEND", "ALL_1", "ALL_2", "ALL_3"]
    """ Numbers of dirtree levels to show simultaneously

    :type: typing.Literal['NONE','BLEND','ALL_1','ALL_2','ALL_3']
    """

    show_hidden: bool
    """ Show hidden dot files

    :type: bool
    """

    sort_method: typing.Literal[
        "FILE_SORT_ALPHA", "FILE_SORT_EXTENSION", "FILE_SORT_TIME", "FILE_SORT_SIZE"
    ]
    """ 

    :type: typing.Literal['FILE_SORT_ALPHA','FILE_SORT_EXTENSION','FILE_SORT_TIME','FILE_SORT_SIZE']
    """

    title: str
    """ Title for the file browser

    :type: str
    """

    use_filter: bool
    """ Enable filtering of files

    :type: bool
    """

    use_filter_backup: bool
    """ Show .blend1, .blend2, etc. files

    :type: bool
    """

    use_filter_blender: bool
    """ Show .blend files

    :type: bool
    """

    use_filter_blendid: bool
    """ Show .blend files items (objects, materials, etc.)

    :type: bool
    """

    use_filter_folder: bool
    """ Show folders

    :type: bool
    """

    use_filter_font: bool
    """ Show font files

    :type: bool
    """

    use_filter_image: bool
    """ Show image files

    :type: bool
    """

    use_filter_movie: bool
    """ Show movie files

    :type: bool
    """

    use_filter_script: bool
    """ Show script files

    :type: bool
    """

    use_filter_sound: bool
    """ Show sound files

    :type: bool
    """

    use_filter_text: bool
    """ Show text files

    :type: bool
    """

    use_library_browsing: bool
    """ Whether we may browse blender files' content or not

    :type: bool
    """

class Filter2DActuator(Actuator, bpy_struct):
    """Actuator to apply screen graphic effects"""

    filter_pass: int
    """ Set filter order

    :type: int
    """

    glsl_shader: Text | None
    """ 

    :type: Text | None
    """

    mode: typing.Literal[
        "ENABLE",
        "DISABLE",
        "REMOVE",
        "MOTIONBLUR",
        "BLUR",
        "SHARPEN",
        "DILATION",
        "EROSION",
        "LAPLACIAN",
        "SOBEL",
        "PREWITT",
        "GRAYSCALE",
        "SEPIA",
        "INVERT",
        "CUSTOMFILTER",
    ]
    """ 

    :type: typing.Literal['ENABLE','DISABLE','REMOVE','MOTIONBLUR','BLUR','SHARPEN','DILATION','EROSION','LAPLACIAN','SOBEL','PREWITT','GRAYSCALE','SEPIA','INVERT','CUSTOMFILTER']
    """

    motion_blur_factor: float
    """ Motion blur factor

    :type: float
    """

    use_motion_blur: bool
    """ Enable/Disable Motion Blur

    :type: bool
    """

class FloatProperty(Property, bpy_struct):
    """RNA floating pointer number property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: float
    """ Default value for this number

    :type: float
    """

    default_array: bpy_prop_array[float]
    """ Default value for this array

    :type: bpy_prop_array[float]
    """

    hard_max: float
    """ Maximum value used by buttons

    :type: float
    """

    hard_min: float
    """ Minimum value used by buttons

    :type: float
    """

    is_array: bool
    """ 

    :type: bool
    """

    precision: int
    """ Number of digits after the dot used by buttons

    :type: int
    """

    soft_max: float
    """ Maximum value used by buttons

    :type: float
    """

    soft_min: float
    """ Minimum value used by buttons

    :type: float
    """

    step: float
    """ Step size used by number buttons, for floats 1/100th of the step size

    :type: float
    """

class FloorConstraint(Constraint, bpy_struct):
    """Use the target object for location limitation"""

    floor_location: typing.Literal[
        "FLOOR_X",
        "FLOOR_Y",
        "FLOOR_Z",
        "FLOOR_NEGATIVE_X",
        "FLOOR_NEGATIVE_Y",
        "FLOOR_NEGATIVE_Z",
    ]
    """ Location of target that object will not pass through

    :type: typing.Literal['FLOOR_X','FLOOR_Y','FLOOR_Z','FLOOR_NEGATIVE_X','FLOOR_NEGATIVE_Y','FLOOR_NEGATIVE_Z']
    """

    offset: float
    """ Offset of floor from object origin

    :type: float
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_rotation: bool
    """ Use the target's rotation to determine floor

    :type: bool
    """

    use_sticky: bool
    """ Immobilize object while constrained

    :type: bool
    """

class FluidFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for the fluid in the simulation"""

    initial_velocity: bpy_prop_array[float]
    """ Initial velocity of fluid

    :type: bpy_prop_array[float]
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    volume_initialization: typing.Literal["VOLUME", "SHELL", "BOTH"]
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: typing.Literal['VOLUME','SHELL','BOTH']
    """

class FluidSettings(bpy_struct):
    """Fluid simulation settings for an object taking part in the simulation"""

    type: typing.Literal[
        "NONE",
        "DOMAIN",
        "FLUID",
        "OBSTACLE",
        "INFLOW",
        "OUTFLOW",
        "PARTICLE",
        "CONTROL",
    ]
    """ Type of participation in the fluid simulation

    :type: typing.Literal['NONE','DOMAIN','FLUID','OBSTACLE','INFLOW','OUTFLOW','PARTICLE','CONTROL']
    """

class FluidSimulationModifier(Modifier, bpy_struct):
    """Fluid simulation modifier"""

    settings: FluidSettings
    """ Settings for how this object is used in the fluid simulation

    :type: FluidSettings
    """

class FluidVertexVelocity(bpy_struct):
    """Velocity of a simulated fluid mesh"""

    velocity: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class FollowPathConstraint(Constraint, bpy_struct):
    """Lock motion to the target path"""

    forward_axis: typing.Literal[
        "FORWARD_X",
        "FORWARD_Y",
        "FORWARD_Z",
        "TRACK_NEGATIVE_X",
        "TRACK_NEGATIVE_Y",
        "TRACK_NEGATIVE_Z",
    ]
    """ Axis that points forward along the path

    :type: typing.Literal['FORWARD_X','FORWARD_Y','FORWARD_Z','TRACK_NEGATIVE_X','TRACK_NEGATIVE_Y','TRACK_NEGATIVE_Z']
    """

    offset: float
    """ Offset from the position corresponding to the time frame

    :type: float
    """

    offset_factor: float
    """ Percentage value defining target position along length of curve

    :type: float
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    up_axis: typing.Literal["UP_X", "UP_Y", "UP_Z"]
    """ Axis that points upward

    :type: typing.Literal['UP_X','UP_Y','UP_Z']
    """

    use_curve_follow: bool
    """ Object will follow the heading and banking of the curve

    :type: bool
    """

    use_curve_radius: bool
    """ Object is scaled by the curve radius

    :type: bool
    """

    use_fixed_location: bool
    """ Object will stay locked to a single point somewhere along the length of the curve regardless of time

    :type: bool
    """

class FollowTrackConstraint(Constraint, bpy_struct):
    """Lock motion to the target motion track"""

    camera: Object | None
    """ Camera to which motion is parented (if empty active scene camera is used)

    :type: Object | None
    """

    clip: MovieClip | None
    """ Movie Clip to get tracking data from

    :type: MovieClip | None
    """

    depth_object: Object | None
    """ Object used to define depth in camera space by projecting onto surface of this object

    :type: Object | None
    """

    frame_method: typing.Literal["STRETCH", "FIT", "CROP"]
    """ How the footage fits in the camera frame

    :type: typing.Literal['STRETCH','FIT','CROP']
    """

    object: str
    """ Movie tracking object to follow (if empty, camera object is used)

    :type: str
    """

    track: str
    """ Movie tracking track to follow

    :type: str
    """

    use_3d_position: bool
    """ Use 3D position of track to parent to

    :type: bool
    """

    use_active_clip: bool
    """ Use active clip defined in scene

    :type: bool
    """

    use_undistorted_position: bool
    """ Parent to undistorted position of 2D track

    :type: bool
    """

class FreestyleLineSet(bpy_struct):
    """Line set for associating lines and style parameters"""

    edge_type_combination: typing.Literal["OR", "AND"]
    """ Specify a logical combination of selection conditions on feature edge types

    :type: typing.Literal['OR','AND']
    """

    edge_type_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges selected by edge types

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    exclude_border: bool
    """ Exclude border edges

    :type: bool
    """

    exclude_contour: bool
    """ Exclude contours

    :type: bool
    """

    exclude_crease: bool
    """ Exclude crease edges

    :type: bool
    """

    exclude_edge_mark: bool
    """ Exclude edge marks

    :type: bool
    """

    exclude_external_contour: bool
    """ Exclude external contours

    :type: bool
    """

    exclude_material_boundary: bool
    """ Exclude edges at material boundaries

    :type: bool
    """

    exclude_ridge_valley: bool
    """ Exclude ridges and valleys

    :type: bool
    """

    exclude_silhouette: bool
    """ Exclude silhouette edges

    :type: bool
    """

    exclude_suggestive_contour: bool
    """ Exclude suggestive contours

    :type: bool
    """

    face_mark_condition: typing.Literal["ONE", "BOTH"]
    """ Specify a feature edge selection condition based on face marks

    :type: typing.Literal['ONE','BOTH']
    """

    face_mark_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges selected by face marks

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    group: Group | None
    """ A group of objects based on which feature edges are selected

    :type: Group | None
    """

    group_negation: typing.Literal["INCLUSIVE", "EXCLUSIVE"]
    """ Specify either inclusion or exclusion of feature edges belonging to a group of objects

    :type: typing.Literal['INCLUSIVE','EXCLUSIVE']
    """

    linestyle: FreestyleLineStyle
    """ Line style settings

    :type: FreestyleLineStyle
    """

    name: str
    """ Line set name

    :type: str
    """

    qi_end: int
    """ Last QI value of the QI range

    :type: int
    """

    qi_start: int
    """ First QI value of the QI range

    :type: int
    """

    select_border: bool
    """ Select border edges (open mesh edges)

    :type: bool
    """

    select_by_edge_types: bool
    """ Select feature edges based on edge types

    :type: bool
    """

    select_by_face_marks: bool
    """ Select feature edges by face marks

    :type: bool
    """

    select_by_group: bool
    """ Select feature edges based on a group of objects

    :type: bool
    """

    select_by_image_border: bool
    """ Select feature edges by image border (less memory consumption)

    :type: bool
    """

    select_by_visibility: bool
    """ Select feature edges based on visibility

    :type: bool
    """

    select_contour: bool
    """ Select contours (outer silhouettes of each object)

    :type: bool
    """

    select_crease: bool
    """ Select crease edges (those between two faces making an angle smaller than the Crease Angle)

    :type: bool
    """

    select_edge_mark: bool
    """ Select edge marks (edges annotated by Freestyle edge marks)

    :type: bool
    """

    select_external_contour: bool
    """ Select external contours (outer silhouettes of occluding and occluded objects)

    :type: bool
    """

    select_material_boundary: bool
    """ Select edges at material boundaries

    :type: bool
    """

    select_ridge_valley: bool
    """ Select ridges and valleys (boundary lines between convex and concave areas of surface)

    :type: bool
    """

    select_silhouette: bool
    """ Select silhouettes (edges at the boundary of visible and hidden faces)

    :type: bool
    """

    select_suggestive_contour: bool
    """ Select suggestive contours (almost silhouette/contour edges)

    :type: bool
    """

    show_render: bool
    """ Enable or disable this line set during stroke rendering

    :type: bool
    """

    visibility: typing.Literal["VISIBLE", "HIDDEN", "RANGE"]
    """ Determine how to use visibility for feature edge selection

    :type: typing.Literal['VISIBLE','HIDDEN','RANGE']
    """

class FreestyleLineStyle(ID, bpy_struct):
    """Freestyle line style, reusable by multiple line sets"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    alpha: float
    """ Base alpha transparency, possibly modified by alpha transparency modifiers

    :type: float
    """

    alpha_modifiers: LineStyleAlphaModifiers
    """ List of alpha transparency modifiers

    :type: LineStyleAlphaModifiers
    """

    angle_max: float
    """ Maximum 2D angle for splitting chains

    :type: float
    """

    angle_min: float
    """ Minimum 2D angle for splitting chains

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    caps: typing.Literal["BUTT", "ROUND", "SQUARE"]
    """ Select the shape of both ends of strokes

    :type: typing.Literal['BUTT','ROUND','SQUARE']
    """

    chain_count: int
    """ Chain count for the selection of first N chains

    :type: int
    """

    chaining: typing.Literal["PLAIN", "SKETCHY"]
    """ Select the way how feature edges are jointed to form chains

    :type: typing.Literal['PLAIN','SKETCHY']
    """

    color: bpy_prop_array[float]
    """ Base line color, possibly modified by line color modifiers

    :type: bpy_prop_array[float]
    """

    color_modifiers: LineStyleColorModifiers
    """ List of line color modifiers

    :type: LineStyleColorModifiers
    """

    dash1: int
    """ Length of the 1st dash for dashed lines

    :type: int
    """

    dash2: int
    """ Length of the 2nd dash for dashed lines

    :type: int
    """

    dash3: int
    """ Length of the 3rd dash for dashed lines

    :type: int
    """

    gap1: int
    """ Length of the 1st gap for dashed lines

    :type: int
    """

    gap2: int
    """ Length of the 2nd gap for dashed lines

    :type: int
    """

    gap3: int
    """ Length of the 3rd gap for dashed lines

    :type: int
    """

    geometry_modifiers: LineStyleGeometryModifiers
    """ List of stroke geometry modifiers

    :type: LineStyleGeometryModifiers
    """

    integration_type: typing.Literal["MEAN", "MIN", "MAX", "FIRST", "LAST"]
    """ Select the way how the sort key is computed for each chain

    :type: typing.Literal['MEAN','MIN','MAX','FIRST','LAST']
    """

    length_max: float
    """ Maximum curvilinear 2D length for the selection of chains

    :type: float
    """

    length_min: float
    """ Minimum curvilinear 2D length for the selection of chains

    :type: float
    """

    material_boundary: bool
    """ If true, chains of feature edges are split at material boundaries

    :type: bool
    """

    node_tree: ShaderNodeTree | None
    """ Node tree for node-based shaders

    :type: ShaderNodeTree | None
    """

    panel: typing.Literal[
        "STROKES", "COLOR", "ALPHA", "THICKNESS", "GEOMETRY", "TEXTURE"
    ]
    """ Select the property panel to be shown

    :type: typing.Literal['STROKES','COLOR','ALPHA','THICKNESS','GEOMETRY','TEXTURE']
    """

    rounds: int
    """ Number of rounds in a sketchy multiple touch

    :type: int
    """

    sort_key: typing.Literal[
        "DISTANCE_FROM_CAMERA", "2D_LENGTH", "PROJECTED_X", "PROJECTED_Y"
    ]
    """ Select the sort key to determine the stacking order of chains

    :type: typing.Literal['DISTANCE_FROM_CAMERA','2D_LENGTH','PROJECTED_X','PROJECTED_Y']
    """

    sort_order: typing.Literal["DEFAULT", "REVERSE"]
    """ Select the sort order

    :type: typing.Literal['DEFAULT','REVERSE']
    """

    split_dash1: int
    """ Length of the 1st dash for splitting

    :type: int
    """

    split_dash2: int
    """ Length of the 2nd dash for splitting

    :type: int
    """

    split_dash3: int
    """ Length of the 3rd dash for splitting

    :type: int
    """

    split_gap1: int
    """ Length of the 1st gap for splitting

    :type: int
    """

    split_gap2: int
    """ Length of the 2nd gap for splitting

    :type: int
    """

    split_gap3: int
    """ Length of the 3rd gap for splitting

    :type: int
    """

    split_length: float
    """ Curvilinear 2D length for chain splitting

    :type: float
    """

    texture_slots: LineStyleTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: LineStyleTextureSlots
    """

    texture_spacing: float
    """ Spacing for textures along stroke length

    :type: float
    """

    thickness: float
    """ Base line thickness, possibly modified by line thickness modifiers

    :type: float
    """

    thickness_modifiers: LineStyleThicknessModifiers
    """ List of line thickness modifiers

    :type: LineStyleThicknessModifiers
    """

    thickness_position: typing.Literal["CENTER", "INSIDE", "OUTSIDE", "RELATIVE"]
    """ Thickness position of silhouettes and border edges (applicable when plain chaining is used with the Same Object option)

    :type: typing.Literal['CENTER','INSIDE','OUTSIDE','RELATIVE']
    """

    thickness_ratio: float
    """ A number between 0 (inside) and 1 (outside) specifying the relative position of stroke thickness

    :type: float
    """

    use_angle_max: bool
    """ Split chains at points with angles larger than the maximum 2D angle

    :type: bool
    """

    use_angle_min: bool
    """ Split chains at points with angles smaller than the minimum 2D angle

    :type: bool
    """

    use_chain_count: bool
    """ Enable the selection of first N chains

    :type: bool
    """

    use_chaining: bool
    """ Enable chaining of feature edges

    :type: bool
    """

    use_dashed_line: bool
    """ Enable or disable dashed line

    :type: bool
    """

    use_length_max: bool
    """ Enable the selection of chains by a maximum 2D length

    :type: bool
    """

    use_length_min: bool
    """ Enable the selection of chains by a minimum 2D length

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes for the line style

    :type: bool
    """

    use_same_object: bool
    """ If true, only feature edges of the same object are joined

    :type: bool
    """

    use_sorting: bool
    """ Arrange the stacking order of strokes

    :type: bool
    """

    use_split_length: bool
    """ Enable chain splitting by curvilinear 2D length

    :type: bool
    """

    use_split_pattern: bool
    """ Enable chain splitting by dashed line patterns

    :type: bool
    """

    use_texture: bool
    """ Enable or disable textured strokes

    :type: bool
    """

class FreestyleModuleSettings(bpy_struct):
    """Style module configuration for specifying a style module"""

    script: Text | None
    """ Python script to define a style module

    :type: Text | None
    """

    use: bool
    """ Enable or disable this style module during stroke rendering

    :type: bool
    """

class FreestyleSettings(bpy_struct):
    """Freestyle settings for a SceneRenderLayer data-block"""

    crease_angle: float
    """ Angular threshold for detecting crease edges

    :type: float
    """

    kr_derivative_epsilon: float
    """ Kr derivative epsilon for computing suggestive contours

    :type: float
    """

    linesets: Linesets
    """ 

    :type: Linesets
    """

    mode: typing.Literal["SCRIPT", "EDITOR"]
    """ Select the Freestyle control mode

    :type: typing.Literal['SCRIPT','EDITOR']
    """

    modules: FreestyleModules
    """ A list of style modules (to be applied from top to bottom)

    :type: FreestyleModules
    """

    sphere_radius: float
    """ Sphere radius for computing curvatures

    :type: float
    """

    use_advanced_options: bool
    """ Enable advanced edge detection options (sphere radius and Kr derivative epsilon)

    :type: bool
    """

    use_culling: bool
    """ If enabled, out-of-view edges are ignored

    :type: bool
    """

    use_material_boundaries: bool
    """ Enable material boundaries

    :type: bool
    """

    use_ridges_and_valleys: bool
    """ Enable ridges and valleys

    :type: bool
    """

    use_smoothness: bool
    """ Take face smoothness into account in view map calculation

    :type: bool
    """

    use_suggestive_contours: bool
    """ Enable suggestive contours

    :type: bool
    """

    use_view_map_cache: bool
    """ Keep the computed view map and avoid re-calculating it if mesh geometry is unchanged

    :type: bool
    """

class Function(bpy_struct):
    """RNA function definition"""

    description: str
    """ Description of the Function's purpose

    :type: str
    """

    identifier: str
    """ Unique name used in the code and scripting

    :type: str
    """

    is_registered: bool
    """ Function is registered as callback as part of type registration

    :type: bool
    """

    is_registered_optional: bool
    """ Function is optionally registered as callback part of type registration

    :type: bool
    """

    parameters: bpy_prop_collection[Property]
    """ Parameters for the function

    :type: bpy_prop_collection[Property]
    """

    use_self: bool
    """ Function does not pass its self as an argument (becomes a static method in python)

    :type: bool
    """

    use_self_type: bool
    """ Function passes its self type as an argument (becomes a class method in python if use_self is false)

    :type: bool
    """

class GPENCIL_PIE_sculpt(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """

class GPENCIL_PIE_settings_palette(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """

class GPENCIL_PIE_tool_palette(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """

class GPENCIL_PIE_tools_more(Menu, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """

class GPENCIL_UL_brush(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class GPENCIL_UL_layer(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class GPENCIL_UL_palettecolor(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class GPUDOFSettings(bpy_struct):
    """Settings for GPU based depth of field"""

    blades: int
    """ Blades for dof effect

    :type: int
    """

    focal_length: float
    """ Focal length for dof effect

    :type: float
    """

    focus_distance: float
    """ Viewport depth of field focus distance

    :type: float
    """

    fstop: float
    """ F-stop for dof effect

    :type: float
    """

    is_hq_supported: bool
    """ Use high quality depth of field

    :type: bool
    """

    sensor: float
    """ Size of sensor

    :type: float
    """

    use_high_quality: bool
    """ Use high quality depth of field

    :type: bool
    """

class GPUFXSettings(bpy_struct):
    """Settings for GPU based compositing"""

    dof: GPUDOFSettings
    """ 

    :type: GPUDOFSettings
    """

    ssao: GPUSSAOSettings
    """ 

    :type: GPUSSAOSettings
    """

    use_dof: bool
    """ Use depth of field on viewport using the values from active camera

    :type: bool
    """

    use_ssao: bool
    """ Use screen space ambient occlusion of field on viewport

    :type: bool
    """

class GPUSSAOSettings(bpy_struct):
    """Settings for GPU based screen space ambient occlusion"""

    attenuation: float
    """ Attenuation constant

    :type: float
    """

    color: bpy_prop_array[float]
    """ Color for screen space ambient occlusion effect

    :type: bpy_prop_array[float]
    """

    distance_max: float
    """ Distance of object that contribute to the SSAO effect

    :type: float
    """

    factor: float
    """ Strength of the SSAO effect

    :type: float
    """

    samples: int
    """ Number of samples

    :type: int
    """

class GPencilBrush(bpy_struct):
    """Collection of brushes being used to control the line style of new strokes"""

    angle: float
    """ Direction of the stroke at which brush gives maximal thickness (0° for horizontal)

    :type: float
    """

    angle_factor: float
    """ Reduce brush thickness by this factor when stroke is perpendicular to 'Angle' direction

    :type: float
    """

    curve_jitter: CurveMapping
    """ Curve used for the jitter effect

    :type: CurveMapping
    """

    curve_sensitivity: CurveMapping
    """ Curve used for the sensitivity

    :type: CurveMapping
    """

    curve_strength: CurveMapping
    """ Curve used for the strength

    :type: CurveMapping
    """

    jitter: float
    """ Jitter factor for new strokes

    :type: float
    """

    line_width: int
    """ Thickness of strokes (in pixels)

    :type: int
    """

    name: str
    """ Brush name

    :type: str
    """

    pen_sensitivity_factor: float
    """ Pressure sensitivity factor for new strokes

    :type: float
    """

    pen_smooth_factor: float
    """ Amount of smoothing to apply to newly created strokes, to reduce jitter/noise

    :type: float
    """

    pen_smooth_steps: int
    """ Number of times to smooth newly created strokes

    :type: int
    """

    pen_subdivision_steps: int
    """ Number of times to subdivide newly created strokes, for less jagged strokes

    :type: int
    """

    random_press: float
    """ Randomness factor for pressure and strength in new strokes

    :type: float
    """

    random_subdiv: float
    """ Randomness factor for new strokes after subdivision

    :type: float
    """

    strength: float
    """ Color strength for new strokes (affect alpha factor of color)

    :type: float
    """

    use_jitter_pressure: bool
    """ Use tablet pressure for jitter

    :type: bool
    """

    use_pressure: bool
    """ Use tablet pressure

    :type: bool
    """

    use_random_pressure: bool
    """ Use random value for pressure

    :type: bool
    """

    use_random_strength: bool
    """ Use random value for strength

    :type: bool
    """

    use_strength_pressure: bool
    """ Use tablet pressure for color strength

    :type: bool
    """

class GPencilFrame(bpy_struct):
    """Collection of related sketches on a particular frame"""

    frame_number: int
    """ The frame on which this sketch appears

    :type: int
    """

    is_edited: bool
    """ Frame is being edited (painted on)

    :type: bool
    """

    select: bool
    """ Frame is selected for editing in the Dope Sheet

    :type: bool
    """

    strokes: GPencilStrokes
    """ Freehand curves defining the sketch on this frame

    :type: GPencilStrokes
    """

    def clear(self):
        """Remove all the grease pencil frame data"""

class GPencilLayer(bpy_struct):
    """Collection of related sketches"""

    active_frame: GPencilFrame | None
    """ Frame currently being displayed for this layer

    :type: GPencilFrame | None
    """

    after_color: bpy_prop_array[float]
    """ Base color for ghosts after the active frame

    :type: bpy_prop_array[float]
    """

    before_color: bpy_prop_array[float]
    """ Base color for ghosts before the active frame

    :type: bpy_prop_array[float]
    """

    frames: GPencilFrames
    """ Sketches for this layer on different frames

    :type: GPencilFrames
    """

    ghost_after_range: int
    """ Maximum number of frames to show after current frame (0 = show only the next sketch, -1 = don't show any frames after current)

    :type: int
    """

    ghost_before_range: int
    """ Maximum number of frames to show before current frame (0 = show only the previous sketch, -1 = don't show any frames before current)

    :type: int
    """

    hide: bool
    """ Set layer Visibility

    :type: bool
    """

    info: str
    """ Layer name

    :type: str
    """

    is_parented: bool
    """ True when the layer parent object is set

    :type: bool
    """

    line_change: int
    """ Thickness change to apply to current strokes (in pixels)

    :type: int
    """

    lock: bool
    """ Protect layer from further editing and/or frame changes

    :type: bool
    """

    lock_frame: bool
    """ Lock current frame displayed by layer

    :type: bool
    """

    matrix_inverse: bpy_prop_array[float]
    """ Parent inverse transformation matrix

    :type: bpy_prop_array[float]
    """

    opacity: float
    """ Layer Opacity

    :type: float
    """

    parent: Object | None
    """ Parent Object

    :type: Object | None
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: typing.Literal["OBJECT", "ARMATURE", "BONE"]
    """ Type of parent relation

    :type: typing.Literal['OBJECT','ARMATURE','BONE']
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    show_points: bool
    """ Draw the points which make up the strokes (for debugging purposes)

    :type: bool
    """

    show_x_ray: bool
    """ Make the layer draw in front of objects

    :type: bool
    """

    tint_color: bpy_prop_array[float]
    """ Color for tinting stroke colors

    :type: bpy_prop_array[float]
    """

    tint_factor: float
    """ Factor of tinting color

    :type: float
    """

    unlock_color: bool
    """ Unprotect selected colors from further editing and/or frame changes

    :type: bool
    """

    use_ghost_custom_colors: bool
    """ Use custom colors for ghost frames

    :type: bool
    """

    use_onion_skinning: bool
    """ Ghost frames on either side of frame

    :type: bool
    """

    use_volumetric_strokes: bool
    """ Draw strokes as a series of circular blobs, resulting in a volumetric effect

    :type: bool
    """

    def clear(self):
        """Remove all the grease pencil layer data"""

class GPencilPalette(bpy_struct):
    """Collection of related palettes"""

    colors: GPencilPaletteColors
    """ Colors of the palette

    :type: GPencilPaletteColors
    """

    name: str
    """ Palette name

    :type: str
    """

class GPencilPaletteColor(bpy_struct):
    """Collection of related colors"""

    alpha: float
    """ Color Opacity

    :type: float
    """

    color: bpy_prop_array[float]
    """ Color for strokes

    :type: bpy_prop_array[float]
    """

    fill_alpha: float
    """ Opacity for filling region bounded by each stroke

    :type: float
    """

    fill_color: bpy_prop_array[float]
    """ Color for filling region bounded by each stroke

    :type: bpy_prop_array[float]
    """

    ghost: bool
    """ Display the color in onion skinning

    :type: bool
    """

    hide: bool
    """ Set color Visibility

    :type: bool
    """

    is_fill_visible: bool
    """ True when opacity of fill is set high enough to be visible

    :type: bool
    """

    is_stroke_visible: bool
    """ True when opacity of stroke is set high enough to be visible

    :type: bool
    """

    lock: bool
    """ Protect color from further editing and/or frame changes

    :type: bool
    """

    name: str
    """ Color name

    :type: str
    """

    use_hq_fill: bool
    """ Fill strokes using high quality to avoid glitches (slower fps during animation play)

    :type: bool
    """

    use_volumetric_strokes: bool
    """ Draw strokes as a series of circular blobs, resulting in a volumetric effect

    :type: bool
    """

class GPencilSculptBrush(bpy_struct):
    """Stroke editing brush"""

    affect_pressure: bool
    """ Affect pressure values as well when smoothing strokes

    :type: bool
    """

    direction: typing.Literal["ADD", "SUBTRACT"]
    """ 

    :type: typing.Literal['ADD','SUBTRACT']
    """

    size: int
    """ Radius of the brush in pixels

    :type: int
    """

    strength: float
    """ Brush strength

    :type: float
    """

    use_falloff: bool
    """ Strength of brush decays with distance from cursor

    :type: bool
    """

    use_pressure_strength: bool
    """ Enable tablet pressure sensitivity for strength

    :type: bool
    """

class GPencilSculptSettings(bpy_struct):
    """Properties for Grease Pencil stroke sculpting tool"""

    affect_position: bool
    """ The brush affects the position of the point

    :type: bool
    """

    affect_strength: bool
    """ The brush affects the color strength of the point

    :type: bool
    """

    affect_thickness: bool
    """ The brush affects the thickness of the point

    :type: bool
    """

    brush: GPencilSculptBrush
    """ 

    :type: GPencilSculptBrush
    """

    selection_alpha: float
    """ Alpha value for selected vertices

    :type: float
    """

    tool: typing.Literal[
        "SMOOTH",
        "THICKNESS",
        "STRENGTH",
        "GRAB",
        "PUSH",
        "TWIST",
        "PINCH",
        "RANDOMIZE",
        "CLONE",
    ]
    """ 

    :type: typing.Literal['SMOOTH','THICKNESS','STRENGTH','GRAB','PUSH','TWIST','PINCH','RANDOMIZE','CLONE']
    """

    use_select_mask: bool
    """ Only sculpt selected stroke points

    :type: bool
    """

class GPencilStroke(bpy_struct):
    """Freehand curve defining part of a sketch"""

    color: GPencilPaletteColor
    """ Color from palette used in Stroke

    :type: GPencilPaletteColor
    """

    colorname: str
    """ Palette color name

    :type: str
    """

    draw_cyclic: bool
    """ Enable cyclic drawing, closing the stroke

    :type: bool
    """

    draw_mode: typing.Literal["SCREEN", "3DSPACE", "2DSPACE", "2DIMAGE"]
    """ 

    :type: typing.Literal['SCREEN','3DSPACE','2DSPACE','2DIMAGE']
    """

    line_width: int
    """ Thickness of stroke (in pixels)

    :type: int
    """

    points: GPencilStrokePoints
    """ Stroke data points

    :type: GPencilStrokePoints
    """

    select: bool
    """ Stroke is selected for viewport editing

    :type: bool
    """

    triangles: bpy_prop_collection[GPencilTriangle]
    """ Triangulation data for HQ fill

    :type: bpy_prop_collection[GPencilTriangle]
    """

class GPencilStrokePoint(bpy_struct):
    """Data point for freehand stroke curve"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    pressure: float
    """ Pressure of tablet at point when drawing it

    :type: float
    """

    select: bool
    """ Point is selected for viewport editing

    :type: bool
    """

    strength: float
    """ Color intensity (alpha factor)

    :type: float
    """

class GPencilTriangle(bpy_struct):
    """Triangulation data for HQ fill"""

    v1: int
    """ First triangle vertex index

    :type: int
    """

    v2: int
    """ Second triangle vertex index

    :type: int
    """

    v3: int
    """ Third triangle vertex index

    :type: int
    """

class GameActuator(Actuator, bpy_struct):
    filename: str
    """ The file to use, depending on the mode (e.g. the blend file to load or a destination for saving a screenshot) - use the "//" prefix for a relative path

    :type: str
    """

    mode: typing.Literal["START", "RESTART", "QUIT", "SAVECFG", "LOADCFG", "SCREENSHOT"]
    """ 

    :type: typing.Literal['START','RESTART','QUIT','SAVECFG','LOADCFG','SCREENSHOT']
    """

class GameBooleanProperty(GameProperty, bpy_struct):
    """Game engine user defined Boolean property"""

    value: bool
    """ Property value

    :type: bool
    """

class GameFloatProperty(GameProperty, bpy_struct):
    """Game engine user defined floating point number property"""

    value: float
    """ Property value

    :type: float
    """

class GameIntProperty(GameProperty, bpy_struct):
    """Game engine user defined integer number property"""

    value: int
    """ Property value

    :type: int
    """

class GameObjectSettings(bpy_struct):
    """Game engine related settings for the object"""

    actuators: bpy_prop_collection[Actuator]
    """ Game engine actuators to act on events

    :type: bpy_prop_collection[Actuator]
    """

    angular_velocity_max: float
    """ Clamp angular velocity to this maximum speed, in angle per second

    :type: float
    """

    angular_velocity_min: float
    """ Clamp angular velocity to this minimum speed (except when totally still), in angle per second

    :type: float
    """

    collision_bounds_type: typing.Literal[
        "BOX", "SPHERE", "CYLINDER", "CONE", "CONVEX_HULL", "TRIANGLE_MESH", "CAPSULE"
    ]
    """ Select the collision shape that better fits the object

    :type: typing.Literal['BOX','SPHERE','CYLINDER','CONE','CONVEX_HULL','TRIANGLE_MESH','CAPSULE']
    """

    collision_group: bpy_prop_array[bool]
    """ The collision group of the object

    :type: bpy_prop_array[bool]
    """

    collision_margin: float
    """ Extra margin around object for collision detection, small amount required for stability

    :type: float
    """

    collision_mask: bpy_prop_array[bool]
    """ The groups this object can collide with

    :type: bpy_prop_array[bool]
    """

    controllers: bpy_prop_collection[Controller]
    """ Game engine controllers to process events, connecting sensors to actuators

    :type: bpy_prop_collection[Controller]
    """

    damping: float
    """ General movement damping

    :type: float
    """

    fall_speed: float
    """ Maximum speed at which the character will fall

    :type: float
    """

    form_factor: float
    """ Form factor scales the inertia tensor

    :type: float
    """

    friction_coefficients: bpy_prop_array[float]
    """ Relative friction coefficients in the in the X, Y and Z directions, when anisotropic friction is enabled

    :type: bpy_prop_array[float]
    """

    jump_max: int
    """ The maximum number of jumps the character can make before it hits the ground

    :type: int
    """

    jump_speed: float
    """ Upward velocity applied to the character when jumping

    :type: float
    """

    lock_location_x: bool
    """ Disable simulation of linear motion along the X axis

    :type: bool
    """

    lock_location_y: bool
    """ Disable simulation of linear motion along the Y axis

    :type: bool
    """

    lock_location_z: bool
    """ Disable simulation of linear motion along the Z axis

    :type: bool
    """

    lock_rotation_x: bool
    """ Disable simulation of angular motion along the X axis

    :type: bool
    """

    lock_rotation_y: bool
    """ Disable simulation of angular motion along the Y axis

    :type: bool
    """

    lock_rotation_z: bool
    """ Disable simulation of angular motion along the Z axis

    :type: bool
    """

    mass: float
    """ Mass of the object

    :type: float
    """

    obstacle_radius: float
    """ Radius of object representation in obstacle simulation

    :type: float
    """

    physics_type: typing.Literal[
        "NO_COLLISION",
        "STATIC",
        "DYNAMIC",
        "RIGID_BODY",
        "SOFT_BODY",
        "OCCLUDER",
        "SENSOR",
        "NAVMESH",
        "CHARACTER",
    ]
    """ Select the type of physical representation

    :type: typing.Literal['NO_COLLISION','STATIC','DYNAMIC','RIGID_BODY','SOFT_BODY','OCCLUDER','SENSOR','NAVMESH','CHARACTER']
    """

    properties: bpy_prop_collection[GameProperty]
    """ Game engine properties

    :type: bpy_prop_collection[GameProperty]
    """

    radius: float
    """ Radius of bounding sphere and material physics

    :type: float
    """

    rotation_damping: float
    """ General rotation damping

    :type: float
    """

    sensors: bpy_prop_collection[Sensor]
    """ Game engine sensor to detect events

    :type: bpy_prop_collection[Sensor]
    """

    show_actuators: bool
    """ Shows actuators for this object in the user interface

    :type: bool
    """

    show_controllers: bool
    """ Shows controllers for this object in the user interface

    :type: bool
    """

    show_debug_state: bool
    """ Print state debug info in the game engine

    :type: bool
    """

    show_sensors: bool
    """ Shows sensors for this object in the user interface

    :type: bool
    """

    show_state_panel: bool
    """ Show state panel

    :type: bool
    """

    soft_body: GameSoftBodySettings
    """ Settings for Bullet soft body simulation

    :type: GameSoftBodySettings
    """

    states_initial: bpy_prop_array[bool]
    """ Initial state when the game starts

    :type: bpy_prop_array[bool]
    """

    states_visible: bpy_prop_array[bool]
    """ State determining which controllers are displayed

    :type: bpy_prop_array[bool]
    """

    step_height: float
    """ Maximum height of steps the character can run over

    :type: float
    """

    use_activity_culling: bool
    """ Disable simulation of angular motion along the Z axis

    :type: bool
    """

    use_actor: bool
    """ Object is detected by the Near and Radar sensor

    :type: bool
    """

    use_all_states: bool
    """ Set all state bits

    :type: bool
    """

    use_anisotropic_friction: bool
    """ Enable anisotropic friction

    :type: bool
    """

    use_collision_bounds: bool
    """ Specify a collision bounds type other than the default

    :type: bool
    """

    use_collision_compound: bool
    """ Add children to form a compound collision object

    :type: bool
    """

    use_ghost: bool
    """ Object does not react to collisions, like a ghost

    :type: bool
    """

    use_material_physics_fh: bool
    """ React to force field physics settings in materials

    :type: bool
    """

    use_obstacle_create: bool
    """ Create representation for obstacle simulation

    :type: bool
    """

    use_record_animation: bool
    """ Record animation objects without physics

    :type: bool
    """

    use_rotate_from_normal: bool
    """ Use face normal to rotate object, so that it points away from the surface

    :type: bool
    """

    use_sleep: bool
    """ Disable auto (de)activation in physics simulation

    :type: bool
    """

    used_states: bpy_prop_array[bool]
    """ States which are being used by controllers

    :type: bpy_prop_array[bool]
    """

    velocity_max: float
    """ Clamp velocity to this maximum speed, in distance per second

    :type: float
    """

    velocity_min: float
    """ Clamp velocity to this minimum speed (except when totally still), in distance per second

    :type: float
    """

class GameProperty(bpy_struct):
    """Game engine user defined object property"""

    name: str
    """ Available as GameObject attributes in the game engine's python API

    :type: str
    """

    show_debug: bool
    """ Print debug information for this property

    :type: bool
    """

    type: typing.Literal["BOOL", "INT", "FLOAT", "STRING", "TIMER"]
    """ 

    :type: typing.Literal['BOOL','INT','FLOAT','STRING','TIMER']
    """

class GameSoftBodySettings(bpy_struct):
    """Soft body simulation settings for an object in the game engine"""

    cluster_iterations: int
    """ Number of cluster iterations

    :type: int
    """

    collision_margin: float
    """ Collision margin for soft body. Small value makes the algorithm unstable

    :type: float
    """

    dynamic_friction: float
    """ Dynamic Friction

    :type: float
    """

    linear_stiffness: float
    """ Linear stiffness of the soft body links

    :type: float
    """

    location_iterations: int
    """ Position solver iterations

    :type: int
    """

    shape_threshold: float
    """ Shape matching threshold

    :type: float
    """

    use_bending_constraints: bool
    """ Enable bending constraints

    :type: bool
    """

    use_cluster_rigid_to_softbody: bool
    """ Enable cluster collision between soft and rigid body

    :type: bool
    """

    use_cluster_soft_to_softbody: bool
    """ Enable cluster collision between soft and soft body

    :type: bool
    """

    use_shape_match: bool
    """ Enable soft body shape matching goal

    :type: bool
    """

    weld_threshold: float
    """ Welding threshold: distance between nearby vertices to be considered equal => set to 0.0 to disable welding test and speed up scene loading (ok if the mesh has no duplicates)

    :type: float
    """

class GameStringProperty(GameProperty, bpy_struct):
    """Game engine user defined text string property"""

    value: str
    """ Property value

    :type: str
    """

class GameTimerProperty(GameProperty, bpy_struct):
    """Game engine user defined timer property"""

    value: float
    """ Property value

    :type: float
    """

class GammaCrossSequence(EffectSequence, Sequence, bpy_struct):
    """Gamma Cross Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class GaussianBlurSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip creating a gaussian blur"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    size_x: float
    """ Size of the blur along X axis

    :type: float
    """

    size_y: float
    """ Size of the blur along Y axis

    :type: float
    """

class GlowSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip creating a glow effect"""

    blur_radius: float
    """ Radius of glow effect

    :type: float
    """

    boost_factor: float
    """ Brightness multiplier

    :type: float
    """

    clamp: float
    """ Brightness limit of intensity

    :type: float
    """

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

    quality: int
    """ Accuracy of the blur effect

    :type: int
    """

    threshold: float
    """ Minimum intensity to trigger a glow

    :type: float
    """

    use_only_boost: bool
    """ Show the glow buffer only

    :type: bool
    """

class GreasePencil(ID, bpy_struct):
    """Freehand annotation sketchbook"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    layers: GreasePencilLayers
    """ 

    :type: GreasePencilLayers
    """

    palettes: GreasePencilPalettes
    """ 

    :type: GreasePencilPalettes
    """

    show_stroke_direction: bool
    """ Show stroke drawing direction with a bigger green dot (start) and smaller red dot (end) points

    :type: bool
    """

    use_onion_skinning: bool
    """ Show ghosts of the frames before and after the current frame, toggle to enable on active layer or disable all

    :type: bool
    """

    use_stroke_edit_mode: bool
    """ Edit Grease Pencil strokes instead of viewport data

    :type: bool
    """

    def clear(self):
        """Remove all the grease pencil data"""

class Group(ID, bpy_struct):
    """Group of Object data-blocks"""

    dupli_offset: bpy_prop_array[float]
    """ Offset from the origin to use when instancing as DupliGroup

    :type: bpy_prop_array[float]
    """

    layers: bpy_prop_array[bool]
    """ Layers visible when this group is instanced as a dupli

    :type: bpy_prop_array[bool]
    """

    objects: GroupObjects
    """ A collection of this groups objects

    :type: GroupObjects
    """

    users_dupli_group: typing.Any
    """ The dupli group this group is used in
(readonly)"""

class Header(bpy_struct):
    """Editor header containing UI elements"""

    bl_idname: str
    """ If this is set, the header gets a custom ID, otherwise it takes the name of the class used to define the panel; for example, if the class name is "OBJECT_HT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_HT_hello"

    :type: str
    """

    bl_space_type: typing.Literal[
        "EMPTY",
        "VIEW_3D",
        "TIMELINE",
        "GRAPH_EDITOR",
        "DOPESHEET_EDITOR",
        "NLA_EDITOR",
        "IMAGE_EDITOR",
        "SEQUENCE_EDITOR",
        "CLIP_EDITOR",
        "TEXT_EDITOR",
        "NODE_EDITOR",
        "LOGIC_EDITOR",
        "PROPERTIES",
        "OUTLINER",
        "USER_PREFERENCES",
        "INFO",
        "FILE_BROWSER",
        "CONSOLE",
    ]
    """ The space where the header is going to be used in

    :type: typing.Literal['EMPTY','VIEW_3D','TIMELINE','GRAPH_EDITOR','DOPESHEET_EDITOR','NLA_EDITOR','IMAGE_EDITOR','SEQUENCE_EDITOR','CLIP_EDITOR','TEXT_EDITOR','NODE_EDITOR','LOGIC_EDITOR','PROPERTIES','OUTLINER','USER_PREFERENCES','INFO','FILE_BROWSER','CONSOLE']
    """

    layout: UILayout
    """ Structure of the header in the UI

    :type: UILayout
    """

    def draw(self, context: Context | None):
        """Draw UI elements into the header UI layout

        :param context:
        :type context: Context | None
        """

class HemiLamp(Lamp, ID, bpy_struct):
    """180 degree constant lamp"""

class Histogram(bpy_struct):
    """Statistical view of the levels of color in an image"""

    mode: typing.Literal["LUMA", "RGB", "R", "G", "B", "A"]
    """ Channels to display when drawing the histogram

    :type: typing.Literal['LUMA','RGB','R','G','B','A']
    """

    show_line: bool
    """ Display lines rather than filled shapes

    :type: bool
    """

class HookModifier(Modifier, bpy_struct):
    """Hook modifier to modify the location of vertices"""

    center: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    falloff_curve: CurveMapping
    """ Custom Lamp Falloff Curve

    :type: CurveMapping
    """

    falloff_radius: float
    """ If not zero, the distance from the hook where influence ends

    :type: float
    """

    falloff_type: typing.Literal[
        "NONE",
        "CURVE",
        "SMOOTH",
        "SPHERE",
        "ROOT",
        "INVERSE_SQUARE",
        "SHARP",
        "LINEAR",
        "CONSTANT",
    ]
    """ 

    :type: typing.Literal['NONE','CURVE','SMOOTH','SPHERE','ROOT','INVERSE_SQUARE','SHARP','LINEAR','CONSTANT']
    """

    matrix_inverse: bpy_prop_array[float]
    """ Reverse the transformation between this object and its target

    :type: bpy_prop_array[float]
    """

    object: Object | None
    """ Parent Object for hook, also recalculates and clears offset

    :type: Object | None
    """

    strength: float
    """ Relative force of the hook

    :type: float
    """

    subtarget: str
    """ Name of Parent Bone for hook (if applicable), also recalculates and clears offset

    :type: str
    """

    use_falloff_uniform: bool
    """ Compensate for non-uniform object scale

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class HueCorrectModifier(SequenceModifier, bpy_struct):
    """Hue correction modifier for sequence strip"""

    curve_mapping: CurveMapping
    """ 

    :type: CurveMapping
    """

class ID(bpy_struct):
    """Base type for data-blocks, defining a unique name, linking from other libraries and garbage collection"""

    is_library_indirect: bool
    """ Is this ID block linked indirectly

    :type: bool
    """

    is_updated: bool
    """ Datablock is tagged for recalculation

    :type: bool
    """

    is_updated_data: bool
    """ Datablock data is tagged for recalculation

    :type: bool
    """

    library: Library
    """ Library file the data-block is linked from

    :type: Library
    """

    name: str
    """ Unique data-block ID name

    :type: str
    """

    preview: ImagePreview
    """ Preview image and icon of this data-block (None if not supported for this type of data)

    :type: ImagePreview
    """

    tag: bool
    """ Tools can use this to tag data for their own purposes (initial state is undefined)

    :type: bool
    """

    use_fake_user: bool
    """ Save this data-block even if it has no users

    :type: bool
    """

    users: int
    """ Number of times this data-block is referenced

    :type: int
    """

    def copy(self) -> typing_extensions.Self:
        """Create a copy of this data-block (not supported for all data-blocks)

        :return: New copy of the ID
        :rtype: typing_extensions.Self
        """

    def user_clear(self):
        """Clear the user count of a data-block so its not saved, on reload the data will be removedThis function is for advanced use only, misuse can crash blender since the user
        count is used to prevent data being removed when it is used.

        """

    def user_remap(self, new_id: typing_extensions.Self):
        """Replace all usage in the .blend file of this ID by new given one

        :param new_id: New ID to use
        :type new_id: typing_extensions.Self
        """

    def user_of_id(self, id: typing_extensions.Self) -> int:
        """Count the number of times that ID uses/references given one

        :param id: ID to count usages
        :type id: typing_extensions.Self
        :return: Number of usages/references of given id by current datablock
        :rtype: int
        """

    def animation_data_create(self) -> AnimData:
        """Create animation data to this ID, note that not all ID types support this

        :return: New animation data or NULL
        :rtype: AnimData
        """

    def animation_data_clear(self):
        """Clear animation on this this ID"""

    def update_tag(
        self, refresh: set[typing.Literal["OBJECT", "DATA", "TIME"]] | None = {}
    ):
        """Tag the ID to update its display data, e.g. when calling `bpy.types.Scene.update`

        :param refresh: Type of updates to perform
        :type refresh: set[typing.Literal['OBJECT','DATA','TIME']] | None
        """

class IKParam(bpy_struct):
    """Base type for IK solver parameters"""

    ik_solver: typing.Literal["LEGACY", "ITASC"]
    """ IK solver for which these parameters are defined

    :type: typing.Literal['LEGACY','ITASC']
    """

class IMAGE_UV_sculpt(Panel, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

    @staticmethod
    def paint_settings(context):
        """

        :param context:
        """

    @staticmethod
    def prop_unified_color(parent, context, brush, prop_name, text=""):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param text:
        """

    @staticmethod
    def prop_unified_color_picker(parent, context, brush, prop_name, value_slider=True):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param value_slider:
        """

    @staticmethod
    def prop_unified_size(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """

    @staticmethod
    def prop_unified_strength(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """

    @staticmethod
    def prop_unified_weight(
        parent, context, brush, prop_name, icon="NONE", text="", slider=False
    ):
        """

        :param parent:
        :param context:
        :param brush:
        :param prop_name:
        :param icon:
        :param text:
        :param slider:
        """

    @staticmethod
    def unified_paint_settings(parent, context):
        """

        :param parent:
        :param context:
        """

class IMAGE_UV_sculpt_curve(Panel, bpy_struct):
    def draw(self, context):
        """

        :param context:
        """

class Image(ID, bpy_struct):
    """Image data-block referencing an external or packed image"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    bindcode: bpy_prop_array[int]
    """ OpenGL bindcode

    :type: bpy_prop_array[int]
    """

    channels: int
    """ Number of channels in pixels buffer

    :type: int
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    depth: int
    """ Image bit depth

    :type: int
    """

    display_aspect: bpy_prop_array[float]
    """ Display Aspect for this image, does not affect rendering

    :type: bpy_prop_array[float]
    """

    field_order: typing.Literal["EVEN", "ODD"]
    """ Order of video fields (select which lines are displayed first)

    :type: typing.Literal['EVEN','ODD']
    """

    file_format: typing.Literal[
        "BMP",
        "IRIS",
        "PNG",
        "JPEG",
        "JPEG2000",
        "TARGA",
        "TARGA_RAW",
        "CINEON",
        "DPX",
        "OPEN_EXR_MULTILAYER",
        "OPEN_EXR",
        "HDR",
        "TIFF",
        "AVI_JPEG",
        "AVI_RAW",
        "FRAMESERVER",
        "H264",
        "FFMPEG",
        "THEORA",
        "XVID",
    ]
    """ Format used for re-saving this file

    :type: typing.Literal['BMP','IRIS','PNG','JPEG','JPEG2000','TARGA','TARGA_RAW','CINEON','DPX','OPEN_EXR_MULTILAYER','OPEN_EXR','HDR','TIFF','AVI_JPEG','AVI_RAW','FRAMESERVER','H264','FFMPEG','THEORA','XVID']
    """

    filepath: str
    """ Image/Movie file name

    :type: str
    """

    filepath_raw: str
    """ Image/Movie file name (without data refreshing)

    :type: str
    """

    fps: int
    """ Speed of the animation in frames per second

    :type: int
    """

    frame_duration: int
    """ Duration (in frames) of the image (1 when not a video/sequence)

    :type: int
    """

    frame_end: int
    """ End frame of an animated texture

    :type: int
    """

    frame_start: int
    """ Start frame of an animated texture

    :type: int
    """

    generated_color: bpy_prop_array[float]
    """ Fill color for the generated image

    :type: bpy_prop_array[float]
    """

    generated_height: int
    """ Generated image height

    :type: int
    """

    generated_type: typing.Literal["BLANK", "UV_GRID", "COLOR_GRID"]
    """ Generated image type

    :type: typing.Literal['BLANK','UV_GRID','COLOR_GRID']
    """

    generated_width: int
    """ Generated image width

    :type: int
    """

    has_data: bool
    """ True if this image has data

    :type: bool
    """

    is_dirty: bool
    """ Image has changed and is not saved

    :type: bool
    """

    is_float: bool
    """ True if this image is stored in float buffer

    :type: bool
    """

    is_multiview: bool
    """ Image has more than one view

    :type: bool
    """

    is_stereo_3d: bool
    """ Image has left and right views

    :type: bool
    """

    mapping: typing.Literal["UV", "REFLECTION"]
    """ Mapping type to use for this image in the game engine

    :type: typing.Literal['UV','REFLECTION']
    """

    packed_file: PackedFile
    """ First packed file of the image

    :type: PackedFile
    """

    packed_files: bpy_prop_collection[ImagePackedFile]
    """ Collection of packed images

    :type: bpy_prop_collection[ImagePackedFile]
    """

    pixels: float
    """ Image pixels in floating point values

    :type: float
    """

    render_slots: RenderSlots
    """ Render slots of the image

    :type: RenderSlots
    """

    resolution: bpy_prop_array[float]
    """ X/Y pixels per meter

    :type: bpy_prop_array[float]
    """

    size: bpy_prop_array[int]
    """ Width and height in pixels, zero when image data cant be loaded

    :type: bpy_prop_array[int]
    """

    source: typing.Literal["FILE", "SEQUENCE", "MOVIE", "GENERATED", "VIEWER"]
    """ Where the image comes from

    :type: typing.Literal['FILE','SEQUENCE','MOVIE','GENERATED','VIEWER']
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    tiles_x: int
    """ Degree of repetition in the X direction

    :type: int
    """

    tiles_y: int
    """ Degree of repetition in the Y direction

    :type: int
    """

    type: typing.Literal[
        "IMAGE", "MULTILAYER", "UV_TEST", "RENDER_RESULT", "COMPOSITING"
    ]
    """ How to generate the image

    :type: typing.Literal['IMAGE','MULTILAYER','UV_TEST','RENDER_RESULT','COMPOSITING']
    """

    use_alpha: bool
    """ Use the alpha channel information from the image or make image fully opaque

    :type: bool
    """

    use_animation: bool
    """ Use as animated texture in the game engine

    :type: bool
    """

    use_clamp_x: bool
    """ Disable texture repeating horizontally

    :type: bool
    """

    use_clamp_y: bool
    """ Disable texture repeating vertically

    :type: bool
    """

    use_deinterlace: bool
    """ Deinterlace movie file on load

    :type: bool
    """

    use_fields: bool
    """ Use fields of the image

    :type: bool
    """

    use_generated_float: bool
    """ Generate floating point buffer

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_tiles: bool
    """ Use of tilemode for faces (default shift-LMB to pick the tile for selected faces)

    :type: bool
    """

    use_view_as_render: bool
    """ Apply render part of display transformation when displaying this image on the screen

    :type: bool
    """

    views_format: typing.Literal["INDIVIDUAL", "STEREO_3D"]
    """ Mode to load image views

    :type: typing.Literal['INDIVIDUAL','STEREO_3D']
    """

    def save_render(self, filepath: str, scene: Scene | None = None):
        """Save image to a specific path using a scenes render settings

        :param filepath: Save path
        :type filepath: str
        :param scene: Scene to take image parameters from
        :type scene: Scene | None
        """

    def save(self):
        """Save image to its source path"""

    def pack(
        self, as_png: bool | None = False, data: str = "", data_len: int | None = 0
    ):
        """Pack an image as embedded data into the .blend file

        :param as_png: as_png, Pack the image as PNG (needed for generated/dirty images)
        :type as_png: bool | None
        :param data: data, Raw data (bytes, exact content of the embedded file)
        :type data: str
        :param data_len: data_len, length of given data (mandatory if data is provided)
        :type data_len: int | None
        """

    def unpack(
        self,
        method: typing.Literal[
            "USE_LOCAL", "WRITE_LOCAL", "USE_ORIGINAL", "WRITE_ORIGINAL"
        ]
        | None = "USE_LOCAL",
    ):
        """Save an image packed in the .blend file to disk

        :param method: method, How to unpack
        :type method: typing.Literal['USE_LOCAL','WRITE_LOCAL','USE_ORIGINAL','WRITE_ORIGINAL'] | None
        """

    def reload(self):
        """Reload the image from its source path"""

    def update(self):
        """Update the display image from the floating point buffer"""

    def scale(self, width: int | None, height: int | None):
        """Scale the image in pixels

        :param width: Width
        :type width: int | None
        :param height: Height
        :type height: int | None
        """

    def gl_touch(
        self, frame: int | None = 0, filter: int | None = 9985, mag: int | None = 9729
    ) -> int:
        """Delay the image from being cleaned from the cache due inactivity

        :param frame: Frame, Frame of image sequence or movie
        :type frame: int | None
        :param filter: Filter, The texture minifying function to use if the image wasn't loaded
        :type filter: int | None
        :param mag: Magnification, The texture magnification function to use if the image wasn't loaded
        :type mag: int | None
        :return: Error, OpenGL error value
        :rtype: int
        """

    def gl_load(
        self, frame: int | None = 0, filter: int | None = 9985, mag: int | None = 9729
    ) -> int:
        """Load the image into OpenGL graphics memory

        :param frame: Frame, Frame of image sequence or movie
        :type frame: int | None
        :param filter: Filter, The texture minifying function
        :type filter: int | None
        :param mag: Magnification, The texture magnification function
        :type mag: int | None
        :return: Error, OpenGL error value
        :rtype: int
        """

    def gl_free(self):
        """Free the image from OpenGL graphics memory"""

    def filepath_from_user(self, image_user: ImageUser | None = None) -> str:
        """Return the absolute path to the filepath of an image frame specified by the image user

        :param image_user: Image user of the image to get filepath for
        :type image_user: ImageUser | None
        :return: File Path, The resulting filepath from the image and it's user
        :rtype: str
        """

    def buffers_free(self):
        """Free the image buffers from memory"""

class ImageFormatSettings(bpy_struct):
    """Settings for image formats"""

    cineon_black: int
    """ Log conversion reference blackpoint

    :type: int
    """

    cineon_gamma: float
    """ Log conversion gamma

    :type: float
    """

    cineon_white: int
    """ Log conversion reference whitepoint

    :type: int
    """

    color_depth: typing.Literal["8", "10", "12", "16", "32"]
    """ Bit depth per channel

    :type: typing.Literal['8','10','12','16','32']
    """

    color_mode: typing.Literal["BW", "RGB", "RGBA"]
    """ Choose BW for saving grayscale images, RGB for saving red, green and blue channels, and RGBA for saving red, green, blue and alpha channels

    :type: typing.Literal['BW','RGB','RGBA']
    """

    compression: int
    """ Amount of time to determine best compression: 0 = no compression with fast file output, 100 = maximum lossless compression with slow file output

    :type: int
    """

    display_settings: ColorManagedDisplaySettings
    """ Settings of device saved image would be displayed on

    :type: ColorManagedDisplaySettings
    """

    exr_codec: typing.Literal[
        "NONE", "PXR24", "ZIP", "PIZ", "RLE", "ZIPS", "B44", "B44A", "DWAA", "DWAB"
    ]
    """ Codec settings for OpenEXR

    :type: typing.Literal['NONE','PXR24','ZIP','PIZ','RLE','ZIPS','B44','B44A','DWAA','DWAB']
    """

    file_format: typing.Literal[
        "BMP",
        "IRIS",
        "PNG",
        "JPEG",
        "JPEG2000",
        "TARGA",
        "TARGA_RAW",
        "CINEON",
        "DPX",
        "OPEN_EXR_MULTILAYER",
        "OPEN_EXR",
        "HDR",
        "TIFF",
        "AVI_JPEG",
        "AVI_RAW",
        "FRAMESERVER",
        "H264",
        "FFMPEG",
        "THEORA",
        "XVID",
    ]
    """ File format to save the rendered images as

    :type: typing.Literal['BMP','IRIS','PNG','JPEG','JPEG2000','TARGA','TARGA_RAW','CINEON','DPX','OPEN_EXR_MULTILAYER','OPEN_EXR','HDR','TIFF','AVI_JPEG','AVI_RAW','FRAMESERVER','H264','FFMPEG','THEORA','XVID']
    """

    jpeg2k_codec: typing.Literal["JP2", "J2K"]
    """ Codec settings for Jpek2000

    :type: typing.Literal['JP2','J2K']
    """

    quality: int
    """ Quality for image formats that support lossy compression

    :type: int
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    tiff_codec: typing.Literal["NONE", "DEFLATE", "LZW", "PACKBITS"]
    """ Compression mode for TIFF

    :type: typing.Literal['NONE','DEFLATE','LZW','PACKBITS']
    """

    use_cineon_log: bool
    """ Convert to logarithmic color space

    :type: bool
    """

    use_jpeg2k_cinema_48: bool
    """ Use Openjpeg Cinema Preset (48fps)

    :type: bool
    """

    use_jpeg2k_cinema_preset: bool
    """ Use Openjpeg Cinema Preset

    :type: bool
    """

    use_jpeg2k_ycc: bool
    """ Save luminance-chrominance-chrominance channels instead of RGB colors

    :type: bool
    """

    use_preview: bool
    """ When rendering animations, save JPG preview images in same directory

    :type: bool
    """

    use_zbuffer: bool
    """ Save the z-depth per pixel (32 bit unsigned int z-buffer)

    :type: bool
    """

    view_settings: ColorManagedViewSettings
    """ Color management settings applied on image before saving

    :type: ColorManagedViewSettings
    """

    views_format: typing.Literal["INDIVIDUAL", "STEREO_3D"]
    """ Format of multiview media

    :type: typing.Literal['INDIVIDUAL','STEREO_3D']
    """

class ImagePackedFile(bpy_struct):
    filepath: str
    """ 

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    def save(self):
        """Save the packed file to its filepath"""

class ImagePaint(Paint, bpy_struct):
    """Properties of image and texture painting mode"""

    canvas: Image | None
    """ Image used as canvas

    :type: Image | None
    """

    clone_image: Image | None
    """ Image used as clone source

    :type: Image | None
    """

    dither: float
    """ Amount of dithering when painting on byte images

    :type: float
    """

    invert_stencil: bool
    """ Invert the stencil layer

    :type: bool
    """

    missing_materials: bool
    """ The mesh is missing materials

    :type: bool
    """

    missing_stencil: bool
    """ Image Painting does not have a stencil

    :type: bool
    """

    missing_texture: bool
    """ Image Painting does not have a texture to paint on

    :type: bool
    """

    missing_uvs: bool
    """ A UV layer is missing on the mesh

    :type: bool
    """

    mode: typing.Literal["MATERIAL", "IMAGE"]
    """ Mode of operation for projection painting

    :type: typing.Literal['MATERIAL','IMAGE']
    """

    normal_angle: int
    """ Paint most on faces pointing towards the view according to this angle

    :type: int
    """

    screen_grab_size: bpy_prop_array[int]
    """ Size to capture the image for re-projecting

    :type: bpy_prop_array[int]
    """

    seam_bleed: int
    """ Extend paint beyond the faces UVs to reduce seams (in pixels, slower)

    :type: int
    """

    stencil_color: bpy_prop_array[float]
    """ Stencil color in the viewport

    :type: bpy_prop_array[float]
    """

    stencil_image: Image | None
    """ Image used as stencil

    :type: Image | None
    """

    use_backface_culling: bool
    """ Ignore faces pointing away from the view (faster)

    :type: bool
    """

    use_clone_layer: bool
    """ Use another UV map as clone source, otherwise use the 3D cursor as the source

    :type: bool
    """

    use_normal_falloff: bool
    """ Paint most on faces pointing towards the view

    :type: bool
    """

    use_occlude: bool
    """ Only paint onto the faces directly under the brush (slower)

    :type: bool
    """

    use_stencil_layer: bool
    """ Set the mask layer from the UV map buttons

    :type: bool
    """

    def detect_data(self) -> bool:
        """Check if required texpaint data exist

        :return:
        :rtype: bool
        """

class ImagePreview(bpy_struct):
    """Preview image and icon"""

    icon_id: int
    """ Unique integer identifying this preview as an icon (zero means invalid)

    :type: int
    """

    icon_pixels: int
    """ Icon pixels, as bytes (always RGBA 32bits)

    :type: int
    """

    icon_pixels_float: float
    """ Icon pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    icon_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    image_pixels: int
    """ Image pixels, as bytes (always RGBA 32bits)

    :type: int
    """

    image_pixels_float: float
    """ Image pixels components, as floats (RGBA concatenated values)

    :type: float
    """

    image_size: bpy_prop_array[int]
    """ Width and height in pixels

    :type: bpy_prop_array[int]
    """

    is_icon_custom: bool
    """ True if this preview icon has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    is_image_custom: bool
    """ True if this preview image has been modified by py script,and is no more auto-generated by Blender

    :type: bool
    """

    def reload(self):
        """Reload the preview from its source path"""

class ImageSequence(Sequence, bpy_struct):
    """Sequence strip to load one or more images"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    directory: str
    """ 

    :type: str
    """

    elements: SequenceElements
    """ 

    :type: SequenceElements
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

    views_format: typing.Literal["INDIVIDUAL", "STEREO_3D"]
    """ Mode to load image views

    :type: typing.Literal['INDIVIDUAL','STEREO_3D']
    """

class ImageTexture(Texture, ID, bpy_struct):
    checker_distance: float
    """ Distance between checker tiles

    :type: float
    """

    crop_max_x: float
    """ Maximum X value to crop the image

    :type: float
    """

    crop_max_y: float
    """ Maximum Y value to crop the image

    :type: float
    """

    crop_min_x: float
    """ Minimum X value to crop the image

    :type: float
    """

    crop_min_y: float
    """ Minimum Y value to crop the image

    :type: float
    """

    extension: typing.Literal["EXTEND", "CLIP", "CLIP_CUBE", "REPEAT", "CHECKER"]
    """ How the image is extrapolated past its original bounds

    :type: typing.Literal['EXTEND','CLIP','CLIP_CUBE','REPEAT','CHECKER']
    """

    filter_eccentricity: int
    """ Maximum eccentricity (higher gives less blur at distant/oblique angles, but is also slower)

    :type: int
    """

    filter_probes: int
    """ Maximum number of samples (higher gives less blur at distant/oblique angles, but is also slower)

    :type: int
    """

    filter_size: float
    """ Multiply the filter size used by MIP Map and Interpolation

    :type: float
    """

    filter_type: typing.Literal["BOX", "EWA", "FELINE", "AREA"]
    """ Texture filter to use for sampling image

    :type: typing.Literal['BOX','EWA','FELINE','AREA']
    """

    image: Image | None
    """ 

    :type: Image | None
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    invert_alpha: bool
    """ Invert all the alpha values in the image

    :type: bool
    """

    repeat_x: int
    """ Repetition multiplier in the X direction

    :type: int
    """

    repeat_y: int
    """ Repetition multiplier in the Y direction

    :type: int
    """

    use_alpha: bool
    """ Use the alpha channel information in the image

    :type: bool
    """

    use_calculate_alpha: bool
    """ Calculate an alpha channel based on RGB values in the image

    :type: bool
    """

    use_checker_even: bool
    """ Even checker tiles

    :type: bool
    """

    use_checker_odd: bool
    """ Odd checker tiles

    :type: bool
    """

    use_derivative_map: bool
    """ Use red and green as derivative values

    :type: bool
    """

    use_filter_size_min: bool
    """ Use Filter Size as a minimal filter value in pixels

    :type: bool
    """

    use_flip_axis: bool
    """ Flip the texture's X and Y axis

    :type: bool
    """

    use_interpolation: bool
    """ Interpolate pixels using selected filter

    :type: bool
    """

    use_mipmap: bool
    """ Use auto-generated MIP maps for the image

    :type: bool
    """

    use_mipmap_gauss: bool
    """ Use Gauss filter to sample down MIP maps

    :type: bool
    """

    use_mirror_x: bool
    """ Mirror the image repetition on the X direction

    :type: bool
    """

    use_mirror_y: bool
    """ Mirror the image repetition on the Y direction

    :type: bool
    """

    use_normal_map: bool
    """ Use image RGB values for normal mapping

    :type: bool
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class ImageUser(bpy_struct):
    """Parameters defining how an Image data-block is used by another data-block"""

    fields_per_frame: int
    """ Number of fields per rendered frame (2 fields is 1 image)

    :type: int
    """

    frame_current: int
    """ Current frame number in image sequence or movie

    :type: int
    """

    frame_duration: int
    """ Number of images of a movie to use

    :type: int
    """

    frame_offset: int
    """ Offset the number of the frame to use in the animation

    :type: int
    """

    frame_start: int
    """ Global starting frame of the movie/sequence, assuming first picture has a #1

    :type: int
    """

    multilayer_layer: int
    """ Layer in multilayer image

    :type: int
    """

    multilayer_pass: int
    """ Pass in multilayer image

    :type: int
    """

    multilayer_view: int
    """ View in multilayer image

    :type: int
    """

    use_auto_refresh: bool
    """ Always refresh image on frame changes

    :type: bool
    """

    use_cyclic: bool
    """ Cycle the images in the movie

    :type: bool
    """

class ImapaintToolCapabilities(bpy_struct):
    """Read-only indications of which brush operations are supported by the current image paint brush"""

    has_accumulate: bool
    """ 

    :type: bool
    """

    has_radius: bool
    """ 

    :type: bool
    """

    has_space_attenuation: bool
    """ 

    :type: bool
    """

class InflowFluidSettings(FluidSettings, bpy_struct):
    """Fluid simulation settings for objects adding fluids in the simulation"""

    inflow_velocity: bpy_prop_array[float]
    """ Initial velocity of fluid

    :type: bpy_prop_array[float]
    """

    use: bool
    """ Object contributes to the fluid simulation

    :type: bool
    """

    use_animated_mesh: bool
    """ Export this mesh as an animated one (slower and enforces No Slip, only use if really necessary [e.g. armatures or parented objects], animated pos/rot/scale F-Curves do not require it)

    :type: bool
    """

    use_local_coords: bool
    """ Use local coordinates for inflow (e.g. for rotating objects)

    :type: bool
    """

    volume_initialization: typing.Literal["VOLUME", "SHELL", "BOTH"]
    """ Volume initialization type (WARNING: complex volumes might require too much memory and break simulation)

    :type: typing.Literal['VOLUME','SHELL','BOTH']
    """

class IntProperty(Property, bpy_struct):
    """RNA integer number property definition"""

    array_length: int
    """ Maximum length of the array, 0 means unlimited

    :type: int
    """

    default: int
    """ Default value for this number

    :type: int
    """

    default_array: bpy_prop_array[int]
    """ Default value for this array

    :type: bpy_prop_array[int]
    """

    hard_max: int
    """ Maximum value used by buttons

    :type: int
    """

    hard_min: int
    """ Minimum value used by buttons

    :type: int
    """

    is_array: bool
    """ 

    :type: bool
    """

    soft_max: int
    """ Maximum value used by buttons

    :type: int
    """

    soft_min: int
    """ Minimum value used by buttons

    :type: int
    """

    step: int
    """ Step size used by number buttons, for floats 1/100th of the step size

    :type: int
    """

class Itasc(IKParam, bpy_struct):
    """Parameters for the iTaSC IK solver"""

    damping_epsilon: float
    """ Singular value under which damping is progressively applied (higher values=more stability, less reactivity - default=0.1)

    :type: float
    """

    damping_max: float
    """ Maximum damping coefficient when singular value is nearly 0 (higher values=more stability, less reactivity - default=0.5)

    :type: float
    """

    feedback: float
    """ Feedback coefficient for error correction, average response time is 1/feedback (default=20)

    :type: float
    """

    iterations: int
    """ Maximum number of iterations for convergence in case of reiteration

    :type: int
    """

    mode: typing.Literal["ANIMATION", "SIMULATION"]
    """ 

    :type: typing.Literal['ANIMATION','SIMULATION']
    """

    precision: float
    """ Precision of convergence in case of reiteration

    :type: float
    """

    reiteration_method: typing.Literal["NEVER", "INITIAL", "ALWAYS"]
    """ Defines if the solver is allowed to reiterate (converge until precision is met) on none, first or all frames

    :type: typing.Literal['NEVER','INITIAL','ALWAYS']
    """

    solver: typing.Literal["SDLS", "DLS"]
    """ Solving method selection: automatic damping or manual damping

    :type: typing.Literal['SDLS','DLS']
    """

    step_count: int
    """ Divide the frame interval into this many steps

    :type: int
    """

    step_max: float
    """ Higher bound for timestep in second in case of automatic substeps

    :type: float
    """

    step_min: float
    """ Lower bound for timestep in second in case of automatic substeps

    :type: float
    """

    use_auto_step: bool
    """ Automatically determine the optimal number of steps for best performance/accuracy trade off

    :type: bool
    """

    velocity_max: float
    """ Maximum joint velocity in rad/s (default=50)

    :type: float
    """

class JoystickSensor(Sensor, bpy_struct):
    """Sensor to detect joystick events"""

    axis_direction: typing.Literal["RIGHTAXIS", "UPAXIS", "LEFTAXIS", "DOWNAXIS"]
    """ The direction of the axis

    :type: typing.Literal['RIGHTAXIS','UPAXIS','LEFTAXIS','DOWNAXIS']
    """

    axis_number: int
    """ Which axis pair to use, 1 is usually the main direction input

    :type: int
    """

    axis_threshold: int
    """ Precision of the axis

    :type: int
    """

    button_number: int
    """ Which button to use

    :type: int
    """

    event_type: typing.Literal["BUTTON", "AXIS", "HAT", "AXIS_SINGLE"]
    """ The type of event this joystick sensor is triggered on

    :type: typing.Literal['BUTTON','AXIS','HAT','AXIS_SINGLE']
    """

    hat_direction: typing.Literal[
        "UP", "DOWN", "LEFT", "RIGHT", "UPRIGHT", "DOWNLEFT", "UPLEFT", "DOWNRIGHT"
    ]
    """ Hat direction

    :type: typing.Literal['UP','DOWN','LEFT','RIGHT','UPRIGHT','DOWNLEFT','UPLEFT','DOWNRIGHT']
    """

    hat_number: int
    """ Which hat to use

    :type: int
    """

    joystick_index: int
    """ Which joystick to use

    :type: int
    """

    single_axis_number: int
    """ Single axis (vertical/horizontal/other) to detect

    :type: int
    """

    use_all_events: bool
    """ Triggered by all events on this joystick's current type (axis/button/hat)

    :type: bool
    """

class Key(ID, bpy_struct):
    """Shape keys data-block containing different shapes of geometric data-blocks"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    eval_time: float
    """ Evaluation time for absolute shape keys

    :type: float
    """

    key_blocks: bpy_prop_collection[ShapeKey]
    """ Shape keys

    :type: bpy_prop_collection[ShapeKey]
    """

    reference_key: ShapeKey
    """ 

    :type: ShapeKey
    """

    use_relative: bool
    """ Make shape keys relative, otherwise play through shapes as a sequence using the evaluation time

    :type: bool
    """

    user: ID
    """ Datablock using these shape keys

    :type: ID
    """

class KeyConfig(bpy_struct):
    """Input configuration, including keymaps"""

    is_user_defined: bool
    """ Indicates that a keyconfig was defined by the user

    :type: bool
    """

    keymaps: KeyMaps
    """ Key maps configured as part of this configuration

    :type: KeyMaps
    """

    name: str
    """ Name of the key configuration

    :type: str
    """

class KeyMap(bpy_struct):
    """Input configuration, including keymaps"""

    is_modal: bool
    """ Indicates that a keymap is used for translate modal events for an operator

    :type: bool
    """

    is_user_modified: bool
    """ Keymap is defined by the user

    :type: bool
    """

    keymap_items: KeyMapItems
    """ Items in the keymap, linking an operator to an input event

    :type: KeyMapItems
    """

    name: str
    """ Name of the key map

    :type: str
    """

    region_type: typing.Literal[
        "WINDOW",
        "HEADER",
        "CHANNELS",
        "TEMPORARY",
        "UI",
        "TOOLS",
        "TOOL_PROPS",
        "PREVIEW",
    ]
    """ Optional region type keymap is associated with

    :type: typing.Literal['WINDOW','HEADER','CHANNELS','TEMPORARY','UI','TOOLS','TOOL_PROPS','PREVIEW']
    """

    show_expanded_children: bool
    """ Children expanded in the user interface

    :type: bool
    """

    show_expanded_items: bool
    """ Expanded in the user interface

    :type: bool
    """

    space_type: typing.Literal[
        "EMPTY",
        "VIEW_3D",
        "TIMELINE",
        "GRAPH_EDITOR",
        "DOPESHEET_EDITOR",
        "NLA_EDITOR",
        "IMAGE_EDITOR",
        "SEQUENCE_EDITOR",
        "CLIP_EDITOR",
        "TEXT_EDITOR",
        "NODE_EDITOR",
        "LOGIC_EDITOR",
        "PROPERTIES",
        "OUTLINER",
        "USER_PREFERENCES",
        "INFO",
        "FILE_BROWSER",
        "CONSOLE",
    ]
    """ Optional space type keymap is associated with

    :type: typing.Literal['EMPTY','VIEW_3D','TIMELINE','GRAPH_EDITOR','DOPESHEET_EDITOR','NLA_EDITOR','IMAGE_EDITOR','SEQUENCE_EDITOR','CLIP_EDITOR','TEXT_EDITOR','NODE_EDITOR','LOGIC_EDITOR','PROPERTIES','OUTLINER','USER_PREFERENCES','INFO','FILE_BROWSER','CONSOLE']
    """

    def active(self) -> typing_extensions.Self:
        """active

        :return: Key Map, Active key map
        :rtype: typing_extensions.Self
        """

    def restore_to_default(self):
        """restore_to_default"""

    def restore_item_to_default(self, item: KeyMapItem):
        """restore_item_to_default

        :param item: Item
        :type item: KeyMapItem
        """

class KeyMapItem(bpy_struct):
    """Item in a Key Map"""

    active: bool | None
    """ Activate or deactivate item

    :type: bool | None
    """

    alt: bool
    """ Alt key pressed

    :type: bool
    """

    any: bool
    """ Any modifier keys pressed

    :type: bool
    """

    ctrl: bool
    """ Control key pressed

    :type: bool
    """

    id: int
    """ ID of the item

    :type: int
    """

    idname: str
    """ Identifier of operator to call on input event

    :type: str
    """

    is_user_defined: bool
    """ Is this keymap item user defined (doesn't just replace a builtin item)

    :type: bool
    """

    is_user_modified: bool
    """ Is this keymap item modified by the user

    :type: bool
    """

    key_modifier: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ Regular key pressed as a modifier

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    map_type: typing.Literal["KEYBOARD", "TWEAK", "MOUSE", "NDOF", "TEXTINPUT", "TIMER"]
    """ Type of event mapping

    :type: typing.Literal['KEYBOARD','TWEAK','MOUSE','NDOF','TEXTINPUT','TIMER']
    """

    name: str
    """ Name of operator (translated) to call on input event

    :type: str
    """

    oskey: bool
    """ Operating system key pressed

    :type: bool
    """

    properties: OperatorProperties
    """ Properties to set when the operator is called

    :type: OperatorProperties
    """

    propvalue: typing.Literal["NONE"]
    """ The value this event translates to in a modal keymap

    :type: typing.Literal['NONE']
    """

    shift: bool
    """ Shift key pressed

    :type: bool
    """

    show_expanded: bool
    """ Show key map event and property details in the user interface

    :type: bool
    """

    type: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ Type of event

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    value: typing.Literal[
        "ANY",
        "NOTHING",
        "PRESS",
        "RELEASE",
        "CLICK",
        "DOUBLE_CLICK",
        "NORTH",
        "NORTH_EAST",
        "EAST",
        "SOUTH_EAST",
        "SOUTH",
        "SOUTH_WEST",
        "WEST",
        "NORTH_WEST",
    ]
    """ 

    :type: typing.Literal['ANY','NOTHING','PRESS','RELEASE','CLICK','DOUBLE_CLICK','NORTH','NORTH_EAST','EAST','SOUTH_EAST','SOUTH','SOUTH_WEST','WEST','NORTH_WEST']
    """

    def compare(self, item: typing_extensions.Self | None) -> bool:
        """compare

        :param item: Item
        :type item: typing_extensions.Self | None
        :return: Comparison result
        :rtype: bool
        """

class KeyboardSensor(Sensor, bpy_struct):
    """Sensor to detect keyboard events"""

    key: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ 

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    log: str
    """ Property that indicates whether to log keystrokes as a string

    :type: str
    """

    modifier_key_1: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ Modifier key code

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    modifier_key_2: typing.Literal[
        "NONE",
        "LEFTMOUSE",
        "MIDDLEMOUSE",
        "RIGHTMOUSE",
        "BUTTON4MOUSE",
        "BUTTON5MOUSE",
        "BUTTON6MOUSE",
        "BUTTON7MOUSE",
        "ACTIONMOUSE",
        "SELECTMOUSE",
        "PEN",
        "ERASER",
        "MOUSEMOVE",
        "INBETWEEN_MOUSEMOVE",
        "TRACKPADPAN",
        "TRACKPADZOOM",
        "MOUSEROTATE",
        "WHEELUPMOUSE",
        "WHEELDOWNMOUSE",
        "WHEELINMOUSE",
        "WHEELOUTMOUSE",
        "EVT_TWEAK_L",
        "EVT_TWEAK_M",
        "EVT_TWEAK_R",
        "EVT_TWEAK_A",
        "EVT_TWEAK_S",
        "A",
        "B",
        "C",
        "D",
        "E",
        "F",
        "G",
        "H",
        "I",
        "J",
        "K",
        "L",
        "M",
        "N",
        "O",
        "P",
        "Q",
        "R",
        "S",
        "T",
        "U",
        "V",
        "W",
        "X",
        "Y",
        "Z",
        "ZERO",
        "ONE",
        "TWO",
        "THREE",
        "FOUR",
        "FIVE",
        "SIX",
        "SEVEN",
        "EIGHT",
        "NINE",
        "LEFT_CTRL",
        "LEFT_ALT",
        "LEFT_SHIFT",
        "RIGHT_ALT",
        "RIGHT_CTRL",
        "RIGHT_SHIFT",
        "OSKEY",
        "GRLESS",
        "ESC",
        "TAB",
        "RET",
        "SPACE",
        "LINE_FEED",
        "BACK_SPACE",
        "DEL",
        "SEMI_COLON",
        "PERIOD",
        "COMMA",
        "QUOTE",
        "ACCENT_GRAVE",
        "MINUS",
        "PLUS",
        "SLASH",
        "BACK_SLASH",
        "EQUAL",
        "LEFT_BRACKET",
        "RIGHT_BRACKET",
        "LEFT_ARROW",
        "DOWN_ARROW",
        "RIGHT_ARROW",
        "UP_ARROW",
        "NUMPAD_2",
        "NUMPAD_4",
        "NUMPAD_6",
        "NUMPAD_8",
        "NUMPAD_1",
        "NUMPAD_3",
        "NUMPAD_5",
        "NUMPAD_7",
        "NUMPAD_9",
        "NUMPAD_PERIOD",
        "NUMPAD_SLASH",
        "NUMPAD_ASTERIX",
        "NUMPAD_0",
        "NUMPAD_MINUS",
        "NUMPAD_ENTER",
        "NUMPAD_PLUS",
        "F1",
        "F2",
        "F3",
        "F4",
        "F5",
        "F6",
        "F7",
        "F8",
        "F9",
        "F10",
        "F11",
        "F12",
        "F13",
        "F14",
        "F15",
        "F16",
        "F17",
        "F18",
        "F19",
        "PAUSE",
        "INSERT",
        "HOME",
        "PAGE_UP",
        "PAGE_DOWN",
        "END",
        "MEDIA_PLAY",
        "MEDIA_STOP",
        "MEDIA_FIRST",
        "MEDIA_LAST",
        "TEXTINPUT",
        "WINDOW_DEACTIVATE",
        "TIMER",
        "TIMER0",
        "TIMER1",
        "TIMER2",
        "TIMER_JOBS",
        "TIMER_AUTOSAVE",
        "TIMER_REPORT",
        "TIMERREGION",
        "NDOF_MOTION",
        "NDOF_BUTTON_MENU",
        "NDOF_BUTTON_FIT",
        "NDOF_BUTTON_TOP",
        "NDOF_BUTTON_BOTTOM",
        "NDOF_BUTTON_LEFT",
        "NDOF_BUTTON_RIGHT",
        "NDOF_BUTTON_FRONT",
        "NDOF_BUTTON_BACK",
        "NDOF_BUTTON_ISO1",
        "NDOF_BUTTON_ISO2",
        "NDOF_BUTTON_ROLL_CW",
        "NDOF_BUTTON_ROLL_CCW",
        "NDOF_BUTTON_SPIN_CW",
        "NDOF_BUTTON_SPIN_CCW",
        "NDOF_BUTTON_TILT_CW",
        "NDOF_BUTTON_TILT_CCW",
        "NDOF_BUTTON_ROTATE",
        "NDOF_BUTTON_PANZOOM",
        "NDOF_BUTTON_DOMINANT",
        "NDOF_BUTTON_PLUS",
        "NDOF_BUTTON_MINUS",
        "NDOF_BUTTON_ESC",
        "NDOF_BUTTON_ALT",
        "NDOF_BUTTON_SHIFT",
        "NDOF_BUTTON_CTRL",
        "NDOF_BUTTON_1",
        "NDOF_BUTTON_2",
        "NDOF_BUTTON_3",
        "NDOF_BUTTON_4",
        "NDOF_BUTTON_5",
        "NDOF_BUTTON_6",
        "NDOF_BUTTON_7",
        "NDOF_BUTTON_8",
        "NDOF_BUTTON_9",
        "NDOF_BUTTON_10",
        "NDOF_BUTTON_A",
        "NDOF_BUTTON_B",
        "NDOF_BUTTON_C",
    ]
    """ Modifier key code

    :type: typing.Literal['NONE','LEFTMOUSE','MIDDLEMOUSE','RIGHTMOUSE','BUTTON4MOUSE','BUTTON5MOUSE','BUTTON6MOUSE','BUTTON7MOUSE','ACTIONMOUSE','SELECTMOUSE','PEN','ERASER','MOUSEMOVE','INBETWEEN_MOUSEMOVE','TRACKPADPAN','TRACKPADZOOM','MOUSEROTATE','WHEELUPMOUSE','WHEELDOWNMOUSE','WHEELINMOUSE','WHEELOUTMOUSE','EVT_TWEAK_L','EVT_TWEAK_M','EVT_TWEAK_R','EVT_TWEAK_A','EVT_TWEAK_S','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T','U','V','W','X','Y','Z','ZERO','ONE','TWO','THREE','FOUR','FIVE','SIX','SEVEN','EIGHT','NINE','LEFT_CTRL','LEFT_ALT','LEFT_SHIFT','RIGHT_ALT','RIGHT_CTRL','RIGHT_SHIFT','OSKEY','GRLESS','ESC','TAB','RET','SPACE','LINE_FEED','BACK_SPACE','DEL','SEMI_COLON','PERIOD','COMMA','QUOTE','ACCENT_GRAVE','MINUS','PLUS','SLASH','BACK_SLASH','EQUAL','LEFT_BRACKET','RIGHT_BRACKET','LEFT_ARROW','DOWN_ARROW','RIGHT_ARROW','UP_ARROW','NUMPAD_2','NUMPAD_4','NUMPAD_6','NUMPAD_8','NUMPAD_1','NUMPAD_3','NUMPAD_5','NUMPAD_7','NUMPAD_9','NUMPAD_PERIOD','NUMPAD_SLASH','NUMPAD_ASTERIX','NUMPAD_0','NUMPAD_MINUS','NUMPAD_ENTER','NUMPAD_PLUS','F1','F2','F3','F4','F5','F6','F7','F8','F9','F10','F11','F12','F13','F14','F15','F16','F17','F18','F19','PAUSE','INSERT','HOME','PAGE_UP','PAGE_DOWN','END','MEDIA_PLAY','MEDIA_STOP','MEDIA_FIRST','MEDIA_LAST','TEXTINPUT','WINDOW_DEACTIVATE','TIMER','TIMER0','TIMER1','TIMER2','TIMER_JOBS','TIMER_AUTOSAVE','TIMER_REPORT','TIMERREGION','NDOF_MOTION','NDOF_BUTTON_MENU','NDOF_BUTTON_FIT','NDOF_BUTTON_TOP','NDOF_BUTTON_BOTTOM','NDOF_BUTTON_LEFT','NDOF_BUTTON_RIGHT','NDOF_BUTTON_FRONT','NDOF_BUTTON_BACK','NDOF_BUTTON_ISO1','NDOF_BUTTON_ISO2','NDOF_BUTTON_ROLL_CW','NDOF_BUTTON_ROLL_CCW','NDOF_BUTTON_SPIN_CW','NDOF_BUTTON_SPIN_CCW','NDOF_BUTTON_TILT_CW','NDOF_BUTTON_TILT_CCW','NDOF_BUTTON_ROTATE','NDOF_BUTTON_PANZOOM','NDOF_BUTTON_DOMINANT','NDOF_BUTTON_PLUS','NDOF_BUTTON_MINUS','NDOF_BUTTON_ESC','NDOF_BUTTON_ALT','NDOF_BUTTON_SHIFT','NDOF_BUTTON_CTRL','NDOF_BUTTON_1','NDOF_BUTTON_2','NDOF_BUTTON_3','NDOF_BUTTON_4','NDOF_BUTTON_5','NDOF_BUTTON_6','NDOF_BUTTON_7','NDOF_BUTTON_8','NDOF_BUTTON_9','NDOF_BUTTON_10','NDOF_BUTTON_A','NDOF_BUTTON_B','NDOF_BUTTON_C']
    """

    target: str
    """ Property that receives the keystrokes in case a string is logged

    :type: str
    """

    use_all_keys: bool
    """ Trigger this sensor on any keystroke

    :type: bool
    """

class Keyframe(bpy_struct):
    """Bezier curve point with two handles defining a Keyframe on an F-Curve"""

    amplitude: float
    """ Amount to boost elastic bounces for 'elastic' easing

    :type: float
    """

    back: float
    """ Amount of overshoot for 'back' easing

    :type: float
    """

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    easing: typing.Literal["AUTO", "EASE_IN", "EASE_OUT", "EASE_IN_OUT"]
    """ Which ends of the segment between this and the next keyframe easing interpolation is applied to

    :type: typing.Literal['AUTO','EASE_IN','EASE_OUT','EASE_IN_OUT']
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the left handle (before the control point)

    :type: bpy_prop_array[float]
    """

    handle_left_type: typing.Literal[
        "FREE", "VECTOR", "ALIGNED", "AUTO", "AUTO_CLAMPED"
    ]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO','AUTO_CLAMPED']
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the right handle (after the control point)

    :type: bpy_prop_array[float]
    """

    handle_right_type: typing.Literal[
        "FREE", "VECTOR", "ALIGNED", "AUTO", "AUTO_CLAMPED"
    ]
    """ Handle types

    :type: typing.Literal['FREE','VECTOR','ALIGNED','AUTO','AUTO_CLAMPED']
    """

    interpolation: typing.Literal[
        "CONSTANT",
        "LINEAR",
        "BEZIER",
        "SINE",
        "QUAD",
        "CUBIC",
        "QUART",
        "QUINT",
        "EXPO",
        "CIRC",
        "BACK",
        "BOUNCE",
        "ELASTIC",
    ]
    """ Interpolation method to use for segment of the F-Curve from this Keyframe until the next Keyframe

    :type: typing.Literal['CONSTANT','LINEAR','BEZIER','SINE','QUAD','CUBIC','QUART','QUINT','EXPO','CIRC','BACK','BOUNCE','ELASTIC']
    """

    period: float
    """ Time between bounces for elastic easing

    :type: float
    """

    select_control_point: bool
    """ Control point selection status

    :type: bool
    """

    select_left_handle: bool
    """ Left handle selection status

    :type: bool
    """

    select_right_handle: bool
    """ Right handle selection status

    :type: bool
    """

    type: typing.Literal["KEYFRAME", "BREAKDOWN", "MOVING_HOLD", "EXTREME", "JITTER"]
    """ Type of keyframe (for visual purposes only)

    :type: typing.Literal['KEYFRAME','BREAKDOWN','MOVING_HOLD','EXTREME','JITTER']
    """

class KeyingSet(bpy_struct):
    """Settings that should be keyframed together"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    is_path_absolute: bool
    """ Keying Set defines specific paths/settings to be keyframed (i.e. is not reliant on context info)

    :type: bool
    """

    paths: KeyingSetPaths
    """ Keying Set Paths to define settings that get keyframed together

    :type: KeyingSetPaths
    """

    type_info: KeyingSetInfo
    """ Callback function defines for built-in Keying Sets

    :type: KeyingSetInfo
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

    def refresh(self):
        """Refresh Keying Set to ensure that it is valid for the current context (call before each use of one)"""

class KeyingSetInfo(bpy_struct):
    """Callback function defines for builtin Keying Sets"""

    bl_description: str
    """ A short description of the keying set

    :type: str
    """

    bl_idname: str
    """ If this is set, the Keying Set gets a custom ID, otherwise it takes the name of the class used to define the Keying Set (for example, if the class name is "BUILTIN_KSI_location", and bl_idname is not set by the script, then bl_idname = "BUILTIN_KSI_location")

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[
        typing.Literal["INSERTKEY_NEEDED", "INSERTKEY_VISUAL", "INSERTKEY_XYZ_TO_RGB"]
    ]
    """ Keying Set options to use when inserting keyframes

    :type: set[typing.Literal['INSERTKEY_NEEDED','INSERTKEY_VISUAL','INSERTKEY_XYZ_TO_RGB']]
    """

    def poll(self, context: Context | None) -> bool:
        """Test if Keying Set can be used or not

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """

    def iterator(self, context: Context | None, ks: KeyingSet | None):
        """Call generate() on the structs which have properties to be keyframed

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        """

    def generate(self, context: Context | None, ks: KeyingSet | None, data: typing.Any):
        """Add Paths to the Keying Set to keyframe the properties of the given data

        :param context:
        :type context: Context | None
        :param ks:
        :type ks: KeyingSet | None
        :param data:
        :type data: typing.Any
        """

class KeyingSetPath(bpy_struct):
    """Path to a setting for use in a Keying Set"""

    array_index: int
    """ Index to the specific setting if applicable

    :type: int
    """

    data_path: str
    """ Path to property setting

    :type: str
    """

    group: str
    """ Name of Action Group to assign setting(s) for this path to

    :type: str
    """

    group_method: typing.Literal["NAMED", "NONE", "KEYINGSET"]
    """ Method used to define which Group-name to use

    :type: typing.Literal['NAMED','NONE','KEYINGSET']
    """

    id: ID | None
    """ ID-Block that keyframes for Keying Set should be added to (for Absolute Keying Sets only)

    :type: ID | None
    """

    id_type: typing.Literal[
        "ACTION",
        "ARMATURE",
        "BRUSH",
        "CAMERA",
        "CACHEFILE",
        "CURVE",
        "FONT",
        "GREASEPENCIL",
        "GROUP",
        "IMAGE",
        "KEY",
        "LAMP",
        "LIBRARY",
        "LINESTYLE",
        "LATTICE",
        "MASK",
        "MATERIAL",
        "META",
        "MESH",
        "MOVIECLIP",
        "NODETREE",
        "OBJECT",
        "PAINTCURVE",
        "PALETTE",
        "PARTICLE",
        "SCENE",
        "SCREEN",
        "SOUND",
        "SPEAKER",
        "TEXT",
        "TEXTURE",
        "WINDOWMANAGER",
        "WORLD",
    ]
    """ Type of ID-block that can be used

    :type: typing.Literal['ACTION','ARMATURE','BRUSH','CAMERA','CACHEFILE','CURVE','FONT','GREASEPENCIL','GROUP','IMAGE','KEY','LAMP','LIBRARY','LINESTYLE','LATTICE','MASK','MATERIAL','META','MESH','MOVIECLIP','NODETREE','OBJECT','PAINTCURVE','PALETTE','PARTICLE','SCENE','SCREEN','SOUND','SPEAKER','TEXT','TEXTURE','WINDOWMANAGER','WORLD']
    """

    use_entire_array: bool
    """ When an 'array/vector' type is chosen (Location, Rotation, Color, etc.), entire array is to be used

    :type: bool
    """

    use_insertkey_needed: bool
    """ Only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_needed: bool
    """ Override default setting to only insert keyframes where they're needed in the relevant F-Curves

    :type: bool
    """

    use_insertkey_override_visual: bool
    """ Override default setting to insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_override_xyz_to_rgb: bool
    """ Override default setting to set color for newly added transformation F-Curves (Location, Rotation, Scale) to be based on the transform axis

    :type: bool
    """

    use_insertkey_visual: bool
    """ Insert keyframes based on 'visual transforms'

    :type: bool
    """

    use_insertkey_xyz_to_rgb: bool
    """ Color for newly added transformation F-Curves (Location, Rotation, Scale) is based on the transform axis

    :type: bool
    """

class KinematicConstraint(Constraint, bpy_struct):
    """Inverse Kinematics"""

    chain_count: int
    """ How many bones are included in the IK effect - 0 uses all bones

    :type: int
    """

    distance: float
    """ Radius of limiting sphere

    :type: float
    """

    ik_type: typing.Literal["COPY_POSE", "DISTANCE"]
    """ 

    :type: typing.Literal['COPY_POSE','DISTANCE']
    """

    iterations: int
    """ Maximum number of solving iterations

    :type: int
    """

    limit_mode: typing.Literal[
        "LIMITDIST_INSIDE", "LIMITDIST_OUTSIDE", "LIMITDIST_ONSURFACE"
    ]
    """ Distances in relation to sphere of influence to allow

    :type: typing.Literal['LIMITDIST_INSIDE','LIMITDIST_OUTSIDE','LIMITDIST_ONSURFACE']
    """

    lock_location_x: bool
    """ Constraint position along X axis

    :type: bool
    """

    lock_location_y: bool
    """ Constraint position along Y axis

    :type: bool
    """

    lock_location_z: bool
    """ Constraint position along Z axis

    :type: bool
    """

    lock_rotation_x: bool
    """ Constraint rotation along X axis

    :type: bool
    """

    lock_rotation_y: bool
    """ Constraint rotation along Y axis

    :type: bool
    """

    lock_rotation_z: bool
    """ Constraint rotation along Z axis

    :type: bool
    """

    orient_weight: float
    """ For Tree-IK: Weight of orientation control for this target

    :type: float
    """

    pole_angle: float
    """ Pole rotation offset

    :type: float
    """

    pole_subtarget: str
    """ 

    :type: str
    """

    pole_target: Object | None
    """ Object for pole rotation

    :type: Object | None
    """

    reference_axis: typing.Literal["BONE", "TARGET"]
    """ Constraint axis Lock options relative to Bone or Target reference

    :type: typing.Literal['BONE','TARGET']
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_location: bool
    """ Chain follows position of target

    :type: bool
    """

    use_rotation: bool
    """ Chain follows rotation of target

    :type: bool
    """

    use_stretch: bool
    """ Enable IK Stretching

    :type: bool
    """

    use_tail: bool
    """ Include bone's tail as last element in chain

    :type: bool
    """

    weight: float
    """ For Tree-IK: Weight of position control for this target

    :type: float
    """

class Lamp(ID, bpy_struct):
    """Lamp data-block for lighting a scene"""

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    color: bpy_prop_array[float]
    """ Light color

    :type: bpy_prop_array[float]
    """

    cycles: CyclesLampSettings
    """ Cycles lamp settings

    :type: CyclesLampSettings
    """

    distance: float
    """ Falloff distance - the light is at half the original intensity at this point

    :type: float
    """

    energy: float
    """ Amount of light that the lamp emits

    :type: float
    """

    node_tree: NodeTree
    """ Node tree for node based lamps

    :type: NodeTree
    """

    texture_slots: LampTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: LampTextureSlots
    """

    type: typing.Literal["POINT", "SUN", "SPOT", "HEMI", "AREA"]
    """ Type of Lamp

    :type: typing.Literal['POINT','SUN','SPOT','HEMI','AREA']
    """

    use_diffuse: bool
    """ Do diffuse shading

    :type: bool
    """

    use_negative: bool
    """ Cast negative light

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes to render the lamp

    :type: bool
    """

    use_own_layer: bool
    """ Illuminate objects only on the same layers the lamp is on

    :type: bool
    """

    use_specular: bool
    """ Create specular highlights

    :type: bool
    """

class LampSkySettings(bpy_struct):
    """Sky related settings for a sun lamp"""

    atmosphere_distance_factor: float
    """ Multiplier to convert blender units to physical distance

    :type: float
    """

    atmosphere_extinction: float
    """ Extinction scattering contribution factor

    :type: float
    """

    atmosphere_inscattering: float
    """ Scatter contribution factor

    :type: float
    """

    atmosphere_turbidity: float
    """ Sky turbidity

    :type: float
    """

    backscattered_light: float
    """ Backscattered light

    :type: float
    """

    horizon_brightness: float
    """ Horizon brightness

    :type: float
    """

    sky_blend: float
    """ Blend factor with sky

    :type: float
    """

    sky_blend_type: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Blend mode for combining sun sky with world sky

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    sky_color_space: typing.Literal["SMPTE", "REC709", "CIE"]
    """ Color space to use for internal XYZ->RGB color conversion

    :type: typing.Literal['SMPTE','REC709','CIE']
    """

    sky_exposure: float
    """ Strength of sky shading exponential exposure correction

    :type: float
    """

    spread: float
    """ Horizon Spread

    :type: float
    """

    sun_brightness: float
    """ Sun brightness

    :type: float
    """

    sun_intensity: float
    """ Sun intensity

    :type: float
    """

    sun_size: float
    """ Sun size

    :type: float
    """

    use_atmosphere: bool
    """ Apply sun effect on atmosphere

    :type: bool
    """

    use_sky: bool
    """ Apply sun effect on sky

    :type: bool
    """

class LampTextureSlot(TextureSlot, bpy_struct):
    """Texture slot for textures in a Lamp data-block"""

    color_factor: float
    """ Amount texture affects color values

    :type: float
    """

    object: Object | None
    """ Object to use for mapping with Object texture coordinates

    :type: Object | None
    """

    shadow_factor: float
    """ Amount texture affects shadow

    :type: float
    """

    texture_coords: typing.Literal["GLOBAL", "VIEW", "OBJECT"]
    """ 

    :type: typing.Literal['GLOBAL','VIEW','OBJECT']
    """

    use_map_color: bool
    """ Let the texture affect the basic color of the lamp

    :type: bool
    """

    use_map_shadow: bool
    """ Let the texture affect the shadow color of the lamp

    :type: bool
    """

class LaplacianDeformModifier(Modifier, bpy_struct):
    """Mesh deform modifier"""

    is_bind: bool
    """ Whether geometry has been bound to anchors

    :type: bool
    """

    iterations: int
    """ 

    :type: int
    """

    vertex_group: str
    """ Name of Vertex Group which determines Anchors

    :type: str
    """

class LaplacianSmoothModifier(Modifier, bpy_struct):
    """Smoothing effect modifier"""

    iterations: int
    """ 

    :type: int
    """

    lambda_border: float
    """ Lambda factor in border

    :type: float
    """

    lambda_factor: float
    """ Smooth factor effect

    :type: float
    """

    use_normalized: bool
    """ Improve and stabilize the enhanced shape

    :type: bool
    """

    use_volume_preserve: bool
    """ Apply volume preservation after smooth

    :type: bool
    """

    use_x: bool
    """ Smooth object along X axis

    :type: bool
    """

    use_y: bool
    """ Smooth object along Y axis

    :type: bool
    """

    use_z: bool
    """ Smooth object along Z axis

    :type: bool
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class Lattice(ID, bpy_struct):
    """Lattice data-block defining a grid for deforming other objects"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    interpolation_type_u: typing.Literal[
        "KEY_LINEAR", "KEY_CARDINAL", "KEY_CATMULL_ROM", "KEY_BSPLINE"
    ]
    """ 

    :type: typing.Literal['KEY_LINEAR','KEY_CARDINAL','KEY_CATMULL_ROM','KEY_BSPLINE']
    """

    interpolation_type_v: typing.Literal[
        "KEY_LINEAR", "KEY_CARDINAL", "KEY_CATMULL_ROM", "KEY_BSPLINE"
    ]
    """ 

    :type: typing.Literal['KEY_LINEAR','KEY_CARDINAL','KEY_CATMULL_ROM','KEY_BSPLINE']
    """

    interpolation_type_w: typing.Literal[
        "KEY_LINEAR", "KEY_CARDINAL", "KEY_CATMULL_ROM", "KEY_BSPLINE"
    ]
    """ 

    :type: typing.Literal['KEY_LINEAR','KEY_CARDINAL','KEY_CATMULL_ROM','KEY_BSPLINE']
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    points: bpy_prop_collection[LatticePoint]
    """ Points of the lattice

    :type: bpy_prop_collection[LatticePoint]
    """

    points_u: int
    """ Point in U direction (can't be changed when there are shape keys)

    :type: int
    """

    points_v: int
    """ Point in V direction (can't be changed when there are shape keys)

    :type: int
    """

    points_w: int
    """ Point in W direction (can't be changed when there are shape keys)

    :type: int
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    use_outside: bool
    """ Only draw, and take into account, the outer vertices

    :type: bool
    """

    vertex_group: str
    """ Vertex group to apply the influence of the lattice

    :type: str
    """

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | None = False,
    ):
        """Transform lattice by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | None
        """

class LatticeModifier(Modifier, bpy_struct):
    """Lattice deformation modifier"""

    object: Object | None
    """ Lattice object to deform with

    :type: Object | None
    """

    strength: float
    """ Strength of modifier effect

    :type: float
    """

    vertex_group: str
    """ Name of Vertex Group which determines influence of modifier per point

    :type: str
    """

class LatticePoint(bpy_struct):
    """Point in the lattice grid"""

    co: bpy_prop_array[float]
    """ Original undeformed location used to calculate the strength of the deform effect (edit/animate the Deformed Location instead)

    :type: bpy_prop_array[float]
    """

    co_deform: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this point is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight_softbody: float
    """ Softbody goal weight

    :type: float
    """

class Library(ID, bpy_struct):
    """External .blend file from which data is linked"""

    filepath: str
    """ Path to the library .blend file

    :type: str
    """

    packed_file: PackedFile
    """ 

    :type: PackedFile
    """

    parent: typing_extensions.Self
    """ 

    :type: typing_extensions.Self
    """

    users_id: typing.Any
    """ ID data blocks which use this library
(readonly)"""

    def reload(self):
        """Reload this library and all its linked datablocks"""

class LimitDistanceConstraint(Constraint, bpy_struct):
    """Limit the distance from target object"""

    distance: float
    """ Radius of limiting sphere

    :type: float
    """

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    limit_mode: typing.Literal[
        "LIMITDIST_INSIDE", "LIMITDIST_OUTSIDE", "LIMITDIST_ONSURFACE"
    ]
    """ Distances in relation to sphere of influence to allow

    :type: typing.Literal['LIMITDIST_INSIDE','LIMITDIST_OUTSIDE','LIMITDIST_ONSURFACE']
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitLocationConstraint(Constraint, bpy_struct):
    """Limit the location of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_max_z: bool
    """ Use the maximum Z value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_min_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitRotationConstraint(Constraint, bpy_struct):
    """Limit the rotation of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_limit_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_limit_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_limit_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LimitScaleConstraint(Constraint, bpy_struct):
    """Limit the scaling of the constrained object"""

    max_x: float
    """ Highest X value to allow

    :type: float
    """

    max_y: float
    """ Highest Y value to allow

    :type: float
    """

    max_z: float
    """ Highest Z value to allow

    :type: float
    """

    min_x: float
    """ Lowest X value to allow

    :type: float
    """

    min_y: float
    """ Lowest Y value to allow

    :type: float
    """

    min_z: float
    """ Lowest Z value to allow

    :type: float
    """

    use_max_x: bool
    """ Use the maximum X value

    :type: bool
    """

    use_max_y: bool
    """ Use the maximum Y value

    :type: bool
    """

    use_max_z: bool
    """ Use the maximum Z value

    :type: bool
    """

    use_min_x: bool
    """ Use the minimum X value

    :type: bool
    """

    use_min_y: bool
    """ Use the minimum Y value

    :type: bool
    """

    use_min_z: bool
    """ Use the minimum Z value

    :type: bool
    """

    use_transform_limit: bool
    """ Transforms are affected by this constraint as well

    :type: bool
    """

class LineStyleAlphaModifier(LineStyleModifier, bpy_struct):
    """Base type to define alpha transparency modifiers"""

class LineStyleAlphaModifier_AlongStroke(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Change alpha transparency along stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_CreaseAngle(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Alpha transparency based on the angle between two adjacent faces"""

    angle_max: float
    """ Maximum angle to modify thickness

    :type: float
    """

    angle_min: float
    """ Minimum angle to modify thickness

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_Curvature_3D(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Alpha transparency based on the radial curvature of 3D mesh surfaces"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curvature_max: float
    """ Maximum Curvature

    :type: float
    """

    curvature_min: float
    """ Minimum Curvature

    :type: float
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_DistanceFromCamera(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Change alpha transparency based on the distance from the camera"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_DistanceFromObject(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Change alpha transparency based on the distance from an object"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    target: Object | None
    """ Target object from which the distance is measured

    :type: Object | None
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_Material(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Change alpha transparency based on a material attribute"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    material_attribute: typing.Literal[
        "LINE",
        "LINE_R",
        "LINE_G",
        "LINE_B",
        "LINE_A",
        "DIFF",
        "DIFF_R",
        "DIFF_G",
        "DIFF_B",
        "SPEC",
        "SPEC_R",
        "SPEC_G",
        "SPEC_B",
        "SPEC_HARD",
        "ALPHA",
    ]
    """ Specify which material attribute is used

    :type: typing.Literal['LINE','LINE_R','LINE_G','LINE_B','LINE_A','DIFF','DIFF_R','DIFF_G','DIFF_B','SPEC','SPEC_R','SPEC_G','SPEC_B','SPEC_HARD','ALPHA']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_Noise(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Alpha transparency based on random noise"""

    amplitude: float
    """ Amplitude of the noise

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    period: float
    """ Period of the noise

    :type: float
    """

    seed: int
    """ Seed for the noise generation

    :type: int
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleAlphaModifier_Tangent(
    LineStyleAlphaModifier, LineStyleModifier, bpy_struct
):
    """Alpha transparency based on the direction of the stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier(LineStyleModifier, bpy_struct):
    """Base type to define line color modifiers"""

class LineStyleColorModifier_AlongStroke(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color along stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_CreaseAngle(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on the underlying crease angle"""

    angle_max: float
    """ Maximum angle to modify thickness

    :type: float
    """

    angle_min: float
    """ Minimum angle to modify thickness

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_Curvature_3D(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on the radial curvature of 3D mesh surfaces"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    curvature_max: float
    """ Maximum Curvature

    :type: float
    """

    curvature_min: float
    """ Minimum Curvature

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_DistanceFromCamera(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on the distance from the camera"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_DistanceFromObject(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on the distance from an object"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    target: Object | None
    """ Target object from which the distance is measured

    :type: Object | None
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_Material(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on a material attribute"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    material_attribute: typing.Literal[
        "LINE",
        "LINE_R",
        "LINE_G",
        "LINE_B",
        "LINE_A",
        "DIFF",
        "DIFF_R",
        "DIFF_G",
        "DIFF_B",
        "SPEC",
        "SPEC_R",
        "SPEC_G",
        "SPEC_B",
        "SPEC_HARD",
        "ALPHA",
    ]
    """ Specify which material attribute is used

    :type: typing.Literal['LINE','LINE_R','LINE_G','LINE_B','LINE_A','DIFF','DIFF_R','DIFF_G','DIFF_B','SPEC','SPEC_R','SPEC_G','SPEC_B','SPEC_HARD','ALPHA']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    use_ramp: bool
    """ Use color ramp to map the BW average into an RGB color

    :type: bool
    """

class LineStyleColorModifier_Noise(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on random noise"""

    amplitude: float
    """ Amplitude of the noise

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    period: float
    """ Period of the noise

    :type: float
    """

    seed: int
    """ Seed for the noise generation

    :type: int
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleColorModifier_Tangent(
    LineStyleColorModifier, LineStyleModifier, bpy_struct
):
    """Change line color based on the direction of a stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    color_ramp: ColorRamp
    """ Color ramp used to change line color

    :type: ColorRamp
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier(LineStyleModifier, bpy_struct):
    """Base type to define stroke geometry modifiers"""

class LineStyleGeometryModifier_2DOffset(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Add two-dimensional offsets to stroke backbone geometry"""

    end: float
    """ Displacement that is applied from the end of the stroke

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    start: float
    """ Displacement that is applied from the beginning of the stroke

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    x: float
    """ Displacement that is applied to the X coordinates of stroke vertices

    :type: float
    """

    y: float
    """ Displacement that is applied to the Y coordinates of stroke vertices

    :type: float
    """

class LineStyleGeometryModifier_2DTransform(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Apply two-dimensional scaling and rotation to stroke backbone geometry"""

    angle: float
    """ Rotation angle

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    pivot: typing.Literal["CENTER", "START", "END", "PARAM", "ABSOLUTE"]
    """ Pivot of scaling and rotation operations

    :type: typing.Literal['CENTER','START','END','PARAM','ABSOLUTE']
    """

    pivot_u: float
    """ Pivot in terms of the stroke point parameter u (0 <= u <= 1)

    :type: float
    """

    pivot_x: float
    """ 2D X coordinate of the absolute pivot

    :type: float
    """

    pivot_y: float
    """ 2D Y coordinate of the absolute pivot

    :type: float
    """

    scale_x: float
    """ Scaling factor that is applied along the X axis

    :type: float
    """

    scale_y: float
    """ Scaling factor that is applied along the Y axis

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_BackboneStretcher(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Stretch the beginning and the end of stroke backbone"""

    backbone_length: float
    """ Amount of backbone stretching

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_BezierCurve(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Replace stroke backbone geometry by a Bezier curve approximation of the original backbone geometry"""

    error: float
    """ Maximum distance allowed between the new Bezier curve and the original backbone geometry

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_Blueprint(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Produce a blueprint using circular, elliptic, and square contour strokes"""

    backbone_length: float
    """ Amount of backbone stretching

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    random_backbone: int
    """ Randomness of the backbone stretching

    :type: int
    """

    random_center: int
    """ Randomness of the center

    :type: int
    """

    random_radius: int
    """ Randomness of the radius

    :type: int
    """

    rounds: int
    """ Number of rounds in contour strokes

    :type: int
    """

    shape: typing.Literal["CIRCLES", "ELLIPSES", "SQUARES"]
    """ Select the shape of blueprint contour strokes

    :type: typing.Literal['CIRCLES','ELLIPSES','SQUARES']
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_GuidingLines(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Modify the stroke geometry so that it corresponds to its main direction line"""

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    offset: float
    """ Displacement that is applied to the main direction line along its normal

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_PerlinNoise1D(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Add one-dimensional Perlin noise to stroke backbone geometry"""

    amplitude: float
    """ Amplitude of the Perlin noise

    :type: float
    """

    angle: float
    """ Displacement direction

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    frequency: float
    """ Frequency of the Perlin noise

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    octaves: int
    """ Number of octaves (i.e., the amount of detail of the Perlin noise)

    :type: int
    """

    seed: int
    """ Seed for random number generation (if negative, time is used as a seed instead)

    :type: int
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_PerlinNoise2D(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Add two-dimensional Perlin noise to stroke backbone geometry"""

    amplitude: float
    """ Amplitude of the Perlin noise

    :type: float
    """

    angle: float
    """ Displacement direction

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    frequency: float
    """ Frequency of the Perlin noise

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    octaves: int
    """ Number of octaves (i.e., the amount of detail of the Perlin noise)

    :type: int
    """

    seed: int
    """ Seed for random number generation (if negative, time is used as a seed instead)

    :type: int
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_Polygonalization(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Modify the stroke geometry so that it looks more 'polygonal'"""

    error: float
    """ Maximum distance between the original stroke and its polygonal approximation

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_Sampling(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Specify a new sampling value that determines the resolution of stroke polylines"""

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    sampling: float
    """ New sampling value to be used for subsequent modifiers

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_Simplification(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Simplify the stroke set"""

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    tolerance: float
    """ Distance below which segments will be merged

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleGeometryModifier_SinusDisplacement(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Add sinus displacement to stroke backbone geometry"""

    amplitude: float
    """ Amplitude of the sinus displacement

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    phase: float
    """ Phase of the sinus displacement

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    wavelength: float
    """ Wavelength of the sinus displacement

    :type: float
    """

class LineStyleGeometryModifier_SpatialNoise(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Add spatial noise to stroke backbone geometry"""

    amplitude: float
    """ Amplitude of the spatial noise

    :type: float
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    octaves: int
    """ Number of octaves (i.e., the amount of detail of the spatial noise)

    :type: int
    """

    scale: float
    """ Scale of the spatial noise

    :type: float
    """

    smooth: bool
    """ If true, the spatial noise is smooth

    :type: bool
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    use_pure_random: bool
    """ If true, the spatial noise does not show any coherence

    :type: bool
    """

class LineStyleGeometryModifier_TipRemover(
    LineStyleGeometryModifier, LineStyleModifier, bpy_struct
):
    """Remove a piece of stroke at the beginning and the end of stroke backbone"""

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    tip_length: float
    """ Length of tips to be removed

    :type: float
    """

    type: typing.Literal[
        "2D_OFFSET",
        "2D_TRANSFORM",
        "BACKBONE_STRETCHER",
        "BEZIER_CURVE",
        "BLUEPRINT",
        "GUIDING_LINES",
        "PERLIN_NOISE_1D",
        "PERLIN_NOISE_2D",
        "POLYGONIZATION",
        "SAMPLING",
        "SIMPLIFICATION",
        "SINUS_DISPLACEMENT",
        "SPATIAL_NOISE",
        "TIP_REMOVER",
    ]
    """ Type of the modifier

    :type: typing.Literal['2D_OFFSET','2D_TRANSFORM','BACKBONE_STRETCHER','BEZIER_CURVE','BLUEPRINT','GUIDING_LINES','PERLIN_NOISE_1D','PERLIN_NOISE_2D','POLYGONIZATION','SAMPLING','SIMPLIFICATION','SINUS_DISPLACEMENT','SPATIAL_NOISE','TIP_REMOVER']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleModifier(bpy_struct):
    """Base type to define modifiers"""

class LineStyleTextureSlot(TextureSlot, bpy_struct):
    """Texture slot for textures in a LineStyle data-block"""

    alpha_factor: float
    """ Amount texture affects alpha

    :type: float
    """

    diffuse_color_factor: float
    """ Amount texture affects diffuse color

    :type: float
    """

    mapping: typing.Literal["FLAT", "CUBE", "TUBE", "SPHERE"]
    """ 

    :type: typing.Literal['FLAT','CUBE','TUBE','SPHERE']
    """

    mapping_x: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    mapping_y: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    mapping_z: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    texture_coords: typing.Literal["WINDOW", "GLOBAL", "ALONG_STROKE", "ORCO"]
    """ Texture coordinates used to map the texture onto the background

    :type: typing.Literal['WINDOW','GLOBAL','ALONG_STROKE','ORCO']
    """

    use_map_alpha: bool
    """ The texture affects the alpha value

    :type: bool
    """

    use_map_color_diffuse: bool
    """ The texture affects basic color of the stroke

    :type: bool
    """

    use_tips: bool
    """ Lower half of the texture is for tips of the stroke

    :type: bool
    """

class LineStyleThicknessModifier(LineStyleModifier, bpy_struct):
    """Base type to define line thickness modifiers"""

class LineStyleThicknessModifier_AlongStroke(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Change line thickness along stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    value_max: float
    """ Maximum output value of the mapping

    :type: float
    """

    value_min: float
    """ Minimum output value of the mapping

    :type: float
    """

class LineStyleThicknessModifier_Calligraphy(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Change line thickness so that stroke looks like made with a calligraphic pen"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    orientation: float
    """ Angle of the main direction

    :type: float
    """

    thickness_max: float
    """ Maximum thickness in the main direction

    :type: float
    """

    thickness_min: float
    """ Minimum thickness in the direction perpendicular to the main direction

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleThicknessModifier_CreaseAngle(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Line thickness based on the angle between two adjacent faces"""

    angle_max: float
    """ Maximum angle to modify thickness

    :type: float
    """

    angle_min: float
    """ Minimum angle to modify thickness

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    thickness_max: float
    """ Maximum thickness

    :type: float
    """

    thickness_min: float
    """ Minimum thickness

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleThicknessModifier_Curvature_3D(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Line thickness based on the radial curvature of 3D mesh surfaces"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curvature_max: float
    """ Maximum Curvature

    :type: float
    """

    curvature_min: float
    """ Minimum Curvature

    :type: float
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    thickness_max: float
    """ Maximum thickness

    :type: float
    """

    thickness_min: float
    """ Minimum thickness

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LineStyleThicknessModifier_DistanceFromCamera(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Change line thickness based on the distance from the camera"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    value_max: float
    """ Maximum output value of the mapping

    :type: float
    """

    value_min: float
    """ Minimum output value of the mapping

    :type: float
    """

class LineStyleThicknessModifier_DistanceFromObject(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Change line thickness based on the distance from an object"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    range_max: float
    """ Upper bound of the input range the mapping is applied

    :type: float
    """

    range_min: float
    """ Lower bound of the input range the mapping is applied

    :type: float
    """

    target: Object | None
    """ Target object from which the distance is measured

    :type: Object | None
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    value_max: float
    """ Maximum output value of the mapping

    :type: float
    """

    value_min: float
    """ Minimum output value of the mapping

    :type: float
    """

class LineStyleThicknessModifier_Material(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Change line thickness based on a material attribute"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    material_attribute: typing.Literal[
        "LINE",
        "LINE_R",
        "LINE_G",
        "LINE_B",
        "LINE_A",
        "DIFF",
        "DIFF_R",
        "DIFF_G",
        "DIFF_B",
        "SPEC",
        "SPEC_R",
        "SPEC_G",
        "SPEC_B",
        "SPEC_HARD",
        "ALPHA",
    ]
    """ Specify which material attribute is used

    :type: typing.Literal['LINE','LINE_R','LINE_G','LINE_B','LINE_A','DIFF','DIFF_R','DIFF_G','DIFF_B','SPEC','SPEC_R','SPEC_G','SPEC_B','SPEC_HARD','ALPHA']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    value_max: float
    """ Maximum output value of the mapping

    :type: float
    """

    value_min: float
    """ Minimum output value of the mapping

    :type: float
    """

class LineStyleThicknessModifier_Noise(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Line thickness based on random noise"""

    amplitude: float
    """ Amplitude of the noise

    :type: float
    """

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    period: float
    """ Period of the noise

    :type: float
    """

    seed: int
    """ Seed for the noise generation

    :type: int
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

    use_asymmetric: bool
    """ Allow thickness to be assigned asymmetrically

    :type: bool
    """

class LineStyleThicknessModifier_Tangent(
    LineStyleThicknessModifier, LineStyleModifier, bpy_struct
):
    """Thickness based on the direction of the stroke"""

    blend: typing.Literal[
        "MIX",
        "ADD",
        "SUBTRACT",
        "MULTIPLY",
        "DIVIDE",
        "DIFFERENCE",
        "MININUM",
        "MAXIMUM",
    ]
    """ Specify how the modifier value is blended into the base value

    :type: typing.Literal['MIX','ADD','SUBTRACT','MULTIPLY','DIVIDE','DIFFERENCE','MININUM','MAXIMUM']
    """

    curve: CurveMapping
    """ Curve used for the curve mapping

    :type: CurveMapping
    """

    expanded: bool
    """ True if the modifier tab is expanded

    :type: bool
    """

    influence: float
    """ Influence factor by which the modifier changes the property

    :type: float
    """

    invert: bool
    """ Invert the fade-out direction of the linear mapping

    :type: bool
    """

    mapping: typing.Literal["LINEAR", "CURVE"]
    """ Select the mapping type

    :type: typing.Literal['LINEAR','CURVE']
    """

    name: str
    """ Name of the modifier

    :type: str
    """

    thickness_max: float
    """ Maximum thickness

    :type: float
    """

    thickness_min: float
    """ Minimum thickness

    :type: float
    """

    type: typing.Literal[
        "ALONG_STROKE",
        "CALLIGRAPHY",
        "CREASE_ANGLE",
        "CURVATURE_3D",
        "DISTANCE_FROM_CAMERA",
        "DISTANCE_FROM_OBJECT",
        "MATERIAL",
        "NOISE",
        "TANGENT",
    ]
    """ Type of the modifier

    :type: typing.Literal['ALONG_STROKE','CALLIGRAPHY','CREASE_ANGLE','CURVATURE_3D','DISTANCE_FROM_CAMERA','DISTANCE_FROM_OBJECT','MATERIAL','NOISE','TANGENT']
    """

    use: bool
    """ Enable or disable this modifier during stroke rendering

    :type: bool
    """

class LocRotScale(KeyingSetInfo, bpy_struct):
    def generate(self, context, ks, data):
        """

        :param context:
        :param ks:
        :param data:
        """

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """

class Location(KeyingSetInfo, bpy_struct):
    @staticmethod
    def generate(ksi, context, ks, data):
        """

        :param ksi:
        :param context:
        :param ks:
        :param data:
        """

    @staticmethod
    def iterator(ksi, context, ks):
        """

        :param ksi:
        :param context:
        :param ks:
        """

    @staticmethod
    def poll(ksi, context):
        """

        :param ksi:
        :param context:
        """

class LockedTrackConstraint(Constraint, bpy_struct):
    """Point toward the target along the track axis, while locking the other axis"""

    head_tail: float
    """ Target along length of bone: Head=0, Tail=1

    :type: float
    """

    lock_axis: typing.Literal["LOCK_X", "LOCK_Y", "LOCK_Z"]
    """ Axis that points upward

    :type: typing.Literal['LOCK_X','LOCK_Y','LOCK_Z']
    """

    subtarget: str
    """ 

    :type: str
    """

    target: Object | None
    """ Target Object

    :type: Object | None
    """

    track_axis: typing.Literal[
        "TRACK_X",
        "TRACK_Y",
        "TRACK_Z",
        "TRACK_NEGATIVE_X",
        "TRACK_NEGATIVE_Y",
        "TRACK_NEGATIVE_Z",
    ]
    """ Axis that points to the target object

    :type: typing.Literal['TRACK_X','TRACK_Y','TRACK_Z','TRACK_NEGATIVE_X','TRACK_NEGATIVE_Y','TRACK_NEGATIVE_Z']
    """

    use_bbone_shape: bool
    """ Follow shape of B-Bone segments when calculating Head/Tail position

    :type: bool
    """

class LodLevel(bpy_struct):
    distance: float
    """ Distance to begin using this level of detail

    :type: float
    """

    object: Object | None
    """ Object to use for this level of detail

    :type: Object | None
    """

    object_hysteresis_percentage: int
    """ Minimum distance change required to transition to the previous level of detail

    :type: int
    """

    use_material: bool
    """ Use the material from this object at this level of detail

    :type: bool
    """

    use_mesh: bool
    """ Use the mesh from this object at this level of detail

    :type: bool
    """

    use_object_hysteresis: bool
    """ Override LoD Hysteresis scene setting for this LoD level

    :type: bool
    """

class MASK_UL_layers(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class MATERIAL_UL_matslots(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class MESH_UL_shape_keys(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class MESH_UL_uvmaps_vcols(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class MESH_UL_vgroups(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class Macro(bpy_struct):
    """Storage of a macro operator being executed, or registered after execution"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ 

    :type: str
    """

    bl_options: set[
        typing.Literal[
            "REGISTER", "UNDO", "BLOCKING", "MACRO", "GRAB_CURSOR", "PRESET", "INTERNAL"
        ]
    ]
    """ Options for this operator type

    :type: set[typing.Literal['REGISTER','UNDO','BLOCKING','MACRO','GRAB_CURSOR','PRESET','INTERNAL']]
    """

    bl_translation_context: str
    """ 

    :type: str
    """

    name: str
    """ 

    :type: str
    """

    properties: OperatorProperties
    """ 

    :type: OperatorProperties
    """

    def report(
        self,
        type: set[
            typing.Literal[
                "DEBUG",
                "INFO",
                "OPERATOR",
                "PROPERTY",
                "WARNING",
                "ERROR",
                "ERROR_INVALID_INPUT",
                "ERROR_INVALID_CONTEXT",
                "ERROR_OUT_OF_MEMORY",
            ]
        ]
        | None,
        message: str,
    ):
        """report

        :param type: Type
        :type type: set[typing.Literal['DEBUG','INFO','OPERATOR','PROPERTY','WARNING','ERROR','ERROR_INVALID_INPUT','ERROR_INVALID_CONTEXT','ERROR_OUT_OF_MEMORY']] | None
        :param message: Report Message
        :type message: str
        """

    @classmethod
    def poll(cls, context: Context | None) -> bool:
        """Test if the operator can be called or not

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """

    def draw(self, context: Context | None):
        """Draw function for the operator

        :param context:
        :type context: Context | None
        """

class MagicTexture(Texture, ID, bpy_struct):
    """Procedural noise texture"""

    noise_depth: int
    """ Depth of the noise

    :type: int
    """

    turbulence: float
    """ Turbulence of the noise

    :type: float
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class MaintainVolumeConstraint(Constraint, bpy_struct):
    """Maintain a constant volume along a single scaling axis"""

    free_axis: typing.Literal["SAMEVOL_X", "SAMEVOL_Y", "SAMEVOL_Z"]
    """ The free scaling axis of the object

    :type: typing.Literal['SAMEVOL_X','SAMEVOL_Y','SAMEVOL_Z']
    """

    volume: float
    """ Volume of the bone at rest

    :type: float
    """

class MarbleTexture(Texture, ID, bpy_struct):
    """Procedural noise texture"""

    marble_type: typing.Literal["SOFT", "SHARP", "SHARPER"]
    """ 

    :type: typing.Literal['SOFT','SHARP','SHARPER']
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_basis_2: typing.Literal["SIN", "SAW", "TRI"]
    """ 

    :type: typing.Literal['SIN','SAW','TRI']
    """

    noise_depth: int
    """ Depth of the cloud calculation

    :type: int
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    noise_type: typing.Literal["SOFT_NOISE", "HARD_NOISE"]
    """ 

    :type: typing.Literal['SOFT_NOISE','HARD_NOISE']
    """

    turbulence: float
    """ Turbulence of the bandnoise and ringnoise types

    :type: float
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class Mask(ID, bpy_struct):
    """Mask data-block defining mask for compositing"""

    active_layer_index: int | None
    """ Index of active layer in list of all mask's layers

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    frame_end: int
    """ Final frame of the mask (used for sequencer)

    :type: int
    """

    frame_start: int
    """ First frame of the mask (used for sequencer)

    :type: int
    """

    layers: MaskLayers
    """ Collection of layers which defines this mask

    :type: MaskLayers
    """

class MaskLayer(bpy_struct):
    """Single layer used for masking pixels"""

    alpha: float
    """ Render Opacity

    :type: float
    """

    blend: typing.Literal[
        "MERGE_ADD",
        "MERGE_SUBTRACT",
        "ADD",
        "SUBTRACT",
        "LIGHTEN",
        "DARKEN",
        "MUL",
        "REPLACE",
        "DIFFERENCE",
    ]
    """ Method of blending mask layers

    :type: typing.Literal['MERGE_ADD','MERGE_SUBTRACT','ADD','SUBTRACT','LIGHTEN','DARKEN','MUL','REPLACE','DIFFERENCE']
    """

    falloff: typing.Literal[
        "SMOOTH", "SPHERE", "ROOT", "INVERSE_SQUARE", "SHARP", "LINEAR"
    ]
    """ Falloff type the feather

    :type: typing.Literal['SMOOTH','SPHERE','ROOT','INVERSE_SQUARE','SHARP','LINEAR']
    """

    hide: bool
    """ Restrict visibility in the viewport

    :type: bool
    """

    hide_render: bool
    """ Restrict renderability

    :type: bool
    """

    hide_select: bool
    """ Restrict selection in the viewport

    :type: bool
    """

    invert: bool
    """ Invert the mask black/white

    :type: bool
    """

    name: str
    """ Unique name of layer

    :type: str
    """

    select: bool
    """ Layer is selected for editing in the Dope Sheet

    :type: bool
    """

    splines: MaskSplines
    """ Collection of splines which defines this layer

    :type: MaskSplines
    """

    use_fill_holes: bool
    """ Calculate holes when filling overlapping curves

    :type: bool
    """

    use_fill_overlap: bool
    """ Calculate self intersections and overlap before filling

    :type: bool
    """

class MaskModifier(Modifier, bpy_struct):
    """Mask modifier to hide parts of the mesh"""

    armature: Object | None
    """ Armature to use as source of bones to mask

    :type: Object | None
    """

    invert_vertex_group: bool
    """ Use vertices that are not part of region defined

    :type: bool
    """

    mode: typing.Literal["VERTEX_GROUP", "ARMATURE"]
    """ 

    :type: typing.Literal['VERTEX_GROUP','ARMATURE']
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class MaskParent(bpy_struct):
    """Parenting settings for masking element"""

    id: ID | None
    """ ID-block to which masking element would be parented to or to it's property

    :type: ID | None
    """

    id_type: typing.Literal["MOVIECLIP"]
    """ Type of ID-block that can be used

    :type: typing.Literal['MOVIECLIP']
    """

    parent: str
    """ Name of parent object in specified data block to which parenting happens

    :type: str
    """

    sub_parent: str
    """ Name of parent sub-object in specified data block to which parenting happens

    :type: str
    """

    type: typing.Literal["POINT_TRACK", "PLANE_TRACK"]
    """ Parent Type

    :type: typing.Literal['POINT_TRACK','PLANE_TRACK']
    """

class MaskSequence(Sequence, bpy_struct):
    """Sequence strip to load a video from a mask"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    mask: Mask | None
    """ Mask that this sequence uses

    :type: Mask | None
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MaskSpline(bpy_struct):
    """Single spline used for defining mask shape"""

    offset_mode: typing.Literal["EVEN", "SMOOTH"]
    """ The method used for calculating the feather offset

    :type: typing.Literal['EVEN','SMOOTH']
    """

    points: MaskSplinePoints
    """ Collection of points

    :type: MaskSplinePoints
    """

    use_cyclic: bool
    """ Make this spline a closed loop

    :type: bool
    """

    use_fill: bool
    """ Make this spline filled

    :type: bool
    """

    use_self_intersection_check: bool
    """ Prevent feather from self-intersections

    :type: bool
    """

    weight_interpolation: typing.Literal["LINEAR", "EASE"]
    """ The type of weight interpolation for spline

    :type: typing.Literal['LINEAR','EASE']
    """

class MaskSplinePoint(bpy_struct):
    """Single point in spline used for defining mask"""

    co: bpy_prop_array[float]
    """ Coordinates of the control point

    :type: bpy_prop_array[float]
    """

    feather_points: bpy_prop_collection[MaskSplinePointUW]
    """ Points defining feather

    :type: bpy_prop_collection[MaskSplinePointUW]
    """

    handle_left: bpy_prop_array[float]
    """ Coordinates of the first handle

    :type: bpy_prop_array[float]
    """

    handle_left_type: typing.Literal[
        "AUTO", "VECTOR", "ALIGNED", "ALIGNED_DOUBLESIDE", "FREE"
    ]
    """ Handle type

    :type: typing.Literal['AUTO','VECTOR','ALIGNED','ALIGNED_DOUBLESIDE','FREE']
    """

    handle_right: bpy_prop_array[float]
    """ Coordinates of the second handle

    :type: bpy_prop_array[float]
    """

    handle_right_type: typing.Literal[
        "AUTO", "VECTOR", "ALIGNED", "ALIGNED_DOUBLESIDE", "FREE"
    ]
    """ Handle type

    :type: typing.Literal['AUTO','VECTOR','ALIGNED','ALIGNED_DOUBLESIDE','FREE']
    """

    handle_type: typing.Literal[
        "AUTO", "VECTOR", "ALIGNED", "ALIGNED_DOUBLESIDE", "FREE"
    ]
    """ Handle type

    :type: typing.Literal['AUTO','VECTOR','ALIGNED','ALIGNED_DOUBLESIDE','FREE']
    """

    parent: MaskParent
    """ 

    :type: MaskParent
    """

    select: bool
    """ Selection status

    :type: bool
    """

    weight: float
    """ Weight of the point

    :type: float
    """

class MaskSplinePointUW(bpy_struct):
    """Single point in spline segment defining feather"""

    select: bool
    """ Selection status

    :type: bool
    """

    u: float
    """ U coordinate of point along spline segment

    :type: float
    """

    weight: float
    """ Weight of feather point

    :type: float
    """

class Material(ID, bpy_struct):
    """Material data-block to define the appearance of geometric objects for rendering"""

    active_node_material: typing_extensions.Self | None
    """ Active node material

    :type: typing_extensions.Self | None
    """

    active_texture: Texture | None
    """ Active texture slot being displayed

    :type: Texture | None
    """

    active_texture_index: int | None
    """ Index of active texture slot

    :type: int | None
    """

    alpha: float
    """ Alpha transparency of the material

    :type: float
    """

    ambient: float
    """ Amount of global ambient color the material receives

    :type: float
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    cycles: CyclesMaterialSettings
    """ Cycles material settings

    :type: CyclesMaterialSettings
    """

    darkness: float
    """ Minnaert darkness

    :type: float
    """

    diffuse_color: bpy_prop_array[float]
    """ Diffuse color of the material

    :type: bpy_prop_array[float]
    """

    diffuse_fresnel: float
    """ Power of Fresnel

    :type: float
    """

    diffuse_fresnel_factor: float
    """ Blending factor of Fresnel

    :type: float
    """

    diffuse_intensity: float
    """ Amount of diffuse reflection

    :type: float
    """

    diffuse_ramp: ColorRamp
    """ Color ramp used to affect diffuse shading

    :type: ColorRamp
    """

    diffuse_ramp_blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Blending method of the ramp and the diffuse color

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    diffuse_ramp_factor: float
    """ Blending factor (also uses alpha in Colorband)

    :type: float
    """

    diffuse_ramp_input: typing.Literal["SHADER", "ENERGY", "NORMAL", "RESULT"]
    """ How the ramp maps on the surface

    :type: typing.Literal['SHADER','ENERGY','NORMAL','RESULT']
    """

    diffuse_shader: typing.Literal[
        "LAMBERT", "OREN_NAYAR", "TOON", "MINNAERT", "FRESNEL"
    ]
    """ 

    :type: typing.Literal['LAMBERT','OREN_NAYAR','TOON','MINNAERT','FRESNEL']
    """

    diffuse_toon_size: float
    """ Size of diffuse toon area

    :type: float
    """

    diffuse_toon_smooth: float
    """ Smoothness of diffuse toon area

    :type: float
    """

    emit: float
    """ Amount of light to emit

    :type: float
    """

    game_settings: MaterialGameSettings
    """ Game material settings

    :type: MaterialGameSettings
    """

    halo: MaterialHalo
    """ Halo settings for the material

    :type: MaterialHalo
    """

    invert_z: bool
    """ Render material's faces with an inverted Z buffer (scanline only)

    :type: bool
    """

    light_group: Group | None
    """ Limit lighting to lamps in this Group

    :type: Group | None
    """

    line_color: bpy_prop_array[float]
    """ Line color used for Freestyle line rendering

    :type: bpy_prop_array[float]
    """

    line_priority: int
    """ The line color of a higher priority is used at material boundaries

    :type: int
    """

    mirror_color: bpy_prop_array[float]
    """ Mirror color of the material

    :type: bpy_prop_array[float]
    """

    node_tree: ShaderNodeTree | None
    """ Node tree for node based materials

    :type: ShaderNodeTree | None
    """

    offset_z: float
    """ Give faces an artificial offset in the Z buffer for Z transparency

    :type: float
    """

    paint_active_slot: int
    """ Index of active texture paint slot

    :type: int
    """

    paint_clone_slot: int
    """ Index of clone texture paint slot

    :type: int
    """

    pass_index: int
    """ Index number for the "Material Index" render pass

    :type: int
    """

    physics: MaterialPhysics
    """ Game physics settings

    :type: MaterialPhysics
    """

    preview_render_type: typing.Literal[
        "FLAT", "SPHERE", "CUBE", "MONKEY", "HAIR", "SPHERE_A"
    ]
    """ Type of preview render

    :type: typing.Literal['FLAT','SPHERE','CUBE','MONKEY','HAIR','SPHERE_A']
    """

    raytrace_mirror: MaterialRaytraceMirror
    """ Raytraced reflection settings for the material

    :type: MaterialRaytraceMirror
    """

    raytrace_transparency: MaterialRaytraceTransparency
    """ Raytraced transparency settings for the material

    :type: MaterialRaytraceTransparency
    """

    roughness: float
    """ Oren-Nayar Roughness

    :type: float
    """

    shadow_buffer_bias: float
    """ Factor to multiply shadow buffer bias with (0 is ignore)

    :type: float
    """

    shadow_cast_alpha: float
    """ Shadow casting alpha, in use for Irregular and Deep shadow buffer

    :type: float
    """

    shadow_only_type: typing.Literal[
        "SHADOW_ONLY_OLD", "SHADOW_ONLY", "SHADOW_ONLY_SHADED"
    ]
    """ How to draw shadows

    :type: typing.Literal['SHADOW_ONLY_OLD','SHADOW_ONLY','SHADOW_ONLY_SHADED']
    """

    shadow_ray_bias: float
    """ Shadow raytracing bias to prevent terminator problems on shadow boundary

    :type: float
    """

    specular_alpha: float
    """ Alpha transparency for specular areas

    :type: float
    """

    specular_color: bpy_prop_array[float]
    """ Specular color of the material

    :type: bpy_prop_array[float]
    """

    specular_hardness: int
    """ How hard (sharp) the specular reflection is

    :type: int
    """

    specular_intensity: float
    """ How intense (bright) the specular reflection is

    :type: float
    """

    specular_ior: float
    """ Specular index of refraction

    :type: float
    """

    specular_ramp: ColorRamp
    """ Color ramp used to affect specular shading

    :type: ColorRamp
    """

    specular_ramp_blend: typing.Literal[
        "MIX",
        "ADD",
        "MULTIPLY",
        "SUBTRACT",
        "SCREEN",
        "DIVIDE",
        "DIFFERENCE",
        "DARKEN",
        "LIGHTEN",
        "OVERLAY",
        "DODGE",
        "BURN",
        "HUE",
        "SATURATION",
        "VALUE",
        "COLOR",
        "SOFT_LIGHT",
        "LINEAR_LIGHT",
    ]
    """ Blending method of the ramp and the specular color

    :type: typing.Literal['MIX','ADD','MULTIPLY','SUBTRACT','SCREEN','DIVIDE','DIFFERENCE','DARKEN','LIGHTEN','OVERLAY','DODGE','BURN','HUE','SATURATION','VALUE','COLOR','SOFT_LIGHT','LINEAR_LIGHT']
    """

    specular_ramp_factor: float
    """ Blending factor (also uses alpha in Colorband)

    :type: float
    """

    specular_ramp_input: typing.Literal["SHADER", "ENERGY", "NORMAL", "RESULT"]
    """ How the ramp maps on the surface

    :type: typing.Literal['SHADER','ENERGY','NORMAL','RESULT']
    """

    specular_shader: typing.Literal["COOKTORR", "PHONG", "BLINN", "TOON", "WARDISO"]
    """ 

    :type: typing.Literal['COOKTORR','PHONG','BLINN','TOON','WARDISO']
    """

    specular_slope: float
    """ The standard deviation of surface slope

    :type: float
    """

    specular_toon_size: float
    """ Size of specular toon area

    :type: float
    """

    specular_toon_smooth: float
    """ Smoothness of specular toon area

    :type: float
    """

    strand: MaterialStrand
    """ Strand settings for the material

    :type: MaterialStrand
    """

    subsurface_scattering: MaterialSubsurfaceScattering
    """ Subsurface scattering settings for the material

    :type: MaterialSubsurfaceScattering
    """

    texture_paint_images: bpy_prop_collection[Image]
    """ Texture images used for texture painting

    :type: bpy_prop_collection[Image]
    """

    texture_paint_slots: bpy_prop_collection[TexPaintSlot]
    """ Texture slots defining the mapping and influence of textures

    :type: bpy_prop_collection[TexPaintSlot]
    """

    texture_slots: MaterialTextureSlots
    """ Texture slots defining the mapping and influence of textures

    :type: MaterialTextureSlots
    """

    translucency: float
    """ Amount of diffuse shading on the back side

    :type: float
    """

    transparency_method: typing.Literal["MASK", "Z_TRANSPARENCY", "RAYTRACE"]
    """ Method to use for rendering transparency

    :type: typing.Literal['MASK','Z_TRANSPARENCY','RAYTRACE']
    """

    type: typing.Literal["SURFACE", "WIRE", "VOLUME", "HALO"]
    """ Material type defining how the object is rendered

    :type: typing.Literal['SURFACE','WIRE','VOLUME','HALO']
    """

    use_cast_approximate: bool
    """ Allow this material to cast shadows when using approximate ambient occlusion

    :type: bool
    """

    use_cast_buffer_shadows: bool
    """ Allow this material to cast shadows from shadow buffer lamps

    :type: bool
    """

    use_cast_shadows: bool
    """ Allow this material to cast shadows

    :type: bool
    """

    use_cast_shadows_only: bool
    """ Make objects with this material appear invisible (not rendered), only casting shadows

    :type: bool
    """

    use_cubic: bool
    """ Use cubic interpolation for diffuse values, for smoother transitions

    :type: bool
    """

    use_diffuse_ramp: bool
    """ Toggle diffuse ramp operations

    :type: bool
    """

    use_face_texture: bool
    """ Replace the object's base color with color from UV map image textures

    :type: bool
    """

    use_face_texture_alpha: bool
    """ Replace the object's base alpha value with alpha from UV map image textures

    :type: bool
    """

    use_full_oversampling: bool
    """ Force this material to render full shading/textures for all anti-aliasing samples

    :type: bool
    """

    use_light_group_exclusive: bool
    """ Material uses the light group exclusively - these lamps are excluded from other scene lighting

    :type: bool
    """

    use_light_group_local: bool
    """ When linked in, material uses local light group with the same name

    :type: bool
    """

    use_mist: bool
    """ Use mist with this material (in world settings)

    :type: bool
    """

    use_nodes: bool
    """ Use shader nodes to render the material

    :type: bool
    """

    use_object_color: bool
    """ Modulate the result with a per-object color

    :type: bool
    """

    use_only_shadow: bool
    """ Render shadows as the material's alpha value, making the material transparent except for shadowed areas

    :type: bool
    """

    use_ray_shadow_bias: bool
    """ Prevent raytraced shadow errors on surfaces with smooth shaded normals (terminator problem)

    :type: bool
    """

    use_raytrace: bool
    """ Include this material and geometry that uses it in raytracing calculations

    :type: bool
    """

    use_shadeless: bool
    """ Make this material insensitive to light or shadow

    :type: bool
    """

    use_shadows: bool
    """ Allow this material to receive shadows

    :type: bool
    """

    use_sky: bool
    """ Render this material with zero alpha, with sky background in place (scanline only)

    :type: bool
    """

    use_specular_ramp: bool
    """ Toggle specular ramp operations

    :type: bool
    """

    use_tangent_shading: bool
    """ Use the material's tangent vector instead of the normal for shading - for anisotropic shading effects

    :type: bool
    """

    use_textures: bpy_prop_array[bool]
    """ Enable/Disable each texture

    :type: bpy_prop_array[bool]
    """

    use_transparency: bool
    """ Render material as transparent

    :type: bool
    """

    use_transparent_shadows: bool
    """ Allow this object to receive transparent shadows cast through other objects

    :type: bool
    """

    use_uv_project: bool
    """ Use to ensure UV interpolation is correct for camera projections (use with UV project modifier)

    :type: bool
    """

    use_vertex_color_light: bool
    """ Add vertex colors as additional lighting

    :type: bool
    """

    use_vertex_color_paint: bool
    """ Replace object base color with vertex colors (multiply with 'texture face' face assigned textures)

    :type: bool
    """

    volume: MaterialVolume
    """ Volume settings for the material

    :type: MaterialVolume
    """

class MaterialGameSettings(bpy_struct):
    """Game Engine settings for a Material data-block"""

    alpha_blend: typing.Literal[
        "OPAQUE", "ADD", "CLIP", "ALPHA", "ALPHA_SORT", "ALPHA_ANTIALIASING"
    ]
    """ Blend Mode for Transparent Faces

    :type: typing.Literal['OPAQUE','ADD','CLIP','ALPHA','ALPHA_SORT','ALPHA_ANTIALIASING']
    """

    face_orientation: typing.Literal["NORMAL", "HALO", "BILLBOARD", "SHADOW"]
    """ Especial face orientation options

    :type: typing.Literal['NORMAL','HALO','BILLBOARD','SHADOW']
    """

    invisible: bool
    """ Make face invisible

    :type: bool
    """

    physics: bool
    """ Use physics properties of materials

    :type: bool
    """

    text: bool
    """ Use material as text in Game Engine

    :type: bool
    """

    use_backface_culling: bool
    """ Hide Back of the face in Game Engine

    :type: bool
    """

class MaterialHalo(bpy_struct):
    """Halo particle effect settings for a Material data-block"""

    add: float
    """ Strength of the add effect

    :type: float
    """

    flare_boost: float
    """ Give the flare extra strength

    :type: float
    """

    flare_seed: int
    """ Offset in the flare seed table

    :type: int
    """

    flare_size: float
    """ Factor by which the flare is larger than the halo

    :type: float
    """

    flare_subflare_count: int
    """ Number of sub-flares

    :type: int
    """

    flare_subflare_size: float
    """ Dimension of the sub-flares, dots and circles

    :type: float
    """

    hardness: int
    """ Hardness of the halo

    :type: int
    """

    line_count: int
    """ Number of star shaped lines rendered over the halo

    :type: int
    """

    ring_count: int
    """ Number of rings rendered over the halo

    :type: int
    """

    seed: int
    """ Randomize ring dimension and line location

    :type: int
    """

    size: float
    """ Dimension of the halo

    :type: float
    """

    star_tip_count: int
    """ Number of points on the star shaped halo

    :type: int
    """

    use_extreme_alpha: bool
    """ Use extreme alpha

    :type: bool
    """

    use_flare_mode: bool
    """ Render halo as a lens flare

    :type: bool
    """

    use_lines: bool
    """ Render star shaped lines over halo

    :type: bool
    """

    use_ring: bool
    """ Render rings over halo

    :type: bool
    """

    use_shaded: bool
    """ Let halo receive light and shadows from external objects

    :type: bool
    """

    use_soft: bool
    """ Soften the edges of halos at intersections with other geometry

    :type: bool
    """

    use_star: bool
    """ Render halo as a star

    :type: bool
    """

    use_texture: bool
    """ Give halo a texture

    :type: bool
    """

    use_vertex_normal: bool
    """ Use the vertex normal to specify the dimension of the halo

    :type: bool
    """

class MaterialPhysics(bpy_struct):
    """Physics settings for a Material data-block"""

    elasticity: float
    """ Elasticity of collisions

    :type: float
    """

    fh_damping: float
    """ Damping of the spring force, when inside the physics distance area

    :type: float
    """

    fh_distance: float
    """ Distance of the physics area

    :type: float
    """

    fh_force: float
    """ Upward spring force, when inside the physics distance area

    :type: float
    """

    friction: float
    """ Coulomb friction coefficient, when inside the physics distance area

    :type: float
    """

    use_fh_normal: bool
    """ Align dynamic game objects along the surface normal, when inside the physics distance area

    :type: bool
    """

class MaterialRaytraceMirror(bpy_struct):
    """Raytraced reflection settings for a Material data-block"""

    depth: int
    """ Maximum allowed number of light inter-reflections

    :type: int
    """

    distance: float
    """ Maximum distance of reflected rays (reflections further than this range fade to sky color or material color)

    :type: float
    """

    fade_to: typing.Literal["FADE_TO_SKY", "FADE_TO_MATERIAL"]
    """ The color that rays with no intersection within the Max Distance take (material color can be best for indoor scenes, sky color for outdoor)

    :type: typing.Literal['FADE_TO_SKY','FADE_TO_MATERIAL']
    """

    fresnel: float
    """ Power of Fresnel for mirror reflection

    :type: float
    """

    fresnel_factor: float
    """ Blending factor for Fresnel

    :type: float
    """

    gloss_anisotropic: float
    """ The shape of the reflection, from 0.0 (circular) to 1.0 (fully stretched along the tangent

    :type: float
    """

    gloss_factor: float
    """ The shininess of the reflection (values < 1.0 give diffuse, blurry reflections)

    :type: float
    """

    gloss_samples: int
    """ Number of cone samples averaged for blurry reflections

    :type: int
    """

    gloss_threshold: float
    """ Threshold for adaptive sampling (if a sample contributes less than this amount [as a percentage], sampling is stopped)

    :type: float
    """

    reflect_factor: float
    """ Amount of mirror reflection for raytrace

    :type: float
    """

    use: bool
    """ Enable raytraced reflections

    :type: bool
    """

class MaterialRaytraceTransparency(bpy_struct):
    """Raytraced refraction settings for a Material data-block"""

    depth: int
    """ Maximum allowed number of light inter-refractions

    :type: int
    """

    depth_max: float
    """ Maximum depth for light to travel through the transparent material before becoming fully filtered (0.0 is disabled)

    :type: float
    """

    falloff: float
    """ Falloff power for transmissivity filter effect (1.0 is linear)

    :type: float
    """

    filter: float
    """ Amount to blend in the material's diffuse color in raytraced transparency (simulating absorption)

    :type: float
    """

    fresnel: float
    """ Power of Fresnel for transparency (Ray or ZTransp)

    :type: float
    """

    fresnel_factor: float
    """ Blending factor for Fresnel

    :type: float
    """

    gloss_factor: float
    """ The clarity of the refraction. Values < 1.0 give diffuse, blurry refractions

    :type: float
    """

    gloss_samples: int
    """ Number of cone samples averaged for blurry refractions

    :type: int
    """

    gloss_threshold: float
    """ Threshold for adaptive sampling. If a sample contributes less than this amount (as a percentage), sampling is stopped

    :type: float
    """

    ior: float
    """ Angular index of refraction for raytraced refraction

    :type: float
    """

class MaterialSlot(bpy_struct):
    """Material slot in an object"""

    link: typing.Literal["OBJECT", "DATA"]
    """ Link material to object or the object's data

    :type: typing.Literal['OBJECT','DATA']
    """

    material: Material | None
    """ Material data-block used by this material slot

    :type: Material | None
    """

    name: str
    """ Material slot name

    :type: str
    """

class MaterialStrand(bpy_struct):
    """Strand settings for a Material data-block"""

    blend_distance: float
    """ Worldspace distance over which to blend in the surface normal

    :type: float
    """

    root_size: float
    """ Start size of strands in pixels or Blender units

    :type: float
    """

    shape: float
    """ Positive values make strands rounder, negative ones make strands spiky

    :type: float
    """

    size_min: float
    """ Minimum size of strands in pixels

    :type: float
    """

    tip_size: float
    """ End size of strands in pixels or Blender units

    :type: float
    """

    use_blender_units: bool
    """ Use Blender units for widths instead of pixels

    :type: bool
    """

    use_surface_diffuse: bool
    """ Make diffuse shading more similar to shading the surface

    :type: bool
    """

    use_tangent_shading: bool
    """ Use direction of strands as normal for tangent-shading

    :type: bool
    """

    uv_layer: str
    """ Name of UV map to override

    :type: str
    """

    width_fade: float
    """ Transparency along the width of the strand

    :type: float
    """

class MaterialSubsurfaceScattering(bpy_struct):
    """Diffuse subsurface scattering settings for a Material data-block"""

    back: float
    """ Back scattering weight

    :type: float
    """

    color: bpy_prop_array[float]
    """ Scattering color

    :type: bpy_prop_array[float]
    """

    color_factor: float
    """ Blend factor for SSS colors

    :type: float
    """

    error_threshold: float
    """ Error tolerance (low values are slower and higher quality)

    :type: float
    """

    front: float
    """ Front scattering weight

    :type: float
    """

    ior: float
    """ Index of refraction (higher values are denser)

    :type: float
    """

    radius: bpy_prop_array[float]
    """ Mean red/green/blue scattering path length

    :type: bpy_prop_array[float]
    """

    scale: float
    """ Object scale factor

    :type: float
    """

    texture_factor: float
    """ Texture scattering blend factor

    :type: float
    """

    use: bool
    """ Enable diffuse subsurface scattering effects in a material

    :type: bool
    """

class MaterialTextureSlot(TextureSlot, bpy_struct):
    """Texture slot for textures in a Material data-block"""

    alpha_factor: float
    """ Amount texture affects alpha

    :type: float
    """

    ambient_factor: float
    """ Amount texture affects ambient

    :type: float
    """

    bump_method: typing.Literal[
        "BUMP_ORIGINAL",
        "BUMP_COMPATIBLE",
        "BUMP_LOW_QUALITY",
        "BUMP_MEDIUM_QUALITY",
        "BUMP_BEST_QUALITY",
    ]
    """ Method to use for bump mapping

    :type: typing.Literal['BUMP_ORIGINAL','BUMP_COMPATIBLE','BUMP_LOW_QUALITY','BUMP_MEDIUM_QUALITY','BUMP_BEST_QUALITY']
    """

    bump_objectspace: typing.Literal[
        "BUMP_VIEWSPACE", "BUMP_OBJECTSPACE", "BUMP_TEXTURESPACE"
    ]
    """ Space to apply bump mapping in

    :type: typing.Literal['BUMP_VIEWSPACE','BUMP_OBJECTSPACE','BUMP_TEXTURESPACE']
    """

    density_factor: float
    """ Amount texture affects density

    :type: float
    """

    diffuse_color_factor: float
    """ Amount texture affects diffuse color

    :type: float
    """

    diffuse_factor: float
    """ Amount texture affects diffuse reflectivity

    :type: float
    """

    displacement_factor: float
    """ Amount texture displaces the surface

    :type: float
    """

    emission_color_factor: float
    """ Amount texture affects emission color

    :type: float
    """

    emission_factor: float
    """ Amount texture affects emission

    :type: float
    """

    emit_factor: float
    """ Amount texture affects emission

    :type: float
    """

    hardness_factor: float
    """ Amount texture affects hardness

    :type: float
    """

    mapping: typing.Literal["FLAT", "CUBE", "TUBE", "SPHERE"]
    """ 

    :type: typing.Literal['FLAT','CUBE','TUBE','SPHERE']
    """

    mapping_x: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    mapping_y: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    mapping_z: typing.Literal["NONE", "X", "Y", "Z"]
    """ 

    :type: typing.Literal['NONE','X','Y','Z']
    """

    mirror_factor: float
    """ Amount texture affects mirror color

    :type: float
    """

    normal_factor: float
    """ Amount texture affects normal values

    :type: float
    """

    normal_map_space: typing.Literal["CAMERA", "WORLD", "OBJECT", "TANGENT"]
    """ Set space of normal map image

    :type: typing.Literal['CAMERA','WORLD','OBJECT','TANGENT']
    """

    object: Object | None
    """ Object to use for mapping with Object texture coordinates

    :type: Object | None
    """

    raymir_factor: float
    """ Amount texture affects ray mirror

    :type: float
    """

    reflection_color_factor: float
    """ Amount texture affects color of out-scattered light

    :type: float
    """

    reflection_factor: float
    """ Amount texture affects brightness of out-scattered light

    :type: float
    """

    scattering_factor: float
    """ Amount texture affects scattering

    :type: float
    """

    specular_color_factor: float
    """ Amount texture affects specular color

    :type: float
    """

    specular_factor: float
    """ Amount texture affects specular reflectivity

    :type: float
    """

    texture_coords: typing.Literal[
        "GLOBAL",
        "OBJECT",
        "UV",
        "ORCO",
        "STRAND",
        "WINDOW",
        "NORMAL",
        "REFLECTION",
        "STRESS",
        "TANGENT",
    ]
    """ 

    :type: typing.Literal['GLOBAL','OBJECT','UV','ORCO','STRAND','WINDOW','NORMAL','REFLECTION','STRESS','TANGENT']
    """

    translucency_factor: float
    """ Amount texture affects translucency

    :type: float
    """

    transmission_color_factor: float
    """ Amount texture affects result color after light has been scattered/absorbed

    :type: float
    """

    use: bool
    """ Enable this material texture slot

    :type: bool
    """

    use_from_dupli: bool
    """ Dupli's instanced from verts, faces or particles, inherit texture coordinate from their parent

    :type: bool
    """

    use_from_original: bool
    """ Dupli's derive their object coordinates from the original object's transformation

    :type: bool
    """

    use_map_alpha: bool
    """ The texture affects the alpha value

    :type: bool
    """

    use_map_ambient: bool
    """ The texture affects the value of ambient

    :type: bool
    """

    use_map_color_diffuse: bool
    """ The texture affects basic color of the material

    :type: bool
    """

    use_map_color_emission: bool
    """ The texture affects the color of emission

    :type: bool
    """

    use_map_color_reflection: bool
    """ The texture affects the color of scattered light

    :type: bool
    """

    use_map_color_spec: bool
    """ The texture affects the specularity color

    :type: bool
    """

    use_map_color_transmission: bool
    """ The texture affects the result color after other light has been scattered/absorbed

    :type: bool
    """

    use_map_density: bool
    """ The texture affects the volume's density

    :type: bool
    """

    use_map_diffuse: bool
    """ The texture affects the value of diffuse reflectivity

    :type: bool
    """

    use_map_displacement: bool
    """ Let the texture displace the surface

    :type: bool
    """

    use_map_emission: bool
    """ The texture affects the volume's emission

    :type: bool
    """

    use_map_emit: bool
    """ The texture affects the emit value

    :type: bool
    """

    use_map_hardness: bool
    """ The texture affects the hardness value

    :type: bool
    """

    use_map_mirror: bool
    """ The texture affects the mirror color

    :type: bool
    """

    use_map_normal: bool
    """ The texture affects the rendered normal

    :type: bool
    """

    use_map_raymir: bool
    """ The texture affects the ray-mirror value

    :type: bool
    """

    use_map_reflect: bool
    """ The texture affects the reflected light's brightness

    :type: bool
    """

    use_map_scatter: bool
    """ The texture affects the volume's scattering

    :type: bool
    """

    use_map_specular: bool
    """ The texture affects the value of specular reflectivity

    :type: bool
    """

    use_map_to_bounds: bool
    """ Map coordinates in object bounds

    :type: bool
    """

    use_map_translucency: bool
    """ The texture affects the translucency value

    :type: bool
    """

    use_map_warp: bool
    """ Let the texture warp texture coordinates of next channels

    :type: bool
    """

    uv_layer: str
    """ UV map to use for mapping with UV texture coordinates

    :type: str
    """

    warp_factor: float
    """ Amount texture affects texture coordinates of next channels

    :type: float
    """

class MaterialVolume(bpy_struct):
    """Volume rendering settings for a Material data-block"""

    asymmetry: float
    """ Back scattering (-1.0) to Forward scattering (1.0) and the range in between

    :type: float
    """

    cache_resolution: int
    """ Resolution of the voxel grid, low resolutions are faster, high resolutions use more memory

    :type: int
    """

    density: float
    """ The base density of the volume

    :type: float
    """

    density_scale: float
    """ Multiplier for the material's density

    :type: float
    """

    depth_threshold: float
    """ Stop ray marching early if transmission drops below this luminance - higher values give speedups in dense volumes at the expense of accuracy

    :type: float
    """

    emission: float
    """ Amount of light that gets emitted by the volume

    :type: float
    """

    emission_color: bpy_prop_array[float]
    """ Color of emitted light

    :type: bpy_prop_array[float]
    """

    light_method: typing.Literal[
        "SHADELESS",
        "SHADOWED",
        "SHADED",
        "MULTIPLE_SCATTERING",
        "SHADED_PLUS_MULTIPLE_SCATTERING",
    ]
    """ Method of shading, attenuating, and scattering light through the volume

    :type: typing.Literal['SHADELESS','SHADOWED','SHADED','MULTIPLE_SCATTERING','SHADED_PLUS_MULTIPLE_SCATTERING']
    """

    ms_diffusion: float
    """ Diffusion factor, the strength of the blurring effect

    :type: float
    """

    ms_intensity: float
    """ Multiplier for multiple scattered light energy

    :type: float
    """

    ms_spread: float
    """ Proportional distance over which the light is diffused

    :type: float
    """

    reflection: float
    """ Multiplier to make out-scattered light brighter or darker (non-physically correct)

    :type: float
    """

    reflection_color: bpy_prop_array[float]
    """ Color of light scattered out of the volume (does not affect transmission)

    :type: bpy_prop_array[float]
    """

    scattering: float
    """ Amount of light that gets scattered out by the volume - the more out-scattering, the shallower the light will penetrate

    :type: float
    """

    step_method: typing.Literal["RANDOMIZED", "CONSTANT"]
    """ Method of calculating the steps through the volume

    :type: typing.Literal['RANDOMIZED','CONSTANT']
    """

    step_size: float
    """ Distance between subsequent volume depth samples

    :type: float
    """

    transmission_color: bpy_prop_array[float]
    """ Result color of the volume, after other light has been scattered/absorbed

    :type: bpy_prop_array[float]
    """

    use_external_shadows: bool
    """ Receive shadows from sources outside the volume (temporary)

    :type: bool
    """

    use_light_cache: bool
    """ Pre-calculate the shading information into a voxel grid, speeds up shading at slightly less accuracy

    :type: bool
    """

class Menu(bpy_struct):
    """Editor menu containing buttons"""

    bl_description: str
    """ 

    :type: str
    """

    bl_idname: str
    """ If this is set, the menu gets a custom ID, otherwise it takes the name of the class used to define the menu (for example, if the class name is "OBJECT_MT_hello", and bl_idname is not set by the script, then bl_idname = "OBJECT_MT_hello")

    :type: str
    """

    bl_label: str
    """ The menu label

    :type: str
    """

    bl_translation_context: str | typing.Any
    """ 

    :type: str | typing.Any
    """

    layout: UILayout
    """ Defines the structure of the menu in the UI

    :type: UILayout
    """

    @classmethod
    def poll(cls, context: Context | None) -> bool:
        """If this method returns a non-null output, then the menu can be drawn

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """

    def draw(self, context: Context | None):
        """Draw UI elements into the menu UI layout

        :param context:
        :type context: Context | None
        """

    def draw_preset(self, context):
        """Define these on the subclass:
        - preset_operator (string)
        - preset_subdir (string)Optionally:
        - preset_extensions (set of strings)
        - preset_operator_defaults (dict of keyword args)

                :param context:
        """

    def path_menu(self, searchpaths, operator, props_default=None, filter_ext=None):
        """

        :param searchpaths:
        :param operator:
        :param props_default:
        :param filter_ext:
        """

    @classmethod
    def append(cls, draw_func):
        """

        :param draw_func:
        """

    @classmethod
    def prepend(cls, draw_func):
        """

        :param draw_func:
        """

    @classmethod
    def remove(cls, draw_func):
        """

        :param draw_func:
        """

class Mesh(ID, bpy_struct):
    """Mesh data-block defining geometric surfaces"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    auto_smooth_angle: float
    """ Maximum angle between face normals that will be considered as smooth (unused if custom split normals data are available)

    :type: float
    """

    auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    edges: MeshEdges
    """ Edges of the mesh

    :type: MeshEdges
    """

    has_custom_normals: bool
    """ True if there are custom split normals data in this mesh

    :type: bool
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    loops: MeshLoops
    """ Loops of the mesh (polygon corners)

    :type: MeshLoops
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    polygon_layers_float: PolygonFloatProperties
    """ 

    :type: PolygonFloatProperties
    """

    polygon_layers_int: PolygonIntProperties
    """ 

    :type: PolygonIntProperties
    """

    polygon_layers_string: PolygonStringProperties
    """ 

    :type: PolygonStringProperties
    """

    polygons: MeshPolygons
    """ Polygons of the mesh

    :type: MeshPolygons
    """

    shape_keys: Key
    """ 

    :type: Key
    """

    show_double_sided: bool
    """ Display the mesh with double or single sided lighting (OpenGL only)

    :type: bool
    """

    show_edge_bevel_weight: bool
    """ Display weights created for the Bevel modifier

    :type: bool
    """

    show_edge_crease: bool
    """ Display creases created for subsurf weighting

    :type: bool
    """

    show_edge_seams: bool
    """ Display UV unwrapping seams

    :type: bool
    """

    show_edge_sharp: bool
    """ Display sharp edges, used with the EdgeSplit modifier

    :type: bool
    """

    show_edges: bool
    """ Display selected edges using highlights in the 3D view and UV editor

    :type: bool
    """

    show_extra_edge_angle: bool
    """ Display selected edge angle, using global values when set in the transform panel

    :type: bool
    """

    show_extra_edge_length: bool
    """ Display selected edge lengths, using global values when set in the transform panel

    :type: bool
    """

    show_extra_face_angle: bool
    """ Display the angles in the selected edges, using global values when set in the transform panel

    :type: bool
    """

    show_extra_face_area: bool
    """ Display the area of selected faces, using global values when set in the transform panel

    :type: bool
    """

    show_extra_indices: bool
    """ Display the index numbers of selected vertices, edges, and faces

    :type: bool
    """

    show_faces: bool
    """ Display all faces as shades in the 3D view and UV editor

    :type: bool
    """

    show_freestyle_edge_marks: bool
    """ Display Freestyle edge marks, used with the Freestyle renderer

    :type: bool
    """

    show_freestyle_face_marks: bool
    """ Display Freestyle face marks, used with the Freestyle renderer

    :type: bool
    """

    show_normal_face: bool
    """ Display face normals as lines

    :type: bool
    """

    show_normal_loop: bool
    """ Display vertex-per-face normals as lines

    :type: bool
    """

    show_normal_vertex: bool
    """ Display vertex normals as lines

    :type: bool
    """

    show_statvis: bool
    """ Display statistical information about the mesh

    :type: bool
    """

    show_weight: bool
    """ Draw weights in editmode

    :type: bool
    """

    skin_vertices: bpy_prop_collection[MeshSkinVertexLayer]
    """ All skin vertices

    :type: bpy_prop_collection[MeshSkinVertexLayer]
    """

    tessface_uv_textures: TessfaceUVTextures
    """ All UV maps for tessellated faces (read-only, for use by renderers)

    :type: TessfaceUVTextures
    """

    tessface_vertex_colors: VertexColors
    """ All tessellated face colors (read-only, for use by renderers)

    :type: VertexColors
    """

    tessfaces: MeshTessFaces
    """ Tessellated faces of the mesh (derived from polygons)

    :type: MeshTessFaces
    """

    texco_mesh: typing_extensions.Self | None
    """ Derive texture coordinates from another mesh

    :type: typing_extensions.Self | None
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    texture_mesh: typing_extensions.Self | None
    """ Use another mesh for texture indices (vertex indices must be aligned)

    :type: typing_extensions.Self | None
    """

    total_edge_sel: int
    """ Selected edge count in editmode

    :type: int
    """

    total_face_sel: int
    """ Selected face count in editmode

    :type: int
    """

    total_vert_sel: int
    """ Selected vertex count in editmode

    :type: int
    """

    use_auto_smooth: bool
    """ Auto smooth (based on smooth/sharp faces/edges and angle between faces), or use custom split normals data if available

    :type: bool
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    use_customdata_edge_bevel: bool
    """ 

    :type: bool
    """

    use_customdata_edge_crease: bool
    """ 

    :type: bool
    """

    use_customdata_vertex_bevel: bool
    """ 

    :type: bool
    """

    use_mirror_topology: bool
    """ Use topology based mirroring (for when both sides of mesh have matching, unique topology)

    :type: bool
    """

    use_mirror_x: bool
    """ X Axis mirror editing

    :type: bool
    """

    use_paint_mask: bool
    """ Face selection masking for painting

    :type: bool
    """

    use_paint_mask_vertex: bool
    """ Vertex selection masking for painting (weight paint only)

    :type: bool
    """

    uv_layer_clone: MeshUVLoopLayer | None
    """ UV loop layer to be used as cloning source

    :type: MeshUVLoopLayer | None
    """

    uv_layer_clone_index: int
    """ Clone UV loop layer index

    :type: int
    """

    uv_layer_stencil: MeshUVLoopLayer | None
    """ UV loop layer to mask the painted area

    :type: MeshUVLoopLayer | None
    """

    uv_layer_stencil_index: int
    """ Mask UV loop layer index

    :type: int
    """

    uv_layers: UVLoopLayers
    """ All UV loop layers

    :type: UVLoopLayers
    """

    uv_texture_clone: MeshTexturePolyLayer | None
    """ UV map to be used as cloning source

    :type: MeshTexturePolyLayer | None
    """

    uv_texture_clone_index: int
    """ Clone UV map index

    :type: int
    """

    uv_texture_stencil: MeshTexturePolyLayer | None
    """ UV map to mask the painted area

    :type: MeshTexturePolyLayer | None
    """

    uv_texture_stencil_index: int
    """ Mask UV map index

    :type: int
    """

    uv_textures: UVTextures
    """ All UV maps

    :type: UVTextures
    """

    vertex_colors: LoopColors
    """ All vertex colors

    :type: LoopColors
    """

    vertex_layers_float: VertexFloatProperties
    """ 

    :type: VertexFloatProperties
    """

    vertex_layers_int: VertexIntProperties
    """ 

    :type: VertexIntProperties
    """

    vertex_layers_string: VertexStringProperties
    """ 

    :type: VertexStringProperties
    """

    vertex_paint_masks: bpy_prop_collection[MeshPaintMaskLayer]
    """ Vertex paint mask

    :type: bpy_prop_collection[MeshPaintMaskLayer]
    """

    vertices: MeshVertices
    """ Vertices of the mesh

    :type: MeshVertices
    """

    edge_keys: typing.Any
    """ (readonly)"""

    def transform(
        self,
        matrix: collections.abc.Iterable[float] | None,
        shape_keys: bool | None = False,
    ):
        """Transform mesh vertices by a matrix (Warning: inverts normals if matrix is negative)

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        :param shape_keys: Transform Shape Keys
        :type shape_keys: bool | None
        """

    def flip_normals(self):
        """Invert winding of all polygons (clears tessellation, does not handle custom normals)"""

    def calc_normals(self):
        """Calculate vertex normals"""

    def create_normals_split(self):
        """Empty split vertex normals"""

    def calc_normals_split(self):
        """Calculate split vertex normals, which preserve sharp edges"""

    def free_normals_split(self):
        """Free split vertex normals"""

    def split_faces(self):
        """Spli faces based on the edge angle"""

    def calc_tangents(self, uvmap: str = ""):
        """Compute tangents and bitangent signs, to be used together with the split normals to get a complete tangent space for normal mapping (split normals are also computed if not yet present)

        :param uvmap: Name of the UV map to use for tangent space computation
        :type uvmap: str
        """

    def free_tangents(self):
        """Free tangents"""

    def calc_tessface(self, free_mpoly: bool | None = False):
        """Calculate face tessellation (supports editmode too)

        :param free_mpoly: Free MPoly, Free data used by polygons and loops. WARNING: This destructive operation removes regular faces, only used on temporary mesh data-blocks to reduce memory footprint of render engines and export scripts
        :type free_mpoly: bool | None
        """

    def calc_smooth_groups(self, use_bitflags: bool | None = False):
        """Calculate smooth groups from sharp edges

                :param use_bitflags: Produce bitflags groups instead of simple numeric values
                :type use_bitflags: bool | None
                :return: poly_groups, Smooth Groups, int array of 1 items in [-inf, inf]

        groups, Total number of groups, int in [0, inf]
        """

    def normals_split_custom_set(self, normals: collections.abc.Iterable[float] | None):
        """Define custom split normals of this mesh (use zero-vectors to keep auto ones)

        :param normals: Normals
        :type normals: collections.abc.Iterable[float] | None
        """

    def normals_split_custom_set_from_vertices(
        self, normals: collections.abc.Iterable[float] | None
    ):
        """Define custom split normals of this mesh, from vertices' normals (use zero-vectors to keep auto ones)

        :param normals: Normals
        :type normals: collections.abc.Iterable[float] | None
        """

    def update(
        self, calc_edges: bool | None = False, calc_tessface: bool | None = False
    ):
        """update

        :param calc_edges: Calculate Edges, Force recalculation of edges
        :type calc_edges: bool | None
        :param calc_tessface: Calculate Tessellation, Force recalculation of tessellation faces
        :type calc_tessface: bool | None
        """

    def unit_test_compare(self, mesh: typing_extensions.Self | None = None) -> str:
        """unit_test_compare

        :param mesh: Mesh to compare to
        :type mesh: typing_extensions.Self | None
        :return: Return value, String description of result of comparison
        :rtype: str
        """

    def validate(
        self, verbose: bool | None = False, clean_customdata: bool | None = True
    ) -> bool:
        """Validate geometry, return True when the mesh has had invalid geometry corrected/removed

        :param verbose: Verbose, Output information about the errors found
        :type verbose: bool | None
        :param clean_customdata: Clean Custom Data, Remove temp/cached custom-data layers, like e.g. normals...
        :type clean_customdata: bool | None
        :return: Result
        :rtype: bool
        """

    def validate_material_indices(self) -> bool:
        """Validate material indices of polygons, return True when the mesh has had invalid indices corrected (to default 0)

        :return: Result
        :rtype: bool
        """

    def from_pydata(
        self, vertices: list | None, edges: list | None, faces: list | None
    ):
        """Make a mesh from a list of vertices/edges/faces
        Until we have a nicer way to make geometry, use this.

                :param vertices: float triplets each representing (X, Y, Z)
        eg: [(0.0, 1.0, 0.5), ...].
                :type vertices: list | None
                :param edges: int pairs, each pair contains two indices to the
        vertices argument. eg: [(1, 2), ...]
                :type edges: list | None
                :param faces: iterator of faces, each faces contains three or more indices to
        the vertices argument. eg: [(5, 6, 8, 9), (1, 2, 3), ...]
                :type faces: list | None
        """

class MeshCacheModifier(Modifier, bpy_struct):
    """Cache Mesh"""

    cache_format: typing.Literal["MDD", "PC2"]
    """ 

    :type: typing.Literal['MDD','PC2']
    """

    deform_mode: typing.Literal["OVERWRITE", "INTEGRATE"]
    """ 

    :type: typing.Literal['OVERWRITE','INTEGRATE']
    """

    eval_factor: float
    """ Evaluation time in seconds

    :type: float
    """

    eval_frame: float
    """ The frame to evaluate (starting at 0)

    :type: float
    """

    eval_time: float
    """ Evaluation time in seconds

    :type: float
    """

    factor: float
    """ Influence of the deformation

    :type: float
    """

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    flip_axis: set[typing.Literal["X", "Y", "Z"]]
    """ 

    :type: set[typing.Literal['X','Y','Z']]
    """

    forward_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ 

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    frame_scale: float
    """ Evaluation time in seconds

    :type: float
    """

    frame_start: float
    """ Add this to the start frame

    :type: float
    """

    interpolation: typing.Literal["NONE", "LINEAR"]
    """ 

    :type: typing.Literal['NONE','LINEAR']
    """

    play_mode: typing.Literal["SCENE", "CUSTOM"]
    """ 

    :type: typing.Literal['SCENE','CUSTOM']
    """

    time_mode: typing.Literal["FRAME", "TIME", "FACTOR"]
    """ Method to control playback time

    :type: typing.Literal['FRAME','TIME','FACTOR']
    """

    up_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ 

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

class MeshColor(bpy_struct):
    """Vertex colors for a face in a Mesh"""

    color1: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color2: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color3: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    color4: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshColorLayer(bpy_struct):
    """Layer of vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshColor]
    """ 

    :type: bpy_prop_collection[MeshColor]
    """

    name: str
    """ Name of Vertex color layer

    :type: str
    """

class MeshDeformModifier(Modifier, bpy_struct):
    """Mesh deformation modifier to deform with other meshes"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    is_bound: bool
    """ Whether geometry has been bound to control cage

    :type: bool
    """

    object: Object | None
    """ Mesh object to deform with

    :type: Object | None
    """

    precision: int
    """ The grid size for binding

    :type: int
    """

    use_dynamic_bind: bool
    """ Recompute binding dynamically on top of other deformers (slower and more memory consuming)

    :type: bool
    """

    vertex_group: str
    """ Vertex group name

    :type: str
    """

class MeshEdge(bpy_struct):
    """Edge in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier

    :type: float
    """

    crease: float
    """ Weight used by the Subsurf modifier for creasing

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this edge

    :type: int
    """

    is_loose: bool
    """ Loose edge

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    use_edge_sharp: bool
    """ Sharp edge for the EdgeSplit modifier

    :type: bool
    """

    use_freestyle_mark: bool
    """ Edge mark for Freestyle line rendering

    :type: bool
    """

    use_seam: bool
    """ Seam edge for UV unwrapping

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    key: typing.Any
    """ (readonly)"""

class MeshLoop(bpy_struct):
    """Loop in a Mesh data-block"""

    bitangent: bpy_prop_array[float]
    """ Bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, use it only if really needed, slower access than bitangent_sign)

    :type: bpy_prop_array[float]
    """

    bitangent_sign: float
    """ Sign of the bitangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents, bitangent = bitangent_sign * cross(normal, tangent))

    :type: float
    """

    edge_index: int
    """ Edge index

    :type: int
    """

    index: int
    """ Index of this loop

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length split normal vector of this vertex for this polygon (must be computed beforehand using calc_normals_split or calc_tangents)

    :type: bpy_prop_array[float]
    """

    tangent: bpy_prop_array[float]
    """ Local space unit length tangent vector of this vertex for this polygon (must be computed beforehand using calc_tangents)

    :type: bpy_prop_array[float]
    """

    vertex_index: int
    """ Vertex index

    :type: int
    """

class MeshLoopColor(bpy_struct):
    """Vertex loop colors in a Mesh"""

    color: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshLoopColorLayer(bpy_struct):
    """Layer of vertex colors in a Mesh data-block"""

    active: bool | None
    """ Sets the layer as active for display and editing

    :type: bool | None
    """

    active_render: bool | None
    """ Sets the layer as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshLoopColor]
    """ 

    :type: bpy_prop_collection[MeshLoopColor]
    """

    name: str
    """ Name of Vertex color layer

    :type: str
    """

class MeshPaintMaskLayer(bpy_struct):
    """Per-vertex paint mask data"""

    data: bpy_prop_collection[MeshPaintMaskProperty]
    """ 

    :type: bpy_prop_collection[MeshPaintMaskProperty]
    """

class MeshPaintMaskProperty(bpy_struct):
    """Floating point paint mask value"""

    value: float
    """ 

    :type: float
    """

class MeshPolygon(bpy_struct):
    """Polygon in a Mesh data-block"""

    area: float
    """ Read only area of this polygon

    :type: float
    """

    center: bpy_prop_array[float]
    """ Center of this polygon

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this polygon

    :type: int
    """

    loop_start: int
    """ Index of the first loop of this polygon

    :type: int
    """

    loop_total: int
    """ Number of loops used by this polygon

    :type: int
    """

    material_index: int
    """ 

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this polygon

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    use_freestyle_mark: bool
    """ Face mark for Freestyle line rendering

    :type: bool
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    edge_keys: typing.Any
    """ (readonly)"""

    loop_indices: typing.Any
    """ (readonly)"""

    def flip(self):
        """Invert winding of this polygon (flip its normal)"""

class MeshPolygonFloatProperty(bpy_struct):
    """User defined floating point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

class MeshPolygonFloatPropertyLayer(bpy_struct):
    """User defined layer of floating point number values"""

    data: bpy_prop_collection[MeshPolygonFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshPolygonIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

class MeshPolygonIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshPolygonIntProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonIntProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshPolygonStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

class MeshPolygonStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshPolygonStringProperty]
    """ 

    :type: bpy_prop_collection[MeshPolygonStringProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshSequenceCacheModifier(Modifier, bpy_struct):
    """Cache Mesh"""

    cache_file: CacheFile | None
    """ 

    :type: CacheFile | None
    """

    object_path: str
    """ Path to the object in the Alembic archive used to lookup geometric data

    :type: str
    """

    read_data: set[typing.Literal["VERT", "POLY", "UV", "COLOR"]]
    """ 

    :type: set[typing.Literal['VERT','POLY','UV','COLOR']]
    """

class MeshSkinVertex(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    radius: bpy_prop_array[float]
    """ Radius of the skin

    :type: bpy_prop_array[float]
    """

    use_loose: bool
    """ If vertex has multiple adjacent edges, it is hulled to them directly

    :type: bool
    """

    use_root: bool
    """ Vertex is a root for rotation calculations and armature generation

    :type: bool
    """

class MeshSkinVertexLayer(bpy_struct):
    """Per-vertex skin data for use with the Skin modifier"""

    data: bpy_prop_collection[MeshSkinVertex]
    """ 

    :type: bpy_prop_collection[MeshSkinVertex]
    """

    name: str
    """ Name of skin layer

    :type: str
    """

class MeshStatVis(bpy_struct):
    distort_max: float
    """ Maximum angle to display

    :type: float
    """

    distort_min: float
    """ Minimum angle to display

    :type: float
    """

    overhang_axis: typing.Literal["POS_X", "POS_Y", "POS_Z", "NEG_X", "NEG_Y", "NEG_Z"]
    """ 

    :type: typing.Literal['POS_X','POS_Y','POS_Z','NEG_X','NEG_Y','NEG_Z']
    """

    overhang_max: float
    """ Maximum angle to display

    :type: float
    """

    overhang_min: float
    """ Minimum angle to display

    :type: float
    """

    sharp_max: float
    """ Maximum angle to display

    :type: float
    """

    sharp_min: float
    """ Minimum angle to display

    :type: float
    """

    thickness_max: float
    """ Maximum for measuring thickness

    :type: float
    """

    thickness_min: float
    """ Minimum for measuring thickness

    :type: float
    """

    thickness_samples: int
    """ Number of samples to test per face

    :type: int
    """

    type: typing.Literal["OVERHANG", "THICKNESS", "INTERSECT", "DISTORT", "SHARP"]
    """ Type of data to visualize/check

    :type: typing.Literal['OVERHANG','THICKNESS','INTERSECT','DISTORT','SHARP']
    """

class MeshTessFace(bpy_struct):
    """TessFace in a Mesh data-block"""

    area: float
    """ Read only area of this face

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this face

    :type: int
    """

    material_index: int
    """ 

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Local space unit length normal vector for this face

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    split_normals: bpy_prop_array[float]
    """ Local space unit length split normals vectors of the vertices of this face (must be computed beforehand using calc_normals_split or calc_tangents, and then calc_tessface)

    :type: bpy_prop_array[float]
    """

    use_smooth: bool
    """ 

    :type: bool
    """

    vertices: bpy_prop_array[int]
    """ Vertex indices

    :type: bpy_prop_array[int]
    """

    vertices_raw: bpy_prop_array[int]
    """ Fixed size vertex indices array

    :type: bpy_prop_array[int]
    """

    center: typing.Any
    """ The midpoint of the face.
(readonly)"""

    edge_keys: typing.Any
    """ (readonly)"""

class MeshTextureFace(bpy_struct):
    """UV map and image texture for a face"""

    image: Image | None
    """ 

    :type: Image | None
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv1: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv2: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv3: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv4: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    uv_raw: bpy_prop_array[float]
    """ Fixed size UV coordinates array

    :type: bpy_prop_array[float]
    """

class MeshTextureFaceLayer(bpy_struct):
    """UV map with assigned image textures in a Mesh data-block"""

    active: bool | None
    """ Set the map as active for display and editing

    :type: bool | None
    """

    active_clone: bool | None
    """ Set the map as active for cloning

    :type: bool | None
    """

    active_render: bool | None
    """ Set the map as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshTextureFace]
    """ 

    :type: bpy_prop_collection[MeshTextureFace]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshTexturePoly(bpy_struct):
    """UV map and image texture for a face"""

    image: Image | None
    """ 

    :type: Image | None
    """

class MeshTexturePolyLayer(bpy_struct):
    """UV map with assigned image textures in a Mesh data-block"""

    active: bool | None
    """ Set the map as active for display and editing

    :type: bool | None
    """

    active_clone: bool | None
    """ Set the map as active for cloning

    :type: bool | None
    """

    active_render: bool | None
    """ Set the map as active for rendering

    :type: bool | None
    """

    data: bpy_prop_collection[MeshTexturePoly]
    """ 

    :type: bpy_prop_collection[MeshTexturePoly]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshUVLoop(bpy_struct):
    pin_uv: bool
    """ 

    :type: bool
    """

    select: bool
    """ 

    :type: bool
    """

    select_edge: bool
    """ 

    :type: bool
    """

    uv: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

class MeshUVLoopLayer(bpy_struct):
    data: bpy_prop_collection[MeshUVLoop]
    """ 

    :type: bpy_prop_collection[MeshUVLoop]
    """

    name: str
    """ Name of UV map

    :type: str
    """

class MeshVertex(bpy_struct):
    """Vertex in a Mesh data-block"""

    bevel_weight: float
    """ Weight used by the Bevel modifier 'Only Vertices' option

    :type: float
    """

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    groups: bpy_prop_collection[VertexGroupElement]
    """ Weights for the vertex groups this vertex is member of

    :type: bpy_prop_collection[VertexGroupElement]
    """

    hide: bool
    """ 

    :type: bool
    """

    index: int
    """ Index of this vertex

    :type: int
    """

    normal: bpy_prop_array[float]
    """ Vertex Normal

    :type: bpy_prop_array[float]
    """

    select: bool
    """ 

    :type: bool
    """

    undeformed_co: bpy_prop_array[float]
    """ For meshes with modifiers applied, the coordinate of the vertex with no deforming modifiers applied, as used for generated texture coordinates

    :type: bpy_prop_array[float]
    """

class MeshVertexFloatProperty(bpy_struct):
    """User defined floating point number value in a float properties layer"""

    value: float
    """ 

    :type: float
    """

class MeshVertexFloatPropertyLayer(bpy_struct):
    """User defined layer of floating point number values"""

    data: bpy_prop_collection[MeshVertexFloatProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexFloatProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshVertexIntProperty(bpy_struct):
    """User defined integer number value in an integer properties layer"""

    value: int
    """ 

    :type: int
    """

class MeshVertexIntPropertyLayer(bpy_struct):
    """User defined layer of integer number values"""

    data: bpy_prop_collection[MeshVertexIntProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexIntProperty]
    """

    name: str
    """ 

    :type: str
    """

class MeshVertexStringProperty(bpy_struct):
    """User defined string text value in a string properties layer"""

    value: str
    """ 

    :type: str
    """

class MeshVertexStringPropertyLayer(bpy_struct):
    """User defined layer of string text values"""

    data: bpy_prop_collection[MeshVertexStringProperty]
    """ 

    :type: bpy_prop_collection[MeshVertexStringProperty]
    """

    name: str
    """ 

    :type: str
    """

class MessageActuator(Actuator, bpy_struct):
    body_message: str
    """ Optional, message body Text

    :type: str
    """

    body_property: str
    """ The message body will be set by the Property Value

    :type: str
    """

    body_type: typing.Literal["TEXT", "PROPERTY"]
    """ Toggle message type: either Text or a PropertyName

    :type: typing.Literal['TEXT','PROPERTY']
    """

    subject: str
    """ Optional, message subject (this is what can be filtered on)

    :type: str
    """

    to_property: str
    """ Optional, send message to objects with this name only, or empty to broadcast

    :type: str
    """

class MessageSensor(Sensor, bpy_struct):
    """Sensor to detect incoming messages"""

    subject: str
    """ Optional subject filter: only accept messages with this subject, or empty to accept all

    :type: str
    """

class MetaBall(ID, bpy_struct):
    """Metaball data-block to defined blobby surfaces"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    cycles: CyclesMeshSettings
    """ Cycles mesh settings

    :type: CyclesMeshSettings
    """

    elements: MetaBallElements
    """ Meta elements

    :type: MetaBallElements
    """

    is_editmode: bool
    """ True when used in editmode

    :type: bool
    """

    materials: IDMaterials
    """ 

    :type: IDMaterials
    """

    render_resolution: float
    """ Polygonization resolution in rendering

    :type: float
    """

    resolution: float
    """ Polygonization resolution in the 3D viewport

    :type: float
    """

    texspace_location: bpy_prop_array[float]
    """ Texture space location

    :type: bpy_prop_array[float]
    """

    texspace_size: bpy_prop_array[float]
    """ Texture space size

    :type: bpy_prop_array[float]
    """

    threshold: float
    """ Influence of meta elements

    :type: float
    """

    update_method: typing.Literal["UPDATE_ALWAYS", "HALFRES", "FAST", "NEVER"]
    """ Metaball edit update behavior

    :type: typing.Literal['UPDATE_ALWAYS','HALFRES','FAST','NEVER']
    """

    use_auto_texspace: bool
    """ Adjust active object's texture space automatically when transforming object

    :type: bool
    """

    def transform(self, matrix: collections.abc.Iterable[float] | None):
        """Transform meta elements by a matrix

        :param matrix: Matrix
        :type matrix: collections.abc.Iterable[float] | None
        """

class MetaElement(bpy_struct):
    """Blobby element in a Metaball data-block"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    hide: bool
    """ Hide element

    :type: bool
    """

    radius: float
    """ 

    :type: float
    """

    rotation: bpy_prop_array[float]
    """ Normalized quaternion rotation

    :type: bpy_prop_array[float]
    """

    size_x: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_y: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    size_z: float
    """ Size of element, use of components depends on element type

    :type: float
    """

    stiffness: float
    """ Stiffness defines how much of the element to fill

    :type: float
    """

    type: typing.Literal["BALL", "CAPSULE", "PLANE", "ELLIPSOID", "CUBE"]
    """ Metaball types

    :type: typing.Literal['BALL','CAPSULE','PLANE','ELLIPSOID','CUBE']
    """

    use_negative: bool
    """ Set metaball as negative one

    :type: bool
    """

class MetaSequence(Sequence, bpy_struct):
    """Sequence strip to group other strips as a single sequence strip"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    sequences: bpy_prop_collection[Sequence]
    """ 

    :type: bpy_prop_collection[Sequence]
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MirrorModifier(Modifier, bpy_struct):
    """Mirroring modifier"""

    merge_threshold: float
    """ Distance within which mirrored vertices are merged

    :type: float
    """

    mirror_object: Object | None
    """ Object to use as mirror

    :type: Object | None
    """

    use_clip: bool
    """ Prevent vertices from going through the mirror during transform

    :type: bool
    """

    use_mirror_merge: bool
    """ Merge vertices within the merge threshold

    :type: bool
    """

    use_mirror_u: bool
    """ Mirror the U texture coordinate around the 0.5 point

    :type: bool
    """

    use_mirror_v: bool
    """ Mirror the V texture coordinate around the 0.5 point

    :type: bool
    """

    use_mirror_vertex_groups: bool
    """ Mirror vertex groups (e.g. .R->.L)

    :type: bool
    """

    use_x: bool
    """ Enable X axis mirror

    :type: bool
    """

    use_y: bool
    """ Enable Y axis mirror

    :type: bool
    """

    use_z: bool
    """ Enable Z axis mirror

    :type: bool
    """

class Modifier(bpy_struct):
    """Modifier affecting the geometry data of an object"""

    name: str
    """ Modifier name

    :type: str
    """

    show_expanded: bool
    """ Set modifier expanded in the user interface

    :type: bool
    """

    show_in_editmode: bool
    """ Display modifier in Edit mode

    :type: bool
    """

    show_on_cage: bool
    """ Adjust edit cage to modifier result

    :type: bool
    """

    show_render: bool
    """ Use modifier during render

    :type: bool
    """

    show_viewport: bool
    """ Display modifier in viewport

    :type: bool
    """

    type: typing.Literal[
        "DATA_TRANSFER",
        "MESH_CACHE",
        "MESH_SEQUENCE_CACHE",
        "NORMAL_EDIT",
        "UV_PROJECT",
        "UV_WARP",
        "VERTEX_WEIGHT_EDIT",
        "VERTEX_WEIGHT_MIX",
        "VERTEX_WEIGHT_PROXIMITY",
        "ARRAY",
        "BEVEL",
        "BOOLEAN",
        "BUILD",
        "DECIMATE",
        "EDGE_SPLIT",
        "MASK",
        "MIRROR",
        "MULTIRES",
        "REMESH",
        "SCREW",
        "SKIN",
        "SOLIDIFY",
        "SUBSURF",
        "TRIANGULATE",
        "WIREFRAME",
        "ARMATURE",
        "CAST",
        "CORRECTIVE_SMOOTH",
        "CURVE",
        "DISPLACE",
        "HOOK",
        "LAPLACIANSMOOTH",
        "LAPLACIANDEFORM",
        "LATTICE",
        "MESH_DEFORM",
        "SHRINKWRAP",
        "SIMPLE_DEFORM",
        "SMOOTH",
        "WARP",
        "WAVE",
        "CLOTH",
        "COLLISION",
        "DYNAMIC_PAINT",
        "EXPLODE",
        "FLUID_SIMULATION",
        "OCEAN",
        "PARTICLE_INSTANCE",
        "PARTICLE_SYSTEM",
        "SMOKE",
        "SOFT_BODY",
        "SURFACE",
    ]
    """ 

    :type: typing.Literal['DATA_TRANSFER','MESH_CACHE','MESH_SEQUENCE_CACHE','NORMAL_EDIT','UV_PROJECT','UV_WARP','VERTEX_WEIGHT_EDIT','VERTEX_WEIGHT_MIX','VERTEX_WEIGHT_PROXIMITY','ARRAY','BEVEL','BOOLEAN','BUILD','DECIMATE','EDGE_SPLIT','MASK','MIRROR','MULTIRES','REMESH','SCREW','SKIN','SOLIDIFY','SUBSURF','TRIANGULATE','WIREFRAME','ARMATURE','CAST','CORRECTIVE_SMOOTH','CURVE','DISPLACE','HOOK','LAPLACIANSMOOTH','LAPLACIANDEFORM','LATTICE','MESH_DEFORM','SHRINKWRAP','SIMPLE_DEFORM','SMOOTH','WARP','WAVE','CLOTH','COLLISION','DYNAMIC_PAINT','EXPLODE','FLUID_SIMULATION','OCEAN','PARTICLE_INSTANCE','PARTICLE_SYSTEM','SMOKE','SOFT_BODY','SURFACE']
    """

    use_apply_on_spline: bool
    """ Apply this and all preceding deformation modifiers on splines' points rather than on filled curve/surface

    :type: bool
    """

class MotionPath(bpy_struct):
    """Cache of the worldspace positions of an element over a frame range"""

    frame_end: int
    """ End frame of the stored range

    :type: int
    """

    frame_start: int
    """ Starting frame of the stored range

    :type: int
    """

    is_modified: bool
    """ Path is being edited

    :type: bool
    """

    length: int
    """ Number of frames cached

    :type: int
    """

    points: bpy_prop_collection[MotionPathVert]
    """ Cached positions per frame

    :type: bpy_prop_collection[MotionPathVert]
    """

    use_bone_head: bool
    """ For PoseBone paths, use the bone head location when calculating this path

    :type: bool
    """

class MotionPathVert(bpy_struct):
    """Cached location on path"""

    co: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    select: bool
    """ Path point is selected for editing

    :type: bool
    """

class MouseActuator(Actuator, bpy_struct):
    local_x: bool
    """ Apply rotation locally

    :type: bool
    """

    local_y: bool
    """ Apply rotation locally

    :type: bool
    """

    max_x: float
    """ Maximum positive rotation allowed by X mouse movement (0 for infinite)

    :type: float
    """

    max_y: float
    """ Maximum positive rotation allowed by Y mouse movement (0 for infinite)

    :type: float
    """

    min_x: float
    """ Maximum negative rotation allowed by X mouse movement (0 for infinite)

    :type: float
    """

    min_y: float
    """ Maximum negative rotation allowed by Y mouse movement (0 for infinite)

    :type: float
    """

    mode: typing.Literal["VISIBILITY", "LOOK"]
    """ 

    :type: typing.Literal['VISIBILITY','LOOK']
    """

    object_axis_x: typing.Literal["OBJECT_AXIS_X", "OBJECT_AXIS_Y", "OBJECT_AXIS_Z"]
    """ Local object axis mouse movement in the X direction will apply to

    :type: typing.Literal['OBJECT_AXIS_X','OBJECT_AXIS_Y','OBJECT_AXIS_Z']
    """

    object_axis_y: typing.Literal["OBJECT_AXIS_X", "OBJECT_AXIS_Y", "OBJECT_AXIS_Z"]
    """ Local object axis mouse movement in the Y direction will apply to

    :type: typing.Literal['OBJECT_AXIS_X','OBJECT_AXIS_Y','OBJECT_AXIS_Z']
    """

    reset_x: bool
    """ Reset the cursor's X position to the center of the screen space after calculating

    :type: bool
    """

    reset_y: bool
    """ Reset the cursor's Y position to the center of the screen space after calculating

    :type: bool
    """

    sensitivity_x: float
    """ Sensitivity of the X axis

    :type: float
    """

    sensitivity_y: float
    """ Sensitivity of the Y axis

    :type: float
    """

    threshold_x: float
    """ Amount of X motion before mouse movement will register

    :type: float
    """

    threshold_y: float
    """ Amount of Y motion before mouse movement will register

    :type: float
    """

    use_axis_x: bool
    """ Calculate mouse movement on the X axis

    :type: bool
    """

    use_axis_y: bool
    """ Calculate mouse movement on the Y axis

    :type: bool
    """

    visible: bool
    """ Make mouse cursor visible

    :type: bool
    """

class MouseSensor(Sensor, bpy_struct):
    """Sensor to detect mouse events"""

    material: str
    """ Only look for objects with this material (blank = all objects)

    :type: str
    """

    mouse_event: typing.Literal[
        "LEFTCLICK",
        "MIDDLECLICK",
        "RIGHTCLICK",
        "WHEELUP",
        "WHEELDOWN",
        "MOVEMENT",
        "MOUSEOVER",
        "MOUSEOVERANY",
    ]
    """ Type of event this mouse sensor should trigger on

    :type: typing.Literal['LEFTCLICK','MIDDLECLICK','RIGHTCLICK','WHEELUP','WHEELDOWN','MOVEMENT','MOUSEOVER','MOUSEOVERANY']
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    use_material: typing.Literal["PROPERTY", "MATERIAL"]
    """ Toggle collision on material or property

    :type: typing.Literal['PROPERTY','MATERIAL']
    """

    use_pulse: bool
    """ Moving the mouse over a different object generates a pulse

    :type: bool
    """

    use_x_ray: bool
    """ Toggle X-Ray option (see through objects that don't have the property)

    :type: bool
    """

class MovieClip(ID, bpy_struct):
    """MovieClip data-block referencing an external movie file"""

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    display_aspect: bpy_prop_array[float]
    """ Display Aspect for this clip, does not affect rendering

    :type: bpy_prop_array[float]
    """

    filepath: str
    """ Filename of the movie or sequence file

    :type: str
    """

    frame_duration: int
    """ Detected duration of movie clip in frames

    :type: int
    """

    frame_offset: int
    """ Offset of footage first frame relative to it's file name (affects only how footage is loading, does not change data associated with a clip)

    :type: int
    """

    frame_start: int
    """ Global scene frame number at which this movie starts playing (affects all data associated with a clip)

    :type: int
    """

    grease_pencil: GreasePencil | None
    """ Grease pencil data for this movie clip

    :type: GreasePencil | None
    """

    proxy: MovieClipProxy
    """ 

    :type: MovieClipProxy
    """

    size: bpy_prop_array[int]
    """ Width and height in pixels, zero when image data cant be loaded

    :type: bpy_prop_array[int]
    """

    source: typing.Literal["SEQUENCE", "MOVIE"]
    """ Where the clip comes from

    :type: typing.Literal['SEQUENCE','MOVIE']
    """

    tracking: MovieTracking
    """ 

    :type: MovieTracking
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this clip

    :type: bool
    """

    use_proxy_custom_directory: bool
    """ Create proxy images in a custom directory (default is movie location)

    :type: bool
    """

class MovieClipProxy(bpy_struct):
    """Proxy parameters for a movie clip"""

    build_100: bool
    """ Build proxy resolution 100% of the original footage dimension

    :type: bool
    """

    build_25: bool
    """ Build proxy resolution 25% of the original footage dimension

    :type: bool
    """

    build_50: bool
    """ Build proxy resolution 50% of the original footage dimension

    :type: bool
    """

    build_75: bool
    """ Build proxy resolution 75% of the original footage dimension

    :type: bool
    """

    build_free_run: bool
    """ Build free run time code index

    :type: bool
    """

    build_free_run_rec_date: bool
    """ Build free run time code index using Record Date/Time

    :type: bool
    """

    build_record_run: bool
    """ Build record run time code index

    :type: bool
    """

    build_undistorted_100: bool
    """ Build proxy resolution 100% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_25: bool
    """ Build proxy resolution 25% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_50: bool
    """ Build proxy resolution 50% of the original undistorted footage dimension

    :type: bool
    """

    build_undistorted_75: bool
    """ Build proxy resolution 75% of the original undistorted footage dimension

    :type: bool
    """

    directory: str
    """ Location to store the proxy files

    :type: str
    """

    quality: int
    """ JPEG quality of proxy images

    :type: int
    """

    timecode: typing.Literal[
        "NONE", "RECORD_RUN", "FREE_RUN", "FREE_RUN_REC_DATE", "FREE_RUN_NO_GAPS"
    ]
    """ 

    :type: typing.Literal['NONE','RECORD_RUN','FREE_RUN','FREE_RUN_REC_DATE','FREE_RUN_NO_GAPS']
    """

class MovieClipScopes(bpy_struct):
    """Scopes for statistical view of a movie clip"""

class MovieClipSequence(Sequence, bpy_struct):
    """Sequence strip to load a video from the clip editor"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    stabilize2d: bool
    """ Use the 2D stabilized version of the clip

    :type: bool
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    undistort: bool
    """ Use the undistorted version of the clip

    :type: bool
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

class MovieClipUser(bpy_struct):
    """Parameters defining how a MovieClip data-block is used by another data-block"""

    frame_current: int
    """ Current frame number in movie or image sequence

    :type: int
    """

    proxy_render_size: typing.Literal[
        "PROXY_25", "PROXY_50", "PROXY_75", "PROXY_100", "FULL"
    ]
    """ Draw preview using full resolution or different proxy resolutions

    :type: typing.Literal['PROXY_25','PROXY_50','PROXY_75','PROXY_100','FULL']
    """

    use_render_undistorted: bool
    """ Render preview using undistorted proxy

    :type: bool
    """

class MovieReconstructedCamera(bpy_struct):
    """Match-moving reconstructed camera data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    matrix: bpy_prop_array[float]
    """ Worldspace transformation matrix

    :type: bpy_prop_array[float]
    """

class MovieSequence(Sequence, bpy_struct):
    """Sequence strip to load a video"""

    alpha_mode: typing.Literal["STRAIGHT", "PREMUL"]
    """ Representation of alpha information in the RGBA pixels

    :type: typing.Literal['STRAIGHT','PREMUL']
    """

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    color_multiply: float
    """ 

    :type: float
    """

    color_saturation: float
    """ 

    :type: float
    """

    colorspace_settings: ColorManagedInputColorspaceSettings
    """ Input color space settings

    :type: ColorManagedInputColorspaceSettings
    """

    crop: SequenceCrop
    """ 

    :type: SequenceCrop
    """

    elements: bpy_prop_collection[SequenceElement]
    """ 

    :type: bpy_prop_collection[SequenceElement]
    """

    filepath: str
    """ 

    :type: str
    """

    mpeg_preseek: int
    """ For MPEG movies, preseek this many frames

    :type: int
    """

    proxy: SequenceProxy
    """ 

    :type: SequenceProxy
    """

    stereo_3d_format: Stereo3dFormat
    """ Settings for stereo 3d

    :type: Stereo3dFormat
    """

    stream_index: int
    """ For files with several movie streams, use the stream with the given index

    :type: int
    """

    strobe: float
    """ Only display every nth frame

    :type: float
    """

    transform: SequenceTransform
    """ 

    :type: SequenceTransform
    """

    use_crop: bool
    """ Crop image before processing

    :type: bool
    """

    use_deinterlace: bool
    """ For video movies to remove fields

    :type: bool
    """

    use_flip_x: bool
    """ Flip on the X axis

    :type: bool
    """

    use_flip_y: bool
    """ Flip on the Y axis

    :type: bool
    """

    use_float: bool
    """ Convert input to float data

    :type: bool
    """

    use_multiview: bool
    """ Use Multiple Views (when available)

    :type: bool
    """

    use_proxy: bool
    """ Use a preview proxy and/or timecode index for this strip

    :type: bool
    """

    use_reverse_frames: bool
    """ Reverse frame order

    :type: bool
    """

    use_translation: bool
    """ Translate image before processing

    :type: bool
    """

    views_format: typing.Literal["INDIVIDUAL", "STEREO_3D"]
    """ Mode to load movie views

    :type: typing.Literal['INDIVIDUAL','STEREO_3D']
    """

class MovieTracking(bpy_struct):
    """Match-moving data for tracking"""

    active_object_index: int | None
    """ Index of active object

    :type: int | None
    """

    camera: MovieTrackingCamera
    """ 

    :type: MovieTrackingCamera
    """

    dopesheet: MovieTrackingDopesheet
    """ 

    :type: MovieTrackingDopesheet
    """

    objects: MovieTrackingObjects
    """ Collection of objects in this tracking data object

    :type: MovieTrackingObjects
    """

    plane_tracks: MovieTrackingPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    settings: MovieTrackingSettings
    """ 

    :type: MovieTrackingSettings
    """

    stabilization: MovieTrackingStabilization
    """ 

    :type: MovieTrackingStabilization
    """

    tracks: MovieTrackingTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingTracks
    """

class MovieTrackingCamera(bpy_struct):
    """Match-moving camera data for tracking"""

    distortion_model: typing.Literal["POLYNOMIAL", "DIVISION"]
    """ Distortion model used for camera lenses

    :type: typing.Literal['POLYNOMIAL','DIVISION']
    """

    division_k1: float
    """ First coefficient of second order division distortion

    :type: float
    """

    division_k2: float
    """ First coefficient of second order division distortion

    :type: float
    """

    focal_length: float
    """ Camera's focal length

    :type: float
    """

    focal_length_pixels: float
    """ Camera's focal length

    :type: float
    """

    k1: float
    """ First coefficient of third order polynomial radial distortion

    :type: float
    """

    k2: float
    """ Second coefficient of third order polynomial radial distortion

    :type: float
    """

    k3: float
    """ Third coefficient of third order polynomial radial distortion

    :type: float
    """

    pixel_aspect: float
    """ Pixel aspect ratio

    :type: float
    """

    principal: bpy_prop_array[float]
    """ Optical center of lens

    :type: bpy_prop_array[float]
    """

    sensor_width: float
    """ Width of CCD sensor in millimeters

    :type: float
    """

    units: typing.Literal["PIXELS", "MILLIMETERS"]
    """ Units used for camera focal length

    :type: typing.Literal['PIXELS','MILLIMETERS']
    """

class MovieTrackingDopesheet(bpy_struct):
    """Match-moving dopesheet data"""

    show_hidden: bool
    """ Include channels from objects/bone that aren't visible

    :type: bool
    """

    show_only_selected: bool
    """ Only include channels relating to selected objects and data

    :type: bool
    """

    sort_method: typing.Literal["NAME", "LONGEST", "TOTAL", "AVERAGE_ERROR"]
    """ Method to be used to sort channels in dopesheet view

    :type: typing.Literal['NAME','LONGEST','TOTAL','AVERAGE_ERROR']
    """

    use_invert_sort: bool
    """ Invert sort order of dopesheet channels

    :type: bool
    """

class MovieTrackingMarker(bpy_struct):
    """Match-moving marker data for tracking"""

    co: bpy_prop_array[float]
    """ Marker position at frame in normalized coordinates

    :type: bpy_prop_array[float]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    is_keyed: bool
    """ Whether the position of the marker is keyframed or tracked

    :type: bool
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

    pattern_bound_box: bpy_prop_array[float]
    """ Pattern area bounding box in normalized coordinates

    :type: bpy_prop_array[float]
    """

    pattern_corners: bpy_prop_array[float]
    """ Array of coordinates which represents pattern's corners in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    search_max: bpy_prop_array[float]
    """ Right-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

    search_min: bpy_prop_array[float]
    """ Left-bottom corner of search area in normalized coordinates relative to marker position

    :type: bpy_prop_array[float]
    """

class MovieTrackingObject(bpy_struct):
    """Match-moving object tracking and reconstruction data"""

    is_camera: bool
    """ Object is used for camera tracking

    :type: bool
    """

    keyframe_a: int
    """ First keyframe used for reconstruction initialization

    :type: int
    """

    keyframe_b: int
    """ Second keyframe used for reconstruction initialization

    :type: int
    """

    name: str
    """ Unique name of object

    :type: str
    """

    plane_tracks: MovieTrackingObjectPlaneTracks
    """ Collection of plane tracks in this tracking data object

    :type: MovieTrackingObjectPlaneTracks
    """

    reconstruction: MovieTrackingReconstruction
    """ 

    :type: MovieTrackingReconstruction
    """

    scale: float
    """ Scale of object solution in camera space

    :type: float
    """

    tracks: MovieTrackingObjectTracks
    """ Collection of tracks in this tracking data object

    :type: MovieTrackingObjectTracks
    """

class MovieTrackingPlaneMarker(bpy_struct):
    """Match-moving plane marker data for tracking"""

    corners: bpy_prop_array[float]
    """ Array of coordinates which represents UI rectangle corners in frame normalized coordinates

    :type: bpy_prop_array[float]
    """

    frame: int
    """ Frame number marker is keyframed on

    :type: int
    """

    mute: bool
    """ Is marker muted for current frame

    :type: bool
    """

class MovieTrackingPlaneTrack(bpy_struct):
    """Match-moving plane track data for tracking"""

    image: Image | None
    """ Image displayed in the track during editing in clip editor

    :type: Image | None
    """

    image_opacity: float
    """ Opacity of the image

    :type: float
    """

    markers: MovieTrackingPlaneMarkers
    """ Collection of markers in track

    :type: MovieTrackingPlaneMarkers
    """

    name: str
    """ Unique name of track

    :type: str
    """

    select: bool
    """ Plane track is selected

    :type: bool
    """

    use_auto_keying: bool
    """ Automatic keyframe insertion when moving plane corners

    :type: bool
    """

class MovieTrackingReconstruction(bpy_struct):
    """Match-moving reconstruction data from tracker"""

    average_error: float
    """ Average error of reconstruction

    :type: float
    """

    cameras: MovieTrackingReconstructedCameras
    """ Collection of solved cameras

    :type: MovieTrackingReconstructedCameras
    """

    is_valid: bool
    """ Is tracking data contains valid reconstruction information

    :type: bool
    """

class MovieTrackingSettings(bpy_struct):
    """Match moving settings"""

    clean_action: typing.Literal["SELECT", "DELETE_TRACK", "DELETE_SEGMENTS"]
    """ Cleanup action to execute

    :type: typing.Literal['SELECT','DELETE_TRACK','DELETE_SEGMENTS']
    """

    clean_error: float
    """ Effect on tracks which have a larger re-projection error

    :type: float
    """

    clean_frames: int
    """ Effect on tracks which are tracked less than the specified amount of frames

    :type: int
    """

    default_correlation_min: float
    """ Default minimum value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    default_frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    default_margin: int
    """ Default distance from image boundary at which marker stops tracking

    :type: int
    """

    default_motion_model: typing.Literal[
        "Perspective", "Affine", "LocRotScale", "LocScale", "LocRot", "Loc"
    ]
    """ Default motion model to use for tracking

    :type: typing.Literal['Perspective','Affine','LocRotScale','LocScale','LocRot','Loc']
    """

    default_pattern_match: typing.Literal["KEYFRAME", "PREV_FRAME"]
    """ Track pattern from given frame when tracking marker to next frame

    :type: typing.Literal['KEYFRAME','PREV_FRAME']
    """

    default_pattern_size: int
    """ Size of pattern area for newly created tracks

    :type: int
    """

    default_search_size: int
    """ Size of search area for newly created tracks

    :type: int
    """

    default_weight: float
    """ Influence of newly created track on a final solution

    :type: float
    """

    distance: float
    """ Distance between two bundles used for scene scaling

    :type: float
    """

    object_distance: float
    """ Distance between two bundles used for object scaling

    :type: float
    """

    refine_intrinsics: typing.Literal[
        "NONE",
        "FOCAL_LENGTH",
        "FOCAL_LENGTH_RADIAL_K1",
        "FOCAL_LENGTH_RADIAL_K1_K2",
        "FOCAL_LENGTH_PRINCIPAL_POINT_RADIAL_K1_K2",
        "FOCAL_LENGTH_PRINCIPAL_POINT",
        "RADIAL_K1_K2",
    ]
    """ Refine intrinsics during camera solving

    :type: typing.Literal['NONE','FOCAL_LENGTH','FOCAL_LENGTH_RADIAL_K1','FOCAL_LENGTH_RADIAL_K1_K2','FOCAL_LENGTH_PRINCIPAL_POINT_RADIAL_K1_K2','FOCAL_LENGTH_PRINCIPAL_POINT','RADIAL_K1_K2']
    """

    show_default_expanded: bool
    """ Show default options expanded in the user interface

    :type: bool
    """

    show_extra_expanded: bool
    """ Show extra options expanded in the user interface

    :type: bool
    """

    speed: typing.Literal["FASTEST", "DOUBLE", "REALTIME", "HALF", "QUARTER"]
    """ Limit speed of tracking to make visual feedback easier (this does not affect the tracking quality)

    :type: typing.Literal['FASTEST','DOUBLE','REALTIME','HALF','QUARTER']
    """

    use_default_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_default_brute: bool
    """ Use a brute-force translation-only initialization when tracking

    :type: bool
    """

    use_default_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_default_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_default_normalization: bool
    """ Normalize light intensities while tracking (slower)

    :type: bool
    """

    use_default_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    use_keyframe_selection: bool
    """ Automatically select keyframes when solving camera/object motion

    :type: bool
    """

    use_tripod_solver: bool
    """ Use special solver to track a stable camera position, such as a tripod

    :type: bool
    """

class MovieTrackingStabilization(bpy_struct):
    """2D stabilization based on tracking markers"""

    active_rotation_track_index: int | None
    """ Index of active track in rotation stabilization tracks list

    :type: int | None
    """

    active_track_index: int | None
    """ Index of active track in translation stabilization tracks list

    :type: int | None
    """

    anchor_frame: int
    """ Reference point to anchor stabilization (other frames will be adjusted relative to this frame's position)

    :type: int
    """

    filter_type: typing.Literal["NEAREST", "BILINEAR", "BICUBIC"]
    """ Interpolation to use for sub-pixel shifts and rotations due to stabilization

    :type: typing.Literal['NEAREST','BILINEAR','BICUBIC']
    """

    influence_location: float
    """ Influence of stabilization algorithm on footage location

    :type: float
    """

    influence_rotation: float
    """ Influence of stabilization algorithm on footage rotation

    :type: float
    """

    influence_scale: float
    """ Influence of stabilization algorithm on footage scale

    :type: float
    """

    rotation_tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    scale_max: float
    """ Limit the amount of automatic scaling

    :type: float
    """

    show_tracks_expanded: bool
    """ Show UI list of tracks participating in stabilization

    :type: bool
    """

    target_position: bpy_prop_array[float]
    """ Known relative offset of original shot, will be subtracted (e.g. for panning shot, can be animated)

    :type: bpy_prop_array[float]
    """

    target_rotation: float
    """ Rotation present on original shot, will be compensated (e.g. for deliberate tilting)

    :type: float
    """

    target_scale: float
    """ Explicitly scale resulting frame to compensate zoom of original shot

    :type: float
    """

    tracks: bpy_prop_collection[MovieTrackingTrack]
    """ Collection of tracks used for 2D stabilization (translation)

    :type: bpy_prop_collection[MovieTrackingTrack]
    """

    use_2d_stabilization: bool
    """ Use 2D stabilization for footage

    :type: bool
    """

    use_autoscale: bool
    """ Automatically scale footage to cover unfilled areas when stabilizing

    :type: bool
    """

    use_stabilize_rotation: bool
    """ Stabilize detected rotation around center of frame

    :type: bool
    """

    use_stabilize_scale: bool
    """ Compensate any scale changes relative to center of rotation

    :type: bool
    """

class MovieTrackingTrack(bpy_struct):
    """Match-moving track data for tracking"""

    average_error: float
    """ Average error of re-projection

    :type: float
    """

    bundle: bpy_prop_array[float]
    """ Position of bundle reconstructed from this track

    :type: bpy_prop_array[float]
    """

    color: bpy_prop_array[float]
    """ Color of the track in the Movie Clip Editor and the 3D viewport after a solve

    :type: bpy_prop_array[float]
    """

    correlation_min: float
    """ Minimal value of correlation between matched pattern and reference that is still treated as successful tracking

    :type: float
    """

    frames_limit: int
    """ Every tracking cycle, this number of frames are tracked

    :type: int
    """

    grease_pencil: GreasePencil | None
    """ Grease pencil data for this track

    :type: GreasePencil | None
    """

    has_bundle: bool
    """ True if track has a valid bundle

    :type: bool
    """

    hide: bool
    """ Track is hidden

    :type: bool
    """

    lock: bool
    """ Track is locked and all changes to it are disabled

    :type: bool
    """

    margin: int
    """ Distance from image boundary at which marker stops tracking

    :type: int
    """

    markers: MovieTrackingMarkers
    """ Collection of markers in track

    :type: MovieTrackingMarkers
    """

    motion_model: typing.Literal[
        "Perspective", "Affine", "LocRotScale", "LocScale", "LocRot", "Loc"
    ]
    """ Default motion model to use for tracking

    :type: typing.Literal['Perspective','Affine','LocRotScale','LocScale','LocRot','Loc']
    """

    name: str
    """ Unique name of track

    :type: str
    """

    offset: bpy_prop_array[float]
    """ Offset of track from the parenting point

    :type: bpy_prop_array[float]
    """

    pattern_match: typing.Literal["KEYFRAME", "PREV_FRAME"]
    """ Track pattern from given frame when tracking marker to next frame

    :type: typing.Literal['KEYFRAME','PREV_FRAME']
    """

    select: bool
    """ Track is selected

    :type: bool
    """

    select_anchor: bool
    """ Track's anchor point is selected

    :type: bool
    """

    select_pattern: bool
    """ Track's pattern area is selected

    :type: bool
    """

    select_search: bool
    """ Track's search area is selected

    :type: bool
    """

    use_alpha_preview: bool
    """ Apply track's mask on displaying preview

    :type: bool
    """

    use_blue_channel: bool
    """ Use blue channel from footage for tracking

    :type: bool
    """

    use_brute: bool
    """ Use a brute-force translation only pre-track before refinement

    :type: bool
    """

    use_custom_color: bool
    """ Use custom color instead of theme-defined

    :type: bool
    """

    use_grayscale_preview: bool
    """ Display what the tracking algorithm sees in the preview

    :type: bool
    """

    use_green_channel: bool
    """ Use green channel from footage for tracking

    :type: bool
    """

    use_mask: bool
    """ Use a grease pencil data-block as a mask to use only specified areas of pattern when tracking

    :type: bool
    """

    use_normalization: bool
    """ Normalize light intensities while tracking. Slower

    :type: bool
    """

    use_red_channel: bool
    """ Use red channel from footage for tracking

    :type: bool
    """

    weight: float
    """ Influence of this track on a final solution

    :type: float
    """

    weight_stab: float
    """ Influence of this track on 2D stabilization

    :type: float
    """

class MulticamSequence(EffectSequence, Sequence, bpy_struct):
    """Sequence strip to perform multicam editing"""

    animation_offset_end: int
    """ Animation end offset (trim end)

    :type: int
    """

    animation_offset_start: int
    """ Animation start offset (trim start)

    :type: int
    """

    input_count: int
    """ 

    :type: int
    """

    multicam_source: int
    """ 

    :type: int
    """

class MultiplySequence(EffectSequence, Sequence, bpy_struct):
    """Multiply Sequence"""

    input_1: Sequence
    """ First input for the effect strip

    :type: Sequence
    """

    input_2: Sequence
    """ Second input for the effect strip

    :type: Sequence
    """

    input_count: int
    """ 

    :type: int
    """

class MultiresModifier(Modifier, bpy_struct):
    """Multiresolution mesh modifier"""

    filepath: str
    """ Path to external displacements file

    :type: str
    """

    is_external: bool
    """ Store multires displacements outside the .blend file, to save memory

    :type: bool
    """

    levels: int
    """ Number of subdivisions to use in the viewport

    :type: int
    """

    render_levels: int
    """ The subdivision level visible at render time

    :type: int
    """

    sculpt_levels: int
    """ Number of subdivisions to use in sculpt mode

    :type: int
    """

    show_only_control_edges: bool
    """ Skip drawing/rendering of interior subdivided edges

    :type: bool
    """

    subdivision_type: typing.Literal["CATMULL_CLARK", "SIMPLE"]
    """ Select type of subdivision algorithm

    :type: typing.Literal['CATMULL_CLARK','SIMPLE']
    """

    total_levels: int
    """ Number of subdivisions for which displacements are stored

    :type: int
    """

    use_subsurf_uv: bool
    """ Use subsurf to subdivide UVs

    :type: bool
    """

class MusgraveTexture(Texture, ID, bpy_struct):
    """Procedural musgrave texture"""

    dimension_max: float
    """ Highest fractal dimension

    :type: float
    """

    gain: float
    """ The gain multiplier

    :type: float
    """

    lacunarity: float
    """ Gap between successive frequencies

    :type: float
    """

    musgrave_type: typing.Literal[
        "MULTIFRACTAL",
        "RIDGED_MULTIFRACTAL",
        "HYBRID_MULTIFRACTAL",
        "FBM",
        "HETERO_TERRAIN",
    ]
    """ Fractal noise algorithm

    :type: typing.Literal['MULTIFRACTAL','RIDGED_MULTIFRACTAL','HYBRID_MULTIFRACTAL','FBM','HETERO_TERRAIN']
    """

    nabla: float
    """ Size of derivative offset used for calculating normal

    :type: float
    """

    noise_basis: typing.Literal[
        "BLENDER_ORIGINAL",
        "ORIGINAL_PERLIN",
        "IMPROVED_PERLIN",
        "VORONOI_F1",
        "VORONOI_F2",
        "VORONOI_F3",
        "VORONOI_F4",
        "VORONOI_F2_F1",
        "VORONOI_CRACKLE",
        "CELL_NOISE",
    ]
    """ Noise basis used for turbulence

    :type: typing.Literal['BLENDER_ORIGINAL','ORIGINAL_PERLIN','IMPROVED_PERLIN','VORONOI_F1','VORONOI_F2','VORONOI_F3','VORONOI_F4','VORONOI_F2_F1','VORONOI_CRACKLE','CELL_NOISE']
    """

    noise_intensity: float
    """ Intensity of the noise

    :type: float
    """

    noise_scale: float
    """ Scaling for noise input

    :type: float
    """

    octaves: float
    """ Number of frequencies used

    :type: float
    """

    offset: float
    """ The fractal offset

    :type: float
    """

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class NODE_UL_interface_sockets(UIList, bpy_struct):
    def draw_item(
        self, context, layout, data, item, icon, active_data, active_propname, index
    ):
        """

        :param context:
        :param layout:
        :param data:
        :param item:
        :param icon:
        :param active_data:
        :param active_propname:
        :param index:
        """

class NandController(Controller, bpy_struct):
    """Controller passing on events based on a logical NAND operation"""

class NearSensor(Sensor, bpy_struct):
    """Sensor to detect nearby objects"""

    distance: float
    """ Trigger distance

    :type: float
    """

    property: str
    """ Only look for objects with this property (blank = all objects)

    :type: str
    """

    reset_distance: float
    """ The distance where the sensor forgets the actor

    :type: float
    """

class NlaStrip(bpy_struct):
    """A container referencing an existing Action"""

    action: Action | None
    """ Action referenced by this strip

    :type: Action | None
    """

    action_frame_end: float
    """ Last frame from action to use

    :type: float
    """

    action_frame_start: float
    """ First frame from action to use

    :type: float
    """

    active: bool | None
    """ NLA Strip is active

    :type: bool | None
    """

    blend_in: float
    """ Number of frames at start of strip to fade in influence

    :type: float
    """

    blend_out: float
    """ 

    :type: float
    """

    blend_type: typing.Literal["REPLACE", "ADD", "SUBTRACT", "MULTIPLY"]
    """ Method used for combining strip's result with accumulated result

    :type: typing.Literal['REPLACE','ADD','SUBTRACT','MULTIPLY']
    """

    extrapolation: typing.Literal["NOTHING", "HOLD", "HOLD_FORWARD"]
    """ Action to take for gaps past the strip extents

    :type: typing.Literal['NOTHING','HOLD','HOLD_FORWARD']
    """

    fcurves: NlaStripFCurves
    """ F-Curves for controlling the strip's influence and timing

    :type: NlaStripFCurves
    """

    frame_end: float
    """ 

    :type: float
    """

    frame_start: float
    """ 

    :type: float
    """

    influence: float
    """ Amount the strip contributes to the current result

    :type: float
    """

    modifiers: bpy_prop_collection[FModifier]
    """ Modifiers affecting all the F-Curves in the referenced Action

    :type: bpy_prop_collection[FModifier]
    """

    mute: bool
    """ NLA Strip is not evaluated

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    repeat: float
    """ Number of times to repeat the action range

    :type: float
    """

    scale: float
    """ Scaling factor for action

    :type: float
    """

    select: bool
    """ NLA Strip is selected

    :type: bool
    """

    strip_time: float
    """ Frame of referenced Action to evaluate

    :type: float
    """

    strips: bpy_prop_collection[NlaStrip]
    """ NLA Strips that this strip acts as a container for (if it is of type Meta)

    :type: bpy_prop_collection[NlaStrip]
    """

    type: typing.Literal["CLIP", "TRANSITION", "META", "SOUND"]
    """ Type of NLA Strip

    :type: typing.Literal['CLIP','TRANSITION','META','SOUND']
    """

    use_animated_influence: bool
    """ Influence setting is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time: bool
    """ Strip time is controlled by an F-Curve rather than automatically determined

    :type: bool
    """

    use_animated_time_cyclic: bool
    """ Cycle the animated time within the action start & end

    :type: bool
    """

    use_auto_blend: bool
    """ Number of frames for Blending In/Out is automatically determined from overlapping strips

    :type: bool
    """

    use_reverse: bool
    """ NLA Strip is played back in reverse order (only when timing is automatically determined)

    :type: bool
    """

    use_sync_length: bool
    """ Update range of frames referenced from action after tweaking strip and its keyframes

    :type: bool
    """

class NlaTrack(bpy_struct):
    """A animation layer containing Actions referenced as NLA strips"""

    active: bool | None
    """ NLA Track is active

    :type: bool | None
    """

    is_solo: bool
    """ NLA Track is evaluated itself (i.e. active Action and all other NLA Tracks in the same AnimData block are disabled)

    :type: bool
    """

    lock: bool
    """ NLA Track is locked

    :type: bool
    """

    mute: bool
    """ NLA Track is not evaluated

    :type: bool
    """

    name: str
    """ 

    :type: str
    """

    select: bool
    """ NLA Track is selected

    :type: bool
    """

    strips: NlaStrips
    """ NLA Strips on this NLA-track

    :type: NlaStrips
    """

class Node(bpy_struct):
    """Node in a node tree"""

    bl_description: str
    """ 

    :type: str
    """

    bl_height_default: float
    """ 

    :type: float
    """

    bl_height_max: float
    """ 

    :type: float
    """

    bl_height_min: float
    """ 

    :type: float
    """

    bl_icon: typing.Literal[
        "NONE",
        "QUESTION",
        "ERROR",
        "CANCEL",
        "TRIA_RIGHT",
        "TRIA_DOWN",
        "TRIA_LEFT",
        "TRIA_UP",
        "ARROW_LEFTRIGHT",
        "PLUS",
        "DISCLOSURE_TRI_DOWN",
        "DISCLOSURE_TRI_RIGHT",
        "RADIOBUT_OFF",
        "RADIOBUT_ON",
        "MENU_PANEL",
        "BLENDER",
        "GRIP",
        "DOT",
        "COLLAPSEMENU",
        "X",
        "GO_LEFT",
        "PLUG",
        "UI",
        "NODE",
        "NODE_SEL",
        "FULLSCREEN",
        "SPLITSCREEN",
        "RIGHTARROW_THIN",
        "BORDERMOVE",
        "VIEWZOOM",
        "ZOOMIN",
        "ZOOMOUT",
        "PANEL_CLOSE",
        "COPY_ID",
        "EYEDROPPER",
        "LINK_AREA",
        "AUTO",
        "CHECKBOX_DEHLT",
        "CHECKBOX_HLT",
        "UNLOCKED",
        "LOCKED",
        "UNPINNED",
        "PINNED",
        "SCREEN_BACK",
        "RIGHTARROW",
        "DOWNARROW_HLT",
        "DOTSUP",
        "DOTSDOWN",
        "LINK",
        "INLINK",
        "PLUGIN",
        "HELP",
        "GHOST_ENABLED",
        "COLOR",
        "LINKED",
        "UNLINKED",
        "HAND",
        "ZOOM_ALL",
        "ZOOM_SELECTED",
        "ZOOM_PREVIOUS",
        "ZOOM_IN",
        "ZOOM_OUT",
        "RENDER_REGION",
        "BORDER_RECT",
        "BORDER_LASSO",
        "FREEZE",
        "STYLUS_PRESSURE",
        "GHOST_DISABLED",
        "NEW",
        "FILE_TICK",
        "QUIT",
        "URL",
        "RECOVER_LAST",
        "FULLSCREEN_ENTER",
        "FULLSCREEN_EXIT",
        "BLANK1",
        "LAMP",
        "MATERIAL",
        "TEXTURE",
        "ANIM",
        "WORLD",
        "SCENE",
        "EDIT",
        "GAME",
        "RADIO",
        "SCRIPT",
        "PARTICLES",
        "PHYSICS",
        "SPEAKER",
        "TEXTURE_SHADED",
        "VIEW3D",
        "IPO",
        "OOPS",
        "BUTS",
        "FILESEL",
        "IMAGE_COL",
        "INFO",
        "SEQUENCE",
        "TEXT",
        "IMASEL",
        "SOUND",
        "ACTION",
        "NLA",
        "SCRIPTWIN",
        "TIME",
        "NODETREE",
        "LOGIC",
        "CONSOLE",
        "PREFERENCES",
        "CLIP",
        "ASSET_MANAGER",
        "OBJECT_DATAMODE",
        "EDITMODE_HLT",
        "FACESEL_HLT",
        "VPAINT_HLT",
        "TPAINT_HLT",
        "WPAINT_HLT",
        "SCULPTMODE_HLT",
        "POSE_HLT",
        "PARTICLEMODE",
        "LIGHTPAINT",
        "SCENE_DATA",
        "RENDERLAYERS",
        "WORLD_DATA",
        "OBJECT_DATA",
        "MESH_DATA",
        "CURVE_DATA",
        "META_DATA",
        "LATTICE_DATA",
        "LAMP_DATA",
        "MATERIAL_DATA",
        "TEXTURE_DATA",
        "ANIM_DATA",
        "CAMERA_DATA",
        "PARTICLE_DATA",
        "LIBRARY_DATA_DIRECT",
        "GROUP",
        "ARMATURE_DATA",
        "POSE_DATA",
        "BONE_DATA",
        "CONSTRAINT",
        "SHAPEKEY_DATA",
        "CONSTRAINT_BONE",
        "CAMERA_STEREO",
        "PACKAGE",
        "UGLYPACKAGE",
        "BRUSH_DATA",
        "IMAGE_DATA",
        "FILE",
        "FCURVE",
        "FONT_DATA",
        "RENDER_RESULT",
        "SURFACE_DATA",
        "EMPTY_DATA",
        "SETTINGS",
        "RENDER_ANIMATION",
        "RENDER_STILL",
        "LIBRARY_DATA_BROKEN",
        "BOIDS",
        "STRANDS",
        "LIBRARY_DATA_INDIRECT",
        "GREASEPENCIL",
        "LINE_DATA",
        "GROUP_BONE",
        "GROUP_VERTEX",
        "GROUP_VCOL",
        "GROUP_UVS",
        "RNA",
        "RNA_ADD",
        "OUTLINER_OB_EMPTY",
        "OUTLINER_OB_MESH",
        "OUTLINER_OB_CURVE",
        "OUTLINER_OB_LATTICE",
        "OUTLINER_OB_META",
        "OUTLINER_OB_LAMP",
        "OUTLINER_OB_CAMERA",
        "OUTLINER_OB_ARMATURE",
        "OUTLINER_OB_FONT",
        "OUTLINER_OB_SURFACE",
        "OUTLINER_OB_SPEAKER",
        "RESTRICT_COLOR_OFF",
        "RESTRICT_COLOR_ON",
        "RESTRICT_VIEW_OFF",
        "RESTRICT_VIEW_ON",
        "RESTRICT_SELECT_OFF",
        "RESTRICT_SELECT_ON",
        "RESTRICT_RENDER_OFF",
        "RESTRICT_RENDER_ON",
        "OUTLINER_DATA_EMPTY",
        "OUTLINER_DATA_MESH",
        "OUTLINER_DATA_CURVE",
        "OUTLINER_DATA_LATTICE",
        "OUTLINER_DATA_META",
        "OUTLINER_DATA_LAMP",
        "OUTLINER_DATA_CAMERA",
        "OUTLINER_DATA_ARMATURE",
        "OUTLINER_DATA_FONT",
        "OUTLINER_DATA_SURFACE",
        "OUTLINER_DATA_SPEAKER",
        "OUTLINER_DATA_POSE",
        "MESH_PLANE",
        "MESH_CUBE",
        "MESH_CIRCLE",
        "MESH_UVSPHERE",
        "MESH_ICOSPHERE",
        "MESH_GRID",
        "MESH_MONKEY",
        "MESH_CYLINDER",
        "MESH_TORUS",
        "MESH_CONE",
        "MESH_CAPSULE",
        "LAMP_POINT",
        "LAMP_SUN",
        "LAMP_SPOT",
        "LAMP_HEMI",
        "LAMP_AREA",
        "META_EMPTY",
        "META_PLANE",
        "META_CUBE",
        "META_BALL",
        "META_ELLIPSOID",
        "META_CAPSULE",
        "SURFACE_NCURVE",
        "SURFACE_NCIRCLE",
        "SURFACE_NSURFACE",
        "SURFACE_NCYLINDER",
        "SURFACE_NSPHERE",
        "SURFACE_NTORUS",
        "CURVE_BEZCURVE",
        "CURVE_BEZCIRCLE",
        "CURVE_NCURVE",
        "CURVE_NCIRCLE",
        "CURVE_PATH",
        "COLOR_RED",
        "COLOR_GREEN",
        "COLOR_BLUE",
        "TRIA_RIGHT_BAR",
        "TRIA_DOWN_BAR",
        "TRIA_LEFT_BAR",
        "TRIA_UP_BAR",
        "FORCE_FORCE",
        "FORCE_WIND",
        "FORCE_VORTEX",
        "FORCE_MAGNETIC",
        "FORCE_HARMONIC",
        "FORCE_CHARGE",
        "FORCE_LENNARDJONES",
        "FORCE_TEXTURE",
        "FORCE_CURVE",
        "FORCE_BOID",
        "FORCE_TURBULENCE",
        "FORCE_DRAG",
        "FORCE_SMOKEFLOW",
        "NODE_INSERT_ON",
        "NODE_INSERT_OFF",
        "MODIFIER",
        "MOD_WAVE",
        "MOD_BUILD",
        "MOD_DECIM",
        "MOD_MIRROR",
        "MOD_SOFT",
        "MOD_SUBSURF",
        "HOOK",
        "MOD_PHYSICS",
        "MOD_PARTICLES",
        "MOD_BOOLEAN",
        "MOD_EDGESPLIT",
        "MOD_ARRAY",
        "MOD_UVPROJECT",
        "MOD_DISPLACE",
        "MOD_CURVE",
        "MOD_LATTICE",
        "CONSTRAINT_DATA",
        "MOD_ARMATURE",
        "MOD_SHRINKWRAP",
        "MOD_CAST",
        "MOD_MESHDEFORM",
        "MOD_BEVEL",
        "MOD_SMOOTH",
        "MOD_SIMPLEDEFORM",
        "MOD_MASK",
        "MOD_CLOTH",
        "MOD_EXPLODE",
        "MOD_FLUIDSIM",
        "MOD_MULTIRES",
        "MOD_SMOKE",
        "MOD_SOLIDIFY",
        "MOD_SCREW",
        "MOD_VERTEX_WEIGHT",
        "MOD_DYNAMICPAINT",
        "MOD_REMESH",
        "MOD_OCEAN",
        "MOD_WARP",
        "MOD_SKIN",
        "MOD_TRIANGULATE",
        "MOD_WIREFRAME",
        "MOD_DATA_TRANSFER",
        "MOD_NORMALEDIT",
        "REC",
        "PLAY",
        "FF",
        "REW",
        "PAUSE",
        "PREV_KEYFRAME",
        "NEXT_KEYFRAME",
        "PLAY_AUDIO",
        "PLAY_REVERSE",
        "PREVIEW_RANGE",
        "ACTION_TWEAK",
        "PMARKER_ACT",
        "PMARKER_SEL",
        "PMARKER",
        "MARKER_HLT",
        "MARKER",
        "SPACE2",
        "SPACE3",
        "KEYINGSET",
        "KEY_DEHLT",
        "KEY_HLT",
        "MUTE_IPO_OFF",
        "MUTE_IPO_ON",
        "VISIBLE_IPO_OFF",
        "VISIBLE_IPO_ON",
        "DRIVER",
        "SOLO_OFF",
        "SOLO_ON",
        "FRAME_PREV",
        "FRAME_NEXT",
        "NLA_PUSHDOWN",
        "IPO_CONSTANT",
        "IPO_LINEAR",
        "IPO_BEZIER",
        "IPO_SINE",
        "IPO_QUAD",
        "IPO_CUBIC",
        "IPO_QUART",
        "IPO_QUINT",
        "IPO_EXPO",
        "IPO_CIRC",
        "IPO_BOUNCE",
        "IPO_ELASTIC",
        "IPO_BACK",
        "IPO_EASE_IN",
        "IPO_EASE_OUT",
        "IPO_EASE_IN_OUT",
        "VERTEXSEL",
        "EDGESEL",
        "FACESEL",
        "LOOPSEL",
        "ROTATE",
        "CURSOR",
        "ROTATECOLLECTION",
        "ROTATECENTER",
        "ROTACTIVE",
        "ALIGN",
        "SMOOTHCURVE",
        "SPHERECURVE",
        "ROOTCURVE",
        "SHARPCURVE",
        "LINCURVE",
        "NOCURVE",
        "RNDCURVE",
        "PROP_OFF",
        "PROP_ON",
        "PROP_CON",
        "SCULPT_DYNTOPO",
        "PARTICLE_POINT",
        "PARTICLE_TIP",
        "PARTICLE_PATH",
        "MAN_TRANS",
        "MAN_ROT",
        "MAN_SCALE",
        "MANIPUL",
        "SNAP_OFF",
        "SNAP_ON",
        "SNAP_NORMAL",
        "SNAP_GRID",
        "SNAP_VERTEX",
        "SNAP_EDGE",
        "SNAP_FACE",
        "SNAP_VOLUME",
        "SNAP_INCREMENT",
        "STICKY_UVS_LOC",
        "STICKY_UVS_DISABLE",
        "STICKY_UVS_VERT",
        "CLIPUV_DEHLT",
        "CLIPUV_HLT",
        "SNAP_PEEL_OBJECT",
        "GRID",
        "PASTEDOWN",
        "COPYDOWN",
        "PASTEFLIPUP",
        "PASTEFLIPDOWN",
        "SNAP_SURFACE",
        "AUTOMERGE_ON",
        "AUTOMERGE_OFF",
        "RETOPO",
        "UV_VERTEXSEL",
        "UV_EDGESEL",
        "UV_FACESEL",
        "UV_ISLANDSEL",
        "UV_SYNC_SELECT",
        "BBOX",
        "WIRE",
        "SOLID",
        "SMOOTH",
        "POTATO",
        "ORTHO",
        "LOCKVIEW_OFF",
        "LOCKVIEW_ON",
        "AXIS_SIDE",
        "AXIS_FRONT",
        "AXIS_TOP",
        "NDOF_DOM",
        "NDOF_TURN",
        "NDOF_FLY",
        "NDOF_TRANS",
        "LAYER_USED",
        "LAYER_ACTIVE",
        "SORTALPHA",
        "SORTBYEXT",
        "SORTTIME",
        "SORTSIZE",
        "LONGDISPLAY",
        "SHORTDISPLAY",
        "GHOST",
        "IMGDISPLAY",
        "SAVE_AS",
        "SAVE_COPY",
        "BOOKMARKS",
        "FONTPREVIEW",
        "FILTER",
        "NEWFOLDER",
        "OPEN_RECENT",
        "FILE_PARENT",
        "FILE_REFRESH",
        "FILE_FOLDER",
        "FILE_BLANK",
        "FILE_BLEND",
        "FILE_IMAGE",
        "FILE_MOVIE",
        "FILE_SCRIPT",
        "FILE_SOUND",
        "FILE_FONT",
        "FILE_TEXT",
        "RECOVER_AUTO",
        "SAVE_PREFS",
        "LINK_BLEND",
        "APPEND_BLEND",
        "IMPORT",
        "EXPORT",
        "EXTERNAL_DATA",
        "LOAD_FACTORY",
        "LOOP_BACK",
        "LOOP_FORWARDS",
        "BACK",
        "FORWARD",
        "FILE_HIDDEN",
        "FILE_BACKUP",
        "DISK_DRIVE",
        "MATPLANE",
        "MATSPHERE",
        "MATCUBE",
        "MONKEY",
        "HAIR",
        "ALIASED",
        "ANTIALIASED",
        "MAT_SPHERE_SKY",
        "WORDWRAP_OFF",
        "WORDWRAP_ON",
        "SYNTAX_OFF",
        "SYNTAX_ON",
        "LINENUMBERS_OFF",
        "LINENUMBERS_ON",
        "SCRIPTPLUGINS",
        "SEQ_SEQUENCER",
        "SEQ_PREVIEW",
        "SEQ_LUMA_WAVEFORM",
        "SEQ_CHROMA_SCOPE",
        "SEQ_HISTOGRAM",
        "SEQ_SPLITVIEW",
        "IMAGE_RGB",
        "IMAGE_RGB_ALPHA",
        "IMAGE_ALPHA",
        "IMAGE_ZDEPTH",
        "IMAGEFILE",
        "BRUSH_ADD",
        "BRUSH_BLOB",
        "BRUSH_BLUR",
        "BRUSH_CLAY",
        "BRUSH_CLAY_STRIPS",
        "BRUSH_CLONE",
        "BRUSH_CREASE",
        "BRUSH_DARKEN",
        "BRUSH_FILL",
        "BRUSH_FLATTEN",
        "BRUSH_GRAB",
        "BRUSH_INFLATE",
        "BRUSH_LAYER",
        "BRUSH_LIGHTEN",
        "BRUSH_MASK",
        "BRUSH_MIX",
        "BRUSH_MULTIPLY",
        "BRUSH_NUDGE",
        "BRUSH_PINCH",
        "BRUSH_SCRAPE",
        "BRUSH_SCULPT_DRAW",
        "BRUSH_SMEAR",
        "BRUSH_SMOOTH",
        "BRUSH_SNAKE_HOOK",
        "BRUSH_SOFTEN",
        "BRUSH_SUBTRACT",
        "BRUSH_TEXDRAW",
        "BRUSH_TEXFILL",
        "BRUSH_TEXMASK",
        "BRUSH_THUMB",
        "BRUSH_ROTATE",
        "BRUSH_VERTEXDRAW",
        "MATCAP_01",
        "MATCAP_02",
        "MATCAP_03",
        "MATCAP_04",
        "MATCAP_05",
        "MATCAP_06",
        "MATCAP_07",
        "MATCAP_08",
        "MATCAP_09",
        "MATCAP_10",
        "MATCAP_11",
        "MATCAP_12",
        "MATCAP_13",
        "MATCAP_14",
        "MATCAP_15",
        "MATCAP_16",
        "MATCAP_17",
        "MATCAP_18",
        "MATCAP_19",
        "MATCAP_20",
        "MATCAP_21",
        "MATCAP_22",
        "MATCAP_23",
        "MATCAP_24",
    ]
    """ The node icon

    :type: typing.Literal['NONE','QUESTION','ERROR','CANCEL','TRIA_RIGHT','TRIA_DOWN','TRIA_LEFT','TRIA_UP','ARROW_LEFTRIGHT','PLUS','DISCLOSURE_TRI_DOWN','DISCLOSURE_TRI_RIGHT','RADIOBUT_OFF','RADIOBUT_ON','MENU_PANEL','BLENDER','GRIP','DOT','COLLAPSEMENU','X','GO_LEFT','PLUG','UI','NODE','NODE_SEL','FULLSCREEN','SPLITSCREEN','RIGHTARROW_THIN','BORDERMOVE','VIEWZOOM','ZOOMIN','ZOOMOUT','PANEL_CLOSE','COPY_ID','EYEDROPPER','LINK_AREA','AUTO','CHECKBOX_DEHLT','CHECKBOX_HLT','UNLOCKED','LOCKED','UNPINNED','PINNED','SCREEN_BACK','RIGHTARROW','DOWNARROW_HLT','DOTSUP','DOTSDOWN','LINK','INLINK','PLUGIN','HELP','GHOST_ENABLED','COLOR','LINKED','UNLINKED','HAND','ZOOM_ALL','ZOOM_SELECTED','ZOOM_PREVIOUS','ZOOM_IN','ZOOM_OUT','RENDER_REGION','BORDER_RECT','BORDER_LASSO','FREEZE','STYLUS_PRESSURE','GHOST_DISABLED','NEW','FILE_TICK','QUIT','URL','RECOVER_LAST','FULLSCREEN_ENTER','FULLSCREEN_EXIT','BLANK1','LAMP','MATERIAL','TEXTURE','ANIM','WORLD','SCENE','EDIT','GAME','RADIO','SCRIPT','PARTICLES','PHYSICS','SPEAKER','TEXTURE_SHADED','VIEW3D','IPO','OOPS','BUTS','FILESEL','IMAGE_COL','INFO','SEQUENCE','TEXT','IMASEL','SOUND','ACTION','NLA','SCRIPTWIN','TIME','NODETREE','LOGIC','CONSOLE','PREFERENCES','CLIP','ASSET_MANAGER','OBJECT_DATAMODE','EDITMODE_HLT','FACESEL_HLT','VPAINT_HLT','TPAINT_HLT','WPAINT_HLT','SCULPTMODE_HLT','POSE_HLT','PARTICLEMODE','LIGHTPAINT','SCENE_DATA','RENDERLAYERS','WORLD_DATA','OBJECT_DATA','MESH_DATA','CURVE_DATA','META_DATA','LATTICE_DATA','LAMP_DATA','MATERIAL_DATA','TEXTURE_DATA','ANIM_DATA','CAMERA_DATA','PARTICLE_DATA','LIBRARY_DATA_DIRECT','GROUP','ARMATURE_DATA','POSE_DATA','BONE_DATA','CONSTRAINT','SHAPEKEY_DATA','CONSTRAINT_BONE','CAMERA_STEREO','PACKAGE','UGLYPACKAGE','BRUSH_DATA','IMAGE_DATA','FILE','FCURVE','FONT_DATA','RENDER_RESULT','SURFACE_DATA','EMPTY_DATA','SETTINGS','RENDER_ANIMATION','RENDER_STILL','LIBRARY_DATA_BROKEN','BOIDS','STRANDS','LIBRARY_DATA_INDIRECT','GREASEPENCIL','LINE_DATA','GROUP_BONE','GROUP_VERTEX','GROUP_VCOL','GROUP_UVS','RNA','RNA_ADD','OUTLINER_OB_EMPTY','OUTLINER_OB_MESH','OUTLINER_OB_CURVE','OUTLINER_OB_LATTICE','OUTLINER_OB_META','OUTLINER_OB_LAMP','OUTLINER_OB_CAMERA','OUTLINER_OB_ARMATURE','OUTLINER_OB_FONT','OUTLINER_OB_SURFACE','OUTLINER_OB_SPEAKER','RESTRICT_COLOR_OFF','RESTRICT_COLOR_ON','RESTRICT_VIEW_OFF','RESTRICT_VIEW_ON','RESTRICT_SELECT_OFF','RESTRICT_SELECT_ON','RESTRICT_RENDER_OFF','RESTRICT_RENDER_ON','OUTLINER_DATA_EMPTY','OUTLINER_DATA_MESH','OUTLINER_DATA_CURVE','OUTLINER_DATA_LATTICE','OUTLINER_DATA_META','OUTLINER_DATA_LAMP','OUTLINER_DATA_CAMERA','OUTLINER_DATA_ARMATURE','OUTLINER_DATA_FONT','OUTLINER_DATA_SURFACE','OUTLINER_DATA_SPEAKER','OUTLINER_DATA_POSE','MESH_PLANE','MESH_CUBE','MESH_CIRCLE','MESH_UVSPHERE','MESH_ICOSPHERE','MESH_GRID','MESH_MONKEY','MESH_CYLINDER','MESH_TORUS','MESH_CONE','MESH_CAPSULE','LAMP_POINT','LAMP_SUN','LAMP_SPOT','LAMP_HEMI','LAMP_AREA','META_EMPTY','META_PLANE','META_CUBE','META_BALL','META_ELLIPSOID','META_CAPSULE','SURFACE_NCURVE','SURFACE_NCIRCLE','SURFACE_NSURFACE','SURFACE_NCYLINDER','SURFACE_NSPHERE','SURFACE_NTORUS','CURVE_BEZCURVE','CURVE_BEZCIRCLE','CURVE_NCURVE','CURVE_NCIRCLE','CURVE_PATH','COLOR_RED','COLOR_GREEN','COLOR_BLUE','TRIA_RIGHT_BAR','TRIA_DOWN_BAR','TRIA_LEFT_BAR','TRIA_UP_BAR','FORCE_FORCE','FORCE_WIND','FORCE_VORTEX','FORCE_MAGNETIC','FORCE_HARMONIC','FORCE_CHARGE','FORCE_LENNARDJONES','FORCE_TEXTURE','FORCE_CURVE','FORCE_BOID','FORCE_TURBULENCE','FORCE_DRAG','FORCE_SMOKEFLOW','NODE_INSERT_ON','NODE_INSERT_OFF','MODIFIER','MOD_WAVE','MOD_BUILD','MOD_DECIM','MOD_MIRROR','MOD_SOFT','MOD_SUBSURF','HOOK','MOD_PHYSICS','MOD_PARTICLES','MOD_BOOLEAN','MOD_EDGESPLIT','MOD_ARRAY','MOD_UVPROJECT','MOD_DISPLACE','MOD_CURVE','MOD_LATTICE','CONSTRAINT_DATA','MOD_ARMATURE','MOD_SHRINKWRAP','MOD_CAST','MOD_MESHDEFORM','MOD_BEVEL','MOD_SMOOTH','MOD_SIMPLEDEFORM','MOD_MASK','MOD_CLOTH','MOD_EXPLODE','MOD_FLUIDSIM','MOD_MULTIRES','MOD_SMOKE','MOD_SOLIDIFY','MOD_SCREW','MOD_VERTEX_WEIGHT','MOD_DYNAMICPAINT','MOD_REMESH','MOD_OCEAN','MOD_WARP','MOD_SKIN','MOD_TRIANGULATE','MOD_WIREFRAME','MOD_DATA_TRANSFER','MOD_NORMALEDIT','REC','PLAY','FF','REW','PAUSE','PREV_KEYFRAME','NEXT_KEYFRAME','PLAY_AUDIO','PLAY_REVERSE','PREVIEW_RANGE','ACTION_TWEAK','PMARKER_ACT','PMARKER_SEL','PMARKER','MARKER_HLT','MARKER','SPACE2','SPACE3','KEYINGSET','KEY_DEHLT','KEY_HLT','MUTE_IPO_OFF','MUTE_IPO_ON','VISIBLE_IPO_OFF','VISIBLE_IPO_ON','DRIVER','SOLO_OFF','SOLO_ON','FRAME_PREV','FRAME_NEXT','NLA_PUSHDOWN','IPO_CONSTANT','IPO_LINEAR','IPO_BEZIER','IPO_SINE','IPO_QUAD','IPO_CUBIC','IPO_QUART','IPO_QUINT','IPO_EXPO','IPO_CIRC','IPO_BOUNCE','IPO_ELASTIC','IPO_BACK','IPO_EASE_IN','IPO_EASE_OUT','IPO_EASE_IN_OUT','VERTEXSEL','EDGESEL','FACESEL','LOOPSEL','ROTATE','CURSOR','ROTATECOLLECTION','ROTATECENTER','ROTACTIVE','ALIGN','SMOOTHCURVE','SPHERECURVE','ROOTCURVE','SHARPCURVE','LINCURVE','NOCURVE','RNDCURVE','PROP_OFF','PROP_ON','PROP_CON','SCULPT_DYNTOPO','PARTICLE_POINT','PARTICLE_TIP','PARTICLE_PATH','MAN_TRANS','MAN_ROT','MAN_SCALE','MANIPUL','SNAP_OFF','SNAP_ON','SNAP_NORMAL','SNAP_GRID','SNAP_VERTEX','SNAP_EDGE','SNAP_FACE','SNAP_VOLUME','SNAP_INCREMENT','STICKY_UVS_LOC','STICKY_UVS_DISABLE','STICKY_UVS_VERT','CLIPUV_DEHLT','CLIPUV_HLT','SNAP_PEEL_OBJECT','GRID','PASTEDOWN','COPYDOWN','PASTEFLIPUP','PASTEFLIPDOWN','SNAP_SURFACE','AUTOMERGE_ON','AUTOMERGE_OFF','RETOPO','UV_VERTEXSEL','UV_EDGESEL','UV_FACESEL','UV_ISLANDSEL','UV_SYNC_SELECT','BBOX','WIRE','SOLID','SMOOTH','POTATO','ORTHO','LOCKVIEW_OFF','LOCKVIEW_ON','AXIS_SIDE','AXIS_FRONT','AXIS_TOP','NDOF_DOM','NDOF_TURN','NDOF_FLY','NDOF_TRANS','LAYER_USED','LAYER_ACTIVE','SORTALPHA','SORTBYEXT','SORTTIME','SORTSIZE','LONGDISPLAY','SHORTDISPLAY','GHOST','IMGDISPLAY','SAVE_AS','SAVE_COPY','BOOKMARKS','FONTPREVIEW','FILTER','NEWFOLDER','OPEN_RECENT','FILE_PARENT','FILE_REFRESH','FILE_FOLDER','FILE_BLANK','FILE_BLEND','FILE_IMAGE','FILE_MOVIE','FILE_SCRIPT','FILE_SOUND','FILE_FONT','FILE_TEXT','RECOVER_AUTO','SAVE_PREFS','LINK_BLEND','APPEND_BLEND','IMPORT','EXPORT','EXTERNAL_DATA','LOAD_FACTORY','LOOP_BACK','LOOP_FORWARDS','BACK','FORWARD','FILE_HIDDEN','FILE_BACKUP','DISK_DRIVE','MATPLANE','MATSPHERE','MATCUBE','MONKEY','HAIR','ALIASED','ANTIALIASED','MAT_SPHERE_SKY','WORDWRAP_OFF','WORDWRAP_ON','SYNTAX_OFF','SYNTAX_ON','LINENUMBERS_OFF','LINENUMBERS_ON','SCRIPTPLUGINS','SEQ_SEQUENCER','SEQ_PREVIEW','SEQ_LUMA_WAVEFORM','SEQ_CHROMA_SCOPE','SEQ_HISTOGRAM','SEQ_SPLITVIEW','IMAGE_RGB','IMAGE_RGB_ALPHA','IMAGE_ALPHA','IMAGE_ZDEPTH','IMAGEFILE','BRUSH_ADD','BRUSH_BLOB','BRUSH_BLUR','BRUSH_CLAY','BRUSH_CLAY_STRIPS','BRUSH_CLONE','BRUSH_CREASE','BRUSH_DARKEN','BRUSH_FILL','BRUSH_FLATTEN','BRUSH_GRAB','BRUSH_INFLATE','BRUSH_LAYER','BRUSH_LIGHTEN','BRUSH_MASK','BRUSH_MIX','BRUSH_MULTIPLY','BRUSH_NUDGE','BRUSH_PINCH','BRUSH_SCRAPE','BRUSH_SCULPT_DRAW','BRUSH_SMEAR','BRUSH_SMOOTH','BRUSH_SNAKE_HOOK','BRUSH_SOFTEN','BRUSH_SUBTRACT','BRUSH_TEXDRAW','BRUSH_TEXFILL','BRUSH_TEXMASK','BRUSH_THUMB','BRUSH_ROTATE','BRUSH_VERTEXDRAW','MATCAP_01','MATCAP_02','MATCAP_03','MATCAP_04','MATCAP_05','MATCAP_06','MATCAP_07','MATCAP_08','MATCAP_09','MATCAP_10','MATCAP_11','MATCAP_12','MATCAP_13','MATCAP_14','MATCAP_15','MATCAP_16','MATCAP_17','MATCAP_18','MATCAP_19','MATCAP_20','MATCAP_21','MATCAP_22','MATCAP_23','MATCAP_24']
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ The node label

    :type: str
    """

    bl_static_type: typing.Literal["CUSTOM"]
    """ Node type (deprecated, use with care)

    :type: typing.Literal['CUSTOM']
    """

    bl_width_default: float
    """ 

    :type: float
    """

    bl_width_max: float
    """ 

    :type: float
    """

    bl_width_min: float
    """ 

    :type: float
    """

    color: bpy_prop_array[float]
    """ Custom color of the node body

    :type: bpy_prop_array[float]
    """

    dimensions: bpy_prop_array[float]
    """ Absolute bounding box dimensions of the node

    :type: bpy_prop_array[float]
    """

    height: float
    """ Height of the node

    :type: float
    """

    hide: bool
    """ 

    :type: bool
    """

    inputs: NodeInputs
    """ 

    :type: NodeInputs
    """

    internal_links: bpy_prop_collection[NodeLink]
    """ Internal input-to-output connections for muting

    :type: bpy_prop_collection[NodeLink]
    """

    label: str
    """ Optional custom node label

    :type: str
    """

    location: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    mute: bool
    """ 

    :type: bool
    """

    name: str
    """ Unique node identifier

    :type: str
    """

    outputs: NodeOutputs
    """ 

    :type: NodeOutputs
    """

    parent: typing_extensions.Self | None
    """ Parent this node is attached to

    :type: typing_extensions.Self | None
    """

    select: bool
    """ Node selection state

    :type: bool
    """

    shading_compatibility: set[typing.Literal["OLD_SHADING", "NEW_SHADING"]]
    """ 

    :type: set[typing.Literal['OLD_SHADING','NEW_SHADING']]
    """

    show_options: bool
    """ 

    :type: bool
    """

    show_preview: bool
    """ 

    :type: bool
    """

    show_texture: bool
    """ Draw node in viewport textured draw mode

    :type: bool
    """

    type: typing.Literal["CUSTOM"]
    """ Node type (deprecated, use bl_static_type or bl_idname for the actual identifier string)

    :type: typing.Literal['CUSTOM']
    """

    use_custom_color: bool
    """ Use custom color for the node

    :type: bool
    """

    width: float
    """ Width of the node

    :type: float
    """

    width_hidden: float
    """ Width of the node in hidden state

    :type: float
    """

    def socket_value_update(self, context: Context | None):
        """Update after property changes

        :param context:
        :type context: Context | None
        """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def poll(cls, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node type can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """

    def poll_instance(self, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """

    def update(self):
        """Update on editor changes"""

    def insert_link(self, link: NodeLink):
        """Handle creation of a link to or from the node

        :param link: Link, Node link that will be inserted
        :type link: NodeLink
        """

    def init(self, context: Context | None):
        """Initialize a new instance of this node

        :param context:
        :type context: Context | None
        """

    def copy(self, node: typing_extensions.Self):
        """Initialize a new instance of this node from an existing node

        :param node: Node, Existing node to copy
        :type node: typing_extensions.Self
        """

    def free(self):
        """Clean up node on removal"""

    def draw_buttons(self, context: Context | None, layout: UILayout):
        """Draw node buttons

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

    def draw_buttons_ext(self, context: Context | None, layout: UILayout):
        """Draw node buttons in the sidebar

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

    def draw_label(self) -> str:
        """Returns a dynamic label string

        :return: Label
        :rtype: str
        """

class NodeCustomGroup(Node, bpy_struct):
    """Base node type for custom registered node group types"""

    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    node_tree: NodeTree | None
    """ 

    :type: NodeTree | None
    """

class NodeFrame(NodeInternal, Node, bpy_struct):
    label_size: int
    """ Font size to use for displaying the label

    :type: int
    """

    shrink: bool
    """ Shrink the frame to minimal bounding box

    :type: bool
    """

    text: Text | None
    """ 

    :type: Text | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

class NodeGroup(NodeInternal, Node, bpy_struct):
    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    node_tree: NodeTree | None
    """ 

    :type: NodeTree | None
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

class NodeGroupInput(NodeInternal, Node, bpy_struct):
    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

class NodeGroupOutput(NodeInternal, Node, bpy_struct):
    interface: PropertyGroup
    """ Interface socket data

    :type: PropertyGroup
    """

    is_active_output: bool
    """ True if this node is used as the active group output

    :type: bool
    """

    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

class NodeInstanceHash(bpy_struct):
    """Hash table containing node instance data"""

class NodeInternal(Node, bpy_struct):
    @classmethod
    def poll(cls, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node type can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """

    def poll_instance(self, node_tree: NodeTree | None) -> bool:
        """If non-null output is returned, the node can be added to the tree

        :param node_tree: Node Tree
        :type node_tree: NodeTree | None
        :return:
        :rtype: bool
        """

    def update(self):
        """Update on editor changes"""

    def draw_buttons(self, context: Context | None, layout: UILayout):
        """Draw node buttons

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

    def draw_buttons_ext(self, context: Context | None, layout: UILayout):
        """Draw node buttons in the sidebar

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

class NodeInternalSocketTemplate(bpy_struct):
    """Type and default value of a node socket"""

    identifier: str
    """ Identifier of the socket

    :type: str
    """

    name: str
    """ Name of the socket

    :type: str
    """

    type: typing.Literal[
        "CUSTOM", "VALUE", "INT", "BOOLEAN", "VECTOR", "STRING", "RGBA", "SHADER"
    ]
    """ Data type of the socket

    :type: typing.Literal['CUSTOM','VALUE','INT','BOOLEAN','VECTOR','STRING','RGBA','SHADER']
    """

class NodeLink(bpy_struct):
    """Link is valid"""

    from_node: Node
    """ 

    :type: Node
    """

    from_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

    is_hidden: bool
    """ Link is hidden due to invisible sockets

    :type: bool
    """

    is_valid: bool
    """ 

    :type: bool
    """

    to_node: Node
    """ 

    :type: Node
    """

    to_socket: NodeSocket
    """ 

    :type: NodeSocket
    """

class NodeOutputFileSlotFile(bpy_struct):
    """Single layer file slot of the file output node"""

    format: ImageFormatSettings
    """ 

    :type: ImageFormatSettings
    """

    path: str
    """ Subpath used for this slot

    :type: str
    """

    use_node_format: bool
    """ 

    :type: bool
    """

class NodeOutputFileSlotLayer(bpy_struct):
    """Multilayer slot of the file output node"""

    name: str
    """ OpenEXR layer name used for this slot

    :type: str
    """

class NodeReroute(NodeInternal, Node, bpy_struct):
    @classmethod
    def is_registered_node_type(cls) -> bool:
        """True if a registered node type

        :return: Result
        :rtype: bool
        """

    @classmethod
    def input_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Input socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

    @classmethod
    def output_template(cls, index: int | None) -> NodeInternalSocketTemplate:
        """Output socket template

        :param index: Index
        :type index: int | None
        :return: result
        :rtype: NodeInternalSocketTemplate
        """

class NodeSetting(PropertyGroup, bpy_struct):
    value: str
    """ Python expression to be evaluated as the initial node setting

    :type: str
    """

class NodeSocket(bpy_struct):
    """Input or output socket of a node"""

    bl_idname: str
    """ 

    :type: str
    """

    enabled: bool
    """ Enable the socket

    :type: bool
    """

    hide: bool
    """ Hide the socket

    :type: bool
    """

    hide_value: bool
    """ Hide the socket input value

    :type: bool
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_linked: bool
    """ True if the socket is connected

    :type: bool
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    link_limit: int
    """ Max number of links allowed for this socket

    :type: int
    """

    name: str
    """ Socket name

    :type: str
    """

    node: Node
    """ Node owning this socket

    :type: Node
    """

    show_expanded: bool
    """ Socket links are expanded in the user interface

    :type: bool
    """

    type: typing.Literal[
        "CUSTOM", "VALUE", "INT", "BOOLEAN", "VECTOR", "STRING", "RGBA", "SHADER"
    ]
    """ Data type

    :type: typing.Literal['CUSTOM','VALUE','INT','BOOLEAN','VECTOR','STRING','RGBA','SHADER']
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

    def draw(self, context: Context | None, layout: UILayout, node: Node, text: str):
        """Draw socket

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        :param node: Node, Node the socket belongs to
        :type node: Node
        :param text: Text, Text label to draw alongside properties
        :type text: str
        """

    def draw_color(self, context: Context | None, node: Node) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context | None
        :param node: Node, Node the socket belongs to
        :type node: Node
        :return: Color
        :rtype: bpy_prop_array[float]
        """

class NodeSocketBool(NodeSocketStandard, NodeSocket, bpy_struct):
    """Boolean value socket of a node"""

    default_value: bool
    """ Input value used for unconnected socket

    :type: bool
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketColor(NodeSocketStandard, NodeSocket, bpy_struct):
    """RGBA color socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloat(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloatAngle(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloatFactor(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloatPercentage(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloatTime(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketFloatUnsigned(NodeSocketStandard, NodeSocket, bpy_struct):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketInt(NodeSocketStandard, NodeSocket, bpy_struct):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketIntFactor(NodeSocketStandard, NodeSocket, bpy_struct):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketIntPercentage(NodeSocketStandard, NodeSocket, bpy_struct):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketIntUnsigned(NodeSocketStandard, NodeSocket, bpy_struct):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketInterface(bpy_struct):
    """Parameters to define node sockets"""

    bl_socket_idname: str
    """ 

    :type: str
    """

    identifier: str
    """ Unique identifier for mapping sockets

    :type: str
    """

    is_output: bool
    """ True if the socket is an output, otherwise input

    :type: bool
    """

    name: str
    """ Socket name

    :type: str
    """

    def draw(self, context: Context | None, layout: UILayout):
        """Draw template settings

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

    def draw_color(self, context: Context | None) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context | None
        :return: Color
        :rtype: bpy_prop_array[float]
        """

    def register_properties(self, data_rna_type: Struct | None):
        """Define RNA properties of a socket

        :param data_rna_type: Data RNA Type, RNA type for special socket properties
        :type data_rna_type: Struct | None
        """

    def init_socket(self, node: Node, socket: NodeSocket, data_path: str):
        """Initialize a node socket instance

        :param node: Node, Node of the socket to initialize
        :type node: Node
        :param socket: Socket, Socket to initialize
        :type socket: NodeSocket
        :param data_path: Data Path, Path to specialized socket data
        :type data_path: str
        """

    def from_socket(self, node: Node, socket: NodeSocket):
        """Setup template parameters from an existing socket

        :param node: Node, Node of the original socket
        :type node: Node
        :param socket: Socket, Original socket
        :type socket: NodeSocket
        """

class NodeSocketInterfaceBool(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Boolean value socket of a node"""

    default_value: bool
    """ Input value used for unconnected socket

    :type: bool
    """

class NodeSocketInterfaceColor(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """RGBA color socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

class NodeSocketInterfaceFloat(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceFloatAngle(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceFloatFactor(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceFloatPercentage(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceFloatTime(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceFloatUnsigned(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Floating point number socket of a node"""

    default_value: float
    """ Input value used for unconnected socket

    :type: float
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceInt(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    max_value: int
    """ Maximum value

    :type: int
    """

    min_value: int
    """ Minimum value

    :type: int
    """

class NodeSocketInterfaceIntFactor(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    max_value: int
    """ Maximum value

    :type: int
    """

    min_value: int
    """ Minimum value

    :type: int
    """

class NodeSocketInterfaceIntPercentage(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    max_value: int
    """ Maximum value

    :type: int
    """

    min_value: int
    """ Minimum value

    :type: int
    """

class NodeSocketInterfaceIntUnsigned(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Integer number socket of a node"""

    default_value: int
    """ Input value used for unconnected socket

    :type: int
    """

    max_value: int
    """ Maximum value

    :type: int
    """

    min_value: int
    """ Minimum value

    :type: int
    """

class NodeSocketInterfaceShader(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """Shader socket of a node"""

class NodeSocketInterfaceStandard(NodeSocketInterface, bpy_struct):
    type: typing.Literal[
        "CUSTOM", "VALUE", "INT", "BOOLEAN", "VECTOR", "STRING", "RGBA", "SHADER"
    ]
    """ Data type

    :type: typing.Literal['CUSTOM','VALUE','INT','BOOLEAN','VECTOR','STRING','RGBA','SHADER']
    """

    def draw(self, context: Context | None, layout: UILayout):
        """Draw template settings

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        """

    def draw_color(self, context: Context | None) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context | None
        :return: Color
        :rtype: bpy_prop_array[float]
        """

class NodeSocketInterfaceString(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """String socket of a node"""

    default_value: str
    """ Input value used for unconnected socket

    :type: str
    """

class NodeSocketInterfaceVector(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorAcceleration(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorDirection(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorEuler(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorTranslation(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorVelocity(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketInterfaceVectorXYZ(
    NodeSocketInterfaceStandard, NodeSocketInterface, bpy_struct
):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    max_value: float
    """ Maximum value

    :type: float
    """

    min_value: float
    """ Minimum value

    :type: float
    """

class NodeSocketShader(NodeSocketStandard, NodeSocket, bpy_struct):
    """Shader socket of a node"""

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketStandard(NodeSocket, bpy_struct):
    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

    def draw(self, context: Context | None, layout: UILayout, node: Node, text: str):
        """Draw socket

        :param context:
        :type context: Context | None
        :param layout: Layout, Layout in the UI
        :type layout: UILayout
        :param node: Node, Node the socket belongs to
        :type node: Node
        :param text: Text, Text label to draw alongside properties
        :type text: str
        """

    def draw_color(self, context: Context | None, node: Node) -> bpy_prop_array[float]:
        """Color of the socket icon

        :param context:
        :type context: Context | None
        :param node: Node, Node the socket belongs to
        :type node: Node
        :return: Color
        :rtype: bpy_prop_array[float]
        """

class NodeSocketString(NodeSocketStandard, NodeSocket, bpy_struct):
    """String socket of a node"""

    default_value: str
    """ Input value used for unconnected socket

    :type: str
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVector(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorAcceleration(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorDirection(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorEuler(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorTranslation(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorVelocity(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVectorXYZ(NodeSocketStandard, NodeSocket, bpy_struct):
    """3D vector socket of a node"""

    default_value: bpy_prop_array[float]
    """ Input value used for unconnected socket

    :type: bpy_prop_array[float]
    """

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeSocketVirtual(NodeSocketStandard, NodeSocket, bpy_struct):
    """Virtual socket of a node"""

    links: typing.Any
    """ List of node links from or to this socket
(readonly)"""

class NodeTree(ID, bpy_struct):
    """Node tree consisting of linked nodes used for shading, textures and compositing"""

    active_input: int | None
    """ Index of the active input

    :type: int | None
    """

    active_output: int | None
    """ Index of the active output

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    bl_description: str
    """ 

    :type: str
    """

    bl_icon: typing.Literal[
        "NONE",
        "QUESTION",
        "ERROR",
        "CANCEL",
        "TRIA_RIGHT",
        "TRIA_DOWN",
        "TRIA_LEFT",
        "TRIA_UP",
        "ARROW_LEFTRIGHT",
        "PLUS",
        "DISCLOSURE_TRI_DOWN",
        "DISCLOSURE_TRI_RIGHT",
        "RADIOBUT_OFF",
        "RADIOBUT_ON",
        "MENU_PANEL",
        "BLENDER",
        "GRIP",
        "DOT",
        "COLLAPSEMENU",
        "X",
        "GO_LEFT",
        "PLUG",
        "UI",
        "NODE",
        "NODE_SEL",
        "FULLSCREEN",
        "SPLITSCREEN",
        "RIGHTARROW_THIN",
        "BORDERMOVE",
        "VIEWZOOM",
        "ZOOMIN",
        "ZOOMOUT",
        "PANEL_CLOSE",
        "COPY_ID",
        "EYEDROPPER",
        "LINK_AREA",
        "AUTO",
        "CHECKBOX_DEHLT",
        "CHECKBOX_HLT",
        "UNLOCKED",
        "LOCKED",
        "UNPINNED",
        "PINNED",
        "SCREEN_BACK",
        "RIGHTARROW",
        "DOWNARROW_HLT",
        "DOTSUP",
        "DOTSDOWN",
        "LINK",
        "INLINK",
        "PLUGIN",
        "HELP",
        "GHOST_ENABLED",
        "COLOR",
        "LINKED",
        "UNLINKED",
        "HAND",
        "ZOOM_ALL",
        "ZOOM_SELECTED",
        "ZOOM_PREVIOUS",
        "ZOOM_IN",
        "ZOOM_OUT",
        "RENDER_REGION",
        "BORDER_RECT",
        "BORDER_LASSO",
        "FREEZE",
        "STYLUS_PRESSURE",
        "GHOST_DISABLED",
        "NEW",
        "FILE_TICK",
        "QUIT",
        "URL",
        "RECOVER_LAST",
        "FULLSCREEN_ENTER",
        "FULLSCREEN_EXIT",
        "BLANK1",
        "LAMP",
        "MATERIAL",
        "TEXTURE",
        "ANIM",
        "WORLD",
        "SCENE",
        "EDIT",
        "GAME",
        "RADIO",
        "SCRIPT",
        "PARTICLES",
        "PHYSICS",
        "SPEAKER",
        "TEXTURE_SHADED",
        "VIEW3D",
        "IPO",
        "OOPS",
        "BUTS",
        "FILESEL",
        "IMAGE_COL",
        "INFO",
        "SEQUENCE",
        "TEXT",
        "IMASEL",
        "SOUND",
        "ACTION",
        "NLA",
        "SCRIPTWIN",
        "TIME",
        "NODETREE",
        "LOGIC",
        "CONSOLE",
        "PREFERENCES",
        "CLIP",
        "ASSET_MANAGER",
        "OBJECT_DATAMODE",
        "EDITMODE_HLT",
        "FACESEL_HLT",
        "VPAINT_HLT",
        "TPAINT_HLT",
        "WPAINT_HLT",
        "SCULPTMODE_HLT",
        "POSE_HLT",
        "PARTICLEMODE",
        "LIGHTPAINT",
        "SCENE_DATA",
        "RENDERLAYERS",
        "WORLD_DATA",
        "OBJECT_DATA",
        "MESH_DATA",
        "CURVE_DATA",
        "META_DATA",
        "LATTICE_DATA",
        "LAMP_DATA",
        "MATERIAL_DATA",
        "TEXTURE_DATA",
        "ANIM_DATA",
        "CAMERA_DATA",
        "PARTICLE_DATA",
        "LIBRARY_DATA_DIRECT",
        "GROUP",
        "ARMATURE_DATA",
        "POSE_DATA",
        "BONE_DATA",
        "CONSTRAINT",
        "SHAPEKEY_DATA",
        "CONSTRAINT_BONE",
        "CAMERA_STEREO",
        "PACKAGE",
        "UGLYPACKAGE",
        "BRUSH_DATA",
        "IMAGE_DATA",
        "FILE",
        "FCURVE",
        "FONT_DATA",
        "RENDER_RESULT",
        "SURFACE_DATA",
        "EMPTY_DATA",
        "SETTINGS",
        "RENDER_ANIMATION",
        "RENDER_STILL",
        "LIBRARY_DATA_BROKEN",
        "BOIDS",
        "STRANDS",
        "LIBRARY_DATA_INDIRECT",
        "GREASEPENCIL",
        "LINE_DATA",
        "GROUP_BONE",
        "GROUP_VERTEX",
        "GROUP_VCOL",
        "GROUP_UVS",
        "RNA",
        "RNA_ADD",
        "OUTLINER_OB_EMPTY",
        "OUTLINER_OB_MESH",
        "OUTLINER_OB_CURVE",
        "OUTLINER_OB_LATTICE",
        "OUTLINER_OB_META",
        "OUTLINER_OB_LAMP",
        "OUTLINER_OB_CAMERA",
        "OUTLINER_OB_ARMATURE",
        "OUTLINER_OB_FONT",
        "OUTLINER_OB_SURFACE",
        "OUTLINER_OB_SPEAKER",
        "RESTRICT_COLOR_OFF",
        "RESTRICT_COLOR_ON",
        "RESTRICT_VIEW_OFF",
        "RESTRICT_VIEW_ON",
        "RESTRICT_SELECT_OFF",
        "RESTRICT_SELECT_ON",
        "RESTRICT_RENDER_OFF",
        "RESTRICT_RENDER_ON",
        "OUTLINER_DATA_EMPTY",
        "OUTLINER_DATA_MESH",
        "OUTLINER_DATA_CURVE",
        "OUTLINER_DATA_LATTICE",
        "OUTLINER_DATA_META",
        "OUTLINER_DATA_LAMP",
        "OUTLINER_DATA_CAMERA",
        "OUTLINER_DATA_ARMATURE",
        "OUTLINER_DATA_FONT",
        "OUTLINER_DATA_SURFACE",
        "OUTLINER_DATA_SPEAKER",
        "OUTLINER_DATA_POSE",
        "MESH_PLANE",
        "MESH_CUBE",
        "MESH_CIRCLE",
        "MESH_UVSPHERE",
        "MESH_ICOSPHERE",
        "MESH_GRID",
        "MESH_MONKEY",
        "MESH_CYLINDER",
        "MESH_TORUS",
        "MESH_CONE",
        "MESH_CAPSULE",
        "LAMP_POINT",
        "LAMP_SUN",
        "LAMP_SPOT",
        "LAMP_HEMI",
        "LAMP_AREA",
        "META_EMPTY",
        "META_PLANE",
        "META_CUBE",
        "META_BALL",
        "META_ELLIPSOID",
        "META_CAPSULE",
        "SURFACE_NCURVE",
        "SURFACE_NCIRCLE",
        "SURFACE_NSURFACE",
        "SURFACE_NCYLINDER",
        "SURFACE_NSPHERE",
        "SURFACE_NTORUS",
        "CURVE_BEZCURVE",
        "CURVE_BEZCIRCLE",
        "CURVE_NCURVE",
        "CURVE_NCIRCLE",
        "CURVE_PATH",
        "COLOR_RED",
        "COLOR_GREEN",
        "COLOR_BLUE",
        "TRIA_RIGHT_BAR",
        "TRIA_DOWN_BAR",
        "TRIA_LEFT_BAR",
        "TRIA_UP_BAR",
        "FORCE_FORCE",
        "FORCE_WIND",
        "FORCE_VORTEX",
        "FORCE_MAGNETIC",
        "FORCE_HARMONIC",
        "FORCE_CHARGE",
        "FORCE_LENNARDJONES",
        "FORCE_TEXTURE",
        "FORCE_CURVE",
        "FORCE_BOID",
        "FORCE_TURBULENCE",
        "FORCE_DRAG",
        "FORCE_SMOKEFLOW",
        "NODE_INSERT_ON",
        "NODE_INSERT_OFF",
        "MODIFIER",
        "MOD_WAVE",
        "MOD_BUILD",
        "MOD_DECIM",
        "MOD_MIRROR",
        "MOD_SOFT",
        "MOD_SUBSURF",
        "HOOK",
        "MOD_PHYSICS",
        "MOD_PARTICLES",
        "MOD_BOOLEAN",
        "MOD_EDGESPLIT",
        "MOD_ARRAY",
        "MOD_UVPROJECT",
        "MOD_DISPLACE",
        "MOD_CURVE",
        "MOD_LATTICE",
        "CONSTRAINT_DATA",
        "MOD_ARMATURE",
        "MOD_SHRINKWRAP",
        "MOD_CAST",
        "MOD_MESHDEFORM",
        "MOD_BEVEL",
        "MOD_SMOOTH",
        "MOD_SIMPLEDEFORM",
        "MOD_MASK",
        "MOD_CLOTH",
        "MOD_EXPLODE",
        "MOD_FLUIDSIM",
        "MOD_MULTIRES",
        "MOD_SMOKE",
        "MOD_SOLIDIFY",
        "MOD_SCREW",
        "MOD_VERTEX_WEIGHT",
        "MOD_DYNAMICPAINT",
        "MOD_REMESH",
        "MOD_OCEAN",
        "MOD_WARP",
        "MOD_SKIN",
        "MOD_TRIANGULATE",
        "MOD_WIREFRAME",
        "MOD_DATA_TRANSFER",
        "MOD_NORMALEDIT",
        "REC",
        "PLAY",
        "FF",
        "REW",
        "PAUSE",
        "PREV_KEYFRAME",
        "NEXT_KEYFRAME",
        "PLAY_AUDIO",
        "PLAY_REVERSE",
        "PREVIEW_RANGE",
        "ACTION_TWEAK",
        "PMARKER_ACT",
        "PMARKER_SEL",
        "PMARKER",
        "MARKER_HLT",
        "MARKER",
        "SPACE2",
        "SPACE3",
        "KEYINGSET",
        "KEY_DEHLT",
        "KEY_HLT",
        "MUTE_IPO_OFF",
        "MUTE_IPO_ON",
        "VISIBLE_IPO_OFF",
        "VISIBLE_IPO_ON",
        "DRIVER",
        "SOLO_OFF",
        "SOLO_ON",
        "FRAME_PREV",
        "FRAME_NEXT",
        "NLA_PUSHDOWN",
        "IPO_CONSTANT",
        "IPO_LINEAR",
        "IPO_BEZIER",
        "IPO_SINE",
        "IPO_QUAD",
        "IPO_CUBIC",
        "IPO_QUART",
        "IPO_QUINT",
        "IPO_EXPO",
        "IPO_CIRC",
        "IPO_BOUNCE",
        "IPO_ELASTIC",
        "IPO_BACK",
        "IPO_EASE_IN",
        "IPO_EASE_OUT",
        "IPO_EASE_IN_OUT",
        "VERTEXSEL",
        "EDGESEL",
        "FACESEL",
        "LOOPSEL",
        "ROTATE",
        "CURSOR",
        "ROTATECOLLECTION",
        "ROTATECENTER",
        "ROTACTIVE",
        "ALIGN",
        "SMOOTHCURVE",
        "SPHERECURVE",
        "ROOTCURVE",
        "SHARPCURVE",
        "LINCURVE",
        "NOCURVE",
        "RNDCURVE",
        "PROP_OFF",
        "PROP_ON",
        "PROP_CON",
        "SCULPT_DYNTOPO",
        "PARTICLE_POINT",
        "PARTICLE_TIP",
        "PARTICLE_PATH",
        "MAN_TRANS",
        "MAN_ROT",
        "MAN_SCALE",
        "MANIPUL",
        "SNAP_OFF",
        "SNAP_ON",
        "SNAP_NORMAL",
        "SNAP_GRID",
        "SNAP_VERTEX",
        "SNAP_EDGE",
        "SNAP_FACE",
        "SNAP_VOLUME",
        "SNAP_INCREMENT",
        "STICKY_UVS_LOC",
        "STICKY_UVS_DISABLE",
        "STICKY_UVS_VERT",
        "CLIPUV_DEHLT",
        "CLIPUV_HLT",
        "SNAP_PEEL_OBJECT",
        "GRID",
        "PASTEDOWN",
        "COPYDOWN",
        "PASTEFLIPUP",
        "PASTEFLIPDOWN",
        "SNAP_SURFACE",
        "AUTOMERGE_ON",
        "AUTOMERGE_OFF",
        "RETOPO",
        "UV_VERTEXSEL",
        "UV_EDGESEL",
        "UV_FACESEL",
        "UV_ISLANDSEL",
        "UV_SYNC_SELECT",
        "BBOX",
        "WIRE",
        "SOLID",
        "SMOOTH",
        "POTATO",
        "ORTHO",
        "LOCKVIEW_OFF",
        "LOCKVIEW_ON",
        "AXIS_SIDE",
        "AXIS_FRONT",
        "AXIS_TOP",
        "NDOF_DOM",
        "NDOF_TURN",
        "NDOF_FLY",
        "NDOF_TRANS",
        "LAYER_USED",
        "LAYER_ACTIVE",
        "SORTALPHA",
        "SORTBYEXT",
        "SORTTIME",
        "SORTSIZE",
        "LONGDISPLAY",
        "SHORTDISPLAY",
        "GHOST",
        "IMGDISPLAY",
        "SAVE_AS",
        "SAVE_COPY",
        "BOOKMARKS",
        "FONTPREVIEW",
        "FILTER",
        "NEWFOLDER",
        "OPEN_RECENT",
        "FILE_PARENT",
        "FILE_REFRESH",
        "FILE_FOLDER",
        "FILE_BLANK",
        "FILE_BLEND",
        "FILE_IMAGE",
        "FILE_MOVIE",
        "FILE_SCRIPT",
        "FILE_SOUND",
        "FILE_FONT",
        "FILE_TEXT",
        "RECOVER_AUTO",
        "SAVE_PREFS",
        "LINK_BLEND",
        "APPEND_BLEND",
        "IMPORT",
        "EXPORT",
        "EXTERNAL_DATA",
        "LOAD_FACTORY",
        "LOOP_BACK",
        "LOOP_FORWARDS",
        "BACK",
        "FORWARD",
        "FILE_HIDDEN",
        "FILE_BACKUP",
        "DISK_DRIVE",
        "MATPLANE",
        "MATSPHERE",
        "MATCUBE",
        "MONKEY",
        "HAIR",
        "ALIASED",
        "ANTIALIASED",
        "MAT_SPHERE_SKY",
        "WORDWRAP_OFF",
        "WORDWRAP_ON",
        "SYNTAX_OFF",
        "SYNTAX_ON",
        "LINENUMBERS_OFF",
        "LINENUMBERS_ON",
        "SCRIPTPLUGINS",
        "SEQ_SEQUENCER",
        "SEQ_PREVIEW",
        "SEQ_LUMA_WAVEFORM",
        "SEQ_CHROMA_SCOPE",
        "SEQ_HISTOGRAM",
        "SEQ_SPLITVIEW",
        "IMAGE_RGB",
        "IMAGE_RGB_ALPHA",
        "IMAGE_ALPHA",
        "IMAGE_ZDEPTH",
        "IMAGEFILE",
        "BRUSH_ADD",
        "BRUSH_BLOB",
        "BRUSH_BLUR",
        "BRUSH_CLAY",
        "BRUSH_CLAY_STRIPS",
        "BRUSH_CLONE",
        "BRUSH_CREASE",
        "BRUSH_DARKEN",
        "BRUSH_FILL",
        "BRUSH_FLATTEN",
        "BRUSH_GRAB",
        "BRUSH_INFLATE",
        "BRUSH_LAYER",
        "BRUSH_LIGHTEN",
        "BRUSH_MASK",
        "BRUSH_MIX",
        "BRUSH_MULTIPLY",
        "BRUSH_NUDGE",
        "BRUSH_PINCH",
        "BRUSH_SCRAPE",
        "BRUSH_SCULPT_DRAW",
        "BRUSH_SMEAR",
        "BRUSH_SMOOTH",
        "BRUSH_SNAKE_HOOK",
        "BRUSH_SOFTEN",
        "BRUSH_SUBTRACT",
        "BRUSH_TEXDRAW",
        "BRUSH_TEXFILL",
        "BRUSH_TEXMASK",
        "BRUSH_THUMB",
        "BRUSH_ROTATE",
        "BRUSH_VERTEXDRAW",
        "MATCAP_01",
        "MATCAP_02",
        "MATCAP_03",
        "MATCAP_04",
        "MATCAP_05",
        "MATCAP_06",
        "MATCAP_07",
        "MATCAP_08",
        "MATCAP_09",
        "MATCAP_10",
        "MATCAP_11",
        "MATCAP_12",
        "MATCAP_13",
        "MATCAP_14",
        "MATCAP_15",
        "MATCAP_16",
        "MATCAP_17",
        "MATCAP_18",
        "MATCAP_19",
        "MATCAP_20",
        "MATCAP_21",
        "MATCAP_22",
        "MATCAP_23",
        "MATCAP_24",
    ]
    """ The node tree icon

    :type: typing.Literal['NONE','QUESTION','ERROR','CANCEL','TRIA_RIGHT','TRIA_DOWN','TRIA_LEFT','TRIA_UP','ARROW_LEFTRIGHT','PLUS','DISCLOSURE_TRI_DOWN','DISCLOSURE_TRI_RIGHT','RADIOBUT_OFF','RADIOBUT_ON','MENU_PANEL','BLENDER','GRIP','DOT','COLLAPSEMENU','X','GO_LEFT','PLUG','UI','NODE','NODE_SEL','FULLSCREEN','SPLITSCREEN','RIGHTARROW_THIN','BORDERMOVE','VIEWZOOM','ZOOMIN','ZOOMOUT','PANEL_CLOSE','COPY_ID','EYEDROPPER','LINK_AREA','AUTO','CHECKBOX_DEHLT','CHECKBOX_HLT','UNLOCKED','LOCKED','UNPINNED','PINNED','SCREEN_BACK','RIGHTARROW','DOWNARROW_HLT','DOTSUP','DOTSDOWN','LINK','INLINK','PLUGIN','HELP','GHOST_ENABLED','COLOR','LINKED','UNLINKED','HAND','ZOOM_ALL','ZOOM_SELECTED','ZOOM_PREVIOUS','ZOOM_IN','ZOOM_OUT','RENDER_REGION','BORDER_RECT','BORDER_LASSO','FREEZE','STYLUS_PRESSURE','GHOST_DISABLED','NEW','FILE_TICK','QUIT','URL','RECOVER_LAST','FULLSCREEN_ENTER','FULLSCREEN_EXIT','BLANK1','LAMP','MATERIAL','TEXTURE','ANIM','WORLD','SCENE','EDIT','GAME','RADIO','SCRIPT','PARTICLES','PHYSICS','SPEAKER','TEXTURE_SHADED','VIEW3D','IPO','OOPS','BUTS','FILESEL','IMAGE_COL','INFO','SEQUENCE','TEXT','IMASEL','SOUND','ACTION','NLA','SCRIPTWIN','TIME','NODETREE','LOGIC','CONSOLE','PREFERENCES','CLIP','ASSET_MANAGER','OBJECT_DATAMODE','EDITMODE_HLT','FACESEL_HLT','VPAINT_HLT','TPAINT_HLT','WPAINT_HLT','SCULPTMODE_HLT','POSE_HLT','PARTICLEMODE','LIGHTPAINT','SCENE_DATA','RENDERLAYERS','WORLD_DATA','OBJECT_DATA','MESH_DATA','CURVE_DATA','META_DATA','LATTICE_DATA','LAMP_DATA','MATERIAL_DATA','TEXTURE_DATA','ANIM_DATA','CAMERA_DATA','PARTICLE_DATA','LIBRARY_DATA_DIRECT','GROUP','ARMATURE_DATA','POSE_DATA','BONE_DATA','CONSTRAINT','SHAPEKEY_DATA','CONSTRAINT_BONE','CAMERA_STEREO','PACKAGE','UGLYPACKAGE','BRUSH_DATA','IMAGE_DATA','FILE','FCURVE','FONT_DATA','RENDER_RESULT','SURFACE_DATA','EMPTY_DATA','SETTINGS','RENDER_ANIMATION','RENDER_STILL','LIBRARY_DATA_BROKEN','BOIDS','STRANDS','LIBRARY_DATA_INDIRECT','GREASEPENCIL','LINE_DATA','GROUP_BONE','GROUP_VERTEX','GROUP_VCOL','GROUP_UVS','RNA','RNA_ADD','OUTLINER_OB_EMPTY','OUTLINER_OB_MESH','OUTLINER_OB_CURVE','OUTLINER_OB_LATTICE','OUTLINER_OB_META','OUTLINER_OB_LAMP','OUTLINER_OB_CAMERA','OUTLINER_OB_ARMATURE','OUTLINER_OB_FONT','OUTLINER_OB_SURFACE','OUTLINER_OB_SPEAKER','RESTRICT_COLOR_OFF','RESTRICT_COLOR_ON','RESTRICT_VIEW_OFF','RESTRICT_VIEW_ON','RESTRICT_SELECT_OFF','RESTRICT_SELECT_ON','RESTRICT_RENDER_OFF','RESTRICT_RENDER_ON','OUTLINER_DATA_EMPTY','OUTLINER_DATA_MESH','OUTLINER_DATA_CURVE','OUTLINER_DATA_LATTICE','OUTLINER_DATA_META','OUTLINER_DATA_LAMP','OUTLINER_DATA_CAMERA','OUTLINER_DATA_ARMATURE','OUTLINER_DATA_FONT','OUTLINER_DATA_SURFACE','OUTLINER_DATA_SPEAKER','OUTLINER_DATA_POSE','MESH_PLANE','MESH_CUBE','MESH_CIRCLE','MESH_UVSPHERE','MESH_ICOSPHERE','MESH_GRID','MESH_MONKEY','MESH_CYLINDER','MESH_TORUS','MESH_CONE','MESH_CAPSULE','LAMP_POINT','LAMP_SUN','LAMP_SPOT','LAMP_HEMI','LAMP_AREA','META_EMPTY','META_PLANE','META_CUBE','META_BALL','META_ELLIPSOID','META_CAPSULE','SURFACE_NCURVE','SURFACE_NCIRCLE','SURFACE_NSURFACE','SURFACE_NCYLINDER','SURFACE_NSPHERE','SURFACE_NTORUS','CURVE_BEZCURVE','CURVE_BEZCIRCLE','CURVE_NCURVE','CURVE_NCIRCLE','CURVE_PATH','COLOR_RED','COLOR_GREEN','COLOR_BLUE','TRIA_RIGHT_BAR','TRIA_DOWN_BAR','TRIA_LEFT_BAR','TRIA_UP_BAR','FORCE_FORCE','FORCE_WIND','FORCE_VORTEX','FORCE_MAGNETIC','FORCE_HARMONIC','FORCE_CHARGE','FORCE_LENNARDJONES','FORCE_TEXTURE','FORCE_CURVE','FORCE_BOID','FORCE_TURBULENCE','FORCE_DRAG','FORCE_SMOKEFLOW','NODE_INSERT_ON','NODE_INSERT_OFF','MODIFIER','MOD_WAVE','MOD_BUILD','MOD_DECIM','MOD_MIRROR','MOD_SOFT','MOD_SUBSURF','HOOK','MOD_PHYSICS','MOD_PARTICLES','MOD_BOOLEAN','MOD_EDGESPLIT','MOD_ARRAY','MOD_UVPROJECT','MOD_DISPLACE','MOD_CURVE','MOD_LATTICE','CONSTRAINT_DATA','MOD_ARMATURE','MOD_SHRINKWRAP','MOD_CAST','MOD_MESHDEFORM','MOD_BEVEL','MOD_SMOOTH','MOD_SIMPLEDEFORM','MOD_MASK','MOD_CLOTH','MOD_EXPLODE','MOD_FLUIDSIM','MOD_MULTIRES','MOD_SMOKE','MOD_SOLIDIFY','MOD_SCREW','MOD_VERTEX_WEIGHT','MOD_DYNAMICPAINT','MOD_REMESH','MOD_OCEAN','MOD_WARP','MOD_SKIN','MOD_TRIANGULATE','MOD_WIREFRAME','MOD_DATA_TRANSFER','MOD_NORMALEDIT','REC','PLAY','FF','REW','PAUSE','PREV_KEYFRAME','NEXT_KEYFRAME','PLAY_AUDIO','PLAY_REVERSE','PREVIEW_RANGE','ACTION_TWEAK','PMARKER_ACT','PMARKER_SEL','PMARKER','MARKER_HLT','MARKER','SPACE2','SPACE3','KEYINGSET','KEY_DEHLT','KEY_HLT','MUTE_IPO_OFF','MUTE_IPO_ON','VISIBLE_IPO_OFF','VISIBLE_IPO_ON','DRIVER','SOLO_OFF','SOLO_ON','FRAME_PREV','FRAME_NEXT','NLA_PUSHDOWN','IPO_CONSTANT','IPO_LINEAR','IPO_BEZIER','IPO_SINE','IPO_QUAD','IPO_CUBIC','IPO_QUART','IPO_QUINT','IPO_EXPO','IPO_CIRC','IPO_BOUNCE','IPO_ELASTIC','IPO_BACK','IPO_EASE_IN','IPO_EASE_OUT','IPO_EASE_IN_OUT','VERTEXSEL','EDGESEL','FACESEL','LOOPSEL','ROTATE','CURSOR','ROTATECOLLECTION','ROTATECENTER','ROTACTIVE','ALIGN','SMOOTHCURVE','SPHERECURVE','ROOTCURVE','SHARPCURVE','LINCURVE','NOCURVE','RNDCURVE','PROP_OFF','PROP_ON','PROP_CON','SCULPT_DYNTOPO','PARTICLE_POINT','PARTICLE_TIP','PARTICLE_PATH','MAN_TRANS','MAN_ROT','MAN_SCALE','MANIPUL','SNAP_OFF','SNAP_ON','SNAP_NORMAL','SNAP_GRID','SNAP_VERTEX','SNAP_EDGE','SNAP_FACE','SNAP_VOLUME','SNAP_INCREMENT','STICKY_UVS_LOC','STICKY_UVS_DISABLE','STICKY_UVS_VERT','CLIPUV_DEHLT','CLIPUV_HLT','SNAP_PEEL_OBJECT','GRID','PASTEDOWN','COPYDOWN','PASTEFLIPUP','PASTEFLIPDOWN','SNAP_SURFACE','AUTOMERGE_ON','AUTOMERGE_OFF','RETOPO','UV_VERTEXSEL','UV_EDGESEL','UV_FACESEL','UV_ISLANDSEL','UV_SYNC_SELECT','BBOX','WIRE','SOLID','SMOOTH','POTATO','ORTHO','LOCKVIEW_OFF','LOCKVIEW_ON','AXIS_SIDE','AXIS_FRONT','AXIS_TOP','NDOF_DOM','NDOF_TURN','NDOF_FLY','NDOF_TRANS','LAYER_USED','LAYER_ACTIVE','SORTALPHA','SORTBYEXT','SORTTIME','SORTSIZE','LONGDISPLAY','SHORTDISPLAY','GHOST','IMGDISPLAY','SAVE_AS','SAVE_COPY','BOOKMARKS','FONTPREVIEW','FILTER','NEWFOLDER','OPEN_RECENT','FILE_PARENT','FILE_REFRESH','FILE_FOLDER','FILE_BLANK','FILE_BLEND','FILE_IMAGE','FILE_MOVIE','FILE_SCRIPT','FILE_SOUND','FILE_FONT','FILE_TEXT','RECOVER_AUTO','SAVE_PREFS','LINK_BLEND','APPEND_BLEND','IMPORT','EXPORT','EXTERNAL_DATA','LOAD_FACTORY','LOOP_BACK','LOOP_FORWARDS','BACK','FORWARD','FILE_HIDDEN','FILE_BACKUP','DISK_DRIVE','MATPLANE','MATSPHERE','MATCUBE','MONKEY','HAIR','ALIASED','ANTIALIASED','MAT_SPHERE_SKY','WORDWRAP_OFF','WORDWRAP_ON','SYNTAX_OFF','SYNTAX_ON','LINENUMBERS_OFF','LINENUMBERS_ON','SCRIPTPLUGINS','SEQ_SEQUENCER','SEQ_PREVIEW','SEQ_LUMA_WAVEFORM','SEQ_CHROMA_SCOPE','SEQ_HISTOGRAM','SEQ_SPLITVIEW','IMAGE_RGB','IMAGE_RGB_ALPHA','IMAGE_ALPHA','IMAGE_ZDEPTH','IMAGEFILE','BRUSH_ADD','BRUSH_BLOB','BRUSH_BLUR','BRUSH_CLAY','BRUSH_CLAY_STRIPS','BRUSH_CLONE','BRUSH_CREASE','BRUSH_DARKEN','BRUSH_FILL','BRUSH_FLATTEN','BRUSH_GRAB','BRUSH_INFLATE','BRUSH_LAYER','BRUSH_LIGHTEN','BRUSH_MASK','BRUSH_MIX','BRUSH_MULTIPLY','BRUSH_NUDGE','BRUSH_PINCH','BRUSH_SCRAPE','BRUSH_SCULPT_DRAW','BRUSH_SMEAR','BRUSH_SMOOTH','BRUSH_SNAKE_HOOK','BRUSH_SOFTEN','BRUSH_SUBTRACT','BRUSH_TEXDRAW','BRUSH_TEXFILL','BRUSH_TEXMASK','BRUSH_THUMB','BRUSH_ROTATE','BRUSH_VERTEXDRAW','MATCAP_01','MATCAP_02','MATCAP_03','MATCAP_04','MATCAP_05','MATCAP_06','MATCAP_07','MATCAP_08','MATCAP_09','MATCAP_10','MATCAP_11','MATCAP_12','MATCAP_13','MATCAP_14','MATCAP_15','MATCAP_16','MATCAP_17','MATCAP_18','MATCAP_19','MATCAP_20','MATCAP_21','MATCAP_22','MATCAP_23','MATCAP_24']
    """

    bl_idname: str
    """ 

    :type: str
    """

    bl_label: str
    """ The node tree label

    :type: str
    """

    grease_pencil: GreasePencil | None
    """ Grease Pencil data-block

    :type: GreasePencil | None
    """

    inputs: NodeTreeInputs
    """ Node tree inputs

    :type: NodeTreeInputs
    """

    links: NodeLinks
    """ 

    :type: NodeLinks
    """

    nodes: Nodes
    """ 

    :type: Nodes
    """

    outputs: NodeTreeOutputs
    """ Node tree outputs

    :type: NodeTreeOutputs
    """

    type: typing.Literal["SHADER", "TEXTURE", "COMPOSITING"]
    """ Node Tree type (deprecated, bl_idname is the actual node tree type identifier)

    :type: typing.Literal['SHADER','TEXTURE','COMPOSITING']
    """

    view_center: bpy_prop_array[float]
    """ 

    :type: bpy_prop_array[float]
    """

    def interface_update(self, context: Context | None):
        """Updated node group interface

        :param context:
        :type context: Context | None
        """

    @classmethod
    def poll(cls, context: Context | None) -> bool:
        """Check visibility in the editor

        :param context:
        :type context: Context | None
        :return:
        :rtype: bool
        """

    def update(self):
        """Update on editor changes"""

    @classmethod
    def get_from_context(cls, context: Context | None):
        """Get a node tree from the context

                :param context:
                :type context: Context | None
                :return: result_1, Active node tree from context, `NodeTree`

        result_2, ID data block that owns the node tree, `ID`

        result_3, Original ID data block selected from the context, `ID`
        """

class NodeTreePath(bpy_struct):
    """Element of the node space tree path"""

    node_tree: NodeTree
    """ Base node tree from context

    :type: NodeTree
    """

class NoiseTexture(Texture, ID, bpy_struct):
    """Procedural noise texture"""

    users_material: typing.Any
    """ Materials that use this texture
(readonly)"""

    users_object_modifier: typing.Any
    """ Object modifiers that use this texture
(readonly)"""

class NorController(Controller, bpy_struct):
    """Controller passing on events based on a logical NOR operation"""

class NormalEditModifier(Modifier, bpy_struct):
    """Modifier affecting/generating custom normals"""

    invert_vertex_group: bool
    """ Invert vertex group influence

    :type: bool
    """

    mix_factor: float
    """ How much of generated normals to mix with exiting ones

    :type: float
    """

    mix_limit: float
    """ Maximum angle between old and new normals

    :type: float
    """

    mix_mode: typing.Literal["COPY", "ADD", "SUB", "MUL"]
    """ How to mix generated normals with existing ones

    :type: typing.Literal['COPY','ADD','SUB','MUL']
    """

    mode: typing.Literal["RADIAL", "DIRECTIONAL"]
    """ How to affect (generate) normals

    :type: typing.Literal['RADIAL','DIRECTIONAL']
    """

    offset: bpy_prop_array[float]
    """ Offset from object's center

    :type: bpy_prop_array[float]
    """

    target: Object | None
    """ Target object used to affect normals

    :type: Object | None
    """

    use_direction_parallel: bool
    """ Use same direction for all normals, from origin to target's center (Directional mode only)

    :type: bool
    """

    vertex_group: str
    """ Vertex group name for selecting/weighting the affected areas

    :type: str
    """

class Object(ID, bpy_struct):
    """Object data-block defining an object in a scene"""

    active_material: Material | None
    """ Active material being displayed

    :type: Material | None
    """

    active_material_index: int | None
    """ Index of active material slot

    :type: int | None
    """

    active_shape_key: ShapeKey | None
    """ Current shape key

    :type: ShapeKey | None
    """

    active_shape_key_index: int | None
    """ Current shape key index

    :type: int | None
    """

    animation_data: AnimData
    """ Animation data for this data-block

    :type: AnimData
    """

    animation_visualization: AnimViz
    """ Animation data for this data-block

    :type: AnimViz
    """

    bound_box: bpy_prop_array[float]
    """ Object's bounding box in object-space coordinates, all values are -1.0 when not available

    :type: bpy_prop_array[float]
    """

    collision: CollisionSettings
    """ Settings for using the object as a collider in physics simulation

    :type: CollisionSettings
    """

    color: bpy_prop_array[float]
    """ Object color and alpha, used when faces have the ObColor mode enabled

    :type: bpy_prop_array[float]
    """

    constraints: ObjectConstraints
    """ Constraints affecting the transformation of the object

    :type: ObjectConstraints
    """

    cycles: CyclesObjectSettings
    """ Cycles object settings

    :type: CyclesObjectSettings
    """

    cycles_visibility: CyclesVisibilitySettings
    """ Cycles visibility settings

    :type: CyclesVisibilitySettings
    """

    data: Armature | Camera | Curve | GreasePencil | Lattice | Mesh | MetaBall | Speaker | SurfaceCurve | TextCurve | None
    """ 

    :type: Armature | Camera | Curve | GreasePencil | Lattice | Mesh | MetaBall | Speaker | SurfaceCurve | TextCurve | None
    """

    delta_location: bpy_prop_array[float]
    """ Extra translation added to the location of the object

    :type: bpy_prop_array[float]
    """

    delta_rotation_euler: bpy_prop_array[float]
    """ Extra rotation added to the rotation of the object (when using Euler rotations)

    :type: bpy_prop_array[float]
    """

    delta_rotation_quaternion: bpy_prop_array[float]
    """ Extra rotation added to the rotation of the object (when using Quaternion rotations)

    :type: bpy_prop_array[float]
    """

    delta_scale: bpy_prop_array[float]
    """ Extra scaling added to the scale of the object

    :type: bpy_prop_array[float]
    """

    dimensions: bpy_prop_array[float]
    """ Absolute bounding box dimensions of the object

    :type: bpy_prop_array[float]
    """

    draw_bounds_type: typing.Literal["BOX", "SPHERE", "CYLINDER", "CONE", "CAPSULE"]
    """ Object boundary display type

    :type: typing.Literal['BOX','SPHERE','CYLINDER','CONE','CAPSULE']
    """

    draw_type: typing.Literal["BOUNDS", "WIRE", "SOLID", "TEXTURED"]
    """ Maximum draw type to display object with in viewport

    :type: typing.Literal['BOUNDS','WIRE','SOLID','TEXTURED']
    """

    dupli_faces_scale: float
    """ Scale the DupliFace objects

    :type: float
    """

    dupli_frames_end: int
    """ End frame for DupliFrames

    :type: int
    """

    dupli_frames_off: int
    """ Recurring frames to exclude from the Dupliframes

    :type: int
    """

    dupli_frames_on: int
    """ Number of frames to use between DupOff frames

    :type: int
    """

    dupli_frames_start: int
    """ Start frame for DupliFrames

    :type: int
    """

    dupli_group: Group | None
    """ Instance an existing group

    :type: Group | None
    """

    dupli_list: bpy_prop_collection[DupliObject]
    """ Object duplis

    :type: bpy_prop_collection[DupliObject]
    """

    dupli_type: typing.Literal["NONE", "FRAMES", "VERTS", "FACES", "GROUP"]
    """ If not None, object duplication method to use

    :type: typing.Literal['NONE','FRAMES','VERTS','FACES','GROUP']
    """

    empty_draw_size: float
    """ Size of display for empties in the viewport

    :type: float
    """

    empty_draw_type: typing.Literal[
        "PLAIN_AXES",
        "ARROWS",
        "SINGLE_ARROW",
        "CIRCLE",
        "CUBE",
        "SPHERE",
        "CONE",
        "IMAGE",
    ]
    """ Viewport display style for empties

    :type: typing.Literal['PLAIN_AXES','ARROWS','SINGLE_ARROW','CIRCLE','CUBE','SPHERE','CONE','IMAGE']
    """

    empty_image_offset: bpy_prop_array[float]
    """ Origin offset distance

    :type: bpy_prop_array[float]
    """

    field: FieldSettings
    """ Settings for using the object as a field in physics simulation

    :type: FieldSettings
    """

    game: GameObjectSettings
    """ Game engine related settings for the object

    :type: GameObjectSettings
    """

    grease_pencil: GreasePencil | None
    """ Grease Pencil data-block

    :type: GreasePencil | None
    """

    hide: bool
    """ Restrict visibility in the viewport

    :type: bool
    """

    hide_render: bool
    """ Restrict renderability

    :type: bool
    """

    hide_select: bool
    """ Restrict selection in the viewport

    :type: bool
    """

    image_user: ImageUser
    """ Parameters defining which layer, pass and frame of the image is displayed

    :type: ImageUser
    """

    is_duplicator: bool
    """ 

    :type: bool
    """

    layers: bpy_prop_array[bool]
    """ Layers the object is on

    :type: bpy_prop_array[bool]
    """

    layers_local_view: bpy_prop_array[bool]
    """ 3D local view layers the object is on

    :type: bpy_prop_array[bool]
    """

    location: bpy_prop_array[float]
    """ Location of the object

    :type: bpy_prop_array[float]
    """

    lock_location: bpy_prop_array[bool]
    """ Lock editing of location in the interface

    :type: bpy_prop_array[bool]
    """

    lock_rotation: bpy_prop_array[bool]
    """ Lock editing of rotation in the interface

    :type: bpy_prop_array[bool]
    """

    lock_rotation_w: bool
    """ Lock editing of 'angle' component of four-component rotations in the interface

    :type: bool
    """

    lock_rotations_4d: bool
    """ Lock editing of four component rotations by components (instead of as Eulers)

    :type: bool
    """

    lock_scale: bpy_prop_array[bool]
    """ Lock editing of scale in the interface

    :type: bpy_prop_array[bool]
    """

    lod_levels: bpy_prop_collection[LodLevel]
    """ A collection of detail levels to automatically switch between

    :type: bpy_prop_collection[LodLevel]
    """

    material_slots: bpy_prop_collection[MaterialSlot]
    """ Material slots in the object

    :type: bpy_prop_collection[MaterialSlot]
    """

    matrix_basis: bpy_prop_array[float]
    """ Matrix access to location, rotation and scale (including deltas), before constraints and parenting are applied

    :type: bpy_prop_array[float]
    """

    matrix_local: bpy_prop_array[float]
    """ Parent relative transformation matrix - WARNING: Only takes into account 'Object' parenting, so e.g. in case of bone parenting you get a matrix relative to the Armature object, not to the actual parent bone

    :type: bpy_prop_array[float]
    """

    matrix_parent_inverse: bpy_prop_array[float]
    """ Inverse of object's parent matrix at time of parenting

    :type: bpy_prop_array[float]
    """

    matrix_world: bpy_prop_array[float]
    """ Worldspace transformation matrix

    :type: bpy_prop_array[float]
    """

    mode: typing.Literal[
        "OBJECT",
        "EDIT",
        "POSE",
        "SCULPT",
        "VERTEX_PAINT",
        "WEIGHT_PAINT",
        "TEXTURE_PAINT",
        "PARTICLE_EDIT",
        "GPENCIL_EDIT",
    ]
    """ Object interaction mode

    :type: typing.Literal['OBJECT','EDIT','POSE','SCULPT','VERTEX_PAINT','WEIGHT_PAINT','TEXTURE_PAINT','PARTICLE_EDIT','GPENCIL_EDIT']
    """

    modifiers: ObjectModifiers
    """ Modifiers affecting the geometric data of the object

    :type: ObjectModifiers
    """

    motion_path: MotionPath
    """ Motion Path for this element

    :type: MotionPath
    """

    parent: typing_extensions.Self | None
    """ Parent Object

    :type: typing_extensions.Self | None
    """

    parent_bone: str
    """ Name of parent bone in case of a bone parenting relation

    :type: str
    """

    parent_type: typing.Literal[
        "OBJECT", "ARMATURE", "LATTICE", "VERTEX", "VERTEX_3", "BONE"
    ]
    """ Type of parent relation

    :type: typing.Literal