# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

Version 1 of HydPy-H-Land closely emulates the "land components" of HBV96
:cite:p:`ref-Lindstrom1997HBV96` while providing additional functionalities (for
example, the land-use type |SEALED|).  We implemented it on behalf of the `German
Federal Institute of Hydrology (BfG)`_ for modelling large river basins in central
Europe.

The following list summarises the main components of |hland_v1|:

 * Calculate the current from average potential evapotranspiration values.
 * Apply different correction factors on the liquid and the frozen amount of
   precipitation.
 * Simulate interception via simple "bucket" storages.
 * Reflect local spatial heterogeneity of snow by defining snow classes.
 * Redistribute snow from higher to lower zones to prevent "snow towers" in high
   mountain areas.
 * Use a degree-day factor reflecting the seasonal variability of radiation for
   calculating snowmelt.
 * Consider both the melting of ice and the (re)freezing of water within the snow layer.
 * Assume a linear relationship between soil evaporation and relative soil moisture (as
   long as actual evaporation does not exceed potential evaporation).
 * Apply a saturation excess mechanism for the generation of direct runoff.
 * Calculate a (nearly) complete and immediate runoff response for sealed areas.
 * Provide an optional "response area" option, which modifies the usual direct runoff
   and percolation calculation.
 * Distinguish between an upper zone layer related to direct runoff and a lower zone
   layer related to base flow.
 * Pass percolation from the upper to the lower zone layer and capillary rise from the
   upper zone layer to the soil layer.
 * Consider water areas as "groundwater lakes" that are part of the lower zone layer.
 * In contrast to the original HBV96 implementation, handle both the upper and the
   lower zone layer as nonlinear storages.
 * Conceptualise the melting of glacial ice with an additional application of the
   degree-day method.
 * Concentrate surface runoff with a Unit Hydrograph approach.


The following figure shows the general structure of H-Land Version 1.  Note that zones
of type |FIELD| and |FOREST| are based on the same set of process equations:

.. image:: HydPy-H-Land_Version-1.png

Integration tests
=================

.. how_to_understand_integration_tests::

The following integration tests rely on the data used for testing the application model
|lland_v2|.  Hopefully, this eases drawing comparisons between both models.

We perform all integration tests over five days with a simulation step of one hour:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

First, we prepare the model instance and build the connections to element `land` and
node `outlet`:

>>> from hydpy.models.hland_v1 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model

We perform most tests using a single zone of 1 km³ at an altitude of 100 m, consisting
of a single snow class:

>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)

We set the reference elevation levels for precipitation (|ZRelP|), temperature
(|ZRelT|), and evaporation (|ZRelE|) all to 200 m:

>>> zrelp(2.0)
>>> zrelt(2.0)
>>> zrele(2.0)

We assume a runoff coefficient of one for zones defined as |SEALED|:

>>> psi(1.0)

We initialise a test function object, which prepares and runs the tests and prints
their results for the given sequences:

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.p, fluxes.rt, fluxes.qt
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = "%d/%m %H:00"

.. _hland_v1_field:

field
_____

In the first example, we select |FIELD| as the only zone type (note that the results
for the land-use type |FOREST| would be the same):

>>> zonetype(FIELD)

The following set of control parameter values tries to configure application model
|hland_v1| in a manner that allows retracing the influence of the different implemented
methods on the shown results:

>>> pcorr(1.2)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcalt(0.6)
>>> ecorr(0.8)
>>> ecalt(-0.1)
>>> epf(0.1)
>>> etf(0.1)
>>> ered(0.5)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> lp(0.8)
>>> beta(2.0)
>>> percmax(0.5)
>>> cflux(0.1)
>>> alpha(2.0)
>>> k(0.001)
>>> k4(0.005)
>>> gamma(0.0)
>>> maxbaz(3.0)

Initially, relative soil moisture is 50 %, the lower zone layer contains only 10 mm,
and all other storages are empty:

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.uz, 0.0),
...               (states.lz, 10.0),
...               (logs.quh, 0.05))

As mentioned above, the values of the input sequences |P|, |hland_inputs.T|, and |EPN|
stem from :ref:`here <lland_v2_acker_summer>`. For educational purposes, we again use
unrealistically high values of |EPN|.  For the sake of simplicity, we define |TN| to be
constantly 1 °C below |hland_inputs.T|:

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> inputs.tn.series = inputs.t.series - 1.0
>>> inputs.epn.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

We memorise the initial conditions to check later if |hland_v1| holds the water balance:

>>> derived.uh.update()
>>> test.reset_inits()
>>> conditions = sequences.conditions

In the first example, we disable the |RespArea| option and set a relatively large value
for the accuracy-related parameter |RecStep|:

>>> resparea(False)
>>> recstep(100)

The following results show the response of application model |hland_v1| to the given
extreme precipitation event.  The intense evaporation weakens the response markedly.
One striking difference to other models like |lland_v2| is the block-like appearance of
percolation (|Perc|), which is one reason for the unusual transitions between event
periods (consisting of both runoff components |Q0| and |Q1|) and the subsequent pure
base flow periods (consisting of |Q1| only):

.. integration-test::

    >>> test("hland_v1_field")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |     outuh |        rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |      0.5 |  0.000002 | 0.0 | 0.047958 |   0.04796 |  0.046243 |  0.046243 | 0.012845 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.212506 |  9.543605 | 0.012845 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 |      0.5 |  0.000007 | 0.0 | 0.050218 |  0.050225 |  0.047958 |  0.047958 | 0.013322 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.173779 |  9.993387 | 0.013322 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 |      0.5 |   0.00007 | 0.0 | 0.052467 |  0.052537 |  0.050236 |  0.050236 | 0.013954 | 1.954058 | 0.0 | 0.0 |  86.559232 |  0.578886 |  10.44092 | 0.013954 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 |      0.5 |  0.004477 | 0.0 | 0.054705 |  0.059182 |    0.0535 |    0.0535 | 0.014861 | 1.976736 | 0.0 | 0.0 |  96.800491 |   2.37297 | 10.886215 | 0.014861 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 |      0.5 |   0.00942 | 0.0 | 0.056931 |  0.066351 |  0.059298 |  0.059298 | 0.016472 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.834367 | 11.329284 | 0.016472 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 |      0.5 |  0.004491 | 0.0 | 0.059146 |  0.063637 |  0.064155 |  0.064155 | 0.017821 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.454943 | 11.770138 | 0.017821 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |      0.5 |  0.004082 | 0.0 | 0.061351 |  0.065432 |  0.064639 |  0.064639 | 0.017955 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.730216 | 12.208787 | 0.017955 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |      0.5 |  0.005447 | 0.0 | 0.063544 |  0.068991 |  0.065824 |  0.065824 | 0.018284 | 1.936885 | 0.0 | 0.0 | 102.050789 |   1.78762 | 12.645243 | 0.018284 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 |      0.5 |  0.035664 | 0.0 | 0.065726 |   0.10139 |    0.0754 |    0.0754 | 0.020944 | 1.976535 | 0.0 | 0.0 | 111.178794 |  4.407884 | 13.079517 | 0.020944 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |      0.5 |  0.106278 | 0.0 | 0.067898 |  0.174175 |  0.110365 |  0.110365 | 0.030657 |  1.94692 | 0.0 | 0.0 | 114.059742 |   5.03627 | 13.511619 | 0.030657 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 |      0.5 |  0.153262 | 0.0 | 0.070058 |   0.22332 |  0.168922 |  0.168922 | 0.046923 |  1.94647 | 0.0 | 0.0 | 116.771503 |  5.637831 | 13.941561 | 0.046923 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 |      0.5 |  0.167893 | 0.0 | 0.072208 |  0.240101 |  0.216128 |  0.216128 | 0.060036 | 1.864353 | 0.0 | 0.0 |  117.66711 |    5.3965 | 14.369353 | 0.060036 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 |      0.5 |  0.129717 | 0.0 | 0.074347 |  0.204064 |  0.228364 |  0.228364 | 0.063434 | 0.939422 | 0.0 | 0.0 | 117.324353 |  4.725616 | 14.795007 | 0.063434 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 |      0.5 |  0.086003 | 0.0 | 0.076475 |  0.162478 |  0.202831 |  0.202831 | 0.056342 |      0.0 | 0.0 | 0.0 | 115.630662 |  4.098276 | 15.218532 | 0.056342 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 |      0.5 |  0.054998 | 0.0 | 0.078593 |  0.133591 |    0.1653 |    0.1653 | 0.045917 |      0.0 | 0.0 | 0.0 | 114.507032 |  3.501093 | 15.639939 | 0.045917 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 |      0.5 |  0.033312 | 0.0 |   0.0807 |  0.114012 |  0.135659 |  0.135659 | 0.037683 |      0.0 | 0.0 | 0.0 | 113.146516 |  2.925034 | 16.059239 | 0.037683 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 |      0.5 |  0.018617 | 0.0 | 0.082796 |  0.101413 |  0.115563 |  0.115563 | 0.032101 |      0.0 | 0.0 | 0.0 | 109.560616 |  2.362991 | 16.476443 | 0.032101 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 |      0.5 |  0.014291 | 0.0 | 0.084882 |  0.099173 |  0.103715 |  0.103715 |  0.02881 | 0.162204 | 0.0 | 0.0 | 110.559972 |  2.487517 | 16.891561 |  0.02881 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 |      0.5 |  0.027763 | 0.0 | 0.086958 |  0.114721 |  0.103126 |  0.103126 | 0.028646 |      0.0 | 0.0 | 0.0 | 112.061866 |  3.497147 | 17.304603 | 0.028646 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 |      0.5 |  0.033174 | 0.0 | 0.089023 |  0.122197 |  0.112927 |  0.112927 | 0.031369 |      0.0 | 0.0 | 0.0 | 110.237768 |  2.920004 |  17.71558 | 0.031369 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 |      0.5 |  0.275437 | 0.0 | 0.091078 |  0.366515 |  0.174829 |  0.174829 | 0.048564 |  1.84926 | 0.0 | 0.0 | 125.943496 |  8.971084 | 18.124502 | 0.048564 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 |      0.5 |  4.821721 | 0.0 | 0.093123 |  4.914843 |  1.322962 |  1.322962 | 0.367489 | 1.979276 | 0.0 | 0.0 |  153.03683 | 21.420577 |  18.53138 | 0.367489 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 |      0.5 |  7.878038 | 0.0 | 0.095157 |  7.973195 |  4.583738 |  4.583738 |  1.27326 | 1.092627 | 0.0 | 0.0 | 156.642106 |  18.71271 | 18.936223 |  1.27326 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 |      0.5 |  4.840645 | 0.0 | 0.097181 |  4.937827 |  6.619035 |  6.619035 | 1.838621 | 1.133348 | 0.0 | 0.0 | 157.537874 | 15.417906 | 19.339042 | 1.838621 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 |      0.5 |    3.9218 | 0.0 | 0.099195 |  4.020995 |  5.408613 |  5.408613 | 1.502392 | 1.900086 | 0.0 | 0.0 | 160.560437 | 15.966933 | 19.739846 | 1.502392 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 |      0.5 |  7.677383 | 0.0 | 0.101199 |  7.778582 |  5.059755 |  5.059755 | 1.405487 |  1.99141 | 0.0 | 0.0 | 168.349822 | 21.873957 | 20.138647 | 1.405487 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 |      0.5 | 11.510418 | 0.0 | 0.103193 | 11.613611 |  7.795791 |  7.795791 | 2.165498 | 1.987608 | 0.0 | 0.0 | 173.686572 | 22.807202 | 20.535454 | 2.165498 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |      0.5 |  8.843986 | 0.0 | 0.105177 |  8.949163 | 10.169283 | 10.169283 | 2.824801 | 1.963945 | 0.0 | 0.0 | 175.517585 | 19.086184 | 20.930277 | 2.824801 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 |      0.5 |  4.947934 | 0.0 | 0.107151 |  5.055086 |  8.675912 |  8.675912 | 2.409976 | 1.940232 | 0.0 | 0.0 | 176.010193 | 15.336902 | 21.323125 | 2.409976 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |      0.5 |  3.725205 | 0.0 | 0.109116 |   3.83432 |  5.649155 |  5.649155 |  1.56921 | 1.958069 | 0.0 | 0.0 | 177.299835 | 15.562521 |  21.71401 |  1.56921 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 |      0.5 |  3.288302 | 0.0 |  0.11107 |  3.399372 |  4.008946 |  4.008946 | 1.113596 |  1.94565 | 0.0 | 0.0 | 177.961502 | 14.251047 |  22.10294 | 1.113596 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 |      0.5 |  2.220294 | 0.0 | 0.113015 |  2.333308 |  3.259124 |  3.259124 | 0.905312 | 1.930107 | 0.0 | 0.0 | 178.049897 | 11.947062 | 22.489925 | 0.905312 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 |      0.5 |  1.351723 | 0.0 |  0.11495 |  1.466673 |  2.377626 |  2.377626 | 0.660452 | 1.927284 | 0.0 | 0.0 | 178.059289 | 10.217297 | 22.874975 | 0.660452 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 |      0.5 |  0.912964 | 0.0 | 0.116875 |  1.029839 |  1.562184 |  1.562184 |  0.43394 | 1.929004 | 0.0 | 0.0 | 178.142807 |  9.206601 |   23.2581 |  0.43394 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 |      0.5 |  0.896634 | 0.0 | 0.118791 |  1.015425 |   1.12371 |   1.12371 | 0.312142 | 1.942479 | 0.0 | 0.0 | 178.699157 |  10.00506 |  23.63931 | 0.312142 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 |      0.5 |  0.851988 | 0.0 | 0.120697 |  0.972684 |   1.00913 |   1.00913 | 0.280314 | 1.876663 | 0.0 | 0.0 | 178.732328 |  8.975912 | 24.018613 | 0.280314 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 |      0.5 |  0.604807 | 0.0 | 0.122593 |    0.7274 |  0.927674 |  0.927674 | 0.257687 | 1.616372 | 0.0 | 0.0 | 178.574145 |  7.951736 |  24.39602 | 0.257687 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 |      0.5 |  0.419019 | 0.0 |  0.12448 |  0.543499 |  0.741041 |  0.741041 | 0.205845 | 0.636076 | 0.0 | 0.0 |  178.09471 |  7.022004 |  24.77154 | 0.205845 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 |      0.5 |  0.290078 | 0.0 | 0.126358 |  0.416436 |   0.55613 |   0.55613 |  0.15448 |      0.0 | 0.0 | 0.0 | 174.913943 |  6.220974 | 25.145182 |  0.15448 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 |      0.5 |  0.201588 | 0.0 | 0.128226 |  0.329814 |  0.425423 |  0.425423 | 0.118173 |      0.0 | 0.0 | 0.0 | 171.590941 |  5.506843 | 25.516957 | 0.118173 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 |      0.5 |  0.172588 | 0.0 | 0.130085 |  0.302673 |  0.343032 |  0.343032 | 0.095287 |      0.0 | 0.0 | 0.0 | 169.638537 |  5.621644 | 25.886872 | 0.095287 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 |      0.5 |  0.148076 | 0.0 | 0.131934 |   0.28001 |  0.303668 |  0.303668 | 0.084352 |      0.0 | 0.0 | 0.0 | 165.937749 |  4.958388 | 26.254937 | 0.084352 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 |      0.5 |  0.100609 | 0.0 | 0.133775 |  0.234383 |  0.274907 |  0.274907 | 0.076363 |      0.0 | 0.0 | 0.0 | 162.378092 |  4.340748 | 26.621163 | 0.076363 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 |      0.5 |  0.066455 | 0.0 | 0.135606 |  0.202061 |   0.23734 |   0.23734 | 0.065928 |      0.0 | 0.0 | 0.0 |  156.29702 |  3.755482 | 26.985557 | 0.065928 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |      0.5 |  0.042037 | 0.0 | 0.137428 |  0.179465 |  0.204222 |  0.204222 | 0.056728 |      0.0 | 0.0 | 0.0 | 152.360068 |  3.191593 | 27.348129 | 0.056728 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 |      0.5 |  0.024952 | 0.0 | 0.139241 |  0.164192 |  0.181093 |  0.181093 | 0.050303 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.642821 | 27.708889 | 0.050303 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 |      0.5 |  0.013481 | 0.0 | 0.141044 |  0.154525 |  0.165438 |  0.165438 | 0.045955 |      0.0 | 0.0 | 0.0 | 145.646557 |  2.103421 | 28.067844 | 0.045955 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 |      0.5 |  0.006295 | 0.0 | 0.142839 |  0.149135 |  0.155476 |  0.155476 | 0.043188 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.569949 | 28.425005 | 0.043188 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 |      0.5 |  0.002299 | 0.0 | 0.144625 |  0.146924 |  0.149841 |  0.149841 | 0.041623 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.039914 |  28.78038 | 0.041623 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 |      0.5 |  0.000516 | 0.0 | 0.146402 |  0.146918 |  0.147414 |  0.147414 | 0.040948 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.511573 | 29.133978 | 0.040948 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.483697 |  0.000032 | 0.0 | 0.148088 |   0.14812 |  0.147186 |  0.147186 | 0.040885 |      0.0 | 0.0 | 0.0 |  144.27384 |       0.0 | 29.469586 | 0.040885 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   0.0635 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.147348 |  0.147348 |  0.147681 |  0.147681 | 0.041023 |      0.0 | 0.0 | 0.0 |  144.21034 |       0.0 | 29.322238 | 0.041023 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062642 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.146611 |  0.146611 |  0.147356 |  0.147356 | 0.040932 |      0.0 | 0.0 | 0.0 | 144.147698 |       0.0 | 29.175627 | 0.040932 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061918 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145878 |  0.145878 |  0.146612 |  0.146612 | 0.040726 |      0.0 | 0.0 | 0.0 |  144.08578 |       0.0 | 29.029749 | 0.040726 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063555 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145149 |  0.145149 |  0.145879 |  0.145879 | 0.040522 |      0.0 | 0.0 | 0.0 | 144.022225 |       0.0 |   28.8846 | 0.040522 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   0.0638 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.144423 |  0.144423 |   0.14515 |   0.14515 | 0.040319 |      0.0 | 0.0 | 0.0 | 143.958424 |       0.0 | 28.740177 | 0.040319 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026911 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.143701 |  0.143701 |  0.144424 |  0.144424 | 0.040118 | 1.484581 | 0.0 | 0.0 | 143.931513 |       0.0 | 28.596476 | 0.040118 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.03133 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142982 |  0.142982 |  0.143702 |  0.143702 | 0.039917 | 1.414925 | 0.0 | 0.0 | 143.900183 |       0.0 | 28.453494 | 0.039917 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031873 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142267 |  0.142267 |  0.142983 |  0.142983 | 0.039718 | 1.344047 | 0.0 | 0.0 |  143.86831 |       0.0 | 28.311226 | 0.039718 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.11649 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.141556 |  0.141556 |  0.142268 |  0.142268 | 0.039519 | 1.084943 | 0.0 | 0.0 |  143.75182 |       0.0 |  28.16967 | 0.039519 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480578 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140848 |  0.140848 |  0.141557 |  0.141557 | 0.039321 | 0.846748 | 0.0 | 0.0 | 143.271242 |       0.0 | 28.028822 | 0.039321 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751285 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140144 |  0.140144 |  0.140849 |  0.140849 | 0.039125 |      0.0 | 0.0 | 0.0 | 141.519957 |       0.0 | 27.888678 | 0.039125 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.789957 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.139443 |  0.139443 |  0.140145 |  0.140145 | 0.038929 |      0.0 | 0.0 | 0.0 |     137.73 |       0.0 | 27.749234 | 0.038929 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.905386 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138746 |  0.138746 |  0.139444 |  0.139444 | 0.038734 |      0.0 | 0.0 | 0.0 | 132.824614 |       0.0 | 27.610488 | 0.038734 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.818153 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138052 |  0.138052 |  0.138747 |  0.138747 | 0.038541 |      0.0 | 0.0 | 0.0 | 127.006461 |       0.0 | 27.472436 | 0.038541 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.563202 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137362 |  0.137362 |  0.138053 |  0.138053 | 0.038348 |      0.0 | 0.0 | 0.0 | 121.443259 |       0.0 | 27.335074 | 0.038348 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.792752 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.136675 |  0.136675 |  0.137363 |  0.137363 | 0.038156 |      0.0 | 0.0 | 0.0 | 114.650507 |       0.0 | 27.198398 | 0.038156 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.615733 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135992 |  0.135992 |  0.136676 |  0.136676 | 0.037966 |      0.0 | 0.0 | 0.0 | 109.034774 |       0.0 | 27.062406 | 0.037966 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   4.4747 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135312 |  0.135312 |  0.135993 |  0.135993 | 0.037776 |      0.0 | 0.0 | 0.0 | 104.560075 |       0.0 | 26.927094 | 0.037776 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.437651 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.134635 |  0.134635 |  0.135313 |  0.135313 | 0.037587 |      0.0 | 0.0 | 0.0 | 101.122424 |       0.0 | 26.792459 | 0.037587 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556348 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133962 |  0.133962 |  0.134636 |  0.134636 | 0.037399 |      0.0 | 0.0 | 0.0 |  99.566076 |       0.0 | 26.658496 | 0.037399 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680454 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133292 |  0.133292 |  0.133963 |  0.133963 | 0.037212 |      0.0 | 0.0 | 0.0 |  98.885622 |       0.0 | 26.525204 | 0.037212 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170108 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.132626 |  0.132626 |  0.133293 |  0.133293 | 0.037026 |      0.0 | 0.0 | 0.0 |  98.715514 |       0.0 | 26.392578 | 0.037026 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046123 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131963 |  0.131963 |  0.132627 |  0.132627 | 0.036841 |      0.0 | 0.0 | 0.0 |  98.669391 |       0.0 | 26.260615 | 0.036841 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044177 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131303 |  0.131303 |  0.131964 |  0.131964 | 0.036657 |      0.0 | 0.0 | 0.0 |  98.625215 |       0.0 | 26.129312 | 0.036657 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043503 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.130647 |  0.130647 |  0.131304 |  0.131304 | 0.036473 |      0.0 | 0.0 | 0.0 |  98.581712 |       0.0 | 25.998665 | 0.036473 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043671 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.129993 |  0.129993 |  0.130647 |  0.130647 | 0.036291 |      0.0 | 0.0 | 0.0 |  98.538041 |       0.0 | 25.868672 | 0.036291 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_resparea:

contributing area
_________________

We can substantially change the functioning of |hland_v1| by enabling its |RespArea|
option, which decreases |Perc| but increases |Q0| in dry periods (more concretely: in
periods with dry soils).  Hence the simulated result of |Perc| appears less "block-like"
and reaches its maximum at the same time as the result of |SM| does, whereas |Q0| shows
more pronounced peaks in the initial subperiod when the soil is not saturated yet:

.. integration-test::

    >>> resparea(True)
    >>> test("hland_v1_resparea")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |    outuh |       rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249737 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 | 0.061111 | 0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249482 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 | 0.088833 | 0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24923 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.04975 |  0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24898 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 | 0.049502 | 0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248728 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 | 0.049254 | 0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248461 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 | 0.049008 | 0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247705 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 | 0.048763 | 0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.242731 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 | 0.048519 | 0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.231785 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 | 0.048276 | 0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.214416 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 | 0.048035 | 0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.196315 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 | 0.047795 | 0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.177581 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 | 0.047556 | 0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.168204 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 | 0.047318 | 0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.156772 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 | 0.047081 | 0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.144898 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 | 0.046846 | 0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.141386 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 | 0.046612 | 0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.138698 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 | 0.046379 | 0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.137057 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 | 0.046147 | 0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.136597 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 | 0.045916 | 0.045916 | 0.012754 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.1554 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |   0.0777 |  0.016629 | 0.0 | 0.045846 |  0.062475 | 0.049468 | 0.049468 | 0.013741 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.618179 |  9.123417 | 0.013741 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.166931 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 | 0.083466 |  0.100569 | 0.0 | 0.046034 |  0.146603 | 0.077439 | 0.077439 | 0.021511 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.895425 |  9.160848 | 0.021511 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.187313 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 | 0.093656 |  0.275044 | 0.0 | 0.046273 |  0.321317 | 0.166733 | 0.166733 | 0.046315 | 1.954058 | 0.0 | 0.0 |  86.559232 |  1.431901 |  9.208232 | 0.046315 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234258 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 | 0.117129 |  0.903885 | 0.0 | 0.046627 |  0.950512 | 0.422313 | 0.422313 | 0.117309 | 1.976736 | 0.0 | 0.0 |  96.800491 |  2.709448 |  9.278734 | 0.117309 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234744 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 | 0.117372 |  0.805699 | 0.0 | 0.046981 |  0.852679 |  0.78895 |  0.78895 | 0.219153 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.757194 |  9.349126 | 0.219153 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.237676 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 | 0.118838 |   0.30497 | 0.0 |  0.04734 |   0.35231 | 0.763227 | 0.763227 | 0.212007 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.458453 |  9.420624 | 0.212007 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.250981 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |  0.12549 |  0.292533 | 0.0 | 0.047731 |  0.340263 | 0.460826 | 0.460826 | 0.128007 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.819784 |  9.498384 | 0.128007 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.260359 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |  0.13018 |  0.365199 | 0.0 | 0.048143 |  0.413342 |  0.35918 |  0.35918 | 0.099772 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.887257 |   9.58042 | 0.099772 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.309018 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 | 0.154509 |  1.047457 | 0.0 | 0.048675 |  1.096132 | 0.548833 | 0.548833 | 0.152454 | 1.976535 | 0.0 | 0.0 | 111.178794 |  3.841219 |  9.686255 | 0.152454 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.325241 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |  0.16262 |  1.415468 | 0.0 | 0.049244 |  1.464713 | 1.026307 | 1.026307 | 0.285085 |  1.94692 | 0.0 | 0.0 | 114.059742 |  3.497794 |  9.799631 | 0.285085 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.34089 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 | 0.170445 |  1.088288 | 0.0 |  0.04985 |  1.138139 | 1.310234 | 1.310234 | 0.363954 |  1.94647 | 0.0 | 0.0 | 116.771503 |  3.493884 |  9.920225 | 0.363954 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.346139 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 | 0.173069 |  0.792891 | 0.0 | 0.050466 |  0.843358 | 1.145204 | 1.145204 | 0.318112 | 1.864353 | 0.0 | 0.0 |  117.66711 |  2.954486 | 10.042828 | 0.318112 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.344125 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 | 0.172063 |  0.439147 | 0.0 | 0.051074 |  0.490221 |  0.83039 |  0.83039 | 0.230664 | 0.939422 | 0.0 | 0.0 | 117.324353 |   2.30211 | 10.163816 | 0.230664 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334261 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 | 0.167131 |  0.238811 | 0.0 | 0.051655 |  0.290465 | 0.524306 | 0.524306 | 0.145641 |      0.0 | 0.0 | 0.0 | 115.630662 |   1.85483 | 10.279292 | 0.145641 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.327797 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 | 0.163898 |  0.135278 | 0.0 | 0.052216 |  0.187494 | 0.311973 | 0.311973 | 0.086659 |      0.0 | 0.0 | 0.0 | 114.507032 |   1.51347 | 10.390974 | 0.086659 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.320053 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 | 0.160027 |  0.078912 | 0.0 | 0.052755 |  0.131667 |  0.19797 |  0.19797 | 0.054992 |      0.0 | 0.0 | 0.0 | 113.146516 |  1.231784 | 10.498246 | 0.054992 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.300088 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 | 0.150044 |  0.050702 | 0.0 | 0.053241 |  0.103943 | 0.137912 | 0.137912 | 0.038309 |      0.0 | 0.0 | 0.0 | 109.560616 |  0.987612 | 10.595049 | 0.038309 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.305588 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 | 0.152794 |  0.063517 | 0.0 | 0.053739 |  0.117256 | 0.113063 | 0.113063 | 0.031406 | 0.162204 | 0.0 | 0.0 | 110.559972 |  1.410119 | 10.694103 | 0.031406 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.313947 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 | 0.156973 |  0.280085 | 0.0 | 0.054255 |  0.334341 | 0.162539 | 0.162539 |  0.04515 |      0.0 | 0.0 | 0.0 | 112.061866 |  2.510453 | 10.796821 |  0.04515 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.303809 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 | 0.151905 |  0.384322 | 0.0 | 0.054744 |  0.439066 | 0.309372 | 0.309372 | 0.085937 |      0.0 | 0.0 | 0.0 | 110.237768 |  1.930258 | 10.893982 | 0.085937 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.396544 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 | 0.198272 |  2.037496 | 0.0 | 0.055461 |  2.092957 | 0.783325 | 0.783325 |  0.21759 |  1.84926 | 0.0 | 0.0 | 125.943496 |  6.521007 | 11.036793 |  0.21759 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.585507 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 | 0.292753 |  9.477037 | 0.0 | 0.056648 |  9.533685 | 3.378921 | 3.378921 | 0.938589 | 1.979276 | 0.0 | 0.0 |  153.03683 | 14.522431 | 11.272899 | 0.938589 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.613419 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 | 0.306709 |  8.550769 | 0.0 | 0.057898 |  8.608667 |  7.67463 |  7.67463 | 2.131842 | 1.092627 | 0.0 | 0.0 | 156.642106 | 11.335124 |  11.52171 | 2.131842 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.620455 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 | 0.310227 |  4.140784 | 0.0 |  0.05916 |  4.199943 |  7.83451 |  7.83451 | 2.176253 | 1.133348 | 0.0 | 0.0 | 157.537874 |  8.929955 | 11.772778 | 2.176253 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.644491 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 | 0.322246 |  3.415715 | 0.0 | 0.060475 |   3.47619 | 5.018826 | 5.018826 | 1.394118 | 1.900086 | 0.0 | 0.0 | 160.560437 | 10.162822 | 12.034548 | 1.394118 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.708542 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 | 0.354271 |  7.923725 | 0.0 | 0.061944 |  7.985669 |  4.63913 |  4.63913 | 1.288647 |  1.99141 | 0.0 | 0.0 | 168.349822 | 15.969232 | 12.326875 | 1.288647 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.754176 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 | 0.377088 | 11.304212 | 0.0 |  0.06352 | 11.367732 | 7.735132 | 7.735132 | 2.148648 | 1.987608 | 0.0 | 0.0 | 173.686572 | 17.231595 | 12.640443 | 2.148648 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.770161 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |  0.38508 |  8.161988 | 0.0 | 0.065128 |  8.227115 | 9.918248 | 9.918248 | 2.755069 | 1.963945 | 0.0 | 0.0 | 175.517585 | 14.307494 | 12.960395 | 2.755069 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.77449 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 | 0.387245 |  4.333944 | 0.0 | 0.066738 |  4.400682 | 8.074712 | 8.074712 | 2.242975 | 1.940232 | 0.0 | 0.0 | 176.010193 | 11.284958 | 13.280902 | 2.242975 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.785881 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |  0.39294 |  3.332947 | 0.0 | 0.068369 |  3.401316 | 5.028919 | 5.028919 | 1.396922 | 1.958069 | 0.0 | 0.0 | 177.299835 | 12.009895 | 13.605473 | 1.396922 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.791757 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 | 0.395879 |   3.05638 | 0.0 | 0.070007 |  3.126387 | 3.562302 | 3.562302 | 0.989528 |  1.94565 | 0.0 | 0.0 | 177.961502 | 11.034464 | 13.931345 | 0.989528 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792544 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 | 0.396272 |  2.001337 | 0.0 | 0.071638 |  2.072975 | 2.953391 | 2.953391 | 0.820386 | 1.930107 | 0.0 | 0.0 | 178.049897 |  9.053164 | 14.255979 | 0.820386 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792628 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 | 0.396314 |  1.154152 | 0.0 | 0.073261 |  1.227413 | 2.119164 | 2.119164 | 0.588657 | 1.927284 | 0.0 | 0.0 | 178.059289 |  7.624656 | 14.579032 | 0.588657 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.793371 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 | 0.396686 |  0.759428 | 0.0 | 0.074879 |  0.834307 | 1.327959 | 1.327959 | 0.368877 | 1.929004 | 0.0 | 0.0 | 178.142807 |  6.870811 | 14.900839 | 0.368877 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798335 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 | 0.399167 |   0.80239 | 0.0 |   0.0765 |   0.87889 | 0.931571 | 0.931571 |  0.25877 | 1.942479 | 0.0 | 0.0 | 178.699157 |  7.864347 | 15.223506 |  0.25877 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798631 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 | 0.399316 |  0.800248 | 0.0 | 0.078114 |  0.878362 | 0.868865 | 0.868865 | 0.241351 | 1.876663 | 0.0 | 0.0 | 178.732328 |  6.987622 | 15.544708 | 0.241351 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.797218 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 | 0.398609 |  0.553214 | 0.0 | 0.079717 |   0.63293 | 0.823939 | 0.823939 | 0.228872 | 1.616372 | 0.0 | 0.0 | 178.574145 |  6.116431 |   15.8636 | 0.228872 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792943 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 | 0.396472 |  0.375567 | 0.0 |   0.0813 |  0.456868 | 0.648346 | 0.648346 | 0.180096 | 0.636076 | 0.0 | 0.0 |  178.09471 |  5.333679 | 16.178771 | 0.180096 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.764872 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 | 0.382436 |  0.278561 | 0.0 | 0.082806 |  0.361367 |  0.47477 |  0.47477 | 0.131881 |      0.0 | 0.0 | 0.0 | 174.913943 |   4.66173 | 16.478401 | 0.131881 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.736086 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 | 0.368043 |  0.208651 | 0.0 | 0.084232 |  0.292883 | 0.367371 | 0.367371 | 0.102047 |      0.0 | 0.0 | 0.0 | 171.590941 |  4.072493 | 16.762212 | 0.102047 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.719431 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 | 0.359715 |  0.198136 | 0.0 |  0.08561 |  0.283746 | 0.306071 | 0.306071 |  0.08502 |      0.0 | 0.0 | 0.0 | 169.638537 |  4.302031 | 17.036318 |  0.08502 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.688383 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 | 0.344192 |  0.199364 | 0.0 | 0.086903 |  0.286266 | 0.286337 | 0.286337 | 0.079538 |      0.0 | 0.0 | 0.0 | 165.937749 |  3.743294 | 17.293607 | 0.079538 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.659166 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 | 0.329583 |  0.149065 | 0.0 | 0.088116 |  0.237181 | 0.274798 | 0.274798 | 0.076333 |      0.0 | 0.0 | 0.0 | 162.378092 |  3.247615 | 17.535074 | 0.076333 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.610719 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 | 0.305359 |  0.121494 | 0.0 | 0.089202 |  0.210696 | 0.242203 | 0.242203 | 0.067279 |      0.0 | 0.0 | 0.0 |  156.29702 |  2.801951 | 17.751232 | 0.067279 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.58034 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |  0.29017 |  0.090079 | 0.0 | 0.090207 |  0.180286 | 0.209824 | 0.209824 | 0.058284 |      0.0 | 0.0 | 0.0 | 152.360068 |   2.39985 | 17.951194 | 0.058284 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.548803 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 | 0.274402 |  0.066087 | 0.0 | 0.091128 |  0.157215 | 0.181917 | 0.181917 | 0.050532 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.035542 | 18.134468 | 0.050532 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.530323 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 | 0.265161 |  0.043836 | 0.0 | 0.091998 |  0.135834 |  0.15759 |  0.15759 | 0.043775 |      0.0 | 0.0 | 0.0 | 145.646557 |  1.700626 | 18.307631 | 0.043775 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.522211 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 | 0.261106 |  0.025957 | 0.0 | 0.092844 |  0.118801 |   0.1368 |   0.1368 |    0.038 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.386387 | 18.475893 |    0.038 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520914 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 | 0.260457 |  0.013472 | 0.0 | 0.093682 |  0.107154 | 0.119998 | 0.119998 | 0.033333 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.084722 | 18.642669 | 0.033333 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520639 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 | 0.260319 |   0.00595 | 0.0 | 0.094515 |  0.100465 | 0.108256 | 0.108256 | 0.030071 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.790627 | 18.808473 | 0.030071 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520374 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.260187 |  0.001991 | 0.0 | 0.095343 |  0.097335 | 0.101256 | 0.101256 | 0.028127 |      0.0 | 0.0 | 0.0 |  144.27384 |  0.500604 | 18.973317 | 0.028127 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520116 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063512 | 0.027863 | -0.027863 | 0.260058 |   0.00037 | 0.0 | 0.096167 |  0.096537 | 0.097853 | 0.097853 | 0.027181 |      0.0 | 0.0 | 0.0 | 144.238191 |  0.212312 | 19.137208 | 0.027181 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.519866 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062667 | 0.027881 | -0.027881 |  0.18442 |  0.000012 | 0.0 | 0.096608 |   0.09662 | 0.096733 | 0.096733 |  0.02687 |      0.0 | 0.0 | 0.0 | 144.203406 |       0.0 |  19.22502 |  0.02687 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.519419 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061942 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096125 |  0.096125 | 0.096492 | 0.096492 | 0.026803 |      0.0 | 0.0 | 0.0 | 144.141463 |       0.0 | 19.128895 | 0.026803 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518961 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063579 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095644 |  0.095644 | 0.096128 | 0.096128 | 0.026702 |      0.0 | 0.0 | 0.0 | 144.077884 |       0.0 |  19.03325 | 0.026702 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518501 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063825 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095166 |  0.095166 | 0.095645 | 0.095645 | 0.026568 |      0.0 | 0.0 | 0.0 | 144.014059 |       0.0 | 18.938084 | 0.026568 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518307 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026921 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09469 |   0.09469 | 0.095167 | 0.095167 | 0.026435 | 1.484581 | 0.0 | 0.0 | 143.987137 |       0.0 | 18.843393 | 0.026435 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518082 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031342 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094217 |  0.094217 | 0.094691 | 0.094691 | 0.026303 | 1.414925 | 0.0 | 0.0 | 143.955795 |       0.0 | 18.749176 | 0.026303 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517852 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031885 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093746 |  0.093746 | 0.094217 | 0.094217 | 0.026172 | 1.344047 | 0.0 | 0.0 |  143.92391 |       0.0 | 18.655431 | 0.026172 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517014 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.116535 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093277 |  0.093277 | 0.093746 | 0.093746 | 0.026041 | 1.084943 | 0.0 | 0.0 | 143.807375 |       0.0 | 18.562153 | 0.026041 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.513563 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480764 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092811 |  0.092811 | 0.093278 | 0.093278 |  0.02591 | 0.846748 | 0.0 | 0.0 | 143.326611 |       0.0 | 18.469343 |  0.02591 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.501086 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751726 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092347 |  0.092347 | 0.092811 | 0.092811 | 0.025781 |      0.0 | 0.0 | 0.0 | 141.574885 |       0.0 | 18.376996 | 0.025781 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.474607 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.791428 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091885 |  0.091885 | 0.092347 | 0.092347 | 0.025652 |      0.0 | 0.0 | 0.0 | 137.783457 |       0.0 | 18.285111 | 0.025652 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.441402 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.90729 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091426 |  0.091426 | 0.091885 | 0.091885 | 0.025524 |      0.0 | 0.0 | 0.0 | 132.876167 |       0.0 | 18.193685 | 0.025524 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403579 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.820411 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090968 |  0.090968 | 0.091426 | 0.091426 | 0.025396 |      0.0 | 0.0 | 0.0 | 127.055756 |       0.0 | 18.102717 | 0.025396 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.368998 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.565361 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090514 |  0.090514 | 0.090969 | 0.090969 | 0.025269 |      0.0 | 0.0 | 0.0 | 121.490395 |       0.0 | 18.012203 | 0.025269 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.328874 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.795388 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090061 |  0.090061 | 0.090514 | 0.090514 | 0.025143 |      0.0 | 0.0 | 0.0 | 114.695007 |       0.0 | 17.922142 | 0.025143 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.297445 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.617912 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089611 |  0.089611 | 0.090062 | 0.090062 | 0.025017 |      0.0 | 0.0 | 0.0 | 109.077094 |       0.0 | 17.832532 | 0.025017 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.273532 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.476436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089163 |  0.089163 | 0.089611 | 0.089611 | 0.024892 |      0.0 | 0.0 | 0.0 | 104.600658 |       0.0 | 17.743369 | 0.024892 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.255842 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.438985 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088717 |  0.088717 | 0.089163 | 0.089163 | 0.024768 |      0.0 | 0.0 | 0.0 | 101.161673 |       0.0 | 17.654652 | 0.024768 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.248028 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556952 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088273 |  0.088273 | 0.088717 | 0.088717 | 0.024644 |      0.0 | 0.0 | 0.0 |  99.604721 |       0.0 | 17.566379 | 0.024644 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.244649 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680718 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087832 |  0.087832 | 0.088274 | 0.088274 |  0.02452 |      0.0 | 0.0 | 0.0 |  98.924003 |       0.0 | 17.478547 |  0.02452 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243808 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170174 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087393 |  0.087393 | 0.087832 | 0.087832 | 0.024398 |      0.0 | 0.0 | 0.0 |  98.753829 |       0.0 | 17.391154 | 0.024398 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.24358 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046141 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086956 |  0.086956 | 0.087393 | 0.087393 | 0.024276 |      0.0 | 0.0 | 0.0 |  98.707688 |       0.0 | 17.304198 | 0.024276 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243362 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044194 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086521 |  0.086521 | 0.086956 | 0.086956 | 0.024155 |      0.0 | 0.0 | 0.0 |  98.663494 |       0.0 | 17.217677 | 0.024155 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243147 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043519 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086088 |  0.086088 | 0.086521 | 0.086521 | 0.024034 |      0.0 | 0.0 | 0.0 |  98.619975 |       0.0 | 17.131589 | 0.024034 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.242932 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043688 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085658 |  0.085658 | 0.086089 | 0.086089 | 0.023914 |      0.0 | 0.0 | 0.0 |  98.576287 |       0.0 | 17.045931 | 0.023914 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_low_accuracy:

low accuracy
____________

The following example indicates caution when reducing the value of |RecStep| to save
computation time.  Setting |RecStep| to 1, which is the smallest possible value, results
in low accuracies.  You can see this by comparing the time series of |Q0| calculated in
this and the last example.  The time series of this example is much more wrong, as the
peak of |Q0| is high above the peak of |R|, which is physically impossible:

.. integration-test::

    >>> recstep(1)
    >>> test("hland_v1_low_accuracy")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |     outuh |        rt |       qt |       ic |  sp |  wc |         sm |       uz |        lz |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249737 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 |      0.0 | 0.0 | 0.0 |  99.947431 |      0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249482 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 |      0.0 | 0.0 | 0.0 |  99.896406 |      0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24923 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 |      0.0 | 0.0 | 0.0 |  99.845834 |      0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24898 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.795804 |      0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248728 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |  99.745284 |      0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248461 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.691779 |      0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247705 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 |      0.0 | 0.0 | 0.0 |  99.539873 |      0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.242731 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  98.535555 |      0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.231785 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  96.288059 |      0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.214416 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  92.610082 |      0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.196315 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 |      0.0 | 0.0 | 0.0 |  88.614886 |      0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.177581 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |   84.28067 |      0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.168204 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  82.025433 |      0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.156772 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  79.188775 |      0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.144898 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  76.130856 |      0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.141386 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  75.202777 |      0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.138698 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 |      0.0 | 0.0 | 0.0 |  74.484461 |      0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.137057 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 |      0.0 | 0.0 | 0.0 |  74.042552 |      0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.136597 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 1.006935 | 0.0 | 0.0 |  73.918192 |      0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.1554 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |   0.0777 |  0.068166 | 0.0 | 0.045846 |  0.114013 |  0.060921 |  0.060921 | 0.016922 | 1.903859 | 0.0 | 0.0 |  78.841725 | 0.566641 |  9.123417 | 0.016922 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.166931 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 | 0.083466 |  0.181105 | 0.0 | 0.046034 |   0.22714 |  0.123968 |  0.123968 | 0.034436 | 1.941161 | 0.0 | 0.0 |  81.714476 | 0.763351 |  9.160848 | 0.034436 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.187313 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 | 0.093656 |  0.594339 | 0.0 | 0.046273 |  0.640612 |  0.293883 |  0.293883 | 0.081634 | 1.954058 | 0.0 | 0.0 |  86.559232 | 0.980532 |  9.208232 | 0.081634 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234258 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 | 0.117129 |  2.459152 | 0.0 | 0.046627 |  2.505778 |  0.963211 |  0.963211 | 0.267559 | 1.976736 | 0.0 | 0.0 |  96.800491 | 0.702812 |  9.278734 | 0.267559 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234744 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 | 0.117372 |  0.013306 | 0.0 | 0.046981 |  0.060286 |  1.547854 |  1.547854 |  0.42996 | 1.919551 | 0.0 | 0.0 |  96.900842 | 0.542951 |  9.349126 |  0.42996 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.237676 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 | 0.118838 |  0.012336 | 0.0 |  0.04734 |  0.059676 |  0.603593 |  0.603593 | 0.167665 | 1.925086 | 0.0 | 0.0 |  97.504096 | 0.536843 |  9.420624 | 0.167665 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.250981 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |  0.12549 |  0.106781 | 0.0 | 0.047731 |  0.154511 |  0.080886 |  0.080886 | 0.022468 | 1.942997 | 0.0 | 0.0 | 100.195975 | 1.083926 |  9.498384 | 0.022468 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.260359 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |  0.13018 |  0.197648 | 0.0 | 0.048143 |  0.245791 |  0.153721 |  0.153721 |   0.0427 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.31895 |   9.58042 |   0.0427 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.309018 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 | 0.154509 |  2.732817 | 0.0 | 0.048675 |  2.781491 |  0.788996 |  0.788996 | 0.219165 | 1.976535 | 0.0 | 0.0 | 111.178794 | 1.587552 |  9.686255 | 0.219165 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.325241 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |  0.16262 |  0.546805 | 0.0 | 0.049244 |  0.596049 |  1.732349 |  1.732349 | 0.481208 |  1.94692 | 0.0 | 0.0 | 114.059742 |  2.11279 |  9.799631 | 0.481208 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.34089 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 | 0.170445 |  0.825003 | 0.0 |  0.04985 |  0.874853 |  1.143659 |  1.143659 | 0.317683 |  1.94647 | 0.0 | 0.0 | 116.771503 | 2.372166 |  9.920225 | 0.317683 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.346139 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 | 0.173069 |   0.43648 | 0.0 | 0.050466 |  0.486946 |  0.726695 |  0.726695 |  0.20186 | 1.864353 | 0.0 | 0.0 |  117.66711 | 2.189179 | 10.042828 |  0.20186 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.344125 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 | 0.172063 |  0.189312 | 0.0 | 0.051074 |  0.240387 |  0.518357 |  0.518357 | 0.143988 | 0.939422 | 0.0 | 0.0 | 117.324353 | 1.786638 | 10.163816 | 0.143988 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334261 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 | 0.167131 |  0.105245 | 0.0 | 0.051655 |    0.1569 |  0.276625 |  0.276625 |  0.07684 |      0.0 | 0.0 | 0.0 | 115.630662 | 1.472924 | 10.279292 |  0.07684 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.327797 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 | 0.163898 |  0.057724 | 0.0 | 0.052216 |   0.10994 |  0.165017 |  0.165017 | 0.045838 |      0.0 | 0.0 | 0.0 | 114.507032 | 1.209117 | 10.390974 | 0.045838 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.320053 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 | 0.160027 |  0.031087 | 0.0 | 0.052755 |  0.083842 |  0.114576 |  0.114576 | 0.031827 |      0.0 | 0.0 | 0.0 | 113.146516 | 0.975258 | 10.498246 | 0.031827 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.300088 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 | 0.150044 |  0.017681 | 0.0 | 0.053241 |  0.070923 |   0.08677 |   0.08677 | 0.024103 |      0.0 | 0.0 | 0.0 | 109.560616 | 0.764105 | 10.595049 | 0.024103 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.305588 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 | 0.152794 |  0.068463 | 0.0 | 0.053739 |  0.122202 |  0.085189 |  0.085189 | 0.023664 | 0.162204 | 0.0 | 0.0 | 110.559972 | 1.181666 | 10.694103 | 0.023664 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.313947 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 | 0.156973 |  0.543518 | 0.0 | 0.054255 |  0.597773 |  0.216489 |  0.216489 | 0.060136 |      0.0 | 0.0 | 0.0 | 112.061866 | 2.018568 | 10.796821 | 0.060136 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.303809 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 | 0.151905 |  0.215943 | 0.0 | 0.054744 |  0.270687 |  0.419405 |  0.419405 | 0.116501 |      0.0 | 0.0 | 0.0 | 110.237768 | 1.606751 | 10.893982 | 0.116501 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.396544 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 | 0.198272 |  8.234997 | 0.0 | 0.055461 |  8.290458 |  2.125544 |  2.125544 | 0.590429 |  1.84926 | 0.0 | 0.0 | 125.943496 |      0.0 | 11.036793 | 0.590429 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.585507 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 | 0.292753 | 17.478461 | 0.0 | 0.056648 | 17.535108 |  8.562653 |  8.562653 | 2.378515 | 1.979276 | 0.0 | 0.0 |  153.03683 |      0.0 | 11.272899 | 2.378515 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.613419 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 | 0.306709 |  0.668443 | 0.0 | 0.057898 |  0.726341 |  11.74546 |  11.74546 | 3.262628 | 1.092627 | 0.0 | 0.0 | 156.642106 | 4.695019 |  11.52171 | 3.262628 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.620455 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 | 0.310227 |  1.113349 | 0.0 |  0.05916 |  1.172509 |  4.560771 |  4.560771 | 1.266881 | 1.133348 | 0.0 | 0.0 | 157.537874 | 5.317284 | 11.772778 | 1.266881 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.644491 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 | 0.322246 |  3.697378 | 0.0 | 0.060475 |  3.757853 |  1.647881 |  1.647881 | 0.457745 | 1.900086 | 0.0 | 0.0 | 160.560437 | 6.268488 | 12.034548 | 0.457745 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.708542 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 | 0.354271 | 19.998623 | 0.0 | 0.061944 | 20.060568 |  6.806157 |  6.806157 | 1.890599 |  1.99141 | 0.0 | 0.0 | 168.349822 |      0.0 | 12.326875 | 1.890599 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.754176 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 | 0.377088 |  4.626295 | 0.0 |  0.06352 |  4.689814 | 13.022019 | 13.022019 | 3.617228 | 1.987608 | 0.0 | 0.0 | 173.686572 |  7.94028 | 12.640443 | 3.617228 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.770161 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |  0.38508 |   5.00981 | 0.0 | 0.065128 |  5.074938 |   8.19112 |   8.19112 | 2.275311 | 1.963945 | 0.0 | 0.0 | 175.517585 | 8.168358 | 12.960395 | 2.275311 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.77449 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 | 0.387245 |  1.833774 | 0.0 | 0.066738 |  1.900512 |  4.283927 |  4.283927 |  1.18998 | 1.940232 | 0.0 | 0.0 | 176.010193 | 7.645991 | 13.280902 |  1.18998 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.785881 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |  0.39294 |  3.303079 | 0.0 | 0.068369 |  3.371448 |  2.932815 |  2.932815 | 0.814671 | 1.958069 | 0.0 | 0.0 | 177.299835 | 8.400796 | 13.605473 | 0.814671 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.791757 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 | 0.395879 |  2.320194 | 0.0 | 0.070007 |  2.390201 |  2.826519 |  2.826519 | 0.785144 |  1.94565 | 0.0 | 0.0 | 177.961502 |  8.16155 | 13.931345 | 0.785144 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792544 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 | 0.396272 |  1.100126 | 0.0 | 0.071638 |  1.171764 |  2.337492 |  2.337492 | 0.649303 | 1.930107 | 0.0 | 0.0 | 178.049897 | 7.081462 | 14.255979 | 0.649303 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792628 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 | 0.396314 |  0.633402 | 0.0 | 0.073261 |  0.706664 |  1.339172 |  1.339172 | 0.371992 | 1.927284 | 0.0 | 0.0 | 178.059289 | 6.173703 | 14.579032 | 0.371992 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.793371 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 | 0.396686 |  0.472482 | 0.0 | 0.074879 |   0.54736 |  0.774619 |  0.774619 | 0.215172 | 1.929004 | 0.0 | 0.0 | 178.142807 | 5.706804 | 14.900839 | 0.215172 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798335 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 | 0.399167 |  0.830047 | 0.0 |   0.0765 |  0.906547 |   0.66258 |   0.66258 |  0.18405 | 1.942479 | 0.0 | 0.0 | 178.699157 | 6.672683 | 15.223506 |  0.18405 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798631 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 | 0.399316 |  0.563435 | 0.0 | 0.078114 |  0.641549 |   0.76784 |   0.76784 | 0.213289 | 1.876663 | 0.0 | 0.0 | 178.732328 | 6.032771 | 15.544708 | 0.213289 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.797218 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 | 0.398609 |  0.368358 | 0.0 | 0.079717 |  0.448075 |  0.657443 |  0.657443 | 0.182623 | 1.616372 | 0.0 | 0.0 | 178.574145 | 5.346435 |   15.8636 | 0.182623 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792943 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 | 0.396472 |  0.241689 | 0.0 |   0.0813 |  0.322989 |  0.463272 |  0.463272 | 0.128687 | 0.636076 | 0.0 | 0.0 |  178.09471 | 4.697562 | 16.178771 | 0.128687 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.764872 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 | 0.382436 |  0.178198 | 0.0 | 0.082806 |  0.261004 |  0.337012 |  0.337012 | 0.093614 |      0.0 | 0.0 | 0.0 | 174.913943 | 4.125975 | 16.478401 | 0.093614 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.736086 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 | 0.368043 |  0.131736 | 0.0 | 0.084232 |  0.215968 |  0.264771 |  0.264771 | 0.073547 |      0.0 | 0.0 | 0.0 | 171.590941 | 3.613653 | 16.762212 | 0.073547 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.719431 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 | 0.359715 |  0.177258 | 0.0 |  0.08561 |  0.262867 |  0.236398 |  0.236398 | 0.065666 |      0.0 | 0.0 | 0.0 | 169.638537 |  3.86407 | 17.036318 | 0.065666 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.688383 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 | 0.344192 |  0.131965 | 0.0 | 0.086903 |  0.218868 |  0.242668 |  0.242668 | 0.067408 |      0.0 | 0.0 | 0.0 | 165.937749 | 3.372732 | 17.293607 | 0.067408 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.659166 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 | 0.329583 |  0.096755 | 0.0 | 0.088116 |  0.184871 |  0.221091 |  0.221091 | 0.061414 |      0.0 | 0.0 | 0.0 | 162.378092 | 2.929363 | 17.535074 | 0.061414 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.610719 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 | 0.305359 |  0.077624 | 0.0 | 0.089202 |  0.166826 |  0.188416 |  0.188416 | 0.052338 |      0.0 | 0.0 | 0.0 |  156.29702 | 2.527568 | 17.751232 | 0.052338 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.58034 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |  0.29017 |  0.055641 | 0.0 | 0.090207 |  0.145848 |  0.166174 |  0.166174 |  0.04616 |      0.0 | 0.0 | 0.0 | 152.360068 | 2.159906 | 17.951194 |  0.04616 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.548803 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 | 0.274402 |  0.039036 | 0.0 | 0.091128 |  0.130164 |  0.147025 |  0.147025 |  0.04084 |      0.0 | 0.0 | 0.0 | 148.162531 | 1.822648 | 18.134468 |  0.04084 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.530323 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 | 0.265161 |  0.024087 | 0.0 | 0.091998 |  0.116085 |  0.130521 |  0.130521 | 0.036256 |      0.0 | 0.0 | 0.0 | 145.646557 | 1.507481 | 18.307631 | 0.036256 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.522211 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 | 0.261106 |  0.012726 | 0.0 | 0.092844 |  0.105569 |  0.116877 |  0.116877 | 0.032466 |      0.0 | 0.0 | 0.0 | 144.528382 | 1.206472 | 18.475893 | 0.032466 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520914 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 | 0.260457 |  0.005478 | 0.0 | 0.093682 |   0.09916 |  0.106482 |  0.106482 | 0.029578 |      0.0 | 0.0 | 0.0 | 144.348787 | 0.912801 | 18.642669 | 0.029578 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520639 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 | 0.260319 |  0.001727 | 0.0 | 0.094515 |  0.096242 |  0.099936 |  0.099936 |  0.02776 |      0.0 | 0.0 | 0.0 |  144.31062 | 0.622929 | 18.808473 |  0.02776 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520374 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.260187 |  0.000267 | 0.0 | 0.095343 |   0.09561 |   0.09675 |   0.09675 | 0.026875 |      0.0 | 0.0 | 0.0 |  144.27384 | 0.334631 | 18.973317 | 0.026875 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520116 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063512 | 0.027863 | -0.027863 | 0.260058 |  0.000001 | 0.0 | 0.096167 |  0.096168 |  0.095874 |  0.095874 | 0.026632 |      0.0 | 0.0 | 0.0 | 144.238191 | 0.046709 | 19.137208 | 0.026632 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.519866 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062667 | 0.027881 | -0.027881 | 0.018828 |       0.0 | 0.0 |  0.09578 |   0.09578 |  0.095958 |  0.095958 | 0.026655 |      0.0 | 0.0 | 0.0 | 144.203406 |      0.0 | 19.060256 | 0.026655 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.519419 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061942 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095301 |  0.095301 |   0.09576 |   0.09576 |   0.0266 |      0.0 | 0.0 | 0.0 | 144.141463 |      0.0 | 18.964955 |   0.0266 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518961 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063579 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094825 |  0.094825 |  0.095302 |  0.095302 | 0.026473 |      0.0 | 0.0 | 0.0 | 144.077884 |      0.0 |  18.87013 | 0.026473 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518501 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063825 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094351 |  0.094351 |  0.094825 |  0.094825 |  0.02634 |      0.0 | 0.0 | 0.0 | 144.014059 |      0.0 |  18.77578 |  0.02634 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518307 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026921 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093879 |  0.093879 |  0.094351 |  0.094351 | 0.026209 | 1.484581 | 0.0 | 0.0 | 143.987137 |      0.0 | 18.681901 | 0.026209 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518082 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031342 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09341 |   0.09341 |  0.093879 |  0.093879 | 0.026078 | 1.414925 | 0.0 | 0.0 | 143.955795 |      0.0 | 18.588491 | 0.026078 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517852 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031885 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092942 |  0.092942 |   0.09341 |   0.09341 | 0.025947 | 1.344047 | 0.0 | 0.0 |  143.92391 |      0.0 | 18.495549 | 0.025947 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517014 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.116535 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092478 |  0.092478 |  0.092943 |  0.092943 | 0.025817 | 1.084943 | 0.0 | 0.0 | 143.807375 |      0.0 | 18.403071 | 0.025817 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.513563 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480764 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092015 |  0.092015 |  0.092478 |  0.092478 | 0.025688 | 0.846748 | 0.0 | 0.0 | 143.326611 |      0.0 | 18.311056 | 0.025688 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.501086 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751726 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091555 |  0.091555 |  0.092016 |  0.092016 |  0.02556 |      0.0 | 0.0 | 0.0 | 141.574885 |      0.0 |   18.2195 |  0.02556 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.474607 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.791428 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091098 |  0.091098 |  0.091556 |  0.091556 | 0.025432 |      0.0 | 0.0 | 0.0 | 137.783457 |      0.0 | 18.128403 | 0.025432 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.441402 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.90729 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090642 |  0.090642 |  0.091098 |  0.091098 | 0.025305 |      0.0 | 0.0 | 0.0 | 132.876167 |      0.0 | 18.037761 | 0.025305 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403579 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.820411 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090189 |  0.090189 |  0.090643 |  0.090643 | 0.025178 |      0.0 | 0.0 | 0.0 | 127.055756 |      0.0 | 17.947572 | 0.025178 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.368998 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.565361 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089738 |  0.089738 |  0.090189 |  0.090189 | 0.025053 |      0.0 | 0.0 | 0.0 | 121.490395 |      0.0 | 17.857834 | 0.025053 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.328874 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.795388 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089289 |  0.089289 |  0.089738 |  0.089738 | 0.024927 |      0.0 | 0.0 | 0.0 | 114.695007 |      0.0 | 17.768545 | 0.024927 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.297445 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.617912 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088843 |  0.088843 |   0.08929 |   0.08929 | 0.024803 |      0.0 | 0.0 | 0.0 | 109.077094 |      0.0 | 17.679702 | 0.024803 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.273532 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.476436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088399 |  0.088399 |  0.088843 |  0.088843 | 0.024679 |      0.0 | 0.0 | 0.0 | 104.600658 |      0.0 | 17.591304 | 0.024679 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.255842 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.438985 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087957 |  0.087957 |  0.088399 |  0.088399 | 0.024555 |      0.0 | 0.0 | 0.0 | 101.161673 |      0.0 | 17.503347 | 0.024555 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.248028 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556952 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087517 |  0.087517 |  0.087957 |  0.087957 | 0.024433 |      0.0 | 0.0 | 0.0 |  99.604721 |      0.0 | 17.415831 | 0.024433 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.244649 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680718 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087079 |  0.087079 |  0.087517 |  0.087517 |  0.02431 |      0.0 | 0.0 | 0.0 |  98.924003 |      0.0 | 17.328751 |  0.02431 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243808 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170174 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086644 |  0.086644 |   0.08708 |   0.08708 | 0.024189 |      0.0 | 0.0 | 0.0 |  98.753829 |      0.0 | 17.242108 | 0.024189 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.24358 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046141 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086211 |  0.086211 |  0.086644 |  0.086644 | 0.024068 |      0.0 | 0.0 | 0.0 |  98.707688 |      0.0 | 17.155897 | 0.024068 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243362 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044194 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085779 |  0.085779 |  0.086211 |  0.086211 | 0.023948 |      0.0 | 0.0 | 0.0 |  98.663494 |      0.0 | 17.070118 | 0.023948 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243147 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043519 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085351 |  0.085351 |   0.08578 |   0.08578 | 0.023828 |      0.0 | 0.0 | 0.0 |  98.619975 |      0.0 | 16.984767 | 0.023828 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.242932 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043688 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.084924 |  0.084924 |  0.085351 |  0.085351 | 0.023709 |      0.0 | 0.0 | 0.0 |  98.576287 |      0.0 | 16.899843 | 0.023709 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_sealed:

sealed area
___________

|hland_v1| tends to underestimate the runoff response to small or medium-sized rain
events that fall on dry soils.  One means to improve the situation is to activate the
:ref:`contributing area <hland_v1_resparea>` option.  Unfortunately, this option has the
side-effect that decreasing the soil moisture increases the direct discharge released by
the upper zone storage.  Defining sealed areas provides an alternative to the
"contributing area" approach.  The following example shows that a zone classified as
|SEALED| converts all precipitation, which is not intercepted or stored within a snow
cover, to runoff and directly passes it to the runoff concentration module.  Thus, the
definition of sealed surfaces causes higher dynamics in the simulated flood hydrograph
that is most perceptible during dry conditions:

.. integration-test::

    >>> recstep(100)
    >>> zonetype(SEALED)
    >>> test("hland_v1_sealed")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |        sr |  ea |  cf | inuz | perc |  q0 |  el |  q1 |      inuh |     outuh |        rt |       qt |       ic |  sp |  wc |  sm |  uz |  lz |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.05 |      0.05 | 0.013889 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.013889 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.05 |      0.05 | 0.013889 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.013889 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.006935 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  5.659735 |  5.659735 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.659735 |  1.257719 |  1.257719 | 0.349366 | 1.903859 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.349366 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  3.349059 |  3.349059 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.349059 |  3.888533 |  3.888533 | 1.080148 | 1.941161 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.080148 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  5.762361 |  5.762361 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.762361 |  4.398832 |  4.398832 | 1.221898 | 1.954058 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.221898 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 | 12.546858 | 12.546858 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.546858 |  6.733738 |  6.733738 | 1.870483 | 1.976736 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.870483 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |  0.095536 |  0.095536 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.095536 |  8.272231 |  8.272231 | 2.297842 | 1.919551 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.297842 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.751151 |  0.751151 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.751151 |  3.008189 |  3.008189 | 0.835608 | 1.925086 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.835608 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  3.489086 |  3.489086 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.489086 |  1.213889 |  1.213889 | 0.337191 | 1.942997 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.337191 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |  2.437797 |  2.437797 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.437797 |  2.647036 |  2.647036 | 0.735288 | 1.936885 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.735288 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 | 12.292085 | 12.292085 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.292085 |  4.861259 |  4.861259 |  1.35035 | 1.976535 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.35035 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  4.134535 |  4.134535 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  4.134535 |  8.289455 |  8.289455 | 2.302626 |  1.94692 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.302626 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |   3.98612 |   3.98612 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   3.98612 |  5.914343 |  5.914343 | 1.642873 |  1.94647 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.642873 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |   1.37207 |   1.37207 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   1.37207 |  3.438201 |  3.438201 | 0.955056 | 1.864353 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.955056 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  1.648066 |  1.648066 | 0.457796 | 0.939422 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.457796 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.304904 |  0.304904 | 0.084696 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.084696 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |    2.2768 |    2.2768 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    2.2768 |  0.505956 |  0.505956 | 0.140543 | 0.162204 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.140543 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  5.171404 |  5.171404 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.171404 |   2.41409 |   2.41409 | 0.670581 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.670581 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  3.378958 |  3.378958 | 0.938599 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.938599 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |   22.5916 |   22.5916 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   22.5916 |  6.169556 |  6.169556 | 1.713766 |  1.84926 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.713766 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 |  44.87446 |  44.87446 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  44.87446 | 22.522991 | 22.522991 | 6.256386 | 1.979276 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 6.256386 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  9.720876 |  9.720876 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  9.720876 | 32.110806 | 32.110806 | 8.919668 | 1.092627 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 8.919668 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  3.369427 |  3.369427 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.369427 |  16.12135 |  16.12135 | 4.478153 | 1.133348 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 4.478153 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  8.043348 |  8.043348 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  8.043348 |  5.819509 |  5.819509 |  1.61653 | 1.900086 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.61653 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 21.878086 | 21.878086 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 21.878086 | 10.079085 | 10.079085 | 2.799746 |  1.99141 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.799746 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 |  18.28661 |  18.28661 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  18.28661 | 18.005594 | 18.005594 | 5.001554 | 1.987608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 5.001554 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  7.472008 |  7.472008 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  7.472008 | 16.681471 | 16.681471 | 4.633742 | 1.963945 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 4.633742 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  2.221145 |  2.221145 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.221145 |  8.708394 |  8.708394 | 2.418998 | 1.940232 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 2.418998 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  5.761432 |  5.761432 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.761432 |  4.174734 |  4.174734 | 1.159648 | 1.958069 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.159648 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  3.165669 |  3.165669 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.165669 |  4.397865 |  4.397865 | 1.221629 |  1.94565 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 1.221629 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |   0.53965 |   0.53965 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.53965 |  3.158945 |  3.158945 | 0.877485 | 1.930107 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.877485 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.167707 |  0.167707 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.167707 |  1.040556 |  1.040556 | 0.289043 | 1.927284 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.289043 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.521284 |  0.521284 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.521284 |  0.328934 |  0.328934 | 0.091371 | 1.929004 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.091371 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.780204 |  2.780204 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.780204 |  0.944694 |  0.944694 | 0.262415 | 1.942479 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.262415 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.417679 |  0.417679 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.417679 |  1.753216 |  1.753216 | 0.487005 | 1.876663 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.487005 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.114263 |  0.114263 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.114263 |  0.875259 |  0.875259 | 0.243127 | 1.616372 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.243127 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.156297 |  0.156297 | 0.043416 | 0.636076 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.043416 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.025392 |  0.025392 | 0.007053 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.007053 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |    1.0888 |    1.0888 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    1.0888 |  0.241956 |  0.241956 |  0.06721 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.06721 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.604889 |  0.604889 | 0.168025 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.168025 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.241956 |  0.241956 |  0.06721 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.06721 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.484581 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.414925 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.344047 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.084943 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 0.846748 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_ilake:

internal lake
_____________

In the following example, we demonstrate the functionality of zones of type |ILAKE|.
For such "internal lakes", only the lower zone storage (|LZ|) is relevant (all other
storage values are zero).  Precipitation (|PC|) adds directly to, and evaporation
(|EPC|) subtracts directly from |LZ|.  The latter occurs even when |LZ| is empty,
possibly resulting in negative storage values in drought periods.  The only condition
preventing lake evaporation (|EL|) is the occurrence of lake ice, which exists when the
actual temperature ("TC") is below the threshold temperature (|TTIce|).  In our example,
we set |TTIce| to the unrealistic value of 13°C, resulting in a deviation between the
graphs of |EPC| and |EL| for the last day of the simulation period:

.. integration-test::

    >>> zonetype(ILAKE)
    >>> ttice(13.0)
    >>> test("hland_v1_ilake")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc | cfact | swe | gact | contriarea |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg | glmelt | melt | refr |     in_ |       r |  sr |  ea |  cf | inuz | perc |  q0 |       el |       q1 |     inuh |    outuh |       rt |       qt |  ic |  sp |  wc |  sm |  uz |         lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.08411 | 0.049579 | 0.049579 | 0.061018 | 0.061018 | 0.016949 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.86631 | 0.016949 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.081683 | 0.048923 | 0.048923 | 0.088416 | 0.088416 |  0.02456 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.735704 |  0.02456 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080999 | 0.048274 | 0.048274 | 0.048925 | 0.048925 |  0.01359 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.606431 |  0.01359 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080171 | 0.047631 | 0.047631 | 0.048275 | 0.048275 |  0.01341 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.478629 |  0.01341 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080999 | 0.046988 | 0.046988 | 0.047631 | 0.047631 | 0.013231 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.350643 | 0.013231 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.085826 | 0.046324 | 0.046324 | 0.046984 | 0.046984 | 0.013051 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.218493 | 0.013051 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.243802 | 0.044873 | 0.044873 | 0.046149 | 0.046149 | 0.012819 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.929818 | 0.012819 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.614337 | 0.036577 | 0.036577 | 0.043352 | 0.043352 | 0.012042 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.278903 | 0.012042 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.649436 | 0.018147 | 0.018147 | 0.034325 | 0.034325 | 0.009535 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.611319 | 0.009535 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.111623 |      0.0 |      0.0 |  0.01821 |  0.01821 | 0.005058 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.500304 | 0.005058 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.902395 |      0.0 |      0.0 | 0.004033 | 0.004033 |  0.00112 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -9.402699 |  0.00112 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.825713 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -17.228412 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.281384 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -21.509796 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.533225 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -27.043021 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.178491 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -33.221512 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.950491 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -34.934403 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.528277 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -36.46268 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.949262 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -37.411942 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.537465 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -36.405007 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.0 |  6.6528 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.6528 |  6.6528 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.096141 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.848348 |      0.0 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.0 |  3.4452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.4452 |  3.4452 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.058839 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -26.461986 |      0.0 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.045942 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -20.686729 |      0.0 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.0 | 12.5928 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.5928 | 12.5928 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.023264 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.117193 |      0.0 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080449 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.078842 |      0.0 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074914 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.322156 |      0.0 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.0 |   3.564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.564 |   3.564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.057003 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.815158 |      0.0 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.0 |  2.4948 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.4948 |  2.4948 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.063115 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -1.383473 |      0.0 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.0 | 12.3552 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.3552 | 12.3552 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.023465 | 0.054741 | 0.054741 | 0.012165 | 0.012165 | 0.003379 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.893521 | 0.003379 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 | 0.0 |   4.158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   4.158 |   4.158 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05308 | 0.074992 | 0.074992 | 0.047077 | 0.047077 | 0.013077 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.923448 | 0.013077 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 | 0.0 |  4.0392 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.0392 |  4.0392 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05353 | 0.094546 | 0.094546 | 0.074837 | 0.074837 | 0.020788 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.814573 | 0.020788 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.0 |  1.4256 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.4256 |  1.4256 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.135647 | 0.100523 | 0.100523 | 0.091529 | 0.091529 | 0.025425 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  20.004003 | 0.025425 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  1.04373 | 0.095395 | 0.095395 | 0.098055 | 0.098055 | 0.027238 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.983677 | 0.027238 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.543633 |   0.0822 |   0.0822 | 0.093603 | 0.093603 | 0.026001 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  16.357844 | 0.026001 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.612569 | 0.073726 | 0.073726 | 0.083249 | 0.083249 | 0.023125 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.671548 | 0.023125 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.960038 | 0.065934 | 0.065934 | 0.073878 | 0.073878 | 0.020522 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.120776 | 0.020522 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.130246 | 0.040547 | 0.040547 | 0.062024 | 0.062024 | 0.017229 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.068783 | 0.017229 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.837796 | 0.052539 | 0.052539 | 0.048853 | 0.048853 |  0.01357 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.455248 |  0.01357 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 | 0.0 |  7.0092 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.0092 |  7.0092 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.655358 | 0.069045 | 0.069045 | 0.053542 | 0.053542 | 0.014873 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.740045 | 0.014873 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 | 0.0 |  1.3068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.3068 |  1.3068 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.965278 | 0.060408 | 0.060408 | 0.063458 | 0.063458 | 0.017627 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  12.021159 | 0.017627 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 | 0.0 | 24.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 24.5916 | 24.5916 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.15074 |  0.18231 |  0.18231 | 0.089417 | 0.089417 | 0.024838 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.279708 | 0.024838 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.0 | 45.0252 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.0252 | 45.0252 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.020724 | 0.406421 | 0.406421 | 0.205023 | 0.205023 | 0.056951 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.877763 | 0.056951 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.0 |  9.7416 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.7416 |  9.7416 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.907373 |  0.44856 |  0.44856 | 0.365983 | 0.365983 | 0.101662 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   89.26343 | 0.101662 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.866652 | 0.463368 | 0.463368 | 0.442486 | 0.442486 | 0.122913 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   92.21021 | 0.122913 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.0 |    8.91 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    8.91 |    8.91 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.099914 | 0.505101 | 0.505101 | 0.469351 | 0.469351 | 0.130375 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 100.515194 | 0.130375 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 | 0.0 |  21.978 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  21.978 |  21.978 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.00859 | 0.612423 | 0.612423 | 0.519677 | 0.519677 | 0.144355 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  121.87218 | 0.144355 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.0 | 18.2952 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.2952 | 18.2952 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.012392 | 0.700775 | 0.700775 | 0.608208 | 0.608208 | 0.168947 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.454213 | 0.168947 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.0 |  7.4844 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.4844 |  7.4844 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.036055 | 0.734513 | 0.734513 | 0.688638 | 0.688638 | 0.191288 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 146.168045 | 0.191288 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.0 |  2.2572 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.2572 |  2.2572 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.059768 | 0.741827 | 0.741827 | 0.728641 | 0.728641 |   0.2024 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 147.623649 |   0.2024 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.041931 | 0.767015 | 0.767015 | 0.745799 | 0.745799 | 0.207166 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 152.635904 | 0.207166 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.2076 |  3.2076 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05435 | 0.778946 | 0.778946 | 0.764069 | 0.764069 | 0.212241 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.010208 | 0.212241 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.069893 | 0.777672 | 0.777672 | 0.776011 | 0.776011 | 0.215559 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.756644 | 0.215559 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.072716 | 0.774608 | 0.774608 | 0.777274 | 0.777274 | 0.215909 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.146921 | 0.215909 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.070996 |  0.77335 |  0.77335 | 0.775009 | 0.775009 |  0.21528 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 153.896575 |  0.21528 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.8512 |  2.8512 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.057521 | 0.783451 | 0.783451 | 0.775874 | 0.775874 | 0.215521 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.906803 | 0.215521 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.123337 | 0.781293 | 0.781293 | 0.780727 | 0.780727 | 0.216869 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.477372 | 0.216869 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.383628 | 0.776657 | 0.776657 | 0.780743 | 0.780743 | 0.216873 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.554688 | 0.216873 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.980296 | 0.767872 | 0.767872 | 0.775735 | 0.775735 | 0.215482 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  152.80652 | 0.215482 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.509758 | 0.746484 | 0.746484 | 0.765071 | 0.765071 |  0.21252 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 148.550278 |  0.21252 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.513746 | 0.726965 | 0.726965 | 0.746899 | 0.746899 | 0.207472 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 144.665968 | 0.207472 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.0888 |  3.0888 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.253813 | 0.722505 | 0.722505 | 0.730311 | 0.730311 | 0.202864 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  143.77845 | 0.202864 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.131769 | 0.702391 | 0.702391 | 0.719026 | 0.719026 |  0.19973 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  139.77589 |  0.19973 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.754888 | 0.681887 | 0.681887 | 0.702305 | 0.702305 | 0.195085 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.695514 | 0.195085 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.278083 | 0.648869 | 0.648869 | 0.679106 | 0.679106 | 0.188641 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 129.124962 | 0.188641 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.052029 | 0.625365 | 0.625365 | 0.650983 | 0.650983 | 0.180829 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.447569 | 0.180829 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  4.43234 | 0.600076 | 0.600076 | 0.624968 | 0.624968 | 0.173602 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.415153 | 0.173602 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.744497 | 0.583353 | 0.583353 |  0.60198 |  0.60198 | 0.167217 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 116.087303 | 0.167217 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.257992 | 0.574147 | 0.574147 | 0.585024 | 0.585024 | 0.162507 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 114.255164 | 0.162507 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.229481 | 0.570128 | 0.570128 |   0.5753 |   0.5753 | 0.159805 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.455555 | 0.159805 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.073134 | 0.566912 | 0.566912 | 0.570307 | 0.570307 | 0.158418 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.815508 | 0.158418 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.564078 | 0.564078 | 0.566997 | 0.566997 | 0.157499 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.251431 | 0.157499 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.561257 | 0.561257 | 0.564081 | 0.564081 | 0.156689 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.690174 | 0.156689 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.558451 | 0.558451 |  0.56126 |  0.56126 | 0.155906 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.131723 | 0.155906 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.555659 | 0.555659 | 0.558454 | 0.558454 | 0.155126 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.576064 | 0.155126 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.55288 |  0.55288 | 0.555662 | 0.555662 |  0.15435 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.023184 |  0.15435 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.550116 | 0.550116 | 0.552883 | 0.552883 | 0.153579 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.473068 | 0.153579 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.555087 | 0.555087 | 0.551835 | 0.551835 | 0.153287 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.462381 | 0.153287 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.552312 | 0.552312 | 0.553366 | 0.553366 | 0.153713 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.910069 | 0.153713 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.54955 |  0.54955 | 0.552315 | 0.552315 | 0.153421 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.360518 | 0.153421 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.546803 | 0.546803 | 0.549553 | 0.549553 | 0.152654 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.813716 | 0.152654 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.548227 | 0.548227 |  0.54773 |  0.54773 | 0.152147 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.097089 | 0.152147 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.547861 | 0.547861 | 0.547829 | 0.547829 | 0.152175 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.024428 | 0.152175 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.545716 | 0.545716 | 0.547466 | 0.547466 | 0.152074 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.597512 | 0.152074 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.698554 | 0.516871 | 0.516871 | 0.539783 | 0.539783 |  0.14994 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 102.857287 |  0.14994 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.008524 | 0.479244 | 0.479244 | 0.514919 | 0.514919 | 0.143033 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  95.369519 | 0.143033 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.008402 | 0.441806 | 0.441806 | 0.479286 | 0.479286 | 0.133135 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.919312 | 0.133135 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 8.949367 |  0.39485 |  0.39485 | 0.439691 | 0.439691 | 0.122136 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.575095 | 0.122136 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.837011 |  0.35369 |  0.35369 | 0.396138 | 0.396138 | 0.110038 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  70.384394 | 0.110038 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.566271 | 0.319091 | 0.319091 | 0.355148 | 0.355148 | 0.098652 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  63.499032 | 0.098652 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.260364 | 0.291193 | 0.291193 |  0.32058 |  0.32058 |  0.08905 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  57.947474 |  0.08905 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.462517 | 0.277425 | 0.277425 | 0.294333 | 0.294333 | 0.081759 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  55.207533 | 0.081759 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.093471 |  0.27057 |  0.27057 | 0.278961 | 0.278961 | 0.077489 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  53.843492 | 0.077489 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.27524 | 0.267841 | 0.267841 | 0.271487 | 0.271487 | 0.075413 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   53.30041 | 0.075413 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074757 | 0.266128 | 0.266128 | 0.268067 | 0.268067 | 0.074463 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.959525 | 0.074463 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.264798 | 0.264798 | 0.266213 | 0.266213 | 0.073948 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.694727 | 0.073948 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.263474 | 0.263474 | 0.264799 | 0.264799 | 0.073555 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.431254 | 0.073555 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.262156 | 0.262156 | 0.263475 | 0.263475 | 0.073188 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.169097 | 0.073188 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_snow_classes:

snow classes
____________

In this example, we divide the single zone into two snow classes.  All snow classes of
a specific zone are generally identical in their size and process parameterisation.
The only difference is that they receive different amounts of (frozen or liquid)
precipitation.  The extent of this difference depends on parameter |SFDist|, for which
we apply the standard "linear-mode" value 0.2:

>>> sclass(2)
>>> sfdist(linear=0.2)

We reset the land-use type to |FIELD| and modify the input temperature series to
activate the snow routines.  In the first half of the simulation period, the
temperature is -20°C, and in the second half, it is +20°C:

>>> zonetype(FIELD)
>>> t_series = inputs.t.series.copy()
>>> tn_series = inputs.tn.series.copy()
>>> inputs.t.series[:48] = -20.0
>>> inputs.t.series[48:] = 20.0
>>> inputs.tn.series = inputs.t.series

The second snow class receives more precipitation and thus builds a deeper snow layer.
At the beginning of the warm period, the melting rates of both classes are identical.
Due to the huger ice content, the second class possesses a higher storage capacity and
releases its meltwater later.  Also, it takes longer until it is finally snow-free:

.. integration-test::

    >>> test("hland_v1_snow_classes")
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |       cf |      inuz |     perc |       q0 |  el |       q1 |     inuh |    outuh |       rt |       qt |       ic |                     sp |                   wc |         sm |        uz |        lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.24976 |     0.0 |  0.106749 |  0.07686 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048037 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |     0.05 |     0.05 | 0.061111 | 0.061111 | 0.016975 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.951963 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249527 |     0.0 |  0.103669 | 0.074642 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046629 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |  0.04975 | 0.088833 | 0.088833 | 0.024676 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.905334 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249296 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046216 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049501 | 0.049501 |  0.04975 |  0.04975 |  0.01382 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.859118 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249068 |     0.0 |  0.101749 |  0.07326 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.045723 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049254 | 0.049254 | 0.049502 | 0.049502 |  0.01375 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.813395 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248837 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046174 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049007 | 0.049007 | 0.049254 | 0.049254 | 0.013682 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.767222 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248594 |     0.0 |  0.108927 | 0.078427 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048903 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048762 | 0.048762 | 0.049008 | 0.049008 | 0.013613 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.718319 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.247902 |     0.0 |  0.309422 | 0.222784 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.138848 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048519 | 0.048519 | 0.048763 | 0.048763 | 0.013545 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.579471 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.243352 |     0.0 |  2.048847 |  1.47517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.918104 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048276 | 0.048276 | 0.048519 | 0.048519 | 0.013477 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.661367 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.233313 |     0.0 |  4.631708 |  3.33483 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.056368 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048035 | 0.048035 | 0.048276 | 0.048276 |  0.01341 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.604999 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.21731 |     0.0 |  7.756609 | 5.584759 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.371973 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047794 | 0.047794 | 0.048035 | 0.048035 | 0.013343 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.233026 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.200514 |     0.0 |  8.760224 | 6.307361 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  3.67534 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047556 | 0.047556 | 0.047795 | 0.047795 | 0.013276 |      0.0 |        0.0         0.0 |       0.0        0.0 |  89.557686 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.182991 |     0.0 |  9.932059 | 7.151082 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.002715 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047318 | 0.047318 | 0.047556 | 0.047556 |  0.01321 |      0.0 |        0.0         0.0 |       0.0        0.0 |  85.554971 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.174152 |     0.0 |  5.433749 | 3.912299 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.091979 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047081 | 0.047081 | 0.047318 | 0.047318 | 0.013144 |      0.0 |        0.0         0.0 |       0.0        0.0 |  83.462992 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.163319 |     0.0 |  7.022532 | 5.056223 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.637547 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046846 | 0.046846 | 0.047081 | 0.047081 | 0.013078 |      0.0 |        0.0         0.0 |       0.0        0.0 |  80.825446 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.151996 |     0.0 |  7.841476 | 5.645863 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.852059 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046612 | 0.046612 | 0.046846 | 0.046846 | 0.013013 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.973387 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.148643 |  0.2808 |  2.535001 | 1.774662 |   0.2808 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.864852 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046378 | 0.046378 | 0.046612 | 0.046612 | 0.012948 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.108535 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.14606 |     0.0 |  1.939624 | 1.396529 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.673027 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046147 | 0.046147 | 0.046379 | 0.046379 | 0.012883 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.435507 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.14448 |     0.0 |  1.204763 | 0.867429 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.41439 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045916 | 0.045916 | 0.046147 | 0.046147 | 0.012819 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.021117 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.144049 |  1.8252 |  0.796045 | 0.477532 | 0.477532 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.113445 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045686 | 0.045686 | 0.045916 | 0.045916 | 0.012754 | 1.347668 |        0.0         0.0 |       0.0        0.0 |  75.907672 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.768054    8.652081 |  0.0 |   0.144049 |  7.8624 |  0.237329 | 0.077844 | 0.077844 |  7.210068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045458 | 0.045458 | 0.045686 | 0.045686 | 0.012691 | 1.922156 |   5.768054    8.652081 |       0.0        0.0 |  75.907672 |       0.0 |  9.046105 | 0.012691 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   8.963059   13.444589 |  0.0 |   0.144049 |  4.0716 |   0.10539 | 0.050502 | 0.050502 |  3.993756 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045231 | 0.045231 | 0.045458 | 0.045458 | 0.012627 | 1.949498 |   8.963059   13.444589 |       0.0        0.0 |  75.907672 |       0.0 |  9.000874 | 0.012627 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.426338   21.639507 |  0.0 |   0.144049 |  6.8796 |  0.104361 | 0.037765 | 0.037765 |  6.829098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045004 | 0.045004 | 0.045231 | 0.045231 | 0.012564 | 1.962235 |  14.426338   21.639507 |       0.0        0.0 |  75.907672 |       0.0 |   8.95587 | 0.012564 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.302045   39.453068 |  0.0 |   0.144049 | 14.8824 |  0.104016 | 0.016908 | 0.016908 | 14.844635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044779 | 0.044779 | 0.045005 | 0.045005 | 0.012501 | 1.983092 |  26.302045   39.453068 |       0.0        0.0 |  75.907672 |       0.0 |  8.911091 | 0.012501 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.400839   39.601258 |  0.0 |   0.144049 |  0.1404 |  0.103322 | 0.073355 | 0.073355 |  0.123492 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044555 | 0.044555 |  0.04478 |  0.04478 | 0.012439 | 1.926645 |  26.400839   39.601258 |       0.0        0.0 |  75.907672 |       0.0 |  8.866535 | 0.012439 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.128395   40.692592 |  0.0 |   0.144049 |  0.9828 |  0.103322 | 0.067429 | 0.067429 |  0.909445 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044333 | 0.044333 | 0.044556 | 0.044556 | 0.012377 | 1.932571 |  27.128395   40.692592 |       0.0        0.0 |  75.907672 |       0.0 |  8.822202 | 0.012377 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  30.444052   45.666078 |  0.0 |   0.144049 |   4.212 |  0.103322 | 0.048821 | 0.048821 |  4.144571 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044111 | 0.044111 | 0.044333 | 0.044333 | 0.012315 | 1.951179 |  30.444052   45.666078 |       0.0        0.0 |  75.907672 |       0.0 |  8.778091 | 0.012315 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  32.763716   49.145573 |  0.0 |   0.144049 |  2.9484 |    0.1028 | 0.055116 | 0.055116 |  2.899579 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04389 |  0.04389 | 0.044111 | 0.044111 | 0.012253 | 1.944884 |  32.763716   49.145573 |       0.0        0.0 |  75.907672 |       0.0 |  8.734201 | 0.012253 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  44.400903   66.601354 |  0.0 |   0.144049 | 14.6016 |  0.102451 | 0.017128 | 0.017128 | 14.546484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043671 | 0.043671 | 0.043891 | 0.043891 | 0.012192 | 1.982872 |  44.400903   66.601354 |       0.0        0.0 |  75.907672 |       0.0 |   8.69053 | 0.012192 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    48.3184     72.4776 |  0.0 |   0.144049 |   4.914 |    0.1021 | 0.044973 | 0.044973 |  4.896872 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043453 | 0.043453 | 0.043671 | 0.043671 | 0.012131 | 1.955027 |    48.3184     72.4776 |       0.0        0.0 |  75.907672 |       0.0 |  8.647077 | 0.012131 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  52.101302   78.151953 |  0.0 |   0.144049 |  4.7736 |  0.101749 | 0.045452 | 0.045452 |  4.728627 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043235 | 0.043235 | 0.043453 | 0.043453 |  0.01207 | 1.954548 |  52.101302   78.151953 |       0.0        0.0 |  75.907672 |       0.0 |  8.603842 |  0.01207 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.412781   80.119171 |  0.0 |   0.144049 |  1.6848 |  0.198536 | 0.120782 | 0.120782 |  1.639348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043019 | 0.043019 | 0.043236 | 0.043236 |  0.01201 | 1.879218 |  53.412781   80.119171 |       0.0        0.0 |  75.907672 |       0.0 |  8.560823 |  0.01201 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.1404 |  1.340489 | 0.951696 | 0.951696 |  0.019618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042804 | 0.042804 | 0.043019 | 0.043019 |  0.01195 | 1.048304 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.518019 |  0.01195 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |     0.0 |   3.22827 | 2.324355 | 1.048304 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04259 |  0.04259 | 0.042804 | 0.042804 |  0.01189 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.475429 |  0.01189 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |     0.0 |  2.046603 | 1.473555 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042377 | 0.042377 |  0.04259 |  0.04259 | 0.011831 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.433051 | 0.011831 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.5616 |  2.608661 | 1.775661 |   0.5616 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042165 | 0.042165 | 0.042377 | 0.042377 | 0.011771 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.390886 | 0.011771 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.1404 |  6.588902 | 4.677869 |   0.1404 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041954 | 0.041954 | 0.042165 | 0.042165 | 0.011713 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.348932 | 0.011713 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  55.871996   83.807993 |  0.0 |   0.144049 |  5.0544 |   3.57727 | 1.553726 | 1.553726 |    3.0544 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041745 | 0.041745 | 0.041955 | 0.041955 | 0.011654 | 0.446274 |  55.871996   83.807993 |       0.0        0.0 |  75.907672 |       0.0 |  8.307187 | 0.011654 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |   0.144049 |  8.2836 |  9.350848 | 2.940569 |      2.0 |  6.729874 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041536 | 0.041536 | 0.041745 | 0.041745 | 0.011596 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  8.265651 | 0.011596 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |   0.144049 |  1.5444 |  4.288787 | 2.646028 |   1.5444 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041328 | 0.041328 | 0.041536 | 0.041536 | 0.011538 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  8.224323 | 0.011538 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  82.906135  124.359202 |  0.0 |   0.144049 | 29.0628 |  2.237402 | 0.088084 | 0.088084 |   27.0628 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041122 | 0.041122 | 0.041328 | 0.041328 |  0.01148 | 1.911916 |  82.906135  124.359202 |       0.0        0.0 |  75.907672 |       0.0 |  8.183201 |  0.01148 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 125.404948  188.107422 |  0.0 |   0.144049 | 53.2116 |  2.373612 | 0.008352 | 0.008352 | 53.123516 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040916 | 0.040916 | 0.041122 | 0.041122 | 0.011423 | 1.991648 | 125.404948  188.107422 |       0.0        0.0 |  75.907672 |       0.0 |  8.142285 | 0.011423 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 134.608506  201.912759 |  0.0 |   0.144049 | 11.5128 |  3.050519 | 0.694563 | 0.694563 | 11.504448 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040711 | 0.040711 | 0.040916 | 0.040916 | 0.011366 | 1.305437 | 134.608506  201.912759 |       0.0        0.0 |  75.907672 |       0.0 |  8.101574 | 0.011366 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 138.096376  207.144564 |  0.0 |   0.144049 |  5.0544 |  1.686939 | 0.732693 | 0.732693 |  4.359837 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040508 | 0.040508 | 0.040712 | 0.040712 | 0.011309 | 1.267307 | 138.096376  207.144564 |       0.0        0.0 |  75.907672 |       0.0 |  8.061066 | 0.011309 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 145.934221  218.901332 |  0.0 |   0.144049 |   10.53 |    0.3091 | 0.077646 | 0.077646 |  9.797307 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040305 | 0.040305 | 0.040508 | 0.040508 | 0.011252 | 1.922354 | 145.934221  218.901332 |       0.0        0.0 |  75.907672 |       0.0 |  8.020761 | 0.011252 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 166.651304  249.976957 |  0.0 |   0.144049 |  25.974 |  0.098179 | 0.005264 | 0.005264 | 25.896354 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040104 | 0.040104 | 0.040306 | 0.040306 | 0.011196 | 1.994736 | 166.651304  249.976957 |       0.0        0.0 |  75.907672 |       0.0 |  7.980657 | 0.011196 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 183.944373   275.91656 |  0.0 |   0.144049 | 21.6216 |  0.097998 |  0.00812 |  0.00812 | 21.616336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039903 | 0.039903 | 0.040104 | 0.040104 |  0.01114 |  1.99188 | 183.944373   275.91656 |       0.0        0.0 |  75.907672 |       0.0 |  7.940753 |  0.01114 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 191.014038  286.521056 |  0.0 |   0.144049 |  8.8452 |  0.096723 | 0.028755 | 0.028755 |   8.83708 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039704 | 0.039704 | 0.039904 | 0.039904 | 0.011084 | 1.971245 | 191.014038  286.521056 |       0.0        0.0 |  75.907672 |       0.0 |   7.90105 | 0.011084 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 193.125113   289.68767 |  0.0 |   0.144049 |  2.6676 |  0.095064 |  0.05242 |  0.05242 |  2.638845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039505 | 0.039505 | 0.039704 | 0.039704 | 0.011029 |  1.94758 | 193.125113   289.68767 |       0.0        0.0 |  75.907672 |       0.0 |  7.861544 | 0.011029 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 197.740137  296.610206 |  0.0 |   0.144049 |  5.8212 |  0.095249 | 0.038316 | 0.038316 |   5.76878 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039308 | 0.039308 | 0.039505 | 0.039505 | 0.010974 | 1.961684 |  184.27879  280.841347 | 13.461347  15.768859 |  75.907672 |       0.0 |  7.822237 | 0.010974 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.275564  300.413346 |  0.0 |   0.144049 |  3.2076 |  0.095064 | 0.049664 | 0.049664 |  3.169284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039111 | 0.039111 | 0.039308 | 0.039308 | 0.010919 | 1.950336 | 175.432468  271.995024 | 24.843097  28.418322 |  75.907672 |       0.0 |  7.783126 | 0.010919 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.711033  301.066549 |  0.0 |   0.144049 |   0.594 |  0.094133 | 0.063867 | 0.063867 |  0.544336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038916 | 0.038916 | 0.039111 | 0.039111 | 0.010864 | 1.936133 | 166.586145  263.148701 | 34.124888  37.917848 |  75.907672 |       0.0 |   7.74421 | 0.010864 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.850019  301.275028 |  0.0 |   0.144049 |  0.2376 |  0.094506 | 0.066447 | 0.066447 |  0.173733 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038721 | 0.038721 | 0.038916 | 0.038916 |  0.01081 | 1.933553 | 157.739822  254.302379 | 43.110197   46.97265 |  75.907672 |       0.0 |  7.705489 |  0.01081 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.272061  301.908092 |  0.0 |   0.144049 |   0.594 |  0.095619 | 0.064876 | 0.064876 |  0.527553 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038527 | 0.038527 | 0.038721 | 0.038721 | 0.010756 | 1.935124 | 148.893499  245.456056 | 52.378562  56.452036 |  75.907672 |       0.0 |  7.666961 | 0.010756 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 196.066047  305.251681 |  0.0 |   0.156619 |  2.8512 |  0.097089 | 0.052562 | 0.052562 |  2.786324 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.717537 | 0.535509 | 0.0 |      0.0 | 0.060455 |  0.475054 | 0.078309 | 0.004075 | 0.0 | 0.038726 | 0.042801 |  0.03952 |  0.03952 | 0.010978 | 1.947438 | 140.047176  236.609733 |  56.01887  68.641949 |  79.150155 |   0.39267 |  7.706544 | 0.010978 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 183.681195  305.758847 |  0.0 |   0.178815 |  0.4752 |  0.164153 | 0.112705 | 0.112705 |  0.422638 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.361481 | 0.996327 | 0.0 |      0.0 | 0.057742 |  0.938584 | 0.089408 | 0.105621 | 0.0 |  0.03898 | 0.144601 | 0.064474 | 0.064474 | 0.017909 | 1.887295 | 131.200853   227.76341 | 52.480341  77.995437 |  84.573052 |  1.136225 |  7.756972 | 0.017909 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 171.296343  305.908721 |  0.0 |   0.201396 |  0.2376 |  0.498591 | 0.350557 | 0.350557 |  0.124895 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.242384 | 1.116232 | 0.0 |      0.0 |  0.05515 |  1.061082 | 0.100698 | 0.396479 | 0.0 | 0.039288 | 0.435768 | 0.186682 | 0.186682 | 0.051856 | 1.649443 | 122.354531  218.917087 | 48.941812  86.991634 |  89.754354 |   1.70013 |  7.818382 | 0.051856 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 158.911491   294.09907 |  0.0 |   0.247335 |     0.0 |   1.24415 | 0.895788 | 0.895788 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.097251 | 2.436339 | 0.0 |      0.0 | 0.050292 |  2.386047 | 0.123667 |  1.03346 | 0.0 |  0.03971 |  1.07317 | 0.512709 | 0.512709 | 0.142419 | 0.753656 | 113.508208  210.070764 | 45.403283  84.028306 |  99.465559 |  2.929049 |  7.902339 | 0.142419 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 146.526639  281.714218 |  0.0 |   0.296115 |     0.0 |  4.454434 | 3.207192 | 0.753656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.063207 | 0.0 |      0.0 | 0.045606 |    3.0176 | 0.148057 | 1.829676 | 0.0 | 0.040252 | 1.869928 | 1.108583 | 1.108583 |  0.30794 |      0.0 | 104.661885  201.224442 | 41.864754  80.489777 | 108.832811 |  3.968916 |  8.010144 |  0.30794 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 134.141787  269.329366 |  0.0 |   0.345694 |  0.3564 |  4.621298 | 3.210837 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.667334 | 0.0 |      0.0 | 0.041225 |   3.62611 | 0.172847 | 2.436608 | 0.0 | 0.040915 | 2.477523 | 1.827892 | 1.827892 | 0.507748 |      0.0 |  95.815562  192.378119 | 38.326225  76.951247 | 117.591553 |   4.98557 |  8.142077 | 0.507748 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 121.756935  256.944514 |  0.0 |   0.399704 |  3.0888 |  5.624098 | 2.973312 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.473652 | 4.657765 | 0.0 |      0.0 | 0.036796 |  4.620969 | 0.199852 | 3.137838 | 0.0 |  0.04171 | 3.179547 | 2.498507 | 2.498507 |  0.69403 |      0.0 |  86.969239  183.531796 | 34.787696  73.412718 | 126.444236 |  6.268849 |  8.300219 |  0.69403 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 109.372083  244.559662 |  0.0 |    0.44831 |  0.8316 |  5.698589 | 3.775582 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  4.95027 | 0.0 |      0.0 | 0.033061 |   4.91721 | 0.224155 | 3.709554 | 0.0 | 0.042622 | 3.752176 | 3.150793 | 3.150793 |  0.87522 |      0.0 |  78.122916  174.685473 | 31.249167  69.874189 | 133.911878 |   7.25235 |  8.481752 |  0.87522 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  96.987231   232.17481 |  0.0 |   0.495434 |  0.3564 |   4.93845 | 3.431191 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  5.55225 | 0.0 |      0.0 | 0.029628 |  5.522622 | 0.247717 | 4.192861 | 0.0 | 0.043647 | 4.236508 | 3.732554 | 3.732554 | 1.036821 |      0.0 |  69.276594   165.83915 | 27.710637   66.33566 | 140.774108 |  8.334395 |  8.685821 | 1.036821 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  84.602379  219.789958 |  0.0 |   0.540589 |  0.3564 |   8.25697 | 5.736869 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.135873 | 0.0 |      0.0 | 0.026488 |  6.109385 | 0.270295 | 4.743723 | 0.0 | 0.044781 | 4.788503 | 4.251544 | 4.251544 | 1.180985 |      0.0 |  60.430271  156.992827 | 24.172108  62.797131 | 147.049575 |  9.429763 |  8.911335 | 1.180985 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  72.217527  207.405106 |  0.0 |   0.583413 |     0.0 |  5.142661 | 3.702716 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  6.69512 | 0.0 |      0.0 |  0.02363 |   6.67149 | 0.291706 | 5.300698 | 0.0 | 0.046015 | 5.346713 | 4.789884 | 4.789884 | 1.330523 |      0.0 |  51.583948  148.146505 | 20.633579  59.258602 | 152.762937 | 10.508848 |  9.157027 | 1.330523 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  59.832675  195.020254 |  0.0 |   0.623653 |     0.0 |  5.625335 | 4.050241 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.225482 | 0.0 |      0.0 | 0.021039 |  7.204443 | 0.311826 | 5.846186 | 0.0 | 0.047344 |  5.89353 | 5.344181 | 5.344181 | 1.484495 |      0.0 |  42.737625  139.300182 |  17.09505  55.720073 | 157.943346 | 11.555279 |  9.421509 | 1.484495 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  47.447823  182.635402 |  0.0 |   0.661156 |     0.0 |  3.483198 | 2.507903 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.723844 | 0.0 |      0.0 | 0.018698 |  7.705146 | 0.330578 | 6.372206 | 0.0 |  0.04876 | 6.420967 | 5.889223 | 5.889223 | 1.635895 |      0.0 |  33.891302  130.453859 | 13.556521  52.181544 | 162.623052 | 12.557641 |  9.703326 | 1.635895 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  35.062971   170.25055 |  0.0 |   0.695858 |     0.0 |  1.596589 | 1.149544 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.188324 | 0.0 |      0.0 |  0.01659 |  8.171734 | 0.347929 | 6.873745 | 0.0 | 0.050256 | 6.924001 | 6.415544 | 6.415544 | 1.782096 |      0.0 |  25.044979  121.607536 | 10.017992  48.643014 | 166.836169 | 13.507701 | 10.000999 | 1.782096 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  22.678119  157.865698 |  0.0 |   0.727759 |     0.0 |  0.291248 | 0.209698 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.618094 | 0.0 |      0.0 | 0.014699 |  8.603396 |  0.36388 | 7.347361 | 0.0 | 0.051824 | 7.399186 | 6.917812 | 6.917812 | 1.921615 |      0.0 |  16.198657  112.761213 |  6.479463  45.104485 | 170.617625 | 14.399856 | 10.313054 | 1.921615 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  10.293267  145.480847 |  0.0 |    0.75692 |     0.0 |  0.092819 |  0.06683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.013192 | 0.0 |      0.0 | 0.013005 |  9.000186 |  0.37846 | 7.790813 | 0.0 | 0.053458 | 7.844271 | 7.392497 | 7.392497 | 2.053471 |      0.0 |   7.352334   103.91489 |  2.940934  41.565956 | 174.002291 |  15.23077 | 10.638057 | 2.053471 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  133.095995 |  0.0 |   0.780903 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 7.352334  8.846323 | 0.0   0.0 |  11.33906 |  8.58276 | 0.0 |  0.03273 | 0.011621 |   8.57114 | 0.390452 |  7.84807 | 0.0 | 0.055143 | 7.903212 | 7.758461 | 7.758461 | 2.155128 |      0.0 |        0.0   95.068568 |       0.0  38.027427 | 176.737481 | 15.563388 | 10.973366 | 2.155128 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  120.711143 |  0.0 |    0.79275 |     0.0 |  0.089376 | 0.064351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.835687 | 0.0 | 0.032175 | 0.010953 |  4.824734 | 0.396375 |   6.1996 | 0.0 | 0.056849 | 6.256448 | 7.524167 | 7.524167 | 2.090046 |      0.0 |        0.0   86.222245 |       0.0  34.488898 | 178.072998 | 13.792147 | 11.312892 | 2.090046 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  108.326291 |  0.0 |   0.804026 |     0.0 |  0.088208 |  0.06351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.909045 | 0.0 | 0.031755 | 0.010322 |  4.898723 | 0.402013 | 4.864603 | 0.0 | 0.058575 | 4.923178 | 6.326114 | 6.326114 | 1.757254 |      0.0 |        0.0   77.375922 |       0.0  30.950369 | 179.334946 | 13.424254 |  11.65633 | 1.757254 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    95.93582 |  0.0 |   0.814753 |     0.0 |  0.087226 | 0.062803 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 4.981128 | 0.0 | 0.031401 | 0.009725 |  4.971402 | 0.407377 |  4.53539 | 0.0 | 0.060319 | 4.595709 | 5.146689 | 5.146689 | 1.429636 |      0.0 |        0.0   68.525586 |       0.0  27.410234 | 180.527378 | 13.452889 | 12.003388 | 1.429636 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   83.545349 |  0.0 |   0.824936 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.047589 | 0.0 | 0.032245 | 0.009162 |  5.038426 | 0.412468 | 4.470074 | 0.0 | 0.062079 | 4.532153 | 4.654356 | 4.654356 | 1.292877 |      0.0 |        0.0   59.675249 |       0.0    23.8701 | 181.651942 | 13.608774 | 12.353777 | 1.292877 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   71.154878 |  0.0 |   0.834599 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.110671 | 0.0 | 0.032384 | 0.008632 |  5.102039 | 0.417299 | 4.489519 | 0.0 | 0.063855 | 4.553375 | 4.550992 | 4.550992 | 1.264165 |      0.0 |        0.0   50.824913 |       0.0  20.329965 | 182.712754 | 13.803994 | 12.707221 | 1.264165 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   58.764408 |  0.0 |   0.843935 |  1.5444 |  0.088599 | 0.054662 | 0.054662 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.170536 | 0.0 | 0.013666 | 0.008131 |  5.162404 | 0.421968 | 4.535454 | 0.0 | 0.065646 |   4.6011 | 4.559264 | 4.559264 | 1.266462 | 1.489738 |        0.0   41.974577 |       0.0  16.789831 | 183.731919 | 14.008977 | 13.063543 | 1.266462 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   46.373937 |  0.0 |   0.852765 |     0.0 |  0.088404 | 0.063651 | 0.063651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.228379 | 0.0 | 0.015913 | 0.007651 |  5.220728 | 0.426382 | 4.591621 | 0.0 |  0.06745 | 4.659071 | 4.603377 | 4.603377 | 1.278716 | 1.426087 |        0.0   33.124241 |       0.0  13.249696 | 184.690514 | 14.211702 | 13.422476 | 1.278716 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   33.983466 |  0.0 |   0.861125 |     0.0 |  0.089956 | 0.064768 | 0.064768 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.283078 | 0.0 | 0.016192 | 0.007199 |  5.275879 | 0.430563 | 4.649731 | 0.0 | 0.069265 | 4.718996 | 4.659505 | 4.659505 | 1.294307 | 1.361318 |        0.0   24.273904 |       0.0   9.709562 | 185.593678 | 14.407287 | 13.783773 | 1.294307 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   21.592995 |  0.0 |   0.868639 |     0.0 |  0.328843 | 0.236767 | 0.236767 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.334874 | 0.0 | 0.059192 | 0.006773 |  5.328101 |  0.43432 | 4.710916 | 0.0 |  0.07109 | 4.782007 | 4.719682 | 4.719682 | 1.311023 | 1.124552 |        0.0   15.423568 |       0.0   6.169427 |  186.40162 | 14.590152 | 14.147002 | 1.311023 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    9.202525 |  0.0 |   0.874014 |  0.8316 |  1.475475 | 0.977572 | 0.977572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.381424 | 0.0 | 0.244393 | 0.006392 |  5.375031 | 0.437007 | 4.784772 | 0.0 |  0.07292 | 4.857692 | 4.784823 | 4.784823 | 1.329118 |  0.97858 |        0.0    6.573232 |       0.0   2.629293 | 186.977431 | 14.743404 | 14.511089 | 1.329118 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.864568 |  0.4752 |  3.387629 | 2.325898 |  1.45378 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  6.573232 | 0.0   0.0 |  4.601262 | 4.021568 | 0.0 | 1.599008 | 0.006221 |  4.015346 | 0.432284 | 4.490724 | 0.0 | 0.074717 | 4.565441 | 4.775929 | 4.775929 | 1.326647 |      0.0 |        0.0         0.0 |       0.0        0.0 |  185.96434 | 13.835743 | 14.868656 | 1.326647 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.829149 |  0.1188 |   5.50315 | 3.915475 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.856075 | 0.007018 | -0.007018 | 0.414575 |  3.02507 | 0.0 | 0.076416 | 3.101486 | 4.305062 | 4.305062 | 1.195851 |      0.0 |        0.0         0.0 |       0.0        0.0 | 182.115283 |  10.38908 | 15.206815 | 1.195851 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.784593 |  0.4752 |  7.584338 | 5.207299 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.969699 | 0.008942 | -0.008942 | 0.392297 | 1.668741 | 0.0 | 0.077996 | 1.746736 | 3.125754 | 3.125754 | 0.868265 |      0.0 |        0.0         0.0 |       0.0        0.0 | 177.154526 |    8.3191 | 15.521116 | 0.868265 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.728988 |     0.0 |  8.894918 | 6.404341 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404341 | 0.011423 | -0.011423 | 0.364494 | 1.110617 | 0.0 | 0.079428 | 1.190045 | 1.924083 | 1.924083 | 0.534467 |      0.0 |        0.0         0.0 |       0.0        0.0 | 170.761608 |  6.832566 | 15.806182 | 0.534467 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.675454 |     0.0 |  8.894763 | 6.404229 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404229 | 0.014619 | -0.014619 | 0.337727 | 0.790254 | 0.0 |  0.08072 | 0.870974 | 1.242849 | 1.242849 | 0.345236 |      0.0 |        0.0         0.0 |       0.0        0.0 | 164.371998 |  5.689966 | 16.063189 | 0.345236 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.610054 |     0.0 | 11.358152 |  8.17787 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  8.17787 | 0.017814 | -0.017814 | 0.305027 | 0.621287 | 0.0 | 0.081841 | 0.703128 | 0.904579 | 0.904579 | 0.251272 |      0.0 |        0.0         0.0 |       0.0        0.0 | 156.211942 |  4.745838 | 16.286375 | 0.251272 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.556822 |     0.0 |  9.946398 | 7.161406 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.992837 | 0.021894 | -0.021894 | 0.278411 | 0.476358 | 0.0 | 0.082824 | 0.559182 | 0.708439 | 0.708439 | 0.196789 |      0.0 |        0.0         0.0 |       0.0        0.0 | 149.240999 |  3.969175 | 16.481962 | 0.196789 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.516017 |     0.0 |   8.33363 | 6.000214 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.597688 |  0.02538 |  -0.02538 | 0.258009 | 0.352578 | 0.0 |   0.0837 | 0.436277 | 0.563858 | 0.563858 | 0.156627 |      0.0 |        0.0         0.0 |       0.0        0.0 |  143.66869 |  3.333209 | 16.656271 | 0.156627 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.485668 |     0.0 |  6.676229 | 4.806885 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.317089 | 0.028166 | -0.028166 | 0.242834 | 0.252146 | 0.0 | 0.084496 | 0.336642 | 0.441448 | 0.441448 | 0.122625 |      0.0 |        0.0         0.0 |       0.0        0.0 | 139.379767 |  2.810063 |  16.81461 | 0.122625 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.472309 |     0.0 |  3.125321 | 2.250231 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.960656 |  0.03031 |  -0.03031 | 0.236154 | 0.165423 | 0.0 | 0.085254 | 0.250677 |  0.33968 |  0.33968 | 0.094356 |      0.0 |        0.0         0.0 |       0.0        0.0 | 137.449421 |  2.378176 |  16.96551 | 0.094356 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.46664 |     0.0 |  1.387786 | 0.999206 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.858572 | 0.031275 | -0.031275 |  0.23332 | 0.103924 | 0.0 | 0.085994 | 0.189918 | 0.256278 | 0.256278 | 0.071188 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.622125 |  2.009657 | 17.112836 | 0.071188 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.46539 |     0.0 |  0.349323 | 0.251513 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.214813 | 0.031689 | -0.031689 | 0.232695 | 0.062535 | 0.0 | 0.086728 | 0.149263 | 0.194386 | 0.194386 | 0.053996 |      0.0 |        0.0         0.0 |       0.0        0.0 |    136.439 |  1.682737 | 17.258803 | 0.053996 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.465209 |     0.0 |  0.094878 | 0.068313 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.058267 |  0.03178 |  -0.03178 | 0.232605 | 0.035896 | 0.0 | 0.087457 | 0.123353 |  0.15254 |  0.15254 | 0.042372 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.412514 |  1.382456 | 17.403951 | 0.042372 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.465045 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055823 | 0.031794 | -0.031794 | 0.232523 | 0.019132 | 0.0 | 0.088182 | 0.107315 | 0.125547 | 0.125547 | 0.034874 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.388485 |  1.099007 | 17.548291 | 0.034874 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.464887 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.054986 | 0.031806 | -0.031806 | 0.232444 | 0.008997 | 0.0 | 0.088904 | 0.097901 | 0.108787 | 0.108787 | 0.030219 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.365304 |  0.825761 | 17.691831 | 0.030219 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.464728 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055214 | 0.031817 | -0.031817 | 0.232364 | 0.003401 | 0.0 | 0.089621 | 0.093022 | 0.098909 | 0.098909 | 0.027475 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.341908 |  0.558178 | 17.834574 | 0.027475 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_glacier:

glacier
_______

This example demonstrates the functionality of zones of type |GLACIER|.  |GLACIER| zones
are similar to zones of type |FIELD| or |FOREST| but possess neither an interception nor
a soil module.  Instead, they pass precipitation and meltwater to the upper zone storage
(|UZ|) directly. The snow routines of |GLACIER|, |FIELD|, and |FOREST| zones are
identical.  Additional glacier melt can only occur if no snow covers the glacier.  In
the next test run, we can distinguish the simulation period into three subperiods.  On
the first two days (-20°C), the snow layer builds up.  On the third day (+20°C and
|SP| > 0), the snow melts and (with some time delay) releases meltwater. On the fourth
day (+20°C and |SP| = 0), an increased amount of water is passed to |UZ| due to |GMelt|
being larger than |CFMax|:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test("hland_v1_glacier")
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact | contriarea |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |  cf |      inuz | perc |        q0 |  el |       q1 |      inuh |     outuh |        rt |       qt |  ic |                     sp |                   wc |  sm |        uz |        lz |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.106749 |  0.07686 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.061111 | 0.016975 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |      9.95 | 0.016975 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.103669 | 0.074642 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.088833 | 0.024676 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.90025 | 0.024676 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.04975 |  0.01382 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.850749 |  0.01382 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.101749 |  0.07326 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.049502 |  0.01375 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.801495 |  0.01375 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.049254 | 0.013682 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.752488 | 0.013682 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.108927 | 0.078427 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.049008 | 0.013613 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.703725 | 0.013613 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.309422 | 0.222784 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.048763 | 0.013545 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.655206 | 0.013545 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  2.048847 |  1.47517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.048519 | 0.013477 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.60693 | 0.013477 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  4.631708 |  3.33483 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.048276 |  0.01341 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.558896 |  0.01341 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.756609 | 5.584759 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.048035 | 0.013343 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.511101 | 0.013343 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  8.760224 | 6.307361 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.047795 | 0.013276 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.463546 | 0.013276 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  9.932059 | 7.151082 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.047556 |  0.01321 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.416228 |  0.01321 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  5.433749 | 3.912299 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.047318 | 0.013144 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.369147 | 0.013144 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.022532 | 5.056223 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.047081 | 0.013078 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.322301 | 0.013078 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.841476 | 5.645863 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.046846 | 0.013013 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.27569 | 0.013013 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |  0.2808 |  2.535001 | 1.774662 | 0.0 |  0.2808 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.046612 | 0.012948 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.229311 | 0.012948 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |     0.0 |  1.939624 | 1.396529 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.046379 | 0.012883 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.183165 | 0.012883 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |     0.0 |  1.204763 | 0.867429 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.046147 | 0.012819 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.137249 | 0.012819 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     1.6848      2.5272 | 0.901953 |        1.0 |  1.8252 |  0.796045 | 0.477532 | 0.0 |  1.8252 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.045916 | 0.012754 | 0.0 |     1.6848      2.5272 |       0.0        0.0 | 0.0 |       0.0 |  9.091563 | 0.012754 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    7.97472    11.96208 | 0.901953 |        1.0 |  7.8624 |  0.237329 | 0.077844 | 0.0 |  7.8624 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045458 |  0.045458 |  0.045686 |  0.045686 | 0.012691 | 0.0 |    7.97472    11.96208 |       0.0        0.0 | 0.0 |       0.0 |  9.046105 | 0.012691 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     11.232      16.848 | 0.901953 |        1.0 |  4.0716 |   0.10539 | 0.050502 | 0.0 |  4.0716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045231 |  0.045231 |  0.045458 |  0.045458 | 0.012627 | 0.0 |     11.232      16.848 |       0.0        0.0 | 0.0 |       0.0 |  9.000874 | 0.012627 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   16.73568    25.10352 | 0.901953 |        1.0 |  6.8796 |  0.104361 | 0.037765 | 0.0 |  6.8796 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045004 |  0.045004 |  0.045231 |  0.045231 | 0.012564 | 0.0 |   16.73568    25.10352 |       0.0        0.0 | 0.0 |       0.0 |   8.95587 | 0.012564 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    28.6416     42.9624 | 0.901953 |        1.0 | 14.8824 |  0.104016 | 0.016908 | 0.0 | 14.8824 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044779 |  0.044779 |  0.045005 |  0.045005 | 0.012501 | 0.0 |    28.6416     42.9624 |       0.0        0.0 | 0.0 |       0.0 |  8.911091 | 0.012501 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   28.75392    43.13088 | 0.901953 |        1.0 |  0.1404 |  0.103322 | 0.073355 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044555 |  0.044555 |   0.04478 |   0.04478 | 0.012439 | 0.0 |   28.75392    43.13088 |       0.0        0.0 | 0.0 |       0.0 |  8.866535 | 0.012439 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   29.54016    44.31024 | 0.902305 |        1.0 |  0.9828 |  0.103322 | 0.067429 | 0.0 |  0.9828 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044333 |  0.044333 |  0.044556 |  0.044556 | 0.012377 | 0.0 |   29.54016    44.31024 |       0.0        0.0 | 0.0 |       0.0 |  8.822202 | 0.012377 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   32.90976    49.36464 | 0.902305 |        1.0 |   4.212 |  0.103322 | 0.048821 | 0.0 |   4.212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044111 |  0.044111 |  0.044333 |  0.044333 | 0.012315 | 0.0 |   32.90976    49.36464 |       0.0        0.0 | 0.0 |       0.0 |  8.778091 | 0.012315 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   35.26848    52.90272 | 0.902305 |        1.0 |  2.9484 |    0.1028 | 0.055116 | 0.0 |  2.9484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04389 |   0.04389 |  0.044111 |  0.044111 | 0.012253 | 0.0 |   35.26848    52.90272 |       0.0        0.0 | 0.0 |       0.0 |  8.734201 | 0.012253 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   46.94976    70.42464 | 0.902305 |        1.0 | 14.6016 |  0.102451 | 0.017128 | 0.0 | 14.6016 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043671 |  0.043671 |  0.043891 |  0.043891 | 0.012192 | 0.0 |   46.94976    70.42464 |       0.0        0.0 | 0.0 |       0.0 |   8.69053 | 0.012192 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   50.88096    76.32144 | 0.902305 |        1.0 |   4.914 |    0.1021 | 0.044973 | 0.0 |   4.914 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043453 |  0.043453 |  0.043671 |  0.043671 | 0.012131 | 0.0 |   50.88096    76.32144 |       0.0        0.0 | 0.0 |       0.0 |  8.647077 | 0.012131 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   54.69984    82.04976 | 0.902305 |        1.0 |  4.7736 |  0.101749 | 0.045452 | 0.0 |  4.7736 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043235 |  0.043235 |  0.043453 |  0.043453 |  0.01207 | 0.0 |   54.69984    82.04976 |       0.0        0.0 | 0.0 |       0.0 |  8.603842 |  0.01207 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.04768    84.07152 | 0.902305 |        1.0 |  1.6848 |  0.198536 | 0.120782 | 0.0 |  1.6848 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043019 |  0.043019 |  0.043236 |  0.043236 |  0.01201 | 0.0 |   56.04768    84.07152 |       0.0        0.0 | 0.0 |       0.0 |  8.560823 |  0.01201 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |  0.1404 |  1.340489 | 0.951696 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042804 |  0.042804 |  0.043019 |  0.043019 |  0.01195 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.518019 |  0.01195 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |     0.0 |   3.22827 | 2.324355 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04259 |   0.04259 |  0.042804 |  0.042804 |  0.01189 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.475429 |  0.01189 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |     0.0 |  2.046603 | 1.473555 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042377 |  0.042377 |   0.04259 |   0.04259 | 0.011831 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.433051 | 0.011831 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.60928    84.91392 | 0.902305 |        1.0 |  0.5616 |  2.608661 | 1.775661 | 0.0 |  0.5616 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042165 |  0.042165 |  0.042377 |  0.042377 | 0.011771 | 0.0 |   56.60928    84.91392 |       0.0        0.0 | 0.0 |       0.0 |  8.390886 | 0.011771 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    56.7216     85.0824 | 0.902305 |        1.0 |  0.1404 |  6.588902 | 4.677869 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041954 |  0.041954 |  0.042165 |  0.042165 | 0.011713 | 0.0 |    56.7216     85.0824 |       0.0        0.0 | 0.0 |       0.0 |  8.348932 | 0.011713 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   60.76512    91.14768 | 0.902305 |        1.0 |  5.0544 |   3.57727 | 1.553726 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041745 |  0.041745 |  0.041955 |  0.041955 | 0.011654 | 0.0 |   60.76512    91.14768 |       0.0        0.0 | 0.0 |       0.0 |  8.307187 | 0.011654 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     67.392     101.088 | 0.902305 |        1.0 |  8.2836 |  9.350848 | 2.940569 | 0.0 |  8.2836 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041536 |  0.041536 |  0.041745 |  0.041745 | 0.011596 | 0.0 |     67.392     101.088 |       0.0        0.0 | 0.0 |       0.0 |  8.265651 | 0.011596 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   68.62752   102.94128 | 0.902305 |        1.0 |  1.5444 |  4.288787 | 2.646028 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041328 |  0.041328 |  0.041536 |  0.041536 | 0.011538 | 0.0 |   68.62752   102.94128 |       0.0        0.0 | 0.0 |       0.0 |  8.224323 | 0.011538 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   91.87776   137.81664 | 0.902305 |        1.0 | 29.0628 |  2.237402 | 0.088084 | 0.0 | 29.0628 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041122 |  0.041122 |  0.041328 |  0.041328 |  0.01148 | 0.0 |   91.87776   137.81664 |       0.0        0.0 | 0.0 |       0.0 |  8.183201 |  0.01148 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  134.44704   201.67056 | 0.902305 |        1.0 | 53.2116 |  2.373612 | 0.008352 | 0.0 | 53.2116 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040916 |  0.040916 |  0.041122 |  0.041122 | 0.011423 | 0.0 |  134.44704   201.67056 |       0.0        0.0 | 0.0 |       0.0 |  8.142285 | 0.011423 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  143.65728   215.48592 | 0.902305 |        1.0 | 11.5128 |  3.050519 | 0.694563 | 0.0 | 11.5128 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040711 |  0.040711 |  0.040916 |  0.040916 | 0.011366 | 0.0 |  143.65728   215.48592 |       0.0        0.0 | 0.0 |       0.0 |  8.101574 | 0.011366 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   147.7008    221.5512 | 0.902305 |        1.0 |  5.0544 |  1.686939 | 0.732693 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040508 |  0.040508 |  0.040712 |  0.040712 | 0.011309 | 0.0 |   147.7008    221.5512 |       0.0        0.0 | 0.0 |       0.0 |  8.061066 | 0.011309 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   156.1248    234.1872 | 0.902305 |        1.0 |   10.53 |    0.3091 | 0.077646 | 0.0 |   10.53 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040305 |  0.040305 |  0.040508 |  0.040508 | 0.011252 | 0.0 |   156.1248    234.1872 |       0.0        0.0 | 0.0 |       0.0 |  8.020761 | 0.011252 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    176.904     265.356 | 0.902305 |        1.0 |  25.974 |  0.098179 | 0.005264 | 0.0 |  25.974 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040104 |  0.040104 |  0.040306 |  0.040306 | 0.011196 | 0.0 |    176.904     265.356 |       0.0        0.0 | 0.0 |       0.0 |  7.980657 | 0.011196 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  194.20128   291.30192 | 0.902305 |        1.0 | 21.6216 |  0.097998 |  0.00812 | 0.0 | 21.6216 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039903 |  0.039903 |  0.040104 |  0.040104 |  0.01114 | 0.0 |  194.20128   291.30192 |       0.0        0.0 | 0.0 |       0.0 |  7.940753 |  0.01114 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  201.27744   301.91616 | 0.902305 |        1.0 |  8.8452 |  0.096723 | 0.028755 | 0.0 |  8.8452 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039704 |  0.039704 |  0.039904 |  0.039904 | 0.011084 | 0.0 |  201.27744   301.91616 |       0.0        0.0 | 0.0 |       0.0 |   7.90105 | 0.011084 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  203.41152   305.11728 | 0.902305 |        1.0 |  2.6676 |  0.095064 |  0.05242 | 0.0 |  2.6676 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039505 |  0.039505 |  0.039704 |  0.039704 | 0.011029 | 0.0 |  203.41152   305.11728 |       0.0        0.0 | 0.0 |       0.0 |  7.861544 | 0.011029 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  208.06848   312.10272 | 0.902686 |        1.0 |  5.8212 |  0.095249 | 0.038316 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039308 |  0.039308 |  0.039505 |  0.039505 | 0.010974 | 0.0 | 194.565197  296.270957 | 13.503283  15.831763 | 0.0 |       0.0 |  7.822237 | 0.010974 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  210.63456   315.95184 | 0.902686 |        1.0 |  3.2076 |  0.095064 | 0.049664 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039111 |  0.039111 |  0.039308 |  0.039308 | 0.010919 | 0.0 | 185.718874  287.424634 | 24.915686  28.527206 | 0.0 |       0.0 |  7.783126 | 0.010919 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.10976   316.66464 | 0.902686 |        1.0 |   0.594 |  0.094133 | 0.063867 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038916 |  0.038916 |  0.039111 |  0.039111 | 0.010864 | 0.0 | 176.872552  278.578312 | 34.237208  38.086328 | 0.0 |       0.0 |   7.74421 | 0.010864 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.29984   316.94976 | 0.902686 |        1.0 |  0.2376 |  0.094506 | 0.066447 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038721 |  0.038721 |  0.038916 |  0.038916 |  0.01081 | 0.0 | 168.026229  269.731989 | 43.273611  47.217771 | 0.0 |       0.0 |  7.705489 |  0.01081 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.77504   317.66256 | 0.902686 |        1.0 |   0.594 |  0.095619 | 0.064876 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038527 |  0.038527 |  0.038721 |  0.038721 | 0.010756 | 0.0 | 159.179906  260.885666 | 52.595134  56.776894 | 0.0 |       0.0 |  7.666961 | 0.010756 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 210.467016     321.084 | 0.902686 |        1.0 |  2.8512 |  0.097089 | 0.052562 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.794492 |  1.794492 | 0.0 | 0.0 | 0.0 |  1.794492 |  0.5 |  0.000553 | 0.0 | 0.040835 |  0.041388 |  0.039206 |  0.039206 | 0.010891 | 0.0 | 150.333583  252.039343 | 60.133433  69.044657 | 0.0 |  1.293939 |  8.126127 | 0.010891 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 198.082164   321.65424 | 0.902686 |        1.0 |  0.4752 |  0.164153 | 0.112705 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.382506 |  6.382506 | 0.0 | 0.0 | 0.0 |  6.382506 |  0.5 |   0.11148 | 0.0 | 0.043131 |  0.154611 |  0.065913 |  0.065913 | 0.018309 | 0.0 |  141.48726   243.19302 | 56.594904   78.46122 | 0.0 |  7.064965 |  8.582996 | 0.018309 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 185.697312   321.93936 | 0.902686 |        1.0 |  0.2376 |  0.498591 | 0.350557 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.287466 |  6.287466 | 0.0 | 0.0 | 0.0 |  6.287466 |  0.5 |  0.949881 | 0.0 | 0.045415 |  0.995296 |  0.316269 |  0.316269 | 0.087853 | 0.0 | 132.640937  234.346697 | 53.056375  87.592663 | 0.0 |  11.90255 |  9.037581 | 0.087853 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  173.31246  315.700524 | 0.902686 |        1.0 |     0.0 |   1.24415 | 0.895788 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  9.311844 |  9.311844 | 0.0 | 0.0 | 0.0 |  9.311844 |  0.5 |  3.446062 | 0.0 | 0.047688 |   3.49375 |  1.363689 |  1.363689 | 0.378803 | 0.0 | 123.794615  225.500375 | 49.517846   90.20015 | 0.0 | 17.268331 |  9.489893 | 0.378803 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 160.927608  303.315672 | 0.902686 |        1.0 |     0.0 |  4.454434 | 3.207192 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 |  7.825683 | 0.0 | 0.049949 |  7.875633 |   3.91229 |   3.91229 | 1.086747 | 0.0 | 114.948292  216.654052 | 45.979317  86.661621 | 0.0 |   21.3275 |  9.939944 | 1.086747 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 148.542756   290.93082 | 0.902686 |        1.0 |  0.3564 |  4.621298 | 3.210837 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 |  10.98364 | 0.0 |   0.0522 |  11.03584 |  7.604149 |  7.604149 | 2.112264 | 0.0 | 106.101969  207.807729 | 42.440788  83.123092 | 0.0 | 22.585111 | 10.387744 | 2.112264 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 136.157904  278.545968 | 0.902686 |        1.0 |  3.0888 |  5.624098 | 2.973312 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.473652 | 15.473652 | 0.0 | 0.0 | 0.0 | 15.473652 |  0.5 | 13.386091 | 0.0 | 0.054439 |  13.44053 | 10.867947 | 10.867947 | 3.018874 | 0.0 |  97.255646  198.961406 | 38.902258  79.584562 | 0.0 | 24.172672 | 10.833305 | 3.018874 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 123.773052  266.161116 | 0.902686 |        1.0 |  0.8316 |  5.698589 | 3.775582 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.216452 | 13.216452 | 0.0 | 0.0 | 0.0 | 13.216452 |  0.5 | 13.496721 | 0.0 | 0.056667 | 13.553388 | 12.931234 | 12.931234 | 3.592009 | 0.0 |  88.409323  190.115083 | 35.363729  76.046033 | 0.0 | 23.392403 | 11.276639 | 3.592009 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 111.388201  253.776265 | 0.902686 |        1.0 |  0.3564 |   4.93845 | 3.431191 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 | 12.616985 | 0.0 | 0.058883 | 12.675868 | 13.333304 | 13.333304 | 3.703696 | 0.0 |     79.563   181.26876 |   31.8252  72.507504 | 0.0 |  23.01667 | 11.717755 | 3.703696 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  99.003349  241.391413 | 0.902686 |        1.0 |  0.3564 |   8.25697 | 5.736869 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 | 12.315165 | 0.0 | 0.061089 | 12.376254 | 12.804292 | 12.804292 | 3.556748 | 0.0 |  70.716678  172.422438 | 28.286671  68.968975 | 0.0 | 22.942756 | 12.156667 | 3.556748 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  86.618497  229.006561 | 0.902686 |        1.0 |     0.0 |  5.142661 | 3.702716 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 12.076198 | 0.0 | 0.063283 | 12.139482 | 12.390219 | 12.390219 | 3.441728 | 0.0 |  61.870355  163.576115 | 24.748142  65.430446 | 0.0 |  22.75141 | 12.593383 | 3.441728 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  74.233645  216.621709 | 0.902686 |        1.0 |     0.0 |  5.625335 | 4.050241 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.924071 | 0.0 | 0.065467 | 11.989538 | 12.158777 | 12.158777 | 3.377438 | 0.0 |  53.024032  154.729792 | 21.209613  61.891917 | 0.0 | 22.712191 | 13.027916 | 3.377438 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  61.848793  204.236857 | 0.902686 |        1.0 |     0.0 |  3.483198 | 2.507903 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.892956 | 0.0 |  0.06764 | 11.960595 | 12.016427 | 12.016427 | 3.337896 | 0.0 |  44.177709  145.883469 | 17.671084  58.353388 | 0.0 | 22.704088 | 13.460277 | 3.337896 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  49.463941  191.852005 | 0.902686 |        1.0 |     0.0 |  1.596589 | 1.149544 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.886529 | 0.0 | 0.069801 | 11.956331 | 11.966079 | 11.966079 | 3.323911 | 0.0 |  35.331386  137.037146 | 14.132554  54.814858 | 0.0 |  22.70241 | 13.890476 | 3.323911 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  37.079089  179.467153 | 0.902686 |        1.0 |     0.0 |  0.291248 | 0.209698 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.885199 | 0.0 | 0.071952 | 11.957152 | 11.957461 | 11.957461 | 3.321517 | 0.0 |  26.485063  128.190823 | 10.594025  51.276329 | 0.0 | 22.702063 | 14.318523 | 3.321517 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  24.694237  167.082301 | 0.902686 |        1.0 |     0.0 |  0.092819 |  0.06683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884924 | 0.0 | 0.074093 | 11.959016 | 11.957384 | 11.957384 | 3.321495 | 0.0 |  17.638741  119.344501 |  7.055496    47.7378 | 0.0 | 22.701991 | 14.744431 | 3.321495 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  12.309385  154.697449 | 0.902686 |        1.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884867 | 0.0 | 0.076222 | 11.961089 | 11.959063 | 11.959063 | 3.321962 | 0.0 |   8.792418  110.498178 |  3.516967  44.199271 | 0.0 | 22.701976 | 15.168208 | 3.321962 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  142.312597 | 0.902686 |        1.0 |     0.0 |  0.089376 | 0.064351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 8.792418  8.846323 | 0.0   0.0 | 21.193441 | 21.193441 | 0.0 | 0.0 | 0.0 | 21.193441 |  0.5 | 16.689124 | 0.0 | 0.078341 | 16.767465 | 13.028712 | 13.028712 | 3.619087 | 0.0 |        0.0  101.651855 |       0.0  40.660742 | 0.0 | 26.706293 | 15.589867 | 3.619087 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.927745 | 0.902686 |        1.0 |     0.0 |  0.088208 |  0.06351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 |      0.0  8.846323 | 0.0   0.0 | 15.038749 | 15.038749 | 0.0 | 0.0 | 0.0 | 15.038749 |  0.5 |  16.61309 | 0.0 | 0.080449 |  16.69354 | 15.682954 | 15.682954 | 4.356376 | 0.0 |        0.0   92.805532 |       0.0  37.122213 | 0.0 | 24.631952 | 16.009418 | 4.356376 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  117.537274 | 0.903096 |        1.0 |     0.0 |  0.087226 | 0.062803 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.852812 | 0.0 | 0.082547 | 14.935359 | 16.319261 | 16.319261 | 4.533128 | 0.0 |        0.0   83.955196 |       0.0  33.582078 | 0.0 | 24.324711 | 16.426871 | 4.533128 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  105.146803 | 0.903096 |        1.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.595421 | 0.0 | 0.084634 | 14.680056 | 15.269332 | 15.269332 | 4.241481 | 0.0 |        0.0    75.10486 |       0.0  30.041944 | 0.0 | 24.274862 | 16.842237 | 4.241481 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.756333 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.553761 | 0.0 | 0.086711 | 14.640472 | 14.727994 | 14.727994 | 4.091109 | 0.0 |        0.0   66.254523 |       0.0  26.501809 | 0.0 | 24.266672 | 17.255525 | 4.091109 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   80.365862 | 0.903096 |        1.0 |  1.5444 |  0.088599 | 0.054662 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.589972 | 16.589972 | 0.0 | 0.0 | 0.0 | 16.589972 |  0.5 | 15.394071 | 0.0 | 0.088778 | 15.482849 | 14.836463 | 14.836463 |  4.12124 | 0.0 |        0.0   57.404187 |       0.0  22.961675 | 0.0 | 24.962573 | 17.666748 |  4.12124 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.975391 | 0.903096 |        1.0 |     0.0 |  0.088404 | 0.063651 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 15.130965 | 0.0 | 0.090834 | 15.221798 | 15.237643 | 15.237643 | 4.232679 | 0.0 |        0.0   48.553851 |       0.0   19.42154 | 0.0 |  24.37718 | 18.075914 | 4.232679 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    55.58492 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.639301 | 0.0 |  0.09288 | 14.732181 | 15.171006 | 15.171006 | 4.214168 | 0.0 |        0.0   39.703515 |       0.0  15.881406 | 0.0 |  24.28345 | 18.483034 | 4.214168 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    43.19445 | 0.903096 |        1.0 |     0.0 |  0.328843 | 0.236767 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.560936 | 0.0 | 0.094915 | 14.655852 | 14.824023 | 14.824023 | 4.117784 | 0.0 |        0.0   30.853178 |       0.0  12.341271 | 0.0 | 24.268085 | 18.888119 | 4.117784 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.803979 | 0.903096 |        1.0 |  0.8316 |  1.475475 | 0.977572 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.877172 | 15.877172 | 0.0 | 0.0 | 0.0 | 15.877172 |  0.5 | 15.001429 | 0.0 | 0.096941 |  15.09837 | 14.771151 | 14.771151 | 4.103098 | 0.0 |        0.0   22.002842 |       0.0   8.801137 | 0.0 | 24.643828 | 19.291179 | 4.103098 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   18.413508 | 0.903096 |        1.0 |  0.4752 |  3.387629 | 2.325898 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.520772 | 15.520772 | 0.0 | 0.0 | 0.0 | 15.520772 |  0.5 | 15.122306 | 0.0 | 0.098956 | 15.221262 | 15.027342 | 15.027342 | 4.174262 | 0.0 |        0.0   13.152506 |       0.0   5.261002 | 0.0 | 24.542293 | 19.692223 | 4.174262 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    6.023037 | 0.903096 |        1.0 |  0.1188 |   5.50315 | 3.915475 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.164372 | 15.164372 | 0.0 | 0.0 | 0.0 | 15.164372 |  0.5 | 14.842181 | 0.0 | 0.100961 | 14.943142 | 15.132148 | 15.132148 | 4.203374 | 0.0 |        0.0     4.30217 |       0.0   1.720868 | 0.0 | 24.364484 | 20.091262 | 4.203374 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |  0.4752 |  7.584338 | 5.207299 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0   4.30217 | 0.0   0.0 | 21.187391 | 21.187391 | 0.0 | 0.0 | 0.0 | 21.187391 |  0.5 | 18.130219 | 0.0 | 0.102956 | 18.233175 | 15.736065 | 15.736065 | 4.371129 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 26.921657 | 20.488305 | 4.371129 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  8.894918 | 6.404341 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 18.318022 | 0.0 | 0.104942 | 18.422964 | 17.544232 | 17.544232 | 4.873398 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.804307 | 20.883364 | 4.873398 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  8.894763 | 6.404229 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.341911 | 0.0 | 0.106917 | 17.448828 | 18.164314 | 18.164314 | 5.045643 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.663069 | 21.276447 | 5.045643 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 | 11.358152 |  8.17787 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.219304 | 0.0 | 0.108882 | 17.328187 | 17.638493 | 17.638493 | 4.899581 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.644437 | 21.667565 | 4.899581 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  9.946398 | 7.161406 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.203144 | 0.0 | 0.110838 | 17.313982 | 17.351839 | 17.351839 | 4.819955 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641966 | 22.056727 | 4.819955 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |   8.33363 | 6.000214 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.201001 | 0.0 | 0.112784 | 17.313784 | 17.317094 | 17.317094 | 4.810304 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641638 | 22.443943 | 4.810304 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  6.676229 | 4.806885 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200716 | 0.0 |  0.11472 | 17.315436 | 17.314195 | 17.314195 | 4.809499 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641594 | 22.829224 | 4.809499 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  3.125321 | 2.250231 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200678 | 0.0 | 0.116646 | 17.317324 | 17.315488 | 17.315488 | 4.809858 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641588 | 23.212577 | 4.809858 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  1.387786 | 0.999206 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.118563 | 17.319236 |  17.31733 |  17.31733 | 4.810369 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.594015 | 4.810369 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.349323 | 0.251513 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 |  0.12047 | 17.321143 | 17.319235 | 17.319235 | 4.810899 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.973545 | 4.810899 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.094878 | 0.068313 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.122368 |  17.32304 | 17.321141 | 17.321141 | 4.811428 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.351177 | 4.811428 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.124256 | 17.324928 | 17.323038 | 17.323038 | 4.811955 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.726921 | 4.811955 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.126135 | 17.326807 | 17.324926 | 17.324926 |  4.81248 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.100786 |  4.81248 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.128004 | 17.328676 | 17.326805 | 17.326805 | 4.813001 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.472782 | 4.813001 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_multiple_zones:

multiple zones
______________

All of the above examples deal with a subbasin consisting of a single zone.  Hence, none
of them is suitable to check if, for instance, the aggregation of fluxes addressing
different spatial extents does not introduce errors into the water balance.  The
following example fills this gap.  We restore the initial
:ref:`field example <hland_v1_field>` settings, except defining five zones of different
land-use types and sizes.  The variations between the responses of the land-use
types are as to be expected:

.. integration-test::

    >>> ttice(0.0)
    >>> resparea(False)
    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> inputs.t.series = t_series
    >>> inputs.tn.series = tn_series
    >>> test("hland_v1_multiple_zones")
    |        date |    p |    t |   tn |       epn | tmean |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact | contriarea |                                          pc |                                                    ep |                                              epc |                                     ei |                                                tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                 in_ |                                                   r |                            sr |                                ea |                                cf |      inuz | perc |        q0 |                           el |       q1 |      inuh |     outuh |        rt |        qt |                                     ic |                      sp |                      wc |                                    sm |        uz |        lz |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.11682    0.11682    0.11682    0.11682    0.11682 |  0.08411   0.08411   0.08411   0.08411   0.08411 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624      0.0        0.0 |       0.0        0.0  18.760624      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.052569  0.052569  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.690156 |  0.5 |  0.018623 | 0.0  0.0  0.0   0.08411  0.0 | 0.052083 |  0.063509 |  0.064113 |  0.064113 |  0.267138 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.947431   99.947431  0.0  0.0  0.0 |  4.171533 | 10.364473 |  0.267138 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.113449   0.113449   0.113449   0.113449   0.113449 | 0.081683  0.081683  0.081683  0.081683  0.081683 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109      0.0        0.0 |       0.0        0.0  17.137109      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.051051  0.051051  0.0  0.0  0.0 | 0.050026  0.050026  0.0  0.0  0.0 |  4.246757 |  0.5 |   0.23242 | 0.0  0.0  0.0  0.081683  0.0 | 0.053907 |  0.236249 |  0.137783 |  0.137783 |  0.574094 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.946406   99.946406  0.0  0.0  0.0 |   7.68587 | 10.727468 |  0.574094 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050622  0.050622  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |  4.134013 |  0.5 |  0.794235 | 0.0  0.0  0.0  0.080999  0.0 | 0.055722 |  0.687395 |  0.298117 |  0.298117 |  1.242155 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945811   99.945811  0.0  0.0  0.0 | 10.525649 | 11.088746 |  1.242155 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.111348   0.111348   0.111348   0.111348   0.111348 | 0.080171  0.080171  0.080171  0.080171  0.080171 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496      0.0        0.0 |       0.0        0.0   16.14496      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050105  0.050105  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |   3.99872 |  0.5 |  1.564786 | 0.0  0.0  0.0  0.080171  0.0 | 0.057529 |  1.305523 |  0.724502 |  0.724502 |   3.01876 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945733   99.945733  0.0  0.0  0.0 | 12.459582 | 11.448335 |   3.01876 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050622  0.050622  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |  4.134013 |  0.5 |  2.310317 | 0.0  0.0  0.0  0.080999  0.0 | 0.059327 |  1.903625 |  1.301073 |  1.301073 |  5.421137 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945139   99.945139  0.0  0.0  0.0 | 13.783278 | 11.806009 |  5.421137 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.119203   0.119203   0.119203   0.119203   0.119203 | 0.085826  0.085826  0.085826  0.085826  0.085826 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163      0.0        0.0 |       0.0        0.0  19.933163      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.053639  0.053639  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |   4.94577 |  0.5 |  3.097316 | 0.0  0.0  0.0  0.085826  0.0 | 0.061112 |   2.53489 |  1.910995 |  1.910995 |  7.962478 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.941527   99.941527  0.0  0.0  0.0 | 15.131732 | 12.161208 |  7.962478 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.338613   0.338613   0.338613   0.338613   0.338613 | 0.243802  0.243802  0.243802  0.243802  0.243802 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241      0.0        0.0 |       0.0        0.0  22.278241      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.152363  0.152363  0.0  0.0  0.0 | 0.050029  0.050029  0.0  0.0  0.0 |  5.532038 |  0.5 |  3.941177 | 0.0  0.0  0.0  0.243802  0.0 | 0.062775 |  3.211531 |  2.544974 |  2.544974 | 10.604058 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.839193   99.839193  0.0  0.0  0.0 | 16.222594 | 12.492176 | 10.604058 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.242135   2.242135   2.242135   2.242135   2.242135 | 1.614337  1.614337  1.614337  1.614337  1.614337 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491      0.0        0.0 |       0.0        0.0  25.164491      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.007844  1.007844  0.0  0.0  0.0 |  0.05008   0.05008  0.0  0.0  0.0 |  6.253562 |  0.5 |  4.774207 | 0.0  0.0  0.0  1.614337  0.0 | 0.063451 |  3.878586 |  3.209401 |  3.209401 | 13.372504 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   98.88143    98.88143  0.0  0.0  0.0 | 17.201949 | 12.626677 | 13.372504 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.068662   5.068662   5.068662   5.068662   5.068662 | 3.649436  3.649436  3.649436  3.649436  3.649436 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515      0.0        0.0 |       0.0        0.0  24.713515      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.256538  2.256538  0.0  0.0  0.0 | 0.050559  0.050559  0.0  0.0  0.0 |  6.140459 |  0.5 |  5.310752 | 0.0  0.0  0.0  3.649436  0.0 |  0.06267 |  4.307093 |  3.825576 |  3.825576 | 15.939899 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.675452   96.675452  0.0  0.0  0.0 | 17.531656 | 12.471231 | 15.939899 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  8.488365   8.488365   8.488365   8.488365   8.488365 | 6.111623  6.111623  6.111623  6.111623  6.111623 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546      0.0        0.0 |       0.0        0.0  27.960546      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.694748  3.694748  0.0  0.0  0.0 | 0.051662  0.051662  0.0  0.0  0.0 |   6.95139 |  0.5 |  5.803413 | 0.0  0.0  0.0  6.111623  0.0 | 0.060134 |  4.698855 |  4.298928 |  4.298928 | 17.912199 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  93.032367   93.032367  0.0  0.0  0.0 | 18.179633 |  11.96658 | 17.912199 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   9.58666    9.58666    9.58666    9.58666    9.58666 | 6.902395  6.902395  6.902395  6.902395  6.902395 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108      0.0        0.0 |       0.0        0.0  28.682108      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.015721  4.015721  0.0  0.0  0.0 | 0.053484  0.053484  0.0  0.0  0.0 |  7.130414 |  0.5 |  6.279869 | 0.0  0.0  0.0  6.902395  0.0 | 0.057045 |  5.077138 |   4.69586 |   4.69586 | 19.566082 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   89.07013    89.07013  0.0  0.0  0.0 | 18.530178 | 11.352049 | 19.566082 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 10.869046  10.869046  10.869046  10.869046  10.869046 | 7.825713  7.825713  7.825713  7.825713  7.825713 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967      0.0        0.0 |       0.0        0.0  31.387967      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.359196  4.359196  0.0  0.0  0.0 | 0.055465  0.055465  0.0  0.0  0.0 |  7.805393 |  0.5 |  6.769495 | 0.0  0.0  0.0  7.825713  0.0 | 0.053313 |  5.465355 |  5.079345 |  5.079345 | 21.163939 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  84.766399   84.766399  0.0  0.0  0.0 | 19.066076 | 10.609348 | 21.163939 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.946366   5.946366   5.946366   5.946366   5.946366 | 4.281384  4.281384  4.281384  4.281384  4.281384 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678      0.0        0.0 |       0.0        0.0  33.101678      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.269776  2.269776  0.0  0.0  0.0 | 0.057617  0.057617  0.0  0.0  0.0 |  8.232207 |  0.5 |  7.299567 | 0.0  0.0  0.0  4.281384  0.0 | 0.052131 |   5.88831 |  5.473075 |  5.473075 | 22.804477 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   82.55424    82.55424  0.0  0.0  0.0 | 19.498715 | 10.374162 | 22.804477 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  7.685035   7.685035   7.685035   7.685035   7.685035 | 5.533225  5.533225  5.533225  5.533225  5.533225 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155      0.0        0.0 |       0.0        0.0  27.780155      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.856976  2.856976  0.0  0.0  0.0 | 0.058723  0.058723  0.0  0.0  0.0 |  6.900997 |  0.5 |  7.056489 | 0.0  0.0  0.0  5.533225  0.0 | 0.050061 |  5.691915 |  5.750677 |  5.750677 | 23.961152 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  79.755987   79.755987  0.0  0.0  0.0 | 18.843223 |  9.962211 | 23.961152 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  8.581238   8.581238   8.581238   8.581238   8.581238 | 6.178491  6.178491  6.178491  6.178491  6.178491 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679      0.0        0.0 |       0.0        0.0  21.556679      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.082145  3.082145  0.0  0.0  0.0 | 0.060122  0.060122  0.0  0.0  0.0 |  5.344078 |  0.5 |   6.03778 | 0.0  0.0  0.0  6.178491  0.0 | 0.047541 |  4.874596 |  5.553932 |  5.553932 | 23.141382 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.733964   76.733964  0.0  0.0  0.0 | 17.649521 |    9.4606 | 23.141382 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  2.774152   2.774152   2.774152   2.774152   2.774152 | 1.950491  1.950491  1.950491  1.950491  1.950491 |   0.2376    0.2376  0.0  0.0    0.2376 |       0.0        0.0   0.2376   0.2376        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.786427   0.2376        0.0 |       0.0        0.0  22.786427   0.2376        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.936182  0.936182  0.0  0.0  0.0 | 0.061633  0.061633  0.0  0.0  0.0 |  5.650382 |  0.5 |  5.408841 | 0.0  0.0  0.0  1.950491  0.0 | 0.048222 |  4.372081 |  4.944552 |  4.944552 | 20.602301 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.859415   75.859415  0.0  0.0  0.0 | 17.391062 |  9.596251 | 20.602301 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.122607   2.122607   2.122607   2.122607   2.122607 | 1.528277  1.528277  1.528277  1.528277  1.528277 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804      0.0        0.0 |       0.0        0.0  22.999804      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.725182  0.725182  0.0  0.0  0.0 |  0.06207   0.06207  0.0  0.0  0.0 |  5.703398 |  0.5 |  5.271782 | 0.0  0.0  0.0  1.528277  0.0 | 0.049032 |  4.263189 |  4.459553 |  4.459553 | 18.581469 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.196304   75.196304  0.0  0.0  0.0 | 17.322678 |  9.757464 | 18.581469 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   1.31842    1.31842    1.31842    1.31842    1.31842 | 0.949262  0.949262  0.949262  0.949262  0.949262 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507      0.0        0.0 |       0.0        0.0  21.015507      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.446502  0.446502  0.0  0.0  0.0 | 0.062402  0.062402  0.0  0.0  0.0 |  5.207075 |  0.5 |   5.06116 | 0.0  0.0  0.0  0.949262  0.0 | 0.050252 |   4.09583 |  4.250196 |  4.250196 | 17.709152 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  74.812204   74.812204  0.0  0.0  0.0 | 16.968593 | 10.000175 | 17.709152 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.871144   0.871144   0.871144   0.871144   0.871144 | 0.537465  0.537465  0.537465  0.537465  0.537465 | 0.537465  0.537465  0.0  0.0  0.537465 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.665805   1.5444        0.0 |       0.0        0.0  20.665805   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.125758  0.125758  0.0  0.0  0.0 | 0.062594  0.062594  0.0  0.0  0.0 |  5.119506 |  0.5 |  4.825479 | 0.0  0.0  0.0  0.537465  0.0 | 0.052863 |  3.909722 |  4.091664 |  4.091664 | 17.048599 | 1.006935  1.006935  0.0  0.0  1.006935 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   74.74904    74.74904  0.0  0.0  0.0 |  16.76262 | 10.519731 | 17.048599 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  6.6528   6.6528   6.6528   6.6528   6.6528 |  0.259718   0.259718   0.259718   0.259718   0.259718 | 0.096141  0.096141  0.096141  0.096141  0.096141 | 0.096141  0.096141  0.0  0.0  0.096141 |  5.659735   5.659735   6.6528   6.6528   5.659735 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.659735   5.659735  25.413424   6.6528   5.659735 |  0.790583   0.790583  25.413424   6.6528   5.659735 | 0.0  0.0  0.0  0.0   5.659735 | 0.023939  0.023939  0.0  0.0  0.0 | 0.060191  0.060191  0.0  0.0  0.0 |   6.90115 |  0.5 |  5.315038 | 0.0  0.0  0.0  0.096141  0.0 | 0.059425 |  4.684809 |  4.123321 |  4.123321 | 17.180504 | 1.903859  1.903859  0.0  0.0  1.903859 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  79.654445   79.654445  0.0  0.0  0.0 | 17.848732 | 11.825543 | 17.180504 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.4452   3.4452   3.4452   3.4452   3.4452 |  0.115333   0.115333   0.115333   0.115333   0.115333 | 0.058839  0.058839  0.058839  0.058839  0.058839 | 0.058839  0.058839  0.0  0.0  0.058839 |  3.349059   3.349059   3.4452   3.4452   3.349059 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.349059   3.349059  21.484262   3.4452   3.349059 |   0.53123    0.53123  21.484262   3.4452   3.349059 | 0.0  0.0  0.0  0.0   3.349059 | 0.015175  0.015175  0.0  0.0  0.0 | 0.058764  0.058764  0.0  0.0  0.0 |  5.725415 |  0.5 |  5.556446 | 0.0  0.0  0.0  0.058839  0.0 | 0.063689 |  4.727871 |  4.522137 |  4.522137 | 18.842238 | 1.941161  1.941161  0.0  0.0  1.941161 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  82.515863   82.515863  0.0  0.0  0.0 | 17.517701 | 12.674191 | 18.842238 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.114207   0.114207   0.114207   0.114207   0.114207 | 0.045942  0.045942  0.045942  0.045942  0.045942 | 0.045942  0.045942  0.0  0.0  0.045942 |  5.762361   5.762361   5.8212   5.8212   5.762361 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.762361   5.762361   23.31909   5.8212   5.762361 |  0.980879   0.980879   23.31909   5.8212   5.762361 | 0.0  0.0  0.0  0.0   5.762361 | 0.012541  0.012541  0.0  0.0  0.0 | 0.056351  0.056351  0.0  0.0  0.0 |  6.523168 |  0.5 |  5.638796 | 0.0  0.0  0.0  0.045942  0.0 | 0.069639 |   4.96019 |  4.769928 |  4.769928 | 19.874701 | 1.954058  1.954058  0.0  0.0  1.954058 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  87.341155   87.341155  0.0  0.0  0.0 | 17.902074 | 13.858161 | 19.874701 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 12.5928  12.5928  12.5928  12.5928  12.5928 |  0.113828   0.113828   0.113828   0.113828   0.113828 | 0.023264  0.023264  0.023264  0.023264  0.023264 | 0.023264  0.023264  0.0  0.0  0.023264 | 12.546858  12.546858  12.5928  12.5928  12.546858 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.546858  12.546858  29.910299  12.5928  12.546858 |  2.392836   2.392836  29.910299  12.5928  12.546858 | 0.0  0.0  0.0  0.0  12.546858 | 0.007092  0.007092  0.0  0.0  0.0 | 0.051252  0.051252  0.0  0.0  0.0 |  9.233762 |  0.5 |  6.924651 | 0.0  0.0  0.0  0.023264  0.0 | 0.080412 |  6.451229 |  5.239906 |  5.239906 |  21.83294 | 1.976736  1.976736  0.0  0.0  1.976736 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  97.539338   97.539338  0.0  0.0  0.0 | 19.711185 | 16.001968 |  21.83294 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.080449  0.080449  0.080449  0.080449  0.080449 | 0.080449  0.080449  0.0  0.0  0.080449 |  0.095536   0.095536   0.1188   0.1188   0.095536 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.095536   0.095536  17.075518   0.1188   0.095536 |  0.022723   0.022723  17.075518   0.1188   0.095536 | 0.0  0.0  0.0  0.0   0.095536 | 0.024553  0.024553  0.0  0.0  0.0 | 0.051194  0.051194  0.0  0.0  0.0 |  4.247526 |  0.5 |  6.190949 | 0.0  0.0  0.0  0.080449  0.0 |  0.08218 |  5.035829 |  5.805354 |  5.805354 | 24.188974 | 1.919551  1.919551  0.0  0.0  1.919551 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  97.638792   97.638792  0.0  0.0  0.0 | 17.267763 | 16.353838 | 24.188974 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.074914  0.074914  0.074914  0.074914  0.074914 | 0.074914  0.074914  0.0  0.0  0.074914 |  0.751151   0.751151   0.8316   0.8316   0.751151 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.751151   0.751151  17.794937   0.8316   0.751151 |  0.179024   0.179024  17.794937   0.8316   0.751151 | 0.0  0.0  0.0  0.0   0.751151 | 0.023004  0.023004  0.0  0.0  0.0 | 0.050895  0.050895  0.0  0.0  0.0 |  4.544832 |  0.5 |  4.808754 | 0.0  0.0  0.0  0.074914  0.0 | 0.084453 |  3.975902 |  5.114823 |  5.114823 | 21.311763 | 1.925086  1.925086  0.0  0.0  1.925086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   98.23881    98.23881  0.0  0.0  0.0 | 16.503841 | 16.806055 | 21.311763 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.564    3.564    3.564    3.564    3.564 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.057003  0.057003  0.057003  0.057003  0.057003 | 0.057003  0.057003  0.0  0.0  0.057003 |  3.489086   3.489086    3.564    3.564   3.489086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.489086   3.489086  20.527337    3.564   3.489086 |  0.841817   0.841817  20.527337    3.564   3.489086 | 0.0  0.0  0.0  0.0   3.489086 |  0.01798   0.01798  0.0  0.0  0.0 | 0.049557  0.049557  0.0  0.0  0.0 |  5.726029 |  0.5 |  4.759355 | 0.0  0.0  0.0  0.057003  0.0 | 0.088678 |  4.122856 |  4.244098 |  4.244098 | 17.683742 | 1.942997  1.942997  0.0  0.0  1.942997 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 100.917655  100.917655  0.0  0.0  0.0 | 16.970515 | 17.646948 | 17.683742 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.4948   2.4948   2.4948   2.4948   2.4948 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.063115  0.063115  0.063115  0.063115  0.063115 | 0.063115  0.063115  0.0  0.0  0.063115 |  2.437797   2.437797   2.4948   2.4948   2.437797 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.437797   2.437797  19.187445   2.4948   2.437797 |  0.620686   0.620686  19.187445   2.4948   2.437797 | 0.0  0.0  0.0  0.0   2.437797 | 0.020272  0.020272  0.0  0.0  0.0 | 0.048633  0.048633  0.0  0.0  0.0 |  5.225901 |  0.5 |  4.861905 | 0.0  0.0  0.0  0.063115  0.0 | 0.092115 |  4.138017 |  4.093569 |  4.093569 | 17.056537 | 1.936885  1.936885  0.0  0.0  1.936885 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 102.763127  102.763127  0.0  0.0  0.0 | 16.834511 | 18.330789 | 17.056537 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 12.3552  12.3552  12.3552  12.3552  12.3552 |  0.112116   0.112116   0.112116   0.112116   0.112116 | 0.023465  0.023465  0.023465  0.023465  0.023465 | 0.023465  0.023465  0.0  0.0  0.023465 | 12.292085  12.292085  12.3552  12.3552  12.292085 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.292085  12.292085  28.867384  12.3552  12.292085 |  3.245191   3.245191  28.867384  12.3552  12.292085 | 0.0  0.0  0.0  0.0  12.292085 | 0.008202  0.008202  0.0  0.0  0.0 | 0.044095  0.044095  0.0  0.0  0.0 |  9.617668 |  0.5 |  6.378841 | 0.0  0.0  0.0  0.023465  0.0 | 0.102605 |   6.01831 |  4.552491 |  4.552491 | 18.968713 | 1.976535  1.976535  0.0  0.0  1.976535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.845915  111.845915  0.0  0.0  0.0 | 19.573338 | 20.418431 | 18.968713 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   4.158    4.158    4.158    4.158    4.158 |  0.111732   0.111732   0.111732   0.111732   0.111732 |  0.05308   0.05308   0.05308   0.05308   0.05308 |  0.05308   0.05308  0.0  0.0   0.05308 |  4.134535   4.134535    4.158    4.158   4.134535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.134535   4.134535  20.489723    4.158   4.134535 |  1.293025   1.293025  20.489723    4.158   4.134535 | 0.0  0.0  0.0  0.0   4.134535 | 0.019031  0.019031  0.0  0.0  0.0 | 0.042656  0.042656  0.0  0.0  0.0 |  6.060207 |  0.5 |  6.777964 | 0.0  0.0  0.0   0.05308  0.0 | 0.107167 |   5.79803 |  5.551516 |  5.551516 | 23.131317 |  1.94692   1.94692  0.0  0.0   1.94692 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  114.71105   114.71105  0.0  0.0  0.0 | 18.355581 | 21.326253 | 23.131317 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.0392   4.0392   4.0392   4.0392   4.0392 |  0.111348   0.111348   0.111348   0.111348   0.111348 |  0.05353   0.05353   0.05353   0.05353   0.05353 |  0.05353   0.05353  0.0  0.0   0.05353 |   3.98612    3.98612   4.0392   4.0392    3.98612 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.98612    3.98612  20.190462   4.0392    3.98612 |  1.311296   1.311296  20.190462   4.0392    3.98612 | 0.0  0.0  0.0  0.0    3.98612 | 0.019643  0.019643  0.0  0.0  0.0 | 0.041307  0.041307  0.0  0.0  0.0 |  6.000108 |  0.5 |  5.969917 | 0.0  0.0  0.0   0.05353  0.0 | 0.111621 |  5.145855 |  5.702053 |  5.702053 | 23.758554 |  1.94647   1.94647  0.0  0.0   1.94647 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.407537  117.407537  0.0  0.0  0.0 | 17.885771 | 22.212584 | 23.758554 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.4256   1.4256   1.4256   1.4256   1.4256 |  0.217266   0.217266   0.217266   0.217266   0.217266 | 0.135647  0.135647  0.135647  0.135647  0.135647 | 0.135647  0.135647  0.0  0.0  0.135647 |   1.37207    1.37207   1.4256   1.4256    1.37207 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   1.37207    1.37207  17.757323   1.4256    1.37207 |  0.472834   0.472834  17.757323   1.4256    1.37207 | 0.0  0.0  0.0  0.0    1.37207 | 0.050167  0.050167  0.0  0.0  0.0 | 0.040847  0.040847  0.0  0.0  0.0 |  4.763321 |  0.5 |   5.24437 | 0.0  0.0  0.0  0.135647  0.0 | 0.114127 |  4.393486 |  5.123589 |  5.123589 | 21.348289 | 1.864353  1.864353  0.0  0.0  1.864353 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.297453  118.297453  0.0  0.0  0.0 | 16.904722 | 22.711308 | 21.348289 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   1.46695    1.46695    1.46695    1.46695    1.46695 |  1.04373   1.04373   1.04373   1.04373   1.04373 |  1.04373   1.04373  0.0  0.0   1.04373 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.721215   0.1188        0.0 |       0.0        0.0  16.721215   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.385979  0.385979  0.0  0.0  0.0 | 0.040851  0.040851  0.0  0.0  0.0 |  4.149665 |  0.5 |  4.475785 | 0.0  0.0  0.0   1.04373  0.0 | 0.115039 |  3.687997 |  4.403904 |  4.403904 |   18.3496 | 0.939422  0.939422  0.0  0.0  0.939422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.952325  117.952325  0.0  0.0  0.0 | 16.078602 | 22.892707 |   18.3496 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.532824   3.532824   3.532824   3.532824   3.532824 | 2.543633  2.543633  2.543633  2.543633  2.543633 | 0.939422  0.939422  0.0  0.0  0.939422 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415      0.0        0.0 |       0.0        0.0  16.602415      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.740017  1.740017  0.0  0.0  0.0 | 0.041024  0.041024  0.0  0.0  0.0 |  4.119836 |  0.5 |  4.015724 | 0.0  0.0  0.0  2.543633  0.0 |  0.11479 |  3.319716 |  3.762932 |  3.762932 | 15.678885 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.253332  116.253332  0.0  0.0  0.0 | 15.682714 | 22.843113 | 15.678885 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.239679   2.239679   2.239679   2.239679   2.239679 | 1.612569  1.612569  1.612569  1.612569  1.612569 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876      0.0        0.0 |       0.0        0.0  16.782876      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.172088  1.172088  0.0  0.0  0.0 | 0.041873  0.041873  0.0  0.0  0.0 |  4.164314 |  0.5 |  3.820377 | 0.0  0.0  0.0  1.612569  0.0 | 0.115207 |  3.163827 |  3.366914 |  3.366914 |  14.02881 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.123117  115.123117  0.0  0.0  0.0 | 15.526652 | 22.926111 |  14.02881 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  2.854761   2.854761   2.854761   2.854761   2.854761 | 1.960038  1.960038  1.960038  1.960038  1.960038 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.438537   0.4752        0.0 |       0.0        0.0  17.438537   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.410806  1.410806  0.0  0.0  0.0 | 0.042438  0.042438  0.0  0.0  0.0 |  4.327805 |  0.5 |  3.787265 | 0.0  0.0  0.0  1.960038  0.0 | 0.115713 |  3.137811 |  3.192688 |  3.192688 | 13.302866 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  113.75475   113.75475  0.0  0.0  0.0 | 15.567192 |  23.02685 | 13.302866 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  7.210496   7.210496   7.210496   7.210496   7.210496 | 5.130246  5.130246  5.130246  5.130246  5.130246 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.172367   0.1188        0.0 |       0.0        0.0  17.172367   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.64882   3.64882  0.0  0.0  0.0 | 0.043123  0.043123  0.0  0.0  0.0 |   4.26075 |  0.5 |  3.788625 | 0.0  0.0  0.0  5.130246  0.0 | 0.113698 |  3.137018 |  3.143416 |  3.143416 | 13.097566 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.149053  110.149053  0.0  0.0  0.0 | 15.539317 | 22.625803 | 13.097566 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  3.914748   3.914748   3.914748   3.914748   3.914748 | 1.837796  1.837796  1.837796  1.837796  1.837796 | 1.837796  1.837796  0.0  0.0  1.837796 |    2.2768     2.2768   4.2768   4.2768     2.2768 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    2.2768     2.2768  21.059676   4.2768     2.2768 |    0.6906     0.6906  21.059676   4.2768     2.2768 | 0.0  0.0  0.0  0.0     2.2768 | 0.641962  0.641962  0.0  0.0  0.0 | 0.044132  0.044132  0.0  0.0  0.0 |  5.749769 |  0.5 |  4.227527 | 0.0  0.0  0.0  1.837796  0.0 | 0.117014 |  3.643021 |  3.249639 |  3.249639 | 13.540162 | 0.162204  0.162204  0.0  0.0  0.162204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.137424  111.137424  0.0  0.0  0.0 |  16.56156 | 23.285789 | 13.540162 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.0092   7.0092   7.0092   7.0092   7.0092 | 10.233004  10.233004  10.233004  10.233004  10.233004 | 3.655358  3.655358  3.655358  3.655358  3.655358 |      2.0       2.0  0.0  0.0       2.0 |  5.171404   5.171404   7.0092   7.0092   5.171404 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.171404   5.171404  23.611615   7.0092   5.171404 |  1.596868   1.596868  23.611615   7.0092   5.171404 | 0.0  0.0  0.0  0.0   5.171404 |  2.13852   2.13852  0.0  0.0  0.0 | 0.042644  0.042644  0.0  0.0  0.0 |  7.068572 |  0.5 |  5.253883 | 0.0  0.0  0.0  3.655358  0.0 | 0.120967 |   4.66077 |  3.756742 |  3.756742 | 15.653092 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.616083  112.616083  0.0  0.0  0.0 | 17.876249 | 24.072514 | 15.653092 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.3068   1.3068   1.3068   1.3068   1.3068 |  4.693389   4.693389   4.693389   4.693389   4.693389 | 2.965278  2.965278  2.965278  2.965278  2.965278 |   1.3068    1.3068  0.0  0.0    1.3068 |       0.0        0.0   1.3068   1.3068        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.818984   1.3068        0.0 |       0.0        0.0  17.818984   1.3068        0.0 | 0.0  0.0  0.0  0.0        0.0 |   1.8732    1.8732  0.0  0.0  0.0 | 0.043692  0.043692  0.0  0.0  0.0 |  4.421977 |  0.5 |  5.123278 | 0.0  0.0  0.0  2.965278  0.0 | 0.121321 |  4.211855 |  4.334845 |  4.334845 | 18.061853 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.786575  110.786575  0.0  0.0  0.0 | 16.674948 | 24.142839 | 18.061853 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 24.5916  24.5916  24.5916  24.5916  24.5916 |  2.448478   2.448478   2.448478   2.448478   2.448478 |  0.15074   0.15074   0.15074   0.15074   0.15074 |  0.15074   0.15074  0.0  0.0   0.15074 |   22.5916    22.5916  24.5916  24.5916    22.5916 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   22.5916    22.5916   40.29171  24.5916    22.5916 |  6.932043   6.932043   40.29171  24.5916    22.5916 | 0.0  0.0  0.0  0.0    22.5916 | 0.059581  0.059581  0.0  0.0  0.0 | 0.036777  0.036777  0.0  0.0  0.0 | 15.244377 |  0.5 |  8.705784 | 0.0  0.0  0.0   0.15074  0.0 | 0.140315 |  8.601694 |  5.287134 |  5.287134 | 22.029725 |  1.84926   1.84926  0.0  0.0   1.84926 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 126.423327  126.423327  0.0  0.0  0.0 | 22.713541 | 27.922647 | 22.029725 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 45.0252  45.0252  45.0252  45.0252  45.0252 |  2.597538   2.597538   2.597538   2.597538   2.597538 | 0.020724  0.020724  0.020724  0.020724  0.020724 | 0.020724  0.020724  0.0  0.0  0.020724 |  44.87446   44.87446  45.0252  45.0252   44.87446 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  44.87446   44.87446  60.364388  45.0252   44.87446 | 17.930553  17.930553  60.364388  45.0252   44.87446 | 0.0  0.0  0.0  0.0   44.87446 | 0.009934  0.009934  0.0  0.0  0.0 | 0.023316  0.023316  0.0  0.0  0.0 | 28.521524 |  0.5 | 21.215224 | 0.0  0.0  0.0  0.020724  0.0 | 0.173902 | 20.126118 | 10.187158 | 10.187158 | 42.446491 | 1.979276  1.979276  0.0  0.0  1.979276 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 153.380617  153.380617  0.0  0.0  0.0 | 29.519841 | 34.606527 | 42.446491 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.7416   9.7416   9.7416   9.7416   9.7416 |  3.338304   3.338304   3.338304   3.338304   3.338304 | 0.907373  0.907373  0.907373  0.907373  0.907373 | 0.907373  0.907373  0.0  0.0  0.907373 |  9.720876   9.720876   9.7416   9.7416   9.720876 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.720876   9.720876  24.990557   9.7416   9.720876 |  5.717239   5.717239  24.990557   9.7416   9.720876 | 0.0  0.0  0.0  0.0   9.720876 |  0.44633   0.44633  0.0  0.0  0.0 | 0.021308  0.021308  0.0  0.0  0.0 | 10.519588 |  0.5 | 16.566572 | 0.0  0.0  0.0  0.907373  0.0 | 0.181486 | 14.070702 | 16.219487 | 16.219487 | 67.581197 | 1.092627  1.092627  0.0  0.0  1.092627 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 156.959231  156.959231  0.0  0.0  0.0 | 22.972857 | 36.115645 | 67.581197 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  1.846084   1.846084   1.846084   1.846084   1.846084 | 0.866652  0.866652  0.866652  0.866652  0.866652 | 0.866652  0.866652  0.0  0.0  0.866652 |  3.369427   3.369427   4.2768   4.2768   3.369427 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.369427   3.369427  19.074604   4.2768   3.369427 |  2.075247   2.075247  19.074604   4.2768   3.369427 | 0.0  0.0  0.0  0.0   3.369427 | 0.428652  0.428652  0.0  0.0  0.0 | 0.020873  0.020873  0.0  0.0  0.0 |  6.309431 |  0.5 |  9.257302 | 0.0  0.0  0.0  0.866652  0.0 | 0.185157 |  7.803283 |  14.02359 |  14.02359 | 58.431627 | 1.133348  1.133348  0.0  0.0  1.133348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 157.845632  157.845632  0.0  0.0  0.0 | 19.524987 | 36.846224 | 58.431627 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    8.91     8.91     8.91     8.91     8.91 |  0.338261   0.338261   0.338261   0.338261   0.338261 | 0.099914  0.099914  0.099914  0.099914  0.099914 | 0.099914  0.099914  0.0  0.0  0.099914 |  8.043348   8.043348     8.91     8.91   8.043348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.043348   8.043348  23.437113     8.91   8.043348 |  5.010049   5.010049  23.437113     8.91   8.043348 | 0.0  0.0  0.0  0.0   8.043348 | 0.049957  0.049957  0.0  0.0  0.0 | 0.019561  0.019561  0.0  0.0  0.0 |  9.602145 |  0.5 |  8.193251 | 0.0  0.0  0.0  0.099914  0.0 | 0.192667 |  7.270647 |  9.077679 |  9.077679 | 37.823663 | 1.900086  1.900086  0.0  0.0  1.900086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 160.848534  160.848534  0.0  0.0  0.0 |  20.43388 | 38.340712 | 37.823663 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  21.978   21.978   21.978   21.978   21.978 |  0.107442   0.107442   0.107442   0.107442   0.107442 |  0.00859   0.00859   0.00859   0.00859   0.00859 |  0.00859   0.00859  0.0  0.0   0.00859 | 21.878086  21.878086   21.978   21.978  21.878086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 21.878086  21.878086  36.324652   21.978  21.878086 | 14.150883  14.150883  36.324652   21.978  21.878086 | 0.0  0.0  0.0  0.0  21.878086 | 0.004295  0.004295  0.0  0.0  0.0 | 0.015712  0.015712  0.0  0.0  0.0 | 19.682541 |  0.5 | 13.910461 | 0.0  0.0  0.0   0.00859  0.0 | 0.209539 | 12.782477 |  8.613862 |  8.613862 |  35.89109 |  1.99141   1.99141  0.0  0.0   1.99141 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 168.587153  168.587153  0.0  0.0  0.0 |  25.70596 | 41.698232 |  35.89109 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 18.2952  18.2952  18.2952  18.2952  18.2952 |  0.107243   0.107243   0.107243   0.107243   0.107243 | 0.012392  0.012392  0.012392  0.012392  0.012392 | 0.012392  0.012392  0.0  0.0  0.012392 |  18.28661   18.28661  18.2952  18.2952   18.28661 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  18.28661   18.28661  32.551621  18.2952   18.28661 | 12.993381  12.993381  32.551621  18.2952   18.28661 | 0.0  0.0  0.0  0.0   18.28661 | 0.006196  0.006196  0.0  0.0  0.0 |  0.01306   0.01306  0.0  0.0  0.0 | 17.873146 |  0.5 | 17.355859 | 0.0  0.0  0.0  0.012392  0.0 | 0.223693 | 15.312575 |  12.11987 |  12.11987 | 50.499458 | 1.987608  1.987608  0.0  0.0  1.987608 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 173.887246  173.887246  0.0  0.0  0.0 | 25.723248 | 44.514939 | 50.499458 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.4844   7.4844   7.4844   7.4844   7.4844 |  0.105848   0.105848   0.105848   0.105848   0.105848 | 0.036055  0.036055  0.036055  0.036055  0.036055 | 0.036055  0.036055  0.0  0.0  0.036055 |  7.472008   7.472008   7.4844   7.4844   7.472008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.472008   7.472008  21.109208   7.4844   7.472008 |  5.648235   5.648235  21.109208   7.4844   7.472008 | 0.0  0.0  0.0  0.0   7.472008 | 0.018028  0.018028  0.0  0.0  0.0 | 0.012144  0.012144  0.0  0.0  0.0 |   9.50437 |  0.5 |  12.87116 | 0.0  0.0  0.0  0.036055  0.0 | 0.230038 | 11.009764 | 13.794151 | 13.794151 | 57.475628 | 1.963945  1.963945  0.0  0.0  1.963945 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.705135  175.705135  0.0  0.0  0.0 | 21.856458 | 45.777522 | 57.475628 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.2572   2.2572   2.2572   2.2572   2.2572 |  0.104032   0.104032   0.104032   0.104032   0.104032 | 0.059768  0.059768  0.059768  0.059768  0.059768 | 0.059768  0.059768  0.0  0.0  0.059768 |  2.221145   2.221145   2.2572   2.2572   2.221145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.221145   2.221145  15.069933   2.2572   2.221145 |  1.714296   1.714296  15.069933   2.2572   2.221145 | 0.0  0.0  0.0  0.0   2.221145 | 0.029884  0.029884  0.0  0.0  0.0 | 0.011894  0.011894  0.0  0.0  0.0 |  5.044285 |  0.5 |  7.929774 | 0.0  0.0  0.0  0.059768  0.0 |   0.2326 |  6.708989 | 11.010216 | 11.010216 | 45.875901 | 1.940232  1.940232  0.0  0.0  1.940232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 176.193994  176.193994  0.0  0.0  0.0 | 18.470968 | 46.287412 | 45.875901 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.104235   0.104235   0.104235   0.104235   0.104235 | 0.041931  0.041931  0.041931  0.041931  0.041931 | 0.041931  0.041931  0.0  0.0  0.041931 |  5.761432   5.761432   5.8212   5.8212   5.761432 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.761432   5.761432   18.72961   5.8212   5.761432 |  4.471494   4.471494   18.72961   5.8212   5.761432 | 0.0  0.0  0.0  0.0   5.761432 | 0.020966  0.020966  0.0  0.0  0.0 | 0.011258  0.011258  0.0  0.0  0.0 |  8.027579 |  0.5 |  6.818281 | 0.0  0.0  0.0  0.041931  0.0 | 0.237708 |  6.060581 |  7.520626 |  7.520626 | 31.335942 | 1.958069  1.958069  0.0  0.0  1.958069 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.474225  177.474225  0.0  0.0  0.0 | 19.180266 | 47.303886 | 31.335942 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.2076   3.2076   3.2076   3.2076   3.2076 |  0.104032   0.104032   0.104032   0.104032   0.104032 |  0.05435   0.05435   0.05435   0.05435   0.05435 |  0.05435   0.05435  0.0  0.0   0.05435 |  3.165669   3.165669   3.2076   3.2076   3.165669 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.165669   3.165669  16.025741   3.2076   3.165669 |  2.492735   2.492735  16.025741   3.2076   3.165669 | 0.0  0.0  0.0  0.0   3.165669 | 0.027175  0.027175  0.0  0.0  0.0 | 0.010926  0.010926  0.0  0.0  0.0 |  5.867792 |  0.5 |  6.447511 | 0.0  0.0  0.0   0.05435  0.0 | 0.240915 |  5.593907 |  6.100966 |  6.100966 | 25.420693 |  1.94565   1.94565  0.0  0.0   1.94565 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  178.13091   178.13091  0.0  0.0  0.0 | 18.100547 | 47.942007 | 25.420693 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.594    0.594    0.594    0.594    0.594 |  0.103014   0.103014   0.103014   0.103014   0.103014 | 0.069893  0.069893  0.069893  0.069893  0.069893 | 0.069893  0.069893  0.0  0.0  0.069893 |   0.53965    0.53965    0.594    0.594    0.53965 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.53965    0.53965  12.960798    0.594    0.53965 |  0.428086   0.428086  12.960798    0.594    0.53965 | 0.0  0.0  0.0  0.0    0.53965 | 0.034946  0.034946  0.0  0.0  0.0 | 0.010879  0.010879  0.0  0.0  0.0 |  3.553105 |  0.5 |  4.967601 | 0.0  0.0  0.0  0.069893  0.0 | 0.242227 |  4.236136 |  5.395886 |  5.395886 | 22.482856 | 1.930107  1.930107  0.0  0.0  1.930107 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.218407  178.218407  0.0  0.0  0.0 | 16.186051 | 48.203224 | 22.482856 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.103422   0.103422   0.103422   0.103422   0.103422 | 0.072716  0.072716  0.072716  0.072716  0.072716 | 0.072716  0.072716  0.0  0.0  0.072716 |  0.167707   0.167707   0.2376   0.2376   0.167707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.167707   0.167707  12.784935   0.2376   0.167707 |  0.133167   0.133167  12.784935   0.2376   0.167707 | 0.0  0.0  0.0  0.0   0.167707 | 0.036358  0.036358  0.0  0.0  0.0 | 0.010874  0.010874  0.0  0.0  0.0 |  3.287954 |  0.5 |  3.827142 | 0.0  0.0  0.0  0.072716  0.0 | 0.243277 |  3.299953 |  4.329822 |  4.329822 | 18.040927 | 1.927284  1.927284  0.0  0.0  1.927284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.227463  178.227463  0.0  0.0  0.0 | 15.146863 | 48.412073 | 18.040927 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.594    0.594    0.594    0.594    0.594 |   0.10464    0.10464    0.10464    0.10464    0.10464 | 0.070996  0.070996  0.070996  0.070996  0.070996 | 0.070996  0.070996  0.0  0.0  0.070996 |  0.521284   0.521284    0.594    0.594   0.521284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.521284   0.521284  13.682947    0.594   0.521284 |  0.413965   0.413965  13.682947    0.594   0.521284 | 0.0  0.0  0.0  0.0   0.521284 | 0.035498  0.035498  0.0  0.0  0.0 | 0.010833  0.010833  0.0  0.0  0.0 |  3.723086 |  0.5 |  3.422146 | 0.0  0.0  0.0  0.070996  0.0 | 0.244577 |  3.000741 |  3.441502 |  3.441502 | 14.339591 | 1.929004  1.929004  0.0  0.0  1.929004 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.310117  178.310117  0.0  0.0  0.0 | 14.947803 | 48.670783 | 14.339591 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.8512   2.8512   2.8512   2.8512   2.8512 |  0.106248   0.106248   0.106248   0.106248   0.106248 | 0.057521  0.057521  0.057521  0.057521  0.057521 | 0.057521  0.057521  0.0  0.0  0.057521 |  2.780204   2.780204   2.8512   2.8512   2.780204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.780204   2.780204  16.662296   2.8512   2.780204 |   2.20988    2.20988  16.662296   2.8512   2.780204 | 0.0  0.0  0.0  0.0   2.780204 | 0.028761  0.028761  0.0  0.0  0.0 |  0.01056   0.01056  0.0  0.0  0.0 |  5.815064 |  0.5 |  3.930692 | 0.0  0.0  0.0  0.057521  0.0 | 0.247492 |  3.560893 |  3.191711 |  3.191711 | 13.298794 | 1.942479  1.942479  0.0  0.0  1.942479 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  178.86224   178.86224  0.0  0.0  0.0 | 16.332175 | 49.250959 | 13.298794 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  0.179639   0.179639   0.179639   0.179639   0.179639 | 0.123337  0.123337  0.123337  0.123337  0.123337 | 0.123337  0.123337  0.0  0.0  0.123337 |  0.417679   0.417679   0.4752   0.4752   0.417679 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.417679   0.417679  14.557102   0.4752   0.417679 |  0.334057   0.334057  14.557102   0.4752   0.417679 | 0.0  0.0  0.0  0.0   0.417679 | 0.061669  0.061669  0.0  0.0  0.0 | 0.010527  0.010527  0.0  0.0  0.0 |  3.881922 |  0.5 |  4.079736 | 0.0  0.0  0.0  0.123337  0.0 | 0.248649 |  3.523706 |  3.428151 |  3.428151 | 14.283963 | 1.876663  1.876663  0.0  0.0  1.876663 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.894721  178.894721  0.0  0.0  0.0 | 15.634362 | 49.481147 | 14.283963 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.545628   0.545628   0.545628   0.545628   0.545628 | 0.383628  0.383628  0.383628  0.383628  0.383628 | 0.383628  0.383628  0.0  0.0  0.383628 |  0.114263   0.114263   0.2376   0.2376   0.114263 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.114263   0.114263  14.951382   0.2376   0.114263 |   0.09142    0.09142  14.951382   0.2376   0.114263 | 0.0  0.0  0.0  0.0   0.114263 | 0.191814  0.191814  0.0  0.0  0.0 | 0.010541  0.010541  0.0  0.0  0.0 |  3.798504 |  0.5 |  3.689107 | 0.0  0.0  0.0  0.383628  0.0 | 0.249444 |  3.191718 |  3.458195 |  3.458195 | 14.409145 | 1.616372  1.616372  0.0  0.0  1.616372 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.736291  178.736291  0.0  0.0  0.0 | 15.243759 | 49.639413 | 14.409145 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.361522   1.361522   1.361522   1.361522   1.361522 | 0.980296  0.980296  0.980296  0.980296  0.980296 | 0.980296  0.980296  0.0  0.0  0.980296 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856      0.0        0.0 |       0.0        0.0  15.074856      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.490148  0.490148  0.0  0.0  0.0 | 0.010632  0.010632  0.0  0.0  0.0 |   3.76074 |  0.5 |  3.480795 | 0.0  0.0  0.0  0.980296  0.0 |  0.24964 |  3.017633 |  3.226807 |  3.226807 | 13.445031 | 0.636076  0.636076  0.0  0.0  0.636076 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.256775  178.256775  0.0  0.0  0.0 | 15.023704 | 49.678303 | 13.445031 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  4.874663   4.874663   4.874663   4.874663   4.874663 | 3.509758  3.509758  3.509758  3.509758  3.509758 | 0.636076  0.636076  0.0  0.0  0.636076 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439      0.0        0.0 |       0.0        0.0  14.262439      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.19172   3.19172  0.0  0.0  0.0 | 0.010872  0.010872  0.0  0.0  0.0 |  3.557456 |  0.5 |  3.316135 | 0.0  0.0  0.0  3.509758  0.0 | 0.248027 |    2.8844 |  3.026711 |  3.026711 | 12.611296 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.075927  175.075927  0.0  0.0  0.0 | 14.765026 | 49.357453 | 12.611296 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.057269   5.057269   5.057269   5.057269   5.057269 | 3.513746  3.513746  3.513746  3.513746  3.513746 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.348033   0.3564        0.0 |       0.0        0.0  14.348033   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.335546  3.335546  0.0  0.0  0.0 | 0.012462  0.012462  0.0  0.0  0.0 |  3.577662 |  0.5 |  3.196444 | 0.0  0.0  0.0  3.513746  0.0 | 0.246675 |  2.787385 |  2.892448 |  2.892448 | 12.051868 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 171.752844  171.752844  0.0  0.0  0.0 | 14.646244 |   49.0883 | 12.051868 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.0888   3.0888   3.0888   3.0888   3.0888 |  6.154673   6.154673   6.154673   6.154673   6.154673 | 3.253813  3.253813  3.253813  3.253813  3.253813 |      2.0       2.0  0.0  0.0       2.0 |    1.0888     1.0888   3.0888   3.0888     1.0888 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    1.0888     1.0888  17.441507   3.0888     1.0888 |  0.802964   0.802964  17.441507   3.0888     1.0888 | 0.0  0.0  0.0  0.0     1.0888 | 2.253813  2.253813  0.0  0.0  0.0 | 0.013981  0.013981  0.0  0.0  0.0 |  4.952114 |  0.5 |  3.519341 | 0.0  0.0  0.0  3.253813  0.0 | 0.247466 |  3.119028 |  2.882642 |  2.882642 |  12.01101 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 169.798847  169.798847  0.0  0.0  0.0 | 15.579017 | 49.245832 |  12.01101 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  6.236191   6.236191   6.236191   6.236191   6.236191 | 4.131769  4.131769  4.131769  4.131769  4.131769 |   0.8316    0.8316  0.0  0.0    0.8316 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.184307   0.8316        0.0 |       0.0        0.0  15.184307   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.715969  3.715969  0.0  0.0  0.0 | 0.015101  0.015101  0.0  0.0  0.0 |  3.784751 |  0.5 |  3.656849 | 0.0  0.0  0.0  4.131769  0.0 | 0.246015 |  3.155093 |  3.053344 |  3.053344 | 12.722267 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 166.097979  166.097979  0.0  0.0  0.0 | 15.206919 | 48.956936 | 12.722267 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.404341   5.404341   5.404341   5.404341   5.404341 | 3.754888  3.754888  3.754888  3.754888  3.754888 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.799376   0.3564        0.0 |       0.0        0.0  14.799376   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.576688  3.576688  0.0  0.0  0.0 | 0.016951  0.016951  0.0  0.0  0.0 |  3.687131 |  0.5 |  3.441957 | 0.0  0.0  0.0  3.754888  0.0 |   0.2445 |  2.981765 |  3.108561 |  3.108561 | 12.952339 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 162.538242  162.538242  0.0  0.0  0.0 | 14.952093 | 48.655509 | 12.952339 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  9.035929   9.035929   9.035929   9.035929   9.035929 | 6.278083  6.278083  6.278083  6.278083  6.278083 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.889645   0.3564        0.0 |       0.0        0.0  14.889645   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.099883  6.099883  0.0  0.0  0.0 | 0.018731  0.018731  0.0  0.0  0.0 |  3.708363 |  0.5 |  3.322309 | 0.0  0.0  0.0  6.278083  0.0 | 0.241191 |  2.882958 |  2.998326 |  2.998326 | 12.493023 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  156.45709   156.45709  0.0  0.0  0.0 | 14.838148 | 47.996935 | 12.493023 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.627818   5.627818   5.627818   5.627818   5.627818 | 4.052029  4.052029  4.052029  4.052029  4.052029 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348      0.0        0.0 |       0.0        0.0  16.248348      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.962855  3.962855  0.0  0.0  0.0 | 0.021771  0.021771  0.0  0.0  0.0 |  4.045758 |  0.5 |  3.359201 | 0.0  0.0  0.0  4.052029  0.0 | 0.239233 |  2.910645 |  2.911068 |  2.911068 |  12.12945 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 152.516006  152.516006  0.0  0.0  0.0 | 15.024705 | 47.607412 |  12.12945 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  6.156027   6.156027   6.156027   6.156027   6.156027 |  4.43234   4.43234   4.43234   4.43234   4.43234 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079      0.0        0.0 |       0.0        0.0  16.158079      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.225675  4.225675  0.0  0.0  0.0 | 0.023742  0.023742  0.0  0.0  0.0 |  4.021713 |  0.5 |  3.444646 | 0.0  0.0  0.0   4.43234  0.0 | 0.237014 |   2.97693 |  2.919223 |  2.919223 | 12.163427 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.314073  148.314073  0.0  0.0  0.0 | 15.101772 | 47.165778 | 12.163427 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.811802   3.811802   3.811802   3.811802   3.811802 | 2.744497  2.744497  2.744497  2.744497  2.744497 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542      0.0        0.0 |       0.0        0.0  15.977542      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.544491  2.544491  0.0  0.0  0.0 | 0.025843  0.025843  0.0  0.0  0.0 |  3.975003 |  0.5 |  3.469922 | 0.0  0.0  0.0  2.744497  0.0 | 0.236011 |  2.996215 |  2.966485 |  2.966485 | 12.360356 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 145.795425  145.795425  0.0  0.0  0.0 | 15.106853 | 46.966267 | 12.360356 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.747211   1.747211   1.747211   1.747211   1.747211 | 1.257992  1.257992  1.257992  1.257992  1.257992 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782      0.0        0.0 |       0.0        0.0  14.713782      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.146522  1.146522  0.0  0.0  0.0 | 0.027102  0.027102  0.0  0.0  0.0 |  3.658119 |  0.5 |  3.384447 | 0.0  0.0  0.0  1.257992  0.0 | 0.236076 |  2.927895 |  2.976747 |  2.976747 | 12.403113 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.676006  144.676006  0.0  0.0  0.0 | 14.880525 |  46.97905 | 12.403113 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.318724   0.318724   0.318724   0.318724   0.318724 | 0.229481  0.229481  0.229481  0.229481  0.229481 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753      0.0        0.0 |       0.0        0.0  13.359753      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.207542  0.207542  0.0  0.0  0.0 | 0.027662  0.027662  0.0  0.0  0.0 |  3.319192 |  0.5 |  3.182896 | 0.0  0.0  0.0  0.229481  0.0 | 0.236874 |  2.767399 |  2.907411 |  2.907411 | 12.114214 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.496125  144.496125  0.0  0.0  0.0 | 14.516821 | 47.137964 | 12.114214 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.101575   0.101575   0.101575   0.101575   0.101575 | 0.073134  0.073134  0.073134  0.073134  0.073134 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918      0.0        0.0 |       0.0        0.0  11.734918      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06606   0.06606  0.0  0.0  0.0 | 0.027752  0.027752  0.0  0.0  0.0 |  2.912916 |  0.5 |  2.907373 | 0.0  0.0  0.0  0.073134  0.0 |  0.23778 |  2.547827 |  2.754271 |  2.754271 |  11.47613 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.457817  144.457817  0.0  0.0  0.0 | 14.022363 | 47.318307 |  11.47613 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232      0.0        0.0 |       0.0        0.0  10.832232      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06469   0.06469  0.0  0.0  0.0 | 0.027771  0.027771  0.0  0.0  0.0 |   2.68723 |  0.5 |  2.630798 | 0.0  0.0  0.0  0.071636  0.0 | 0.238683 |  2.327409 |  2.547639 |  2.547639 | 10.615163 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.420898  144.420898  0.0  0.0  0.0 | 13.578795 | 47.497962 | 10.615163 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.097808   0.097808   0.097808   0.097808   0.097808 | 0.070422  0.070422  0.070422  0.070422  0.070422 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083      0.0        0.0 |       0.0        0.0  10.110083      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063577  0.063577  0.0  0.0  0.0 |  0.02779   0.02779  0.0  0.0  0.0 |  2.506679 |  0.5 |  2.399532 | 0.0  0.0  0.0  0.070422  0.0 | 0.239582 |  2.143236 |  2.335464 |  2.335464 |  9.731098 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.385111  144.385111  0.0  0.0  0.0 | 13.185942 |  47.67689 |  9.731098 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096529   0.096529   0.096529   0.096529   0.096529 | 0.069501  0.069501  0.069501  0.069501  0.069501 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472      0.0        0.0 |       0.0        0.0   9.568472      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06273   0.06273  0.0  0.0  0.0 | 0.027807  0.027807  0.0  0.0  0.0 |  2.371262 |  0.5 |  2.209111 | 0.0  0.0  0.0  0.069501  0.0 | 0.240478 |  1.991735 |  2.150496 |  2.150496 |  8.960402 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.350188  144.350188  0.0  0.0  0.0 | 12.848093 | 47.855055 |  8.960402 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.095455   0.095455   0.095455   0.095455   0.095455 | 0.068728  0.068728  0.068728  0.068728  0.068728 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265      0.0        0.0 |       0.0        0.0   9.121265      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.062017  0.062017  0.0  0.0  0.0 | 0.027825  0.027825  0.0  0.0  0.0 |  2.259448 |  0.5 |  2.052834 | 0.0  0.0  0.0  0.068728  0.0 | 0.241369 |  1.867545 |  1.997804 |  1.997804 |  8.324184 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.315995  144.315995  0.0  0.0  0.0 | 12.554706 |  48.03244 |  8.324184 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063669  0.063669  0.0  0.0  0.0 | 0.027842  0.027842  0.0  0.0  0.0 |  2.530363 |  0.5 |  1.996979 | 0.0  0.0  0.0  0.070574  0.0 | 0.242255 |  1.823687 |  1.885397 |  1.885397 |   7.85582 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.280169  144.280169  0.0  0.0  0.0 |  12.58809 | 48.208674 |   7.85582 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063927  0.063927  0.0  0.0  0.0 |  0.02786   0.02786  0.0  0.0  0.0 |  2.575505 |  0.5 |  2.019041 | 0.0  0.0  0.0  0.070878  0.0 | 0.243136 |  1.842159 |  1.837538 |  1.837538 |   7.65641 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.244102  144.244102  0.0  0.0  0.0 | 12.644554 | 48.383985 |   7.65641 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.096957   0.096957   0.096957   0.096957   0.096957 | 0.059819  0.059819  0.059819  0.059819  0.059819 | 0.059819  0.059819  0.0  0.0  0.059819 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.297832   1.5444        0.0 |       0.0        0.0  11.297832   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.02697   0.02697  0.0  0.0  0.0 | 0.027878  0.027878  0.0  0.0  0.0 |  2.803549 |  0.5 |  2.088785 | 0.0  0.0  0.0  0.059819  0.0 | 0.245123 |   1.89981 |  1.850866 |  1.850866 |   7.71194 | 1.484581  1.484581  0.0  0.0  1.484581 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  144.24501   144.24501  0.0  0.0  0.0 | 12.859318 | 48.779516 |   7.71194 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096744   0.096744   0.096744   0.096744   0.096744 | 0.069656  0.069656  0.069656  0.069656  0.069656 | 0.069656  0.069656  0.0  0.0  0.069656 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122      0.0        0.0 |       0.0        0.0   9.663122      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.031404  0.031404  0.0  0.0  0.0 | 0.027877  0.027877  0.0  0.0  0.0 |  2.394872 |  0.5 |  2.085549 | 0.0  0.0  0.0  0.069656  0.0 | 0.245991 |   1.89803 |  1.886603 |  1.886603 |  7.860846 | 1.414925  1.414925  0.0  0.0  1.414925 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.241483  144.241483  0.0  0.0  0.0 | 12.668642 | 48.952146 |  7.860846 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 | 0.070878  0.070878  0.0  0.0  0.070878 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.031955  0.031955  0.0  0.0  0.0 | 0.027879  0.027879  0.0  0.0  0.0 |   2.57549 |  0.5 |  2.049886 | 0.0  0.0  0.0  0.070878  0.0 | 0.246853 |  1.870305 |  1.892265 |  1.892265 |  7.884436 | 1.344047  1.344047  0.0  0.0  1.344047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.237408  144.237408  0.0  0.0  0.0 | 12.694246 | 49.123739 |  7.884436 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.359866   0.359866   0.359866   0.359866   0.359866 | 0.259103  0.259103  0.259103  0.259103  0.259103 | 0.259103  0.259103  0.0  0.0  0.259103 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527      0.0        0.0 |       0.0        0.0  10.656527      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.116811  0.116811  0.0  0.0  0.0 | 0.027881  0.027881  0.0  0.0  0.0 |  2.643221 |  0.5 |  2.074023 | 0.0  0.0  0.0  0.259103  0.0 | 0.247576 |   1.89029 |  1.880907 |  1.880907 |  7.837114 | 1.084943  1.084943  0.0  0.0  1.084943 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.148478  144.148478  0.0  0.0  0.0 | 12.763443 | 49.267719 |  7.837114 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  1.614671   1.614671   1.614671   1.614671   1.614671 | 1.069795  1.069795  1.069795  1.069795  1.069795 | 1.069795  1.069795  0.0  0.0  1.069795 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.126889   0.8316        0.0 |       0.0        0.0  11.126889   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.481998  0.481998  0.0  0.0  0.0 | 0.027926  0.027926  0.0  0.0  0.0 |  2.760778 |  0.5 |  2.126008 | 0.0  0.0  0.0  1.069795  0.0 | 0.248311 |  1.932564 |  1.895243 |  1.895243 |  7.896846 | 0.846748  0.846748  0.0  0.0  0.846748 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.694406  143.694406  0.0  0.0  0.0 | 12.898213 | 49.413951 |  7.896846 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  3.707216   3.707216   3.707216   3.707216   3.707216 | 2.545322  2.545322  2.545322  2.545322  2.545322 | 1.321948  1.321948  0.0  0.0  1.321948 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.409251   0.4752        0.0 |       0.0        0.0  10.409251   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.754875  1.754875  0.0  0.0  0.0 | 0.028153  0.028153  0.0  0.0  0.0 |  2.581198 |  0.5 |   2.14036 | 0.0  0.0  0.0  2.545322  0.0 | 0.247734 |  1.943507 |  1.925601 |  1.925601 |  8.023338 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 141.967683  141.967683  0.0  0.0  0.0 | 12.839051 | 49.299057 |  8.023338 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  6.022315   6.022315   6.022315   6.022315   6.022315 | 4.284859  4.284859  4.284859  4.284859  4.284859 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23347   0.1188        0.0 |       0.0        0.0   10.23347   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.802724  3.802724  0.0  0.0  0.0 | 0.029016  0.029016  0.0  0.0  0.0 |  2.536605 |  0.5 |  2.107668 | 0.0  0.0  0.0  4.284859  0.0 | 0.245662 |  1.915419 |  1.934833 |  1.934833 |  8.061805 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 138.193975  138.193975  0.0  0.0  0.0 | 12.767989 | 48.886815 |  8.061805 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  8.299842   8.299842   8.299842   8.299842   8.299842 | 5.698554  5.698554  5.698554  5.698554  5.698554 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.854204   0.4752        0.0 |       0.0        0.0  11.854204   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.923012  4.923012  0.0  0.0  0.0 | 0.030903  0.030903  0.0  0.0  0.0 |  2.940374 |  0.5 |  2.166627 | 0.0  0.0  0.0  5.698554  0.0 | 0.242846 |  1.959958 |  1.931558 |  1.931558 |  8.048159 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 133.301867  133.301867  0.0  0.0  0.0 | 13.041736 | 48.326347 |  8.048159 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.734061   9.734061   9.734061   9.734061   9.734061 | 7.008524  7.008524  7.008524  7.008524  7.008524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695      0.0        0.0 |       0.0        0.0  15.081695      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.840519  5.840519  0.0  0.0  0.0 | 0.033349  0.033349  0.0  0.0  0.0 |  3.745412 |  0.5 |  2.461314 | 0.0  0.0  0.0  7.008524  0.0 | 0.238769 |  2.191902 |  2.001603 |  2.001603 |  8.340014 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 127.494697  127.494697  0.0  0.0  0.0 | 13.825834 | 47.514932 |  8.340014 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.733892   9.733892   9.733892   9.733892   9.733892 | 7.008402  7.008402  7.008402  7.008402  7.008402 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101      0.0        0.0 |       0.0        0.0  16.075101      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.586176  5.586176  0.0  0.0  0.0 | 0.036253  0.036253  0.0  0.0  0.0 |  3.991586 |  0.5 |  2.865841 | 0.0  0.0  0.0  7.008402  0.0 | 0.234712 |  2.511737 |  2.211433 |  2.211433 |  9.214305 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 121.944773  121.944773  0.0  0.0  0.0 | 14.451578 | 46.707592 |  9.214305 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 12.429676  12.429676  12.429676  12.429676  12.429676 | 8.949367  8.949367  8.949367  8.949367  8.949367 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102      0.0        0.0 |       0.0        0.0  19.868102      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.822986  6.822986  0.0  0.0  0.0 | 0.039028  0.039028  0.0  0.0  0.0 |  4.937755 |  0.5 |  3.418262 | 0.0  0.0  0.0  8.949367  0.0 | 0.229288 |  2.948612 |  2.537746 |  2.537746 | 10.573941 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.160815  115.160815  0.0  0.0  0.0 | 15.471071 | 45.628394 | 10.573941 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 10.884737  10.884737  10.884737  10.884737  10.884737 | 7.837011  7.837011  7.837011  7.837011  7.837011 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006      0.0        0.0 |       0.0        0.0  18.965006      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.642806  5.642806  0.0  0.0  0.0 |  0.04242   0.04242  0.0  0.0  0.0 |  4.709437 |  0.5 |   3.87128 | 0.0  0.0  0.0  7.837011  0.0 | 0.224687 |  3.306732 |  2.931111 |  2.931111 | 12.212962 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 109.560428  109.560428  0.0  0.0  0.0 | 15.809228 | 44.712705 | 12.212962 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.119821   9.119821   9.119821   9.119821   9.119821 | 6.566271  6.566271  6.566271  6.566271  6.566271 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244      0.0        0.0 |       0.0        0.0  19.326244      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.498128  4.498128  0.0  0.0  0.0 |  0.04522   0.04522  0.0  0.0  0.0 |  4.797646 |  0.5 |  4.077238 | 0.0  0.0  0.0  6.566271  0.0 | 0.221016 |  3.468072 |  3.263003 |  3.263003 | 13.595846 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  105.10752   105.10752  0.0  0.0  0.0 | 16.029636 | 43.982222 | 13.595846 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  7.306062   7.306062   7.306062   7.306062   7.306062 | 5.260364  5.260364  5.260364  5.260364  5.260364 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483      0.0        0.0 |       0.0        0.0  19.687483      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.457209  3.457209  0.0  0.0  0.0 | 0.047446  0.047446  0.0  0.0  0.0 |  4.886286 |  0.5 |  4.223748 | 0.0  0.0  0.0  5.260364  0.0 | 0.218297 |  3.582742 |  3.457701 |  3.457701 | 14.407087 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 101.697758  101.697758  0.0  0.0  0.0 | 16.192174 | 43.441016 | 14.407087 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.420163   3.420163   3.420163   3.420163   3.420163 | 2.462517  2.462517  2.462517  2.462517  2.462517 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982      0.0        0.0 |       0.0        0.0  17.790982      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.565959  1.565959  0.0  0.0  0.0 | 0.049151  0.049151  0.0  0.0  0.0 |  4.410882 |  0.5 |  4.165073 | 0.0  0.0  0.0  2.462517  0.0 | 0.217589 |  3.535141 |  3.546682 |  3.546682 |  14.77784 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 100.180949  100.180949  0.0  0.0  0.0 | 15.937983 |  43.30021 |  14.77784 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.518709   1.518709   1.518709   1.518709   1.518709 | 1.093471  1.093471  1.093471  1.093471  1.093471 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862      0.0        0.0 |       0.0        0.0  15.713862      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.684997  0.684997  0.0  0.0  0.0 |  0.04991   0.04991  0.0  0.0  0.0 |  3.891033 |  0.5 |  3.873407 | 0.0  0.0  0.0  1.093471  0.0 | 0.217863 |  3.302065 |  3.493924 |  3.493924 | 14.558019 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.545862   99.545862  0.0  0.0  0.0 | 15.455609 | 43.354709 | 14.558019 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.382278   0.382278   0.382278   0.382278   0.382278 |  0.27524   0.27524   0.27524   0.27524   0.27524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814      0.0        0.0 |       0.0        0.0  13.365814      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.17133   0.17133  0.0  0.0  0.0 | 0.050227  0.050227  0.0  0.0  0.0 |  3.303783 |  0.5 |   3.45946 | 0.0  0.0  0.0   0.27524  0.0 |  0.21872 |  2.971706 |  3.280447 |  3.280447 | 13.668527 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.424759   99.424759  0.0  0.0  0.0 | 14.799933 |  43.52524 | 13.668527 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.103829   0.103829   0.103829   0.103829   0.103829 | 0.074757  0.074757  0.074757  0.074757  0.074757 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647      0.0        0.0 |       0.0        0.0  12.733647      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046478  0.046478  0.0  0.0  0.0 | 0.050288  0.050288  0.0  0.0  0.0 |  3.145696 |  0.5 |  3.098875 | 0.0  0.0  0.0  0.074757  0.0 | 0.219716 |  2.684168 |  2.981222 |  2.981222 | 12.421758 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.428569   99.428569  0.0  0.0  0.0 | 14.346754 | 43.723416 | 12.421758 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146      0.0        0.0 |       0.0        0.0  10.837146      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.044539  0.044539  0.0  0.0  0.0 | 0.050286  0.050286  0.0  0.0  0.0 |  2.671572 |  0.5 |  2.770382 | 0.0  0.0  0.0  0.071636  0.0 | 0.220709 |    2.4223 |  2.689872 |  2.689872 | 11.207802 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.434315   99.434315  0.0  0.0  0.0 | 13.747944 | 43.921045 | 11.207802 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043882  0.043882  0.0  0.0  0.0 | 0.050283  0.050283  0.0  0.0  0.0 |  2.513533 |  0.5 |  2.472123 | 0.0  0.0  0.0  0.070574  0.0 | 0.221698 |  2.184616 |  2.427674 |  2.427674 |  10.11531 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.440716   99.440716  0.0  0.0  0.0 | 13.289355 | 44.117837 |  10.11531 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.044074  0.044074  0.0  0.0  0.0 |  0.05028   0.05028  0.0  0.0  0.0 |   2.55869 |  0.5 |  2.291974 | 0.0  0.0  0.0  0.070878  0.0 | 0.222681 |  2.041415 |  2.205612 |  2.205612 |  9.190051 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.446922   99.446922  0.0  0.0  0.0 |  13.05607 | 44.313602 |  9.190051 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v1_snow_redistribution_1:

snow redistribution 1
_____________________

Working with five zones allows demonstrating the redistribution of snow due to
gravitational and wind forcing. Therefore, we change the zone elevations so that the
first one (|FIELD|) is on 0 m, the second one (|FOREST|) on 1000 m, the third one on
3000 m (|GLACIER|), the fourth one (|ILAKE|) on 0 m, and the fifth one (|SEALED|) on
2000 m above sea level.  This random order is not ideal for clarity but demonstrates
that unsorted configurations do not harm the results:

>>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)

We define a small snow threshold value of 200 mm to trigger snow redistribution despite
the short simulation period:

>>> smax(200.0)

We prepare the redistribution parameter |SRed| so that the slipped snow of one zone
deposits in the zone(s) directly below:

>>> sred(n_zones=1)

We define a constant air temperature of 5 °C for the subbasin.  However, there is a
substantial difference in the individual zone temperatures due to the high elevation
differences:

>>> inputs.t.series = 5.0

There is also a marked increase in precipitation with elevation due to the value
defined for parameter |PCAlt|, which is exaggerated but helps to stress out the role of
redistribution:

>>> pcalt
pcalt(0.1)

In the following simulation results, the highest zone (|GLACIER|) stores the falling
precipitation as frozen snow (|SP|) and releases the amount exceeding 200 mm (|SPL|) to
the second-highest zone (|SEALED|). The |SEALED| zone receives too much snow itself.
Hence, the third-highest zone (|FOREST|) receives the snow released by both higher
zones (|SPG|), but only for a few hours.  Its available capacity also replenishes soon,
and it then starts releasing snow to the |FIELD| zone at sea level (|ILAKE| zones never
play a role in snow redistribution due to their missing snow module).  However, with
all zones except the |ILAKE| zone releasing snow towards the |FIELD| zone, its snow
cover grows rapidly and soon reaches the |SMax| threshold.  Then, the net transport due
to snow redistribution virtually stops, and the snow covers of all zones (again, except
for the |ILAKE| zone) grow with the same speed (see the documentation on method
|Calc_SPG_WCG_SP_WC_V1| for further details).

In the simulation period's second half, the melting of snow (only taking place in the
low elevation zones) outweighs snowfall (which dominates in the high elevation zones).
Still, snow ablation is at even speed across all zones due to the rebalancing action of
snow redistribution above |SRED| (again, please also read the documentation on method
|Calc_SPG_WCG_SP_WC_V1|):

.. integration-test::

    >>> test("hland_v1_snow_redistribution_1",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |   t |   tn |       epn | tmean |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact | contriarea |                                             pc |                                               ep |                                              epc |                                     ei |                                              tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                           refr |                                    in_ |                                     r |                      sr |                      ea |                                cf |      inuz |     perc |       q0 |                           el |       q1 |     inuh |    outuh |       rt |       qt |                                     ic |                                                  sp |                                                  wc |                                    sm |       uz |        lz |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 5.0 | 20.2 |  0.100707 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |     0.05 | 0.046667 |  0.06037 |  0.06037 | 0.251543 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |      9.95 | 0.251543 |
    | 01/01 01:00 |  0.0 | 5.0 | 18.4 |  0.097801 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.04975 | 0.046433 | 0.086244 | 0.086244 | 0.359352 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.90025 | 0.359352 |
    | 01/01 02:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049501 | 0.046201 | 0.046434 | 0.046434 | 0.193473 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.850749 | 0.193473 |
    | 01/01 03:00 |  0.0 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049254 |  0.04597 | 0.046201 | 0.046201 | 0.192506 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.801495 | 0.192506 |
    | 01/01 04:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049007 |  0.04574 |  0.04597 |  0.04597 | 0.191543 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.752488 | 0.191543 |
    | 01/01 05:00 |  0.0 | 5.0 | 21.5 |  0.102761 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048762 | 0.045512 | 0.045741 | 0.045741 | 0.190586 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.703725 | 0.190586 |
    | 01/01 06:00 |  0.0 | 5.0 | 24.1 |  0.291908 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048519 | 0.045284 | 0.045512 | 0.045512 | 0.189633 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.655206 | 0.189633 |
    | 01/01 07:00 |  0.0 | 5.0 | 27.3 |  1.932875 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048276 | 0.045058 | 0.045284 | 0.045284 | 0.188685 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.60693 | 0.188685 |
    | 01/01 08:00 |  0.0 | 5.0 | 26.8 |  4.369536 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048035 | 0.044832 | 0.045058 | 0.045058 | 0.187741 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.558896 | 0.187741 |
    | 01/01 09:00 |  0.0 | 5.0 | 30.4 |  7.317556 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047794 | 0.044608 | 0.044833 | 0.044833 | 0.186802 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.511101 | 0.186802 |
    | 01/01 10:00 |  0.0 | 5.0 | 31.2 |  8.264362 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047556 | 0.044385 | 0.044608 | 0.044608 | 0.185868 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.463546 | 0.185868 |
    | 01/01 11:00 |  0.0 | 5.0 | 34.2 |  9.369867 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047318 | 0.044163 | 0.044385 | 0.044385 | 0.184939 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.416228 | 0.184939 |
    | 01/01 12:00 |  0.0 | 5.0 | 36.1 |  5.126178 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047081 | 0.043942 | 0.044163 | 0.044163 | 0.184014 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.369147 | 0.184014 |
    | 01/01 13:00 |  0.0 | 5.0 | 30.2 |   6.62503 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046846 | 0.043723 | 0.043943 | 0.043943 | 0.183094 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.322301 | 0.183094 |
    | 01/01 14:00 |  0.0 | 5.0 | 23.3 |  7.397619 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046612 | 0.043504 | 0.043723 | 0.043723 | 0.182179 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.27569 | 0.182179 |
    | 01/01 15:00 |  0.2 | 5.0 | 24.4 |   2.39151 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0    1.1856   0.2112       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046529 | 0.043427 | 0.043536 | 0.043536 | 0.181398 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.259332 | 0.181398 |
    | 01/01 16:00 |  0.0 | 5.0 | 24.9 |  1.829834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046297 |  0.04321 | 0.043396 | 0.043396 | 0.180817 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.213035 | 0.180817 |
    | 01/01 17:00 |  0.0 | 5.0 | 22.7 |  1.136569 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046065 | 0.042994 |  0.04321 |  0.04321 | 0.180044 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.16697 | 0.180044 |
    | 01/01 18:00 |  1.3 | 5.0 | 20.6 |  0.750986 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.180822       8.892  0.0       4.552 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0    1.8232    7.7064   1.3728     4.552 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0   0.642378  0.0   1.3728  0.0 |      0.0  0.160595  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.049759  0.0  0.0  0.0 |  0.036945 | 0.036945 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046974 | 0.043842 | 0.043231 | 0.043231 | 0.180128 |    1.584       2.0  0.0  0.0       2.0 |        0.0    0.843444       8.892  0.0       4.552 |        0.0    0.337378         0.0  0.0         0.0 |      100.0  100.531543  0.0  0.0  0.0 |      0.0 |  9.347778 | 0.180128 |
    | 01/01 19:00 |  5.6 | 5.0 | 20.2 |  0.223895 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    10.04271     42.0888  0.0     29.0128 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.9136  14.27328   33.1968   5.9136   24.4608 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.4976  14.27328   33.1968   5.9136   24.4608 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.4976   5.411392  0.0   5.9136  0.0 |   1.3744  1.367268  0.0   5.9136  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047938  0.047712  0.0  0.0  0.0 |  0.992544 |      0.5 |  0.00003 | 0.0  0.0  0.0       0.0  0.0 | 0.053106 |  0.04959 | 0.044931 | 0.044931 | 0.187212 |      2.0       2.0  0.0  0.0       2.0 |        0.0    7.173364     42.0888  0.0     29.0128 |        0.0    2.869346         0.0  0.0         0.0 | 104.171138  104.623379  0.0  0.0  0.0 | 0.492514 | 10.568043 | 0.187212 |
    | 01/01 20:00 |  2.9 | 5.0 | 19.4 |  0.099425 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.658902       59.28  0.0       41.68 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.0624   7.39152   17.1912   3.0624   12.6672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.0624   7.39152   17.1912   3.0624   12.6672 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.0624   2.775328  0.0   3.0624  0.0 |   0.8308  0.759472  0.0   3.0624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046799   0.04668  0.0  0.0  0.0 |  0.564265 |      0.5 | 0.000145 | 0.0  0.0  0.0       0.0  0.0 | 0.057171 | 0.053475 | 0.049176 | 0.049176 |   0.2049 |      2.0       2.0  0.0  0.0       2.0 |        0.0   10.470644       59.28  0.0       41.68 |        0.0    4.188258         0.0  0.0         0.0 | 106.449537  106.685915  0.0  0.0  0.0 | 0.556633 |  11.37693 |   0.2049 |
    | 01/01 21:00 |  4.9 | 5.0 | 18.8 |  0.098454 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.420054     88.3272  0.0     63.0832 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.1744   4.727968  0.0   5.1744  0.0 | 1.465843   1.34533  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.044921  0.044966  0.0  0.0  0.0 |  1.025506 |      0.5 | 0.000612 | 0.0  0.0  0.0       0.0  0.0 | 0.062724 | 0.059031 | 0.053846 | 0.053846 |  0.22436 |      2.0       2.0  0.0  0.0       2.0 |        0.0   16.014324     88.3272  0.0     63.0832 |        0.0     6.40573         0.0  0.0         0.0 | 110.203014  110.113519  0.0  0.0  0.0 | 1.081527 | 12.481978 |  0.22436 |
    | 01/01 22:00 | 10.6 | 5.0 | 18.6 |  0.098128 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.144342     151.164  0.0     109.384 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 11.1936  27.01728   62.8368  11.1936   46.3008 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  11.1936  27.01728   62.8368  11.1936   46.3008 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  11.1936  10.292992  0.0  11.1936  0.0 | 3.398574   3.12006  0.0  11.1936  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.041001  0.041357  0.0  0.0  0.0 |  2.425223 |      0.5 | 0.010504 | 0.0  0.0  0.0       0.0  0.0 | 0.072548 | 0.076115 | 0.061593 | 0.061593 | 0.256637 |      2.0       2.0  0.0  0.0       2.0 |        0.0   27.960244     151.164  0.0     109.384 |        0.0   11.184098         0.0  0.0         0.0 | 118.039041  117.327808  0.0  0.0  0.0 | 2.996247 | 14.437087 | 0.256637 |
    | 01/01 23:00 |  0.1 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    39.35759    151.7568  0.0    109.8208 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056   0.041632  0.0   0.1056  0.0 | 0.036784  0.014327  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040946  0.041322  0.0  0.0  0.0 | -0.010733 |      0.5 | 0.020476 | 0.0  0.0  0.0       0.0  0.0 | 0.074404 | 0.085825 | 0.074476 | 0.074476 | 0.310317 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.112564    151.7568  0.0    109.8208 |        0.0   11.245026         0.0  0.0         0.0 | 118.148803  117.396435  0.0  0.0  0.0 | 2.465038 |  14.80634 | 0.310317 |
    | 02/01 00:00 |  0.7 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   40.514326    155.9064  0.0    112.8784 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.7392   0.627424  0.0   0.7392  0.0 | 0.257965  0.216178  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040685  0.041096  0.0  0.0  0.0 |  0.148894 |      0.5 | 0.011948 | 0.0  0.0  0.0       0.0  0.0 | 0.076703 | 0.081148 | 0.082628 | 0.082628 | 0.344281 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.938804    155.9064  0.0    112.8784 |        0.0   11.575522         0.0  0.0         0.0 | 118.670724  117.848778  0.0  0.0  0.0 | 2.101983 | 15.263809 | 0.344281 |
    | 02/01 01:00 |  3.0 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   45.287766    173.6904  0.0    125.9824 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.168    7.6464    17.784    3.168    13.104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.168    7.6464    17.784    3.168    13.104 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.168    2.87296  0.0    3.168  0.0 | 1.115353  0.997516  0.0    3.168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039638  0.040138  0.0  0.0  0.0 |   0.76734 |      0.5 | 0.011153 | 0.0  0.0  0.0       0.0  0.0 | 0.080725 | 0.084265 |  0.08288 |  0.08288 | 0.345333 |      2.0       2.0  0.0  0.0       2.0 |        0.0   32.348404    173.6904  0.0    125.9824 |        0.0   12.939362         0.0  0.0         0.0 | 120.763009   119.76436  0.0  0.0  0.0 | 2.358171 | 16.064227 | 0.345333 |
    | 02/01 02:00 |  2.1 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   48.645974    186.1392  0.0    135.1552 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.2176   5.35248   12.4488   2.2176    9.1728 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.2176   5.35248   12.4488   2.2176    9.1728 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   2.2176   1.994272  0.0   2.2176  0.0 | 0.808521  0.715121  0.0   2.2176  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.038914  0.039478  0.0  0.0  0.0 |  0.545884 |      0.5 | 0.013393 | 0.0  0.0  0.0       0.0  0.0 | 0.084048 | 0.089159 |  0.08466 |  0.08466 | 0.352751 |      2.0       2.0  0.0  0.0       2.0 |        0.0   34.747124    186.1392  0.0    135.1552 |        0.0    13.89885         0.0  0.0         0.0 | 122.211002  121.082989  0.0  0.0  0.0 | 2.390661 | 16.725551 | 0.352751 |
    | 02/01 03:00 | 10.4 | 5.0 | 17.7 |  0.096652 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  106.141894       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 10.9824  26.50752   61.6512  10.9824   45.4272 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  10.9824  26.50752   61.6512  10.9824   45.4272 |        0.0         0.0     47.7904  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     30.9884         0.0  0.0     19.4176 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  10.9824        0.0  0.0  10.9824  0.0 | 4.100699       0.0  0.0  10.9824  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.035454  0.039459  0.0  0.0  0.0 |  1.680699 |      0.5 | 0.027346 | 0.0  0.0  0.0       0.0  0.0 | 0.093615 | 0.109251 | 0.092537 | 0.092537 | 0.385569 |      2.0       2.0  0.0  0.0       2.0 |        0.0   77.456804       200.0  0.0       200.0 |        0.0    28.68509         0.0  0.0         0.0 | 129.128157  121.122447  0.0  0.0  0.0 | 3.544015 | 18.629421 | 0.385569 |
    | 02/01 04:00 |  3.5 | 5.0 | 17.5 |  0.096321 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  134.445694       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.696    8.9208    20.748    3.696    15.288 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.696    8.9208    20.748    3.696    15.288 |        0.0         0.0      20.748  0.0      15.288 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.383         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.696        0.0  0.0    3.696  0.0 | 1.540685       0.0  0.0    3.696  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.034358  0.039439  0.0  0.0  0.0 |   0.61449 |      0.5 | 0.045862 | 0.0  0.0  0.0       0.0  0.0 |  0.09793 | 0.128091 | 0.108973 | 0.108973 | 0.454053 |      2.0       2.0  0.0  0.0       2.0 |        0.0  100.811004       200.0  0.0       200.0 |        0.0    33.63469         0.0  0.0         0.0 |  131.31783  121.161886  0.0  0.0  0.0 | 3.612643 | 19.488063 | 0.454053 |
    | 02/01 05:00 |  3.4 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  161.940814       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.5904   8.66592   20.1552   3.5904   14.8512 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.5904   8.66592   20.1552   3.5904   14.8512 |        0.0         0.0     20.1552  0.0     14.8512 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.8292         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.5904        0.0  0.0   3.5904  0.0 | 1.547855       0.0  0.0   3.5904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.03332  0.039419  0.0  0.0  0.0 |  0.617917 |      0.5 | 0.048567 | 0.0  0.0  0.0       0.0  0.0 | 0.102148 | 0.134191 |  0.12526 |  0.12526 | 0.521916 |      2.0       2.0  0.0  0.0       2.0 |        0.0  123.499084       200.0  0.0       200.0 |        0.0    38.44173         0.0  0.0         0.0 | 133.393695  121.201305  0.0  0.0  0.0 | 3.681992 | 20.327401 | 0.521916 |
    | 02/01 06:00 |  1.2 | 5.0 | 17.5 |  0.187298 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  171.644974       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.2672   3.05856    7.1136   1.2672    5.2416 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.2672   3.05856    7.1136   1.2672    5.2416 |        0.0         0.0      7.1136  0.0      5.2416 |       0.0        0.0        0.0  0.0        0.0 |        0.0      6.6456         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   1.2672        0.0  0.0   1.2672  0.0 |  0.56371       0.0  0.0   1.2672  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032951  0.039399  0.0  0.0  0.0 |  0.208016 |      0.5 | 0.043417 | 0.0  0.0  0.0       0.0  0.0 | 0.104685 | 0.132439 | 0.132446 | 0.132446 |  0.55186 |      2.0       2.0  0.0  0.0       2.0 |        0.0  131.532524       200.0  0.0       200.0 |        0.0    40.11245         0.0  0.0         0.0 | 134.130136  121.240705  0.0  0.0  0.0 | 3.346592 | 20.832316 |  0.55186 |
    | 02/01 07:00 |  0.1 | 5.0 | 17.8 |  1.264612 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047496       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032906   0.03938  0.0  0.0  0.0 | -0.007047 |      0.5 |  0.02929 | 0.0  0.0  0.0       0.0  0.0 |  0.10638 |  0.12272 | 0.130669 | 0.130669 | 0.544454 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.238644       200.0  0.0       200.0 |        0.0    40.21501         0.0  0.0         0.0 | 134.221146  121.280084  0.0  0.0  0.0 | 2.810254 | 21.169593 | 0.544454 |
    | 02/01 08:00 |  0.0 | 5.0 | 17.8 |  3.045538 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032889   0.03936  0.0  0.0  0.0 | -0.026824 |      0.5 | 0.016481 | 0.0  0.0  0.0       0.0  0.0 | 0.107991 | 0.113976 | 0.122937 | 0.122937 | 0.512237 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.278644       200.0  0.0       200.0 |        0.0    40.17501         0.0  0.0         0.0 | 134.254036  121.319444  0.0  0.0  0.0 | 2.266949 | 21.490174 | 0.512237 |
    | 02/01 09:00 |  0.0 | 5.0 | 18.0 |  1.930758 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032873   0.03934  0.0  0.0  0.0 | -0.026811 |      0.5 | 0.008099 | 0.0  0.0  0.0       0.0  0.0 | 0.109594 | 0.108767 | 0.114762 | 0.114762 | 0.478174 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.318644       200.0  0.0       200.0 |        0.0    40.13501         0.0  0.0         0.0 | 134.286909  121.358784  0.0  0.0  0.0 | 1.732039 | 21.809151 | 0.478174 |
    | 02/01 10:00 |  0.4 | 5.0 | 18.2 |  2.461001 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  175.688374       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 |        0.0         0.0      2.3712  0.0      1.7472 |       0.0        0.0        0.0  0.0        0.0 |        0.0      2.2152         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.4224        0.0  0.0   0.4224  0.0 | 0.190428       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032741  0.039321  0.0  0.0  0.0 |  0.052596 |      0.5 | 0.003479 | 0.0  0.0  0.0       0.0  0.0 |  0.11149 | 0.106841 | 0.109497 | 0.109497 | 0.456236 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.023124       200.0  0.0       200.0 |        0.0    40.66525         0.0  0.0         0.0 | 134.551621  121.398105  0.0  0.0  0.0 | 1.281156 | 22.186575 | 0.456236 |
    | 02/01 11:00 |  0.1 | 5.0 | 18.3 |  6.215945 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  176.497054       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047795       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032695  0.039301  0.0  0.0  0.0 | -0.006809 |      0.5 | 0.001141 | 0.0  0.0  0.0       0.0  0.0 | 0.113151 | 0.106521 | 0.107198 | 0.107198 | 0.446658 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.729244       200.0  0.0       200.0 |        0.0    40.76781         0.0  0.0         0.0 | 134.642122  121.437406  0.0  0.0  0.0 | 0.773206 | 22.517081 | 0.446658 |
    | 02/01 12:00 |  3.6 | 5.0 | 18.0 |  3.374783 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   2.998287       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |        0.0         0.0     21.3408  0.0     15.7248 |       0.0        0.0        0.0  0.0        0.0 |   4.487627   14.327266         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  5.29094        0.0  0.0   3.8016  0.0 |  2.39792       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031232  0.039281  0.0  0.0  0.0 |  0.973026 |      0.5 | 0.001092 | 0.0  0.0  0.0       0.0  0.0 | 0.117444 | 0.110488 | 0.107474 | 0.107474 | 0.447806 |      2.0       2.0  0.0  0.0       2.0 |   2.141634   154.14003       200.0  0.0       200.0 |   0.856653    45.85997         0.0  0.0         0.0 | 137.566374  121.476687  0.0  0.0  0.0 | 1.245141 | 23.371295 | 0.447806 |
    | 02/01 13:00 |  5.9 | 5.0 | 17.8 |  8.821555 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  47.398383       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.2304  15.03792   34.9752   6.2304   25.7712 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.2304  15.03792   34.9752   6.2304   25.7712 |        0.0   11.242668     34.9752  0.0     25.7712 |       0.0   3.795252        0.0  0.0        0.0 |  35.133494         0.0         0.0  0.0         0.0 |  3.036202        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0   6.2304  0.0 |      0.0       0.0  0.0   6.2304  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031217  0.039262  0.0  0.0  0.0 | -0.026094 |      0.5 | 0.001006 | 0.0  0.0  0.0       0.0  0.0 |  0.12345 | 0.116024 | 0.110836 | 0.110836 | 0.461818 |      2.0       2.0  0.0  0.0       2.0 |  34.929135  149.564242       200.0  0.0       200.0 |  12.469248   50.435758         0.0  0.0         0.0 |  137.59759  121.515949  0.0  0.0  0.0 | 0.718041 | 24.566474 | 0.461818 |
    | 02/01 14:00 |  1.1 | 5.0 | 17.7 |  4.046025 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   54.77413       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.1616   2.80368    6.5208   1.1616    4.8048 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.1616   2.80368    6.5208   1.1616    4.8048 |        0.0    2.084747      6.5208  0.0      4.8048 |       0.0   0.718933        0.0  0.0        0.0 |   6.541237         0.0         0.0  0.0         0.0 |  0.575147        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 | 0.902237        0.0  0.0   1.1616  0.0 | 0.427054       0.0  0.0   1.1616  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030964  0.039242  0.0  0.0  0.0 |  0.151957 |      0.5 | 0.000176 | 0.0  0.0  0.0       0.0  0.0 | 0.125805 | 0.117559 | 0.115135 | 0.115135 | 0.479729 |      2.0       2.0  0.0  0.0       2.0 |  39.124378  148.755016       200.0  0.0       200.0 |  15.649751   51.244984         0.0  0.0         0.0 | 138.103738  121.555191  0.0  0.0  0.0 | 0.369822 | 25.035183 | 0.479729 |
    | 02/01 15:00 | 20.7 | 5.0 | 16.8 |  2.110757 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 204.057976  195.707265  204.057976  0.0  204.057976 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 21.8592  52.76016  122.7096  21.8592   90.4176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  21.8592  52.76016  122.7096  21.8592   90.4176 |        0.0   35.903699    122.7096  0.0     90.4176 |       0.0  16.856461        0.0  0.0        0.0 | 114.593284    3.649347    3.649347  0.0    3.649347 | 12.831363   0.408629   0.408629  0.0   0.408629 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.408629  0.0  0.34 |      0.0   8.350711  0.0  21.8592  0.0 |      0.0  3.084683  0.0  21.8592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030948  0.036589  0.0  0.0  0.0 |  1.003136 |      0.5 | 0.000282 | 0.0  0.0  0.0       0.0  0.0 | 0.142932 | 0.133629 | 0.120789 | 0.120789 | 0.503287 |      2.0       2.0  0.0  0.0       2.0 | 151.371669  139.790904  204.057976  0.0  203.989347 |  52.686307   55.916362         0.0  0.0    0.068629 | 138.134686  126.857809  0.0  0.0  0.0 | 0.872675 | 28.443565 | 0.503287 |
    | 02/01 16:00 | 37.9 | 5.0 | 16.4 |  2.239257 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 302.049358  306.424159  311.186287  0.0  311.186287 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 40.0224  96.59952  224.6712  40.0224  165.5472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  40.0224  96.59952  224.6712  40.0224  165.5472 |  27.337389   57.587039  228.729176  0.0  169.573684 | 16.742987  34.719746        0.0  0.0   0.031492 |  94.061255   94.061255   94.061255  0.0   94.061255 | 17.125032  17.125032  17.125032  0.0  17.125032 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 9.136928   4.762128  0.0  40.0224  0.0 | 4.358587  1.915912  0.0  40.0224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028543  0.035148  0.0  0.0  0.0 |  2.431106 |      0.5 | 0.007994 | 0.0  0.0  0.0       0.0  0.0 | 0.172948 | 0.167814 | 0.137655 | 0.137655 | 0.573561 |      2.0       2.0  0.0  0.0       2.0 | 215.749542  218.874399  294.701255  0.0  294.364119 |  86.299817    87.54976   16.485032  0.0   16.822168 | 142.941571  129.739173  0.0  0.0  0.0 | 2.795787 | 34.416674 | 0.573561 |
    | 02/01 17:00 |  8.2 | 5.0 | 16.3 |  2.877848 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 329.940893  329.940893  329.940893  0.0  329.940893 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  8.6592  20.90016   48.6096   8.6592   35.8176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   8.6592  20.90016   48.6096   8.6592   35.8176 |  76.873713   88.721551  152.474401  0.0  139.877384 | 33.834845  38.602768   7.321485  0.0   7.126503 | 102.811951  102.811951  102.811951  0.0  102.811951 | 27.128942  27.128942  27.128942  0.0  27.128942 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   8.6592  0.0 |      0.0       0.0  0.0   8.6592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028529   0.03513  0.0  0.0  0.0 | -0.023597 |      0.5 | 0.016234 | 0.0  0.0  0.0       0.0  0.0 | 0.180411 | 0.181371 |  0.16323 |  0.16323 | 0.680124 |      2.0       2.0  0.0  0.0       2.0 | 239.341786  242.215039  294.288404  0.0  293.456285 |  90.599108   87.725855   35.652489  0.0   36.484608 |   142.9701  129.774303  0.0  0.0  0.0 | 2.255956 | 35.901863 | 0.680124 |
    | 02/01 18:00 |  3.6 | 5.0 | 15.8 |  1.591452 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 340.360733  340.360733  340.360733  0.0  340.360733 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |   95.91277  101.023216  135.927722  0.0   130.29085 | 37.829723  38.093357  15.353971  0.0  15.374843 | 109.363903  109.363903  109.363903  0.0  109.363903 | 30.996831  30.996831  30.996831  0.0  30.996831 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   3.8016  0.0 |      0.0       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028515  0.035113  0.0  0.0  0.0 | -0.023586 |      0.5 | 0.007986 | 0.0  0.0  0.0       0.0  0.0 | 0.184368 | 0.178465 | 0.177713 | 0.177713 | 0.740469 |      2.0       2.0  0.0  0.0       2.0 | 250.446924  254.639245  289.705384  0.0  288.594137 |  89.913809   85.721488   50.655349  0.0   51.766596 | 142.998615  129.809416  0.0  0.0  0.0 | 1.724384 | 36.689152 | 0.740469 |
    | 02/01 19:00 |  7.5 | 5.0 | 15.5 |  0.291604 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 362.068733  362.068733  362.068733  0.0  362.068733 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    7.92    19.116     44.46     7.92     32.76 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |     7.92    19.116     44.46     7.92     32.76 |  106.62793  116.704262   160.49211  0.0  149.102044 | 41.652803  42.772471  24.328623  0.0  24.018689 | 125.425775  125.425775  125.425775  0.0  125.425775 | 36.642959  36.642959  36.642959  0.0  36.642959 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0     7.92  0.0 |      0.0       0.0  0.0     7.92  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028501  0.035095  0.0  0.0  0.0 | -0.023574 |      0.5 | 0.003201 | 0.0  0.0  0.0       0.0  0.0 | 0.191246 | 0.181057 | 0.179687 | 0.179687 | 0.748696 |      2.0       2.0  0.0  0.0       2.0 | 266.898776  271.824758  299.739049  0.0  298.017868 |  95.169957   90.243976   62.329684  0.0   64.050865 | 143.027116  129.844511  0.0  0.0  0.0 | 1.197609 | 38.057906 | 0.748696 |
    | 02/01 20:00 | 18.5 | 5.0 | 15.3 |  0.092622 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 415.615133  415.615133  415.615133  0.0  415.615133 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  19.536   47.1528   109.668   19.536    80.808 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   19.536   47.1528   109.668   19.536    80.808 | 127.016457  149.598796  235.832672  0.0  207.750786 | 54.588277  59.622737  35.904062  0.0  35.125947 | 165.286636  165.286636  165.286636  0.0  165.286636 | 50.328497  50.328497  50.328497  0.0  50.328497 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   19.536  0.0 |      0.0       0.0  0.0   19.536  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028486  0.035078  0.0  0.0  0.0 | -0.023562 |      0.5 | 0.000875 | 0.0  0.0  0.0       0.0  0.0 | 0.206387 | 0.193328 | 0.183208 | 0.183208 | 0.763366 |      2.0       2.0  0.0  0.0       2.0 | 302.822962  308.331798  339.501014  0.0  336.701718 | 112.792171  107.283336    76.11412  0.0   78.913415 | 143.055602  129.879589  0.0  0.0  0.0 | 0.673172 | 41.070948 | 0.763366 |
    | 02/01 21:00 | 15.4 | 5.0 | 15.2 |  0.092451 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 460.188893  460.188893  460.188893  0.0  460.188893 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 16.2624  39.25152   91.2912  16.2624   67.2672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  16.2624  39.25152   91.2912  16.2624   67.2672 | 162.587391  182.453457  260.823055  0.0  236.653243 | 69.290142  72.413197  46.083278  0.0   46.22909 | 197.067169  197.067169  197.067169  0.0  197.067169 | 63.121725  63.121725  63.121725  0.0  63.121725 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0  16.2624  0.0 |      0.0       0.0  0.0  16.2624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028472   0.03506  0.0  0.0  0.0 |  -0.02355 |      0.5 | 0.000096 | 0.0  0.0  0.0       0.0  0.0 | 0.219114 | 0.204583 | 0.193102 | 0.193102 | 0.804592 |      2.0       2.0  0.0  0.0       2.0 | 334.956746   340.28279  367.676327  0.0  364.722844 | 125.232147  119.906104   92.512567  0.0   95.466049 | 143.084074  129.914649  0.0  0.0  0.0 | 0.149526 | 43.603606 | 0.804592 |
    | 02/01 22:00 |  6.3 | 5.0 | 14.5 |  0.091248 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.423613  478.423613  478.423613  0.0  478.423613 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.6528  16.05744   37.3464   6.6528   27.5184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.6528  16.05744   37.3464   6.6528   27.5184 | 191.457675  201.485302   242.21107  0.0  231.390156 | 75.384018  74.761032  55.324223  0.0  56.317137 |  209.32715   209.32715   209.32715  0.0   209.32715 | 69.096463  69.096463  69.096463  0.0  69.096463 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   6.6528  0.0 |      0.0       0.0  0.0   6.6528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028458  0.035043  0.0  0.0  0.0 | -0.023538 | 0.125987 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22331 | 0.208423 | 0.202935 | 0.202935 | 0.845563 |      2.0       2.0  0.0  0.0       2.0 | 350.480227  355.240798  372.778807  0.0  370.518238 | 127.943386  123.182816  105.644807  0.0  107.905375 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.438685 | 0.845563 |
    | 02/01 23:00 |  1.9 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 483.922973  483.922973  483.922973  0.0  483.922973 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.0064   4.84272   11.2632   2.0064    8.2992 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.0064   4.84272   11.2632   2.0064    8.2992 | 204.577508  209.475116  227.189914  0.0  223.156997 | 75.852505  73.791217  62.496899  0.0  63.565816 | 212.731903  212.731903  212.731903  0.0  212.731903 |  71.19107   71.19107   71.19107  0.0   71.19107 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.0064  0.0 |      0.0       0.0  0.0   2.0064  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.223627 | 0.208718 | 0.207635 | 0.207635 | 0.865146 |      2.0       2.0  0.0  0.0       2.0 | 356.288629  360.671665  370.223996  0.0  368.732344 | 127.634344  123.251308  113.698978  0.0  115.190629 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.501687 | 0.865146 |
    | 03/01 00:00 |  4.9 | 5.0 | 13.7 |  0.089858 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 498.105533  498.105533  498.105533  0.0  498.105533 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |  210.59632  218.646568   243.60086  0.0  235.726149 | 78.501053  77.765525  69.369313  0.0  69.600025 | 222.622815  222.622815  222.622815  0.0  222.622815 | 75.482718  75.482718  75.482718  0.0  75.482718 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   5.1744  0.0 |      0.0       0.0  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226204 | 0.211124 | 0.209187 | 0.209187 | 0.871613 |      2.0       2.0  0.0  0.0       2.0 | 365.968141  370.191592  378.933151  0.0  377.372211 | 132.137392  127.913941  119.172382  0.0  120.733323 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.014682 | 0.871613 |
    | 03/01 01:00 |  2.7 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 505.920413  505.920413  505.920413  0.0  505.920413 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.8512   6.88176   16.0056   2.8512   11.7936 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.8512   6.88176   16.0056   2.8512   11.7936 | 219.860457  225.408936  241.299304  0.0  236.521578 | 81.096276  79.578357  72.811829  0.0  73.377556 | 227.796732  227.796732  227.796732  0.0  227.796732 | 78.123681  78.123681  78.123681  0.0  78.123681 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.8512  0.0 |      0.0       0.0  0.0   2.8512  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22711 | 0.211969 | 0.210777 | 0.210777 | 0.878239 |      2.0       2.0  0.0  0.0       2.0 | 371.557433  375.652029  382.076179  0.0  380.780965 |  134.36298  130.268385  123.844234  0.0  125.139448 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.194887 | 0.878239 |
    | 03/01 02:00 |  0.5 | 5.0 | 13.1 |  0.088805 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.367613  507.367613  507.367613  0.0  507.367613 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 224.826819  227.862889  233.713014  0.0   232.22234 | 81.621594  79.331924    75.1714  0.0  75.882073 | 228.380541  228.380541  228.380541  0.0  228.380541 | 78.987072  78.987072  78.987072  0.0  78.987072 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226352 | 0.211261 | 0.211624 | 0.211624 | 0.881768 |      2.0       2.0  0.0  0.0       2.0 | 372.764171  376.771281  380.347707  0.0  379.463166 | 134.603442  130.596333  127.019907  0.0  127.904447 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.043964 | 0.881768 |
    | 03/01 03:00 |  0.2 | 5.0 | 13.3 |  0.089157 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.946493  507.946493  507.946493  0.0  507.946493 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 225.884845  228.536493  231.486744  0.0  230.668948 | 81.693969   79.34088  77.066469  0.0  77.572265 | 228.361491  228.361491  228.361491  0.0  228.361491 | 79.585003  79.585003  79.585003  0.0  79.585003 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225371 | 0.210346 | 0.211215 | 0.211215 | 0.880064 |      2.0       2.0  0.0  0.0       2.0 | 372.893834  376.860918  379.048053  0.0  378.369308 |  135.05266  131.085575  128.898441  0.0  129.577185 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.848765 | 0.880064 |
    | 03/01 04:00 |  0.5 | 5.0 | 13.9 |  0.090207 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 509.393693  509.393693  509.393693  0.0  509.393693 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 226.222235  229.187234  232.470378  0.0  231.354892 | 82.252258  80.033659  78.440115  0.0  78.775601 | 228.971242  228.971242  228.971242  0.0  228.971242 | 80.422452  80.422452  80.422452  0.0  80.422452 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224621 | 0.209646 | 0.210394 | 0.210394 | 0.876641 |      2.0       2.0  0.0  0.0       2.0 | 373.295856  377.246525  379.152916  0.0  378.509658 | 136.097837  132.147168  130.240778  0.0  130.884036 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.699572 | 0.876641 |
    | 03/01 05:00 |  2.4 | 5.0 | 14.7 |  0.091593 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.340253  516.340253  516.340253  0.0  516.340253 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.5344   6.11712   14.2272   2.5344   10.4832 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.5344   6.11712   14.2272   2.5344   10.4832 |  227.45668   232.53804  243.126327  0.0  239.344792 | 84.471413  82.972773  80.494566  0.0  80.532101 | 233.550718  233.550718  233.550718  0.0  233.550718 | 82.789535  82.789535  82.789535  0.0  82.789535 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.5344  0.0 |      0.0       0.0  0.0   2.5344  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225308 | 0.210288 | 0.209944 | 0.209944 | 0.874768 |      2.0       2.0  0.0  0.0       2.0 |  377.04291  380.994883  384.444507  0.0  383.538784 | 139.297343   135.34537  131.895746  0.0  132.801469 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.836321 | 0.874768 |
    | 03/01 06:00 |  0.4 | 5.0 | 15.0 |  0.154861 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.498013  517.498013  517.498013  0.0  517.498013 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 231.117926  233.986173  237.670858  0.0  236.552027 | 85.644727  83.373601  81.040595  0.0  81.535426 | 233.930686  233.930686  233.930686  0.0  233.930686 | 83.567327  83.567327  83.567327  0.0  83.567327 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.4224  0.0 |      0.0       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224483 | 0.209518 | 0.209974 | 0.209974 | 0.874892 |      2.0       2.0  0.0  0.0       2.0 | 377.508687  381.428677  383.715535  0.0  383.004643 | 139.989326  136.069336  133.782478  0.0   134.49337 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.672181 | 0.874892 |
    | 03/01 07:00 |  0.2 | 5.0 | 15.7 |  0.470369 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 231.670569  234.299035  236.486524  0.0  235.768959 | 86.038644  83.708738   82.19709  0.0  82.602655 | 233.905963  233.905963  233.905963  0.0  233.905963 | 84.170931  84.170931  84.170931  0.0  84.170931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.223512 | 0.208611 | 0.209487 | 0.209487 | 0.872864 |      2.0       2.0  0.0  0.0       2.0 | 377.397097  381.300245  382.960574  0.0  382.355247 | 140.679796  136.776649  135.116319  0.0  135.721646 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 |  44.47884 | 0.872864 |
    | 03/01 08:00 |  0.0 | 5.0 | 16.1 |  1.173726 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.705559  234.101923  235.121294  0.0  234.749649 | 86.371334   83.97497  82.955599  0.0  83.327244 | 233.465309  233.465309  233.465309  0.0  233.465309 | 84.611584  84.611584  84.611584  0.0  84.611584 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.222394 | 0.207568 | 0.208581 | 0.208581 | 0.869086 |      2.0       2.0  0.0  0.0       2.0 | 376.809864  380.703631  381.944589  0.0  381.410907 | 141.267029  137.373263  136.132304  0.0  136.665986 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.256446 | 0.869086 |
    | 03/01 09:00 |  0.0 | 5.0 | 15.2 |  4.202296 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.345023  233.735628  234.497523  0.0  234.169865 |  86.73187  84.341265   83.57937  0.0  83.907028 |  233.02539   233.02539   233.02539  0.0   233.02539 | 85.051504  85.051504  85.051504  0.0  85.051504 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.221282 |  0.20653 | 0.207569 | 0.207569 | 0.864871 |      2.0       2.0  0.0  0.0       2.0 | 376.143247  380.033392  381.112456  0.0  380.606431 | 141.933647  138.043501  136.964438  0.0  137.470462 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.035164 | 0.864871 |
    | 03/01 10:00 |  0.3 | 5.0 | 14.9 |  4.359715 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.945213  518.945213  518.945213  0.0  518.945213 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 231.024488  233.747413  235.584149  0.0  234.852459 | 87.369205   85.09412  84.271144  0.0  84.534834 |    233.209     233.209     233.209  0.0     233.209 | 85.736213  85.736213  85.736213  0.0  85.736213 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220402 | 0.205709 | 0.206578 | 0.206578 | 0.860742 |      2.0       2.0  0.0  0.0       2.0 | 375.980775  379.871939  381.155706  0.0  380.613372 | 142.964438  139.073275  137.789507  0.0  138.331841 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.860019 | 0.860742 |
    | 03/01 11:00 |  2.6 | 5.0 | 15.3 |  5.305753 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.470653  526.470653  526.470653  0.0  526.470653 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.7456   6.62688   15.4128   2.7456   11.3568 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.7456   6.62688   15.4128   2.7456   11.3568 | 231.841463   237.12537  248.140487  0.0  244.141138 |  89.84935  88.446724  86.217526  0.0  86.160875 | 238.174723  238.174723  238.174723  0.0  238.174723 | 88.295931  88.295931  88.295931  0.0  88.295931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.7456  0.0 |      0.0       0.0  0.0   2.7456  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.221261 |  0.20651 | 0.206069 | 0.206069 | 0.858622 |      2.0       2.0  0.0  0.0       2.0 | 379.967051  383.881611  387.242741  0.0  386.343757 | 146.503602  142.589042  139.227912  0.0  140.126897 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.030986 | 0.858622 |
    | 03/01 12:00 |  0.7 | 5.0 | 15.3 |  5.376027 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.496733  528.496733  528.496733  0.0  528.496733 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |  235.82443  239.030617  243.869762  0.0  242.326539 | 91.385423  89.224197  86.750491  0.0  87.201714 | 239.167726  239.167726  239.167726  0.0  239.167726 | 89.329007  89.329007  89.329007  0.0  89.329007 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.7392  0.0 |      0.0       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220683 | 0.205971 | 0.206212 | 0.206212 | 0.859218 |      2.0       2.0  0.0  0.0       2.0 | 380.963364  384.844961  387.330305  0.0  386.582544 | 147.533369  143.651772  141.166428  0.0  141.914189 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.915903 | 0.859218 |
    | 03/01 13:00 |  0.3 | 5.0 | 15.4 |  4.658915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.365053  529.365053  529.365053  0.0  529.365053 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 236.881059  239.627403  242.351416  0.0  241.464963 | 91.932474   89.63397  87.923717  0.0   88.34217 | 239.341086  239.341086  239.341086  0.0  239.341086 | 90.023967  90.023967  90.023967  0.0  90.023967 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.219806 | 0.205152 | 0.205909 | 0.205909 | 0.857953 |      2.0       2.0  0.0  0.0       2.0 | 381.076407  384.935604  386.738375  0.0  386.109067 | 148.288646  144.429449  142.626678  0.0  143.255986 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.741355 | 0.857953 |
    | 03/01 14:00 |  0.3 | 5.0 | 15.5 |  7.789594 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 237.187616  239.922259  242.222296  0.0  241.409523 | 92.494237  90.207435  88.921158  0.0  89.265931 | 239.515656  239.515656  239.515656  0.0  239.515656 | 90.717717  90.717717  90.717717  0.0  90.717717 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.218933 | 0.204338 | 0.205153 | 0.205153 | 0.854804 |      2.0       2.0  0.0  0.0       2.0 | 381.057463  384.905961  386.450135  0.0    385.8656 |  149.17591  145.327412  143.783238  0.0  144.367773 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.567679 | 0.854804 |
    | 03/01 15:00 |  0.0 | 5.0 | 17.4 |  4.851567 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 237.325483  239.722357   240.68408  0.0  240.320027 |  92.90789  90.511016  89.549293  0.0  89.913346 | 239.068393  239.068393  239.068393  0.0  239.068393 |  91.16498   91.16498   91.16498  0.0   91.16498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.217838 | 0.203316 | 0.204291 | 0.204291 | 0.851215 |      2.0       2.0  0.0  0.0       2.0 |  380.45339  384.291998  385.474448  0.0  384.953966 | 149.779984  145.941376  144.758925  0.0  145.279407 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 |  43.34984 | 0.851215 |
    | 03/01 16:00 |  0.0 | 5.0 | 17.3 |   5.30692 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.949262  239.339976  240.076415  0.0  239.752255 | 93.284112  90.893398  90.156958  0.0  90.481119 | 238.622132  238.622132  238.622132  0.0  238.622132 | 91.611242  91.611242  91.611242  0.0  91.611242 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.216749 | 0.202299 | 0.203317 | 0.203317 | 0.847154 |      2.0       2.0  0.0  0.0       2.0 | 379.779276  383.614153  384.660165  0.0  384.163843 | 150.454097   146.61922  145.573208  0.0   146.06953 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.133091 | 0.847154 |
    | 03/01 17:00 |  0.0 | 5.0 | 17.1 |  3.286036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.529419  238.917809  239.569273  0.0   239.26016 | 93.703955  91.315564    90.6641  0.0  90.973213 |  238.17587   238.17587   238.17587  0.0   238.17587 | 92.057503  92.057503  92.057503  0.0  92.057503 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215665 | 0.201288 |   0.2023 |   0.2023 | 0.842918 |      2.0       2.0  0.0  0.0       2.0 | 379.078744  382.912214  383.906762  0.0  383.419553 | 151.154629  147.321159  146.326612  0.0   146.81382 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 42.917426 | 0.842918 |
    | 03/01 18:00 |  0.0 | 5.0 | 15.7 |  1.506216 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.715546  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.093122  238.480636  239.100048  0.0  238.796611 | 94.140252  91.752737  91.133325  0.0  91.436763 | 237.729608  237.729608  237.729608  0.0  237.729608 | 92.503765  92.503765  92.503765  0.0  92.503765 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 0.517828        0.0  0.0      0.0  0.0 | 0.265143       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028317       0.0  0.0  0.0  0.0 |  0.098677 | 0.098677 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.21501 | 0.200676 | 0.201377 | 0.201377 | 0.839069 |      2.0       2.0  0.0  0.0       2.0 | 378.368247  382.201186  383.176322  0.0  382.692551 | 151.347299  148.032187  147.057051  0.0  147.540823 | 143.393534  129.949692  0.0  0.0  0.0 |      0.0 | 42.786996 | 0.839069 |
    | 03/01 19:00 |  0.0 | 5.0 | 14.2 |  0.274762 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.835785  530.034209  530.034209  0.0  530.034209 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 235.511104  238.037803  238.645125  0.0  238.343828 | 94.204442  92.195571  91.588249  0.0  91.889545 | 237.229687  237.229687  237.229687  0.0  237.229687 | 92.804522  92.804522  92.804522  0.0  92.804522 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.198424        0.0  0.0      0.0  0.0 | 0.616041       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028012       0.0  0.0  0.0  0.0 |  0.245012 | 0.245012 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.214985 | 0.200653 | 0.200807 | 0.200807 | 0.836695 |      2.0       2.0  0.0  0.0       2.0 | 377.739846  381.433071  382.400885  0.0   381.91841 | 151.095939  148.601138  147.633324  0.0  148.115799 | 144.003929  129.949692  0.0  0.0  0.0 |      0.0 | 42.782022 | 0.836695 |
    | 03/01 20:00 |  0.0 | 5.0 | 12.4 |  0.087565 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  528.03676  529.573277  529.573277  0.0  529.573277 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.882703  237.505353  238.107977  0.0  237.807557 | 93.953081  92.528856  91.926231  0.0  92.226652 | 236.658955  236.658955  236.658955  0.0  236.658955 | 92.914321  92.914321  92.914321  0.0  92.914321 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.536516        0.0  0.0      0.0  0.0 | 0.796574       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.027628       0.0  0.0  0.0  0.0 |  0.320394 | 0.320394 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215283 | 0.200931 |  0.20072 |  0.20072 | 0.836332 |      2.0       2.0  0.0  0.0       2.0 | 377.169114  380.626673  381.591862  0.0  381.109808 | 150.867646  148.946603  147.981414  0.0  148.463468 |   144.7715  129.949692  0.0  0.0  0.0 |      0.0 | 42.841362 | 0.836332 |
    | 03/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 527.188356  528.982309  528.982309  0.0  528.982309 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.311972  236.878229  237.478902  0.0  237.178902 | 93.724789  92.695048  92.094374  0.0  92.394375 | 236.052952  236.052952  236.052952  0.0  236.052952 | 92.929356  92.929356  92.929356  0.0  92.929356 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.793952        0.0  0.0      0.0  0.0 | 0.939977       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.027187       0.0  0.0  0.0  0.0 |  0.380329 | 0.380329 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215837 | 0.201448 | 0.200984 | 0.200984 | 0.837433 |      2.0       2.0  0.0  0.0       2.0 | 376.563112  379.841397  380.805913  0.0  380.323859 | 150.625245  149.140912  148.176396  0.0   148.65845 | 145.652663  129.949692  0.0  0.0  0.0 |      0.0 | 42.951521 | 0.837433 |
    | 03/01 22:00 |  0.0 | 5.0 | 10.6 |  0.084317 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.302602  528.292327  528.292327  0.0  528.292327 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 233.705969  236.229261  236.829108  0.0  236.529311 | 93.482388  92.753048  92.153201  0.0  92.452998 | 235.420271  235.420271  235.420271  0.0  235.420271 | 92.872056  92.872056  92.872056  0.0  92.872056 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.989725        0.0  0.0      0.0  0.0 | 1.055285       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026706       0.0  0.0  0.0  0.0 |  0.428575 | 0.428575 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.216594 | 0.202155 |  0.20149 |  0.20149 | 0.839542 |      2.0       2.0  0.0  0.0       2.0 |  375.93043  379.072407  380.037076  0.0  379.554818 | 150.372172   149.21992  148.255252  0.0  148.737509 | 146.613809  129.949692  0.0  0.0  0.0 |      0.0 | 43.102276 | 0.839542 |
    | 03/01 23:00 |  0.0 | 5.0 | 10.0 |  0.083215 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.388479  527.527048  527.527048  0.0  527.527048 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.001664  0.001664  0.001664  0.001664  0.001664 | 0.001065  0.002397  0.005059  0.001065  0.003728 | 0.001065  0.002397  0.0  0.0  0.003728 |      0.0       0.0       0.0      0.0       0.0 | 233.073287  235.563828  236.163294  0.0  235.863608 | 93.229315  92.728499  92.129034  0.0  92.428719 | 234.767326  234.767326  234.767326  0.0  234.767326 | 92.759722  92.759722  92.759722  0.0  92.759722 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.138569        0.0  0.0      0.0  0.0 | 1.149246       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026198       0.0  0.0  0.0  0.0 |  0.467937 | 0.467937 |      0.0 | 0.0  0.0  0.0  0.001065  0.0 | 0.217516 | 0.203015 | 0.202189 | 0.202189 | 0.842453 | 1.998935  1.997603  0.0  0.0  1.996272 | 375.277485  378.315905  379.281108  0.0  378.798536 | 150.110994  149.211143   148.24594  0.0  148.728512 |  147.62933  129.949692  0.0  0.0  0.0 |      0.0 | 43.285697 | 0.842453 |
    | 04/01 00:00 |  0.0 | 5.0 |  9.5 |  0.082289 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.451307  526.704522  526.704522  0.0  526.704522 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |  0.00576   0.00576   0.00576   0.00576   0.00576 | 0.003687  0.008295  0.017511  0.003687  0.012903 | 0.003687  0.008295  0.0  0.0  0.012903 |      0.0       0.0       0.0      0.0       0.0 | 232.420342   234.88595  235.485218  0.0  235.185602 | 92.968137  92.641098  92.041831  0.0  92.341446 | 234.098982  234.098982  234.098982  0.0  234.098982 |  92.60554   92.60554   92.60554  0.0   92.60554 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.253215        0.0  0.0      0.0  0.0 | 1.227688       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025673       0.0  0.0  0.0  0.0 |   0.50084 |      0.5 |      0.0 | 0.0  0.0  0.0  0.003687  0.0 | 0.218569 | 0.203997 | 0.203042 | 0.203042 | 0.846009 | 1.995248  1.989309  0.0  0.0  1.983369 | 374.608076  377.568937  378.534872  0.0  378.051916 | 149.843231  149.135584  148.169649  0.0  148.652606 |  148.68053  129.949692  0.0  0.0  0.0 |  0.00084 | 43.495173 | 0.846009 |
    | 04/01 01:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.498661  525.837901  525.837901  0.0  525.837901 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.750934  234.198633  234.797784  0.0  234.498215 | 92.700373  92.505888  91.906738  0.0  92.206306 | 233.418521  233.418521  233.418521  0.0  233.418521 |  92.41938   92.41938   92.41938  0.0   92.41938 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.33924        0.0  0.0      0.0  0.0 | 1.292775       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025137   0.00084  0.0  0.0  0.0 |  0.527903 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.219619 | 0.204977 | 0.203997 | 0.203997 | 0.849987 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.927615  376.828825  377.795609  0.0  377.312221 | 149.571046  149.009076  148.042292  0.0   148.52568 | 149.752131  129.950532  0.0  0.0  0.0 | 0.028742 | 43.704126 | 0.849987 |
    | 04/01 02:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.533546  524.938193  524.938193  0.0  524.938193 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.070472  233.503734  234.102806  0.0  233.803273 | 92.428189  92.334167  91.735094  0.0  92.034628 | 232.729153  232.729153  232.729153  0.0  232.729153 |  92.20904   92.20904   92.20904  0.0   92.20904 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.404647        0.0  0.0      0.0  0.0 | 1.348147       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024596  0.028742  0.0  0.0  0.0 |  0.541899 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220663 | 0.205953 | 0.204976 | 0.204976 | 0.854068 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.238247  376.094244  377.061956  0.0  376.578101 | 149.295299  148.843949  147.876237  0.0  148.360092 | 150.833226  129.979274  0.0  0.0  0.0 | 0.070641 | 43.912033 | 0.854068 |
    | 04/01 03:00 |  1.3 | 5.0 | 10.2 |  0.083584 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.784249  527.769652  527.769652  0.0  527.769652 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 1.368048  3.302749    7.7064   1.3728  5.661769 | 230.754143  234.603676  240.293649  0.0  238.161507 | 93.147452  93.637266  92.350944  0.0  92.438455 | 234.709836  234.709836  234.709836  0.0  234.709836 | 93.059816  93.059816  93.059816  0.0  93.059816 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.985403        0.0  0.0   1.3728  0.0 | 1.697997       0.0  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02394   0.03501  0.0  0.0  0.0 |  0.685854 |      0.5 | 0.000006 | 0.0  0.0  0.0       0.0  0.0 | 0.222684 | 0.207843 | 0.206156 | 0.206156 | 0.858983 |      2.0       2.0  0.0  0.0       2.0 | 374.845892  377.695853  379.824543  0.0    379.1282 | 149.938357  150.073799   147.94511  0.0  148.641453 | 152.144572  130.014284  0.0  0.0  0.0 | 0.256489 | 44.314036 | 0.858983 |
    | 04/01 04:00 |  0.0 | 5.0 | 10.1 |    0.0834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.240982   526.62142   526.62142  0.0   526.62142 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.000834  0.000834  0.000834  0.000834  0.000834 | 0.000534  0.001201  0.002535  0.000534  0.001868 | 0.000534  0.001201  0.0  0.0  0.001868 |      0.0       0.0       0.0      0.0       0.0 | 231.988749    234.5668  235.888816  0.0  235.456354 |   92.7955  93.202852  91.880837  0.0  92.313298 |  233.94875   233.94875   233.94875  0.0   233.94875 | 92.672671  92.672671  92.672671  0.0  92.672671 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.380438        0.0  0.0      0.0  0.0 | 1.377558       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023426  0.034993  0.0  0.0  0.0 |  0.552557 |      0.5 | 0.000023 | 0.0  0.0  0.0  0.000534  0.0 | 0.223713 | 0.208817 | 0.207639 | 0.207639 | 0.865163 | 1.999466  1.998799  0.0  0.0  1.998132 | 374.457844  377.117803  378.524477  0.0  377.960595 | 149.783138  149.503617  148.096944  0.0  148.660825 | 153.170878  130.049277  0.0  0.0  0.0 | 0.309024 | 44.518818 | 0.865163 |
    | 04/01 05:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.269541  525.705867  525.705867  0.0  525.705867 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.600701  233.896206  234.768654  0.0  234.418924 | 92.640281  92.725214  91.852766  0.0  92.202497 | 233.254863  233.254863  233.254863  0.0  233.254863 | 92.451004  92.451004  92.451004  0.0  92.451004 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.436327        0.0  0.0      0.0  0.0 | 1.428986       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022911  0.034975  0.0  0.0  0.0 |  0.574206 |      0.5 | 0.000042 | 0.0  0.0  0.0       0.0  0.0 | 0.224737 | 0.209788 | 0.208816 | 0.208816 | 0.870067 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.763958   376.51646  377.650685  0.0  377.136535 | 149.505583  149.189408  148.055182  0.0  148.569332 |  154.20113  130.084253  0.0  0.0  0.0 | 0.383188 | 44.722653 | 0.870067 |
    | 04/01 06:00 |  0.0 | 5.0 | 11.2 |  0.310229 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.290279  524.768818  524.768818  0.0  524.768818 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.906815  233.274208  233.976928  0.0  233.658382 | 92.362726  92.431659  91.728939  0.0  92.047486 |  232.55539   232.55539   232.55539  0.0   232.55539 | 92.213428  92.213428  92.213428  0.0  92.213428 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.47854        0.0  0.0      0.0  0.0 | 1.473367       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022397  0.034958  0.0  0.0  0.0 |  0.592918 |      0.5 |  0.00008 | 0.0  0.0  0.0       0.0  0.0 | 0.225756 |  0.21077 |  0.20979 |  0.20979 | 0.874127 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.064485  375.837642  376.869147  0.0  376.373543 | 149.225794  148.931177  147.899671  0.0  148.395275 | 155.228699   130.11921  0.0  0.0  0.0 | 0.476026 | 44.925468 | 0.874127 |
    | 04/01 07:00 |  0.7 | 5.0 | 10.8 |  1.391958 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.041495  525.840895  525.840895  0.0  525.840895 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 0.738666  1.782959    4.1496   0.7392  3.055732 | 230.409097   233.57062  236.944072  0.0  235.658291 | 92.619848  92.981158  91.974347  0.0  92.166259 | 233.293729  233.293729  233.293729  0.0  233.293729 | 92.547167  92.547167  92.547167  0.0  92.547167 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |   2.7994        0.0  0.0   0.7392  0.0 | 1.686355       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021829   0.03494  0.0  0.0  0.0 |  0.681906 |      0.5 | 0.000188 | 0.0  0.0  0.0       0.0  0.0 | 0.227298 | 0.212295 | 0.210891 | 0.210891 | 0.878712 |      2.0       2.0  0.0  0.0       2.0 | 373.601068  376.386461  378.008404  0.0  377.404713 | 149.440427  149.454435  147.832491  0.0  148.436183 | 156.363574  130.154151  0.0  0.0  0.0 | 0.657744 | 45.232341 | 0.878712 |
    | 04/01 08:00 |  0.4 | 5.0 | 10.4 |  3.195876 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.281753  525.921963  525.921963  0.0  525.921963 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.859201  233.786185  236.354275  0.0  235.421637 | 92.604694   93.07423   91.85782  0.0  92.166459 | 233.378862  233.378862  233.378862  0.0  233.378862 | 92.543101  92.543101  92.543101  0.0  92.543101 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.64021        0.0  0.0   0.4224  0.0 |   1.6138       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021305  0.034923  0.0  0.0  0.0 |  0.651899 |      0.5 |   0.0004 | 0.0  0.0  0.0       0.0  0.0 | 0.228606 | 0.213686 | 0.212265 | 0.212265 | 0.884439 |      2.0       2.0  0.0  0.0       2.0 | 373.772681  376.468418  378.044191  0.0  377.449138 | 149.509072  149.453545  147.877772  0.0  148.472825 | 157.411289  130.189074  0.0  0.0  0.0 | 0.809242 | 45.492649 | 0.884439 |
    | 04/01 09:00 |  0.1 | 5.0 | 10.6 |  5.191651 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.680667  525.195938  525.195938  0.0  525.195938 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 | 230.944361  233.442269  234.809309  0.0  234.301005 | 92.442992  92.734574  91.705454  0.0  92.057758 | 232.863063  232.863063  232.863063  0.0  232.863063 | 92.332875  92.332875  92.332875  0.0  92.332875 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.51527        0.0  0.0   0.1056  0.0 | 1.558104       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020816  0.034905  0.0  0.0  0.0 |  0.628902 |      0.5 | 0.000671 | 0.0  0.0  0.0       0.0  0.0 | 0.229682 | 0.214906 | 0.213648 | 0.213648 |   0.8902 |      2.0       2.0  0.0  0.0       2.0 | 373.343334  376.041531  377.330745  0.0  376.787995 | 149.337334  149.154406  147.865192  0.0  148.407942 | 158.389271  130.223979  0.0  0.0  0.0 | 0.937472 | 45.706625 |   0.8902 |
    | 04/01 10:00 |  0.4 | 5.0 | 12.0 |  7.155036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  522.72229  525.386286  525.386286  0.0  525.386286 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.601546  233.397025  235.757443  0.0  234.863073 | 92.501521  92.818433  91.809695  0.0  92.080065 | 232.979326  232.979326  232.979326  0.0  232.979326 |  92.40696   92.40696   92.40696  0.0   92.40696 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.663996        0.0  0.0   0.4224  0.0 | 1.670802       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020309  0.034888  0.0  0.0  0.0 |  0.676076 |      0.5 | 0.001088 | 0.0  0.0  0.0       0.0  0.0 | 0.230978 | 0.216449 | 0.214978 | 0.214978 | 0.895742 |      2.0       2.0  0.0  0.0       2.0 | 373.373065  376.113112  377.563828  0.0  376.991448 | 149.349226  149.273174  147.822458  0.0  148.394838 | 159.402773  130.258867  0.0  0.0  0.0 | 1.112461 | 45.964561 | 0.895742 |
    | 04/01 11:00 |  0.0 | 5.0 | 16.1 |  8.391432 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 521.869966  524.361672  524.361672  0.0  524.361672 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.515922  232.937273  233.835741  0.0  233.481251 | 92.206369  92.449013  91.550545  0.0  91.905035 | 232.255167  232.255167  232.255167  0.0  232.255167 | 92.106505  92.106505  92.106505  0.0  92.106505 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.491706        0.0  0.0      0.0  0.0 | 1.582809       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019844  0.034871  0.0  0.0  0.0 |  0.639612 |      0.5 | 0.001658 | 0.0  0.0  0.0       0.0  0.0 | 0.231966 | 0.217828 | 0.216413 | 0.216413 |  0.90172 |      2.0       2.0  0.0  0.0       2.0 | 372.764262  375.471006  376.623254  0.0  376.105365 | 149.105705  148.890666  147.738418  0.0  148.256308 | 160.331514  130.293738  0.0  0.0  0.0 | 1.250415 | 46.161167 |  0.90172 |
    | 04/01 12:00 |  0.0 | 5.0 | 17.2 |  8.391286 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.882597  523.403324  523.403324  0.0  523.403324 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.907119  232.260308   232.97307  0.0  232.652712 | 91.962848  92.101364  91.388603  0.0   91.70896 | 231.549904  231.549904  231.549904  0.0  231.549904 |  91.85342   91.85342   91.85342  0.0   91.85342 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.520726        0.0  0.0      0.0  0.0 | 1.619957       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019384  0.034853  0.0  0.0  0.0 |  0.655288 |      0.5 | 0.002348 | 0.0  0.0  0.0       0.0  0.0 | 0.232949 | 0.219298 | 0.217848 | 0.217848 |   0.9077 |      2.0       2.0  0.0  0.0       2.0 | 372.058998  374.800602  375.840088  0.0  375.342557 | 148.823599  148.602722  147.563236  0.0  148.060767 | 161.251667  130.328591  0.0  0.0  0.0 | 1.403354 |  46.35679 |   0.9077 |
    | 04/01 13:00 |  0.0 | 5.0 | 21.4 | 10.715238 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.891085  522.433814  522.433814  0.0  522.433814 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.201855  231.583857   232.22614  0.0  231.918723 | 91.680742  91.819467  91.177184  0.0  91.484601 | 230.841681  230.841681  230.841681  0.0  230.841681 | 91.592133  91.592133  91.592133  0.0  91.592133 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.542729        0.0  0.0      0.0  0.0 | 1.652907       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018929  0.034836  0.0  0.0  0.0 |  0.669212 |      0.5 |   0.0033 | 0.0  0.0  0.0       0.0  0.0 | 0.233927 | 0.220972 | 0.219343 | 0.219343 | 0.913929 |      2.0       2.0  0.0  0.0       2.0 | 371.350775  374.098426  375.095628  0.0  374.605514 |  148.54031  148.335388  147.338185  0.0  147.828299 | 162.160418  130.363427  0.0  0.0  0.0 | 1.569267 | 46.551434 | 0.913929 |
    | 04/01 14:00 |  0.0 | 5.0 | 20.4 |  9.383394 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 518.896368  521.455841  521.455841  0.0  521.455841 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 228.493632  230.884715  231.500165  0.0  231.197678 | 91.397453  91.549099  90.933649  0.0  91.236136 | 230.131169  230.131169  230.131169  0.0  230.131169 | 91.324673  91.324673  91.324673  0.0  91.324673 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.559473        0.0  0.0      0.0  0.0 | 1.682598       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018481  0.034818  0.0  0.0  0.0 |  0.681776 |      0.5 | 0.004578 | 0.0  0.0  0.0       0.0  0.0 |   0.2349 | 0.222903 | 0.221029 | 0.221029 | 0.920953 |      2.0       2.0  0.0  0.0       2.0 | 370.640263  373.384879  374.366632  0.0  373.879005 | 148.256105  148.070962  147.089209  0.0  147.576836 | 163.055775  130.398245  0.0  0.0  0.0 | 1.746464 | 46.745106 | 0.920953 |
    | 04/01 15:00 |  0.0 | 5.0 | 20.8 |  7.861915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 517.899208  520.471428  520.471428  0.0  520.471428 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  227.78312  230.176251  230.781461  0.0  230.480859 | 91.113248   91.27959  90.674381  0.0  90.974982 | 229.418911  229.418911  229.418911  0.0  229.418911 | 91.052517  91.052517  91.052517  0.0  91.052517 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.572221        0.0  0.0      0.0  0.0 | 1.709703       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018041  0.034801  0.0  0.0  0.0 |  0.693259 |      0.5 | 0.006257 | 0.0  0.0  0.0       0.0  0.0 | 0.235868 | 0.225149 | 0.222973 | 0.222973 | 0.929054 |      2.0       2.0  0.0  0.0       2.0 | 369.928006   372.66754  373.644083  0.0  373.157057 | 147.971202  147.803889  146.827346  0.0  147.314371 | 163.936334  130.433046  0.0  0.0  0.0 | 1.933466 | 46.937809 | 0.929054 |
    | 04/01 16:00 |  0.0 | 5.0 | 21.2 |  6.298329 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 516.900183  519.482113  519.482113  0.0  519.482113 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 227.070863  229.463698  230.064988  0.0   229.76511 | 90.828345   91.00773  90.406441  0.0  90.706318 | 228.705322  228.705322  228.705322  0.0  228.705322 | 90.776791  90.776791  90.776791  0.0  90.776791 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58193        0.0  0.0      0.0  0.0 | 1.734742       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017608  0.034783  0.0  0.0  0.0 |  0.703878 |      0.5 | 0.008415 | 0.0  0.0  0.0       0.0  0.0 | 0.236832 | 0.227775 | 0.225234 | 0.225234 | 0.938474 |      2.0       2.0  0.0  0.0       2.0 | 369.214416  371.949163  372.924416  0.0  372.437269 | 147.685766   147.53295  146.557696  0.0  147.044844 |  164.80113  130.467829  0.0  0.0  0.0 | 2.128928 | 47.129548 | 0.938474 |
    | 04/01 17:00 |  0.0 | 5.0 | 19.1 |  2.948416 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 515.899733  518.489063  518.489063  0.0  518.489063 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 226.357273  228.749175  229.348956  0.0   229.04936 | 90.542909  90.732938  90.133156  0.0  90.432753 | 227.990715  227.990715  227.990715  0.0  227.990715 | 90.498348  90.498348  90.498348  0.0  90.498348 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58933        0.0  0.0      0.0  0.0 | 1.758117       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017184  0.034766  0.0  0.0  0.0 |    0.7138 |      0.5 | 0.011137 | 0.0  0.0  0.0       0.0  0.0 | 0.237791 | 0.230847 | 0.227874 | 0.227874 | 0.949477 |      2.0       2.0  0.0  0.0       2.0 | 368.499809  371.230703  372.206175  0.0  371.718624 | 147.399924  147.258359  146.282888  0.0  146.770439 | 165.649527  130.502595  0.0  0.0  0.0 | 2.331592 | 47.320329 | 0.949477 |
    | 04/01 18:00 |  0.0 | 5.0 | 16.8 |  1.309232 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 514.898195  517.493167  517.493167  0.0  517.493167 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 225.642667  228.033583   228.63278  0.0  228.333295 | 90.257067   90.45548  89.856283  0.0  90.155768 | 227.275331  227.275331  227.275331  0.0  227.275331 | 90.217836  90.217836  90.217836  0.0  90.217836 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.594971        0.0  0.0      0.0  0.0 | 1.780135       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016768  0.034749  0.0  0.0  0.0 |  0.723154 |      0.5 | 0.014507 | 0.0  0.0  0.0       0.0  0.0 | 0.238745 | 0.234434 | 0.230962 | 0.230962 | 0.962341 |      2.0       2.0  0.0  0.0       2.0 | 367.784425  370.512451  371.488726  0.0  371.000659 |  147.11377  146.980716  146.004441  0.0  146.492507 | 166.481131  130.537344  0.0  0.0  0.0 | 2.540238 | 47.510156 | 0.962341 |
    | 04/01 19:00 |  0.0 | 5.0 | 14.2 |   0.32955 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 513.895825  516.495101  516.495101  0.0  516.495101 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.927282  227.317342  227.916308  0.0  227.616869 | 89.970913  90.175824  89.576859  0.0  89.876298 | 226.559352  226.559352  226.559352  0.0  226.559352 | 89.935749  89.935749  89.935749  0.0  89.935749 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.599276        0.0  0.0      0.0  0.0 | 1.801036       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01636  0.034731  0.0  0.0  0.0 |  0.732038 |      0.5 |  0.01861 | 0.0  0.0  0.0       0.0  0.0 | 0.239694 | 0.238602 | 0.234563 | 0.234563 | 0.977347 |      2.0       2.0  0.0  0.0       2.0 | 367.068446   369.79446   370.77177  0.0  370.283143 | 146.827379   146.70064  145.723331  0.0  146.211958 | 167.295731  130.572075  0.0  0.0  0.0 | 2.753666 | 47.699034 | 0.977347 |
    | 04/01 20:00 |  0.0 | 5.0 | 13.5 |  0.089508 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.892817  515.495379  515.495379  0.0  515.495379 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.211303  226.600668  227.199539  0.0   226.90012 | 89.684521  89.894432  89.295562  0.0   89.59498 | 225.842917  225.842917  225.842917  0.0  225.842917 | 89.652462  89.652462  89.652462  0.0  89.652462 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.602563        0.0  0.0      0.0  0.0 | 1.821004       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015961  0.034714  0.0  0.0  0.0 |   0.74053 |      0.5 | 0.023524 | 0.0  0.0  0.0       0.0  0.0 | 0.240638 | 0.243415 | 0.238745 | 0.238745 | 0.994772 |      2.0       2.0  0.0  0.0       2.0 | 366.352012   369.07671  370.055148  0.0   369.56594 | 146.540805   146.41867  145.440231  0.0   145.92944 | 168.093251  130.606789  0.0  0.0  0.0 | 2.970672 | 47.886967 | 0.994772 |
    | 04/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.889319  514.494394  514.494394  0.0  514.494394 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 223.494869  225.883686  226.482514  0.0  226.183107 | 89.397948  89.611693  89.012865  0.0  89.312273 | 225.126134  225.126134  225.126134  0.0  225.126134 |  89.36826   89.36826   89.36826  0.0   89.36826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.605074        0.0  0.0      0.0  0.0 | 1.840182       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015571  0.034697  0.0  0.0  0.0 |  0.748689 |      0.5 |  0.02932 | 0.0  0.0  0.0       0.0  0.0 | 0.241578 | 0.248928 |  0.24357 |  0.24357 | 1.014877 |      2.0       2.0  0.0  0.0       2.0 | 365.635228  368.359157  369.338768  0.0  368.848967 | 146.254091  146.135237  145.155625  0.0  145.645427 | 168.873714  130.641486  0.0  0.0  0.0 | 3.190041 | 48.073961 | 1.014877 |
    | 04/01 22:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.885446  513.492442  513.492442  0.0  513.492442 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.778085  225.166476  225.765282  0.0  225.465882 | 89.111234  89.327917  88.729111  0.0  89.028512 | 224.409082  224.409082  224.409082  0.0  224.409082 |  89.08336   89.08336   89.08336  0.0   89.08336 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.606996        0.0  0.0      0.0  0.0 | 1.858679       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015189  0.034679  0.0  0.0  0.0 |  0.756561 |      0.5 | 0.036054 | 0.0  0.0  0.0       0.0  0.0 | 0.242513 | 0.255189 | 0.249094 | 0.249094 | 1.037893 |      2.0       2.0  0.0  0.0       2.0 | 364.918176  367.641762  368.622568  0.0  368.132167 |  145.96727   145.85068  144.869874  0.0  145.360275 |  169.63722  130.676165  0.0  0.0  0.0 | 3.410548 |  48.26002 | 1.037893 |
    | 04/01 23:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.881284  512.489751  512.489751  0.0  512.489751 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.061033  224.449095  225.047887  0.0  224.748492 | 88.824413  89.043347  88.444555  0.0   88.74395 | 223.691823  223.691823  223.691823  0.0  223.691823 | 88.797929  88.797929  88.797929  0.0  88.797929 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.608467        0.0  0.0      0.0  0.0 | 1.876583       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014815  0.034662  0.0  0.0  0.0 |  0.764182 |      0.5 | 0.043768 | 0.0  0.0  0.0       0.0  0.0 | 0.243443 | 0.262228 | 0.255362 | 0.255362 | 1.064007 |      2.0       2.0  0.0  0.0       2.0 | 364.200917   366.92449  367.906504  0.0  367.415498 | 145.680367  145.565261  144.583248  0.0  145.074254 |  170.38392  130.710827  0.0  0.0  0.0 | 3.630962 | 48.445148 | 1.064007 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> sclass(2)
>>> sfdist(linear=0.2)

.. _hland_v1_snow_redistribution_2:

snow redistribution 2
_____________________

The example :ref:`hland_v1_snow_redistribution_1` assumes uniform snow distributions
within all zones.  Here, we introduce some heterogeneity by defining two snow classes
for each zone:

>>> sclass(2)
>>> sfdist(linear=0.2)

Inspecting the snow pack's evolution reveals that we individually apply the threshold
parameter |SMax| for each snow class.  Hence, one snow class can still receive snowfall
or redistributed snow while the second one already routes it to the next lower zone:

.. integration-test::

    >>> test("hland_v1_snow_redistribution_2", axis1=(states.sp, states.wc))
    |        date |    p |   t |   tn |       epn | tmean |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                                                                      swe |                          gact | contriarea |                                             pc |                                               ep |                                              epc |                                     ei |                                              tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                                                        melt |                                                                   refr |                                     in_ |                                     r |                      sr |                      ea |                                cf |      inuz |     perc |       q0 |                           el |       q1 |     inuh |    outuh |       rt |       qt |                                     ic |                                                                                                       sp |                                                                                                       wc |                                    sm |       uz |        lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 5.0 | 20.2 |  0.100707 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |     0.05 | 0.046667 |  0.06037 |  0.06037 | 0.251543 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |      9.95 | 0.251543 |
    | 01/01 01:00 |  0.0 | 5.0 | 18.4 |  0.097801 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.04975 | 0.046433 | 0.086244 | 0.086244 | 0.359352 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.90025 | 0.359352 |
    | 01/01 02:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049501 | 0.046201 | 0.046434 | 0.046434 | 0.193473 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.850749 | 0.193473 |
    | 01/01 03:00 |  0.0 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049254 |  0.04597 | 0.046201 | 0.046201 | 0.192506 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.801495 | 0.192506 |
    | 01/01 04:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049007 |  0.04574 |  0.04597 |  0.04597 | 0.191543 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.752488 | 0.191543 |
    | 01/01 05:00 |  0.0 | 5.0 | 21.5 |  0.102761 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048762 | 0.045512 | 0.045741 | 0.045741 | 0.190586 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.703725 | 0.190586 |
    | 01/01 06:00 |  0.0 | 5.0 | 24.1 |  0.291908 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048519 | 0.045284 | 0.045512 | 0.045512 | 0.189633 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.655206 | 0.189633 |
    | 01/01 07:00 |  0.0 | 5.0 | 27.3 |  1.932875 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048276 | 0.045058 | 0.045284 | 0.045284 | 0.188685 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.60693 | 0.188685 |
    | 01/01 08:00 |  0.0 | 5.0 | 26.8 |  4.369536 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048035 | 0.044832 | 0.045058 | 0.045058 | 0.187741 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.558896 | 0.187741 |
    | 01/01 09:00 |  0.0 | 5.0 | 30.4 |  7.317556 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047794 | 0.044608 | 0.044833 | 0.044833 | 0.186802 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.511101 | 0.186802 |
    | 01/01 10:00 |  0.0 | 5.0 | 31.2 |  8.264362 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047556 | 0.044385 | 0.044608 | 0.044608 | 0.185868 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.463546 | 0.185868 |
    | 01/01 11:00 |  0.0 | 5.0 | 34.2 |  9.369867 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047318 | 0.044163 | 0.044385 | 0.044385 | 0.184939 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.416228 | 0.184939 |
    | 01/01 12:00 |  0.0 | 5.0 | 36.1 |  5.126178 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047081 | 0.043942 | 0.044163 | 0.044163 | 0.184014 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.369147 | 0.184014 |
    | 01/01 13:00 |  0.0 | 5.0 | 30.2 |   6.62503 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046846 | 0.043723 | 0.043943 | 0.043943 | 0.183094 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.322301 | 0.183094 |
    | 01/01 14:00 |  0.0 | 5.0 | 23.3 |  7.397619 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046612 | 0.043504 | 0.043723 | 0.043723 | 0.182179 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.27569 | 0.182179 |
    | 01/01 15:00 |  0.2 | 5.0 | 24.4 |   2.39151 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0    1.1856   0.2112       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046529 | 0.043427 | 0.043536 | 0.043536 | 0.181398 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.259332 | 0.181398 |
    | 01/01 16:00 |  0.0 | 5.0 | 24.9 |  1.829834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046297 |  0.04321 | 0.043396 | 0.043396 | 0.180817 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.213035 | 0.180817 |
    | 01/01 17:00 |  0.0 | 5.0 | 22.7 |  1.136569 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0  0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046065 | 0.042994 |  0.04321 |  0.04321 | 0.180044 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0     0.94848  0.0         0.0         0.0         0.0     1.42272  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0         0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.16697 | 0.180044 |
    | 01/01 18:00 |  1.3 | 5.0 | 20.6 |  0.750986 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    0.955857      7.1136  0.0      3.6416         0.0    1.405786     10.6704  0.0      5.4624 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0    1.8232    7.7064   1.3728     4.552 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |       0.0   0.642378  0.0   1.3728  0.0 |      0.0  0.160595  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.049759  0.0  0.0  0.0 |  0.036945 | 0.036945 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046974 | 0.043842 | 0.043231 | 0.043231 | 0.180128 |    1.584       2.0  0.0  0.0       2.0 |        0.0    0.682755      7.1136  0.0      3.6416         0.0    1.004133     10.6704  0.0      5.4624 |        0.0    0.273102         0.0  0.0         0.0         0.0    0.401653         0.0  0.0         0.0 |      100.0  100.531543  0.0  0.0  0.0 |      0.0 |  9.347778 | 0.180128 |
    | 01/01 19:00 |  5.6 | 5.0 | 20.2 |  0.223895 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    8.056568    33.67104  0.0    23.21024         0.0   12.028852    50.50656  0.0    34.81536 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.9136  14.27328   33.1968   5.9136   24.4608 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.4976  14.27328   33.1968   5.9136   24.4608 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    5.4976   5.411392  0.0   5.9136  0.0 |   1.3744  1.367268  0.0   5.9136  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047938  0.047712  0.0  0.0  0.0 |  0.992544 |      0.5 |  0.00003 | 0.0  0.0  0.0       0.0  0.0 | 0.053106 |  0.04959 | 0.044931 | 0.044931 | 0.187212 |      2.0       2.0  0.0  0.0       2.0 |        0.0    5.754691    33.67104  0.0    23.21024         0.0    8.592037    50.50656  0.0    34.81536 |        0.0    2.301877         0.0  0.0         0.0         0.0    3.436815         0.0  0.0         0.0 | 104.171138  104.623379  0.0  0.0  0.0 | 0.492514 | 10.568043 | 0.187212 |
    | 01/01 20:00 |  2.9 | 5.0 | 19.4 |  0.099425 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   11.760721      47.424  0.0      33.344         0.0   17.557082      71.136  0.0      50.016 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.0624   7.39152   17.1912   3.0624   12.6672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.0624   7.39152   17.1912   3.0624   12.6672 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    3.0624   2.775328  0.0   3.0624  0.0 |   0.8308  0.759472  0.0   3.0624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046799   0.04668  0.0  0.0  0.0 |  0.564265 |      0.5 | 0.000145 | 0.0  0.0  0.0       0.0  0.0 | 0.057171 | 0.053475 | 0.049176 | 0.049176 |   0.2049 |      2.0       2.0  0.0  0.0       2.0 |        0.0    8.400515      47.424  0.0      33.344         0.0   12.540773      71.136  0.0      50.016 |        0.0    3.360206         0.0  0.0         0.0         0.0    5.016309         0.0  0.0         0.0 | 106.449537  106.685915  0.0  0.0  0.0 | 0.556633 |  11.37693 |   0.2049 |
    | 01/01 21:00 |  4.9 | 5.0 | 18.8 |  0.098454 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   17.980843    70.66176  0.0    50.46656         0.0   26.859264   105.99264  0.0    75.69984 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    5.1744   4.727968  0.0   5.1744  0.0 | 1.465843   1.34533  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.044921  0.044966  0.0  0.0  0.0 |  1.025506 |      0.5 | 0.000612 | 0.0  0.0  0.0       0.0  0.0 | 0.062724 | 0.059031 | 0.053846 | 0.053846 |  0.22436 |      2.0       2.0  0.0  0.0       2.0 |        0.0   12.843459    70.66176  0.0    50.46656         0.0   19.185189   105.99264  0.0    75.69984 |        0.0    5.137384         0.0  0.0         0.0         0.0    7.674076         0.0  0.0         0.0 | 110.203014  110.113519  0.0  0.0  0.0 | 1.081527 | 12.481978 |  0.22436 |
    | 01/01 22:00 | 10.6 | 5.0 | 18.6 |  0.098128 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   31.371473    120.9312  0.0     87.5072         0.0    46.91721    181.3968  0.0    131.2608 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 11.1936  27.01728   62.8368  11.1936   46.3008 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  11.1936  27.01728   62.8368  11.1936   46.3008 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |   11.1936  10.292992  0.0  11.1936  0.0 | 3.398574   3.12006  0.0  11.1936  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.041001  0.041357  0.0  0.0  0.0 |  2.425223 |      0.5 | 0.010504 | 0.0  0.0  0.0       0.0  0.0 | 0.072548 | 0.076115 | 0.061593 | 0.061593 | 0.256637 |      2.0       2.0  0.0  0.0       2.0 |        0.0   22.408195    120.9312  0.0     87.5072         0.0   33.512293    181.3968  0.0    131.2608 |        0.0    8.963278         0.0  0.0         0.0         0.0   13.404917         0.0  0.0         0.0 | 118.039041  117.327808  0.0  0.0  0.0 | 2.996247 | 14.437087 | 0.256637 |
    | 01/01 23:00 |  0.1 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   31.553272   121.40544  0.0    87.85664         0.0   47.161908   182.10816  0.0   131.78496 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    0.1056   0.041632  0.0   0.1056  0.0 | 0.036784  0.014327  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040946  0.041322  0.0  0.0  0.0 | -0.010733 |      0.5 | 0.020476 | 0.0  0.0  0.0       0.0  0.0 | 0.074404 | 0.085825 | 0.074476 | 0.074476 | 0.310317 |      2.0       2.0  0.0  0.0       2.0 |        0.0   22.538051   121.40544  0.0    87.85664         0.0   33.687077   182.10816  0.0   131.78496 |        0.0    9.015221         0.0  0.0         0.0         0.0   13.474831         0.0  0.0         0.0 | 118.148803  117.396435  0.0  0.0  0.0 | 2.465038 |  14.80634 | 0.310317 |
    | 02/01 00:00 |  0.7 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   32.489861   124.72512  0.0    90.30272         0.0   48.538791   187.08768  0.0   135.45408 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    0.7392   0.627424  0.0   0.7392  0.0 | 0.257965  0.216178  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040685  0.041096  0.0  0.0  0.0 |  0.148894 |      0.5 | 0.011948 | 0.0  0.0  0.0       0.0  0.0 | 0.076703 | 0.081148 | 0.082628 | 0.082628 | 0.344281 |      2.0       2.0  0.0  0.0       2.0 |        0.0   23.207043   124.72512  0.0    90.30272         0.0   34.670565   187.08768  0.0   135.45408 |        0.0    9.282817         0.0  0.0         0.0         0.0   13.868226         0.0  0.0         0.0 | 118.670724  117.848778  0.0  0.0  0.0 | 2.101983 | 15.263809 | 0.344281 |
    | 02/01 01:00 |  3.0 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   36.319813   138.95232  0.0  110.900096         0.0   54.255719       200.0  0.0  166.350144 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.168    7.6464    17.784    3.168    13.104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.168    7.6464    17.784    3.168    13.104 |        0.0         0.0     4.21424  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0    12.64272 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |     3.168    2.87296  0.0    3.168  0.0 | 1.115353  0.997516  0.0    3.168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039638  0.040138  0.0  0.0  0.0 |   0.76734 |      0.5 | 0.011153 | 0.0  0.0  0.0       0.0  0.0 | 0.080725 | 0.084265 |  0.08288 |  0.08288 | 0.345333 |      2.0       2.0  0.0  0.0       2.0 |        0.0   25.942723   138.95232  0.0  110.900096         0.0   38.754085       200.0  0.0  166.350144 |        0.0   10.377089         0.0  0.0         0.0         0.0   15.501634         0.0  0.0         0.0 | 120.763009   119.76436  0.0  0.0  0.0 | 2.358171 | 16.064227 | 0.345333 |
    | 02/01 02:00 |  2.1 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   39.612147   148.91136  0.0  136.164608         0.0    59.16622       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.2176   5.35248   12.4488   2.2176    9.1728 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.2176   5.35248   12.4488   2.2176    9.1728 |        0.0         0.0     7.46928  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0    0.530864         0.0  0.0   20.284384 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    2.2176   1.781926  0.0   2.2176  0.0 | 0.808521  0.638977  0.0   2.2176  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.038914  0.039546  0.0  0.0  0.0 |   0.52048 |      0.5 | 0.013178 | 0.0  0.0  0.0       0.0  0.0 | 0.084048 | 0.088987 | 0.084622 | 0.084622 | 0.352591 |      2.0       2.0  0.0  0.0       2.0 |        0.0    28.29439   148.91136  0.0  136.164608         0.0   42.261586       200.0  0.0       200.0 |        0.0   11.317756         0.0  0.0         0.0         0.0   16.904634         0.0  0.0         0.0 | 122.211002  120.946856  0.0  0.0  0.0 | 2.365472 | 16.725551 | 0.352591 |
    | 02/01 03:00 | 10.4 | 5.0 | 17.7 |  0.096652 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   85.714495   198.23232  0.0       200.0         0.0  128.319743       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 10.9824  26.50752   61.6512  10.9824   45.4272 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  10.9824  26.50752   61.6512  10.9824   45.4272 |        0.0         0.0    36.99072  0.0    27.25632 |       0.0        0.0        0.0  0.0        0.0 |        0.0   31.120416         0.0  0.0   13.746816 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |   10.9824        0.0  0.0  10.9824  0.0 | 4.100699       0.0  0.0  10.9824  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.035454  0.039527  0.0  0.0  0.0 |  1.680677 |      0.5 | 0.026682 | 0.0  0.0  0.0       0.0  0.0 | 0.093615 |  0.10872 | 0.092323 | 0.092323 | 0.384679 |      2.0       2.0  0.0  0.0       2.0 |        0.0   62.575747   198.23232  0.0       200.0         0.0   93.663621       200.0  0.0       200.0 |        0.0   23.138748         0.0  0.0         0.0         0.0   34.656122         0.0  0.0         0.0 | 129.128157  120.986382  0.0  0.0  0.0 | 3.519467 | 18.629421 | 0.384679 |
    | 02/01 04:00 |  3.5 | 5.0 | 17.5 |  0.096321 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  107.827231       200.0  0.0       200.0         0.0  161.488847       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.696    8.9208    20.748    3.696    15.288 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.696    8.9208    20.748    3.696    15.288 |        0.0         0.0    19.86416  0.0      15.288 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.72012         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |     3.696        0.0  0.0    3.696  0.0 | 1.540685       0.0  0.0    3.696  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.034358  0.039507  0.0  0.0  0.0 |  0.614467 |      0.5 | 0.044942 | 0.0  0.0  0.0       0.0  0.0 |  0.09793 | 0.127355 | 0.108476 | 0.108476 | 0.451983 |      2.0       2.0  0.0  0.0       2.0 |        0.0   80.736803       200.0  0.0       200.0         0.0  120.885205       200.0  0.0       200.0 |        0.0   27.090428         0.0  0.0         0.0         0.0   40.603642         0.0  0.0         0.0 |  131.31783  121.025889  0.0  0.0  0.0 | 3.588992 | 19.488063 | 0.451983 |
    | 02/01 05:00 |  3.4 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  129.823327       200.0  0.0       200.0         0.0  194.482991       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.5904   8.66592   20.1552   3.5904   14.8512 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.5904   8.66592   20.1552   3.5904   14.8512 |        0.0         0.0     20.1552  0.0     14.8512 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.8292         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |    3.5904        0.0  0.0   3.5904  0.0 | 1.547855       0.0  0.0   3.5904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.03332  0.039487  0.0  0.0  0.0 |  0.617894 |      0.5 | 0.047646 | 0.0  0.0  0.0       0.0  0.0 | 0.102148 | 0.133455 | 0.124569 | 0.124569 | 0.519039 |      2.0       2.0  0.0  0.0       2.0 |        0.0   98.895267       200.0  0.0       200.0         0.0  148.102901       200.0  0.0       200.0 |        0.0    30.92806         0.0  0.0         0.0         0.0    46.38009         0.0  0.0         0.0 | 133.393695  121.065376  0.0  0.0  0.0 |  3.65924 | 20.327401 | 0.519039 |
    | 02/01 06:00 |  1.2 | 5.0 | 17.5 |  0.187298 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  137.586655       200.0  0.0       200.0    0.833614       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.2672   3.05856    7.1136   1.2672    5.2416 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.2672   3.05856    7.1136   1.2672    5.2416 |        0.0         0.0      7.1136  0.0      5.2416 |       0.0        0.0        0.0  0.0        0.0 |   2.451193    3.581608         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 1.960955  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |  3.301586        0.0  0.0   1.2672  0.0 | 1.468701       0.0  0.0   1.2672  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032387  0.039467  0.0  0.0  0.0 |  0.585308 |      0.5 | 0.049748 | 0.0  0.0  0.0       0.0  0.0 | 0.104685 | 0.137505 | 0.132999 | 0.132999 | 0.554163 |      2.0       2.0  0.0  0.0       2.0 |        0.0  105.330019       200.0  0.0       200.0    0.595439  151.607046       200.0  0.0       200.0 |        0.0   32.256636         0.0  0.0         0.0    0.238175   48.392954         0.0  0.0         0.0 | 135.258968  121.104844  0.0  0.0  0.0 |   3.6948 | 20.832316 | 0.554163 |
    | 02/01 07:00 |  0.1 | 5.0 | 17.8 |  1.264612 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  138.233599       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0    0.115851      0.5928  0.0      0.4368 |       0.0   0.037077        0.0  0.0        0.0 |   0.358505     0.22152         0.0  0.0         0.0 |  0.029662        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 0.286804  0.0  0.0  0.0  0.0  1.025644  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |  0.910573        0.0  0.0   0.1056  0.0 | 0.416473       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032123  0.039448  0.0  0.0  0.0 |  0.146997 |      0.5 | 0.042797 | 0.0  0.0  0.0       0.0  0.0 |  0.10638 | 0.133525 |  0.13572 |  0.13572 | 0.565502 |      2.0       2.0  0.0  0.0       2.0 |        0.0  105.902915       200.0  0.0       200.0         0.0  151.550128       200.0  0.0       200.0 |        0.0   32.330684         0.0  0.0         0.0         0.0   48.449872         0.0  0.0         0.0 | 135.785191  121.144291  0.0  0.0  0.0 | 3.298999 | 21.169593 | 0.565502 |
    | 02/01 08:00 |  0.0 | 5.0 | 17.8 |  3.045538 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  138.233599       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032107  0.039428  0.0  0.0  0.0 | -0.026521 |      0.5 | 0.027724 | 0.0  0.0  0.0       0.0  0.0 | 0.107991 | 0.122971 | 0.132064 | 0.132064 | 0.550267 |      2.0       2.0  0.0  0.0       2.0 |        0.0  105.942915       200.0  0.0       200.0         0.0  151.590128       200.0  0.0       200.0 |        0.0   32.290684         0.0  0.0         0.0         0.0   48.409872         0.0  0.0         0.0 | 135.817299  121.183719  0.0  0.0  0.0 | 2.744755 | 21.490174 | 0.550267 |
    | 02/01 09:00 |  0.0 | 5.0 | 18.0 |  1.930758 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  138.233599       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0  0.0       0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032091  0.039408  0.0  0.0  0.0 | -0.026507 |      0.5 | 0.015259 | 0.0  0.0  0.0       0.0  0.0 | 0.109594 | 0.114495 | 0.123433 | 0.123433 | 0.514303 |      2.0       2.0  0.0  0.0       2.0 |        0.0  105.982915       200.0  0.0       200.0         0.0  151.630128       200.0  0.0       200.0 |        0.0   32.250684         0.0  0.0         0.0         0.0   48.369872         0.0  0.0         0.0 |  135.84939  121.223127  0.0  0.0  0.0 | 2.202988 | 21.809151 | 0.514303 |
    | 02/01 10:00 |  0.4 | 5.0 | 18.2 |  2.461001 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  140.821375       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 |        0.0    0.462589      2.3712  0.0      1.7472 |       0.0   0.149123        0.0  0.0        0.0 |   1.433367     0.88608         0.0  0.0         0.0 |  0.119298        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 1.146694  0.0  0.0  0.0  0.0  1.720041  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |  1.975066        0.0  0.0   0.4224  0.0 | 0.911249       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031543  0.039388  0.0  0.0  0.0 |  0.353414 |      0.5 | 0.009596 | 0.0  0.0  0.0       0.0  0.0 |  0.11149 | 0.111734 | 0.115765 | 0.115765 | 0.482354 |      2.0       2.0  0.0  0.0       2.0 |        0.0  108.154499       200.0  0.0       200.0         0.0  151.284086       200.0  0.0       200.0 |        0.0   32.666876         0.0  0.0         0.0         0.0   48.715914         0.0  0.0         0.0 |  136.94475  121.262516  0.0  0.0  0.0 | 2.046807 | 22.186575 | 0.482354 |
    | 02/01 11:00 |  0.1 | 5.0 | 18.3 |  6.215945 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  141.468319       200.0  0.0       200.0         0.0       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0    0.115604      0.5928  0.0      0.4368 |       0.0   0.037324        0.0  0.0        0.0 |   0.358307     0.22152         0.0  0.0         0.0 |  0.029859        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 0.286646  0.0  0.0  0.0  0.0  0.429969  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |  0.493766        0.0  0.0   0.1056  0.0 | 0.231501       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031396  0.039369  0.0  0.0  0.0 |  0.070254 |      0.5 | 0.006179 | 0.0  0.0  0.0       0.0  0.0 | 0.113151 | 0.110551 | 0.112085 | 0.112085 |  0.46702 |      2.0       2.0  0.0  0.0       2.0 |        0.0  108.727395       200.0  0.0       200.0         0.0  151.227662       200.0  0.0       200.0 |        0.0   32.740924         0.0  0.0         0.0         0.0   48.772338         0.0  0.0         0.0 | 137.238412  121.301884  0.0  0.0  0.0 | 1.610882 | 22.517081 |  0.46702 |
    | 02/01 12:00 |  3.6 | 5.0 | 18.0 |  3.374783 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  11.082346  164.758303       200.0  0.0       200.0   18.265715       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |        0.0    4.072225     21.3408  0.0     15.7248 |       0.0   1.433183        0.0  0.0        0.0 |  12.827444     7.97472         0.0  0.0         0.0 |  1.146547        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |   3.10156        0.0  0.0   3.8016  0.0 | 1.460399       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.03056  0.039349  0.0  0.0  0.0 |   0.58265 |      0.5 | 0.004498 | 0.0  0.0  0.0       0.0  0.0 | 0.117444 | 0.113213 | 0.111405 | 0.111405 | 0.464189 |      2.0       2.0  0.0  0.0       2.0 |   7.915962  127.951651       200.0  0.0       200.0   13.046939  147.975436       200.0  0.0       200.0 |   3.166385   36.806652         0.0  0.0         0.0    5.218776   52.024564         0.0  0.0         0.0 | 138.910133  121.341233  0.0  0.0  0.0 | 1.689033 | 23.371295 | 0.464189 |
    | 02/01 13:00 |  5.9 | 5.0 | 17.8 |  8.821555 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  32.456618       200.0       200.0  0.0       200.0   51.969317       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.2304  15.03792   34.9752   6.2304   25.7712 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.2304  15.03792   34.9752   6.2304   25.7712 |        0.0    6.452309     34.9752  0.0     25.7712 |       0.0   2.570443        0.0  0.0        0.0 |  22.016663    11.60568         0.0  0.0         0.0 |  2.056355        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |   2.76448        0.0  0.0   6.2304  0.0 | 1.333587       0.0  0.0   6.2304  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.029829  0.039329  0.0  0.0  0.0 |  0.530123 |      0.5 | 0.004929 | 0.0  0.0  0.0       0.0  0.0 |  0.12345 | 0.119163 | 0.113943 | 0.113943 | 0.474764 |      2.0       2.0  0.0  0.0       2.0 |  23.183298  156.504516       200.0  0.0       200.0   37.120941  143.063075       200.0  0.0       200.0 |   9.273319   43.495484         0.0  0.0         0.0   14.848376   56.936925         0.0  0.0         0.0 | 140.370856  121.380563  0.0  0.0  0.0 | 1.714227 | 24.566474 | 0.474764 |
    | 02/01 14:00 |  1.1 | 5.0 | 17.7 |  4.046025 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  36.484296       200.0       200.0  0.0       200.0    59.65303  199.072114       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.1616   2.80368    6.5208   1.1616    4.8048 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.1616   2.80368    6.5208   1.1616    4.8048 |        0.0    2.068992      6.5208  0.0      4.8048 |       0.0   0.734688        0.0  0.0        0.0 |   6.528633         0.0         0.0  0.0         0.0 |  0.587751        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0  0.0  0.04       0.0  0.0       0.0 |  2.422288   0.463943  0.0   1.1616  0.0 | 1.193218  0.170885  0.0   1.1616  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0292  0.039163  0.0  0.0  0.0 |  0.528915 |      0.5 | 0.005145 | 0.0  0.0  0.0       0.0  0.0 | 0.125805 | 0.121534 | 0.118368 | 0.118368 | 0.493198 |      2.0       2.0  0.0  0.0       2.0 |  26.060212  155.786281       200.0  0.0       200.0   42.609307  142.194367       200.0  0.0       200.0 |  10.424085   44.213719         0.0  0.0         0.0   17.043723   56.877747         0.0  0.0         0.0 | 141.629127  121.712784  0.0  0.0  0.0 | 1.737997 | 25.035183 | 0.493198 |
    | 02/01 15:00 | 20.7 | 5.0 | 16.8 |  2.110757 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.336199  201.336199  201.336199  0.0  201.336199  201.336199  183.251808  201.336199  0.0  201.336199 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 21.8592  52.76016  122.7096  21.8592   90.4176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  21.8592  52.76016  122.7096  21.8592   90.4176 |        0.0   35.415424    122.7096  0.0     90.4176 |       0.0  16.880793        0.0  0.0        0.0 | 118.119894    1.201078    1.201078  0.0    1.201078 | 13.288441   0.135121   0.135121  0.0   0.135121 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.135121  0.0  0.135121  0.0  0.04  0.135121  0.0  0.135121 |       0.0   9.042195  0.0  21.8592  0.0 |      0.0  3.348777  0.0  21.8592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.029185  0.036297  0.0  0.0  0.0 |     1.092 |      0.5 |  0.00858 | 0.0  0.0  0.0       0.0  0.0 | 0.142932 | 0.140267 |  0.12517 |  0.12517 | 0.521541 |      2.0       2.0  0.0  0.0       2.0 | 156.176776  145.238288  201.336199  0.0  201.336199  144.040545  130.894149  201.336199  0.0  201.336199 |  45.159423   56.097911         0.0  0.0         0.0   57.295654    52.35766         0.0  0.0         0.0 | 141.658312  127.442499  0.0  0.0  0.0 | 2.321417 | 28.443565 | 0.521541 |
    | 02/01 16:00 | 37.9 | 5.0 | 16.4 |  2.239257 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 308.251745  308.251745  308.251745  0.0  308.251745  287.018778  298.937676  308.251745  0.0  308.251745 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 40.0224  96.59952  224.6712  40.0224  165.5472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  40.0224  96.59952  224.6712  40.0224  165.5472 |  25.418337   55.456773  226.007399  0.0  166.883399 | 15.940261  33.436751        0.0  0.0        0.0 |  91.832644   91.832644   91.832644  0.0   91.832644 | 16.419101  16.419101  16.419101  0.0  16.419101 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 | 10.616484   4.657035  0.0  40.0224  0.0 | 5.326045  1.890941  0.0  40.0224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026526  0.034896  0.0  0.0  0.0 |  2.826815 |      0.5 | 0.046687 | 0.0  0.0  0.0       0.0  0.0 | 0.172948 | 0.198768 | 0.149104 | 0.149104 | 0.621268 |      2.0       2.0  0.0  0.0       2.0 | 223.340535  220.575828  292.472644  0.0  292.172644  205.013413  213.526911  292.472644  0.0  292.172644 |   84.91121   87.675917   15.779101  0.0   16.079101   82.005365   85.410764   15.779101  0.0   16.079101 | 146.975276  130.243488  0.0  0.0  0.0 | 4.601545 | 34.416674 | 0.621268 |
    | 02/01 17:00 |  8.2 | 5.0 | 16.3 |  2.877848 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 326.469628  326.469628  326.469628  0.0  326.469628  326.469628  326.469628  326.469628  0.0  326.469628 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  8.6592  20.90016   48.6096   8.6592   35.8176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   8.6592  20.90016   48.6096   8.6592   35.8176 |   74.38254   86.808172  149.932079  0.0  137.340728 | 31.911921  37.686698   6.929266  0.0   6.728617 | 100.483258  100.483258  100.483258  0.0  100.483258 |  25.98637   25.98637   25.98637  0.0   25.98637 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   8.6592  0.0 |      0.0       0.0  0.0   8.6592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026512  0.034878  0.0  0.0  0.0 | -0.022673 |      0.5 | 0.079641 | 0.0  0.0  0.0       0.0  0.0 | 0.180411 | 0.232097 | 0.193174 | 0.193174 | 0.804892 |      2.0       2.0  0.0  0.0       2.0 | 239.860197  240.808642  292.032328  0.0  291.278098  236.003199  239.144749  292.514518  0.0  291.667452 |  86.609431   85.660986     34.4373  0.0   35.191531   90.466429   87.324879   33.955111  0.0   34.802177 | 147.001789  130.278366  0.0  0.0  0.0 |  3.99923 | 35.901863 | 0.804892 |
    | 02/01 18:00 |  3.6 | 5.0 | 15.8 |  1.591452 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 336.889468  336.889468  336.889468  0.0  336.889468  336.889468  336.889468  336.889468  0.0  336.889468 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |  93.845639   98.612172  133.282589  0.0  127.654716 | 36.425589  37.033136  14.527839  0.0  14.539712 | 107.013798  107.013798  107.013798  0.0  107.013798 | 29.875671  29.875671  29.875671  0.0  29.875671 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.8016  0.0 |      0.0       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026499  0.034861  0.0  0.0  0.0 | -0.022662 |      0.5 | 0.051285 | 0.0  0.0  0.0       0.0  0.0 | 0.184368 | 0.213104 |  0.22047 |  0.22047 | 0.918624 |      2.0       2.0  0.0  0.0       2.0 | 250.253392  253.270707  287.605435  0.0  286.594862  247.254334  251.652975  288.365427  0.0   287.19845 |  86.636076   83.618762   49.284033  0.0   50.294606   89.635134   85.236494   48.524041  0.0   49.691018 | 147.028288  130.313227  0.0  0.0  0.0 | 3.425284 | 36.689152 | 0.918624 |
    | 02/01 19:00 |  7.5 | 5.0 | 15.5 |  0.291604 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 358.597468  358.597468  358.597468  0.0  358.597468  358.597468  358.597468  358.597468  0.0  358.597468 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    7.92    19.116     44.46     7.92     32.76 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |     7.92    19.116     44.46     7.92     32.76 | 104.462001  114.311393  158.111918  0.0  146.716819 | 40.347467  41.694075   23.23755  0.0  22.932649 | 123.123704  123.123704  123.123704  0.0  123.123704 | 35.473764  35.473764  35.473764  0.0  35.473764 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     7.92  0.0 |      0.0       0.0  0.0     7.92  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026486  0.034843  0.0  0.0  0.0 |  -0.02265 |      0.5 | 0.031333 | 0.0  0.0  0.0       0.0  0.0 | 0.191246 | 0.203562 | 0.215204 | 0.215204 | 0.896684 |      2.0       2.0  0.0  0.0       2.0 | 266.602199  270.820148  297.299456  0.0  295.760615  263.536947  268.656155  298.894978  0.0  297.046466 |  91.995269   87.777321   61.298012  0.0   62.836853   95.060522   89.941313    59.70249  0.0   61.551002 | 147.054774  130.348071  0.0  0.0  0.0 | 2.871301 | 38.057906 | 0.896684 |
    | 02/01 20:00 | 18.5 | 5.0 | 15.3 |  0.092622 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 412.143868  412.143868  412.143868  0.0  412.143868  412.143868  412.143868  412.143868  0.0  412.143868 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  19.536   47.1528   109.668   19.536    80.808 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   19.536   47.1528   109.668   19.536    80.808 | 124.847277  147.275108   233.67818  0.0  205.568884 | 53.286191   58.47516  34.587288  0.0  33.836585 | 163.072326  163.072326  163.072326  0.0  163.072326 | 49.071542  49.071542  49.071542  0.0  49.071542 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   19.536  0.0 |      0.0       0.0  0.0   19.536  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026473  0.034826  0.0  0.0  0.0 | -0.022639 |      0.5 | 0.017719 | 0.0  0.0  0.0       0.0  0.0 | 0.206387 | 0.206803 | 0.206403 | 0.206403 | 0.860011 |      2.0       2.0  0.0  0.0       2.0 | 303.208153  308.168705  336.244867  0.0  333.719349  298.689104  304.540434  339.353859  0.0  336.390618 | 108.935715  103.975163   75.899001  0.0   78.424519  113.454764  107.603434   72.790009  0.0    75.75325 | 147.081246  130.382897  0.0  0.0  0.0 | 2.330944 | 41.070948 | 0.860011 |
    | 02/01 21:00 | 15.4 | 5.0 | 15.2 |  0.092451 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 456.717628  456.717628  456.717628  0.0  456.717628  456.717628  456.717628  456.717628  0.0  456.717628 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 16.2624  39.25152   91.2912  16.2624   67.2672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  16.2624  39.25152   91.2912  16.2624   67.2672 | 160.435684  180.220545  258.686782  0.0  234.523005 | 67.970584  71.174843  44.748286  0.0  44.888063 | 194.895689  194.895689  194.895689  0.0  194.895689 |  61.82194   61.82194   61.82194  0.0   61.82194 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  16.2624  0.0 |      0.0       0.0  0.0  16.2624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026459  0.034809  0.0  0.0  0.0 | -0.022628 |      0.5 | 0.008913 | 0.0  0.0  0.0       0.0  0.0 | 0.219114 | 0.211637 | 0.207157 | 0.207157 | 0.863153 |      2.0       2.0  0.0  0.0       2.0 | 335.184272  340.132441  364.248539  0.0  361.574036  330.941008  336.601545  367.630401  0.0  364.495699 | 121.533356  116.585187    92.46909  0.0   95.143592   125.77662  120.116084   89.087227  0.0    92.22193 | 147.107706  130.417705  0.0  0.0  0.0 | 1.799403 | 43.603606 | 0.863153 |
    | 02/01 22:00 |  6.3 | 5.0 | 14.5 |  0.091248 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 474.952348  474.952348  474.952348  0.0  474.952348  474.952348  474.952348  474.952348  0.0  474.952348 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.6528  16.05744   37.3464   6.6528   27.5184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.6528  16.05744   37.3464   6.6528   27.5184 | 189.302288  199.301152  240.052149  0.0   229.25814 |  74.06814  73.473916  54.011879  0.0  54.977888 | 207.163895  207.163895  207.163895  0.0  207.163895 | 67.788453  67.788453  67.788453  0.0  67.788453 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   6.6528  0.0 |      0.0       0.0  0.0   6.6528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026446  0.034791  0.0  0.0  0.0 | -0.022616 |      0.5 | 0.003711 | 0.0  0.0  0.0       0.0  0.0 | 0.224913 | 0.212888 |  0.21084 |  0.21084 | 0.878502 |      2.0       2.0  0.0  0.0       2.0 | 349.906798   354.48677  369.797283  0.0  367.755759  347.249709  352.205023  372.277949  0.0  369.842285 |  125.04555  120.465579  105.155065  0.0  107.196589  127.702639  122.747326    102.6744  0.0  105.110063 | 147.134152  130.452496  0.0  0.0  0.0 | 1.273075 | 44.757665 | 0.878502 |
    | 02/01 23:00 |  1.9 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 480.451708  480.451708  480.451708  0.0  480.451708  480.451708  480.451708  480.451708  0.0  480.451708 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.0064   4.84272   11.2632   2.0064    8.2992 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.0064   4.84272   11.2632   2.0064    8.2992 | 202.410637  207.298851  225.048404  0.0  221.033172 | 74.548112  72.496218  61.167144  0.0  62.218377 | 210.571305  210.571305  210.571305  0.0  210.571305 | 69.880403  69.880403  69.880403  0.0  69.880403 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0  0.0  2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.0064  0.0 |      0.0       0.0  0.0   2.0064  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026433  0.034774  0.0  0.0  0.0 | -0.022605 |      0.5 | 0.001095 | 0.0  0.0  0.0       0.0  0.0 | 0.227364 | 0.213083 | 0.212653 | 0.212653 | 0.886054 |      2.0       2.0  0.0  0.0       2.0 | 355.072994  359.389209  367.755469  0.0  366.393683  353.712864  358.195652  369.171966  0.0  367.559029 | 125.378714  121.062499   112.69624  0.0  114.058025  126.738845  122.256056  111.279742  0.0   112.89268 | 147.160585   130.48727  0.0  0.0  0.0 | 0.749376 |   45.2455 | 0.886054 |
    | 03/01 00:00 |  4.9 | 5.0 | 13.7 |  0.089858 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 494.634268  494.634268  494.634268  0.0  494.634268  494.634268  494.634268  494.634268  0.0  494.634268 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 | 208.438672  216.488143  241.479855  0.0  233.607201 | 77.187436  76.452686  68.019053  0.0  68.247707 | 220.476361  220.476361  220.476361  0.0  220.476361 | 74.157907  74.157907  74.157907  0.0  74.157907 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.1744  0.0 |      0.0       0.0  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02642  0.034756  0.0  0.0  0.0 | -0.022594 |      0.5 | 0.000147 | 0.0  0.0  0.0       0.0  0.0 | 0.232066 | 0.216713 | 0.213846 | 0.213846 | 0.891025 |      2.0       2.0  0.0  0.0       2.0 | 364.674744  368.868831  376.368061  0.0  374.970733  363.492525  367.779829  377.926787  0.0  376.206699 | 129.959525  125.765438  118.266208  0.0  119.663535  131.141743   126.85444  116.707482  0.0  118.427569 | 147.187004  130.522026  0.0  0.0  0.0 | 0.226634 | 46.181205 | 0.891025 |
    | 03/01 01:00 |  2.7 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.449148  502.449148  502.449148  0.0  502.449148  502.449148  502.449148  502.449148  0.0  502.449148 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.8512   6.88176   16.0056   2.8512   11.7936 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.8512   6.88176   16.0056   2.8512   11.7936 | 217.713614  223.262163  239.172285  0.0  234.398296 | 79.771854  78.253865  71.467583  0.0  72.029572 | 225.656298  225.656298  225.656298  0.0  225.656298 |  76.79285   76.79285   76.79285  0.0   76.79285 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.8512  0.0 |      0.0       0.0  0.0   2.8512  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026406  0.034739  0.0  0.0  0.0 | -0.022582 | 0.204051 | 0.000001 | 0.0  0.0  0.0       0.0  0.0 | 0.233817 |  0.21823 | 0.216244 | 0.216244 | 0.901015 |      2.0       2.0  0.0  0.0       2.0 | 370.084749  374.172798    379.6833  0.0  378.517282   369.27392  373.409412  380.870774  0.0  379.443353 | 132.364399  128.276351  122.765849  0.0  123.931866  133.175228  129.039737  121.578375  0.0  123.005795 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 46.529603 | 0.901015 |
    | 03/01 02:00 |  0.5 | 5.0 | 13.1 |  0.088805 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.896348  503.896348  503.896348  0.0  503.896348  503.896348  503.896348  503.896348  0.0  503.896348 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 222.682821  225.718424  231.586821  0.0  230.098608 | 80.294327  78.005125  73.826327  0.0  74.534541 | 226.242067  226.242067  226.242067  0.0  226.242067 | 77.654281  77.654281  77.654281  0.0  77.654281 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.233025 |  0.21749 | 0.217729 | 0.217729 | 0.907203 |      2.0       2.0  0.0  0.0       2.0 | 371.083665  375.098508  378.244626  0.0  377.421907  370.699529  374.734187  378.827938  0.0  377.873647 | 132.812683   128.79784  125.651722  0.0  126.474441  133.196819  129.162161   125.06841  0.0  126.022701 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 46.372007 | 0.907203 |
    | 03/01 03:00 |  0.2 | 5.0 | 13.3 |  0.089157 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.475228  504.475228  504.475228  0.0  504.475228  504.475228  504.475228  504.475228  0.0  504.475228 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 223.743784  226.395354  229.361443  0.0  228.543636 | 80.363764  78.010754  75.720505  0.0  76.226313 | 226.225254  226.225254  226.225254  0.0  226.225254 | 78.249974  78.249974  78.249974  0.0  78.249974 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.232011 | 0.216543 | 0.217444 | 0.217444 | 0.906018 |      2.0       2.0  0.0  0.0       2.0 | 371.114614  375.095366    377.0869  0.0  376.436186  370.937552  374.926409  377.364486  0.0  376.649804 | 133.360614  129.379862  127.388328  0.0  128.039042  133.537676  129.548819  127.110742  0.0  127.825424 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 46.170167 | 0.906018 |
    | 03/01 04:00 |  0.5 | 5.0 | 13.9 |  0.090207 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 505.922428  505.922428  505.922428  0.0  505.922428  505.922428  505.922428  505.922428  0.0  505.922428 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 224.085986  227.051343  230.343431  0.0  229.227376 | 80.917243  78.698285  77.095797  0.0  77.431852 | 226.837921  226.837921  226.837921  0.0  226.837921 | 79.084507  79.084507  79.084507  0.0  79.084507 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.231228 | 0.215813 | 0.216591 | 0.216591 | 0.902465 |      2.0       2.0  0.0  0.0       2.0 | 371.496822  375.462905  377.210889  0.0  376.591841  371.365247  375.335226  377.437476  0.0  376.763239 | 134.425606  130.459523  128.711539  0.0  129.330588  134.557181  130.587202  128.484952  0.0  129.159189 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 46.014368 | 0.902465 |
    | 03/01 05:00 |  2.4 | 5.0 | 14.7 |  0.091593 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 512.868988  512.868988  512.868988  0.0  512.868988  512.868988  512.868988  512.868988  0.0  512.868988 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.5344   6.11712   14.2272   2.5344   10.4832 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.5344   6.11712   14.2272   2.5344   10.4832 | 225.329557  230.412493  240.999736  0.0  237.217269 | 83.127271  81.627055  79.149892  0.0  79.188359 | 231.424172  231.424172  231.424172  0.0  231.424172 | 81.444816  81.444816  81.444816  0.0  81.444816 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.5344  0.0 |      0.0       0.0  0.0   2.5344  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.231882 | 0.216423 | 0.216111 | 0.216111 | 0.900462 |      2.0       2.0  0.0  0.0       2.0 | 375.350192  379.313635  382.301753  0.0   381.47122   375.00714  378.979213  382.929884  0.0  381.944066 | 137.518796  133.555353  130.567235  0.0  131.397769  137.861848  133.889775  129.939104  0.0  130.924922 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 46.144543 | 0.900462 |
    | 03/01 06:00 |  0.4 | 5.0 | 15.0 |  0.154861 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.026748  514.026748  514.026748  0.0  514.026748  514.026748  514.026748  514.026748  0.0  514.026748 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 |   228.9932  231.861015  235.547348  0.0  234.429363 | 84.298188  82.027493   79.69284  0.0  80.186825 | 231.806267  231.806267  231.806267  0.0  231.806267 | 82.220481  82.220481  82.220481  0.0  82.220481 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.4224  0.0 |      0.0       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.231024 | 0.215623 |  0.21611 |  0.21611 | 0.900457 |      2.0       2.0  0.0  0.0       2.0 | 375.735643  379.669681   381.71749  0.0   381.04525  375.553856  379.492231  382.054384  0.0  381.298243 | 138.291105  134.357068  132.309259  0.0  132.981498  138.472892  134.534517  131.972364  0.0  132.728505 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.973861 | 0.900457 |
    | 03/01 07:00 |  0.2 | 5.0 | 15.7 |  0.470369 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.605628  514.605628  514.605628  0.0  514.605628  514.605628  514.605628  514.605628  0.0  514.605628 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 229.547107  232.175403  234.367247  0.0   233.64953 | 84.690841  82.361105  80.845101  0.0  81.250818 | 231.783725  231.783725  231.783725  0.0  231.783725 | 82.821904  82.821904  82.821904  0.0  82.821904 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.23002 | 0.214685 | 0.215592 | 0.215592 | 0.898302 |      2.0       2.0  0.0  0.0       2.0 | 375.581735  379.500193  381.039017  0.0  380.452896  375.487032  379.407641  381.217012  0.0  380.586186 | 139.023893  135.105435  133.566612  0.0  134.152732  139.118596  135.197987  133.388616  0.0  134.019442 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.774013 | 0.898302 |
    | 03/01 08:00 |  0.0 | 5.0 | 16.1 |  1.173726 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.605628  514.605628  514.605628  0.0  514.605628  514.605628  514.605628  514.605628  0.0  514.605628 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.584023  231.980242  233.003706  0.0  232.631714 | 85.021605  82.625387  81.601923  0.0  81.973914 | 231.344917  231.344917  231.344917  0.0  231.344917 | 83.260712  83.260712  83.260712  0.0  83.260712 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22887 | 0.213612 | 0.214655 | 0.214655 | 0.894397 |      2.0       2.0  0.0  0.0       2.0 | 374.966696  378.876577  380.074636  0.0  379.546842   374.92989  378.840607  380.143814  0.0  379.598645 | 139.638932  135.729051  134.530992  0.0  135.058786  139.675738  135.765021  134.461814  0.0  135.006983 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.545143 | 0.894397 |
    | 03/01 09:00 |  0.0 | 5.0 | 15.2 |  4.202296 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.605628  514.605628  514.605628  0.0  514.605628  514.605628  514.605628  514.605628  0.0  514.605628 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.225715  231.616288  232.380866  0.0  232.052887 | 85.379913   82.98934  82.224762  0.0  82.552742 | 230.906863  230.906863  230.906863  0.0  230.906863 | 83.698766  83.698766  83.698766  0.0  83.698766 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.227726 | 0.212544 | 0.213613 | 0.213613 | 0.890055 |      2.0       2.0  0.0  0.0       2.0 | 374.289609  378.196156  379.261779  0.0  378.756653  374.275304  378.182177  379.288664  0.0  378.776786 | 140.316019  136.409472  135.343849  0.0  135.848975  140.330324  136.423451  135.316964  0.0  135.828842 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.317417 | 0.890055 |
    | 03/01 10:00 |  0.3 | 5.0 | 14.9 |  4.359715 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.473948  515.473948  515.473948  0.0  515.473948  515.473948  515.473948  515.473948  0.0  515.473948 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 228.908147  231.631296  233.468287  0.0  232.736445 | 86.014282  83.738972  82.915741  0.0  83.179584 | 231.092863  231.092863  231.092863  0.0  231.092863 | 84.381085  84.381085  84.381085  0.0  84.381085 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226813 | 0.211692 | 0.212592 | 0.212592 |   0.8858 |      2.0       2.0  0.0  0.0       2.0 | 374.140855  378.047843  379.276889  0.0  378.742886  374.099523  378.007546  379.359507  0.0  378.804191 | 141.333093  137.426106   136.19706  0.0  136.731063  141.374425  137.466403  136.114441  0.0  136.669757 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.135861 |   0.8858 |
    | 03/01 11:00 |  2.6 | 5.0 | 15.3 |  5.305753 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.999388  522.999388  522.999388  0.0  522.999388  522.999388  522.999388  522.999388  0.0  522.999388 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.7456   6.62688   15.4128   2.7456   11.3568 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.7456   6.62688   15.4128   2.7456   11.3568 | 229.733266  235.018716  246.026526  0.0  242.026733 | 88.486282  87.082112  84.860222  0.0  84.804015 | 236.065193  236.065193  236.065193  0.0  236.065193 | 86.934196  86.934196  86.934196  0.0  86.934196 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.7456  0.0 |      0.0       0.0  0.0   2.7456  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22764 | 0.212464 | 0.212053 | 0.212053 | 0.883555 |      2.0       2.0  0.0  0.0       2.0 | 378.266105  382.188921  385.096258  0.0  384.273417   377.94416  381.880061  385.723071  0.0  384.744179 | 144.733283  140.810467   137.90313  0.0  138.725972  145.055228  141.119327  137.276317  0.0   138.25521 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.300449 | 0.883555 |
    | 03/01 12:00 |  0.7 | 5.0 | 15.3 |  5.376027 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.025468  525.025468  525.025468  0.0  525.025468  525.025468  525.025468  525.025468  0.0  525.025468 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 | 233.718136  236.924044  241.761037  0.0  240.218888 | 90.020452  87.859505  85.387951  0.0  85.838101 | 237.060681  237.060681  237.060681  0.0  237.060681 | 87.964787  87.964787  87.964787  0.0  87.964787 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.7392  0.0 |      0.0       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22703 | 0.211895 | 0.212166 | 0.212166 | 0.884026 |      2.0       2.0  0.0  0.0       2.0 | 379.202922  383.095498  385.297818  0.0  384.597496  378.998466  382.899238  385.699999  0.0  384.898886 | 145.822547   141.92997   139.72765  0.0  140.427972  146.027002   142.12623  139.325469  0.0  140.126582 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 45.179018 | 0.884026 |
    | 03/01 13:00 |  0.3 | 5.0 | 15.4 |  4.658915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.893788  525.893788  525.893788  0.0  525.893788  525.893788  525.893788  525.893788  0.0  525.893788 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 234.775466  237.521631  240.248354  0.0  239.361965 | 90.566802  88.268477  86.555514  0.0  86.973904 | 237.236222  237.236222  237.236222  0.0  237.236222 | 88.657566  88.657566  88.657566  0.0  88.657566 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226121 | 0.211047 | 0.211833 | 0.211833 | 0.882637 |      2.0       2.0  0.0  0.0       2.0 | 379.270791  383.143022  384.793068  0.0  384.189045   379.15814  383.034817  385.017284  0.0  384.356652 | 146.622997  142.750767   141.10072  0.0  141.704743  146.735648  142.858971  140.876504  0.0  141.537136 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 44.998154 | 0.882637 |
    | 03/01 14:00 |  0.3 | 5.0 | 15.5 |  7.789594 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.762108  526.762108  526.762108  0.0  526.762108  526.762108  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 235.084155  237.818826  240.121584  0.0  239.308549 | 91.126433  88.839602  87.550605  0.0   87.89564 | 237.413029  237.413029  237.413029  0.0  237.413029 | 89.349079  89.349079  89.349079  0.0  89.349079 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225217 | 0.210203 | 0.211048 | 0.211048 | 0.879365 |      2.0       2.0  0.0  0.0       2.0 | 379.235119  383.097316   384.53651  0.0   383.96923  379.157594  383.022848  384.693533  0.0  384.086228 |  147.52699  143.664793  142.225598  0.0  142.792878  147.604514   143.73926  142.068575  0.0   142.67588 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 44.818194 | 0.879365 |
    | 03/01 15:00 |  0.0 | 5.0 | 17.4 |  4.851567 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.762108  526.762108  526.762108  0.0  526.762108  526.762108  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  235.22383  237.620584  238.585148  0.0  238.220837 | 91.538278  89.141524  88.176961  0.0  88.541271 | 236.967521  236.967521  236.967521  0.0  236.967521 | 89.794588  89.794588  89.794588  0.0  89.794588 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224091 | 0.209152 | 0.210157 | 0.210157 | 0.875652 |      2.0       2.0  0.0  0.0       2.0 |  378.60778  382.461155  383.607586  0.0  383.092202  378.578346  382.432882  383.667204  0.0  383.136624 | 148.154328  144.300953  143.154523  0.0  143.669906  148.183762  144.329227  143.094904  0.0  143.625485 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 44.594103 | 0.875652 |
    | 03/01 16:00 |  0.0 | 5.0 | 17.3 |   5.30692 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.762108  526.762108  526.762108  0.0  526.762108  526.762108  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.849594  237.240287  237.978704  0.0  237.654287 | 91.912514  89.521821  88.783404  0.0  89.107821 |  236.52304   236.52304   236.52304  0.0   236.52304 | 90.239068  90.239068  90.239068  0.0  90.239068 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.222971 | 0.208106 | 0.209153 | 0.209153 |  0.87147 |      2.0       2.0  0.0  0.0       2.0 | 377.925113  381.775138  382.810413  0.0  382.314733  377.913937  381.764403  382.833049  0.0  382.331599 | 148.836995   144.98697  143.951695  0.0  144.447375  148.848171  144.997705   143.92906  0.0  144.430509 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 44.371133 |  0.87147 |
    | 03/01 17:00 |  0.0 | 5.0 | 17.1 |  3.286036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.762108  526.762108  526.762108  0.0  526.762108  526.762108  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.431784  236.820176   237.47273  0.0  237.163459 | 92.330324  89.941932  89.289378  0.0  89.598649 | 236.078559  236.078559  236.078559  0.0  236.078559 | 90.683549  90.683549  90.683549  0.0  90.683549 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.221856 | 0.207065 | 0.208107 | 0.208107 | 0.867112 |      2.0       2.0  0.0  0.0       2.0 | 377.221439  381.070192  382.063263  0.0  381.575064  377.217195  381.066116  382.071857  0.0  381.581468 |  149.54067  145.691916  144.698845  0.0  145.187044  149.544913  145.695992  144.690251  0.0   145.18064 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 44.149277 | 0.867112 |
    | 03/01 18:00 |  0.0 | 5.0 | 15.7 |  1.506216 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.762108  526.762108  526.762108  0.0  526.762108  526.762108  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 233.997429  236.384948  237.004901  0.0  236.701381 | 92.764679   90.37716  89.757207  0.0  90.060727 | 235.634079  235.634079  235.634079  0.0  235.634079 | 91.128029  91.128029  91.128029  0.0  91.128029 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220746 |  0.20603 | 0.207066 | 0.207066 | 0.862777 |      2.0       2.0  0.0  0.0       2.0 | 376.509788  380.358059  381.335106  0.0  380.849748  376.508177  380.356512  381.338369  0.0  380.852179 |  150.25232  146.404049  145.427002  0.0   145.91236  150.253931  146.405597  145.423739  0.0  145.909929 | 147.213411  130.556765  0.0  0.0  0.0 |      0.0 | 43.928531 | 0.862777 |
    | 03/01 19:00 |  0.0 | 5.0 | 14.2 |  0.274762 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.113153  526.762108  526.762108  0.0  526.762108  526.112297  526.762108  526.762108  0.0  526.762108 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 233.556794   235.94398  236.551556  0.0   236.25022 | 93.205315  90.818128  90.210552  0.0  90.511889 | 235.189598  235.189598  235.189598  0.0  235.189598 |  91.57251   91.57251   91.57251  0.0   91.57251 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  0.649383        0.0  0.0      0.0  0.0 | 0.351832       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026245       0.0  0.0  0.0  0.0 |  0.135662 | 0.135662 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220224 | 0.205542 | 0.206152 | 0.206152 | 0.858965 |      2.0       2.0  0.0  0.0       2.0 |  375.79511  379.643197   380.61416  0.0  380.129881  375.794498  379.642609  380.615399  0.0  380.130804 | 150.318044  147.118911  146.147948  0.0  146.632228  150.317799  147.119499  146.146709  0.0  146.631305 | 147.537206  130.556765  0.0  0.0  0.0 |      0.0 | 43.824588 | 0.858965 |
    | 03/01 20:00 |  0.0 | 5.0 | 12.4 |  0.087565 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.262558  526.512346  526.512346  0.0  526.512346  525.262558  526.512346  526.512346  0.0  526.512346 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 232.937661  235.500833   236.10371  0.0  235.803202 | 93.175064  91.261275  90.658399  0.0  90.958906 | 234.677382  234.677382  234.677382  0.0  234.677382 | 91.834963  91.834963  91.834963  0.0  91.834963 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  1.249787        0.0  0.0      0.0  0.0 |  0.68011       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025947       0.0  0.0  0.0  0.0 |  0.272568 | 0.272568 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220291 | 0.205605 | 0.205665 | 0.205665 | 0.856936 |      2.0       2.0  0.0  0.0       2.0 | 375.187541  378.859564  379.828217  0.0  379.344347  375.187541  378.859341  379.828688  0.0  379.344697 | 150.075017  147.652782  146.684128  0.0  147.167999  150.075017  147.653005  146.683658  0.0  147.167648 |  148.13283  130.556765  0.0  0.0  0.0 |      0.0 | 43.837927 | 0.856936 |
    | 03/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 524.457282  526.031658  526.031658  0.0  526.031658  524.457282  526.031658  526.031658  0.0  526.031658 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 232.330399  234.946606  235.547523  0.0  235.247418 | 92.932159   91.56574  90.964822  0.0  91.264928 | 234.102185  234.102185  234.102185  0.0  234.102185 | 91.929473  91.929473  91.929473  0.0  91.929473 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  1.574376        0.0  0.0      0.0  0.0 | 0.863677       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025578       0.0  0.0  0.0  0.0 |  0.349208 | 0.349208 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220686 | 0.205974 | 0.205673 | 0.205673 | 0.856971 |      2.0       2.0  0.0  0.0       2.0 | 374.612344  378.055074  379.023025  0.0  378.539223  374.612344  378.054989  379.023203  0.0  378.539356 | 149.844938  147.976584  147.008633  0.0  147.492435  149.844938  147.976669  147.008455  0.0  147.492302 | 148.869108  130.556765  0.0  0.0  0.0 |      0.0 | 43.916561 | 0.856971 |
    | 03/01 22:00 |  0.0 | 5.0 | 10.6 |  0.084317 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 523.604354  525.426129  525.426129  0.0  525.426129  523.604354  525.426129  525.426129  0.0  525.426129 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.755201  234.316522  234.916535  0.0  234.616663 |  92.70208  91.715136  91.115123  0.0  91.414995 | 233.492951  233.492951  233.492951  0.0  233.492951 | 91.933178  91.933178  91.933178  0.0  91.933178 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  1.821774        0.0  0.0      0.0  0.0 | 1.009355       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025159       0.0  0.0  0.0  0.0 |  0.410081 | 0.410081 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22134 | 0.206584 | 0.206028 | 0.206028 | 0.858448 |      2.0       2.0  0.0  0.0       2.0 |  374.00311  377.271476  378.239496  0.0  377.755552   374.00311  377.271444  378.239564  0.0  377.755603 | 149.601244  148.154652  147.186632  0.0  147.670577  149.601244  148.154685  147.186565  0.0  147.670526 | 149.706687  130.556765  0.0  0.0  0.0 |      0.0 |  44.04672 | 0.858448 |
    | 03/01 23:00 |  0.0 | 5.0 | 10.0 |  0.083215 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.715305  524.725446  524.725446  0.0  524.725446  522.715305  524.725446  524.725446  0.0  524.725446 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.001664  0.001664  0.001664  0.001664  0.001664 | 0.001065  0.002397  0.005059  0.001065  0.003728 | 0.001065  0.002397  0.0  0.0  0.003728 |      0.0       0.0       0.0      0.0       0.0 | 231.145967   233.66556  234.265141  0.0  233.965401 | 92.458387  91.760568  91.160988  0.0  91.460727 | 232.857915  232.857915  232.857915  0.0  232.857915 | 91.867531  91.867531  91.867531  0.0  91.867531 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0  0.0  2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.010142        0.0  0.0      0.0  0.0 | 1.126287       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024705       0.0  0.0  0.0  0.0 |  0.458993 | 0.458993 |      0.0 | 0.0  0.0  0.0  0.001065  0.0 |   0.2222 | 0.207387 | 0.206627 | 0.206627 | 0.860945 | 1.998935  1.997603  0.0  0.0  1.996272 | 373.368075  376.503821  377.472292  0.0  376.988082  373.368075  376.503809  377.472318  0.0  376.988101 |  149.34723  148.221625  147.253154  0.0  147.737364   149.34723  148.221637  147.253128  0.0  147.737345 | 150.615246  130.556765  0.0  0.0  0.0 |      0.0 |  44.21779 | 0.860945 |
    | 04/01 00:00 |  0.0 | 5.0 |  9.5 |  0.082289 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 521.797292  523.952315  523.952315  0.0  523.952315  521.797292  523.952315  523.952315  0.0  523.952315 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |  0.00576   0.00576   0.00576   0.00576   0.00576 | 0.003687  0.008295  0.017511  0.003687  0.012903 | 0.003687  0.008295  0.0  0.0  0.012903 |      0.0       0.0       0.0      0.0       0.0 | 230.510932  232.998743  233.598091  0.0  233.298437 | 92.204373  91.726703  91.127355  0.0   91.42701 | 232.203257  232.203257  232.203257  0.0  232.203257 | 91.749058  91.749058  91.749058  0.0  91.749058 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.155023        0.0  0.0      0.0  0.0 | 1.222165       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024226       0.0  0.0  0.0  0.0 |  0.499141 | 0.499141 |      0.0 | 0.0  0.0  0.0  0.003687  0.0 | 0.223225 | 0.208344 | 0.207421 | 0.207421 | 0.864254 | 1.995248  1.989309  0.0  0.0  1.983369 | 372.712352  375.748332  376.717466  0.0  376.232908  372.712352  375.748327  376.717476  0.0  376.232916 | 149.084941  148.203983  147.234849  0.0  147.719407  149.084941  148.203988  147.234839  0.0  147.719399 |  151.57233  130.556765  0.0  0.0  0.0 |      0.0 | 44.421873 | 0.864254 |
    | 04/01 01:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.859293   523.12346   523.12346  0.0   523.12346  520.859293   523.12346   523.12346  0.0   523.12346 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.855209  232.319884  232.919091  0.0  232.619495 | 91.942083  91.632431  91.033224  0.0   91.33282 | 231.533257  231.533257  231.533257  0.0  231.533257 | 91.590203  91.590203  91.590203  0.0  91.590203 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.264167        0.0  0.0      0.0  0.0 | 1.300434       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023732       0.0  0.0  0.0  0.0 |  0.531959 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224252 | 0.209302 | 0.208344 | 0.208344 |   0.8681 | 1.995248  1.989309  0.0  0.0  1.983369 | 372.042352  375.001703  375.971636  0.0  375.486673  372.042352  375.001702  375.971639  0.0  375.486676 | 148.816941  148.121756  147.151824  0.0  147.636787  148.816941  148.121758   147.15182  0.0  147.636784 | 152.559795  130.556765  0.0  0.0  0.0 | 0.031959 | 44.626192 |   0.8681 |
    | 04/01 02:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.905378  522.252626  522.252626  0.0  522.252626  519.905378  522.252626  522.252626  0.0  522.252626 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.185209  231.631454  232.230564  0.0  231.931012 | 91.674084  91.492006  90.892896  0.0  91.192448 |  230.85189   230.85189   230.85189  0.0   230.85189 | 91.400737  91.400737  91.400737  0.0  91.400737 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.347249        0.0  0.0      0.0  0.0 | 1.365775       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023229  0.031959  0.0  0.0  0.0 |  0.548741 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225274 | 0.210256 | 0.209301 | 0.209301 | 0.872088 | 1.995248  1.989309  0.0  0.0  1.983369 | 371.360984  374.262139  375.232962  0.0  374.747552  371.360984  374.262138  375.232964  0.0  374.747553 | 148.544394  147.990488  147.019664  0.0  147.505075  148.544394  147.990488  147.019663  0.0  147.505073 | 153.564498  130.588725  0.0  0.0  0.0 |   0.0807 |  44.82949 | 0.872088 |
    | 04/01 03:00 |  1.3 | 5.0 | 10.2 |  0.083584 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.267317  525.106162  525.106162  0.0  525.106162  522.057913  525.106162  525.106162  0.0  525.106162 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 1.368048  3.302749    7.7064   1.3728  5.661769 | 228.878721  232.738243   238.42324  0.0  236.291773 | 92.394705  92.817132  91.535787  0.0  91.622623 | 232.839082  232.839082  232.839082  0.0  232.839082 |  92.26708   92.26708   92.26708  0.0   92.26708 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.943547        0.0  0.0   1.3728  0.0 | 1.735372       0.0  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022614  0.034706  0.0  0.0  0.0 |  0.702081 |      0.5 | 0.000008 | 0.0  0.0  0.0       0.0  0.0 | 0.227271 | 0.212126 | 0.210459 | 0.210459 | 0.876914 |      2.0       2.0  0.0  0.0       2.0 | 373.048084  375.927358  377.833841  0.0  377.176765   372.89851  375.789496  378.156569  0.0  377.416497 | 149.219234  149.178804  147.272321  0.0  147.929397  149.159404  149.316666  146.949593  0.0  147.689665 | 154.795286   130.62343  0.0  0.0  0.0 | 0.282773 | 45.226904 | 0.876914 |
    | 04/01 04:00 |  0.0 | 5.0 | 10.1 |    0.0834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 521.627499  523.974029  523.974029  0.0  523.974029  521.627499  523.974029  523.974029  0.0  523.974029 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.000834  0.000834  0.000834  0.000834  0.000834 | 0.000534  0.001201  0.002535  0.000534  0.001868 | 0.000534  0.001201  0.0  0.0  0.001868 |      0.0       0.0       0.0      0.0       0.0 | 230.116154  232.703212  234.026144  0.0   233.59364 | 92.046462   92.40295  91.080018  0.0  91.512522 | 232.081976  232.081976  232.081976  0.0  232.081976 | 91.892053  91.892053  91.892053  0.0  91.892053 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |   2.34653        0.0  0.0      0.0  0.0 | 1.405664       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022132  0.034688  0.0  0.0  0.0 |  0.564909 |      0.5 | 0.000032 | 0.0  0.0  0.0  0.000534  0.0 | 0.228277 | 0.213084 | 0.211923 | 0.211923 | 0.883013 | 1.999466  1.998799  0.0  0.0  1.998132 | 372.591071  375.303445  376.629578  0.0  376.079313  372.591071  375.250937  376.752497  0.0  376.170621 | 149.036428  148.670584  147.344451  0.0  147.894715  149.036428  148.723092  147.221532  0.0  147.803407 | 155.758284  130.658119  0.0  0.0  0.0 |  0.34765 | 45.427123 | 0.883013 |
    | 04/01 05:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.661161  523.071517  523.071517  0.0  523.071517  520.661161  523.071517  523.071517  0.0  523.071517 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.733928  232.034522  232.908706  0.0  232.558704 | 91.893571  91.939507  91.065323  0.0  91.415325 | 231.391734  231.391734  231.391734  0.0  231.391734 | 91.679783  91.679783  91.679783  0.0  91.679783 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.410357        0.0  0.0      0.0  0.0 |  1.46192       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021647  0.034671  0.0  0.0  0.0 |  0.588557 |      0.5 | 0.000061 | 0.0  0.0  0.0       0.0  0.0 | 0.229278 | 0.214042 | 0.213084 | 0.213084 |  0.88785 | 1.999466  1.998799  0.0  0.0  1.998132 | 371.900829  374.684424  375.790607  0.0  375.280572  371.900829  374.664382  375.837525  0.0  375.315424 | 148.760332  148.387093   147.28091  0.0  147.790945  148.760332  148.407135  147.233992  0.0  147.756093 | 156.728367   130.69279  0.0  0.0  0.0 | 0.436146 | 45.626416 |  0.88785 |
    | 04/01 06:00 |  0.0 | 5.0 | 11.2 |  0.310229 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.685819  522.144457  522.144457  0.0  522.144457  519.685819  522.144457  522.144457  0.0  522.144457 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.043686  231.415063  232.118967  0.0  231.800222 | 91.617474  91.656455   90.95255  0.0  91.271296 | 230.695062  230.695062  230.695062  0.0  230.695062 | 91.449395  91.449395  91.449395  0.0  91.449395 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.458638        0.0  0.0      0.0  0.0 | 1.509836       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021161  0.034654  0.0  0.0  0.0 |   0.60873 |      0.5 |  0.00012 | 0.0  0.0  0.0       0.0  0.0 | 0.230275 | 0.215019 | 0.214046 | 0.214046 |  0.89186 | 1.999466  1.998799  0.0  0.0  1.998132 | 371.204156  373.998234  375.021191  0.0  374.526175  371.204156  373.990571   375.03913  0.0  374.539501 | 148.481663  148.146223  147.123266  0.0  147.618282  148.481663  148.153886  147.105327  0.0  147.604956 |  157.69833  130.727443  0.0  0.0  0.0 | 0.544756 | 45.824712 |  0.89186 |
    | 04/01 07:00 |  0.7 | 5.0 | 10.8 |  1.391958 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.495506  523.224188  523.224188  0.0  523.224188  520.382119  523.224188  523.224188  0.0  523.224188 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 0.738666  1.782959    4.1496   0.7392  3.055732 | 228.549767  231.714425  235.085646  0.0  233.800213 | 91.874718  92.212991  91.208411  0.0  91.399976 | 231.435668  231.435668  231.435668  0.0  231.435668 |  91.78852   91.78852   91.78852  0.0   91.78852 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.785376        0.0  0.0   0.7392  0.0 | 1.731721       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020624  0.034636  0.0  0.0  0.0 |  0.701412 |      0.5 | 0.000277 | 0.0  0.0  0.0       0.0  0.0 | 0.231794 | 0.216563 | 0.215145 | 0.215145 | 0.896438 |      2.0       2.0  0.0  0.0       2.0 | 371.782504  374.580205  376.078215  0.0  375.496078  371.701514  374.502508   376.26135  0.0  375.631973 | 148.713002  148.643983  147.145973  0.0  147.728111  148.680605  148.721681  146.962839  0.0  147.592215 | 158.772609  130.762079  0.0  0.0  0.0 | 0.745891 | 46.127089 | 0.896438 |
    | 04/01 08:00 |  0.4 | 5.0 | 10.4 |  3.195876 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.714011   523.31065   523.31065  0.0   523.31065  520.649151   523.31065   523.31065  0.0   523.31065 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 229.000717  231.930961  234.498326  0.0  233.565824 | 91.860496  92.312747  91.097063  0.0  91.405564 | 231.522172  231.522172  231.522172  0.0  231.522172 | 91.788478  91.788478  91.788478  0.0  91.788478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.629069        0.0  0.0   0.4224  0.0 | 1.656888       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020128  0.034619  0.0  0.0  0.0 |  0.670444 |      0.5 | 0.000581 | 0.0  0.0  0.0       0.0  0.0 |  0.23308 | 0.218006 | 0.216541 | 0.216541 | 0.902252 |      2.0       2.0  0.0  0.0       2.0 |  371.93858   374.65803  376.119495  0.0  375.544242  371.892251  374.585663  376.290161  0.0  375.670904 | 148.775432   148.65262  147.191155  0.0  147.766408    148.7569  148.724987  147.020489  0.0  147.639746 | 159.764917  130.796698  0.0  0.0  0.0 | 0.915754 | 46.382923 | 0.902252 |
    | 04/01 09:00 |  0.1 | 5.0 | 10.6 |  5.191651 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.090778   522.58891   522.58891  0.0   522.58891  520.074555   522.58891   522.58891  0.0   522.58891 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |  229.08724  231.587168  232.955246  0.0  232.446783 | 91.699942  91.978362  90.948204  0.0  91.300667 | 231.007492  231.007492  231.007492  0.0  231.007492 | 91.581418  91.581418  91.581418  0.0  91.581418 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.506243        0.0  0.0   0.1056  0.0 | 1.599286       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019664  0.034602  0.0  0.0  0.0 |  0.646642 |      0.5 | 0.000974 | 0.0  0.0  0.0       0.0  0.0 | 0.234133 | 0.219303 | 0.217974 | 0.217974 | 0.908224 |      2.0       2.0  0.0  0.0       2.0 | 371.493413  374.213669  375.444589  0.0   374.91149  371.481825  374.175312  375.535158  0.0  374.978674 | 148.597365  148.375241   147.14432  0.0   147.67742   148.59273  148.413597  147.053752  0.0  147.610236 | 160.691539    130.8313  0.0  0.0  0.0 | 1.061422 | 46.592448 | 0.908224 |
    | 04/01 10:00 |  0.4 | 5.0 | 12.0 |  7.155036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.157575   522.78273   522.78273  0.0   522.78273  520.092701   522.78273   522.78273  0.0   522.78273 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 228.746404  231.543019  233.903121  0.0  233.008798 | 91.758662  92.065411  91.056989  0.0  91.327311 | 231.124789  231.124789  231.124789  0.0  231.124789 | 91.657941  91.657941  91.657941  0.0  91.657941 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.657592        0.0  0.0   0.4224  0.0 | 1.715593       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019183  0.034584  0.0  0.0  0.0 |  0.695309 |      0.5 | 0.001569 | 0.0  0.0  0.0       0.0  0.0 | 0.235407 | 0.220968 | 0.219385 | 0.219385 | 0.914104 |      2.0       2.0  0.0  0.0       2.0 | 371.541125  374.294271  375.654862  0.0  375.097928  371.494786  374.236811  375.790621  0.0  375.198617 |  148.61645  148.488459  147.127868  0.0  147.684802  148.597914  148.545919  146.992109  0.0  147.584113 | 161.652721  130.865884  0.0  0.0  0.0 | 1.255162 | 46.845955 | 0.914104 |
    | 04/01 11:00 |  0.0 | 5.0 | 16.1 |  8.391432 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.274466  521.760579  521.760579  0.0  521.760579  519.274466  521.760579  521.760579  0.0  521.760579 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 228.660813  231.083481  231.983203  0.0  231.628508 | 91.464325  91.699249  90.799527  0.0  91.154222 | 230.401239  230.401239  230.401239  0.0  230.401239 |  91.35934   91.35934   91.35934  0.0   91.35934 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.486113        0.0  0.0      0.0  0.0 | 1.624153       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018743  0.034567  0.0  0.0  0.0 |  0.657398 |      0.5 | 0.002379 | 0.0  0.0  0.0       0.0  0.0 | 0.236373 | 0.222518 | 0.220943 | 0.220943 | 0.920594 |      2.0       2.0  0.0  0.0       2.0 | 370.910333   373.63429  374.754808  0.0  374.241743  370.910333  373.612307  374.806746  0.0  374.280264 | 148.364133  148.126289  147.005771  0.0  147.518836  148.364133  148.148272  146.953833  0.0  147.480315 | 162.533424  130.900451  0.0  0.0  0.0 | 1.410181 | 47.038154 | 0.920594 |
    | 04/01 12:00 |  0.0 | 5.0 | 17.2 |  8.391286 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 518.287991  520.804382  520.804382  0.0  520.804382  518.287991  520.804382  520.804382  0.0  520.804382 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  228.05319  230.406922  231.120718  0.0  230.800183 | 91.221276  91.353657  90.639861  0.0  90.960396 | 229.696614  229.696614  229.696614  0.0  229.696614 | 91.107768  91.107768  91.107768  0.0  91.107768 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.516391        0.0  0.0      0.0  0.0 | 1.661894       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018306   0.03455  0.0  0.0  0.0 |  0.673312 |      0.5 |  0.00336 | 0.0  0.0  0.0       0.0  0.0 | 0.237334 | 0.224199 | 0.222547 | 0.222547 |  0.92728 |      2.0       2.0  0.0  0.0       2.0 | 370.205708  372.957204  373.986718  0.0  373.490051  370.205708  372.948777  374.006627  0.0  373.504817 | 148.082283  147.847178  146.817664  0.0  147.314331  148.082283  147.855605  146.797755  0.0  147.299565 | 163.406226  130.935001  0.0  0.0  0.0 | 1.580133 | 47.229392 |  0.92728 |
    | 04/01 13:00 |  0.0 | 5.0 | 21.4 | 10.715238 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  517.29717  519.836539  519.836539  0.0  519.836539   517.29717  519.836539  519.836539  0.0  519.836539 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 227.348565  229.731081  230.373967  0.0  230.066447 | 90.939426  91.073301  90.430415  0.0  90.737935 | 228.988884  228.988884  228.988884  0.0  228.988884 | 90.847655  90.847655  90.847655  0.0  90.847655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.539369        0.0  0.0      0.0  0.0 |  1.69513       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017875  0.034532  0.0  0.0  0.0 |  0.687346 |      0.5 | 0.004693 | 0.0  0.0  0.0       0.0  0.0 |  0.23829 | 0.226158 | 0.224261 | 0.224261 |  0.93442 |      2.0       2.0  0.0  0.0       2.0 | 369.497979  372.252412  373.247767  0.0  372.757037  369.497979  372.249176  373.255413  0.0  372.762707 | 147.799192  147.584128  146.588772  0.0  147.079503  147.799192  147.587363  146.581127  0.0  147.073832 |  164.26834  130.969534  0.0  0.0  0.0 | 1.762786 | 47.419673 |  0.93442 |
    | 04/01 14:00 |  0.0 | 5.0 | 20.4 |  9.383394 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 516.302986  518.859859  518.859859  0.0  518.859859  516.302986  518.859859  518.859859  0.0  518.859859 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 226.640836  229.032391  229.648145  0.0  229.345608 | 90.656334  90.804148  90.188395  0.0  90.490931 | 228.278753  228.278753  228.278753  0.0  228.278753 | 90.581106  90.581106  90.581106  0.0  90.581106 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.556873        0.0  0.0      0.0  0.0 | 1.724872       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01745  0.034515  0.0  0.0  0.0 |  0.699921 |      0.5 |  0.00646 | 0.0  0.0  0.0       0.0  0.0 | 0.239241 |  0.22846 | 0.226234 | 0.226234 | 0.942642 |      2.0       2.0  0.0  0.0       2.0 | 368.787847  371.537778  372.520727  0.0  372.031926  368.787847  371.536533  372.523669  0.0  372.034107 | 147.515139  147.322081  146.339132  0.0  146.827933  147.515139  147.323326   146.33619  0.0  146.825752 |  165.11779  131.004049  0.0  0.0  0.0 | 1.956247 | 47.609003 | 0.942642 |
    | 04/01 15:00 |  0.0 | 5.0 | 20.8 |  7.861915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 515.306235  517.876446  517.876446  0.0  517.876446  515.306235  517.876446  517.876446  0.0  517.876446 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 225.930705  228.324244  228.929592  0.0   228.62897 | 90.372282  90.535615  89.930267  0.0  90.230889 | 227.566788  227.566788  227.566788  0.0  227.566788 | 90.309659  90.309659  90.309659  0.0  90.309659 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.570211        0.0  0.0      0.0  0.0 | 1.751848       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017032  0.034498  0.0  0.0  0.0 |  0.711341 |      0.5 | 0.008748 | 0.0  0.0  0.0       0.0  0.0 | 0.240188 | 0.231174 | 0.228551 | 0.228551 | 0.952297 |      2.0       2.0  0.0  0.0       2.0 | 368.075882  370.819939  371.798828  0.0  371.310414  368.075882   370.81946  371.799961  0.0  371.311255 | 147.230353  147.056507  146.077619  0.0  146.566033  147.230353  147.056987  146.076485  0.0  146.565192 | 165.953185  131.038547  0.0  0.0  0.0 |  2.15884 | 47.797387 | 0.952297 |
    | 04/01 16:00 |  0.0 | 5.0 | 21.2 |  6.298329 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 514.307524  516.887904  516.887904  0.0  516.887904  514.307524  516.887904  516.887904  0.0  516.887904 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  225.21874  227.611912  228.213257  0.0  227.913375 | 90.087496  90.264534   89.66319  0.0  89.963072 | 226.853422  226.853422  226.853422  0.0  226.853422 | 90.034481  90.034481  90.034481  0.0  90.034481 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |   2.58038        0.0  0.0      0.0  0.0 | 1.776621       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016622  0.034481  0.0  0.0  0.0 |   0.72184 |      0.5 | 0.011651 | 0.0  0.0  0.0       0.0  0.0 |  0.24113 | 0.234375 | 0.231282 | 0.231282 | 0.963676 |      2.0       2.0  0.0  0.0       2.0 | 367.362517  370.101302  371.079341  0.0   370.59072  367.362517  370.101117  371.079779  0.0  370.591044 | 146.945007  146.786602  145.808562  0.0  146.297184  146.945007  146.786787  145.808124  0.0  146.296859 | 166.773565  131.073028  0.0  0.0  0.0 | 2.369029 | 47.984829 | 0.963676 |
    | 04/01 17:00 |  0.0 | 5.0 | 19.1 |  2.948416 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 513.307313   515.89545   515.89545  0.0   515.89545  513.307313   515.89545   515.89545  0.0   515.89545 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.505374  226.897545  227.497342  0.0  227.197748 |  89.80215  89.990359  89.390562  0.0  89.690156 | 226.138986  226.138986  226.138986  0.0  226.138986 | 89.756463  89.756463  89.756463  0.0  89.756463 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.588137        0.0  0.0      0.0  0.0 | 1.799623       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016219  0.034463  0.0  0.0  0.0 |  0.731597 |      0.5 | 0.015259 | 0.0  0.0  0.0       0.0  0.0 | 0.242067 | 0.238137 |   0.2345 |   0.2345 | 0.977082 |      2.0       2.0  0.0  0.0       2.0 | 366.648081  369.382687  370.361121  0.0  369.872058  366.648081  369.382615   370.36129  0.0  369.872183 | 146.659232  146.512763  145.534329  0.0  146.023392  146.659232  146.512835   145.53416  0.0  146.023267 | 167.578297  131.107491  0.0  0.0  0.0 | 2.585367 | 48.171333 | 0.977082 |
    | 04/01 18:00 |  0.0 | 5.0 | 16.8 |  1.309232 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.305956  514.900013  514.900013  0.0  514.900013  512.305956  514.900013  514.900013  0.0  514.900013 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 223.790938  226.182066  226.781259  0.0   226.48178 | 89.516375  89.713384  89.114191  0.0   89.41367 | 225.423732  225.423732  225.423732  0.0  225.423732 | 89.476281  89.476281  89.476281  0.0  89.476281 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.594056        0.0  0.0      0.0  0.0 | 1.821189       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015824  0.034446  0.0  0.0  0.0 |  0.740753 |      0.5 | 0.019664 | 0.0  0.0  0.0       0.0  0.0 |    0.243 | 0.242531 | 0.238277 | 0.238277 | 0.992821 |      2.0       2.0  0.0  0.0       2.0 | 365.932826   368.66433  369.643645  0.0  369.154047  365.932826  368.664303  369.643711  0.0  369.154096 |  146.37313  146.235682  145.256368  0.0  145.745965   146.37313   146.23571  145.256302  0.0  145.745917 | 168.366989  131.141937  0.0  0.0  0.0 | 2.806456 | 48.356905 | 0.992821 |
    | 04/01 19:00 |  0.0 | 5.0 | 14.2 |   0.32955 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.303722  513.902299  513.902299  0.0  513.902299  511.303722  513.902299  513.902299  0.0  513.902299 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 223.075683  225.465906  226.064859  0.0  225.765428 | 89.230273  89.434107  88.835153  0.0  89.134584 |  224.70785   224.70785   224.70785  0.0   224.70785 | 89.194449  89.194449  89.194449  0.0  89.194449 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.598577        0.0  0.0      0.0  0.0 | 1.841575       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015438  0.034429  0.0  0.0  0.0 |  0.749414 |      0.5 | 0.024948 | 0.0  0.0  0.0       0.0  0.0 | 0.243927 | 0.247624 | 0.242686 | 0.242686 | 1.011192 |      2.0       2.0  0.0  0.0       2.0 | 365.216944  367.946265  368.926656  0.0  368.436484  365.216944  367.946255  368.926681  0.0  368.436503 | 146.086778  145.956033  144.975643  0.0  145.465815  146.086778  145.956044  144.975618  0.0  145.465796 | 169.139429  131.176366  0.0  0.0  0.0 | 3.030922 | 48.541549 | 1.011192 |
    | 04/01 20:00 |  0.0 | 5.0 | 13.5 |  0.089508 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.300814  512.902846  512.902846  0.0  512.902846  510.300814  512.902846  512.902846  0.0  512.902846 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.359801   224.74929  225.348144  0.0  225.048735 | 88.943921  89.153009  88.554155  0.0  88.853564 | 223.991487  223.991487  223.991487  0.0  223.991487 | 88.911359  88.911359  88.911359  0.0  88.911359 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.602032        0.0  0.0      0.0  0.0 | 1.860983       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01506  0.034412  0.0  0.0  0.0 |  0.757664 |      0.5 | 0.031182 | 0.0  0.0  0.0       0.0  0.0 | 0.244851 | 0.253473 | 0.247792 | 0.247792 | 1.032466 |      2.0       2.0  0.0  0.0       2.0 | 364.500582  367.228459  368.210007  0.0  367.719242  364.500582  367.228455  368.210017  0.0  367.719249 | 145.800233  145.674387  144.692839  0.0  145.183604  145.800233  145.674391   144.69283  0.0  145.183597 | 169.895538  131.210778  0.0  0.0  0.0 | 3.257404 |  48.72527 | 1.032466 |
    | 04/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.297391  511.902065  511.902065  0.0  511.902065  509.297391  511.902065  511.902065  0.0  511.902065 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 221.643439  224.032349  224.631159  0.0  224.331761 | 88.657376  88.870497  88.271687  0.0  88.571085 | 223.274756  223.274756  223.274756  0.0  223.274756 | 88.627309  88.627309  88.627309  0.0  88.627309 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.604674        0.0  0.0      0.0  0.0 | 1.879565       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01469  0.034395  0.0  0.0  0.0 |  0.765567 |      0.5 | 0.038423 | 0.0  0.0  0.0       0.0  0.0 | 0.245769 | 0.260123 | 0.253651 | 0.253651 | 1.056879 |      2.0       2.0  0.0  0.0       2.0 |  363.78385  366.510865  367.493607  0.0  367.002239   363.78385  366.510863  367.493611  0.0  367.002242 |  145.51354    145.3912  144.408458  0.0  144.899825   145.51354  145.391201  144.408454  0.0  144.899822 | 170.635336  131.245173  0.0  0.0  0.0 | 3.484547 | 48.908072 | 1.056879 |
    | 04/01 22:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  508.29357  510.900267  510.900267  0.0  510.900267   508.29357  510.900267  510.900267  0.0  510.900267 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 220.926708  223.315167  223.913954  0.0  223.614563 | 88.370683  88.586897  87.988111  0.0  88.287501 | 222.557741  222.557741  222.557741  0.0  222.557741 | 88.342526  88.342526  88.342526  0.0  88.342526 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.606697        0.0  0.0      0.0  0.0 | 1.897442       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014328  0.034377  0.0  0.0  0.0 |  0.773172 |      0.5 | 0.046707 | 0.0  0.0  0.0       0.0  0.0 | 0.246683 | 0.267603 | 0.260308 | 0.260308 | 1.084615 |      2.0       2.0  0.0  0.0       2.0 | 363.066835  365.793438  366.777395  0.0  366.285418  363.066835  365.793437  366.777397  0.0  366.285419 | 145.226734  145.106829  144.122872  0.0  144.614849  145.226734   145.10683   144.12287  0.0  144.614848 | 171.358919   131.27955  0.0  0.0  0.0 | 3.711012 |  49.08996 | 1.084615 |
    | 04/01 23:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.289443  509.897691  509.897691  0.0  509.897691  507.289443  509.897691  509.897691  0.0  509.897691 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 220.209693  222.597804  223.196576  0.0  222.897191 | 88.083877  88.302463  87.703691  0.0  88.003076 | 221.840508  221.840508  221.840508  0.0  221.840508 | 88.057184  88.057184  88.057184  0.0  88.057184 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0  0.0  2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34  0.0  0.04      0.64  0.0      0.34 |  2.608248        0.0  0.0      0.0  0.0 | 1.914707       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013974   0.03436  0.0  0.0  0.0 |  0.780519 |      0.5 | 0.056043 | 0.0  0.0  0.0       0.0  0.0 | 0.247593 | 0.275921 | 0.267789 | 0.267789 | 1.115789 |      2.0       2.0  0.0  0.0       2.0 | 362.349602  365.076141  366.061327  0.0  365.568735  362.349602  365.076141  366.061328  0.0  365.568735 | 144.939841   144.82155  143.836364  0.0  144.328956  144.939841   144.82155  143.836363  0.0  144.328956 | 172.066434   131.31391  0.0  0.0  0.0 | 3.935488 | 49.270939 | 1.115789 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from site-packages
from typing import *

# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(modeltools.AdHocModel):
    """HBV96 version of HydPy-H-Land (|hland_v1|)."""

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_TMean_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_EP_V1,
        hland_model.Calc_EPC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_CF_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_InUZ_V1,
        hland_model.Calc_ContriArea_V1,
        hland_model.Calc_Q0_Perc_UZ_V1,
        hland_model.Calc_LZ_V1,
        hland_model.Calc_EL_LZ_V1,
        hland_model.Calc_Q1_LZ_V1,
        hland_model.Calc_InUH_V1,
        hland_model.Calc_OutUH_QUH_V1,
        hland_model.Calc_RT_V1,
        hland_model.Calc_QT_V1,
    )
    ADD_METHODS = ()
    OUTLET_METHODS = (hland_model.Pass_Q_v1,)
    SENDER_METHODS = ()
    SUBMODELS = ()

    def check_waterbalance(
        self,
        initial_conditions: Dict[str, Dict[str, ArrayFloat]],
    ) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \sum_{k=1}^{NmbZones} RelZoneAreas^k \cdot \left(
            \sum_{t=t0}^{t1} \big( PC_t^k + GLMelt_t^k - EI_t^k - EA_t^k - EL_t^k \big)
            + \big( IC_{t0}^k - IC_{t1}^k \big)
            + \frac{1}{SClass} \cdot \sum_{c=1}^{SClass}
            \Big( \big( SP_{t0}^{c,k} - SP_{t1}^{c,k} \big)
            + \big( WC_{t0}^{c,k} - WC_{t1}^{c,k} \big) \Big)
            + \big( SM_{t0}^k - SM_{t1}^k \big) \right)
            + RelUpperZoneArea \cdot \big( UZ_{t0}^k - UZ_{t1}^k \big)
            + RelLowerZoneArea \cdot \big( LZ_{t0} - LZ_{t1} \big)
            - \sum_{t=t0}^{t1} RT_t
            + \sum_{i=1}^{Len(UH)} \big( QUH_{t0}^i - QUH_{t1}^i \big)

        The returned error should always be in scale with numerical precision so
        that it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run
        via property |Sequences.conditions|.  See the integration tests of the
        application model |hland_v1| for some examples.
        """
        derived = self.parameters.derived
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["states"]
        areas = derived.relzoneareas.value
        idxs_lake = self.parameters.control.zonetype.values == ILAKE
        idxs_land = ~idxs_lake
        idxs_glac = self.parameters.control.zonetype.values == GLACIER
        idxs_seal = self.parameters.control.zonetype.values == SEALED
        idxs_soil = idxs_land * ~idxs_glac * ~idxs_seal
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - (last.uz - first["uz"]) * derived.relupperzonearea
            - (last.lz - first["lz"]) * derived.rellowerzonearea
            - numpy.sum(self.sequences.logs.quh - initial_conditions["logs"]["quh"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_v1|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
