from __future__ import annotations
from dataclasses import dataclass
from typing import Any, Dict, Optional
import httpx

from .errors import *
# internal models live under ./internal (generated by openapi-generator-cli)

@dataclass(frozen=True)
class _Config:
    base_url: str
    token: Optional[str] = None
    timeout: float = 15.0

class _HTTP:
    def __init__(self, cfg: _Config):
        self._cfg = cfg
        self._client = httpx.Client(timeout=cfg.timeout)

    def request(self, method: str, path: str, *, params: Dict[str, Any] | None = None, json: Any | None = None, headers: Dict[str, str] | None = None):
        url = self._cfg.base_url.rstrip("/") + path
        headers = headers or {}
        if self._cfg.token:
            headers["Authorization"] = f"Bearer {self._cfg.token}"
        r = self._client.request(method, url, params=params, json=json, headers=headers)
        if r.status_code >= 400:
            raise map_error(r)
        # try json else bytes
        try:
            return r.json()
        except Exception:
            return r.content

class UapiClient:
    """领域驱动的 UAPI 客户端。
    - 子模块按 Tag 划分：如 `client.image`
    - 方法名来源于 operationId 的惯用命名 (snake_case)
    - 错误以具名异常抛出
    """

    def __init__(self, base_url: str, token: str | None = None, timeout: float = 15.0):
        self._http = _HTTP(_Config(base_url, token, timeout))
        # 动态挂载每个 Tag 的 API 门面
        _clipzy_zai_xian_jian_tie_ban = _ClipzyZaiXianJianTieBanApi(self._http)
        self.clipzy_zai_xian_jian_tie_ban = _clipzy_zai_xian_jian_tie_ban
        setattr(self, "Clipzy 在线剪贴板", _clipzy_zai_xian_jian_tie_ban)
        _convert = _ConvertApi(self._http)
        self.convert = _convert
        setattr(self, "Convert", _convert)
        _daily = _DailyApi(self._http)
        self.daily = _daily
        setattr(self, "Daily", _daily)
        _game = _GameApi(self._http)
        self.game = _game
        setattr(self, "Game", _game)
        _image = _ImageApi(self._http)
        self.image = _image
        setattr(self, "Image", _image)
        _misc = _MiscApi(self._http)
        self.misc = _misc
        setattr(self, "Misc", _misc)
        _network = _NetworkApi(self._http)
        self.network = _network
        setattr(self, "Network", _network)
        _poem = _PoemApi(self._http)
        self.poem = _poem
        setattr(self, "Poem", _poem)
        _random = _RandomApi(self._http)
        self.random = _random
        setattr(self, "Random", _random)
        _social = _SocialApi(self._http)
        self.social = _social
        setattr(self, "Social", _social)
        _status = _StatusApi(self._http)
        self.status = _status
        setattr(self, "Status", _status)
        _text = _TextApi(self._http)
        self.text = _text
        setattr(self, "Text", _text)
        _translate = _TranslateApi(self._http)
        self.translate = _translate
        setattr(self, "Translate", _translate)
        _webparse = _WebparseApi(self._http)
        self.webparse = _webparse
        setattr(self, "WebParse", _webparse)
        _min_gan_ci_shi_bie = _MinGanCiShiBieApi(self._http)
        self.min_gan_ci_shi_bie = _min_gan_ci_shi_bie
        setattr(self, "敏感词识别", _min_gan_ci_shi_bie)
        _zhi_neng_sou_suo = _ZhiNengSouSuoApi(self._http)
        self.zhi_neng_sou_suo = _zhi_neng_sou_suo
        setattr(self, "智能搜索", _zhi_neng_sou_suo)


class _ClipzyZaiXianJianTieBanApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_clipzy_get(self, **kwargs):
        """步骤2 (方法一): 获取加密数据
        **此接口用于“最高安全等级”方法。**

您提供第一步中获得的ID，它会返回存储在服务器上的**加密数据**。您需要在自己的客户端中，使用您自己保管的密钥来解密它。
        """
        params = {}
        body = None
        
        if "query" == "query" and "id" in kwargs:
            params["id"] = kwargs["id"]
        
        path = "/api/v1/api/get"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_clipzy_raw(self, **kwargs):
        """步骤2 (方法二): 获取原始文本
        **此接口用于“方便自动化”方法。**

您提供第一步获得的ID，并附上您自己保管的**解密密钥**作为 `key` 参数。服务器会直接为您解密，并返回纯文本内容。

> [!IMPORTANT]
> 查看文档首页的 **cURL 示例**，了解此接口最典型的用法。
        """
        params = {}
        body = None
        
        if "path" == "query" and "id" in kwargs:
            params["id"] = kwargs["id"]
        
        if "query" == "query" and "key" in kwargs:
            params["key"] = kwargs["key"]
        
        path = "/api/v1/api/raw/{id}"
        
        if "id" in kwargs:
            path = path.replace("{" + "id" + "}", str(kwargs["id"]))
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_clipzy_store(self, **kwargs):
        """步骤1：上传加密数据
        这是所有流程的第一步。您的客户端应用需要先在本地准备好 **加密后的数据**，然后调用此接口进行上传。成功后，您会得到一个用于后续操作的唯一ID。

> [!NOTE]
> 您发送给此接口的应该是**密文**，而不是原始文本。请参考文档首页的JavaScript示例来了解如何在客户端进行加密。
        """
        params = {}
        body = None
        
        path = "/api/v1/api/store"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _ConvertApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_convert_unixtime(self, **kwargs):
        """Unix时间戳与日期字符串双向转换
        时间戳和日期字符串，哪个用着更顺手？别纠结了，这个接口让你轻松拥有两种格式！

## 功能概述
这是一个非常智能的转换器。你给它一个 Unix 时间戳，它还你一个人类可读的日期时间；你给它一个日期时间字符串，它还你一个 Unix 时间戳。它会自动识别你输入的是哪种格式。

## 使用须知
这个接口非常智能，能够自动识别输入格式：

- **输入时间戳**：支持10位秒级（如 `1672531200`）和13位毫秒级（如 `1672531200000`）。
- **输入日期字符串**：为了确保准确性，推荐使用 `YYYY-MM-DD HH:mm:ss` 标准格式（如 `2023-01-01 08:00:00`）。

> [!TIP]
> 无论你输入哪种格式，响应中都会同时包含标准日期字符串和秒级Unix时间戳，方便你按需取用。

## 错误处理指南
- **400 Bad Request**: 如果你提供的 `time` 参数既不是有效的时间戳，也不是我们支持的日期格式，就会收到这个错误。请检查你的输入值。
        """
        params = {}
        body = None
        
        if "query" == "query" and "time" in kwargs:
            params["time"] = kwargs["time"]
        
        path = "/api/v1/convert/unixtime"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_convert_json(self, **kwargs):
        """美化并格式化JSON字符串
        还在为一团乱麻的 JSON 字符串头疼吗？这个接口能瞬间让它变得井井有条，赏心悦目。

## 功能概述
你只需要提供一个原始的、可能是压缩过的或者格式混乱的 JSON 字符串，这个 API 就会返回一个经过美化（带标准缩进和换行）的版本。这在调试 API 响应、或者需要在前端界面清晰展示 JSON 数据时非常有用。

## 使用须知
> [!NOTE]
> **请求格式**
> 请注意，待格式化的 JSON 字符串需要被包裹在另一个 JSON 对象中，作为 `content` 字段的值提交。具体格式请参考请求体示例。

## 错误处理指南
- **400 Bad Request**: 最常见的原因是你提供的 `content` 字符串本身不是一个有效的 JSON。请仔细检查括号、引号是否正确闭合，或者有没有多余的逗号等语法错误。
        """
        params = {}
        body = None
        
        path = "/api/v1/convert/json"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _DailyApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_daily_news_image(self, **kwargs):
        """生成每日新闻摘要图片
        想用一张图快速了解天下大事？这个接口为你一键生成今日新闻摘要，非常适合用在早报、数字看板或应用首页等场景。

## 功能概述
此接口会实时抓取各大平台的热点新闻，并动态地将它们渲染成一张清晰、美观的摘要图片。你调用接口，直接就能得到一张可以展示的图片。

## 使用须知
调用此接口时，请务必注意以下两点：

1.  **响应格式是图片**：接口成功时直接返回 `image/jpeg` 格式的二进制数据，而非 JSON。请确保你的客户端能正确处理二进制流，例如直接在 `<img>` 标签中显示，或保存为 `.jpg` 文件。

2.  **设置合理超时**：由于涉及实时新闻抓取和图片渲染，处理过程可能耗时数秒。建议将客户端请求超时时间设置为至少10秒，以防止因等待过久而失败。

> [!IMPORTANT]
> 未能正确处理图片响应或超时设置过短，是导致调用失败的最常见原因。
        """
        params = {}
        body = None
        
        path = "/api/v1/daily/news-image"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _GameApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_game_epic_free(self, **kwargs):
        """获取Epic Games免费游戏
        白嫖党的福音来了！想第一时间知道Epic商店本周送了哪些游戏大作吗？

## 功能概述
这个接口帮你实时追踪Epic Games商店的每周免费游戏活动。无需任何参数，调用后即可获得当前所有免费游戏的完整信息，包括游戏封面、原价、剩余时间等，再也不用担心错过心仪的免费游戏了！

## 使用场景
- 开发游戏资讯应用或网站
- 制作Epic免费游戏推送机器人
- 为用户提供游戏收藏建议
- 构建个人游戏库管理工具

> [!TIP]
> **关于时间格式**
> 为了方便不同场景的使用，我们同时提供了可读的时间字符串（如 `2025/01/10 00:00:00`）和13位毫秒时间戳。前端显示用字符串，程序逻辑用时间戳
        """
        params = {}
        body = None
        
        path = "/api/v1/game/epic-free"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_game_minecraft_historyid(self, **kwargs):
        """查询Minecraft玩家历史用户名
        想知道某个大佬以前叫什么名字吗？这个接口可以帮你追溯一个 Minecraft 玩家的“黑历史”！

## 功能概述
通过提供一个玩家的 UUID，你可以获取到该玩家所有曾用名及其变更时间的列表。这对于识别回归的老玩家或者社区管理非常有用。

## 使用须知
> [!NOTE]
> **UUID 格式**
> 查询时，请务必提供玩家的 **32位无破折号** Minecraft UUID，例如 `ee9b4ed1aac1491eb7611471be374b80`。
        """
        params = {}
        body = None
        
        if "query" == "query" and "uuid" in kwargs:
            params["uuid"] = kwargs["uuid"]
        
        path = "/api/v1/game/minecraft/historyid"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_game_minecraft_serverstatus(self, **kwargs):
        """查询Minecraft服务器状态
        想在加入服务器前看看有多少人在线？或者检查一下服务器开没开？用这个接口就对了！

## 功能概述
你可以通过提供服务器地址（域名或IP），来获取一个 Minecraft Java 版服务器的实时状态。返回信息非常丰富，包括服务器是否在线、当前玩家数、最大玩家数、服务器版本、MOTD（每日消息）以及服务器图标等。
        """
        params = {}
        body = None
        
        if "query" == "query" and "server" in kwargs:
            params["server"] = kwargs["server"]
        
        path = "/api/v1/game/minecraft/serverstatus"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_game_minecraft_userinfo(self, **kwargs):
        """查询Minecraft玩家信息
        只需要一个玩家的用户名，就能快速获取到他的正版皮肤和独一无二的UUID！

## 功能概述
这是一个基础但非常实用的接口。通过玩家当前的游戏内名称（Username），你可以查询到他对应的UUID（唯一标识符）和当前皮肤的URL地址。这是构建许多其他玩家相关服务的第一步。
        """
        params = {}
        body = None
        
        if "query" == "query" and "username" in kwargs:
            params["username"] = kwargs["username"]
        
        path = "/api/v1/game/minecraft/userinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_game_steam_summary(self, **kwargs):
        """获取Steam用户公开摘要
        想在你的网站或应用中展示用户的 Steam 个人资料？这个接口就是为你准备的。

## 功能概述
通过一个用户的 Steam 标识（支持多种格式），你可以获取到他公开的个人资料摘要，包括昵称、头像、在线状态、真实姓名（如果公开）和个人资料主页URL等信息。

## 支持的参数格式
接口现在支持以下几种标识符格式：
- **`steamid`**: 64位SteamID（如 `76561197960287930`）
- **`id`**: 自定义URL名称（如 `gabelogannewell`）
- **`id3`**: Steam ID3格式（如 `STEAM_0:0:22202`）
- 完整的个人资料链接
- 好友代码

## 使用须知

> [!IMPORTANT]
> **API Key 安全**
> 此接口需要一个 Steam Web API Key。我们强烈建议由后端统一配置和调用，以避免在客户端泄露。当然，你也可以通过 `key` 查询参数临时提供一个Key来覆盖后端配置。

在处理响应时，请注意以下数字代码的含义：
- **`personastate` (用户状态)**: 0-离线, 1-在线, 2-忙碌, 3-离开, 4-打盹, 5-想交易, 6-想玩。
- **`communityvisibilitystate` (社区可见性)**: 1-私密, 3-公开 (API通常只能查到这两种状态)。
        """
        params = {}
        body = None
        
        if "query" == "query" and "steamid" in kwargs:
            params["steamid"] = kwargs["steamid"]
        
        if "query" == "query" and "id" in kwargs:
            params["id"] = kwargs["id"]
        
        if "query" == "query" and "id3" in kwargs:
            params["id3"] = kwargs["id3"]
        
        if "query" == "query" and "key" in kwargs:
            params["key"] = kwargs["key"]
        
        path = "/api/v1/game/steam/summary"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _ImageApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_avatar_gravatar(self, **kwargs):
        """获取Gravatar头像
        提供一个超高速、高可用的Gravatar头像代理服务。内置了强大的ETag条件缓存，确保用户在更新Gravatar头像后能几乎立刻看到变化，同时最大化地利用缓存。
        """
        params = {}
        body = None
        
        if "query" == "query" and "email" in kwargs:
            params["email"] = kwargs["email"]
        
        if "query" == "query" and "hash" in kwargs:
            params["hash"] = kwargs["hash"]
        
        if "query" == "query" and "s" in kwargs:
            params["s"] = kwargs["s"]
        
        if "query" == "query" and "d" in kwargs:
            params["d"] = kwargs["d"]
        
        if "query" == "query" and "r" in kwargs:
            params["r"] = kwargs["r"]
        
        path = "/api/v1/avatar/gravatar"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_image_bing_daily(self, **kwargs):
        """获取必应每日壁纸
        每天都想换张新壁纸？让必应的美图点亮你的一天吧！

## 功能概述
这个接口会获取 Bing 搜索引擎当天全球同步的每日壁纸，并直接以图片形式返回。你可以用它来做应用的启动页、网站背景，或者任何需要每日更新精美图片的地方。

## 使用须知

> [!NOTE]
> **响应格式是图片**
> 请注意，此接口成功时直接返回图片二进制数据（通常为 `image/jpeg`），而非 JSON 格式。请确保客户端能够正确处理。

我们内置了备用方案：如果从必应官方获取图片失败，系统会尝试返回一张预存的高质量风景图，以保证服务的稳定性。
        """
        params = {}
        body = None
        
        path = "/api/v1/image/bing-daily"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_image_motou(self, **kwargs):
        """生成摸摸头GIF (QQ号方式)
        想在线rua一下好友的头像吗？这个趣味接口可以满足你。

## 功能概述
此接口通过GET方法，专门用于通过QQ号生成摸摸头GIF。你只需要提供一个QQ号码，我们就会自动获取其公开头像，并制作成一个可爱的动图。

## 使用须知
- **响应格式**：接口成功时直接返回 `image/gif` 格式的二进制数据。
- **背景颜色**：你可以通过 `bg_color` 参数来控制GIF的背景。使用 `transparent` 选项可以让它更好地融入各种聊天背景中。
        """
        params = {}
        body = None
        
        if "query" == "query" and "qq" in kwargs:
            params["qq"] = kwargs["qq"]
        
        if "query" == "query" and "bg_color" in kwargs:
            params["bg_color"] = kwargs["bg_color"]
        
        path = "/api/v1/image/motou"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_image_qrcode(self, **kwargs):
        """动态生成二维码
        无论是网址、文本还是联系方式，通通可以变成一个二维码！这是一个非常灵活的二维码生成工具。

## 功能概述
你提供一段文本内容，我们为你生成对应的二维码图片。你可以自定义尺寸，并选择不同的返回格式以适应不同场景。

## 使用须知

> [!IMPORTANT]
> **关键参数 `format`**
> 此参数决定了成功响应的内容类型和结构，请务必根据你的需求选择并正确处理响应：
> - **`image`** (默认): 直接返回 `image/png` 格式的图片二进制数据，适合在 `<img>` 标签中直接使用。
> - **`json`**: 返回一个包含 Base64 Data URI 的 JSON 对象，适合需要在前端直接嵌入CSS或HTML的场景。
> - **`json_url`**: 返回一个包含图片临时URL的JSON对象，适合需要图片链接的场景。
        """
        params = {}
        body = None
        
        if "query" == "query" and "text" in kwargs:
            params["text"] = kwargs["text"]
        
        if "query" == "query" and "size" in kwargs:
            params["size"] = kwargs["size"]
        
        if "query" == "query" and "format" in kwargs:
            params["format"] = kwargs["format"]
        
        path = "/api/v1/image/qrcode"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_image_tobase_64(self, **kwargs):
        """将在线图片转换为Base64
        看到一张网上的图片，想把它转换成 Base64 编码以便嵌入到你的 HTML 或 CSS 中？用这个接口就对了。

## 功能概述
你提供一个公开可访问的图片 URL，我们帮你把它下载下来，并转换成包含 MIME 类型的 Base64 Data URI 字符串返回给你。
        """
        params = {}
        body = None
        
        if "query" == "query" and "url" in kwargs:
            params["url"] = kwargs["url"]
        
        path = "/api/v1/image/tobase64"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_image_compress(self, **kwargs):
        """无损压缩图片
        还在为图片体积和加载速度发愁吗？体验一下我们强大的**无损压缩服务**，它能在几乎不牺牲任何肉眼可感知的画质的前提下，将图片体积压缩到极致。

## 功能概述
你只需要上传一张常见的图片（如 PNG, JPG），选择一个压缩等级，就能获得一个体积小到惊人的压缩文件。这对于需要大量展示高清图片的网站、App 或小程序来说，是优化用户体验、节省带宽和存储成本的利器。

## 使用须知
> [!TIP]
> 为了给您最好的压缩效果，我们的算法需要进行复杂计算，处理时间可能会稍长一些，请耐心等待。

> [!WARNING]
> **服务排队提醒**
> 这是一个计算密集型服务。在高并发时，您的请求可能会被排队等待处理。如果您需要将其集成到对延迟敏感的生产服务中，请注意这一点。

### 请求与响应格式
- 请求必须使用 `multipart/form-data` 格式上传文件。
- 成功响应将直接返回压缩后的文件二进制流 (`application/octet-stream`)，并附带 `Content-Disposition` 头，建议客户端根据此头信息保存文件。

## 参数详解
### `level` (压缩等级)
这是一个从 `1` 到 `5` 的整数，它决定了压缩的强度和策略，数字越小，压缩率越高。所有等级都经过精心调校，以在最大化压缩率的同时保证出色的视觉质量。
- `1`: **极限压缩** (推荐，体积最小，画质优异)
- `2`: **高效压缩**
- `3`: **智能均衡** (默认选项)
- `4`: **画质优先**
- `5`: **专业保真** (压缩率稍低，保留最多图像信息)

## 错误处理指南
- **400 Bad Request**: 通常因为没有上传文件，或者 `level` 参数不在 1-5 的范围内。
- **500 Internal Server Error**: 如果在压缩过程中服务器发生内部错误，会返回此状态码。
        """
        params = {}
        body = None
        
        if "query" == "query" and "level" in kwargs:
            params["level"] = kwargs["level"]
        
        if "query" == "query" and "format" in kwargs:
            params["format"] = kwargs["format"]
        
        path = "/api/v1/image/compress"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_image_frombase_64(self, **kwargs):
        """通过Base64编码上传图片
        当你需要在前端处理完图片（比如裁剪、加滤镜后），不通过传统表单，而是直接上传图片的场景，这个接口就派上用场了。

## 功能概述
你只需要将图片的 Base64 编码字符串发送过来，我们就会把它解码、保存为图片文件，并返回一个可供访问的公开 URL。

## 使用须知

> [!IMPORTANT]
> **关于 `imageData` 格式**
> 你发送的 `imageData` 字符串必须是完整的 Base64 Data URI 格式，它需要包含 MIME 类型信息，例如 `data:image/png;base64,iVBORw0KGgo...`。缺少 `data:image/...;base64,` 前缀将导致解码失败。
        """
        params = {}
        body = None
        
        path = "/api/v1/image/frombase64"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_image_motou(self, **kwargs):
        """生成摸摸头GIF (图片上传或URL方式)
        除了使用QQ头像，你还可以通过上传自己的图片或提供图片URL来制作独一无二的摸摸头GIF。

## 功能概述
此接口通过POST方法，支持两种方式生成GIF：
1.  **图片URL**：在表单中提供 `image_url` 字段。
2.  **上传图片**：在表单中上传 `file` 文件。

## 使用须知
- **响应格式**：接口成功时直接返回 `image/gif` 格式的二进制数据。
- **参数优先级**：如果同时提供了 `image_url` 和上传的 `file` 文件，系统将 **优先使用 `image_url`**。
- **背景颜色**：同样支持 `bg_color` 表单字段来控制GIF背景。
        """
        params = {}
        body = None
        
        path = "/api/v1/image/motou"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_image_speechless(self, **kwargs):
        """生成你们怎么不说话了表情包
        你们怎么不说话了？是不是都在偷偷玩Uapi，求求你们不要玩Uapi了

## 效果展示
![示例](https://uapis.cn/static/uploads/33580466897f1e5815296f235b582815.png)

## 使用须知
- **响应格式**：接口成功时直接返回 `image/jpeg` 格式的二进制数据。
- **文字内容**：至少需要提供 `top_text`（上方文字）或 `bottom_text`（下方文字）之一。
- **梗图逻辑**：上方描述某个行为，下方通常以「们」开头表示劝阻，形成戏谑的对比效果。
        """
        params = {}
        body = None
        
        path = "/api/v1/image/speechless"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_image_svg(self, **kwargs):
        """SVG转图片
        需要将灵活的 SVG 矢量图形转换为常见的光栅图像格式吗？这个接口可以帮你轻松实现。

## 功能概述
上传一个 SVG 文件，并指定目标格式（如 PNG、JPEG 等），接口将返回转换后的图像。你还可以调整输出图像的尺寸和（对于JPEG）压缩质量，以满足不同场景的需求。
        """
        params = {}
        body = None
        
        if "query" == "query" and "format" in kwargs:
            params["format"] = kwargs["format"]
        
        if "query" == "query" and "width" in kwargs:
            params["width"] = kwargs["width"]
        
        if "query" == "query" and "height" in kwargs:
            params["height"] = kwargs["height"]
        
        if "query" == "query" and "quality" in kwargs:
            params["quality"] = kwargs["quality"]
        
        path = "/api/v1/image/svg"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _MiscApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_history_programmer(self, **kwargs):
        """获取指定日期的程序员历史事件
        想查看程序员历史上某个特定日期发生的大事件？指定月份和日期，我们就能告诉你！

## 功能概述
通过指定月份和日期，获取该日发生的程序员相关历史事件。同样使用AI智能筛选，确保事件的相关性和重要性。
        """
        params = {}
        body = None
        
        if "query" == "query" and "month" in kwargs:
            params["month"] = kwargs["month"]
        
        if "query" == "query" and "day" in kwargs:
            params["day"] = kwargs["day"]
        
        path = "/api/v1/history/programmer"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_history_programmer_today(self, **kwargs):
        """获取今天的程序员历史事件
        想知道程序员历史上的今天发生了什么大事吗？这个接口告诉你答案！

## 功能概述
我们使用AI智能筛选从海量历史事件中挑选出与程序员、计算机科学相关的重要事件。每个事件都经过重要性评分和相关性评分，确保内容质量。
        """
        params = {}
        body = None
        
        path = "/api/v1/history/programmer/today"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_hotboard(self, **kwargs):
        """获取多平台实时热榜
        想快速跟上网络热点？这个接口让你一网打尽各大主流平台的实时热榜/热搜！

## 功能概述
你只需要指定一个平台类型，就能获取到该平台当前的热榜数据列表。每个热榜条目都包含标题、热度值和原始链接。非常适合用于制作信息聚合类应用或看板。

## 可选值
`type` 参数接受多种不同的值，每种值对应一个不同的热榜来源。以下是目前支持的所有值：

| 分类       | 支持的 type 值 |
|------------|-----------------------------------------------------------------------------------------------------------------------------------|
| 视频/社区  | bilibili（哔哩哔哩弹幕网）, acfun（A站弹幕视频网站）, weibo（新浪微博热搜）, zhihu（知乎热榜）, zhihu-daily（知乎日报热榜）, douyin（抖音热榜）, kuaishou（快手热榜）, douban-movie（豆瓣电影榜单）, douban-group（豆瓣小组话题）, tieba（百度贴吧热帖）, hupu（虎扑热帖）, miyoushe（米游社话题榜）, ngabbs（NGA游戏论坛热帖）, v2ex（V2EX技术社区热帖）, 52pojie（吾爱破解热帖）, hostloc（全球主机交流论坛）, coolapk（酷安热榜） |
| 新闻/资讯  | baidu（百度热搜）, thepaper（澎湃新闻热榜）, toutiao（今日头条热榜）, qq-news（腾讯新闻热榜）, sina（新浪热搜）, sina-news（新浪新闻热榜）, netease-news（网易新闻热榜）, huxiu（虎嗅网热榜）, ifanr（爱范儿热榜） |
| 技术/IT    | sspai（少数派热榜）, ithome（IT之家热榜）, ithome-xijiayi（IT之家·喜加一栏目）, juejin（掘金社区热榜）, jianshu（简书热榜）, guokr（果壳热榜）, 36kr（36氪热榜）, 51cto（51CTO热榜）, csdn（CSDN博客热榜）, nodeseek（NodeSeek 技术社区）, hellogithub（HelloGitHub 项目推荐） |
| 游戏       | lol（英雄联盟热帖）, genshin（原神热榜）, honkai（崩坏3热榜）, starrail（星穹铁道热榜） |
| 其他       | weread（微信读书热门书籍）, weatheralarm（天气预警信息）, earthquake（地震速报）, history（历史上的今天） |

        """
        params = {}
        body = None
        
        if "query" == "query" and "type" in kwargs:
            params["type"] = kwargs["type"]
        
        path = "/api/v1/misc/hotboard"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_phoneinfo(self, **kwargs):
        """查询手机号码归属地信息
        想知道一个手机号码来自哪里？是移动、联通还是电信？这个接口可以告诉你答案。

## 功能概述
提供一个国内的手机号码，我们会查询并返回它的归属地（省份和城市）以及所属的运营商信息。
        """
        params = {}
        body = None
        
        if "query" == "query" and "phone" in kwargs:
            params["phone"] = kwargs["phone"]
        
        path = "/api/v1/misc/phoneinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_randomnumber(self, **kwargs):
        """生成高度可定制的随机数
        需要一个简单的随机数，还是需要一串不重复的、带小数的随机数？这个接口都能满足你！

## 功能概述
这是一个强大的随机数生成器。你可以指定生成的范围（最大/最小值）、数量、是否允许重复、以及是否生成小数（并指定小数位数）。

## 流程图
```mermaid
graph TD
    A[开始] --> B{参数校验};
    B --> |通过| C{是否允许小数?};
    C --> |是| D[生成随机小数];
    C --> |否| E[生成随机整数];
    D --> F{是否允许重复?};
    E --> F;
    F --> |是| G[直接生成指定数量];
    F --> |否| H[生成不重复的数字];
    G --> I[返回结果];
    H --> I;
    B --> |失败| J[返回 400 错误];
```
## 使用须知
> [!WARNING]
> **不重复生成的逻辑限制**
> 当设置 `allow_repeat=false` 时，请确保取值范围 `(max - min + 1)` 大于或等于你请求的数量 `count`。否则，系统将无法生成足够的不重复数字，请求会失败并返回 400 错误。
        """
        params = {}
        body = None
        
        if "query" == "query" and "min" in kwargs:
            params["min"] = kwargs["min"]
        
        if "query" == "query" and "max" in kwargs:
            params["max"] = kwargs["max"]
        
        if "query" == "query" and "count" in kwargs:
            params["count"] = kwargs["count"]
        
        if "query" == "query" and "allow_repeat" in kwargs:
            params["allow_repeat"] = kwargs["allow_repeat"]
        
        if "query" == "query" and "allow_decimal" in kwargs:
            params["allow_decimal"] = kwargs["allow_decimal"]
        
        if "query" == "query" and "decimal_places" in kwargs:
            params["decimal_places"] = kwargs["decimal_places"]
        
        path = "/api/v1/misc/randomnumber"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_timestamp(self, **kwargs):
        """转换时间戳 (旧版，推荐使用/convert/unixtime)
        这是一个用于将Unix时间戳转换为人类可读日期时间的旧版接口。

## 功能概述
输入一个秒级或毫秒级的时间戳，返回其对应的本地时间和UTC时间。

> [!WARNING]
> **接口已过时**：这个接口已被新的 `/convert/unixtime` 取代。新接口功能更强大，支持双向转换。我们建议你迁移到新接口。

[👉 前往新版接口文档](/docs/api-reference/get-convert-unixtime)
        """
        params = {}
        body = None
        
        if "query" == "query" and "ts" in kwargs:
            params["ts"] = kwargs["ts"]
        
        path = "/api/v1/misc/timestamp"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_tracking_carriers(self, **kwargs):
        """获取支持的快递公司列表
        不确定系统支持哪些快递公司？这个接口返回完整的支持列表。

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。

## 功能概述
获取系统当前支持的所有快递公司列表，包括每家公司的标准编码（code）和中文名称（name）。

## 使用建议
- **推荐缓存**：这个列表基本不会频繁变动，建议在应用启动时调用一次并缓存到本地
- **应用场景**：适合用于构建快递公司选择器、下拉菜单等UI组件
- **缓存时长**：建议缓存24小时或更久
        """
        params = {}
        body = None
        
        path = "/api/v1/misc/tracking/carriers"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_tracking_detect(self, **kwargs):
        """识别快递公司
        不确定手里的快递单号属于哪家快递公司？这个接口专门做识别，不查物流。

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。

## 功能概述
输入快递单号，系统会根据单号规则快速识别出最可能的快递公司。如果存在多个可能的匹配结果，还会在 `alternatives` 字段中返回备选项，供你参考选择。

## 使用须知
- **识别速度快**：只做规则匹配，不查询物流信息，响应速度通常在100ms内
- **准确率高**：基于各快递公司的单号规则进行智能识别，准确率超过95%
- **备选方案**：当单号规则可能匹配多家快递公司时，会提供所有可能的选项
        """
        params = {}
        body = None
        
        if "query" == "query" and "tracking_number" in kwargs:
            params["tracking_number"] = kwargs["tracking_number"]
        
        path = "/api/v1/misc/tracking/detect"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_tracking_query(self, **kwargs):
        """查询快递物流信息
        买了东西想知道快递到哪儿了？这个接口帮你实时追踪物流状态。

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。

## 功能概述
提供一个快递单号，系统会自动识别快递公司并返回完整的物流轨迹信息。支持中通、圆通、韵达、申通、极兔、顺丰、京东、EMS、德邦等60+国内外主流快递公司。

## 使用须知
- **自动识别**：不知道是哪家快递？系统会根据单号规则自动识别快递公司（推荐使用）
- **手动指定**：如果已知快递公司，可以传递 `carrier_code` 参数，查询速度会更快
- **查询时效**：物流信息实时查询，响应时间通常在1-2秒内
        """
        params = {}
        body = None
        
        if "query" == "query" and "tracking_number" in kwargs:
            params["tracking_number"] = kwargs["tracking_number"]
        
        if "query" == "query" and "carrier_code" in kwargs:
            params["carrier_code"] = kwargs["carrier_code"]
        
        path = "/api/v1/misc/tracking/query"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_weather(self, **kwargs):
        """查询实时天气信息
        出门前，查一下天气总是个好习惯。这个接口为你提供精准、实时的天气数据。

## 功能概述
你可以通过城市名称或高德地图的Adcode来查询指定地区的实时天气状况，包括天气现象、温度、湿度、风向和风力等。

## 使用须知
- **参数优先级**：当你同时提供了 `city` (城市名) 和 `adcode` (城市编码) 两个参数时，系统会 **优先使用 `adcode`** 进行查询，因为它更精确。
- **查询范围**：为了保证查询的准确性，我们的服务仅支持标准的“省”、“市”、“区/县”级别的行政区划名称查询，不保证能查询到乡镇或具体地点。

## 错误处理指南
- **410 Gone**: 这个特殊的错误码意味着你查询的地区无效或不受我们支持。比如你输入了“火星”，或者某个我们无法识别的村庄名称。这个状态码告诉你，这个“资源”是永久性地不可用了。
        """
        params = {}
        body = None
        
        if "query" == "query" and "city" in kwargs:
            params["city"] = kwargs["city"]
        
        if "query" == "query" and "adcode" in kwargs:
            params["adcode"] = kwargs["adcode"]
        
        path = "/api/v1/misc/weather"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_misc_worldtime(self, **kwargs):
        """查询全球任意时区的时间
        需要和国外的朋友开会，想知道他那边现在几点？用这个接口一查便知。

## 功能概述
根据标准的时区名称（例如 'Asia/Shanghai' 或 'Europe/London'），获取该时区的当前准确时间、UTC偏移量、星期等信息。
        """
        params = {}
        body = None
        
        if "query" == "query" and "city" in kwargs:
            params["city"] = kwargs["city"]
        
        path = "/api/v1/misc/worldtime"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _NetworkApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_network_dns(self, **kwargs):
        """执行DNS解析查询
        想知道一个域名指向了哪个IP？或者它的邮件服务器是谁？这个接口就像一个在线的 `dig` 或 `nslookup` 工具。

## 功能概述
你可以查询指定域名的各种DNS记录，包括 `A` (IPv4), `AAAA` (IPv6), `CNAME` (别名), `MX` (邮件交换), `NS` (域名服务器) 和 `TXT` (文本记录)。
        """
        params = {}
        body = None
        
        if "query" == "query" and "domain" in kwargs:
            params["domain"] = kwargs["domain"]
        
        if "query" == "query" and "type" in kwargs:
            params["type"] = kwargs["type"]
        
        path = "/api/v1/network/dns"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_icp(self, **kwargs):
        """查询域名ICP备案信息
        想知道一个网站的背后运营主体是谁吗？如果它是在中国大陆运营的，ICP备案信息可以告诉你答案。

## 功能概述
提供一个域名，查询其在中国工信部的ICP（Internet Content Provider）备案信息。这对于商业合作前的背景调查、验证网站合法性等场景很有帮助。

> [!NOTE]
> **查询范围**
> 此查询仅对在中国大陆工信部进行过备案的域名有效。对于国外域名或未备案的域名，将查询不到结果。
        """
        params = {}
        body = None
        
        if "query" == "query" and "domain" in kwargs:
            params["domain"] = kwargs["domain"]
        
        path = "/api/v1/network/icp"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_ipinfo(self, **kwargs):
        """查询指定IP或域名的归属信息
        想知道一个IP地址或域名来自地球的哪个角落？这个接口可以帮你定位它。你可以选择使用默认的GeoIP数据库，也可以指定 `source=commercial` 参数来查询更详细的商业级IP归属信息。

## 功能概述
提供一个公网IPv4、IPv6地址或域名，我们会利用GeoIP数据库查询并返回它的地理位置（国家、省份、城市）、经纬度、以及所属的运营商（ISP）和自治系统（ASN）信息。这在网络安全分析、访问来源统计等领域非常有用。

当使用 `source=commercial` 参数时，接口将调用高性能商业API，提供更精确的市、区、运营商、时区、海拔等信息。请注意，商业查询的响应时间可能会稍长。
        """
        params = {}
        body = None
        
        if "query" == "query" and "ip" in kwargs:
            params["ip"] = kwargs["ip"]
        
        if "query" == "query" and "source" in kwargs:
            params["source"] = kwargs["source"]
        
        path = "/api/v1/network/ipinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_myip(self, **kwargs):
        """获取你的公网IP及归属信息
        想知道你自己的出口公网IP是多少吗？这个接口就是你的“网络身份证”。你可以选择使用默认的GeoIP数据库，也可以指定 `source=commercial` 参数来查询更详细的商业级IP归属信息。

## 功能概述
调用此接口，它会返回你（即发起请求的客户端）的公网IP地址，并附带与 `/network/ipinfo` 接口相同的地理位置和网络归属信息。非常适合用于在网页上向用户展示他们自己的IP和地理位置。

当使用 `source=commercial` 参数时，接口将调用高性能商业API，提供更精确的市、区、运营商、时区、海拔等信息。请注意，商业查询的响应时间可能会稍长。
        """
        params = {}
        body = None
        
        if "query" == "query" and "source" in kwargs:
            params["source"] = kwargs["source"]
        
        path = "/api/v1/network/myip"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_ping(self, **kwargs):
        """从服务器Ping指定主机
        想知道从我们的服务器到你的服务器网络延迟高不高？这个工具可以帮你测试网络连通性。

## 功能概述
这个接口会从我们的服务器节点对你指定的主机（域名或IP地址）执行 ICMP Ping 操作。它会返回最小、最大、平均延迟以及丢包率等关键指标，是诊断网络问题的得力助手。
        """
        params = {}
        body = None
        
        if "query" == "query" and "host" in kwargs:
            params["host"] = kwargs["host"]
        
        path = "/api/v1/network/ping"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_pingmyip(self, **kwargs):
        """从服务器Ping你的客户端IP
        这是一个非常方便的快捷接口，想知道你的网络到我们服务器的回程延迟吗？点一下就行！

## 功能概述
这个接口是 `/network/myip` 和 `/network/ping` 的结合体。它会自动获取你客户端的公网IP，然后从我们的服务器Ping这个IP，并返回延迟数据。这对于快速判断你本地网络到服务器的连接质量非常有用。
        """
        params = {}
        body = None
        
        path = "/api/v1/network/pingmyip"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_portscan(self, **kwargs):
        """扫描远程主机的指定端口
        想检查一下你的服务器上某个端口（比如SSH的22端口或者Web的80端口）是否对外开放？这个工具可以帮你快速确认。

## 功能概述
你可以指定一个主机和端口号，我们的服务器会尝试连接该端口，并告诉你它是开放的（open）、关闭的（closed）还是超时了（timeout）。这对于网络服务配置检查和基本的安全扫描很有用。
        """
        params = {}
        body = None
        
        if "query" == "query" and "host" in kwargs:
            params["host"] = kwargs["host"]
        
        if "query" == "query" and "port" in kwargs:
            params["port"] = kwargs["port"]
        
        if "query" == "query" and "protocol" in kwargs:
            params["protocol"] = kwargs["protocol"]
        
        path = "/api/v1/network/portscan"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_urlstatus(self, **kwargs):
        """检查URL的可访问性状态
        你的网站或API还好吗？用这个接口给它做个快速“体检”吧。

## 功能概述
提供一个URL，我们会向它发起一个请求，并返回其HTTP响应状态码。这是一种简单而有效的服务可用性监控方法。

> [!TIP]
> **性能优化**：为了提高效率并减少对目标服务器的负载，我们实际发送的是 `HEAD` 请求，而不是 `GET` 请求。`HEAD` 请求只会获取响应头，而不会下载整个页面内容，因此速度更快。
        """
        params = {}
        body = None
        
        if "query" == "query" and "url" in kwargs:
            params["url"] = kwargs["url"]
        
        path = "/api/v1/network/urlstatus"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_whois(self, **kwargs):
        """查询域名的WHOIS注册信息
        想知道一个域名是什么时候注册的、注册商是谁、什么时候到期？WHOIS信息可以告诉你这一切。

## 功能概述
这是一个在线的WHOIS查询工具。你可以通过如下两种方式获取WHOIS信息：

- **默认行为**（不带参数）：`GET /api/v1/network/whois?domain=google.com`
  - 返回一个JSON对象，`whois` 字段为原始、未处理的WHOIS文本字符串。
- **JSON格式化**：`GET /api/v1/network/whois?domain=google.com&format=json`
  - 返回一个JSON对象，`whois` 字段为解析后的JSON对象，包含WHOIS信息中的键值对。

这样你既可以获得最全的原始信息，也可以方便地处理结构化数据。
        """
        params = {}
        body = None
        
        if "query" == "query" and "domain" in kwargs:
            params["domain"] = kwargs["domain"]
        
        if "query" == "query" and "format" in kwargs:
            params["format"] = kwargs["format"]
        
        path = "/api/v1/network/whois"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_network_wxdomain(self, **kwargs):
        """检查域名在微信中的访问状态
        准备在微信里推广你的网站？最好先检查一下域名是否被“拉黑”了。

## 功能概述
这个接口可以帮你查询一个域名在微信内置浏览器中的访问状态，即是否被微信封禁。这对于做微信生态推广和营销的开发者来说至关重要。
        """
        params = {}
        body = None
        
        if "query" == "query" and "domain" in kwargs:
            params["domain"] = kwargs["domain"]
        
        path = "/api/v1/network/wxdomain"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _PoemApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_saying(self, **kwargs):
        """随机获取一句诗词或名言
        想在你的应用里每天展示一句不一样的话，给用户一点小小的惊喜吗？这个“一言”接口就是为此而生。

## 功能概述
每次调用，它都会从我们精心收集的、包含数千条诗词、动漫台词、名人名言的语料库中，随机返回一条。你可以用它来做网站首页的Slogan、应用的启动语，或者任何需要灵感点缀的地方。
        """
        params = {}
        body = None
        
        path = "/api/v1/saying"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _RandomApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_answerbook_ask(self, **kwargs):
        """获取答案之书的神秘答案 (GET)
        想要获得人生问题的神秘答案吗？答案之书API提供了一个神奇8球风格的问答服务，你可以提问并获得随机的神秘答案。

## 功能概述
通过向答案之书提问，你将获得一个充满智慧（或许）的随机答案。这个API支持通过查询参数或POST请求体两种方式提问。

## 使用须知

> [!TIP]
> **提问技巧**
> - 提出明确的问题会获得更好的体验
> - 问题不能为空
> - 支持中文问题
> - 答案具有随机性，仅供娱乐参考
        """
        params = {}
        body = None
        
        if "query" == "query" and "question" in kwargs:
            params["question"] = kwargs["question"]
        
        path = "/api/v1/answerbook/ask"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_random_image(self, **kwargs):
        """随机二次元、风景、动漫图片壁纸
        需要一张随机图片作为占位符或者背景吗？这个接口是你的不二之选。

## 功能概述
这是一个非常简单的接口，它会从我们庞大的图库和精选外部图床中随机挑选一张图片，然后通过 302 重定向让你直接访问到它。这使得它可以非常方便地直接用在 HTML 的 `<img>` 标签中。

你可以通过 `/api/v1/random/image?category=acg&type=4k` 这样的请求获取由UapiPro服务器提供的图片，也可以通过 `/api/v1/random/image?category=ai_drawing` 获取由外部图床精选的图片。

如果你不提供任何 category 参数，程序会从所有图片（包括本地的和URL的）中随机抽取一张（**全局随机图片不包含ikun和AI绘画**）。

> [!TIP]
> 如果你需要更精确地控制图片类型，请使用 `/image/random/{category}/{type}` 接口。

### 支持的主类别与子类别
- **UapiPro服务器图片**
  - **furry**（福瑞）
    - z4k
    - szs8k
    - s4k
    - 4k
  - **bq**（表情包/趣图）
    - youshou
    - xiongmao
    - waiguoren
    - maomao
    - ikun
    - eciyuan
  - **acg**（二次元动漫）
    - pc
    - mb
- **外部图床精选图片**
  - **ai_drawing**: AI绘画。
  - **general_anime**: 动漫图。
  - **landscape**: 风景图。
  - **mobile_wallpaper**: 手机壁纸。
  - **pc_wallpaper**: 电脑壁纸。
- **混合动漫**
  - **anime**: 混合了UapiPro服务器的acg和外部图床的general_anime分类下的图片。

> [!NOTE]
> 默认全局随机（未指定category参数）时，不会包含ikun和AI绘画（ai_drawing）类别的图片。

        """
        params = {}
        body = None
        
        if "query" == "query" and "category" in kwargs:
            params["category"] = kwargs["category"]
        
        if "query" == "query" and "type" in kwargs:
            params["type"] = kwargs["type"]
        
        path = "/api/v1/random/image"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_random_string(self, **kwargs):
        """生成高度可定制的随机字符串
        无论是需要生成一个安全的随机密码、一个唯一的Token，还是一个简单的随机ID，这个接口都能满足你。

## 功能概述
你可以精确地控制生成字符串的长度和字符集类型，非常灵活。

## 使用须知

> [!TIP]
> **字符集类型 `type` 详解**
> 你可以通过 `type` 参数精确控制生成的字符集：
> - **`numeric`**: 纯数字 (0-9)
> - **`lower`**: 纯小写字母 (a-z)
> - **`upper`**: 纯大写字母 (A-Z)
> - **`alpha`**: 大小写字母 (a-zA-Z)
> - **`alphanumeric`** (默认): 数字和大小写字母 (0-9a-zA-Z)
> - **`hex`**: 十六进制字符 (0-9a-f)
        """
        params = {}
        body = None
        
        if "query" == "query" and "length" in kwargs:
            params["length"] = kwargs["length"]
        
        if "query" == "query" and "type" in kwargs:
            params["type"] = kwargs["type"]
        
        path = "/api/v1/random/string"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_answerbook_ask(self, **kwargs):
        """获取答案之书的神秘答案 (POST)
        通过POST请求向答案之书提问并获得神秘答案。

## 功能概述
与GET方式相同，但通过JSON请求体发送问题，适合在需要发送较长问题或希望避免URL编码问题的场景中使用。

## 请求体格式
请求体必须是有效的JSON格式，包含question字段。
        """
        params = {}
        body = None
        
        path = "/api/v1/answerbook/ask"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _SocialApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_github_repo(self, **kwargs):
        """获取GitHub仓库信息
        需要快速获取一个GitHub仓库的核心信息？这个接口为你聚合了最有价值的数据，避免了多次调用GitHub官方API的麻烦，并且内置了缓存优化，速度更快、更稳定。

### 聚合高价值数据
一次请求，即可获得以下信息：
- **核心指标**: `star`, `fork`, `open_issues` 等关键统计数据。
- **项目详情**: 描述、主页、分支、语言、话题标签、开源协议。
- **参与者信息**: 获取协作者(`collaborators`)和推断的维护者(`maintainers`)列表，包括他们的公开邮箱（如果可用）。

> [!NOTE]
> `collaborators` 字段在私有仓库或权限受限时可能为空。`maintainers` 是根据最新提交记录推断的，仅供参考。

### 性能与稳定性
我们内置了多级缓存，有效避免触发GitHub的API速率限制。对于需要更高请求额度的用户，可以联系我们定制接口。
        """
        params = {}
        body = None
        
        if "query" == "query" and "repo" in kwargs:
            params["repo"] = kwargs["repo"]
        
        path = "/api/v1/github/repo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_bilibili_archives(self, **kwargs):
        """获取Bilibili用户投稿列表
        想要获取UP主的所有投稿视频？或者想在你的应用里展示创作者的作品集？这个接口能帮你轻松实现。

## 功能概述
通过用户的 `mid`（用户ID），你可以获取该UP主的投稿视频列表。接口支持关键词搜索、分页加载和多种排序方式，让你能够灵活地展示和分析创作者的内容。

## 参数说明
- **`mid` (用户ID)**: B站用户的mid，必填参数。
- **`keywords` (搜索关键词)**: 可选，用于在该UP主的投稿中搜索特定关键词。
- **`orderby` (排序方式)**: 
  - `pubdate`: 按最新发布排序（默认）
  - `views`: 按最多播放排序
- **`ps` (每页条数)**: 默认20条。
- **`pn` (页码)**: 默认1，用于分页。

## 响应体字段说明
- **`total` (总稿件数)**: UP主的投稿总数。
- **`page` (当前页码)**: 当前返回的页码。
- **`size` (每页数量)**: 每页返回的视频数量。
- **`videos` (视频列表)**: 包含当前页的所有视频信息：
  - `aid`: 视频的AV号
  - `bvid`: 视频的BV号
  - `title`: 视频标题
  - `cover`: 封面URL
  - `duration`: 时长（秒）
  - `play_count`: 播放量
  - `publish_time`: 发布时间戳
  - `create_time`: 创建时间戳
  - `state`: 视频状态
  - `is_ugc_pay`: 是否付费视频（0=免费，1=付费）
  - `is_interactive`: 是否为互动视频
        """
        params = {}
        body = None
        
        if "query" == "query" and "mid" in kwargs:
            params["mid"] = kwargs["mid"]
        
        if "query" == "query" and "keywords" in kwargs:
            params["keywords"] = kwargs["keywords"]
        
        if "query" == "query" and "orderby" in kwargs:
            params["orderby"] = kwargs["orderby"]
        
        if "query" == "query" and "ps" in kwargs:
            params["ps"] = kwargs["ps"]
        
        if "query" == "query" and "pn" in kwargs:
            params["pn"] = kwargs["pn"]
        
        path = "/api/v1/social/bilibili/archives"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_bilibili_liveroom(self, **kwargs):
        """获取Bilibili直播间信息
        想知道你喜欢的主播开播了吗？或者想在你的应用里集成B站直播间状态？这个接口能满足你。

## 功能概述
这是一个智能接口，你可以用主播的 `mid` (用户ID) 或者直播间的 `room_id` (长号或短号)来查询。它会返回直播间的详细信息，包括是否在直播、标题、封面、人气、分区等。

## 响应体字段说明
- **`live_status` (直播状态)**: `0` 代表未开播，`1` 代表直播中，`2` 代表轮播中。
        """
        params = {}
        body = None
        
        if "query" == "query" and "mid" in kwargs:
            params["mid"] = kwargs["mid"]
        
        if "query" == "query" and "room_id" in kwargs:
            params["room_id"] = kwargs["room_id"]
        
        path = "/api/v1/social/bilibili/liveroom"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_bilibili_replies(self, **kwargs):
        """获取Bilibili视频评论
        想要分析B站视频的评论区？这个接口可以帮你轻松获取评论数据，包括热门评论和最新评论，还支持分页加载。

## 功能概述
通过视频的 `oid`（通常就是视频的`aid`），你可以分页获取该视频的评论区内容。你可以指定排序方式和分页参数，来精确地获取你需要的数据。

## 参数说明
- **`sort` (排序方式)**: `0`=按时间排序, `1`=按点赞数排序, `2`=按回复数排序。默认为按时间排序。

## 响应体字段说明
- **`hots` (热门评论)**: 仅在请求第一页时，可能会返回热门评论列表。其结构与 `replies` 中的对象一致。
- **`replies` (评论列表)**: 这是一个数组，包含了当前页的评论。其中：
  - `root`: 指向根评论的ID。如果评论本身就是根评论，则为 `0`。
  - `parent`: 指向该条回复所回复的上一级评论ID。如果评论是根评论，则为 `0`。
        """
        params = {}
        body = None
        
        if "query" == "query" and "oid" in kwargs:
            params["oid"] = kwargs["oid"]
        
        if "query" == "query" and "sort" in kwargs:
            params["sort"] = kwargs["sort"]
        
        if "query" == "query" and "ps" in kwargs:
            params["ps"] = kwargs["ps"]
        
        if "query" == "query" and "pn" in kwargs:
            params["pn"] = kwargs["pn"]
        
        path = "/api/v1/social/bilibili/replies"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_bilibili_userinfo(self, **kwargs):
        """查询Bilibili用户信息
        想在你的应用里集成B站用户资料展示？这个接口可以轻松获取用户的公开信息。

## 功能概述
通过一个用户的UID（那一串纯数字ID），你可以查询到该用户的昵称、性别、头像、等级、签名等一系列公开的基本信息。
        """
        params = {}
        body = None
        
        if "query" == "query" and "uid" in kwargs:
            params["uid"] = kwargs["uid"]
        
        path = "/api/v1/social/bilibili/userinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_bilibili_videoinfo(self, **kwargs):
        """获取Bilibili视频详细信息
        想在你的应用里展示B站视频的详细信息吗？无论是封面、标题，还是播放量、UP主信息，这个接口都能一网打尽。

## 功能概述
通过提供视频的 `aid` 或 `bvid`，你可以获取到该视频的完整元数据，包括多P信息、UP主资料、数据统计等。

## 响应体字段说明
- **`copyright` (视频类型)**: `1` 代表原创，`2` 代表转载。
- **`owner` (UP主信息)**: 包含 `mid`, `name`, `face` 等UP主的基本资料。
- **`stat` (数据统计)**: 包含了播放、弹幕、评论、点赞、投币、收藏、分享等核心数据。
- **`pages` (分P列表)**: 这是一个数组，包含了视频的每一个分P的信息，即使是单P视频也会有一个元素。
        """
        params = {}
        body = None
        
        if "query" == "query" and "aid" in kwargs:
            params["aid"] = kwargs["aid"]
        
        if "query" == "query" and "bvid" in kwargs:
            params["bvid"] = kwargs["bvid"]
        
        path = "/api/v1/social/bilibili/videoinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_qq_groupinfo(self, **kwargs):
        """获取QQ群名称、头像、简介
        想在你的应用里展示QQ群信息？这个接口让你轻松获取群名称、群头像、群简介等公开信息。它能帮你快速构建社群导航站、群聊推荐系统，或是为你的数据分析工具提供可靠的数据源。无论是展示群聊卡片、生成加群链接，还是进行社群数据统计，这个接口都能满足你的需求。

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。

## 功能概述
你只需要提供一个QQ群号（5-12位纯数字），接口就会返回该群的完整公开信息。我们会先验证群号的有效性，确保返回的数据准确可靠。接口的响应速度快，数据结构清晰，非常适合集成到各类应用场景中。

## 返回数据说明
接口会返回以下QQ群的关键信息：
- **群基础信息**: 包括群号、群名称，让你能够准确识别和展示群聊。
- **视觉素材**: 提供群头像URL（标准100x100尺寸），可直接用于在你的界面中展示群聊图标。
- **群介绍资料**: 包含群描述/简介和群标签，帮助用户了解群聊的主题和特色。
- **便捷入口**: 返回加群链接（二维码URL），方便用户一键加入感兴趣的群聊。
- **数据时效**: 提供最后更新时间戳，让你了解数据的新鲜度。

所有返回的数据都遵循标准的JSON格式，字段命名清晰，便于解析和使用。无论你是在做网页端、移动端还是后端服务，都能轻松集成。
        """
        params = {}
        body = None
        
        if "query" == "query" and "group_id" in kwargs:
            params["group_id"] = kwargs["group_id"]
        
        path = "/api/v1/social/qq/groupinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_social_qq_userinfo(self, **kwargs):
        """独家获取QQ号头像、昵称
        这是一个功能丰富的QQ用户信息查询接口，能够获取QQ用户的详细公开信息。

> [!VIP]
> 我们在近日优化了此接口，速度应该会更加快了。 

## 功能概述
通过QQ号查询用户的详细信息，包括基础资料、等级信息、VIP状态等。返回的信息丰富全面，适合用于用户画像分析、社交应用集成等场景。

## 数据字段说明
- **基础信息**: 昵称、个性签名、头像、年龄、性别
- **联系信息**: QQ邮箱、个性域名(QID)
- **等级信息**: QQ等级、VIP状态和等级
- **时间信息**: 注册时间、最后更新时间
        """
        params = {}
        body = None
        
        if "query" == "query" and "qq" in kwargs:
            params["qq"] = kwargs["qq"]
        
        path = "/api/v1/social/qq/userinfo"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _StatusApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_status_ratelimit(self, **kwargs):
        """获取API限流器实时状态
        想了解我们API的当前负载情况吗？这个接口为你提供了服务的“心电图”。

## 功能概述
此接口返回我们后端自适应限流器的实时状态。你可以看到当前并发请求数、并发上限、系统负载、请求接受/拒绝数等核心指标。这对于监控API健康状况和性能表现至关重要。

> [!IMPORTANT]
> 此接口为管理接口，需要提供有效的管理员级别API密钥才能访问。

### 认证方式
请在请求头中添加 `Authorization: Bearer <你的API密钥>`。
        """
        params = {}
        body = None
        
        if "header" == "query" and "Authorization" in kwargs:
            params["Authorization"] = kwargs["Authorization"]
        
        path = "/api/v1/status/ratelimit"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_status_usage(self, **kwargs):
        """获取API端点使用统计
        想知道哪个API接口最受欢迎吗？这个接口提供了详细的“账单”。

## 功能概述
此接口用于获取每个API端点（Endpoint）的使用情况统计。你可以查询所有端点的列表，也可以通过 `path` 参数指定查询某一个特定端点。返回信息包括调用次数和平均处理时长
        """
        params = {}
        body = None
        
        if "query" == "query" and "path" in kwargs:
            params["path"] = kwargs["path"]
        
        path = "/api/v1/status/usage"
        
        return self._http.request("GET", path, params=params, json=body)
    

class _TextApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_text_md_5(self, **kwargs):
        """计算文本的MD5哈希值(GET)
        一个快速计算文本 MD5 哈希值的工具，适用于短文本且不关心参数暴露的场景。

## 功能概述
通过GET请求的查询参数传入文本，返回其32位小写的MD5哈希值。

> [!NOTE]
> 对于较长或敏感的文本，我们推荐使用本接口的 POST 版本，以避免URL长度限制和参数暴露问题。
        """
        params = {}
        body = None
        
        if "query" == "query" and "text" in kwargs:
            params["text"] = kwargs["text"]
        
        path = "/api/v1/text/md5"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_text_aes_decrypt(self, **kwargs):
        """使用AES算法解密文本
        收到了用AES加密的密文？把它、密钥和随机数（nonce）交给我们，就能还原出原始内容。

## 功能概述
这是一个标准的AES解密接口。你需要提供经过Base64编码的密文、加密时使用的密钥和nonce（随机数，通常为16字节字符串）。

> [!IMPORTANT]
> **关于密钥 `key`**
> 我们支持 AES-128, AES-192, 和 AES-256。请确保你提供的密钥 `key` 的长度（字节数）正好是 **16**、**24** 或 **32**，以分别对应这三种加密强度。
> 
> **关于随机数 `nonce`**
> 通常为16字节字符串，需与加密时一致。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/aes/decrypt"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_aes_encrypt(self, **kwargs):
        """使用AES算法加密文本
        需要安全地传输或存储一些文本信息？AES加密是一个可靠的选择。

## 功能概述
这是一个标准的AES加密接口。你提供需要加密的明文和密钥，我们返回经过Base64编码的密文。

> [!IMPORTANT]
> **关于密钥 `key`**
> 我们支持 AES-128, AES-192, 和 AES-256。请确保你提供的密钥 `key` 的长度（字节数）正好是 **16**、**24** 或 **32**，以分别对应这三种加密强度。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/aes/encrypt"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_analyze(self, **kwargs):
        """多维度分析文本内容
        想知道一篇文章有多少字、多少个词、或者多少行？这个接口可以帮你快速统计。

## 功能概述
你提供一段文本，我们会从多个维度进行分析，并返回其字符数、词数、句子数、段落数和行数。这对于文本编辑、内容管理等场景非常有用。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/analyze"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_base_64_decode(self, **kwargs):
        """解码Base64编码的文本
        这是一个简单实用的 Base64 解码工具。

## 功能概述
你提供一个 Base64 编码的字符串，我们帮你解码成原始的 UTF-8 文本。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/base64/decode"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_base_64_encode(self, **kwargs):
        """将文本进行Base64编码
        这是一个简单实用的 Base64 编码工具。

## 功能概述
你提供一段原始文本，我们帮你转换成 Base64 编码的字符串。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/base64/encode"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_md_5(self, **kwargs):
        """计算文本的MD5哈希值 (POST)
        一个用于计算文本 MD5 哈希值的标准工具，推荐使用此版本。

## 功能概述
通过POST请求的表单体传入文本，返回其32位小写的MD5哈希值。相比GET版本，此方法更适合处理较长或包含敏感信息的文本。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/md5"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_text_md_5_verify(self, **kwargs):
        """校验MD5哈希值
        下载了一个文件，想确认它在传输过程中有没有损坏？校验MD5值是最常用的方法。

## 功能概述
你提供原始文本和一个MD5哈希值，我们帮你计算文本的哈希，并与你提供的哈希进行比对，告诉你它们是否匹配。这在文件完整性校验等场景下非常有用。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/md5/verify"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _TranslateApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_ai_translate_languages(self, **kwargs):
        """获取AI翻译支持的语言和配置
        获取AI智能翻译服务支持的完整语言列表、翻译风格选项、上下文场景选项以及性能指标信息。这个接口对于需要在前端动态展示翻译配置选项的应用非常有用，它会返回当前AI翻译服务所支持的所有语言代码、原生名称、翻译风格说明、上下文场景描述，以及服务的性能特征和限制信息。通过此接口，开发者可以构建用户友好的翻译界面，让用户选择合适的翻译参数。
        """
        params = {}
        body = None
        
        path = "/api/v1/ai/translate/languages"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_ai_translate(self, **kwargs):
        """AI智能翻译
        这是一个商业级的AI智能翻译服务，采用最新的神经网络翻译技术和大语言模型，提供远超传统机器翻译的质量。它不仅能够智能处理单个文本翻译，还支持高效的批量文本翻译，并且具备上下文感知、风格适配、格式保留等高级功能。

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者深度集成和测试。未来，它将转为付费API，为用户提供更稳定、更智能的翻译服务。

## 功能概述

- **智能双模式**: 支持单个文本翻译和批量文本翻译的统一接口设计，自动识别请求类型并提供相应的翻译服务。系统会根据输入自动判断是处理单条文本还是批量文本，无需使用不同的接口。
- **多风格适配**: 提供随意口语化、专业商务、学术正式、文学艺术四种翻译风格，能够根据不同场景需求调整翻译的语言风格和表达方式。
- **上下文感知**: 支持通用、商务、技术、医疗、法律、市场营销、娱乐、教育、新闻等九种专业领域的上下文翻译，确保术语准确性和表达地道性。
- **高质量保证**: 内置质量评估系统，对每次翻译结果进行流畅度、准确度、完整性评分，并提供置信度分数和替代翻译建议。
- **智能解释**: 提供关键词组翻译注释、文化背景说明和语法结构分析，帮助用户理解翻译逻辑和文化差异。
- **高效批量**: 批量翻译支持最多50条文本，总计10万字符，配备智能并发控制（1-10并发）和失败重试机制。
- **快速模式**: 提供快速模式选项，在保证95%+准确率的前提下，响应时间缩短至800ms内，适合实时翻译和聊天应用。
- **格式保留**: 智能识别并保持原文的格式结构，包括换行、缩进、特殊符号等，确保翻译后的文本保持良好的可读性。
        """
        params = {}
        body = None
        
        if "query" == "query" and "target_lang" in kwargs:
            params["target_lang"] = kwargs["target_lang"]
        
        path = "/api/v1/ai/translate"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_translate_stream(self, **kwargs):
        """流式翻译（中英互译）
        想让翻译结果像打字机一样逐字显示出来？这个流式翻译接口能实现这种效果。

## 功能概述
不同于传统翻译API一次性返回完整结果，这个接口会实时地、一个字一个字地把翻译内容推给你（就像ChatGPT回复消息那样），非常适合用在聊天应用、直播字幕等需要即时反馈的场景。

## 它能做什么
- **中英互译**：支持中文和英文之间的双向翻译
- **自动识别**：不确定源语言？设置为 `auto` 让我们自动检测
- **逐字返回**：翻译结果会像打字机一样逐字流式返回，用户体验更流畅
- **音频朗读**：部分翻译结果会附带音频链接，方便朗读

## 支持的语言
目前专注于中英互译，支持以下选项：
- `中文`（简体/繁体）
- `英文`
- `auto`（自动检测）
        """
        params = {}
        body = None
        
        path = "/api/v1/translate/stream"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_translate_text(self, **kwargs):
        """多语言文本翻译
        需要跨越语言的鸿沟进行交流？这个翻译接口是你可靠的'同声传译'。

## 功能概述
你可以将一段源语言文本（我们能自动检测源语言）翻译成你指定的任何目标语言。无论是中译英、日译法，都不在话下。

## 支持的语言
我们支持超过100种语言的互译，包括但不限于：中文（简体/繁体）、英语、日语、韩语、法语、德语、西班牙语、俄语、阿拉伯语等主流语言，以及各种小语种。详见下方参数列表。
        """
        params = {}
        body = None
        
        if "query" == "query" and "to_lang" in kwargs:
            params["to_lang"] = kwargs["to_lang"]
        
        path = "/api/v1/translate/text"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _WebparseApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_web_tomarkdown_async_status(self, **kwargs):
        """查询网页转换任务状态和结果
        提交了URL转Markdown任务后，想知道处理进度和结果？这个接口可以帮你实时追踪。

## 功能概述

通过之前提交任务时获得的任务ID，你可以查询该任务的当前状态、处理进度以及最终结果。任务结果会在缓存中保存30分钟，期间可以重复查询，非常方便。

任务有五种状态：等待处理（pending）时进度为0%；处理中（processing）时进度在10-90%之间；已完成（completed）时进度为100%并返回Markdown内容；失败（failed）时会返回错误信息；超时（timeout）表示任务处理时间超过60秒已被取消。建议采用指数退避策略进行轮询，初始延迟1秒，每次延迟增加20%，最大延迟5秒。当状态为已完成、失败或超时时停止轮询。

系统会自动管理任务生命周期，单个任务最长处理时间为60秒，任务结果保存30分钟后自动清理，每5分钟清理一次过期任务。

## 任务状态说明

| 状态 | 说明 | 进度 | 轮询建议 |
|------|------|------|----------|
| `pending` | 等待处理 | 0% | 立即开始轮询 |
| `processing` | 处理中 | 10-90% | 每2-5秒轮询一次 |
| `completed` | 已完成 | 100% | 停止轮询，获取结果 |
| `failed` | 失败 | 100% | 停止轮询，查看错误信息 |
| `timeout` | 超时 | 100% | 停止轮询，任务已取消 |
        """
        params = {}
        body = None
        
        if "path" == "query" and "task_id" in kwargs:
            params["task_id"] = kwargs["task_id"]
        
        path = "/api/v1/web/tomarkdown/async/{task_id}"
        
        if "task_id" in kwargs:
            path = path.replace("{" + "task_id" + "}", str(kwargs["task_id"]))
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_webparse_extractimages(self, **kwargs):
        """提取网页中的所有图片
        想一次性“打包”一个网页上的所有图片吗？这个接口可以帮你实现。

## 功能概述
你提供一个网页的URL，我们会访问该页面，解析其HTML内容，并提取出所有 `<img>` 标签中的图片链接，然后将这些链接列表返回给你。非常适合用于制作图片采集器或素材下载工具。
        """
        params = {}
        body = None
        
        if "query" == "query" and "url" in kwargs:
            params["url"] = kwargs["url"]
        
        path = "/api/v1/webparse/extractimages"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def get_webparse_metadata(self, **kwargs):
        """抓取并解析网页的元数据
        当你在应用中需要展示一个链接的预览时（就像微信或Telegram里那样），这个接口能帮你轻松获取所需信息。

## 功能概述
你提供一个网页的URL，我们会抓取并解析它的 `<head>` 部分，提取出关键的元数据（Metadata），如页面标题（Title）、描述（Description）、关键词（Keywords）以及网站图标（Favicon）等。
        """
        params = {}
        body = None
        
        if "query" == "query" and "url" in kwargs:
            params["url"] = kwargs["url"]
        
        path = "/api/v1/webparse/metadata"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_web_tomarkdown_async(self, **kwargs):
        """深度抓取网页转Markdown
        想要将复杂的网页转换为结构清晰的Markdown？这个接口采用异步处理模式，特别适合处理大型网页、复杂网站或需要长时间处理的转换任务。

## 功能概述

> [!VIP]
>本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。

UAPI Pro平台推出的异步网页转Markdown API能够将任意网页URL转换为结构清晰、格式优美的Markdown文本。提交任务后立即返回任务ID，不会阻塞客户端等待。您可以通过任务ID实时查询转换进度和处理状态，支持长达60秒的处理时间，轻松应对大型网站、需要JS渲染的单页应用等复杂页面。任务结果会缓存30分钟，期间可重复查询，过期任务自动清理无需手动管理。

此API采用先进算法，自动识别并抓取网页主体内容，精准剔除广告、导航栏、页眉页脚等无关元素。完美保留原文的格式，包括标题、列表、代码块、表格、引用、图片等，并输出为兼容性强的GitHub Flavored Markdown (GFM) 格式。同时会自动解析并提取文章标题、作者、发布日期、站点名称等关键元数据，并将其格式化为标准的YAML Front Matter，方便后续处理和CMS集成。

## 使用流程

调用本接口提交URL转换任务后，会立即获得一个唯一的任务ID。随后使用任务ID调用查询接口，获取任务状态和进度。任务完成后，从查询接口的响应中获取Markdown内容。
        """
        params = {}
        body = None
        
        if "query" == "query" and "url" in kwargs:
            params["url"] = kwargs["url"]
        
        path = "/api/v1/web/tomarkdown/async"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _MinGanCiShiBieApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_sensitive_word_analyze_query(self, **kwargs):
        """查询参数分析
        通过URL查询参数分析单个关键词，便于GET请求调用。
        """
        params = {}
        body = None
        
        if "query" == "query" and "keyword" in kwargs:
            params["keyword"] = kwargs["keyword"]
        
        path = "/api/v1/sensitive-word/analyze-query"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_sensitive_word_analyze(self, **kwargs):
        """分析敏感词
        分析单个或多个关键词的敏感程度，返回详细的风险评分和分析结果。

> [!VIP]
> 本API基于先进的分析模型，提供三级缓存策略和并发处理能力。

## 功能概述

- **模型驱动**: 使用先进的分析模型进行语义分析。
- **高性能**: 采用三级缓存策略（持久化存储 → 统一缓存 → 模型分析），确保高频请求的响应速度。
- **并发支持**: 支持批量并发处理，单次最多可分析100个关键词。
- **详细评分**: 提供色情、辱骂、暴力三个维度的详细风险评分。
- **变体识别**: 能够自动识别关键词的常见变体形式，如拼音、缩写等。

## 风险评分说明

返回的 `s` 字段包含三个维度的风险评分，范围均为0.0至1.0：

- **s[0] - 色情风险**: 评估内容涉及色情信息的程度。
- **s[1] - 辱骂/仇恨言论风险**: 评估内容是否包含侮辱性或仇恨性言论。
- **s[2] - 暴力/威胁风险**: 评估内容是否涉及暴力或威胁信息。

风险等级可参考：0.0-0.3为低风险，0.3-0.7为中等风险，0.7-1.0为高风险。

## 响应字段说明

| 字段 | 类型 | 说明 |
|------|------|------|
| `results` | array | 分析结果对象的数组。 |
| `results[].k` | string | 您在请求中提供的原始关键词。 |
| `results[].r` | string | 模型对该关键词的分析过程和判断理由的简要说明。 |
| `results[].s` | array[float] | 一个包含三个浮点数的数组，分别代表[色情, 辱骂, 暴力]三个维度的风险评分。分值范围从0.0到1.0，越高代表风险越大。 |
| `results[].v` | array[string] | 模型识别出的该关键词的常见变体形式，例如拼音、谐音、缩写等。 |
| `results[].t` | array[string] | 根据分析结果为关键词附加的分类标签，便于进行程序化处理和过滤。 |
| `results[].d` | string | 对整体分析结果的一句简短总结，适合直接展示给用户或记录在日志中。 |
| `total` | integer | 本次请求成功分析的关键词总数。 |
      
        """
        params = {}
        body = None
        
        path = "/api/v1/sensitive-word/analyze"
        
        return self._http.request("POST", path, params=params, json=body)
    
    def post_sensitive_word_quick_check(self, **kwargs):
        """敏感词检测（快速）
        在你的社区或应用中，需要来过滤掉不和谐的声音吗？这个接口可以助你一臂之力。

## 功能概述

我们对敏感词检测接口进行了大幅升级，现在采用高效的 **Aho-Corasick 算法**，实现了多模式字符串匹配。这意味着你不再需要手动编写复杂的正则表达式，系统会自动高效地检测出文本中的所有敏感词。

### 主要特性

- **高性能算法**：基于 Aho-Corasick 算法，单次扫描即可检测多个敏感词模式
- **简繁体支持**：自动识别和处理简体中文、繁体中文内容
- **多模匹配**：无需编写正则表达式，系统内置智能匹配逻辑
- **快速响应**：相比传统方法，检测速度显著提升

无论是论坛、社交平台还是评论系统，这个接口都能帮你快速构建内容审核功能。
        """
        params = {}
        body = None
        
        path = "/api/v1/text/profanitycheck"
        
        return self._http.request("POST", path, params=params, json=body)
    

class _ZhiNengSouSuoApi:
    def __init__(self, http: _HTTP):
        self._http = http

    
    def get_search_engines(self, **kwargs):
        """获取搜索引擎信息
        获取 UAPI Pro Search 引擎的详细信息，包括支持的功能特性、参数限制和使用说明。

## 功能概述

此接口返回搜索引擎的完整配置信息，你可以用它来：
- 了解搜索引擎支持哪些功能（如站内搜索、文件类型过滤等）
- 获取参数的默认值和限制范围
- 查看当前引擎版本和可用状态

适合在应用初始化时调用，或用于动态配置搜索界面。
      
        """
        params = {}
        body = None
        
        path = "/api/v1/search/engines"
        
        return self._http.request("GET", path, params=params, json=body)
    
    def post_search_aggregate(self, **kwargs):
        """智能搜索
        想在你的应用中集成搜索功能？我们提供了一个强大的搜索引擎API，让你可以轻松实现实时网页搜索。

## 功能概述

UAPI Pro Search 是自研的智能搜索引擎，采用机器学习算法对搜索结果进行智能排序，确保最相关的内容排在前面。你可以用它搜索任何关键词，也可以限定在特定网站或特定文件类型中搜索。

- **实时网页搜索**: 毫秒级响应，快速返回搜索结果
- **智能排序**: 采用机器学习回归排序算法，结果更精准
- **站内搜索**: 支持 `site:` 操作符，在指定网站内搜索
- **文件类型过滤**: 支持 `filetype:` 操作符，快速找到 PDF、Word 等特定格式文件

> [!VIP]
> 本API目前处于**限时免费**阶段，我们鼓励开发者集成和测试。未来，它将转为付费API，为用户提供更稳定和强大的服务。
      
        """
        params = {}
        body = None
        
        path = "/api/v1/search/aggregate"
        
        return self._http.request("POST", path, params=params, json=body)
    

