"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FullRegionS3CodeCommitBackup = void 0;
const path = require("path");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const S3_BUCKET_ENV = 'SCRIPTS_BUCKET';
const S3_KEY_ENV = 'SCRIPTS_BUCKET_KEY';
class FullRegionS3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, repositoryNames = [], computeType } = props;
        const asset = new aws_s3_assets_1.Asset(this, 'ScriptsDirectory', {
            path: path.join(__dirname, '..', 'scripts'),
        });
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'FullRegionBackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            environmentVariables: {
                BACKUP_BUCKET: {
                    value: backupBucket.bucketName,
                },
                REPOSITORY_NAMES: {
                    value: repositoryNames.join(' '),
                },
                [S3_BUCKET_ENV]: { value: asset.s3BucketName },
                [S3_KEY_ENV]: { value: asset.s3ObjectKey },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [
                            `echo "Downloading scripts from s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}}"`,
                            `aws s3 cp s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}} ./`,
                            `unzip ./$(basename \${${S3_KEY_ENV}})`,
                        ],
                    },
                    build: {
                        commands: ['chmod +x backup_codecommit.sh', './backup_codecommit.sh'],
                    },
                },
            }),
        });
        asset.grantRead(this.backupProject);
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.FullRegionS3CodeCommitBackup = FullRegionS3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,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