import warnings
from os import PathLike
from typing import Union

import pulp
from pulp.apis import LpSolver_CMD, PulpSolverError, constants

from quantagonia import HybridSolver
from quantagonia.parameters import HybridSolverParameters


class HybridSolver_CMD(LpSolver_CMD):  # noqa: N801 name is not CameCase due to convention of pulp
    """The HybridSolver command to be passed to the :code:`solve` method of PuLP.

    Args:
        api_key (str): A string containing the API key.
        params (HybridSolverParameters): (optional) The parameters for the solver.
        keepFiles (bool): (optional) If True, files are saved in the current directory and not deleted after solving.
        obfuscate (bool): (optional) If True, constraints and variable names are obfuscated.

    """

    name = "HybridSolver_CMD"

    def __init__(
        self,
        api_key: str,
        params: dict = None,
        # To replicate the signature of the parent class, this is camelcase
        keepFiles: bool = False,  # noqa: N803
        obfuscate: bool = True,
    ):

        self.hybrid_solver = HybridSolver(api_key)
        self.params = params
        if self.params is None:
            self.params = HybridSolverParameters()
        self.obfuscate = obfuscate
        LpSolver_CMD.__init__(
            self,
            mip=True,
            path="",
            keepFiles=keepFiles,
        )

    # overrides pulp method hence the camelCase
    def defaultPath(self) -> str:  # noqa: N802
        return self.executableExtension("qqvm")

    def available(self) -> Union[str, PathLike[str], None, bytes]:
        """True if the solver is available"""
        return self.executable(self.path)

    # overrides pulp method hence the camelCase
    def actualSolve(self, lp: pulp.LpProblem) -> int:  # noqa: N802
        """Solve a well formulated lp problem"""
        var_lp = False  # When lp files are written, qqvm-bolt loses the ordering of variables. This results in wrong
        # reading of solutions as the assumed ordering is not present. In order to support varLP=True, one would have to
        # reimplement the readsol method.

        if var_lp:
            tmp_mps, tmp_sol, tmp_options, tmp_log = self.create_tmp_files(
                lp.name, "lp", "sol", "HybridSolver", "HybridSolver_log"
            )
        else:
            tmp_mps, tmp_sol, tmp_options, tmp_log = self.create_tmp_files(
                lp.name, "mps", "sol", "HybridSolver", "HybridSolver_log"
            )

        if not var_lp:
            if lp.sense == constants.LpMaximize:
                # we swap the objectives
                # because it does not handle maximization.

                print("INFO: HybridSolver_CMD solving equivalent minimization form.")

                # don't print: 'Overwriting previously set objective' warning
                with warnings.catch_warnings():
                    warnings.simplefilter("ignore")
                    lp += -lp.objective

        lp.checkDuplicateVars()
        lp.checkLengthVars(52)

        # flag for renaming in writeMPS() should be {0,1}
        rename = 1
        if not self.obfuscate:
            rename = 0

        rename_map = {}

        if var_lp:
            lp.writeLP(filename=tmp_mps)  # , mpsSense=constants.LpMinimize)
        else:
            ret_tpl = lp.writeMPS(filename=tmp_mps, rename=rename)  # , mpsSense=constants.LpMinimize)
            rename_map = ret_tpl[1]

        if lp.isMIP():
            if not self.mip:
                warnings.warn("HybridSolver_CMD cannot solve the relaxation of a problem")

        ########################################################################
        # actual solve operation (local or cloud)
        result, _ = self.hybrid_solver.solve(tmp_mps, self.params)
        ########################################################################

        if not var_lp:
            if lp.sense == constants.LpMaximize:
                print("INFO: Transforming solution value back to original maximization form.")
                # don't print: 'Overwriting previously set objective' warning
                with warnings.catch_warnings():
                    warnings.simplefilter("ignore")
                    lp += -lp.objective

        # parse solution
        content = result["solver_log"].splitlines()

        sol_status_key = "Solution Status"
        try:
            sol_status = next(line for line in content if sol_status_key in line).split()[3]
        except:
            raise PulpSolverError("Pulp: Error while executing", self.path)

        has_sol_key = "Best solution found"
        has_sol = True if len([line for line in content if has_sol_key in line]) >= 1 else False

        # map HybridSolver Status to pulp status
        if sol_status.lower() == "optimal":  # optimal
            status, status_sol = (
                constants.LpStatusOptimal,
                constants.LpSolutionOptimal,
            )
        elif sol_status.lower() == "time limit reached" and has_sol:  # feasible
            # Following the PuLP convention
            status, status_sol = (
                constants.LpStatusOptimal,
                constants.LpSolutionIntegerFeasible,
            )
        elif sol_status.lower() == "time limit reached" and not has_sol:  # feasible
            # Following the PuLP convention
            status, status_sol = (
                constants.LpStatusOptimal,
                constants.LpSolutionNoSolutionFound,
            )
        elif sol_status.lower() == "infeasible":  # infeasible
            status, status_sol = (
                constants.LpStatusInfeasible,
                constants.LpSolutionNoSolutionFound,
            )
        elif sol_status.lower() == "unbounded":  # unbounded
            status, status_sol = (
                constants.LpStatusUnbounded,
                constants.LpSolutionNoSolutionFound,
            )
        else:
            raise RuntimeError(f"Uncatched solution status: {sol_status}")

        # remap obfuscated variable names to original variable names
        for orig_name, obfuscated_name in rename_map.items():
            result["solution"][orig_name] = result["solution"].pop(obfuscated_name)

        self.delete_tmp_files(tmp_mps, tmp_sol, tmp_options, tmp_log)
        lp.assignStatus(status, status_sol)

        if status == constants.LpStatusOptimal:
            lp.assignVarsVals(result["solution"])

        return status
