import math


def compound_discrete(p0:float, r:float, m:int, years:float) -> float:
    """
    Calculate compound interest with discrete compounding.
    
    This function calculates the future value of an investment with 
    discrete compounding: FV = PV * (1 + r/m)^(m*years)
    
    Args:
        p0: Present value (initial investment)
        r: Annual interest rate (as decimal, e.g., 0.05 for 5%)
        m: Number of compounding periods per year (e.g., 12 for monthly)
        years: Time period in years
        
    Returns:
        Future value after compounding
        
    Example:
        >>> compound_discrete(1000.0, 0.05, 12, 1.0)
        1051.161...
    """
    return p0 * (1 + r / m) ** (m * years)

def compound_continuous(p0:float, r:float, t:float) -> float:
    """
    Calculate compound interest with continuous compounding.
    
    This function calculates the future value of an investment with 
    continuous compounding: FV = PV * e^(r*t)
    
    Args:
        p0: Present value (initial investment)
        r: Annual interest rate (as decimal, e.g., 0.05 for 5%)
        t: Time period in years
        
    Returns:
        Future value after continuous compounding
        
    Example:
        >>> compound_continuous(1000.0, 0.05, 1.0)
        1051.271...
    """
    return p0 * math.exp(r * t)
