"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DailyCloudWatchLogsArchiveStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_secure_bucket_1 = require("@gammarer/aws-secure-bucket");
const aws_secure_log_bucket_1 = require("@gammarer/aws-secure-log-bucket");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const scheduler = require("aws-cdk-lib/aws-scheduler");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const log_archiver_function_1 = require("./funcs/log-archiver-function");
class DailyCloudWatchLogsArchiveStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, props);
        // 👇 Get current account & region
        //const account = cdk.Stack.of(this).account;
        //const stackName: string = cdk.Stack.of(this).stackName;
        //const region = cdk.Stack.of(this).region;
        const account = this.account;
        const region = this.region;
        const randomNameKey = crypto.createHash('shake256', { outputLength: 4 })
            .update(`${account}-${region}-${cdk.Names.uniqueId(scope)}-${cdk.Names.uniqueId(this)}`)
            .digest('hex');
        // 👇 Create Backup S3 Bucket
        const logArchiveBucket = new aws_secure_log_bucket_1.SecureLogBucket(this, 'LogArchiveBucket', {
            bucketName: `log-archive-${randomNameKey}`,
            encryption: aws_secure_bucket_1.SecureBucketEncryption.S3_MANAGED,
        });
        logArchiveBucket.addToResourcePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal(`logs.${region}.amazonaws.com`),
            ],
            actions: [
                's3:GetBucketAcl',
            ],
            resources: [
                logArchiveBucket.bucketArn,
            ],
        }));
        logArchiveBucket.addToResourcePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal(`logs.${region}.amazonaws.com`),
            ],
            actions: [
                's3:PutObject',
            ],
            resources: [
                `${logArchiveBucket.bucketArn}/*`,
            ],
            conditions: {
                StringEquals: {
                    's3:x-amz-acl': 'bucket-owner-full-control',
                },
            },
        }));
        // 👇 Create Lambda Execution role.
        const lambdaExecutionRole = new iam.Role(this, 'LambdaExecutionRole', {
            roleName: `daily-cw-logs-archive-lambda-exec-${randomNameKey}-role`,
            description: 'daily CloudWatch Logs archive machine exec role.',
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
            inlinePolicies: {
                ['log-export-policy']: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                'logs:CreateExportTask',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
                ['put-bucket-policy']: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                's3:GetBucketAcl',
                                's3:PutObject',
                            ],
                            resources: [
                                logArchiveBucket.bucketArn,
                            ],
                        }),
                    ],
                }),
            },
        });
        // 👇 Create Lambda Function
        const lambdaFunction = new log_archiver_function_1.LogArchiverFunction(this, 'LogArchiveFunction', {
            functionName: `daily-cw-logs-archive-${randomNameKey}-func`,
            description: 'A function to archive logs s3 bucket from CloudWatch Logs.',
            environment: {
                BUCKET_NAME: logArchiveBucket.bucketName,
            },
            role: lambdaExecutionRole,
        });
        // 👇 Create Lambda Function Log Group
        new logs.LogGroup(this, 'LambdaFunctionLogGroup', {
            // logGroupName: lambdaFunction.logGroup.logGroupName, // <- If you specify this line to Custom:LogRotation resource created.
            logGroupName: `/aws/lambda/${lambdaFunction.functionName}`,
            retention: logs.RetentionDays.ONE_MONTH,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const succeed = new sfn.Succeed(this, 'Succeed');
        // 👇 Get CloudWatch Logs Resources
        const getLogGroupResources = new tasks.CallAwsService(this, 'GetResources', {
            iamResources: ['*'],
            iamAction: 'tag:GetResources',
            service: 'resourcegroupstaggingapi',
            action: 'getResources',
            parameters: {
                ResourceTypeFilters: [
                    'logs:log-group',
                ],
                TagFilters: [
                    {
                        'Key.$': '$.tagKey',
                        'Values.$': '$.tagValues',
                    },
                ],
            },
            resultPath: '$.Result',
            resultSelector: {
                'TargetResources.$': '$..ResourceTagMappingList[*].ResourceARN',
            },
        });
        // Log Group Export Map
        const logGroupExportMap = new sfn.Map(this, 'LogGroupExportMap', {
            itemsPath: sfn.JsonPath.stringAt('$.Result.TargetResources'),
            maxConcurrency: 1,
        });
        // 👇 Get Log Group Name
        const getLogGroupName = new sfn.Pass(this, 'GetLogGroupName', {
            parameters: {
                'TargetLogGroupName.$': "States.ArrayGetItem(States.StringSplit($, ':'), 6)",
            },
        });
        logGroupExportMap.iterator(getLogGroupName);
        // 👇 Invoke Lambda Function
        const invokeLambdaFunction = new tasks.LambdaInvoke(this, 'InvokeLambdaFunction', {
            lambdaFunction: lambdaFunction,
            outputPath: '$.Payload',
            payload: sfn.TaskInput.fromJsonPathAt('$'),
            retryOnServiceExceptions: true,
        });
        getLogGroupName.next(invokeLambdaFunction);
        // 👇 Describe Export Tasks
        const describeExportTasks = new tasks.CallAwsService(this, 'DescribeExportTasks', {
            iamResources: ['*'],
            iamAction: 'logs:DescribeExportTasks',
            service: 'cloudwatchlogs',
            action: 'describeExportTasks',
            parameters: {
                'TaskId.$': '$.TaskId',
            },
            resultPath: '$.Result',
            resultSelector: {
                'DescribeExportTasksStatus.$': '$.ExportTasks[0].Status.Code',
            },
        });
        invokeLambdaFunction.next(describeExportTasks);
        const exportRunningWait = new sfn.Wait(this, 'ExportRunningWait', {
            time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(10)),
        });
        const exportPendingWait = new sfn.Wait(this, 'ExportPendingWait', {
            time: sfn.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(3)),
        });
        // 👇 Export Status Check
        const exportTaskStatusCheck = new sfn.Choice(this, 'ExportTaskStatusCheck')
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'FAILED'), getLogGroupName)
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'RUNNING'), exportRunningWait
            .next(describeExportTasks))
            .when(sfn.Condition.stringEquals('$.Result.DescribeExportTasksStatus', 'PENDING'), exportPendingWait
            .next(describeExportTasks))
            .otherwise(succeed);
        describeExportTasks.next(exportTaskStatusCheck);
        getLogGroupResources.next(logGroupExportMap);
        //
        const machine = new sfn.StateMachine(this, 'StateMachine', {
            stateMachineName: `daily-cw-logs-archive-${randomNameKey}-machine`,
            definition: getLogGroupResources,
        });
        // 👇 auto generated role name & description renaming.
        const role = machine.node.findChild('Role');
        const cfnRole = role.node.defaultChild;
        cfnRole.addPropertyOverride('RoleName', `daily-cw-logs-archive-machine-${randomNameKey}-role`);
        cfnRole.addPropertyOverride('Description', 'daily CloudWatch Logs archive machine role.');
        const policy = role.node.findChild('DefaultPolicy');
        const cfnPolicy = policy.node.defaultChild;
        cfnPolicy.addPropertyOverride('PolicyName', `daily-cw-logs-archive-machine-${randomNameKey}-default-policy`);
        // 👇 EventBridge Scheduler IAM Role (StateMachine Start Execution)
        const schedulerExecutionRole = new iam.Role(this, 'SchedulerExecutionRole', {
            roleName: `daily-cw-logs-archive-${randomNameKey}-schedule-role`,
            description: 'daily CloudWatch Log archive schedule',
            assumedBy: new iam.ServicePrincipal('scheduler.amazonaws.com'),
            inlinePolicies: {
                'state-machine-exec-policy': new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            effect: iam.Effect.ALLOW,
                            actions: [
                                'states:StartExecution',
                            ],
                            resources: [
                                machine.stateMachineArn,
                            ],
                        }),
                    ],
                }),
            },
        });
        // 👇 Schedule
        new scheduler.CfnSchedule(this, 'Schedule', {
            name: `daily-cw-logs-archive-${randomNameKey}-schedule`,
            description: 'daily CloudWatch Logs archive schedule',
            state: 'ENABLED',
            flexibleTimeWindow: {
                mode: 'OFF',
            },
            scheduleExpressionTimezone: 'UTC',
            scheduleExpression: 'cron(1 13 * * ? *)',
            target: {
                arn: machine.stateMachineArn,
                roleArn: schedulerExecutionRole.roleArn,
                input: JSON.stringify({
                    tagKey: props.targetResourceTag.key,
                    tagValues: props.targetResourceTag.values,
                }),
                retryPolicy: {
                    maximumEventAgeInSeconds: 60,
                    maximumRetryAttempts: 0,
                },
            },
        });
    }
}
exports.DailyCloudWatchLogsArchiveStack = DailyCloudWatchLogsArchiveStack;
_a = JSII_RTTI_SYMBOL_1;
DailyCloudWatchLogsArchiveStack[_a] = { fqn: "@gammarer/aws-daily-cloud-watch-logs-archive-stack.DailyCloudWatchLogsArchiveStack", version: "2.5.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMsbUVBQXFFO0FBQ3JFLDJFQUFrRTtBQUNsRSxtQ0FBbUM7QUFDbkMsNkNBQXVDO0FBQ3ZDLDJDQUEyQztBQUMzQyw2Q0FBNkM7QUFDN0MsdURBQXVEO0FBQ3ZELHFEQUFxRDtBQUNyRCw2REFBNkQ7QUFFN0QseUVBQW9FO0FBV3BFLE1BQWEsK0JBQWdDLFNBQVEsR0FBRyxDQUFDLEtBQUs7SUFDNUQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEyQztRQUNuRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixrQ0FBa0M7UUFDbEMsNkNBQTZDO1FBQzdDLHlEQUF5RDtRQUN6RCwyQ0FBMkM7UUFDM0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQztRQUM3QixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO1FBRTNCLE1BQU0sYUFBYSxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsVUFBVSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsRUFBRSxDQUFDO2FBQ3JFLE1BQU0sQ0FBQyxHQUFHLE9BQU8sSUFBSSxNQUFNLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQzthQUN2RixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFakIsNkJBQTZCO1FBQzdCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSx1Q0FBZSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUNyRSxVQUFVLEVBQUUsZUFBZSxhQUFhLEVBQUU7WUFDMUMsVUFBVSxFQUFFLDBDQUFzQixDQUFDLFVBQVU7U0FDOUMsQ0FBQyxDQUFDO1FBQ0gsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzNELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsTUFBTSxnQkFBZ0IsQ0FBQzthQUN6RDtZQUNELE9BQU8sRUFBRTtnQkFDUCxpQkFBaUI7YUFDbEI7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsZ0JBQWdCLENBQUMsU0FBUzthQUMzQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzNELE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsTUFBTSxnQkFBZ0IsQ0FBQzthQUN6RDtZQUNELE9BQU8sRUFBRTtnQkFDUCxjQUFjO2FBQ2Y7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsR0FBRyxnQkFBZ0IsQ0FBQyxTQUFTLElBQUk7YUFDbEM7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLGNBQWMsRUFBRSwyQkFBMkI7aUJBQzVDO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLG1DQUFtQztRQUNuQyxNQUFNLG1CQUFtQixHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7WUFDcEUsUUFBUSxFQUFFLHFDQUFxQyxhQUFhLE9BQU87WUFDbkUsV0FBVyxFQUFFLGtEQUFrRDtZQUMvRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDM0QsZUFBZSxFQUFFO2dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMENBQTBDLENBQUM7YUFDdkY7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztvQkFDNUMsVUFBVSxFQUFFO3dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSzs0QkFDeEIsT0FBTyxFQUFFO2dDQUNQLHVCQUF1Qjs2QkFDeEI7NEJBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO3lCQUNqQixDQUFDO3FCQUNIO2lCQUNGLENBQUM7Z0JBQ0YsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQztvQkFDNUMsVUFBVSxFQUFFO3dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDdEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSzs0QkFDeEIsT0FBTyxFQUFFO2dDQUNQLGlCQUFpQjtnQ0FDakIsY0FBYzs2QkFDZjs0QkFDRCxTQUFTLEVBQUU7Z0NBQ1QsZ0JBQWdCLENBQUMsU0FBUzs2QkFDM0I7eUJBQ0YsQ0FBQztxQkFDSDtpQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCw0QkFBNEI7UUFDNUIsTUFBTSxjQUFjLEdBQUcsSUFBSSwyQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDekUsWUFBWSxFQUFFLHlCQUF5QixhQUFhLE9BQU87WUFDM0QsV0FBVyxFQUFFLDREQUE0RDtZQUN6RSxXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLGdCQUFnQixDQUFDLFVBQVU7YUFDekM7WUFDRCxJQUFJLEVBQUUsbUJBQW1CO1NBQzFCLENBQUMsQ0FBQztRQUVILHNDQUFzQztRQUN0QyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQ2hELDZIQUE2SDtZQUM3SCxZQUFZLEVBQUUsZUFBZSxjQUFjLENBQUMsWUFBWSxFQUFFO1lBQzFELFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7WUFDdkMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTztTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLE9BQU8sR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRWpELG1DQUFtQztRQUNuQyxNQUFNLG9CQUFvQixHQUFHLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzFFLFlBQVksRUFBRSxDQUFDLEdBQUcsQ0FBQztZQUNuQixTQUFTLEVBQUUsa0JBQWtCO1lBQzdCLE9BQU8sRUFBRSwwQkFBMEI7WUFDbkMsTUFBTSxFQUFFLGNBQWM7WUFDdEIsVUFBVSxFQUFFO2dCQUNWLG1CQUFtQixFQUFFO29CQUNuQixnQkFBZ0I7aUJBQ2pCO2dCQUNELFVBQVUsRUFBRTtvQkFDVjt3QkFDRSxPQUFPLEVBQUUsVUFBVTt3QkFDbkIsVUFBVSxFQUFFLGFBQWE7cUJBQzFCO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUUsVUFBVTtZQUN0QixjQUFjLEVBQUU7Z0JBQ2QsbUJBQW1CLEVBQUUsMENBQTBDO2FBQ2hFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsdUJBQXVCO1FBQ3ZCLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUMvRCxTQUFTLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsMEJBQTBCLENBQUM7WUFDNUQsY0FBYyxFQUFFLENBQUM7U0FDbEIsQ0FBQyxDQUFDO1FBRUgsd0JBQXdCO1FBQ3hCLE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDNUQsVUFBVSxFQUFFO2dCQUNWLHNCQUFzQixFQUFFLG9EQUFvRDthQUM3RTtTQUNGLENBQUMsQ0FBQztRQUVILGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUMsQ0FBQztRQUU1Qyw0QkFBNEI7UUFDNUIsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ2hGLGNBQWMsRUFBRSxjQUFjO1lBQzlCLFVBQVUsRUFBRSxXQUFXO1lBQ3ZCLE9BQU8sRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUM7WUFDMUMsd0JBQXdCLEVBQUUsSUFBSTtTQUMvQixDQUFDLENBQUM7UUFFSCxlQUFlLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFFM0MsMkJBQTJCO1FBQzNCLE1BQU0sbUJBQW1CLEdBQUcsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUNoRixZQUFZLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDbkIsU0FBUyxFQUFFLDBCQUEwQjtZQUNyQyxPQUFPLEVBQUUsZ0JBQWdCO1lBQ3pCLE1BQU0sRUFBRSxxQkFBcUI7WUFDN0IsVUFBVSxFQUFFO2dCQUNWLFVBQVUsRUFBRSxVQUFVO2FBQ3ZCO1lBQ0QsVUFBVSxFQUFFLFVBQVU7WUFDdEIsY0FBYyxFQUFFO2dCQUNkLDZCQUE2QixFQUFFLDhCQUE4QjthQUM5RDtTQUNGLENBQUMsQ0FBQztRQUVILG9CQUFvQixDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBRS9DLE1BQU0saUJBQWlCLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUNoRSxJQUFJLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsc0JBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDbEQsQ0FBQyxDQUFDO1FBRUgsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ2hFLElBQUksRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUNqRCxDQUFDLENBQUM7UUFFSCx5QkFBeUI7UUFDekIsTUFBTSxxQkFBcUIsR0FBRyxJQUFJLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLHVCQUF1QixDQUFDO2FBQ3hFLElBQUksQ0FDSCxHQUFHLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxvQ0FBb0MsRUFBRSxRQUFRLENBQUMsRUFDMUUsZUFBZSxDQUNoQjthQUNBLElBQUksQ0FDSCxHQUFHLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxvQ0FBb0MsRUFBRSxTQUFTLENBQUMsRUFDM0UsaUJBQWlCO2FBQ2QsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQzdCO2FBQ0EsSUFBSSxDQUNILEdBQUcsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLG9DQUFvQyxFQUFFLFNBQVMsQ0FBQyxFQUMzRSxpQkFBaUI7YUFDZCxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FDN0I7YUFDQSxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFdEIsbUJBQW1CLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFFaEQsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFFN0MsRUFBRTtRQUNGLE1BQU0sT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3pELGdCQUFnQixFQUFFLHlCQUF5QixhQUFhLFVBQVU7WUFDbEUsVUFBVSxFQUFFLG9CQUFvQjtTQUNqQyxDQUFDLENBQUM7UUFDSCxzREFBc0Q7UUFDdEQsTUFBTSxJQUFJLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFhLENBQUM7UUFDeEQsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUEyQixDQUFDO1FBQ3RELE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLEVBQUUsaUNBQWlDLGFBQWEsT0FBTyxDQUFDLENBQUM7UUFDL0YsT0FBTyxDQUFDLG1CQUFtQixDQUFDLGFBQWEsRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBQzFGLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBZSxDQUFDO1FBQ2xFLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBNkIsQ0FBQztRQUM1RCxTQUFTLENBQUMsbUJBQW1CLENBQUMsWUFBWSxFQUFFLGlDQUFpQyxhQUFhLGlCQUFpQixDQUFDLENBQUM7UUFFN0csbUVBQW1FO1FBQ25FLE1BQU0sc0JBQXNCLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUMxRSxRQUFRLEVBQUUseUJBQXlCLGFBQWEsZ0JBQWdCO1lBQ2hFLFdBQVcsRUFBRSx1Q0FBdUM7WUFDcEQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1lBQzlELGNBQWMsRUFBRTtnQkFDZCwyQkFBMkIsRUFBRSxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7b0JBQ2xELFVBQVUsRUFBRTt3QkFDVixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7NEJBQ3RCLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUs7NEJBQ3hCLE9BQU8sRUFBRTtnQ0FDUCx1QkFBdUI7NkJBQ3hCOzRCQUNELFNBQVMsRUFBRTtnQ0FDVCxPQUFPLENBQUMsZUFBZTs2QkFDeEI7eUJBQ0YsQ0FBQztxQkFDSDtpQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxjQUFjO1FBQ2QsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDMUMsSUFBSSxFQUFFLHlCQUF5QixhQUFhLFdBQVc7WUFDdkQsV0FBVyxFQUFFLHdDQUF3QztZQUNyRCxLQUFLLEVBQUUsU0FBUztZQUNoQixrQkFBa0IsRUFBRTtnQkFDbEIsSUFBSSxFQUFFLEtBQUs7YUFDWjtZQUNELDBCQUEwQixFQUFFLEtBQUs7WUFDakMsa0JBQWtCLEVBQUUsb0JBQW9CO1lBQ3hDLE1BQU0sRUFBRTtnQkFDTixHQUFHLEVBQUUsT0FBTyxDQUFDLGVBQWU7Z0JBQzVCLE9BQU8sRUFBRSxzQkFBc0IsQ0FBQyxPQUFPO2dCQUN2QyxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQztvQkFDcEIsTUFBTSxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHO29CQUNuQyxTQUFTLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixDQUFDLE1BQU07aUJBQzFDLENBQUM7Z0JBQ0YsV0FBVyxFQUFFO29CQUNYLHdCQUF3QixFQUFFLEVBQUU7b0JBQzVCLG9CQUFvQixFQUFFLENBQUM7aUJBQ3hCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDOztBQXJRSCwwRUFzUUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCB7IFNlY3VyZUJ1Y2tldEVuY3J5cHRpb24gfSBmcm9tICdAZ2FtbWFyZXIvYXdzLXNlY3VyZS1idWNrZXQnO1xuaW1wb3J0IHsgU2VjdXJlTG9nQnVja2V0IH0gZnJvbSAnQGdhbW1hcmVyL2F3cy1zZWN1cmUtbG9nLWJ1Y2tldCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsb2dzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1sb2dzJztcbmltcG9ydCAqIGFzIHNjaGVkdWxlciBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc2NoZWR1bGVyJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyB0YXNrcyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucy10YXNrcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IExvZ0FyY2hpdmVyRnVuY3Rpb24gfSBmcm9tICcuL2Z1bmNzL2xvZy1hcmNoaXZlci1mdW5jdGlvbic7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGFpbHlDbG91ZFdhdGNoTG9nc0FyY2hpdmVTdGFja1Byb3BzIGV4dGVuZHMgY2RrLlN0YWNrUHJvcHMge1xuICByZWFkb25seSB0YXJnZXRSZXNvdXJjZVRhZzogVGFyZ2V0UmVzb3VyY2VUYWdQcm9wZXJ0eTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBUYXJnZXRSZXNvdXJjZVRhZ1Byb3BlcnR5IHtcbiAgcmVhZG9ubHkga2V5OiBzdHJpbmc7XG4gIHJlYWRvbmx5IHZhbHVlczogc3RyaW5nW107XG59XG5cbmV4cG9ydCBjbGFzcyBEYWlseUNsb3VkV2F0Y2hMb2dzQXJjaGl2ZVN0YWNrIGV4dGVuZHMgY2RrLlN0YWNrIHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IERhaWx5Q2xvdWRXYXRjaExvZ3NBcmNoaXZlU3RhY2tQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgLy8g8J+RhyBHZXQgY3VycmVudCBhY2NvdW50ICYgcmVnaW9uXG4gICAgLy9jb25zdCBhY2NvdW50ID0gY2RrLlN0YWNrLm9mKHRoaXMpLmFjY291bnQ7XG4gICAgLy9jb25zdCBzdGFja05hbWU6IHN0cmluZyA9IGNkay5TdGFjay5vZih0aGlzKS5zdGFja05hbWU7XG4gICAgLy9jb25zdCByZWdpb24gPSBjZGsuU3RhY2sub2YodGhpcykucmVnaW9uO1xuICAgIGNvbnN0IGFjY291bnQgPSB0aGlzLmFjY291bnQ7XG4gICAgY29uc3QgcmVnaW9uID0gdGhpcy5yZWdpb247XG5cbiAgICBjb25zdCByYW5kb21OYW1lS2V5ID0gY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYWtlMjU2JywgeyBvdXRwdXRMZW5ndGg6IDQgfSlcbiAgICAgIC51cGRhdGUoYCR7YWNjb3VudH0tJHtyZWdpb259LSR7Y2RrLk5hbWVzLnVuaXF1ZUlkKHNjb3BlKX0tJHtjZGsuTmFtZXMudW5pcXVlSWQodGhpcyl9YClcbiAgICAgIC5kaWdlc3QoJ2hleCcpO1xuXG4gICAgLy8g8J+RhyBDcmVhdGUgQmFja3VwIFMzIEJ1Y2tldFxuICAgIGNvbnN0IGxvZ0FyY2hpdmVCdWNrZXQgPSBuZXcgU2VjdXJlTG9nQnVja2V0KHRoaXMsICdMb2dBcmNoaXZlQnVja2V0Jywge1xuICAgICAgYnVja2V0TmFtZTogYGxvZy1hcmNoaXZlLSR7cmFuZG9tTmFtZUtleX1gLFxuICAgICAgZW5jcnlwdGlvbjogU2VjdXJlQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICAgIH0pO1xuICAgIGxvZ0FyY2hpdmVCdWNrZXQuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICBwcmluY2lwYWxzOiBbXG4gICAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbChgbG9ncy4ke3JlZ2lvbn0uYW1hem9uYXdzLmNvbWApLFxuICAgICAgXSxcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3MzOkdldEJ1Y2tldEFjbCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgIGxvZ0FyY2hpdmVCdWNrZXQuYnVja2V0QXJuLFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgbG9nQXJjaGl2ZUJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIHByaW5jaXBhbHM6IFtcbiAgICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKGBsb2dzLiR7cmVnaW9ufS5hbWF6b25hd3MuY29tYCksXG4gICAgICBdLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnczM6UHV0T2JqZWN0JyxcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgYCR7bG9nQXJjaGl2ZUJ1Y2tldC5idWNrZXRBcm59LypgLFxuICAgICAgXSxcbiAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgJ3MzOngtYW16LWFjbCc6ICdidWNrZXQtb3duZXItZnVsbC1jb250cm9sJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgLy8g8J+RhyBDcmVhdGUgTGFtYmRhIEV4ZWN1dGlvbiByb2xlLlxuICAgIGNvbnN0IGxhbWJkYUV4ZWN1dGlvblJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0xhbWJkYUV4ZWN1dGlvblJvbGUnLCB7XG4gICAgICByb2xlTmFtZTogYGRhaWx5LWN3LWxvZ3MtYXJjaGl2ZS1sYW1iZGEtZXhlYy0ke3JhbmRvbU5hbWVLZXl9LXJvbGVgLFxuICAgICAgZGVzY3JpcHRpb246ICdkYWlseSBDbG91ZFdhdGNoIExvZ3MgYXJjaGl2ZSBtYWNoaW5lIGV4ZWMgcm9sZS4nLFxuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlJyksXG4gICAgICBdLFxuICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgWydsb2ctZXhwb3J0LXBvbGljeSddOiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICdsb2dzOkNyZWF0ZUV4cG9ydFRhc2snLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICAgIFsncHV0LWJ1Y2tldC1wb2xpY3knXTogbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IGlhbS5FZmZlY3QuQUxMT1csXG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAnczM6R2V0QnVja2V0QWNsJyxcbiAgICAgICAgICAgICAgICAnczM6UHV0T2JqZWN0JyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICAgICAgbG9nQXJjaGl2ZUJ1Y2tldC5idWNrZXRBcm4sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyDwn5GHIENyZWF0ZSBMYW1iZGEgRnVuY3Rpb25cbiAgICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IG5ldyBMb2dBcmNoaXZlckZ1bmN0aW9uKHRoaXMsICdMb2dBcmNoaXZlRnVuY3Rpb24nLCB7XG4gICAgICBmdW5jdGlvbk5hbWU6IGBkYWlseS1jdy1sb2dzLWFyY2hpdmUtJHtyYW5kb21OYW1lS2V5fS1mdW5jYCxcbiAgICAgIGRlc2NyaXB0aW9uOiAnQSBmdW5jdGlvbiB0byBhcmNoaXZlIGxvZ3MgczMgYnVja2V0IGZyb20gQ2xvdWRXYXRjaCBMb2dzLicsXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBCVUNLRVRfTkFNRTogbG9nQXJjaGl2ZUJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgfSxcbiAgICAgIHJvbGU6IGxhbWJkYUV4ZWN1dGlvblJvbGUsXG4gICAgfSk7XG5cbiAgICAvLyDwn5GHIENyZWF0ZSBMYW1iZGEgRnVuY3Rpb24gTG9nIEdyb3VwXG4gICAgbmV3IGxvZ3MuTG9nR3JvdXAodGhpcywgJ0xhbWJkYUZ1bmN0aW9uTG9nR3JvdXAnLCB7XG4gICAgICAvLyBsb2dHcm91cE5hbWU6IGxhbWJkYUZ1bmN0aW9uLmxvZ0dyb3VwLmxvZ0dyb3VwTmFtZSwgLy8gPC0gSWYgeW91IHNwZWNpZnkgdGhpcyBsaW5lIHRvIEN1c3RvbTpMb2dSb3RhdGlvbiByZXNvdXJjZSBjcmVhdGVkLlxuICAgICAgbG9nR3JvdXBOYW1lOiBgL2F3cy9sYW1iZGEvJHtsYW1iZGFGdW5jdGlvbi5mdW5jdGlvbk5hbWV9YCxcbiAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdWNjZWVkID0gbmV3IHNmbi5TdWNjZWVkKHRoaXMsICdTdWNjZWVkJyk7XG5cbiAgICAvLyDwn5GHIEdldCBDbG91ZFdhdGNoIExvZ3MgUmVzb3VyY2VzXG4gICAgY29uc3QgZ2V0TG9nR3JvdXBSZXNvdXJjZXMgPSBuZXcgdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0dldFJlc291cmNlcycsIHtcbiAgICAgIGlhbVJlc291cmNlczogWycqJ10sXG4gICAgICBpYW1BY3Rpb246ICd0YWc6R2V0UmVzb3VyY2VzJyxcbiAgICAgIHNlcnZpY2U6ICdyZXNvdXJjZWdyb3Vwc3RhZ2dpbmdhcGknLFxuICAgICAgYWN0aW9uOiAnZ2V0UmVzb3VyY2VzJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgUmVzb3VyY2VUeXBlRmlsdGVyczogW1xuICAgICAgICAgICdsb2dzOmxvZy1ncm91cCcsXG4gICAgICAgIF0sXG4gICAgICAgIFRhZ0ZpbHRlcnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnS2V5LiQnOiAnJC50YWdLZXknLFxuICAgICAgICAgICAgJ1ZhbHVlcy4kJzogJyQudGFnVmFsdWVzJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICAgIHJlc3VsdFBhdGg6ICckLlJlc3VsdCcsXG4gICAgICByZXN1bHRTZWxlY3Rvcjoge1xuICAgICAgICAnVGFyZ2V0UmVzb3VyY2VzLiQnOiAnJC4uUmVzb3VyY2VUYWdNYXBwaW5nTGlzdFsqXS5SZXNvdXJjZUFSTicsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gTG9nIEdyb3VwIEV4cG9ydCBNYXBcbiAgICBjb25zdCBsb2dHcm91cEV4cG9ydE1hcCA9IG5ldyBzZm4uTWFwKHRoaXMsICdMb2dHcm91cEV4cG9ydE1hcCcsIHtcbiAgICAgIGl0ZW1zUGF0aDogc2ZuLkpzb25QYXRoLnN0cmluZ0F0KCckLlJlc3VsdC5UYXJnZXRSZXNvdXJjZXMnKSxcbiAgICAgIG1heENvbmN1cnJlbmN5OiAxLFxuICAgIH0pO1xuXG4gICAgLy8g8J+RhyBHZXQgTG9nIEdyb3VwIE5hbWVcbiAgICBjb25zdCBnZXRMb2dHcm91cE5hbWUgPSBuZXcgc2ZuLlBhc3ModGhpcywgJ0dldExvZ0dyb3VwTmFtZScsIHtcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgJ1RhcmdldExvZ0dyb3VwTmFtZS4kJzogXCJTdGF0ZXMuQXJyYXlHZXRJdGVtKFN0YXRlcy5TdHJpbmdTcGxpdCgkLCAnOicpLCA2KVwiLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGxvZ0dyb3VwRXhwb3J0TWFwLml0ZXJhdG9yKGdldExvZ0dyb3VwTmFtZSk7XG5cbiAgICAvLyDwn5GHIEludm9rZSBMYW1iZGEgRnVuY3Rpb25cbiAgICBjb25zdCBpbnZva2VMYW1iZGFGdW5jdGlvbiA9IG5ldyB0YXNrcy5MYW1iZGFJbnZva2UodGhpcywgJ0ludm9rZUxhbWJkYUZ1bmN0aW9uJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb246IGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgb3V0cHV0UGF0aDogJyQuUGF5bG9hZCcsXG4gICAgICBwYXlsb2FkOiBzZm4uVGFza0lucHV0LmZyb21Kc29uUGF0aEF0KCckJyksXG4gICAgICByZXRyeU9uU2VydmljZUV4Y2VwdGlvbnM6IHRydWUsXG4gICAgfSk7XG5cbiAgICBnZXRMb2dHcm91cE5hbWUubmV4dChpbnZva2VMYW1iZGFGdW5jdGlvbik7XG5cbiAgICAvLyDwn5GHIERlc2NyaWJlIEV4cG9ydCBUYXNrc1xuICAgIGNvbnN0IGRlc2NyaWJlRXhwb3J0VGFza3MgPSBuZXcgdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0Rlc2NyaWJlRXhwb3J0VGFza3MnLCB7XG4gICAgICBpYW1SZXNvdXJjZXM6IFsnKiddLFxuICAgICAgaWFtQWN0aW9uOiAnbG9nczpEZXNjcmliZUV4cG9ydFRhc2tzJyxcbiAgICAgIHNlcnZpY2U6ICdjbG91ZHdhdGNobG9ncycsXG4gICAgICBhY3Rpb246ICdkZXNjcmliZUV4cG9ydFRhc2tzJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgJ1Rhc2tJZC4kJzogJyQuVGFza0lkJyxcbiAgICAgIH0sXG4gICAgICByZXN1bHRQYXRoOiAnJC5SZXN1bHQnLFxuICAgICAgcmVzdWx0U2VsZWN0b3I6IHtcbiAgICAgICAgJ0Rlc2NyaWJlRXhwb3J0VGFza3NTdGF0dXMuJCc6ICckLkV4cG9ydFRhc2tzWzBdLlN0YXR1cy5Db2RlJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBpbnZva2VMYW1iZGFGdW5jdGlvbi5uZXh0KGRlc2NyaWJlRXhwb3J0VGFza3MpO1xuXG4gICAgY29uc3QgZXhwb3J0UnVubmluZ1dhaXQgPSBuZXcgc2ZuLldhaXQodGhpcywgJ0V4cG9ydFJ1bm5pbmdXYWl0Jywge1xuICAgICAgdGltZTogc2ZuLldhaXRUaW1lLmR1cmF0aW9uKER1cmF0aW9uLnNlY29uZHMoMTApKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGV4cG9ydFBlbmRpbmdXYWl0ID0gbmV3IHNmbi5XYWl0KHRoaXMsICdFeHBvcnRQZW5kaW5nV2FpdCcsIHtcbiAgICAgIHRpbWU6IHNmbi5XYWl0VGltZS5kdXJhdGlvbihEdXJhdGlvbi5zZWNvbmRzKDMpKSxcbiAgICB9KTtcblxuICAgIC8vIPCfkYcgRXhwb3J0IFN0YXR1cyBDaGVja1xuICAgIGNvbnN0IGV4cG9ydFRhc2tTdGF0dXNDaGVjayA9IG5ldyBzZm4uQ2hvaWNlKHRoaXMsICdFeHBvcnRUYXNrU3RhdHVzQ2hlY2snKVxuICAgICAgLndoZW4oXG4gICAgICAgIHNmbi5Db25kaXRpb24uc3RyaW5nRXF1YWxzKCckLlJlc3VsdC5EZXNjcmliZUV4cG9ydFRhc2tzU3RhdHVzJywgJ0ZBSUxFRCcpLFxuICAgICAgICBnZXRMb2dHcm91cE5hbWUsXG4gICAgICApXG4gICAgICAud2hlbihcbiAgICAgICAgc2ZuLkNvbmRpdGlvbi5zdHJpbmdFcXVhbHMoJyQuUmVzdWx0LkRlc2NyaWJlRXhwb3J0VGFza3NTdGF0dXMnLCAnUlVOTklORycpLFxuICAgICAgICBleHBvcnRSdW5uaW5nV2FpdFxuICAgICAgICAgIC5uZXh0KGRlc2NyaWJlRXhwb3J0VGFza3MpLFxuICAgICAgKVxuICAgICAgLndoZW4oXG4gICAgICAgIHNmbi5Db25kaXRpb24uc3RyaW5nRXF1YWxzKCckLlJlc3VsdC5EZXNjcmliZUV4cG9ydFRhc2tzU3RhdHVzJywgJ1BFTkRJTkcnKSxcbiAgICAgICAgZXhwb3J0UGVuZGluZ1dhaXRcbiAgICAgICAgICAubmV4dChkZXNjcmliZUV4cG9ydFRhc2tzKSxcbiAgICAgIClcbiAgICAgIC5vdGhlcndpc2Uoc3VjY2VlZCk7XG5cbiAgICBkZXNjcmliZUV4cG9ydFRhc2tzLm5leHQoZXhwb3J0VGFza1N0YXR1c0NoZWNrKTtcblxuICAgIGdldExvZ0dyb3VwUmVzb3VyY2VzLm5leHQobG9nR3JvdXBFeHBvcnRNYXApO1xuXG4gICAgLy9cbiAgICBjb25zdCBtYWNoaW5lID0gbmV3IHNmbi5TdGF0ZU1hY2hpbmUodGhpcywgJ1N0YXRlTWFjaGluZScsIHtcbiAgICAgIHN0YXRlTWFjaGluZU5hbWU6IGBkYWlseS1jdy1sb2dzLWFyY2hpdmUtJHtyYW5kb21OYW1lS2V5fS1tYWNoaW5lYCxcbiAgICAgIGRlZmluaXRpb246IGdldExvZ0dyb3VwUmVzb3VyY2VzLFxuICAgIH0pO1xuICAgIC8vIPCfkYcgYXV0byBnZW5lcmF0ZWQgcm9sZSBuYW1lICYgZGVzY3JpcHRpb24gcmVuYW1pbmcuXG4gICAgY29uc3Qgcm9sZSA9IG1hY2hpbmUubm9kZS5maW5kQ2hpbGQoJ1JvbGUnKSBhcyBpYW0uUm9sZTtcbiAgICBjb25zdCBjZm5Sb2xlID0gcm9sZS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBpYW0uQ2ZuUm9sZTtcbiAgICBjZm5Sb2xlLmFkZFByb3BlcnR5T3ZlcnJpZGUoJ1JvbGVOYW1lJywgYGRhaWx5LWN3LWxvZ3MtYXJjaGl2ZS1tYWNoaW5lLSR7cmFuZG9tTmFtZUtleX0tcm9sZWApO1xuICAgIGNmblJvbGUuYWRkUHJvcGVydHlPdmVycmlkZSgnRGVzY3JpcHRpb24nLCAnZGFpbHkgQ2xvdWRXYXRjaCBMb2dzIGFyY2hpdmUgbWFjaGluZSByb2xlLicpO1xuICAgIGNvbnN0IHBvbGljeSA9IHJvbGUubm9kZS5maW5kQ2hpbGQoJ0RlZmF1bHRQb2xpY3knKSBhcyBpYW0uUG9saWN5O1xuICAgIGNvbnN0IGNmblBvbGljeSA9IHBvbGljeS5ub2RlLmRlZmF1bHRDaGlsZCBhcyBpYW0uQ2ZuUG9saWN5O1xuICAgIGNmblBvbGljeS5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdQb2xpY3lOYW1lJywgYGRhaWx5LWN3LWxvZ3MtYXJjaGl2ZS1tYWNoaW5lLSR7cmFuZG9tTmFtZUtleX0tZGVmYXVsdC1wb2xpY3lgKTtcblxuICAgIC8vIPCfkYcgRXZlbnRCcmlkZ2UgU2NoZWR1bGVyIElBTSBSb2xlIChTdGF0ZU1hY2hpbmUgU3RhcnQgRXhlY3V0aW9uKVxuICAgIGNvbnN0IHNjaGVkdWxlckV4ZWN1dGlvblJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1NjaGVkdWxlckV4ZWN1dGlvblJvbGUnLCB7XG4gICAgICByb2xlTmFtZTogYGRhaWx5LWN3LWxvZ3MtYXJjaGl2ZS0ke3JhbmRvbU5hbWVLZXl9LXNjaGVkdWxlLXJvbGVgLFxuICAgICAgZGVzY3JpcHRpb246ICdkYWlseSBDbG91ZFdhdGNoIExvZyBhcmNoaXZlIHNjaGVkdWxlJyxcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzY2hlZHVsZXIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgaW5saW5lUG9saWNpZXM6IHtcbiAgICAgICAgJ3N0YXRlLW1hY2hpbmUtZXhlYy1wb2xpY3knOiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICdzdGF0ZXM6U3RhcnRFeGVjdXRpb24nLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICBtYWNoaW5lLnN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIPCfkYcgU2NoZWR1bGVcbiAgICBuZXcgc2NoZWR1bGVyLkNmblNjaGVkdWxlKHRoaXMsICdTY2hlZHVsZScsIHtcbiAgICAgIG5hbWU6IGBkYWlseS1jdy1sb2dzLWFyY2hpdmUtJHtyYW5kb21OYW1lS2V5fS1zY2hlZHVsZWAsXG4gICAgICBkZXNjcmlwdGlvbjogJ2RhaWx5IENsb3VkV2F0Y2ggTG9ncyBhcmNoaXZlIHNjaGVkdWxlJyxcbiAgICAgIHN0YXRlOiAnRU5BQkxFRCcsXG4gICAgICBmbGV4aWJsZVRpbWVXaW5kb3c6IHtcbiAgICAgICAgbW9kZTogJ09GRicsXG4gICAgICB9LFxuICAgICAgc2NoZWR1bGVFeHByZXNzaW9uVGltZXpvbmU6ICdVVEMnLFxuICAgICAgc2NoZWR1bGVFeHByZXNzaW9uOiAnY3JvbigxIDEzICogKiA/ICopJyxcbiAgICAgIHRhcmdldDoge1xuICAgICAgICBhcm46IG1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICByb2xlQXJuOiBzY2hlZHVsZXJFeGVjdXRpb25Sb2xlLnJvbGVBcm4sXG4gICAgICAgIGlucHV0OiBKU09OLnN0cmluZ2lmeSh7XG4gICAgICAgICAgdGFnS2V5OiBwcm9wcy50YXJnZXRSZXNvdXJjZVRhZy5rZXksXG4gICAgICAgICAgdGFnVmFsdWVzOiBwcm9wcy50YXJnZXRSZXNvdXJjZVRhZy52YWx1ZXMsXG4gICAgICAgIH0pLFxuICAgICAgICByZXRyeVBvbGljeToge1xuICAgICAgICAgIG1heGltdW1FdmVudEFnZUluU2Vjb25kczogNjAsXG4gICAgICAgICAgbWF4aW11bVJldHJ5QXR0ZW1wdHM6IDAsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG59Il19