"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('tsc');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    static clearTscCompilationCache() {
        this.tscCompiled = false;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        const pathJoin = osPathJoin(options.osPlatform);
        let tscCommand = '';
        let relativeEntryPath = this.relativeEntryPath;
        if (this.props.preCompilation) {
            let tsconfig = this.relativeTsconfigPath;
            if (!tsconfig) {
                const findConfig = util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
                if (!findConfig) {
                    throw new Error('Cannot find a tsconfig.json, please specify the prop: tsconfig');
                }
                tsconfig = path.relative(this.projectRoot, findConfig);
            }
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
            if (!Bundling.tscCompiled) {
                // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next ts file.
                tscCommand = `${options.tscRunner} --project ${pathJoin(options.inputDir, tsconfig)} --rootDir ./ --outDir ./`;
                Bundling.tscCompiled = true;
            }
        }
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_c = this.props.sourceMapMode) !== null && _c !== void 0 ? _c : this.props.sourceMap;
        const sourceMapMode = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_e = this.props.sourcesContent) !== null && _e !== void 0 ? _e : true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${pathJoin(options.inputDir, relativeEntryPath)}"`,
            `--target=${(_f = this.props.target) !== null && _f !== void 0 ? _f : toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_g = this.relativeDepsLockFilePath) !== null && _g !== void 0 ? _g : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.beforeBundling(options.inputDir, options.outputDir)) !== null && _j !== void 0 ? _j : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_l = (this.props.nodeModules && ((_k = this.props.commandHooks) === null || _k === void 0 ? void 0 : _k.beforeInstall(options.inputDir, options.outputDir)))) !== null && _l !== void 0 ? _l : [],
            depsCommand,
            ...(_o = (_m = this.props.commandHooks) === null || _m === void 0 ? void 0 : _m.afterBundling(options.inputDir, options.outputDir)) !== null && _o !== void 0 ? _o : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
Bundling.tscCompiled = false;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > "${filePath}"`;
        }
        return `echo '${stringifiedData}' > "${filePath}"`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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