from typing import Dict, List, Union

from belvo.resources.base import Resource


class Accounts(Resource):
    endpoint = "/api/accounts/"

    def create(
        self,
        link: str,
        *,
        token: str = None,
        save_data: bool = True,
        raise_exception: bool = False,
        **kwargs: Dict,
    ) -> Union[List[Dict], Dict]:
        """Retrieve accounts for a given link

        To fetch accounts you will make use of the .create() method and will retrieve all account data available from the banking or gig institution.

        If the account already exists in our records, only its balance and collected_at will be updated.

        > For more information detailed information regarding the Accounts resource, check our [Accounts DevPortal documentation](https://developers.belvo.com/reference/accounts-1).

        Example:

            ```python
            # Fetch accounts for a Link
            accounts = client.Accounts.create("b91835f5-6f83-4d9b-a0ad-a5a249f18b7c")
            ```

        Args:
            link (str): The `link.id` that you want to get information for (UUID).
            token (str, optional): The MFA token generated by the bank in order to access the institution. Defaults to None.
            save_data (bool, optional): Indicates whether or not to persist the data in Belvo. Defaults to `True`.
            raise_exception (bool, optional): Indicates whether to raise an exception or return the API error. Defaults to `False`.

        Returns:
            Dict: The details of the object. For more information on the response from the API, see our [Accounts API documentation](https://developers.belvo.com/reference/retrieveaccounts).
        """

        data = {"link": link, "save_data": save_data}

        if token:
            data.update(token=token)

        return self.session.post(
            self.endpoint, data=data, raise_exception=raise_exception, **kwargs
        )
