"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file. In the next major release of the CDK, these feature flags
// will be removed and will become the default behavior.
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.FUTURE_FLAGS = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * This map includes context keys and values for feature flags that enable
 * capabilities "from the future", which we could not introduce as the default
 * behavior due to backwards compatibility for existing projects.
 *
 * New projects generated through `cdk init` will include these flags in their
 * generated `cdk.json` file.
 *
 * When we release the next major version of the CDK, we will flip the logic of
 * these features and clean up the `cdk.json` generated by `cdk init`.
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true',
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: 'true',
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: 'true',
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The set of defaults that should be applied if the feature flag is not
 * explicitly configured.
 */
const FUTURE_FLAGS_DEFAULTS = {
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: false,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: false,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: false,
    [exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: false,
    [exports.DOCKER_IGNORE_SUPPORT]: false,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: false,
    [exports.KMS_DEFAULT_KEY_POLICIES]: false,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: false,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: false,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: false,
};
function futureFlagDefault(flag) {
    return FUTURE_FLAGS_DEFAULTS[flag];
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,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