"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk_1.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicyArns: [
                cdk_1.Stack.of(this).formatArn({
                    service: 'iam',
                    region: '',
                    account: 'aws',
                    resource: 'policy',
                    resourceName: 'service-role/AWSLambdaBasicExecutionRole',
                })
            ]
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement()
            .addAction('s3:PutBucketNotification')
            .addAllResources());
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.Standard, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs8.10',
                Timeout: 300,
            }
        });
        this.functionArn = resource.getAtt('Arn').toString();
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    const s3 = new (require('aws-sdk').S3)();
    const https = require("https");
    const url = require("url");
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse("FAILED", err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse("SUCCESS");
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || "See the details in CloudWatch Log Stream: " + context.logStreamName,
            PhysicalResourceId: context.logStreamName,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: "PUT",
            headers: {
                "content-type": "",
                "content-length": responseBody.length
            }
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on("error", (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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