"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.Lambda:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.Queue:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.Topic:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: new cdk.Token(() => this.renderNotificationConfiguration())
                }
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
        }
        if (rule.prefix) {
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
        }
    }
    return {
        Key: {
            FilterRules: renderedRules
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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