"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/cdk");
const cdk_1 = require("@aws-cdk/cdk");
const os_1 = require("os");
const s3 = require("../lib");
const lib_1 = require("../lib");
module.exports = {
    'default bucket'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'CFN properties are type-validated during resolution'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            bucketName: new cdk.Token(() => 5).toString() // Oh no
        });
        test.throws(() => {
            assert_1.SynthUtils.synthesize(stack);
        }, /bucketName: 5 should be a string/);
        test.done();
    },
    'bucket without encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with managed encryption'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'valid bucket names'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'abc.xyz-34ab'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '124.pp--33'
        }));
        test.done();
    },
    'bucket validation skips tokenized values'(test) {
        const stack = new cdk.Stack();
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: new cdk.Token(() => '_BUCKET').toString()
        }));
        test.done();
    },
    'fails with message on invalid bucket names'(test) {
        const stack = new cdk.Stack();
        const bucket = `-buckEt.-${new Array(65).join('$')}`;
        const expectedErrors = [
            `Invalid S3 bucket name (value: ${bucket})`,
            'Bucket name must be at least 3 and no more than 63 characters',
            'Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) (offset: 5)',
            'Bucket name must start and end with a lowercase character or number (offset: 0)',
            `Bucket name must start and end with a lowercase character or number (offset: ${bucket.length - 1})`,
            'Bucket name must not have dash next to period, or period next to dash, or consecutive periods (offset: 7)',
        ].join(os_1.EOL);
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            bucketName: bucket
            // tslint:disable-next-line:only-arrow-functions
        }), function (err) {
            return expectedErrors === err.message;
        });
        test.done();
    },
    'fails if bucket name has less than 3 or more than 63 characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'a'
        }), /at least 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: new Array(65).join('x')
        }), /no more than 63/);
        test.done();
    },
    'fails if bucket name has invalid characters'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'b@cket'
        }), /offset: 1/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucKet'
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'bučket'
        }), /offset: 2/);
        test.done();
    },
    'fails if bucket name does not start or end with lowercase character or number'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '-ucket'
        }), /offset: 0/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'bucke.'
        }), /offset: 5/);
        test.done();
    },
    'fails only if bucket name has the consecutive symbols (..), (.-), (-.)'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: 'buc..ket'
        }), /offset: 3/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: 'buck.-et'
        }), /offset: 4/);
        test.throws(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: 'b-.ucket'
        }), /offset: 1/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: 'bu--cket'
        }));
        test.done();
    },
    'fails only if bucket name resembles IP address'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new s3.Bucket(stack, 'MyBucket1', {
            bucketName: '1.2.3.4'
        }), /must not resemble an IP address/);
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket2', {
            bucketName: '1.2.3'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket3', {
            bucketName: '1.2.3.a'
        }));
        test.doesNotThrow(() => new s3.Bucket(stack, 'MyBucket4', {
            bucketName: '1000.2.3.4'
        }));
        test.done();
    },
    'fails if encryption key is used with managed encryption'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.KmsManaged,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'fails if encryption key is used with encryption set to unencrypted'(test) {
        const stack = new cdk.Stack();
        const myKey = new kms.Key(stack, 'MyKey');
        test.throws(() => new s3.Bucket(stack, 'MyBucket', {
            encryption: s3.BucketEncryption.Unencrypted,
            encryptionKey: myKey
        }), /encryptionKey is specified, so 'encryption' must be set to KMS/);
        test.done();
    },
    'encryptionKey can specify kms key'(test) {
        const stack = new cdk.Stack();
        const encryptionKey = new kms.Key(stack, 'MyKey', { description: 'hello, world' });
        new s3.Bucket(stack, 'MyBucket', { encryptionKey, encryption: s3.BucketEncryption.Kms });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyKey6AB29FA6": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "hello, world",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain"
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": [
                                                "MyKey6AB29FA6",
                                                "Arn"
                                            ]
                                        },
                                        "SSEAlgorithm": "aws:kms"
                                    }
                                }
                            ]
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with versioning turned on'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            versioned: true
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "VersioningConfiguration": {
                            "Status": "Enabled"
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAll'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAll,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "BlockPublicPolicy": true,
                            "IgnorePublicAcls": true,
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with block public access set to BlockAcls'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BlockAcls,
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "BlockPublicAcls": true,
                            "IgnorePublicAcls": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'bucket with custom block public access setting'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', {
            blockPublicAccess: new s3.BlockPublicAccess({ restrictPublicBuckets: true })
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "PublicAccessBlockConfiguration": {
                            "RestrictPublicBuckets": true,
                        }
                    },
                    "DeletionPolicy": "Retain",
                }
            }
        });
        test.done();
    },
    'permissions': {
        'addPermission creates a bucket policy'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyBucketPolicyE7FBAC7B": {
                        "Type": "AWS::S3::BucketPolicy",
                        "Properties": {
                            "Bucket": {
                                "Ref": "MyBucketF68F3FF0"
                            },
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "bar",
                                        "Effect": "Allow",
                                        "Resource": "foo"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                    }
                }
            });
            test.done();
        },
        'forBucket returns a permission statement associated with the bucket\'s ARN'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const x = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            test.deepEqual(stack.resolve(x), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }
            });
            test.done();
        },
        'arnForObjects returns a permission statement associated with objects in the bucket'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const p = new iam.PolicyStatement().addResource(bucket.arnForObjects('hello/world')).addAction('s3:GetObject');
            test.deepEqual(stack.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [{ 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] }, '/hello/world']
                    ]
                }
            });
            test.done();
        },
        'arnForObjects accepts multiple arguments and FnConcats them'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Unencrypted });
            const user = new iam.User(stack, 'MyUser');
            const team = new iam.Group(stack, 'MyTeam');
            const resource = bucket.arnForObjects(`home/${team.groupName}/${user.userName}/*`);
            const p = new iam.PolicyStatement().addResource(resource).addAction('s3:GetObject');
            test.deepEqual(stack.resolve(p), {
                Action: 's3:GetObject',
                Effect: 'Allow',
                Resource: {
                    'Fn::Join': [
                        '',
                        [
                            { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                            '/home/',
                            { Ref: 'MyTeam01DD6685' },
                            '/',
                            { Ref: 'MyUserDC45028B' },
                            '/*'
                        ]
                    ]
                }
            });
            test.done();
        }
    },
    'removal policy can be used to specify behavior upon delete'(test) {
        const stack = new cdk.Stack();
        new s3.Bucket(stack, 'MyBucket', { removalPolicy: cdk.RemovalPolicy.Orphan, encryption: s3.BucketEncryption.Unencrypted });
        assert_1.expect(stack).toMatch({
            Resources: {
                MyBucketF68F3FF0: {
                    Type: 'AWS::S3::Bucket',
                    DeletionPolicy: 'Retain'
                }
            }
        });
        test.done();
    },
    'import/export': {
        'static import(ref) allows importing an external/existing bucket'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn });
            // this is a no-op since the bucket is external
            bucket.addToResourcePolicy(new iam.PolicyStatement().addResource('foo').addAction('bar'));
            const p = new iam.PolicyStatement().addResource(bucket.bucketArn).addAction('s3:ListBucket');
            // it is possible to obtain a permission statement for a ref
            test.deepEqual(stack.resolve(p), {
                Action: 's3:ListBucket',
                Effect: 'Allow',
                Resource: 'arn:aws:s3:::my-bucket'
            });
            test.deepEqual(bucket.bucketArn, bucketArn);
            test.deepEqual(stack.resolve(bucket.bucketName), 'my-bucket');
            test.deepEqual(assert_1.SynthUtils.synthesize(stack).template, {}, 'the ref is not a real resource');
            test.done();
        },
        'import can also be used to import arbitrary ARNs'(test) {
            const stack = new cdk.Stack();
            const bucket = s3.Bucket.fromBucketAttributes(stack, 'ImportedBucket', { bucketArn: 'arn:aws:s3:::my-bucket' });
            bucket.addToResourcePolicy(new iam.PolicyStatement().addAllResources().addAction('*'));
            // at this point we technically didn't create any resources in the consuming stack.
            assert_1.expect(stack).toMatch({});
            // but now we can reference the bucket
            // you can even use the bucket name, which will be extracted from the arn provided.
            const user = new iam.User(stack, 'MyUser');
            user.addToPolicy(new iam.PolicyStatement()
                .addResource(bucket.arnForObjects(`my/folder/${bucket.bucketName}`))
                .addAction('s3:*'));
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": "s3:*",
                                        "Effect": "Allow",
                                        "Resource": "arn:aws:s3:::my-bucket/my/folder/my-bucket"
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        },
                    }
                }
            });
            test.done();
        },
    },
    'grantRead'(test) {
        const stack = new cdk.Stack();
        const reader = new iam.User(stack, 'Reader');
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.grantRead(reader);
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ReaderF7BF189D": {
                    "Type": "AWS::IAM::User"
                },
                "ReaderDefaultPolicy151F3818": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::GetAtt": [
                                                "MyBucketF68F3FF0",
                                                "Arn"
                                            ]
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::GetAtt": [
                                                            "MyBucketF68F3FF0",
                                                            "Arn"
                                                        ]
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "ReaderDefaultPolicy151F3818",
                        "Users": [
                            {
                                "Ref": "ReaderF7BF189D"
                            }
                        ]
                    }
                },
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                },
            }
        });
        test.done();
    },
    'grantReadWrite': {
        'can be used to grant reciprocal permissions to an identity'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket');
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'grant permissions to non-identity principal'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            // WHEN
            bucket.grantRead(new iam.OrganizationPrincipal('o-1234'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                PolicyDocument: {
                    "Version": "2012-10-17",
                    "Statement": [
                        {
                            "Action": ["s3:GetObject*", "s3:GetBucket*", "s3:List*"],
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": [
                                { "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] },
                                { "Fn::Join": ["", [{ "Fn::GetAtt": ["MyBucketF68F3FF0", "Arn"] }, "/*"]] }
                            ]
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                "KeyPolicy": {
                    "Statement": [
                        {
                            "Action": ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*",
                                "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion"],
                            "Effect": "Allow",
                            "Principal": {
                                "AWS": {
                                    "Fn::Join": ["", [
                                            "arn:", { "Ref": "AWS::Partition" }, ":iam::", { "Ref": "AWS::AccountId" }, ":root"
                                        ]]
                                }
                            },
                            "Resource": "*"
                        },
                        {
                            "Action": ["kms:Decrypt", "kms:DescribeKey"],
                            "Effect": "Allow",
                            "Resource": "*",
                            "Principal": "*",
                            "Condition": { "StringEquals": { "aws:PrincipalOrgID": "o-1234" } },
                        }
                    ],
                    "Version": "2012-10-17"
                },
            }));
            test.done();
        },
        'if an encryption key is included, encrypt/decrypt permissions are also added both ways'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
            const user = new iam.User(stack, 'MyUser');
            bucket.grantReadWrite(user);
            assert_1.expect(stack).toMatch({
                "Resources": {
                    "MyBucketKeyC17130CF": {
                        "Type": "AWS::KMS::Key",
                        "Properties": {
                            "Description": "Created by MyBucket",
                            "KeyPolicy": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kms:Create*",
                                            "kms:Describe*",
                                            "kms:Enable*",
                                            "kms:List*",
                                            "kms:Put*",
                                            "kms:Update*",
                                            "kms:Revoke*",
                                            "kms:Disable*",
                                            "kms:Get*",
                                            "kms:Delete*",
                                            "kms:ScheduleKeyDeletion",
                                            "kms:CancelKeyDeletion"
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:",
                                                        {
                                                            "Ref": "AWS::Partition"
                                                        },
                                                        ":iam::",
                                                        {
                                                            "Ref": "AWS::AccountId"
                                                        },
                                                        ":root"
                                                    ]
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Principal": {
                                            "AWS": {
                                                "Fn::GetAtt": [
                                                    "MyUserDC45028B",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        "Resource": "*"
                                    }
                                ],
                                "Version": "2012-10-17"
                            }
                        },
                        "DeletionPolicy": "Retain"
                    },
                    "MyBucketF68F3FF0": {
                        "Type": "AWS::S3::Bucket",
                        "DeletionPolicy": "Retain",
                        "Properties": {
                            "BucketEncryption": {
                                "ServerSideEncryptionConfiguration": [
                                    {
                                        "ServerSideEncryptionByDefault": {
                                            "KMSMasterKeyID": {
                                                "Fn::GetAtt": [
                                                    "MyBucketKeyC17130CF",
                                                    "Arn"
                                                ]
                                            },
                                            "SSEAlgorithm": "aws:kms"
                                        }
                                    }
                                ]
                            }
                        }
                    },
                    "MyUserDC45028B": {
                        "Type": "AWS::IAM::User"
                    },
                    "MyUserDefaultPolicy7B897426": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "s3:GetObject*",
                                            "s3:GetBucket*",
                                            "s3:List*",
                                            "s3:DeleteObject*",
                                            "s3:PutObject*",
                                            "s3:Abort*"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": [
                                            {
                                                "Fn::GetAtt": [
                                                    "MyBucketF68F3FF0",
                                                    "Arn"
                                                ]
                                            },
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        {
                                                            "Fn::GetAtt": [
                                                                "MyBucketF68F3FF0",
                                                                "Arn"
                                                            ]
                                                        },
                                                        "/*"
                                                    ]
                                                ]
                                            }
                                        ]
                                    },
                                    {
                                        "Action": [
                                            "kms:Decrypt",
                                            "kms:DescribeKey",
                                            "kms:Encrypt",
                                            "kms:ReEncrypt*",
                                            "kms:GenerateDataKey*",
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::GetAtt": [
                                                "MyBucketKeyC17130CF",
                                                "Arn"
                                            ]
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "MyUserDefaultPolicy7B897426",
                            "Users": [
                                {
                                    "Ref": "MyUserDC45028B"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
    },
    'more grants'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket', { encryption: s3.BucketEncryption.Kms });
        const putter = new iam.User(stack, 'Putter');
        const writer = new iam.User(stack, 'Writer');
        const deleter = new iam.User(stack, 'Deleter');
        bucket.grantPut(putter);
        bucket.grantWrite(writer);
        bucket.grantDelete(deleter);
        const resources = assert_1.SynthUtils.synthesize(stack).template.Resources;
        const actions = (id) => resources[id].Properties.PolicyDocument.Statement[0].Action;
        test.deepEqual(actions('WriterDefaultPolicyDC585BCE'), ['s3:DeleteObject*', 's3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('PutterDefaultPolicyAB138DD3'), ['s3:PutObject*', 's3:Abort*']);
        test.deepEqual(actions('DeleterDefaultPolicyCD33B8A0'), 's3:DeleteObject*');
        test.done();
    },
    'cross-stack permissions'(test) {
        const app = new cdk.App();
        const stackA = new cdk.Stack(app, 'stackA');
        const bucketFromStackA = new s3.Bucket(stackA, 'MyBucket');
        const stackB = new cdk.Stack(app, 'stackB');
        const user = new iam.User(stackB, 'UserWhoNeedsAccess');
        bucketFromStackA.grantRead(user);
        assert_1.expect(stackA).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            },
            "Outputs": {
                "ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58": {
                    "Value": {
                        "Fn::GetAtt": [
                            "MyBucketF68F3FF0",
                            "Arn"
                        ]
                    },
                    "Export": {
                        "Name": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                    }
                }
            }
        });
        assert_1.expect(stackB).toMatch({
            "Resources": {
                "UserWhoNeedsAccessF8959C3D": {
                    "Type": "AWS::IAM::User"
                },
                "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                    "Type": "AWS::IAM::Policy",
                    "Properties": {
                        "PolicyDocument": {
                            "Statement": [
                                {
                                    "Action": [
                                        "s3:GetObject*",
                                        "s3:GetBucket*",
                                        "s3:List*"
                                    ],
                                    "Effect": "Allow",
                                    "Resource": [
                                        {
                                            "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                        },
                                        {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    {
                                                        "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyBucketF68F3FF0Arn0F7E8E58"
                                                    },
                                                    "/*"
                                                ]
                                            ]
                                        }
                                    ]
                                }
                            ],
                            "Version": "2012-10-17"
                        },
                        "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                        "Users": [
                            {
                                "Ref": "UserWhoNeedsAccessF8959C3D"
                            }
                        ]
                    }
                }
            }
        });
        test.done();
    },
    'urlForObject returns a token with the S3 URL of the token'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        new cdk.CfnOutput(stack, 'BucketURL', { value: bucket.urlForObject() });
        new cdk.CfnOutput(stack, 'MyFileURL', { value: bucket.urlForObject('my/file.txt') });
        new cdk.CfnOutput(stack, 'YourFileURL', { value: bucket.urlForObject('/your/file.txt') }); // "/" is optional
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyBucketF68F3FF0": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain"
                }
            },
            "Outputs": {
                "BucketURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                }
                            ]
                        ]
                    },
                },
                "MyFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/my/file.txt"
                            ]
                        ]
                    }
                },
                "YourFileURL": {
                    "Value": {
                        "Fn::Join": [
                            "",
                            [
                                "https://s3.",
                                {
                                    "Ref": "AWS::Region"
                                },
                                ".",
                                {
                                    "Ref": "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    "Ref": "MyBucketF68F3FF0"
                                },
                                "/your/file.txt"
                            ]
                        ]
                    },
                }
            }
        });
        test.done();
    },
    'grantPublicAccess': {
        'by default, grants s3:GetObject to all objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"keyPrefix" can be used to only grant access to certain objects'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('only/access/these/*');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/only/access/these/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        '"allowedActions" can be used to specify actions explicitly'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            bucket.grantPublicAccess('*', 's3:GetObject', 's3:PutObject');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": ["s3:GetObject", "s3:PutObject"],
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'returns the PolicyStatement which can be then customized'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'b');
            // WHEN
            const result = bucket.grantPublicAccess();
            result.resourceStatement.addCondition('IpAddress', { "aws:SourceIp": "54.240.143.0/24" });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
                "PolicyDocument": {
                    "Statement": [
                        {
                            "Action": "s3:GetObject",
                            "Effect": "Allow",
                            "Principal": "*",
                            "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["bC3BBCC65", "Arn"] }, "/*"]] },
                            "Condition": {
                                "IpAddress": { "aws:SourceIp": "54.240.143.0/24" }
                            }
                        }
                    ],
                    "Version": "2012-10-17"
                }
            }));
            test.done();
        },
        'throws when blockPublicPolicy is set to true'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'MyBucket', {
                blockPublicAccess: new s3.BlockPublicAccess({ blockPublicPolicy: true })
            });
            // THEN
            test.throws(() => bucket.grantPublicAccess(), /blockPublicPolicy/);
            test.done();
        }
    },
    'website configuration': {
        'only index doc'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html'
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html"
                }
            }));
            test.done();
        },
        'fails if only error doc is specified'(test) {
            const stack = new cdk.Stack();
            test.throws(() => {
                new s3.Bucket(stack, 'Website', {
                    websiteErrorDocument: 'error.html'
                });
            }, /"websiteIndexDocument" is required if "websiteErrorDocument" is set/);
            test.done();
        },
        'error and index docs'(test) {
            const stack = new cdk.Stack();
            new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index2.html',
                websiteErrorDocument: 'error.html',
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                WebsiteConfiguration: {
                    IndexDocument: "index2.html",
                    ErrorDocument: "error.html"
                }
            }));
            test.done();
        },
        'exports the WebsiteURL'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'Website', {
                websiteIndexDocument: 'index.html'
            });
            test.deepEqual(stack.resolve(bucket.bucketWebsiteUrl), { 'Fn::GetAtt': ['Website32962D0B', 'WebsiteURL'] });
            test.done();
        }
    },
    'Bucket.fromBucketArn'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketArn(stack, 'my-bucket', 'arn:aws:s3:::my_corporate_bucket');
        // THEN
        test.deepEqual(bucket.bucketName, 'my_corporate_bucket');
        test.deepEqual(bucket.bucketArn, 'arn:aws:s3:::my_corporate_bucket');
        test.done();
    },
    'Bucket.fromBucketName'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const bucket = lib_1.Bucket.fromBucketName(stack, 'imported-bucket', 'my-bucket-name');
        // THEN
        test.deepEqual(bucket.bucketName, 'my-bucket-name');
        test.deepEqual(stack.resolve(bucket.bucketArn), {
            'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket-name']]
        });
        test.done();
    },
    'if a kms key is specified, it implies bucket is encrypted with kms (dah)'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const key = new kms.Key(stack, 'k');
        // THEN
        new lib_1.Bucket(stack, 'b', { encryptionKey: key });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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