"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'Bucket with expiration days'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    expirationInDays: 30
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationInDays: 30,
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Can use addLifecycleRule() to add a lifecycle rule'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        const bucket = new lib_1.Bucket(stack, 'Bucket');
        bucket.addLifecycleRule({
            expirationInDays: 30
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationInDays: 30,
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Bucket with expiration date'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    expirationDate: new Date('2018-01-01')
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        ExpirationDate: '2018-01-01T00:00:00',
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Bucket with transition rule'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN
        new lib_1.Bucket(stack, 'Bucket', {
            lifecycleRules: [{
                    transitions: [{
                            storageClass: lib_1.StorageClass.Glacier,
                            transitionInDays: 30
                        }]
                }]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
            LifecycleConfiguration: {
                Rules: [{
                        Transitions: [{
                                StorageClass: 'GLACIER',
                                TransitionInDays: 30
                            }],
                        Status: "Enabled"
                    }]
            }
        }));
        test.done();
    },
    'Noncurrent rule on nonversioned bucket fails'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        // WHEN: Fail because of lack of versioning
        test.throws(() => {
            new lib_1.Bucket(stack, 'Bucket1', {
                lifecycleRules: [{
                        noncurrentVersionExpirationInDays: 10
                    }]
            });
        });
        // WHEN: Succeeds because versioning is enabled
        new lib_1.Bucket(stack, 'Bucket2', {
            versioned: true,
            lifecycleRules: [{
                    noncurrentVersionExpirationInDays: 10
                }]
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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