"use strict";
const cdk = require("@aws-cdk/cdk");
const util_1 = require("../lib/util");
module.exports = {
    parseBucketArn: {
        'explicit arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'my:bucket:arn';
            test.deepEqual(util_1.parseBucketArn(stack, { bucketArn }), bucketArn);
            test.done();
        },
        'produce arn from bucket name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'hello';
            test.deepEqual(stack.resolve(util_1.parseBucketArn(stack, { bucketName })), {
                'Fn::Join': ['',
                    ['arn:',
                        { Ref: 'AWS::Partition' },
                        ':s3:::hello']]
            });
            test.done();
        },
        'fails if neither arn nor name are provided'(test) {
            const stack = new cdk.Stack();
            test.throws(() => util_1.parseBucketArn(stack, {}), /Cannot determine bucket ARN. At least `bucketArn` or `bucketName` is needed/);
            test.done();
        }
    },
    parseBucketName: {
        'explicit name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'foo';
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketName })), 'foo');
            test.done();
        },
        'extract bucket name from string arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketArn })), 'my-bucket');
            test.done();
        },
        'can parse bucket name even if it contains a token'(test) {
            const stack = new cdk.Stack();
            const bucketArn = `arn:aws:s3:::${new cdk.Token({ Ref: 'my-bucket' })}`;
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketArn })), {
                "Fn::Select": [
                    0,
                    {
                        "Fn::Split": [
                            "/",
                            {
                                "Fn::Select": [
                                    5,
                                    {
                                        "Fn::Split": [
                                            ":",
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:aws:s3:::",
                                                        {
                                                            Ref: "my-bucket"
                                                        }
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    }
                ]
            });
            test.done();
        },
        'fails if ARN has invalid format'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'invalid-arn';
            test.throws(() => util_1.parseBucketName(stack, { bucketArn }), /ARNs must have at least 6 components/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC51dGlsLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC51dGlsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvQ0FBcUM7QUFFckMsc0NBQThEO0FBRTlELGlCQUFTO0lBQ1AsY0FBYyxFQUFFO1FBQ2QsY0FBYyxDQUFDLElBQVU7WUFDdkIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxTQUFTLEdBQUcsZUFBZSxDQUFDO1lBQ2xDLElBQUksQ0FBQyxTQUFTLENBQUMscUJBQWMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxTQUFTLEVBQUUsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQ2hFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw4QkFBOEIsQ0FBQyxJQUFVO1lBQ3ZDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQztZQUMzQixJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMscUJBQWMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDLEVBQUU7Z0JBQ25FLFVBQVUsRUFDUixDQUFDLEVBQUU7b0JBQ0QsQ0FBQyxNQUFNO3dCQUNMLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO3dCQUN6QixhQUFhLENBQUMsQ0FBQzthQUN0QixDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNENBQTRDLENBQUMsSUFBVTtZQUNyRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLHFCQUFjLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxFQUFFLDZFQUE2RSxDQUFDLENBQUM7WUFDNUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsZUFBZSxFQUFFO1FBRWYsZUFBZSxDQUFDLElBQVU7WUFDeEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDO1lBQ3pCLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxzQkFBZSxDQUFDLEtBQUssRUFBRSxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQztZQUM3RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUNBQXFDLENBQUMsSUFBVTtZQUM5QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFNBQVMsR0FBRyx3QkFBd0IsQ0FBQztZQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsc0JBQWUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7WUFDbEYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1EQUFtRCxDQUFDLElBQVU7WUFDNUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxTQUFTLEdBQUcsZ0JBQWdCLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLEdBQUcsRUFBRSxXQUFXLEVBQUUsQ0FBQyxFQUFFLENBQUM7WUFFeEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLHNCQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQyxFQUFFO2dCQUNuRSxZQUFZLEVBQUU7b0JBQ1osQ0FBQztvQkFDRDt3QkFDRSxXQUFXLEVBQUU7NEJBQ1gsR0FBRzs0QkFDSDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsR0FBRzs0Q0FDSDtnREFDRSxVQUFVLEVBQUU7b0RBQ1YsRUFBRTtvREFDRjt3REFDRSxlQUFlO3dEQUNmOzREQUNFLEdBQUcsRUFBRSxXQUFXO3lEQUNqQjtxREFDRjtpREFDRjs2Q0FDRjt5Q0FDRjtxQ0FDRjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1lBQzFDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sU0FBUyxHQUFHLGFBQWEsQ0FBQztZQUNoQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLHNCQUFlLENBQUMsS0FBSyxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1lBQ2pHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgcGFyc2VCdWNrZXRBcm4sIHBhcnNlQnVja2V0TmFtZSB9IGZyb20gJy4uL2xpYi91dGlsJztcblxuZXhwb3J0ID0ge1xuICBwYXJzZUJ1Y2tldEFybjoge1xuICAgICdleHBsaWNpdCBhcm4nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYnVja2V0QXJuID0gJ215OmJ1Y2tldDphcm4nO1xuICAgICAgdGVzdC5kZWVwRXF1YWwocGFyc2VCdWNrZXRBcm4oc3RhY2ssIHsgYnVja2V0QXJuIH0pLCBidWNrZXRBcm4pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdwcm9kdWNlIGFybiBmcm9tIGJ1Y2tldCBuYW1lJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IGJ1Y2tldE5hbWUgPSAnaGVsbG8nO1xuICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sucmVzb2x2ZShwYXJzZUJ1Y2tldEFybihzdGFjaywgeyBidWNrZXROYW1lIH0pKSwge1xuICAgICAgICAnRm46OkpvaW4nOlxuICAgICAgICAgIFsnJyxcbiAgICAgICAgICAgIFsnYXJuOicsXG4gICAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sXG4gICAgICAgICAgICAgICc6czM6OjpoZWxsbyddXVxuICAgICAgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2ZhaWxzIGlmIG5laXRoZXIgYXJuIG5vciBuYW1lIGFyZSBwcm92aWRlZCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiBwYXJzZUJ1Y2tldEFybihzdGFjaywge30pLCAvQ2Fubm90IGRldGVybWluZSBidWNrZXQgQVJOLiBBdCBsZWFzdCBgYnVja2V0QXJuYCBvciBgYnVja2V0TmFtZWAgaXMgbmVlZGVkLyk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9XG4gIH0sXG5cbiAgcGFyc2VCdWNrZXROYW1lOiB7XG5cbiAgICAnZXhwbGljaXQgbmFtZScodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXROYW1lID0gJ2Zvbyc7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlQnVja2V0TmFtZShzdGFjaywgeyBidWNrZXROYW1lIH0pKSwgJ2ZvbycpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdleHRyYWN0IGJ1Y2tldCBuYW1lIGZyb20gc3RyaW5nIGFybicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXRBcm4gPSAnYXJuOmF3czpzMzo6Om15LWJ1Y2tldCc7XG4gICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5yZXNvbHZlKHBhcnNlQnVja2V0TmFtZShzdGFjaywgeyBidWNrZXRBcm4gfSkpLCAnbXktYnVja2V0Jyk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBwYXJzZSBidWNrZXQgbmFtZSBldmVuIGlmIGl0IGNvbnRhaW5zIGEgdG9rZW4nKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgYnVja2V0QXJuID0gYGFybjphd3M6czM6Ojoke25ldyBjZGsuVG9rZW4oeyBSZWY6ICdteS1idWNrZXQnIH0pfWA7XG5cbiAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUocGFyc2VCdWNrZXROYW1lKHN0YWNrLCB7IGJ1Y2tldEFybiB9KSksIHtcbiAgICAgICAgXCJGbjo6U2VsZWN0XCI6IFtcbiAgICAgICAgICAwLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFwiRm46OlNwbGl0XCI6IFtcbiAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBcIkZuOjpTZWxlY3RcIjogW1xuICAgICAgICAgICAgICAgICAgNSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgXCJGbjo6U3BsaXRcIjogW1xuICAgICAgICAgICAgICAgICAgICAgIFwiOlwiLFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgXCJhcm46YXdzOnMzOjo6XCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIm15LWJ1Y2tldFwiXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9KTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZmFpbHMgaWYgQVJOIGhhcyBpbnZhbGlkIGZvcm1hdCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXRBcm4gPSAnaW52YWxpZC1hcm4nO1xuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gcGFyc2VCdWNrZXROYW1lKHN0YWNrLCB7IGJ1Y2tldEFybiB9KSwgL0FSTnMgbXVzdCBoYXZlIGF0IGxlYXN0IDYgY29tcG9uZW50cy8pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcbn07XG4iXX0=