"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStore = exports.DeletionProtectionMode = exports.ValidationSettingsMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
const cedar_helpers_1 = require("./cedar-helpers");
const policy_1 = require("./policy");
const permissions_1 = require("./private/permissions");
const RELEVANT_HTTP_METHODS = ['get', 'post', 'put', 'patch', 'delete', 'head'];
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
var ValidationSettingsMode;
(function (ValidationSettingsMode) {
    ValidationSettingsMode["OFF"] = "OFF";
    ValidationSettingsMode["STRICT"] = "STRICT";
})(ValidationSettingsMode || (exports.ValidationSettingsMode = ValidationSettingsMode = {}));
var DeletionProtectionMode;
(function (DeletionProtectionMode) {
    DeletionProtectionMode["ENABLED"] = "ENABLED";
    DeletionProtectionMode["DISABLED"] = "DISABLED";
})(DeletionProtectionMode || (exports.DeletionProtectionMode = DeletionProtectionMode = {}));
class PolicyStoreBase extends core_1.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.policyStoreArn],
            scope: this,
        });
    }
    grantAuth(grantee) {
        return this.grant(grantee, ...permissions_1.AUTH_ACTIONS);
    }
    grantRead(grantee) {
        return this.grant(grantee, ...permissions_1.READ_ACTIONS);
    }
    grantWrite(grantee) {
        return this.grant(grantee, ...permissions_1.WRITE_ACTIONS.concat(permissions_1.READ_ACTIONS));
    }
}
class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope, id, policyStoreId) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreId });
    }
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope, id, policyStoreArn) {
        return PolicyStore.fromPolicyStoreAttributes(scope, id, { policyStoreArn });
    }
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope, id, attrs) {
        class Import extends PolicyStoreBase {
            constructor(policyStoreArn, policyStoreId) {
                super(scope, id);
                this.policyStoreArn = policyStoreArn;
                this.policyStoreId = policyStoreId;
            }
        }
        let policyStoreId;
        let policyStoreArn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.policyStoreId) {
            if (!attrs.policyStoreArn) {
                throw new Error('One of policyStoreId or policyStoreArn is required!');
            }
            policyStoreArn = attrs.policyStoreArn;
            const maybeId = stack.splitArn(attrs.policyStoreArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for PolicyStore must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            policyStoreId = maybeId;
        }
        else {
            if (attrs.policyStoreArn) {
                throw new Error('Only one of policyStoreArn or policyStoreId can be provided');
            }
            policyStoreId = attrs.policyStoreId;
            policyStoreArn = stack.formatArn({
                resource: 'policy-store',
                resourceName: attrs.policyStoreId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(policyStoreArn, policyStoreId);
    }
    /**
     * This method generates a schema based on an swagger file. It makes the same assumptions and decisions
     * made in the Amazon Verified Permissions console. This feature is built for swagger files generated from an Amazon API Gateway
     * export. It's possible that some swagger files generated by other tools will not work. In that case, please
     * file an issue.
     * @param swaggerFilePath absolute path to a swagger file in the local directory structure, in json format
     * @param groupEntityTypeName optional parameter to specify the group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromOpenApiSpec(swaggerFilePath, groupEntityTypeName) {
        const openApiSpecString = fs.readFileSync(swaggerFilePath, 'utf-8');
        const openApiSpec = JSON.parse(openApiSpecString);
        if (!openApiSpec.paths) {
            throw new Error('Invalid OpenAPI spec - missing paths object');
        }
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(openApiSpec.info.title);
        const pathUrls = Object.keys(openApiSpec.paths);
        const actionNames = [];
        for (const pathUrl of pathUrls) {
            const pathDef = openApiSpec.paths[pathUrl];
            if (!pathDef) {
                continue;
            }
            let pathVerbs = Object.keys(pathDef);
            if (pathVerbs.includes('x-amazon-apigateway-any-method')) {
                pathVerbs = RELEVANT_HTTP_METHODS;
            }
            for (const httpVerb of pathVerbs) {
                if (!RELEVANT_HTTP_METHODS.includes(httpVerb)) {
                    continue;
                }
                const actionName = `${httpVerb} ${pathUrl}`;
                actionNames.push(actionName);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    /**
     * This method generates a schema based on an AWS CDK RestApi construct. It makes the same assumptions
     * and decisions made in the Amazon Verified Permissions console.
     *
     * @param restApi The RestApi construct instance from which to generate the schema.
     * @param groupEntityTypeName Specifies a group entity type name. If passed, the schema's User type will have a parent of this type.
     */
    static schemaFromRestApi(restApi, groupEntityTypeName) {
        const namespace = (0, cedar_helpers_1.cleanUpApiNameForNamespace)(restApi.restApiName);
        const actionNames = [];
        for (const method of restApi.methods) {
            const pathVerb = method.httpMethod.toLowerCase();
            const pathUrl = method.resource.path;
            if (pathVerb === 'any') {
                for (const verb of RELEVANT_HTTP_METHODS) {
                    actionNames.push(`${verb} ${pathUrl}`);
                }
            }
            if (RELEVANT_HTTP_METHODS.includes(pathVerb)) {
                actionNames.push(`${pathVerb} ${pathUrl}`);
            }
        }
        return (0, cedar_helpers_1.buildSchema)(namespace, actionNames, groupEntityTypeName);
    }
    constructor(scope, id, props = {
        validationSettings: {
            mode: ValidationSettingsMode.OFF,
        },
        deletionProtection: DeletionProtectionMode.DISABLED,
    }) {
        super(scope, id);
        if (props.schema) {
            (0, cedar_helpers_1.checkParseSchema)(props.schema.cedarJson);
        }
        this.policyStore = new aws_verifiedpermissions_1.CfnPolicyStore(this, id, {
            schema: props.schema
                ? {
                    cedarJson: props.schema.cedarJson,
                }
                : undefined,
            validationSettings: {
                mode: props.validationSettings.mode,
            },
            description: props.description,
            deletionProtection: {
                mode: (props.deletionProtection) ? props.deletionProtection : DeletionProtectionMode.DISABLED,
            },
            tags: props.tags,
        });
        this.policyStoreArn = this.getResourceArnAttribute(this.policyStore.attrArn, {
            resource: 'policy-store',
            resourceName: this.physicalName,
            service: 'verifiedpermissions',
        });
        this.policyStoreName = this.getResourceNameAttribute(this.policyStore.ref);
        this.policyStoreId = this.policyStore.attrPolicyStoreId;
        this.schema = props.schema;
        this.validationSettings = props.validationSettings;
        this.description = props.description;
        this.deletionProtection = (props.deletionProtection) ? props.deletionProtection : DeletionProtectionMode.DISABLED;
    }
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions) {
        let policies = policyDefinitions.map((policyOption) => {
            return new policy_1.Policy(this, policyOption.policyId, {
                policyStore: this,
                definition: policyOption.policyConfiguration,
            });
        });
        return policies;
    }
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath) {
        if (!fs.statSync(absolutePath).isDirectory()) {
            throw new Error(`The path ${absolutePath} does not appear to be a directory`);
        }
        const policyFileNames = fs
            .readdirSync(absolutePath)
            .map((f) => path.join(absolutePath, f))
            .filter((f) => !fs.statSync(f).isDirectory() && f.endsWith('.cedar'));
        if (this.validationSettings.mode === ValidationSettingsMode.STRICT) {
            if (!this.schema) {
                throw new Error('A schema must exist when adding policies to a policy store with strict validation mode.');
            }
            for (const policyFile of policyFileNames) {
                const policyStatement = fs.readFileSync(policyFile, 'utf-8');
                (0, cedar_helpers_1.validatePolicy)(policyStatement, this.schema.cedarJson);
            }
        }
        const policies = policyFileNames.map((cedarFile) => policy_1.Policy.fromFile(this, cedarFile, {
            path: cedarFile,
            policyStore: this,
        }));
        return policies;
    }
}
exports.PolicyStore = PolicyStore;
_a = JSII_RTTI_SYMBOL_1;
PolicyStore[_a] = { fqn: "@cdklabs/cdk-verified-permissions.PolicyStore", version: "0.2.1" };
//# sourceMappingURL=data:application/json;base64,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