from typing import List, Tuple
from shttst.text.shmart_nums import normalize_numbers
import re

rules_def: List[Tuple[str, str]] = [
    ('\'', ''),
    ('”', '"'),
    ('„', '"'),
    ('–', '-'),
    ('—', '-'),
    ('...', '…'),
    ('»', '"'),
    ('’', "'"),
    ('«', '"'),
    ('\(', '"'),
    ('\)', '"'),
    ('odziem', 'od_ziem'),
    ('ęzis', 'ęz_is'),
    ('arzn', 'ar_zn'),
    ('arzl', 'ar_zl'),
    ('arzł', 'ar_zł'),
    ('asilo', 'as_ilo'),
    (' tarzan', ' tar_zan'),
    ('kontratak', 'kontr_atak'),
    ('mierzi', 'mier_zi'),
    ('mierzł', 'mier_zł'),
    ('odzown', 'od_zown'),
    ('zinte', 'z_inte'),
    (' zign', ' z_ign'),
    (' bezi', ' be_zi'),
    (' ziry', ' z_iry'),
    ('zide', 'z_ide'),
    ('ziszcz', 'z_iszcz'),
    (' nadzor', ' nad_zor'),
    ('hunter', 'hanter'),
    (' slums', ' slams'),
    (' bruce', ' brus'),
    (' hulk', ' halk'),
    ('she-hulk', 'szi-halk'),
    ('lady-hulk', 'lejdi-halk'),
    ('girl-hulk', 'gerl-halk'),
    (' walter', ' łolter'),
    (' steve', ' stiw'),
    (' roger', ' rodżer'),
    ('jennifer', 'dżenifer'),
    (' jen ', ' dżen '),
    (' yen ', ' jen '),
    ('cheetos', 'czitos'),
    (' holiway', 'holiłej'),
    (' blonsky', 'blonski'),
    ('carring', 'karing'),
    ('izaac', 'izaak'),
    ('sprite', 'sprajt'),
    ('avonlea', 'avonli'),
    (' chubb', 'czab'),
    (' nigel', ' najdżel'),
    ('lady ', 'lejdi '), #unsafe
    ('pauline', 'polin'),
    (' jerome', ' dżerom'),
    ('rodney', 'rodnyj'),
    ('morgarath', 'morgaraf'),
    (' horace', ' horys'),
    (' mount ', ' maunt '),
    ('beethoven', 'betoven'),
    ('mozart', 'mocart'),
    (' howard', ' hałard'),
    (' clara', ' klara'),
    (' duncan', ' dankan'),
    (' hackham', ' hakam'),
    ('slipsunder', 'slipsander'),
    (' george', 'dżordż'),
    (' sir ', ' ser '),
    (' jenny', ' dżeny'),
    (' rosie ', ' roz_i '),
    (' melanie ', ' melani '),
    (' caldwell ', ' kaldłel '),
    (' kitty ', ' ki_ti '),
    (' sarah ', ' sara '),
    ('cruiser', 'kruzer'),
    (' cushing ', ' kaszing '),
    (' chandler ', ' czendler '),
    (' newport ', ' niuport '),
    (' david ', ' dejwid '),
    (' fielding ', ' filding '),
    ('antoinette', 'antłenet'),
    ('wallace', 'łoles'),
    ('permettez-moi', 'per-me-te-mła'),
    ('carter', 'karter'),
    ('cordom', 'kordom'),
    ('wensley', 'łenzli'),
    ('casting', 'kasting'),
    ('klermont', 'klermąn'),
    ('melbourne', 'melbern'),
    ('shirley', 'szirlij'),
    (' alice', ' alis'),
    ('auschwitz', 'ałszwic'),
    ('beethoven', 'betowen'),
    ('benjamin', 'bendżamin'),
    ('birmingham', 'birmingam'),
    ('bismarck', 'bismark'),
    (' blake', ' blejk'),
    ('brewster', 'bruster'),
    ('brooklyn', 'bruklin'),
    (' brown', ' brałn'),
    (' bush', ' busz'),
    (' carlos', ' karlos'),
    ('castaneda', 'kastaneda'),
    (' carson', ' karson'),
    (' carter', ' karter'),
    ('catalina', ' katalina'),
    ('charles', 'czarls'),
    ('charlie', 'czarli'),
    ('charlotte', 'szarlot'),
    (' chase', ' czejs'),
    ('chelsea', 'czels_i'),
    (' chopin', ' szopen'),
    (' colorado', ' kolorado'),
    (' columb', ' kolumb'),
    ('connor', 'konor'),
    ('conrad', 'konrad'),
    ('crusoe', 'kruzo'),
    (' duncan', ' dankan'),
    ('eiffla', 'ajfla'),
    ('everest', 'ewrest'),
    ('facebook', 'fejsbuk'),
    (' federic', ' federik'),
    (' freya', ' freja'),
    (' gray', ' grej'),
    ('greenwich', 'grinłicz'),
    (' grey', ' grej'),
    ('isabel', 'izabel'),
    ('jacob', ' dżejkob'),
    (' james', ' dżejms'),
    (' jeremy', ' dżeremi'),
    (' jenkins', ' dżenkins'),
    (' jim', ' dżim'),
    (' joe ', ' dżo '),
    (' john', ' dżon'),
    (' justin', ' dżastin'),
    ('liverpool', 'liwerpul'),
    ('lorraine', 'loren'),
    (' louis', ' lu_łiz'),
    (' lucy ', ' lus_i '),
    ('malcolm', 'malkolm'),
    ('massachusetts', 'masaczjusets'),
    (' michael', ' majkel'),
    ('microsoft', 'majkrosoft'),
    ('raymond', 'rejmond'),
    (' mike', ' majk'),
    (' miles', ' majls'),
    (' mount', ' małnt'),
    (' movie', ' muwi'),
    (' murphy', ' merfi'),
    (' nicol', ' nikol'),
    (' owen', ' ołen'),
    (' pierre', ' pier'),
    ('requiem', 'rekfiem'),
    (' richard', ' riczard'),
    (' schubert', ' szubert'),
    (' schwartz', ' szwarc'),
    (' scotland', ' skotland'),
    (' schroeder', ' szreder'),
    (' schlesinger', ' szlezinger'),
    (' scott', ' skot'),
    ('siergiej', 'sjergiej'),
    (' skype', ' skajp'),
    (' spice', ' spajs'),
    (' steal ', ' stil '),
    (' stephen', ' stiwen'),
    (' steven', ' stiwen'),
    (' stuart', ' stju_art'),
    (' tequil', ' tekil'),
    (' tracy', ' trejs_i'),
    (' train', ' trejn'),
    (' true ', ' tru '),
    (' victor', ' wiktor'),
    (' wells', ' łels'),
    (' william', ' łyliam'),
    (' wilson', ' łylson'),
    (' world', ' łerld'),
    ('sawyer', ' sojer'),
    ('manhattan', 'manhatan'),
    ('morningside', 'morningsajd'),
    ('highway', 'hajłej'),
    (' frau ', ' frał '),
    ('deutsche', 'dojcze'),
    ('strauss', 'sztrałs'),
    ('rozenthal', 'rozental'),
    ('louis', 'lułis'),
    ('hollywood', 'holiłud'),
    (' madame ', ' madam '),
    ('steel', 'stil'),
    ('dashwood', 'daszłud'),
    ('middleton', 'midlton'),
    ('street', 'strit'),
    ('jennings', 'dżenings'),
    ('temple ', 'templ'),
    ('t-shirt', 'tiszert'),
    ('smith', 'smif'),
    ('thatcher', 'taczer'),
    ('cardiff', 'kardif'),
    ('cardiff', 'kardif'),
    (' hill ', ' hil '),
    ('douglas', 'daglas'),
    ('lunch', 'lancz'),
    ('york', 'jork'),
    ('crowmont', 'krołmont'),
    (' brian', 'brajan'),
    (' withmore', 'łifmor'),
    (' whitmore', 'łitmor'),
    ('whitemore', 'łajtmor'),
    (' cowell', 'kołel'),
    ('scrooge', 'skrudż'),
    ('marley', 'marlej'),
    (' pepsi', ' peps_i'),
    ('cornhill', 'korn_hill'),
    (' camden', ' kamden'),
    (' town ', ' tałn '),
    ('debbie', 'debi'),
    (' cowboy', ' kałboj'),
    (' pool', ' pul'),
    (' chi ', ' czi '),
    ('ernshow', 'ernszoł'),
    ('speech', 'spicz'),
    ('bright', 'brajt'),
    ('white', 'łajt'),
    ('francesc', 'franczesk'),
    ('skellige', 'skelige'),
    ('percival', 'pers_iwal'),
    ('chivay', 'cziwaj'),
    ('djinn', 'dżin'),
    ('incognito', 'inkognito'),
    ('vilgefortz', 'vilgeforc'),
    (' ciri', ' c_iri'),
    (' metz ', ' mec '),
    (' dick', ' dik'),
    ('pudding', 'pading'),
    ('thyssen', 'fyssen'),
    ('lorenzo', 'lorenco'),
    ('schiller', 'sziler'),
    ('caesar', 'cezar'),
    ('hemicrania', 'hemikrania'),
    ('procurator', 'prokurator'),
    ('aurillac', 'aurillak'),
    ('stratonic', 'stratonik'),
    ('gaziet', 'gasjet'),
    ('ad acta', 'ad akta'),
    ('de facto', 'de fakto'),
    ('goethe', 'gete'),
    ('ambrosij', 'ambroz_ij'),
    ('colosseum', 'koloseum'),
    ('au naturel', 'o natjurel'),
    ('shakespeare', 'szekspir'),
    (' menu ', ' meni '),
    (' kaer ', ' ker '),
    (' morhen', ' moren'),
    (' tizian', ' ticjan'),
    (' toussaint', ' tusą'),
    ('vsky', 'wski'),
    ('thank you', 'fenk ju'),
    (' nein ', ' najn '),
    ('qun', 'kun'),
    ('lays', 'lejs'),
    (' chip', ' czip'),
    (' voodoo', ' wudu'),
    (' runner', ' raner'),
    (' runner', ' raner'),
    ('afterlife', 'afterlajf'),
    (' night', ' najt'),
    (' city', ' s_iti'),
    (' wasil', ' was_il'),
    (' jazz', ' dżez'),
    (' de volaille', ' de wolajl'),
    ('kalkstein', 'kalksztajn'),
    ('goldstein', 'goldsztajn'),
    ("scoia'tael", 'skoja tel'), 
    ("abigail", 'abigejl'),
    ('beatles', 'bitels'),
    ('w ogóle', 'wogóle'),
    ('starkhaven', 'starkhejwen'),
    (' north', ' norf'),
    (' oak', ' ołk'),
    ('rainbow', 'rejnboł'),
    ('underground', 'andergrand'),
    ('hand ', 'hend '),
    ('hobby', 'hobbi'),
    ('winston', 'łinston'),
    ('status quo', 'status kwo'),

    # (' ', ''),
    ('m.in.', 'między innymi '),
    (' etc.', 'et_cetera '),
    (' pt.', ' pod tytułem '),
    (' vhs ', ' fał ha es '),
    (' nr ', ' numer '),
    
    # (' ', ''),
    ('é', 'e'),
    ('ü', 'y'),
    ('ö', 'y'),
    ('ä', 'e'),
    ('x', 'ks'),
    ('sh', 'sz'),
    ('zż', 'z-ż'),
    ('rż', 'r-ż'),
    ('zsz', 'z_sz'),
    ('ższ', 'ż_sz'),
    ('ww', 'w\'w'),
    ('nn', 'n\'n'),
    ('ss', 's\'s'),
    ('tc', 't\'c'),
    ('dd', 'd\'d'),
    ('pp', 'p\'p'),
    ('tt', 't\'t'),
    ('zz', 'z\'z'),
]
    
def gen_rule(rule_tuple: Tuple[str, str]) -> Tuple[re.Pattern, str]:
    from_rule, to_rule = rule_tuple
    prefix, suffix = False, False
    
    from_rule = from_rule.replace('.', '\\.')

    if from_rule[0] == ' ':
        prefix = True
    
    if from_rule[-1] == ' ':
        suffix = True
    
    _prefix = '([ \,\.\;\!\:\?]|^)' if prefix else ''
    _suffix = '([ \,\.\;\!\:\?]|$)' if suffix else ''

    _from_rule = (f'{_prefix}{from_rule.strip()}{_suffix}')

    _to_prefix = '\g<1>' if _prefix != '' else ''
    _to_suffix = '\g<1>' if _prefix == '' and _suffix != '' else '\g<2>' if _suffix != '' else ''

    _to_rule = (f'{_to_prefix}{to_rule.strip()}{_to_suffix}')

    return re.compile(_from_rule), _to_rule

rules = [*map(gen_rule, rules_def)]

prelower_rules = [
    (' GLKiH', ' Gie eL Ka i Ha'),
    (' GSzG', 'Gie es zet Gie'),
    (' SWD', ' eS Wu De'),
    (' DNA', ' De En A'),
    (' USA', ' U eS A'),
    ('Red Mount', 'Red Małnt'),
    (' Zin', ' Z_in'),
    ('Will', 'Łil'),
    ('Andy', 'Endi'),
    ('Betty', 'Beti'),
    (' Vincent', ' Winsent'),
    ('Trump', 'Tramp'),
    ('Fieldstone', 'Fildston'),
    ('West Side', 'Łest Sajd'),
    ('Hudson', 'Hadson'),
    ('Hofmeister', 'Hofmajster'),
    ('Lucy ', 'Lus_i '),
    ('Lady ', 'Lejdi '),
    ('Marianne ', 'Marian '),
    ('Cleveland', 'Kliwlend'),
    ('Willoughby', 'Łilołbi'),
    ('Jake ', 'Dżejk '),
    ('Jessie', 'Dżes_i'),
    ('Andrews', 'Endrjus'),
    ('Carmen', 'Karmen'),
    ('Doug ', 'Dag '),
    ('Craig', 'Krejg'),
    ('Sophie', 'Sofi'),
    ('Carlit', 'Karlit'),
    (' Misty ', ' Misti '),
    ('Joe', 'Dżo'),
    ('Kate ', 'Kejt '),
    ('Oate', 'Ołen'),
    ('Maisy', 'Mejz_i'),
    (' Simon', ' Sajmon'),
    (' Boyce', ' Bojs'),
    (' Yennefer', ' Jenefer'),
    (' Neyd', ' Nejd'),
    (' Chaplin', ' Czaplyn'),
    (' SOS ', ' es o es '),
    (' Col', ' Kol'),
    (' SS ', ' eS eS '),
    (' Schtreck ', ' Sztrek '),
    ('Mary Poppins', 'Mery Popiins'),
    ('Mein Kampf', 'Majn Kampf'),
    ('La Manche', 'La-Mansz'),
    ('Führer', 'Firer'),
    ('Ahnenerbe', 'Anen_Erbe'),
    ('MacMillan', 'Mak_Milan'),
    (' Mac ', ' Mak '),
    (' Buck', ' Bak'),
    (' Randy', ' Randi'),
    (' Nellie', ' Neli'),
    (' Cooper', ' Kuper'),
    (' Jack ', ' Dżak '),
    (' Randy', ' Randij'),
    (' Michigan', ' Miszigen'),
    (' Bobby', ' Bobi'),
    (' Buster', ' Baster'),
    (' Keaton', ' Kiton'),
    (' Goebbel', ' Gebel'),
    (' Heil', ' Hajl'),
    (' Riefenstahl', ' Rifensztal'),
    (' Einstein', ' Ajnsztajn'),
    (' Walt', ' Łolt'),
    (' Disney', ' Di_znyj'),
    ('Dmitry', 'Dmitri'),
    ('Copperfield', 'Koperfild'),
    ('We all live in our yellow submarine, Yellow submarine, yellow submarine', 'Łi ol liw in ałer jeloł submarin, jeloł submarin, jeloł submarin'),
    ('And our friends are all aboard, Many more of them live next door', 'end ałer frends ar ol e-bord, men_i mor of wem liw next dor'),
    ('And the band begins to play', 'end we band begins tu plej'),
    ('My God… White skin, this is really white skin', 'maj god… łajt skin, dis is rili łajt skin'),
    ('Не asked, where did you come from', 'hi askt, łer did ju kam from'),
    ("A white face! After so many years! Please, help me! I 'm a citizen of the United States", 'e łajt fejs! after soł men_i jers! pliz, help mi! ajm e s_itizen of we junajted stejts'),
    ("We were shot down twenty years ago… In the desert.", 'łi łer szot dałn tłenti jers egoł... in we dezert.'),
    ("Pilots were killed, and I was imprisoned by these savages", 'pajlots łer kilt, end aj łos imprizont baj diz sawydżys'),
    ("You are Russian, right", 'ju ar raszjan, rajt'),
    ("I 'm lieutenant Jenny Tahoma", 'ajm lewtenant dżen_i tahoma'),
    ("Are you from military", 'ar ju from militari'),
    ("Did you come here with a group", 'did ju kam hir łyf e grup'),
    ("How many people are with you", 'hał men_i pipyl ar łyf ju'),
    ("It will kill you", 'it łil kil ju'),
    ("Bitte geben Sie mir ein Schluck", 'bite gejben z_i mir ajn szluk'),
    ("Wasser", 'Wassaa'),
]

rules2 = [*map(gen_rule, prelower_rules)]

def shmart_cleaner(text: str) -> str:
    if not len(text):
        return '___'
    text = normalize_numbers(text)
    for _, (rule_from, rule_to) in enumerate(rules2):
        text = rule_from.sub(rule_to, text)

    text = text[1:] if text[0] == '-' else text
    text = text.lower()

    for _, (rule_from, rule_to) in enumerate(rules):
        text = rule_from.sub(rule_to, text)
    
    if text.strip()[-1] not in ['.', ',', '!', '-', '?', ';', ':']:
        text += ','
    
    return text

if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--text', type=str, help='text to clean', default='Podziemny krąg.')
    args = parser.parse_args()

    print(shmart_cleaner(args.text))