'''
Author: bo-qian bqian@shu.edu.cn
Date: 2025-07-01 15:38:16
LastEditors: bo-qian bqian@shu.edu.cn
LastEditTime: 2025-07-10 15:21:59
FilePath: /boviz/src/boviz/cli.py
Description: This module provides a command-line interface for boviz, allowing users to initialize a new plotting project with a template.
Copyright (c) 2025 by Bo Qian, All Rights Reserved. 
'''

import argparse
from pathlib import Path

TEMPLATE_CODE = '''\
import os
from boviz import *

base_dir = os.path.dirname(os.path.abspath(__file__))
csv_path = os.path.join(base_dir, 'data', 'example.csv')
# exodus_path = os.path.join(base_dir, "data/test_two_particle_viscos_sintering.e")

#1. 绘制初始粒子分布示意图
plot_initial_particle_schematic(
    coordinates=[[90, 90], [150, 90]],
    radii=[30, 30],
    domain=[240, 180],
    title="Initial Particle Distribution",
    show=False,
    save=True
)

# 2. 多曲线对比：不同实验和模拟条件下的收缩率对比
plot_curves_csv(
    path=[csv_path, csv_path, csv_path, csv_path],
    label=["Exp 800K", "Exp 900K", "Sim 800K", "Sim 900K"],
    x=[0, 0, 0, 0],
    y=[1, 2, 3, 4],
    xy_label=["Time (s)", "Shrinkage Ratio"],
    title_figure="Shrinkage Comparison at Two Temperatures",
    use_marker=[True, True, False, False],
    legend_ncol=2,
    save=True,
    show=False
)

# 3. 单曲线绘图：绘制单条模拟曲线
plot_curves_csv(
    path=[csv_path],
    label=["Sim 800K"],
    x=[0],
    y=[3],
    title_figure="Shrinkage at 800K",
    save=True,
    show=False
)

# 4. 样式演示：展示不同颜色、marker、线型等样式
plot_curves_csv(
    path=[csv_path, csv_path],
    label=["Exp 800K", "Exp 900K"],
    x=[0, 0],
    y=[1, 2],
    xy_label=["Time (s)", "Shrinkage Ratio"],
    use_marker=[True, True],
    title_figure="Style Demo",
    save=True,
    show=False
)

# 5. 残差分析图：展示两条曲线的残差
plot_curves_csv(
    path=[csv_path, csv_path],
    label=["Sim 800K", "Sim 900K"],
    x=[0, 0],
    y=[3, 4],
    xy_label=["Time (s)", "Shrinkage Ratio"],
    title_figure="Residual Analysis",
    show=False,
    save=True,
    show_residual=True
)

# 6. 直接传入数据进行绘图
x = np.linspace(0, 4*np.pi, 200)
y = np.sin(x)
plot_curves(
    data=[(x, y)],
    label=[r"$\sin(x)$"],
    xy_label=(r"$x$", r"$\sin(x)$"),
    title_figure="Sine Wave Example",
    save=True,
    show=False
)

# # 7. 绘制热图示例
# plot_heatmap_exodus2d(
#     path=exodus_path,
#     variable="Real_Pressure",
#     time_step=5,
#     cmap='jet',
#     save=True,
#     show=True,
#     font_style='sans',
#     show_ticks=False
# )
'''

EXAMPLE_CSV = '''\
Time,Shrinkage_A,Shrinkage_B,Shrinkage_C,Radius_A,Radius_B,Residual_A,Residual_B
0.0,1.0209447982777509,1.0,1.022544492624045,1.0,1.0,0.09145300179805697,0.11871277304679126
0.05025125628140704,0.93078700119278,0.9950230947479447,0.850618700274092,0.9852911718279044,0.9551664160337779,0.03734231526877581,0.05188018637730255
0.10050251256281408,0.9337694925771136,0.9901335064388825,1.0672554599457287,0.9712699284723975,0.9365957375051759,0.0018022256126726242,0.03991562571165031
0.15075376884422112,0.9124100769886012,0.9853175857320949,0.9646029534111357,0.9578748463510758,0.9223459546850981,0.013251253394357189,0.04799346557144779
0.20100502512562815,0.9395107513382187,0.9805617572379888,0.9646678928631606,0.9450523818378945,0.9103328320675559,0.027040150820159755,0.013400072751493028
0.2512562814070352,0.918517171991616,0.9758525511919605,0.8330566604083536,0.9327555784779354,0.8997490585765828,0.04687877579412846,0.07079871530129744
0.30150753768844224,0.8792210653099232,0.9711766348875863,1.0178464777963514,0.9209430289074864,0.8901805959425307,0.05115864797784769,0.01637864646243115
0.35175879396984927,0.9465036609532331,0.966520843789521,0.9167977808873513,0.909578033536796,0.8813814864416436,0.03915920900295252,0.02165595915130574
0.4020100502512563,0.8648709552514344,0.9618722122472979,0.7468935697743287,0.898627912870938,0.8731914750103518,0.0015216709695828187,0.16149898160677276
0.45226130653266333,0.7297325631564083,0.9572180037322346,0.7770582968531939,0.8880634409836181,0.8654992481013339,0.014842237638382058,0.08999962204482043
0.5025125628140704,0.8385407139823808,0.952545740520853,0.8857389370870143,0.8778583754001418,0.8582237589983328,0.02046920010532116,0.11684521529429073
0.5527638190954774,0.8618799267031556,0.947843232749624,0.9049301929477894,0.8679890643480241,0.8513038239771477,0.09879875311791815,0.009929764081485224
0.6030150753768845,0.8120563476107203,0.9430986067674376,0.9060042435981566,0.8584341165818943,0.8446919093701961,0.018986359993188442,0.01401429853297645
0.6532663316582915,0.8421197200350728,0.9383003327139714,0.7334766122527371,0.8491741221877361,0.8383502141469663,0.114070050811105,0.19307785814648543
0.7035175879396985,0.8199321162854664,0.9334372512540906,0.8347236873679941,0.8401914152032126,0.8322480893772356,0.09373711334555605,0.006485559972505346
0.7537688442211056,0.8434832280195514,0.9284985994005441,0.8222161664438415,0.8314698707566196,0.8263602759480302,0.06641088494373783,0.09481741774573418
0.8040201005025126,0.7091516423587425,0.9234740353595257,0.8887021066846745,0.822994730870864,0.8206656641351119,0.0074426369810544615,0.10809377189368738
0.8542713567839196,0.8144569222301328,0.9183536623361391,0.776692338404998,0.8147524542051896,0.8151463977322682,0.03508069652592302,0.073705638590358
0.9045226130653267,0.8968948831829472,0.9131280512394333,0.7301604661839558,0.8067305858926198,0.8097872125155275,0.01700055455711631,0.029023520822439725
0.9547738693467337,0.7522909859589794,0.907788262229454,0.6981179559511411,0.7989176443318138,0.8045749382144927,0.04922967894025382,0.011753916404921183
1.0050251256281408,0.6919002519040867,0.9023258650516879,0.8439697017018186,0.7913030223503886,0.7994981171531658,0.02360888478788168,0.1486274798146839
1.0552763819095479,0.7576304833084876,0.8967329581073316,0.8055380258575947,0.7838769006045072,0.794546707798629,0.018984659599647014,0.03866413348372867
1.105527638190955,0.7879593958948011,0.8910021862110196,0.8034090356309292,0.7766301714407065,0.7897118511954652,0.015154656406218981,0.1731722575973963
1.155778894472362,0.6452574781504304,0.8851267569909556,0.7400612495142068,0.7695543717389426,0.7849856847117047,0.06297819996911212,0.013240218568550444
1.206030150753769,0.5869490048614198,0.8791004558898244,0.7405120288851479,0.7626416234947964,0.7803611918850615,0.04520567816145864,0.03045080451141218
1.256281407035176,0.6742331720605388,0.8729176597283959,0.5880146281850022,0.7558845810946715,0.7758320801688898,0.042633534039092126,0.15175743435602576
1.306532663316583,0.6343243696465698,0.8665733487973595,0.8053731494888088,0.7492763843991841,0.7713926804919332,0.03368916252678005,0.0035781363245920464
1.35678391959799,0.6386333140263972,0.8600631174466541,0.6158615015309037,0.7428106168834638,0.7670378640552942,0.0450828211262279,0.051228660246413975
1.407035175879397,0.5825874070448445,0.8533831831453447,0.6614376191149248,0.7364812681940731,0.7627629728832873,0.10297633293217337,0.0008451363498873521
1.4572864321608041,0.6621401144672502,0.8465303939889643,0.689977285080868,0.7302827005748862,0.7585637614473913,0.01729401759866454,0.047783871307394386
1.5075376884422111,0.6766371306335031,0.8395022346351646,0.5928612953156425,0.724209618691892,0.7544363472789826,0.061042281871111416,0.08138119124950707
1.5577889447236182,0.6127298923812176,0.8322968306524806,0.8001731493389628,0.7182570424521909,0.7503771689349215,0.02719961686315634,0.06959620893315555
1.6080402010050252,0.6088321318668333,0.8249129512710338,0.6274251391829577,0.7124202824675812,0.7463829500207034,0.038117151660807466,0.11310788226081876
1.6582914572864322,0.7081434077229812,0.8173500105280241,0.6067908972831708,0.7066949178598578,0.7424506682372146,0.014688723010190863,0.0056638867980286466
1.7085427135678393,0.5631089378955199,0.809608066804923,0.64058553168321,0.7010767761446486,0.7385775286194518,0.01838870232737677,0.03669615146204832
1.7587939698492463,0.667627256495273,0.8016878207573402,0.46705999308017854,0.6955619149644885,0.7347609402935347,0.003589484761932637,0.013494287989343797
1.8090452261306533,0.6098397212696849,0.7935906116425939,0.6639772957844278,0.690146605470818,0.7309984962026678,0.04789697684585741,0.06333996306365457
1.8592964824120604,0.4933066288247342,0.7853184120540673,0.628306763724403,0.6848273171794484,0.7272879553512855,0.003413449956965813,0.01370299766556954
1.9095477386934674,0.5869091567682225,0.7768738210754518,0.6307383248838679,0.6796007041454492,0.7236272271953355,0.005685354832301275,0.01620005097939397
1.9597989949748744,0.5745421073016223,0.7682600558719739,0.6341535915207429,0.6744635923218718,0.7200143578699169,0.041242833736057916,0.0008390698571727638
2.0100502512562817,0.5364000668608677,0.7594809417396463,0.48842299103788384,0.669412967982713,0.7164475179966656,0.0070512918284863935,0.031115177634543564
2.0603015075376887,0.5164709901922427,0.7505409006374838,0.5639972614372138,0.6644459671043785,0.712924991854903,0.0026551687292387977,0.04686958768631601
2.1105527638190957,0.544011767148791,0.7414449382314499,0.6860908894250208,0.6595598656119689,0.7094451677346187,0.02966127523742708,0.039651092346787244
2.1608040201005028,0.4989536226946192,0.7321986294826655,0.6284932321872049,0.6547520704072156,0.7060065293173672,0.0028509515762108976,0.11988387173616225
2.21105527638191,0.46734845101030037,0.7228081028160838,0.5703964645669753,0.6500201111040873,0.7026076479542953,0.04911894298411106,0.061159444347240674
2.261306532663317,0.6482786392103428,0.7132800229094233,0.5281088965394642,0.6453616324061207,0.6992471757297487,0.11193469648515511,0.012701128352875186
2.311557788944724,0.4574703999444425,0.7036215721456343,0.5083209896939455,0.6407743870665956,0.6959238392149281,0.026104007393431085,0.023006874815430588
2.361809045226131,0.4870642251680577,0.6938404307755529,0.5636453980809554,0.6362562293788792,0.6926364338295035,0.05514027102087267,0.0856938249277266
2.412060301507538,0.47525485479777824,0.6839447558406517,0.6071918183722557,0.6318051091497389,0.6893838187403922,0.07848394880097831,0.01412785870436662
2.462311557788945,0.5162832426556985,0.6739431589089295,0.5672747806904705,0.6274190661132502,0.6861649122364457,0.010663149163452028,0.15975395450357213
2.512562814070352,0.5362033168845872,0.6638446826799806,0.5585194987286081,0.6230962247472047,0.6829786875258792,0.0007421940425200996,0.0678829320011391
2.562814070351759,0.4446609646408357,0.653658776518142,0.505363227773571,0.6188347894577095,0.679824168910159,0.011646308921456579,0.08052805201355576
2.613065326633166,0.3967517944632962,0.6433952709753217,0.5446184095805382,0.6146330401010338,0.6767004282939326,0.03926773865213555,0.10243546773621204
2.663316582914573,0.4453262267350198,0.6330643513676673,0.45840264679163284,0.6104893278147503,0.6736065819956185,0.03330235902012565,0.018339091003952725
2.71356783919598,0.4639069473980026,0.622676530472621,0.5384108296145927,0.6064020711328872,0.6705417878275923,0.03807350297479813,0.08268792573579528
2.763819095477387,0.39093752613578997,0.6122426204151328,0.468521905699049,0.6023697523621818,0.6675052424186277,0.04434348008144762,0.0895041674109679
2.814070351758794,0.3179502775722342,0.6017737038138494,0.3969056421363847,0.5983909141986588,0.6644961787544712,0.04314835997558304,0.10666392433750473
2.864321608040201,0.4674092807739292,0.5912811042599684,0.37005219448754506,0.5944641565656545,0.6615138639152143,0.030937910473723036,0.06969239328063026
2.9145728643216082,0.4215066963899487,0.5807763562031315,0.42950304946247514,0.5905881336561232,0.6585575969905548,0.03569399362320074,0.031634860792342866
2.9648241206030153,0.4540571836379348,0.5702711743202334,0.42476361819746467,0.5867615511635855,0.6556267071561435,0.009143041170309011,0.09422765118513002
3.0150753768844223,0.44406100726954606,0.5597774224443233,0.5492355372968523,0.5829831636874654,0.6527205518960604,0.004701239804278709,0.09837561050775107
3.0653266331658293,0.3978729253551041,0.549307082131894,0.46098827297463474,0.5792517722998003,0.6498385153580806,0.01829106852297188,0.04486852072080594
3.1155778894472363,0.3123439808132314,0.5388722209477665,0.42555121449142597,0.5755662222614268,0.646980006829798,0.0029814962420955313,0.05544272815240161
3.1658291457286434,0.4136913548114169,0.5284849605474892,0.3971401151257297,0.5719254008767591,0.6441444593249309,0.1036824897286617,0.06351809004292508
3.2160804020100504,0.3622128288987594,0.5181574446376889,0.4213407278739692,0.5683282354771859,0.6413313282702238,0.023428969507356037,0.017621788893376487
3.2663316582914574,0.38687713731430484,0.5079018068951161,0.4392732333292392,0.5647736915239435,0.6385400902843328,0.07966860227064582,0.08597084581204291
3.3165829145728645,0.33707322544956964,0.4977301389252352,0.3514831747174872,0.5612607708220712,0.6357702420409412,0.024528942189122174,0.022380515188034012
3.3668341708542715,0.29259697050432243,0.4876544583411095,0.4908804875537351,0.5577885098377301,0.6330212992091082,0.07180284597765092,0.08049472699512393
3.4170854271356785,0.33627620466163877,0.4776866770430317,0.4722267452681855,0.5543559781117946,0.6302927954645363,0.03600428224357248,0.01901126006021853
3.4673366834170856,0.3829703196150572,0.4678385697788435,0.3971675107477449,0.5509622767631763,0.6275842815660388,0.026089580573919892,0.05654053696926411
3.5175879396984926,0.372781498642361,0.4581217430641791,0.37658576526896853,0.5476065370758616,0.6248953244920297,0.0259116378083547,0.056866009573258214
3.5678391959798996,0.3784355339392126,0.44854760454096304,0.45726398883615066,0.5442879191641041,0.622225506632335,0.01480099829482799,0.007643601881924323
3.6180904522613067,0.3003692108101906,0.43912733285138733,0.36426107657724166,0.5410056107106416,0.6195744250310551,0.034036960725786436,0.00013534045748443875
3.6683417085427137,0.25450406729454206,0.4298718481032887,0.3456139307852943,0.5377588257731962,0.6169416906765909,0.06555219641735577,0.136108093789273
3.7185929648241207,0.3141400894960919,0.4207917830013622,0.39882875405334717,0.5345468036548688,0.614326927835291,0.028545997008118152,0.039110564776871776
3.7688442211055277,0.33614323516546585,0.4118974547169604,0.355513312392413,0.5313688078343695,0.6117297734254903,0.0396502309013602,0.0735298605227585
3.819095477386935,0.3353840966628386,0.4031988375673689,0.36692416640044584,0.5282241249523163,0.6091498764289853,0.07309219984687042,0.03545090719218082
3.869346733668342,0.35086170773957326,0.39470553657340585,0.36076387145690025,0.5251120638501179,0.6065868973372472,0.03350170509147917,0.1381724386609329
3.919597989949749,0.33991745147078467,0.38642676196197334,0.4149817606837244,0.5220319546581946,0.6040405076298966,0.052257241188312736,0.17303937845990178
3.969849246231156,0.3140027517341931,0.3783713046778105,0.3516995482934533,0.518983147930534,0.6015103892831757,0.010172407946542401,0.06580338653470225
4.020100502512563,0.33215354791386353,0.3705475129661445,0.3929891642814765,0.5159650138227816,0.5989962343063315,0.08613016677101318,0.0762224588418334
4.07035175879397,0.2965353651041895,0.36296327008524004,0.3681328324511959,0.512976941311265,0.5964977443040016,0.057628268373450114,0.008475885621904792
4.120603015075377,0.3049306329641761,0.35562597320498934,0.3476330049188865,0.5100183374505303,0.594014630062837,0.07791920118909124,0.019054421428699533
4.170854271356784,0.26229464289392634,0.34854251354469934,0.38801537360529464,0.5070886266671328,0.5915466111607453,0.024473488844794757,0.04297547330096282
4.2211055276381915,0.23785858032658297,0.3417192578000988,0.4139116209040377,0.5041872500875795,0.589093415597258,0.04148364795655554,0.04102052718786601
4.2713567839195985,0.2596441290120736,0.3351620309063378,0.29454591384443113,0.5013136648984586,0.5866547794436423,0.030040016902486588,0.029894889349171158
4.3216080402010055,0.30457828467005915,0.32887610018038005,0.37143344198311007,0.4984673437369257,0.5842304465114836,0.01853300976352635,0.13031620507734348
4.371859296482413,0.31778682830971516,0.32286616088270637,0.3557869648781942,0.4956477741098342,0.581820168038562,0.011349994723150158,0.05376097391915419
4.42211055276382,0.28948821383741535,0.3171363232346659,0.3316717197972997,0.49285445783990733,0.5794237023909303,0.012117431259037659,0.16261513079955783
4.472361809045227,0.20967544398016993,0.31169010092413785,0.30428334195416845,0.49008691053745956,0.5770408147801858,0.04751608953339149,0.029351527740602696
4.522613065326634,0.300069275981074,0.3065304011284101,0.31804242793974097,0.4873446610962613,0.5746712769949984,0.0196942869218072,0.09291596478444848
4.572864321608041,0.2541782622987943,0.3016595160793521,0.35437976636863666,0.4846272512122376,0.5723148671460256,0.011830700564348116,0.08311361633778733
4.623115577889448,0.2524274605153138,0.2970791161920659,0.32127356065656715,0.4819342349237702,0.5699713694234093,0.021645392268469975,0.03310518541283996
4.673366834170855,0.14444658243991654,0.2927902447742539,0.2833650605642774,0.47926517817244907,0.5676405738661012,0.02192667800648812,0.15404878148302525
4.723618090452262,0.17092354633571957,0.2887933143295523,0.3452810770475188,0.4766196583831931,0.5653222761423236,0.003922130078184631,0.062325664238841054
4.773869346733669,0.20840608804176824,0.28508810446405464,0.33359215738381665,0.473997264062721,0.5630162773405092,0.058836401843328515,0.0027180217636243527
4.824120603015076,0.1774971542130949,0.2816737614012061,0.30306038313124356,0.47139759441542217,0.560722383770123,0.04960370777115906,0.05365508283567286
4.874371859296483,0.2004230179220044,0.27854879910618635,0.3008133507313949,0.4688202589757272,0.5584404067717934,0.09740882204743354,0.039823433833792284
4.92462311557789,0.23960070254455418,0.27571110201683924,0.31537204989585743,0.4662648772561365,0.556170162536231,0.10025281949356,0.03592416185237296
4.974874371859297,0.21659054688268906,0.27315792937415173,0.3189430897314952,0.4637310784101132,0.5539114719314429,0.034230172708657754,0.05792632597379673
5.025125628140704,0.22247092593703602,0.2708859211412455,0.3125036896893103,0.46121850090909344,0.5516641603377797,0.0477374475559193,0.02415763993370682
5.075376884422111,0.23325492919025875,0.2688911054958388,0.31738225565174816,0.4587267922329099,0.5494280574903887,0.03703606722925771,0.04000556686621442
5.125628140703518,0.17307064389164023,0.2671689078771609,0.26068594041757837,0.4562556085729663,0.5472029973286696,0.007647423850988876,0.09279290138560206
5.175879396984925,0.25291637457677235,0.265714161564381,0.2689722368175585,0.4538046145475373,0.5449888178523554,0.04880383212000101,0.043019481709809765
5.226130653266332,0.24472180410895006,0.26452111975974524,0.2713041759479546,0.451373482928605,0.5427853609838664,0.05926195302899706,0.07163963546194728
5.276381909547739,0.12157671029525455,0.2635834691458207,0.2701578028049197,0.4489618943796744,0.5405924724366071,0.0024723390953717023,0.013930211866198326
5.326633165829146,0.17410848656926237,0.2628943448825214,0.3045040550814903,0.4465695372040436,0.5384100015888929,0.019634178115846605,0.02820506501106623
5.376884422110553,0.15570682310716694,0.26244634700595926,0.24331160278806646,0.4441961071030315,0.5362378013632179,0.006500469028884761,0.19116114565042883
5.42713567839196,0.1952069955780907,0.26223155818762484,0.2724902599826162,0.44184130694369494,0.5340757281105883,0.0485093964322103,0.02878516843931022
5.477386934673367,0.2114298876665932,0.262241562808968,0.27281258529062624,0.4395048465355915,0.5319236414996644,0.06342102156181435,0.044751062303240836
5.527638190954774,0.1891031353773233,0.2624674673031302,0.30867340033670937,0.4371864424161679,0.529781404410469,0.006957383334758637,0.006120183394719674
5.577889447236181,0.16934772574747295,0.26289992171238,0.25592701281722685,0.4348858176443777,0.5276488828324344,0.03783032643952941,0.09533568193667905
5.628140703517588,0.17073313480441923,0.26352914240673775,0.26524143863470045,0.4326027016021533,0.5255259457665746,0.005563380000111119,0.12855408281481295
5.678391959798995,0.22151050225218494,0.26434493590634117,0.264141103676647,0.4303368298033742,0.5234124651315775,0.004473609638956794,0.054001863795044604
5.728643216080402,0.08477167750309866,0.26533672374732165,0.2306173824961552,0.42808794370999603,0.521308315673631,0.08178550098330555,0.0009819810848300334
5.778894472361809,0.22543650371592328,0.26649356832832466,0.21168088274097815,0.425855790555021,0.5192133748797993,0.0023443964874747644,0.07278896266501898
5.8291457286432165,0.22161197049789091,0.26780419967233404,0.22088882876419233,0.4236401231720046,0.5171275228947827,0.03646199197604743,0.06032430400409404
5.8793969849246235,0.160780314903849,0.2692570430361494,0.22707863319843538,0.4214406998308131,0.5150506424408989,0.011372308420656618,0.020009215224109905
5.9296482412060305,0.17745203081624394,0.27084024729772754,0.2040931342456095,0.41925728407935714,0.5129826187411365,0.005887302818904104,0.06960686858421383
5.9798994974874375,0.21477491851264965,0.27254171404963434,0.22138178465338645,0.4170896445910446,0.5109233394451362,0.05064632153640321,0.01005877673706175
6.030150753768845,0.14837743573668233,0.2743491273250741,0.18647831661939243,0.41493755501770335,0.5088726945579651,0.06310099611063544,0.10042813978132314
6.080402010050252,0.16801450424542635,0.2762499838813636,0.22596247693533855,0.4128007938477437,0.5068305763715575,0.021511117836206812,0.05153840749979935
6.130653266331659,0.13777348449639915,0.2782316239643159,0.20341334503353517,0.41067914426933494,0.5047968793986993,0.09248120820840905,0.10477499866396159
6.180904522613066,0.1403811812510717,0.28028126247578106,0.21041705720371634,0.4085723940383875,0.5027715003094426,0.00747858476684247,0.07165448029699609
6.231155778894473,0.12733188658269307,0.28238602046557865,0.20561130962473423,0.40648033535113737,0.5007543378698429,0.037386021406691554,0.002240403454562261
6.28140703517588,0.1936814956883173,0.2845329568682336,0.22125245818419803,0.404402764721143,0.49874529288291447,0.02754855871068998,0.01259688754109908
6.331658291457287,0.17348779376915105,0.28670910040431535,0.23798111883885903,0.402339482860512,0.4967442681317068,0.006763474069997318,0.1281128791781081
6.381909547738694,0.13219143836078306,0.2889014815657597,0.22565532639903157,0.4002902945651846,0.4947511683244109,0.036505841104217854,0.10429908962789783
6.432160804020101,0.17778893486628533,0.29109716460434626,0.17870833149081863,0.3982550086041078,0.49276590004140686,0.02924842888924627,0.01462788465725484
6.482412060301508,0.08792434037357544,0.293283279442495,0.20307821193739423,0.396233437612145,0.49078837168416867,0.034816175206325725,0.05303654868635464
6.532663316582915,0.1471710383851964,0.2954470534257411,0.21562063643830992,0.3942253979865684,0.48881849342594885,0.06038116304388228,0.06373463516962363
6.582914572864322,0.17116846117164233,0.29757584283665206,0.2036939409788639,0.39223070978699304,0.48685617716416685,0.13733031632783108,0.0870088958070863
6.633165829145729,0.23491713309498585,0.29965716409054766,0.18540163710690077,0.3902491966386159,0.48490133647442923,0.03842921833542416,0.01748850643909971
6.683417085427136,0.024243474136171117,0.30167872453419403,0.19976382392197217,0.38828068563862905,0.4829538865661155,0.042398298012925495,0.16857267721502328
6.733668341708543,0.14783034506529852,0.30362845276964223,0.1853508320672203,0.386325007265684,0.48101374423946275,0.08238040682089218,0.059584759865364774
6.78391959798995,0.07006915236845562,0.3054945284265804,0.2002631972936741,0.3843819952922871,0.47908082784409056,0.051363848542672,0.006784769410566944
6.834170854271357,0.05556086658742769,0.3072654113079672,0.19360277334083875,0.3824514867000147,0.4771550572389035,0.02988536129517095,0.023776580165338904
6.884422110552764,0.0886837545904657,0.3089298698352891,0.18275210309973866,0.380533321597438,0.4752363537533201,0.006775341009002768,0.0005253055714815856
6.934673366834171,0.1065797705370216,0.3104770087215607,0.20050261562994845,0.37862734314065516,0.47332464014977227,0.035686570172060596,0.008147068991130048
6.984924623115578,0.07962439686728003,0.3118962958021328,0.16052978259077988,0.37673339745633294,0.47141984058742503,0.03632268077210549,0.22944496512503734
7.035175879396985,0.11805702350518604,0.31317758795550077,0.16128496142369655,0.3748513335671614,0.46952188058706945,0.0015733564216703591,0.09516334097537861
7.085427135678392,0.13978684745525308,0.3143111560486062,0.1725949204633635,0.3729810033196328,0.4676306869971413,0.02344007718587842,0.1500673743764607
7.135678391959799,0.0721655153928557,0.3152877088435851,0.13709682077878144,0.37112226131405723,0.46574618796082323,0.00044489431944467404,0.02477925249599344
7.185929648241206,0.12543373159857096,0.3160984158055401,0.1525599394066463,0.36927496483673417,0.46386831288418673,0.031845361425056486,0.024609909314467503
7.236180904522613,0.1733831323554618,0.31673492875369025,0.1790327040750901,0.3674389737941989,0.46199699240533554,0.08414718823423192,0.16644308086108525
7.28643216080402,0.08791264090495227,0.31718940230117076,0.13341672238500016,0.365614150649468,0.4601321583645116,0.04508150685815779,0.03802798778299485
7.336683417085427,0.06774999906489262,0.3174545130318157,0.15806515450904035,0.36380036036021435,0.45827374377512664,0.05023723780342812,0.01610581823645577
7.386934673366834,0.16889558452973397,0.31752347736544395,0.16632990156288324,0.36199747031879714,0.4564216827956864,0.02527605654731682,0.12116636496527308
7.437185929648241,0.09381812385689507,0.31739006806648334,0.14602821153474466,0.3602053502940855,0.4545759107025711,0.011074046951096282,0.03912275916150316
7.4874371859296485,0.1307907395272664,0.3170486293541907,0.16703593258256932,0.3584238723750076,0.4527363638636439,0.06285683574911392,0.03374970186770117
7.5376884422110555,0.07131004144525588,0.3164940905762582,0.14915332269901235,0.3566529109157681,0.4509029797126539,0.054052800107380006,0.06198912360909445
7.5879396984924625,0.11063686298700051,0.3157219784112243,0.15086506545823547,0.35489234248267054,0.44907569672440617,0.01199093642708893,0.16846291671918928
7.63819095477387,0.13309518349040472,0.31472842756882075,0.14218505233024045,0.353142045802493,0.447254454390671,0.02287616584282282,0.04908721258273215
7.688442211055277,0.10862434322493682,0.3135101899611825,0.15719520312878102,0.3514019017123605,0.4454391931968045,0.0824351974190266,0.12728808652690557
7.738693467336684,0.06301193649995501,0.31206464232170517,0.12358266849837081,0.3496717931110629,0.44362985459905613,0.024793300247563953,0.07562227478500767
7.788944723618091,0.0793438636810195,0.31038979225225805,0.13449243890194315,0.34795160491176713,0.44182638100253824,0.055179978237872164,0.032914764299220564
7.839195979899498,0.0849296043928592,0.30848428268342654,0.15588060553089955,0.34624122399608026,0.4400287157398337,0.016821685346385687,0.0026206303456452746
7.889447236180905,0.04925961712491072,0.3063473947364659,0.15587835767963437,0.3445405391694122,0.43823680305022095,0.05342932299829024,0.041154915263012945
7.939698492462312,0.07968496741689288,0.30397904897968375,0.11387443606199414,0.3428494411175972,0.4364505880594919,0.052795959722532995,0.08720032511200959
7.989949748743719,0.0220252890192532,0.3013798050760236,0.13715469154278478,0.341167822364732,0.4346700167603449,0.016244031298854194,0.015984710044387347
8.040201005025127,0.08050502863992404,0.29855085982268237,0.13118594331216868,0.3394955772321878,0.43289503599333123,0.008345428945330371,0.11238794130542833
8.090452261306533,0.17194582224060037,0.2954940435876572,0.12516106851963876,0.3378326017987593,0.43112559342833734,0.017167845316208536,0.061116240643410626
8.14070351758794,0.08627274394196591,0.29221181515216127,0.13000463426373268,0.33617879386191096,0.42936163754658274,0.07100258789829059,0.048520412740302346
8.190954773869347,0.12356436558502672,0.28870725497187816,0.14515442627338318,0.3345340529000881,0.4276031176231182,0.010847796794511767,0.0842278159461919
8.241206030150755,0.0714549080255366,0.28498405687401146,0.12395647126561667,0.3328982800360514,0.42584998370980587,0.021367792351754077,0.08690419647559329
8.291457286432161,0.1834146433413999,0.28104651821104126,0.10892574658212852,0.3312713780012103,0.4241021866187662,0.034646571192497214,0.06963275384526206
8.341708542713569,0.045460797529358186,0.2768995284959905,0.14002759672938125,0.3296532511009145,0.4223596779062746,0.07462136172360107,0.13646149232524818
8.391959798994975,-0.006668511677120723,0.2725485565478443,0.11170328614020199,0.3280438051806789,0.42062240985709587,0.08738037259257703,0.014318089545088244
8.442211055276383,0.08804892712128562,0.26799963617952094,0.13142941827759289,0.32644294759330805,0.4188903354692375,0.09896949695975771,0.011703954561204426
8.492462311557789,0.08053227478076344,0.2632593504644793,0.13761089038976085,0.32485058716689486,0.4171634084391136,0.05856486168080475,0.02807276172942578
8.542713567839197,0.042093052940108074,0.258334814621631,0.12255278357932842,0.32326663417366197,0.4154415831471008,0.016226430572385618,0.0421562050160143
8.592964824120603,0.13876962179540214,0.2532336575617206,0.12194143719037313,0.3216910002996235,0.4137248146434781,0.08904428253692065,0.05108237174677233
8.643216080402011,0.09338068285797815,0.247964002141708,0.12112642875882852,0.3201235986150377,0.41201305863473425,0.07172828673809409,0.03953793484688261
8.693467336683417,0.005275241247451809,0.24253444417696152,0.12299171840239556,0.318564343545629,0.4103062714702346,0.036971950960068854,0.12350568991740142
8.743718592964825,0.027348314720468128,0.2369540302641935,0.10433494601050525,0.31701315084455206,0.40860441012923243,0.0947802680294259,0.05417590165846841
8.793969849246231,0.037350484196303164,0.2312322344710864,0.10112941403279027,0.31546993756507946,0.4069074322082181,4.740441905731953e-06,0.015787412268143907
8.84422110552764,0.11535642510219385,0.22537893395140587,0.11849098767847487,0.3139346220339855,0.4052152959085905,0.058651700734734295,0.03604839290012779
8.894472361809045,0.06737164867792983,0.2194043835471242,0.10504675797304722,0.31240712382561,0.40352796002464475,0.023243745420766102,0.1111063038761658
8.944723618090453,0.12016707059108844,0.2133191894416211,0.09898309454406383,0.310887363736578,0.40184538393186486,0.008202155589820575,0.04963149412524555
8.99497487437186,0.07589763647509859,0.20713428193043998,0.11166532493629253,0.3093752637611561,0.40016752757551177,0.0127750403147877,0.07133785467069191
9.045226130653267,0.07192738314952522,0.20086088737829086,0.09390828646874448,0.3078707470672276,0.3984943514594973,0.039970120605574616,0.07895086772001304
9.095477386934673,-0.003484133763075284,0.19451049943306112,0.113095049052314,0.3063737379728674,0.39682581663553684,0.052435924713293405,0.0603233130986561
9.145728643216081,0.12977474579607773,0.18809484956945682,0.12909670842977133,0.30488416192349643,0.39516188469257063,0.008388738332320848,0.03760696638278936
9.195979899497488,0.11294415562759114,0.18162587703660477,0.0842627788205477,0.3034019454696033,0.3935025177464464,0.0025392455038834006,0.09782792455333471
9.246231155778895,0.1634431312113413,0.17511569828543594,0.10130036327196984,0.30192701624501117,0.391847678429856,0.02634619903711574,0.00532314874384173
9.296482412060302,0.08554047029252905,0.16857657595299985,0.11542478622946663,0.3004593029456778,0.39019732988251676,0.09861980410166339,0.05796065512924803
9.34673366834171,0.022079141283382953,0.16202088748196328,0.0933363109010238,0.298998735309011,0.388551435741592,0.03248057559132277,0.0494389015459046
9.396984924623116,0.011909915822690825,0.1554610934544833,0.08892198912933562,0.2975452440936849,0.38690996013234347,0.02572305374875209,0.09307819483467214
9.447236180904524,0.10897483698203239,0.14890970572035084,0.09245961134203215,0.2960987610599438,0.3852728676590068,0.021452530222302868,0.056223958334962985
9.49748743718593,-0.015926672122622852,0.14237925539983665,0.09937765919965845,0.29465921895037794,0.38364012339588704,0.09614650520342814,0.050042130664608275
9.547738693467338,0.071963907134743,0.13588226084197189,0.08912179259987506,0.2932265514711577,0.382011692878665,0.00853311906406967,0.021645213684255763
9.597989949748744,0.0567778961690902,0.12943119561912533,0.09693028825976016,0.2918006932737153,0.3803875420959083,0.022244452191961566,0.05678168256241234
9.648241206030152,0.006763773479069299,0.1230384566386307,0.07893294511391093,0.290381579936858,0.37876763748078446,0.019598683265174197,0.10294073898284255
9.698492462311558,-0.025601148572716857,0.11671633245193981,0.09212924503203447,0.28896914794930395,0.3771519459029655,0.034994032261565534,0.04154804031096798
9.748743718592966,0.11248127925329127,0.11047697184126248,0.11016868146930495,0.28756333469262607,0.37554043466072307,0.036643879409956,0.0408076441111178
9.798994974874372,0.12653788966236273,0.10433235276296898,0.08891550856072965,0.28616407842459457,0.3739330714732051,0.07886836761391981,0.07754916891946767
9.84924623115578,0.08236896150915565,0.09829425172611685,0.10858174528582236,0.28477131826290547,0.3723298244728914,0.020083854484003016,0.006101729606710677
9.899497487437186,0.030565710168135037,0.09237421368337875,0.06831959777731408,0.2833849941692854,0.3707306621982225,0.08823145072169943,0.08413811222277685
9.949748743718594,0.06910478628433699,0.08658352251034154,0.08305687633688313,0.2820050469339618,0.3691355535863954,0.003160238004895635,0.0890686560344958
10.0,0.12994086212047606,0.08093317214767573,0.07427988631504226,0.28063141816048887,0.367544467966324,0.044903461097547934,0.03585440361794555
'''

def init_project(project_name: str):
    project_path = Path(project_name)
    if project_path.exists():
        print(f"❌ 文件夹 {project_name} 已存在。")
        return

    # 创建目录
    (project_path / 'data').mkdir(parents=True)
    
    # 写入模板文件
    (project_path / 'plot.py').write_text(TEMPLATE_CODE, encoding='utf-8')
    (project_path / 'data' / 'example.csv').write_text(EXAMPLE_CSV, encoding='utf-8')

    print(f"✅ 成功创建绘图项目：{project_name}")

def main():
    parser = argparse.ArgumentParser(description='boviz CLI 工具')
    parser.add_argument('command', choices=['init'], help='可用命令：init')
    parser.add_argument('project_name', help='要创建的项目名称')

    args = parser.parse_args()

    if args.command == 'init':
        init_project(args.project_name)

if __name__ == '__main__':
    main()
