"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Parcel code
 */
class Bundling {
    static parcel(options) {
        var _a;
        // Original package.json path and content
        let pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        pkgPath = path.join(pkgPath, 'package.json');
        const originalPkg = fs.readFileSync(pkgPath);
        const originalPkgJson = JSON.parse(originalPkg.toString());
        // Update engines.node in package.json to set the right Babel target
        setEngines(options.nodeVersion, pkgPath, originalPkgJson);
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(options.projectRoot, path.resolve(options.entry)));
        try {
            const command = [
                'parcel', 'build', containerEntryPath.replace(/\\/g, '/'),
                '--out-dir', cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
                '--out-file', 'index.js',
                '--global', options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !options.minify && '--no-minify',
                !options.sourceMaps && '--no-source-maps',
                ...(options.cacheDir ? ['--cache-dir', '/parcel-cache'] : []),
            ].filter(Boolean);
            return lambda.Code.fromAsset(options.projectRoot, {
                assetHashType: cdk.AssetHashType.BUNDLE,
                bundling: {
                    image: cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel-bundler'), {
                        buildArgs: {
                            NODE_TAG: (_a = options.nodeDockerTag) !== null && _a !== void 0 ? _a : `${process.versions.node}-alpine`,
                        },
                    }),
                    environment: options.environment,
                    volumes: options.cacheDir
                        ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                        : [],
                    workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
                    command,
                },
            });
        }
        finally {
            restorePkg(pkgPath, originalPkg);
        }
    }
}
exports.Bundling = Bundling;
function setEngines(nodeVersion, pkgPath, originalPkgJson) {
    // Update engines.node (Babel target)
    const updateData = {
        engines: {
            node: `>= ${nodeVersion}`,
        },
    };
    // Write new package.json
    if (Object.keys(updateData).length !== 0) {
        fs.writeFileSync(pkgPath, JSON.stringify({
            ...originalPkgJson,
            ...updateData,
        }, null, 2));
    }
}
function restorePkg(pkgPath, originalPkg) {
    fs.writeFileSync(pkgPath, originalPkg);
}
//# sourceMappingURL=data:application/json;base64,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