"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    static bundle(options) {
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(options.solutionDir, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash
                ? cdk.AssetHashType.CUSTOM
                : cdk.AssetHashType.SOURCE,
            exclude: ['**/bin/', '**/obj/'],
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    constructor(props) {
        this.props = props;
        Bundling.runsLocally = Bundling.runsLocally ?? (0, util_1.getDotNetLambdaTools)();
        const { solutionDir, projectDir } = props;
        this.relativeProjectPath = path.relative(path.resolve(solutionDir), path.resolve(projectDir));
        this.relativeProjectPath =
            this.relativeProjectPath === '' ? '.' : this.relativeProjectPath;
        this.msbuildParameters = props.msbuildParameters ?? [];
        if (props.runtime.family === aws_lambda_1.RuntimeFamily.OTHER) {
            this.msbuildParameters.push('--self-contained', '/p:AssemblyName=bootstrap');
        }
        const environment = {
            ...props.environment,
        };
        // Docker Bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? props.dockerImage ??
                cdk.DockerImage.fromRegistry(Bundling.defaultBuildImage)
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR, props.architecture);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) {
            // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(solutionDir, outputDir, props.architecture, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('dotnet build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [osPlatform === 'win32' ? '/c' : '-c', localCommand], {
                        env: { ...process.env, ...(environment ?? {}) },
                        stdio: [
                            // show output
                            'ignore', // ignore stdio
                            process.stderr, // redirect stdout to stderr
                            'inherit', // inherit stderr
                        ],
                        cwd: props.solutionDir,
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    createBundlingCommand(inputDir, outputDir, architecture, osPlatform = 'linux') {
        const pathJoin = osPathJoin(osPlatform);
        const projectLocation = this.relativeProjectPath.replace(/\\/g, '/');
        const packageFile = pathJoin(outputDir, 'package.zip');
        const dotnetPackageCommand = [
            'dotnet',
            'lambda',
            'package',
            '--project-location',
            projectLocation,
            '-farch',
            architecture.name,
            '--output-package',
            packageFile,
            this.msbuildParameters.length > 0
                ? `--msbuild-parameters "${this.msbuildParameters.join(' ')}"`
                : '',
        ]
            .filter((c) => !!c)
            .join(' ');
        const unzipCommand = osPlatform === 'win32'
            ? [
                'powershell',
                '-command',
                'Expand-Archive',
                packageFile,
                outputDir,
            ].join(' ')
            : ['unzip', '-od', outputDir, packageFile].filter((c) => !!c).join(' ');
        const deleteCommand = osPlatform === 'win32'
            ? ['powershell', '-command', 'Remove-Item', packageFile]
                .filter((c) => !!c)
                .join(' ')
            : ['rm', packageFile].filter((c) => !!c).join(' ');
        return chain([
            ...(this.props.commandHooks?.beforeBundling(inputDir, outputDir) ?? []),
            dotnetPackageCommand,
            unzipCommand,
            deleteCommand,
            ...(this.props.commandHooks?.afterBundling(inputDir, outputDir) ?? []),
        ]);
    }
}
exports.Bundling = Bundling;
Bundling.defaultBuildImage = 'public.ecr.aws/sam/build-dotnet7';
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter((c) => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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