from typing import Dict, Optional, Union

from pydantic import ConfigDict, Field, HttpUrl
from pydantic.dataclasses import dataclass

from ltiaas.resources.deep_linking.enums import ContentItemType
from ltiaas.utils.serialization import Submittable, Serializable


@dataclass
class Icon(Submittable):
    url: HttpUrl = Field()
    width: int = Field()
    height: int = Field()


@dataclass
class Thumbnail(Submittable):
    url: HttpUrl = Field()
    width: int = Field()
    height: int = Field()


@dataclass
class Available(Submittable):
    start_date_time: Optional[str] = None
    end_date_time: Optional[str] = None


@dataclass
class Submission(Submittable):
    start_date_time: Optional[str] = None
    end_date_time: Optional[str] = None


@dataclass
class Iframe(Submittable):
    width: Optional[int] = None
    height: Optional[int] = None


@dataclass
class Window(Submittable):
    target_name: Optional[str] = None
    width: Optional[int] = None
    height: Optional[int] = None
    window_features: Optional[str] = None


@dataclass(config=ConfigDict(extra="allow"))
class LTIResourceLinkContentItem(Submittable):
    type: str = ContentItemType.LTI_RESOURCE_LINK
    url: Optional[HttpUrl] = None
    title: Optional[str] = None
    text: Optional[str] = None
    icon: Optional[Icon] = None
    thumbnail: Optional[Thumbnail] = None
    available: Optional[Available] = None
    submission: Optional[Submission] = None
    iframe: Optional[Iframe] = None
    window: Optional[Window] = None
    custom: Optional[Dict[str, Union[str, int]]] = None


@dataclass(config=ConfigDict(extra="allow"))
class LinkContentItem(Submittable):
    url: HttpUrl = Field()
    type: str = ContentItemType.LINK
    title: Optional[str] = None
    text: Optional[str] = None
    icon: Optional[Icon] = None
    thumbnail: Optional[Thumbnail] = None
    embed: Optional[dict] = None
    iframe: Optional[Iframe] = None
    window: Optional[Window] = None


@dataclass(config=ConfigDict(extra="allow"))
class FileContentItem(Submittable):
    url: HttpUrl = Field()
    type: str = ContentItemType.FILE
    title: Optional[str] = None
    text: Optional[str] = None
    icon: Optional[Icon] = None
    thumbnail: Optional[Thumbnail] = None
    expires_at: Optional[str] = None


@dataclass(config=ConfigDict(extra="allow"))
class ImageContentItem(Submittable):
    url: HttpUrl = Field()
    type: str = ContentItemType.IMAGE
    title: Optional[str] = None
    text: Optional[str] = None
    icon: Optional[Icon] = None
    thumbnail: Optional[Thumbnail] = None
    width: Optional[int] = None
    height: Optional[int] = None


@dataclass(config=ConfigDict(extra="allow"))
class HTMLFragmentContentItem(Submittable):
    html: str = Field()
    type: str = ContentItemType.HTML_FRAGMENT
    title: Optional[str] = None
    text: Optional[str] = None


ContentItem = Union[
    LTIResourceLinkContentItem,
    LinkContentItem,
    FileContentItem,
    ImageContentItem,
    HTMLFragmentContentItem,
]


@dataclass
class DeepLinkingForm(Serializable):
    form: str = Field()


@dataclass
class DeepLinkingFormComponents(Serializable):
    message: str = Field()
    target: str = Field()