"use strict";

/**
 * A view which gives the user hints about comments in other revisions.
 */
RB.DiffCommentsHintView = Backbone.View.extend({
  events: {
    'click .revision': '_onRevisionSelected',
    'click .interdiff': '_onInterdiffSelected'
  },
  template: _.template("<div class=\"box-container\">\n <div class=\"box important\">\n  <div class=\"box-inner comments-hint\">\n   <h1><%- unpublishedCommentsHeader %></h1>\n   <p><%- unpublishedCommentsText %></p>\n   <ul>\n   </ul>\n  </div>\n </div>\n</div>"),

  /**
   * Initialize the view.
   */
  initialize: function initialize() {
    this.listenTo(this.model, 'change', this.render);
  },

  /**
   * Render the view.
   *
   * Returns:
   *     RB.DiffCommentsHintView:
   *     This object, for chaining.
   */
  render: function render() {
    var revisionText = gettext("Revision %s");
    var interdiffText = gettext("Interdiff revision %(oldRevision)s - %(newRevision)s");

    if (this.model.get('hasOtherComments')) {
      this.$el.html(this.template({
        unpublishedCommentsHeader: gettext("You have unpublished comments on other revisions"),
        unpublishedCommentsText: gettext("Your review consists of comments on the following revisions:")
      }));
      var $ul = this.$('ul');
      this.model.get('diffsetsWithComments').forEach(function (diffset) {
        var $li = $('<li/>').addClass('revision').data('revision', diffset.revision).text(interpolate(revisionText, [diffset.revision])).appendTo($ul);

        if (diffset.isCurrent) {
          $li.addClass('current');
        }
      });
      this.model.get('interdiffsWithComments').forEach(function (interdiff) {
        var $li = $('<li/>').addClass('interdiff').data({
          'first-revision': interdiff.oldRevision,
          'second-revision': interdiff.newRevision
        }).text(interpolate(interdiffText, {
          oldRevision: interdiff.oldRevision,
          newRevision: interdiff.newRevision
        }, true)).appendTo($ul);

        if (interdiff.isCurrent) {
          $li.addClass('current');
        }
      });
    } else {
      this.$el.empty();
    }

    return this;
  },

  /**
   * Callback for when a single revision is selected.
   *
   * Args:
   *     ev (Event):
   *         The click event.
   */
  _onRevisionSelected: function _onRevisionSelected(ev) {
    var $target = $(ev.currentTarget);

    if (!$target.hasClass('current')) {
      this.trigger('revisionSelected', [0, $target.data('revision')]);
    }
  },

  /**
   * Callback for when an interdiff is selected.
   *
   * Args:
   *     ev (Event):
   *         The click event.
   */
  _onInterdiffSelected: function _onInterdiffSelected(ev) {
    var $target = $(ev.currentTarget);

    if (!$target.hasClass('current')) {
      this.trigger('revisionSelected', [$target.data('first-revision'), $target.data('second-revision')]);
    }
  }
});

//# sourceMappingURL=diffCommentsHintView.js.map