# -*- coding: utf-8 -*-
# Copyright 2025 Google LLC
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import dataclasses
import json  # type: ignore
import logging
from typing import Any, Callable, Dict, List, Optional, Sequence, Tuple, Union
import warnings

from google.api_core import exceptions as core_exceptions
from google.api_core import gapic_v1, rest_helpers, rest_streaming
from google.api_core import retry as retries
from google.auth import credentials as ga_credentials  # type: ignore
from google.auth.transport.requests import AuthorizedSession  # type: ignore
import google.protobuf
from google.protobuf import json_format
from requests import __version__ as requests_version

from google.cloud.cloudquotas_v1beta.types import (
    quota_adjuster_settings as gac_quota_adjuster_settings,
)
from google.cloud.cloudquotas_v1beta.types import quota_adjuster_settings

from .base import DEFAULT_CLIENT_INFO as BASE_DEFAULT_CLIENT_INFO
from .rest_base import _BaseQuotaAdjusterSettingsManagerRestTransport

try:
    OptionalRetry = Union[retries.Retry, gapic_v1.method._MethodDefault, None]
except AttributeError:  # pragma: NO COVER
    OptionalRetry = Union[retries.Retry, object, None]  # type: ignore

try:
    from google.api_core import client_logging  # type: ignore

    CLIENT_LOGGING_SUPPORTED = True  # pragma: NO COVER
except ImportError:  # pragma: NO COVER
    CLIENT_LOGGING_SUPPORTED = False

_LOGGER = logging.getLogger(__name__)

DEFAULT_CLIENT_INFO = gapic_v1.client_info.ClientInfo(
    gapic_version=BASE_DEFAULT_CLIENT_INFO.gapic_version,
    grpc_version=None,
    rest_version=f"requests@{requests_version}",
)

if hasattr(DEFAULT_CLIENT_INFO, "protobuf_runtime_version"):  # pragma: NO COVER
    DEFAULT_CLIENT_INFO.protobuf_runtime_version = google.protobuf.__version__


class QuotaAdjusterSettingsManagerRestInterceptor:
    """Interceptor for QuotaAdjusterSettingsManager.

    Interceptors are used to manipulate requests, request metadata, and responses
    in arbitrary ways.
    Example use cases include:
    * Logging
    * Verifying requests according to service or custom semantics
    * Stripping extraneous information from responses

    These use cases and more can be enabled by injecting an
    instance of a custom subclass when constructing the QuotaAdjusterSettingsManagerRestTransport.

    .. code-block:: python
        class MyCustomQuotaAdjusterSettingsManagerInterceptor(QuotaAdjusterSettingsManagerRestInterceptor):
            def pre_get_quota_adjuster_settings(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_get_quota_adjuster_settings(self, response):
                logging.log(f"Received response: {response}")
                return response

            def pre_update_quota_adjuster_settings(self, request, metadata):
                logging.log(f"Received request: {request}")
                return request, metadata

            def post_update_quota_adjuster_settings(self, response):
                logging.log(f"Received response: {response}")
                return response

        transport = QuotaAdjusterSettingsManagerRestTransport(interceptor=MyCustomQuotaAdjusterSettingsManagerInterceptor())
        client = QuotaAdjusterSettingsManagerClient(transport=transport)


    """

    def pre_get_quota_adjuster_settings(
        self,
        request: quota_adjuster_settings.GetQuotaAdjusterSettingsRequest,
        metadata: Sequence[Tuple[str, Union[str, bytes]]],
    ) -> Tuple[
        quota_adjuster_settings.GetQuotaAdjusterSettingsRequest,
        Sequence[Tuple[str, Union[str, bytes]]],
    ]:
        """Pre-rpc interceptor for get_quota_adjuster_settings

        Override in a subclass to manipulate the request or metadata
        before they are sent to the QuotaAdjusterSettingsManager server.
        """
        return request, metadata

    def post_get_quota_adjuster_settings(
        self, response: quota_adjuster_settings.QuotaAdjusterSettings
    ) -> quota_adjuster_settings.QuotaAdjusterSettings:
        """Post-rpc interceptor for get_quota_adjuster_settings

        DEPRECATED. Please use the `post_get_quota_adjuster_settings_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the QuotaAdjusterSettingsManager server but before
        it is returned to user code. This `post_get_quota_adjuster_settings` interceptor runs
        before the `post_get_quota_adjuster_settings_with_metadata` interceptor.
        """
        return response

    def post_get_quota_adjuster_settings_with_metadata(
        self,
        response: quota_adjuster_settings.QuotaAdjusterSettings,
        metadata: Sequence[Tuple[str, Union[str, bytes]]],
    ) -> Tuple[
        quota_adjuster_settings.QuotaAdjusterSettings,
        Sequence[Tuple[str, Union[str, bytes]]],
    ]:
        """Post-rpc interceptor for get_quota_adjuster_settings

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the QuotaAdjusterSettingsManager server but before it is returned to user code.

        We recommend only using this `post_get_quota_adjuster_settings_with_metadata`
        interceptor in new development instead of the `post_get_quota_adjuster_settings` interceptor.
        When both interceptors are used, this `post_get_quota_adjuster_settings_with_metadata` interceptor runs after the
        `post_get_quota_adjuster_settings` interceptor. The (possibly modified) response returned by
        `post_get_quota_adjuster_settings` will be passed to
        `post_get_quota_adjuster_settings_with_metadata`.
        """
        return response, metadata

    def pre_update_quota_adjuster_settings(
        self,
        request: gac_quota_adjuster_settings.UpdateQuotaAdjusterSettingsRequest,
        metadata: Sequence[Tuple[str, Union[str, bytes]]],
    ) -> Tuple[
        gac_quota_adjuster_settings.UpdateQuotaAdjusterSettingsRequest,
        Sequence[Tuple[str, Union[str, bytes]]],
    ]:
        """Pre-rpc interceptor for update_quota_adjuster_settings

        Override in a subclass to manipulate the request or metadata
        before they are sent to the QuotaAdjusterSettingsManager server.
        """
        return request, metadata

    def post_update_quota_adjuster_settings(
        self, response: gac_quota_adjuster_settings.QuotaAdjusterSettings
    ) -> gac_quota_adjuster_settings.QuotaAdjusterSettings:
        """Post-rpc interceptor for update_quota_adjuster_settings

        DEPRECATED. Please use the `post_update_quota_adjuster_settings_with_metadata`
        interceptor instead.

        Override in a subclass to read or manipulate the response
        after it is returned by the QuotaAdjusterSettingsManager server but before
        it is returned to user code. This `post_update_quota_adjuster_settings` interceptor runs
        before the `post_update_quota_adjuster_settings_with_metadata` interceptor.
        """
        return response

    def post_update_quota_adjuster_settings_with_metadata(
        self,
        response: gac_quota_adjuster_settings.QuotaAdjusterSettings,
        metadata: Sequence[Tuple[str, Union[str, bytes]]],
    ) -> Tuple[
        gac_quota_adjuster_settings.QuotaAdjusterSettings,
        Sequence[Tuple[str, Union[str, bytes]]],
    ]:
        """Post-rpc interceptor for update_quota_adjuster_settings

        Override in a subclass to read or manipulate the response or metadata after it
        is returned by the QuotaAdjusterSettingsManager server but before it is returned to user code.

        We recommend only using this `post_update_quota_adjuster_settings_with_metadata`
        interceptor in new development instead of the `post_update_quota_adjuster_settings` interceptor.
        When both interceptors are used, this `post_update_quota_adjuster_settings_with_metadata` interceptor runs after the
        `post_update_quota_adjuster_settings` interceptor. The (possibly modified) response returned by
        `post_update_quota_adjuster_settings` will be passed to
        `post_update_quota_adjuster_settings_with_metadata`.
        """
        return response, metadata


@dataclasses.dataclass
class QuotaAdjusterSettingsManagerRestStub:
    _session: AuthorizedSession
    _host: str
    _interceptor: QuotaAdjusterSettingsManagerRestInterceptor


class QuotaAdjusterSettingsManagerRestTransport(
    _BaseQuotaAdjusterSettingsManagerRestTransport
):
    """REST backend synchronous transport for QuotaAdjusterSettingsManager.

    The Quotas Adjuster Settings API is an infrastructure service
    for Google  Cloud that lets service consumers view and update
    their quota adjuster  settings.

    - Update quota adjuster settings.
    - Get the name of the configurations.

    This class defines the same methods as the primary client, so the
    primary client can load the underlying transport implementation
    and call it.

    It sends JSON representations of protocol buffers over HTTP/1.1
    """

    def __init__(
        self,
        *,
        host: str = "cloudquotas.googleapis.com",
        credentials: Optional[ga_credentials.Credentials] = None,
        credentials_file: Optional[str] = None,
        scopes: Optional[Sequence[str]] = None,
        client_cert_source_for_mtls: Optional[Callable[[], Tuple[bytes, bytes]]] = None,
        quota_project_id: Optional[str] = None,
        client_info: gapic_v1.client_info.ClientInfo = DEFAULT_CLIENT_INFO,
        always_use_jwt_access: Optional[bool] = False,
        url_scheme: str = "https",
        interceptor: Optional[QuotaAdjusterSettingsManagerRestInterceptor] = None,
        api_audience: Optional[str] = None,
    ) -> None:
        """Instantiate the transport.

        Args:
            host (Optional[str]):
                 The hostname to connect to (default: 'cloudquotas.googleapis.com').
            credentials (Optional[google.auth.credentials.Credentials]): The
                authorization credentials to attach to requests. These
                credentials identify the application to the service; if none
                are specified, the client will attempt to ascertain the
                credentials from the environment.

            credentials_file (Optional[str]): Deprecated. A file with credentials that can
                be loaded with :func:`google.auth.load_credentials_from_file`.
                This argument is ignored if ``channel`` is provided. This argument will be
                removed in the next major version of this library.
            scopes (Optional(Sequence[str])): A list of scopes. This argument is
                ignored if ``channel`` is provided.
            client_cert_source_for_mtls (Callable[[], Tuple[bytes, bytes]]): Client
                certificate to configure mutual TLS HTTP channel. It is ignored
                if ``channel`` is provided.
            quota_project_id (Optional[str]): An optional project to use for billing
                and quota.
            client_info (google.api_core.gapic_v1.client_info.ClientInfo):
                The client info used to send a user-agent string along with
                API requests. If ``None``, then default info will be used.
                Generally, you only need to set this if you are developing
                your own client library.
            always_use_jwt_access (Optional[bool]): Whether self signed JWT should
                be used for service account credentials.
            url_scheme: the protocol scheme for the API endpoint.  Normally
                "https", but for testing or local servers,
                "http" can be specified.
        """
        # Run the base constructor
        # TODO(yon-mg): resolve other ctor params i.e. scopes, quota, etc.
        # TODO: When custom host (api_endpoint) is set, `scopes` must *also* be set on the
        # credentials object
        super().__init__(
            host=host,
            credentials=credentials,
            client_info=client_info,
            always_use_jwt_access=always_use_jwt_access,
            url_scheme=url_scheme,
            api_audience=api_audience,
        )
        self._session = AuthorizedSession(
            self._credentials, default_host=self.DEFAULT_HOST
        )
        if client_cert_source_for_mtls:
            self._session.configure_mtls_channel(client_cert_source_for_mtls)
        self._interceptor = interceptor or QuotaAdjusterSettingsManagerRestInterceptor()
        self._prep_wrapped_messages(client_info)

    class _GetQuotaAdjusterSettings(
        _BaseQuotaAdjusterSettingsManagerRestTransport._BaseGetQuotaAdjusterSettings,
        QuotaAdjusterSettingsManagerRestStub,
    ):
        def __hash__(self):
            return hash(
                "QuotaAdjusterSettingsManagerRestTransport.GetQuotaAdjusterSettings"
            )

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None,
        ):
            uri = transcoded_request["uri"]
            method = transcoded_request["method"]
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
            )
            return response

        def __call__(
            self,
            request: quota_adjuster_settings.GetQuotaAdjusterSettingsRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, Union[str, bytes]]] = (),
        ) -> quota_adjuster_settings.QuotaAdjusterSettings:
            r"""Call the get quota adjuster
            settings method over HTTP.

                Args:
                    request (~.quota_adjuster_settings.GetQuotaAdjusterSettingsRequest):
                        The request object. Request for getting
                    QuotaAdjusterSettings
                    retry (google.api_core.retry.Retry): Designation of what errors, if any,
                        should be retried.
                    timeout (float): The timeout for this request.
                    metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                        sent along with the request as metadata. Normally, each value must be of type `str`,
                        but for metadata keys ending with the suffix `-bin`, the corresponding values must
                        be of type `bytes`.

                Returns:
                    ~.quota_adjuster_settings.QuotaAdjusterSettings:
                        The QuotaAdjusterSettings resource
                    defines the settings for the Quota
                    Adjuster.

            """

            http_options = (
                _BaseQuotaAdjusterSettingsManagerRestTransport._BaseGetQuotaAdjusterSettings._get_http_options()
            )

            request, metadata = self._interceptor.pre_get_quota_adjuster_settings(
                request, metadata
            )
            transcoded_request = _BaseQuotaAdjusterSettingsManagerRestTransport._BaseGetQuotaAdjusterSettings._get_transcoded_request(
                http_options, request
            )

            # Jsonify the query params
            query_params = _BaseQuotaAdjusterSettingsManagerRestTransport._BaseGetQuotaAdjusterSettings._get_query_params_json(
                transcoded_request
            )

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(
                logging.DEBUG
            ):  # pragma: NO COVER
                request_url = "{host}{uri}".format(
                    host=self._host, uri=transcoded_request["uri"]
                )
                method = transcoded_request["method"]
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                    "payload": request_payload,
                    "requestMethod": method,
                    "requestUrl": request_url,
                    "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.api.cloudquotas_v1beta.QuotaAdjusterSettingsManagerClient.GetQuotaAdjusterSettings",
                    extra={
                        "serviceName": "google.api.cloudquotas.v1beta.QuotaAdjusterSettingsManager",
                        "rpcName": "GetQuotaAdjusterSettings",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = QuotaAdjusterSettingsManagerRestTransport._GetQuotaAdjusterSettings._get_response(
                self._host,
                metadata,
                query_params,
                self._session,
                timeout,
                transcoded_request,
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = quota_adjuster_settings.QuotaAdjusterSettings()
            pb_resp = quota_adjuster_settings.QuotaAdjusterSettings.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_get_quota_adjuster_settings(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            resp, _ = self._interceptor.post_get_quota_adjuster_settings_with_metadata(
                resp, response_metadata
            )
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(
                logging.DEBUG
            ):  # pragma: NO COVER
                try:
                    response_payload = (
                        quota_adjuster_settings.QuotaAdjusterSettings.to_json(response)
                    )
                except:
                    response_payload = None
                http_response = {
                    "payload": response_payload,
                    "headers": dict(response.headers),
                    "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.api.cloudquotas_v1beta.QuotaAdjusterSettingsManagerClient.get_quota_adjuster_settings",
                    extra={
                        "serviceName": "google.api.cloudquotas.v1beta.QuotaAdjusterSettingsManager",
                        "rpcName": "GetQuotaAdjusterSettings",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    class _UpdateQuotaAdjusterSettings(
        _BaseQuotaAdjusterSettingsManagerRestTransport._BaseUpdateQuotaAdjusterSettings,
        QuotaAdjusterSettingsManagerRestStub,
    ):
        def __hash__(self):
            return hash(
                "QuotaAdjusterSettingsManagerRestTransport.UpdateQuotaAdjusterSettings"
            )

        @staticmethod
        def _get_response(
            host,
            metadata,
            query_params,
            session,
            timeout,
            transcoded_request,
            body=None,
        ):
            uri = transcoded_request["uri"]
            method = transcoded_request["method"]
            headers = dict(metadata)
            headers["Content-Type"] = "application/json"
            response = getattr(session, method)(
                "{host}{uri}".format(host=host, uri=uri),
                timeout=timeout,
                headers=headers,
                params=rest_helpers.flatten_query_params(query_params, strict=True),
                data=body,
            )
            return response

        def __call__(
            self,
            request: gac_quota_adjuster_settings.UpdateQuotaAdjusterSettingsRequest,
            *,
            retry: OptionalRetry = gapic_v1.method.DEFAULT,
            timeout: Optional[float] = None,
            metadata: Sequence[Tuple[str, Union[str, bytes]]] = (),
        ) -> gac_quota_adjuster_settings.QuotaAdjusterSettings:
            r"""Call the update quota adjuster
            settings method over HTTP.

                Args:
                    request (~.gac_quota_adjuster_settings.UpdateQuotaAdjusterSettingsRequest):
                        The request object. Request for updating
                    QuotaAdjusterSettings
                    retry (google.api_core.retry.Retry): Designation of what errors, if any,
                        should be retried.
                    timeout (float): The timeout for this request.
                    metadata (Sequence[Tuple[str, Union[str, bytes]]]): Key/value pairs which should be
                        sent along with the request as metadata. Normally, each value must be of type `str`,
                        but for metadata keys ending with the suffix `-bin`, the corresponding values must
                        be of type `bytes`.

                Returns:
                    ~.gac_quota_adjuster_settings.QuotaAdjusterSettings:
                        The QuotaAdjusterSettings resource
                    defines the settings for the Quota
                    Adjuster.

            """

            http_options = (
                _BaseQuotaAdjusterSettingsManagerRestTransport._BaseUpdateQuotaAdjusterSettings._get_http_options()
            )

            request, metadata = self._interceptor.pre_update_quota_adjuster_settings(
                request, metadata
            )
            transcoded_request = _BaseQuotaAdjusterSettingsManagerRestTransport._BaseUpdateQuotaAdjusterSettings._get_transcoded_request(
                http_options, request
            )

            body = _BaseQuotaAdjusterSettingsManagerRestTransport._BaseUpdateQuotaAdjusterSettings._get_request_body_json(
                transcoded_request
            )

            # Jsonify the query params
            query_params = _BaseQuotaAdjusterSettingsManagerRestTransport._BaseUpdateQuotaAdjusterSettings._get_query_params_json(
                transcoded_request
            )

            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(
                logging.DEBUG
            ):  # pragma: NO COVER
                request_url = "{host}{uri}".format(
                    host=self._host, uri=transcoded_request["uri"]
                )
                method = transcoded_request["method"]
                try:
                    request_payload = type(request).to_json(request)
                except:
                    request_payload = None
                http_request = {
                    "payload": request_payload,
                    "requestMethod": method,
                    "requestUrl": request_url,
                    "headers": dict(metadata),
                }
                _LOGGER.debug(
                    f"Sending request for google.api.cloudquotas_v1beta.QuotaAdjusterSettingsManagerClient.UpdateQuotaAdjusterSettings",
                    extra={
                        "serviceName": "google.api.cloudquotas.v1beta.QuotaAdjusterSettingsManager",
                        "rpcName": "UpdateQuotaAdjusterSettings",
                        "httpRequest": http_request,
                        "metadata": http_request["headers"],
                    },
                )

            # Send the request
            response = QuotaAdjusterSettingsManagerRestTransport._UpdateQuotaAdjusterSettings._get_response(
                self._host,
                metadata,
                query_params,
                self._session,
                timeout,
                transcoded_request,
                body,
            )

            # In case of error, raise the appropriate core_exceptions.GoogleAPICallError exception
            # subclass.
            if response.status_code >= 400:
                raise core_exceptions.from_http_response(response)

            # Return the response
            resp = gac_quota_adjuster_settings.QuotaAdjusterSettings()
            pb_resp = gac_quota_adjuster_settings.QuotaAdjusterSettings.pb(resp)

            json_format.Parse(response.content, pb_resp, ignore_unknown_fields=True)

            resp = self._interceptor.post_update_quota_adjuster_settings(resp)
            response_metadata = [(k, str(v)) for k, v in response.headers.items()]
            (
                resp,
                _,
            ) = self._interceptor.post_update_quota_adjuster_settings_with_metadata(
                resp, response_metadata
            )
            if CLIENT_LOGGING_SUPPORTED and _LOGGER.isEnabledFor(
                logging.DEBUG
            ):  # pragma: NO COVER
                try:
                    response_payload = (
                        gac_quota_adjuster_settings.QuotaAdjusterSettings.to_json(
                            response
                        )
                    )
                except:
                    response_payload = None
                http_response = {
                    "payload": response_payload,
                    "headers": dict(response.headers),
                    "status": response.status_code,
                }
                _LOGGER.debug(
                    "Received response for google.api.cloudquotas_v1beta.QuotaAdjusterSettingsManagerClient.update_quota_adjuster_settings",
                    extra={
                        "serviceName": "google.api.cloudquotas.v1beta.QuotaAdjusterSettingsManager",
                        "rpcName": "UpdateQuotaAdjusterSettings",
                        "metadata": http_response["headers"],
                        "httpResponse": http_response,
                    },
                )
            return resp

    @property
    def get_quota_adjuster_settings(
        self,
    ) -> Callable[
        [quota_adjuster_settings.GetQuotaAdjusterSettingsRequest],
        quota_adjuster_settings.QuotaAdjusterSettings,
    ]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._GetQuotaAdjusterSettings(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def update_quota_adjuster_settings(
        self,
    ) -> Callable[
        [gac_quota_adjuster_settings.UpdateQuotaAdjusterSettingsRequest],
        gac_quota_adjuster_settings.QuotaAdjusterSettings,
    ]:
        # The return type is fine, but mypy isn't sophisticated enough to determine what's going on here.
        # In C++ this would require a dynamic_cast
        return self._UpdateQuotaAdjusterSettings(self._session, self._host, self._interceptor)  # type: ignore

    @property
    def kind(self) -> str:
        return "rest"

    def close(self):
        self._session.close()


__all__ = ("QuotaAdjusterSettingsManagerRestTransport",)
