"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return encoding_1.unresolved(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === 'string') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === 'number') {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every(x => typeof x === 'string')) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return isResolvableObject(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    constructor() {
    }
}
exports.Token = Token;
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return resolve_1.resolve(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: (options.preparing !== undefined ? options.preparing : false),
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return isResolvableObject(obj);
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({ produce: context => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== 'number' ? resolved : `${resolved}`;
                } });
        }
        else {
            return typeof x !== 'number' ? x : `${x}`;
        }
    }
    constructor() {
    }
}
exports.Tokenization = Tokenization;
function isResolvableObject(x) {
    return typeof (x) === 'object' && x !== null && typeof x.resolve === 'function';
}
exports.isResolvableObject = isResolvableObject;
function withResolved(...args) {
    if (args.length < 2) {
        return;
    }
    const argArray = args.slice(0, args.length - 1);
    if (argArray.some(Token.isUnresolved)) {
        return;
    }
    args[args.length - 1].apply(arguments, argArray);
}
exports.withResolved = withResolved;
//# sourceMappingURL=data:application/json;base64,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