"""LICENSE
Copyright 2017 Hermann Krumrey <hermann@krumreyh.com>

This file is part of bundesliga-tippspiel.

bundesliga-tippspiel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

bundesliga-tippspiel is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with bundesliga-tippspiel.  If not, see <http://www.gnu.org/licenses/>.
LICENSE"""

from typing import Dict, Any, Optional, List, Type
from flask import abort, redirect, url_for, request
from werkzeug.wrappers import Response
from puffotter.flask.base import db
from puffotter.flask.enums import AlertSeverity
from puffotter.flask.db.ModelMixin import ModelMixin
from bundesliga_tippspiel.Config import Config
from bundesliga_tippspiel.exceptions import ActionException
from bundesliga_tippspiel.db.match_data.Match import Match


class Action:
    """
    A framework class for actions that can be used by normal site requests
    as well as API calls to achieve stuff™.
    """

    def validate_data(self):
        """
        Validates user-provided data
        :return: None
        :raises ActionException: if any data discrepancies are found
        """
        raise NotImplementedError()  # pragma: no cover

    def _execute(self) -> Dict[str, Any]:
        """
        Executes the actual action
        :return: A JSON-compatible dictionary containing the response
        :raises ActionException: if anything went wrong
        """
        raise NotImplementedError()  # pragma: no cover

    def execute(self) -> Dict[str, Any]:
        """
        Executes the action after validating user-provided data
        :return: A JSON-compatible dictionary containing the response
        :raises ActionException: if anything went wrong
        """
        self.validate_data()
        return self._execute()

    @classmethod
    def from_dict(cls, data: Dict[str, Any]):
        """
        Generates an action from a dictionary
        :param data: The dictionary containing the relevant data
        :return: The generated Action object
        """
        try:
            return cls._from_dict(data)
        except (ValueError, TypeError):
            raise ActionException(
                "invalid parameters",
                "Ungültige Parameter"
            )

    @classmethod
    def _from_dict(cls, data: Dict[str, Any]):
        """
        Generates an action from a dictionary
        :param data: The dictionary containing the relevant data
        :return: The generated Action object
        """
        raise NotImplementedError()  # pragma: no cover

    @classmethod
    def from_site_request(cls):
        """
        Generates an Action object from a site request
        :return: The generated Action object
        """
        try:

            if request.method == "GET":
                data = request.args
            elif request.method == "POST":
                data = request.form
            else:  # pragma: no cover
                raise KeyError()
            return cls.from_dict(data)

        except (KeyError, TypeError, ValueError):
            abort(400)

    def execute_with_redirects(
            self,
            success_url: str,
            success_msg: str,
            failure_url: str
    ) -> Response:
        """
        Executes the action and subsequently redirects accordingly
        :param success_url: The URL to which to redirect upon success
        :param success_msg: The message to flash on success
        :param failure_url: The URl to which to redirect upon failure
        :return: The redirect
        """
        try:
            self.execute()
            ActionException(
                success_msg, success_msg, 200, AlertSeverity.SUCCESS
            ).flash()
            return redirect(url_for(success_url))

        except ActionException as e:
            e.flash()
            return redirect(url_for(failure_url))

    @staticmethod
    def handle_id_fetch(_id: int, query_cls: Type[db.Model]) -> db.Model:
        """
        Handles fetching a single object by using it's ID
        Raises an ActionException if an ID does not exist
        :return: The object identified by the ID
        :raises ActionException: Without fail
        """
        result = query_cls.query.get(_id)
        if result is None:
            raise ActionException(
                "ID does not exist",
                "Die angegebene ID existiert nicht",
                404
            )
        else:
            return result

    @staticmethod
    def check_id_or_filters(
            _id: Optional[int], filters: List[Optional[Any]]
    ):
        """
        Checks that no filters are applied if a specific ID was provided
        :param _id: The specific ID
        :param filters: A list of filters
        :return: None
        """
        has_filter = len(list(filter(lambda x: x is not None, filters))) > 0
        if has_filter and _id is not None:
            raise ActionException(
                "Can't filter specific ID",
                "Eine spezifische ID kann nicht gefiltered werden"
            )

    @staticmethod
    def resolve_and_check_matchday(matchday: Optional[int]) -> Optional[int]:
        """
        Checks the bound of a matchday
        :param matchday: The matchday to check
        :return: None
        :raises ActionException: If the matchday is invalid
        """
        if matchday is not None:
            if matchday == -1:
                all_matches = Match.query\
                    .filter_by(season=Config.season()).all()
                filtered = list(filter(lambda x: not x.started, all_matches))
                if len(filtered) > 0:
                    matchday = min(filtered, key=lambda x: x.matchday).matchday
                else:
                    matchday = 34
            elif not 0 < matchday < 35:
                raise ActionException(
                    "Matchday out of bounds",
                    "Den angegebenen Spieltag gibt es nicht"
                )
        return matchday

    @staticmethod
    def get_current_matchday() -> int:
        """
        :return: The current matchday
        """
        matchday = Action.resolve_and_check_matchday(-1)
        assert matchday is not None
        return matchday

    def prepare_get_response(self, result: List[ModelMixin], keyword: str) \
            -> Dict[str, Any]:
        """
        Prepares a GetAction response by
        :param result: The result to wrap in a response dictionary
        :param keyword: The keyword to use, e.g: bet|match|player etc.
        :return: The wrapped response dictionary
        """
        key = "{}s".format(keyword)
        if keyword in ["match"]:
            key = "{}es".format(keyword)

        response: Dict[str, Any] = {key: result}

        if getattr(self, "id", None) is not None:
            response[keyword] = result[0]

        return response


# noinspection PyAbstractClass
class GetAction(Action):
    """
    Special Action class for 'Get' Actions
    """
    def __init__(self, _id: Optional[int]):
        """
        :param _id: The ID to get
        """
        self.id = None if _id is None else int(_id)
