# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN
.. _`LARSIM`: http://www.larsim.de/en/the-model/

One can understand |lland_knauf| as an extended, more complicated
version of the application model |lland_dd|, using precipitation, temperature, wind
speed, relative humidity, and actual and possible sunshine duration as meteorological
input parameters.  The main difference is that |lland_knauf| models the energy balance
of snow processes in more detail :cite:p:`ref-LUBW2006a`, taking not only the air
temperature but also forcings like the soil heat flux or global radiation into account
explicitly.  This methodology includes calculating age-dependent snow albedo values,
which |lland_knauf| can provide to actual evapotranspiration submodels as
|evap_aet_morsim| for fine-tuning their radiation energy balance.  The second
difference is that |lland_knauf| models the soil temperature to simulate the effect of
soil water freezing on runoff generation.  We created |lland_knauf| on behalf of the
`German Federal Institute of Hydrology (BfG)`_ in the context of optimising the control
of the Kiel Canal (Nord-Ostsee-Kanal).

The following list summarises the main components of |lland_knauf|:

 * Multiple routines for adjusting the meteorological input data
 * Mixed precipitation within a definable temperature range
 * Energy balance of the snow layer after Knauf :cite:p:`ref-LUBW2006a`
 * Dynamical adjustment of the snow albedo
 * Optional refinement of the snow-surface temperature
 * Optional (re)freezing of liquid water within the snow layer
 * A simple snow retention routine
 * Usage of a submodel following the |AETModel_V1| interface for calculating
   interception evaporation, soil evapotranspiration, and water evaporation
 * Direct runoff generation based on the Xinanjiang model :cite:p:`ref-zhao1977flood`
 * Optional inclusion of a soil submodel following the |SoilModel_V1| interface
 * Frost-sealing of the upper soil layer
 * One base flow, two interflow and two direct flow components
 * A freely configurable capillary rise routine
 * Options to limit the capacity of the base flow storage
 * Separate linear storages for modelling runoff concentration
 * Additional evaporation from water areas within the subcatchment
 * Optional evaporation from inflowing runoff

Some notes for `LARSIM`_ users: |lland_knauf| is similar to many `LARSIM`_ models used
for forecasting (relying on a combination of options equivalent to the features listed
above) but not identical.  Often, we strive for more flexibility. One example is
modifying parameter |KTSchnee| to control if and how |lland_knauf| adjusts the snow
surface temperature.  However, there are also differences due to technical reasons.
One relevant difference is that *HydPy* does not include "future values" in the current
simulation timestep. For example, `LARSIM`_ fills nightly gaps regarding cloudiness
with the average value of the present calendar day, while |lland_knauf| uses the
average over the 24 hours preceding the current simulation step.  Hence, do not expect
the results of |lland_knauf| and `LARSIM`_ to be identical.  When switching from one
model system to the other, the need to fine-tune some parameters via calibration might
arise.

|lland_knauf| requires a submodel that complies with the |RadiationModel_V1| or the
|RadiationModel_V4| interface and provides the time series of possible sunshine
duration, actual sunshine duration, and global radiation.  An obvious choice is
|meteo_glob_morsim|, which estimates these meteorological properties according to
`LARSIM`_.

.. note::

    Some details of |lland_knauf| are still under discussion and might change in the
    future.

Integration tests
=================

.. how_to_understand_integration_tests::

Many of the following integration tests are similar to those for the application model
|lland_dd|.  When there are no substantial differences, we do not repeat our
explanations, so please note the respective sections of the documentation on
|lland_dd|.  Some additional tests cover structural differences between land-use types
that only exist for |lland_knauf|; others show various possible snow-related
configurations of |lland_knauf|.  Furthermore, we perform daily and hourly tests to
point out differences and ensure |lland_knauf| works well for different simulation step
sizes.

Note that while our daily simulations always cover an entire month, our hourly
simulations cover only three days, focussing on the "most interesting" part of the
daily simulation.  To ensure the best possible comparability, we use the same input
data (we aggregated hourly values to daily values), the same parameter values (possibly
adapted to the change in the time step size), and the same initial conditions (we
extract the required initial conditions of the respective hourly simulation from the
longer daily simulation).

daily simulation
________________

The following general setup is identical to the one of |lland_dd|, except that we now
start with a different period and a daily simulation time step:

>>> from hydpy import pub
>>> pub.timegrids = "1997-08-01", "1997-09-01", "1d"
>>> from hydpy.models.lland_knauf import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> inlet = Node("inlet")
>>> land = Element("land", inlets=inlet, outlets=outlet)
>>> land.model = model
>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)

.. _lland_knauf_acker_summer_daily:

acre (summer)
-------------

First, we set the values of those parameters also required by the application model
|lland_dd| (see integration test :ref:`lland_dd_acker_summer`):

>>> lnk(ACKER)
>>> kg(0.94)
>>> kt(0.0)
>>> hinz(0.2)
>>> lai.acker_aug = 3.5
>>> tgr(0.0)
>>> trefn(0.0)
>>> tsp(4.0)
>>> pwmax(1.43)
>>> wmax(309.0)
>>> fk(199.0)
>>> pwp(119.4)
>>> kapgrenz(option="0_WMax/10")
>>> kapmax(0.08)
>>> beta(0.001/24)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(r_dmax=0.55)
>>> dmin(r_dmin=10.0)
>>> bsf(0.3)
>>> volbmax(inf)
>>> gsbmax(1.0)
>>> gsbgrad1(inf)
>>> gsbgrad2(inf)
>>> a1(0.0)
>>> a2(0.5)
>>> tind(0.191056)
>>> eqb(35000.0)
>>> eqi1(400.0)
>>> eqi2(400.0)
>>> eqd1(200.0)
>>> eqd2(50.0)

In contrast to the documentation on |lland_dd|, we set parameter |NegQ| to |True| in
most examples.  That means we favour negative discharge estimates over water balance
errors whenever the base flow storage cannot meet the water demand of the capillary
rise of groundwater into the soil:

>>> negq(True)

Next, we set the values of the parameters specific to |lland_knauf|, beginning with
those required for the energy-accounting snow modelling approach after Knauf
:cite:p:`ref-LUBW2006a`:

>>> measuringheightwindspeed(10.0)
>>> wg2z.aug = -2.9
>>> p1strahl(0.5)
>>> p2strahl(1.0/35.0)
>>> p1wind(0.6)
>>> p2wind(1.0/70.0)
>>> turb0(2.0)
>>> turb1(1.6)
>>> albedo0snow(0.9)
>>> snowagingfactor(0.35)
>>> refreezeflag(True)
>>> ktschnee(inf)

Now, we set the values of the parameters controlling the frost sealing of the upper
soil layer:

>>> fvf(0.3)
>>> bsff(2.0)

All integration tests performed in daily simulation steps rely on pre-calculated
radiation-related data.  We use the |meteo_psun_sun_glob_io| submodel to provide
|lland_knauf| with this data:

>>> with model.add_radiationmodel_v4("meteo_psun_sun_glob_io") as submodel_meteo_psun_sun_glob_io:
...     pass

We select |evap_aet_morsim|, which implements the MORECS method with some
LARSIM-specific modifications, as the submodel for calculating the different
evapotranspiration components.  Note that |evap_aet_morsim| and |lland_knauf| can share
the same submodel instance due to them requiring the same radiation-related data and
|meteo_psun_sun_glob_io| being a sharable submodel that complies with
|SharableSubmodelInterface|:

>>> with model.add_aetmodel_v1("evap_aet_morsim"):
...     albedo.acker_aug = 0.24
...     cropheight.acker_aug = 0.4
...     surfaceresistance.acker = 40.0
...     emissivity(0.95)
...     averagesoilheatflux.aug = 2.9
...     soilmoisturelimit(119.4 / 309.0)
...     model.add_radiationmodel_v4(submodel_meteo_psun_sun_glob_io)

We initialise a test function object that prepares and runs the tests and prints and
plots their results:

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.nied
>>> IntegrationTest.plotting_options.axis2 = fluxes.qah
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

Due to preparing a simulation with a daily simulation step size, we only need to define
the state sequences' initial values (not the log sequences' initial values):

>>> test.inits = ((states.inzp, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.esnow, 0.0),
...               (states.taus, 0.0),
...               (states.ebdn, 350.0),
...               (states.bowa, 72.0),
...               (states.sdg1, 0.0),
...               (states.sdg2, 0.0),
...               (states.sig1, 0.0),
...               (states.sig2, 0.0),
...               (states.sbg, 0.0))

We prepare those input time series usually available as measurements:

>>> inputs.nied.series = (
...     13.3, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 2.7,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 0.3, 0.0, 0.1, 10.5, 6.5, 15.0, 9.2, 0.0,
...     0.0)
>>> inputs.teml.series = (
...     16.1, 17.3, 18.9, 17.8, 18.5, 19.9, 20.8, 20.9, 22.3, 23.0, 23.1, 23.2, 23.2,
...     21.7, 20.8, 19.5, 21.0, 21.6, 21.1, 22.4, 22.5, 19.6, 20.3, 24.1, 23.7, 21.1,
...     20.9, 20.2, 16.1, 17.0, 18.2)
>>> inputs.windspeed.series = (
...     2.4, 1.6, 1.5, 2.8, 3.0, 3.5, 3.8, 3.3, 1.8, 1.9, 3.0, 2.5, 1.6, 2.7, 3.2, 1.3,
...     1.9, 2.0, 2.6, 3.6, 2.5, 2.1, 1.5, 2.2, 1.9, 2.3, 2.1, 2.6, 2.6, 2.1, 2.1)
>>> inputs.relativehumidity.series = (
...     86.2, 85.8, 82.4, 77.6, 74.9, 77.9, 73.5, 80.1, 72.9, 68.5, 68.6, 66.0, 69.0,
...     75.6, 81.4, 79.0, 75.9, 70.0, 70.3, 69.0, 62.1, 84.5, 83.6, 76.5, 76.6, 84.2,
...     85.8, 86.5, 89.6, 78.3, 78.5)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     6.3, 1.7, 4.5, 12.4, 13.9, 13.0, 13.8, 12.3, 13.1, 12.8, 13.1, 13.3, 12.7,
...     10.2, 9.4, 10.3, 11.1, 11.0, 8.5, 11.3, 12.4, 0.1, 6.7, 10.4, 6.5, 4.9, 6.6,
...     0.3, 0.1, 5.0, 3.8)

>>> aetinputs = model.aetmodel.sequences.inputs
>>> aetinputs.atmosphericpressure.series = (
...     1007.0, 1013.0, 1016.0, 1017.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1020.0,
...     1021.0, 1022.0, 1019.0, 1015.0, 1016.0, 1019.0, 1019.0, 1018.0, 1019.0, 1019.0,
...     1017.0, 1015.0, 1015.0, 1011.0, 1008.0, 1006.0, 1005.0, 1000.0, 1004.0, 1011.0,
...     1014.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

The following two input time series stem from the results of |meteo_glob_morsim|
calculated in the :ref:`meteo_glob_morsim_daily_simulation_summer` example, which
addresses the same place and time:

>>> model.radiationmodel.sequences.inputs.possiblesunshineduration.series = (
...     15.69078473, 15.62882066, 15.56613383, 15.50275437, 15.43871163, 15.37403418,
...     15.3087498, 15.24288548, 15.17646742, 15.10952106, 15.04207104, 14.97414122,
...     14.90575471, 14.83693386, 14.76770029, 14.69807488, 14.62807779, 14.5577285,
...     14.48704578, 14.41604776, 14.34475191, 14.27317506, 14.20133345, 14.12924271,
...     14.05691791, 13.98437357, 13.91162366, 13.83868165, 13.76556053, 13.69227282,
...     13.61883057)
>>> model.radiationmodel.sequences.inputs.globalradiation.series = (
...     190.2514903, 127.6607822, 164.4012978, 269.0205016, 287.8486672, 274.6742631,
...     284.0478917, 262.9712633, 272.2564135, 267.0192178, 269.6265466, 270.8782909,
...     261.6971638, 227.9621155, 216.3563213, 226.6379036, 235.5285025, 232.8494284,
...     199.7134896, 233.7719021, 246.1212861, 90.58551, 171.9449932, 216.5498126,
...     166.9168296, 145.9856109, 165.5438903, 87.6933456, 84.356885, 142.3092025,
...     126.7220785)

The following results illustrate the behaviour of |lland_knauf| for relatively dry and
hot summer condions. Compared to the result table of integration test
:ref:`lland_dd_acker_summer` of the application model |lland_dd|, there are many more
columns because of to the higher number of input, flux, and state sequences.  These are
mainly due to the increased data requirements and the more complex calculations of the
Knauf :cite:p:`ref-LUBW2006a` approach. Therefore, |lland_knauf| calculates some "daily
values", representing the averages or sums over the last 24 hours:

.. integration-test::

    >>> parameters.update()
    >>> test.reset_inits()
    >>> conditions = model.conditions
    >>> conditions_acker_summer = test(
    ...     "lland_knauf_acker_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |   nbes | sbes |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga |       qah |        qa | inzp | wats | waes | esnow | taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 |       sbg | inlet |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                       0.0 |                      0.0 |              0.0 | 11.802 |  0.0 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |  0.196005 |  0.002269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.565338 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.002269 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |  0.263667 |  0.003052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |  0.30167 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.003052 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |  0.140696 |  0.001628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 | 0.160975 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.001628 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |  0.075077 |  0.000869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 | 0.085898 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000869 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |  0.040062 |  0.000464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 | 0.045836 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000464 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |  0.021377 |  0.000247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 | 0.024459 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000247 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |  0.011407 |  0.000132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 | 0.013051 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000132 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |  0.006087 |   0.00007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 | 0.006964 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00007 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |  0.003248 |  0.000038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 | 0.003716 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000038 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |  0.001733 |   0.00002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 | 0.001983 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00002 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000925 |  0.000011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 | 0.001058 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000011 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000494 |  0.000006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 | 0.000565 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000006 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000263 |  0.000003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 | 0.000301 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |  0.000141 |  0.000002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 | 0.000161 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |  0.000002 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                       0.0 |                      0.0 |              0.0 |  1.838 |  0.0 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |   0.01292 |   0.00015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.037134 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |   0.00015 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 |  0.017319 |    0.0002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 | 0.019815 |  0.0 |  0.0 |  0.0 |       0.0 |   0.0 |    0.0002 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 |  0.009173 |  0.000106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 | 0.010574 |  0.0 |  0.0 |  0.0 | -0.037922 |   0.0 |  0.000106 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |  0.004367 |  0.000051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 | 0.005642 |  0.0 |  0.0 |  0.0 | -0.276602 |   0.0 |  0.000051 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 |  0.000839 |   0.00001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 | 0.003011 |  0.0 |  0.0 |  0.0 | -0.721723 |   0.0 |   0.00001 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |   -0.0023 | -0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 | 0.001607 |  0.0 |  0.0 |  0.0 | -1.342015 |   0.0 | -0.000027 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 | -0.005541 | -0.000064 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 | 0.000857 |  0.0 |  0.0 |  0.0 | -2.162781 |   0.0 | -0.000064 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                       0.0 |                      0.0 |              0.0 |   1.18 |  0.0 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 | -0.005779 | -0.000067 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.010337 |  0.0 |  0.0 |  0.0 | -3.188768 |   0.0 | -0.000067 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 | -0.008334 | -0.000096 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 | 0.005516 |  0.0 |  0.0 |  0.0 | -4.141203 |   0.0 | -0.000096 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 | -0.014111 | -0.000163 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 | 0.002943 |  0.0 |  0.0 |  0.0 | -5.155306 |   0.0 | -0.000163 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 | -0.019204 | -0.000222 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 | 0.001571 |  0.0 |  0.0 |  0.0 | -6.310376 |   0.0 | -0.000222 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                       0.0 |                      0.0 |              0.0 |   9.17 |  0.0 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 |  0.002769 |  0.000032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.078493 |  0.0 |  0.0 |  0.0 | -7.557056 |   0.0 |  0.000032 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                       0.0 |                      0.0 |              0.0 |   5.41 |  0.0 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |   0.03159 |  0.000366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.109268 |  0.0 |  0.0 |  0.0 | -8.257199 |   0.0 |  0.000366 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                       0.0 |                      0.0 |              0.0 |   13.4 |  0.0 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 |  0.101121 |   0.00117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.290685 |  0.0 |  0.0 |  0.0 | -8.686544 |   0.0 |   0.00117 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                       0.0 |                      0.0 |              0.0 |  7.948 |  0.0 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 |  0.168598 |  0.001951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 |  0.34013 |  0.0 |  0.0 |  0.0 | -8.655423 |   0.0 |  0.001951 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 |  0.127624 |  0.001477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 | 0.181497 |  0.0 |  0.0 |  0.0 | -8.624414 |   0.0 |  0.001477 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |   0.05375 |  0.000622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 | 0.096849 |  0.0 |  0.0 |  0.0 | -8.593516 |   0.0 |  0.000622 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_acker_routing_daily:

:ref:`acre (routing) <lland_dd_acker_routing>`
----------------------------------------------

The following calculation shows the possibility of routing inflowing runoff, discussed
in the documentation of |lland_dd|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_knauf_acker_routing_daily",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa)
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |   qz |   qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |   nbes | sbes |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga |      qah |       qa | inzp | wats | waes | esnow | taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 |       sbg | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.02 | 1.728 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                       0.0 |                      0.0 |              0.0 | 11.802 |  0.0 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 | 1.924005 | 0.022269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.565338 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.022269 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.02 | 1.728 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 | 1.991667 | 0.023052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |  0.30167 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.023052 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.02 | 1.728 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 | 1.868696 | 0.021628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 | 0.160975 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.021628 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.02 | 1.728 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 | 1.803077 | 0.020869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 | 0.085898 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020869 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.02 | 1.728 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 | 1.768062 | 0.020464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 | 0.045836 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020464 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.02 | 1.728 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 | 1.749377 | 0.020247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 | 0.024459 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020247 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.02 | 1.728 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 | 1.739407 | 0.020132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 | 0.013051 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020132 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.02 | 1.728 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 | 1.734087 |  0.02007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 | 0.006964 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02007 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 | 1.731248 | 0.020038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 | 0.003716 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020038 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.02 | 1.728 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 | 1.729733 |  0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 | 0.001983 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02002 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728925 | 0.020011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 | 0.001058 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020011 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.02 | 1.728 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728494 | 0.020006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 | 0.000565 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020006 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.02 | 1.728 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728263 | 0.020003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 | 0.000301 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020003 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.02 | 1.728 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 | 1.728141 | 0.020002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 | 0.000161 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 | 0.020002 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.02 | 1.728 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                       0.0 |                      0.0 |              0.0 |  1.838 |  0.0 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |  1.74092 |  0.02015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.037134 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |  0.02015 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.02 | 1.728 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 | 1.745319 |   0.0202 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 | 0.019815 |  0.0 |  0.0 |  0.0 |       0.0 |  0.02 |   0.0202 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.02 | 1.728 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 | 1.737173 | 0.020106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 | 0.010574 |  0.0 |  0.0 |  0.0 | -0.037922 |  0.02 | 0.020106 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.02 | 1.728 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 | 1.732367 | 0.020051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 | 0.005642 |  0.0 |  0.0 |  0.0 | -0.276602 |  0.02 | 0.020051 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.02 | 1.728 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 | 1.728839 |  0.02001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 | 0.003011 |  0.0 |  0.0 |  0.0 | -0.721723 |  0.02 |  0.02001 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.02 | 1.728 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |   1.7257 | 0.019973 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 | 0.001607 |  0.0 |  0.0 |  0.0 | -1.342015 |  0.02 | 0.019973 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.02 | 1.728 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 | 1.722459 | 0.019936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 | 0.000857 |  0.0 |  0.0 |  0.0 | -2.162781 |  0.02 | 0.019936 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.02 | 1.728 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                       0.0 |                      0.0 |              0.0 |   1.18 |  0.0 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 | 1.722221 | 0.019933 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.010337 |  0.0 |  0.0 |  0.0 | -3.188768 |  0.02 | 0.019933 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.02 | 1.728 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 | 1.719666 | 0.019904 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 | 0.005516 |  0.0 |  0.0 |  0.0 | -4.141203 |  0.02 | 0.019904 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.02 | 1.728 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 | 1.713889 | 0.019837 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 | 0.002943 |  0.0 |  0.0 |  0.0 | -5.155306 |  0.02 | 0.019837 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.02 | 1.728 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 | 1.708796 | 0.019778 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 | 0.001571 |  0.0 |  0.0 |  0.0 | -6.310376 |  0.02 | 0.019778 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.02 | 1.728 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                       0.0 |                      0.0 |              0.0 |   9.17 |  0.0 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 | 1.730769 | 0.020032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.078493 |  0.0 |  0.0 |  0.0 | -7.557056 |  0.02 | 0.020032 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.02 | 1.728 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                       0.0 |                      0.0 |              0.0 |   5.41 |  0.0 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |  1.75959 | 0.020366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.109268 |  0.0 |  0.0 |  0.0 | -8.257199 |  0.02 | 0.020366 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.02 | 1.728 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                       0.0 |                      0.0 |              0.0 |   13.4 |  0.0 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 | 1.829121 |  0.02117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.290685 |  0.0 |  0.0 |  0.0 | -8.686544 |  0.02 |  0.02117 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.02 | 1.728 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                       0.0 |                      0.0 |              0.0 |  7.948 |  0.0 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 | 1.896598 | 0.021951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 |  0.34013 |  0.0 |  0.0 |  0.0 | -8.655423 |  0.02 | 0.021951 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.02 | 1.728 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 | 1.855624 | 0.021477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 | 0.181497 |  0.0 |  0.0 |  0.0 | -8.624414 |  0.02 | 0.021477 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.02 | 1.728 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |  1.78175 | 0.020622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 | 0.096849 |  0.0 |  0.0 |  0.0 | -8.593516 |  0.02 | 0.020622 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_acker_heavy_rain_daily:

acre (heavy rain)
-----------------

Integration test :ref:`lland_knauf_acker_summer_daily` deals with dry summer
conditions.  To show how |lland_knauf| works for warm but wet conditions, we set the
precipitation input time series constantly to 20 mm.  Now, the soil water content
(|BoWa|) rises from its initial value of 72 mm and nearly reaches its maximum value of
309 mm (|WMax|), resulting in a relevant response of all runoff components:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.0
    >>> nied = inputs.nied.series.copy()
    >>> inputs.nied.series = 20.0
    >>> conditions_acker_heavy_rain = test(
    ...     "lland_knauf_acker_heavy_rain_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |     qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |     qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga |       qah |       qa | inzp | wats | waes | esnow | taus |       ebdn |       bowa |      sdg1 |     sdg2 |     sig1 |     sig2 |      sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 18.8 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.217805 |      0.0 |      0.0 |      0.0 |  0.0 |  1.217805 |  1.217805 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 | 0.003629 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  87.162066 |  0.904285 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 | 18.8 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.294457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |   1.48087 |      0.0 |      0.0 |      0.0 |  0.0 |   1.48087 |   1.48087 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802994 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802994 | 0.009294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 102.486739 |  1.582162 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 | 18.8 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.922357 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.761525 |      0.0 |      0.0 |      0.0 |  0.0 |  1.761525 |  1.761525 |      0.0 |      0.0 |      0.0 |      0.0 |  1.191401 |      0.0 |      0.0 |      0.0 |      0.0 |  1.191401 | 0.013789 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 116.902858 |  2.152285 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 | 18.8 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.397171 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.040918 |      0.0 |      0.0 |      0.0 |  0.0 |  2.040918 |  2.040918 |      0.0 |      0.0 |      0.0 |      0.0 |   1.52923 |      0.0 |      0.0 |      0.0 |      0.0 |   1.52923 | 0.017699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 129.564768 |  2.663974 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 | 18.8 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.917105 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.300286 | 0.101438 |      0.0 | 0.010165 |  0.0 |  2.300286 |  2.300286 |      0.0 | 0.101438 |      0.0 | 0.010165 |  1.834649 |      0.0 | 0.014384 |      0.0 | 0.000018 |  1.849051 | 0.021401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 141.335775 |  3.129611 |      0.0 | 0.087054 |      0.0 | 0.010147 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 | 18.8 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.913569 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.554694 | 0.110654 |      0.0 | 0.021936 |  0.0 |  2.554694 |  2.554694 |      0.0 | 0.110654 |      0.0 | 0.021936 |  2.117313 |      0.0 | 0.039153 |      0.0 | 0.000076 |  2.156542 |  0.02496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 152.834923 |  3.566992 |      0.0 | 0.158555 |      0.0 | 0.032007 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 | 18.8 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.551444 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.817235 | 0.119656 |      0.0 | 0.033435 |  0.0 |  2.817235 |  2.817235 |      0.0 | 0.119656 |      0.0 | 0.033435 |  2.388893 |      0.0 |   0.0597 |      0.0 | 0.000175 |  2.448767 | 0.028342 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 163.413153 |  3.995333 |      0.0 | 0.218512 |      0.0 | 0.065267 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 | 18.8 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 |  3.76789 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.072622 | 0.127938 |      0.0 | 0.044013 |  0.0 |  3.072622 |  3.072622 |      0.0 | 0.127938 |      0.0 | 0.044013 |  2.654416 |      0.0 | 0.077033 |      0.0 | 0.000313 |  2.731761 | 0.031618 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 174.500689 |   4.41354 |      0.0 | 0.269417 |      0.0 | 0.108967 |   0.0 | 0.031618 |
    | 1997-09-08 | 20.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 | 18.8 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.120968 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.356609 | 0.136619 |      0.0 | 0.055101 |  0.0 |  3.356609 |  3.356609 |      0.0 | 0.136619 |      0.0 | 0.055101 |  2.922574 |      0.0 | 0.091984 |      0.0 | 0.000489 |  3.015047 | 0.034896 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 184.931393 |  4.847575 |      0.0 | 0.314052 |      0.0 | 0.163579 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 | 18.8 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.450324 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.641281 | 0.144785 |      0.0 | 0.065531 |  0.0 |  3.641281 |  3.641281 |      0.0 | 0.144785 |      0.0 | 0.065531 |  3.198291 |      0.0 | 0.105171 |      0.0 | 0.000704 |  3.304166 | 0.038243 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 194.729472 |  5.290565 |      0.0 | 0.353666 |      0.0 | 0.228407 |   0.0 | 0.038243 |
    | 1997-11-08 | 20.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 | 18.8 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.992996 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.926597 | 0.152456 |      0.0 | 0.075329 |  0.0 |  3.926597 |  3.926597 |      0.0 | 0.152456 |      0.0 | 0.075329 |   3.47835 |      0.0 | 0.116936 |      0.0 | 0.000953 |  3.596239 | 0.041623 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 203.682094 |  5.738812 |      0.0 | 0.389186 |      0.0 | 0.302783 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 | 18.8 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.979765 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.204896 | 0.159465 |  0.00956 | 0.084282 |  0.0 |  4.204896 |  4.204896 |      0.0 | 0.159465 |  0.00956 | 0.084282 |  3.759055 |      0.0 | 0.127503 | 0.001356 | 0.001236 |  3.889149 | 0.045013 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 212.344126 |  6.184653 |      0.0 | 0.421149 | 0.008204 | 0.385829 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 | 18.8 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.198392 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.492791 | 0.166247 | 0.045997 | 0.092944 |  0.0 |  4.492791 |  4.492791 |      0.0 | 0.166247 | 0.045997 | 0.092944 |  4.041108 |      0.0 | 0.137079 | 0.008733 | 0.001549 |  4.188469 | 0.048478 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 221.447755 |  6.636336 |      0.0 | 0.450317 | 0.045468 | 0.477225 |   0.0 | 0.048478 |
    | 1997-14-08 | 20.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 | 18.8 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.701991 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.818593 | 0.173374 | 0.100359 | 0.102048 |  0.0 |  4.818593 |  4.818593 |      0.0 | 0.173374 | 0.100359 | 0.102048 |  4.335644 |      0.0 | 0.145951 | 0.026485 | 0.001893 |  4.509972 | 0.052199 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  230.65139 |  7.119285 |      0.0 | 0.477741 | 0.119342 |  0.57738 |   0.0 | 0.052199 |
    | 1997-15-08 | 20.0 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 | 18.8 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.155199 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.176976 |  0.18058 | 0.168029 | 0.111251 |  0.0 |  5.176976 |  5.176976 |      0.0 |  0.18058 | 0.168029 | 0.111251 |  4.653151 |      0.0 | 0.154363 |  0.05599 | 0.002268 |  4.865773 | 0.056317 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 239.959354 |  7.643111 |      0.0 | 0.503957 |  0.23138 | 0.686363 |   0.0 | 0.056317 |
    | 1997-16-08 | 20.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 | 18.8 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.708353 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.575883 | 0.187867 | 0.247357 | 0.120559 |  0.0 |  5.575883 |  5.575883 |      0.0 | 0.187867 | 0.247357 | 0.120559 |  5.000154 |      0.0 | 0.162462 | 0.097435 | 0.002675 |  5.262727 | 0.060911 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 249.219334 |  8.218839 |      0.0 | 0.529362 | 0.381303 | 0.804247 |   0.0 | 0.060911 |
    | 1997-17-08 | 20.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 | 18.8 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.378596 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.018762 | 0.195117 | 0.335816 | 0.129819 |  0.0 |  6.018762 |  6.018762 |      0.0 | 0.195117 | 0.335816 | 0.129819 |  5.382685 |      0.0 | 0.170337 | 0.150384 | 0.003114 |  5.706521 | 0.066048 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 257.261224 |  8.854915 |      0.0 | 0.554142 | 0.566735 | 0.930953 |   0.0 | 0.066048 |
    | 1997-18-08 | 20.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 | 18.8 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.779721 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.451575 | 0.201413 | 0.419628 | 0.137861 |  0.0 |  6.451575 |  6.451575 |      0.0 | 0.201413 | 0.419628 | 0.137861 |  5.790771 |      0.0 | 0.177909 | 0.212245 | 0.003582 |  6.184507 |  0.07158 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 264.371025 |   9.51572 |      0.0 | 0.577646 | 0.774118 | 1.065232 |   0.0 |  0.07158 |
    | 1997-19-08 | 20.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 | 18.8 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.628366 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.883257 | 0.206979 | 0.498739 | 0.144971 |  0.0 |  6.883257 |  6.883257 |      0.0 | 0.206979 | 0.498739 | 0.144971 |  6.210098 |      0.0 | 0.185033 | 0.279355 | 0.004076 |  6.678562 | 0.077298 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 271.108713 | 10.188879 |      0.0 | 0.599593 | 0.993502 | 1.206126 |   0.0 | 0.077298 |
    | 1997-20-08 | 20.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 | 18.8 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.624935 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.348961 | 0.212254 | 0.577799 | 0.151709 |  0.0 |  7.348961 |  7.348961 |      0.0 | 0.212254 | 0.577799 | 0.151709 |  6.643946 |      0.0 | 0.191696 | 0.349693 | 0.004593 |  7.189927 | 0.083217 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 276.293054 | 10.893894 |      0.0 | 0.620152 | 1.221608 | 1.353242 |   0.0 | 0.083217 |
    | 1997-21-08 | 20.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 | 18.8 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.756903 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.757566 | 0.216313 | 0.641218 | 0.156893 |  0.0 |  7.757566 |  7.757566 |      0.0 | 0.216313 | 0.641218 | 0.156893 |  7.077951 |      0.0 | 0.197812 | 0.420163 | 0.005129 |  7.701056 | 0.089133 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 280.864161 | 11.573509 |      0.0 | 0.638653 | 1.442663 | 1.505006 |   0.0 | 0.089133 |
    | 1997-22-08 | 20.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 | 18.8 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.361124 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.166722 | 0.219892 | 0.698934 | 0.161464 |  0.0 |  8.166722 |  8.166722 |      0.0 | 0.219892 | 0.698934 | 0.161464 |  7.500252 |      0.0 | 0.203306 | 0.487924 | 0.005681 |  8.197163 | 0.094875 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 288.356024 |  12.23998 |      0.0 |  0.65524 | 1.653673 | 1.660789 |   0.0 | 0.094875 |
    | 1997-23-08 | 20.0 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 | 18.8 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.097454 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.984347 | 0.225758 | 0.797042 | 0.168956 |  0.0 |  8.984347 |  8.984347 |      0.0 | 0.225758 | 0.797042 | 0.168956 |  8.021581 |      0.0 | 0.208608 | 0.558706 | 0.006253 |  8.795147 | 0.101796 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 294.182468 | 13.202746 |      0.0 |  0.67239 | 1.892009 | 1.823492 |   0.0 | 0.101796 |
    | 1997-24-08 | 20.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 | 18.8 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.649811 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  9.835059 | 0.230319 | 0.876255 | 0.174782 |  0.0 |  9.835059 |  9.835059 |      0.0 | 0.230319 | 0.876255 | 0.174782 |  8.689617 |      0.0 | 0.213877 | 0.634173 | 0.006846 |  9.544513 | 0.110469 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 | 297.516241 | 14.348187 |      0.0 | 0.688832 | 2.134091 | 1.991428 |   0.0 | 0.110469 |
    | 1997-25-08 | 20.0 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 | 18.8 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.895865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 10.483053 | 0.232929 | 0.922693 | 0.178116 |  0.0 | 10.483053 | 10.483053 |      0.0 | 0.232929 | 0.922693 | 0.178116 |  9.390662 |      0.0 | 0.218678 | 0.706002 | 0.007454 | 10.322796 | 0.119477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 300.903585 | 15.440578 |      0.0 | 0.703083 | 2.350782 |  2.16209 |   0.0 | 0.119477 |
    | 1997-26-08 | 20.0 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 | 18.8 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.071449 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 11.394349 | 0.235581 | 0.970688 | 0.181504 |  0.0 | 11.394349 | 11.394349 |      0.0 | 0.235581 | 0.970688 | 0.181504 | 10.134751 |      0.0 | 0.222895 | 0.771208 | 0.008071 | 11.136926 |   0.1289 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 304.150015 | 16.700175 |      0.0 | 0.715769 | 2.550261 | 2.335523 |   0.0 |   0.1289 |
    | 1997-27-08 | 20.0 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 | 18.8 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.108859 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 13.250015 | 0.238123 | 1.017441 |  0.18475 |  0.0 | 13.250015 |      12.0 | 1.250015 | 0.238123 | 1.017441 |  0.18475 | 10.878137 | 0.792807 | 0.226675 |   0.8316 | 0.008698 | 12.737917 |  0.14743 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 305.450827 | 17.822039 | 0.457208 | 0.727217 | 2.736102 | 2.511574 |   0.0 |  0.14743 |
    | 1997-28-08 | 20.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 | 18.8 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 |  1.33161 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.550827 | 0.239141 |  1.03638 | 0.186051 |  0.0 | 14.550827 |      12.0 | 2.550827 | 0.239141 |  1.03638 | 0.186051 | 11.401361 | 2.037971 | 0.229904 | 0.884372 | 0.009332 |  14.56294 | 0.168553 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 306.206818 | 18.420678 | 0.970064 | 0.736454 | 2.888109 | 2.688293 |   0.0 | 0.168553 |
    | 1997-29-08 | 20.0 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 | 18.8 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 0.881773 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.306818 | 0.239733 | 1.047439 | 0.186807 |  0.0 | 15.306818 |      12.0 | 3.306818 | 0.239733 | 1.047439 | 0.186807 |  11.68056 | 2.988724 | 0.232478 | 0.926908 | 0.009966 | 15.838636 | 0.183318 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 306.644248 | 18.740118 | 1.288158 |  0.74371 |  3.00864 | 2.865134 |   0.0 | 0.183318 |
    | 1997-30-08 | 20.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 | 18.8 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.913351 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.744248 | 0.240076 | 1.053857 | 0.187244 |  0.0 | 15.744248 |      12.0 | 3.744248 | 0.240076 | 1.053857 | 0.187244 | 11.829543 | 3.558462 | 0.234482 | 0.960303 |   0.0106 | 16.593391 | 0.192053 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 305.605473 | 18.910575 | 1.473943 | 0.749303 | 3.102193 | 3.041778 |   0.0 | 0.192053 |
    | 1997-31-08 | 20.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 | 18.8 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.882123 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.705473 | 0.239262 | 1.038639 | 0.186205 |  0.0 | 14.705473 |      12.0 | 2.705473 | 0.239262 | 1.038639 | 0.186205 | 11.909042 | 3.070354 | 0.235874 | 0.983359 | 0.011231 | 16.209861 | 0.187614 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 305.653771 | 19.001532 | 1.109062 | 0.752692 | 3.157473 | 3.216752 |   0.0 | 0.187614 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_acker_heavy_garto_daily:

acre (GARTO)
------------

One can pep up |lland_knauf| with submodels following the |SoilModel_V1| interface for
the reasons discussed in the :ref:`lland_dd_acker_garto` example of application model
|lland_dd|.  As in the :ref:`lland_dd_acker_garto` example, we prepare a
|ga_garto_submodel1| model and assume loam soil for illustration. for illustration.
We set the soil depth in agreement with the maximum soil water content (309.0 mm) and
the initial relative soil moisture in agreement with the initial water content (72 mm)
of the previous :ref:`lland_knauf_acker_heavy_rain_daily` example:

>>> from hydpy import pub
>>> with model.add_soilmodel_v1("ga_garto_submodel1"):
...     nmbbins(4)
...     with pub.options.parameterstep("1m"):
...         dt(1.0)
...     sealed(False)
...     soildepth(309.0 / 0.434)
...     residualmoisture(0.027)
...     saturationmoisture(0.434)
...     saturatedconductivity(13.2)
...     poresizedistribution(0.252)
...     airentrypotential(111.5)
...     states.moisture = 72.0 / 309.0 * 0.434
...     states.frontdepth = 0.0
...     states.moisturechange = 0.0

When comparing the |lland_dd| examples :ref:`lland_dd_acker_summer` and
:ref:`lland_dd_acker_garto`, we see an increase in direct runoff generation due to
including infiltration excess when using |ga_garto_submodel1|.  However, comparing the
following results with the :ref:`lland_knauf_acker_heavy_rain_daily` example, we see a
decrease in direct runoff generation.  This behaviour results from the lower rainfall
rates, which never exceed the soil's (saturated) conductivity.  Hence,
|ga_garto_submodel1| always lets the surface water supplied by |lland_knauf| infiltrate
and creates no additional direct runoff component.  But, starting from August 13, it
calculates extra groundwater recharge.  At this time, the (partly saturated) wetting
front reaches the soil's bottom and enables the continuous percolation of all rainfall
through the soil column:

.. integration-test::

    >>> test("lland_knauf_acker_garto_daily",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa)
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada |      qdb |     qib1 |     qib2 |       qbb | qkap |     qdgz |    qdgz1 | qdgz2 |    qigz1 |    qigz2 |      qbgz |    qdga1 | qdga2 |    qiga1 |    qiga2 |     qbga |      qah |       qa | inzp | wats | waes | esnow | taus |       ebdn |       bowa |     sdg1 | sdg2 |     sig1 |     sig2 |        sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 18.8 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.720128 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 1.217805 |      0.0 |      0.0 |  0.000003 |  0.0 | 1.217805 | 1.217805 |   0.0 |      0.0 |      0.0 |  0.000003 |  0.31352 |   0.0 |      0.0 |      0.0 |      0.0 |  0.31352 | 0.003629 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  87.162064 | 0.904285 |  0.0 |      0.0 |      0.0 |   0.000003 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 | 18.8 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.294457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.48087 |      0.0 |      0.0 |  0.000003 |  0.0 |  1.48087 |  1.48087 |   0.0 |      0.0 |      0.0 |  0.000003 | 0.802994 |   0.0 |      0.0 |      0.0 |      0.0 | 0.802994 | 0.009294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 102.486734 | 1.582161 |  0.0 |      0.0 |      0.0 |   0.000005 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 | 18.8 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.922357 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 1.761525 |      0.0 |      0.0 |  0.000003 |  0.0 | 1.761525 | 1.761525 |   0.0 |      0.0 |      0.0 |  0.000003 | 1.191401 |   0.0 |      0.0 |      0.0 |      0.0 | 1.191401 | 0.013789 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 |  116.90285 | 2.152285 |  0.0 |      0.0 |      0.0 |   0.000008 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 | 18.8 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.397171 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.040918 |      0.0 |      0.0 |  0.000003 |  0.0 | 2.040918 | 2.040918 |   0.0 |      0.0 |      0.0 |  0.000003 | 1.529229 |   0.0 |      0.0 |      0.0 |      0.0 | 1.529229 | 0.017699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 129.564759 | 2.663974 |  0.0 |      0.0 |      0.0 |    0.00001 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 | 18.8 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.917105 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.300286 | 0.101438 |      0.0 |  0.010167 |  0.0 | 2.300286 | 2.300286 |   0.0 | 0.101438 |      0.0 |  0.010167 | 1.834649 |   0.0 | 0.014384 |      0.0 | 0.000018 | 1.849051 | 0.021401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 141.335762 | 3.129611 |  0.0 | 0.087054 |      0.0 |   0.010159 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 | 18.8 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.913569 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.554693 | 0.110654 |      0.0 |  0.021938 |  0.0 | 2.554693 | 2.554693 |   0.0 | 0.110654 |      0.0 |  0.021938 | 2.117313 |   0.0 | 0.039153 |      0.0 | 0.000076 | 2.156542 |  0.02496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 152.834908 | 3.566991 |  0.0 | 0.158555 |      0.0 |   0.032022 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 | 18.8 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.551444 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 2.817235 | 0.119656 |      0.0 |  0.033437 |  0.0 | 2.817235 | 2.817235 |   0.0 | 0.119656 |      0.0 |  0.033437 | 2.388893 |   0.0 |   0.0597 |      0.0 | 0.000175 | 2.448767 | 0.028342 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 163.413136 | 3.995333 |  0.0 | 0.218512 |      0.0 |   0.065285 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 | 18.8 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 |  3.76789 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.072622 | 0.127938 |      0.0 |  0.044016 |  0.0 | 3.072622 | 3.072622 |   0.0 | 0.127938 |      0.0 |  0.044016 | 2.654415 |   0.0 | 0.077033 |      0.0 | 0.000313 | 2.731761 | 0.031618 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 |  174.50067 |  4.41354 |  0.0 | 0.269417 |      0.0 |   0.108988 |   0.0 | 0.031618 |
    | 1997-09-08 | 20.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 | 18.8 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.120968 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.356608 | 0.136619 |      0.0 |  0.055103 |  0.0 | 3.356608 | 3.356608 |   0.0 | 0.136619 |      0.0 |  0.055103 | 2.922574 |   0.0 | 0.091984 |      0.0 | 0.000489 | 3.015046 | 0.034896 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 184.931372 | 4.847574 |  0.0 | 0.314052 |      0.0 |   0.163602 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 | 18.8 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.450324 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.64128 | 0.144785 |      0.0 |  0.065534 |  0.0 |  3.64128 |  3.64128 |   0.0 | 0.144785 |      0.0 |  0.065534 |  3.19829 |   0.0 | 0.105171 |      0.0 | 0.000704 | 3.304165 | 0.038243 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 194.729449 | 5.290564 |  0.0 | 0.353666 |      0.0 |   0.228432 |   0.0 | 0.038243 |
    | 1997-11-08 | 20.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 | 18.8 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.992996 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 3.926596 | 0.152456 |      0.0 |  0.075332 |  0.0 | 3.926596 | 3.926596 |   0.0 | 0.152456 |      0.0 |  0.075332 |  3.47835 |   0.0 | 0.116936 |      0.0 | 0.000953 | 3.596238 | 0.041623 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 203.682069 | 5.738811 |  0.0 | 0.389186 |      0.0 |   0.302811 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 | 18.8 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.979765 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.204895 | 0.159465 |  0.00956 |  0.084285 |  0.0 | 4.204895 | 4.204895 |   0.0 | 0.159465 |  0.00956 |  0.084285 | 3.759054 |   0.0 | 0.127503 | 0.001356 | 0.001236 | 3.889148 | 0.045013 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 212.344099 | 6.184652 |  0.0 | 0.421149 | 0.008204 |   0.385859 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 | 18.8 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.198392 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.49279 | 0.166247 | 0.045997 |  1.325619 |  0.0 |  4.49279 |  4.49279 |   0.0 | 0.166247 | 0.045997 |  1.325619 | 4.041107 |   0.0 | 0.137079 | 0.008733 | 0.003758 | 4.190677 | 0.048503 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 220.215055 | 6.636335 |  0.0 | 0.450317 | 0.045468 |    1.70772 |   0.0 | 0.048503 |
    | 1997-14-08 | 20.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 | 18.8 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.701991 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.772928 | 0.172409 | 0.092206 | 10.848312 |  0.0 | 4.772928 | 4.772928 |   0.0 | 0.172409 | 0.092206 | 10.848312 | 4.323887 |   0.0 | 0.145814 | 0.025329 | 0.025563 | 4.520593 | 0.052322 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 | 218.727208 | 7.085376 |  0.0 | 0.476912 | 0.112345 |  12.530469 |   0.0 | 0.052322 |
    | 1997-15-08 | 20.0 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 | 18.8 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.155199 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.718485 | 0.171244 | 0.082679 |  9.337522 |  0.0 | 4.718485 | 4.718485 |   0.0 | 0.171244 | 0.082679 |  9.337522 | 4.519299 |   0.0 | 0.152816 | 0.042002 | 0.061629 | 4.775746 | 0.055275 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 219.362078 | 7.284562 |  0.0 |  0.49534 | 0.153022 |  21.806363 |   0.0 | 0.055275 |
    | 1997-16-08 | 20.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 | 18.8 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.708353 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.741627 | 0.171741 | 0.086702 |  9.982112 |  0.0 | 4.741627 | 4.741627 |   0.0 | 0.171741 | 0.086702 |  9.982112 | 4.618155 |   0.0 | 0.157853 | 0.053536 | 0.096016 |  4.92556 | 0.057009 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 219.771543 | 7.408034 |  0.0 | 0.509228 | 0.186188 |  31.692458 |   0.0 | 0.057009 |
    | 1997-17-08 | 20.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 | 18.8 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.378596 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.756623 | 0.172062 |  0.08933 | 10.397901 |  0.0 | 4.756623 | 4.756623 |   0.0 | 0.172062 |  0.08933 | 10.397901 | 4.679602 |   0.0 | 0.161642 | 0.062847 |  0.13218 |  5.03627 |  0.05829 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 219.077031 | 7.485056 |  0.0 | 0.519648 | 0.212671 |  41.958179 |   0.0 |  0.05829 |
    | 1997-18-08 | 20.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 | 18.8 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.779721 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.731221 | 0.171518 | 0.084888 |  9.692687 |  0.0 | 4.731221 | 4.731221 |   0.0 | 0.171518 | 0.084888 |  9.692687 | 4.708984 |   0.0 | 0.164373 | 0.069355 | 0.167694 | 5.110406 | 0.059148 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 218.716996 | 7.507292 |  0.0 | 0.526794 | 0.228204 |  51.483172 |   0.0 | 0.059148 |
    | 1997-19-08 | 20.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 | 18.8 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.628366 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.718114 | 0.171236 | 0.082615 |  9.327154 |  0.0 | 4.718114 | 4.718114 |   0.0 | 0.171236 | 0.082615 |  9.327154 | 4.715981 |   0.0 | 0.166259 | 0.073219 | 0.201164 | 5.156622 | 0.059683 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 |  218.88951 | 7.509426 |  0.0 | 0.531771 |   0.2376 |  60.609163 |   0.0 | 0.059683 |
    | 1997-20-08 | 20.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 | 18.8 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.624935 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.724389 | 0.171371 | 0.083701 |  9.502299 |  0.0 | 4.724389 | 4.724389 |   0.0 | 0.171371 | 0.083701 |  9.502299 | 4.718591 |   0.0 | 0.167619 | 0.075905 | 0.234173 | 5.196288 | 0.060142 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 217.882815 | 7.515224 |  0.0 | 0.535523 | 0.245396 |  69.877289 |   0.0 | 0.060142 |
    | 1997-21-08 | 20.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 | 18.8 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 4.756903 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.687908 | 0.170583 | 0.077428 |  8.480382 |  0.0 | 4.687908 | 4.687908 |   0.0 | 0.170583 | 0.077428 |  8.480382 | 4.711903 |   0.0 | 0.168519 | 0.077117 | 0.265545 | 5.223084 | 0.060452 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 217.809611 | 7.491228 |  0.0 | 0.537587 | 0.245707 |  78.092126 |   0.0 | 0.060452 |
    | 1997-22-08 | 20.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 | 18.8 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.361124 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.685268 | 0.170526 | 0.076978 |  8.406087 |  0.0 | 4.685268 | 4.685268 |   0.0 | 0.170526 | 0.076978 |  8.406087 | 4.700032 |   0.0 | 0.169067 | 0.077137 | 0.294843 | 5.241079 | 0.060661 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 221.209629 | 7.476464 |  0.0 | 0.539046 | 0.245548 |   86.20337 |   0.0 | 0.060661 |
    | 1997-23-08 | 20.0 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 | 18.8 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.097454 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.809732 | 0.173188 | 0.098766 | 11.858545 |  0.0 | 4.809732 | 4.809732 |   0.0 | 0.173188 | 0.098766 | 11.858545 | 4.725189 |   0.0 | 0.169838 | 0.080183 | 0.330091 | 5.305301 | 0.061404 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 220.271945 | 7.561006 |  0.0 | 0.542396 | 0.264131 |  97.731823 |   0.0 | 0.061404 |
    | 1997-24-08 | 20.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 | 18.8 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 3.649811 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.775024 | 0.172454 | 0.092578 |  10.90609 |  0.0 | 4.775024 | 4.775024 |   0.0 | 0.172454 | 0.092578 |  10.90609 | 4.755683 |   0.0 | 0.170637 | 0.084314 | 0.369686 |  5.38032 | 0.062272 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 | 218.775987 | 7.580347 |  0.0 | 0.544213 | 0.272394 | 108.268228 |   0.0 | 0.062272 |
    | 1997-25-08 | 20.0 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 | 18.8 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.895865 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.720259 | 0.171282 | 0.082986 |  9.387044 |  0.0 | 4.720259 | 4.720259 |   0.0 | 0.171282 | 0.082986 |  9.387044 | 4.750605 |   0.0 |  0.17096 | 0.085181 | 0.404712 | 5.411457 | 0.062633 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 219.618551 | 7.550001 |  0.0 | 0.544536 | 0.270199 |  117.25056 |   0.0 | 0.062633 |
    | 1997-26-08 | 20.0 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 | 18.8 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.071449 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.751014 | 0.171942 | 0.088345 | 10.242541 |  0.0 | 4.751014 | 4.751014 |   0.0 | 0.171942 | 0.088345 | 10.242541 | 4.744369 |   0.0 | 0.171141 | 0.085349 | 0.438425 | 5.439285 | 0.062955 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 |  220.39326 | 7.556645 |  0.0 | 0.545337 | 0.273195 | 127.054676 |   0.0 | 0.062955 |
    | 1997-27-08 | 20.0 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 | 18.8 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 2.108859 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.779498 | 0.172549 | 0.093371 | 11.029304 |  0.0 | 4.779498 | 4.779498 |   0.0 | 0.172549 | 0.093371 | 11.029304 | 4.754801 |   0.0 | 0.171443 | 0.086869 |  0.47496 | 5.488074 | 0.063519 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 |  220.30968 | 7.581342 |  0.0 | 0.546443 | 0.279696 |  137.60902 |   0.0 | 0.063519 |
    | 1997-28-08 | 20.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 | 18.8 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 |  1.33161 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.776415 | 0.172483 | 0.092824 | 10.944416 |  0.0 | 4.776415 | 4.776415 |   0.0 | 0.172483 | 0.092824 | 10.944416 | 4.765526 |   0.0 | 0.171731 | 0.088544 | 0.512621 | 5.538422 | 0.064102 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 221.091931 | 7.592231 |  0.0 | 0.547195 | 0.283976 | 148.040815 |   0.0 | 0.064102 |
    | 1997-29-08 | 20.0 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 | 18.8 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 0.881773 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.805359 | 0.173096 | 0.097982 | 11.738981 |  0.0 | 4.805359 | 4.805359 |   0.0 | 0.173096 | 0.097982 | 11.738981 | 4.778056 |   0.0 | 0.172021 | 0.090429 | 0.551418 | 5.591924 | 0.064721 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 221.494741 | 7.619534 |  0.0 |  0.54827 | 0.291529 | 159.228378 |   0.0 | 0.064721 |
    | 1997-30-08 | 20.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 | 18.8 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.913351 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.820344 | 0.173411 | 0.100674 | 12.148195 |  0.0 | 4.820344 | 4.820344 |   0.0 | 0.173411 | 0.100674 | 12.148195 | 4.794648 |   0.0 | 0.172355 | 0.092846 | 0.592233 | 5.652082 | 0.065418 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 220.438767 |  7.64523 |  0.0 | 0.549325 | 0.299357 |  170.78434 |   0.0 | 0.065418 |
    | 1997-31-08 | 20.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 | 18.8 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                       0.0 |                      0.0 |              0.0 | 18.1 |  0.0 | 0.7 | 1.882123 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 4.781177 | 0.172584 | 0.093669 | 11.075523 |  0.0 | 4.781177 | 4.781177 |   0.0 | 0.172584 | 0.093669 | 11.075523 | 4.796549 |   0.0 | 0.172523 | 0.093963 | 0.631711 | 5.694745 | 0.065911 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 220.533691 | 7.629859 |  0.0 | 0.549387 | 0.299063 | 181.228153 |   0.0 | 0.065911 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> del model.soilmodel
>>> inputs.nied.series = nied

.. _lland_knauf_wasser_daily:

water
-----

|lland.DOCNAME.long| defines three types of water areas, |WASSER|, |FLUSS|, and |SEE|.
|lland_knauf| calculates their evaporation based on the same submodel interface as
|lland_dd|.  So, please read the detailed discussions in integration test
:ref:`lland_dd_wasser` to better understand the following test results for water type
|WASSER|:

.. integration-test::

    >>> lnk(WASSER)
    >>> negq(False)
    >>> model.aetmodel.parameters.control.surfaceresistance.wasser_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.wasser_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.wasser_aug = 0.7
    >>> conditions_wasser = test(
    ...     "lland_knauf_wasser_daily",
    ...     axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 11.424803 | 0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.008458 | 0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.073812 | 0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.282 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.792904 |  0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   5.01244 | 0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.305141 | 0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.040224 | 0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_wasser_routing_daily:

water (routing)
---------------

The following calculation shows the possibility of subtracting evaporation from
inflowing runoff, discussed in the integration test :ref:`lland_dd_wasser_routing` of
|lland_dd|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_knauf_wasser_routing_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |   qz |   qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.02 | 1.728 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.152803 | 0.152232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.152232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.02 | 1.728 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.95655 | 0.011071 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.011071 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.02 | 1.728 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.635635 | 0.007357 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.007357 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.02 | 1.728 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.256106 | 0.002964 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002964 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.02 | 1.728 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.02 | 1.728 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.02 | 1.728 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.02 | 1.728 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.02 | 1.728 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.02 | 1.728 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.02 | 1.728 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.822 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.02 | 1.728 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.02 | 1.728 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.055264 |  0.00064 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.00064 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.02 | 1.728 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  2.736458 | 0.031672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.031672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.02 | 1.728 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.444099 |  0.00514 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.00514 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.02 | 1.728 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.217257 | 0.002515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002515 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.02 | 1.728 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.133428 | 0.001544 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.001544 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.02 | 1.728 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.186433 | 0.002158 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.002158 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.02 | 1.728 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.02 | 1.728 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.02 | 1.728 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  2.801812 | 0.032428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.032428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.02 | 1.728 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.92579 | 0.010715 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.010715 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.02 | 1.728 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.055768 | 0.000645 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.000645 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.02 | 1.728 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.447898 | 0.005184 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.005184 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.02 | 1.728 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 10.520904 |  0.12177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 |  0.12177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.02 | 1.728 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   6.74044 | 0.078014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.078014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.02 | 1.728 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 15.033141 | 0.173995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.173995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.02 | 1.728 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  9.768224 | 0.113058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.113058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.02 | 1.728 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.82735 | 0.009576 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.009576 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.02 | 1.728 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.815922 | 0.009444 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |  0.02 | 0.009444 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_wasser_negq_daily:

water (negative runoff)
-----------------------

The following calculation shows the possibility of calculating negative discharge
values, discussed in the integration test :ref:`lland_dd_wasser_negq` of |lland_dd|:

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_knauf_wasser_negq_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |       qah |        qa | inzp | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |    outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 11.424803 |  0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.77145 | -0.008929 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.008929 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.092365 | -0.012643 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.012643 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.471894 | -0.017036 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017036 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.729733 |  -0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.02002 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.826087 | -0.021135 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021135 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -2.041956 | -0.023634 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.023634 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.800053 | -0.020834 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.020834 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.82186 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -1.82186 | -0.021086 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021086 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.891469 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.891469 | -0.021892 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021892 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.106575 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -2.106575 | -0.024382 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.024382 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.036929 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.942929 | -0.022488 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.022488 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.808559 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.808559 | -0.020932 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.020932 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.672736 |  -0.01936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.01936 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.008458 |  0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.283901 |  -0.01486 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  -0.01486 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.510743 | -0.017485 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017485 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.594572 | -0.018456 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.018456 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.541567 | -0.017842 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.017842 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.939383 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.939383 | -0.022447 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.022447 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.826811 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.826811 | -0.021144 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.021144 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  1.073812 |  0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.80221 | -0.009285 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.009285 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.672232 | -0.019355 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.019355 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -1.280102 | -0.014816 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.014816 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.792904 |   0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   5.01244 |  0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 13.305141 |  0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  8.040224 |  0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  -0.90065 | -0.010424 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.010424 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | -0.912078 | -0.010556 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | -0.010556 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_see_daily:

lakes
-----

The following example focuses on water-type |SEE| (for further information, see
integration test :ref:`lland_dd_see` of |lland_dd|):

.. integration-test::

    >>> lnk(SEE)
    >>> model.aetmodel.parameters.control.surfaceresistance.see_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.see_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.see_aug = 0.7
    >>> negq(False)
    >>> test("lland_knauf_see_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 |        sbg | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 11.424803 |   0.0 |   0.0 |   0.0 |   0.0 |  0.020478 | 0.020478 | 0.000237 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  11.404325 |   0.0 | 0.000237 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.77145 |   0.0 |   0.0 |   0.0 |   0.0 |  0.039475 | 0.039475 | 0.000457 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  10.593401 |   0.0 | 0.000457 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.092365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035994 | 0.035994 | 0.000417 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   9.465042 |   0.0 | 0.000417 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.471894 |   0.0 |   0.0 |   0.0 |   0.0 |  0.031272 | 0.031272 | 0.000362 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   7.961876 |   0.0 | 0.000362 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.729733 |   0.0 |   0.0 |   0.0 |   0.0 |  0.025424 | 0.025424 | 0.000294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   6.206719 |   0.0 | 0.000294 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826087 |   0.0 |   0.0 |   0.0 |   0.0 |  0.018963 | 0.018963 | 0.000219 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   4.361669 |   0.0 | 0.000219 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.041956 |   0.0 |   0.0 |   0.0 |   0.0 |  0.011966 | 0.011966 | 0.000138 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   2.307747 |   0.0 | 0.000138 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.800053 |   0.0 |   0.0 |   0.0 |   0.0 |  0.005041 | 0.005041 | 0.000058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.502653 |   0.0 | 0.000058 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.820396 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -1.82186 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001465 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -1.317743 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.883357 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.891469 |   0.0 |   0.0 |   0.0 |   0.0 | -0.008111 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |    -3.2011 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.091331 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.106575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.015244 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -5.292431 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 2.014486 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.942929 |   0.0 |   0.0 |   0.0 |   0.0 | -0.022443 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -7.212917 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.779476 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.808559 |   0.0 |   0.0 |   0.0 |   0.0 | -0.029083 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -8.992393 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.637521 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672736 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035215 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -10.629914 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.493266 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.008458 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036276 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   -9.58518 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.247259 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.283901 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036642 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -10.832439 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.469226 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.510743 |   0.0 |   0.0 |   0.0 |   0.0 | -0.041517 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -12.301666 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.547641 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.594572 |   0.0 |   0.0 |   0.0 |   0.0 | -0.046931 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -13.849307 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.489186 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.541567 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.05238 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -15.338493 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.880954 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.939383 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058428 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -17.219448 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.761846 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826811 |   0.0 |   0.0 |   0.0 |   0.0 | -0.064965 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -18.981293 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.740109 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.073812 |   0.0 |   0.0 |   0.0 |   0.0 | -0.066078 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -17.841402 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.018853 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.80221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.065357 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -18.578255 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.602675 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672232 |   0.0 |   0.0 |   0.0 |   0.0 | -0.069557 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -20.180931 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.299507 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.280102 |   0.0 |   0.0 |   0.0 |   0.0 | -0.074595 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -21.386437 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.016237 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.792904 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06086 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | -12.532674 |   0.0 |      0.0 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 1.061645 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   5.01244 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035916 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  -7.484319 |   0.0 |      0.0 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.791893 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 13.305141 |   0.0 |   0.0 |   0.0 |   0.0 | -0.002966 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   5.823788 |   0.0 |      0.0 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.040224 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035276 | 0.035276 | 0.000408 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  13.828736 |   0.0 | 0.000408 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.90065 |   0.0 |   0.0 |   0.0 |   0.0 |  0.047929 | 0.047929 | 0.000555 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  12.880157 |   0.0 | 0.000555 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.912078 |   0.0 |   0.0 |   0.0 |   0.0 |   0.04451 |  0.04451 | 0.000515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  11.923568 |   0.0 | 0.000515 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_fluss_daily:

streams
-------

The following example focuses on water-type |FLUSS| (for further information, see
integration test :ref:`lland_dd_fluss` of |lland_dd|):

.. integration-test::

    >>> lnk(FLUSS)
    >>> model.aetmodel.parameters.control.surfaceresistance.fluss_aug = 0.0
    >>> model.aetmodel.parameters.control.cropheight.fluss_aug = 0.05
    >>> model.aetmodel.parameters.control.albedo.fluss_aug = 0.7
    >>> test("lland_knauf_fluss_daily",
    ...       axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |       evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |     qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa |      sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.077197 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 11.424803 | 11.424803 |      0.0 |   0.0 |   0.0 |  0.0 |  2.941276 |      0.0 |   0.0 |   0.0 |  0.0 | 2.941276 | 0.034043 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.483526 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034043 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |   0.86545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.77145 |  -0.77145 |      0.0 |   0.0 |   0.0 |  0.0 |  3.758016 |      0.0 |   0.0 |   0.0 |  0.0 | 3.758016 | 0.043496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  3.954061 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043496 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.092365 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.092365 | -1.092365 |      0.0 |   0.0 |   0.0 |  0.0 |  1.562905 |      0.0 |   0.0 |   0.0 |  0.0 | 1.562905 | 0.018089 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.298791 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.018089 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.565894 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.471894 | -1.471894 |      0.0 |   0.0 |   0.0 |  0.0 |  0.226808 |      0.0 |   0.0 |   0.0 |  0.0 | 0.226808 | 0.002625 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.39991 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002625 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.097905 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.729733 | -1.729733 |      0.0 |   0.0 |   0.0 |  0.0 | -0.631827 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.497815 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.657403 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826087 | -1.826087 |      0.0 |   0.0 |   0.0 |  0.0 | -1.168684 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.155218 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.511091 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.041956 | -2.041956 |      0.0 |   0.0 |   0.0 |  0.0 | -1.530864 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.666309 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.093098 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.800053 | -1.800053 |      0.0 |   0.0 |   0.0 |  0.0 | -1.706955 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.759408 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.065871 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -1.82186 |  -1.82186 |      0.0 |   0.0 |   0.0 |  0.0 | -1.755989 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.825279 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.086838 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.891469 | -1.891469 |      0.0 |   0.0 |   0.0 |  0.0 | -1.804631 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.912116 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.206065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.106575 | -2.106575 |      0.0 |   0.0 |   0.0 |  0.0 | -1.900509 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.118182 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.082443 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.942929 | -1.942929 |      0.0 |   0.0 |   0.0 |  0.0 | -1.954486 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.106625 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.105944 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.808559 | -1.808559 |      0.0 |   0.0 |   0.0 |  0.0 | -1.914503 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -3.000681 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.157388 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672736 | -1.672736 |      0.0 |   0.0 |   0.0 |  0.0 | -1.830125 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.843292 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.463086 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.008458 |  1.008458 |      0.0 |   0.0 |   0.0 |  0.0 | -1.066456 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.768378 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.595002 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.283901 | -1.283901 |      0.0 |   0.0 |   0.0 |  0.0 | -0.688899 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -1.36338 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.485942 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.510743 | -1.510743 |      0.0 |   0.0 |   0.0 |  0.0 | -1.024801 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.849322 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.321551 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.594572 | -1.594572 |      0.0 |   0.0 |   0.0 |  0.0 |  -1.27302 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.170873 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.132224 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.541567 | -1.541567 |      0.0 |   0.0 |   0.0 |  0.0 | -1.409342 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.303098 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.365956 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.939383 | -1.939383 |      0.0 |   0.0 |   0.0 |  0.0 | -1.573427 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.669054 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.111688 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826811 | -1.826811 |      0.0 |   0.0 |   0.0 |  0.0 | -1.715123 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.780741 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  -0.21427 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.073812 |  1.073812 |      0.0 |   0.0 |   0.0 |  0.0 | -1.020458 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.686471 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.557521 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.80221 |  -0.80221 |      0.0 |   0.0 |   0.0 |  0.0 | -0.526689 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.961992 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.793059 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672232 | -1.672232 |      0.0 |   0.0 |   0.0 |  0.0 | -0.879173 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.755051 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.226008 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.280102 | -1.280102 |      0.0 |   0.0 |   0.0 |  0.0 | -1.148095 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.887059 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  1.077096 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.792904 |  8.792904 |      0.0 |   0.0 |   0.0 |  0.0 |    1.3836 |      0.0 |   0.0 |   0.0 |  0.0 |   1.3836 | 0.016014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  5.522245 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.016014 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |   1.09756 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   5.01244 |   5.01244 |      0.0 |   0.0 |   0.0 |  0.0 |   3.86595 |      0.0 |   0.0 |   0.0 |  0.0 |  3.86595 | 0.044745 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  6.668735 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044745 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.794859 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.305141 |      12.0 | 1.305141 |   0.0 |   0.0 |  0.0 |  6.199583 |  0.82777 |   0.0 |   0.0 |  0.0 | 7.027354 | 0.081335 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 12.469152 | 0.477371 |  0.0 |  0.0 | 0.0 |   0.0 | 0.081335 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.607776 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.040224 |  8.040224 |      0.0 |   0.0 |   0.0 |  0.0 |  7.885403 | 0.438667 |   0.0 |   0.0 |  0.0 |  8.32407 | 0.096343 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 12.623973 | 0.038704 |  0.0 |  0.0 | 0.0 |   0.0 | 0.096343 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |   0.90065 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.90065 |  -0.90065 |      0.0 |   0.0 |   0.0 |  0.0 |  5.655812 | 0.035566 |   0.0 |   0.0 |  0.0 | 5.691378 | 0.065872 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   6.06751 | 0.003138 |  0.0 |  0.0 | 0.0 |   0.0 | 0.065872 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.912078 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.912078 | -0.912078 |      0.0 |   0.0 |   0.0 |  0.0 |  2.595008 | 0.002884 |   0.0 |   0.0 |  0.0 | 2.597892 | 0.030068 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  2.560423 | 0.000254 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030068 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_vers_daily:

sealed surfaces
---------------

As also shown in the integration test :ref:`lland_dd_vers` of |lland_dd|, sealed
surfaces route the not intercepted water to the linear storages for direct discharge
immediately:

.. integration-test::

    >>> lnk(VERS)
    >>> negq(True)
    >>> model.aetmodel.parameters.control.surfaceresistance.vers_aug = 500
    >>> model.aetmodel.parameters.control.cropheight.vers_aug = 5.0
    >>> model.aetmodel.parameters.control.albedo.vers_aug = 0.10
    >>> lai.vers_aug = 10.0
    >>> model.aetmodel.parameters.control.leafareaindex.vers_aug = 10.0
    >>> test("lland_knauf_vers_daily",
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |   nbes | sbes |   evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |    qdb | qib1 | qib2 | qbb | qkap |   qdgz |  qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | wats | waes | esnow | taus |       ebdn | bowa |      sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |             86.2 |       2.4 |                15.690785 |              6.3 | 0.0 | 0.0 |                   6.3 |                     15.690785 |       190.25149 | 12.502 | 16.1 |    1.923819 |           1.923819 |                18.326791 |                          0.0 |            15.797694 | 11.733333 |      -26.2 |                       0.0 |                      0.0 |              0.0 | 10.502 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 10.502 | 10.502 |  0.0 |  0.0 | 0.0 |  0.0 | 10.502 | 10.502 |   0.0 |   0.0 |   0.0 |  0.0 | 2.703704 |      0.0 |   0.0 |   0.0 |  0.0 | 2.703704 | 0.031293 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  0.0 |  7.798296 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031293 |
    | 1997-02-08 |  0.1 | 17.3 |             85.8 |       1.6 |                15.628821 |              1.7 | 0.0 | 0.0 |                   1.7 |                     15.628821 |      127.660782 |  0.094 | 17.3 |    1.282546 |           1.282546 |                19.779139 |                          0.0 |            16.970501 | 18.443733 |     6.8624 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 3.637039 |      0.0 |   0.0 |   0.0 |  0.0 | 3.637039 | 0.042095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 |  0.0 |  4.161257 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.042095 |
    | 1997-03-08 |  0.0 | 18.9 |             82.4 |       1.5 |                15.566134 |              4.5 | 0.0 | 0.0 |                   4.5 |                     15.566134 |      164.401298 |    0.0 | 18.9 |    1.202387 |           1.202387 |                21.871576 |                          0.0 |            18.022179 | 15.632162 | -19.607027 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.940764 |      0.0 |   0.0 |   0.0 |  0.0 | 1.940764 | 0.022463 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 |  0.0 |  2.220493 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.022463 |
    | 1997-04-08 |  0.1 | 17.8 |             77.6 |       2.8 |                15.502754 |             12.4 | 0.0 | 0.0 |                  12.4 |                     15.502754 |      269.020502 |  0.094 | 17.8 |    2.244455 |           2.244455 |                20.413369 |                          0.0 |            15.840774 | 20.443786 |  15.862716 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.035613 |      0.0 |   0.0 |   0.0 |  0.0 | 1.035613 | 0.011986 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 |  0.0 |  1.184879 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011986 |
    | 1997-05-08 |  0.0 | 18.5 |             74.9 |       3.0 |                15.438712 |             13.9 | 0.0 | 0.0 |                  13.9 |                     15.438712 |      287.848667 |    0.0 | 18.5 |    2.404774 |           2.404774 |                21.331125 |                          0.0 |            15.977013 | 15.040124 | -20.759257 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.552615 |      0.0 |   0.0 |   0.0 |  0.0 | 0.552615 | 0.006396 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 |  0.0 |  0.632265 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.006396 |
    | 1997-06-08 |  0.0 | 19.9 |             77.9 |       3.5 |                15.374034 |             13.0 | 0.0 | 0.0 |                  13.0 |                     15.374034 |      274.674263 |    0.0 | 19.9 |    2.805569 |           2.805569 |                23.275348 |                          0.0 |            18.131496 |  20.18359 |   1.701539 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.294881 |      0.0 |   0.0 |   0.0 |  0.0 | 0.294881 | 0.003413 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 |  0.0 |  0.337383 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003413 |
    | 1997-07-08 |  0.0 | 20.8 |             73.5 |       3.8 |                 15.30875 |             13.8 | 0.0 | 0.0 |                  13.8 |                      15.30875 |      284.047892 |    0.0 | 20.8 |    3.046047 |           3.046047 |                24.605348 |                          0.0 |            18.084931 | 18.858347 | -11.649921 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.157352 |      0.0 |   0.0 |   0.0 |  0.0 | 0.157352 | 0.001821 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  0.0 |  0.180032 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001821 |
    | 1997-08-08 |  0.0 | 20.9 |             80.1 |       3.3 |                15.242885 |             12.3 | 0.0 | 0.0 |                  12.3 |                     15.242885 |      262.971263 |    0.0 | 20.9 |    2.645251 |           2.645251 |                 24.75714 |                          0.0 |            19.830469 | 21.378324 |   2.869942 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083965 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083965 | 0.000972 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 |  0.0 |  0.096067 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000972 |
    | 1997-09-08 |  0.0 | 22.3 |             72.9 |       1.8 |                15.176467 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.176467 |      272.256414 |    0.0 | 22.3 |    1.442864 |           1.442864 |                 26.96965 |                          0.0 |            19.660875 |  19.71658 | -15.500518 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044804 |      0.0 |   0.0 |   0.0 |  0.0 | 0.044804 | 0.000519 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 |  0.0 |  0.051262 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000519 |
    | 1997-10-08 |  0.0 | 23.0 |             68.5 |       1.9 |                15.109521 |             12.8 | 0.0 | 0.0 |                  12.8 |                     15.109521 |      267.019218 |    0.0 | 23.0 |    1.523023 |           1.523023 |                28.139149 |                          0.0 |            19.275317 |  23.34553 |   2.073177 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.023908 |      0.0 |   0.0 |   0.0 |  0.0 | 0.023908 | 0.000277 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 |  0.0 |  0.027354 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000277 |
    | 1997-11-08 |  0.0 | 23.1 |             68.6 |       3.0 |                15.042071 |             13.1 | 0.0 | 0.0 |                  13.1 |                     15.042071 |      269.626547 |    0.0 | 23.1 |    2.404774 |           2.404774 |                28.309776 |                          0.0 |            19.420506 | 21.913255 |  -7.120473 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.012758 |      0.0 |   0.0 |   0.0 |  0.0 | 0.012758 | 0.000148 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 |  0.0 |  0.014596 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000148 |
    | 1997-12-08 |  0.1 | 23.2 |             66.0 |       2.5 |                14.974141 |             13.3 | 0.0 | 0.0 |                  13.3 |                     14.974141 |      270.878291 |  0.094 | 23.2 |    2.003978 |           2.003978 |                28.481304 |                          0.0 |             18.79766 | 23.128751 |  -0.427496 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006808 |      0.0 |   0.0 |   0.0 |  0.0 | 0.006808 | 0.000079 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 |  0.0 |  0.007789 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000079 |
    | 1997-13-08 |  0.0 | 23.2 |             69.0 |       1.6 |                14.905755 |             12.7 | 0.0 | 0.0 |                  12.7 |                     14.905755 |      261.697164 |    0.0 | 23.2 |    1.282546 |           1.282546 |                28.481304 |                          0.0 |              19.6521 | 22.416669 |  -4.699983 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003633 |      0.0 |   0.0 |   0.0 |  0.0 | 0.003633 | 0.000042 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 |  0.0 |  0.004156 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000042 |
    | 1997-14-08 |  0.0 | 21.7 |             75.6 |       2.7 |                14.836934 |             10.2 | 0.0 | 0.0 |                  10.2 |                     14.836934 |      227.962116 |    0.0 | 21.7 |    2.164296 |           2.164296 |                 26.00118 |                          0.0 |            19.656892 | 22.935065 |   7.410388 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001938 |      0.0 |   0.0 |   0.0 |  0.0 | 0.001938 | 0.000022 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  0.0 |  0.002218 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000022 |
    | 1997-15-08 |  2.7 | 20.8 |             81.4 |       3.2 |                  14.7677 |              9.4 | 0.0 | 0.0 |                   9.4 |                       14.7677 |      216.356321 |  2.538 | 20.8 |    2.565092 |           2.565092 |                24.605348 |                          0.0 |            20.028753 | 19.965673 |  -5.005962 |                       0.0 |                      0.0 |              0.0 |  0.538 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.538 |  0.538 |  0.0 |  0.0 | 0.0 |  0.0 |  0.538 |  0.538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.139541 |      0.0 |   0.0 |   0.0 |  0.0 | 0.139541 | 0.001615 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 |  0.0 |  0.400677 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001615 |
    | 1997-16-08 |  0.0 | 19.5 |             79.0 |       1.3 |                14.698075 |             10.3 | 0.0 | 0.0 |                  10.3 |                     14.698075 |      226.637904 |    0.0 | 19.5 |    1.042069 |           1.042069 |                22.704676 |                          0.0 |            17.936694 |  20.57219 |   6.433141 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.186871 |      0.0 |   0.0 |   0.0 |  0.0 | 0.186871 | 0.002163 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 |  0.0 |  0.213806 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002163 |
    | 1997-17-08 |  0.0 | 21.0 |             75.9 |       1.9 |                14.628078 |             11.1 | 0.0 | 0.0 |                  11.1 |                     14.628078 |      235.528503 |    0.0 | 21.0 |    1.523023 |           1.523023 |                24.909749 |                          0.0 |            18.906499 | 17.884246 | -18.694526 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.099717 |      0.0 |   0.0 |   0.0 |  0.0 | 0.099717 | 0.001154 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 |  0.0 |  0.114089 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001154 |
    | 1997-18-08 |  0.0 | 21.6 |             70.0 |       2.0 |                14.557728 |             11.0 | 0.0 | 0.0 |                  11.0 |                     14.557728 |      232.849428 |    0.0 | 21.6 |    1.603182 |           1.603182 |                25.842752 |                          0.0 |            18.089927 | 22.433069 |   4.998415 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05321 |      0.0 |   0.0 |   0.0 |  0.0 |  0.05321 | 0.000616 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 |  0.0 |  0.060879 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000616 |
    | 1997-19-08 |  0.0 | 21.1 |             70.3 |       2.6 |                14.487046 |              8.5 | 0.0 | 0.0 |                   8.5 |                     14.487046 |       199.71349 |    0.0 | 21.1 |    2.084137 |           2.084137 |                25.063177 |                          0.0 |            17.619414 | 20.158326 |  -5.650046 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.028393 |      0.0 |   0.0 |   0.0 |  0.0 | 0.028393 | 0.000329 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 |  0.0 |  0.032486 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000329 |
    | 1997-20-08 |  0.0 | 22.4 |             69.0 |       3.6 |                14.416048 |             11.3 | 0.0 | 0.0 |                  11.3 |                     14.416048 |      233.771902 |    0.0 | 22.4 |    2.885728 |           2.885728 |                27.134082 |                          0.0 |            18.722517 | 20.950339 |  -8.697966 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015151 |      0.0 |   0.0 |   0.0 |  0.0 | 0.015151 | 0.000175 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 |  0.0 |  0.017335 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000175 |
    | 1997-21-08 |  0.0 | 22.5 |             62.1 |       2.5 |                14.344752 |             12.4 | 0.0 | 0.0 |                  12.4 |                     14.344752 |      246.121286 |    0.0 | 22.5 |    2.003978 |           2.003978 |                27.299387 |                          0.0 |             16.95292 | 22.620153 |    0.72092 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008085 |      0.0 |   0.0 |   0.0 |  0.0 | 0.008085 | 0.000094 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 |  0.0 |   0.00925 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000094 |
    | 1997-22-08 |  2.0 | 19.6 |             84.5 |       2.1 |                14.273175 |              0.1 | 0.0 | 0.0 |                   0.1 |                     14.273175 |        90.58551 |   1.88 | 19.6 |    1.683342 |           1.683342 |                22.846186 |                          0.0 |            19.305027 | 21.577328 |  11.863971 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |  1.88 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |      0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |  0.00005 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 |  0.0 |  0.004936 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00005 |
    | 1997-23-08 |  0.3 | 20.3 |             83.6 |       1.5 |                14.201333 |              6.7 | 0.0 | 0.0 |                   6.7 |                     14.201333 |      171.944993 |  0.282 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |            19.945708 | 17.325305 | -17.848171 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.282 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002302 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002302 | 0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 |  0.0 |  0.002634 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000027 |
    | 1997-24-08 |  0.0 | 24.1 |             76.5 |       2.2 |                14.129243 |             10.4 | 0.0 | 0.0 |                  10.4 |                     14.129243 |      216.549813 |    0.0 | 24.1 |    1.763501 |           1.763501 |                30.066256 |                          0.0 |            23.000686 | 21.630378 | -14.817732 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001228 |      0.0 |   0.0 |   0.0 |  0.0 | 0.001228 | 0.000014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  0.0 |  0.001405 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000014 |
    | 1997-25-08 |  0.1 | 23.7 |             76.6 |       1.9 |                14.056918 |              6.5 | 0.0 | 0.0 |                   6.5 |                     14.056918 |       166.91683 |  0.094 | 23.7 |    1.523023 |           1.523023 |                29.352596 |                          0.0 |            22.484088 | 25.062685 |   8.176109 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000655 |      0.0 |   0.0 |   0.0 |  0.0 | 0.000655 | 0.000008 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 |  0.0 |   0.00075 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000008 |
    | 1997-26-08 | 10.5 | 21.1 |             84.2 |       2.3 |                13.984374 |              4.9 | 0.0 | 0.0 |                   4.9 |                     13.984374 |      145.985611 |   9.87 | 21.1 |     1.84366 |            1.84366 |                25.063177 |                          0.0 |            21.103195 | 21.872765 |   4.636593 |                       0.0 |                      0.0 |              0.0 |   7.87 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   7.87 |   7.87 |  0.0 |  0.0 | 0.0 |  0.0 |   7.87 |   7.87 |   0.0 |   0.0 |   0.0 |  0.0 | 2.026454 |      0.0 |   0.0 |   0.0 |  0.0 | 2.026454 | 0.023454 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 |  0.0 |  5.844296 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023454 |
    | 1997-27-08 |  6.5 | 20.9 |             85.8 |       2.1 |                13.911624 |              6.6 | 0.0 | 0.0 |                   6.6 |                     13.911624 |       165.54389 |   6.11 | 20.9 |    1.683342 |           1.683342 |                 24.75714 |                          0.0 |            21.241626 | 19.702227 |   -7.18664 |                       0.0 |                      0.0 |              0.0 |   4.11 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   4.11 |   4.11 |  0.0 |  0.0 | 0.0 |  0.0 |   4.11 |   4.11 |   0.0 |   0.0 |   0.0 |  0.0 |  3.78382 |      0.0 |   0.0 |   0.0 |  0.0 |  3.78382 | 0.043794 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 |  0.0 |  6.170475 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043794 |
    | 1997-28-08 | 15.0 | 20.2 |             86.5 |       2.6 |                13.838682 |              0.3 | 0.0 | 0.0 |                   0.3 |                     13.838682 |       87.693346 |   14.1 | 20.2 |    2.084137 |           2.084137 |                23.711531 |                          0.0 |            20.510475 | 20.936779 |   4.420674 |                       0.0 |                      0.0 |              0.0 |   12.1 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   12.1 |   12.1 |  0.0 |  0.0 | 0.0 |  0.0 |   12.1 |   12.0 |   0.1 |   0.0 |   0.0 |  0.0 | 5.967201 | 0.063424 |   0.0 |   0.0 |  0.0 | 6.030624 | 0.069799 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 |  0.0 | 12.203275 | 0.036576 |  0.0 |  0.0 | 0.0 |   0.0 | 0.069799 |
    | 1997-29-08 |  9.2 | 16.1 |             89.6 |       2.6 |                13.765561 |              0.1 | 0.0 | 0.0 |                   0.1 |                     13.765561 |       84.356885 |  8.648 | 16.1 |    2.084137 |           2.084137 |                18.326791 |                          0.0 |            16.420805 | 18.828425 |   16.37055 |                       0.0 |                      0.0 |              0.0 |  6.648 |  0.0 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  6.648 |  6.648 |  0.0 |  0.0 | 0.0 |  0.0 |  6.648 |  6.648 |   0.0 |   0.0 |   0.0 |  0.0 | 7.402977 | 0.033611 |   0.0 |   0.0 |  0.0 | 7.436588 | 0.086072 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 |  0.0 | 11.448297 | 0.002965 |  0.0 |  0.0 | 0.0 |   0.0 | 0.086072 |
    | 1997-30-08 |  0.0 | 17.0 |             78.3 |       2.1 |                13.692273 |              5.0 | 0.0 | 0.0 |                   5.0 |                     13.692273 |      142.309202 |    0.0 | 17.0 |    1.683342 |           1.683342 |                19.406929 |                          0.0 |            15.195625 | 13.278507 |  -22.32896 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 5.339359 | 0.002725 |   0.0 |   0.0 |  0.0 | 5.342084 |  0.06183 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 |  0.0 |  6.108938 |  0.00024 |  0.0 |  0.0 | 0.0 |   0.0 |  0.06183 |
    | 1997-31-08 |  0.0 | 18.2 |             78.5 |       2.1 |                13.618831 |              3.8 | 0.0 | 0.0 |                   3.8 |                     13.618831 |      126.722078 |    0.0 | 18.2 |    1.683342 |           1.683342 |                20.933479 |                          0.0 |            16.432781 | 18.874047 |   4.044283 |                       0.0 |                      0.0 |              0.0 |    0.0 |  0.0 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849141 | 0.000221 |   0.0 |   0.0 |  0.0 | 2.849362 | 0.032979 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 |  0.0 |  3.259797 | 0.000019 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032979 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_acker_winter_daily:

acre (winter)
-------------

None of the examples discussed so far dealt with snow processes.  Thus, we now switch
from a warm August to a snowy December and adapt the initial conditions and the input
time series accordingly (those of global radiation and the astronomically possible
sunshine duration stem from the example
:ref:`meteo_glob_morsim_daily_simulation_winter` of |meteo_glob_morsim|):

>>> pub.timegrids = "2010-12-01", "2011-01-01", "1d"
>>> lnk(ACKER)
>>> model.aetmodel.parameters.control.cropheight.acker_dec = 0.05
>>> model.aetmodel.parameters.control.albedo.acker_dec = 0.1
>>> lai.acker_dec = 0.3
>>> model.aetmodel.parameters.control.leafareaindex.acker_dec = 0.3
>>> wg2z.dec = 8.6
>>> model.aetmodel.parameters.control.averagesoilheatflux.dec = -8.6
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

>>> test.inits = ((states.inzp, 0.0),
...               (states.wats, 0.0),
...               (states.waes, 0.0),
...               (states.esnow, 0.0),
...               (states.taus, 0.0),
...               (states.ebdn, 0.0),
...               (states.bowa, 72.0),
...               (states.sdg1, 0.0),
...               (states.sdg2, 0.0),
...               (states.sig1, 0.0),
...               (states.sig2, 0.0),
...               (states.sbg, 0.0))

>>> inputs.nied.series = (
...     0.0, 11.1, 0.4, 1.2, 6.4, 0.0, 0.0, 0.0, 0.7, 3.3, 9.4, 0.2, 2.9, 0.0, 0.0,
...     6.7, 0.0, 0.1, 0.0, 0.2, 0.2, 2.4, 7.0, 0.2, 0.0, 1.2, 0.9, 0.0, 0.0, 0.2, 0.9)
>>> inputs.teml.series = (
...     -5.8, -3.0, -5.6, -3.8, 0.2, -1.1, -2.9, -2.9, -2.1, -3.1, 5.1, 0.2, -3.5,
...     -3.9, -6.3, -3.5, -7.3, -5.8, -7.9, -5.7, -5.6, -4.8, -1.1, -2.9, -9.1, -8.6,
...     -5.3, -7.7, -9.2, -7.4, 1.7)
>>> inputs.windspeed.series = (
...     6.9, 5.2, 2.0, 4.3, 4.0, 2.4, 1.9, 1.8, 2.7, 3.0, 7.4, 5.7, 2.2, 3.1, 2.8, 3.7,
...     0.9, 1.3, 1.9, 2.4, 2.3, 4.6, 8.6, 7.6, 3.8, 1.5, 1.3, 3.0, 2.7, 1.4, 4.2)
>>> inputs.relativehumidity.series = (
...     70.3, 85.0, 95.9, 90.6, 96.5, 99.0, 99.2, 99.1, 96.9, 95.0, 90.6, 78.7, 90.5,
...     88.1, 92.8, 98.1, 96.6, 97.7, 95.8, 97.5, 95.0, 93.8, 94.9, 85.4, 87.2, 92.7,
...     97.8, 95.4, 94.9, 96.0, 96.5)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     3.5, 0.0, 0.1, 3.7, 0.0, 0.1, 0.0, 0.0, 4.4, 0.1, 0.2, 0.7, 0.0, 1.2, 3.0, 0.0,
...     0.0, 0.0, 0.0, 0.2, 0.8, 0.0, 0.0, 0.0, 3.2, 4.3, 0.0, 3.4, 0.0, 0.0, 4.0)

>>> aetinputs.atmosphericpressure.series = (
...     1015.0, 1007.0, 1005.0, 1008.0, 996.0, 998.0, 1004.0, 1006.0, 1009.0, 1018.0,
...     1004.0, 1012.0, 1020.0, 1025.0, 1023.0, 1000.0, 991.0, 997.0, 997.0, 997.0,
...     1004.0, 1007.0, 1004.0, 1008.0, 1018.0, 1019.0, 1017.0, 1022.0, 1024.0, 1021.0,
...     1016.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

>>> model.radiationmodel.sequences.inputs.possiblesunshineduration.series = (
...     7.64102946, 7.6096444, 7.57982377, 7.55159603, 7.52498861, 7.50002776,
...     7.47673851, 7.45514453, 7.43526809, 7.41712991, 7.40074913, 7.38614323,
...     7.3733279, 7.36231706, 7.35312273, 7.34575502, 7.34022207, 7.33653002,
...     7.33468296, 7.33468296, 7.33653002, 7.34022207, 7.34575502, 7.35312273,
...     7.36231706, 7.3733279, 7.38614323, 7.40074913, 7.41712991, 7.43526809,
...     7.45514453)
>>> model.radiationmodel.sequences.inputs.globalradiation.series = (
...     31.9427981, 9.8738984, 16.6834998, 31.8074275, 9.5325105, 16.1414844,
...     9.3414287, 9.2567644, 33.405815, 15.5911584, 15.8915442, 17.8235189, 8.941324,
...     19.6680679, 26.8315815, 8.8377384, 8.817416, 8.8041854, 8.7980439, 15.4647459,
...     17.8795412, 8.8221612, 8.8443961, 8.8737425, 27.7596417, 32.3287553, 9.0045823,
...     28.9823209, 9.1276454, 9.1999895, 32.0622345)

When comparing the following results with those of integration test
:ref:`lland_dd_acker_winter` on |lland_dd|, it is not immediately clear what
differences are due to the much higher complexities of the snow module of
|lland_knauf|.  But, at least, one can see that |lland_knauf| allows for the
(re)freezing of liquid water within the snow layer.  Refreezing occurs around
December 7, where the  frozen water equivalent (|WATS|) grows until it reaches to total
water equivalent (|WAeS|) (a note for `LARSIM`_ users: currently, `LARSIM`_ does not
implement such a freezing feature; you can disable it in |lland_knauf| through setting
|RefreezeFlag| to |False|):

.. integration-test::

    >>> test.reset_inits()
    >>> conditions = model.conditions
    >>> conditions_acker_winter = test(
    ...     "lland_knauf_acker_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |         tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |   nbes |    sbes | evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |            5.53098 |                 3.958114 |                          0.0 |             2.782554 |        0.0 |       34.8 |                       0.0 |                      0.0 |              0.0 |    0.0 |     0.0 | 0.0 | 0.553793 |       0.0 |       0.0 |          nan |          nan |       0.0 |       0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |           4.168274 |                 4.893489 |                     3.819164 |             4.159465 |    -7.5456 |  -7.683496 |                  1.229506 |                27.862843 |       -26.633337 | 10.374 |  10.374 | 0.0 |      0.0 |  -0.16818 | -0.752835 |    -6.265017 |     0.875479 |       0.0 |       0.0 | 0.406695 |      0.0 |  -5.192276 | -28.305216 |   -6.864155 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |  10.54218 |  10.54218 | -1.572176 | 0.084415 |  -9.916504 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           1.603182 |                 4.019239 |                     3.571122 |              3.85445 |  -2.855953 |  25.666656 |                  2.321028 |                37.526691 |       -35.205663 |  0.376 |   0.376 | 0.0 |      0.0 | -0.073734 | -0.050934 |    -7.133729 |     0.860879 |       0.0 |       0.0 | 0.487379 |      0.0 |  -2.276422 |  -7.001615 |   25.927626 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 10.991915 | 10.991915 |  -1.88408 | 0.141806 |  -26.98316 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           3.446842 |                 4.608124 |                     3.236431 |              4.17496 |   -7.77115 |   3.742499 |                  4.733805 |                55.893456 |       -51.159651 |  1.128 |   1.128 | 0.0 |      0.0 | -0.402072 | -0.103687 |      -8.3949 |     0.851173 |       0.0 |       0.0 | 0.636671 |      0.0 | -12.413282 | -34.530433 |    4.215936 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 12.521987 | 12.521987 | -2.461204 | 0.183441 | -22.125659 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           3.206365 |                 6.197477 |                     4.912922 |             5.980565 |   -6.37219 | -20.550929 |                   1.46096 |                15.889112 |       -14.428152 |  6.016 |  2.7072 | 0.0 |      0.0 | -0.433968 |  0.045165 |    -2.947035 |     0.846739 |       0.0 |       0.0 | 0.403076 | 0.403076 | -13.397994 | -22.438939 |  -21.408781 | 0.977284 | 0.286525 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 15.998208 | 18.971955 |       0.0 | 0.203488 |   7.025269 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           1.923819 |                 5.634754 |                     5.151568 |             5.578406 |        0.0 |  13.867614 |                  3.011254 |                 28.07219 |       -25.060937 |    0.0 |     0.0 | 0.0 |      0.0 | -0.123565 |       0.0 |    -2.311269 |     0.813446 |       0.0 |       0.0 | 0.317535 | 0.317535 |  -3.814858 |  -6.150958 |   15.095121 | 0.994317 |   0.2966 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 16.422008 |  19.09552 |       0.0 |  0.37888 |   1.757655 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           1.523023 |                 4.930236 |                     4.623647 |             4.890794 |        0.0 |  22.532218 |                  1.983491 |                32.375277 |       -30.391785 |    0.0 |     0.0 | 0.0 |      0.0 |  -0.06757 |       0.0 |     -3.75537 |     0.787667 |       0.0 |       0.0 | 0.511759 | 0.511759 |  -2.086105 |  -3.795136 |   24.510544 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 16.993687 |  19.16309 |       0.0 | 0.554272 | -12.174563 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           1.442864 |                 4.930236 |                     4.225088 |             4.885864 |  -3.506274 |   8.636691 |                  2.155749 |                27.170321 |       -25.014571 |    0.0 |     0.0 | 0.0 |      0.0 |   -0.1623 |       0.0 |    -4.945723 |     0.767116 |       0.0 |       0.0 | 0.660412 | 0.660412 |  -5.010735 |  -8.814165 |   11.189672 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 17.803619 |  19.32539 |       0.0 | 0.729664 | -12.211254 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           2.164296 |                 5.233109 |                     3.897456 |             5.070883 |  -3.516841 |  14.906113 |                  8.184701 |                58.773789 |       -50.589088 |  0.658 |   0.658 | 0.0 |      0.0 | -0.365433 | -0.033425 |    -6.001193 |     0.754992 |       0.0 |       0.0 | 0.807756 | 0.807756 | -11.282099 | -21.311727 |   17.995262 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |  19.62175 | 20.348822 |       0.0 | 0.853031 | -18.517366 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           2.404774 |                 4.856984 |                      4.02737 |             4.614135 |  -5.333001 |   1.443584 |                  3.649898 |                28.624554 |       -24.974656 |  3.102 |   3.102 | 0.0 |      0.0 | -0.195603 | -0.232614 |    -5.573599 |     0.765899 |       0.0 |       0.0 | 0.843328 | 0.727072 |  -6.038888 |  -14.46471 |    4.471057 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 23.646425 | 23.646425 | -0.449415 | 0.741299 |  -11.36095 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           5.931775 |                 8.790172 |                       6.1078 |             7.963896 |  -3.271954 | -19.631722 |                  3.997614 |                -0.754591 |         4.752205 |  8.836 |     0.0 | 0.0 |      0.0 | -1.215856 |  2.183707 |          0.0 |     0.748444 | 21.469498 | 21.469498 |      0.0 |      0.0 | -37.537452 | -58.603285 | -100.892942 | 0.945448 | 0.268161 | 30.468752 | 9.782248 |  0.0 |  0.0 | 0.0 |  0.0 | 9.782248 | 9.782248 |   0.0 |   0.0 |   0.0 |  0.0 | 2.518406 |   0.0 |   0.0 |   0.0 |  0.0 | 2.518406 | 0.029148 | 0.06 |  2.258412 |  3.229529 |       0.0 | 0.927266 |  16.870772 |  92.13271 | 7.263842 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029148 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |            4.56907 |                 6.197477 |                     5.176439 |             4.877414 |        0.0 |   13.47911 |                  4.716666 |                36.311921 |       -31.595255 |  0.188 |  0.0846 | 0.0 |      0.0 |  0.158712 |  0.000762 |    -2.246518 |     0.735368 |       0.0 |       0.0 | 0.061309 | 0.061309 |   4.899959 | -22.778338 |   13.716876 | 0.961225 | 0.277186 |  0.067022 |  0.02238 |  0.0 |  0.0 | 0.0 |  0.0 |  0.02238 |  0.02238 |   0.0 |   0.0 |   0.0 |  0.0 | 3.393537 |   0.0 |   0.0 |   0.0 |  0.0 | 3.393537 | 0.039277 | 0.06 |   2.29043 |  3.191795 |       0.0 | 1.094855 |  11.991662 | 92.177353 | 3.892684 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.039277 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           1.763501 |                 4.713362 |                     4.320033 |             4.265593 |        0.0 |  27.920468 |                  2.237038 |                 35.79395 |       -33.556911 |  2.726 |   2.726 | 0.0 |      0.0 |  0.014964 | -0.230795 |    -4.653411 |     0.749809 |       0.0 |       0.0 |  0.19865 |  0.19865 |   0.461981 |  -5.561289 |   28.457603 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.815505 |   0.0 |   0.0 |   0.0 |  0.0 | 1.815505 | 0.021013 | 0.06 |  5.201893 |  5.902832 |       0.0 | 0.911303 |  -7.328805 | 92.177353 | 2.077179 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021013 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |           2.484933 |                  4.57351 |                     3.839218 |             4.029262 |  -2.110696 |  24.517738 |                  5.193227 |                46.426577 |        -41.23335 |    0.0 |     0.0 | 0.0 |      0.0 | -0.064742 |       0.0 |    -6.196986 |     0.735956 |       0.0 |       0.0 | 0.256166 | 0.256166 |    -1.9988 | -13.726539 |   25.508011 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968774 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968774 | 0.011213 | 0.06 |  5.517923 |  5.967574 |       0.0 | 1.086695 | -23.246543 | 92.177353 | 1.108406 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.011213 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           2.244455 |                 3.808888 |                     2.826211 |             3.534648 |  -6.695004 |  20.484563 |                  7.398757 |                57.721383 |       -50.322626 |    0.0 |     0.0 | 0.0 |      0.0 | -0.225804 |       0.0 |   -10.109098 |     0.724252 |       0.0 |       0.0 |  0.40603 |  0.40603 |  -6.971296 | -21.297159 |   22.054171 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516948 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516948 | 0.005983 | 0.06 |  6.148107 |  6.193377 |       0.0 | 1.262087 | -35.131106 | 92.177353 | 0.591458 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.005983 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           2.965888 |                 4.713362 |                     3.578876 |             4.623808 | -10.117759 | -18.071922 |                  1.897806 |                22.175395 |       -20.277589 |  6.298 |   6.298 | 0.0 |      0.0 | -0.401816 | -0.533216 |    -7.105772 |     0.785261 |       0.0 |       0.0 | 0.557439 | 0.045271 | -12.405366 | -24.322444 |  -16.450222 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275849 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275849 | 0.003193 | 0.06 | 12.893193 | 12.893193 | -1.979911 | 0.572947 |  -8.459184 | 92.177353 | 0.315608 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003193 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.721432 |                  3.52531 |                     3.261432 |             3.405449 |  -2.436245 |  35.163253 |                  1.957901 |                41.672557 |       -39.714657 |    0.0 |     0.0 | 0.0 |      0.0 | -0.025897 |       0.0 |    -8.296787 |     0.777951 |       0.0 |       0.0 | 0.669376 |      0.0 |  -0.799522 |  -3.144158 |   35.770977 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147196 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147196 | 0.001704 | 0.06 |  12.91909 |  12.91909 | -2.587635 | 0.632541 | -35.022438 | 92.177353 | 0.168412 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001704 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           1.042069 |                 3.958114 |                     2.837467 |             3.867077 | -10.086462 |  -0.163598 |                  2.001268 |                24.668697 |       -22.667429 |  0.094 |   0.094 | 0.0 |      0.0 | -0.215254 | -0.013188 |   -10.059196 |     0.772691 |       0.0 |       0.0 | 0.819112 |      0.0 |  -6.645584 |  -15.61979 |    0.402055 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078546 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078546 | 0.000909 | 0.06 | 13.228344 | 13.228344 | -3.166476 | 0.678288 |  -26.25884 | 92.177353 | 0.089867 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000909 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           1.523023 |                 3.364296 |                      2.72315 |             3.222996 |  -7.562546 |  18.071092 |                  2.044019 |                36.101415 |       -34.057396 |    0.0 |     0.0 | 0.0 |      0.0 | -0.126427 |       0.0 |   -10.574395 |     0.767673 |       0.0 |       0.0 | 0.875308 |      0.0 |  -3.903213 | -11.865853 |   18.288329 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041913 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041913 | 0.000485 | 0.06 | 13.354771 | 13.354771 | -3.383713 | 0.724389 | -35.729932 | 92.177353 | 0.047954 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000485 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           1.923819 |                 3.988573 |                     2.930765 |             3.888858 |  -10.29022 |  -3.828076 |                  3.642564 |                 28.19957 |       -24.557007 |  0.188 |   0.188 | 0.0 |      0.0 | -0.277358 | -0.025922 |    -9.652208 |      0.76446 |       0.0 |       0.0 | 0.817964 |      0.0 |  -8.562936 | -20.069747 |   -4.075676 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022365 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022365 | 0.000259 | 0.06 | 13.820128 | 13.820128 | -3.162035 | 0.755264 | -23.301856 | 92.177353 | 0.025589 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000259 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |            1.84366 |                 4.019239 |                     3.128794 |             3.818277 |  -6.710934 |   12.68525 |                  4.271427 |                38.730209 |       -34.458782 |  0.188 |   0.188 | 0.0 |      0.0 | -0.194557 | -0.025467 |    -8.825143 |       0.7611 |       0.0 |       0.0 | 0.773574 |      0.0 |  -6.006607 | -15.963992 |   12.488183 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011934 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011934 | 0.000138 | 0.06 | 14.202685 | 14.202685 | -2.990435 | 0.788759 | -27.387106 | 92.177353 | 0.013654 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000138 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |            3.68732 |                 4.272186 |                     3.442548 |             4.007311 |  -7.887486 |   -1.69314 |                  1.989984 |                30.085589 |       -28.095605 |  2.256 |   2.256 | 0.0 |      0.0 | -0.254336 | -0.261947 |    -7.605297 |     0.774434 |       0.0 |       0.0 | 0.783269 |      0.0 |  -7.852177 | -22.161033 |   -1.917606 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006368 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006368 | 0.000074 | 0.06 | 16.713021 | 16.713021 | -3.027916 | 0.662853 | -17.093966 | 92.177353 | 0.007286 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000074 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           6.893685 |                 5.634754 |                     4.876829 |             5.347381 |  -4.923062 | -11.265049 |                  1.665037 |                25.124268 |       -23.459231 |   6.58 |  5.0995 | 0.0 |      0.0 | -0.349525 | -0.214608 |    -3.045554 |     0.811741 |       0.0 |       0.0 | 0.472213 | 0.472213 | -10.790985 | -25.350365 |  -12.682119 |  0.99104 | 0.294648 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003398 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003398 | 0.000039 | 0.06 |  22.61913 | 23.642546 |       0.0 |  0.38925 |   2.771083 | 92.177353 | 0.003888 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000039 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           6.092093 |                 4.930236 |                     4.510204 |             4.210422 |        0.0 |  24.507813 |                   1.88611 |                36.555032 |       -34.668922 |  0.188 |   0.188 | 0.0 |      0.0 |  0.200759 | -0.013188 |    -4.084636 |      0.78745 |       0.0 |       0.0 | 0.635341 | 0.635341 |   6.198095 | -13.916328 |   26.950689 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 | 0.000021 | 0.06 | 23.244981 | 23.629787 |       0.0 | 0.555937 |  -13.13673 | 92.177353 | 0.002075 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000021 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           3.046047 |                 3.061705 |                     2.608657 |             2.669806 |  -3.783378 |  43.959797 |                  6.469629 |                71.439587 |       -64.969959 |    0.0 |     0.0 | 0.0 |      0.0 | -0.023961 |       0.0 |   -11.110011 |     0.766941 |       0.0 |       0.0 | 1.669599 | 0.384806 |  -0.739762 | -13.816162 |   50.414034 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000968 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000968 | 0.000011 | 0.06 | 23.653748 | 23.653748 | -4.966676 | 0.731329 | -48.496527 | 92.177353 | 0.001107 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000011 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           1.202387 |                 3.184716 |                     1.819251 |             2.952232 |    -13.967 |   9.228637 |                  7.422691 |                55.663436 |       -48.240745 |  1.128 |   1.128 | 0.0 |      0.0 | -0.253432 | -0.234661 |   -15.505106 |       0.7704 |       0.0 |       0.0 | 2.404399 |      0.0 |  -7.824276 | -27.094386 |   13.322083 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 | 0.000006 | 0.06 |  25.03518 |  25.03518 | -9.294782 | 0.699032 | -49.125164 | 92.177353 | 0.000591 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000006 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           1.042069 |                 4.112496 |                     2.537766 |             4.022021 | -14.148047 | -16.174968 |                  2.040737 |                15.774476 |       -13.733739 |  0.846 |   0.846 | 0.0 |      0.0 | -0.310303 | -0.108462 |   -11.452219 |     0.773367 |       0.0 |       0.0 |   1.8547 |      0.0 |  -9.580069 | -22.562094 |  -18.408425 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000276 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000276 | 0.000003 | 0.06 | 26.191484 | 26.191484 | -7.169788 | 0.672269 | -24.350197 | 92.177353 | 0.000315 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000003 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           2.404774 |                 3.417227 |                      2.55336 |             3.260035 |  -7.012857 |  26.180199 |                  6.701461 |                61.689685 |       -54.988224 |    0.0 |     0.0 | 0.0 |      0.0 | -0.235576 |       0.0 |   -11.376223 |     0.768774 |       0.0 |       0.0 | 1.864481 |      0.0 |  -7.272988 | -21.497223 |   26.218013 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 | 0.000002 | 0.06 | 26.427059 | 26.427059 | -7.207601 |  0.71406 | -41.930396 | 92.177353 | 0.000168 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           2.164296 |                 3.037615 |                     2.253172 |             2.882696 | -12.075954 |   5.050047 |                  2.150584 |                 34.61242 |       -32.461836 |    0.0 |     0.0 | 0.0 |      0.0 | -0.196048 |       0.0 |   -12.917629 |     0.764388 |       0.0 |       0.0 | 2.136149 |      0.0 |  -6.052657 | -20.308937 |    6.100242 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 | 0.000001 | 0.06 | 26.623108 | 26.623108 | -8.257797 | 0.755971 | -38.380443 | 92.177353 |  0.00009 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           1.122228 |                 3.498009 |                     2.475816 |             3.358088 | -11.053568 |   4.228358 |                  2.189479 |                28.188233 |       -25.998754 |  0.188 |   0.188 | 0.0 |      0.0 | -0.190901 | -0.033653 |   -11.758294 |     0.762013 |       0.0 |       0.0 |  1.97269 |      0.0 |   -5.89375 | -16.542186 |    3.562818 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |      0.0 | 0.06 | 27.002009 | 27.002009 | -7.625909 | 0.779531 | -34.008802 | 92.177353 | 0.000048 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |           3.366683 |                 6.907942 |                     4.470314 |             6.666164 |  -9.794535 | -33.554262 |                  7.749127 |                41.691328 |       -33.942201 |  0.846 | 0.06345 | 0.0 |      0.0 | -0.924662 |  0.067075 |    -4.202158 |      0.75831 |       0.0 |       0.0 | 0.752906 | 0.752906 |  -28.54733 | -43.597429 |  -38.202558 | 0.973661 | 0.284405 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |      0.0 | 0.06 | 28.742042 | 28.772671 |       0.0 | 0.817554 |   8.145461 | 92.177353 | 0.000026 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_nadelw_winter_daily:

conifers (winter)
-----------------

Next, we repeat the integration test :ref:`lland_knauf_acker_winter_daily` but select
the land use type |NADELW| instead of |ACKER|.  Like for |LAUBW| and |MISCHW|,
|lland_knauf| now modifies the global radiation, the net longwave radiation, and the
wind speed to take the shadowing effects of the tree canopies into account.  Here, the
net effect of these modifications is that the total water equivalent's peak amounts are
two to three times smaller than for land use type |ACKER|.  Also, snow evaporation
(|EvS|) coincides with interception (|EvI|) and soil evaporation (|EvB|), which never
happens at non-forest sites, but this is a feature |evap_aet_morsim|, not of
|lland_knauf|:

.. integration-test::

    >>> lnk(NADELW)
    >>> model.aetmodel.parameters.control.surfaceresistance.nadelw = 56.0
    >>> model.aetmodel.parameters.control.cropheight.nadelw = 10.0
    >>> model.aetmodel.parameters.control.albedo.nadelw = 0.12
    >>> lai.nadelw = 11.0
    >>> model.aetmodel.parameters.control.leafareaindex.nadelw = 11.0
    >>> conditions_nadelw_winter = test(
    ...     "lland_knauf_nadelw_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |     nbes |     sbes |       evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |      wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |           2.449434 |                 3.958114 |                          0.0 |             2.782554 |       0.0 |       34.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.675021 |       0.0 |       0.0 |          nan |          nan |       0.0 |       0.0 |      0.0 |      0.0 |        0.0 |        0.0 |        0.0 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |            1.84595 |                 4.893489 |                     4.009414 |             4.159465 |   -7.5456 | -11.481814 |                  0.228337 |                 3.616419 |        -3.388082 |    8.234 |    8.234 |  1.794683 |      0.0 | -0.042372 | -0.597537 |    -5.631964 |     0.875479 |       0.0 |       0.0 | 0.290182 |      0.0 |  -1.308176 | -13.037488 | -10.957582 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.405317 |  8.276372 |  8.276372 | -1.121769 | 0.084415 |  -6.118186 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           0.709981 |                 4.019239 |                     4.166471 |              3.85445 | -1.762038 |  20.202401 |                  0.435485 |                17.344132 |       -16.908647 |      0.0 |      0.0 |  0.129738 |      0.0 |  0.055781 |       0.0 |    -5.129104 |     0.859447 |       0.0 |       0.0 | 0.265633 |      0.0 |   1.722139 |   1.476714 |    20.1075 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.651579 |  8.220592 |  8.220592 | -1.026867 | 0.147764 | -17.720587 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           1.526459 |                 4.608124 |                     4.014615 |              4.17496 | -5.103529 |   3.069047 |                  0.918227 |                13.393648 |       -12.475422 |      0.0 |      0.0 |  0.584565 |      0.0 | -0.040607 |       0.0 |    -5.615037 |     0.844555 |       0.0 |       0.0 | 0.288839 |      0.0 |  -1.253665 |  -8.063001 |   3.158756 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 1.195014 |  8.261198 |  8.261198 | -1.116576 | 0.213614 | -12.189634 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           1.419962 |                 6.197477 |                     5.323469 |             5.980565 | -3.510615 |  -9.848154 |                  0.289543 |                 3.867776 |        -3.578233 | 5.011014 | 2.254956 |  0.342799 |      0.0 | -0.160024 |   0.03762 |    -1.869256 |     0.836446 |       0.0 |       0.0 | 0.187585 | 0.187585 |  -4.940454 |  -8.839733 | -10.201955 | 0.979763 | 0.287981 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 1.857201 | 10.832795 | 13.432236 |       0.0 | 0.252752 |    6.25852 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           0.851977 |                 5.634754 |                     5.474524 |             5.578406 |       0.0 |   8.946827 |                  0.582856 |                12.038612 |       -11.455757 |      0.0 |      0.0 |  0.033701 |      0.0 | -0.019917 |       0.0 |    -1.491138 |     0.805566 |       0.0 |       0.0 | 0.149667 | 0.149667 |  -0.614896 |   -1.31546 |     9.5254 | 0.980884 |  0.28864 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   1.8235 | 10.998746 | 13.452153 |       0.0 | 0.428144 |   5.911693 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           0.674482 |                 4.930236 |                     5.036854 |             4.890794 |       0.0 |  15.681283 |                  0.379164 |                15.392937 |       -15.013773 |      0.0 |      0.0 |  0.001011 |      0.0 |  0.025639 |       0.0 |    -2.613547 |     0.781441 |       0.0 |       0.0 | 0.260254 | 0.260254 |   0.791547 |   0.882037 |  16.687357 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 1.822489 | 11.237542 | 13.426514 |       0.0 | 0.603536 |  -1.169591 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           0.638983 |                 4.930236 |                      4.98944 |             4.885864 | -0.336842 |  14.420543 |                  0.409024 |                14.831397 |       -14.422374 |      0.0 |      0.0 |  0.005587 |      0.0 |  0.017846 |       0.0 |    -2.740266 |     0.762073 |       0.0 |       0.0 | 0.267884 | 0.267884 |   0.550961 |   0.482776 |  15.456111 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 1.816902 | 11.490133 | 13.408669 |       0.0 | 0.778928 |  -6.990133 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           0.958474 |                 5.233109 |                     4.791747 |             5.070883 | -2.013158 |    7.60331 |                  1.562377 |                16.297167 |       -14.734789 | 0.274902 | 0.274902 | -0.145824 |      0.0 | -0.056229 | -0.013965 |    -3.280377 |     0.748164 |       0.0 |       0.0 | 0.320391 | 0.320391 |  -1.735965 |   -4.17093 |   8.827894 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.345824 | 12.135087 | 13.739799 |       0.0 | 0.930578 |  -5.993443 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           1.064971 |                 4.856984 |                     4.725379 |             4.614135 | -1.726112 |  10.440055 |                  0.687586 |                13.074297 |       -12.386711 | 3.247824 | 3.247824 |  0.333091 |      0.0 |  0.023489 | -0.243549 |    -3.466121 |     0.762533 |       0.0 |       0.0 | 0.403366 | 0.403366 |   0.725197 |  -1.356094 |  11.755814 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 1.866909 | 15.764449 | 16.964134 |       0.0 | 0.774318 |  -7.833498 | 71.324979 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           2.626929 |                 8.790172 |                       6.1078 |             7.963896 | -2.256047 | -13.536284 |                  0.750864 |               -11.576952 |        12.327816 | 8.502909 |      0.0 |       2.2 | 0.020689 | -0.656354 |  1.378787 |          0.0 |     0.745581 | 13.469568 | 13.469568 |      0.0 |      0.0 | -20.263798 | -31.635741 | -64.227355 | 0.953751 | 0.272893 | 22.74617 | 7.343664 |  0.0 |  0.0 | 0.0 |  0.0 | 7.343664 | 7.343664 |   0.0 |   0.0 |   0.0 |  0.0 | 1.890601 |   0.0 |   0.0 |   0.0 |  0.0 | 1.890601 | 0.021882 |      0.0 |  2.361698 |  3.377227 |       0.0 | 0.961631 |  14.302786 | 86.706796 | 5.453063 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021882 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |           2.023445 |                 6.197477 |                     5.568922 |             4.877414 |       0.0 |   7.556989 |                  0.885774 |                 9.846507 |        -8.960734 |      0.0 |      0.0 |     0.188 | 0.683908 |  0.206468 |       0.0 |    -1.259498 |     0.732401 |       0.0 |       0.0 | 0.034647 | 0.034647 |   6.374329 |  -7.644139 |   7.690923 | 0.950379 | 0.270966 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.543248 |   0.0 |   0.0 |   0.0 |  0.0 | 2.543248 | 0.029436 |      0.0 |  2.249843 |   3.17076 |       0.0 | 1.137023 |  15.345798 | 86.022888 | 2.909814 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029436 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           0.780979 |                 4.713362 |                     4.810831 |             4.265593 |       0.0 |  19.364316 |                  0.454567 |                16.130859 |       -15.676292 |    0.526 |    0.526 |  0.463667 |      0.0 |  0.101005 | -0.044533 |    -3.227386 |     0.726252 |       0.0 |       0.0 | 0.093316 | 0.093316 |   3.118348 |   0.885877 |  19.680517 | 0.985186 | 0.291177 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.357105 |   0.0 |   0.0 |   0.0 |  0.0 | 1.357105 | 0.015707 | 1.736333 |  2.790766 |  3.595755 |       0.0 | 1.230056 |   4.581482 | 86.022888 | 1.552709 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.015707 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |            1.10047 |                  4.57351 |                     4.646441 |             4.029262 |       0.0 |  22.140455 |                  1.037523 |                  18.6965 |       -17.658977 |      0.0 |      0.0 |   0.74798 |      0.0 |  0.132317 |       0.0 |    -3.690076 |     0.715953 |       0.0 |       0.0 | 0.101676 | 0.101676 |   4.085059 |   0.789473 |  22.533509 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.724166 |   0.0 |   0.0 |   0.0 |  0.0 | 0.724166 | 0.008382 | 0.988353 |  2.786006 |  3.463438 |       0.0 | 1.405448 |  -8.958973 | 86.022888 | 0.828543 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.008382 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           0.993973 |                 3.808888 |                       3.8451 |             3.534648 | -2.580184 |  21.581439 |                  1.459756 |                21.257217 |       -19.797461 |      0.0 |      0.0 |  0.185292 |      0.0 |  0.063542 |       0.0 |    -6.177091 |     0.707053 |       0.0 |       0.0 | 0.160142 | 0.160142 |   1.961759 |   0.441288 |  22.200508 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.386423 |   0.0 |   0.0 |   0.0 |  0.0 | 0.386423 | 0.004472 | 0.803061 |  2.892097 |  3.399896 |       0.0 |  1.58084 | -21.940412 | 86.022888 |  0.44212 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004472 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           1.313465 |                 4.713362 |                     4.106014 |             4.623808 | -6.318839 |  -5.989098 |                  0.410116 |                  6.34533 |        -5.935214 | 4.901061 | 4.901061 |  0.125607 |      0.0 | -0.121069 | -0.414945 |    -5.320656 |     0.750126 |       0.0 |       0.0 | 0.293333 | 0.293333 |   -3.73781 |  -7.467498 |  -5.270094 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.2062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.2062 | 0.002387 | 2.074393 |  8.204433 |  8.422026 |       0.0 | 0.907629 |  -7.351314 | 86.022888 |  0.23592 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002387 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.319491 |                  3.52531 |                     3.867045 |             3.405449 | -2.117178 |  23.915569 |                  0.431941 |                20.577376 |       -20.145435 |      0.0 |      0.0 |  0.027374 |      0.0 |   0.06608 |       0.0 |    -6.103107 |     0.736223 |       0.0 |       0.0 | 0.329975 | 0.217594 |    2.04011 |   3.005622 |  25.191167 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.11003 |   0.0 |   0.0 |   0.0 |  0.0 |  0.11003 | 0.001274 | 2.047019 |  8.355946 |  8.355946 | -0.434437 | 1.083021 | -22.666883 | 86.022888 | 0.125889 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001274 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           0.461488 |                 3.958114 |                     3.588204 |             3.867077 | -6.528062 |   3.264901 |                  0.442354 |                 9.530158 |        -9.087804 |      0.0 |      0.0 |  0.029554 |      0.0 | -0.043534 |       0.0 |    -7.072213 |     0.729458 |       0.0 |       0.0 | 0.369786 |      0.0 |  -1.344041 |  -3.483801 |   4.259962 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.058713 |   0.0 |   0.0 |   0.0 |  0.0 | 0.058713 |  0.00068 | 2.111465 |   8.39948 |   8.39948 | -1.429498 | 1.180569 | -17.331784 | 86.022888 | 0.067176 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00068 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           0.674482 |                 3.364296 |                     3.393896 |             3.222996 | -4.991554 |  16.777576 |                  0.448009 |                16.106832 |       -15.658823 |      0.0 |      0.0 |  0.112396 |      0.0 |  0.029999 |       0.0 |    -7.787816 |     0.725807 |       0.0 |       0.0 | 0.409325 |      0.0 |   0.926166 |   0.345432 |  16.930421 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.03133 |   0.0 |   0.0 |   0.0 |  0.0 |  0.03133 | 0.000363 | 1.999069 |  8.369481 |  8.369481 | -1.582343 | 1.237097 |  -25.50936 | 86.022888 | 0.035846 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000363 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           0.851977 |                 3.988573 |                     3.514448 |             3.888858 | -7.346696 |  -0.041947 |                  0.796911 |                 8.389385 |        -7.592473 |      0.0 |      0.0 |  0.092286 |      0.0 | -0.071784 |       0.0 |    -7.339705 |     0.722526 |       0.0 |       0.0 | 0.384394 |      0.0 |  -2.216201 |  -5.514594 |  -0.138321 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.016718 |   0.0 |   0.0 |   0.0 |  0.0 | 0.016718 | 0.000193 | 2.094783 |  8.441265 |  8.441265 | -1.485969 | 1.290472 | -16.867413 | 86.022888 | 0.019128 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000193 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |           0.816478 |                 4.019239 |                     3.770073 |             3.818277 | -4.857815 |   9.450477 |                  0.930128 |                13.244216 |       -12.314088 | 0.082783 | 0.082783 |  0.167051 |      0.0 | -0.009086 | -0.011214 |    -6.432895 |     0.719882 |       0.0 |       0.0 | 0.343125 |      0.0 |  -0.280508 |  -2.753853 |   9.279726 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008921 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008921 | 0.000103 | 2.032949 |  8.533134 |  8.533134 | -1.326432 | 1.335375 |  -17.71789 | 86.022888 | 0.010207 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000103 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |           1.632956 |                 4.272186 |                     3.939138 |             4.007311 | -5.102752 |   4.559358 |                   0.43617 |                10.388081 |        -9.951911 | 2.088949 | 2.088949 |  0.580986 |      0.0 | -0.017927 |  -0.24255 |    -5.862645 |     0.733783 |       0.0 |       0.0 | 0.389676 |      0.0 |  -0.553455 |  -4.901695 |    4.49676 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00476 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00476 | 0.000055 | 1.619014 |  10.64001 |  10.64001 | -1.506385 | 1.117179 | -13.677248 | 86.022888 | 0.005446 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000055 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           3.052917 |                 5.634754 |                     5.020182 |             5.347381 | -3.939047 |  -7.686385 |                  0.372322 |                 6.831659 |        -6.459337 | 5.999014 | 4.649236 |  1.086744 |      0.0 | -0.128418 | -0.195659 |    -2.657983 |     0.773324 |       0.0 |       0.0 | 0.299268 | 0.299268 |  -3.964676 | -10.726197 |  -8.231536 | 0.991563 | 0.294959 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00254 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00254 | 0.000029 | 1.113256 | 15.708823 | 16.767441 |       0.0 | 0.672649 |   2.609137 | 86.022888 | 0.002906 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000029 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           2.697927 |                 4.930236 |                     4.859622 |             4.210422 |       0.0 |  18.556505 |                  0.403012 |                14.293546 |       -13.890533 |      0.0 |      0.0 |  1.301256 | 0.210203 |  0.233775 |       0.0 |    -3.092751 |     0.755451 |       0.0 |       0.0 | 0.345052 | 0.345052 |   7.217399 |  -1.217546 |  19.890385 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001355 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001355 | 0.000016 |      0.0 | 15.830048 | 16.533666 |       0.0 | 0.848041 |  -7.347368 | 85.812685 | 0.001551 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000016 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           1.348964 |                 3.061705 |                     3.406529 |             2.669806 | -2.116042 |  33.744938 |                  1.336909 |                 27.26332 |       -25.926411 |      0.0 |      0.0 |       0.0 | 0.151738 |  0.174644 |       0.0 |    -7.740198 |     0.740676 |       0.0 |       0.0 | 0.834984 | 0.703618 |   5.391841 |   5.654519 |  36.972772 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000723 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000723 | 0.000008 |      0.0 | 16.359022 | 16.359022 | -0.507831 | 1.023433 | -32.492307 | 85.660946 | 0.000828 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000008 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           0.532486 |                 3.184716 |                     2.803445 |             2.952232 | -9.357784 |   5.116691 |                  1.608566 |                15.598098 |       -13.989532 |      0.0 |      0.0 |  -0.08782 |      0.0 |  -0.02419 |       0.0 |   -10.210566 |      0.73208 |       0.0 |       0.0 | 1.045218 |      0.0 |  -0.746832 |  -4.593297 |   8.649402 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000386 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000386 | 0.000004 |  1.21582 | 16.383212 | 16.383212 | -4.040543 | 1.141686 | -29.008997 | 85.660946 | 0.000442 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000004 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           0.461488 |                 4.112496 |                     3.425855 |             4.022021 | -8.354591 |  -4.121585 |                  0.453115 |                 4.686537 |        -4.233422 |      0.0 |      0.0 |  0.030409 |      0.0 | -0.093066 |       0.0 |     -7.66766 |     0.729044 |       0.0 |       0.0 | 0.786071 |      0.0 |  -2.873249 |  -6.483554 |  -5.123381 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000206 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000206 | 0.000002 | 2.031411 | 16.476278 | 16.476278 | -3.038746 | 1.186839 | -16.287412 | 85.660946 | 0.000236 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           1.064971 |                 3.417227 |                     3.373812 |             3.260035 | -4.690775 |   19.03839 |                  1.477129 |                20.476117 |       -18.998988 |      0.0 |      0.0 |  0.014952 |      0.0 |  0.024024 |       0.0 |     -7.86384 |     0.725565 |       0.0 |       0.0 | 0.810762 |      0.0 |   0.741708 |  -0.606855 |  19.133841 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00011 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00011 | 0.000001 | 2.016459 | 16.452253 | 16.452253 | -3.134197 | 1.240961 | -26.725803 | 85.660946 | 0.000126 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           0.958474 |                 3.037615 |                     2.932544 |             2.882696 | -7.697031 |  11.685178 |                  0.470725 |                14.120895 |       -13.650171 |      0.0 |      0.0 |  0.205881 |      0.0 |  0.010041 |       0.0 |    -9.644561 |     0.722308 |       0.0 |       0.0 | 0.992905 |      0.0 |   0.310004 |  -1.570882 |  12.389293 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000059 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000059 | 0.000001 | 1.810578 | 16.442212 | 16.442212 | -3.838313 | 1.294107 | -29.810981 | 85.660946 | 0.000067 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           0.496987 |                 3.498009 |                        3.082 |             3.358088 | -8.585562 |   2.584381 |                  0.479155 |                 8.687356 |        -8.208202 |      0.0 |      0.0 |  0.077402 |      0.0 | -0.043993 |       0.0 |    -9.016293 |     0.719558 |       0.0 |       0.0 | 0.927658 |      0.0 |  -1.358221 |  -4.517826 |   2.332154 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000031 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000031 |      0.0 | 1.921177 | 16.486206 | 16.486206 | -3.586087 | 1.340999 | -23.795361 | 85.660946 | 0.000036 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |            1.49096 |                 6.907942 |                     5.130481 |             6.666164 | -6.853064 | -26.920086 |                  1.685226 |                 1.916828 |        -0.231602 | 0.567177 | 0.042538 |  0.111701 |      0.0 | -0.383932 |  0.044969 |    -2.366383 |     0.716979 |       0.0 |       0.0 | 0.260314 | 0.260314 | -11.853233 | -17.833268 | -29.454898 | 0.962088 | 0.277684 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000017 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000017 |      0.0 | 2.088299 | 17.167039 | 17.437315 |       0.0 | 1.386749 |  11.724724 | 85.660946 | 0.000019 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_knauf_acker_winter_ktschnee_daily:

acre (snow surface temperature)
-------------------------------

In integration test :ref:`lland_knauf_acker_winter_daily`, |lland_knauf| assumes that
the snow layer has a constant temperature over its complete depth.  For less simplified
modelling of the processes within the snow layer, one can enable an iterative
adjustment of the snow surface temperature (see method |Return_TempSSurface_V1|).
Therefore, one must set the thermal conductivity of the top snow layer (|KTSchnee|) to
a value smaller than |numpy.inf|.  The lower the conductivity, the higher the possible
differences between the bulk and the surface temperature of the snow layer. Setting it
to zero (which is not advisable) would prevent any thermal exchange.  Setting it to
|numpy.inf| results in an infinitely fast exchange, avoiding any temperature
differences within the snow layer.

We show what happens when we use the default |KTSchnee| value used by `LARSIM`_.  In
integration test :ref:`lland_knauf_acker_winter_daily`, with :math:`KTSchnee = inf`,
nearly the complete frozen water equivalent melts on December 11, when the average air
temperature is 5 °C.  In our next test, with :math:`KTSchnee = 5 W/m²/K`, the limited
thermal conductivity of the top snow layer weakens the energy flux into the snow layer
and fewer snow melts.  The snow surface temperature rises faster than the bulk
temperature, and the higher surface temperature results in smaller sensible
(|WSensSnow|) and latent (|WLatSnow|) heat fluxes from the atmosphere into the snow
layer and larger longwave radiation losses (|NetLongwaveRadiationSnow|):

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(5.0)
    >>> conditions_acker_winter_ktschnee = test(
    ...     "lland_knauf_acker_winter_ktschnee_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |   nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |   nbes |    sbes |  evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |     schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |     qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |             70.3 |       6.9 |                 7.641029 |              3.5 | 0.0 | 0.0 |                   3.5 |                      7.641029 |       31.942798 |    0.0 | -5.8 |     5.53098 |            5.53098 |                 3.958114 |                          0.0 |             2.782554 |       0.0 |       34.8 |                       0.0 |                      0.0 |              0.0 |    0.0 |     0.0 |  0.0 | 0.553793 |       0.0 |       0.0 |          nan |          nan |       0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |             85.0 |       5.2 |                 7.609644 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.609644 |        9.873898 | 10.434 | -3.0 |    4.168274 |           4.168274 |                 4.893489 |                     3.874827 |             4.159465 |   -7.5456 |  -4.524364 |                  1.229506 |                28.674485 |       -27.444979 | 10.374 |  10.374 |  0.0 |      0.0 | -0.140671 | -0.752835 |     -6.07696 |     0.875479 |       0.0 |      0.0 | 0.440874 |      0.0 |  -4.342973 | -26.674903 |   -3.572896 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 10.514671 | 10.514671 | -1.704304 | 0.084415 | -13.075636 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |             95.9 |       2.0 |                 7.579824 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.579824 |         16.6835 |  0.376 | -5.6 |    1.603182 |           1.603182 |                 4.019239 |                     3.243675 |              3.85445 | -3.765783 |  12.660208 |                  2.311282 |                32.300381 |       -29.989098 |  0.376 |   0.376 |  0.0 |      0.0 |  -0.15895 | -0.050934 |    -8.366401 |     0.861463 |       0.0 |      0.0 | 0.400426 |      0.0 |   -4.90731 | -12.628873 |   12.452915 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 11.049621 | 11.049621 | -1.547945 | 0.139393 | -17.135843 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |             90.6 |       4.3 |                 7.551596 |              3.7 | 0.0 | 0.0 |                   3.7 |                      7.551596 |       31.807427 |  1.128 | -3.8 |    3.446842 |           3.446842 |                 4.608124 |                     3.313259 |              4.17496 | -4.935123 |   8.499536 |                  4.759177 |                57.155746 |       -52.396569 |  1.128 |   1.128 |  0.0 |      0.0 | -0.369159 | -0.103687 |    -8.095503 |     0.850375 |       0.0 |      0.0 | 0.484009 |      0.0 | -11.397129 | -32.280484 |    8.718956 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |  12.54678 |  12.54678 | -1.871052 | 0.186998 |  -17.03538 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |             96.5 |       4.0 |                 7.524989 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.524989 |        9.532511 |  6.016 |  0.2 |    3.206365 |           3.206365 |                 6.197477 |                     5.232243 |             5.980565 | -4.906189 |  -7.697363 |                  1.485911 |                19.686689 |       -18.200779 |  6.016 |  2.7072 |  0.0 |      0.0 | -0.304172 |  0.045165 |    -2.102229 |     0.844122 |       0.0 |      0.0 | 0.496132 | 0.496132 |  -9.390793 | -16.415317 |   -7.605332 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 16.008195 | 18.866952 |       0.0 | 0.215645 |  -0.738016 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |             99.0 |       2.4 |                 7.500028 |              0.1 | 0.0 | 0.0 |                   0.1 |                      7.500028 |       16.141484 |    0.0 | -1.1 |    1.923819 |           1.923819 |                 5.634754 |                     4.935069 |             5.578406 | -0.212549 |    6.90279 |                   3.04348 |                25.486763 |       -22.443283 |    0.0 |     0.0 |  0.0 |      0.0 | -0.186239 |       0.0 |    -2.886897 |      0.81145 |       0.0 |      0.0 | 0.185379 | 0.185379 |  -5.749809 |  -9.074059 |    7.619416 | 0.996898 | 0.298142 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 16.353424 | 19.053191 |       0.0 | 0.391037 |   0.959194 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |             99.2 |       1.9 |                 7.476739 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.476739 |        9.341429 |    0.0 | -2.9 |    1.523023 |           1.523023 |                 4.930236 |                      4.29691 |             4.890794 |       0.0 |   12.29543 |                  1.998184 |                28.103018 |       -26.104834 |    0.0 |     0.0 |  0.0 |      0.0 | -0.150212 |       0.0 |    -4.724073 |     0.786094 |       0.0 |      0.0 | 0.279053 | 0.279053 |  -4.637544 |  -8.093116 |   13.374174 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 16.763604 | 19.203403 |       0.0 | 0.566429 |  -2.736236 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |             99.1 |       1.8 |                 7.455145 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.455145 |        9.256764 |    0.0 | -2.9 |    1.442864 |           1.442864 |                 4.930236 |                     4.242856 |             4.885864 | -0.788036 |  10.465352 |                  2.167515 |                 27.41168 |       -25.244165 |    0.0 |     0.0 |  0.0 |      0.0 | -0.157936 |       0.0 |     -4.89058 |     0.765845 |       0.0 |      0.0 | 0.343075 | 0.343075 |  -4.875997 |  -8.576577 |   11.791592 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 17.247371 | 19.361339 |       0.0 | 0.741821 |  -4.601588 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |             96.9 |       2.7 |                 7.435268 |              4.4 | 0.0 | 0.0 |                   4.4 |                      7.435268 |       33.405815 |  0.658 | -2.1 |    2.164296 |           2.164296 |                 5.233109 |                     3.812675 |             5.070883 | -1.325257 |   12.55224 |                  8.219632 |                57.539469 |       -49.319838 |  0.658 |   0.658 |  0.0 |      0.0 | -0.391835 | -0.033425 |    -6.287098 |     0.753946 |       0.0 |      0.0 | 0.473439 | 0.473439 |  -12.09724 | -22.873591 |   14.349006 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 | 18.738535 | 20.411175 |       0.0 |   0.8645 |  -8.553827 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |             95.0 |       3.0 |                  7.41713 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.41713 |       15.591158 |  3.102 | -3.1 |    2.404774 |           2.404774 |                 4.856984 |                     4.122087 |             4.614135 | -2.463502 |   6.595367 |                  3.663944 |                29.948168 |       -26.284224 |  3.102 |   3.102 |  0.0 |      0.0 | -0.164028 | -0.232614 |    -5.269486 |     0.764999 |       0.0 |      0.0 | 0.561607 | 0.561607 |  -5.064075 | -12.686369 |     8.53378 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 0.06 |  22.55842 | 23.677203 |       0.0 | 0.750014 |  -6.549194 | 71.446207 |      0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |             90.6 |       7.4 |                 7.400749 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.400749 |       15.891544 |  8.836 |  5.1 |    5.931775 |           5.931775 |                 8.790172 |                       6.1078 |             7.963896 | -1.886168 | -11.317008 |                  4.012368 |                -0.754591 |         4.766959 |  8.836 |     0.0 | 0.06 | 0.339079 | -1.215856 |   2.12077 |          0.0 |     0.747516 | 23.724162 | 22.55842 |      0.0 |      0.0 | -37.537452 | -58.603285 | -100.907696 | 0.956775 | 0.274625 | 33.729059 | 11.067249 |  0.0 |  0.0 | 0.0 |  0.0 | 11.067249 | 11.067249 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849225 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849225 | 0.032977 |  0.0 |       0.0 |       0.0 |       0.0 | 0.938276 |  13.367813 | 93.768939 | 8.218023 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032977 |
    | 2010-12-12 |  0.2 |  0.2 |             78.7 |       5.7 |                 7.386143 |              0.7 | 0.0 | 0.0 |                   0.7 |                      7.386143 |       17.823519 |  0.188 |  0.2 |     4.56907 |            4.56907 |                 6.197477 |                     5.049903 |             4.877414 |       0.0 |   7.031305 |                  4.735182 |                 34.81614 |       -30.080958 |  0.128 |  0.0576 | 0.06 | 0.519712 |  0.082368 |  0.000519 |    -2.578857 |      0.73433 |       0.0 |      0.0 | 0.000786 | 0.000786 |   2.826483 | -25.872578 |    7.034863 | 0.951702 | 0.271721 |  0.045632 |  0.015061 |  0.0 |  0.0 | 0.0 |  0.0 |  0.015061 |  0.015061 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836673 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836673 | 0.044406 |  0.0 |       0.0 |       0.0 |       0.0 | 1.109428 |  14.936509 | 93.279798 | 4.396411 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044406 |
    | 2010-13-12 |  2.9 | -3.5 |             90.5 |       2.2 |                 7.373328 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.373328 |        8.941324 |  2.726 | -3.5 |    1.763501 |           1.763501 |                 4.713362 |                     3.977478 |             4.265593 |       0.0 |  15.675795 |                  2.222322 |                31.068606 |       -28.846284 |  2.666 |   2.666 |  0.0 |      0.0 | -0.079193 | -0.225715 |    -5.736346 |     0.751455 |       0.0 |      0.0 | 0.043585 |      0.0 |  -2.444945 |  -10.78277 |   15.618569 | 0.974582 | 0.284943 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050438 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050438 | 0.023732 | 0.06 |  2.745193 |  2.745193 | -0.168489 | 0.892404 |   7.860714 | 93.279798 | 2.345974 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023732 |
    | 2010-14-12 |  0.0 | -3.9 |             88.1 |       3.1 |                 7.362317 |              1.2 | 0.0 | 0.0 |                   1.2 |                      7.362317 |       19.668068 |    0.0 | -3.9 |    2.484933 |           2.484933 |                  4.57351 |                      3.68122 |             4.029262 |       0.0 |  18.367586 |                  5.038442 |                44.083324 |       -39.044881 |    0.0 |     0.0 |  0.0 |      0.0 | -0.118567 |       0.0 |    -6.741741 |     0.743826 |       0.0 |      0.0 | 0.052586 |      0.0 |  -3.660562 | -16.981937 |   18.402382 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094136 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094136 | 0.012664 | 0.06 |   2.86376 |   2.86376 | -0.203285 | 0.983299 |  -1.906872 | 93.279798 | 1.251837 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012664 |
    | 2010-15-12 |  0.0 | -6.3 |             92.8 |       2.8 |                 7.353123 |              3.0 | 0.0 | 0.0 |                   3.0 |                      7.353123 |       26.831581 |    0.0 | -6.3 |    2.244455 |           2.244455 |                 3.808888 |                     2.883853 |             3.534648 | -0.549179 |  25.311295 |                  7.049178 |                58.770158 |        -51.72098 |    0.0 |     0.0 |  0.0 |      0.0 | -0.207431 |       0.0 |    -9.855386 |     0.737281 |       0.0 |      0.0 | 0.085437 |      0.0 |   -6.40407 | -19.878621 |   25.438289 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583843 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583843 | 0.006757 | 0.06 |  3.071191 |  3.071191 | -0.330279 |  1.06855 | -18.618167 | 93.279798 | 0.667994 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.006757 |
    | 2010-16-12 |  6.7 | -3.5 |             98.1 |       3.7 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.837738 |  6.298 | -3.5 |    2.965888 |           2.965888 |                 4.713362 |                     3.936783 |             4.623808 | -5.362032 |   1.170579 |                  1.775591 |                27.488061 |       -25.712469 |  6.298 |   6.298 |  0.0 |      0.0 | -0.264187 | -0.533216 |    -5.870438 |      0.79909 |       0.0 |      0.0 | 0.325801 |      0.0 |  -8.156322 | -15.989601 |    1.566547 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311545 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311545 | 0.003606 | 0.06 |  9.633378 |  9.633378 | -1.259462 | 0.471345 | -11.188747 | 93.279798 | 0.356449 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003606 |
    | 2010-17-12 |  0.0 | -7.3 |             96.6 |       0.9 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.817416 |    0.0 | -7.3 |    0.721432 |           0.721432 |                  3.52531 |                     2.800277 |             3.405449 | -3.222359 |  18.972885 |                  1.850724 |                33.637078 |       -31.786354 |    0.0 |     0.0 |  0.0 |      0.0 | -0.108821 |       0.0 |   -10.224747 |     0.790106 |       0.0 |      0.0 | 0.384863 |      0.0 |  -3.359648 |  -9.225504 |   19.201202 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166244 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166244 | 0.001924 | 0.06 |  9.742199 |  9.742199 | -1.487779 | 0.535793 | -21.561632 | 93.279798 | 0.190205 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001924 |
    | 2010-18-12 |  0.1 | -5.8 |             97.7 |       1.3 |                  7.33653 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.33653 |        8.804185 |  0.094 | -5.8 |    1.042069 |           1.042069 |                 3.958114 |                     3.033407 |             3.867077 |  -6.20975 |   8.041899 |                  1.910103 |                28.161536 |       -26.251433 |  0.094 |   0.094 |  0.0 |      0.0 |  -0.17429 | -0.013188 |    -9.217539 |     0.783046 |       0.0 |      0.0 | 0.464706 |      0.0 |  -5.380896 | -12.533174 |    8.337363 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.08871 |   0.0 |   0.0 |   0.0 |  0.0 |  0.08871 | 0.001027 | 0.06 | 10.010489 | 10.010489 | -1.796431 | 0.590539 | -21.003531 | 93.279798 | 0.101496 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001027 |
    | 2010-19-12 |  0.0 | -7.9 |             95.8 |       1.9 |                 7.334683 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.334683 |        8.798044 |    0.0 | -7.9 |    1.523023 |           1.523023 |                 3.364296 |                     2.624919 |             3.222996 | -6.049017 |  13.476044 |                   1.96695 |                34.224679 |        -32.25773 |    0.0 |     0.0 |  0.0 |      0.0 | -0.151273 |       0.0 |   -11.032678 |     0.776433 |       0.0 |      0.0 | 0.519604 |      0.0 |   -4.67028 | -13.899182 |   13.688268 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047336 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047336 | 0.000548 | 0.06 | 10.161762 | 10.161762 | -2.008656 | 0.645482 | -25.879575 | 93.279798 | 0.054159 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000548 |
    | 2010-20-12 |  0.2 | -5.7 |             97.5 |       2.4 |                 7.334683 |              0.2 | 0.0 | 0.0 |                   0.2 |                      7.334683 |       15.464746 |  0.188 | -5.7 |    1.923819 |           1.923819 |                 3.988573 |                     3.099124 |             3.888858 | -7.453318 |   4.070387 |                  3.527764 |                31.141438 |       -27.613674 |  0.188 |   0.188 |  0.0 |      0.0 |  -0.22862 | -0.025922 |    -8.946047 |     0.771883 |       0.0 |      0.0 | 0.526638 |      0.0 |  -7.058235 | -16.483783 |    4.071656 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.025259 |   0.0 |   0.0 |   0.0 |  0.0 | 0.025259 | 0.000292 | 0.06 | 10.578381 | 10.578381 | -2.035847 | 0.685542 | -21.349962 | 93.279798 |   0.0289 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000292 |
    | 2010-21-12 |  0.2 | -5.6 |             95.0 |       2.3 |                  7.33653 |              0.8 | 0.0 | 0.0 |                   0.8 |                       7.33653 |       17.879541 |  0.188 | -5.6 |     1.84366 |            1.84366 |                 4.019239 |                     3.035743 |             3.818277 | -6.148789 |   8.394877 |                  4.156278 |                37.130933 |       -32.974655 |  0.188 |   0.188 |  0.0 |      0.0 | -0.220814 | -0.025467 |      -9.2078 |      0.76754 |       0.0 |      0.0 | 0.508508 |      0.0 |  -6.817243 | -17.858089 |    8.299323 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 | 0.000156 | 0.06 | 10.987195 | 10.987195 |  -1.96576 | 0.725649 | -21.144839 | 93.279798 | 0.015421 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000156 |
    | 2010-22-12 |  2.4 | -4.8 |             93.8 |       4.6 |                 7.340222 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.340222 |        8.822161 |  2.256 | -4.8 |     3.68732 |            3.68732 |                 4.272186 |                     3.529214 |             4.007311 | -6.089714 |   3.316366 |                  1.944525 |                31.445051 |       -29.500526 |  2.256 |   2.256 |  0.0 |      0.0 | -0.215306 | -0.261947 |    -7.285755 |     0.779586 |       0.0 |      0.0 | 0.550453 |      0.0 |   -6.64721 | -19.636748 |    3.216569 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007192 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007192 | 0.000083 | 0.06 | 13.458501 | 13.458501 | -2.127909 | 0.618819 | -15.861205 | 93.279798 | 0.008229 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000083 |
    | 2010-23-12 |  7.0 | -1.1 |             94.9 |       8.6 |                 7.345755 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.345755 |        8.844396 |   6.58 | -1.1 |    6.893685 |           6.893685 |                 5.634754 |                     4.983961 |             5.347381 | -4.568027 |  -4.739886 |                  1.641983 |                26.425029 |       -24.783046 |   6.58 |  5.0995 |  0.0 |      0.0 | -0.269948 | -0.214608 |    -2.754979 |     0.814348 |       0.0 |      0.0 | 0.508846 | 0.508846 |  -8.334182 |   -21.5642 |   -5.115337 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 | 0.000044 | 0.06 | 19.323706 |  20.30845 |       0.0 | 0.373461 |  -2.521319 | 93.279798 | 0.004391 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000044 |
    | 2010-24-12 |  0.2 | -2.9 |             85.4 |       7.6 |                 7.353123 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.353123 |        8.873743 |  0.188 | -2.9 |    6.092093 |           6.092093 |                 4.930236 |                     4.287255 |             4.210422 |  -0.72614 |  10.298965 |                  1.867788 |                 33.61027 |       -31.742482 |  0.188 |   0.188 |  0.0 |      0.0 |  0.051454 | -0.013188 |    -4.753678 |     0.789515 |       0.0 |      0.0 | 0.328384 | 0.328384 |   1.588548 | -21.775808 |   11.555222 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 | 0.000024 | 0.06 | 19.790283 | 20.444996 |       0.0 | 0.540228 |  -4.220284 | 93.279798 | 0.002343 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000024 |
    | 2010-25-12 |  0.0 | -9.1 |             87.2 |       3.8 |                 7.362317 |              3.2 | 0.0 | 0.0 |                   3.2 |                      7.362317 |       27.759642 |    0.0 | -9.1 |    3.046047 |           3.046047 |                 3.061705 |                     2.334322 |             2.669806 | -1.215442 |  29.045464 |                  6.423384 |                65.877607 |       -59.454222 |    0.0 |     0.0 |  0.0 |      0.0 |  -0.13146 |       0.0 |   -12.483738 |     0.768607 |       0.0 |      0.0 | 0.799708 | 0.654712 |  -4.058584 | -23.258712 |   32.136926 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 | 0.000013 | 0.06 | 20.576455 | 20.576455 | -0.560514 |  0.71562 | -24.665748 | 93.279798 |  0.00125 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000013 |
    | 2010-26-12 |  1.2 | -8.6 |             92.7 |       1.5 |                 7.373328 |              4.3 | 0.0 | 0.0 |                   4.3 |                      7.373328 |       32.328755 |  1.128 | -8.6 |    1.202387 |           1.202387 |                 3.184716 |                     1.979308 |             2.952232 | -7.103735 |  17.705249 |                  7.586074 |                59.614784 |        -52.02871 |  1.128 |   1.128 |  0.0 |      0.0 |  -0.21763 | -0.234661 |   -14.492413 |     0.765346 |       0.0 |      0.0 |  1.36557 |      0.0 |  -6.718935 | -23.120761 |   22.189013 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 | 0.000007 | 0.06 | 21.922085 | 21.922085 | -5.278939 | 0.746643 | -33.770997 | 93.279798 | 0.000667 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000007 |
    | 2010-27-12 |  0.9 | -5.3 |             97.8 |       1.3 |                 7.386143 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.386143 |        9.004582 |  0.846 | -5.3 |    1.042069 |           1.042069 |                 4.112496 |                     2.963655 |             4.022021 | -9.726047 |   -0.54134 |                  2.097285 |                23.751486 |       -21.654201 |  0.846 |   0.846 |  0.0 |      0.0 | -0.221265 | -0.108462 |    -9.511473 |     0.767087 |       0.0 |      0.0 | 1.372825 |      0.0 |  -6.831183 | -15.444774 |   -0.621756 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 | 0.000004 | 0.06 |  22.98935 |  22.98935 | -5.306985 | 0.729943 | -24.629657 | 93.279798 | 0.000356 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000004 |
    | 2010-28-12 |  0.0 | -7.7 |             95.4 |       3.0 |                 7.400749 |              3.4 | 0.0 | 0.0 |                   3.4 |                      7.400749 |       28.982321 |    0.0 | -7.7 |    2.404774 |           2.404774 |                 3.417227 |                     2.358788 |             3.260035 | -7.093341 |  14.368183 |                  6.890648 |                57.727703 |       -50.837054 |    0.0 |     0.0 |  0.0 |      0.0 | -0.300438 |       0.0 |   -12.355553 |     0.762246 |       0.0 |      0.0 | 1.364907 |      0.0 |  -9.275493 | -27.223988 |   14.337573 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 | 0.000002 | 0.06 | 23.289788 | 23.289788 | -5.276375 | 0.777186 |  -30.39784 | 93.279798 |  0.00019 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |             94.9 |       2.7 |                  7.41713 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.41713 |        9.127645 |    0.0 | -9.2 |    2.164296 |           2.164296 |                 3.037615 |                     2.327708 |             2.882696 | -8.754578 |   9.943979 |                  2.212531 |                36.211017 |       -33.998487 |    0.0 |     0.0 |  0.0 |      0.0 | -0.172836 |       0.0 |   -12.518592 |     0.757601 |       0.0 |      0.0 | 1.517385 |      0.0 |  -5.336019 | -18.129048 |   10.533419 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 | 0.000001 | 0.06 | 23.462624 | 23.462624 | -5.865815 | 0.825017 | -31.741819 | 93.279798 | 0.000101 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |             96.0 |       1.4 |                 7.435268 |              0.0 | 0.0 | 0.0 |                   0.0 |                      7.435268 |        9.199989 |  0.188 | -7.4 |    1.122228 |           1.122228 |                 3.498009 |                     2.537041 |             3.358088 | -9.141644 |   6.346345 |                  2.255141 |                29.415788 |       -27.160647 |  0.188 |   0.188 |  0.0 |      0.0 | -0.177654 | -0.033653 |   -11.455764 |     0.754876 |       0.0 |      0.0 | 1.509439 |      0.0 |  -5.484757 | -15.393912 |    6.281978 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 | 0.000001 | 0.06 | 23.828278 | 23.828278 | -5.835101 | 0.854297 | -29.488164 | 93.279798 | 0.000054 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000001 |
    | 2010-31-12 |  0.9 |  1.7 |             96.5 |       4.2 |                 7.455145 |              4.0 | 0.0 | 0.0 |                   4.0 |                      7.455145 |       32.062235 |  0.846 |  1.7 |    3.366683 |           3.366683 |                 6.907942 |                     4.947106 |             6.666164 | -8.492591 |  -14.28406 |                  7.981016 |                47.683043 |       -39.702027 |  0.846 | 0.06345 |  0.0 |      0.0 | -0.723887 |  0.067075 |    -2.854311 |     0.751077 |       0.0 |      0.0 | 0.973699 |  0.78255 | -22.348747 | -33.641298 |  -16.288018 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |      0.0 | 0.06 | 25.398165 | 25.398165 | -0.738933 | 0.896706 |  -6.604104 | 93.279798 | 0.000029 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> ktschnee(inf)

hourly simulation
_________________

We change the simulation time step size from one day to one hour. Consequently, we need
to change the values of all time-dependent control parameters by the factor 1/24 (or,
in single cases, 24).  Also, we need to |FixedParameter.restore| all time-dependent
fixed parameter values:

>>> pub.timegrids = "1997-08-03", "1997-08-06", "1h"
>>> for parameter in control:
...     if parameter.TIME == True:
...         parameter.value = parameter.value/24
...     if parameter.TIME == False:
...         parameter.value = parameter.value*24
>>> for parameter in fixed:
...     parameter.restore()
>>> for parameter in model.aetmodel.parameters.fixed:
...     parameter.restore()

For all integration tests performed in hourly simulation steps, we switch to
calculating the radiation-related data on the fly using submodel |meteo_glob_morsim|
with parameter values taken from its documentation:

>>> with model.add_radiationmodel_v1("meteo_glob_morsim") as submodel_meteo_glob_morsim:
...     latitude(54.1)
...     longitude(9.7)
...     angstromconstant(0.25)
...     angstromfactor(0.5)
...     angstromalternative(0.15)

As |meteo_psun_sun_glob_io|, |meteo_glob_morsim| complies with
|SharableSubmodelInterface| and is hence "sharable":

>>> model.aetmodel.add_radiationmodel_v1(submodel_meteo_glob_morsim)

>>> test = IntegrationTest(land)

.. _lland_knauf_acker_summer_hourly:

acre (summer)
-------------

The following input values agree with the ones of the respective sub-period
of the daily integration test :ref:`lland_knauf_acker_summer_daily` (global radiation
and possible sunshine duration stem from
:ref:`meteo_glob_morsim_hourly_simulation_summer`):

>>> inputs.nied.series = 0.0
>>> inputs.teml.series = (
...     16.9, 16.6, 16.4, 16.3, 16.0, 15.9, 16.0, 16.6, 17.4, 19.0, 20.3, 21.4, 21.3,
...     21.8, 22.9, 22.7, 22.5, 21.9, 21.4, 20.7, 19.4, 17.8, 17.0, 16.4, 15.9, 15.4,
...     15.2, 14.9, 14.3, 14.1, 14.3, 15.4, 16.8, 18.2, 19.5, 20.3, 21.0, 21.8, 21.9,
...     21.8, 22.2, 21.8, 20.9, 19.7, 17.9, 15.8, 14.8, 14.1, 13.4, 12.8, 12.3, 12.0,
...     11.7, 11.6, 12.6, 14.8, 17.1, 19.4, 20.6, 21.4, 22.5, 23.1, 23.4, 23.8, 24.0,
...     24.0, 23.7, 22.8, 21.3, 19.3, 18.3, 17.4)
>>> inputs.windspeed.series = (
...     0.8, 0.8, 0.8, 0.8, 0.8, 0.6, 0.9, 0.9, 0.9, 1.3, 1.5, 1.2, 1.3, 1.5, 1.9, 1.9,
...     2.3, 2.4, 2.5, 2.5, 2.2, 1.7, 1.7, 2.3, 2.3, 2.2, 2.3, 2.0, 2.3, 2.5, 2.4, 2.5,
...     2.5, 2.9, 3.1, 3.3, 3.3, 3.2, 2.5, 2.9, 3.6, 4.2, 4.2, 3.6, 3.2, 2.2, 2.2, 2.1,
...     1.8, 1.8, 1.6, 1.2, 1.9, 1.3, 1.6, 1.9, 3.2, 4.0, 4.6, 4.1, 4.1, 4.4, 4.6, 4.5,
...     4.7, 4.5, 4.1, 3.8, 2.5, 1.9, 2.2, 2.7)
>>> inputs.relativehumidity.series = (
...     95.1, 94.9, 95.9, 96.7, 97.2, 97.5, 97.7, 97.4, 96.8, 86.1, 76.8, 71.8, 67.5,
...     66.1, 63.4, 62.4, 61.1, 62.1, 67.0, 74.5, 81.2, 86.9, 90.1, 90.9, 88.0, 89.2,
...     88.8, 89.7, 93.0, 93.6, 94.6, 93.3, 86.9, 78.9, 75.8, 73.4, 68.3, 61.3, 60.6,
...     58.8, 57.2, 57.2, 59.9, 66.1, 71.7, 80.6, 85.9, 89.3, 91.4, 93.1, 94.6, 95.6,
...     96.2, 95.7, 95.1, 96.3, 89.9, 79.2, 73.4, 67.3, 62.2, 59.6, 55.4, 52.1, 49.5,
...     48.1, 49.4, 57.0, 65.9, 73.0, 76.7, 81.8)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.5, 0.7, 0.8, 0.5, 0.4, 0.5,
...     0.5, 0.3, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.9,
...     1.0, 1.0, 0.9, 0.8, 0.9, 0.8, 0.9, 0.9, 0.9, 1.0, 1.0, 1.0, 0.3, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6, 0.9, 1.0, 0.9, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 0.5, 0.0, 0.0, 0.0)

>>> aetinputs.atmosphericpressure.series = (
...     1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1017.0, 1017.0, 1017.0, 1016.0, 1016.0, 1016.0, 1016.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1019.0, 1019.0,
...     1019.0, 1019.0, 1019.0, 1019.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

>>> inlet.sequences.sim.series = 0.0

In this and the following summer examples, we use the initial conditions of August 3rd,
calculated by the corresponding daily integration tests.  However, this is possible for
most state sequences but not for the (now higher resolved) log sequences.  Hence, we
need to define them manually:

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.1, 0.2, 0.1, 0.2, 0.2, 0.3, 0.0,
...       0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedpossiblesunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.5, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...       1.0, 1.0, 1.0, 1.0, 1.0, 0.2, 0.0, 0.0, 0.0]),
...     (model.radiationmodel.sequences.logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556, 138.888889, 222.222222,
...       305.555556, 333.333333, 388.888889, 527.777778, 444.444444, 250.0,
...       222.222222, 166.666667, 111.111111, 55.555556, 27.777778, 0.0, 0.0, 0.0,
...       0.0]),
...     (model.aetmodel.sequences.logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.8, 55.6, 138.9, 222.2, 305.6, 333.3, 388.9,
...       527.8, 444.4, 250.0, 222.2, 166.7, 111.1, 55.6, 27.8, 0.0, 0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedairtemperature,
...      [[13.2], [13.2], [13.1], [12.6], [12.7], [13.0], [13.5], [14.8], [16.2],
...       [17.7], [18.8], [19.4], [20.4], [21.0], [21.5], [21.2], [20.4], [20.7],
...       [20.2], [19.7], [19.0], [18.0], [17.5], [17.1]]),
...     (model.aetmodel.sequences.logs.loggedrelativehumidity,
...      [95.1, 94.5, 94.8, 96.4, 96.6, 97.1, 97.1, 96.7, 92.2, 88.5, 81.1, 76.5, 75.1,
...       70.8, 68.9, 69.2, 75.0, 74.0, 77.4, 81.4, 85.3, 90.1, 92.3, 93.8]),
...     (model.aetmodel.sequences.logs.loggedwindspeed2m,
...      [0.8, 1.0, 1.2, 1.3, 0.9, 1.1, 1.3, 1.3, 1.9, 2.2, 1.8, 2.3, 2.4, 2.5, 2.4,
...       2.5, 2.1, 2.2, 1.7, 1.7, 1.3, 1.3, 0.7, 0.8]))

|ESnow| and |EBdn| are the only state sequences with values depending on the current
simulation step size.  According to their unit WT/m², we must multiply their conditions
saved for August 3rd and December 10th (referring to Wd/m²) by 24 (to adapt them to
Wh/m²).  We do so by trick to ensure we do not miss one of the relevant condition
dictionaries:

>>> for key, value in locals().copy().items():
...     if key.startswith("conditions_") and ("states" in value.get("model", {})):
...         value["model"]["states"]["esnow"] *= 24
...         value["model"]["states"]["ebdn"] *= 24

This integration test deals with a dry situation.  Hence, the soil water content
(|BoWa|) shows a pronounced decline.  This decline is sharpest around noon when
evapotranspiration from the soil (|EvB|) reaches its maximum:

.. integration-test::

    >>> test("lland_knauf_acker_summer_hourly",
    ...      axis1=(fluxes.evb, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_summer)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation | nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp | wats | waes | esnow | taus |        ebdn |      bowa |     sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                          0.0 |              18.3393 | 15.632162 |  -7.607027 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007792 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007792 | 0.002165 | 0.002953 |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 79.623355 | 0.293878 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002165 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                          0.0 |            17.955254 | 15.688646 |  -5.468121 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007591 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007591 | 0.002109 | 0.005626 |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 79.623355 | 0.286287 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002109 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 |  0.0 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                          0.0 |            17.914926 | 15.719464 |  -4.083217 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007395 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007395 | 0.002054 | 0.009722 |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 | 79.623355 | 0.278892 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002054 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 |  0.0 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                          0.0 |            17.949613 | 15.733663 |  -3.398025 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007204 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007204 | 0.002001 | 0.014967 |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 79.623355 | 0.271688 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002001 |
    | 1997-08-03 04:00:00 |  0.0 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 |  0.0 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                          0.0 |            17.700204 | 15.739639 |  -1.562167 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007018 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007018 | 0.001949 | 0.019872 |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 79.623355 |  0.26467 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001949 |
    | 1997-08-03 05:00:00 |  0.0 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 |  0.0 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                          0.0 |            17.641681 | 15.723585 |  -1.058491 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006837 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006837 | 0.001899 |  0.01488 |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 | 79.623355 | 0.257833 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001899 |
    | 1997-08-03 06:00:00 |  0.0 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 |  0.0 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                          0.0 |            17.791254 | 15.701487 |   -1.79108 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |   0.01488 | 0.008643 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00666 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00666 |  0.00185 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 79.614713 | 0.251173 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00185 |
    | 1997-08-03 07:00:00 |  0.0 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 |  0.0 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                          0.0 |            18.428259 |  15.68818 |  -5.470922 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.043948 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006488 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006488 | 0.001802 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 | 79.570765 | 0.244685 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001802 |
    | 1997-08-03 08:00:00 |  0.0 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 |  0.0 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                          0.0 |            19.267642 | 15.719031 | -10.085816 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.073907 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00632 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00632 | 0.001756 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 79.496858 | 0.238365 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001756 |
    | 1997-08-03 09:00:00 |  0.0 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 |  0.0 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                          0.0 |            18.949356 | 15.805261 | -19.168437 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.160936 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006157 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006157 |  0.00171 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 79.335922 | 0.232208 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00171 |
    | 1997-08-03 10:00:00 |  0.0 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 |  0.0 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |             18.32333 | 16.000482 | -25.797109 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.275473 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005998 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005998 | 0.001666 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 79.060449 | 0.226209 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001666 |
    | 1997-08-03 11:00:00 |  0.0 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 |  0.0 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                          0.0 |            18.329406 | 16.275247 | -30.748518 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.348731 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005843 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005843 | 0.001623 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 78.711718 | 0.220366 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001623 |
    | 1997-08-03 12:00:00 |  0.0 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 |  0.0 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                          0.0 |            17.126444 | 16.609429 | -28.143424 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.360693 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005692 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005692 | 0.001581 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 78.351026 | 0.214674 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001581 |
    | 1997-08-03 13:00:00 |  0.0 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 |  0.0 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                          0.0 |             17.29206 |  16.91235 | -29.325898 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.303239 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005545 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005545 |  0.00154 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 78.047787 | 0.209129 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00154 |
    | 1997-08-03 14:00:00 |  0.0 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 |  0.0 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                          0.0 |            17.732611 | 17.229461 | -34.023233 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.282369 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005402 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005402 | 0.001501 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 77.765418 | 0.203727 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001501 |
    | 1997-08-03 15:00:00 |  0.0 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 |  0.0 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                          0.0 |            17.242763 |  17.60294 |  -30.58236 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.266636 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005262 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005262 | 0.001462 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 77.498782 | 0.198464 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001462 |
    | 1997-08-03 16:00:00 |  0.0 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 |  0.0 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                          0.0 |            16.679926 | 17.935128 |  -27.38923 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.218982 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005127 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005127 | 0.001424 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 77.279799 | 0.193338 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001424 |
    | 1997-08-03 17:00:00 |  0.0 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 |  0.0 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                          0.0 |            16.345078 | 18.228999 | -22.026006 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.147975 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004994 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004994 | 0.001387 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 77.131824 | 0.188344 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001387 |
    | 1997-08-03 18:00:00 |  0.0 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 |  0.0 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                          0.0 |            17.104042 | 18.458511 | -17.648933 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.095816 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004865 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004865 | 0.001351 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 77.036008 | 0.183479 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001351 |
    | 1997-08-03 19:00:00 |  0.0 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 |  0.0 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                          0.0 |            18.218504 | 18.635498 |  -12.38701 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.058452 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004739 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004739 | 0.001317 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 76.977556 | 0.178739 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001317 |
    | 1997-08-03 20:00:00 |  0.0 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 |  0.0 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                          0.0 |            18.321912 | 18.749342 |  -3.903945 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.032413 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004617 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004617 | 0.001282 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 76.945143 | 0.174122 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001282 |
    | 1997-08-03 21:00:00 |  0.0 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                          0.0 |            17.739217 |  18.76139 |   5.768339 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.013439 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004498 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004498 | 0.001249 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 76.931705 | 0.169625 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001249 |
    | 1997-08-03 22:00:00 |  0.0 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                          0.0 |            17.485643 |  18.65737 |   9.944218 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.007132 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004382 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004382 | 0.001217 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 76.924573 | 0.165243 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001217 |
    | 1997-08-03 23:00:00 |  0.0 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                          0.0 |            16.980884 | 18.503239 |  12.619435 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.009242 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004268 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004268 | 0.001186 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 76.915331 | 0.160975 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001186 |
    | 1997-08-04 00:00:00 |  0.0 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                          0.0 |            15.922748 | 18.317006 |  14.502035 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.015057 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004158 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004158 | 0.001155 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 76.900275 | 0.156817 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001155 |
    | 1997-08-04 01:00:00 |  0.0 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                          0.0 |            15.630887 | 18.108181 |  16.249089 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.011515 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004051 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004051 | 0.001125 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 76.888759 | 0.152766 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001125 |
    | 1997-08-04 02:00:00 |  0.0 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                          0.0 |            15.362061 | 17.878392 |  16.070354 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.012795 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003946 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003946 | 0.001096 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 76.875964 |  0.14882 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001096 |
    | 1997-08-04 03:00:00 |  0.0 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 |  0.0 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                          0.0 |            15.220845 | 17.650748 |  16.504489 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.008851 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003844 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003844 | 0.001068 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 76.867113 | 0.144976 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001068 |
    | 1997-08-04 04:00:00 |  0.0 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 |  0.0 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                          0.0 |             15.18057 | 17.417894 |  18.707366 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.004365 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003745 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003745 |  0.00104 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 76.862748 | 0.141231 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00104 |
    | 1997-08-04 05:00:00 |  0.0 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 |  0.0 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                          0.0 |            15.081661 | 17.158606 |  18.351635 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.010118 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003648 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003648 | 0.001013 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 |  76.85263 | 0.137583 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001013 |
    | 1997-08-04 06:00:00 |  0.0 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 |  0.0 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                          0.0 |            15.441741 | 16.903586 |  15.621518 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.023223 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003554 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003554 | 0.000987 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 76.829407 | 0.134029 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000987 |
    | 1997-08-04 07:00:00 |  0.0 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 |  0.0 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                          0.0 |            16.349347 | 16.681328 |   7.687968 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |   0.1075 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003462 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003462 | 0.000962 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 76.721908 | 0.130567 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000962 |
    | 1997-08-04 08:00:00 |  0.0 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 |  0.0 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                          0.0 |            16.651956 | 16.554272 |  -1.474365 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.198377 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003373 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003373 | 0.000937 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 | 76.523531 | 0.127194 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000937 |
    | 1997-08-04 09:00:00 |  0.0 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 |  0.0 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                          0.0 |            16.516515 | 16.537165 |  -9.977011 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.274475 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003286 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003286 | 0.000913 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 76.249056 | 0.123909 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000913 |
    | 1997-08-04 10:00:00 |  0.0 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 |  0.0 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                          0.0 |            17.210144 | 16.622089 | -17.267466 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.30313 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003201 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003201 | 0.000889 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 75.945926 | 0.120708 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000889 |
    | 1997-08-04 11:00:00 |  0.0 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 |  0.0 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                          0.0 |            17.512141 | 16.794499 | -21.033009 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.297686 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003118 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003118 | 0.000866 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 |  75.64824 |  0.11759 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000866 |
    | 1997-08-04 12:00:00 |  0.0 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 |  0.0 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                          0.0 |            17.013358 | 17.012095 | -23.927432 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.327488 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003037 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003037 | 0.000844 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 75.320752 | 0.114552 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000844 |
    | 1997-08-04 13:00:00 |  0.0 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 |  0.0 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                          0.0 |            16.036358 | 17.264424 | -27.213457 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.348705 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002959 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002959 | 0.000822 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 74.972048 | 0.111593 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000822 |
    | 1997-08-04 14:00:00 |  0.0 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 |  0.0 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                          0.0 |             15.95027 | 17.556185 | -26.062888 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.361278 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002883 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002883 | 0.000801 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 74.610769 | 0.108711 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000801 |
    | 1997-08-04 15:00:00 |  0.0 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 |  0.0 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                          0.0 |            15.382347 |  17.83414 |  -23.79516 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.320086 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002808 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002808 |  0.00078 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 74.290683 | 0.105903 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00078 |
    | 1997-08-04 16:00:00 |  0.0 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 |  0.0 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                          0.0 |            15.333082 | 18.084882 | -24.690709 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.271017 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002736 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002736 |  0.00076 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 74.019667 | 0.103167 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00076 |
    | 1997-08-04 17:00:00 |  0.0 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 |  0.0 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                          0.0 |            14.963779 |  18.34637 | -20.721778 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.220995 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002665 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002665 |  0.00074 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 73.798672 | 0.100502 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00074 |
    | 1997-08-04 18:00:00 |  0.0 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 |  0.0 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                          0.0 |            14.829527 | 18.560232 |  -14.03861 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.158739 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002596 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002596 | 0.000721 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 73.639933 | 0.097906 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000721 |
    | 1997-08-04 19:00:00 |  0.0 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 |  0.0 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                          0.0 |            15.195376 | 18.693895 |   -6.03663 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.093951 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002529 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002529 | 0.000703 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 73.545981 | 0.095377 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000703 |
    | 1997-08-04 20:00:00 |  0.0 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 |  0.0 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                          0.0 |            14.728848 | 18.731535 |   4.989208 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.057093 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002464 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002464 | 0.000684 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 73.488889 | 0.092914 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000684 |
    | 1997-08-04 21:00:00 |  0.0 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                          0.0 |            14.490772 | 18.636864 |  17.021185 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.019046 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0024 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0024 | 0.000667 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 73.469843 | 0.090514 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000667 |
    | 1997-08-04 22:00:00 |  0.0 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                          0.0 |            14.482324 |  18.39781 |  21.586859 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.007576 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002338 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002338 | 0.000649 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 73.462266 | 0.088176 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000649 |
    | 1997-08-04 23:00:00 |  0.0 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                          0.0 |            14.388807 | 18.103968 |  24.023805 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002278 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002278 | 0.000633 | 0.000116 |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 73.462266 | 0.085898 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000633 |
    | 1997-08-05 00:00:00 |  0.0 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                          0.0 |            14.071374 | 17.780882 |  26.285292 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002219 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002219 | 0.000616 | 0.009139 |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 73.462266 | 0.083679 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000616 |
    | 1997-08-05 01:00:00 |  0.0 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                          0.0 |            13.781417 | 17.430658 |  27.783951 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002162 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002162 |   0.0006 | 0.022403 |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 73.462266 | 0.081518 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0006 |
    | 1997-08-05 02:00:00 |  0.0 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                          0.0 |            13.550859 | 17.062451 |  28.574706 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002106 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002106 | 0.000585 | 0.040121 |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 73.462266 | 0.079412 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000585 |
    | 1997-08-05 03:00:00 |  0.0 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 |  0.0 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                          0.0 |            13.425939 | 16.684755 |  28.108527 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002051 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002051 |  0.00057 | 0.060837 |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 73.462266 | 0.077361 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00057 |
    | 1997-08-05 04:00:00 |  0.0 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 |  0.0 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                          0.0 |            13.245001 | 16.312652 |  27.675913 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001998 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001998 | 0.000555 | 0.080415 |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 73.462266 | 0.075362 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000555 |
    | 1997-08-05 05:00:00 |  0.0 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 |  0.0 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                          0.0 |            13.089234 | 15.945741 |  26.074448 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001947 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001947 | 0.000541 | 0.089679 |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 73.462266 | 0.073416 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000541 |
    | 1997-08-05 06:00:00 |  0.0 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 |  0.0 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                          0.0 |            13.893903 | 15.598048 |  17.988287 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001896 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001896 | 0.000527 | 0.037317 |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 73.462266 | 0.071519 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000527 |
    | 1997-08-05 07:00:00 |  0.0 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 |  0.0 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                          0.0 |            16.235713 | 15.347388 |   3.284331 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.037317 | 0.079358 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001847 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001847 | 0.000513 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 73.382908 | 0.069672 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000513 |
    | 1997-08-05 08:00:00 |  0.0 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 |  0.0 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                          0.0 |            17.557751 | 15.273176 | -10.960941 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.185679 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0018 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0018 |   0.0005 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 73.197229 | 0.067872 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0005 |
    | 1997-08-05 09:00:00 |  0.0 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 |  0.0 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                          0.0 |            17.870634 | 15.369908 | -24.180553 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |   0.2631 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001753 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001753 | 0.000487 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 |  72.93413 | 0.066119 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000487 |
    | 1997-08-05 10:00:00 |  0.0 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 |  0.0 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                          0.0 |            17.839281 | 15.625274 | -29.848354 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |   0.3322 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001708 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001708 | 0.000474 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 |  72.60193 | 0.064411 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000474 |
    | 1997-08-05 11:00:00 |  0.0 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 |  0.0 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                          0.0 |            17.180627 | 15.948655 | -32.708072 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.369947 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001664 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001664 | 0.000462 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 | 72.231982 | 0.062747 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000462 |
    | 1997-08-05 12:00:00 |  0.0 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 |  0.0 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                          0.0 |            16.980219 | 16.306352 | -37.161891 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.389289 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001621 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001621 |  0.00045 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 71.842694 | 0.061126 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00045 |
    | 1997-08-05 13:00:00 |  0.0 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 |  0.0 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                          0.0 |            16.872626 | 16.717494 | -38.295035 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.41464 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001579 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001579 | 0.000439 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 | 71.428054 | 0.059547 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000439 |
    | 1997-08-05 14:00:00 |  0.0 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 |  0.0 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                          0.0 |            15.970201 | 17.142235 | -37.546592 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.419328 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001538 | 0.000427 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 71.008726 | 0.058009 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000427 |
    | 1997-08-05 15:00:00 |  0.0 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 |  0.0 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                          0.0 |            15.384928 | 17.557994 | -37.452038 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.39307 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001498 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001498 | 0.000416 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 70.615656 | 0.056511 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000416 |
    | 1997-08-05 16:00:00 |  0.0 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 |  0.0 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                          0.0 |            14.793787 | 17.972618 | -36.164291 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.33964 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00146 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00146 | 0.000405 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 70.276016 | 0.055051 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000405 |
    | 1997-08-05 17:00:00 |  0.0 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 |  0.0 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                          0.0 |            14.375377 |  18.37179 | -33.769262 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.275457 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001422 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001422 | 0.000395 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 70.000559 | 0.053629 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000395 |
    | 1997-08-05 18:00:00 |  0.0 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 |  0.0 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                          0.0 |            14.500182 | 18.742221 | -29.746675 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.207398 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001385 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001385 | 0.000385 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 69.793161 | 0.052244 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000385 |
    | 1997-08-05 19:00:00 |  0.0 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 |  0.0 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                          0.0 |            15.846331 | 19.064381 | -22.413715 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.133656 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00135 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00135 | 0.000375 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 69.659505 | 0.050894 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000375 |
    | 1997-08-05 20:00:00 |  0.0 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 |  0.0 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                          0.0 |            16.720484 | 19.298545 | -12.008727 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.066112 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001315 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001315 | 0.000365 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 69.593394 |  0.04958 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000365 |
    | 1997-08-05 21:00:00 |  0.0 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                          0.0 |            16.369483 |  19.40785 |   0.647101 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.03182 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001281 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001281 | 0.000356 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 69.561574 | 0.048299 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000356 |
    | 1997-08-05 22:00:00 |  0.0 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                          0.0 |            16.157086 | 19.365285 |    6.39171 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 |  0.02777 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001248 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001248 | 0.000347 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 69.533804 | 0.047051 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000347 |
    | 1997-08-05 23:00:00 |  0.0 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 |  0.0 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                          0.0 |            16.281953 | 19.253784 |  11.122707 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |       0.0 | 0.021514 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001215 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001215 | 0.000338 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 |  69.51229 | 0.045836 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000338 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_summer))
0.0

.. _lland_knauf_acker_heavy_rain_hourly:

acre (heavy rain)
-----------------

When comparing the soil moisture content (|BoWa|) on August 6, 00:00, we notice only
minor differences between the daily (:ref:`lland_knauf_acker_summer_daily`, 70.2 mm)
and the hourly (:ref:`lland_knauf_acker_summer_daily`, 69.5) simulation results for a
dry situation.  However, when we compare the differences between our daily
(:ref:`lland_knauf_acker_heavy_rain_daily`, 152.8 mm) and our hourly test for a wet
situation, we see much more pronounced differences, which are mainly due to the
prioritisation of throughfall (|NBes|) over interception evaporation (|EvI|):

.. integration-test::

    >>> inputs.nied.series = 20.0 / 24.0
    >>> control.negq(False)
    >>> test("lland_knauf_acker_heavy_rain_hourly",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_heavy_rain)
    |                date |     nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |     nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |     nbes | sbes |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |     wada |      qdb |     qib1 | qib2 |      qbb | qkap |     qdgz |    qdgz1 | qdgz2 |    qigz1 | qigz2 |     qbgz |    qdga1 | qdga2 |    qiga1 | qiga2 |     qbga |      qah |       qa |     inzp | wats | waes | esnow | taus |        ebdn |       bowa |     sdg1 | sdg2 |     sig1 | sig2 |      sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.833333 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                          0.0 |              18.3393 | 15.632162 |  -7.607027 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.007403 |      0.0 |  0.0 |      0.0 |  0.0 | 0.007403 | 0.007403 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040965 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040965 | 0.011379 | 0.702953 |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 102.562669 |   1.5486 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011379 |
    | 1997-08-03 01:00:00 | 0.833333 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                          0.0 |            17.955254 | 15.688646 |  -5.468121 |                       0.0 |                      0.0 |              0.0 | 0.786286 |  0.0 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786286 | 0.070171 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070171 | 0.070171 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040912 |   0.0 |      0.0 |   0.0 |      0.0 | 0.040912 | 0.011364 | 0.702673 |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 103.278784 |  1.57786 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011364 |
    | 1997-08-03 02:00:00 | 0.833333 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.783333 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                          0.0 |            17.914926 | 15.719464 |  -4.083217 |                       0.0 |                      0.0 |              0.0 | 0.786007 |  0.0 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786007 | 0.070721 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070721 | 0.070721 |   0.0 |      0.0 |   0.0 |      0.0 | 0.041675 |   0.0 |      0.0 |   0.0 |      0.0 | 0.041675 | 0.011576 | 0.704096 |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 |  103.99407 | 1.606906 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011576 |
    | 1997-08-03 03:00:00 | 0.833333 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 | 0.783333 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                          0.0 |            17.949613 | 15.733663 |  -3.398025 |                       0.0 |                      0.0 |              0.0 | 0.787429 |  0.0 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.787429 | 0.071426 |      0.0 |  0.0 |      0.0 |  0.0 | 0.071426 | 0.071426 |   0.0 |      0.0 |   0.0 |      0.0 | 0.042434 |   0.0 |      0.0 |   0.0 |      0.0 | 0.042434 | 0.011787 | 0.705245 |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 104.710073 | 1.635897 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011787 |
    | 1997-08-03 04:00:00 | 0.833333 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 | 0.783333 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                          0.0 |            17.700204 | 15.739639 |  -1.562167 |                       0.0 |                      0.0 |              0.0 | 0.788579 |  0.0 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788579 |  0.07211 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07211 |  0.07211 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043192 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043192 | 0.011998 | 0.704905 |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 105.426542 | 1.664815 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.011998 |
    | 1997-08-03 05:00:00 | 0.833333 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 | 0.783333 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                          0.0 |            17.641681 | 15.723585 |  -1.058491 |                       0.0 |                      0.0 |              0.0 | 0.788239 |  0.0 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788239 |  0.07266 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07266 |  0.07266 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043946 |   0.0 |      0.0 |   0.0 |      0.0 | 0.043946 | 0.012207 | 0.695008 |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 |  106.14212 | 1.693529 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012207 |
    | 1997-08-03 06:00:00 | 0.833333 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 | 0.783333 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                          0.0 |            17.791254 | 15.701487 |   -1.79108 |                       0.0 |                      0.0 |              0.0 | 0.778341 |  0.0 |  0.026203 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.778341 | 0.072319 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072319 | 0.072319 |   0.0 |      0.0 |   0.0 |      0.0 | 0.044683 |   0.0 |      0.0 |   0.0 |      0.0 | 0.044683 | 0.012412 | 0.673797 |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 106.848143 | 1.721164 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012412 |
    | 1997-08-03 07:00:00 | 0.833333 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 | 0.783333 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                          0.0 |            18.428259 |  15.68818 |  -5.470922 |                       0.0 |                      0.0 |              0.0 |  0.75713 |  0.0 |  0.057366 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.75713 | 0.070894 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070894 | 0.070894 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045379 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045379 | 0.012605 | 0.642634 |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 |  107.53438 | 1.746679 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012605 |
    | 1997-08-03 08:00:00 | 0.833333 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 | 0.783333 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                          0.0 |            19.267642 | 15.719031 | -10.085816 |                       0.0 |                      0.0 |              0.0 | 0.725967 |  0.0 |  0.096008 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.725967 | 0.068482 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068482 | 0.068482 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046007 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046007 |  0.01278 | 0.603992 |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 108.191865 | 1.769154 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01278 |
    | 1997-08-03 09:00:00 | 0.833333 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 | 0.783333 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                          0.0 |            18.949356 | 15.805261 | -19.168437 |                       0.0 |                      0.0 |              0.0 | 0.687326 |  0.0 |  0.222649 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687326 | 0.065294 |      0.0 |  0.0 |      0.0 |  0.0 | 0.065294 | 0.065294 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046546 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046546 | 0.012929 | 0.477351 |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 108.813897 | 1.787903 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012929 |
    | 1997-08-03 10:00:00 | 0.833333 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 | 0.783333 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |             18.32333 | 16.000482 | -25.797109 |                       0.0 |                      0.0 |              0.0 | 0.560684 |  0.0 |  0.389789 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.560684 | 0.053594 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053594 | 0.053594 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046878 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046878 | 0.013022 | 0.310211 |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 109.320987 | 1.794618 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013022 |
    | 1997-08-03 11:00:00 | 0.833333 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 | 0.783333 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                          0.0 |            18.329406 | 16.275247 | -30.748518 |                       0.0 |                      0.0 |              0.0 | 0.393544 |  0.0 |   0.46611 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.393544 | 0.037795 |      0.0 |  0.0 |      0.0 |  0.0 | 0.037795 | 0.037795 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046847 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046847 | 0.013013 |  0.23389 |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 109.676737 | 1.785566 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013013 |
    | 1997-08-03 12:00:00 | 0.833333 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 | 0.783333 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                          0.0 |            17.126444 | 16.609429 | -28.143424 |                       0.0 |                      0.0 |              0.0 | 0.317224 |  0.0 |  0.490849 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.317224 | 0.030572 |      0.0 |  0.0 |      0.0 |  0.0 | 0.030572 | 0.030572 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046519 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046519 | 0.012922 | 0.209151 |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 109.963389 | 1.769618 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012922 |
    | 1997-08-03 13:00:00 | 0.833333 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 | 0.783333 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                          0.0 |             17.29206 |  16.91235 | -29.325898 |                       0.0 |                      0.0 |              0.0 | 0.292485 |  0.0 |  0.424547 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.292485 | 0.028272 |      0.0 |  0.0 |      0.0 |  0.0 | 0.028272 | 0.028272 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046078 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046078 | 0.012799 | 0.275453 |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 110.227602 | 1.751813 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012799 |
    | 1997-08-03 14:00:00 | 0.833333 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 | 0.783333 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                          0.0 |            17.732611 | 17.229461 | -34.023233 |                       0.0 |                      0.0 |              0.0 | 0.358786 |  0.0 |  0.415176 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.358786 | 0.034791 |      0.0 |  0.0 |      0.0 |  0.0 | 0.034791 | 0.034791 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045702 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045702 | 0.012695 | 0.284824 |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 110.551596 | 1.740902 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012695 |
    | 1997-08-03 15:00:00 | 0.833333 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 | 0.783333 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                          0.0 |            17.242763 |  17.60294 |  -30.58236 |                       0.0 |                      0.0 |              0.0 | 0.368157 |  0.0 |  0.393194 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.368157 | 0.035827 |      0.0 |  0.0 |      0.0 |  0.0 | 0.035827 | 0.035827 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045434 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045434 |  0.01262 | 0.306806 |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 110.883927 | 1.731295 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01262 |
    | 1997-08-03 16:00:00 | 0.833333 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 | 0.783333 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                          0.0 |            16.679926 | 17.935128 |  -27.38923 |                       0.0 |                      0.0 |              0.0 | 0.390139 |  0.0 |   0.34217 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.390139 | 0.038106 |      0.0 |  0.0 |      0.0 |  0.0 | 0.038106 | 0.038106 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045215 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045215 |  0.01256 |  0.35783 |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 111.235959 | 1.724186 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01256 |
    | 1997-08-03 17:00:00 | 0.833333 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 | 0.783333 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                          0.0 |            16.345078 | 18.228999 | -22.026006 |                       0.0 |                      0.0 |              0.0 | 0.441163 |  0.0 |  0.236219 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.441163 | 0.043264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.043264 | 0.043264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045098 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045098 | 0.012527 | 0.463781 |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 111.633858 | 1.722352 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012527 |
    | 1997-08-03 18:00:00 | 0.833333 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 | 0.783333 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                          0.0 |            17.104042 | 18.458511 | -17.648933 |                       0.0 |                      0.0 |              0.0 | 0.547114 |  0.0 |  0.156077 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.547114 | 0.053912 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053912 | 0.053912 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045189 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045189 | 0.012553 | 0.543923 |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 112.127061 | 1.731075 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012553 |
    | 1997-08-03 19:00:00 | 0.833333 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 | 0.783333 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                          0.0 |            18.218504 | 18.635498 |  -12.38701 |                       0.0 |                      0.0 |              0.0 | 0.627256 |  0.0 |   0.09611 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.627256 |  0.06216 |      0.0 |  0.0 |      0.0 |  0.0 |  0.06216 |  0.06216 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045521 |   0.0 |      0.0 |   0.0 |      0.0 | 0.045521 | 0.012645 |  0.60389 |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 112.692157 | 1.747713 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012645 |
    | 1997-08-03 20:00:00 | 0.833333 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 | 0.783333 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                          0.0 |            18.321912 | 18.749342 |  -3.903945 |                       0.0 |                      0.0 |              0.0 | 0.687223 |  0.0 |  0.049374 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687223 | 0.068533 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068533 | 0.068533 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046034 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046034 | 0.012787 | 0.650626 |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 113.310848 | 1.770212 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012787 |
    | 1997-08-03 21:00:00 | 0.833333 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                          0.0 |            17.739217 |  18.76139 |   5.768339 |                       0.0 |                      0.0 |              0.0 | 0.733959 |  0.0 |  0.019253 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.733959 | 0.073692 |      0.0 |  0.0 |      0.0 |  0.0 | 0.073692 | 0.073692 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046682 |   0.0 |      0.0 |   0.0 |      0.0 | 0.046682 | 0.012967 | 0.680747 |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 113.971116 | 1.797222 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.012967 |
    | 1997-08-03 22:00:00 | 0.833333 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                          0.0 |            17.485643 |  18.65737 |   9.944218 |                       0.0 |                      0.0 |              0.0 | 0.764081 |  0.0 |  0.010292 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.764081 | 0.077264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.077264 | 0.077264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.047426 |   0.0 |      0.0 |   0.0 |      0.0 | 0.047426 | 0.013174 | 0.689708 |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 114.657933 | 1.827059 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013174 |
    | 1997-08-03 23:00:00 | 0.833333 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                          0.0 |            16.980884 | 18.503239 |  12.619435 |                       0.0 |                      0.0 |              0.0 | 0.773041 |  0.0 |  0.014655 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.773041 | 0.078739 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078739 | 0.078739 |   0.0 |      0.0 |   0.0 |      0.0 | 0.048216 |   0.0 |      0.0 |   0.0 |      0.0 | 0.048216 | 0.013393 | 0.685345 |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 115.352235 | 1.857583 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013393 |
    | 1997-08-04 00:00:00 | 0.833333 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                          0.0 |            15.922748 | 18.317006 |  14.502035 |                       0.0 |                      0.0 |              0.0 | 0.768678 |  0.0 |  0.024014 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.768678 | 0.078864 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078864 | 0.078864 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049006 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049006 | 0.013613 | 0.675986 |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 116.042049 | 1.887441 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013613 |
    | 1997-08-04 01:00:00 | 0.833333 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                          0.0 |            15.630887 | 18.108181 |  16.249089 |                       0.0 |                      0.0 |              0.0 | 0.759319 |  0.0 |  0.018207 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.759319 | 0.078462 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078462 | 0.078462 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049772 |   0.0 |      0.0 |   0.0 |      0.0 | 0.049772 | 0.013826 | 0.681793 |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 116.722906 | 1.916131 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.013826 |
    | 1997-08-04 02:00:00 | 0.833333 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                          0.0 |            15.362061 | 17.878392 |  16.070354 |                       0.0 |                      0.0 |              0.0 | 0.765127 |  0.0 |  0.020574 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765127 | 0.079624 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079624 | 0.079624 |   0.0 |      0.0 |   0.0 |      0.0 | 0.050528 |   0.0 |      0.0 |   0.0 |      0.0 | 0.050528 | 0.014036 | 0.679426 |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 117.408409 | 1.945227 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014036 |
    | 1997-08-04 03:00:00 | 0.833333 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.783333 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                          0.0 |            15.220845 | 17.650748 |  16.504489 |                       0.0 |                      0.0 |              0.0 | 0.762759 |  0.0 |  0.013655 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.762759 | 0.079941 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079941 | 0.079941 |   0.0 |      0.0 |   0.0 |      0.0 | 0.051284 |   0.0 |      0.0 |   0.0 |      0.0 | 0.051284 | 0.014246 | 0.686345 |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 118.091227 | 1.973884 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014246 |
    | 1997-08-04 04:00:00 | 0.833333 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 | 0.783333 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                          0.0 |             15.18057 | 17.417894 |  18.707366 |                       0.0 |                      0.0 |              0.0 | 0.769679 |  0.0 |  0.007264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.769679 | 0.081237 |      0.0 |  0.0 |      0.0 |  0.0 | 0.081237 | 0.081237 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052041 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052041 | 0.014456 | 0.692736 |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 118.779668 |  2.00308 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 | 0.014456 |
    | 1997-08-04 05:00:00 | 0.833333 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 | 0.783333 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                          0.0 |            15.081661 | 17.158606 |  18.351635 |                       0.0 |                      0.0 |              0.0 | 0.776069 |  0.0 |  0.018172 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.776069 | 0.082493 |      0.0 |  0.0 |      0.0 |  0.0 | 0.082493 | 0.082493 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052811 |   0.0 |      0.0 |   0.0 |      0.0 | 0.052811 |  0.01467 | 0.681828 |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 | 119.473244 | 2.032762 |  0.0 |      0.0 |  0.0 |      0.0 |   0.0 |  0.01467 |
    | 1997-08-04 06:00:00 | 0.833333 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 | 0.783333 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                          0.0 |            15.441741 | 16.903586 |  15.621518 |                       0.0 |                      0.0 |              0.0 | 0.765161 |  0.0 |   0.04094 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765161 | 0.081907 | 0.003897 |  0.0 | 0.000003 |  0.0 | 0.081907 | 0.081907 |   0.0 | 0.003897 |   0.0 | 0.000003 |  0.05357 |   0.0 | 0.000025 |   0.0 |      0.0 | 0.053596 | 0.014888 |  0.65906 |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 120.152598 | 2.061098 |  0.0 | 0.003872 |  0.0 | 0.000003 |   0.0 | 0.014888 |
    | 1997-08-04 07:00:00 | 0.833333 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 | 0.783333 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                          0.0 |            16.349347 | 16.681328 |   7.687968 |                       0.0 |                      0.0 |              0.0 | 0.742393 |  0.0 |  0.189733 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.742393 | 0.080011 |  0.00392 |  0.0 | 0.000031 |  0.0 | 0.080011 | 0.080011 |   0.0 |  0.00392 |   0.0 | 0.000031 | 0.054278 |   0.0 | 0.000076 |   0.0 |      0.0 | 0.054354 | 0.015098 | 0.510267 |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 120.811029 | 2.086831 |  0.0 | 0.007716 |  0.0 | 0.000034 |   0.0 | 0.015098 |
    | 1997-08-04 08:00:00 | 0.833333 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 | 0.783333 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                          0.0 |            16.651956 | 16.554272 |  -1.474365 |                       0.0 |                      0.0 |              0.0 |   0.5936 |  0.0 |  0.343753 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   0.5936 | 0.064359 | 0.003941 |  0.0 | 0.000059 |  0.0 | 0.064359 | 0.064359 |   0.0 | 0.003941 |   0.0 | 0.000059 |  0.05474 |   0.0 | 0.000126 |   0.0 |      0.0 | 0.054865 |  0.01524 | 0.356247 |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 |  121.33627 | 2.096451 |  0.0 | 0.011531 |  0.0 | 0.000093 |   0.0 |  0.01524 |
    | 1997-08-04 09:00:00 | 0.833333 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 | 0.783333 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                          0.0 |            16.516515 | 16.537165 |  -9.977011 |                       0.0 |                      0.0 |              0.0 | 0.439581 |  0.0 |  0.494205 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.439581 |  0.04788 | 0.003958 |  0.0 | 0.000081 |  0.0 |  0.04788 |  0.04788 |   0.0 | 0.003958 |   0.0 | 0.000081 | 0.054774 |   0.0 | 0.000176 |   0.0 |      0.0 |  0.05495 | 0.015264 | 0.205795 |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 121.723932 | 2.089557 |  0.0 | 0.015313 |  0.0 | 0.000174 |   0.0 | 0.015264 |
    | 1997-08-04 10:00:00 | 0.833333 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 | 0.783333 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                          0.0 |            17.210144 | 16.622089 | -17.267466 |                       0.0 |                      0.0 |              0.0 | 0.289128 |  0.0 |  0.550473 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.289128 | 0.031594 | 0.003971 |  0.0 | 0.000097 |  0.0 | 0.031594 | 0.031594 |   0.0 | 0.003971 |   0.0 | 0.000097 | 0.054385 |   0.0 | 0.000225 |   0.0 |      0.0 |  0.05461 | 0.015169 | 0.149527 |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 121.977399 | 2.066766 |  0.0 | 0.019059 |  0.0 | 0.000271 |   0.0 | 0.015169 |
    | 1997-08-04 11:00:00 | 0.833333 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 | 0.783333 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                          0.0 |            17.512141 | 16.794499 | -21.033009 |                       0.0 |                      0.0 |              0.0 |  0.23286 |  0.0 |  0.548409 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.23286 | 0.025504 | 0.003979 |  0.0 | 0.000107 |  0.0 | 0.025504 | 0.025504 |   0.0 | 0.003979 |   0.0 | 0.000107 | 0.053717 |   0.0 | 0.000274 |   0.0 |      0.0 | 0.053991 | 0.014997 | 0.151591 |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 | 122.180668 | 2.038553 |  0.0 | 0.022764 |  0.0 | 0.000378 |   0.0 | 0.014997 |
    | 1997-08-04 12:00:00 | 0.833333 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 | 0.783333 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                          0.0 |            17.013358 | 17.012095 | -23.927432 |                       0.0 |                      0.0 |              0.0 | 0.234924 |  0.0 |  0.597585 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.234924 | 0.025783 | 0.003986 |  0.0 | 0.000116 |  0.0 | 0.025783 | 0.025783 |   0.0 | 0.003986 |   0.0 | 0.000116 | 0.052992 |   0.0 | 0.000322 |   0.0 |      0.0 | 0.053314 | 0.014809 | 0.102415 |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 122.385708 | 2.011344 |  0.0 | 0.026428 |  0.0 | 0.000494 |   0.0 | 0.014809 |
    | 1997-08-04 13:00:00 | 0.833333 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 | 0.783333 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                          0.0 |            16.036358 | 17.264424 | -27.213457 |                       0.0 |                      0.0 |              0.0 | 0.185749 |  0.0 |  0.621828 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.185749 | 0.020423 | 0.003992 |  0.0 | 0.000124 |  0.0 | 0.020423 | 0.020423 |   0.0 | 0.003992 |   0.0 | 0.000124 |  0.05222 |   0.0 |  0.00037 |   0.0 |      0.0 | 0.052589 | 0.014608 | 0.078172 |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 122.546917 | 1.979547 |  0.0 | 0.030051 |  0.0 | 0.000618 |   0.0 | 0.014608 |
    | 1997-08-04 14:00:00 | 0.833333 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 | 0.783333 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                          0.0 |             15.95027 | 17.556185 | -26.062888 |                       0.0 |                      0.0 |              0.0 | 0.161505 |  0.0 |  0.589213 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.161505 | 0.017784 | 0.003998 |  0.0 | 0.000131 |  0.0 | 0.017784 | 0.017784 |   0.0 | 0.003998 |   0.0 | 0.000131 | 0.051364 |   0.0 | 0.000417 |   0.0 |      0.0 | 0.051781 | 0.014384 | 0.110787 |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 122.686509 | 1.945967 |  0.0 | 0.033632 |  0.0 | 0.000749 |   0.0 | 0.014384 |
    | 1997-08-04 15:00:00 | 0.833333 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 | 0.783333 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                          0.0 |            15.382347 |  17.83414 |  -23.79516 |                       0.0 |                      0.0 |              0.0 | 0.194121 |  0.0 |  0.551455 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.194121 | 0.021409 | 0.004002 |  0.0 | 0.000137 |  0.0 | 0.021409 | 0.021409 |   0.0 | 0.004002 |   0.0 | 0.000137 | 0.050544 |   0.0 | 0.000463 |   0.0 |      0.0 | 0.051007 | 0.014169 | 0.148545 |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 122.855082 | 1.916832 |  0.0 | 0.037171 |  0.0 | 0.000886 |   0.0 | 0.014169 |
    | 1997-08-04 16:00:00 | 0.833333 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 | 0.783333 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                          0.0 |            15.333082 | 18.084882 | -24.690709 |                       0.0 |                      0.0 |              0.0 | 0.231878 |  0.0 |  0.505795 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.231878 |  0.02562 | 0.004008 |  0.0 | 0.000144 |  0.0 |  0.02562 |  0.02562 |   0.0 | 0.004008 |   0.0 | 0.000144 | 0.049846 |   0.0 | 0.000509 |   0.0 |      0.0 | 0.050355 | 0.013988 | 0.194205 |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 123.057187 | 1.892607 |  0.0 | 0.040669 |  0.0 |  0.00103 |   0.0 | 0.013988 |
    | 1997-08-04 17:00:00 | 0.833333 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 | 0.783333 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                          0.0 |            14.963779 |  18.34637 | -20.721778 |                       0.0 |                      0.0 |              0.0 | 0.277538 |  0.0 |   0.44464 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.277538 | 0.030734 | 0.004014 |  0.0 | 0.000152 |  0.0 | 0.030734 | 0.030734 |   0.0 | 0.004014 |   0.0 | 0.000152 | 0.049286 |   0.0 | 0.000555 |   0.0 |      0.0 | 0.049841 | 0.013845 |  0.25536 |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 123.299825 | 1.874054 |  0.0 | 0.044129 |  0.0 | 0.001182 |   0.0 | 0.013845 |
    | 1997-08-04 18:00:00 | 0.833333 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 | 0.783333 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                          0.0 |            14.829527 | 18.560232 |  -14.03861 |                       0.0 |                      0.0 |              0.0 | 0.338693 |  0.0 |   0.32482 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.338693 | 0.037607 | 0.004022 |  0.0 | 0.000162 |  0.0 | 0.037607 | 0.037607 |   0.0 | 0.004022 |   0.0 | 0.000162 | 0.048896 |   0.0 |   0.0006 |   0.0 |      0.0 | 0.049496 | 0.013749 |  0.37518 |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 123.596727 | 1.862765 |  0.0 | 0.047551 |  0.0 | 0.001344 |   0.0 | 0.013749 |
    | 1997-08-04 19:00:00 | 0.833333 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 | 0.783333 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                          0.0 |            15.195376 | 18.693895 |   -6.03663 |                       0.0 |                      0.0 |              0.0 | 0.458513 |  0.0 |  0.183135 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.458513 | 0.051088 | 0.004032 |  0.0 | 0.000175 |  0.0 | 0.051088 | 0.051088 |   0.0 | 0.004032 |   0.0 | 0.000175 |  0.04878 |   0.0 | 0.000644 |   0.0 |      0.0 | 0.049424 | 0.013729 | 0.516865 |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 123.999945 | 1.865073 |  0.0 | 0.050939 |  0.0 | 0.001519 |   0.0 | 0.013729 |
    | 1997-08-04 20:00:00 | 0.833333 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 | 0.783333 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                          0.0 |            14.728848 | 18.731535 |   4.989208 |                       0.0 |                      0.0 |              0.0 | 0.600198 |  0.0 |  0.100625 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.600198 | 0.067185 | 0.004045 |  0.0 | 0.000192 |  0.0 | 0.067185 | 0.067185 |   0.0 | 0.004045 |   0.0 | 0.000192 | 0.049048 |   0.0 | 0.000689 |   0.0 |      0.0 | 0.049737 | 0.013816 | 0.599375 |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 124.528721 |  1.88321 |  0.0 | 0.054295 |  0.0 |  0.00171 |   0.0 | 0.013816 |
    | 1997-08-04 21:00:00 | 0.833333 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                          0.0 |            14.490772 | 18.636864 |  17.021185 |                       0.0 |                      0.0 |              0.0 | 0.682708 |  0.0 |  0.029975 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.682708 |  0.07685 | 0.004062 |  0.0 | 0.000214 |  0.0 |  0.07685 |  0.07685 |   0.0 | 0.004062 |   0.0 | 0.000214 | 0.049642 |   0.0 | 0.000732 |   0.0 |      0.0 | 0.050374 | 0.013993 | 0.670025 |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 125.130304 | 1.910418 |  0.0 | 0.057625 |  0.0 | 0.001924 |   0.0 | 0.013993 |
    | 1997-08-04 22:00:00 | 0.833333 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                          0.0 |            14.482324 |  18.39781 |  21.586859 |                       0.0 |                      0.0 |              0.0 | 0.753358 |  0.0 |  0.012059 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.753358 | 0.085333 | 0.004082 |  0.0 | 0.000239 |  0.0 | 0.085333 | 0.085333 |   0.0 | 0.004082 |   0.0 | 0.000239 | 0.050455 |   0.0 | 0.000776 |   0.0 |      0.0 | 0.051231 | 0.014231 | 0.687941 |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 125.794008 | 1.945297 |  0.0 | 0.060931 |  0.0 | 0.002162 |   0.0 | 0.014231 |
    | 1997-08-04 23:00:00 | 0.833333 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                          0.0 |            14.388807 | 18.103968 |  24.023805 |                       0.0 |                      0.0 |              0.0 | 0.771275 |  0.0 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.771275 | 0.087941 | 0.004104 |  0.0 | 0.000266 |  0.0 | 0.087941 | 0.087941 |   0.0 | 0.004104 |   0.0 | 0.000266 | 0.051389 |   0.0 | 0.000819 |   0.0 |      0.0 | 0.052209 | 0.014502 | 0.700116 |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 126.472972 | 1.981848 |  0.0 | 0.064216 |  0.0 | 0.002428 |   0.0 | 0.014502 |
    | 1997-08-05 00:00:00 | 0.833333 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                          0.0 |            14.071374 | 17.780882 |  26.285292 |                       0.0 |                      0.0 |              0.0 |  0.78345 |  0.0 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.78345 | 0.089929 | 0.004126 |  0.0 | 0.000295 |  0.0 | 0.089929 | 0.089929 |   0.0 | 0.004126 |   0.0 | 0.000295 | 0.052359 |   0.0 | 0.000862 |   0.0 |      0.0 | 0.053221 | 0.014784 | 0.709023 |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 127.162072 | 2.019418 |  0.0 |  0.06748 |  0.0 | 0.002723 |   0.0 | 0.014784 |
    | 1997-08-05 01:00:00 | 0.833333 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                          0.0 |            13.781417 | 17.430658 |  27.783951 |                       0.0 |                      0.0 |              0.0 | 0.792356 |  0.0 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792356 | 0.091568 | 0.004148 |  0.0 | 0.000323 |  0.0 | 0.091568 | 0.091568 |   0.0 | 0.004148 |   0.0 | 0.000323 | 0.053351 |   0.0 | 0.000904 |   0.0 |      0.0 | 0.054256 | 0.015071 | 0.713264 |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 127.858389 | 2.057635 |  0.0 | 0.070724 |  0.0 | 0.003046 |   0.0 | 0.015071 |
    | 1997-08-05 02:00:00 | 0.833333 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                          0.0 |            13.550859 | 17.062451 |  28.574706 |                       0.0 |                      0.0 |              0.0 | 0.796598 |  0.0 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.796598 | 0.092684 | 0.004171 |  0.0 | 0.000352 |  0.0 | 0.092684 | 0.092684 |   0.0 | 0.004171 |   0.0 | 0.000352 | 0.054353 |   0.0 | 0.000947 |   0.0 |      0.0 |   0.0553 | 0.015361 | 0.717718 |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 128.557779 | 2.095966 |  0.0 | 0.073948 |  0.0 | 0.003398 |   0.0 | 0.015361 |
    | 1997-08-05 03:00:00 | 0.833333 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.783333 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                          0.0 |            13.425939 | 16.684755 |  28.108527 |                       0.0 |                      0.0 |              0.0 | 0.801051 |  0.0 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.801051 | 0.093837 | 0.004194 |  0.0 | 0.000382 |  0.0 | 0.093837 | 0.093837 |   0.0 | 0.004194 |   0.0 | 0.000382 | 0.055358 |   0.0 | 0.000989 |   0.0 | 0.000001 | 0.056347 | 0.015652 | 0.720716 |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 129.260418 | 2.134445 |  0.0 | 0.077153 |  0.0 | 0.003779 |   0.0 | 0.015652 |
    | 1997-08-05 04:00:00 | 0.833333 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 | 0.783333 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                          0.0 |            13.245001 | 16.312652 |  27.675913 |                       0.0 |                      0.0 |              0.0 | 0.804049 |  0.0 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.804049 | 0.094829 | 0.004217 |  0.0 | 0.000411 |  0.0 | 0.094829 | 0.094829 |   0.0 | 0.004217 |   0.0 | 0.000411 | 0.056365 |   0.0 |  0.00103 |   0.0 | 0.000001 | 0.057396 | 0.015943 | 0.719578 |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 129.965011 | 2.172909 |  0.0 |  0.08034 |  0.0 | 0.004189 |   0.0 | 0.015943 |
    | 1997-08-05 05:00:00 | 0.833333 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 | 0.783333 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                          0.0 |            13.089234 | 15.945741 |  26.074448 |                       0.0 |                      0.0 |              0.0 | 0.802911 |  0.0 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.802911 | 0.095337 |  0.00424 |  0.0 |  0.00044 |  0.0 | 0.095337 | 0.095337 |   0.0 |  0.00424 |   0.0 |  0.00044 | 0.057365 |   0.0 | 0.001072 |   0.0 | 0.000001 | 0.058437 | 0.016233 | 0.709263 |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 130.667906 | 2.210881 |  0.0 | 0.083507 |  0.0 | 0.004628 |   0.0 | 0.016233 |
    | 1997-08-05 06:00:00 | 0.833333 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 | 0.783333 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                          0.0 |            13.893903 | 15.598048 |  17.988287 |                       0.0 |                      0.0 |              0.0 | 0.792597 |  0.0 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792597 | 0.094743 | 0.004263 |  0.0 | 0.000469 |  0.0 | 0.094743 | 0.094743 |   0.0 | 0.004263 |   0.0 | 0.000469 | 0.058338 |   0.0 | 0.001113 |   0.0 | 0.000001 | 0.059452 | 0.016514 | 0.647639 |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 131.361028 | 2.247286 |  0.0 | 0.086656 |  0.0 | 0.005097 |   0.0 | 0.016514 |
    | 1997-08-05 07:00:00 | 0.833333 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 | 0.783333 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                          0.0 |            16.235713 | 15.347388 |   3.284331 |                       0.0 |                      0.0 |              0.0 | 0.730972 |  0.0 |  0.166779 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.730972 | 0.087933 | 0.004285 |  0.0 | 0.000498 |  0.0 | 0.087933 | 0.087933 |   0.0 | 0.004285 |   0.0 | 0.000498 |  0.05919 |   0.0 | 0.001154 |   0.0 | 0.000001 | 0.060345 | 0.016763 | 0.533221 |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 131.999284 | 2.276028 |  0.0 | 0.089787 |  0.0 | 0.005595 |   0.0 | 0.016763 |
    | 1997-08-05 08:00:00 | 0.833333 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 | 0.783333 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                          0.0 |            17.557751 | 15.273176 | -10.960941 |                       0.0 |                      0.0 |              0.0 | 0.616554 |  0.0 |  0.358863 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.616554 | 0.074584 | 0.004306 |  0.0 | 0.000525 |  0.0 | 0.074584 | 0.074584 |   0.0 | 0.004306 |   0.0 | 0.000525 | 0.059759 |   0.0 | 0.001195 |   0.0 | 0.000001 | 0.060955 | 0.016932 | 0.341137 |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 132.536423 | 2.290853 |  0.0 | 0.092898 |  0.0 | 0.006119 |   0.0 | 0.016932 |
    | 1997-08-05 09:00:00 | 0.833333 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 | 0.783333 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                          0.0 |            17.870634 | 15.369908 | -24.180553 |                       0.0 |                      0.0 |              0.0 | 0.424471 |  0.0 |  0.541806 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.424471 | 0.051569 | 0.004324 |  0.0 | 0.000547 |  0.0 | 0.051569 | 0.051569 |   0.0 | 0.004324 |   0.0 | 0.000547 | 0.059844 |   0.0 | 0.001236 |   0.0 | 0.000001 |  0.06108 | 0.016967 | 0.158194 |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 | 132.904454 | 2.282578 |  0.0 | 0.095986 |  0.0 | 0.006665 |   0.0 | 0.016967 |
    | 1997-08-05 10:00:00 | 0.833333 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 | 0.783333 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                          0.0 |            17.839281 | 15.625274 | -29.848354 |                       0.0 |                      0.0 |              0.0 | 0.241527 |  0.0 |       0.7 | 0.010395 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.241527 | 0.029423 | 0.004336 |  0.0 | 0.000563 |  0.0 | 0.029423 | 0.029423 |   0.0 | 0.004336 |   0.0 | 0.000563 | 0.059343 |   0.0 | 0.001276 |   0.0 | 0.000001 |  0.06062 | 0.016839 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 | 133.101265 | 2.252658 |  0.0 | 0.099045 |  0.0 | 0.007227 |   0.0 | 0.016839 |
    | 1997-08-05 11:00:00 | 0.833333 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 | 0.783333 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                          0.0 |            17.180627 | 15.948655 | -32.708072 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.029871 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 |  0.05832 |   0.0 | 0.001316 |   0.0 | 0.000001 | 0.059637 | 0.016566 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 |  133.13965 | 2.204502 |  0.0 | 0.102071 |  0.0 | 0.007797 |   0.0 | 0.016566 |
    | 1997-08-05 12:00:00 | 0.833333 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 | 0.783333 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                          0.0 |            16.980219 | 16.306352 | -37.161891 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.046934 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010168 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010168 | 0.010168 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.057076 |   0.0 | 0.001355 |   0.0 | 0.000001 | 0.058433 | 0.016231 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 133.160966 | 2.157594 |  0.0 | 0.105059 |  0.0 | 0.008368 |   0.0 | 0.016231 |
    | 1997-08-05 13:00:00 | 0.833333 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 | 0.783333 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                          0.0 |            16.872626 | 16.717494 | -38.295035 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.091262 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 |  0.01017 | 0.004344 |  0.0 | 0.000573 |  0.0 |  0.01017 |  0.01017 |   0.0 | 0.004344 |   0.0 | 0.000573 | 0.055864 |   0.0 | 0.001394 |   0.0 | 0.000001 |  0.05726 | 0.015905 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 |  133.13795 | 2.111899 |  0.0 | 0.108009 |  0.0 |  0.00894 |   0.0 | 0.015905 |
    | 1997-08-05 14:00:00 | 0.833333 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 | 0.783333 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                          0.0 |            15.970201 | 17.142235 | -37.546592 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.105785 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010167 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010167 | 0.010167 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.054684 |   0.0 | 0.001432 |   0.0 | 0.000001 | 0.056118 | 0.015588 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 133.100415 | 2.067382 |  0.0 |  0.11092 |  0.0 | 0.009511 |   0.0 | 0.015588 |
    | 1997-08-05 15:00:00 | 0.833333 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 | 0.783333 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                          0.0 |            15.384928 | 17.557994 | -37.452038 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.063817 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.053534 |   0.0 |  0.00147 |   0.0 | 0.000001 | 0.055006 | 0.015279 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 133.104855 | 2.024012 |  0.0 | 0.113791 |  0.0 |  0.01008 |   0.0 | 0.015279 |
    | 1997-08-05 16:00:00 | 0.833333 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 | 0.783333 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                          0.0 |            14.793787 | 17.972618 | -36.164291 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |       0.7 | 0.001774 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.052414 |   0.0 | 0.001508 |   0.0 | 0.000002 | 0.053923 | 0.014979 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 133.171337 | 1.981763 |  0.0 | 0.116626 |  0.0 |  0.01065 |   0.0 | 0.014979 |
    | 1997-08-05 17:00:00 | 0.833333 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 | 0.783333 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                          0.0 |            14.375377 |  18.37179 | -33.769262 |                       0.0 |                      0.0 |              0.0 | 0.083333 |  0.0 |  0.559369 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010171 | 0.004344 |  0.0 | 0.000574 |  0.0 | 0.010171 | 0.010171 |   0.0 | 0.004344 |   0.0 | 0.000574 | 0.051323 |   0.0 | 0.001544 |   0.0 | 0.000002 | 0.052869 | 0.014686 | 0.140631 |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 133.239582 | 1.940611 |  0.0 | 0.119426 |  0.0 | 0.011222 |   0.0 | 0.014686 |
    | 1997-08-05 18:00:00 | 0.833333 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 | 0.783333 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                          0.0 |            14.500182 | 18.742221 | -29.746675 |                       0.0 |                      0.0 |              0.0 | 0.223964 |  0.0 |  0.406457 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.223964 | 0.027368 | 0.004346 |  0.0 | 0.000577 |  0.0 | 0.027368 | 0.027368 |   0.0 | 0.004346 |   0.0 | 0.000577 | 0.050483 |   0.0 | 0.001581 |   0.0 | 0.000002 | 0.052065 | 0.014463 | 0.293543 |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 133.431255 | 1.917496 |  0.0 | 0.122191 |  0.0 | 0.011797 |   0.0 | 0.014463 |
    | 1997-08-05 19:00:00 | 0.833333 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 | 0.783333 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                          0.0 |            15.846331 | 19.064381 | -22.413715 |                       0.0 |                      0.0 |              0.0 | 0.376876 |  0.0 |  0.257526 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.376876 | 0.046166 | 0.004353 |  0.0 | 0.000585 |  0.0 | 0.046166 | 0.046166 |   0.0 | 0.004353 |   0.0 | 0.000585 | 0.050129 |   0.0 | 0.001617 |   0.0 | 0.000002 | 0.051748 | 0.014374 | 0.442474 |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 133.757028 | 1.913532 |  0.0 | 0.124927 |  0.0 |  0.01238 |   0.0 | 0.014374 |
    | 1997-08-05 20:00:00 | 0.833333 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 | 0.783333 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                          0.0 |            16.720484 | 19.298545 | -12.008727 |                       0.0 |                      0.0 |              0.0 | 0.525807 |  0.0 |  0.102206 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.525807 | 0.064647 | 0.004363 |  0.0 | 0.000598 |  0.0 | 0.064647 | 0.064647 |   0.0 | 0.004363 |   0.0 | 0.000598 | 0.050267 |   0.0 | 0.001652 |   0.0 | 0.000002 | 0.051921 | 0.014423 | 0.597794 |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 134.213226 | 1.927912 |  0.0 | 0.127638 |  0.0 | 0.012976 |   0.0 | 0.014423 |
    | 1997-08-05 21:00:00 | 0.833333 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                          0.0 |            16.369483 |  19.40785 |   0.647101 |                       0.0 |                      0.0 |              0.0 | 0.681127 |  0.0 |  0.046254 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.681127 | 0.084156 | 0.004378 |  0.0 | 0.000617 |  0.0 | 0.084156 | 0.084156 |   0.0 | 0.004378 |   0.0 | 0.000617 | 0.050891 |   0.0 | 0.001688 |   0.0 | 0.000002 | 0.052581 | 0.014606 | 0.653746 |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 134.805202 | 1.961177 |  0.0 | 0.130328 |  0.0 | 0.013591 |   0.0 | 0.014606 |
    | 1997-08-05 22:00:00 | 0.833333 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                          0.0 |            16.157086 | 19.365285 |    6.39171 |                       0.0 |                      0.0 |              0.0 |  0.73708 |  0.0 |  0.042508 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.73708 | 0.091597 | 0.004398 |  0.0 | 0.000642 |  0.0 | 0.091597 | 0.091597 |   0.0 | 0.004398 |   0.0 | 0.000642 | 0.051847 |   0.0 | 0.001723 |   0.0 | 0.000002 | 0.053572 | 0.014881 | 0.657492 |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 135.445645 | 2.000927 |  0.0 | 0.133003 |  0.0 | 0.014231 |   0.0 | 0.014881 |
    | 1997-08-05 23:00:00 | 0.833333 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.783333 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                          0.0 |            16.281953 | 19.253784 |  11.122707 |                       0.0 |                      0.0 |              0.0 | 0.740825 |  0.0 |  0.035577 |      0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.740825 | 0.092616 | 0.004418 |  0.0 | 0.000669 |  0.0 | 0.092616 | 0.092616 |   0.0 | 0.004418 |   0.0 | 0.000669 | 0.052887 |   0.0 | 0.001758 |   0.0 | 0.000002 | 0.054647 |  0.01518 | 0.664423 |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 | 136.088768 | 2.040656 |  0.0 | 0.135664 |  0.0 | 0.014898 |   0.0 |  0.01518 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_heavy_rain))
0.0

.. _lland_knauf_water_hourly:

water
-----

For our hourly integration test on water type |WASSER|, we deviate from the input data
of the daily simulation (:ref:`lland_knauf_wasser_daily`) in setting precipitation to
the constant value of 0.05 mm/h.  The actual interception evaporation (|EvI|) does not
show a diurnal pattern due to submodel |evap_aet_morsim| relying on daily aggregated
input data:

.. integration-test::

    >>> lnk(WASSER)
    >>> inputs.nied.series = 0.05
    >>> test("lland_knauf_water_hourly",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...      use_conditions=conditions_wasser)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |  tz |  wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow | nbes | sbes |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | sdg1 | sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.05 | 16.9 |             95.1 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.9 |    0.641273 |           0.641273 |                19.284227 |                          0.0 |              18.3393 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.040545 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006455 | 0.001793 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001793 |
    | 1997-08-03 01:00:00 | 0.05 | 16.6 |             94.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.6 |    0.641273 |           0.641273 |                18.920184 |                          0.0 |            17.955254 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.04041 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00659 | 0.001831 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001831 |
    | 1997-08-03 02:00:00 | 0.05 | 16.4 |             95.9 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.7 |             0.0 | 0.047 | 16.4 |    0.641273 |           0.641273 |                 18.68084 |                          0.0 |            17.914926 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.040055 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006945 | 0.001929 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001929 |
    | 1997-08-03 03:00:00 | 0.05 | 16.3 |             96.7 |       0.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                          15.5 |             0.0 | 0.047 | 16.3 |    0.641273 |           0.641273 |                18.562165 |                          0.0 |            17.949613 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.039459 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007541 | 0.002095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002095 |
    | 1997-08-03 04:00:00 | 0.05 | 16.0 |             97.2 |       0.8 |                 0.429734 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |        1.943686 | 0.047 | 16.0 |    0.641273 |           0.641273 |                18.210086 |                          0.0 |            17.700204 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.038271 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008729 | 0.002425 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002425 |
    | 1997-08-03 05:00:00 | 0.05 | 15.9 |             97.5 |       0.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       21.932441 | 0.047 | 15.9 |    0.480955 |           0.480955 |                18.094032 |                          0.0 |            17.641681 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.036944 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010056 | 0.002793 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002793 |
    | 1997-08-03 06:00:00 | 0.05 | 16.0 |             97.7 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |       57.256187 | 0.047 | 16.0 |    0.721432 |           0.721432 |                18.210086 |                          0.0 |            17.791254 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.035268 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011732 | 0.003259 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003259 |
    | 1997-08-03 07:00:00 | 0.05 | 16.6 |             97.4 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      109.332844 | 0.047 | 16.6 |    0.721432 |           0.721432 |                18.920184 |                          0.0 |            18.428259 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.033685 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013315 | 0.003699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003699 |
    | 1997-08-03 08:00:00 | 0.05 | 17.4 |             96.8 |       0.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   1.3 |                     14.929734 |      170.949152 | 0.047 | 17.4 |    0.721432 |           0.721432 |                19.904589 |                          0.0 |            19.267642 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.032054 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.014946 | 0.004152 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004152 |
    | 1997-08-03 09:00:00 | 0.05 | 19.0 |             86.1 |       1.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   1.5 |                     14.929734 |      311.762624 | 0.047 | 19.0 |    1.042069 |           1.042069 |                22.008543 |                          0.0 |            18.949356 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.031746 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015254 | 0.004237 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004237 |
    | 1997-08-03 10:00:00 | 0.05 | 20.3 |             76.8 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   1.7 |                     14.929734 |      501.583299 | 0.047 | 20.3 |    1.202387 |           1.202387 |                23.858503 |                          0.0 |             18.32333 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.031752 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015248 | 0.004236 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004236 |
    | 1997-08-03 11:00:00 | 0.05 | 21.4 |             71.8 |       1.2 |                      1.0 |              0.7 | 0.0 | 0.0 |                   2.2 |                     14.929734 |      615.018727 | 0.047 | 21.4 |    0.961909 |           0.961909 |                25.528421 |                          0.0 |            18.329406 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.031812 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015188 | 0.004219 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.004219 |
    | 1997-08-03 12:00:00 | 0.05 | 21.3 |             67.5 |       1.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   2.8 |                     14.929734 |      626.544326 | 0.047 | 21.3 |    1.042069 |           1.042069 |                 25.37251 |                          0.0 |            17.126444 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.033619 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013381 | 0.003717 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003717 |
    | 1997-08-03 13:00:00 | 0.05 | 21.8 |             66.1 |       1.5 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.2 |                     14.929734 |      496.133417 | 0.047 | 21.8 |    1.202387 |           1.202387 |                26.160453 |                          0.0 |             17.29206 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.035172 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011828 | 0.003286 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003286 |
    | 1997-08-03 14:00:00 | 0.05 | 22.9 |             63.4 |       1.9 |                      1.0 |              0.4 | 0.0 | 0.0 |                   3.4 |                     14.929734 |      419.520994 | 0.047 | 22.9 |    1.523023 |           1.523023 |                27.969419 |                          0.0 |            17.732611 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.036854 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.010146 | 0.002818 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002818 |
    | 1997-08-03 15:00:00 | 0.05 | 22.7 |             62.4 |       1.9 |                      1.0 |              0.5 | 0.0 | 0.0 |                   3.8 |                     14.929734 |      387.887354 | 0.047 | 22.7 |    1.523023 |           1.523023 |                27.632633 |                          0.0 |            17.242763 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.038963 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008037 | 0.002233 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.002233 |
    | 1997-08-03 16:00:00 | 0.05 | 22.5 |             61.1 |       2.3 |                      1.0 |              0.5 | 0.0 | 0.0 |                   4.1 |                     14.929734 |      278.496873 | 0.047 | 22.5 |     1.84366 |            1.84366 |                27.299387 |                          0.0 |            16.679926 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.041278 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.005722 |  0.00159 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00159 |
    | 1997-08-03 17:00:00 | 0.05 | 21.9 |             62.1 |       2.4 |                      1.0 |              0.3 | 0.0 | 0.0 |                   4.4 |                     14.929734 |      137.138608 | 0.047 | 21.9 |    1.923819 |           1.923819 |                26.320577 |                          0.0 |            16.345078 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |  0.04299 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00401 | 0.001114 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.001114 |
    | 1997-08-03 18:00:00 | 0.05 | 21.4 |             67.0 |       2.5 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.5 |                     14.929734 |       51.080715 | 0.047 | 21.4 |    2.003978 |           2.003978 |                25.528421 |                          0.0 |            17.104042 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |   0.0444 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0026 | 0.000722 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000722 |
    | 1997-08-03 19:00:00 | 0.05 | 20.7 |             74.5 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.429734 |       13.632816 | 0.047 | 20.7 |    2.003978 |           2.003978 |                24.454368 |                          0.0 |            18.218504 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 | 0.046089 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000911 | 0.000253 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000253 |
    | 1997-08-03 20:00:00 | 0.05 | 19.4 |             81.2 |       2.2 |                   0.1364 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |        0.185943 | 0.047 | 19.4 |    1.763501 |           1.763501 |                22.563931 |                          0.0 |            18.321912 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 21:00:00 | 0.05 | 17.8 |             86.9 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 17.8 |    1.362705 |           1.362705 |                20.413369 |                          0.0 |            17.739217 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 22:00:00 | 0.05 | 17.0 |             90.1 |       1.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 17.0 |    1.362705 |           1.362705 |                19.406929 |                          0.0 |            17.485643 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-03 23:00:00 | 0.05 | 16.4 |             90.9 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 16.4 |     1.84366 |            1.84366 |                 18.68084 |                          0.0 |            16.980884 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 00:00:00 | 0.05 | 15.9 |             88.0 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.9 |     1.84366 |            1.84366 |                18.094032 |                          0.0 |            15.922748 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 01:00:00 | 0.05 | 15.4 |             89.2 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.4 |    1.763501 |           1.763501 |                17.523416 |                          0.0 |            15.630887 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 02:00:00 | 0.05 | 15.2 |             88.8 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 15.2 |     1.84366 |            1.84366 |                17.299618 |                          0.0 |            15.362061 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 03:00:00 | 0.05 | 14.9 |             89.7 |       2.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.566134 |             0.0 | 0.047 | 14.9 |    1.603182 |           1.603182 |                16.968612 |                          0.0 |            15.220845 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 04:00:00 | 0.05 | 14.3 |             93.0 |       2.3 |                 0.398044 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |        1.590516 | 0.047 | 14.3 |     1.84366 |            1.84366 |                16.323194 |                          0.0 |             15.18057 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 05:00:00 | 0.05 | 14.1 |             93.6 |       2.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   4.5 |                     15.534444 |       19.950903 | 0.047 | 14.1 |    2.003978 |           2.003978 |                16.112886 |                          0.0 |            15.081661 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 06:00:00 | 0.05 | 14.3 |             94.6 |       2.4 |                      1.0 |              0.1 | 0.0 | 0.0 |                   4.6 |                     15.534444 |       62.985774 | 0.047 | 14.3 |    1.923819 |           1.923819 |                16.323194 |                          0.0 |            15.441741 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 07:00:00 | 0.05 | 15.4 |             93.3 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   5.5 |                     15.534444 |      285.776389 | 0.047 | 15.4 |    2.003978 |           2.003978 |                17.523416 |                          0.0 |            16.349347 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 08:00:00 | 0.05 | 16.8 |             86.9 |       2.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                   6.5 |                     15.534444 |      475.225247 | 0.047 | 16.8 |    2.003978 |           2.003978 |                19.162205 |                          0.0 |            16.651956 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 09:00:00 | 0.05 | 18.2 |             78.9 |       2.9 |                      1.0 |              1.0 | 0.0 | 0.0 |                   7.3 |                     15.534444 |      614.006719 | 0.047 | 18.2 |    2.324615 |           2.324615 |                20.933479 |                          0.0 |            16.516515 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 10:00:00 | 0.05 | 19.5 |             75.8 |       3.1 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.7 |                     15.534444 |      643.806508 | 0.047 | 19.5 |    2.484933 |           2.484933 |                22.704676 |                          0.0 |            17.210144 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 11:00:00 | 0.05 | 20.3 |             73.4 |       3.3 |                      1.0 |              0.8 | 0.0 | 0.0 |                   7.8 |                     15.534444 |       600.26298 | 0.047 | 20.3 |    2.645251 |           2.645251 |                23.858503 |                          0.0 |            17.512141 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 12:00:00 | 0.05 | 21.0 |             68.3 |       3.3 |                      1.0 |              0.9 | 0.0 | 0.0 |                   7.9 |                     15.534444 |       620.51414 | 0.047 | 21.0 |    2.645251 |           2.645251 |                24.909749 |                          0.0 |            17.013358 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 13:00:00 | 0.05 | 21.8 |             61.3 |       3.2 |                      1.0 |              0.8 | 0.0 | 0.0 |                   8.2 |                     15.534444 |      599.441386 | 0.047 | 21.8 |    2.565092 |           2.565092 |                26.160453 |                          0.0 |            16.036358 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 14:00:00 | 0.05 | 21.9 |             60.6 |       2.5 |                      1.0 |              0.9 | 0.0 | 0.0 |                   8.7 |                     15.534444 |      613.551223 | 0.047 | 21.9 |    2.003978 |           2.003978 |                26.320577 |                          0.0 |             15.95027 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 15:00:00 | 0.05 | 21.8 |             58.8 |       2.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.1 |                     15.534444 |      517.107047 | 0.047 | 21.8 |    2.324615 |           2.324615 |                26.160453 |                          0.0 |            15.382347 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 16:00:00 | 0.05 | 22.2 |             57.2 |       3.6 |                      1.0 |              0.9 | 0.0 | 0.0 |                   9.5 |                     15.534444 |      377.836616 | 0.047 | 22.2 |    2.885728 |           2.885728 |                26.806088 |                          0.0 |            15.333082 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 17:00:00 | 0.05 | 21.8 |             57.2 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  10.2 |                     15.534444 |      252.204639 | 0.047 | 21.8 |    3.366683 |           3.366683 |                26.160453 |                          0.0 |            14.963779 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 18:00:00 | 0.05 | 20.9 |             59.9 |       4.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  11.1 |                     15.534444 |      128.576715 | 0.047 | 20.9 |    3.366683 |           3.366683 |                 24.75714 |                          0.0 |            14.829527 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 19:00:00 | 0.05 | 19.7 |             66.1 |       3.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.1 |                     15.534444 |       42.072585 | 0.047 | 19.7 |    2.885728 |           2.885728 |                22.988466 |                          0.0 |            15.195376 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 20:00:00 | 0.05 | 17.9 |             71.7 |       3.2 |                 0.104711 |              0.3 | 0.0 | 0.0 |                  12.4 |                     15.502754 |        0.812898 | 0.047 | 17.9 |    2.565092 |           2.565092 |                20.542326 |                          0.0 |            14.728848 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 21:00:00 | 0.05 | 15.8 |             80.6 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 15.8 |    1.763501 |           1.763501 |                17.978626 |                          0.0 |            14.490772 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 22:00:00 | 0.05 | 14.8 |             85.9 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 14.8 |    1.763501 |           1.763501 |                16.859515 |                          0.0 |            14.482324 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-04 23:00:00 | 0.05 | 14.1 |             89.3 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 14.1 |    1.683342 |           1.683342 |                16.112886 |                          0.0 |            14.388807 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 00:00:00 | 0.05 | 13.4 |             91.4 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 13.4 |    1.442864 |           1.442864 |                15.395377 |                          0.0 |            14.071374 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 01:00:00 | 0.05 | 12.8 |             93.1 |       1.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.8 |    1.442864 |           1.442864 |                14.802811 |                          0.0 |            13.781417 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 02:00:00 | 0.05 | 12.3 |             94.6 |       1.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.3 |    1.282546 |           1.282546 |                14.324376 |                          0.0 |            13.550859 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 03:00:00 | 0.05 | 12.0 |             95.6 |       1.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.502754 |             0.0 | 0.047 | 12.0 |    0.961909 |           0.961909 |                 14.04387 |                          0.0 |            13.425939 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 04:00:00 | 0.05 | 11.7 |             96.2 |       1.9 |                 0.366022 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |        1.482348 | 0.047 | 11.7 |    1.523023 |           1.523023 |                13.768192 |                          0.0 |            13.245001 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 05:00:00 | 0.05 | 11.6 |             95.7 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  12.4 |                     15.470733 |       21.135568 | 0.047 | 11.6 |    1.042069 |           1.042069 |                 13.67736 |                          0.0 |            13.089234 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 06:00:00 | 0.05 | 12.6 |             95.1 |       1.6 |                      1.0 |              0.6 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      125.883451 | 0.047 | 12.6 |    1.282546 |           1.282546 |                14.609782 |                          0.0 |            13.893903 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 07:00:00 | 0.05 | 14.8 |             96.3 |       1.9 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      305.954943 | 0.047 | 14.8 |    1.523023 |           1.523023 |                16.859515 |                          0.0 |            16.235713 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 08:00:00 | 0.05 | 17.1 |             89.9 |       3.2 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |      511.104575 | 0.047 | 17.1 |    2.565092 |           2.565092 |                19.530313 |                          0.0 |            17.557751 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 09:00:00 | 0.05 | 19.4 |             79.2 |       4.0 |                      1.0 |              0.9 | 0.0 | 0.0 |                  12.8 |                     15.470733 |       629.51924 | 0.047 | 19.4 |    3.206365 |           3.206365 |                22.563931 |                          0.0 |            17.870634 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 10:00:00 | 0.05 | 20.6 |             73.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  12.9 |                     15.470733 |       765.81659 | 0.047 | 20.6 |     3.68732 |            3.68732 |                24.304197 |                          0.0 |            17.839281 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 11:00:00 | 0.05 | 21.4 |             67.3 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.1 |                     15.470733 |      767.145843 | 0.047 | 21.4 |    3.286524 |           3.286524 |                25.528421 |                          0.0 |            17.180627 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 12:00:00 | 0.05 | 22.5 |             62.2 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.2 |                     15.470733 |      736.263854 | 0.047 | 22.5 |    3.286524 |           3.286524 |                27.299387 |                          0.0 |            16.980219 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 13:00:00 | 0.05 | 23.1 |             59.6 |       4.4 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.4 |                     15.470733 |      768.006183 | 0.047 | 23.1 |    3.527001 |           3.527001 |                28.309776 |                          0.0 |            16.872626 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 14:00:00 | 0.05 | 23.4 |             55.4 |       4.6 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.5 |                     15.470733 |       735.92937 | 0.047 | 23.4 |     3.68732 |            3.68732 |                28.827078 |                          0.0 |            15.970201 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 15:00:00 | 0.05 | 23.8 |             52.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.6 |                     15.470733 |      620.374953 | 0.047 | 23.8 |    3.607161 |           3.607161 |                29.529613 |                          0.0 |            15.384928 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 16:00:00 | 0.05 | 24.0 |             49.5 |       4.7 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      450.095263 | 0.047 | 24.0 |    3.767479 |           3.767479 |                29.886439 |                          0.0 |            14.793787 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 17:00:00 | 0.05 | 24.0 |             48.1 |       4.5 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      273.614141 | 0.047 | 24.0 |    3.607161 |           3.607161 |                29.886439 |                          0.0 |            14.375377 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 18:00:00 | 0.05 | 23.7 |             49.4 |       4.1 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |      133.368104 | 0.047 | 23.7 |    3.286524 |           3.286524 |                29.352596 |                          0.0 |            14.500182 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 19:00:00 | 0.05 | 22.8 |             57.0 |       3.8 |                      1.0 |              1.0 | 0.0 | 0.0 |                  13.7 |                     15.470733 |       40.326502 | 0.047 | 22.8 |    3.046047 |           3.046047 |                27.800581 |                          0.0 |            15.846331 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 20:00:00 | 0.05 | 21.3 |             65.9 |       2.5 |                 0.072689 |              0.5 | 0.0 | 0.0 |                  13.9 |                     15.438712 |        0.867122 | 0.047 | 21.3 |    2.003978 |           2.003978 |                 25.37251 |                          0.0 |            16.720484 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 21:00:00 | 0.05 | 19.3 |             73.0 |       1.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 19.3 |    1.523023 |           1.523023 |                22.423949 |                          0.0 |            16.369483 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 22:00:00 | 0.05 | 18.3 |             76.7 |       2.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 18.3 |    1.763501 |           1.763501 |                21.065302 |                          0.0 |            16.157086 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 1997-08-05 23:00:00 | 0.05 | 17.4 |             81.8 |       2.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                  13.9 |                     15.438712 |             0.0 | 0.047 | 17.4 |    2.164296 |           2.164296 |                19.904589 |                          0.0 |            16.281953 | nan | 0.0 |                       0.0 |                      0.0 |              0.0 |  0.0 |  0.0 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          nan |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

    >>> inputs.nied.series = 0.0

.. _lland_knauf_acker_winter_hourly:

acre (winter)
-------------

Our hourly integration tests covering snow processes focus on the subperiod from
December 10th to December 12, for which the daily simulation suggested that some
melting occurs:

>>> pub.timegrids = "2010-12-10", "2010-12-13", "1h"

The following input values agree with the ones of the respective sub-period of the
daily integration test :ref:`lland_knauf_acker_winter_daily` (global radiation and
possible sunshine duration stem from
:ref:`meteo_glob_morsim_hourly_simulation_winter`):

>>> inputs.nied.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.1, 0.0, 0.3, 0.2, 0.4, 0.7, 0.5, 1.0, 2.0, 2.1, 1.8, 1.3, 0.5, 0.2, 0.5, 0.2,
...     0.1, 0.3, 0.1, 0.1, 0.0, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.1,
...     0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.teml.series = (
...     -3.7, -4.8, -5.6, -6.2, -7.3, -8.3, -8.6, -8.5, -9.5, -9.2, -6.9, -4.0, -2.8,
...     -1.4, -0.4, 0.5, 1.0, 1.1, 1.0, 1.4, 1.6, 2.0, 2.4, 2.5, 2.7, 3.2, 3.9, 4.4,
...     4.8, 4.8, 5.1, 5.1, 5.4, 5.3, 5.5, 6.1, 6.4, 6.4, 6.5, 6.3, 5.9, 5.5, 5.1, 4.8,
...     4.9, 4.9, 4.7, 3.8, 3.9, 3.7, 3.5, 3.3, 3.3, 3.0, 1.9, 1.2, 0.8, 0.8, 0.5, 0.3,
...     -0.1, -0.4, -0.6, -0.8, -1.3, -1.6, -2.0, -2.1, -2.2, -2.6, -3.1, -3.9)
>>> inputs.windspeed.series = (
...     3.4, 2.9, 2.5, 3.0, 2.5, 2.3, 2.1, 1.4, 1.1, 1.0, 1.4, 1.3, 1.3, 1.4, 2.3, 2.8,
...     3.1, 3.5, 4.4, 3.8, 5.7, 5.6, 5.8, 6.2, 5.6, 5.2, 5.7, 6.6, 7.0, 7.3, 7.2, 7.6,
...     7.3, 7.1, 7.1, 8.9, 9.5, 7.9, 9.6, 9.1, 8.7, 8.9, 7.9, 7.1, 6.4, 7.2, 7.0, 6.8,
...     6.4, 6.3, 5.7, 5.2, 4.6, 4.9, 6.3, 5.8, 6.1, 5.9, 6.4, 6.8, 7.0, 6.3, 7.3, 7.5,
...     7.0, 6.0, 4.8, 4.2, 4.9, 4.0, 3.4, 3.4)
>>> inputs.relativehumidity.series = (
...     87.5, 90.7, 89.7, 91.3, 94.2, 93.2, 96.5, 95.9, 96.8, 94.3, 96.7, 97.4, 93.9,
...     93.4, 93.3, 94.1, 94.6, 94.4, 96.9, 98.3, 98.7, 99.6, 99.4, 99.2, 99.2, 99.6,
...     99.8, 99.8, 99.0, 97.6, 97.4, 96.3, 95.2, 96.3, 95.5, 91.6, 90.0, 85.7, 82.3,
...     82.5, 82.2, 76.5, 81.7, 83.4, 82.8, 85.8, 87.6, 85.6, 85.3, 86.2, 89.3, 91.4,
...     90.9, 89.2, 85.6, 86.3, 91.2, 87.5, 84.5, 76.7, 74.8, 69.7, 66.9, 64.4, 65.4,
...     67.6, 70.2, 70.4, 69.5, 71.6, 74.5, 80.2)

>>> model.radiationmodel.sequences.inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.2,
...     0.2, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0)

>>> aetinputs.atmosphericpressure.series = (
...     1017.0, 1018.0, 1018.0, 1019.0, 1020.0, 1020.0, 1021.0, 1021.0, 1022.0, 1022.0,
...     1022.0, 1022.0, 1022.0, 1021.0, 1020.0, 1020.0, 1019.0, 1018.0, 1016.0, 1015.0,
...     1014.0, 1013.0, 1011.0, 1010.0, 1008.0, 1007.0, 1006.0, 1005.0, 1005.0, 1004.0,
...     1004.0, 1004.0, 1003.0, 1003.0, 1003.0, 1002.0, 1002.0, 1002.0, 1002.0, 1003.0,
...     1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1003.0, 1002.0, 1002.0, 1002.0,
...     1002.0, 1003.0, 1004.0, 1005.0, 1006.0, 1008.0, 1009.0, 1010.0, 1011.0, 1012.0,
...     1013.0, 1014.0, 1015.0, 1015.0, 1017.0, 1017.0, 1018.0, 1019.0, 1019.0, 1020.0,
...     1020.0, 1020.0)
>>> aetinputs.windspeed.series = inputs.windspeed.series
>>> aetinputs.relativehumidity.series = inputs.relativehumidity.series

In this and the following winter examples, we use the initial conditions of December
10th, calculated by the corresponding daily integration tests.  However, this is
possible for the state sequences but not for the (now higher resolved) log sequences.
Hence, we need to define them manually:

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.04, 0.25, 0.59, 0.91,
...       0.97, 1.0, 0.65, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedpossiblesunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.4, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...       1.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (model.radiationmodel.sequences.logs.loggedunadjustedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 27.777778, 55.555556,
...       111.111111, 166.666667, 138.888889, 55.555556, 0.0,0.0, 0.0, 0.0, 0.0, 0.0,
...       0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 27.8, 55.6, 111.1, 166.7, 138.9,
...       55.6, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (model.aetmodel.sequences.logs.loggedairtemperature,
...      [[-2.8], [-2.6], [-2.2], [-1.8], [-1.8], [-2.4], [-1.5], [-1.4], [-0.8],
...       [-0.8], [-0.9], [0.9], [-0.8], [-1.0], [-1.2], [-1.8], [-2.8], [-3.8],
...       [-4.3], [-4.0], [-3.9], [-1.6], [-1.6], [2.6]]),
...     (model.aetmodel.sequences.logs.loggedrelativehumidity,
...      [99.4, 99.0, 99.6, 99.6, 99.3, 99.3, 99.8, 99.7, 100.0, 99.6, 99.3, 98.9,
...       96.6, 93.0, 91.4, 92.0, 94.5, 96.5, 98.2, 96.9, 98.0, 95.0, 91.8, 88.5]),
...     (model.aetmodel.sequences.logs.loggedwindspeed2m,
...      [2.2, 2.0, 2.5, 2.6, 2.3, 1.7, 2.8, 1.9, 2.6, 2.9, 3.5, 3.4, 3.5, 3.0, 3.9,
...       3.6, 2.2, 2.4, 2.5, 1.5, 2.5, 3.0, 3.2, 3.1]))

Compared to the daily simulation :ref:`lland_knauf_acker_winter_daily`, the hourly
simulation estimates a slightly more pronounced melting rate for land-use type |ACKER|,
resulting in a complete depletion of the snow cover:

.. integration-test::

    >>> lnk(ACKER)
    >>> test("lland_knauf_acker_winter_hourly",
    ...      axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_acker_winter)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |     nbes |    sbes |      evi |      evb |       evs |    wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp |      wats |      waes |       esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |            2.72541 |                  4.64297 |                     4.116366 |             4.062599 | -5.333001 |  -0.271934 |                       0.0 |                73.930701 |       -73.930701 |      0.0 |     0.0 |      0.0 |      0.0 |  0.000812 |      0.0 |    -5.287679 |     0.754324 |      0.0 |      0.0 | 0.697428 | 0.697428 |   0.601912 | -10.098682 |   64.433931 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.318368 |  20.34801 |         0.0 | 0.860339 | -435.544856 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           2.324615 |                 4.272186 |                     3.961817 |             3.874873 | -5.226538 |   3.367603 |                       0.0 |                76.614193 |       -76.614193 |      0.0 |     0.0 |      0.0 |      0.0 |  0.001181 |      0.0 |    -5.787805 |     0.753661 |      0.0 |      0.0 | 0.738023 | 0.029642 |    0.87519 |  -5.649638 |   71.839745 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.346829 | 20.346829 |  -65.721984 | 0.867647 | -430.312459 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           2.003978 |                 4.019239 |                     3.242304 |             3.605258 |  -5.16375 |  19.248256 |                       0.0 |                70.174552 |       -70.174552 |      0.0 |     0.0 |      0.0 |      0.0 | -0.004489 |      0.0 |    -8.371792 |     0.753423 |      0.0 |      0.0 | 1.065896 |      0.0 |   -3.32582 | -14.430962 |   52.417771 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.351317 | 20.351317 |  -98.891498 | 0.870296 | -440.960715 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           2.404774 |                 3.838328 |                     2.930365 |             3.504393 | -5.291529 |  26.174386 |                       0.0 |                67.449822 |       -67.449822 |      0.0 |     0.0 |      0.0 |      0.0 | -0.007973 |      0.0 |    -9.653926 |      0.75323 |      0.0 |      0.0 | 1.229409 |      0.0 |  -5.907804 | -20.197311 |   41.344707 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.359291 | 20.359291 | -114.061819 | 0.872433 | -458.535101 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           2.003978 |                  3.52531 |                     2.689111 |             3.320842 | -5.502421 |   31.37453 |                       0.0 |                67.810291 |       -67.810291 |      0.0 |     0.0 |      0.0 |      0.0 | -0.007812 |      0.0 |   -10.731509 |     0.753055 |      0.0 |      0.0 | 1.367172 |      0.0 |  -5.788675 | -17.865691 |   44.155926 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.367103 | 20.367103 | -126.843215 | 0.874387 | -481.309631 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |            1.84366 |                  3.26061 |                     2.483423 |             3.038889 | -5.775716 |  35.667788 |                       0.0 |                69.178008 |       -69.178008 |      0.0 |     0.0 |      0.0 |      0.0 | -0.006531 |      0.0 |   -11.720347 |     0.752893 |      0.0 |      0.0 | 1.493721 |      0.0 |  -4.839077 | -16.930224 |   47.408708 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.373634 | 20.373634 | -138.584134 | 0.876201 | -508.377419 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 06:00:00 |  0.0 | -8.6 |             96.5 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.6 |    1.683342 |           1.683342 |                 3.184716 |                     2.368771 |             3.073251 | -6.100529 |  37.218337 |                       0.0 |                 67.36275 |        -67.36275 |      0.0 |     0.0 |      0.0 |      0.0 | -0.007854 |      0.0 |   -12.303585 |     0.752741 |      0.0 |      0.0 | 1.568556 |      0.0 |  -5.819209 | -17.382209 |   44.161332 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.381488 | 20.381488 |  -145.52713 | 0.877894 | -536.995756 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 07:00:00 |  0.0 | -8.5 |             95.9 |       1.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  3.76 |                           7.4 |             0.0 |   0.0 | -8.5 |    1.122228 |           1.122228 |                 3.209838 |                     2.300653 |             3.078235 | -6.443949 |  37.308926 |                       0.0 |                62.599151 |       -62.599151 |      0.0 |     0.0 |      0.0 |      0.0 |  -0.00701 |      0.0 |   -12.662103 |     0.752596 |      0.0 |      0.0 | 1.614885 |      0.0 |  -5.194397 | -15.797532 |   41.607223 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.388498 | 20.388498 | -149.825426 | 0.879522 | -565.704682 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 08:00:00 |  0.0 | -9.5 |             96.8 |       1.1 |                 0.355232 |              0.0 | 0.0 | 0.0 |                  2.76 |                      6.755232 |         1.10761 |   0.0 | -9.5 |     0.88175 |            0.88175 |                 2.966351 |                      2.20792 |             2.871427 | -6.788456 |    38.2632 |                  0.274184 |                 61.03516 |       -60.760976 |      0.0 |     0.0 |      0.0 |      0.0 | -0.005376 |      0.0 |   -13.165656 |     0.752454 |      0.0 |      0.0 | 1.679684 |      0.0 |  -3.983039 | -12.502822 |   44.275115 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.393873 | 20.393873 |  -155.83734 |  0.88111 | -595.367883 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 09:00:00 |  0.0 | -9.2 |             94.3 |       1.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                  1.79 |                      6.755232 |       18.663203 |   0.0 | -9.2 |    0.801591 |           0.801591 |                 3.037615 |                     2.243439 |             2.864471 | -7.144415 |  34.957135 |                   4.62254 |                53.276439 |       -48.653899 |      0.0 |     0.0 |      0.0 |      0.0 | -0.004842 |      0.0 |   -12.970604 |     0.752318 |      0.0 |      0.0 | 1.655235 |      0.0 |  -3.587876 |  -12.37718 |   32.688843 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.398716 | 20.398716 | -153.569049 | 0.882643 | -621.725017 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 10:00:00 |  0.0 | -6.9 |             96.7 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.88 |                      6.755232 |       56.267907 |   0.0 | -6.9 |    1.122228 |           1.122228 |                 3.636412 |                     2.591869 |             3.516411 |   -7.4607 |  22.377585 |                 13.944315 |                 37.73515 |       -23.790835 |      0.0 |     0.0 |      0.0 |      0.0 | -0.008335 |      0.0 |   -11.190298 |      0.75218 |      0.0 |      0.0 | 1.428382 |      0.0 |  -6.176117 | -16.284101 |    1.330618 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.407051 | 20.407051 | -132.522081 | 0.884197 | -635.502602 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 11:00:00 |  0.0 | -4.0 |             97.4 |       1.3 |                      1.0 |              0.1 | 0.0 | 0.0 |                  0.39 |                      6.755232 |      107.168237 |   0.0 | -4.0 |    1.042069 |           1.042069 |                 4.539126 |                     3.203405 |             4.421109 | -7.626031 |   5.397073 |                 26.575128 |                24.912069 |         1.663059 |      0.0 |     0.0 |      0.0 |      0.0 | -0.010607 |      0.0 |    -8.525543 |     0.752024 |      0.0 |      0.0 | 1.088685 |      0.0 |  -7.859627 |  -16.59657 |  -26.119256 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.417658 | 20.417658 | -101.005752 | 0.885954 | -632.299675 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 12:00:00 |  0.0 | -2.8 |             93.9 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.14 |                      6.755232 |       91.473333 |   0.0 | -2.8 |    1.042069 |           1.042069 |                 4.967227 |                     3.625978 |             4.664226 | -7.587596 |  -3.902967 |                 22.700284 |                21.894455 |         0.805828 |      0.0 |     0.0 |      0.0 |      0.0 | -0.009044 |      0.0 |    -6.937102 |     0.751837 |      0.0 |      0.0 | 0.886306 |      0.0 |  -6.701336 | -15.172033 |  -22.679197 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.426702 | 20.426702 |  -82.229522 | 0.888069 | -619.796708 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 13:00:00 |  0.0 | -1.4 |             93.4 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       80.475535 |   0.0 | -1.4 |    1.122228 |           1.122228 |                 5.511494 |                     4.004272 |             5.147736 |  -7.43756 | -10.733024 |                 19.987895 |                17.777493 |         2.210401 |      0.0 |     0.0 |      0.0 |      0.0 | -0.010309 |      0.0 |    -5.648723 |     0.751628 |      0.0 |      0.0 | 0.722019 |      0.0 |  -7.638562 | -16.126303 |  -25.975266 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.437011 | 20.437011 |   -66.98728 | 0.890442 | -600.463685 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 14:00:00 |  0.0 | -0.4 |             93.3 |       2.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       43.120591 |   0.0 | -0.4 |     1.84366 |            1.84366 |                 5.931876 |                     4.319014 |              5.53444 | -7.205564 | -15.294273 |                 10.719943 |                 15.34479 |        -4.624847 |      0.0 |     0.0 |      0.0 |      0.0 |  -0.01429 |      0.0 |    -4.656519 |     0.751396 |      0.0 |      0.0 | 0.595496 |      0.0 | -10.588489 | -21.069153 |  -27.032796 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.451302 | 20.451302 |  -55.248758 | 0.893072 | -576.569411 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 15:00:00 |  0.0 |  0.5 |             94.1 |       2.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.355232 |       10.320091 |   0.0 |  0.5 |    2.244455 |           2.244455 |                 6.334173 |                     4.593267 |             5.960456 | -6.918833 | -18.455959 |                  2.568223 |                12.225037 |        -9.656814 |      0.0 |     0.0 |      0.0 |      0.0 | -0.018157 |      0.0 |     -3.84284 |     0.751143 |      0.0 |      0.0 | 0.491782 |      0.0 |  -13.45367 | -24.281376 |  -28.078232 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.469459 | 20.469459 |  -45.626485 | 0.895952 | -549.513452 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 16:00:00 |  0.1 |  1.0 |             94.6 |       3.1 |                 0.061898 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |        0.028102 | 0.094 |  1.0 |    2.484933 |           2.484933 |                 6.567919 |                     4.830306 |             6.213251 | -6.594161 | -20.523962 |                  0.006996 |                11.454094 |       -11.447098 |    0.094 |  0.0235 |      0.0 |      0.0 |  -0.01963 | 0.095635 |    -3.173501 |     0.751051 |      0.0 |      0.0 | 0.409757 |   0.0705 | -14.545218 | -24.940394 |  -28.038514 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.583089 | 20.583089 |  -31.475465 | 0.897007 |  -520.38949 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 17:00:00 |  0.0 |  1.1 |             94.4 |       3.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |   0.0 |  1.1 |    2.805569 |           2.805569 |                 6.615571 |                     5.065454 |             6.245099 | -6.244674 | -22.242326 |                       0.0 |                 13.68212 |        -13.68212 |      0.0 |     0.0 |      0.0 |      0.0 | -0.018182 |      0.0 |     -2.53762 |     0.750726 |      0.0 |      0.0 | 0.326841 |      0.0 | -13.472116 | -23.604189 |  -23.394185 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.601271 | 20.601271 |  -30.323606 | 0.900733 | -489.547164 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 18:00:00 |  0.3 |  1.0 |             96.9 |       4.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.282 |  1.0 |    3.527001 |           3.527001 |                 6.567919 |                     5.196863 |             6.364313 | -5.874566 | -22.086098 |                       0.0 |                14.855317 |       -14.855317 |    0.282 |  0.0705 |      0.0 |      0.0 | -0.021195 | 0.286904 |     -2.19355 |     0.750944 |      0.0 |      0.0 | 0.289558 |   0.2115 | -15.704581 | -24.408946 |   -25.25821 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.904466 | 20.904466 |   -7.242091 | 0.898226 | -458.861066 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 19:00:00 |  0.2 |  1.4 |             98.3 |       3.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  1.4 |    3.046047 |           3.046047 |                 6.760362 |                     5.554322 |             6.645436 | -5.506333 | -25.267422 |                       0.0 |                15.478179 |       -15.478179 |    0.188 |  0.0282 |      0.0 |      0.0 | -0.017815 | 0.283107 |    -1.295096 |     0.750638 |      0.0 |      0.0 | 0.172233 |   0.1598 | -13.199933 | -18.525212 |  -16.246966 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.110281 | 21.110281 |   -1.153551 | 0.901733 | -424.993644 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 20:00:00 |  0.4 |  1.6 |             98.7 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.376 |  1.6 |     4.56907 |            4.56907 |                 6.858436 |                     5.831352 |             6.769276 | -5.099924 | -26.799806 |                       0.0 |                16.898085 |       -16.898085 |    0.376 |  0.0376 |      0.0 |      0.0 | -0.020742 | 0.664621 |    -0.633289 |     0.750307 |      0.0 |      0.0 | 0.086491 | 0.086491 | -15.369294 | -20.793068 |  -19.264276 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.254871 | 21.507023 |         0.0 | 0.905549 | -389.593838 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 21:00:00 |  0.7 |  2.0 |             99.6 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.658 |  2.0 |    4.488911 |           4.488911 |                 7.058357 |                     6.000528 |             7.030123 | -4.675126 | -26.594708 |                       0.0 |                15.419061 |       -15.419061 |    0.658 |     0.0 |      0.0 |      0.0 | -0.022456 | 1.530508 |    -0.242675 |     0.749673 |      0.0 |      0.0 | 0.035045 | 0.035045 | -16.639054 | -20.592817 |   -21.81281 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.311486 | 22.187479 |         0.0 | 0.912885 |  -354.39913 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 22:00:00 |  0.5 |  2.4 |             99.4 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  2.4 |    4.649229 |           4.649229 |                 7.263399 |                       6.1078 |             7.219819 |  -4.25279 | -25.516737 |                       0.0 |                 13.71239 |        -13.71239 |     0.47 |     0.0 |      0.0 |      0.0 | -0.024931 | 1.311864 |          0.0 |     0.749044 | 0.034587 | 0.034587 |      0.0 |      0.0 | -18.473108 |  -22.65304 |  -27.413757 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.300311 | 22.682411 |         0.0 | 0.920215 | -320.282392 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 23:00:00 |  1.0 |  2.5 |             99.2 |       6.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.94 |  2.5 |    4.969866 |           4.969866 |                 7.315473 |                       6.1078 |             7.256949 | -3.843389 | -23.060332 |                       0.0 |                13.067893 |       -13.067893 |     0.94 |     0.0 |      0.0 |      0.0 | -0.027164 |  2.73305 |          0.0 |     0.748418 | 0.125157 | 0.125157 |      0.0 |      0.0 |  -20.12751 | -24.879463 |   -31.93908 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.199504 | 23.649575 |         0.0 | 0.927567 |  -288.62206 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 00:00:00 |  2.0 |  2.7 |             99.2 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  1.88 |  2.7 |    4.488911 |           4.488911 |                 7.420609 |                       6.1078 |             7.361244 | -3.463465 | -20.780788 |                       0.0 |                11.601106 |       -11.601106 |     1.88 |     0.0 |      0.0 |      0.0 | -0.027338 | 5.903388 |          0.0 |     0.747794 | 0.200158 | 0.200158 |      0.0 |      0.0 |  -20.25663 | -24.792095 |  -33.447619 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.021833 | 25.556913 |         0.0 | 0.934958 | -259.241272 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 01:00:00 |  2.1 |  3.2 |             99.6 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.974 |  3.2 |    4.168274 |           4.168274 |                 7.689299 |                       6.1078 |             7.658542 | -3.110895 | -18.665372 |                       0.0 |                 7.733351 |        -7.733351 |    1.974 |     0.0 |      0.0 |      0.0 | -0.031933 | 7.346438 |          0.0 |     0.747174 | 0.348686 | 0.348686 |      0.0 |      0.0 | -23.661007 | -27.741565 |  -43.669221 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.697126 | 27.562846 |         0.0 | 0.942357 |   -231.9759 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 02:00:00 |  1.8 |  3.9 |             99.8 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.692 |  3.9 |     4.56907 |            4.56907 |                  8.07986 |                       6.1078 |               8.0637 | -2.783711 | -16.702265 |                       0.0 |                 2.415944 |        -2.415944 |    1.692 |     0.0 |      0.0 |      0.0 | -0.043255 | 7.674404 |          0.0 |     0.746559 | 0.613483 | 0.613483 |      0.0 |      0.0 | -32.050365 | -36.310997 |  -65.945417 |      1.0 |      0.3 | 0.536027 | 0.182986 |  0.0 |  0.0 | 0.0 |  0.0 | 0.182986 | 0.182986 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002374 |      0.0 |   0.0 |   0.0 |  0.0 | 0.002374 | 0.000659 |     0.06 | 20.113339 | 28.762074 |         0.0 |  0.94975 | -206.673635 | 71.799248 | 0.180612 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 | 0.000659 |
    | 2010-12-11 03:00:00 |  1.3 |  4.4 |             99.8 |       6.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.222 |  4.4 |    5.290502 |           5.290502 |                  8.36943 |                       6.1078 |             8.352691 | -2.480084 | -14.880502 |                       0.0 |                -1.360594 |         1.360594 |    1.222 |     0.0 |      0.0 |      0.0 | -0.055801 |      0.0 |          0.0 |     0.745956 | 0.796222 | 0.796222 |      0.0 |      0.0 | -41.346526 | -46.045135 |  -88.752255 |      1.0 |      0.3 | 2.362807 |  0.80844 |  0.0 |  0.0 | 0.0 |  0.0 |  0.80844 |      0.5 |  0.30844 |   0.0 |   0.0 |  0.0 | 0.011151 | 0.015595 |   0.0 |   0.0 |  0.0 | 0.026746 |  0.00743 |     0.06 | 19.354593 | 27.677068 |         0.0 | 0.957058 | -183.193134 | 73.353615 | 0.669461 | 0.292845 |  0.0 |  0.0 | 0.0 |   0.0 |  0.00743 |
    | 2010-12-11 04:00:00 |  0.5 |  4.8 |             99.0 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  4.8 |    5.611139 |           5.611139 |                 8.607627 |                       6.1078 |             8.521551 | -2.198318 | -13.189906 |                       0.0 |                -4.039084 |         4.039084 |     0.47 |     0.0 |      0.0 |      0.0 |  -0.06294 |      0.0 |          0.0 |     0.745357 | 0.971986 | 0.971986 |      0.0 |      0.0 | -46.635995 | -52.693545 | -103.368624 |      1.0 |      0.3 | 1.863101 | 0.639047 |  0.0 |  0.0 | 0.0 |  0.0 | 0.639047 |      0.5 | 0.139047 |   0.0 |   0.0 |  0.0 | 0.023779 | 0.036136 |   0.0 |   0.0 |  0.0 | 0.059914 | 0.016643 |     0.06 | 18.424411 | 26.346907 |         0.0 | 0.964366 | -161.403228 | 74.577669 | 1.145683 | 0.395756 |  0.0 |  0.0 | 0.0 |   0.0 | 0.016643 |
    | 2010-12-11 05:00:00 |  0.2 |  4.8 |             97.6 |       7.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  4.8 |    5.851616 |           5.851616 |                 8.607627 |                       6.1078 |             8.401044 | -1.936839 | -11.621032 |                       0.0 |                -3.389325 |         3.389325 |    0.188 |     0.0 |      0.0 |      0.0 | -0.061894 |      0.0 |          0.0 |     0.744763 | 0.993442 | 0.993442 |      0.0 |      0.0 | -45.860636 | -54.540411 | -103.790373 |      1.0 |      0.3 | 1.611959 | 0.554058 |  0.0 |  0.0 | 0.0 |  0.0 | 0.554058 |      0.5 | 0.054058 |   0.0 |   0.0 |  0.0 |  0.03608 | 0.042067 |   0.0 |   0.0 |  0.0 | 0.078147 | 0.021707 |     0.06 | 17.471917 | 24.984842 |         0.0 | 0.971674 | -141.182196 | 75.635571 | 1.609603 | 0.407747 |  0.0 |  0.0 | 0.0 |   0.0 | 0.021707 |
    | 2010-12-11 06:00:00 |  0.5 |  5.1 |             97.4 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  5.1 |    5.771457 |           5.771457 |                 8.790172 |                       6.1078 |             8.561627 | -1.694186 | -10.165118 |                       0.0 |                -5.587229 |         5.587229 |     0.47 |     0.0 |      0.0 |      0.0 |  -0.06548 |      0.0 |          0.0 |     0.744173 | 1.091159 | 1.091159 |      0.0 |      0.0 |  -48.51811 | -57.295089 | -111.400428 |      1.0 |      0.3 | 2.034447 | 0.700932 |  0.0 |  0.0 | 0.0 |  0.0 | 0.700932 |      0.5 | 0.200932 |   0.0 |   0.0 |  0.0 | 0.048063 | 0.050685 |   0.0 |   0.0 |  0.0 | 0.098748 |  0.02743 |     0.06 | 16.423689 | 23.485875 |         0.0 | 0.978982 | -122.417078 | 76.969086 | 2.061539 | 0.557994 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02743 |
    | 2010-12-11 07:00:00 |  0.2 |  5.1 |             96.3 |       7.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  5.1 |    6.092093 |           6.092093 |                 8.790172 |                       6.1078 |             8.464935 | -1.469005 |   -8.81403 |                       0.0 |                -5.066448 |         5.066448 |    0.188 |     0.0 |      0.0 |      0.0 | -0.065772 |      0.0 |          0.0 |     0.743587 | 1.130642 | 1.130642 |      0.0 |      0.0 | -48.734564 | -59.911482 | -113.712494 |      1.0 |      0.3 | 1.809346 | 0.624825 |  0.0 |  0.0 | 0.0 |  0.0 | 0.624825 |      0.5 | 0.124825 |   0.0 |   0.0 |  0.0 | 0.059737 |  0.06177 |   0.0 |   0.0 |  0.0 | 0.121507 | 0.033752 |     0.06 | 15.335875 | 21.930301 |         0.0 |  0.98629 | -105.003048 | 78.153607 | 2.501802 |  0.62105 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033752 |
    | 2010-12-11 08:00:00 |  0.1 |  5.4 |             95.2 |       7.3 |                 0.347041 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        0.900734 | 0.094 |  5.4 |    5.851616 |           5.851616 |                 8.976116 |                       6.1078 |             8.545263 | -1.260037 |  -7.560219 |                  0.231483 |                -6.835277 |          7.06676 |    0.094 |     0.0 |      0.0 |      0.0 | -0.065786 |      0.0 |          0.0 |     0.743006 | 1.181417 | 1.181417 |      0.0 |      0.0 | -48.744743 | -61.357963 | -117.169466 |      1.0 |      0.3 | 1.788494 | 0.618999 |  0.0 |  0.0 | 0.0 |  0.0 | 0.618999 |      0.5 | 0.118999 |   0.0 |   0.0 |  0.0 |  0.07111 | 0.067742 |   0.0 |   0.0 |  0.0 | 0.138852 |  0.03857 |     0.06 | 14.196918 | 20.301593 |         0.0 | 0.993598 |  -88.842829 | 79.323102 | 2.930693 | 0.672307 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03857 |
    | 2010-12-11 09:00:00 |  0.3 |  5.3 |             96.3 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        17.26751 | 0.282 |  5.3 |    5.691298 |           5.691298 |                 8.913754 |                       6.1078 |             8.583945 | -1.066114 |  -6.396684 |                  4.447615 |                -6.596815 |        11.044431 |    0.282 |     0.0 |      0.0 |      0.0 | -0.065321 |      0.0 |          0.0 |     0.742429 |  1.20622 |  1.20622 |      0.0 |      0.0 | -48.400444 | -58.862205 | -118.307081 |      1.0 |      0.3 | 2.012444 | 0.698193 |  0.0 |  0.0 | 0.0 |  0.0 | 0.698193 |      0.5 | 0.198193 |   0.0 |   0.0 |  0.0 | 0.082188 |  0.07684 |   0.0 |   0.0 |  0.0 | 0.159029 | 0.044175 |     0.06 | 13.032497 |  18.63647 |         0.0 | 1.000906 |  -73.846145 | 80.637354 | 3.348504 | 0.793659 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044175 |
    | 2010-12-11 10:00:00 |  0.1 |  5.5 |             95.5 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        52.61761 | 0.094 |  5.5 |    5.691298 |           5.691298 |                 9.038862 |                       6.1078 |             8.632113 | -0.886154 |  -5.316922 |                 13.582931 |                -7.748872 |        21.331804 |    0.094 |     0.0 |      0.0 |      0.0 | -0.066592 |      0.0 |          0.0 |     0.741856 | 1.362829 | 1.362829 |      0.0 |      0.0 | -49.341985 | -61.083421 | -131.757209 |      1.0 |      0.3 |  2.04981 | 0.712957 |  0.0 |  0.0 | 0.0 |  0.0 | 0.712957 |      0.5 | 0.212957 |   0.0 |   0.0 |  0.0 | 0.092981 | 0.089648 |   0.0 |   0.0 |  0.0 | 0.182629 |  0.05073 |     0.06 | 11.711365 | 16.747253 |         0.0 | 1.008214 |  -59.929222 | 81.974207 | 3.755524 | 0.916968 |  0.0 |  0.0 | 0.0 |   0.0 |  0.05073 |
    | 2010-12-11 11:00:00 |  0.1 |  6.1 |             91.6 |       8.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.135633 | 0.094 |  6.1 |    7.134162 |           7.134162 |                 9.423511 |                       6.1078 |             8.631936 | -0.719151 |  -4.314904 |                 20.732126 |               -12.026271 |        32.758397 |    0.094 |     0.0 |      0.0 |      0.0 | -0.080429 |      0.0 |          0.0 |     0.741287 | 1.830905 | 1.830905 |      0.0 |      0.0 |  -59.59435 | -81.829421 | -174.182168 |      1.0 |      0.3 | 2.724768 | 0.950702 |  0.0 |  0.0 | 0.0 |  0.0 | 0.950702 |      0.5 | 0.450702 |   0.0 |   0.0 |  0.0 | 0.103494 | 0.113924 |   0.0 |   0.0 |  0.0 | 0.217418 | 0.060394 |     0.06 |  9.927912 | 14.196914 |         0.0 | 1.015522 |  -47.014318 | 83.748272 | 4.152029 | 1.253746 |  0.0 |  0.0 | 0.0 |   0.0 | 0.060394 |
    | 2010-12-11 12:00:00 |  0.0 |  6.4 |             90.0 |       9.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.276118 |   0.0 |  6.4 |    7.615117 |           7.615117 |                 9.621181 |                       6.1078 |             8.659063 | -0.564172 |  -3.385031 |                 20.813797 |               -13.533625 |        34.347422 |      0.0 |     0.0 |      0.0 |      0.0 | -0.085956 |      0.0 |          0.0 |     0.740722 | 1.998661 | 1.998661 |      0.0 |      0.0 | -63.690154 | -90.778796 | -188.816371 |      1.0 |      0.3 |  2.87539 | 1.006791 |  0.0 |  0.0 | 0.0 |  0.0 | 1.006791 |      0.5 | 0.506791 |   0.0 |   0.0 |  0.0 | 0.113736 | 0.150232 |   0.0 |   0.0 |  0.0 | 0.263968 | 0.073325 |     0.06 |  7.977259 |  11.40748 |         0.0 |  1.02283 |  -35.029287 | 85.616871 | 4.538293 | 1.610305 |  0.0 |  0.0 | 0.0 |   0.0 | 0.073325 |
    | 2010-12-11 13:00:00 |  0.1 |  6.4 |             85.7 |       7.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       75.835403 | 0.094 |  6.4 |    6.332571 |           6.332571 |                 9.621181 |                       6.1078 |             8.245352 | -0.420351 |  -2.522109 |                  19.70493 |               -11.239491 |        30.944421 |    0.094 |     0.0 |      0.0 |      0.0 | -0.061599 |      0.0 |          0.0 |     0.740162 | 1.635197 | 1.635197 |      0.0 |      0.0 | -45.642119 | -77.645525 | -154.232066 |      1.0 |      0.3 | 2.444959 | 0.858816 |  0.0 |  0.0 | 0.0 |  0.0 | 0.858816 |      0.5 | 0.358816 |   0.0 |   0.0 |  0.0 | 0.123714 | 0.178188 |   0.0 |   0.0 |  0.0 | 0.301902 | 0.083862 |     0.06 |  6.376308 |  9.118121 |         0.0 | 1.030138 |  -23.907179 | 87.203013 | 4.914579 | 1.790933 |  0.0 |  0.0 | 0.0 |   0.0 | 0.083862 |
    | 2010-12-11 14:00:00 |  0.0 |  6.5 |             82.3 |       9.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       42.227911 |   0.0 |  6.5 |    7.695276 |           7.695276 |                 9.687876 |                       6.1078 |             7.973122 | -0.286886 |  -1.721317 |                 10.995923 |               -10.124562 |        21.120486 |      0.0 |     0.0 |      0.0 |      0.0 | -0.063414 |      0.0 |          0.0 |     0.739605 |  1.71827 |  1.71827 |      0.0 |      0.0 | -46.987277 | -93.030869 | -161.138633 |      1.0 |      0.3 | 2.474215 | 0.871763 |  0.0 |  0.0 | 0.0 |  0.0 | 0.871763 |      0.5 | 0.371763 |   0.0 |   0.0 |  0.0 | 0.133434 | 0.196795 |   0.0 |   0.0 |  0.0 | 0.330229 |  0.09173 |     0.06 |  4.690433 |  6.707319 |         0.0 | 1.037446 |  -13.585862 | 88.805465 | 5.281145 | 1.965902 |  0.0 |  0.0 | 0.0 |   0.0 |  0.09173 |
    | 2010-12-11 15:00:00 |  0.1 |  6.3 |             82.5 |       9.1 |                      1.0 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.40894 |       12.613921 | 0.094 |  6.3 |     7.29448 |            7.29448 |                 9.554889 |                       6.1078 |             7.882784 |  -0.16303 |  -0.978182 |                  3.291568 |                -7.130286 |        10.421854 |    0.094 |     0.0 |      0.0 |      0.0 | -0.057639 |      0.0 |          0.0 |     0.739053 | 1.490447 | 1.490447 |      0.0 |      0.0 |  -42.70834 | -86.128361 | -139.258555 |      1.0 |      0.3 | 2.243655 | 0.792804 |  0.0 |  0.0 | 0.0 |  0.0 | 0.792804 |      0.5 | 0.292804 |   0.0 |   0.0 |  0.0 | 0.142902 | 0.210193 |   0.0 |   0.0 |  0.0 | 0.353095 | 0.098082 |     0.06 |  3.227485 |  4.615304 |         0.0 | 1.044754 |    -4.00768 | 90.256316 | 5.638243 | 2.048514 |  0.0 |  0.0 | 0.0 |   0.0 | 0.098082 |
    | 2010-12-11 16:00:00 |  0.0 |  5.9 |             82.2 |       8.7 |                 0.053708 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |        0.021912 |   0.0 |  5.9 |    6.973844 |           6.973844 |                 9.293725 |                       6.1078 |             7.639442 | -0.048092 |  -0.288553 |                   0.00573 |                -3.915548 |         3.921278 |      0.0 |     0.0 |      0.0 |      0.0 | -0.047871 |      0.0 |          0.0 |     0.738504 | 1.258233 | 1.258233 |      0.0 |      0.0 | -35.470286 | -77.633085 | -117.024649 | 0.999342 | 0.299606 | 1.817936 | 0.643268 |  0.0 |  0.0 | 0.0 |  0.0 | 0.643268 |      0.5 | 0.143268 |   0.0 |   0.0 |  0.0 | 0.152127 | 0.210843 |   0.0 |   0.0 |  0.0 | 0.362969 | 0.100825 |     0.06 |  1.989677 |  2.845238 |         0.0 | 1.052062 |    4.880873 | 91.430985 | 5.986117 | 1.980938 |  0.0 |  0.0 | 0.0 |   0.0 | 0.100825 |
    | 2010-12-11 17:00:00 |  0.0 |  5.5 |             76.5 |       8.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.5 |    7.134162 |           7.134162 |                 9.038862 |                       6.1078 |              6.91473 |       0.0 |        0.0 |                       0.0 |                 2.340465 |        -2.340465 |      0.0 |     0.0 |      0.0 |      0.0 | -0.025712 |      0.0 |          0.0 |     0.737959 | 0.975359 | 0.975359 |      0.0 |      0.0 | -19.051448 | -73.780626 |  -90.491608 | 0.998184 | 0.298911 | 1.407367 | 0.498039 |  0.0 |  0.0 | 0.0 |  0.0 | 0.498039 | 0.498039 |      0.0 |   0.0 |   0.0 |  0.0 | 0.161087 | 0.196883 |   0.0 |   0.0 |  0.0 |  0.35797 | 0.099436 |     0.06 |  1.023485 |  1.463583 |         0.0 |  1.05937 |   13.480873 | 92.340313 | 6.323068 | 1.784056 |  0.0 |  0.0 | 0.0 |   0.0 | 0.099436 |
    | 2010-12-11 18:00:00 |  0.0 |  5.1 |             81.7 |       7.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.1 |    6.332571 |           6.332571 |                 8.790172 |                       6.1078 |              7.18157 |       0.0 |        0.0 |                       0.0 |                 2.381726 |        -2.381726 |      0.0 |     0.0 |      0.0 |      0.0 | -0.030943 |      0.0 |          0.0 |     0.737419 | 0.888357 | 0.888357 |      0.0 |      0.0 | -22.927701 | -61.873778 |  -82.419753 | 0.997025 | 0.298218 | 1.297208 | 0.458993 |  0.0 |  0.0 | 0.0 |  0.0 | 0.458993 | 0.458993 |      0.0 |   0.0 |   0.0 |  0.0 | 0.169284 | 0.177315 |   0.0 |   0.0 |  0.0 | 0.346599 | 0.096277 |     0.06 |  0.137985 |  0.197318 |         0.0 | 1.066678 |   22.080873 | 93.178528 | 6.612777 | 1.606741 |  0.0 |  0.0 | 0.0 |   0.0 | 0.096277 |
    | 2010-12-11 19:00:00 |  0.0 |  4.8 |             83.4 |       7.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.8 |    5.691298 |           5.691298 |                 8.607627 |                       6.1078 |             7.178761 |       0.0 |        0.0 |                       0.0 |                 3.699875 |        -3.699875 |      0.0 |     0.0 |     0.06 | 0.003724 | -0.028252 |      0.0 |          0.0 |     0.736881 | 0.760344 | 0.137985 |      0.0 |      0.0 | -20.933744 | -53.309167 |  -70.543036 | 0.995866 | 0.297525 | 0.225571 | 0.079724 |  0.0 |  0.0 | 0.0 |  0.0 | 0.079724 | 0.079724 |      0.0 |   0.0 |   0.0 |  0.0 | 0.171848 | 0.159692 |   0.0 |   0.0 |  0.0 |  0.33154 | 0.092094 |      0.0 |       0.0 |       0.0 |         0.0 | 1.073986 |   30.680873 |  93.32065 | 6.520654 |  1.44705 |  0.0 |  0.0 | 0.0 |   0.0 | 0.092094 |
    | 2010-12-11 20:00:00 |  0.0 |  4.9 |             82.8 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.130184 |           5.130184 |                 8.668101 |                          0.0 |             7.177187 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.025203 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.990747 | 0.294474 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.168434 |  0.14382 |   0.0 |   0.0 |  0.0 | 0.312254 | 0.086737 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |   68.680873 | 93.295447 | 6.352219 | 1.303229 |  0.0 |  0.0 | 0.0 |   0.0 | 0.086737 |
    | 2010-12-11 21:00:00 |  0.0 |  4.9 |             85.8 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.771457 |           5.771457 |                 8.668101 |                          0.0 |              7.43723 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.021597 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.985627 | 0.291438 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.164083 | 0.129526 |   0.0 |   0.0 |  0.0 | 0.293609 | 0.081558 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  106.680873 | 93.273849 | 6.188136 | 1.173703 |  0.0 |  0.0 | 0.0 |   0.0 | 0.081558 |
    | 2010-12-11 22:00:00 |  0.1 |  4.7 |             87.6 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  4.7 |    5.611139 |           5.611139 |                 8.547525 |                          0.0 |             7.487632 |       0.0 |      -28.2 |                       0.0 |                      0.0 |              0.0 |    0.034 |     0.0 | 0.050942 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.980669 | 0.288513 |    0.034 | 0.011735 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011735 | 0.011735 |      0.0 |   0.0 |   0.0 |  0.0 | 0.159997 | 0.116653 |   0.0 |   0.0 |  0.0 |  0.27665 | 0.076847 | 0.009058 |       0.0 |       0.0 |         0.0 |      nan |  143.480873 | 93.296114 | 6.039874 | 1.057051 |  0.0 |  0.0 | 0.0 |   0.0 | 0.076847 |
    | 2010-12-11 23:00:00 |  0.1 |  3.8 |             85.6 |       6.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.8 |     5.45082 |            5.45082 |                 8.023017 |                          0.0 |             6.867703 |       0.0 |      -22.8 |                       0.0 |                      0.0 |              0.0 | 0.043058 |     0.0 | 0.055793 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.976438 | 0.286029 | 0.043058 | 0.014763 |  0.0 |  0.0 | 0.0 |  0.0 | 0.014763 | 0.014763 |      0.0 |   0.0 |   0.0 |  0.0 | 0.156207 | 0.105059 |   0.0 |   0.0 |  0.0 | 0.261265 | 0.072574 | 0.004207 |       0.0 |       0.0 |         0.0 |      nan |  174.880873 | 93.324409 | 5.898431 | 0.951992 |  0.0 |  0.0 | 0.0 |   0.0 | 0.072574 |
    | 2010-12-12 00:00:00 |  0.0 |  3.9 |             85.3 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.9 |    5.130184 |           5.130184 |                  8.07986 |                          0.0 |             6.892121 |       0.0 |      -23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.004207 |  0.01859 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.972127 | 0.283509 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.152362 | 0.094617 |   0.0 |   0.0 |  0.0 | 0.246979 | 0.068605 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  206.880873 | 93.305819 | 5.746069 | 0.857375 |  0.0 |  0.0 | 0.0 |   0.0 | 0.068605 |
    | 2010-12-12 01:00:00 |  0.0 |  3.7 |             86.2 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.7 |    5.050025 |           5.050025 |                 7.966527 |                          0.0 |             6.867146 |       0.0 |      -22.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.018522 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.967977 | 0.281094 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.148426 | 0.085213 |   0.0 |   0.0 |  0.0 | 0.233639 |   0.0649 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  237.680873 | 93.287297 | 5.597643 | 0.772162 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0649 |
    | 2010-12-12 02:00:00 |  0.1 |  3.5 |             89.3 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.5 |     4.56907 |            4.56907 |                 7.854597 |                          0.0 |             7.014155 |       0.0 |      -21.0 |                       0.0 |                      0.0 |              0.0 |    0.034 |     0.0 |  0.03356 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.963989 | 0.278782 |    0.034 | 0.011431 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011431 | 0.011431 |      0.0 |   0.0 |   0.0 |  0.0 |  0.14474 | 0.076744 |   0.0 |   0.0 |  0.0 | 0.221484 | 0.061523 |  0.02644 |       0.0 |       0.0 |         0.0 |      nan |  267.280873 | 93.309866 | 5.464334 | 0.695418 |  0.0 |  0.0 | 0.0 |   0.0 | 0.061523 |
    | 2010-12-12 03:00:00 |  0.0 |  3.3 |             91.4 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |    4.168274 |           4.168274 |                 7.744054 |                          0.0 |             7.078066 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.023813 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.960163 | 0.276574 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.141148 | 0.069117 |   0.0 |   0.0 |  0.0 | 0.210265 | 0.058407 | 0.002628 |       0.0 |       0.0 |         0.0 |      nan |  295.680873 | 93.309866 | 5.323185 | 0.626301 |  0.0 |  0.0 | 0.0 |   0.0 | 0.058407 |
    | 2010-12-12 04:00:00 |  0.0 |  3.3 |             90.9 |       4.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |     3.68732 |            3.68732 |                 7.744054 |                          0.0 |             7.039346 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.002628 |  0.00878 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.956336 | 0.274374 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.137502 | 0.062247 |   0.0 |   0.0 |  0.0 |  0.19975 | 0.055486 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  324.080873 | 93.301086 | 5.185683 | 0.564054 |  0.0 |  0.0 | 0.0 |   0.0 | 0.055486 |
    | 2010-12-12 05:00:00 |  0.0 |  3.0 |             89.2 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.0 |    3.927797 |           3.927797 |                 7.580812 |                          0.0 |             6.762084 |       0.0 |      -18.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.012215 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.952753 | 0.272321 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.133951 |  0.05606 |   0.0 |   0.0 |  0.0 | 0.190011 | 0.052781 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  350.680873 | 93.288871 | 5.051732 | 0.507993 |  0.0 |  0.0 | 0.0 |   0.0 | 0.052781 |
    | 2010-12-12 06:00:00 |  0.0 |  1.9 |             85.6 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.9 |    5.050025 |           5.050025 |                 7.007901 |                          0.0 |             5.998763 |       0.0 |      -11.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.017456 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.950058 | 0.270783 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.130491 | 0.050489 |   0.0 |   0.0 |  0.0 | 0.180979 | 0.050272 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  370.680873 | 93.271415 | 4.921241 | 0.457505 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050272 |
    | 2010-12-12 07:00:00 |  0.0 |  1.2 |             86.3 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.2 |    4.649229 |           4.649229 |                 6.663527 |                          0.0 |             5.750624 |       0.0 |       -7.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.015363 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.947929 | 0.269571 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.12712 | 0.045471 |   0.0 |   0.0 |  0.0 | 0.172591 | 0.047942 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  386.480873 | 93.256052 | 4.794122 | 0.412034 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047942 |
    | 2010-12-12 08:00:00 |  0.0 |  0.8 |             91.2 |       6.1 |                 0.339738 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        0.894145 |   0.0 |  0.8 |    4.889707 |           4.889707 |                 6.473522 |                          0.0 |             5.903852 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.009666 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.946124 | 0.268545 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.123836 | 0.040951 |   0.0 |   0.0 |  0.0 | 0.164788 | 0.045774 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  399.880873 | 93.246386 | 4.670285 | 0.371082 |  0.0 |  0.0 | 0.0 |   0.0 | 0.045774 |
    | 2010-12-12 09:00:00 |  0.0 |  0.8 |             87.5 |       5.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        17.71254 |   0.0 |  0.8 |    4.729388 |           4.729388 |                 6.473522 |                          0.0 |             5.664332 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.017185 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.944318 | 0.267521 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.120637 | 0.036881 |   0.0 |   0.0 |  0.0 | 0.157519 | 0.043755 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  413.280873 | 93.229201 | 4.549648 | 0.334201 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043755 |
    | 2010-12-12 10:00:00 |  0.1 |  0.5 |             84.5 |       6.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       54.302585 | 0.094 |  0.5 |    5.130184 |           5.130184 |                 6.334173 |                     5.554612 |             5.352376 |       0.0 |   7.766328 |                  5.556421 |                28.031613 |       -22.475192 |    0.034 | 0.01275 |      0.0 |      0.0 |  0.004904 | 0.006889 |    -1.294388 |     0.897677 |      0.0 |      0.0 | 0.000192 | 0.000192 |   3.633499 |  -18.31764 |    7.791051 | 0.944206 | 0.267458 | 0.015768 | 0.005136 |  0.0 |  0.0 | 0.0 |  0.0 | 0.005136 | 0.005136 |      0.0 |   0.0 |   0.0 |  0.0 | 0.117588 | 0.033216 |   0.0 |   0.0 |  0.0 | 0.150804 |  0.04189 |     0.06 |  0.009461 |  0.013329 |         0.0 |  0.00743 |  414.114545 | 93.239832 | 4.437196 | 0.300985 |  0.0 |  0.0 | 0.0 |   0.0 |  0.04189 |
    | 2010-12-12 11:00:00 |  0.0 |  0.3 |             76.7 |       6.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.504146 |   0.0 |  0.3 |     5.45082 |            5.45082 |                  6.24275 |                     5.354685 |             4.788189 |       0.0 |  10.742078 |                  8.523311 |                31.005039 |       -22.481729 |      0.0 |     0.0 |     0.06 | 0.008326 |  0.013329 |      0.0 |    -1.790346 |     0.895425 |      0.0 |      0.0 | 0.000193 | 0.000193 |  10.689492 | -22.411256 |   10.759964 | 0.944495 | 0.267621 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.114617 | 0.029914 |   0.0 |   0.0 |  0.0 | 0.144531 | 0.040148 |      0.0 |       0.0 |       0.0 |         0.0 | 0.014738 |  411.972468 | 93.231506 | 4.322579 | 0.271071 |  0.0 |  0.0 | 0.0 |   0.0 | 0.040148 |
    | 2010-12-12 12:00:00 |  0.0 | -0.1 |             74.8 |       7.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.665294 |   0.0 | -0.1 |    5.611139 |           5.611139 |                 6.063393 |                          0.0 |             4.535418 |       0.0 |        0.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.042749 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.943417 | 0.267011 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.111656 | 0.026941 |   0.0 |   0.0 |  0.0 | 0.138597 | 0.038499 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  419.972468 | 93.188757 | 4.210923 |  0.24413 |  0.0 |  0.0 | 0.0 |   0.0 | 0.038499 |
    | 2010-12-12 13:00:00 |  0.0 | -0.4 |             69.7 |       6.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       77.078959 |   0.0 | -0.4 |    5.050025 |           5.050025 |                 5.931876 |                          0.0 |             4.134518 |       0.0 |        2.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.046734 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.942582 | 0.266538 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.108772 | 0.024264 |   0.0 |   0.0 |  0.0 | 0.133035 | 0.036954 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  426.172468 | 93.142023 | 4.102151 | 0.219866 |  0.0 |  0.0 | 0.0 |   0.0 | 0.036954 |
    | 2010-12-12 14:00:00 |  0.0 | -0.6 |             66.9 |       7.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.3 |                      7.393446 |       60.254042 |   0.0 | -0.6 |    5.851616 |           5.851616 |                 5.845604 |                          0.0 |             3.910709 |       0.0 |        3.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.047231 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.941908 | 0.266157 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.105962 | 0.021852 |   0.0 |   0.0 |  0.0 | 0.127814 | 0.035504 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  431.172468 | 93.094792 | 3.996189 | 0.198014 |  0.0 |  0.0 | 0.0 |   0.0 | 0.035504 |
    | 2010-12-12 15:00:00 |  0.0 | -0.8 |             64.4 |       7.5 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.4 |                      7.393446 |       14.837301 |   0.0 | -0.8 |    6.011934 |           6.011934 |                 5.760442 |                          0.0 |             3.709725 |       0.0 |        4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 |  0.04262 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.941396 | 0.265868 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.103225 |  0.01968 |   0.0 |   0.0 |  0.0 | 0.122905 |  0.03414 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  434.972468 | 93.052172 | 3.892964 | 0.178333 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03414 |
    | 2010-12-12 16:00:00 |  0.0 | -1.3 |             65.4 |       7.0 |                 0.046405 |              0.2 | 0.0 | 0.0 |                   0.6 |                      7.386143 |        0.168856 |   0.0 | -1.3 |    5.611139 |           5.611139 |                 5.552313 |                          0.0 |             3.631213 |       0.0 |        7.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 |  0.03666 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.941288 | 0.265807 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.100559 | 0.017724 |   0.0 |   0.0 |  0.0 | 0.118283 | 0.032856 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  435.772468 | 93.015512 | 3.792406 | 0.160609 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032856 |
    | 2010-12-12 17:00:00 |  0.0 | -1.6 |             67.6 |       6.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -1.6 |    4.809547 |           4.809547 |                 5.430651 |                          0.0 |              3.67112 |       0.0 |        9.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.031633 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.941423 | 0.265883 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.097961 | 0.015963 |   0.0 |   0.0 |  0.0 | 0.113924 | 0.031646 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  434.772468 | 92.983879 | 3.694445 | 0.144646 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031646 |
    | 2010-12-12 18:00:00 |  0.0 | -2.0 |             70.2 |       4.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.0 |    3.847638 |           3.847638 |                 5.272101 |                          0.0 |             3.701015 |       0.0 |       12.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.025662 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.941881 | 0.266142 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.095431 | 0.014376 |   0.0 |   0.0 |  0.0 | 0.109807 | 0.030502 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  431.372468 | 92.958217 | 3.599014 |  0.13027 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030502 |
    | 2010-12-12 19:00:00 |  0.0 | -2.1 |             70.4 |       4.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.1 |    3.366683 |           3.366683 |                 5.233109 |                          0.0 |             3.684109 |       0.0 |       12.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.023754 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |  0.94242 | 0.266447 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.092966 | 0.012947 |   0.0 |   0.0 |  0.0 | 0.105913 |  0.02942 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  427.372468 | 92.934463 | 3.506048 | 0.117323 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02942 |
    | 2010-12-12 20:00:00 |  0.0 | -2.2 |             69.5 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.2 |    3.927797 |           3.927797 |                 5.194372 |                          0.0 |             3.610088 |       0.0 |       13.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.026193 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |  0.94304 | 0.266797 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.090564 | 0.011661 |   0.0 |   0.0 |  0.0 | 0.102225 | 0.028396 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  422.772468 |  92.90827 | 3.415484 | 0.105662 |  0.0 |  0.0 | 0.0 |   0.0 | 0.028396 |
    | 2010-12-12 21:00:00 |  0.0 | -2.6 |             71.6 |       4.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.6 |    3.206365 |           3.206365 |                 5.041946 |                          0.0 |             3.610033 |       0.0 |       15.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.021428 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.943983 | 0.267331 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.088225 | 0.010502 |   0.0 |   0.0 |  0.0 | 0.098727 | 0.027424 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  415.772468 | 92.886842 | 3.327259 | 0.095161 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027424 |
    | 2010-12-12 22:00:00 |  0.0 | -3.1 |             74.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.1 |     2.72541 |            2.72541 |                 4.856984 |                          0.0 |             3.618453 |       0.0 |       18.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.016826 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |  0.94533 | 0.268095 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.085946 | 0.009458 |   0.0 |   0.0 |  0.0 | 0.095404 | 0.026501 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  405.772468 | 92.870016 | 3.241313 | 0.085703 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026501 |
    | 2010-12-12 23:00:00 |  0.0 | -3.9 |             80.2 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.9 |     2.72541 |            2.72541 |                  4.57351 |                          0.0 |             3.667955 |       0.0 |       23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.011894 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.947324 | 0.269227 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083726 | 0.008518 |   0.0 |   0.0 |  0.0 | 0.092244 | 0.025623 |      0.0 |       0.0 |       0.0 |         0.0 |      nan |  390.972468 | 92.858122 | 3.157587 | 0.077185 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025623 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_winter))
0.0

.. _lland_knauf_nadelw_winter_hourly:

conifers (winter)
-----------------

By comparing the hourly simulation results for |ACKER|
(:ref:`lland_knauf_acker_winter_hourly`) and |NADELW| (this example), you can see the
shading effect of the tree canopies.  During the nighttime, the decreased longwave
radiation loss (|NetLongwaveRadiationSnow|) under the canopy results in an earlier
start of the snow melt.  But then, during the daytime, the decreased shortwave
radiation gain (|NetShortwaveRadiationSnow|) results in a lower melting rate, so that
the complete melting process finishes even later for |NADELW| than for |ACKER|:

.. integration-test::

    >>> lnk(NADELW)
    >>> test("lland_knauf_nadelw_winter_hourly",
    ...      axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_nadelw_winter)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |     nbes |     sbes |       evi |      evb |       evs |     wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp |      wats |      waes |      esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |           1.206967 |                  4.64297 |                     4.898742 |             4.062599 | -1.726112 |   7.557321 |                       0.0 |                25.509942 |       -25.509942 | 0.145824 | 0.145824 |  0.023437 |      0.0 |  0.007808 | -0.313238 |    -2.985665 |     0.748702 |      0.0 |      0.0 |   0.2895 |   0.2895 |   5.785121 |   2.808156 |    34.10322 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.176563 | 12.563343 | 13.877816 |        0.0 | 0.924224 | -142.799953 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           1.029472 |                 4.272186 |                     4.763125 |             3.874873 | -1.713599 |   9.879594 |                       0.0 |                28.571909 |       -28.571909 |      0.0 |      0.0 |  0.007952 |      0.0 |  0.007695 |       0.0 |    -3.360198 |     0.748083 |      0.0 |      0.0 | 0.319529 | 0.319529 |   5.701688 |    5.25118 |   39.524777 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.168611 | 12.875729 | 13.870121 |        0.0 | 0.931532 | -144.079547 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           0.887476 |                 4.019239 |                     4.647896 |             3.605258 | -1.728955 |  11.741778 |                       0.0 |                30.735107 |       -30.735107 |      0.0 |      0.0 |  0.006645 |      0.0 |   0.00847 |       0.0 |    -3.685918 |     0.747468 |      0.0 |      0.0 | 0.342918 | 0.342918 |   6.275778 |   6.546089 |   43.556973 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.161967 | 13.210575 | 13.861651 |        0.0 |  0.93884 | -147.221325 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           1.064971 |                 3.838328 |                     4.543664 |             3.504393 | -1.766656 |  13.320647 |                       0.0 |                31.943116 |       -31.943116 |      0.0 |      0.0 |  0.006325 |      0.0 |  0.009144 |       0.0 |    -3.986764 |     0.746858 |      0.0 |      0.0 | 0.362103 | 0.362103 |   6.774963 |   8.197725 |   46.915804 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.155641 | 13.563725 | 13.852507 |        0.0 | 0.946148 | -151.941972 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           0.887476 |                  3.52531 |                      4.40958 |             3.320842 | -1.823304 |   15.35748 |                       0.0 |                  34.8128 |         -34.8128 |      0.0 |      0.0 | -0.003363 |      0.0 |  0.008844 |       0.0 |    -4.382884 |     0.746253 |      0.0 |      0.0 | 0.387862 | 0.288782 |    6.55326 |   9.976426 |   51.342486 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.159004 | 13.843663 | 13.843663 |  -9.192467 | 0.953456 | -158.699451 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |           0.816478 |                  3.26061 |                     4.145514 |             3.038889 | -1.904393 |   19.74499 |                       0.0 |                35.601434 |       -35.601434 |      0.0 |      0.0 |  -0.00234 |      0.0 |  0.008691 |       0.0 |    -5.195225 |     0.745832 |      0.0 |      0.0 | 0.450044 |      0.0 |   6.439675 |  10.265518 |   52.306627 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.161345 | 13.834972 | 13.834972 | -41.754104 | 0.958565 | -169.844441 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 06:00:00 |  0.0 | -8.6 |             96.5 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.6 |    1.683342 |            0.74548 |                 3.184716 |                     3.696753 |             3.073251 | -2.038133 |  27.894852 |                       0.0 |                30.242232 |       -30.242232 |      0.0 |      0.0 | -0.008564 |      0.0 |  0.004729 |       0.0 |    -6.687275 |     0.745608 |      0.0 |      0.0 | 0.578932 |      0.0 |   3.503625 |   6.106886 |   39.852742 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.169908 | 13.830243 | 13.830243 | -53.711994 |   0.9613 | -189.139294 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 07:00:00 |  0.0 | -8.5 |             95.9 |       1.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  3.76 |                           7.4 |             0.0 |   0.0 | -8.5 |    1.122228 |           0.496987 |                 3.209838 |                     3.560334 |             3.078235 | -2.269672 |  29.418242 |                       0.0 |                27.215865 |       -27.215865 |      0.0 |      0.0 | -0.007643 |      0.0 |  0.003201 |       0.0 |    -7.172712 |      0.74541 |      0.0 |      0.0 | 0.620745 |      0.0 |   2.371692 |   3.710007 |   33.297564 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.177552 | 13.827043 | 13.827043 | -57.591316 | 0.963718 | -209.957536 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 08:00:00 |  0.0 | -9.5 |             96.8 |       1.1 |                 0.355232 |              0.0 | 0.0 | 0.0 |                  2.76 |                      6.755232 |         1.10761 |   0.0 | -9.5 |     0.88175 |           0.390489 |                 2.966351 |                     3.414235 |             2.871427 |  -2.51949 |  31.150462 |                  0.052408 |                28.309773 |       -28.257365 |      0.0 |      0.0 | -0.006643 |      0.0 |  0.003384 |       0.0 |    -7.711234 |      0.74522 |      0.0 |      0.0 | 0.667195 |      0.0 |   2.507562 |   4.695122 |   35.460049 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.184195 | 13.823658 | 13.823658 | -61.900902 | 0.966049 | -232.507998 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 09:00:00 |  0.0 | -9.2 |             94.3 |       1.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                  1.79 |                      6.755232 |       18.663203 |   0.0 | -9.2 |    0.801591 |            0.35499 |                 3.037615 |                     3.394343 |             2.864471 | -2.790096 |  29.976203 |                  0.883706 |                25.420194 |       -24.536487 |      0.0 |      0.0 | -0.000866 |      0.0 |  0.003232 |       0.0 |     -7.78613 |     0.745037 |      0.0 |      0.0 | 0.673511 |      0.0 |   2.394837 |   3.630797 |   30.562122 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.185061 | 13.820426 | 13.820426 |  -62.48682 |  0.96829 | -253.884202 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 10:00:00 |  0.0 | -6.9 |             96.7 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.88 |                      6.755232 |       56.267907 |   0.0 | -6.9 |    1.122228 |           0.496987 |                 3.636412 |                     3.646117 |             3.516411 |  -3.04661 |  22.913797 |                  2.666188 |                17.444979 |       -14.778791 |      0.0 |      0.0 |  0.004438 |      0.0 |  0.000861 |       0.0 |    -6.865577 |     0.744856 |      0.0 |      0.0 | 0.593743 |      0.0 |   0.638093 |    0.09622 |   15.513103 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.180623 | 13.819565 | 13.819565 | -55.086127 | 0.970518 | -268.197998 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 11:00:00 |  0.0 | -4.0 |             97.4 |       1.3 |                      1.0 |              0.1 | 0.0 | 0.0 |                  0.39 |                      6.755232 |      107.168237 |   0.0 | -4.0 |    1.042069 |           0.461488 |                 4.539126 |                     4.115866 |             4.421109 | -3.218376 |  12.425367 |                  5.081886 |                 9.858721 |        -4.776834 |      0.0 |      0.0 |   0.01142 |      0.0 | -0.001985 |       0.0 |     -5.28927 |     0.744663 |      0.0 |      0.0 | 0.457393 |      0.0 |  -1.471131 |  -3.530512 |   -0.224809 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.169204 | 13.821551 | 13.821551 | -42.435951 | 0.972904 | -272.023365 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 12:00:00 |  0.0 | -2.8 |             93.9 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.14 |                      6.755232 |       91.473333 |   0.0 | -2.8 |    1.042069 |           0.461488 |                 4.967227 |                     4.432347 |             4.664226 |  -3.26428 |     6.3036 |                  4.341363 |                 8.098255 |        -3.756891 |      0.0 |      0.0 |  0.016046 |      0.0 | -0.001508 |       0.0 |     -4.31488 |     0.744444 |      0.0 |      0.0 | 0.373186 |      0.0 |  -1.117555 |  -4.148318 |   -1.508982 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.153158 | 13.823059 | 13.823059 | -34.623368 | 0.975617 | -269.726965 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 13:00:00 |  0.0 | -1.4 |             93.4 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       80.475535 |   0.0 | -1.4 |    1.122228 |           0.496987 |                 5.511494 |                     4.743594 |             5.147736 | -3.236724 |   1.069147 |                  3.822995 |                 5.293878 |        -1.470883 |      0.0 |      0.0 |   0.01742 |      0.0 | -0.002683 |       0.0 |    -3.414915 |     0.744204 |      0.0 |      0.0 | 0.295382 |      0.0 |  -1.988183 |  -5.632047 |   -6.149346 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.135738 | 13.825742 | 13.825742 | -27.404875 | 0.978599 | -262.196113 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 14:00:00 |  0.0 | -0.4 |             93.3 |       2.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       43.120591 |   0.0 | -0.4 |     1.84366 |           0.816478 |                 5.931876 |                     5.015537 |              5.53444 | -3.146353 |    -2.8558 |                  2.050581 |                  3.75039 |         -1.69981 |      0.0 |      0.0 |  0.020303 |      0.0 | -0.004075 |       0.0 |    -2.670387 |     0.743937 |      0.0 |      0.0 | 0.231027 |      0.0 |  -3.019601 |  -7.506726 |   -8.826518 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.115435 | 13.829817 | 13.829817 | -21.434156 | 0.981918 | -250.740313 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 15:00:00 |  0.0 |  0.5 |             94.1 |       2.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.355232 |       10.320091 |   0.0 |  0.5 |    2.244455 |           0.993973 |                 6.334173 |                     5.274495 |             5.960456 | -3.008884 |  -6.090003 |                  0.491336 |                 2.264562 |        -1.773226 |      0.0 |      0.0 |  0.017639 |      0.0 |  -0.00585 |       0.0 |    -1.993883 |      0.74364 |      0.0 |      0.0 |  0.17255 |      0.0 |  -4.334611 |  -8.953931 |  -11.515316 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.097796 | 13.835667 | 13.835667 | -16.008844 | 0.985622 |  -236.05031 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 16:00:00 |  0.1 |  1.0 |             94.6 |       3.1 |                 0.061898 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |        0.028102 | 0.094 |  1.0 |    2.484933 |            1.10047 |                 6.567919 |                     5.477296 |             6.213251 | -2.832604 |  -8.089914 |                   0.00134 |                 2.038921 |        -2.037581 |      0.0 |      0.0 |  0.016676 |      0.0 | -0.006574 |       0.0 |    -1.484285 |     0.743306 |      0.0 |      0.0 | 0.128504 |      0.0 |  -4.871229 |   -9.34278 |  -12.176427 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |  2.17512 | 13.842242 | 13.842242 |  -11.92233 | 0.989814 | -219.360396 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 17:00:00 |  0.0 |  1.1 |             94.4 |       3.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |   0.0 |  1.1 |    2.805569 |           1.242466 |                 6.615571 |                     5.593793 |             6.245099 | -2.632325 |  -8.599681 |                       0.0 |                  2.85358 |         -2.85358 |      0.0 |      0.0 |  0.020112 |      0.0 |  -0.00617 |       0.0 |    -1.199045 |     0.742934 |      0.0 |      0.0 | 0.103858 |      0.0 |  -4.571373 |  -9.168466 |   -10.88626 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.155008 | 13.848411 | 13.848411 |  -9.635751 | 0.994504 | -202.160716 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 18:00:00 |  0.3 |  1.0 |             96.9 |       4.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.282 |  1.0 |    3.527001 |           1.561958 |                 6.567919 |                     5.687577 |             6.364313 | -2.425929 |  -8.716337 |                       0.0 |                 4.184493 |        -4.184493 | 0.237008 | 0.059252 |  0.012607 |      0.0 | -0.007232 |  0.241129 |    -0.973206 |     0.742995 |      0.0 |      0.0 | 0.086864 | 0.086864 |  -5.358718 |  -8.877714 |  -10.051939 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.187393 | 14.001712 | 14.092651 |        0.0 | 0.993733 | -184.844379 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 19:00:00 |  0.2 |  1.4 |             98.3 |       3.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  1.4 |    3.046047 |           1.348964 |                 6.760362 |                     5.944048 |             6.645436 | -2.218133 | -11.076856 |                       0.0 |                 4.822385 |        -4.822385 | 0.175393 | 0.026309 |  0.004103 |      0.0 | -0.006928 |  0.264123 |     -0.37199 |     0.742623 |      0.0 |      0.0 | 0.033773 | 0.033773 |  -5.133236 |  -7.368539 |   -7.679391 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.195897 | 14.068621 | 14.274972 |        0.0 | 0.998434 | -165.167523 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 20:00:00 |  0.4 |  1.6 |             98.7 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.376 |  1.6 |     4.56907 |           2.023445 |                 6.858436 |                     6.072976 |             6.769276 |  -1.98201 | -11.421868 |                       0.0 |                 5.135542 |        -5.135542 | 0.371897 |  0.03719 |  0.005169 |      0.0 | -0.008662 |  0.657369 |    -0.078366 |     0.742339 |      0.0 |      0.0 | 0.007449 | 0.007449 |  -6.418516 |  -8.790461 |  -10.073436 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.194831 | 14.121607 | 14.655532 |        0.0 | 1.002047 | -145.145655 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 21:00:00 |  0.7 |  2.0 |             99.6 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.658 |  2.0 |    4.488911 |           1.987946 |                 7.058357 |                       6.1078 |             7.030123 | -1.741748 | -10.450487 |                       0.0 |                 3.399829 |        -3.399829 | 0.652831 |      0.0 | -0.001152 |      0.0 |  -0.01135 |  1.518485 |          0.0 |     0.741764 | 0.069407 | 0.069407 |      0.0 |      0.0 |  -8.409795 | -10.361428 |  -15.371395 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.201152 | 14.062619 | 15.319713 |        0.0 | 1.009393 | -126.095168 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 22:00:00 |  0.5 |  2.4 |             99.4 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  2.4 |    4.649229 |           2.058944 |                 7.263399 |                       6.1078 |             7.219819 | -1.513142 |  -9.078852 |                       0.0 |                 1.378924 |        -1.378924 | 0.471152 |      0.0 |  0.000424 |      0.0 | -0.013984 |  1.315078 |          0.0 |     0.741193 | 0.150094 | 0.150094 |      0.0 |      0.0 | -10.361775 | -12.706346 |  -21.689197 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.199576 | 13.924845 | 15.804849 |        0.0 | 1.016733 | -108.416315 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 23:00:00 |  1.0 |  2.5 |             99.2 |       6.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.94 |  2.5 |    4.969866 |           2.200941 |                 7.315473 |                       6.1078 |             7.256949 | -1.300996 |  -7.805975 |                       0.0 |                 0.884234 |        -0.884234 | 0.939576 |      0.0 |  0.002314 |      0.0 | -0.015071 |  2.731818 |          0.0 |     0.740625 | 0.204927 | 0.204927 |      0.0 |      0.0 | -11.167257 | -13.803762 |  -24.086786 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.197686 | 13.732268 | 16.759496 |        0.0 | 1.024099 |  -92.010341 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 00:00:00 |  2.0 |  2.7 |             99.2 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  1.88 |  2.7 |    4.488911 |           1.987946 |                 7.420609 |                       6.1078 |             7.361244 | -1.104124 |  -6.624745 |                       0.0 |                -0.139391 |         0.139391 | 1.877686 |      0.0 |  0.001802 |      0.0 | -0.015425 |  5.896122 |          0.0 |     0.740057 | 0.267604 | 0.267604 |      0.0 |      0.0 | -11.428977 | -13.987928 |  -25.556296 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 | 2.198198 | 13.475808 | 18.652607 |        0.0 | 1.031513 |  -76.785596 | 71.324979 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 01:00:00 |  2.1 |  3.2 |             99.6 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.974 |  3.2 |    4.168274 |            1.84595 |                 7.689299 |                       6.1078 |             7.658542 | -0.921427 |  -5.528563 |                       0.0 |                -2.742905 |         2.742905 | 1.972198 |      0.0 | -0.001192 |      0.0 | -0.018246 |  2.299196 |          0.0 |     0.739498 |  0.31133 |  0.31133 |      0.0 |      0.0 | -13.519673 | -15.851265 |  -32.113842 |      1.0 |      0.3 | 1.800023 | 0.615028 |  0.0 |  0.0 | 0.0 |  0.0 | 0.615028 |      0.5 | 0.115028 |   0.0 |   0.0 |  0.0 | 0.006486 | 0.005816 |   0.0 |   0.0 |  0.0 | 0.012302 | 0.003417 | 2.201192 | 13.176943 | 18.843028 |        0.0 | 1.038859 |  -62.657033 | 72.509973 | 0.493514 | 0.109212 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003417 |
    | 2010-12-11 02:00:00 |  1.8 |  3.9 |             99.8 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.692 |  3.9 |     4.56907 |           2.023445 |                  8.07986 |                       6.1078 |               8.0637 | -0.751884 |  -4.511306 |                       0.0 |                -6.386972 |         6.386972 | 1.693192 |      0.0 | -0.002401 |      0.0 | -0.024333 |       0.0 |          0.0 |     0.738946 | 0.434711 | 0.434711 |      0.0 |      0.0 | -18.029533 | -20.426299 |  -44.842804 |      1.0 |      0.3 | 2.315631 | 0.793324 |  0.0 |  0.0 | 0.0 |  0.0 | 0.793324 |      0.5 | 0.293324 |   0.0 |   0.0 |  0.0 | 0.019234 | 0.025685 |   0.0 |   0.0 |  0.0 | 0.044919 | 0.012478 | 2.202401 | 12.758686 | 18.244922 |        0.0 | 1.046167 |  -49.545727 |  74.03228 |  0.97428 | 0.376851 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012478 |
    | 2010-12-11 03:00:00 |  1.3 |  4.4 |             99.8 |       6.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.222 |  4.4 |    5.290502 |           2.342937 |                  8.36943 |                       6.1078 |             8.352691 | -0.594549 |  -3.567292 |                       0.0 |                -8.998234 |         8.998234 | 1.224401 |      0.0 | -0.002123 |      0.0 | -0.030654 |       0.0 |          0.0 |     0.738399 | 0.575982 | 0.575982 |      0.0 |      0.0 | -22.713156 | -25.294274 |  -57.005664 |      1.0 |      0.3 |  2.04944 | 0.703976 |  0.0 |  0.0 | 0.0 |  0.0 | 0.703976 |      0.5 | 0.203976 |   0.0 |   0.0 |  0.0 | 0.031652 | 0.047768 |   0.0 |   0.0 |  0.0 |  0.07942 | 0.022061 | 2.202123 | 12.203172 | 17.450537 |        0.0 | 1.053475 |  -37.378435 | 75.377744 | 1.442628 | 0.533059 |  0.0 |  0.0 | 0.0 |   0.0 | 0.022061 |
    | 2010-12-11 04:00:00 |  0.5 |  4.8 |             99.0 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  4.8 |    5.611139 |           2.484933 |                 8.607627 |                       6.1078 |             8.521551 | -0.448541 |  -2.691247 |                       0.0 |               -11.031041 |        11.031041 | 0.472123 |      0.0 |  0.005749 |      0.0 | -0.034262 |       0.0 |          0.0 |     0.737855 | 0.672692 | 0.672692 |      0.0 |      0.0 | -25.386793 | -28.684284 |  -65.102118 |      1.0 |      0.3 | 1.435961 | 0.494232 |  0.0 |  0.0 | 0.0 |  0.0 | 0.494232 | 0.494232 |      0.0 |   0.0 |   0.0 |  0.0 | 0.043675 |  0.05298 |   0.0 |   0.0 |  0.0 | 0.096655 | 0.026849 | 2.194251 | 11.553119 | 16.520961 |        0.0 | 1.060783 |  -26.087187 | 76.319473 | 1.893185 | 0.480079 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026849 |
    | 2010-12-11 05:00:00 |  0.2 |  4.8 |             97.6 |       7.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  4.8 |    5.851616 |            2.59143 |                 8.607627 |                       6.1078 |             8.401044 | -0.313046 |  -1.878277 |                       0.0 |               -10.910371 |        10.910371 | 0.182251 |      0.0 |  0.020265 |      0.0 |  -0.03348 |       0.0 |          0.0 |     0.737314 | 0.682721 | 0.682721 |      0.0 |      0.0 | -24.807089 | -29.502182 |  -65.219643 |      1.0 |      0.3 | 1.160521 | 0.400044 |  0.0 |  0.0 | 0.0 |  0.0 | 0.400044 | 0.400044 |      0.0 |   0.0 |   0.0 |  0.0 | 0.054092 | 0.047714 |   0.0 |   0.0 |  0.0 | 0.101806 |  0.02828 | 2.179735 | 10.892427 |  15.57617 |        0.0 | 1.068091 |   -15.60891 |  77.07995 | 2.239136 | 0.432365 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02828 |
    | 2010-12-11 06:00:00 |  0.5 |  5.1 |             97.4 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  5.1 |    5.771457 |           2.555931 |                 8.790172 |                       6.1078 |             8.561627 | -0.187307 |  -1.123842 |                       0.0 |               -12.474442 |        12.474442 | 0.449735 |      0.0 |  0.022285 |      0.0 | -0.035493 |       0.0 |          0.0 |     0.736778 | 0.740543 | 0.740543 |      0.0 |      0.0 | -26.298897 | -31.056396 |  -69.829736 |      1.0 |      0.3 | 1.511124 | 0.521811 |  0.0 |  0.0 | 0.0 |  0.0 | 0.521811 |      0.5 | 0.021811 |   0.0 |   0.0 |  0.0 | 0.064325 | 0.044075 |   0.0 |   0.0 |  0.0 |   0.1084 | 0.030111 | 2.177715 | 10.175017 | 14.550274 |        0.0 | 1.075399 |   -5.885068 | 78.069263 | 2.674812 | 0.410101 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030111 |
    | 2010-12-11 07:00:00 |  0.2 |  5.1 |             96.3 |       7.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  5.1 |    6.092093 |           2.697927 |                 8.790172 |                       6.1078 |             8.464935 | -0.070621 |  -0.423725 |                       0.0 |               -12.377725 |        12.377725 | 0.165715 |      0.0 |  0.035391 |      0.0 | -0.035367 |       0.0 |          0.0 |     0.736245 | 0.758524 | 0.758524 |      0.0 |      0.0 | -26.205129 | -32.215085 |  -70.797939 | 0.999577 | 0.299746 | 1.253041 | 0.433106 |  0.0 |  0.0 | 0.0 |  0.0 | 0.433106 | 0.433106 |      0.0 |   0.0 |   0.0 |  0.0 | 0.074711 | 0.040759 |   0.0 |   0.0 |  0.0 |  0.11547 | 0.032075 | 2.164609 |  9.439381 | 13.498315 |        0.0 | 1.082707 |    3.138657 | 78.889198 | 3.033207 | 0.369342 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032075 |
    | 2010-12-11 08:00:00 |  0.1 |  5.4 |             95.2 |       7.3 |                 0.347041 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        0.900734 | 0.094 |  5.4 |    5.851616 |            2.59143 |                 8.976116 |                       6.1078 |             8.545263 |       0.0 |        0.0 |                  0.044209 |               -13.865856 |        13.910065 | 0.058609 |      0.0 |  0.045778 |      0.0 | -0.035585 |       0.0 |          0.0 |     0.735717 | 0.791862 | 0.791862 |      0.0 |      0.0 |  -26.36717 | -33.189955 |  -73.467189 | 0.998418 | 0.299052 | 1.193957 | 0.412531 |  0.0 |  0.0 | 0.0 |  0.0 | 0.412531 | 0.412531 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083702 | 0.036708 |   0.0 |   0.0 |  0.0 |  0.12041 | 0.033447 | 2.154222 |  8.670316 | 12.398552 |        0.0 | 1.090015 |   11.738657 | 79.670624 | 3.362036 | 0.332633 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033447 |
    | 2010-12-11 09:00:00 |  0.3 |  5.3 |             96.3 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        17.26751 | 0.282 |  5.3 |    5.691298 |           2.520432 |                 8.913754 |                       6.1078 |             8.583945 |       0.0 |        0.0 |                  0.849194 |               -13.434609 |        14.283803 | 0.236222 |      0.0 |   0.03653 |      0.0 | -0.035482 |       0.0 |          0.0 |     0.735191 |  0.78195 |  0.78195 |      0.0 |      0.0 | -26.290556 | -31.973262 |  -72.547621 |  0.99726 | 0.298358 | 1.357611 | 0.468964 |  0.0 |  0.0 | 0.0 |  0.0 | 0.468964 | 0.468964 |      0.0 |   0.0 |   0.0 |  0.0 | 0.092928 |  0.03306 |   0.0 |   0.0 |  0.0 | 0.125988 | 0.034997 |  2.16347 |   7.91094 | 11.312645 |        0.0 | 1.097323 |   20.338657 | 80.559271 | 3.738073 | 0.299574 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034997 |
    | 2010-12-11 10:00:00 |  0.1 |  5.5 |             95.5 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        52.61761 | 0.094 |  5.5 |    5.691298 |           2.520432 |                 9.038862 |                       6.1078 |             8.632113 |       0.0 |        0.0 |                  2.592763 |               -14.422901 |        17.015665 |  0.05747 |      0.0 |  0.051465 |      0.0 | -0.036172 |       0.0 |          0.0 |      0.73467 | 0.829913 | 0.829913 |      0.0 |      0.0 |  -26.80199 | -33.179801 |  -76.997455 | 0.996101 | 0.297665 |  1.24804 |    0.431 |  0.0 |  0.0 | 0.0 |  0.0 |    0.431 |    0.431 |      0.0 |   0.0 |   0.0 |  0.0 | 0.102148 | 0.029774 |   0.0 |   0.0 |  0.0 | 0.131923 | 0.036645 | 2.148535 |  7.103669 | 10.158247 |        0.0 | 1.104631 |   28.938657 | 81.376311 | 4.066924 | 0.269799 |  0.0 |  0.0 | 0.0 |   0.0 | 0.036645 |
    | 2010-12-11 11:00:00 |  0.1 |  6.1 |             91.6 |       8.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.135633 | 0.094 |  6.1 |    7.134162 |           3.159415 |                 9.423511 |                       6.1078 |             8.631936 |       0.0 |        0.0 |                  3.956437 |               -17.550317 |        21.506754 | 0.042535 |      0.0 |  0.119414 |      0.0 | -0.042299 |       0.0 |          0.0 |     0.734152 | 1.033487 | 1.033487 |      0.0 |      0.0 | -31.341975 | -43.035887 |  -95.884616 | 0.994942 | 0.296973 | 1.526575 | 0.527165 |  0.0 |  0.0 | 0.0 |  0.0 | 0.527165 |      0.5 | 0.027165 |   0.0 |   0.0 |  0.0 | 0.111538 | 0.028188 |   0.0 |   0.0 |  0.0 | 0.139726 | 0.038813 | 2.080586 |  6.095459 |  8.716506 |        0.0 | 1.111939 |   37.538657 | 82.375722 | 4.455386 | 0.268775 |  0.0 |  0.0 | 0.0 |   0.0 | 0.038813 |
    | 2010-12-11 12:00:00 |  0.0 |  6.4 |             90.0 |       9.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.276118 |   0.0 |  6.4 |    7.615117 |           3.372409 |                 9.621181 |                       6.1078 |             8.659063 |       0.0 |        0.0 |                  3.971041 |               -19.002428 |        22.973469 |      0.0 |      0.0 |  0.149935 |      0.0 | -0.044819 |       0.0 |          0.0 |     0.733638 | 1.115741 | 1.115741 |      0.0 |      0.0 |  -33.20903 | -47.333467 | -103.515966 | 0.993784 | 0.296282 | 1.603713 | 0.553878 |  0.0 |  0.0 | 0.0 |  0.0 | 0.553878 |      0.5 | 0.053878 |   0.0 |   0.0 |  0.0 | 0.121572 | 0.029437 |   0.0 |   0.0 |  0.0 |  0.15101 | 0.041947 | 1.930651 |  5.005323 |  7.157612 |        0.0 | 1.119247 |   46.138657 | 83.425557 | 4.833813 | 0.293216 |  0.0 |  0.0 | 0.0 |   0.0 | 0.041947 |
    | 2010-12-11 13:00:00 |  0.1 |  6.4 |             85.7 |       7.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       75.835403 | 0.094 |  6.4 |    6.332571 |           2.804424 |                 9.621181 |                       6.1078 |             8.245352 |       0.0 |        0.0 |                  3.758565 |               -18.576375 |         22.33494 |      0.0 |      0.0 |  0.175679 |      0.0 | -0.032937 |       0.0 |          0.0 |     0.733127 | 0.951276 | 0.951276 |      0.0 |      0.0 | -24.404983 | -41.517304 |  -88.257227 | 0.992625 | 0.295591 | 1.366584 | 0.471962 |  0.0 |  0.0 | 0.0 |  0.0 | 0.471962 | 0.471962 |      0.0 |   0.0 |   0.0 |  0.0 | 0.130984 | 0.029142 |   0.0 |   0.0 |  0.0 | 0.160126 | 0.044479 | 1.848972 |  4.072703 |  5.823965 |        0.0 | 1.126555 |   54.738657 | 84.320179 | 5.174792 | 0.264074 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044479 |
    | 2010-12-11 14:00:00 |  0.0 |  6.5 |             82.3 |       9.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       42.227911 |   0.0 |  6.5 |    7.695276 |           3.407908 |                 9.687876 |                       6.1078 |             7.973122 |       0.0 |        0.0 |                  2.096884 |               -18.760881 |        20.857765 |      0.0 |      0.0 |  0.251837 |      0.0 | -0.033021 |       0.0 |          0.0 |      0.73262 |  1.01066 |  1.01066 |      0.0 |      0.0 | -24.466815 | -48.442242 |  -93.766822 | 0.991466 | 0.294902 | 1.453439 | 0.501941 |  0.0 |  0.0 | 0.0 |  0.0 | 0.501941 |      0.5 | 0.001941 |   0.0 |   0.0 |  0.0 | 0.140155 | 0.026344 |   0.0 |   0.0 |  0.0 | 0.166499 |  0.04625 | 1.597135 |  3.079403 |  4.403547 |        0.0 | 1.133863 |   63.338657 | 85.271676 | 5.534637 | 0.239671 |  0.0 |  0.0 | 0.0 |   0.0 |  0.04625 |
    | 2010-12-11 15:00:00 |  0.1 |  6.3 |             82.5 |       9.1 |                      1.0 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.40894 |       12.613921 | 0.094 |  6.3 |     7.29448 |           3.230413 |                 9.554889 |                       6.1078 |             7.882784 |       0.0 |        0.0 |                  0.627542 |               -17.422092 |        18.049634 |      0.0 |      0.0 |  0.228998 |      0.0 | -0.030224 |       0.0 |          0.0 |     0.732116 | 0.922709 | 0.922709 |      0.0 |      0.0 |  -22.39469 |  -45.16256 |  -85.606884 | 0.990308 | 0.294213 |  1.32853 | 0.458757 |  0.0 |  0.0 | 0.0 |  0.0 | 0.458757 | 0.458757 |      0.0 |   0.0 |   0.0 |  0.0 | 0.148915 | 0.023821 |   0.0 |   0.0 |  0.0 | 0.172736 | 0.047982 | 1.462137 |  2.171497 |  3.105241 |        0.0 | 1.141171 |   71.938657 |  86.14145 | 5.844478 |  0.21585 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047982 |
    | 2010-12-11 16:00:00 |  0.0 |  5.9 |             82.2 |       8.7 |                 0.053708 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |        0.021912 |   0.0 |  5.9 |    6.973844 |           3.088417 |                 9.293725 |                       6.1078 |             7.639442 |       0.0 |        0.0 |                  0.001092 |               -15.264686 |        15.265778 |      0.0 |      0.0 |  0.217297 |      0.0 | -0.025254 |       0.0 |          0.0 |     0.731615 | 0.807655 | 0.807655 |      0.0 |      0.0 | -18.712038 | -40.954652 |  -74.932468 | 0.989149 | 0.293525 |  1.16434 | 0.401944 |  0.0 |  0.0 | 0.0 |  0.0 | 0.401944 | 0.401944 |      0.0 |   0.0 |   0.0 |  0.0 | 0.156182 | 0.021453 |   0.0 |   0.0 |  0.0 | 0.177635 | 0.049343 |  1.24484 |  1.374933 |  1.966155 |        0.0 | 1.148479 |   80.538657 | 86.903846 |  6.09024 | 0.194397 |  0.0 |  0.0 | 0.0 |   0.0 | 0.049343 |
    | 2010-12-11 17:00:00 |  0.0 |  5.5 |             76.5 |       8.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.5 |    7.134162 |           3.159415 |                 9.038862 |                       6.1078 |              6.91473 |       0.0 |        0.0 |                       0.0 |               -12.549167 |        12.549167 |      0.0 |      0.0 |  0.290187 |      0.0 | -0.013522 |       0.0 |          0.0 |     0.731118 |  0.66149 |  0.66149 |      0.0 |      0.0 | -10.019574 | -38.802849 |   -61.37159 |  0.98799 | 0.292837 | 0.952437 | 0.328616 |  0.0 |  0.0 | 0.0 |  0.0 | 0.328616 | 0.328616 |      0.0 |   0.0 |   0.0 |  0.0 | 0.161579 | 0.019321 |   0.0 |   0.0 |  0.0 |   0.1809 |  0.05025 | 0.954653 |   0.71835 |   1.02724 |        0.0 | 1.155787 |   89.138657 | 87.527667 | 6.257277 | 0.175076 |  0.0 |  0.0 | 0.0 |   0.0 |  0.05025 |
    | 2010-12-11 18:00:00 |  0.0 |  5.1 |             81.7 |       7.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.1 |    6.332571 |           2.804424 |                 8.790172 |                       6.1078 |              7.18157 |       0.0 |        0.0 |                       0.0 |               -10.994493 |        10.994493 |      0.0 |      0.0 |   0.19716 |      0.0 | -0.016545 |       0.0 |          0.0 |     0.730625 | 0.607237 | 0.607237 |      0.0 |      0.0 | -12.259513 | -33.084102 |  -56.338108 | 0.986832 | 0.292151 | 0.882335 | 0.304225 |  0.0 |  0.0 | 0.0 |  0.0 | 0.304225 | 0.304225 |      0.0 |   0.0 |   0.0 |  0.0 | 0.165577 | 0.017401 |   0.0 |   0.0 |  0.0 | 0.182978 | 0.050827 | 0.757493 |  0.112903 |  0.161451 |        0.0 | 1.163095 |   97.738657 | 88.105777 | 6.395924 | 0.157676 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050827 |
    | 2010-12-11 19:00:00 |  0.0 |  4.8 |             83.4 |       7.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.8 |    5.691298 |           2.520432 |                 8.607627 |                       6.1078 |             7.178761 |       0.0 |        0.0 |                       0.0 |                -9.593806 |         9.593806 |      0.0 |      0.0 |  0.158751 |      0.0 | -0.015346 |       0.0 |          0.0 |     0.730134 | 0.538078 | 0.112903 |      0.0 |      0.0 | -11.370965 | -28.956917 |  -49.921688 | 0.985673 | 0.291465 | 0.176797 | 0.060877 |  0.0 |  0.0 | 0.0 |  0.0 | 0.060877 | 0.060877 |      0.0 |   0.0 |   0.0 |  0.0 | 0.166002 | 0.015671 |   0.0 |   0.0 |  0.0 | 0.181673 | 0.050465 | 0.598742 |       0.0 |       0.0 |        0.0 | 1.170403 |  106.338657 | 88.221697 | 6.290799 | 0.142005 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050465 |
    | 2010-12-11 20:00:00 |  0.0 |  4.9 |             82.8 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.130184 |           2.271939 |                 8.668101 |                          0.0 |             7.177187 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.149006 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.980553 | 0.288445 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.162497 | 0.014114 |   0.0 |   0.0 |  0.0 |  0.17661 | 0.049058 | 0.449736 |       0.0 |       0.0 |        0.0 |      nan |  144.338657 | 88.221697 | 6.128303 | 0.127891 |  0.0 |  0.0 | 0.0 |   0.0 | 0.049058 |
    | 2010-12-11 21:00:00 |  0.0 |  4.9 |             85.8 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.771457 |           2.555931 |                 8.668101 |                          0.0 |              7.43723 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.137783 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.975433 | 0.285441 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.158299 | 0.012711 |   0.0 |   0.0 |  0.0 |  0.17101 | 0.047503 | 0.311953 |       0.0 |       0.0 |        0.0 |      nan |  182.338657 | 88.221697 | 5.970003 |  0.11518 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047503 |
    | 2010-12-11 22:00:00 |  0.1 |  4.7 |             87.6 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  4.7 |    5.611139 |           2.484933 |                 8.547525 |                          0.0 |             7.487632 |       0.0 |      -28.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.115717 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.970475 | 0.282547 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.15421 | 0.011448 |   0.0 |   0.0 |  0.0 | 0.165658 | 0.046016 | 0.290236 |       0.0 |       0.0 |        0.0 |      nan |  219.138657 | 88.221697 | 5.815793 | 0.103732 |  0.0 |  0.0 | 0.0 |   0.0 | 0.046016 |
    | 2010-12-11 23:00:00 |  0.1 |  3.8 |             85.6 |       6.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.8 |     5.45082 |           2.413935 |                 8.023017 |                          0.0 |             6.867703 |       0.0 |      -22.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.126391 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.966245 | 0.280089 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.150227 |  0.01031 |   0.0 |   0.0 |  0.0 | 0.160537 | 0.044594 | 0.257845 |       0.0 |       0.0 |        0.0 |      nan |  250.538657 | 88.221697 | 5.665566 | 0.093423 |  0.0 |  0.0 | 0.0 |   0.0 | 0.044594 |
    | 2010-12-12 00:00:00 |  0.0 |  3.9 |             85.3 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.9 |    5.130184 |           2.271939 |                  8.07986 |                          0.0 |             6.892121 |       0.0 |      -23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.121951 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.961933 | 0.277595 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.146346 | 0.009285 |   0.0 |   0.0 |  0.0 | 0.155632 | 0.043231 | 0.135894 |       0.0 |       0.0 |        0.0 |      nan |  282.538657 | 88.221697 |  5.51922 | 0.084138 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043231 |
    | 2010-12-12 01:00:00 |  0.0 |  3.7 |             86.2 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.7 |    5.050025 |            2.23644 |                 7.966527 |                          0.0 |             6.867146 |       0.0 |      -22.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.111611 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.957784 | 0.275205 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.142566 | 0.008362 |   0.0 |   0.0 |  0.0 | 0.150929 | 0.041925 | 0.024283 |       0.0 |       0.0 |        0.0 |      nan |  313.338657 | 88.221697 | 5.376653 | 0.075775 |  0.0 |  0.0 | 0.0 |   0.0 | 0.041925 |
    | 2010-12-12 02:00:00 |  0.1 |  3.5 |             89.3 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.5 |     4.56907 |           2.023445 |                 7.854597 |                          0.0 |             7.014155 |       0.0 |      -21.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.076995 |      0.0 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.953796 | 0.272918 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.138884 | 0.007531 |   0.0 |   0.0 |  0.0 | 0.146415 | 0.040671 | 0.041288 |       0.0 |       0.0 |        0.0 |      nan |  342.938657 | 88.221697 |  5.23777 | 0.068244 |  0.0 |  0.0 | 0.0 |   0.0 | 0.040671 |
    | 2010-12-12 03:00:00 |  0.0 |  3.3 |             91.4 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |    4.168274 |            1.84595 |                 7.744054 |                          0.0 |             7.078066 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |  0.041288 | 0.004487 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.949969 | 0.270733 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.135296 | 0.006783 |   0.0 |   0.0 |  0.0 | 0.142079 | 0.039466 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  371.338657 | 88.217211 | 5.102474 | 0.061461 |  0.0 |  0.0 | 0.0 |   0.0 | 0.039466 |
    | 2010-12-12 04:00:00 |  0.0 |  3.3 |             90.9 |       4.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |     3.68732 |           1.632956 |                 7.744054 |                          0.0 |             7.039346 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.017853 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.946143 | 0.268556 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.131801 | 0.006109 |   0.0 |   0.0 |  0.0 |  0.13791 | 0.038308 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  399.738657 | 88.199357 | 4.970672 | 0.055353 |  0.0 |  0.0 | 0.0 |   0.0 | 0.038308 |
    | 2010-12-12 05:00:00 |  0.0 |  3.0 |             89.2 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.0 |    3.927797 |           1.739453 |                 7.580812 |                          0.0 |             6.762084 |       0.0 |      -18.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.021509 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.942559 | 0.266525 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.128397 | 0.005501 |   0.0 |   0.0 |  0.0 | 0.133898 | 0.037194 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  426.338657 | 88.177848 | 4.842276 | 0.049851 |  0.0 |  0.0 | 0.0 |   0.0 | 0.037194 |
    | 2010-12-12 06:00:00 |  0.0 |  1.9 |             85.6 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.9 |    5.050025 |            2.23644 |                 7.007901 |                          0.0 |             5.998763 |       0.0 |      -11.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.029436 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.939865 | 0.265004 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.12508 | 0.004955 |   0.0 |   0.0 |  0.0 | 0.130035 | 0.036121 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  446.338657 | 88.148412 | 4.717195 | 0.044897 |  0.0 |  0.0 | 0.0 |   0.0 | 0.036121 |
    | 2010-12-12 07:00:00 |  0.0 |  1.2 |             86.3 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.2 |    4.649229 |           2.058944 |                 6.663527 |                          0.0 |             5.750624 |       0.0 |       -7.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 |  0.02618 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.937736 | 0.263805 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.121849 | 0.004462 |   0.0 |   0.0 |  0.0 | 0.126311 | 0.035087 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  462.138657 | 88.122232 | 4.595346 | 0.040434 |  0.0 |  0.0 | 0.0 |   0.0 | 0.035087 |
    | 2010-12-12 08:00:00 |  0.0 |  0.8 |             91.2 |       6.1 |                 0.339738 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        0.894145 |   0.0 |  0.8 |    4.889707 |           2.165441 |                 6.473522 |                          0.0 |             5.903852 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.012681 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |  0.93593 |  0.26279 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.118702 | 0.004019 |   0.0 |   0.0 |  0.0 |  0.12272 | 0.034089 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  475.538657 | 88.109551 | 4.476644 | 0.036416 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034089 |
    | 2010-12-12 09:00:00 |  0.0 |  0.8 |             87.5 |       5.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        17.71254 |   0.0 |  0.8 |    4.729388 |           2.094443 |                 6.473522 |                          0.0 |             5.664332 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.004616 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.934125 | 0.261777 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.115636 | 0.003619 |   0.0 |   0.0 |  0.0 | 0.119255 | 0.033126 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  488.938657 | 88.104935 | 4.361009 | 0.032796 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033126 |
    | 2010-12-12 10:00:00 |  0.1 |  0.5 |             84.5 |       6.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       54.302585 | 0.094 |  0.5 |    5.130184 |           2.271939 |                 6.334173 |                          0.0 |             5.352376 |       0.0 |       -3.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |     0.094 | 0.002006 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.932562 | 0.260902 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.112649 |  0.00326 |   0.0 |   0.0 |  0.0 | 0.115908 | 0.032197 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  500.538657 | 88.102928 |  4.24836 | 0.029537 |  0.0 |  0.0 | 0.0 |   0.0 | 0.032197 |
    | 2010-12-12 11:00:00 |  0.0 |  0.3 |             76.7 |       6.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.504146 |   0.0 |  0.3 |     5.45082 |           2.413935 |                  6.24275 |                          0.0 |             4.788189 |       0.0 |       -1.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.008524 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.931161 | 0.260118 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.109739 | 0.002936 |   0.0 |   0.0 |  0.0 | 0.112674 | 0.031298 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  510.938657 | 88.094404 | 4.138621 | 0.026601 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031298 |
    | 2010-12-12 12:00:00 |  0.0 | -0.1 |             74.8 |       7.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.665294 |   0.0 | -0.1 |    5.611139 |           2.484933 |                 6.063393 |                          0.0 |             4.535418 |       0.0 |        0.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.008234 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.930083 | 0.259516 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.106904 | 0.002644 |   0.0 |   0.0 |  0.0 | 0.109548 |  0.03043 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  518.938657 |  88.08617 | 4.031717 | 0.023957 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03043 |
    | 2010-12-12 13:00:00 |  0.0 | -0.4 |             69.7 |       6.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       77.078959 |   0.0 | -0.4 |    5.050025 |            2.23644 |                 5.931876 |                          0.0 |             4.134518 |       0.0 |        2.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 |  0.00886 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.929248 |  0.25905 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.104143 | 0.002381 |   0.0 |   0.0 |  0.0 | 0.106524 |  0.02959 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  525.138657 |  88.07731 | 3.927575 | 0.021576 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02959 |
    | 2010-12-12 14:00:00 |  0.0 | -0.6 |             66.9 |       7.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.3 |                      7.393446 |       60.254042 |   0.0 | -0.6 |    5.851616 |            2.59143 |                 5.845604 |                          0.0 |             3.910709 |       0.0 |        3.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.008652 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.928574 | 0.258675 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.101453 | 0.002144 |   0.0 |   0.0 |  0.0 | 0.103597 | 0.028777 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  530.138657 | 88.068657 | 3.826122 | 0.019432 |  0.0 |  0.0 | 0.0 |   0.0 | 0.028777 |
    | 2010-12-12 15:00:00 |  0.0 | -0.8 |             64.4 |       7.5 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.4 |                      7.393446 |       14.837301 |   0.0 | -0.8 |    6.011934 |           2.662428 |                 5.760442 |                          0.0 |             3.709725 |       0.0 |        4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.008092 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.928062 |  0.25839 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.098832 | 0.001931 |   0.0 |   0.0 |  0.0 | 0.100763 |  0.02799 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  533.938657 | 88.060566 |  3.72729 | 0.017501 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02799 |
    | 2010-12-12 16:00:00 |  0.0 | -1.3 |             65.4 |       7.0 |                 0.046405 |              0.2 | 0.0 | 0.0 |                   0.6 |                      7.386143 |        0.168856 |   0.0 | -1.3 |    5.611139 |           2.484933 |                 5.552313 |                          0.0 |             3.631213 |       0.0 |        7.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.058495 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.927954 |  0.25833 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.096279 | 0.001739 |   0.0 |   0.0 |  0.0 | 0.098018 | 0.027227 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  534.738657 | 88.002071 | 3.631011 | 0.015761 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027227 |
    | 2010-12-12 17:00:00 |  0.0 | -1.6 |             67.6 |       6.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -1.6 |    4.809547 |           2.129942 |                 5.430651 |                          0.0 |              3.67112 |       0.0 |        9.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.053186 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.928089 | 0.258405 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.093792 | 0.001566 |   0.0 |   0.0 |  0.0 | 0.095359 | 0.026489 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  533.738657 | 87.948885 | 3.537219 | 0.014195 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026489 |
    | 2010-12-12 18:00:00 |  0.0 | -2.0 |             70.2 |       4.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.0 |    3.847638 |           1.703954 |                 5.272101 |                          0.0 |             3.701015 |       0.0 |       12.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.044432 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.928547 |  0.25866 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.091369 | 0.001411 |   0.0 |   0.0 |  0.0 |  0.09278 | 0.025772 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  530.338657 | 87.904453 | 3.445849 | 0.012784 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025772 |
    | 2010-12-12 19:00:00 |  0.0 | -2.1 |             70.4 |       4.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.1 |    3.366683 |            1.49096 |                 5.233109 |                          0.0 |             3.684109 |       0.0 |       12.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.041868 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.929086 |  0.25896 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.089009 | 0.001271 |   0.0 |   0.0 |  0.0 |  0.09028 | 0.025078 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  526.338657 | 87.862585 |  3.35684 | 0.011513 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025078 |
    | 2010-12-12 20:00:00 |  0.0 | -2.2 |             69.5 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.2 |    3.927797 |           1.739453 |                 5.194372 |                          0.0 |             3.610088 |       0.0 |       13.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 |  0.04524 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.929706 | 0.259306 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.08671 | 0.001144 |   0.0 |   0.0 |  0.0 | 0.087854 | 0.024404 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  521.738657 | 87.817346 |  3.27013 | 0.010369 |  0.0 |  0.0 | 0.0 |   0.0 | 0.024404 |
    | 2010-12-12 21:00:00 |  0.0 | -2.6 |             71.6 |       4.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.6 |    3.206365 |           1.419962 |                 5.041946 |                          0.0 |             3.610033 |       0.0 |       15.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.038136 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.930649 | 0.259832 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.08447 | 0.001031 |   0.0 |   0.0 |  0.0 | 0.085501 |  0.02375 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  514.738657 |  87.77921 |  3.18566 | 0.009338 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02375 |
    | 2010-12-12 22:00:00 |  0.0 | -3.1 |             74.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.1 |     2.72541 |           1.206967 |                 4.856984 |                          0.0 |             3.618453 |       0.0 |       18.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.030876 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.931996 | 0.260585 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.082288 | 0.000928 |   0.0 |   0.0 |  0.0 | 0.083216 | 0.023116 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  504.738657 | 87.748333 | 3.103371 |  0.00841 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023116 |
    | 2010-12-12 23:00:00 |  0.0 | -3.9 |             80.2 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.9 |     2.72541 |           1.206967 |                  4.57351 |                          0.0 |             3.667955 |       0.0 |       23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |      0.0 |       0.0 | 0.022342 |       0.0 |       0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |  0.93399 | 0.261701 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.080163 | 0.000836 |   0.0 |   0.0 |  0.0 | 0.080999 |   0.0225 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  489.938657 | 87.725991 | 3.023209 | 0.007574 |  0.0 |  0.0 | 0.0 |   0.0 |   0.0225 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_nadelw_winter))
0.0

.. _lland_knauf_acker_winter_ktschnee_hourly:

acre (snow surface temperature)
-------------------------------

Due to the reasons discussed in the daily integration test
:ref:`lland_knauf_acker_winter_ktschnee_daily`, enabling the iterative adjustment of
the snow surface temperature results in a more delayed and smoothed snow melt, which
becomes even more apparent in the following hourly simulation results:

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(5.0)
    >>> test("lland_knauf_acker_winter_ktschnee_hourly",
    ...      axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...      use_conditions=conditions_acker_winter_ktschnee)
    |                date | nied | teml | relativehumidity | windspeed | possiblesunshineduration | sunshineduration |  qz | qzh | dailysunshineduration | dailypossiblesunshineduration | globalradiation |  nkor | tkor | windspeed2m | reducedwindspeed2m | saturationvapourpressure | saturationvapourpressuresnow | actualvapourpressure |        tz |         wg | netshortwaveradiationsnow | netlongwaveradiationsnow | netradiationsnow |     nbes |    sbes |      evi |      evb |       evs |    wnied | tempssurface | actualalbedo |  schmpot |     schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |    qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga |      qah |       qa |     inzp |      wats |      waes |      esnow |     taus |        ebdn |      bowa |     sdg1 |     sdg2 | sig1 | sig2 | sbg | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-12-10 00:00:00 |  0.0 | -3.7 |             87.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -3.7 |     2.72541 |            2.72541 |                  4.64297 |                     3.363254 |             4.062599 | -2.463502 |  -1.105542 |                       0.0 |                62.692548 |       -62.692548 |      0.0 |     0.0 |      0.0 |      0.0 | -0.010566 |      0.0 |    -7.903964 |     0.753286 |      0.0 |      0.0 | 0.315045 | 0.315045 |  -7.828989 | -26.739968 |   28.123592 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 19.063443 | 20.421741 |        0.0 | 0.871808 | -195.586317 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 01:00:00 |  0.0 | -4.8 |             90.7 |       2.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -4.8 |    2.324615 |           2.324615 |                 4.272186 |                     3.108892 |             3.874873 | -2.347036 |   0.789683 |                       0.0 |                63.332342 |       -63.332342 |      0.0 |     0.0 |      0.0 |      0.0 | -0.010406 |      0.0 |    -8.906126 |     0.752632 |      0.0 |      0.0 | 0.337879 | 0.337879 |  -7.710448 | -23.484511 |   32.137384 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 19.411209 | 20.432147 |        0.0 | 0.879116 |    -187.776 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 02:00:00 |  0.0 | -5.6 |             89.7 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -5.6 |    2.003978 |           2.003978 |                 4.019239 |                     2.893939 |             3.605258 | -2.253312 |    2.48681 |                       0.0 |                64.162713 |       -64.162713 |      0.0 |     0.0 |      0.0 |      0.0 | -0.008797 |      0.0 |    -9.811456 |     0.751983 |      0.0 |      0.0 | 0.358185 | 0.358185 |  -6.517954 | -21.926379 |    35.71838 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 19.777905 | 20.440944 |        0.0 | 0.886424 |  -181.66281 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 03:00:00 |  0.0 | -6.2 |             91.3 |       3.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -6.2 |    2.404774 |           2.404774 |                 3.838328 |                     2.842032 |             3.504393 | -2.179954 |   3.246941 |                       0.0 |                65.855536 |       -65.855536 |      0.0 |     0.0 |      0.0 |      0.0 |   -0.0092 |      0.0 |   -10.039008 |     0.751338 |      0.0 |      0.0 | 0.359381 | 0.359381 |   -6.81692 | -22.449129 |   36.589487 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.146349 | 20.450144 |        0.0 | 0.893732 |  -176.30975 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 04:00:00 |  0.0 | -7.3 |             94.2 |       2.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -7.3 |    2.003978 |           2.003978 |                  3.52531 |                     2.620406 |             3.320842 | -2.115717 |    4.99765 |                       0.0 |                66.490605 |       -66.490605 |      0.0 |     0.0 |      0.0 |      0.0 | -0.008662 |      0.0 |   -11.054095 |     0.750699 |      0.0 |      0.0 | 0.382953 | 0.303795 |  -6.418232 |  -19.54519 |   40.527183 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.458806 | 20.458806 |  -7.344112 |  0.90104 |   -172.7074 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 05:00:00 |  0.0 | -8.3 |             93.2 |       2.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.3 |     1.84366 |            1.84366 |                  3.26061 |                     2.403099 |             3.038889 | -2.072489 |   8.654176 |                       0.0 |                67.536466 |       -67.536466 |      0.0 |     0.0 |      0.0 |      0.0 | -0.007475 |      0.0 |   -12.126375 |     0.750183 |      0.0 |      0.0 | 0.449974 |      0.0 |  -5.538842 | -18.940005 |   43.057619 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.466281 | 20.466281 | -41.747555 | 0.906978 | -172.761577 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 06:00:00 |  0.0 | -8.6 |             96.5 |       2.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                  4.41 |                           7.5 |             0.0 |   0.0 | -8.6 |    1.683342 |           1.683342 |                 3.184716 |                     2.263265 |             3.073251 | -2.073139 |  18.634133 |                       0.0 |                65.118509 |       -65.118509 |      0.0 |     0.0 |      0.0 |      0.0 |  -0.00903 |      0.0 |   -12.862916 |       0.7499 |      0.0 |      0.0 | 0.663239 |      0.0 |  -6.690723 | -20.007343 |   38.420442 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.475311 | 20.475311 | -61.533864 | 0.910254 |  -182.79571 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 07:00:00 |  0.0 | -8.5 |             95.9 |       1.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                  3.76 |                           7.4 |             0.0 |   0.0 | -8.5 |    1.122228 |           1.122228 |                 3.209838 |                     2.180922 |             3.078235 | -2.193549 |    23.8747 |                       0.0 |                59.988532 |       -59.988532 |      0.0 |     0.0 |      0.0 |      0.0 |  -0.00809 |      0.0 |   -13.315795 |     0.749663 |      0.0 |      0.0 | 0.790866 |      0.0 |  -5.994221 | -18.278661 |    35.71565 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.483401 | 20.483401 | -73.374814 | 0.912994 |  -198.07041 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 08:00:00 |  0.0 | -9.5 |             96.8 |       1.1 |                 0.355232 |              0.0 | 0.0 | 0.0 |                  2.76 |                      6.755232 |         1.10761 |   0.0 | -9.5 |     0.88175 |            0.88175 |                 2.966351 |                     2.038566 |             2.871427 | -2.376845 |  27.233906 |                  0.277515 |                 57.18978 |       -56.912264 |      0.0 |     0.0 |      0.0 |      0.0 | -0.006748 |      0.0 |   -14.135913 |     0.749447 |      0.0 |      0.0 | 0.886433 |      0.0 |  -4.999672 | -15.812174 |   36.100418 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.490148 | 20.490148 | -82.241326 | 0.915513 | -216.704316 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 09:00:00 |  0.0 | -9.2 |             94.3 |       1.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                  1.79 |                      6.755232 |       18.663203 |   0.0 | -9.2 |    0.801591 |           0.801591 |                 3.037615 |                     2.194304 |             2.864471 | -2.600452 |  27.387926 |                  4.679929 |                52.197338 |       -47.517409 |      0.0 |     0.0 |      0.0 |      0.0 | -0.005225 |      0.0 |   -13.241173 |     0.749243 |      0.0 |      0.0 | 0.918687 |      0.0 |   -3.87174 | -13.265335 |   30.380333 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.495374 | 20.495374 | -85.233733 |  0.91789 | -235.492242 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 10:00:00 |  0.0 | -6.9 |             96.7 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.88 |                      6.755232 |       56.267907 |   0.0 | -6.9 |    1.122228 |           1.122228 |                 3.636412 |                     2.927774 |             3.516411 | -2.825907 |  22.482917 |                   14.1208 |                 44.00809 |       -29.887291 |      0.0 |     0.0 |      0.0 |      0.0 | -0.005307 |      0.0 |    -9.665074 |     0.749043 |      0.0 |      0.0 | 0.842992 |      0.0 |  -3.932205 | -10.495016 |    15.46007 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 |  20.50068 |  20.50068 | -78.210886 | 0.920223 | -249.375159 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 11:00:00 |  0.0 | -4.0 |             97.4 |       1.3 |                      1.0 |              0.1 | 0.0 | 0.0 |                  0.39 |                      6.755232 |      107.168237 |   0.0 | -4.0 |    1.042069 |           1.042069 |                 4.539126 |                      3.99401 |             4.421109 | -2.992502 |  14.682709 |                 26.916882 |                37.055806 |       -10.138924 |      0.0 |     0.0 |      0.0 |      0.0 |  -0.00372 |      0.0 |    -5.682221 |     0.748835 |      0.0 |      0.0 | 0.697809 |      0.0 |  -2.756694 |  -6.169225 |    1.213005 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.504401 | 20.504401 | -64.741182 | 0.922663 | -255.457868 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 12:00:00 |  0.0 | -2.8 |             93.9 |       1.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                  0.14 |                      6.755232 |       91.473333 |   0.0 | -2.8 |    1.042069 |           1.042069 |                 4.967227 |                     4.326118 |             4.664226 | -3.065494 |   9.455994 |                 22.995716 |                31.873954 |        -8.878238 |      0.0 |     0.0 |      0.0 |      0.0 | -0.002945 |      0.0 |    -4.634871 |     0.748607 |      0.0 |      0.0 | 0.595531 |      0.0 |  -2.182308 |  -6.729041 |   -0.033111 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.507346 | 20.507346 | -55.252077 |  0.92534 | -256.313862 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 13:00:00 |  0.0 | -1.4 |             93.4 |       1.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       80.475535 |   0.0 | -1.4 |    1.122228 |           1.122228 |                 5.511494 |                     4.749947 |             5.147736 | -3.075766 |   5.301671 |                 20.250642 |                27.676623 |        -7.425981 |      0.0 |     0.0 |      0.0 |      0.0 | -0.003586 |      0.0 |    -3.397094 |     0.748363 |      0.0 |      0.0 | 0.508085 |      0.0 |  -2.657304 |  -7.580098 |   -2.811421 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.510932 | 20.510932 | -47.138985 | 0.928224 | -253.015532 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 14:00:00 |  0.0 | -0.4 |             93.3 |       2.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      6.755232 |       43.120591 |   0.0 | -0.4 |     1.84366 |            1.84366 |                 5.931876 |                     4.984779 |              5.53444 | -3.036186 |   2.506013 |                 10.862062 |                23.791116 |       -12.929054 |      0.0 |     0.0 |      0.0 |      0.0 | -0.006463 |      0.0 |     -2.75278 |       0.7481 |      0.0 |      0.0 | 0.443291 |      0.0 |  -4.788508 | -11.645922 |   -3.505376 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.517395 | 20.517395 | -41.127596 | 0.931326 | -246.921546 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 15:00:00 |  0.0 |  0.5 |             94.1 |       2.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.355232 |       10.320091 |   0.0 |  0.5 |    2.244455 |           2.244455 |                 6.334173 |                     5.225042 |             5.960456 | -2.963059 |   0.424693 |                    2.6025 |                19.926984 |       -17.324484 |      0.0 |     0.0 |      0.0 |      0.0 | -0.009767 |      0.0 |    -2.120771 |     0.747822 |      0.0 |      0.0 | 0.389507 |      0.0 |  -7.236762 | -14.653069 |   -4.565347 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.527162 | 20.527162 | -36.137556 | 0.934628 | -238.746238 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 16:00:00 |  0.1 |  1.0 |             94.6 |       3.1 |                 0.061898 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |        0.028102 | 0.094 |  1.0 |    2.484933 |           2.484933 |                 6.567919 |                     5.429219 |             6.213251 | -2.864955 |  -1.220485 |                   0.00709 |                18.521969 |       -18.514879 |    0.094 |  0.0235 |      0.0 |      0.0 | -0.011129 | 0.095635 |    -1.603567 |     0.747711 |      0.0 |      0.0 | 0.344614 |   0.0705 |  -8.246115 | -15.558634 |    -5.28987 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.632291 | 20.632291 | -25.431703 | 0.935949 | -228.925753 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 17:00:00 |  0.0 |  1.1 |             94.4 |       3.5 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |   0.0 |  1.1 |    2.805569 |           2.805569 |                 6.615571 |                     5.555809 |             6.245099 | -2.747109 |  -3.851975 |                       0.0 |                19.321816 |       -19.321816 |      0.0 |     0.0 |      0.0 |      0.0 | -0.010624 |      0.0 |    -1.291468 |     0.747368 |      0.0 |      0.0 | 0.271783 |      0.0 |  -7.872025 |  -15.51802 |   -4.068229 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.642915 | 20.642915 | -25.215449 | 0.940034 | -216.473779 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 18:00:00 |  0.3 |  1.0 |             96.9 |       4.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.282 |  1.0 |    3.527001 |           3.527001 |                 6.567919 |                     5.666507 |             6.364313 | -2.597685 |  -3.820631 |                       0.0 |                20.167827 |       -20.167827 |    0.282 |  0.0705 |      0.0 |      0.0 | -0.012669 | 0.286904 |    -1.023655 |     0.747575 |      0.0 |      0.0 |  0.25936 |   0.2115 |  -9.386911 | -15.467207 |   -4.686291 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.937584 | 20.937584 |  -4.440385 | 0.937567 | -204.053148 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 19:00:00 |  0.2 |  1.4 |             98.3 |       3.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  1.4 |    3.046047 |           3.046047 |                 6.760362 |                     5.910147 |             6.645436 | -2.448638 |  -9.271823 |                       0.0 |                19.346458 |       -19.346458 |    0.188 |  0.0282 |      0.0 |      0.0 | -0.012005 | 0.283107 |     -0.45013 |     0.747252 |      0.0 |      0.0 | 0.120321 | 0.120321 |  -8.895282 | -12.717194 |   -2.266018 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.098087 | 21.137589 |        0.0 | 0.941423 | -186.181325 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 20:00:00 |  0.4 |  1.6 |             98.7 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.376 |  1.6 |     4.56907 |            4.56907 |                 6.858436 |                       6.1078 |             6.769276 | -2.234176 | -11.218166 |                       0.0 |                19.815098 |       -19.815098 |    0.376 |  0.0376 |      0.0 |      0.0 | -0.014629 | 0.664621 |          0.0 |     0.746937 |      0.0 |      0.0 | 0.049932 | 0.049932 | -10.839285 | -14.896819 |   -5.921007 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.200024 | 21.528217 |        0.0 | 0.945204 | -166.363158 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 21:00:00 |  0.7 |  2.0 |             99.6 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.658 |  2.0 |    4.488911 |           4.488911 |                 7.058357 |                       6.1078 |             7.030123 | -1.996358 | -11.978147 |                       0.0 |                16.539253 |       -16.539253 |    0.658 |     0.0 |      0.0 |      0.0 | -0.020116 | 1.530508 |          0.0 |     0.746328 | 0.067722 | 0.067722 |      0.0 |      0.0 | -14.905456 | -18.364515 |  -16.730719 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 21.151463 | 22.206334 |        0.0 | 0.952539 | -145.785011 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 22:00:00 |  0.5 |  2.4 |             99.4 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  2.4 |    4.649229 |           4.649229 |                 7.263399 |                       6.1078 |             7.219819 |  -1.74942 | -10.496521 |                       0.0 |                 13.71239 |        -13.71239 |     0.47 |     0.0 |      0.0 |      0.0 | -0.024931 | 1.311864 |          0.0 |     0.745725 | 0.196481 | 0.196481 |      0.0 |      0.0 | -18.473108 |  -22.65304 |  -27.413757 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.978021 | 22.701265 |        0.0 |  0.95987 |  -126.68849 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-10 23:00:00 |  1.0 |  2.5 |             99.2 |       6.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.94 |  2.5 |    4.969866 |           4.969866 |                 7.315473 |                       6.1078 |             7.256949 | -1.520262 |  -9.121571 |                       0.0 |                13.067893 |       -13.067893 |     0.94 |     0.0 |      0.0 |      0.0 | -0.027164 |  2.73305 |          0.0 |     0.745124 | 0.275395 | 0.275395 |      0.0 |      0.0 |  -20.12751 | -24.879463 |   -31.93908 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.726413 | 23.668429 |        0.0 | 0.967221 | -108.966919 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 00:00:00 |  2.0 |  2.7 |             99.2 |       5.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  1.88 |  2.7 |    4.488911 |           4.488911 |                 7.420609 |                       6.1078 |             7.361244 | -1.307603 |  -7.845618 |                       0.0 |                11.601106 |       -11.601106 |     1.88 |     0.0 |      0.0 |      0.0 | -0.027338 | 5.903388 |          0.0 |     0.744525 | 0.339579 | 0.339579 |      0.0 |      0.0 |  -20.25663 | -24.792095 |  -33.447619 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 20.408649 | 25.575768 |        0.0 | 0.974611 |  -92.521301 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 01:00:00 |  2.1 |  3.2 |             99.6 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.974 |  3.2 |    4.168274 |           4.168274 |                 7.689299 |                       6.1078 |             7.658542 | -1.110256 |  -6.661534 |                       0.0 |                 7.733351 |        -7.733351 |    1.974 |     0.0 |      0.0 |      0.0 | -0.031933 | 7.346438 |          0.0 |      0.74393 | 0.478068 | 0.478068 |      0.0 |      0.0 | -23.661007 | -27.741565 |  -43.669221 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |      0.0 |      0.0 |     0.06 | 19.953682 | 27.581701 |        0.0 | 0.982008 |  -77.259767 | 71.446207 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |   0.0 |      0.0 |
    | 2010-12-11 02:00:00 |  1.8 |  3.9 |             99.8 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.692 |  3.9 |     4.56907 |            4.56907 |                  8.07986 |                       6.1078 |               8.0637 | -0.927117 |  -5.562703 |                       0.0 |                 2.415944 |        -2.415944 |    1.692 |     0.0 |      0.0 |      0.0 | -0.043255 | 4.318281 |          0.0 |     0.743342 | 0.697376 | 0.697376 |      0.0 |      0.0 | -32.050365 | -36.310997 |  -65.945417 |      1.0 |      0.3 | 1.738694 | 0.594177 |  0.0 |  0.0 | 0.0 |  0.0 | 0.594177 |      0.5 | 0.094177 |   0.0 |   0.0 |  0.0 | 0.006486 | 0.004762 |   0.0 |   0.0 |  0.0 | 0.011248 | 0.003124 |     0.06 | 19.285498 | 27.578262 |        0.0 | 0.989364 |  -63.097064 | 72.590725 | 0.493514 | 0.089415 |  0.0 |  0.0 | 0.0 |   0.0 | 0.003124 |
    | 2010-12-11 03:00:00 |  1.3 |  4.4 |             99.8 |       6.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 1.222 |  4.4 |    5.290502 |           5.290502 |                  8.36943 |                       6.1078 |             8.352691 | -0.757165 |  -4.542989 |                       0.0 |                -1.360594 |         1.360594 |    1.222 |     0.0 |      0.0 |      0.0 | -0.055801 |      0.0 |          0.0 |     0.742762 | 0.907645 | 0.907645 |      0.0 |      0.0 | -41.346526 | -46.045135 |  -88.752255 |      1.0 |      0.3 | 2.522558 | 0.864506 |  0.0 |  0.0 | 0.0 |  0.0 | 0.864506 |      0.5 | 0.364506 |   0.0 |   0.0 |  0.0 | 0.019234 | 0.027317 |   0.0 |   0.0 |  0.0 |  0.04655 | 0.012931 |     0.06 | 18.415038 | 26.333505 |        0.0 | 0.996672 |  -49.954075 | 74.248777 |  0.97428 | 0.426604 |  0.0 |  0.0 | 0.0 |   0.0 | 0.012931 |
    | 2010-12-11 04:00:00 |  0.5 |  4.8 |             99.0 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  4.8 |    5.611139 |           5.611139 |                 8.607627 |                       6.1078 |             8.521551 | -0.599449 |  -3.596693 |                       0.0 |                -4.039084 |         4.039084 |     0.47 |     0.0 |      0.0 |      0.0 |  -0.06294 |      0.0 |          0.0 |     0.742187 | 1.075386 | 1.075386 |      0.0 |      0.0 | -46.635995 | -52.693545 | -103.368624 |      1.0 |      0.3 | 2.011478 | 0.691195 |  0.0 |  0.0 | 0.0 |  0.0 | 0.691195 |      0.5 | 0.191195 |   0.0 |   0.0 |  0.0 | 0.031652 | 0.052067 |   0.0 |   0.0 |  0.0 | 0.083719 | 0.023255 |     0.06 | 17.381096 | 24.854967 |        0.0 |  1.00398 |  -37.757382 |  75.56906 | 1.442628 | 0.565732 |  0.0 |  0.0 | 0.0 |   0.0 | 0.023255 |
    | 2010-12-11 05:00:00 |  0.2 |  4.8 |             97.6 |       7.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  4.8 |    5.851616 |           5.851616 |                 8.607627 |                       6.1078 |             8.401044 | -0.453089 |  -2.718531 |                       0.0 |                -3.389325 |         3.389325 |    0.188 |     0.0 |      0.0 |      0.0 | -0.061894 |      0.0 |          0.0 |     0.741616 | 1.089397 | 1.089397 |      0.0 |      0.0 | -45.860636 | -54.540411 | -103.790373 |      1.0 |      0.3 | 1.749717 | 0.602606 |  0.0 |  0.0 | 0.0 |  0.0 | 0.602606 |      0.5 | 0.102606 |   0.0 |   0.0 |  0.0 |  0.04375 | 0.061415 |   0.0 |   0.0 |  0.0 | 0.105165 | 0.029213 |     0.06 | 16.332268 | 23.355144 |        0.0 | 1.011288 |   -26.43885 | 76.716172 | 1.898878 | 0.606923 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029213 |
    | 2010-12-11 06:00:00 |  0.5 |  5.1 |             97.4 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 |  0.47 |  5.1 |    5.771457 |           5.771457 |                 8.790172 |                       6.1078 |             8.561627 | -0.317266 |  -1.903597 |                       0.0 |                -5.587229 |         5.587229 |     0.47 |     0.0 |      0.0 |      0.0 |  -0.06548 |      0.0 |          0.0 |     0.741049 | 1.180205 | 1.180205 |      0.0 |      0.0 |  -48.51811 | -57.295089 | -111.400428 |      1.0 |      0.3 | 2.162425 | 0.746631 |  0.0 |  0.0 | 0.0 |  0.0 | 0.746631 |      0.5 | 0.246631 |   0.0 |   0.0 |  0.0 | 0.055536 | 0.072791 |   0.0 |   0.0 |  0.0 | 0.128327 | 0.035646 |     0.06 | 15.194544 | 21.728199 |        0.0 | 1.018596 |  -15.935253 | 78.131966 | 2.343342 | 0.780763 |  0.0 |  0.0 | 0.0 |   0.0 | 0.035646 |
    | 2010-12-11 07:00:00 |  0.2 |  5.1 |             96.3 |       7.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.41713 |             0.0 | 0.188 |  5.1 |    6.092093 |           6.092093 |                 8.790172 |                       6.1078 |             8.464935 | -0.191223 |  -1.147338 |                       0.0 |                -5.066448 |         5.066448 |    0.188 |     0.0 |      0.0 |      0.0 | -0.065772 |      0.0 |          0.0 |     0.740486 | 1.213277 | 1.213277 |      0.0 |      0.0 | -48.734564 | -59.911482 | -113.712494 |      1.0 |      0.3 | 1.928238 |  0.66742 |  0.0 |  0.0 | 0.0 |  0.0 |  0.66742 |      0.5 |  0.16742 |   0.0 |   0.0 |  0.0 | 0.067016 | 0.086064 |   0.0 |   0.0 |  0.0 |  0.15308 | 0.042522 |     0.06 | 14.023589 | 20.053733 |        0.0 | 1.025904 |   -6.187915 | 79.392784 | 2.776326 | 0.862119 |  0.0 |  0.0 | 0.0 |   0.0 | 0.042522 |
    | 2010-12-11 08:00:00 |  0.1 |  5.4 |             95.2 |       7.3 |                 0.347041 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        0.900734 | 0.094 |  5.4 |    5.851616 |           5.851616 |                 8.976116 |                       6.1078 |             8.545263 | -0.074255 |   -0.44553 |                  0.234256 |                -6.835277 |         7.069533 |    0.094 |     0.0 |      0.0 |      0.0 | -0.065786 |      0.0 |          0.0 |     0.739927 | 1.258132 | 1.258132 |      0.0 |      0.0 | -48.744743 | -61.357963 | -117.172238 | 0.999615 | 0.299769 | 1.899031 | 0.658457 |  0.0 |  0.0 | 0.0 |  0.0 | 0.658457 |      0.5 | 0.158457 |   0.0 |   0.0 |  0.0 | 0.078201 | 0.093697 |   0.0 |   0.0 |  0.0 | 0.171897 | 0.047749 |     0.06 | 12.807334 | 18.314488 |        0.0 | 1.033212 |    2.857615 | 80.633358 | 3.198125 |  0.92688 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047749 |
    | 2010-12-11 09:00:00 |  0.3 |  5.3 |             96.3 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        17.26751 | 0.282 |  5.3 |    5.691298 |           5.691298 |                 8.913754 |                       6.1078 |             8.583945 |       0.0 |        0.0 |                   4.50039 |                -6.596815 |        11.097205 |    0.282 |     0.0 |      0.0 |      0.0 | -0.065321 |      0.0 |          0.0 |     0.739372 | 1.275735 | 1.275735 |      0.0 |      0.0 | -48.400444 | -58.862205 | -118.359855 | 0.998456 | 0.299075 | 2.112808 | 0.733082 |  0.0 |  0.0 | 0.0 |  0.0 | 0.733082 |      0.5 | 0.233082 |   0.0 |   0.0 |  0.0 | 0.089096 | 0.103906 |   0.0 |   0.0 |  0.0 | 0.193002 | 0.053612 |     0.06 | 11.572728 | 16.549002 |        0.0 |  1.04052 |   11.457615 | 82.013083 | 3.609029 | 1.056056 |  0.0 |  0.0 | 0.0 |   0.0 | 0.053612 |
    | 2010-12-11 10:00:00 |  0.1 |  5.5 |             95.5 |       7.1 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                       7.40894 |        52.61761 | 0.094 |  5.5 |    5.691298 |           5.691298 |                 9.038862 |                       6.1078 |             8.632113 |       0.0 |        0.0 |                 13.742589 |                -7.748872 |        21.491462 |    0.094 |     0.0 |      0.0 |      0.0 | -0.066592 |      0.0 |          0.0 |     0.738821 | 1.421858 | 1.421858 |      0.0 |      0.0 | -49.341985 | -61.083421 | -131.916867 | 0.997298 | 0.298381 | 2.135439 | 0.741527 |  0.0 |  0.0 | 0.0 |  0.0 | 0.741527 |      0.5 | 0.241527 |   0.0 |   0.0 |  0.0 |  0.09971 | 0.117172 |   0.0 |   0.0 |  0.0 | 0.216882 | 0.060245 |     0.06 | 10.191716 | 14.574154 |        0.0 | 1.047828 |   20.057615 | 83.406996 | 4.009319 | 1.180411 |  0.0 |  0.0 | 0.0 |   0.0 | 0.060245 |
    | 2010-12-11 11:00:00 |  0.1 |  6.1 |             91.6 |       8.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.135633 | 0.094 |  6.1 |    7.134162 |           7.134162 |                 9.423511 |                       6.1078 |             8.631936 |       0.0 |        0.0 |                 20.973538 |               -12.026271 |        32.999809 |    0.094 |     0.0 |      0.0 |      0.0 | -0.080429 |      0.0 |          0.0 |     0.738275 | 1.880015 | 1.880015 |      0.0 |      0.0 |  -59.59435 | -81.829421 |  -174.42358 | 0.996139 | 0.297688 | 2.797257 | 0.972721 |  0.0 |  0.0 | 0.0 |  0.0 | 0.972721 |      0.5 | 0.472721 |   0.0 |   0.0 |  0.0 |  0.11005 | 0.141221 |   0.0 |   0.0 |  0.0 | 0.251271 | 0.069797 |     0.06 |  8.357571 | 11.951326 |        0.0 | 1.055136 |   28.657615 | 85.231532 | 4.399269 | 1.511911 |  0.0 |  0.0 | 0.0 |   0.0 | 0.069797 |
    | 2010-12-11 12:00:00 |  0.0 |  6.4 |             90.0 |       9.5 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       80.276118 |   0.0 |  6.4 |    7.615117 |           7.615117 |                 9.621181 |                       6.1078 |             8.659063 |       0.0 |        0.0 |                 21.053906 |               -13.533625 |         34.58753 |      0.0 |     0.0 |      0.0 |      0.0 | -0.085956 |      0.0 |          0.0 |     0.737731 | 2.037735 | 2.037735 |      0.0 |      0.0 | -63.690154 | -90.778796 |  -189.05648 |  0.99498 | 0.296996 | 2.934918 | 1.022435 |  0.0 |  0.0 | 0.0 |  0.0 | 1.022435 |      0.5 | 0.522435 |   0.0 |   0.0 |  0.0 | 0.120123 | 0.176682 |   0.0 |   0.0 |  0.0 | 0.296804 | 0.082446 |     0.06 |   6.36529 |  9.102364 |        0.0 | 1.062444 |   37.257615 | 87.144015 | 4.779146 | 1.857665 |  0.0 |  0.0 | 0.0 |   0.0 | 0.082446 |
    | 2010-12-11 13:00:00 |  0.1 |  6.4 |             85.7 |       7.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       75.835403 | 0.094 |  6.4 |    6.332571 |           6.332571 |                 9.621181 |                       6.1078 |             8.245352 |       0.0 |        0.0 |                 19.930141 |               -11.239491 |        31.169633 |    0.094 |     0.0 |      0.0 |      0.0 | -0.061599 |      0.0 |          0.0 |     0.737192 | 1.664809 | 1.664809 |      0.0 |      0.0 | -45.642119 | -77.645525 | -154.457277 | 0.993822 | 0.296304 | 2.490788 | 0.868999 |  0.0 |  0.0 | 0.0 |  0.0 | 0.868999 |      0.5 | 0.368999 |   0.0 |   0.0 |  0.0 | 0.129935 | 0.203288 |   0.0 |   0.0 |  0.0 | 0.333223 | 0.092562 |     0.06 |   4.73229 |  6.767175 |        0.0 | 1.069752 |   45.857615 | 88.765804 |  5.14921 | 2.023376 |  0.0 |  0.0 | 0.0 |   0.0 | 0.092562 |
    | 2010-12-11 14:00:00 |  0.0 |  6.5 |             82.3 |       9.6 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.0 |                       7.40894 |       42.227911 |   0.0 |  6.5 |    7.695276 |           7.695276 |                 9.687876 |                       6.1078 |             7.973122 |       0.0 |        0.0 |                  11.12044 |               -10.124562 |        21.245002 |      0.0 |     0.0 |      0.0 |      0.0 | -0.063414 |      0.0 |          0.0 |     0.736657 | 1.738166 | 1.738166 |      0.0 |      0.0 | -46.987277 | -93.030869 | -161.263149 | 0.992663 | 0.295614 | 2.508869 | 0.876496 |  0.0 |  0.0 | 0.0 |  0.0 | 0.876496 |      0.5 | 0.376496 |   0.0 |   0.0 |  0.0 | 0.139494 | 0.220136 |   0.0 |   0.0 |  0.0 | 0.359631 | 0.099897 |     0.06 |  3.022182 |  4.321721 |        0.0 |  1.07706 |   54.457615 | 90.398177 | 5.509716 | 2.179736 |  0.0 |  0.0 | 0.0 |   0.0 | 0.099897 |
    | 2010-12-11 15:00:00 |  0.1 |  6.3 |             82.5 |       9.1 |                      1.0 |              0.1 | 0.0 | 0.0 |                   0.1 |                       7.40894 |       12.613921 | 0.094 |  6.3 |     7.29448 |            7.29448 |                 9.554889 |                       6.1078 |             7.882784 |       0.0 |        0.0 |                  3.328499 |                -7.130286 |        10.458785 |    0.094 |     0.0 |      0.0 |      0.0 | -0.057639 |      0.0 |          0.0 |     0.736125 | 1.501388 | 1.501388 |      0.0 |      0.0 |  -42.70834 | -86.128361 | -139.295486 | 0.991504 | 0.294924 |  2.26962 | 0.793839 |  0.0 |  0.0 | 0.0 |  0.0 | 0.793839 |      0.5 | 0.293839 |   0.0 |   0.0 |  0.0 | 0.148807 | 0.231498 |   0.0 |   0.0 |  0.0 | 0.380304 |  0.10564 |     0.06 |  1.541077 |   2.20374 |        0.0 | 1.084368 |   63.057615 | 91.873958 | 5.860909 | 2.242077 |  0.0 |  0.0 | 0.0 |   0.0 |  0.10564 |
    | 2010-12-11 16:00:00 |  0.0 |  5.9 |             82.2 |       8.7 |                 0.053708 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |        0.021912 |   0.0 |  5.9 |    6.973844 |           6.973844 |                 9.293725 |                       6.1078 |             7.639442 |       0.0 |        0.0 |                  0.005794 |                -3.915548 |         3.921342 |      0.0 |     0.0 |      0.0 |      0.0 | -0.047871 |      0.0 |          0.0 |     0.735597 | 1.261344 | 1.261344 |      0.0 |      0.0 | -35.470286 | -77.633085 | -117.024713 | 0.990346 | 0.294235 | 1.842904 | 0.645046 |  0.0 |  0.0 | 0.0 |  0.0 | 0.645046 |      0.5 | 0.145046 |   0.0 |   0.0 |  0.0 | 0.157878 |  0.23017 |   0.0 |   0.0 |  0.0 | 0.388049 | 0.107791 |     0.06 |  0.285809 |  0.408707 |        0.0 | 1.091676 |   71.657615 | 93.071815 | 6.203031 | 2.156953 |  0.0 |  0.0 | 0.0 |   0.0 | 0.107791 |
    | 2010-12-11 17:00:00 |  0.0 |  5.5 |             76.5 |       8.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.5 |    7.134162 |           7.134162 |                 9.038862 |                       6.1078 |              6.91473 |       0.0 |        0.0 |                       0.0 |                 2.340465 |        -2.340465 |      0.0 |     0.0 |     0.06 | 0.021724 | -0.025712 |      0.0 |          0.0 |     0.735073 | 0.975359 | 0.285809 |      0.0 |      0.0 | -19.051448 | -73.780626 |  -90.491608 | 0.989187 | 0.293547 | 0.434419 | 0.151944 |  0.0 |  0.0 | 0.0 |  0.0 | 0.151944 | 0.151944 |      0.0 |   0.0 |   0.0 |  0.0 | 0.162201 | 0.214376 |   0.0 |   0.0 |  0.0 | 0.376577 | 0.104605 |      0.0 |       0.0 |       0.0 |        0.0 | 1.098984 |   80.257615 | 93.332566 | 6.192774 | 1.942577 |  0.0 |  0.0 | 0.0 |   0.0 | 0.104605 |
    | 2010-12-11 18:00:00 |  0.0 |  5.1 |             81.7 |       7.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  5.1 |    6.332571 |           6.332571 |                 8.790172 |                          0.0 |              7.18157 |       0.0 |      -30.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.029422 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.983905 | 0.290421 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.159965 |  0.19307 |   0.0 |   0.0 |  0.0 | 0.353035 | 0.098065 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  119.457615 | 93.303144 |  6.03281 | 1.749507 |  0.0 |  0.0 | 0.0 |   0.0 | 0.098065 |
    | 2010-12-11 19:00:00 |  0.0 |  4.8 |             83.4 |       7.1 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.8 |    5.691298 |           5.691298 |                 8.607627 |                          0.0 |             7.178761 |       0.0 |      -28.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.025147 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.978866 | 0.287454 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.155833 | 0.173881 |   0.0 |   0.0 |  0.0 | 0.329714 | 0.091587 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  156.857615 | 93.277997 | 5.876977 | 1.575626 |  0.0 |  0.0 | 0.0 |   0.0 | 0.091587 |
    | 2010-12-11 20:00:00 |  0.0 |  4.9 |             82.8 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.130184 |           5.130184 |                 8.668101 |                          0.0 |             7.177187 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.025203 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.973747 | 0.284455 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.151807 | 0.156599 |   0.0 |   0.0 |  0.0 | 0.308407 | 0.085668 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  194.857615 | 93.252793 |  5.72517 | 1.419027 |  0.0 |  0.0 | 0.0 |   0.0 | 0.085668 |
    | 2010-12-11 21:00:00 |  0.0 |  4.9 |             85.8 |       7.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  4.9 |    5.771457 |           5.771457 |                 8.668101 |                          0.0 |              7.43723 |       0.0 |      -29.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.021597 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.968627 | 0.281471 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.147886 | 0.141035 |   0.0 |   0.0 |  0.0 | 0.288921 | 0.080256 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  232.857615 | 93.231196 | 5.577284 | 1.277992 |  0.0 |  0.0 | 0.0 |   0.0 | 0.080256 |
    | 2010-12-11 22:00:00 |  0.1 |  4.7 |             87.6 |       7.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  4.7 |    5.611139 |           5.611139 |                 8.547525 |                          0.0 |             7.487632 |       0.0 |      -28.2 |                       0.0 |                      0.0 |              0.0 |    0.034 |     0.0 | 0.050942 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.963669 | 0.278597 |    0.034 | 0.011424 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011424 | 0.011424 |      0.0 |   0.0 |   0.0 |  0.0 | 0.144214 | 0.127018 |   0.0 |   0.0 |  0.0 | 0.271232 | 0.075342 | 0.009058 |       0.0 |       0.0 |        0.0 |      nan |  269.657615 | 93.253772 | 5.444493 | 1.150974 |  0.0 |  0.0 | 0.0 |   0.0 | 0.075342 |
    | 2010-12-11 23:00:00 |  0.1 |  3.8 |             85.6 |       6.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.8 |     5.45082 |            5.45082 |                 8.023017 |                          0.0 |             6.867703 |       0.0 |      -22.8 |                       0.0 |                      0.0 |              0.0 | 0.043058 |     0.0 | 0.055793 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.959438 | 0.276157 | 0.043058 | 0.014371 |  0.0 |  0.0 | 0.0 |  0.0 | 0.014371 | 0.014371 |      0.0 |   0.0 |   0.0 |  0.0 | 0.140822 | 0.114394 |   0.0 |   0.0 |  0.0 | 0.255216 | 0.070893 | 0.004207 |       0.0 |       0.0 |        0.0 |      nan |  301.057615 | 93.282459 | 5.318041 |  1.03658 |  0.0 |  0.0 | 0.0 |   0.0 | 0.070893 |
    | 2010-12-12 00:00:00 |  0.0 |  3.9 |             85.3 |       6.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.9 |    5.130184 |           5.130184 |                  8.07986 |                          0.0 |             6.892121 |       0.0 |      -23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.004207 |  0.01859 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.955127 |  0.27368 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.13737 | 0.103024 |   0.0 |   0.0 |  0.0 | 0.240394 | 0.066776 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  333.057615 | 93.263869 | 5.180672 | 0.933556 |  0.0 |  0.0 | 0.0 |   0.0 | 0.066776 |
    | 2010-12-12 01:00:00 |  0.0 |  3.7 |             86.2 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.7 |    5.050025 |           5.050025 |                 7.966527 |                          0.0 |             6.867146 |       0.0 |      -22.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.018522 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.950977 | 0.271307 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.133821 | 0.092785 |   0.0 |   0.0 |  0.0 | 0.226606 | 0.062946 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  363.857615 | 93.245347 | 5.046851 | 0.840771 |  0.0 |  0.0 | 0.0 |   0.0 | 0.062946 |
    | 2010-12-12 02:00:00 |  0.1 |  3.5 |             89.3 |       5.7 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 | 0.094 |  3.5 |     4.56907 |            4.56907 |                 7.854597 |                          0.0 |             7.014155 |       0.0 |      -21.0 |                       0.0 |                      0.0 |              0.0 |    0.034 |     0.0 |  0.03356 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.946989 | 0.269037 |    0.034 | 0.011125 |  0.0 |  0.0 | 0.0 |  0.0 | 0.011125 | 0.011125 |      0.0 |   0.0 |   0.0 |  0.0 | 0.130509 | 0.083563 |   0.0 |   0.0 |  0.0 | 0.214072 | 0.059464 |  0.02644 |       0.0 |       0.0 |        0.0 |      nan |  393.457615 | 93.268222 | 4.927466 | 0.757208 |  0.0 |  0.0 | 0.0 |   0.0 | 0.059464 |
    | 2010-12-12 03:00:00 |  0.0 |  3.3 |             91.4 |       5.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |    4.168274 |           4.168274 |                 7.744054 |                          0.0 |             7.078066 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.023813 |      0.0 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.943163 | 0.266867 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.127281 | 0.075258 |   0.0 |   0.0 |  0.0 | 0.202539 | 0.056261 | 0.002628 |       0.0 |       0.0 |        0.0 |      nan |  421.857615 | 93.268222 | 4.800186 |  0.68195 |  0.0 |  0.0 | 0.0 |   0.0 | 0.056261 |
    | 2010-12-12 04:00:00 |  0.0 |  3.3 |             90.9 |       4.6 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.3 |     3.68732 |            3.68732 |                 7.744054 |                          0.0 |             7.039346 |       0.0 |      -19.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 | 0.002628 |  0.00878 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.939337 | 0.264706 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.123993 | 0.067778 |   0.0 |   0.0 |  0.0 | 0.191771 |  0.05327 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  450.257615 | 93.259442 | 4.676193 | 0.614172 |  0.0 |  0.0 | 0.0 |   0.0 |  0.05327 |
    | 2010-12-12 05:00:00 |  0.0 |  3.0 |             89.2 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  3.0 |    3.927797 |           3.927797 |                 7.580812 |                          0.0 |             6.762084 |       0.0 |      -18.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.012215 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.935753 |  0.26269 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.12079 | 0.061042 |   0.0 |   0.0 |  0.0 | 0.181832 | 0.050509 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  476.857615 | 93.247227 | 4.555403 | 0.553131 |  0.0 |  0.0 | 0.0 |   0.0 | 0.050509 |
    | 2010-12-12 06:00:00 |  0.0 |  1.9 |             85.6 |       6.3 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.9 |    5.050025 |           5.050025 |                 7.007901 |                          0.0 |             5.998763 |       0.0 |      -11.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.017456 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.933058 | 0.261179 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.11767 | 0.054975 |   0.0 |   0.0 |  0.0 | 0.172645 | 0.047957 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  496.857615 | 93.229771 | 4.437733 | 0.498156 |  0.0 |  0.0 | 0.0 |   0.0 | 0.047957 |
    | 2010-12-12 07:00:00 |  0.0 |  1.2 |             86.3 |       5.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.400749 |             0.0 |   0.0 |  1.2 |    4.649229 |           4.649229 |                 6.663527 |                          0.0 |             5.750624 |       0.0 |       -7.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.015363 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.930929 | 0.259989 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.11463 | 0.049511 |   0.0 |   0.0 |  0.0 | 0.164141 | 0.045595 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  512.657615 | 93.214408 | 4.323102 | 0.448645 |  0.0 |  0.0 | 0.0 |   0.0 | 0.045595 |
    | 2010-12-12 08:00:00 |  0.0 |  0.8 |             91.2 |       6.1 |                 0.339738 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        0.894145 |   0.0 |  0.8 |    4.889707 |           4.889707 |                 6.473522 |                          0.0 |             5.903852 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.009666 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.929124 | 0.258981 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.111669 |  0.04459 |   0.0 |   0.0 |  0.0 |  0.15626 | 0.043405 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  526.057615 | 93.204742 | 4.211433 | 0.404055 |  0.0 |  0.0 | 0.0 |   0.0 | 0.043405 |
    | 2010-12-12 09:00:00 |  0.0 |  0.8 |             87.5 |       5.9 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |        17.71254 |   0.0 |  0.8 |    4.729388 |           4.729388 |                 6.473522 |                          0.0 |             5.664332 |       0.0 |       -4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.017184 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.927319 | 0.257976 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.108785 | 0.040158 |   0.0 |   0.0 |  0.0 | 0.148943 | 0.041373 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  539.457615 | 93.187558 | 4.102648 | 0.363896 |  0.0 |  0.0 | 0.0 |   0.0 | 0.041373 |
    | 2010-12-12 10:00:00 |  0.1 |  0.5 |             84.5 |       6.4 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       54.302585 | 0.094 |  0.5 |    5.130184 |           5.130184 |                 6.334173 |                     5.484759 |             5.352376 |       0.0 |   3.989026 |                  5.556421 |                27.251025 |       -21.694604 |    0.034 | 0.01275 |      0.0 |      0.0 |   0.00321 | 0.006889 |    -1.465853 |     0.897677 |      0.0 |      0.0 | 0.000099 | 0.000099 |   2.378474 | -20.068003 |    4.005075 | 0.926697 |  0.25763 | 0.015768 | 0.004993 |  0.0 |  0.0 | 0.0 |  0.0 | 0.004993 | 0.004993 |      0.0 |   0.0 |   0.0 |  0.0 |  0.10604 | 0.036167 |   0.0 |   0.0 |  0.0 | 0.142207 | 0.039502 |     0.06 |  0.010587 |  0.015023 |        0.0 |  0.00743 |  544.068589 | 93.198333 | 4.001601 | 0.327729 |  0.0 |  0.0 | 0.0 |   0.0 | 0.039502 |
    | 2010-12-12 11:00:00 |  0.0 |  0.3 |             76.7 |       6.8 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.504146 |   0.0 |  0.3 |     5.45082 |            5.45082 |                  6.24275 |                     5.263686 |             4.788189 |       0.0 |   5.507593 |                  8.523311 |                29.958696 |       -21.435386 |      0.0 |     0.0 |      0.0 |      0.0 |  0.012109 |      0.0 |    -2.021526 |     0.895425 |      0.0 |      0.0 | 0.000112 | 0.000112 |   8.972394 | -24.889809 |     5.51797 | 0.926281 | 0.257399 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.103365 | 0.032573 |   0.0 |   0.0 |  0.0 | 0.135937 |  0.03776 |     0.06 |  0.002075 |  0.002913 |        0.0 | 0.014738 |  547.160996 | 93.198333 | 3.898236 | 0.295157 |  0.0 |  0.0 | 0.0 |   0.0 |  0.03776 |
    | 2010-12-12 12:00:00 |  0.0 | -0.1 |             74.8 |       7.0 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       81.665294 |   0.0 | -0.1 |    5.611139 |           5.611139 |                 6.063393 |                      5.07808 |             4.535418 |       0.0 |   6.828335 |                  8.721431 |                31.460569 |       -22.739138 |      0.0 |     0.0 |     0.06 | 0.010695 |  0.002913 |      0.0 |    -2.504219 |     0.893205 |      0.0 |      0.0 | 0.000027 | 0.000027 |  10.484764 | -26.393086 |    6.830815 | 0.926042 | 0.257266 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.100695 | 0.029335 |   0.0 |   0.0 |  0.0 |  0.13003 | 0.036119 |      0.0 |       0.0 |       0.0 |        0.0 | 0.022046 |  548.932661 | 93.187638 | 3.797541 | 0.265821 |  0.0 |  0.0 | 0.0 |   0.0 | 0.036119 |
    | 2010-12-12 13:00:00 |  0.0 | -0.4 |             69.7 |       6.3 |                      1.0 |              0.0 | 0.0 | 0.0 |                   0.1 |                      7.393446 |       77.078959 |   0.0 | -0.4 |    5.050025 |           5.050025 |                 5.931876 |                          0.0 |             4.134518 |       0.0 |        2.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.046734 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.925207 | 0.256802 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.098094 |  0.02642 |   0.0 |   0.0 |  0.0 | 0.124513 | 0.034587 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  555.132661 | 93.140904 | 3.699448 | 0.239402 |  0.0 |  0.0 | 0.0 |   0.0 | 0.034587 |
    | 2010-12-12 14:00:00 |  0.0 | -0.6 |             66.9 |       7.3 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.3 |                      7.393446 |       60.254042 |   0.0 | -0.6 |    5.851616 |           5.851616 |                 5.845604 |                          0.0 |             3.910709 |       0.0 |        3.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.047231 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.924533 | 0.256428 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 |  0.09556 | 0.023794 |   0.0 |   0.0 |  0.0 | 0.119354 | 0.033154 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  560.132661 | 93.093673 | 3.603888 | 0.215608 |  0.0 |  0.0 | 0.0 |   0.0 | 0.033154 |
    | 2010-12-12 15:00:00 |  0.0 | -0.8 |             64.4 |       7.5 |                      1.0 |              0.2 | 0.0 | 0.0 |                   0.4 |                      7.393446 |       14.837301 |   0.0 | -0.8 |    6.011934 |           6.011934 |                 5.760442 |                          0.0 |             3.709725 |       0.0 |        4.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 |  0.04262 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.924021 | 0.256144 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.093092 | 0.021429 |   0.0 |   0.0 |  0.0 |  0.11452 | 0.031811 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  563.932661 | 93.051053 | 3.510796 | 0.194179 |  0.0 |  0.0 | 0.0 |   0.0 | 0.031811 |
    | 2010-12-12 16:00:00 |  0.0 | -1.3 |             65.4 |       7.0 |                 0.046405 |              0.2 | 0.0 | 0.0 |                   0.6 |                      7.386143 |        0.168856 |   0.0 | -1.3 |    5.611139 |           5.611139 |                 5.552313 |                          0.0 |             3.631213 |       0.0 |        7.8 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 |  0.03666 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.923913 | 0.256085 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.090687 | 0.019299 |   0.0 |   0.0 |  0.0 | 0.109986 | 0.030552 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  564.732661 | 93.014393 | 3.420109 |  0.17488 |  0.0 |  0.0 | 0.0 |   0.0 | 0.030552 |
    | 2010-12-12 17:00:00 |  0.0 | -1.6 |             67.6 |       6.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -1.6 |    4.809547 |           4.809547 |                 5.430651 |                          0.0 |              3.67112 |       0.0 |        9.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.031633 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.924048 | 0.256159 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.088344 | 0.017381 |   0.0 |   0.0 |  0.0 | 0.105725 | 0.029368 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  563.732661 |  92.98276 | 3.331765 | 0.157499 |  0.0 |  0.0 | 0.0 |   0.0 | 0.029368 |
    | 2010-12-12 18:00:00 |  0.0 | -2.0 |             70.2 |       4.8 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.0 |    3.847638 |           3.847638 |                 5.272101 |                          0.0 |             3.701015 |       0.0 |       12.0 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.025662 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.924506 | 0.256413 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.086062 | 0.015654 |   0.0 |   0.0 |  0.0 | 0.101716 | 0.028254 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  560.332661 | 92.957098 | 3.245703 | 0.141845 |  0.0 |  0.0 | 0.0 |   0.0 | 0.028254 |
    | 2010-12-12 19:00:00 |  0.0 | -2.1 |             70.4 |       4.2 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.1 |    3.366683 |           3.366683 |                 5.233109 |                          0.0 |             3.684109 |       0.0 |       12.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.023754 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.925045 | 0.256712 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.083839 | 0.014098 |   0.0 |   0.0 |  0.0 | 0.097937 | 0.027205 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  556.332661 | 92.933345 | 3.161863 | 0.127747 |  0.0 |  0.0 | 0.0 |   0.0 | 0.027205 |
    | 2010-12-12 20:00:00 |  0.0 | -2.2 |             69.5 |       4.9 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.2 |    3.927797 |           3.927797 |                 5.194372 |                          0.0 |             3.610088 |       0.0 |       13.2 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.026193 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.925665 | 0.257057 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.081674 | 0.012697 |   0.0 |   0.0 |  0.0 |  0.09437 | 0.026214 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  551.732661 | 92.907151 |  3.08019 | 0.115051 |  0.0 |  0.0 | 0.0 |   0.0 | 0.026214 |
    | 2010-12-12 21:00:00 |  0.0 | -2.6 |             71.6 |       4.0 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -2.6 |    3.206365 |           3.206365 |                 5.041946 |                          0.0 |             3.610033 |       0.0 |       15.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.021428 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.926608 | 0.257581 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.079564 | 0.011435 |   0.0 |   0.0 |  0.0 | 0.090999 | 0.025277 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  544.732661 | 92.885723 | 3.000626 | 0.103616 |  0.0 |  0.0 | 0.0 |   0.0 | 0.025277 |
    | 2010-12-12 22:00:00 |  0.0 | -3.1 |             74.5 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.1 |     2.72541 |            2.72541 |                 4.856984 |                          0.0 |             3.618453 |       0.0 |       18.6 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.016826 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.927955 |  0.25833 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.077509 | 0.010298 |   0.0 |   0.0 |  0.0 | 0.087807 | 0.024391 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  534.732661 | 92.868897 | 2.923117 | 0.093318 |  0.0 |  0.0 | 0.0 |   0.0 | 0.024391 |
    | 2010-12-12 23:00:00 |  0.0 | -3.9 |             80.2 |       3.4 |                      0.0 |              0.0 | 0.0 | 0.0 |                   0.6 |                      7.386143 |             0.0 |   0.0 | -3.9 |     2.72541 |            2.72541 |                  4.57351 |                          0.0 |             3.667955 |       0.0 |       23.4 |                       0.0 |                      0.0 |              0.0 |      0.0 |     0.0 |      0.0 | 0.011894 |       0.0 |      0.0 |          nan |          nan |      0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 | 0.929949 | 0.259442 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   0.0 |   0.0 |  0.0 | 0.075507 | 0.009275 |   0.0 |   0.0 |  0.0 | 0.084781 |  0.02355 |      0.0 |       0.0 |       0.0 |        0.0 |      nan |  519.932661 | 92.857003 |  2.84761 | 0.084043 |  0.0 |  0.0 | 0.0 |   0.0 |  0.02355 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_winter_ktschnee))
0.0
"""
# import...
# ...from site-packages
import numpy

# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import aetinterfaces
from hydpy.interfaces import radiationinterfaces
from hydpy.interfaces import soilinterfaces

# ...from lland
from hydpy.models.lland import lland_model
from hydpy.models.lland import lland_masks
from hydpy.models.lland.lland_constants import *


class Model(
    lland_model.Main_RadiationModel_V1,
    lland_model.Main_RadiationModel_V4,
    lland_model.Main_AETModel_V1B,
    lland_model.Main_SoilModel_V1,
    lland_model.Sub_TempModel_V1,
    lland_model.Sub_PrecipModel_V1,
    lland_model.Sub_IntercModel_V1,
    lland_model.Sub_SoilWaterModel_V1,
    lland_model.Sub_SnowCoverModel_V1,
    lland_model.Sub_SnowAlbedoModel_V1,
):
    """|lland_knauf.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="L-Knauf",
        description="adoption of LARSIM with Knauf-based snow modelling",
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = (lland_model.Pick_QZ_V1,)
    RECEIVER_METHODS = ()
    INTERFACE_METHODS = (
        lland_model.Get_Temperature_V1,
        lland_model.Get_MeanTemperature_V1,
        lland_model.Get_Precipitation_V1,
        lland_model.Get_InterceptedWater_V1,
        lland_model.Get_SoilWater_V1,
        lland_model.Get_SnowCover_V1,
        lland_model.Get_SnowAlbedo_V1,
    )
    ADD_METHODS = (
        lland_model.Return_NetLongwaveRadiationSnow_V1,
        lland_model.Return_EnergyGainSnowSurface_V1,
        lland_model.Return_SaturationVapourPressure_V1,
        lland_model.Return_NetRadiation_V1,
        lland_model.Return_WSensSnow_V1,
        lland_model.Return_WLatSnow_V1,
        lland_model.Return_WSurf_V1,
        lland_model.Return_TempS_V1,
        lland_model.Return_WG_V1,
        lland_model.Return_BackwardEulerError_V1,
        lland_model.Return_ESnow_V1,
        lland_model.Return_TempSSurface_V1,
        lland_model.Return_SG_V1,
        lland_model.Calc_BoWa_Default_V1,
        lland_model.Calc_BoWa_SoilModel_V1,
        lland_model.Calc_EvI_Inzp_AETModel_V1,
        lland_model.Calc_EvB_AETModel_V1,
    )
    RUN_METHODS = (
        lland_model.Process_RadiationModel_V1,
        lland_model.Calc_PossibleSunshineDuration_V1,
        lland_model.Calc_SunshineDuration_V1,
        lland_model.Calc_GlobalRadiation_V1,
        lland_model.Calc_QZH_V1,
        lland_model.Update_LoggedSunshineDuration_V1,
        lland_model.Calc_DailySunshineDuration_V1,
        lland_model.Update_LoggedPossibleSunshineDuration_V1,
        lland_model.Calc_DailyPossibleSunshineDuration_V1,
        lland_model.Calc_NKor_V1,
        lland_model.Calc_TKor_V1,
        lland_model.Calc_WindSpeed2m_V1,
        lland_model.Calc_ReducedWindSpeed2m_V1,
        lland_model.Calc_SaturationVapourPressure_V1,
        lland_model.Calc_ActualVapourPressure_V1,
        lland_model.Calc_NBes_Inzp_V1,
        lland_model.Calc_SNRatio_V1,
        lland_model.Calc_SBes_V1,
        lland_model.Calc_WATS_V1,
        lland_model.Calc_WaDa_WAeS_V1,
        lland_model.Calc_WNied_ESnow_V1,
        lland_model.Calc_TempS_V1,
        lland_model.Update_TauS_V1,
        lland_model.Calc_ActualAlbedo_V1,
        lland_model.Calc_NetShortwaveRadiationSnow_V1,
        lland_model.Calc_RLAtm_V1,
        lland_model.Calc_TZ_V1,
        lland_model.Calc_WG_V1,
        lland_model.Update_ESnow_V1,
        lland_model.Calc_SchmPot_V2,
        lland_model.Calc_Schm_WATS_V1,
        lland_model.Calc_GefrPot_V1,
        lland_model.Calc_Gefr_WATS_V1,
        lland_model.Calc_EvS_WAeS_WATS_V1,
        lland_model.Update_WaDa_WAeS_V1,
        lland_model.Update_ESnow_V2,
        lland_model.Calc_EvI_Inzp_V1,
        lland_model.Calc_EvB_V1,
        lland_model.Update_EBdn_V1,
        lland_model.Calc_SFF_V1,
        lland_model.Calc_FVG_V1,
        lland_model.Calc_QKap_V1,
        lland_model.Calc_QBB_V1,
        lland_model.Calc_QIB1_V1,
        lland_model.Calc_QIB2_V1,
        lland_model.Calc_QDB_V1,
        lland_model.Update_QDB_V1,
        lland_model.Calc_BoWa_V1,
        lland_model.Calc_QBGZ_V1,
        lland_model.Calc_QIGZ1_V1,
        lland_model.Calc_QIGZ2_V1,
        lland_model.Calc_QDGZ_V1,
        lland_model.Calc_QBGA_SBG_QBGZ_QDGZ_V1,
        lland_model.Calc_QIGA1_SIG1_V1,
        lland_model.Calc_QIGA2_SIG2_V1,
        lland_model.Calc_QDGZ1_QDGZ2_V1,
        lland_model.Calc_QDGA1_SDG1_V1,
        lland_model.Calc_QDGA2_SDG2_V1,
        lland_model.Calc_QAH_V1,
        lland_model.Calc_QA_V1,
    )
    OUTLET_METHODS = (lland_model.Pass_QA_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1, soilinterfaces.SoilModel_V1)
    SUBMODELS = (lland_model.PegasusESnow, lland_model.PegasusTempSSurface)

    idx_hru = modeltools.Idx_HRU()
    radiationmodel = modeltools.SubmodelProperty(
        radiationinterfaces.RadiationModel_V1, radiationinterfaces.RadiationModel_V4
    )
    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)
    soilmodel = modeltools.SubmodelProperty(soilinterfaces.SoilModel_V1, optional=True)

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \sum_{t=t0}^{t1} \Bigg(
            NKor_t
            + QZH_t
            - \bigg( \sum_{k=1}^{nhru} fhru^k \cdot \Big( EvI_t^k + EvS_t^k + EvB_t^k \Big) \bigg)
            - QAH_t
            \Bigg)
            + \sum_{k=1}^{nhru} fhru^k \cdot \bigg(
            \Big( Inzp_{t0}^k - Inzp_{t1}^k \Big)
            + \Big( WAeS_{t0}^{k} - WAeS_{t1}^{k} \Big)
            + \Big( BoWa_{t0}^{k} - BoWa_{t1}^{k} \Big)
            \bigg) -
            \bigg(
            \Big( SDG2_{t0} - SDG2_{t1} \Big)
            + \Big( SDG1_{t0} - SDG1_{t1} \Big)
            + \Big( SIG2_{t0} - SIG2_{t1} \Big)
            + \Big( SIG1_{t0} - SIG1_{t1} \Big)
            + \Big( SBG_{t0} - SBG_{t1} \Big)
            \bigg)

        The returned error should usually be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.  The only
        exception we are aware of is the "generation" of additional water when the
        base flow storage cannot meet the water demand required for the calculated
        capillary rise (see :ref:`lland_dd_acker_qkap_negq-false`).

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |lland_knauf| for some examples.
        """
        control = self.parameters.control
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        idxs_water = numpy.isin(control.lnk.values, [WASSER, FLUSS, SEE])
        idxs_land = numpy.invert(idxs_water)
        idxs_soil = numpy.invert(
            numpy.isin(control.lnk.values, [VERS, WASSER, FLUSS, SEE])
        )
        return (
            numpy.sum(fluxes.nkor.evalseries * control.fhru)
            + numpy.sum(fluxes.qzh.evalseries)
            - numpy.sum(fluxes.evi.evalseries * control.fhru)
            - numpy.sum((fluxes.evs.evalseries * control.fhru)[:, idxs_land])
            - numpy.sum((fluxes.evb.evalseries * control.fhru)[:, idxs_soil])
            - numpy.sum(fluxes.qah.evalseries)
            - numpy.sum(((last.inzp - first["inzp"]) * control.fhru)[idxs_land])
            - numpy.sum(((last.waes - first["waes"]) * control.fhru)[idxs_land])
            - numpy.sum(((last.bowa - first["bowa"]) * control.fhru)[idxs_soil])
            - (last.sdg1 - first["sdg1"])
            - (last.sdg2 - first["sdg2"])
            - (last.sig1 - first["sig1"])
            - (last.sig2 - first["sig2"])
            - (last.sbg - first["sbg"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |lland_knauf|."""

    CLASSES = lland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
